C This file is part of the ESP-r system.
C Copyright Energy Systems Research Unit, University of
C Strathclyde, Glasgow, Scotland, 2001-.

C ESP-r is free software.  You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation
C (version 2 or later).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.


C This file contains a collection of low-level ESP-r routines that are
C applicable to all versions of ESP-r (that is, X11, GTK and noX). Code
C pretaining to version-specific configurations should be placed in the
C libGTK, libX11 and libNONGtk files.

C  st2name: Given `string' swap blanks & wildcards to _ return as `name'.
C  st2file: Given `string' strip blanks & wildcards and return as `name'.
C  hasblanks: Given a string returns true if it contains blanks or wildcards.
C  backslashit: Given `string' swap / to \ : return as `name'.
C  iprevblnk: Given a string, return position of blank just before ipos.
C  inextblnk: Given a string, return position of blank just after ipos.
C  icutstr: Given a string, cuts icut characters from position ipos.

C  EASKXORGTKF: Generic file edit/browse for X11 or GTK using easkf calls.
C  iCountWords: Counts the number of space/tab/comma-separated
C           words in a string.

C  isadll:  Checks if module is being used as a dll (silent running).
C  isunix:  Checks if machine type is Unix or NT/W7.
C  usrhome: Returns user's home directory.
C  usrname: Returns user's login name.
C  usrdir:  Returns user's current directory.
C  esppid:  Find current process number.
C  tstamp:  Date stamp with message.
C  dstamp:  Get date stamp in the form: Fri Jan 23 09:34:31 1998.
C  comparedate: is passed two date strings (generated by call to dstamp)
C    and returns 1 if first date is more current, 0 if the same, otherwise -1.
C  getsecs: Get computer clock seconds.

C  iEGetArrW: integer function that breaks a string into an array.
C  iEGetArrW_2500: Same as iEGetArrW but for strings 2500 ch long. 

C  DAYCLK:  Print day, month, day no. and time based on the julian day & time.
C  FDROOT:  Given a file name see if it contains a path.
C  CMDTOROOT Parse command line file name and return model root name.
C  EFOPSEQ: Open a sequential file with existance flag & path check.
C  EFOPRAN: Open a random access file with existance flag & path check.
C  ADDPATH: Return file name appended onto the path and logical concat.
C  GETTOKENS checks a string, returning nb of tokens and array of tokens.
C  C2FSTR:  Convert c function returned string to fortran format.
C  EKPAGE:  Maps key characters, pages & array indexs in long display lists.
C  KEYIND:  Decodes EMENU index and returns the array index of the item.
C  EPMENSV: saves menu definitions (common block PMENU).
C  EPMENRC: recovers menu definitions (common block PMENU) from PMENUSV.
C  EPAGE:   Screen control: page without waiting.
C  EPAGEW:  Screen control: Wait before paging.
C  EPWAIT:  Screen control: Wait without paging.
C  EPAGEND: Screen control: Page then close window if open.

C  ang3vtx: Get angle between three vertex.
C  CLOSE3D: Calc min dist between two lines in 3D return dist and closest points.
C  CROW:    Function returns shortest dist between two points P(3) and Q(3).
C  CROWXYZ: Function returns shortest dist between two points Px,Py,Pz and Qx,Qy,Qz.
C  UVXYZ:   Returns Unit vector Ux,Uy,Uz from two points Px,Py,Pz and Qx,Qy,Qz.
C  UVAB:    Returns Unit vector U of vector A.
C  PLNDIS:  Finds distance DIST from a point (x,y,z) to a plane (eq EQN).
C  PLNDANG: Finds dihedral angel between two planes given their equations.
C  AVER:    Returns the centre of gravity of an polygon array.
C  CROSS:   Performs a cross-product on vectors A() & B() returning in C().
C  CROSS2:  Performs a cross-product on vectors passing parameters as ax,ay,az etc.
C  DOT3(a,b,product) Return dot product of two vectors a & b.
C  ZEROS:   Clear a 4x4 array prior to doing vieweing transforms.
C  REQUAL:  Checks tolerance between two real numbers. Function returning .TRUE. or .FALSE. 
C  ECLOSE:  Checks tolerance between two real numbers.
C  ECLOSE3: Checks tolerance between two real vectors (3 numbers).
C  ESIND:   Function returning SIN of angle where angle is given in degrees.
C  ECOSD:   Function returning COS of angle where angle is given in degrees.
C  ETAND:   Function returning TAN of angle where angle is given in degrees.
C  IFAX :   Integer function returning the integer part of it's argument.
C  EAZALT:  Computes the solar azimuth & altitude angles at current time.
C  AGNXYZ:  Given solar azi & elev return viewing coords @ 1000m.
C  ORTTRN:  Multiplies a point (XM,YM,ZM) by the transform matrix
C           TMAT to return the point XO,YO,Z0.
C  VECTRN:  Transforms a vector VECIN by the 4x4 (homogeneious) matrix
C           TMAT and returns the vector VECOUT.
C  VECPLN:  Returns the point of intersection X,Y,Z between a line
C           defined by X1,Y1,Z1 & X2,Y2,Z2 and a plane defined in PEQN.
C  HMATMUL: Multiplies the homogenous (4x4) matrices A by B returning C.
C  HREVMAT: Takes the homogenous perspective transformation PER and
C           returns it's inverse REP making use of CROUT.
C  CROUT:   Inverts a nonsymetric square matrix A (order N), returning
C           the matrix B and IERR =-1 if matrix is singular.
C  DPACC:   Provides double precision accumulation of inner products for
C           CROUT in the form SUM(+,-)SUM(+,-)AB.
C  EYEMAT:  Provides transform eyepoint - viewpoint....

C  INTSTR: Converts integer into string (10 char) w/o leading blanks.
C  RELSTR: Converts a real into a string (12 char) w/o leading blanks.
C  REL10STR: Converts a real into a string (10 char) w/o leading blanks.
C  REL16STR: Converts a real into a string (16 char) w/o leading blanks.
C  EXPSTR: Converts a exponential into a string (10 char) w/o leading blanks.
C  ARLIST: takes a real array (rlist) and builds a packed string.
C  AILIST: takes an int array (ilist) and builds a packed string.
C  ASLIST: takes an string array (list*24) and builds a packed string.
C  ASLIST2 takes the range (inst to inil) items of an string array (list)
C  ASFLIST: takes an string array (list*48) and builds a packed string.
C  STRIPC: Strips comments from a ASCII file str (124 char) & returns data.
C  LSTRIPC: Strips comments from a ASCII file str (248 char) & returns data.
C  STRIPC400 strips comments from a ASCII file (400 char long) string and returns the data.
C  STRIPC1K strips comments from a ASCII file (1000 char long) string and returns the data.
C  STRIPC2500 strips comments from a ASCII file (2500 char long) string and returns the data.
C  CHARCH: Routine to check a string for a specific number of data items.
C  NOYES:  INTEGER FUNCTION to read the answer Y,y,1,N,n,0 to a question.
C  IFIRST: Function returning ASCII value for 1st char in a string ISTR.

C  EASKPER: Provides interface to specification of a period of days.
C eAskPerYear: Provides interface to specification of a multiyear period of days.
C easkPerGen: Handles specification of both single-year and multi-year
C           simulation periods.
C  EDAY:   Returns the year day number when passed the day of month & month.
C  EDAYR: returns the day and month numbers from the day-of-year.
C  EWEEKD: returns the day of the week given the day of month, month
C          and year as integers.
C  EDAYCH: Checks for errors in the users specification of the day & month.
C  DATTIM: returns UNIX time via a string in the form : 16 Sep 73 14:23.
C  STDATE: Takes the day of year and returns two descriptive strings:
C          DESCR takes the form '12 Jan' & DESCR1 takes the form 'Fri 12 Jan'.
C  ESTIME: Takes an integer timestep and returns two string descriptions:
C          DESCRH in the form '12h28' & DESCRD which takes the form of 12.46.
C  EDTIME: takes an real time and returns two string descriptions:
C          DESCRH in the form '12h28' and DESCRD which takes the form of 12.46,
C  EPERSTR: creates three strings representing the start and
C           stop time of a diary period.

C  EGETW:  Finds first word after pos k in a string (' ' ',' '|' or tab separated).
C  EGETP:  Finds first phrase after pos k in a string (delimeter separated).
C  EGETDQ:  Finds first quoted phrase after pos k in a string (delimeter separated).
C  EGETWI: As EGETW for an integer with range checking & error messages.
C  EGETWR: As EGETW for a real with range checking & error messages.
C  EGETWRA: Recovers (IRVA) reals of real array (RVA) from an ASCII file.
C  EGETWIA: Recovers (IRVA) int of array (IVA) from an ASCII file.
C EGETAGWIA recovers (IRVA) integers of integer array (IVA) from a string
C   TSTR (from position K) and if TSTR does not hold all of the array then
C   it continues reading from an ASCII file (unit IAF) reading as
C   many lines as necessary to recover the data.
C EGETAGWRA recovers (IRVA) reals of real array (RVA) from a string
C   TSTR (from position K) and if TSTR does not hold all of the array then
C   it continues reading from an ASCII file (unit IAF) reading as
C   many lines as necessary to recover the data.
C EGETAGWRA1K recovers (IRVA) reals of real array (RVA) from a string
C   TSTR (from position K) and if TSTR does not hold all of the array then
C   it continues reading from an ASCII file (unit IAF) into 1K buffer reading as
C   many lines as necessary to recover the data.
C EGETAGWSA recovers (ISVA) words into string array (SVA) from a string
C   TSTR (from position K) and if TSTR does not hold all of the array then
C   it continues reading from an ASCII file (unit IAF) reading as
C   many lines as necessary to recover the data
C EGETAGWPA recovers (ISVA) phrases into string array (SVA) from a string
C   TSTR (from position K) and if TSTR does not hold all of the array then
C   it continues reading from an ASCII file (unit IAF) reading as
C   many lines as necessary to recover the data
C  EGETRM: Returns the remainder (RSTR with no leading blanks) from
C          a text string (TSTR) after position k.
C EGETXMLTAGE: gets first XML tag from a STRING of characters.
C EGETWXML: gets first WORD from a STRING of characters.
C EGETWIXML: gets a word from a STRING and converts to an integer.
C EGETWRXML: gets a word from a STRING and converts to a real.
C EGETEQDQXML: gets tag & quoted PHRASE from a STRING.
C EGETRMXML: gets remainder of a string until a '<'.


C  ERPFREE: Is used to close any file.
C  EFDELET: Delete the current file opened under IUN and return ISTAT.
C  FPOPEN:  is used to open a file with a name.
C  FPRAND:  is used to open a file with a name for random access.

C  SITELL2S: Takes site lat & long and returns descrptive string.
C  SIGFIG:  Returns number to specified number of significant figures.
C  SIpre:   Returns suitable SI prefix for number supplied.
C  pronam:  Returns the characters of a string after the last
C           occurance of '/' or '\'
C DNOTZERO: Function returns non zero value with the same sign (dbl precision).
C ANOTZERO: Function returns non zero value with the same sign (sngl precision).

C  ASKTIM:  Enquire month and day and time (real for view).
C  SOLAIR:  Returns solair temperature.
C  LISTAS:  General read & display of an ascii file.
C  SDELIM:  Replaces blanks in a string A with alternative delimiter.
C  EDDISP:  As edisp with text separated with current delimiter.
C EDISP248 Displays a 248 char block of text (text or graphic).
C EDDISP248 Displays a 248 char block of text (text or graphic) with current delimiter.
C clrtextbuf: Clears the graphic text buffer common blocks.

C  UPDVIEW: Called from C to pass back updates to common VIEWPX & GFONT
C  WIREPK:  Called from 'C' upon a wireframe control button pick.
C  EPROMPT: Does nothing, for compatibility only.
C  ISEOF:   Checks for an EOF from a READ statement.

C ******************** st2name ********************
C Takes string and swap blanks & wildcards to _ : return as `name'.

      SUBROUTINE st2name(string,name)
      CHARACTER*(*) string,name
      character phrase*124

C Clear variables and get string lengths.
      name = ' '
      phrase = ' '
      ilname = LEN(name)
      illstr = max(1,lnblnk(string))

C Strip off any leading blanks from string, substitute `_'
C between words and for any ` \ / ~ & * ( ) ^ # < > ' ` " '.
C Stop copying if end of name reached.
      K=0
      DO 99 I=1,illstr
        IF(string(I:I).NE.' '.OR.K.GE.1)THEN
          if(ichar(string(I:I)).lt.32)goto 100
          if(I.gt.ilname)goto 100
          K=K+1
          if(string(I:I).eq.' ')then
            phrase(K:K)='_'
          elseif(string(I:I).eq.'/')then
            phrase(K:K)='_'
          elseif(string(I:I).eq.char(34))then
            phrase(K:K)='_'
          elseif(string(I:I).eq.char(39))then
            phrase(K:K)='_'
          elseif(string(I:I).eq.char(44))then
            phrase(K:K)='_'
          elseif(string(I:I).eq.char(91))then
            phrase(K:K)='_'
          elseif(string(I:I).eq.char(92))then
            phrase(K:K)='_'
          elseif(string(I:I).eq.char(93))then
            phrase(K:K)='_'
          elseif(string(I:I).eq.char(96))then
            phrase(K:K)='_'
          elseif(string(I:I).eq.'*')then
            phrase(K:K)='_'
          elseif(string(I:I).eq.'~')then
            phrase(K:K)='_'
          elseif(string(I:I).eq.'&')then
            phrase(K:K)='_'
          elseif(string(I:I).eq.'(')then
            phrase(K:K)='_'
          elseif(string(I:I).eq.')')then
            phrase(K:K)='_'
          elseif(string(I:I).eq.'>')then
            phrase(K:K)='_'
          elseif(string(I:I).eq.'<')then
            phrase(K:K)='_'
          elseif(string(I:I).eq.'^')then
            phrase(K:K)='_'
          elseif(string(I:I).eq.'#')then
            phrase(K:K)='_'
          else
            phrase(K:K)=string(I:I)
          endif
        ENDIF
 99   CONTINUE
100   continue
      LN=max(1,lnblnk(phrase))
      write(name,'(a)') phrase(1:LN)

      return
      end

C ******************** hashname ********************
C Takes an entity name (inname), a desired length and returns shortened name
C via removing - _ or spaces if required.

      subroutine hashname(lenin,lendesire,inname,outname,ier)
      integer lendesire,lenin          ! requested length & input length
      character (len=lenin) :: inname  ! entity name (of lenin size)
      character (len=lendesire) :: outname ! shortened name (of lendesire size)
      integer ier                      ! error is non-zero
      character phrase*124             ! working string

      illstr = max(1,lnblnk(inname))   ! get length to loop over
      phrase = '  '                    ! clear phrase
      if(lnblnk(inname).le.lendesire)then
        write(outname(1:lendesire),'(a)') inname(1:lendesire) ! no need to shorten
        ier=0
      else
        K=0
        DO I=1,illstr
          if(inname(I:I).NE.' '.OR.K.GE.1)then
            if(K.gt.lendesire)goto 100
            if(inname(I:I).eq.' '.or.inname(I:I).eq.'-'.or.
     &         inname(I:I).eq.'_')then
              continue   ! strip out blank or - or _
            else
              K=K+1
              phrase(K:K)=inname(I:I)
            endif
          endif
        ENDDO
  100   continue
        if(lnblnk(phrase).gt.lendesire)then
          ier=1; LN=lendesire                ! signal truncation
        else
          ier=0; LN=lnblnk(phrase)
        endif
        write(outname,'(a)') phrase(1:LN)    ! truncate phrase if needed
      endif

      return
      end      ! of hashname

C ********** st2file
C Given `string' strip blanks & wildcards and return as `name'.
C Useful to check file names.

      SUBROUTINE st2file(string,name)
      CHARACTER*(*) string,name
      character phrase*124

C Clear variables and get string lengths.
      name = ' '
      phrase = ' '
      ilname = LEN(name)
      illstr = max(1,lnblnk(string))

C Strip off any leading blanks from string, compact spaces and tabs
C between words and substitute _ for any ` & * ( ) ^ # < > ' ` " ' or commas.
C Stop copying if end of name reached.
      K=0
      DO 99 I=1,illstr
        if(ichar(string(I:I)).eq.0)goto 99
        if(ichar(string(I:I)).eq.9)goto 99
        if(ichar(string(I:I)).eq.32)goto 99
        if(ichar(string(I:I)).lt.32)goto 100
        if(I.gt.ilname)goto 100

        K=K+1
        if(string(I:I).eq.char(34))then
          phrase(K:K)='_'
        elseif(string(I:I).eq.char(39))then
          phrase(K:K)='_'
        elseif(string(I:I).eq.char(44))then
          phrase(K:K)='_'
        elseif(string(I:I).eq.char(91))then
          phrase(K:K)='_'
        elseif(string(I:I).eq.char(93))then
          phrase(K:K)='_'
        elseif(string(I:I).eq.char(96))then
          phrase(K:K)='_'
        elseif(string(I:I).eq.'*')then
          phrase(K:K)='_'
        elseif(string(I:I).eq.'~')then
          phrase(K:K)='_'
        elseif(string(I:I).eq.'&')then
          phrase(K:K)='_'
        elseif(string(I:I).eq.'(')then
          phrase(K:K)='_'
        elseif(string(I:I).eq.')')then
          phrase(K:K)='_'
        elseif(string(I:I).eq.'>')then
          phrase(K:K)='_'
        elseif(string(I:I).eq.'<')then
          phrase(K:K)='_'
        elseif(string(I:I).eq.'^')then
          phrase(K:K)='_'
        elseif(string(I:I).eq.'#')then
          phrase(K:K)='_'
        else
          phrase(K:K)=string(I:I)
        endif
 99   CONTINUE
100   continue
      LN=max(1,lnblnk(phrase))
      write(name,'(a)') phrase(1:LN)

      return
      end


C ********** hasblanks
C Given `string' returns true if it includes blanks & wildcards.

      SUBROUTINE hasblanks(string,blanks)
      CHARACTER*(*) string
      logical blanks

C Clear variables and get string lengths.
      illstr = max(1,lnblnk(string))
      blanks=.false.

C Strip off any leading blanks from string, compact spaces and tabs
C between words and substitute _ for any ` & * ( ) ^ # < > ' ` " ' or commas.
C Stop copying if end of name reached.
      K=0
      DO 99 I=1,illstr
        if(ichar(string(I:I)).eq.0)goto 99
        if(ichar(string(I:I)).eq.9)goto 99
        if(ichar(string(I:I)).eq.32)goto 99
        if(ichar(string(I:I)).lt.32)goto 100

        K=K+1
        if(string(I:I).eq.char(34))then
          blanks=.true.
        elseif(string(I:I).eq.char(39))then
          blanks=.true.
        elseif(string(I:I).eq.char(44))then
          blanks=.true.
        elseif(string(I:I).eq.char(91))then
          blanks=.true.
        elseif(string(I:I).eq.char(93))then
          blanks=.true.
        elseif(string(I:I).eq.char(96))then
          blanks=.true.
        elseif(string(I:I).eq.'*')then
          blanks=.true.
        elseif(string(I:I).eq.'~')then
          blanks=.true.
        elseif(string(I:I).eq.'&')then
          blanks=.true.
        elseif(string(I:I).eq.'(')then
          blanks=.true.
        elseif(string(I:I).eq.')')then
          blanks=.true.
        elseif(string(I:I).eq.'>')then
          blanks=.true.
        elseif(string(I:I).eq.'<')then
          blanks=.true.
        elseif(string(I:I).eq.'^')then
          blanks=.true.
        elseif(string(I:I).eq.'#')then
          blanks=.true.
        else
          continue
        endif
 99   CONTINUE
100   continue

      return
      end


C ******************** backslashit ********************
C Given `string' swap / to \ : return as `name'.

      SUBROUTINE backslashit(string,name)
      CHARACTER*(*) string,name
      character phrase*144,bs*1

C Clear variables and get string lengths.
      name = ' '
      phrase = ' '
      ilname = LEN(name)
      illstr = max(1,lnblnk(string))

C Make up a \ character.
      bs = char(92)

C Strip off any leading blanks from string, substitute \ for /
C between words and dtop copying if end of name reached.
      K=0
      DO 99 I=1,illstr
        IF(string(I:I).NE.' '.OR.K.GE.1)THEN
          if(ichar(string(I:I)).lt.32)goto 100
          if(I.gt.ilname)goto 100
          K=K+1
          if(string(I:I).eq.'/')then
            phrase(K:K)=bs
          else
            phrase(K:K)=string(I:I)
          endif
        ENDIF
 99   CONTINUE
100   continue
      LN=max(1,lnblnk(phrase))
      write(name,'(a)') phrase(1:LN)

      return
      end

C ******************** iprevblnk ********************
C Given 'string', return position of blank just before ipos.

      function iprevblnk(string,ipos)
      character*(*) string
      character a*1,b*1
      integer right

C right is the defined length of string.
      right=len(string)
      lnb=lnblnk(string)

C If requested position beyond the length of the string return the
C declared length of the string.
      if(ipos.gt.right)then
        iprevblnk=right
        return
      endif

C If requrested position is beyond lnblnk then return lnblnk.
      if(ipos.gt.lnb)then
        iprevblnk=lnb
        return
      endif

C Set position for checking to ipos
      iprevblnk=ipos
      right=ipos
 42   continue
      right=right-1
      if(right.le.0)then

C If position zero reached set to one and return.
        iprevblnk = 1
        return
      else

C If the current character is non blank then continue leftwards.
        a=string(right:right)
        if(right.gt.1)then
          b=string(right-1:right-1)
          if(ichar(a).gt.32)then
            goto 42
          elseif(ichar(a).eq.32.or.ichar(a).eq.9)then

C If the current character is a blank and the character to its
C left is also a blank carry on stepping leftwards, otherwise return.
            if(ichar(b).eq.32.or.ichar(b).eq.9)then
              goto 42
            else
              iprevblnk = right
              return
            endif
          endif
        else

C If the current position is 1 do not bother with position zero.
          if(ichar(a).gt.32)then
            goto 42
          elseif(ichar(a).eq.32.or.ichar(a).eq.9)then
            iprevblnk = right
            return
          endif
        endif
      endif
      end

C ******************** inextblnk ********************
C Given 'string', return position of blank just after ipos.
C (or the end of the string if that happens first).

      function inextblnk(string,ipos)
      character*(*) string
      character a*1,b*1
      integer right

C right is the defined length of string.
      right=len(string)
      lnb=lnblnk(string)

C If requested position beyond the length of the string return the
C declared length of the string.
      if(ipos.gt.right)then
        inextblnk=right
        return
      endif

C If requrested position is beyond lnblnk then return lnblnk.
      if(ipos.gt.lnb)then
        inextblnk=lnb
        return
      endif

C Set position for checking to ipos
      inextblnk=ipos
      right=ipos
 42   continue
      right=right+1
      if(right.ge.lnb)then

C If position zero reached end of string set to lnb and return.
        inextblnk = lnb
        return
      else

C If the current character is non blank then continue rightwards.
        a=string(right:right)
        b=string(right+1:right+1)
        if(ichar(a).gt.32)then
          goto 42
        elseif(ichar(a).eq.32.or.ichar(a).eq.9)then

C If the current character is a blank and the character to its
C left is also a blank carry on stepping leftwards, otherwise return.
          if(ichar(b).eq.32.or.ichar(b).eq.9)then
            goto 42
          else
            inextblnk = right
            return
          endif
        endif
      endif
      end

C ******************** icutstr ********************
C Given 'string', cuts icut characters from position ipos
C and shifts latter portion of the string down icut chars. Returns
C zero if no error.

      function icutstr(string,ipos,icut)
      character*(*) string
      integer right

C right is the defined length of string.
      right=len(string)
      lnb=lnblnk(string)

C If requested text to cut is beyond the length of the string return
C with no action.
      if(ipos+icut.gt.right)then
        icutstr=1
        return
      endif

C If requested text to cut starts within the string, but carrys on
C beyond the lnblnk then shorten string to ipos.
      if(ipos.lt.lnb.and.ipos+icut.gt.lnb)then
        write(string,'(a)') string(1:ipos)
        icutstr=0
        return
      else
        write(string,'(2a)') string(1:ipos),string(ipos+icut:lnb)
        icutstr=0
        return
      endif
      end



C The following subroutines are used to open a user-specified file
C with a meaningful error message if this is not possible.

C << with a bit of work the 'e' version calls can do this
C << task as long as the path is temporarily cleared
C << this would simplify much of the logic.

C  FPOPEN  open an ascii file - use EFOPSEQ instead for model files.
C  FPRAND  open a binary file - use EFOPRAN instead for model files.

C File opening parameter conventions:
C  IUN   is the unit number on which the file is to be opened;
C  ISTAT   returns a status value as follows:-

C   >0     file was successfully opened
C   -1     input line was a ?
C   -2     input line was not recognisable as a file title,
C          and an error message has been output
C   -3     input line (file name)was 'NONE' NOTHING OPENED,RETURN

C   -300   file already exists (  IXIST  =2), and
C          an error message has been output
C   -301   file does not exist, but no error message has
C          been output (  IXIST=0  )

C other <0 error opening file, and an error message has been
C          output (value returned is minus the value returned in the
C          second argument of a call to   ERRSNS)

C MODE determines the access mode of the file as follows:-
C
C  1-3  read and write
C    4  append (write only on to end of file)

C   FOR FORMATTED I/O LET MODE -VE BE FOR UNFORMATTED I/O
C   -1 TO -4   AS ABOVE FOR UNFORMATTED SEQUENTIAL ACCESS

C   >4 <0   Length of direct access record

C IXIST determines whether the file is expected to exist:-
C   0  file is expected to exist; return with
C      ISTAT=-301   if it does not, but
C      do not output an error message.  This
C      can be used to test for existence of a default
C      file.

C   1  file is expected to exist; error if it does not
C   2  file is expected not to exist; error if it does
C   3  file used if it exists; zero length file created if not

C ******************** FPOPEN ********************
C Opens the file with standard parameter conventions for
C IUN, ISTAT, MODE, and IXIST;  FNARG is the file name.

        SUBROUTINE FPOPEN(IUN,ISTAT,MODE,IXIST,FNARG)
        CHARACTER*(*) FNARG
        LOGICAL        XST
        logical unixok

        lfn=max(1,LNBLNK(fnarg))

        call isunix(unixok)
        IF (FNARG(1:min(lfn,7)).eq.'UNKNOWN'
     &        .or.FNARG(1:min(lfn,2)).eq.'  ') THEN
          if(unixok)WRITE(6,*)' !+Files named UNKNOWN cannot be opened'
          ISTAT=-400
          RETURN
        ENDIF

C Check if file exists and its type
        INQUIRE (FILE=FNARG,EXIST=XST)

        IF (XST.AND.(IXIST.EQ.2)) THEN
            if(unixok)WRITE(6,*)' !+File already exists',fnarg
            ISTAT=-300
            RETURN
        ENDIF

        IF (.NOT.XST.AND.(IXIST.LE.1)) THEN

C File does not exist, set error and return.
          IF ( IXIST .EQ. 1 )THEN
            if(unixok)call edisp(6,' File doesn`t exist: ')
            if(unixok)call edisp(6,FNARG)
          ENDIF
          ISTAT=-301
          RETURN
        ENDIF

C Check mode:   mode 1-4,5-8 is sequential access
        IF ((MODE.LT.0).OR.(MODE.GT.10)) THEN

C Open file for direct access.
            IF (MODE.GT.0) THEN
              MODE = MODE - 10
              IF (XST) THEN

C ASCII RECORDS,4 CHAR PER WORD,RECL IN CHARS
                OPEN (IUN,FILE=fnarg(1:lfn),ACCESS='DIRECT',RECL=MODE*4,
     &                  FORM='FORMATTED',STATUS='OLD',IOSTAT=ISTAT)
              ELSE
                OPEN (IUN,FILE=fnarg(1:lfn),ACCESS='DIRECT',RECL=MODE*4,
     &                  FORM='FORMATTED',STATUS='NEW',IOSTAT=ISTAT)
              ENDIF
            ELSE

C BINARY UNFORMATTED D.A. FILE
C ON VAX/VMS  RECL=  -MODE WORDS PER RECORD
C ON UNIX     RECL=  -MODE*4  BYTES PER RECORD

              MODE= -MODE
              IF (XST) THEN
                OPEN (IUN,FILE=fnarg(1:lfn),ACCESS='DIRECT',RECL=MODE*4,
     &               FORM='UNFORMATTED',STATUS='OLD',IOSTAT=ISTAT)
              ELSE
                OPEN (IUN,FILE=fnarg(1:lfn),ACCESS='DIRECT',RECL=MODE*4,
     &               FORM='UNFORMATTED',STATUS='NEW',IOSTAT=ISTAT)
              ENDIF
            ENDIF
        ELSE

c Sequential access file.
C***V3.4  ADD READONLY FOR MODE=1 OR 5 FOR VAX SYSTEMS
C  THIS SHOULD BE REMOVED IF YOUR SYSTEM DOESNT SUPPORT 'READONLY'
         IF(MODE.EQ.1)THEN
C  MODE=1-4 FORMATTED (DEFAULT)
            IF (XST) THEN
              OPEN (IUN,FILE=fnarg(1:lfn),ACCESS='SEQUENTIAL',
     &              STATUS='OLD',IOSTAT=ISTAT)
            ELSE
              OPEN (IUN,FILE=fnarg(1:lfn),ACCESS='SEQUENTIAL',
     &              STATUS='NEW',IOSTAT=ISTAT)
            ENDIF
         ELSE If(MODE.LT.5) THEN
C READ & WRITE MODE=1-4 FORMATTED (DEFAULT)
            IF (XST) THEN
              OPEN (IUN,FILE=fnarg(1:lfn),ACCESS='SEQUENTIAL',
     &              STATUS='OLD',IOSTAT=ISTAT)
            ELSE
              OPEN (IUN,FILE=fnarg(1:lfn),ACCESS='SEQUENTIAL',
     &              STATUS='NEW',IOSTAT=ISTAT)
            ENDIF

C MODE =5  READONLY UNFORMATTED.
         ELSE IF (MODE .EQ. 5) THEN
           IF (XST) THEN
             OPEN (IUN,FILE=fnarg(1:lfn),FORM='UNFORMATTED',
     &             ACCESS='SEQUENTIAL',STATUS='OLD',IOSTAT=ISTAT)
           ELSE
             OPEN (IUN,FILE=fnarg(1:lfn),FORM='UNFORMATTED',
     &             ACCESS='SEQUENTIAL',STATUS='NEW',IOSTAT=ISTAT)
           ENDIF

         ELSE IF (MODE .LT. 9) THEN
c READ & WRITE Unformatted.
            IF (XST) THEN
              OPEN (IUN,FILE=fnarg(1:lfn),FORM='UNFORMATTED',
     &              ACCESS='SEQUENTIAL',STATUS='OLD',IOSTAT=ISTAT)
            ELSE
              OPEN (IUN,FILE=fnarg(1:lfn),FORM='UNFORMATTED',
     &              ACCESS='SEQUENTIAL',STATUS='NEW',IOSTAT=ISTAT)
            ENDIF
          endif
        ENDIF

C***3.5
      IF(ISTAT.EQ.0 .AND. .NOT. XST) ISTAT=1

      RETURN
      END

C ******************** FPRAND ********************
C Opens a random access file with name FNARG.
C  LENG determines the record length and type as follows:-
C       <0      -( LENG ) characters per record, ASCII records.

C       =0      128 words per record,   IMAGE   MODE RECORDS
C               (BUFFER COUNT IS FORCED TO 1 TO ENSURE WRITES
C               ARE PERFORMED IN ORDER REQUESTED)
C
C       >0      (LENG  ) words per record, ff BINARY   records
C IUN,ISTAT, and IXIST are the same as standard conventions.

        SUBROUTINE FPRAND(IUN,ISTAT,LENG,IXIST,FNARG)
        CHARACTER*(*) FNARG
        logical unixok

        lfn = lnblnk(FNARG)

        call isunix(unixok)
        IF (FNARG(1:min(lfn,7)).eq.'UNKNOWN') THEN
          if(unixok)WRITE(6,*)' !+Files named UNKNOWN cannot be opened'
          ISTAT=-400
          RETURN
        ENDIF

C Convert length to mode for FPOPEN.
        IF (LENG.EQ.0) MODE=-128
        IF (LENG.GT.0) MODE=-LENG
        IF (LENG.LT.0) MODE=10-LENG
C Open file.
        CALL FPOPEN (IUN,ISTAT,MODE,IXIST,FNARG)
        RETURN
        END


c ******************** ERPFREE ********************
C Used to close a file.
C IUN (integer) is the file unit number
C ISTAT (integer) is its status 0=OK, 1=if IUN was zero.

      SUBROUTINE ERPFREE(IUN,ISTAT)
      LOGICAL OPND
      IF(IUN.NE.0)THEN
        INQUIRE(IUN,OPENED=OPND)
        IF(OPND)CLOSE(IUN)
        ISTAT=0
      ELSE
        CALL USRMSG(' ',' Unable to free error channel! ','W')
        ISTAT=1
      ENDIF
      RETURN
      END

c ******************** EFDELET ********************
C Delete file IUN and return ISTAT for compatibility.
C IUN (integer) is the file unit number
C ISTAT (integer) is its status 0=OK, 1=if IUN was zero.

      SUBROUTINE EFDELET(IUN,ISTAT)
      LOGICAL OPND
      IF(IUN.NE.0)THEN
        INQUIRE(IUN,OPENED=OPND)
        IF(OPND)CLOSE(IUN,STATUS='DELETE')
        ISTAT=0
      ELSE
        CALL USRMSG(' ',' Unable to delete error channel! ','W')
        ISTAT=1
      ENDIF
      RETURN
      END

C ******************** EASKXORGTKF ********************
C Generic file edit/browse for X11 or GTK using easkf calls.
C The current implementation only opens .cfg files for existing models.
C If in GTK or text mode, it redirects to the previous implementation.

      SUBROUTINE EASKXORGTKF(FILEN,PROMP1,PROMP2,DFILE,RETFILE,ERMSG,
     &  IER,NHELP)

      integer lnblnk  ! function definition

C Parameters.
      CHARACTER*(*) FILEN   ! initial file name (never changed)
      CHARACTER*(*) PROMP1,PROMP2  ! prompts for dialogs
      CHARACTER*(*) DFILE   ! default file name
      CHARACTER*(*) RETFILE ! returned file name
      CHARACTER*(*) ERMSG   ! message in case of error
      integer ier           ! zero is ok, -3 is cancel
      integer NHELP         ! number of help lines

      COMMON/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER outs248*248
      character llitem*144     ! local string buffer
      character LTMP72*72      ! local string buffer
      character LTMP96*96      ! local string buffer
      character LTMP124*124    ! local string buffer
      integer iglib      ! for detecting GTK or X11
      INTEGER llt,lfn    ! length of the local string buffer
      integer ISTRW

C Depending on whether it is X11 or GTK or text setup dialogs. The
C X11 and text will work with 96 or 124 char string.
      iglib = igraphiclib()  ! find out if X11 or GTK or text support only.
      lfn=lnblnk(FILEN)
      if(lfn.lt.72)then
        write(LTMP72,'(a)') FILEN(1:lfn)   ! for X11 and text editing
      else
        write(LTMP72,'(a)') FILEN(1:72)    ! for X11 and text editing
      endif
      if(lfn.lt.96)then
        write(LTMP96,'(a)') FILEN(1:lfn)   ! for X11 and text editing
      else
        write(LTMP96,'(a)') FILEN(1:96)    ! for X11 and text editing
      endif
      if(lfn.lt.124)then
        write(LTMP124,'(a)') FILEN(1:lfn)  ! for X11 and text editing
      else
        write(LTMP124,'(a)') FILEN(1:124)  ! for X11 and text editing
      endif
      write(llitem,'(a)') FILEN(1:lfn)     ! for GTK dialog

C If in text or GTK mode echo FILEN.
      llt=lnblnk(llitem)
      if(iglib.eq.2.or.iglib.eq.3)then
        call edisp(iuout,'  ')
        write(outs248,'(2a)') 'Current file or folder:',llitem(1:llt)
        call edisp248(iuout,outs248,100)
      endif
      if(iglib.eq.1.or.iglib.eq.3)then

C Adjust dialog depending on initial length of file name. If shorter
C than 32 char the use shortest width.
        if(llt.lt.42)then
          ISTRW=72
          CALL EASKF(LTMP72,PROMP1,PROMP2,ISTRW,DFILE,ERMSG,IER,NHELP)
          write(llitem,'(a)') LTMP72(1:lnblnk(LTMP72))
        elseif(llt.lt.96)then
          ISTRW=96
          CALL EASKF(LTMP96,PROMP1,PROMP2,ISTRW,DFILE,ERMSG,IER,NHELP)
          write(llitem,'(a)') LTMP96(1:lnblnk(LTMP96))
        elseif(llt.ge.96.and.llt.lt.124)then
          ISTRW=124
          CALL EASKF(LTMP124,PROMP1,PROMP2,ISTRW,DFILE,ERMSG,IER,NHELP)
          write(llitem,'(a)') LTMP124(1:lnblnk(LTMP124))
        elseif(llt.ge.124.and.llt.le.144)then
          ISTRW=144
          CALL EASKF(llitem,PROMP1,PROMP2,ISTRW,DFILE,ERMSG,IER,NHELP)
        endif
      elseif(iglib.eq.2)then
        ISTRW=144
        CALL EASKF(llitem,PROMP1,PROMP2,ISTRW,DFILE,ERMSG,IER,NHELP)
      else
        ISTRW=96
        CALL EASKF(LTMP96,PROMP1,PROMP2,ISTRW,DFILE,ERMSG,IER,NHELP)
        write(llitem,'(a)') LTMP96(1:lnblnk(LTMP96))
      endif

C If user request cancel set RFILE ot FILEN.
      if(ier.eq.-3)then
        write(RETFILE,'(a)') FILEN(1:lnblnk(FILEN))
        return
      endif

C If ier is zero then set RFILE to llitem.
      if(ier.eq.0)then
        write(RETFILE,'(a)') llitem(1:lnblnk(llitem))
        return
      endif
      
      RETURN
      END


C ******************** EASKXFWIN ********************
C Generic file edit/browse for X11 only using a separate window for file selection.

      SUBROUTINE EASKXFWIN(FILEN,PROMP1,PROMP2,DFILE,RETFILE,ERMSG,
     &  IER,NHELP)

      integer lnblnk  ! function definition

C Parameters
      CHARACTER*(*) FILEN   ! initial file name (never changed)
      CHARACTER*(*) PROMP1,PROMP2  ! prompts for dialogs
      CHARACTER*(*) DFILE   ! default file name
      CHARACTER*(*) RETFILE ! returned file name
      CHARACTER*(*) ERMSG   ! message in case of error
      integer ier           ! zero is ok, -3 is cancel
      integer NHELP         ! number of help lines

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      common/SPAD/MMOD,LIMIT,LIMTTY  
      CHARACTER outs248*248
      character llitem*144     ! local string buffer
      character LTMP72*72      ! local string buffer
      character LTMP96*96      ! local string buffer
      character LTMP124*144    ! local string buffer
      integer iglib      ! for detecting GTK or X11
      INTEGER llt,lfn    ! length of the local string buffer
      integer ISTRW
      character file_return*144 ! temporary file name and path from dialog box
  
C Depending on whether it is X11 or GTK or text setup dialogs. The
C X11 and text will work with 96 or 124 char string.
      iglib = igraphiclib()  ! find out if X11 or GTK or text support only.
      lfn=lnblnk(FILEN)

C If in text or GTK mode echo FILEN.
      llt=lnblnk(llitem)
      if(iglib.ne.1.or.MMOD.ne.8)then
        CALL EASKXORGTKF(FILEN,PROMP1,PROMP2,DFILE,RETFILE,ERMSG,
     &  IER,NHELP)
        RETURN
      else !if in graphic mode and x11: open dialog in separate window

C Call popup file browser to select model cfg file. Note: prompts are
C filed in xdialogfilebox.
        call xdialogfilebox(file_return,argc,argv,ier)
        LTMP124=file_return
        write(llitem,'(a)') file_return(1:lnblnk(file_return))
      endif

C If user request cancel set RFILE ot FILEN.
      if(ier.eq.-3)then
        write(RETFILE,'(a)') FILEN(1:lnblnk(FILEN))
        return
      endif

C If ier is zero then set RFILE to llitem.
      if(ier.eq.0)then
        write(RETFILE,'(a)') llitem(1:lnblnk(llitem))
        return
      endif
      
      RETURN
      END

C ******************** bSPEqual ********************
C Determines if two SP numbers are within machine round-off error.
C The floating point computations in this function are
C based on the following paper:

C    Goldberg, David (1991). 'What every computer scientest should know
C    about floating point arithmetic", Computing Surveys, March.

      logical function bSPEqual(fFloat1, fFloat2)
      implicit none

C Passed floats
      real fFloat1, fFloat2

C Significand portions of passed floats
      real fSig1, fSig2

C Exponential portion of passed floats.
      integer iExp1, iExp2

C Upper and lower bounds for exponential root-search
      integer iExpLB, iExpUB
      
C Signs associated with passed floats
      integer iSign1, iSign2
      integer bit_size

      logical bExpFound

C Flag for initialization
      logical bInitialized
      data bInitialized/.false./
      save bInitialized

C min-exponent for system
      integer iMin_exp
      save iMin_exp

C Round-off error
      real fRoundoffError

      logical bClose

C Components for float:
      integer iFloatBasis
      integer iFloatPrecision
      save iFloatBasis
      save iFloatPrecision
      
      call eclose ( fFloat1, fFloat2, 1.0E-03, bClose )
      if ( .not. bclose ) then
        bSPEqual = .false.
        return
      endif

C Establish the minimum exponent distinguisable on the system
      if ( .not. bInitialized ) then

C Initialize base and precision parameters according to
C IEEE 754.
        iFloatBasis = 2
        
        select case  ( bit_size( iExp1 ) )

C Single precision
          case (32)

            iFloatPrecision = 24
            iMin_exp        = -126
           
C Single-extended precision
          case (43)

            iFloatPrecision = 32
            iMin_exp        = -1022

C Double precisision
          case (64)

            iFloatPrecision = 43
            iMin_exp        = -1022

C Double-extended precision
          case (79)

            iFloatPrecision = 79
            iMin_exp        = -16382

          case default

C Perhaps a better error message could go here?
            STOP "bSPEqual: System is not IEEE 754 conformant."

        end select

        bInitialized = .true.

      endif

C Establish signs for arguments, and convert to absolute avalues.

      if ( fFloat1 .LT. 0.0 ) then
        iSign1 = -1
        fFloat1 = fFloat1 * (-1.0)
      else
        iSign1 =  1
      endif

      if ( fFloat2 .LT. 0.0 ) then
        iSign2 = -1
        fFloat2 = fFloat2 * (-1.0)
      else
        iSign2 =  1
      endif 

C Determine the exponent of the first arguement.
      bExpFound = .false. 
      iExpLB = -1 * iMin_exp
      iExpUB =      iMin_exp + 1

      do while ( .not. bExpFound )

        iExp1 = int( float ( iExpLB + iExpUB ) / 2.0 )

        fSig1 = fFloat1 / 2.0 ** float ( iExp1 )

        if ( fSig1 * real ( iSign1 ) .lt. 1.0 ) then
          iExpLB = iExp1
        elseif ( fSig1 * real ( iSign1 ) .ge. 2.0 ) then
          iExpUB = iExp1
        else
          bExpFound = .true.
        endif 
      enddo

C Determine the exponent of second arguement.
      bExpFound = .false. 
      iExpLB = -1 * iMin_exp
      iExpUB =      iMin_exp + 1

      do while ( .not. bExpFound )

        iExp2 = int( float ( iExpLB + iExpUB ) / 2.0 )

        fSig2 = fFloat2 / 2.0 ** float ( iExp2 )

        if ( fSig2 * real ( iSign2 ) .lt. 1.0 ) then
          iExpLB = iExp2
        elseif ( fSig2 * real ( iSign2 ) .ge. 2.0 ) then
          iExpUB = iExp2
        else
          bExpFound = .true.
        endif 
      enddo

C Compute potential round-off error:
C     ( ( basis / 2 ) * basis ^ ( - precision ) ) * basis ^ exponent 
C (See Goldberg 1991).
      fRoundoffError =
     &    (   ( real ( iFloatBasis ) / 2.0 )
     &      * real ( iFloatBasis ) ** real ( -1 * iFloatPrecision )
     &     ) *
     &     real ( iFloatBasis ) ** real ( max ( iExp1, iExp2 ) )

C Finally, compare difference between arguements with potential
C round-off error. Report numbers as equal if round-off error
C exceeds difference.
      if ( ABS ( fFloat1 - fFloat2 ) .lt. fRoundoffError ) then
        bSPEqual = .true.
      else
        bSPEqual = .false.
      endif 

      return
      end


C ******************** DAYCLK ********************
C Print the day, month, day no. and time based on the day
C of the year IYD (integer) and clock time in hours TIME (real) on the
C output channel ITU (integer).

      SUBROUTINE DAYCLK(IYD,TIME,ITU)

      CHARACTER*5 DAY(4)
      CHARACTER*5 RMONTH(12)
      character outs*124

      DATA DAY/'st of','nd of','rd of','th of'/
      DATA RMONTH/' Jan.',' Feb.',' Mar.',' Apr.',' May.',' June',
     &' July',' Aug.',' Sept',' Oct.',' Nov.',' Dec.'/

      CALL EDAYR(IYD,ID,IM)
      IF(TIME.LE.24.0)goto  1
      TIME=TIME-24.0
      IYD=IYD+1
    1 IS=4
      IF(ID.EQ.1.OR.ID.EQ.21.OR.ID.EQ.31)IS=1
      IF(ID.EQ.2.OR.ID.EQ.22)IS=2
      IF(ID.EQ.3.OR.ID.EQ.23)IS=3
      IHR=IFAX(TIME)
      REM=(TIME-IHR)*60.0
      IMIN=INT(REM)
      IF(IMIN.LT.10.0)THEN
        WRITE(outs,4)IYD,ID,DAY(IS),RMONTH(IM),IHR,IMIN
    4   FORMAT(' Day No. = ',I3,' (i.e. ',I2,A5,A5,'),  Time ',
     &         I2,':0',I1,' Hours')
        call edisp(itu,outs)
      ELSE
        WRITE(outs,3)IYD,ID,DAY(IS),RMONTH(IM),IHR,IMIN
    3   FORMAT(' Day No. = ',I3,' (i.e. ',I2,A5,A5,'),  Time ',
     &         I2,':',I2,' Hours')
        call edisp(itu,outs)
      ENDIF
      RETURN
      END

C ******************** DAYCLKSTR ********************
C Return string (36 char) of the day, month, day no. and time based on
C the day of the year IYD (integer) and clock time in hours TIME (real).

      SUBROUTINE DAYCLKSTR(IYD,TIME,clickstr)

      CHARACTER*5 DAY(4)
      CHARACTER*5 RMONTH(12)
C      character outs*124
      character clickstr*36

      DATA DAY/'st of','nd of','rd of','th of'/
      DATA RMONTH/' Jan.',' Feb.',' Mar.',' Apr.',' May.',' June',
     &' July',' Aug.',' Sept',' Oct.',' Nov.',' Dec.'/

      CALL EDAYR(IYD,ID,IM)
      IF(TIME.LE.24.0)goto  1
      TIME=TIME-24.0
      IYD=IYD+1
    1 IS=4
      IF(ID.EQ.1.OR.ID.EQ.21.OR.ID.EQ.31)IS=1
      IF(ID.EQ.2.OR.ID.EQ.22)IS=2
      IF(ID.EQ.3.OR.ID.EQ.23)IS=3
      IHR=IFAX(TIME)
      REM=(TIME-IHR)*60.0
      IMIN=INT(REM)
      IF(IMIN.LT.10.0)THEN
        WRITE(clickstr,4)IYD,ID,DAY(IS),RMONTH(IM),IHR,IMIN
    4   FORMAT('Day ',I3,' (',I2,A5,A5,'), Time ',
     &         I2,':0',I1,' ')
      ELSE
        WRITE(clickstr,3)IYD,ID,DAY(IS),RMONTH(IM),IHR,IMIN
    3   FORMAT('Day ',I3,' (',I2,A5,A5,'), Time ',
     &         I2,':',I2,' ')
      ENDIF
      RETURN
      END

C ******************** FDROOT ********************
C Given a file name (fstring) see if it contains a (path) and a (filen).
C Note: mingw compilers require fstring to be declaired length (not *(*).

      SUBROUTINE fdroot(fstring,path,filen)
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      CHARACTER*(*) path,filen,fstring
      character fs*1
      logical unixok
      integer ilbfstr  ! last actual char in fstring
      integer ipos     ! the leftward moving character position
      integer ilstr    ! declaired length of fstring
      integer ilenfilen ! declaired length of filen to be returned
      integer ilenpath ! declaired length of path to be returned

C Set fileseparator depending on OS.
      call isunix(unixok)
      if(unixok)then
        fs = char(47)
      else
        fs = char(92)
      endif

C Clear variables and get string lengths.
      filen = ' '
      ilbfstr = lnblnk(fstring)
      ipos = ilbfstr
      ilstr = LEN(fstring)
      ilenfilen = LEN(filen)
      ilenpath = LEN(path)

C Loop backwards from last non-blank character until a separator is found.
 344  continue
      ipos=ipos-1
      if(fstring(ipos:ipos).eq.fs)then
        filen = ' '
        write(filen,'(a)',IOSTAT=IOS,ERR=1) fstring(ipos+1:ilbfstr)
        path = ' '
        if(ipos.gt.ilenpath)then
          write(path,'(a)',IOSTAT=IOS,ERR=1) fstring(1:ilenpath)
        else
          write(path,'(a)',IOSTAT=IOS,ERR=1) fstring(1:ipos)
        endif
        return
      elseif(fstring(ipos:ipos).eq.'/')then
        filen = ' '
        write(filen,'(a)',IOSTAT=IOS,ERR=1) fstring(ipos+1:ilbfstr)
        path = ' '
        if(ipos.gt.ilenpath)then
          write(path,'(a)',IOSTAT=IOS,ERR=1) fstring(1:ilenpath)
        else
          write(path,'(a)',IOSTAT=IOS,ERR=1) fstring(1:ipos)
        endif
        return
      else

C Arrived at the first character. There have been no file separators
C so filen is fstring (truncated if necessary) and the path becomes
C the equivalent of where-i-am-now.
        if(ipos.eq.1)then
          if(ilstr.ge.ilenfilen)then
            filen = ' '
            write(filen,'(a)',IOSTAT=IOS,ERR=1)fstring(1:ilenfilen)
            write(path,'(a1,a1)',IOSTAT=IOS,ERR=1) '.',fs
            return
          elseif(ilstr.lt.ilenfilen)then
            write(filen,'(a)')fstring(1:lnblnk(fstring))
            write(path,'(a1,a1)',IOSTAT=IOS,ERR=1) '.',fs
            return
          endif
        elseif(ipos.gt.1)then
          goto 344
        endif
      endif

      return

   1  if(IOS.eq.2)then
        call edisp(iuout,
     &  'fdroot: permission error composing path or file from string.')
        call edisp(iuout,fstring)
      else
        call edisp(iuout,
     &    'fdroot: error composing path or file from string.')
        call edisp(iuout,fstring)
      endif
      return
      end

C ******************** CMDTOROOT  *****************
C Parse command line file name and return root name.

      SUBROUTINE cmdtoroot(inf,root)
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      CHARACTER*(*) inf

      character root*32
      character ltcmdfl*144
      character thecfgis*72
      character fs*1
      character uname*24
      character path*72
      logical unixok

C Get underlying indicators.
      uname=' '
      call usrname(uname)
      call esppid(ipid)
      call isunix(unixok)

C Take command line file name as initial configuration file.
      if(inf(1:2).eq.'  '.or.inf(1:4).eq.'UNKN')then
        thecfgis='UNKNOWN'
        write(root,'(a)') uname(1:lnblnk(uname))  ! No cfg file so user name must suffice.
      else
        ltcmdfl=inf                        ! Take command line file
        call fdroot(ltcmdfl,path,thecfgis) ! Find the filename (no path)
        lcfgr=lnblnk(thecfgis)
        lcfgl=lcfgr-3
        if(lcfgr.gt.4)then
          if(thecfgis(lcfgl:lcfgr).eq.'.cfg')then  ! Take away the .cfg.
            if(lcfgl-1.le.32)then
              write(root,'(a)') thecfgis(1:lcfgl-1)
            else
              write(root,'(a)') thecfgis(1:32)
            endif
          else
            if(lcfgr.le.32)then
              write(root,'(a)')  thecfgis(1:lcfgr)
            else
              write(root,'(a)') thecfgis(1:32)
            endif
          endif
        else
          if(lcfgr.le.32)then
            write(root,'(a)') thecfgis(1:lcfgr)
          else
            write(root,'(a)') thecfgis(1:32)
          endif
        endif
      endif

      return
      end

C ******************** EFOPSEQ ********************
C Open a sequential ASCII file (SFILE) and return appropriate
C messages and error status (IEXIST).
C IUN is the file unit number, SFILE is the file name.
C IEXIST is a flag denoting:
C   0 - should exist, no message but error -301 if not.
C   1 - should exist, message & error -301 if not.
C   2 - should not exist, error 300 & message if exists.
C   3 - if exists use otherwise create.
C   4 - if exists ask user before overwriting otherwise create.
C IER is the error status (0 is OK).
C If the path is other than `./' but is not `/usr/esru' then
C concatenate path with sfile.

      SUBROUTINE EFOPSEQ(IUN,SFILE,IEXIST,IER)
#include "help.h"

      LOGICAL XST,OK,concat
      CHARACTER*(*) SFILE
      character outs*144
      CHARACTER LFIL*72
      character longtfile*144

      LFIL=' '
      longtfile=' '

C Add path to file if required then get length of file name.
      XST=.false.
      if(SFILE(1:2).eq.'  ')then
        IER=-301
        return
      elseif(SFILE(1:4).eq.'UNKN')then
        IER=-301
        return
      elseif(SFILE(1:4).eq.'unkn')then
        IER=-301
        return
      endif

      call addpath(SFILE,longtfile,concat)

 10   ltf=max(1,LNBLNK(longtfile))
      IF(IUN.NE.0)CALL ERPFREE(IUN,ISTAT)
      INQUIRE (FILE=longtfile(1:ltf),EXIST=XST)

C Depending on whether the file exists or not process according
C to the value of IEXIST:
      IF(IEXIST.EQ.0.OR.IEXIST.EQ.1)THEN
        IF(XST)THEN
          OPEN(IUN,FILE=longtfile,ACCESS='SEQUENTIAL',
     &            STATUS='OLD',IOSTAT=ISTAT)
        ELSE

C File should exist, print message if IEXIST=1 and not found.
          IF(IEXIST.EQ.1)THEN
            CALL LUSRMSG('Could not find',longtfile,'W')
            IER=-301
            RETURN
          elseif(IEXIST.EQ.0)THEN
            IER=-301
            RETURN
          ENDIF
        ENDIF
      ELSEIF(IEXIST.EQ.2)THEN

C File should not exist, error=-300 if found.
        IF(XST)THEN
          CALL LUSRMSG(longtfile,'already exists (and should not)!',
     &      'W')
          IER=-300
          RETURN
        ELSE
          OPEN(IUN,FILE=longtfile,ACCESS='SEQUENTIAL',
     &            STATUS='NEW',IOSTAT=ISTAT)
        ENDIF
      ELSEIF(IEXIST.EQ.3)THEN

C Use existing file or create a new one.
        IF(XST)THEN
          OPEN(IUN,FILE=longtfile,ACCESS='SEQUENTIAL',
     &            STATUS='OLD',IOSTAT=ISTAT)
        ELSE
          OPEN(IUN,FILE=longtfile,ACCESS='SEQUENTIAL',
     &            STATUS='NEW',IOSTAT=ISTAT)
        ENDIF
      ELSEIF(IEXIST.EQ.4)THEN

C IEXIST=4 if file exists, ask user if it should be deleted and
C overwritten with a file by the same name.
        IF(XST)THEN
          OPEN(IUN,FILE=longtfile,ACCESS='SEQUENTIAL',
     &            STATUS='OLD',IOSTAT=ISTAT)
          H(1)='Click YES to overwrite file with updated data. '
          H(2)=' '
          H(3)='Click NO to return to file name input. '
          H(4)=' '
          NHELP=4
          CALL EASKOK(longtfile,'Overwrite file?',OK,NHELP)
          IF(OK)THEN
            CALL EFDELET(IUN,ISTAT)
            OPEN(IUN,FILE=longtfile,ACCESS='SEQUENTIAL',
     &           STATUS='NEW',IOSTAT=ISTAT)
          ELSE

C Close the initial file before asking the user for new file.
  83        CALL ERPFREE(IUN,ISTAT)
            H(1)='Enter new file name and click OK. '
            H(2)=' '
            write(LFIL,'(a)')SFILE(1:lnblnk(SFILE))
            CALL EASKS(LFIL,' ','Revised file name?',72,' ',
     &        'revised file name',IER,2)
            IF(LFIL(1:2).eq.'  ')goto 83
            call addpath(LFIL,longtfile,concat)
            H(1)='Click YES to use updated file name. '
            H(2)=' '
            NHELP=2
            CALL EASKOK(' ','Preserve this revised file name?',OK,NHELP)
            if(OK)SFILE=LFIL
            goto 10
          ENDIF
        ELSE
          OPEN(IUN,FILE=longtfile,ACCESS='SEQUENTIAL',
     &            STATUS='NEW',IOSTAT=ISTAT)
        ENDIF
      ELSE
        IER=1
        RETURN
      ENDIF

C If problem in opening then report difficulty. F90 ISTAT
C values can be positive numbers, but F77 ISTAT tend to be
C negative values.
      IF(ISTAT.ne.0)THEN
        write(outs,'(a,i5,a)') 'Error (',ISTAT,') opening the file'
        CALL LUSRMSG(outs,longtfile,'W')
        IER=ISTAT
      ENDIF

      RETURN
      END

C ******************** EFOPRAN ********************
C Open a random access file (SFILE) and return appropriate
C messages and error status (IEXIST).
C IUN is the file unit number, SFILE is the file name, IRW is
C record width.  IEXIST is a flag denoting:
C   0 - should exist, no message but error -301 if not.
C   1 - should exist, message & error -301 if not.
C   2 - should not exist, error 300 & message if exists.
C   3 - if exists use otherwise create.
C   4 - if exists ask user before overwriting otherwise create.
C IRW is the record width to use when opening the file.
C IER is the error status (0 is OK). Note - if the file opens correctly
C     the value of ier is not altered from its state in the calling
C     code.  Thus the ier state should always be set prior to a call
C     to efopran.
C If the path is other than `./' but is not `/usr/esru' then
C concatenate path with sfile.

      SUBROUTINE EFOPRAN(IUN,SFILE,IRW,IEXIST,IER)
#include "help.h"

      LOGICAL XST,OK,concat
      CHARACTER*(*) SFILE
      CHARACTER LFIL*72
      character longtfile*144

      LFIL=' '
      longtfile=' '
      XST=.false.
      if(SFILE(1:2).eq.'  ')then
        IER=-301
        return
      elseif(SFILE(1:4).eq.'UNKN')then
        IER=-301
        return
      elseif(SFILE(1:4).eq.'unkn')then
        IER=-301
        return
      endif

      call addpath(SFILE,longtfile,concat)

C Length of file.
  10  ltf=max(1,LNBLNK(longtfile))
      IF(IUN.NE.0)CALL ERPFREE(IUN,ISTAT)
      INQUIRE (FILE=longtfile(1:ltf),EXIST=XST)

C Depending on whether the file exists or not process according
C to the value of IEXIST:
      IF(IEXIST.EQ.0.OR.IEXIST.EQ.1)THEN
        IF(XST)THEN
          OPEN(IUN,FILE=longtfile(1:ltf),ACCESS='DIRECT',
     &        RECL=IRW*4,FORM='UNFORMATTED',STATUS='OLD',IOSTAT=ISTAT)
        ELSE

C File should exist, print message if IEXIST=1 and not found.
          IF(IEXIST.EQ.1)THEN
            CALL LUSRMSG('Could not find',longtfile,'W')
            IER=-301
            RETURN
          elseif(IEXIST.EQ.0)THEN
            IER=-301
            RETURN
          ENDIF
        ENDIF
      ELSEIF(IEXIST.EQ.2)THEN

C File should not exist, error=-300 if found.
        IF(XST)THEN
          CALL LUSRMSG(longtfile,
     &      'already exists (and should not)!','W')
          IER=-300
          RETURN
        ELSE
          OPEN(IUN,FILE=longtfile(1:ltf),ACCESS='DIRECT',
     &        RECL=IRW*4,FORM='UNFORMATTED',STATUS='NEW',IOSTAT=ISTAT)
        ENDIF
      ELSEIF(IEXIST.EQ.3)THEN

C Use existing file or create a new one.
        IF(XST)THEN
          OPEN(IUN,FILE=longtfile(1:ltf),ACCESS='DIRECT',
     &        RECL=IRW*4,FORM='UNFORMATTED',STATUS='OLD',IOSTAT=ISTAT)
        ELSE
          OPEN(IUN,FILE=longtfile(1:ltf),ACCESS='DIRECT',
     &        RECL=IRW*4,FORM='UNFORMATTED',STATUS='NEW',IOSTAT=ISTAT)
        ENDIF
      ELSEIF(IEXIST.EQ.4)THEN

C IEXIST=4 if file exists, ask user if it should be deleted and
C overwritten with a file by the same name.
        IF(XST)THEN
          OPEN(IUN,FILE=longtfile(1:ltf),ACCESS='DIRECT',
     &        RECL=IRW*4,FORM='UNFORMATTED',STATUS='OLD',IOSTAT=ISTAT)
          H(1)='The displayed file exists. Confirm if you want'
          H(2)='to overwrite it. Otherwise, say `no` and enter'
          H(3)='another file name.'
          NHELP=3
          CALL EASKOK(longtfile,'Overwrite this file?',OK,NHELP)
          IF(OK)THEN
            CALL EFDELET(IUN,ISTAT)
            OPEN(IUN,FILE=longtfile(1:ltf),ACCESS='DIRECT',
     &        RECL=IRW*4,FORM='UNFORMATTED',STATUS='NEW',IOSTAT=ISTAT)
          ELSE

C Close the initial file before asking the user for new file.
C Pass back through with alternative file name.
  83        CALL ERPFREE(IUN,ISTAT)
            H(1)='Specify the file relative to the location of the'
            H(2)='model configuration file'
            write(LFIL,'(a)')SFILE(1:lnblnk(SFILE))
            CALL EASKS(LFIL,' ','File name?',72,' ','file name',IER,2)
            if(LFIL(1:2).eq.'  ')goto 83
            call addpath(LFIL,longtfile,concat)
            goto 10
          ENDIF
        ELSE
          OPEN(IUN,FILE=longtfile(1:ltf),ACCESS='DIRECT',
     &        RECL=IRW*4,FORM='UNFORMATTED',STATUS='NEW',IOSTAT=ISTAT)
        ENDIF
      ELSE
        IER=1
        RETURN
      ENDIF

C If problem in opening then report difficulty.
      IF(ISTAT.LT.0)THEN
        CALL LUSRMSG('Error opening file ',longtfile,'W')
        IER=ISTAT
      ENDIF

      RETURN
      END

C ******************** FINDFIL ********************
C Check for existance of a file SFILE (with path) & return XST.
C If the path is other than `./' but is not `/usr/esru' then
C concatenate path with sfile before looking.  If file name is
C blank or UNKNOWN then return XST=.FALSE.

      SUBROUTINE FINDFIL(SFILE,XST)

      LOGICAL xst,concat
      CHARACTER SFILE*72,longtfile*144
      integer ltf   ! position of last character in the string.

      xst=.false.
      if(SFILE(1:2).eq.'  ')return
      if(SFILE(1:4).eq.'UNKN')return
      if(SFILE(1:4).eq.'unkn')return
      call addpath(SFILE,longtfile,concat)

C Does the file exist?
      ltf=max(1,LNBLNK(longtfile))
      INQUIRE (FILE=longtfile(1:ltf),EXIST=xst)

      RETURN
      END

C ******************** addpath ********************
C Return file name appended onto the path and logical concat.
C IUOUT is the message channel, SFILE is the file name.
C If the path does not begin with '/' or '?:' then
C concatenate path with sfile.
C If path is '  ' or './' do not concatenate
C This version includes cross-platform logic.

      SUBROUTINE addpath(SFILE,tfile,concat)
      IMPLICIT NONE

C Functions
      integer lnblnk

C Declare calling parameters
      CHARACTER*(*), INTENT(OUT) :: sfile,tfile
      LOGICAL, INTENT(OUT) :: concat
C Declare local variables
      LOGICAL :: unixok
      CHARACTER :: fs,bs,a1
      CHARACTER(72) :: path
      CHARACTER(12) :: tp, tp1
      CHARACTER(124):: outs
      INTEGER :: LN,LNS,I,LT,ios,iuin,iuout,ieout,pos

C Common blocks
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      common/rpath/path
      
C Define path and file name lengths and check operating system.
      LN=max(1,lnblnk(path))
      LNS=max(1,lnblnk(sfile))
      call isunix(unixok)

C Set folder separator (fs) to \ or / and the folders to ignore as required.
      fs = char(47)
      bs = char(92)
      write(tp,'(2a)') '.',fs
      write(tp1,'(2a)') '.',bs
      write(a1,'(a)') sfile(1:1)
C Debug.
C      write(*,*)'tp',tp,'tp1',tp1,'a1',a1

C Assume that we will add the path to the file name.
      concat = .true.

      if (path(1:2).eq.tp(1:2).or.path(1:2).eq.tp1(1:2).or.
     &    path(1:2).eq.'  ') then

C Path is defined starting ./ or .\ or we are running locally (path='  ')
C therefore do not add path.
        concat = .false.
      elseif(unixok)then

C If running on unix and sfile begins with a / then an absolute path has
C been defined - do not add path.
        if (a1.eq.'/') then
          concat = .false.
        elseif (((ICHAR(a1).gt.64.and.ICHAR(a1).lt.91).or.
     &           (ICHAR(a1).gt.96.and.ICHAR(a1).lt.123)).and.
     &            sfile(2:2).eq.':') then

C Special case: windows absolute path being used on unix - substitute /usr/ for initial c:\
C and change all \ to /.
          concat = .false.
          write (tfile,'(2a)') '/usr/',sfile(4:LNS)
          LT=max(1,lnblnk(tfile))
          sfile=tfile
          do 10 I=1,LT
            if (tfile(I:I).eq.bs) then
              write (sfile(I:I),'(a2)') fs
            endif
 10       continue
          write(outs,*)
     &      'addpath: warning found windows path changing to unix.'
          call edisp(iuout,outs)
          write(outs,*) 'addpath: ',sfile
          call edisp(iuout,outs)
        endif
      else

C If running on windows and sfile begins ?: then an absolute path has
C been defined - do not add path.
        if (((ICHAR(a1).gt.64.and.ICHAR(a1).lt.91).or.
     &       (ICHAR(a1).gt.96.and.ICHAR(a1).lt.123)).and.
     &        sfile(2:2).eq.':') then
          concat = .false.
        elseif (a1.eq.'/') then
C make sure all \ (bs) are replaced with / (fs)
          do
            pos=SCAN(sfile,bs)
            if (pos.ne.0) then
              sfile(pos:pos)=fs
            else 
              EXIT
            end if
          end do      

C Special case: unix absolute path being used on windows - substitute c: for initial /
C Limit feedback text length to fit within outs text buffer.
          concat = .false.
          write (tfile,'(2a)') 'C:',sfile(5:LNS)
          sfile=tfile
          write(outs,*)
     &      'addpath: warning found unix path changing to windows.'
          call edisp(iuout,outs)
          LNS=lnblnk(sfile)
          if(LNS.gt.112) LNS=112
          write(outs,*) 'addpath: ',sfile(1:LNS)
          call edisp(iuout,outs)
        endif
        
      endif

      if(concat)then
        LN=max(1,lnblnk(path))
        LNS=max(1,lnblnk(sfile))
        write(tfile,'(2a)',iostat=ios,err=1)path(1:LN),sfile(1:LNS)
      else
        write(tfile,'(a)',iostat=ios,err=1)sfile(1:LNS)
      endif
      return
   1  if(IOS.eq.2)then
        call lusrmsg('addpath: permission error writing file name',
     &    tfile,'W')
      else
        call lusrmsg('addpath: error writing file name',tfile,'W')
      endif
      return
      end

C ******************** GETTOKENS ********************
C Checks a character string (A), returning the number of data
C items (IW) separated by a file separator and an array of tokens/words
C (up to 12 32character words returned) that were in the string.
C Useful for parsing file paths. Note to keep from
C overwriting the string passed, deal with a copy.

      SUBROUTINE GETTOKENS(A,IW,WORDS)
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      CHARACTER*(*) A
      CHARACTER*32 WORDS(12)
      CHARACTER B*248,C*1,fs*1,outs*248
      logical unixok

C Determine the file separator and clear the WORDS to return.
      call isunix(unixok)
      if(unixok)then
        fs = char(47)
      else
        fs = char(92)
      endif
      do 9 i=1,12
        WORDS(i)='  '
  9   continue

      lastcc = MIN0(LNBLNK(A),246)
      WRITE(B,'(A)',IOSTAT=ios,ERR=1) A(1:lastcc+1)

C Loop through B and convert all fileseparators to blanks.
      DO 100 I=1,LNBLNK(B)+1
        C=B(I:I)
        IF(ICHAR(C).EQ.47.or.ICHAR(C).eq.92) B(I:I)=' '
  100 CONTINUE

C Loop through B and see how many 'words'.
      K=0
      IW=0
  102 IW=IW+1

C Start by skipping blanks before the word.
   10 K=K+1
      IF(K.ge.248) GOTO 3
      C=B(K:K)
      IF(C.EQ.' ') GOTO 10

C Loop character by character until separator is found.
   20 K=K+1
      C=B(K:K)
      if(K.GE.LNBLNK(B))then

C Have determined the number of tokes so loop through each and
C assign WORDS.
        if(IW.gt.0)then
          KK=0
          do 42 ij=1,IW
            CALL EGETW(B,KK,WORDS(ij),'W','token',IER)
            if(ier.ne.0)goto 2
  42      continue
        endif
        RETURN
      endif
      IF(C.EQ.' ')GOTO 102
      GOTO 20
   1  if(IOS.eq.2)then
        write(outs,*) 
     &   'GETTOKENS: internal write permission error: ',A(1:lnblnk(A))
        call edisp248(iuout,outs,100)
      else
        write(outs,*) 
     &    'GETTOKENS: error in internal write: ',A(1:lnblnk(A))
        call edisp248(iuout,outs,100)
      endif
      return
   2  if(IOS.eq.2)then
        write(outs,*) 
     &  'GETTOKENS: permission error in getting token: ',A(1:lnblnk(A))
        call edisp248(iuout,outs,100)
      else
        write(outs,*) 
     &  'GETTOKENS: error in getting token: ',A(1:lnblnk(A))
        call edisp248(iuout,outs,100)
      endif
      return
    3 write(outs,*) 
     &  'GETTOKENS: ran past end of internal 248 char buffer: ',
     &   A(1:lnblnk(A))
      call edisp248(iuout,outs,100)
      return
      end

C ******************** c2fstr ********************
C Converts c function returned string to fortran format.
C Strip off the last character (end of line mark) as well as any
C leading blanks from string returned.

C Noticed problems in the native windows compile with null
C characters. This might be because of the direct copy was
C used if windows detected and this preserves unprintable
C characters. Revised to use the same logic for both Unix and Dos.

      SUBROUTINE c2fstr(cstr,fstr)
      CHARACTER*(*) cstr,fstr

      fstr = ' '    ! clear the return string
      K=0
      DO 99 I=1,LEN(cstr)
        IF(cstr(I:I).NE.' '.OR.K.GE.1)THEN
          if(ichar(cstr(I:I)).lt.32)goto 100
          K=K+1
          if(K.gt.LEN(fstr))goto 100
          fstr(K:K)=cstr(I:I)
        ENDIF
 99   CONTINUE
 100  return
      end


C ******************** terminalmode ********************
C Given an index `mode` return a terminal text string.
C this would be called based on the fortran setting of mode in prj.

      SUBROUTINE terminalmode(mode,tmode)
      CHARACTER*(*) tmode

      integer mode          ! passed from tchild

      tmode = ' '
      if(mode.eq.-1)then
        tmode = 'text'
      elseif(mode.eq.-2)then
        tmode = 'page'
      elseif(mode.eq.8)then
        tmode = 'graphic'
      elseif(mode.eq.-6)then
        tmode = 'script'
      endif

      RETURN
      END

C ******************** EKPAGE ********************
C Maps key characters, pages & array indexs in long display lists.
C No menu should have more than 26 data selections (plus heading and
C control lines) so that key characters (a-z) for arrays will not be
C repeated. Notes on variables:
C IPACT   - page option, if IPACT=CREATE then only update the common
C           block PMENU, otherwise ask for action to be taken.
C IPM,MPM - current and last page of menu.
C IST     - data array index at the beginning of the menu.

      SUBROUTINE EKPAGE(IPACT)
#include "epara.h"
#include "help.h"

      COMMON/SPAD/MMOD,LIMIT,LIMTTY

C If creating menu don't ask questions. Allow movement forward or backwards
C in the control menu.
      IGTP=1  ! if creating assume first page
      IF(IPACT.EQ.CREATE)THEN
        IX=1
      ELSEIF(IPACT.lt.0)then
        IX=3
        IGTP=IABS(IPACT)
      ELSE

C If only two pages then automatically switch to the page not being displayed,
C otherwise ask what to do.
        if (MPM.eq.2) then
          if (IPM.eq.1) then
            IX=5
          else
            IX=2
          endif
        elseif (MPM.eq.3) then

C If three pages and on 1st jump to middle, if at end jump to middle
C and if in middle ask user what to do.
          if (IPM.eq.1) then
            IX=5
          elseif (IPM.eq.3) then
            IX=2
          else
            H(1)='In order to manage long lists of selections or data'
            H(2)='a number of viewports may be necessary.'
            CALL EASKMBOX(' ',' Menu-page options:','previous','next',
     &        'goto page',' ',' ',' ',' ',' ',IW,2)
            IF(IW.EQ.3)then
              H(1)='If page greater than length then last page assumed.'
              CALL EASKI(IGTP,' ',' Which page ? ',1,'F',1,'-',1,
     &          'menu page',IER,1)
              IX=3
            elseif(IW.EQ.2)then
              IX=5
            elseif(IW.EQ.1)then
              IX=2
            endif
            CALL USRMSG(' ',' ','-')
          endif
        else

C If lots of pages track the 1st and last page otherwise ask user.
          if (IPM.eq.1) then
            IX=5
          elseif (IPM.eq.MPM) then
            IX=2
          else
            H(1)='To manage long selection lists, several pages'
            H(2)='may be offered here.'
            CALL EASKMBOX(' ','Options:','first page',
     &        'previous','next','go to','last page',' ',' ',' ',IW,2)
            if(IW.eq.5)then
              IGTP=MPM   ! set it to last page
              IX=3
            elseif(IW.EQ.4)then
              H(1)='If the specified page is greater than the total'
              H(2)='available then the last page is assumed.'
              CALL EASKI(IGTP,' ','Which page?',1,'F',1,'-',1,
     &          'menu page',IER,2)
              IX=3
            elseif(IW.EQ.3)then
              IX=5
            elseif(IW.EQ.2)then
              IX=2
            elseif(IW.EQ.1)then
              IGTP=1   ! set it to first page
              IX=3
            endif
          endif
          CALL USRMSG(' ',' ','-')
        endif
      ENDIF
      IF(IX.EQ.1)THEN

C Build list from beginning of the item array, find out how many lines
C are allowed in a menu on this screen (MFULL), how many items can
C fit in the main section of the menu (MIFULL) and limit this to 26
C so that the automaticly supplied key character remains within the
C range a-z.  If more items that allowed set tag for paging.
        IF(MMOD.LT.8)THEN
          MFULL=LIMTTY
        ELSEIF(MMOD.EQ.8)THEN
          MFULL=30
        ENDIF
        MIFULL=MFULL-(MHEAD+MCTL)
        IF(MIFULL.GT.26)THEN
          MIFULL=26
          MFULL=MIFULL+MHEAD+MCTL
        ENDIF
        IF(ILEN.LE.MIFULL)THEN
          IPFLG=0
        ELSE
          IPFLG=1
        ENDIF
        IST=1
      ELSEIF(IX.EQ.2)THEN

C Move to previous page and rebuild list.
        IPACT=PREV
        IF(IPM.GT.1)THEN
          IST=IST-MIFULL
        ENDIF
      ELSEIF(IX.EQ.3)THEN

C Move to a particular page. Begin at first and increment.
        if(IGTP.eq.1)then
          IST=1
        else
          IST=1
          do 42 ij=2,IGTP
            IF((IST+MIFULL).LE.ILEN)IST=IST+MIFULL
  42      continue
        endif
      ELSEIF(IX.EQ.5)THEN

C Move to the next page and rebuild list if not the last page.
        IPACT=NEXT
        IF((IST+MIFULL).LE.ILEN)THEN
          IST=IST+MIFULL
        ENDIF
      ELSE

C Unknown option.
        RETURN
      ENDIF

C Determine current page and last page of menu. Base the current page
C number on the starting item of the current menu unless the last page.
      PAGE=(FLOAT(IST+MIFULL-1)/FLOAT(MIFULL))
      IF(PAGE.LT.1.0)PAGE=1.0
      IPM=INT(PAGE)
      PAGE=(FLOAT(ILEN)/FLOAT(MIFULL))
      IF(PAGE.LT.1.0)PAGE=1.0
      IF((PAGE-AINT(PAGE)).GT.0.0)PAGE=AINT(PAGE)+1.0
      MPM=INT(PAGE)

      RETURN
      END

C ******************** KEYIND ********************
C Decodes the index INO and returns the array index IA of the item
C displayed and whether INO is within the list.
C MENUL - length of the menu.
C INO   - index passed from the calling code.
C IA    - array index of the data
C INOUT - 0 if outside the current menu, 1 if within the menu.

      SUBROUTINE KEYIND(MENUL,INO,IA,INOUT)
#include "epara.h"

C Assume blank returned key, 0 array index, outwith list.
      IA=0
      INOUT=0

      IF(INO.GT.MHEAD.AND.INO.LT.(MENUL-MCTL+1))THEN
        INOUT=1
        IA=INO-MHEAD+IST-1
      ENDIF
      RETURN
      END

C ******************** EPMENSV ********************
C Pushes the current state of common block PMENU into common block
C PMENUSV. Up to 5 levels are maintained.
C This is useful if a second menu is about to be called and
C knowledge of the first is required.  See also EPMENRC which pops the
C information back into common PMENU.

      SUBROUTINE EPMENSV
#include "epara.h"

      COMMON/PMENUSV/MHEAD1(10),MCTL1(10),MIFUL1(10),MFUL1(10),IST1(10),
     &               ILEN1(10),IPM1(10),MPM1(10),IPFLG1(10),MILEV
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      CHARACTER outs*124
      MILEV=MILEV+1
      if(MILEV.gt.10)then

C Debug.
        write(outs,*) 'Number of nested menu levels is',MILEV,
     &    ' which exceeds allowable menu depth.'
        call edisp(iuout,outs)

      endif
      do 42 j=10,2,-1
        i=j-1
        MHEAD1(j)=MHEAD1(i)
        MCTL1(j)=MCTL1(i)
        MIFUL1(j)=MIFUL1(i)
        MFUL1(j)=MFUL1(i)
        IST1(j)=IST1(i)
        ILEN1(j)=ILEN1(i)
        IPM1(j)=IPM1(i)
        MPM1(j)=MPM1(i)
        IPFLG1(j)=IPFLG1(i)
  42  continue
      MHEAD1(1)=MHEAD
      MCTL1(1)=MCTL
      MIFUL1(1)=MIFULL
      MFUL1(1)=MFULL
      IST1(1)=IST
      ILEN1(1)=ILEN
      IPM1(1)=IPM
      MPM1(1)=MPM
      IPFLG1(1)=IPFLG

      RETURN

      END

C ******************** EPMENRC ********************
C Recovers menu definitions (common block PMENU) from common
C block PMENUSV. This is useful if returning from a second menu.
C See also EPMENSV.

      SUBROUTINE EPMENRC
#include "epara.h"

      COMMON/PMENUSV/MHEAD1(10),MCTL1(10),MIFUL1(10),MFUL1(10),IST1(10),
     &               ILEN1(10),IPM1(10),MPM1(10),IPFLG1(10),MILEV
      MILEV=MILEV-1
      if(MILEV.le.0)then
        MILEV=0
      endif
      MHEAD=MHEAD1(1)
      MCTL=MCTL1(1)
      MIFULL=MIFUL1(1)
      MFULL=MFUL1(1)
      IST=IST1(1)
      ILEN=ILEN1(1)
      IPM=IPM1(1)
      MPM=MPM1(1)
      IPFLG=IPFLG1(1)
      do 42 i=2,10
        j=i-1
        MHEAD1(j)=MHEAD1(i)
        MCTL1(j)=MCTL1(i)
        MIFUL1(j)=MIFUL1(i)
        MFUL1(j)=MFUL1(i)
        IST1(j)=IST1(i)
        ILEN1(j)=ILEN1(i)
        IPM1(j)=IPM1(i)
        MPM1(j)=MPM1(i)
        IPFLG1(j)=IPFLG1(i)
  42  continue
      RETURN

      END

C ******************** EPAGE ********************
C Clear screen without waiting.

      SUBROUTINE EPAGE

      IPAGE=1
      IWAIT=0
      IEND=0
      CALL EMPAGE(IPAGE,IWAIT,IEND)
      RETURN
      END

C ******************** EPAGEW ********************
C Wait before clearing screen.

      SUBROUTINE EPAGEW

      IPAGE=1
      IWAIT=1
      IEND=0
      CALL EMPAGE(IPAGE,IWAIT,IEND)
      RETURN
      END

C ******************** EPWAIT ********************
C Wait but do not clear screen.

      SUBROUTINE EPWAIT

      IPAGE=0
      IWAIT=1
      IEND=0
      CALL EMPAGE(IPAGE,IWAIT,IEND)
      RETURN
      END

C ******************** EPAGEND ********************
C Clear screen then close window if open.

      SUBROUTINE EPAGEND

      IPAGE=1
      IWAIT=0
      IEND=1
      CALL EMPAGE(IPAGE,IWAIT,IEND)
      RETURN
      END
      

C ******************** ang3vtx ********************
C Recover angle between three vertex (i.e. between two lines).
C Angel between lines: finds angle A between two lines'
C given 3 vertex as in:    * 1
C                           a \
C                     3 *------* 2

      SUBROUTINE ang3vtx(x1,y1,z1,x2,y2,z2,x3,y3,z3,ang)

      dimension vd21(3),vd32(3)
      PI = 4.0 * ATAN(1.0)
      RAD = PI/180.
      vdx21=x1-x2
      vdy21=y1-y2
      vdz21=z1-z2
      vdx32=x3-x2
      vdy32=y3-y2
      vdz32=z3-z2
      CROW21 = CROWXYZ(x1,y1,z1,x2,y2,z2)
      if (abs(crow21).lt.0.001) then
        return
      endif
      CROW32 = CROWXYZ(x2,y2,z2,x3,y3,z3)
      if (abs(crow32).lt.0.001) then
        return
      endif

C Make vectors into unit vectors.
      vd21(1)=vdx21/crow21
      vd21(2)=vdy21/crow21
      vd21(3)=vdz21/crow21
      vd32(1)=vdx32/crow32
      vd32(2)=vdy32/crow32
      vd32(3)=vdz32/crow32

C Get dot product.
      call dot3(vd21,vd32,prod)
      ang=ACOS(prod)/RAD
      return
      end

C ******************** CLOSE3D ********************
C Calculates the minimum distance between two lines in 3D return
C the distance and closest points.
C PA,VA define line A (point and vector), SA is parametric variable at closest point
C PB,VB define line B (point and vector), SB is parametric variable at closest point
C DIST minimum distance between lines - if negative then error
C CA,CB are closest points on two lines (if intesect DIST=0 and CA=CB)
C Theory:
C  At intersection the values of the parametric variables will
C  produce the same x,y,z values.  Two equations are formed and
C  solved (these are easily derived from the parametric form of
C  the line equations but the variable names are used below).
C    SA.VA.VB - SB.VBsq = RHSa
C    SA.VAsq - SB.VA.VB = RHSb

      SUBROUTINE CLOSE3D(PA,VA,SA,PB,VB,SB,DIST,CA,CB)

      DIMENSION PA(3),VA(3),PB(3),VB(3),CA(3),CB(3)
      DIMENSION PD(3)

C Check for parallel lines.
C Calculate some data here for use later.
      VAsq=0.
      VBsq=0.
      VAVB=0.0
      do 5 I=1,3
        VAVB=VAVB+VA(I)*VB(I)
        VAsq=VAsq+VA(I)*VA(I)
        VBsq=VBsq+VB(I)*VB(I)
 5    continue
      ALEN=sqrt(VAsq)
      BLEN=sqrt(VBsq)
      DOT=VAVB/(ALEN*BLEN)
      DOT=abs(DOT)-1.
      if (abs(DOT).lt.0.0001) then

C Check if colinear.
        DOT=0.
        ALEN=0.
        do 7 I=1,3
          DOT=DOT+(PA(I)-PB(I))*VB(I)
          ALEN=ALEN+(PA(I)-PB(I))*(PA(I)-PB(I))
 7      continue
        ALEN=sqrt(ALEN)
        DOT=DOT/(ALEN*BLEN)
        DOT=abs(DOT)-1.
        if (abs(DOT).gt.0.0001) then
          DIST=-99.
          return
        endif
      endif

C Calculate closest distance between test line and current edge.
C PD is the distance from the test point to the initial point on
C the edge.
      do 10 I=1,3
        PD(I)=PB(I)-PA(I)
 10   continue
      RHSa=0.0
      RHSb=0.0
      do 20 I=1,3
        RHSa=RHSa+PD(I)*VB(I)
        RHSb=RHSb+PD(I)*VA(I)
 20   continue

C VAVB will equal 0.0 if lines are at right angles this simplifies
C solution of simultaneous equations.
      if (abs(VAVB).lt.0.0001) then
        SB=(RHSa/VBsq)*(-1.)
        SA=RHSb/VAsq
      else

C Solve simultaneous equations.
        SB=(((VAVB*RHSb)/VAsq)-RHSa) / (VBsq-((VAVB*VAVB)/VAsq))
        SA=(RHSa+VBsq*SB)/VAVB
      endif

C Calc closest points.
      do 30 I=1,3
        CA(I)=SA*VA(I)+PA(I)
        CB(I)=SB*VB(I)+PB(I)
 30   continue

C Calc distance between them.
      DIST=CROW(CA,CB)

      return
      end

C ******************** CROW ********************
C Function returning the distance 'as the crow flies'
C between the two points P and Q in X Y Z space.

      FUNCTION CROW(P,Q)
      DIMENSION  P(3), Q(3)
      CROW = ((P(1)-Q(1))*(P(1)-Q(1))) + ((P(2)-Q(2))*(P(2)-Q(2))) +
     &       ((P(3)-Q(3))*(P(3)-Q(3)))
      CROW = SQRT(CROW)

      RETURN
      END

C ******************** CROWXYZ ********************
C Function returning the distance 'as the crow flies'
C between the two X Y Z points.

      FUNCTION CROWXYZ(PX,PY,PZ,QX,QY,QZ)
      CROWXYZ = ((PX-QX)*(PX-QX))+((PY-QY)*(PY-QY))+((PZ-QZ)*(PZ-QZ))
      CROWXYZ = SQRT(CROWXYZ)

      RETURN
      END

C ******************** UVXYZ ********************
C Returns the unit vector from two points along a
C line expressed as X Y Z points.

      subroutine UVXYZ(PX,PY,PZ,QX,QY,QZ,UX,UY,UZ)
      CROWXYZ = ((PX-QX)*(PX-QX))+((PY-QY)*(PY-QY))+((PZ-QZ)*(PZ-QZ))
      CROWXYZ = SQRT(CROWXYZ)
      if (abs(crowxyz).lt.0.001) then
        return
      endif
      UX=(PX-QX)/crowxyz
      UY=(PY-QY)/crowxyz
      UZ=(PZ-QZ)/crowxyz

      RETURN
      END

C ******************** UVAB ********************
C UVAB: Subroutine returning Unit normal vector U of vector A.
C ierr (integer) = -1 if vector is very short.

      subroutine UVAB(A,U,ierr)
      dimension A(3),U(3)
      ierr=0
      S = A(1)*A(1) + A(2)*A(2) + A(3)*A(3)
      S = SQRT(S)
      if (abs(S).lt.0.001) then
        ierr=-1
        return
      endif
      U(1)=A(1)/S
      U(2)=A(2)/S
      U(3)=A(3)/S

      RETURN
      END

C ******************** PLNDANG ********************
C Finds dihedral angle between two planes given their equations.

      subroutine plndang(aeqn,beqn,dang)
      dimension aeqn(4),beqn(4)
      PI = 4.0 * ATAN(1.0)
      RAD = PI/180.0
      COSDAN = AEQN(1)*BEQN(1)+AEQN(2)*BEQN(2)+AEQN(3)*BEQN(3)
      DANG = ACOS(COSDAN) * RAD
      return
      end

C ******************** PLNDIS ********************
C Finds distance DIST from a point (x,y,z) to a plane (eq EQN).
C If DIST=0 then one the plane, if DIST > 0 then point on the side
C of the normal vector, if DIST < 0 then on the opposite side.

      subroutine PLNDIS(EQN,X,Y,Z,DIST)

      DIMENSION EQN(4)
      DIST =   EQN(1)*X + EQN(2)*Y + EQN(3)*Z - EQN(4)
      RETURN
      END

C ******************** AVER ********************
C Returns the centre of gravity CG for an array
C of vertex points PNT with NP points in it.

      SUBROUTINE AVER(MNV,NP,PNT,CG)

      DIMENSION PNT(MNV,3),CG(3)

      DO 20 J=1,3
        CG(J)=0.
        DO 10 K=1,NP
          CG(J)=CG(J)+PNT(K,J)
   10   CONTINUE
        CG(J)=CG(J)/NP
   20 CONTINUE

      RETURN
      END

C ******************** CROSS ********************
C Performs a cross-product of two vectors A(X,Y,Z) and B(X,Y,Z),
C and returns the result in C(X,Y,Z).

      SUBROUTINE CROSS(A,B,C)

      DIMENSION A(3),B(3),C(3)
      C(1)=0.0
      C(2)=0.0
      C(3)=0.0
      C(1)=A(2)*B(3)-A(3)*B(2)
      C(2)=A(3)*B(1)-A(1)*B(3)
      C(3)=A(1)*B(2)-A(2)*B(1)
 
      RETURN
      END

C ******************** CROSS2 ********************
C Performs a cross-product of two vectors AX,AY,AZ and BX,BY,BZ,
C and returns the result in CX,CY,CZ.

      subroutine cross2(ax,ay,az, bx,by,bz, cx,cy,cz)
      cx = ay*bz - az*by
      cy = az*bx - ax*bz
      cz = ax*by - ay*bx
      return
      end

C ******************** DOT3 ********************
C Return dot product of two vectors a & b.

      subroutine dot3(a,b,product)
      dimension a(3),b(3)
      product=a(1)*b(1)+a(2)*b(2)+a(3)*b(3)
      return
      end

C ******************** ZEROS ********************
C Clear a 4x4 array prior to doing viewing transforms.

      SUBROUTINE ZEROS(A)
      DIMENSION A(4,4)
      A(1,1)=0.0
      A(1,2)=0.0
      A(1,3)=0.0
      A(1,4)=0.0

      A(2,1)=0.0
      A(2,2)=0.0
      A(2,3)=0.0
      A(2,4)=0.0

      A(3,1)=0.0
      A(3,2)=0.0
      A(3,3)=0.0
      A(3,4)=0.0

      A(4,1)=0.0
      A(4,2)=0.0
      A(4,3)=0.0
      A(4,4)=0.0

      RETURN
      END
      
C ******************** REQUAL ********************
C Allows two real numbers R1 & R2 to be checked for closeness
C to a given tolerance TOL and returns .TRUE. or .FALSE.

      LOGICAL FUNCTION REQUAL(R1,R2,TOLERANCE)

      IMPLICIT NONE
C Declare calling parameters 
      REAL, INTENT(IN)  :: R1,R2,TOLERANCE
      
      CALL ECLOSE(R1,R2,TOLERANCE,REQUAL)

      RETURN
      END

C ******************** ECLOSE ********************
C Allows two real numbers R1 & R2 to be checked for closeness
C to a given tolerance TOL and returns CLOSE = .TRUE. or .FALSE.

      SUBROUTINE ECLOSE(R1,R2,TOL,CLOSE)

      LOGICAL CLOSE

      IF(ABS(R1-R2).LT.TOL)THEN
        CLOSE = .TRUE.
      ELSE
        CLOSE = .FALSE.
      ENDIF

      RETURN
      END

C ******************** ECLOSE3 ********************
C Allows two real vectors R1 & R2  & R3 to be checked for closeness
C to a given tolerance TOL with X1 X2 & X3 and returns CLOSE = .TRUE. or .FALSE.

      SUBROUTINE ECLOSE3(R1,R2,R3,X1,X2,X3,TOL,CLOSE)

      LOGICAL CLOSE,CLOSEA,CLOSEB,CLOSEC
      real R1,R2,R3  ! the vector to test
      real X1,X2,X3  ! the vector to compare against
      real TOL       ! how close

      call eclose(R1,X1,TOL,CLOSEA)   ! test first pair
      call eclose(R2,X2,TOL,CLOSEB)   ! test second pair
      call eclose(R3,X3,TOL,CLOSEC)   ! test third pair
      if(CLOSEA.and.CLOSEB.and.CLOSEC)then   
        CLOSE=.true.  ! all are close
      else
        CLOSE=.false. ! at least one is different
      endif
      
      RETURN
      END

C ******************** ESIND ********************
C Returns SIN of angle where angle is in degrees.

      FUNCTION ESIND (DEG)

      PI = 4.0 * ATAN(1.0)
      RAD = PI/180.
      ESIND = SIN (RAD*DEG)

      RETURN
      END

C ******************** ECOSD ********************
C Returns COS of angle where angle is in degrees.

      FUNCTION ECOSD (DEG)

      PI = 4.0 * ATAN(1.0)
      RAD = PI/180.
      ECOSD = COS (RAD*DEG)

      RETURN
      END

C ******************** ETAND ********************
C Returns TAN of angle where angle is in degrees.

      FUNCTION ETAND (DEG)

      PI = 4.0 * ATAN(1.0)
      RAD = PI/180.
      ETAND = TAN (RAD*DEG)

      RETURN
      END

C ******************** IFAX ********************
C An integer function returning the integer part of it's
C argument truncated towards negative infinity as follows:
C     V        IFAX(V)
C    1.0        1
C     .9        0
C     .1        0
C      0        0
C    -.1       -1
C    -.9       -1
C   -1         -1
C   -1.1       -2

      FUNCTION IFAX(V)

        IFAX = INT(V)
        IF (V.LT.0) IFAX=IFAX-1

      RETURN
      END

C ******************** EAZALT ********************
C Computes the solar azimuth and altitude angles at the current
C time (REAL). The 'ISUNUP' variable determines whether the sun
C is up (=1) or down (=0). The solar angles are computed relative
C to local mean time (e.g. Greenwich is the reference time zone for
C the UK).

      SUBROUTINE EAZALTS(TIMEH,ID,SLAT,SLON,ISUNUP,SAZI1,SALT1)

      implicit none
      
      real esind,ecosd  ! function definitions

C Parameters:
      integer ID,ISUNUP
      real TIMEH,SLAT,SLON,SAZI1,SALT1

C Local
      real A,ABST,AZMUTH,B,CDEC,CDTIME,CSLAT,DAY,DEC,EQT,HOUR,PI,R
      real SABST,SALT,SAZI,SDEC,SSLAT,TIME,TIMCOE,XX,YY
      logical CLOSE
      
      PI = 4.0 * ATAN(1.0)  ! pattern used in AZALT
      SAZI1=0.0      ! set initial values
      SALT1=0.0
      ISUNUP=1
      R=PI/180.0    ! compute radians
      DAY=FLOAT(ID) ! cast julian day to local
      HOUR=TIMEH    ! cast passed time to hour fraction

c Equation of time.
      A=1.978*DAY-160.22
      B=0.989*DAY-80.11
      EQT=0.1645*ESIND(A)-0.1255*ECOSD(B)-0.025*ESIND(B)

c Declination.
      A=280.1+0.9863*DAY
      DEC=23.45*ESIND(A)
      SDEC=SIN(DEC*R)
      CDEC=COS(DEC*R)

c Solar altitude. If SALT < 0.1 assume sun not yet up.
      TIME=HOUR+(EQT+SLON/15.0)
      TIMCOE=15.0*(12.0-TIME)
      CDTIME=COS(TIMCOE*R)
      ABST=ABS(TIMCOE)
      SABST=SIN(ABST*R)
      SSLAT=SIN(SLAT*R)
      CSLAT=COS(SLAT*R)
      SALT=ASIN(SSLAT*SDEC+CSLAT*CDEC*CDTIME)/R
      IF(SALT.LT.0.1)goto 1

C Solar aziumth.
      AZMUTH=(CDEC*SABST)/ECOSD(SALT)
      IF(AZMUTH.LT.-1.0)AZMUTH=-1.0
      IF(AZMUTH.GT.1.0)AZMUTH=1.0
      SAZI=ASIN(AZMUTH)/R

C Correct the azimuthal angle for time of day and whether in north or
C south hemispheres.
      XX=CDTIME
      CALL ECLOSE(SLAT,0.0,0.01,CLOSE)
      IF(CLOSE)goto 13
      CALL ECLOSE(SLAT,90.0,0.01,CLOSE)
      IF(CLOSE)goto 8
      YY=(CSLAT/SSLAT)*(SDEC/CDEC)
      goto 9
    8 YY=0.0
      goto 9
   13 YY=10.0*(SDEC/CDEC)

C This is one place where gfortran 4.3 complains.
    9 IF(YY-XX)3,4,5
    3 IF(SLAT.GE.0.0)goto 6
      goto 7
    5 IF(SLAT.LT.0.0)goto 6
      goto 7
    4 IF(TIME.LE.12.0)SAZI=90.0
      IF(TIME.GT.12.0)SAZI=270.0
      goto 2
    6 IF(TIME.LE.12.0)SAZI=180.0-SAZI
      IF(TIME.GT.12.0)SAZI=180.0+SAZI
      goto 2
    7 IF(TIME.GT.12.0)SAZI=360.0-SAZI
    2 SAZI1=SAZI
      SALT1=SALT
      goto 11
    1 ISUNUP=0
   11 CONTINUE
      RETURN
      END

C ********** AGNXYZ
C Given the solar azimuth and altitude, returns viewing coords
C at a far distance from the site origin.
      SUBROUTINE ANGXYZ(YAZI,SALT,X,Y,Z)

      DATA SUNDIS /1000./

      PI = 4.0 * ATAN(1.0)

      RAD = PI/180.
      RYAZI = YAZI*RAD
      RSALT = SALT*RAD

      Z = SUNDIS*SIN(RSALT)
      XYDIS = SUNDIS*COS(RSALT)

      IF (XYDIS .LT. 1E-6)THEN
        X = 0.
        Y = 0.
        RETURN
      ELSE
        X = XYDIS*SIN(RYAZI)
        Y = XYDIS*COS(RYAZI)
        RETURN
      ENDIF

      END

C ******************** ORTTRN ********************
C Multiplies a point (XM,YM,ZM) by the transform matrix
C TMAT to return the point XO,YO,Z0.  If the points are to close
C together then IER=-1.

      SUBROUTINE ORTTRN(XM,YM,ZM,TMAT,XO,YO,ZO,IERR)

      DIMENSION TMAT(4,4)

      IERR = 0

      XO = XM*TMAT(1,1)+YM*TMAT(2,1)+ZM*TMAT(3,1)+TMAT(4,1)
      YO = XM*TMAT(1,2)+YM*TMAT(2,2)+ZM*TMAT(3,2)+TMAT(4,2)
      ZO = XM*TMAT(1,3)+YM*TMAT(2,3)+ZM*TMAT(3,3)+TMAT(4,3)
      T4 = XM*TMAT(1,4)+YM*TMAT(2,4)+ZM*TMAT(3,4)+TMAT(4,4)

      IF (ABS(T4) .LT. 10E-6)THEN
        IERR = -1
        RETURN
      ENDIF

      XO = XO/T4
      YO = YO/T4
      ZO = ZO/T4

      RETURN
      END

C ******************** VECTRN ********************
C Transforms a vector VECIN by the 4x4 (homogeneious) matrix TMAT
C and returns the vector VECOUT. If IERR is < 0 then a fatal error.

      SUBROUTINE VECTRN(VECIN,TMAT,VECOUT,IERR)

      DIMENSION VECIN(3),VECOUT(3),TMAT(4,4)

      IERR = 0

      VECOUT(1) = VECIN(1)*TMAT(1,1) + VECIN(2)*TMAT(2,1) +
     &            VECIN(3)*TMAT(3,1) + TMAT(4,1)

      VECOUT(2) = VECIN(1)*TMAT(1,2) + VECIN(2)*TMAT(2,2) +
     &            VECIN(3)*TMAT(3,2) + TMAT(4,2)

      VECOUT(3) = VECIN(1)*TMAT(1,3) + VECIN(2)*TMAT(2,3) +
     &            VECIN(3)*TMAT(3,3) + TMAT(4,3)

      T4 = VECIN(1)*TMAT(1,4) + VECIN(2)*TMAT(2,4) +
     &            VECIN(3)*TMAT(3,4) + TMAT(4,4)

      IF (ABS(T4) .LT. 10E-6)THEN
        IERR = -1
        RETURN
      ELSE
        VECOUT(1) = VECOUT(1)/T4
        VECOUT(2) = VECOUT(2)/T4
        VECOUT(3) = VECOUT(3)/T4
        RETURN
      ENDIF

      END

C ******************** VECPLN ********************
C Returns the point of intersection X,Y,Z between a line defined
C by X1,Y1,Z1 & X2,Y2,Z2 and a plane defined in PEQN.
C IERR is -1 if denominator is close to zero or negative.
C IERR is 0 if no problem found.

      SUBROUTINE  VECPLN(X1,Y1,Z1,X2,Y2,Z2,PEQN,X,Y,Z,IERR)

      DIMENSION  PEQN(4)

      IERR=0

      F = X2 - X1
      G = Y2 - Y1
      H = Z2 - Z1

      DENOM = SQRT(F*F + G*G + H*H)

      F = F/DENOM
      G = G/DENOM
      H = H/DENOM

      DENOM = PEQN(1)*F + PEQN(2)*G + PEQN(3)*H

      IF (ABS(DENOM).LT.0.0001)THEN
        IERR=-1
        RETURN
      ENDIF

      T = -(PEQN(1)*X1+PEQN(2)*Y1+PEQN(3)*Z1-PEQN(4))/DENOM

      X = X1 + F*T
      Y = Y1 + G*T
      Z = Z1 + H*T

      RETURN
      END

C ******************** HMATMUL ********************
C Multiplies the homogenous (4x4) matrices A by B returning C.
C Matrix A is premultiplied and matrix B is postmultiplied.

      SUBROUTINE HMATMUL(A,B,C)

      DIMENSION A(4,4),B(4,4),C(4,4)

      DO 20 I=1,4
        C(I,1)=A(I,1)*B(1,1)+A(I,2)*B(2,1)+A(I,3)*B(3,1)+A(I,4)*B(4,1)
        C(I,2)=A(I,1)*B(1,2)+A(I,2)*B(2,2)+A(I,3)*B(3,2)+A(I,4)*B(4,2)
        C(I,3)=A(I,1)*B(1,3)+A(I,2)*B(2,3)+A(I,3)*B(3,3)+A(I,4)*B(4,3)
        C(I,4)=A(I,1)*B(1,4)+A(I,2)*B(2,4)+A(I,3)*B(3,4)+A(I,4)*B(4,4)
   20 CONTINUE

      RETURN
      END

C ******************** HREVMAT ********************
C Takes the homogenous perspective transformation PER and
C returns it's inverse REP making use of CROUT.

      SUBROUTINE HREVMAT(PER,REP,IERR)

      DIMENSION PER(4,4),REP(4,4),A(4,4),WS(4)

      IA=4
      IB=4
      DO 20 J=1,4
        A(J,1)=PER(J,1)
        A(J,2)=PER(J,2)
        A(J,3)=PER(J,3)
        A(J,4)=PER(J,4)
   20 CONTINUE

      CALL CROUT(A,4,WS,REP,IA,IB,IFL)

C IERR > 0, if ok, IERR=-1 if matrix singular.
      IERR=0
      IF(IFL.LT.0)IERR = -1
      RETURN

      END

C ******************** CROUT ********************
C Inverts a nonsymetric square matrix A (order N), returning
C the matrix B and IERR =-1 if matrix is singular.
C WS is a workspace vector of dimension N, IA is the
C first dimension of A and IB is the second dimension.
C based on crout factorization based on code from
C Wilkinson and Reinsch 'Linear Algebra', Springer Verlag, 1971, PP 93-110.

C The decomposition of A=LU, where L
C is a lower triangular matrix, and U is a unit
C upper triangular matrix, is performed and
C overwritten on A, omitting the unit diagonal of
C U. A record of any interchanges made to the rows
C of A is kept in WS(I), such that the I-th row and
C the WS(I)-th row were interchanged at the I-th step.
C The factorisation will fail if A, modified by the
C the rounding errors, is singular (or almost singular).
C Makes use of subroutine DPACC to accumulate
C sums of inner products.

      SUBROUTINE CROUT(A,N,WS,B,IA,IB,IER)

      DOUBLE PRECISION D1
      DIMENSION A(IA,N),B(IB,N),WS(N)

      common/OUTIN/IUOUT,IUIN,IEOUT

C Machine dependent rounding error, such that 1.0+EPS > 1.0
      EPS=2.0E-10
      IER=0
      N1=N
      DO 10 I = 1,N
        DO 20 J = 1,N
          B(I,J) = 0.0
   20   CONTINUE
        B(I,I) = 1.0
   10 CONTINUE

      KK=IA*N
      D1=1.D0
      ID=0
      DO 30 I=1,N
        I1=I
        I2=I1+IA
        CALL DPACC(A,I1,I2,KK,A,I1,I2,KK,0.0,Y,N,1)
        WS(I)=1.0/SQRT(Y)
  30  CONTINUE

      DO 40 K=1,N
        LL=K-1
        L=K
        K1=K+1
        X=0.0
        DO 50 I=K,N
          I1=I
          I2=I1+IA
          I3=LL*IA+1
          I4=I3+1
          CALL DPACC(A,I1,I2,KK,A,I3,I4,KK,A(I,K),Y,LL,2)
          A(I,K)=Y
          Y=ABS(Y*WS(I))
          IF(Y.LE.X)GO TO 50
          X=Y
          L=I
50      CONTINUE

        IF(L.EQ.K)GO TO 60
        D1=-D1
        DO 70 J=1,N
          Y=A(K,J)
          A(K,J)=A(L,J)
          A(L,J)=Y
  70    CONTINUE

        WS(L)=WS(K)
60      WS(K)=L
        D1=D1*DBLE(A(K,K))
        IF(X.LT.(8.0*EPS))THEN
          IER=-1
          call edisp(iuout,' ')
          call edisp(iuout,
     &      'CROUT: matrix is singular and cannot be inverted')
          RETURN
        ENDIF

80      IF(DABS(D1).LT.1.D0)GO TO 90
        D1=D1*0.625D-1
        ID=ID+4
        GO TO 80

90      IF(DABS(D1).GE.0.625D-4)GO TO 85
        D1=D1*0.16D2
        ID=ID-4
        GO TO 90

85      X=-1.0/A(K,K)
        IF(K.EQ.N)GO TO 40
        DO 100 J=K1,N
          I1=K
          I2=I1+IA
          I3=(J-1)*IA+1
          I4=I3+1
          CALL DPACC(A,I1,I2,KK,A,I3,I4,KK,A(K,J),Y,LL,3)
          A(K,J)=X*Y
  100   CONTINUE

40    CONTINUE

      L=IB*N
      DO 110 I=1,N
        J=INT(WS(I)+0.5)
        IF(J.EQ.I)GO TO 110
        DO 120 K=1,N1
          X=B(I,K)
          B(I,K)=B(J,K)
          B(J,K)=X
  120   CONTINUE
110   CONTINUE

      DO 130 K=1,N1
        K1=K-1
        DO 140 I=1,N
          I0=I-1
          I1=I
          I2=I1+IA
          I3=K1*IB+1
          I4=I3+1
          CALL DPACC(A,I1,I2,KK,B,I3,I4,L,B(I,K),X,I0,4)
          B(I,K)=X/A(I,I)
  140   CONTINUE

        I=N+1
        DO 150 I5=1,N
          I=I-1
          I0=N-I
          I1=I*IA+I
          I2=I1+N
          I3=K1*IB+I+1
          I4=I3+1
          CALL DPACC(A,I1,I2,KK,B,I3,I4,L,B(I,K),X,I0,4)
          B(I,K)=X
  150   CONTINUE
130   CONTINUE

      RETURN
      END

C ******************** DPACC ********************
C Provides double precision accumulation of inner products for
C CROUT in the form SUM(+,-)SUM(+,-)AB.
C A is the vector on left, I & J numerical identifiers of first two
C elements of A in the multiplaction and IJ is the dimension of A.
C For vector B the parameters K,L,KL are analogous to I,J,IJ.
C X is the quantity to be added to the product of the specified
C elements of vectors A & B.
C SUM is the result, N is a counter, IND is an indicator as follows:
C   IF IND=1 SUM=AB+X
C   IF IND=2 SUM=X-AB
C   IF IND=3 SUM=AB-X
C   IF IND=4 SUM=-AB-X = -(AB+X)

      SUBROUTINE DPACC(A,I,J,IJ,B,K,L,KL,X,SUM,N,IND)

      DOUBLE PRECISION P,Q,R
      DIMENSION A(IJ),B(KL)

      R=0.D0
      IF(I.GT.IJ.OR.K.GT.KL)GO TO 10
      IF(N.GT.0)THEN
        M=J-I
        MM1=L-K
        DO 20 IK=1,N
          I1=I+(IK-1)*M
          P=DBLE(A(I1))
          I1=K+(IK-1)*MM1
          Q=DBLE(B(I1))
          R=R+P*Q
   20   CONTINUE
      ENDIF

10    P=DBLE(X)
      IF(IND.EQ.1)THEN
        SUM=SNGL(P+R)
      ELSEIF(IND.EQ.2)THEN
        SUM=SNGL(P-R)
      ELSEIF(IND.EQ.3)THEN
        SUM=SNGL(R-P)
      ELSEIF(IND.EQ.4)THEN
        SUM=SNGL(-P-R)
      ENDIF

      RETURN
      END

C ******************** EYEMAT ********************
C Provides the transform eyepoint -> viewpoint.

      SUBROUTINE EYEMAT(EP,VP,SCALE,TMAT,RMAT)
      common/OUTIN/IUOUT,IUIN,IEOUT

      DIMENSION  EP(3),VP(3),TMAT(4,4),RMAT(4,4)
      character outs*124

      DELTA = 10E-6

      VPRIME = SQRT((EP(1)-VP(1))**2+(EP(2)-VP(2))**2)
      IF (ABS(VPRIME).GT.DELTA) GOTO 90
      VPRIME = 0.
      SINTET = 0.
      COSTET = -1.
      GOTO 100

   90 SINTET = (EP(1)-VP(1))/VPRIME
      COSTET = (EP(2)-VP(2))/VPRIME

  100 CONTINUE

      EPRIME = EP(3)-VP(3)
      IF (ABS(VPRIME).GT.DELTA) GOTO 110
      IF (ABS(EPRIME).GT.DELTA) GOTO 110
      SINPHI = 1.0
      COSPHI = 0.0
      write(outs,*) ' EP is ',EP,' VP is ',VP
      call edisp(iuout,' ')
      call edisp(iuout,outs)
      call edisp(iuout,'Eye point & viewed point too close together')
      call edisp(iuout,'so a view || to the Z-axis is assumed.')
      GOTO 130

  110 DENOM = SQRT(EPRIME**2+VPRIME**2)
      COSPHI = VPRIME/DENOM
      SINPHI = EPRIME/DENOM

  130 CONTINUE

      TMAT(1,1) = -COSTET
      TMAT(1,2) = -SINTET*SINPHI
      TMAT(1,3) = -SINTET*COSPHI
      TMAT(1,4) = 0.0

      TMAT(2,1) = SINTET
      TMAT(2,2) = -COSTET*SINPHI
      TMAT(2,3) = -COSTET*COSPHI
      TMAT(2,4) = 0.0

      TMAT(3,1) = 0.0
      TMAT(3,2) = COSPHI
      TMAT(3,3) = -SINPHI
      TMAT(3,4) = 0.0

      TMAT(4,1) = EP(1)*COSTET - EP(2)*SINTET
      TMAT(4,2) = (EP(1)*SINTET + EP(2)*COSTET)*SINPHI - EP(3)*COSPHI
      TMAT(4,3) = (EP(1)*SINTET + EP(2)*COSTET)*COSPHI + EP(3)*SINPHI
      TMAT(4,4) = 1.0/SCALE

      RMAT(1,1) = -COSTET
      RMAT(1,2) = SINTET
      RMAT(1,3) = 0.0
      RMAT(1,4) = 0.0

      RMAT(2,1) = -SINPHI*SINTET
      RMAT(2,2) = -COSTET*SINPHI
      RMAT(2,3) = COSPHI
      RMAT(2,4) = 0.0

      RMAT(3,1) = -COSPHI*SINTET
      RMAT(3,2) = -COSTET*COSPHI
      RMAT(3,3) = -SINPHI
      RMAT(3,4) = 0.0

      RMAT(4,1) = EP(1)*SCALE
      RMAT(4,2) = EP(2)*SCALE
      RMAT(4,3) = EP(3)*SCALE
      RMAT(4,4) = SCALE

      RETURN
      END

C ******************** INTSTR ********************
C Converts an integer into a string (10 char long) with no
C leading blanks. ISWD is the length of the resulting string.

      SUBROUTINE INTSTR(INTIN,FSTR,ISWD,IFLAG)

      CHARACTER*10 CSTR, FSTR
      character outs*124
      common/OUTIN/IUOUT,IUIN,IEOUT

      IFLAG=0
      ISWD=0
      fstr = ' '

C Internal write to STR.
      WRITE(CSTR,'(I10)',ERR=999)INTIN
      K=0
      DO 99 I=1,LEN(cstr)
        IF(cstr(I:I).NE.' '.OR.K.GE.1)THEN
          if(ichar(cstr(I:I)).lt.32)goto 100
          K=K+1
          fstr(K:K)=cstr(I:I)
        ENDIF
 99   CONTINUE

  100 ISWD=max(1,LNBLNK(FSTR))
      RETURN

 999  WRITE(outs,*) ' INTSTR: invalid integer: ',INTIN
      call edisp(iuout,outs)
      IFLAG=1
      RETURN

      END

C ******************** RELSTR ********************
C Converts a real into a string (12 char) with no leading
C blanks. ISWD is the actual length of the resulting string. Takes
C the magnitude of the number into account.

      SUBROUTINE RELSTR(RELIN,FSTR,ISWD,IFLAG)

      CHARACTER*12 CSTR, FSTR
      character outs*124
      logical close
      common/OUTIN/IUOUT,IUIN,IEOUT

      IFLAG=0
      ISWD=0
      fstr = ' '

C Internal write to STR. If trivally close to zero set, otherwise
C write out in an appropriate format given the magnitude of the
C real value.
      rval=relin
      CALL ECLOSE(rval,0.00,0.000001,CLOSE)
      if(close)then
        rval=0.00
        fstr='0.00'
        iswd=4
        return
      endif
      if(abs(rval).gt.1000000.0)then
        WRITE(CSTR,'(1PE12.6)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.1000000.0.and.abs(rval).gt.10000.0)then
        WRITE(CSTR,'(F12.3)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.10000.0.and.abs(rval).gt.0.1)then
        WRITE(CSTR,'(F11.5)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.0.1.and.abs(rval).gt.0.00001)then
        WRITE(CSTR,'(F12.10)',IOSTAT=ios,ERR=999)RVAL
      else
        WRITE(CSTR,'(F12.4)',IOSTAT=ios,ERR=999)RVAL
      endif
      K=0
      DO 99 I=1,LEN(cstr)
        IF(cstr(I:I).NE.' '.OR.K.GE.1)THEN
          if(ichar(cstr(I:I)).lt.32)goto 100
          K=K+1
          fstr(K:K)=cstr(I:I)
        ENDIF
 99   CONTINUE

  100 ISWD=max(1,LNBLNK(FSTR))
      RETURN

 999  if(IOS.eq.2)then
        WRITE(outs,*) ' RELSTR: permission issue: ',RELIN
      else
        WRITE(outs,*) ' RELSTR: invalid real or > 12 char: ',RELIN
      endif
      call edisp(iuout,outs)
      IFLAG=1

      RETURN
      END


C ******************** REL16STR ********************
C Converts a real into a string (16 char) with no leading
C blanks. ISWD is the actual length of the resulting string. Takes
C the magnitude of the number into account.

      SUBROUTINE REL16STR(RELIN,FSTR,ISWD,IFLAG)

      CHARACTER*16 CSTR, FSTR
      character outs*124
      logical close

      common/OUTIN/IUOUT,IUIN,IEOUT

      IFLAG=0
      ISWD=0
      fstr = ' '

C Internal write to STR. If trivally close to zero set, otherwise
C write out in an appropriate format given the magnitude of the
C real value.
      rval=relin
      CALL ECLOSE(rval,0.00,0.000001,CLOSE)
      if(close)then
        rval=0.00
        fstr='0.00'
        iswd=4
        return
      endif
      if(abs(rval).gt.1000000.0)then
        WRITE(CSTR,'(1PE14.6)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.1000000.0.and.abs(rval).gt.10000.0)then
        WRITE(CSTR,'(G15.5)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.10000.0.and.abs(rval).gt.10.0)then
        WRITE(CSTR,'(G14.5)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.10.0.and.abs(rval).gt.0.1)then
        WRITE(CSTR,'(G14.5)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.0.1.and.abs(rval).gt.0.00001)then
        WRITE(CSTR,'(F10.7)',IOSTAT=ios,ERR=999)RVAL
      elseif(rval.le.0.00001)then
        WRITE(CSTR,'(1PE14.6)',IOSTAT=ios,ERR=999)RVAL
      else
        WRITE(CSTR,'(F14.4)',IOSTAT=ios,ERR=999)RVAL
      endif
      K=0
      DO 99 I=1,LEN(cstr)
        IF(cstr(I:I).NE.' '.OR.K.GE.1)THEN
          if(ichar(cstr(I:I)).lt.32)goto 100
          K=K+1
          fstr(K:K)=cstr(I:I)
        ENDIF
 99   CONTINUE

  100 ISWD=max(1,LNBLNK(FSTR))
      RETURN

 999  if(IOS.eq.2)then
        WRITE(outs,*) ' REL16STR: permission issue: ',RELIN
      else
        WRITE(outs,*) ' REL16STR: invalid real or > 16 char: ',RELIN
      endif
      call edisp(iuout,outs)
      IFLAG=1
      RETURN

      END

C ******************** REL10STR ********************
C Converts a real into a string (10 char) with no leading
C blanks. ISWD is the actual length of the resulting string. Takes
C the magnitude of the number into account but typically only
C writes to 2 decimal places.

      SUBROUTINE REL10STR(RELIN,FSTR,ISWD,IFLAG)

      CHARACTER*10 CSTR, FSTR
      character outs*124
      logical close

      common/OUTIN/IUOUT,IUIN,IEOUT

      IFLAG=0
      ISWD=0
      fstr = ' '

C Internal write to STR. If trivally close to zero set, otherwise
C write out in an appropriate format given the magnitude of the
C real value.
      rval=relin
      CALL ECLOSE(rval,0.00,0.000001,CLOSE)
      if(close)then
        rval=0.00
        fstr='0.00'
        iswd=4
        return
      endif
      if(abs(rval).gt.1000000.0)then
        WRITE(CSTR,'(1PE10.4)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.1000000.0.and.abs(rval).gt.10000.0)then
        WRITE(CSTR,'(G10.2)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.10000.0.and.abs(rval).gt.10.0)then
        WRITE(CSTR,'(G10.2)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.10.0.and.abs(rval).gt.0.1)then
        WRITE(CSTR,'(G10.2)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.0.1.and.abs(rval).gt.0.001)then
        WRITE(CSTR,'(F10.3)',IOSTAT=ios,ERR=999)RVAL
      elseif(rval.le.0.001)then
        WRITE(CSTR,'(1PE10.5)',IOSTAT=ios,ERR=999)RVAL
      else
        WRITE(CSTR,'(F10.4)',IOSTAT=ios,ERR=999)RVAL
      endif
      K=0
      DO 99 I=1,LEN(cstr)
        IF(cstr(I:I).NE.' '.OR.K.GE.1)THEN
          if(ichar(cstr(I:I)).lt.32)goto 100
          K=K+1
          fstr(K:K)=cstr(I:I)
        ENDIF
 99   CONTINUE

  100 ISWD=max(1,LNBLNK(FSTR))
      RETURN

 999  if(IOS.eq.2)then
        WRITE(outs,*) ' REL10STR: permission issue: ',RELIN
      else
        WRITE(outs,*) ' REL10STR: invalid real or > 10 char: ',RELIN
      endif
      call edisp(iuout,outs)
      IFLAG=1

      RETURN
      END


C ******************** REL12STR ********************
C Converts a real into a string (12 char) with no leading
C blanks. ISWD is the actual length of the resulting string. Takes
C the magnitude of the number into account.

      SUBROUTINE REL12STR(RELIN,FSTR,ISWD,IFLAG)

      CHARACTER*12 CSTR, FSTR
      character outs*124
      logical close

      common/OUTIN/IUOUT,IUIN,IEOUT

      IFLAG=0
      ISWD=0
      fstr = ' '

C Internal write to STR. If trivally close to zero set, otherwise
C write out in an appropriate format given the magnitude of the
C real value.
      rval=relin
      CALL ECLOSE(rval,0.00,0.000001,CLOSE)
      if(close)then
        rval=0.00
        fstr='0.00'
        iswd=4
        return
      endif
      if(abs(rval).gt.1000000.0)then
        WRITE(CSTR,'(1PE12.6)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.1000000.0.and.abs(rval).gt.10000.0)then
        WRITE(CSTR,'(G12.5)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.10000.0.and.abs(rval).gt.10.0)then
        WRITE(CSTR,'(G12.5)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.10.0.and.abs(rval).gt.0.1)then
        WRITE(CSTR,'(G12.4)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.0.1.and.abs(rval).gt.0.00001)then
        WRITE(CSTR,'(F10.6)',IOSTAT=ios,ERR=999)RVAL
      elseif(rval.le.0.00001)then
        WRITE(CSTR,'(1PE12.6)',IOSTAT=ios,ERR=999)RVAL
      else
        WRITE(CSTR,'(F14.4)',IOSTAT=ios,ERR=999)RVAL
      endif
      K=0
      DO 99 I=1,LEN(cstr)
        IF(cstr(I:I).NE.' '.OR.K.GE.1)THEN
          if(ichar(cstr(I:I)).lt.32)goto 100
          K=K+1
          fstr(K:K)=cstr(I:I)
        ENDIF
 99   CONTINUE

  100 ISWD=max(1,LNBLNK(FSTR))
      RETURN

 999  if(IOS.eq.2)then
        WRITE(outs,*) ' REL12STR: permission issue: ',RELIN
      else
        WRITE(outs,*) ' REL12STR: invalid real or > 12 char: ',RELIN
      endif
      call edisp(iuout,outs)
      IFLAG=1

      RETURN
      END

C ******************** RELESTR ********************
C Converts a real into a string (16 char) with no leading
C blanks. ISWD is the actual length of the resulting string. Takes
C the magnitude of the number into account.

      SUBROUTINE RELESTR(RELIN,FSTR,ISWD,IFLAG)

      CHARACTER*16 CSTR, FSTR
      character outs*124
      logical close

      common/OUTIN/IUOUT,IUIN,IEOUT

      IFLAG=0
      ISWD=0
      fstr = ' '

C Internal write to STR. If trivally close to zero set, otherwise
C write out in an appropriate format given the magnitude of the
C real value.
      rval=relin
      CALL ECLOSE(rval,0.00,0.000001,CLOSE)
      if(close)then
        rval=0.00
        fstr='0.00'
        iswd=4
        return
      endif
      if(abs(rval).gt.1000000.0)then
        WRITE(CSTR,'(E11.4)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.1000000.0.and.abs(rval).gt.10000.0)then
        WRITE(CSTR,'(F8.0)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.10000.0.and.abs(rval).gt.100.)then
        WRITE(CSTR,'(f6.0)',IOSTAT=ios,ERR=999)RVAL
      elseif(abs(rval).le.100.0.and.abs(rval).gt.0.001)then
        WRITE(CSTR,'(F7.3)',IOSTAT=ios,ERR=999)RVAL
      else
        WRITE(CSTR,'(E11.4)',IOSTAT=ios,ERR=999)RVAL
      endif
      K=0
      DO 99 I=1,LEN(cstr)
        IF(cstr(I:I).NE.' '.OR.K.GE.1)THEN
          if(ichar(cstr(I:I)).lt.32)goto 100
          K=K+1
          fstr(K:K)=cstr(I:I)
        ENDIF
 99   CONTINUE

  100 ISWD=max(1,LNBLNK(FSTR))
      RETURN

 999  if(IOS.eq.2)then
        WRITE(outs,*) ' RELESTR: permission issue: ',RELIN
      else
        WRITE(outs,*) ' RELESTR: invalid real or > 12 char: ',RELIN
      endif
      call edisp(iuout,outs)
      IFLAG=1
      RETURN

      END


C ******************** EXPSTR ********************
C Converts a exponential into a string (10 char) with no leading
C blanks. ISWD is the actual length of the resulting string.

      SUBROUTINE EXPSTR(RELIN,FSTR,ISWD,IFLAG)

      character*10 CSTR, FSTR
      character outs*124

      common/OUTIN/IUOUT,IUIN,IEOUT

      ISWD=0
      fstr = ' '

C Internal write to STR.
      WRITE(CSTR,'(1PE10.3)',IOSTAT=ISTAT,ERR=999)RELIN
      K=0
      DO 99 I=1,LEN(cstr)
        IF(cstr(I:I).NE.' '.OR.K.GE.1)THEN
          if(ichar(cstr(I:I)).lt.32)goto 100
          K=K+1
          fstr(K:K)=cstr(I:I)
        ENDIF
 99   CONTINUE

  100 ISWD=max(1,LNBLNK(FSTR))
      RETURN

 999  WRITE(outs,*) ' EXPSTR: invalid real or > 10 char: ',RELIN
      call edisp(iuout,outs)
      IFLAG=1
      RETURN

      END

C ******************** ARLIST ********************
C Takes the first (inst to inrl) items of a real array (rlist)
C of array size (inrs) and builds a packed string (pckstr)
C and returns pckstr and the actual character width (length). If itrunc
C is zero then all items written, else the index of the last item
C which was written. Delm is delimeter between items.

      SUBROUTINE ARLIST(inst,inrl,rlist,inrs,delm,pckstr,length,itrunc)

      dimension rlist(inrs)
      character*(*) pckstr
      CHARACTER item*16,delm*1

      itrunc=0
      LS=LEN(pckstr)
      length=0

C Proceed to fill as much of pckstr as possible, set ierr to one
C if the array cannot be written out fully in the space available.
      pckstr=' '
      ix=1
      ixl=0
      do 43 i=inst,inrl
        item=' '
        rval=rlist(i)
        call rel16str(rval,item,lna,ier)

C Write next portion of pckstr and if not at the end write a
C separating comma. If attempting to write past the end of
C pckstr then set itrunc to the last successfully written
C rlist index.
        if(lna.eq.1)then
          ixl=ix
        else
          ixl=ix+(lna-1)
        endif
        if(ixl+1.lt.LS)then
          write(pckstr(ix:ixl),'(a)')item(1:lna)
          if(i.lt.inrl)then
            if(delm.eq.'T')then
              write(pckstr(ixl+1:ixl+1),'(a)') CHAR(9)
            elseif(delm.eq.'S')then
              write(pckstr(ixl+1:ixl+1),'(a)') ' '
            elseif(delm.eq.'C')then
              write(pckstr(ixl+1:ixl+1),'(a)') ','
            endif
            ix=ix+lna+1
          else
            ix=ix+lna+1
          endif
        else
          length=ix
          itrunc = i-1
          goto 1
        endif
  43  continue
      length=ixl
      return

C The full array could not be packed into the string so return
C the position this happened so calling code can deal with it.
   1  continue

      return
      END

C ******************** ARLIST2 ********************
C Takes the first (inst to inrl) items of a real array (rlist)
C of array size (inrs) and builds a packed string (pckstr) with 12 char
C string representations of the values and returns
C pckstr and the actual character width (length). If itrunc
C is zero then all items written, else the index of the last item
C which was written. Delm is delimeter between items.

      SUBROUTINE ARLIST2(inst,inrl,rlist,inrs,delm,pckstr,length,itrunc)

      dimension rlist(inrs)
      character*(*) pckstr
      CHARACTER item*12,delm*1  ! Note 12 character

      itrunc=0
      LS=LEN(pckstr)
      length=0

C Proceed to fill as much of pckstr as possible, set ierr to one
C if the array cannot be written out fully in the space available.
      pckstr=' '
      ix=1
      ixl=0
      do 43 i=inst,inrl
        item=' '
        rval=rlist(i)
        call rel12str(rval,item,lna,ier)

C Write next portion of pckstr and if not at the end write a
C separating comma. If attempting to write past the end of
C pckstr then set itrunc to the last successfully written
C rlist index.
        if(lna.eq.1)then
          ixl=ix
        else
          ixl=ix+(lna-1)
        endif
        if(ixl+1.lt.LS)then
          write(pckstr(ix:ixl),'(a)')item(1:lna)
          if(i.lt.inrl)then
            if(delm.eq.'T')then
              write(pckstr(ixl+1:ixl+1),'(a)') CHAR(9)
            elseif(delm.eq.'S')then
              write(pckstr(ixl+1:ixl+1),'(a)') ' '
            elseif(delm.eq.'C')then
              write(pckstr(ixl+1:ixl+1),'(a)') ','
            endif
            ix=ix+lna+1
          else
            ix=ix+lna+1
          endif
        else
          length=ix
          itrunc = i-1
          goto 1
        endif
  43  continue
      length=ixl
      return

C The full array could not be packed into the string so return
C the position this happened so calling code can deal with it.
   1  continue

      return
      END


C ******************** ARLIST3 ********************
C Takes the first (inst to inrl) items of a real array (rlist)
C of array size (inrs) and builds a packed string (pckstr) with 10 char
C string representations of the values and returns
C pckstr and the actual character width (length). If itrunc
C is zero then all items written, else the index of the last item
C which was written. Delm is delimeter between items.

      SUBROUTINE ARLIST3(inst,inrl,rlist,inrs,delm,pckstr,length,itrunc)

      dimension rlist(inrs)
      character*(*) pckstr
      CHARACTER item*10,delm*1  ! Note 10 character

      itrunc=0
      LS=LEN(pckstr)
      length=0

C Proceed to fill as much of pckstr as possible, set ierr to one
C if the array cannot be written out fully in the space available.
      pckstr=' '
      ix=1
      ixl=0
      do 43 i=inst,inrl
        item=' '
        rval=rlist(i)
        call rel10str(rval,item,lna,ier)

C Write next portion of pckstr and if not at the end write a
C separating comma. If attempting to write past the end of
C pckstr then set itrunc to the last successfully written
C rlist index.
        if(lna.eq.1)then
          ixl=ix
        else
          ixl=ix+(lna-1)
        endif
        if(ixl+1.lt.LS)then
          write(pckstr(ix:ixl),'(a)')item(1:lna)
          if(i.lt.inrl)then
            if(delm.eq.'T')then
              write(pckstr(ixl+1:ixl+1),'(a)') CHAR(9)
            elseif(delm.eq.'S')then
              write(pckstr(ixl+1:ixl+1),'(a)') ' '
            elseif(delm.eq.'C')then
              write(pckstr(ixl+1:ixl+1),'(a)') ','
            endif
            ix=ix+lna+1
          else
            ix=ix+lna+1
          endif
        else
          length=ix
          itrunc = i-1
          goto 1
        endif
  43  continue
      length=ixl
      return

C The full array could not be packed into the string so return
C the position this happened so calling code can deal with it.
   1  continue

      return
      END


C ******************** AELIST ********************
C Takes the first (inst to inrl) items of a real array (rlist)
C of array size (inrs) and builds a packed string (pckstr)
C and returns pckstr and the actual character width (length). If itrunc
C is zero then all items written, else the index of the last item
C which was written. Delm is delimeter between items.
C This subroutine is the same as ARLIST but the final string written
C out is shorter.

      SUBROUTINE AELIST(inst,inrl,rlist,inrs,delm,pckstr,length,itrunc)

      dimension rlist(inrs)
      character*(*) pckstr
      CHARACTER item*16,delm*1

      itrunc=0
      LS=LEN(pckstr)
      length=0

C Proceed to fill as much of pckstr as possible, set ierr to one
C if the array cannot be written out fully in the space available.
      pckstr=' '
      ix=1
      ixl=0
      do 43 i=inst,inrl
        item=' '
        rval=rlist(i)
        call relestr(rval,item,lna,ier)

C Write next portion of pckstr and if not at the end write a
C separating comma. If attempting to write past the end of
C pckstr then set itrunc to the last successfully written
C rlist index.
        if(lna.eq.1)then
          ixl=ix
        else
          ixl=ix+(lna-1)
        endif
        if(ixl+1.lt.LS)then
          write(pckstr(ix:ixl),'(a)')item(1:lna)
          if(i.lt.inrl)then
            if(delm.eq.'T')then
              write(pckstr(ixl+1:ixl+1),'(a)') CHAR(9)
            elseif(delm.eq.'S')then
              write(pckstr(ixl+1:ixl+1),'(a)') ' '
            elseif(delm.eq.'C')then
              write(pckstr(ixl+1:ixl+1),'(a)') ','
            endif
            ix=ix+lna+1
          else
            ix=ix+lna+1
          endif
        else
          length=ix
          itrunc = i-1
          goto 1
        endif
  43  continue
      length=ixl
      return

C The full array could not be packed into the string so return
C the position this happened so calling code can deal with it.
   1  continue

      return
      END

C ******************** AILIST ********************
C Takes the range (inst to inil) items of an integer array (ilist)
C of array size (inisz) and builds a packed string (pckstr) and
C returns pckstr and written character width (length). If itrunc
C is zero then all items written, else the index of the last item
C which was written. Delm is delimeter between items.

      SUBROUTINE AILIST(inst,inil,ilist,inisz,delm,pckstr,length,itrunc)

      dimension ilist(inisz)
      character*(*) pckstr
      CHARACTER item*10,delm*1

      itrunc=0
      LS=LEN(pckstr)
      length=0

C Proceed to fill as much of pckstr as possible, set itrunc to the
C last sucessful items of the array cannot be written out fully in
C the space available.
      pckstr=' '
      ix=1
      ixl=0
      do 43 i=inst,inil
        item=' '
        ival=ilist(i)
        CALL INTSTR(ival,item,lna,IER)

C Write next portion of pckstr and if not at the end write a
C separating comma before looping back for another item. If a
C single character then ixl is ix.
        if(lna.eq.1)then
          ixl=ix
        else
          ixl=ix+(lna-1)
        endif
        if(ixl+1.lt.LS)then
          write(pckstr(ix:ixl),'(a)')item(1:lna)
          if(i.lt.inil)then
            if(delm.eq.'T')then
              write(pckstr(ixl+1:ixl+1),'(a)') CHAR(9)
            elseif(delm.eq.'S')then
              write(pckstr(ixl+1:ixl+1),'(a)') ' '
            elseif(delm.eq.'C')then
              write(pckstr(ixl+1:ixl+1),'(a)') ','
            endif
            ix=ix+lna+1
          else
            ix=ix+lna
          endif
        else
          length=ix
          itrunc = i-1
          goto 1
        endif
  43  continue
      length=ixl
      return

C The full array could not be packed into the string so return
C the position this happened so calling code can deal with it.
   1  continue

      return
      END

C ******************** ASLIST ********************
C Takes the range (inst to inil) items of an string array (list)
C of array size (inisz) and builds a packed string (pckstr) and
C returns pckstr and written character width (length). If itrunc
C is zero then all items written, else the index of the last item
C which was written. Delm is delimeter between items. It is assumed
C that each string array item is less than 32 characters wide.

      SUBROUTINE ASLIST(inst,inil,list,inisz,delm,pckstr,length,itrunc)

      dimension list(inisz)
      character*(*) pckstr,list
      CHARACTER item*32,delm*1

      itrunc=0
      LS=LEN(pckstr)
      length=0

C Proceed to fill as much of pckstr as possible, set itrunc to the
C last sucessful items of the array cannot be written out fully in
C the space available.
      pckstr=' '
      ix=1
      ixl=0
      do 43 i=inst,inil
        lna=max(1,LNBLNK(list(i)))
        if(lna.gt.32) lna=32
        write(item,'(a)') list(i)(1:lna)

C Write next portion of pckstr and if not at the end write a
C separating comma before looping back for another item. If a
C single character then ixl is ix.
        if(lna.eq.1)then
          ixl=ix
        else
          ixl=ix+(lna-1)
        endif
        if(ixl+1.lt.LS)then
          write(pckstr(ix:ixl),'(a)')item(1:lna)
          if(i.lt.inil)then
            if(delm.eq.'T')then
              write(pckstr(ixl+1:ixl+1),'(a)') CHAR(9)
            elseif(delm.eq.'S')then
              write(pckstr(ixl+1:ixl+1),'(a)') ' '
            elseif(delm.eq.'C')then
              write(pckstr(ixl+1:ixl+1),'(a)') ','
            endif
            ix=ix+lna+1
          else
            ix=ix+lna+1
          endif
        else
          length=ix
          itrunc = i-1
          goto 1
        endif
  43  continue
      length=ixl
      return

C The full array could not be packed into the string so return
C the position this happened so calling code can deal with it.
   1  continue

      return
      END

C ******************** ASLIST2 ********************
C Takes the range (inst to inil) items of an string array (list)
C of 2_dimensional array size (inisz,inisz2) and builds a packed string
C (pckstr) of the row (inirw) and
C returns pckstr and written character width (length). If itrunc
C is zero then all items written, else the index of the last item
C which was written. Delm is delimeter between items. It is assumed
C that each string array item is less than 24 characters wide.

      SUBROUTINE ASLIST2(inst,inil,list,inisz,inisz2,inirw,
     &delm,pckstr,length,itrunc)

      dimension list(inisz,inisz2)
      character*(*) pckstr,list
      CHARACTER item*24,delm*1

      itrunc=0
      LS=LEN(pckstr)
      length=0

C Proceed to fill as much of pckstr as possible, set itrunc to the
C last sucessful items of the array cannot be written out fully in
C the space available.
      pckstr=' '
      ix=1
      ixl=0
      do 43 i=inst,inil
        lna=max(1,LNBLNK(list(i,inirw)))
        if(lna.gt.24) lna=24
        write(item,'(a)') list(i,inirw)(1:lna)

C Write next portion of pckstr and if not at the end write a
C separating comma before looping back for another item. If a
C single character then ixl is ix.
        if(lna.eq.1)then
          ixl=ix
        else
          ixl=ix+(lna-1)
        endif
        if(ixl+1.lt.LS)then
          write(pckstr(ix:ixl),'(a)')item(1:lna)
          if(i.lt.inil)then
            if(delm.eq.'T')then
              write(pckstr(ixl+1:ixl+1),'(a)') CHAR(9)
            elseif(delm.eq.'S')then
              write(pckstr(ixl+1:ixl+1),'(a)') ' '
            elseif(delm.eq.'C')then
              write(pckstr(ixl+1:ixl+1),'(a)') ','
            endif
            ix=ix+lna+1
          else
            ix=ix+lna+1
          endif
        else
          length=ix
          itrunc = i-1
          goto 1
        endif
  43  continue
      length=ixl
      return

C The full array could not be packed into the string so return
C the position this happened so calling code can deal with it.
   1  continue

      return
      END

C ******************** ASFLIST ********************
C Takes the range (inst to inil) items of an string array (list)
C of array size (inisz) and builds a packed string (pckstr) and
C returns pckstr and written character width (length). If itrunc
C is zero then all items written, else the index of the last item
C which was written. Delm is delimeter between items. It is assumed
C that each string array item is less than 48 characters wide.

      SUBROUTINE ASFLIST(inst,inil,list,inisz,delm,pckstr,length,itrunc)

      dimension list(inisz)
      character*(*) pckstr,list
      CHARACTER item*48,delm*1

      itrunc=0
      LS=LEN(pckstr)
      length=0

C Proceed to fill as much of pckstr as possible, set itrunc to the
C last sucessful items of the array cannot be written out fully in
C the space available.
      pckstr=' '
      ix=1
      ixl=0
      do 43 i=inst,inil
        lna=max(1,LNBLNK(list(i)))
        if(lna.gt.48) lna=48
        write(item,'(a)') list(i)(1:lna)

C Write next portion of pckstr and if not at the end write a
C separating comma before looping back for another item. If a
C single character then ixl is ix.
        if(lna.eq.1)then
          ixl=ix
        else
          ixl=ix+(lna-1)
        endif
        if(ixl+1.lt.LS)then
          write(pckstr(ix:ixl),'(a)')item(1:lna)
          if(i.lt.inil)then
            if(delm.eq.'T')then
              write(pckstr(ixl+1:ixl+1),'(a)') CHAR(9)
            elseif(delm.eq.'S')then
              write(pckstr(ixl+1:ixl+1),'(a)') ' '
            elseif(delm.eq.'C')then
              write(pckstr(ixl+1:ixl+1),'(a)') ','
            endif
            ix=ix+lna+1
          else
            ix=ix+lna+1
          endif
        else
          length=ix
          itrunc = i-1
          goto 1  ! cannot write more to the packed string
        endif
  43  continue
      length=ixl
      return

   1  continue

      return
      END

C ******************** STRIPC ********************
C Strips comments from a ASCII file string and returns the data.
C It assumes that if a string begins with a '#' then the whole line is
C a comment an the next line is read.  If a ' #' is discovered within
C a line the rest of the line is removed.
C IER=0 if ok. MSG is a text string used in error messages. If
C IR=0 then acts silently, otherwise notes when EOF found.
C IEXP is the number of expected items in the line:
C   IEXP = 0 means don't care or already know no. items - don't check
C   IEXP >0  means a specific number of items expected (error if not)
C   IEXP = 99 check number of items and return in ITEMS

      SUBROUTINE STRIPC(INPCH,OUTSTR,IEXP,ITEMS,IR,MSG,IER)
#include "espriou.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      CHARACTER*124 tmp,STRING,OUTSTR
      character loutstr*248,outs*124
      CHARACTER*(*) MSG
      integer iCountWords
      logical unixok

C Read a line of the file, strip off any trailing blanks, if the first
C character is a # then read the next line from the file.
      IER=0
    8 READ(INPCH,10,IOSTAT=ISTAT,ERR=101,END=102)STRING
   10 FORMAT(A124)
      tmp=STRING(1:LNBLNK(STRING))

C Take the string and check for a #, discarding any text which follows.
      iloc = INDEX(tmp,'#')
      if(iloc.eq.1)then
        goto 8
      elseif(iloc.eq.0)then
        OUTSTR=tmp
      elseif(iloc.gt.1)then
        OUTSTR=tmp(1:ILOC-1)
      endif

C Find out the number of separate words/string groupings.
      if(IEXP.eq.99)then
        items = iCountWords(OUTSTR)
      elseif(IEXP.eq.0)then
        ITEMS=0
      elseif(IEXP.gt.0)then
        items = iCountWords(OUTSTR)
        if(IEXP.ne.ITEMS)then
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(3A,I3,3A)')'In ',currentfile(1:LN),
     &        ' expecting',IEXP,' items (',MSG(1:LNM),') in...'
            CALL LUSRMSG(loutstr,OUTSTR,'F')
          else
            WRITE(loutstr,'(A,I3,3A)')' Expecting',IEXP,' items (',
     &         MSG(1:LNM),') in...'
            CALL LUSRMSG(loutstr,OUTSTR,'F')
          endif
        endif
      endif

    4 RETURN

  101 IER=1
      IF(IR.EQ.1)THEN
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5A)')' In ',currentfile(1:LN),
     &      ' error reading (',MSG(1:LNM),') in...'
        else
          WRITE(loutstr,'(A,A)',IOSTAT=IOS,ERR=1)' Error reading: ',
     &      MSG(1:LNM),') in...'
        endif
        CALL LUSRMSG(loutstr,OUTSTR,'W')
      ENDIF
      goto 4

  102 IER=2
      IF(IR.EQ.1)THEN
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5A)')' In ',currentfile(1:LN),
     &      ' error reading (',MSG(1:LNM),') EOF sensed.'
        else
          WRITE(loutstr,'(3A)',IOSTAT=IOS,ERR=1)' Error reading ',
     &      MSG(1:LNM),' EOF sensed.'
        endif
        CALL LUSRMSG(' ',loutstr,'-')
      ENDIF
      goto 4

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(outs,*) 'STRIPC: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,outstr(1:72))
      else
        write(outs,*) 'STRIPC: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,outstr(1:72))
      endif

      END

C ******************** LSTRIPC ********************
C Strips comments from a ASCII file (long) string and returns the data.
C It assumes that if a string begins with a '#' then the whole line is
C a comment an the next line is read.  If a ' #' is discovered within
C a line the rest of the line is removed.
C IER=0 if ok. MSG is a text string used in error messages. If
C IR=0 then acts silently, otherwise notes when EOF found.
C IEXP is the number of expected items in the line:
C   IEXP = 0 means don't care or already know no. items - don't check
C   IEXP >0  means a specific number of items expected (error if not)
C   IEXP = 99 check number of items and return in ITEMS

      SUBROUTINE LSTRIPC(INPCH,OUTSTR,IEXP,ITEMS,IR,MSG,IER)
#include "espriou.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      CHARACTER*248 OUTSTR,STRING,TMP
      CHARACTER loutstr*248,outs*124
      CHARACTER*(*) MSG
      logical unixok
      integer iCountWords

C Read a line of the file, strip off any trailing blanks, if the first
C character is a # then read the next line from the file.
      IER=0
    8 READ(INPCH,10,IOSTAT=ISTAT,ERR=101,END=102)STRING
   10 FORMAT(A)
      tmp=STRING(1:LNBLNK(STRING))

C Take the string and check for a #, discarding any text which follows.
      iloc = INDEX(tmp,'#')
      if(iloc.eq.1)then
        goto 8
      elseif(iloc.eq.0)then
        OUTSTR=tmp
      elseif(iloc.gt.1)then
        OUTSTR=tmp(1:ILOC-1)
      endif

C Find out the number of separate words/string groupings.
      if(IEXP.eq.99)then
        items = iCountWords(OUTSTR)
      elseif(IEXP.eq.0)then
        ITEMS=0
      elseif(IEXP.gt.0)then
        items = iCountWords(OUTSTR)
        if(IEXP.ne.ITEMS)then
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(3A,I3,3A)')' In ',currentfile(1:LN),
     &        ' expecting',IEXP,' items (',MSG(1:LNM),') in...'
            CALL LUSRMSG(loutstr,OUTSTR,'F')
          else
            WRITE(loutstr,'(A,I3,3A)')' Expecting',IEXP,' items (',
     &         MSG(1:LNM),') in...'
            CALL LUSRMSG(loutstr,OUTSTR,'F')
          endif
        endif
      endif

    4 RETURN

  101 IER=1
      IF(IR.EQ.1)THEN
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5A)')' In ',currentfile(1:LN),
     &      ' error reading (',MSG(1:LNM),') in...'
        else
          WRITE(loutstr,'(A,A)',IOSTAT=IOS,ERR=1)' Error reading: ',
     &      MSG(1:LNM),') in...'
        endif
        CALL LUSRMSG(loutstr,OUTSTR,'W')
      ENDIF
      goto 4

  102 IER=2
      IF(IR.EQ.1)THEN
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5A)')' In ',currentfile(1:LN),
     &      ' error reading (',MSG(1:LNM),') EOF sensed.'
        else
          WRITE(loutstr,'(3A)',IOSTAT=IOS,ERR=1)' Error reading ',
     &      MSG(1:LNM),' EOF sensed.'
        endif
        CALL LUSRMSG(' ',loutstr,'-')
      ENDIF
      goto 4

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(outs,*) 'LSTRIPC: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,outstr(1:72))
      else
        write(outs,*) 'LSTRIPC: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,outstr(1:72))
      endif

      return
      END

C ******************** STRIPC400 ********************
C data. Strips comments from a ASCII file (400 char long) string and returns the
C It assumes that if a string begins with a '#' then the whole line is
C a comment an the next line is read.  If a ' #' is discovered within
C a line the rest of the line is removed.
C IER=0 if ok. MSG is a text string used in error messages. If
C IR=0 then acts silently, otherwise notes when EOF found.
C IEXP is the number of expected items in the line:
C   IEXP = 0 means don't care or already know no. items - don't check
C   IEXP >0  means a specific number of items expected (error if not)
C   IEXP = 99 check number of items and return in ITEMS

      SUBROUTINE STRIPC400(INPCH,OUTSTR,IEXP,ITEMS,IR,MSG,IER)
#include "espriou.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      CHARACTER*400 OUTSTR,STRING,TMP
      CHARACTER loutstr*248,outs*124
      character truncstr*248   ! truncate outsr for reporting.
      CHARACTER*(*) MSG
      logical unixok
      integer iCountWords

C Read a line of the file, strip off any trailing blanks, if the first
C character is a # then read the next line from the file.
      IER=0
    8 READ(INPCH,10,IOSTAT=ISTAT,ERR=101,END=102)STRING
   10 FORMAT(A)
      tmp=STRING(1:LNBLNK(STRING))

C Take the string and check for a #, discarding any text which follows.
      iloc = INDEX(tmp,'#')
      if(iloc.eq.1)then
        goto 8
      elseif(iloc.eq.0)then
        OUTSTR=tmp
      elseif(iloc.gt.1)then
        OUTSTR=tmp(1:ILOC-1)
      endif
      lastcc = MIN0(LNBLNK(OUTSTR),247)
      write(truncstr,'(a)') OUTSTR(1:lastcc)

C Find out the number of separate words/string groupings.
      if(IEXP.eq.99)then
        items = iCountWords(OUTSTR)
      elseif(IEXP.eq.0)then
        ITEMS=0
      elseif(IEXP.gt.0)then
        items = iCountWords(OUTSTR)
        if(IEXP.ne.ITEMS)then
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(3A,I3,3A)')' In ',currentfile(1:LN),
     &        ' expecting',IEXP,' items (',MSG(1:LNM),') in...'
            CALL LUSRMSG(loutstr,truncstr,'F')
          else
            WRITE(loutstr,'(A,I3,3A)')' Expecting',IEXP,' items (',
     &         MSG(1:LNM),') in...'
            CALL LUSRMSG(loutstr,truncstr,'F')
          endif
        endif
      endif

    4 RETURN

  101 IER=1
      IF(IR.EQ.1)THEN
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5A)')' In ',currentfile(1:LN),
     &      ' error reading (',MSG(1:LNM),') in...'
        else
          WRITE(loutstr,'(A,A)',IOSTAT=IOS,ERR=1)' Error reading: ',
     &      MSG(1:LNM),') in...'
        endif
        CALL LUSRMSG(loutstr,OUTSTR,'W')
      ENDIF
      goto 4

  102 IER=2
      IF(IR.EQ.1)THEN
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5A)')' In ',currentfile(1:LN),
     &      ' error reading (',MSG(1:LNM),') EOF sensed.'
        else
          WRITE(loutstr,'(3A)',IOSTAT=IOS,ERR=1)' Error reading ',
     &      MSG(1:LNM),' EOF sensed.'
        endif
        CALL LUSRMSG(' ',loutstr,'-')
      ENDIF
      goto 4

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(outs,*) 'LSTRIPC4: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,outstr(1:72))
      else
        write(outs,*) 'LSTRIPC4: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,outstr(1:72))
      endif

      return
      END


C ******************** STRIPC1K ********************
C Strips comments from a ASCII file (1000 char long) string and returns the
C data. It assumes that if a string begins with a '#' then the whole line is
C a comment an the next line is read.  If a ' #' is discovered within
C a line the rest of the line is removed.
C IER=0 if ok. MSG is a text string used in error messages. If
C IR=0 then acts silently, otherwise notes when EOF found.
C IEXP is the number of expected items in the line:
C   IEXP = 0 means don't care or already know no. items - don't check
C   IEXP >0  means a specific number of items expected (error if not)
C   IEXP = 99 check number of items and return in ITEMS

      SUBROUTINE STRIPC1K(INPCH,OUTSTR,IEXP,ITEMS,IR,MSG,IER)
#include "espriou.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      CHARACTER*1000 OUTSTR,STRING,TMP
      CHARACTER loutstr*248,outs*124
      character truncstr*248   ! truncate outsr for reporting.
      CHARACTER*(*) MSG
      logical unixok
      integer iCountWords

C Read a line of the file, strip off any trailing blanks, if the first
C character is a # then read the next line from the file.
      IER=0
    8 READ(INPCH,10,IOSTAT=ISTAT,ERR=101,END=102)STRING
   10 FORMAT(A)
      tmp=STRING(1:LNBLNK(STRING))

C Take the string and check for a #, discarding any text which follows.
      iloc = INDEX(tmp,'#')
      if(iloc.eq.1)then
        goto 8
      elseif(iloc.eq.0)then
        OUTSTR=tmp
      elseif(iloc.gt.1)then
        OUTSTR=tmp(1:ILOC-1)
      endif

C For messages we need to remember the initial part of the long string.
      lastcc = MIN0(LNBLNK(OUTSTR),247)
      write(truncstr,'(a)') OUTSTR(1:lastcc)

C Find out the number of separate words/string groupings.
      if(IEXP.eq.99)then
        ITEMS = iCountWords(OUTSTR)
      elseif(IEXP.eq.0)then
        ITEMS=0
      elseif(IEXP.gt.0)then
        ITEMS = iCountWords(OUTSTR)
        if(IEXP.ne.ITEMS)then
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(3A,I3,3A)')' In ',currentfile(1:LN),
     &        ' expecting',IEXP,' items (',MSG(1:LNM),') in...'
            CALL LUSRMSG(loutstr,truncstr,'F')
          else
            WRITE(loutstr,'(A,I3,3A)')' Expecting',IEXP,' items (',
     &         MSG(1:LNM),') in...'
            CALL LUSRMSG(loutstr,truncstr,'F')
          endif
        endif
      endif

    4 RETURN

  101 IER=1
      IF(IR.EQ.1)THEN
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5A)')' In ',currentfile(1:LN),
     &      ' error reading (',MSG(1:LNM),') in...'
        else
          WRITE(loutstr,'(A,A)',IOSTAT=IOS,ERR=1)' Error reading: ',
     &      MSG(1:LNM),') in...'
        endif
        lastcc = MIN0(LNBLNK(STRING),247)
        write(truncstr,'(a)') STRING(1:lastcc)
        CALL LUSRMSG(loutstr,truncstr,'W')
      ENDIF
      goto 4

  102 IER=2
      IF(IR.EQ.1)THEN
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5A)')' In ',currentfile(1:LN),
     &      ' error reading (',MSG(1:LNM),') EOF sensed.'
        else
          WRITE(loutstr,'(3A)',IOSTAT=IOS,ERR=1)' Error reading ',
     &      MSG(1:LNM),' EOF sensed.'
        endif
        CALL LUSRMSG(' ',loutstr,'-')
      ENDIF
      goto 4

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(outs,*) 'LSTRIPC4: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,outstr(1:72))
      else
        write(outs,*) 'LSTRIPC4: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,outstr(1:72))
      endif

      return
      END

C ******************** STRIPC2500 ********************
C Strips comments from a ASCII file (2500 char long) string and returns the
C data. It assumes that if a string begins with a '#' then the whole line is
C a comment an the next line is read.  If a ' #' is discovered within
C a line the rest of the line is removed.
C IER=0 if ok. MSG is a text string used in error messages. If
C IR=0 then acts silently, otherwise notes when EOF found.
C IEXP is the number of expected items in the line:
C   IEXP = 0 means don't care or already know no. items - don't check
C   IEXP >0  means a specific number of items expected (error if not)
C   IEXP = 99 check number of items and return in ITEMS

      SUBROUTINE STRIPC2500(INPCH,OUTSTR,IEXP,ITEMS,IR,MSG,IER)
#include "espriou.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      CHARACTER*2500 OUTSTR,STRING,TMP
      CHARACTER loutstr*248,outs*124
      character truncstr*248   ! truncate outsr for reporting.
      CHARACTER*(*) MSG
      logical unixok
      integer iCountWords

C Read a line of the file, strip off any trailing blanks, if the first
C character is a # then read the next line from the file.
      IER=0
    8 READ(INPCH,10,IOSTAT=ISTAT,ERR=101,END=102)STRING
   10 FORMAT(A)
      tmp=STRING(1:LNBLNK(STRING))

C Take the string and check for a #, discarding any text which follows.
      iloc = INDEX(tmp,'#')
      if(iloc.eq.1)then
        goto 8
      elseif(iloc.eq.0)then
        OUTSTR=tmp
      elseif(iloc.gt.1)then
        OUTSTR=tmp(1:ILOC-1)
      endif

C For messages we need to remember the initial part of the long string.
      lastcc = MIN0(LNBLNK(OUTSTR),247)
      write(truncstr,'(a)') OUTSTR(1:lastcc)

C Find out the number of separate words/string groupings.
      if(IEXP.eq.99)then
        ITEMS = iCountWords(OUTSTR)
      elseif(IEXP.eq.0)then
        ITEMS=0
      elseif(IEXP.gt.0)then
        ITEMS = iCountWords(OUTSTR)
        if(IEXP.ne.ITEMS)then
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(3A,I3,3A)')' In ',currentfile(1:LN),
     &        ' expecting',IEXP,' items (',MSG(1:LNM),') in...'
            CALL LUSRMSG(loutstr,truncstr,'F')
          else
            WRITE(loutstr,'(A,I3,3A)')' Expecting',IEXP,' items (',
     &         MSG(1:LNM),') in...'
            CALL LUSRMSG(loutstr,truncstr,'F')
          endif
        endif
      endif

    4 RETURN

  101 IER=1
      IF(IR.EQ.1)THEN
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5A)')' In ',currentfile(1:LN),
     &      ' error reading (',MSG(1:LNM),') in...'
        else
          WRITE(loutstr,'(A,A)',IOSTAT=IOS,ERR=1)' Error reading: ',
     &      MSG(1:LNM),') in...'
        endif
        lastcc = MIN0(LNBLNK(STRING),247)
        write(truncstr,'(a)') STRING(1:lastcc)
        CALL LUSRMSG(loutstr,truncstr,'W')
      ENDIF
      goto 4

  102 IER=2
      IF(IR.EQ.1)THEN
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5A)')' In ',currentfile(1:LN),
     &      ' error reading (',MSG(1:LNM),') EOF sensed.'
        else
          WRITE(loutstr,'(3A)',IOSTAT=IOS,ERR=1)' Error reading ',
     &      MSG(1:LNM),' EOF sensed.'
        endif
        CALL LUSRMSG(' ',loutstr,'-')
      ENDIF
      goto 4

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(outs,*) 
     &  'STRIPC2500: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,outstr(1:72))
      else
        write(outs,*) 'STRIPC2500: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,outstr(1:72))
      endif

      return
      END


C ******************** CHARCH ********************
C Older routine to check a string for a number of data items.
C This is an older version of CHITMS, however it modifies the string A.

      SUBROUTINE CHARCH(A,ND,IERR)

      CHARACTER*72 A

C Number of commas required.
      NC=ND

C Assign field (comma) counter, character counter
C and field start indicator.
      IC=0
      ICC=0
      ISC=0

      DO 10 I=1,72

C Jump if required number of commas found.
      IF(IC.EQ.NC)goto 10

      ICC=I
      IF(A(I:I).EQ.',')goto 1
      IF(A(I:I).EQ.' ')goto 2
      goto 3

    2 IF(ISC.EQ.0)goto 10
      A(I:I)=','
      goto 1

    3 IF(ISC.EQ.0)ISC=I
      goto 10

    1 IC=IC+1
      ISC=0

   10 CONTINUE

      IERR=-ICC
      IF(IC.NE.NC)IERR=1

      RETURN
      END

C ******************** NOYES ********************
C An INTEGER FUNCTION used to read the answer to a
C question. A line is read from the user and the value
C of the function is as follows:-
C   1 The answer was '1', ' Y', or '  YES'
C   0 The answer was '0', ' N', or '  NO'
C  -1 otherwise.

      FUNCTION NOYES(J)

      PARAMETER ( IUIN=5 )
      CHARACTER IA*3

C Attributes of session keystroke file.
      logical ikopened     ! Has keystoke file been started/requested.
      integer ikcount      ! Number of entries.
      integer ikout        ! File unit for keystrokes.
      character ikfile*72  ! The name of the keystroke file.
      common/logk/ikopened,ikcount,ikout,ikfile

      READ(IUIN,100)IA
      if(ikopened)then    ! Capture keystroke.
        ikcount=ikcount+1
        write(ikout,'(a)') ia(1:lnblnk(ia))
      endif
 100  FORMAT(A3)
      IF(IA.EQ.'1'.OR.IA.EQ.'Y'.OR.IA.EQ.'y'
     &            .OR.IA.EQ.'yes'.OR.IA.EQ.'YES')THEN
        NOYES=1
        RETURN
      ELSEIF(IA.EQ.'0'.OR.IA.EQ.'N'.OR.IA.EQ.'n'
     &                .OR.IA.EQ.'no'.OR.IA.EQ.'NO')THEN
        NOYES=0
        RETURN
      ELSE
        NOYES=-1
        RETURN
      ENDIF
      END

c ******************** IFIRST ********************
C Returns the ASCII value for the first character in
C a string ISTR.

      FUNCTION IFIRST(ISTR)

      CHARACTER*1 JSTR
      CHARACTER*(*) ISTR
      JSTR=ISTR(1:1)
      IFIRST=ICHAR(JSTR)

      RETURN
      END


C ******************** EASKPER ********************
C Provides legacy interface to the more general eAskPerGen below.

      SUBROUTINE eAskPer(PROMP1,IBDOY,IEDOY,IFDAY,IER)

      CHARACTER*(*) PROMP1
      integer ibdoy,iedoy,ifday,iBYr,iEYr,ier
      logical bMY_enabled
      parameter ( bMY_enabled = .false. )

C Note: initialize iBYr and iEYr even though the are not 
C returned. They're not referenced when multiyear simulations are disabled
      iBYr=2000
      iEYr=2000
      call eAskPerGen(PROMP1,IBDOY,IEDOY,
     &                 iBYr,iEYr,IFDAY,bMY_enabled,IER)

      return
      end


C ******************** EAskPerYear ********************
C Provides multi-year capable interface to the more general
C eAskPerGen below.

      SUBROUTINE eAskPerYear(PROMP1,ibdoy,iedoy,iBYr,iEYr,ifday,ier)

      CHARACTER*(*) PROMP1
      integer ibdoy,iedoy,iBYr,iEYr,ifday,ier

      logical bMY_enabled
      parameter ( bMY_enabled = .true. )

C initial values of iBYr iEYr are not need for single-year simulation
      call eAskPerGen(PROMP1,ibdoy,iedoy,iBYr,iEYr,ifday,bMY_enabled,
     &                IER)

      return
      end


C ******************** eAskPerGen ********************
C Provides interface to specification of a multiyear period of days. It
C returns IBDOY and IEDOY based on the current settings of IFDAY.
C iBYr,iEYr are the begining year and ending year of the period.
C Promp1 gives the context of the request for a period.

      SUBROUTINE eAskPerGen(PROMP1,ibdoy,iedoy,iBYr,iEYr,ifday,
     &                      bMY_enabled,IER)
#include "help.h"

      integer ibdoy,iedoy,iBYr,iEYr,ifday,ier
      CHARACTER*(*) PROMP1
      character HOLD*16

C Error flag.
      logical bDateOK

C Flag indicating if multi-year simulations are in use.
      logical bMY_enabled

      IER=0

      bDateOk = .false.

      do while ( .not. bDateOK )

C Initialise success flag and character buffer index.
        bDateOK = .true.
        k=0

C IFDAY describes how the date will be formatted.
C   IFDAY = 0 -> DDD (1->365)
C   IFDAY = 1 -> DD MM (1->31,1->12)

        IF(ifday.EQ.0)THEN
C Day of year to be provided

          if ( .not. bMY_enabled ) then
C Single year version
            H(1)='Period  must be within a calendar year.'

C Get start & end year-day.
            write(HOLD,'(I4,I4)')ibdoy,iedoy
            CALL EASKS(HOLD,PROMP1,'Start and end year-day?',16,
     &      ' 1  1 ','start & end day of year',IER,1)

            CALL EGETWI(HOLD,K,ibdoy,1,365,'F','start doy',IER)
            if ( ier .ne. 0 ) bDateOK = .false.

            CALL EGETWI(HOLD,K,iedoy,1,365,'F','end doy',IER)
            if ( ier .ne. 0 ) bDateOK = .false.

          else

C Multi-year version.
            H(1)='Provide day-of-year and year for start'
            H(2)='and end dates.'

C Get start year-day and year.
            write(HOLD,'(I4,I5)') ibdoy,iBYr
            CALL EASKS(HOLD,PROMP1,'Start year-day and year?',16,
     &      ' 1  2000 ','start day and year',IER,2)

            CALL EGETWI(HOLD,K,ibdoy,1,365,'F','start doy',IER)
            if  ( ier .ne. 0 ) BDateok = .false.

            CALL EGETWI(HOLD,K,iBYr,1900,2100,'F','start year',IER)
            if  ( ier .ne. 0 ) BDateok = .false.

C Reset string index K.
            K=0

C Get end year-day and year.
            write(HOLD,'(I4,I5)') iedoy,iEYr
            CALL EASKS(HOLD,PROMP1,'End year-day and year?',16,
     &      ' 365 2007 ','end day and year',IER,2)

            CALL EGETWI(HOLD,K,iedoy,1,365,'F','end doy',IER)
            if  ( ier .ne. 0 ) BDateok = .false.

            CALL EGETWI(HOLD,K,iEYr,1900,2100,'F','end year',IER)
            if  ( ier .ne. 0 ) BDateok = .false.

          endif !<- matches if ( .not. bMY_enabled )...

        ELSEIF ( ifday .eq. 1 .or. ifday .eq. 2 ) THEN

C Day & month or day, month & year to be provided
C Start date - convert to dd mm format.
          CALL EDAYR(ibdoy,IBD,IBM)

          if ( .not. bMY_enabled ) then

C Get day & month.
            write(HOLD,'(I2,A,I2)') IBD,' ',IBM
            H(1)='Period must be within a calendar year.'
            H(2)='( `1 1` to `31 12` )'
            CALL EASKS(HOLD,PROMP1,'Start day & month?',
     &        16,' 1  1 ','start day and month',ier,2)

          else

C Get day, month & year. Append year to string.
            write (HOLD, '(I2,A,I2,A,I4)') IBD,' ',IBM,' ',iBYr

            H(1)='Specify simulation start day, month and year.'
            H(2)='Leap years are not supported.'
            CALL EASKS(HOLD,PROMP1,'Start day, month & year?',
     &        16,' 1 1 2007','start day, month and year',IER,2)
          endif

C Error handling.
          if ( ier .ne. 0 ) bDateOK = .false.


C Convert stings to integers and flag errors..
          K=0
          CALL EGETWI(HOLD,K,IBD,1,31,'F','start dom',ier)
          if ( ier .ne. 0 ) bDateOK = .false.

          CALL EGETWI(HOLD,K,IBM,1,12,'F','start month',ier)
          if ( ier .ne. 0 ) bDateOK = .false.

C Convert year string, if necessary.
          if ( bMY_enabled ) then

            CALL EGETWI ( HOLD, K, iBYr, 1900, 2100, 'F',
     &        'start year', ier)

            if ( ier .ne. 0 ) bDateOK = .false.

          endif

C Check range, then convert to ibdoy.
          if ( bDateOK ) CALL EDAYCH(IBD,IBM,ier)

          if ( ier .eq. 1 ) bDateOK = .false.

          if ( bDateOK ) CALL EDAY(IBD,IBM,ibdoy)

C Only ask for end date if start-date successfully defined
          if ( bDateOK ) then

C End date, set at least equal to the beginning day.
            if ( .not. bMY_enabled .and. iedoy.lt.ibdoy ) then
              iedoy = ibdoy
            elseif ( bMY_enabled ) then

C Convert end-date into EDAYR friendly format and
C check that end date is not before start-date.
              iedoy = iedoy - ( iEYr - iBYr ) * 365
              if ( ( iEYr .eq. iBYr .and. iedoy .lt. ibdoy ) .or.
     &             ( iEYr .lt. iBYr ) ) then
                iedoy=ibdoy
                iEyr=iBYr
                if (iedoy .gt. 365 ) then
                  iedoy = 1
                  iEYr = iEyr + 1
                endif
              endif
            endif ! <- matches ' elseif (bMY_enabled)...

C Convert end-day to day-month format
            CALL EDAYR(iedoy,IED,IEM)

            if ( .not. bMY_enabled ) then
              write(HOLD,'(I2,A,I2)') IED,' ',IEM
              H(1)='Period  must be within a calendar year.'
              H(2)='( `1 1` to `31 12` )'
              CALL EASKS(HOLD,PROMP1,'End day & month?',
     &          16,' 1  1 ','start day and month',ier,2)

            else

C Get day, month & year  and append year to string.
              write (HOLD, '(I2,A,I2,A,I4)') IED,' ',IEM,' ',iEYr
              H(1)='Specify simulation start day, month and year'
              H(2)='in `DD MM YYYY` format. Note: Leap years are'
              H(3)='not presently supported.'

              CALL EASKS(HOLD,PROMP1,
     &          ' End day, month & year?',
     &          16,' 31 12 2007','end day, month and year',IER,3)
            endif

            if ( ier .ne. 0 ) bDateOK = .false.

C Convert strings to integers and errortrap
            K=0
            CALL EGETWI(HOLD,K,IED,1,31,'F','end dom',IER)
            if ( ier .ne. 0 ) bDateOK = .false.
            CALL EGETWI(HOLD,K,IEM,1,12,'F','end month',IER)
            if ( ier .ne. 0 ) bDateOK = .false.

C Convert year string, if necessary
            if ( bMY_enabled ) then

              CALL EGETWI ( HOLD, K, iEYr, 1900, 2100, 'F',
     &          'end year', ier)
              if ( ier .ne. 0 ) bDateOK = .false.
            endif

C Check range, then convert to iedoy.
            if ( bDateOK )  CALL EDAYCH(IED,IEM, ier )
            if ( ier .ne. 0 ) bDateOK = .false.
            if ( bDateOK ) CALL EDAY(IED,IEM,iedoy)

C For multi-year simulations, convert iedoy:
            iedoy = iedoy + ( iEYr - iBYr ) * 365

          endif
        endif
      enddo

      RETURN
      END

c ******************** EDAY ********************
C Returns year day number IYDN when passed the day of the month
C IDAYN and the month number IMTHN. 1st Jan= 1, 31st Dec=365, leap
C years NOT considered.

      SUBROUTINE EDAY(IDAYN,IMTHN,IYDN)
      DIMENSION MONTH(12)
      DATA MONTH/31,28,31,30,31,30,31,31,30,31,30,31/
      IYDN=IDAYN
      IF(IMTHN.EQ.1)RETURN
      DO 10 I=2,IMTHN
        IYDN=IYDN+MONTH(I-1)
   10 CONTINUE
      RETURN
      END

C ******************** EDAYR ********************
C Returns the day and month numbers from the day-of-year where:
C day-of-year 1 = 1st January and day-of-year 365 = 31st December.
C Leap years are NOT considered.

      SUBROUTINE EDAYR(IYDN,IDAYN,IMTHN)
      DIMENSION MONTH(12),IACTOT(11)
      DATA MONTH/31,28,31,30,31,30,31,31,30,31,30,31/
      DATA IACTOT/31,59,90,120,151,181,212,243,273,304,334/
      IMTHN=1

C Determine the month number.
      DO 10 I=1,11
        IF(IYDN.GT.IACTOT(I))IMTHN=IMTHN+1
   10 CONTINUE

C Determine the day of the month.
      IF(IMTHN.EQ.1)THEN
        IDAYN=IYDN
        RETURN
      ELSE
        IDAYN=0
        J=IMTHN-1
        DO 20 I=1,J
          IDAYN=IDAYN+MONTH(I)
   20   CONTINUE
        IDAYN=IYDN-IDAYN
        RETURN
      ENDIF
      END

C ******************** EWEEKD ********************
C Returns the day of the week (IDAY) given the day of month, month
C and year (ND,NM,NY) as integers.
C MON=1, TUE=2, WED=3, THU=4, FRI=5, SAT=6 AND SUN=7.

      SUBROUTINE EWEEKD(ND,NM,NY,IDAY)
      common/OUTIN/IUOUT,IUIN,IEOUT
      DIMENSION MMTH(12),MYR(5)
      character outs*124
      DATA MMTH/1,4,4,0,2,5,0,3,6,1,4,6/
      DATA MYR/2,4,2,0,6/

      IF(ND.EQ.0.OR.NM.EQ.0.OR.NY.EQ.0)goto 50
      IL1=MOD(NY,100)
      IL2=MOD(IL1,4)
      IL3=MOD(NY,1000)
      LEAP=0
      IF(IL2.NE.0)goto 11
      IF(IL3.EQ.0.OR.IL1.NE.0)LEAP=1
   11 IF(LEAP.EQ.0.AND.NM.EQ.2.AND.ND.EQ.29)goto 50
      II=IL1/4
      IDAY=IL1+II+ND+MMTH(NM)
      IF(LEAP.EQ.1.AND.(NM.EQ.1.OR.NM.EQ.2))IDAY=IDAY-1
      IL2=INT(.01*(NY-IL1+1))
      IL2=IL2-15
      IF(IL2.LE.0.OR.IL2.GT.5)goto 50
      IL3=MYR(IL2)
      IF(IL2.NE.2.OR.NM.GT.9)goto 12
      IF(NM.EQ.9.AND.ND.GE.14)goto 12
      IF(NM.EQ.9.AND.ND.GT.2)goto 50
      IL3=1
   12 IDAY=IDAY+IL3+5
      IDAY=MOD(IDAY,7)+1
      goto  10
   50 WRITE(outs,'(a,3I6,a)')' The date ',ND,NM,NY,' is illegal.'
      call edisp(iuout,outs)
   10 CONTINUE
      RETURN
      END

C ******************** EDAYCH ********************
C Checks for errors in the users specification of the day and
C month under consideration.
C IERR set to 1 if ID or IM are outwith the allowable range.

      SUBROUTINE EDAYCH(ID,IM,IERR)
      common/OUTIN/IUOUT,IUIN,IEOUT
      DIMENSION MNTH(12)
      DATA MNTH/31,28,31,30,31,30,31,31,30,31,30,31/
      IERR=0
      IF(IM.LT.1.OR.IM.GT.12)THEN
        call edisp(iuout,' Month value outwith allowable range. ')
        IERR=1
      ENDIF
      IF(ID.LT.1.OR.ID.GT.MNTH(IM))THEN
        call edisp(iuout,' Day value outwith allowable monthly range.')
        IERR=1
      ENDIF
      RETURN
      END

C ******************** DATTIM ********************
C Returns UNIX time via a string in the form : 16 Sep 73 14:23.

      SUBROUTINE DATTIM(DT)
      CHARACTER*(*) DT
      CHARACTER*24 TUNIX

      CALL FDATE(TUNIX)
      DT(1:6) = TUNIX(5:10)
      DT(7:7) = ' '
      DT(10:15) = TUNIX(11:16)
      RETURN
      END

C ******************** STDATE ********************
C Takes the day of year and returns two descriptive strings.
C DESCR takes the form '12-Jan' and DESCR1 takes the form 'Fri-12-Jan',
C DESCR2 takes the form '12/01/97'.

      SUBROUTINE STDATE(IYEAR,IDOY,DESCR,DESCR1,DESCR2)

      dimension MTHNAM(12),DAYNAM(7)
      character MTHNAM*3,DAYNAM*3,DESCR*7,DESCR1*10,DESCR2*8
      character YRNAME*4
      DATA MTHNAM/'Jan','Feb','Mar','Apr','May','Jun','Jul','Aug',
     &            'Sep','Oct','Nov','Dec'/
      DATA DAYNAM/'Mon','Tue','Wed','Thu','Fri','Sat','Sun'/

C Determine whether IDAY is a weekday, saturday or sunday.
C Determine the month and day of month via EDAYR.
      CALL EDAYR(IDOY,IDAYN,IMTHN)
      CALL EWEEKD(IDAYN,IMTHN,IYEAR,IDWK)
      write(YRNAME,'(i4)') IYEAR

C Build up the descriptive text strings if a known month. Trap out
C of range idwk and imthn to prevent string array crashes.
      if(imthn.gt.0.and.imthn.le.12)then
        WRITE(DESCR,'(i2.2,2a)',ERR=1)IDAYN,'-',MTHNAM(IMTHN)
        if(idwk.gt.0.and.idwk.le.7)then
          WRITE(DESCR1,'(2a,i2.2,2a)',ERR=1)DAYNAM(IDWK),'-',IDAYN,'-',
     &      MTHNAM(IMTHN)
          WRITE(DESCR2,'(i2.2,a,i2.2,2a)',ERR=1)IDAYN,'/',IMTHN,'/',
     &      YRNAME(3:4)
        else
          WRITE(DESCR1,'(a,i2.2,2a)',ERR=1) 'XXX-',IDAYN,'-',
     &      MTHNAM(IMTHN)
          WRITE(DESCR2,'(i2.2,a,i2.2,2a)',ERR=1)IDAYN,'/',IMTHN,'/',
     &      YRNAME(3:4)
        endif
      else
        WRITE(DESCR,'(i2.2,a)',ERR=1)IDAYN,'-XXX'
        WRITE(DESCR2,'(i2.2,a,i2.2,2a)',ERR=1)IDAYN,'/',IMTHN,'/',
     &    YRNAME(3:4)
        if(idwk.gt.0.and.idwk.le.7)then
          WRITE(DESCR1,'(2a,i2.2,a)',ERR=1)DAYNAM(IDWK),'-',IDAYN,'-XXX'
        else
          WRITE(DESCR1,'(a,i2.2,a)',ERR=1) 'XXX-',IDAYN,'-XXX'
        endif
      endif

      RETURN
 1    call usrmsg('STDATE: problem writing date strings.',' ','W')
      END

C ******************** ESTIME ********************
C Takes an integer timestep and returns two string descriptions:
C DESCRH in the form '12h28' and DESCRD which takes the form of 12.46,
C DESCRJ in the form '12:28' as well as the time as a fraction
C of a day TIMER.
C STIME takes timestep averaging IDAVER into account ie. 0=averaging,
C 1=no averaging. NTS is the number of timesteps per hour.

      SUBROUTINE ESTIME(NTS,IDAVER,ITIME,DESCRH,DESCRD,DESCRJ,TIMER)
      CHARACTER*5 DESCRH, DESCRD, DESCRJ

      ADJUST=0.
      IF(IDAVER.EQ.0)ADJUST=1.0/(FLOAT(NTS)*2.0)
      TIMER=FLOAT(ITIME)/FLOAT(NTS)
      TIMER=TIMER-ADJUST
      MIN=INT((TIMER-AINT(TIMER))*60.)
      WRITE(DESCRH,'(i2.2,a,i2.2)',ERR=1)INT(TIMER),'h',MIN
      WRITE(DESCRJ,'(i2.2,a,i2.2)',ERR=1)INT(TIMER),':',MIN
      WRITE(DESCRD,'(f5.2)',IOSTAT=IOS,ERR=1)TIMER
      TIMER=TIMER/24.0
      RETURN

 1    if(IOS.eq.2)then
        call usrmsg('ESTIME: permission writing time strings.',' ','W')
      else
        call usrmsg('ESTIME: problem writing time strings.',' ','W')
      endif
      return

      END

C ******************** EDTIME ********************
C Takes an real time and returns two string descriptions:
C DESCRH in the form '12h28' and DESCRD which takes the form of 12.46,
C DESCRJ in the form '12:28' as well as the time as a fraction
C of a day TIMER.

      SUBROUTINE EDTIME(TIME,DESCRH,DESCRD,DESCRJ,TIMER)
      CHARACTER*5 DESCRH, DESCRD, DESCRJ

      MIN=INT((TIME-AINT(TIME))*60.)
      WRITE(DESCRH,'(i2.2,a,i2.2)',ERR=1)INT(TIME),'h',MIN
      WRITE(DESCRJ,'(i2.2,a,i2.2)',ERR=1)INT(TIME),':',MIN
      WRITE(DESCRD,'(f5.2)',IOSTAT=IOS,ERR=1)TIME
      TIMER=TIME/24.0
      RETURN

 1    if(IOS.eq.2)then
        call usrmsg('EDTIME: permission writing time strings.',' ','W')
      else
        call usrmsg('EDTIME: problem writing time strings.',' ','W')
      endif
      return

      END

C ******************** EPERSTR ********************
C Creates three strings representing the start and stop
C time of a diary period based on the preferred time & date
C display format.
C IFDAY  0 gives 'DOY 10', 1 gives '10 Jan', 2 gives 'Fri 10 Jan'
C IFDAY  3 gives '10/01/21'
C IFTIME 0 gives '10h30', 1 gives '10.50', 2 gives '0.4375'
C IFTIME 3 gives '10:30'
C PERST1 (14 char) is:' 10h00  15h30 ',' 10.00  15.50 ',' 0.4375 0.6458'
C PERST3 (44 char):
C if IFDAY=0 then it is: 'period: DOY 100 to DOY 112, 1990'
C if IFDAY=1 then it is: 'period: 10 Jan to 31 Jan, 1990'
C if IFDAY=2 then it is: 'period: Mon 10 Jan to Mon 17 Jan, 1990'
C if IFDAY=4 then it is: 'period: 10/01/90 to 17/01/90'
C PERST2 (44 char) includes the time of day but not the year.
C IER=0 OK, IER=1 problem.
C BTIM and PETIM are in terms of decimal fractions of a day.

      SUBROUTINE EPERSTR(IYEAR,IBDOY,IBTIM,IEDOY,IETIM,NTSPH,
     &           IFDAY,IFTIME,PERST1,PERST2,PERST3,IER)
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      character T1H*5, T1D*5, T2H*5, T2D*5, T1J*5, T2J*5
      character PERST1*14,PERST2*44,PERST3*44,DS*7,DS1*10,DE*7,DE1*10
      character DS2*8,DE2*8
      character outs*124

      IER=0

C Generate view period string based on IBDOY,IEDOY,BTIM,ETIM
      CALL STDATE(IYEAR,IBDOY,DS,DS1,DS2)
      CALL STDATE(IYEAR,IEDOY,DE,DE1,DE2)
      CALL ESTIME(NTSPH,1,IBTIM,T1H,T1D,T1J,TIMER1)
      CALL ESTIME(NTSPH,1,IETIM,T2H,T2D,T2J,TIMER2)
      FD1=FLOAT(IBDOY)+TIMER1
      FD2=FLOAT(IEDOY)+TIMER2
      IF(IFTIME.EQ.0)THEN
        WRITE(PERST1,1,IOSTAT=IOS,ERR=10)T1H,T2H
    1   FORMAT(' ',A5,'  ',A5,' ')
      ELSEIF(IFTIME.EQ.1)THEN
        WRITE(PERST1,1,IOSTAT=IOS,ERR=10)T1D,T2D
      ELSEIF(IFTIME.EQ.2)THEN
        WRITE(PERST1,2,IOSTAT=IOS,ERR=10)TIMER1,TIMER2
    2   FORMAT(2F7.4)
      ELSEIF(IFTIME.EQ.3)THEN
        WRITE(PERST1,1,IOSTAT=IOS,ERR=10)T1J,T2J
      ENDIF

      IF(IFDAY.EQ.0)THEN
        WRITE(PERST3,3,IOSTAT=IOS,ERR=10)IBDOY,IEDOY,IYEAR
    3   FORMAT('period: DOY ',I3,' to DOY ',I3,', ',I4)
        IF(IFTIME.EQ.0)THEN
          WRITE(PERST2,4,IOSTAT=IOS,ERR=10)IBDOY,T1H,IEDOY,T2H
    4     FORMAT('period: DOY ',I3,' @ ',A5,' to DOY ',I3,' @ ',A5)
        ELSEIF(IFTIME.EQ.1)THEN
          WRITE(PERST2,4,IOSTAT=IOS,ERR=10)IBDOY,T1D,IEDOY,T2D
        ELSEIF(IFTIME.EQ.2)THEN
          WRITE(PERST2,5,IOSTAT=IOS,ERR=10)FD1,FD2
    5     FORMAT('period: DOY ',F10.6,' to DOY ',F10.6)
        ELSEIF(IFTIME.EQ.3)THEN
          WRITE(PERST2,42,IOSTAT=IOS,ERR=10)IBDOY,T1J,IEDOY,T2J
   42     FORMAT('period: DOY ',I3,' @ ',A5,' to DOY ',I3,' @ ',A5)
        ENDIF
      ELSEIF(IFDAY.EQ.1)THEN
        WRITE(PERST3,6,IOSTAT=IOS,ERR=10)DS,DE,IYEAR
    6   FORMAT('period: ',A7,' to ',A7,', ',I4)
        IF(IFTIME.EQ.0)THEN
          WRITE(PERST2,7,IOSTAT=IOS,ERR=10)DS,T1H,DE,T2H
    7     FORMAT('period: ',A7,' @ ',A5,' to ',A7,' @ ',A5)
        ELSEIF(IFTIME.EQ.1.OR.IFTIME.EQ.2)THEN
          WRITE(PERST2,7,IOSTAT=IOS,ERR=10)DS,T1D,DE,T2D
        ELSEIF(IFTIME.EQ.3)THEN
          WRITE(PERST2,47,IOSTAT=IOS,ERR=10)DS,T1J,DE,T2J
   47     FORMAT('period: ',A7,' @ ',A5,' to ',A7,' @ ',A5)
        ENDIF
      ELSEIF(IFDAY.EQ.2)THEN
        WRITE(PERST3,8,IOSTAT=IOS,ERR=10)DS1,DE1,IYEAR
    8   FORMAT('period: ',A10,' to ',A10,', ',I4)
        IF(IFTIME.EQ.0)THEN
          WRITE(PERST2,9,IOSTAT=IOS,ERR=10)DS1,T1H,DE1,T2H
    9     FORMAT('period: ',A10,'@',A5,' - ',A10,'@',A5)
        ELSEIF(IFTIME.EQ.1.OR.IFTIME.EQ.2)THEN
          WRITE(PERST2,9,IOSTAT=IOS,ERR=10)DS1,T1D,DE1,T2D
        ELSEIF(IFTIME.EQ.3)THEN
          WRITE(PERST2,49,IOSTAT=IOS,ERR=10)DS1,T1J,DE1,T2J
   49     FORMAT('period: ',A10,'@',A5,' - ',A10,'@',A5)
        ENDIF
      ELSEIF(IFDAY.EQ.3)THEN
        WRITE(PERST3,48,IOSTAT=IOS,ERR=10)DS1,DE1,IYEAR
   48   FORMAT('period: ',A10,' to ',A10,', ',I4)
        WRITE(PERST2,50,IOSTAT=IOS,ERR=10)T1J,DS2,T2J,DE2
   50   FORMAT('period: ',A5,' on ',A10,' to ',A5,' on ',A10)
      ENDIF
      RETURN

  10  if(IOS.eq.2)then
        write(outs,*) 'EPERSTR: permission error writing strings.'
        call edisp(iuout,outs)
      else
        write(outs,*) 'EPERSTR: error writing warning or strings.'
        call edisp(iuout,outs)
      endif
      return

      END

C ******************** EGETW ********************
C Gets first WORD after position K from the STRING of
C characters. Words are separated by blanks, commas, |, or tab: WORD,WORD,WORD
C or WORD WORD WORD or WORD, WORD, WORD  are all valid.  Provides a warning
C message if ACT='W', a failure message if ACT='F' and does
C no message if ACT='-'.  Modified after:
C G.N. Walton, US Nat. Institute of Standards and Technology
C     LS     - maximum length of STRING
C     L      - current position in WORD
C     LW     - maximum length of WORD

      SUBROUTINE EGETW(STRING,K,WORD,ACT,MSG,IER)
#include "espriou.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER*(*) WORD, STRING, MSG
      CHARACTER ACT*1,A*1,loutstr*248,outs*124
      logical unixok

      WORD=' '
      LS=LEN(STRING)
      LW=LEN(WORD)
      L=0

C Start by skipping blanks and tabs before the word.
   10 K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(A.EQ.' '.OR.A.EQ.','.OR.(ICHAR(A).EQ.9).OR.
     &   (ICHAR(A).eq.124)) GOTO 10

C Copy WORD from STRING, character by character until separator found.
   20 L=L+1
      IF(L.GT.LW) GOTO 100
      WORD(L:L)=A
      K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(A.EQ.' '.OR.A.EQ.','.OR.(ICHAR(A).EQ.9).OR.
     &   (ICHAR(A).eq.124)) GO TO 100
      GOTO 20

  100 RETURN

  999 ier=1
      if(ACT.EQ.'-')then
        RETURN
      elseif(ACT.EQ.'W')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: Past end of line for ',
     &      MSG(1:LNM),' (word) in...'
        endif
      elseif(ACT.EQ.'F')then
        ier=1
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: Past end of line for ',
     &      MSG(1:LNM),' (word) in...'
        endif
      endif
      CALL EDISP248(iuout,loutstr,100)
      if(lnblnk(STRING).gt.123)then
        CALL EDISP248(iuout,STRING,100)
      else
        CALL EDISP(iuout,STRING)
      endif
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      LNM=max(1,lnblnk(STRING))
      if(IOS.eq.2)then
        write(outs,*) 'EGETW: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      else
        write(outs,*) 'EGETW: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      endif
      return

      END

C ******************** EGETP ********************
C Gets first PHRASE after position K from the STRING of
C characters. Phrases are separated by tabs or commas. Provides a warning
C message if ACT='W', a failure message if ACT='F' and does
C no message if ACT='-'.

      SUBROUTINE EGETP(STRING,K,PHRASE,ACT,MSG,ier)
#include "espriou.h"
      common/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER*(*) PHRASE, STRING, MSG
      CHARACTER ACT*1,A*1,loutstr*248,outs*124
      logical unixok

C LS is th maximum length of STRING, L the current position,
C LW the maximum length of PHRASE.
      ier=0
      PHRASE=' '
      LS=LEN(STRING)
      LW=LEN(PHRASE)
      L=0

C Start by skipping blanks and tabs before the PHRASE.
   10 K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(A.EQ.' '.OR.A.EQ.','.OR.(ICHAR(A).EQ.9)) GOTO 10

C Copy PHRASE from STRING, character by character until tab or a
C comma is found.
   20 L=L+1
      IF(L.GT.LW) GOTO 100
      PHRASE(L:L)=A
      K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(ICHAR(A).EQ.9.or.A.eq.',') GO TO 100
      GOTO 20

  100 RETURN

  999 if(ACT.EQ.'-')then
        RETURN
      elseif(ACT.EQ.'W')then
        ier=1
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' (phrase) in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: Past end of line for ',
     &      MSG(1:LNM),' (phrase) in...'
        endif
      elseif(ACT.EQ.'F')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' (phrase) in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: Past end of line for ',
     &      MSG(1:LNM),' (phrase) in...'
        endif
      endif
      CALL EDISP248(iuout,loutstr,100)
      if(lnblnk(STRING).gt.123)then
        CALL EDISP248(iuout,STRING,100)
      else
        CALL EDISP(iuout,STRING)
      endif
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      LNM=max(1,lnblnk(STRING))
      if(IOS.eq.2)then
        write(outs,*) 'EGETP: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      else
        write(outs,*) 'EGETP: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      endif
      return

      END

C ******************** EGETDQ ********************
C Gets first quoted PHRASE after position K from the STRING of
C characters. Phrases can contain spaces and commas but not tabs.
C Provides a warning message if ACT='W', a failure message if ACT='F' and
C no message if ACT='-'.  The string returned has the quotes removed.
C Note it uses an internal string buffer which assumes that the
C phrase is less than 248 characters long.

      SUBROUTINE EGETDQ(STRING,K,PHRASE,ACT,MSG,ier)
#include "espriou.h"
      common/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER*(*) PHRASE, STRING, MSG
      CHARACTER ACT*1,A*1,B*1,loutstr*124,outs*124
      character dq*1,sqleft*1,sqright*1
      character buffer*248
      integer lnofstr
      logical unixok

      dq = char(34)  ! double quote
      sqleft = char(96)  ! single quote left
      sqright = char(39)  ! single quote right

C LS is th maximum length of STRING, L the current position,
C LW the maximum length of PHRASE.
      ier=0
      PHRASE=' '
      LS=LEN(STRING)
      LW=LEN(PHRASE)
      L=0

C Start by skipping blanks, commas and tabs before the PHRASE.
   10 K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(A.EQ.' '.OR.A.EQ.','.OR.(ICHAR(A).EQ.9)) GOTO 10

C Copy PHRASE from STRING, character by character until 
C a double quote or single quote is found (a quoted phrase 
C could include a comma, but not a tab)
   20 L=L+1
      IF(L.GE.LW) GOTO 100   ! at end of available phrase chars
      PHRASE(L:L)=A
      K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(ICHAR(A).EQ.9) GOTO 100
      if(A.eq.dq.or.A.eq.sqleft.or.A.eq.sqright)then

C Found a quote so increment L add quote it to the phrase and then process.
        L=L+1
        PHRASE(L:L)=A
        GOTO 100
      endif
      GOTO 20

  100 continue

C Test if the first and last characters are double or single quotes.
C If the last character is not a quote but the first is at least
C cut out the intial quote when writing phrase.
      A=PHRASE(1:1)
      B=PHRASE(L:L)
      if(A.eq.dq.or.A.eq.sqleft.or.A.eq.sqright)then
        if(B.eq.dq.or.B.eq.sqright.or.B.eq.sqleft)then
          lnofstr=lnblnk(PHRASE)-1
          write(buffer,'(a)') PHRASE(2:lnofstr)
          write(phrase,'(a)') buffer(1:lnblnk(buffer))
        else
          lnofstr=lnblnk(PHRASE)
          write(buffer,'(a)') PHRASE(2:lnofstr)
          write(phrase,'(a)') buffer(1:lnblnk(buffer))
        endif
      endif
      RETURN

  999 if(ACT.EQ.'-')then
        RETURN
      elseif(ACT.EQ.'W')then
        ier=1
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' (phrase) in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: Past end of line for ',
     &      MSG(1:LNM),' (phrase) in...'
        endif
      elseif(ACT.EQ.'F')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' (phrase) in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: Past end of line for ',
     &      MSG(1:LNM),' (phrase) in...'
        endif
      endif
      CALL EDISP248(iuout,loutstr,100)
      if(lnblnk(STRING).gt.123)then
        CALL EDISP248(iuout,STRING,100)
      else
        CALL EDISP(iuout,STRING)
      endif
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      LNM=max(1,lnblnk(STRING))
      if(IOS.eq.2)then
        write(outs,*) 'EGETDQ: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      else
        write(outs,*) 'EGETDQ: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      endif
      return

      END

C ******************** EGETWI ********************
C Gets first word after position K from the STRING of
C characters and converts it into an integer IV, tests it against
C the minimum MN and the maximum MX and provides a warning
C message if ACT='W', a failure message if ACT='F' and does
C no range checking if ACT='-'. Words may be separated by blanks,
C commas, or tab: WORD,WORD,WORD or WORD WORD WORD or WORD, WORD, WORD
C are all valid.

      SUBROUTINE EGETWI(STRING,K,IV,MN,MX,ACT,MSG,IER)
#include "espriou.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER*(*) STRING, MSG
      CHARACTER ACT*1,STR1*10,STR2*10,WORD*20,loutstr*248,outs*124
      logical unixok

C Pick up line and lenght for error messages.
      ils=max(1,lnblnk(STRING))
      if(ils.gt.105)ils=105

      IER=0
      WORD=' '
      CALL EGETW(STRING,K,WORD,'-','integer',IER)
      IF(IER.NE.0)RETURN
      read(WORD,*,IOSTAT=IOS,ERR=1002)iv

C If range checking disabled, jump out of routine.
      IF(ACT.EQ.'-')RETURN

C Make up reporting string.
      CALL INTSTR(IV,STR1,IW1,IER)

C Check IV against minimum and respond based on ACT.
      IF(IV.LT.MN)THEN
        CALL INTSTR(MN,STR2,IW2,IER)
        IF(ACT.EQ.'W')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(4a)',IOSTAT=IOS,ERR=1)
     &        'WARNING: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(loutstr,'(a,a)',IOSTAT=IOS,ERR=1)' WARNING in: ',
     &      STRING(1:ils)
          endif
          CALL EDISP248(iuout,loutstr,100)
          write(loutstr,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &      ' value (',STR1(1:IW1),') < normal minimum ',STR2(1:IW2),'!'
          CALL EDISP248(iuout,loutstr,100)
        ELSEIF(ACT.EQ.'F')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(4a)',IOSTAT=IOS,ERR=1)
     &        'FAILURE: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(loutstr,'(a,a)',IOSTAT=IOS,ERR=1)' FAILURE in: ',
     &        STRING(1:ils)
          endif
          CALL EDISP248(iuout,loutstr,100)
          write(loutstr,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &       ' value (',STR1(1:IW1),') < allowable minimum ',
     &       STR2(1:IW2),'!'
          CALL EDISP248(iuout,loutstr,100)
          IER=1
          RETURN
        ENDIF
      ELSEIF(IV.GT.MX)THEN
        CALL INTSTR(MX,STR2,IW2,IER)
        IF(ACT.EQ.'W')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(4a)',IOSTAT=IOS,ERR=1)
     &        'WARNING: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(loutstr,'(a,a)',IOSTAT=IOS,ERR=1)' WARNING in: ',
     &        STRING(1:ils)
          endif
          CALL EDISP248(iuout,loutstr,100)
          write(loutstr,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &      ' value (',STR1(1:IW1),') > normal maximum ',STR2(1:IW2),'!'
          CALL EDISP248(iuout,loutstr,100)
        ELSEIF(ACT.EQ.'F')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(4a)',IOSTAT=IOS,ERR=1)
     &        'FAILURE: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(loutstr,'(a,a)',IOSTAT=IOS,ERR=1)' FAILURE in: ',
     &        STRING(1:ils)
          endif
          CALL EDISP248(iuout,loutstr,100)
          write(loutstr,'(7a)',IOSTAT=IOS,ERR=1) ' the ',MSG(1:LNM),
     &       ' value (',STR1(1:IW1),') > allowable maximum ',
     &       STR2(1:IW2),'!'
          CALL EDISP248(iuout,loutstr,100)
          IER=1
          RETURN
        ENDIF
      ENDIF

  100 RETURN

 1002 if(currentfile(1:2).ne.'  ')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        WRITE(loutstr,'(6a)',IOSTAT=IOS,ERR=1) 'Failed in ',
     &    currentfile(1:LN),': conversion of ',
     &    MSG(1:LNM),' value in ',WORD
      else
        LNM=max(1,lnblnk(MSG))
        WRITE(loutstr,1003,IOSTAT=IOS,ERR=1)MSG(1:LNM),WORD
 1003   FORMAT(' Failed: conversion of ',A,' value in ',A)
      endif
      CALL EDISP248(iuout,loutstr,100)
      IER=2
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      LNM=max(1,lnblnk(STRING))
      if(IOS.eq.2)then
        write(outs,*) 'EGETWI: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      else
        write(outs,*) 'EGETWI: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      endif
      return

      END

C ******************** EGETWR ********************
C Gets first word after position K from the STRING of
C characters and converts it into a real number RV, tests it against
C the minimum RMN and the maximum RMX and provides a warning
C message if RACT='W', a failure message if RACT='F' and does
C no range checking if RACT='-'. Words may be separated by blanks,
C commas, or tab: WORD,WORD,WORD or WORD WORD WORD or WORD, WORD, WORD
C are all valid.

      SUBROUTINE EGETWR(STRING,K,RV,RMN,RMX,RACT,MSG,IER)
#include "espriou.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

C LOUTSTR is for messages to the user and should be long enough
c to prevent truncation of messge contents.
      CHARACTER*(*) STRING, MSG
      CHARACTER RACT*1,STR1*16,STR2*16,WORD*20,LOUTSTR*248,outs*124
      logical unixok

C Pick up line and length for error messages.
      ils=max(1,lnblnk(STRING))
      if(ils.gt.230)ils=230

      IER=0
      WORD=' '
      CALL EGETW(STRING,K,WORD,'-','real',IER)
      IF(IER.NE.0) goto 1002
      read(WORD,*,ERR=1002)rv

C If range checking disabled, jump out of routine.
      IF(RACT.EQ.'-')RETURN

C Make up reporting string.
      CALL REL16STR(RV,STR1,IW1,IER)

C Check RV against minimum and respond based on RACT.
      IF(RV.LT.RMN)THEN
        CALL REL16STR(RMN,STR2,IW2,IER)
        IF(RACT.EQ.'W')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(LOUTSTR,'(4a)',IOSTAT=IOS,ERR=1)
     &        'WARNING: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(LOUTSTR,'(a,a)',IOSTAT=IOS,ERR=1)' WARNING in: ',
     &        STRING(1:ils)
          endif
          call edisp248(iuout,LOUTSTR,100)
          write(LOUTSTR,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &      ' value (',STR1(1:IW1),') < normal minimum ',STR2(1:IW2),'!'
          call edisp248(iuout,LOUTSTR,100)
        ELSEIF(RACT.EQ.'F')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(LOUTSTR,'(4a)',IOSTAT=IOS,ERR=1)
     &        'FAILURE: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(LOUTSTR,'(a,a)',IOSTAT=IOS,ERR=1)' FAILURE in: ',
     &      STRING(1:ils)
          endif
          call edisp248(iuout,LOUTSTR,100)
          write(LOUTSTR,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &       ' value (',STR1(1:IW1),') < allowable minumum ',
     &       STR2(1:IW2),'!'
          call edisp248(iuout,loutstr,100)
          IER=1
          RETURN
        ENDIF
      ELSEIF(RV.GT.RMX)THEN
        CALL REL16STR(RMX,STR2,IW2,IER)
        IF(RACT.EQ.'W')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(LOUTSTR,'(4a)',IOSTAT=IOS,ERR=1)
     &        'WARNING: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(LOUTSTR,'(a,a)',IOSTAT=IOS,ERR=1)' WARNING in: ',
     &      STRING(1:ils)
          endif
          call edisp248(iuout,LOUTSTR,100)
          write(LOUTSTR,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &       ' value (',STR1(1:IW1),') > normal  maximum ',
     &       STR2(1:IW2),'!'
          call edisp248(iuout,LOUTSTR,100)
        ELSEIF(RACT.EQ.'F')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(LOUTSTR,'(4a)',IOSTAT=IOS,ERR=1)
     &        'FAILURE: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(LOUTSTR,'(a,a)',IOSTAT=IOS,ERR=1)' FAILURE in: ',
     &      STRING(1:ils)
          endif
          call edisp248(iuout,LOUTSTR,100)
          write(LOUTSTR,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &       ' value (',STR1(1:IW1),') > allowable maximum ',
     &       STR2(1:IW2),'!'
          call edisp248(iuout,loutstr,100)
          IER=1
          RETURN
        ENDIF
      ENDIF

  100 RETURN

 1002 if(currentfile(1:2).ne.'  ')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        WRITE(LOUTSTR,'(6a)',IOSTAT=IOS,ERR=1) 'Failed in ',
     &    currentfile(1:LN),': conversion of ',
     &    MSG(1:LNM),' value in ',WORD
      else
        LNM=max(1,lnblnk(MSG))
        WRITE(LOUTSTR,1003,IOSTAT=IOS,ERR=1)MSG(1:LNM),WORD
 1003   FORMAT(' Failed: conversion of ',A,' value in ',A)
      endif
      CALL EDISP248(iuout,LOUTSTR,100)
      IER=2
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      LNM=max(1,lnblnk(STRING))
      if(IOS.eq.2)then
        write(outs,*) 'EGETWR: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      else
        write(outs,*) 'EGETWR: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      endif
      return

      END

C ******************** EGETWRA ********************
C Recovers (IRVA) reals of real array (RVA) from an ASCII file
C (unit IAF) reading as many lines as necessary to recover the data.
C IER=2 if EOF is reached before INUM items have been parsed. IER=1 if
C there was a problem reading it. Each value is tested against
C the minimum RMN and the maximum RMX and provides a warning
C message if RACT='W', a failure message if RACT='F' and does
C no range checking if RACT='-'. Words may be separated by blanks,
C commas, or tab: WORD,WORD,WORD or WORD WORD WORD or WORD, WORD, WORD
C are all valid.  Will deal with input lines upto 1K wide.
C Note: all reads are able to accept commented files.

      SUBROUTINE EGETWRA(IAF,RVA,IRVA,RMN,RMX,RACT,MSG,IER)
      common/OUTIN/IUOUT,IUIN,IEOUT
      DIMENSION RVA(*)
      CHARACTER*(*) MSG
      CHARACTER OUTS*248,MSG1*124,RACT*1,lkouts*1000
      logical unixok

      IF(IRVA.LE.0)GOTO 99

C Read a line from the file, if no problems set position to 0 and begin
C to parse words.
      CALL STRIPC1K(IAF,lkouts,0,ND,1,MSG,IER)
      LN=max(1,LNBLNK(lkouts))
      IF(IER.NE.0)RETURN
      K=0
      DO 12 KV=1,IRVA

C If character position is < actual length of string parse another
C word, otherwise read another line from the file.
        IF(K.LT.LN)THEN
          CALL EGETWR(lkouts,K,VAL,RMN,RMX,RACT,MSG,IERV)
          IF(IERV.NE.0) THEN
            CALL STRIPC1K(IAF,lkouts,0,ND,0,MSG,IER)
            LN=max(1,LNBLNK(lkouts))
            IF(IER.NE.0)RETURN
            K=0
            CALL EGETWR(lkouts,K,VAL,RMN,RMX,RACT,MSG,IERV)
          ENDIF
        ELSE
          CALL STRIPC1K(IAF,lkouts,0,ND,0,MSG,IER)
          LN=max(1,LNBLNK(lkouts))
          IF(IER.NE.0)RETURN
          K=0
          CALL EGETWR(lkouts,K,VAL,RMN,RMX,RACT,MSG,IERV)
        ENDIF
        IF(IERV.NE.0) GOTO 1001
        RVA(KV)=VAL
   12 CONTINUE

  100 RETURN

   99 CALL EDISP(iuout,'EGETWRA array size zero, skipping read.')
      GOTO 100

 1001 LNM=max(1,lnblnk(MSG))
      WRITE(MSG1,1002,IOSTAT=IOS,ERR=1)MSG(1:LNM),VAL,K
 1002 FORMAT(' Problem reading ',A,' value= ',F9.3,' @ pos ',
     &       I3,' in the string:')
      CALL EDISP(iuout,MSG1)
      if(LNM.ge.124)then
        CALL EDISP248(iuout,lkouts,100)
      else
        CALL EDISP(iuout,lkouts)
      endif
      IER=1
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(6,*) 'EGETWRA: permission error writing warning re:',OUTS
      else
        write(6,*) 'EGETWRA: error writing warning re: ',OUTS
      endif
      return

      END

C ******************** EGETWIA ********************
C Recovers (IRVA) integers of integer array (IVA) from an
C ASCII file (unit IAF) reading as many lines as necessary to
C recover the data.
C IER=2 if EOF is reached before IRVA items have been parsed. IER=1 if
C there was a problem reading it. Each value is tested against
C the minimum IRMN and the maximum IRMX and provides a warning
C message if IACT='W', a failure message if IACT='F' and does
C no range checking if IACT='-'. Words may be separated by blanks,
C commas, or tab: WORD,WORD,WORD or WORD WORD WORD or WORD, WORD, WORD
C are all valid.  Will deal with input lines upto 1K wide.
C Note: all reads are able to accept commented files.

      SUBROUTINE EGETWIA(IAF,IVA,IRVA,IRMN,IRMX,IACT,MSG,IER)
      common/OUTIN/IUOUT,IUIN,IEOUT
      DIMENSION IVA(*)
      CHARACTER*(*) MSG
      CHARACTER OUTS*248,MSG1*124,IACT*1,out*124,lkouts*1000
      logical unixok

      IF(IRVA.LE.0)GOTO 99

C Read a line from the file, if no problems set position to 0 and begin
C to parse words.
      CALL STRIPC1K(IAF,lkouts,0,ND,1,MSG,IER)
      LN=max(1,LNBLNK(lkouts))
      IF(IER.NE.0)RETURN
      K=0
      DO 12 KV=1,IRVA

C If character position is < actual length of string parse another
C word, otherwise read another line from the file.
        IF(K.LT.LN)THEN
          CALL EGETWI(lkouts,K,IVAL,IRMN,IRMX,IACT,MSG,IERV)
          IF(IERV.NE.0) THEN
            CALL STRIPC1K(IAF,lkouts,0,ND,0,MSG,IER)
            LN=max(1,LNBLNK(lkouts))
            IF(IER.NE.0)RETURN
            K=0
            CALL EGETWI(lkouts,K,IVAL,IRMN,IRMX,IACT,MSG,IERV)
          ENDIF
        ELSE
          CALL STRIPC1K(IAF,lkouts,0,ND,0,MSG,IER)
          LN=max(1,LNBLNK(lkouts))
          IF(IER.NE.0)RETURN
          K=0
          CALL EGETWI(lkouts,K,IVAL,IRMN,IRMX,IACT,MSG,IERV)
        ENDIF
        IF(IERV.NE.0) GOTO 1001
        IVA(KV)=IVAL
   12 CONTINUE

  100 RETURN

   99 CALL EDISP(iuout,'EGETWIA: array size zero, skipping read.')
      GOTO 100

 1001 LNM=max(1,lnblnk(MSG))
      WRITE(MSG1,1002,IOSTAT=IOS,ERR=1)MSG(1:LNM),IVAL,K
 1002 FORMAT(' Problem reading ',A,' value= ',I8,' @ pos ',
     &       I3,' in the string:')
      CALL EDISP(iuout,MSG1)
      if(LNM.ge.124)then
        write(out,*) 'EGETWIA: error writing ',lkouts(1:72),'...'
        CALL EDISP(iuout,out)
      else
        CALL EDISP(iuout,OUTS)
      endif
      IER=1
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(6,*) 'EGETWIA: permission error writing warning re:',OUTS
      else
        write(6,*) 'EGETWIA: error writing warning re: ',OUTS
      endif
      return

      END


C ******************** EGETWIA1K ********************
C Recovers (IRVA) integers of integer array (IVA) from an
C ASCII file (unit IAF) reading as many lines as necessary into a 1K
C buffer to recover the data.
C IER=2 if EOF is reached before IRVA items have been parsed. IER=1 if
C there was a problem reading it. Each value is tested against
C the minimum IRMN and the maximum IRMX and provides a warning
C message if IACT='W', a failure message if IACT='F' and does
C no range checking if IACT='-'. Words may be separated by blanks,
C commas, or tab: WORD,WORD,WORD or WORD WORD WORD or WORD, WORD, WORD
C are all valid.  Will deal with input lines upto 1K wide.
C Note: all reads are able to accept commented files.

      SUBROUTINE EGETWIA1K(IAF,IVA,IRVA,IRMN,IRMX,IACT,MSG,IER)
      common/OUTIN/IUOUT,IUIN,IEOUT
      DIMENSION IVA(*)
      CHARACTER*(*) MSG
      CHARACTER OUTS*248,MSG1*124,IACT*1,out*124,lkouts*1000
      logical unixok

      IF(IRVA.LE.0)GOTO 99

C Read a line from the file, if no problems set position to 0 and begin
C to parse words.
      CALL STRIPC1K(IAF,lkouts,0,ND,1,MSG,IER)
C      CALL LSTRIPC(IAF,OUTS,0,ND,1,MSG,IER)
      LN=max(1,LNBLNK(lkouts))
      IF(IER.NE.0)RETURN
      K=0
      DO 12 KV=1,IRVA

C If character position is < actual length of string parse another
C word, otherwise read another line from the file.
        IF(K.LT.LN)THEN
          CALL EGETWI(lkouts,K,IVAL,IRMN,IRMX,IACT,MSG,IERV)
          IF(IERV.NE.0) THEN
            CALL STRIPC1K(IAF,lkouts,0,ND,0,MSG,IER)
C            CALL LSTRIPC(IAF,OUTS,0,ND,0,MSG,IER)
            LN=max(1,LNBLNK(lkouts))
            IF(IER.NE.0)RETURN
            K=0
            CALL EGETWI(lkouts,K,IVAL,IRMN,IRMX,IACT,MSG,IERV)
          ENDIF
        ELSE
          CALL STRIPC1K(IAF,lkouts,0,ND,0,MSG,IER)
C          CALL LSTRIPC(IAF,OUTS,0,ND,0,MSG,IER)
          LN=max(1,LNBLNK(lkouts))
          IF(IER.NE.0)RETURN
          K=0
          CALL EGETWI(lkouts,K,IVAL,IRMN,IRMX,IACT,MSG,IERV)
        ENDIF
        IF(IERV.NE.0) GOTO 1001
        IVA(KV)=IVAL
   12 CONTINUE

  100 RETURN

   99 CALL EDISP(iuout,'EGETWIA1K: array size zero, skipping read.')
      GOTO 100

 1001 LNM=max(1,lnblnk(MSG))
      WRITE(MSG1,1002,IOSTAT=IOS,ERR=1)MSG(1:LNM),IVAL,K
 1002 FORMAT(' Problem reading ',A,' value= ',I8,' @ pos ',
     &       I3,' in the string:')
      CALL EDISP(iuout,MSG1)
      if(LNM.ge.124)then
        write(out,*) 'EGETWIA: error writing ',lkouts(1:72),'...'
        CALL EDISP(iuout,out)
      else
        CALL EDISP(iuout,OUTS)
      endif
      IER=1
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(6,*) 'EGETWIA: permission error writing warning re:',OUTS
      else
        write(6,*) 'EGETWIA: error writing warning re: ',OUTS
      endif
      return

      END


C ******************** EGETAGWIA ********************
C Recovers (IRVA) integers of integer array (IVA) from a string
C TSTR (from position K) and if TSTR does not hold all of the array then
C it continues reading from an ASCII file (unit IAF) reading as
C many lines as necessary to recover the data.
C IER=2 if EOF is reached before IRVA items have been parsed. IER=1 if
C there was a problem reading it. Each value is tested against
C the minimum IRMN and the maximum IRMX and provides a warning
C message if IACT='W', a failure message if IACT='F' and does
C no range checking if IACT='-'. Words may be separated by blanks,
C commas, or tab: WORD,WORD,WORD or WORD WORD WORD or WORD, WORD, WORD
C are all valid.
C Note: all reads are able to accept commented files.

      SUBROUTINE EGETAGWIA(TSTR,K,IAF,IRVA,IVA,IRMN,IRMX,IACT,MSG,IER)
      common/OUTIN/IUOUT,IUIN,IEOUT
      DIMENSION IVA(*)
      CHARACTER*(*) TSTR,MSG
      CHARACTER OUTS*248,IACT*1,MSG1*124,out*124
      logical swapover
      logical unixok

      IF(IRVA.LE.0)GOTO 99

C LN is the actual length of TSTR
C Initially set swapover = false. When TSTR exhaused set swapover
C = true and read from outs.
      swapover=.false.
      ier=0
      LN=max(1,lnblnk(TSTR))
      DO 12 KV=1,IRVA

C If character position is < actual length of string parse another
C word, otherwise read another line from the file.
        IF(K.LT.LN)THEN
          if(swapover)then
            CALL EGETWI(outs,K,IVAL,IRMN,IRMX,IACT,MSG,IERV)
          else
            CALL EGETWI(TSTR,K,IVAL,IRMN,IRMX,IACT,MSG,IERV)
          endif
          IF(IERV.NE.0) THEN
            CALL LSTRIPC(IAF,OUTS,0,ND,0,MSG,IER)
            LN=max(1,LNBLNK(OUTS))
            IF(IER.NE.0)RETURN
            K=0
            CALL EGETWI(OUTS,K,IVAL,IRMN,IRMX,IACT,MSG,IERV)
            swapover=.true.
          ENDIF
        ELSE
          CALL LSTRIPC(IAF,OUTS,0,ND,0,MSG,IER)
          LN=max(1,LNBLNK(OUTS))
          IF(IER.NE.0)RETURN
          K=0
          CALL EGETWI(OUTS,K,IVAL,IRMN,IRMX,IACT,MSG,IERV)
          swapover=.true.
        ENDIF
        IF(IERV.NE.0) GOTO 1001
        IVA(KV)=IVAL
   12 CONTINUE

  100 RETURN

   99 CALL EDISP(iuout,'EGETAGWIA: array size zero, skipping read.')
      GOTO 100

 1001 LNM=max(1,lnblnk(MSG))
      WRITE(MSG1,1002,IOSTAT=IOS,ERR=1)MSG(1:LNM),IVAL,K
 1002 FORMAT(' Problem reading ',A,' value= ',I8,' @ pos ',
     &       I3,' in the string:')
      CALL EDISP(iuout,MSG1)
      if(LNM.ge.124)then
        write(out,*) 'EGETAGWIA: error writing ',OUTS(1:72),'...'
        CALL EDISP(iuout,out)
      else
        CALL EDISP(iuout,OUTS)
      endif
      IER=1
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(6,*) 'EGETAGWIA: permission error writing warning ',OUTS
      else
        write(6,*) 'EGETAGWIA: error writing warning re: ',OUTS
      endif
      return

      end

C ******************** EGETAGWRA ********************
C Recovers (IRVA) reals of real array (RVA) from a string
C TSTR (from position K) and if TSTR does not hold all of the array then
C it continues reading from an ASCII file (unit IAF) reading as
C many lines as necessary to recover the data.
C IER=2 if EOF is reached before IRVA items have been parsed. IER=1 if
C there was a problem reading it. Each value is tested against
C the minimum RMN and the maximum RMX and provides a warning
C message if IACT='W', a failure message if IACT='F' and does
C no range checking if RACT='-'. Words may be separated by blanks,
C commas, or tab: WORD,WORD,WORD or WORD WORD WORD or WORD, WORD, WORD
C are all valid.
C Note: all reads are able to accept commented files.

      SUBROUTINE EGETAGWRA(TSTR,K,IAF,IRVA,RVA,RMN,RMX,RACT,MSG,IER)
      common/OUTIN/IUOUT,IUIN,IEOUT
      DIMENSION RVA(*)
      CHARACTER*(*) TSTR,MSG
      CHARACTER OUTS*248,RACT*1,MSG1*124,out*124
      logical swapover
      logical unixok

      IF(IRVA.LE.0)GOTO 99

C LN is the actual length of TSTR.
      ier=0
      LN=max(1,lnblnk(TSTR))

C Initially set swapover = false. When TSTR exhaused set swapover
C = true and read from outs.
      swapover=.false.
      DO 12 KV=1,IRVA

C If character position is < actual length of string parse another
C word, otherwise read another line from the file.
        IF(K.LT.LN)THEN
          if(swapover)then
            CALL EGETWR(OUTS,K,VAL,RMN,RMX,RACT,MSG,IERV)
          else
            CALL EGETWR(TSTR,K,VAL,RMN,RMX,RACT,MSG,IERV)
          endif
          IF(IERV.NE.0) THEN
            CALL LSTRIPC(IAF,OUTS,0,ND,0,MSG,IER)
            LN=max(1,LNBLNK(OUTS))
            IF(IER.NE.0)RETURN
            K=0
            CALL EGETWR(OUTS,K,VAL,RMN,RMX,RACT,MSG,IERV)
            swapover=.true.
          ENDIF
        ELSE
          CALL LSTRIPC(IAF,OUTS,0,ND,0,MSG,IER)
          LN=max(1,LNBLNK(OUTS))
          IF(IER.NE.0)RETURN
          K=0
          CALL EGETWR(OUTS,K,VAL,RMN,RMX,RACT,MSG,IERV)
          swapover=.true.
        ENDIF
        IF(IERV.NE.0) GOTO 1001
        RVA(KV)=VAL
   12 CONTINUE

  100 RETURN

   99 CALL EDISP(iuout,'EGETAGWRA: array size zero, skipping read.')
      GOTO 100

 1001 LNM=max(1,lnblnk(MSG))
      WRITE(MSG1,1002,IOSTAT=IOS,ERR=1)MSG(1:LNM),VAL,K
 1002 FORMAT(' Problem reading ',A,' value= ',F9.3,' @ pos ',
     &       I3,' in the string:')
      CALL EDISP(iuout,MSG1)
      if(LNM.ge.124)then
        write(out,*) 'EGETAGWRA: error writing ',OUTS(1:72),'...'
        CALL EDISP(iuout,out)
      else
        CALL EDISP(iuout,OUTS)
      endif
      IER=1
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(6,*) 'EGETAGWRA: permission error writing warning ',OUTS
      else
        write(6,*) 'EGETAGWRA: error writing warning re: ',OUTS
      endif
      return

      end

C ******************** EGETAGWRA1K ********************
C Recovers (IRVA) reals of real array (RVA) from a string
C TSTR (from position K) and if TSTR does not hold all of the array then
C it continues reading from an ASCII file (unit IAF) into a 1K buffer,
C reading as many lines as necessary to recover the data.
C IER=2 if EOF is reached before IRVA items have been parsed. IER=1 if
C there was a problem reading it. Each value is tested against
C the minimum RMN and the maximum RMX and provides a warning
C message if IACT='W', a failure message if IACT='F' and does
C no range checking if RACT='-'. Words may be separated by blanks,
C commas, or tab: WORD,WORD,WORD or WORD WORD WORD or WORD, WORD, WORD
C are all valid.
C Note: all reads are able to accept commented files.

      SUBROUTINE EGETAGWRA1K(TSTR,K,IAF,IRVA,RVA,RMN,RMX,RACT,MSG,IER)
      common/OUTIN/IUOUT,IUIN,IEOUT
      DIMENSION RVA(*)
      CHARACTER*(*) TSTR,MSG
      CHARACTER OUTS*1000,RACT*1,MSG1*124,out*124,out2*248
      logical swapover
      logical unixok

      IF(IRVA.LE.0)GOTO 99

C LN is the actual length of TSTR.
      ier=0
      LN=max(1,lnblnk(TSTR))

C Initially set swapover = false. When TSTR exhaused set swapover
C = true and read from outs.
      swapover=.false.
      DO 12 KV=1,IRVA

C If character position is < actual length of string parse another
C word, otherwise read another line from the file.
        IF(K.LT.LN)THEN
          if(swapover)then
            CALL EGETWR(OUTS,K,VAL,RMN,RMX,RACT,MSG,IERV)
          else
            CALL EGETWR(TSTR,K,VAL,RMN,RMX,RACT,MSG,IERV)
          endif
          IF(IERV.NE.0) THEN
            CALL STRIPC1K(IAF,OUTS,0,ND,0,MSG,IER)
            LN=max(1,LNBLNK(OUTS))
            IF(IER.NE.0)RETURN
            K=0
            CALL EGETWR(OUTS,K,VAL,RMN,RMX,RACT,MSG,IERV)
            swapover=.true.
          ENDIF
        ELSE
          CALL STRIPC1K(IAF,OUTS,0,ND,0,MSG,IER)
          LN=max(1,LNBLNK(OUTS))
          IF(IER.NE.0)RETURN
          K=0
          CALL EGETWR(OUTS,K,VAL,RMN,RMX,RACT,MSG,IERV)
          swapover=.true.
        ENDIF
        IF(IERV.NE.0) GOTO 1001
        RVA(KV)=VAL
   12 CONTINUE

  100 RETURN

   99 CALL EDISP(iuout,'EGETAGWRA1K: array size zero, skipping read.')
      GOTO 100

 1001 LNM=max(1,lnblnk(MSG))
      WRITE(MSG1,1002,IOSTAT=IOS,ERR=1)MSG(1:LNM),VAL,K
 1002 FORMAT(' Problem reading ',A,' value= ',F9.3,' @ pos ',
     &       I3,' in the string:')
      CALL EDISP(iuout,MSG1)
      if(LNM.ge.124)then
        write(out,*) 'EGETAGWRA1K: error writing ',OUTS(1:72),'...'
        CALL EDISP(iuout,out)
      elseif(LNM.gt.124.and.LNM.le.248)then
        CALL EDISP(iuout,OUTS)
      else
        write(out2,*) 'EGETAGWRA1K: error writing ',OUTS(1:200),'...'
        CALL EDISP(iuout,out2)
      endif
      IER=1
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(6,*) 'EGETAGWRA: permission error writing warning ',OUTS
      else
        write(6,*) 'EGETAGWRA: error writing warning re: ',OUTS
      endif
      return

      end


C ******************** EGETAGWSA ********************
C Recovers (ISVA) words into string array (SVA) from a string
C TSTR (from position K) and if TSTR does not hold all of the array then
C it continues reading from an ASCII file (unit IAF) reading as
C many lines as necessary to recover the data

C The string array SVN is an array of inisz words assumed to be less than
C or equal to *32 characters (that is the size of the buffer). This 
C will work with space or comma or tab separations.

C IER=2 if EOF is reached before ISVA items have been parsed. IER=1 if
C there was a problem reading it. Each value is tested against
C a blank stringt and provides a warning message if SACT='W', a failure
C message if SACT='F' and does no blank checking if SACT='-'.
C Phrases may be separated by spaces commas, or tab: word,word,word or
C word, word, word, or word<tab>word<tab>word are all valid.
C Note: all reads are able to accept commented files.

      SUBROUTINE EGETAGWSA(TSTR,K,IAF,ISVA,SVA,inisz,SACT,MSG,IER)
      common/OUTIN/IUOUT,IUIN,IEOUT
      dimension SVA(inisz)
      CHARACTER*(*) SVA
      CHARACTER*(*) TSTR,MSG
      CHARACTER OUTS*248,SACT*1,MSG1*124,buffer*32,out*124
      logical swapover
      logical unixok

      IF(ISVA.LE.0)GOTO 99

C LN is the actual length of TSTR.
      ier=0
      LN=max(1,lnblnk(TSTR))

C Initially set swapover = false. When TSTR exhaused set swapover
C = true and read from outs.
      swapover=.false.
      DO 12 KV=1,ISVA

C If character position is < actual length of string parse another
C word, otherwise read another line from the file.
        IF(K.LT.LN)THEN
          if(swapover)then
            CALL EGETW(OUTS,K,buffer,SACT,MSG,IERV)
          else
            CALL EGETW(TSTR,K,buffer,SACT,MSG,IERV)
          endif
          IF(IERV.NE.0) THEN
            CALL LSTRIPC(IAF,OUTS,0,ND,0,MSG,IER)
            LN=max(1,LNBLNK(OUTS))
            IF(IER.NE.0)RETURN
            K=0
            CALL EGETW(OUTS,K,buffer,SACT,MSG,IERV)
            swapover=.true.
          ENDIF
        ELSE
          CALL LSTRIPC(IAF,OUTS,0,ND,0,MSG,IER)
          LN=max(1,LNBLNK(OUTS))
          IF(IER.NE.0)RETURN
          K=0
          CALL EGETW(OUTS,K,buffer,SACT,MSG,IERV)
          swapover=.true.
        ENDIF
        IF(IERV.NE.0) GOTO 1001
        write(SVA(KV),'(a)') buffer(1:lnblnk(buffer))
   12 CONTINUE

  100 RETURN

   99 CALL EDISP(iuout,'EGETAGWSA array size zero, skipping read.')
      GOTO 100

 1001 LNM=max(1,lnblnk(MSG))
      WRITE(MSG1,1002,IOSTAT=IOS,ERR=1)MSG(1:LNM),buffer,K
 1002 FORMAT(' Problem reading ',A,' word= ',A,' @ pos ',
     &       I3,' in the string:')
      CALL EDISP(iuout,MSG1)
      if(LNM.ge.124)then
        write(out,*) 'EGETAGWSA: error writing ',OUTS(1:72),'...'
        CALL EDISP(iuout,out)
      else
        CALL EDISP(iuout,OUTS)
      endif
      IER=1
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(6,*) 'EGETAGWSA: permission error writing warning ',OUTS
      else
        write(6,*) 'EGETAGWSA: error writing warning re: ',OUTS
      endif
      return

      end


C ******************** EGETAGWPA ********************
C Recovers (ISVA) phrases into string array (SVA) from a string
C TSTR (from position K) and if TSTR does not hold all of the array then
C it continues reading from an ASCII file (unit IAF) reading as
C many lines as necessary to recover the data

C The string array SVN is an array of inisz items assumed to be less than
C or equal to *32 characters (that is the size of the buffer). This is
C flexable enough to work with phrases (and so it requires comma or
C tab separation in the file).

C IER=2 if EOF is reached before ISVA items have been parsed. IER=1 if
C there was a problem reading it. Each value is tested against
C a blank stringt and provides a warning message if SACT='W', a failure
C message if SACT='F' and does no blank checking if SACT='-'.
C Phrases may be separated by commas, or tab: phrase,phrase,phrase or
C phrase, phrase, phrase, or phrase<tab>phrase<tab>phrase are all valid.
C Note: all reads are able to accept commented files.

      SUBROUTINE EGETAGWPA(TSTR,K,IAF,ISVA,SVA,inisz,SACT,MSG,IER)
      common/OUTIN/IUOUT,IUIN,IEOUT
      dimension SVA(inisz)
      CHARACTER*(*) SVA
      CHARACTER*(*) TSTR,MSG
      CHARACTER OUTS*248,SACT*1,MSG1*124,buffer*32,out*124
      logical swapover
      logical unixok

      IF(ISVA.LE.0)GOTO 99

C LN is the actual length of TSTR.
      ier=0
      LN=max(1,lnblnk(TSTR))

C Initially set swapover = false. When TSTR exhaused set swapover
C = true and read from outs.
      swapover=.false.
      DO 12 KV=1,ISVA

C If character position is < actual length of string parse another
C word, otherwise read another line from the file.
        IF(K.LT.LN)THEN
          if(swapover)then
            CALL EGETP(OUTS,K,buffer,SACT,MSG,IERV)
          else
            CALL EGETP(TSTR,K,buffer,SACT,MSG,IERV)
          endif
          IF(IERV.NE.0) THEN
            CALL LSTRIPC(IAF,OUTS,0,ND,0,MSG,IER)
            LN=max(1,LNBLNK(OUTS))
            IF(IER.NE.0)RETURN
            K=0
            CALL EGETP(OUTS,K,buffer,SACT,MSG,IERV)
            swapover=.true.
          ENDIF
        ELSE
          CALL LSTRIPC(IAF,OUTS,0,ND,0,MSG,IER)
          LN=max(1,LNBLNK(OUTS))
          IF(IER.NE.0)RETURN
          K=0
          CALL EGETP(OUTS,K,buffer,SACT,MSG,IERV)
          swapover=.true.
        ENDIF
        IF(IERV.NE.0) GOTO 1001
        write(SVA(KV),'(a)') buffer(1:lnblnk(buffer))
   12 CONTINUE

  100 RETURN

   99 CALL EDISP(iuout,'EGETAGWPA: array size zero, skipping read.')
      GOTO 100

 1001 LNM=max(1,lnblnk(MSG))
      WRITE(MSG1,1002,IOSTAT=IOS,ERR=1)MSG(1:LNM),buffer,K
 1002 FORMAT(' Problem reading ',A,' phrase= ',A,' @ pos ',
     &       I3,' in the string:')
      CALL EDISP(iuout,MSG1)
      if(LNM.ge.124)then
        write(out,*) 'EGETAGWPA: error writing ',OUTS(1:72),'...'
        CALL EDISP(iuout,out)
      else
        CALL EDISP(iuout,OUTS)
      endif
      IER=1
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(6,*) 'EGETAGWPA: permission error writing warning ',OUTS
      else
        write(6,*) 'EGETAGWPA: error writing warning re: ',OUTS
      endif
      return

      end

C ******************** EGETRM ********************
C Returns the remainder of a text string TSTR after position k
C in RSTR where RSTR has no leading blanks.

      SUBROUTINE EGETRM(TSTR,K,RSTR,ACT,MSG,IER)
#include "espriou.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER*(*) TSTR,RSTR,MSG
      character ACT*1,A*1,loutstr*248
      logical unixok

C LS is th maximum length of STRING, L the current position,
C LR the maximum length of RSTR.
      RSTR=' '
      LS=LEN(TSTR)
      LR=LEN(RSTR)
      L=0

C Start by skipping blanks and tabs before TSTR.
   10 K=K+1
      IF(K.GT.LS) GOTO 999
      A=TSTR(K:K)
      IF(A.EQ.' '.OR.A.EQ.','.OR.(ICHAR(A).EQ.9)) GOTO 10

C Copy RSTR from TSTR character by character until either end
C of RSTR or TSTR.
   20 L=L+1
      IF(L.GT.LR)goto 100
      RSTR(L:L)=A
      K=K+1
      IF(K.GT.LS) goto 100
      A=TSTR(K:K)
      GOTO 20

  100 RETURN

  999 if(ACT.EQ.'-')then
        RETURN
      elseif(ACT.EQ.'W')then
        ier=1
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1) 'WARNING in ',
     &      currentfile(1:LN),': no chars found for ',
     &      MSG(1:LNM),' string in:'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &     'WARNING: no characters found for ',MSG(1:LNM),' string in:'
        endif
      elseif(ACT.EQ.'F')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1) 'FAILURE in ',
     &      currentfile(1:LN),': no chars found for ',
     &      MSG(1:LNM),' string in:'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: no characters found for ',MSG(1:LNM),' string in:'
        endif
      endif
      CALL EDISP248(iuout,loutstr,100)
      if(lnblnk(TSTR).gt.123)then
        CALL EDISP248(iuout,TSTR,100)
      else
        CALL EDISP(iuout,TSTR)
      endif
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(loutstr,*) 'EGETRM: permission error writing warning ',
     &    RSTR
      else
        write(loutstr,*) 'EGETRM: error writing warning re: ',RSTR
      endif
      call edisp248(iuout,loutstr,100)
      return

      END

C ******************** EGETXMLTAG ********************
C EGETXMLTAGE gets first XML tag after position K from the STRING of
C characters. Strips the leading < and the trailing >. Returns k at
C the trailing > Spaces within tag are ok. Provides a warning
C message if ACT='W', a failure message if ACT='F' and does
C no message if ACT='-'.

      SUBROUTINE EGETXMLTAG(STRING,K,PHRASE,ACT,MSG,ier)
#include "espriou.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER*(*) PHRASE, STRING, MSG
      CHARACTER ACT*1,A*1,loutstr*248,outs*124
      logical unixok

C LS is th maximum length of STRING, L the current position,
C LW the maximum length of PHRASE.
      ier=0
      PHRASE=' '
      LS=LEN(STRING)
      LW=LEN(PHRASE)
      L=0

C Start by skipping blanks tabs before the PHRASE.
   10 K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(A.EQ.' '.OR.A.EQ.','.OR.(ICHAR(A).EQ.9)) GOTO 10

      if(A.EQ.'<') then
        K=K+1            ! increment to next character
        A=STRING(K:K)
        goto 20          ! proceed to copy
      elseif(A.EQ.'>') then
        continue
      endif

C Copy PHRASE from STRING, character by character until tab or a
C comma or > is found.
   20 L=L+1
      IF(L.GT.LW) return
      if(A.EQ.'>') then
        return          ! do not save > in phrase
      endif
      PHRASE(L:L)=A     ! copy character into the phrase
      K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(ICHAR(A).EQ.9.or.A.eq.',') GO TO 100
      if(A.EQ.'>') then
        continue
      endif
      GOTO 20

  100 RETURN

  999 if(ACT.EQ.'-')then
        RETURN
      elseif(ACT.EQ.'W')then
        ier=1
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' (phrase) in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: Past end of line for ',
     &      MSG(1:LNM),' (phrase) in...'
        endif
      elseif(ACT.EQ.'F')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' (phrase) in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: Past end of line for ',
     &      MSG(1:LNM),' (phrase) in...'
        endif
      endif
      CALL EDISP248(iuout,loutstr,100)
      if(lnblnk(STRING).gt.123)then
        CALL EDISP248(iuout,STRING,100)
      else
        CALL EDISP(iuout,STRING)
      endif
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      LNM=max(1,lnblnk(STRING))
      if(IOS.eq.2)then
        write(outs,*) 'EGETXMLTAG: permission error writing warning: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      else
        write(outs,*) 'EGETXMLTAG: error writing warning: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      endif
      return

      END


C ******************** EGETWXML ********************
C Gets first WORD after position K from the STRING of
C characters. Words are separated by blanks, commas, |, <, or tab: WORD,WORD,WORD
C or WORD WORD WORD or WORD, WORD, WORD  WORD> are all valid.  Provides a warning
C message if ACT='W', a failure message if ACT='F' and does
C no message if ACT='-'.  Modified after:
C G.N. Walton, US Nat. Institute of Standards and Technology
C     LS     - maximum length of STRING
C     L      - current position in WORD
C     LW     - maximum length of WORD

      SUBROUTINE EGETWXML(STRING,K,WORD,ACT,MSG,IER)
#include "espriou.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER*(*) WORD, STRING, MSG
      CHARACTER ACT*1,A*1,loutstr*248,outs*124
      logical unixok

      WORD=' '
      LS=LEN(STRING)
      LW=LEN(WORD)
      L=0

C Start by skipping blanks and tabs before the word.
   10 K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(A.EQ.' '.OR.A.EQ.','.OR.(ICHAR(A).EQ.9).OR.
     &   (ICHAR(A).eq.124)) GOTO 10

C Copy WORD from STRING, character by character until separator found.
C A < is considered a separator as it represents start of a subsequent
C XML tag.
   20 L=L+1
      IF(L.GT.LW) GOTO 100
      WORD(L:L)=A
      K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(A.EQ.' '.OR.A.EQ.','.OR.(ICHAR(A).EQ.9).OR.
     &   (ICHAR(A).eq.124).OR.A.EQ.'<') GO TO 100
      GOTO 20

  100 RETURN

  999 ier=1
      if(ACT.EQ.'-')then
        RETURN
      elseif(ACT.EQ.'W')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: Past end of line for ',
     &      MSG(1:LNM),' (word) in...'
        endif
      elseif(ACT.EQ.'F')then
        ier=1
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: Past end of line for ',
     &      MSG(1:LNM),' (word) in...'
        endif
      endif
      CALL EDISP248(iuout,loutstr,100)
      if(lnblnk(STRING).gt.123)then
        CALL EDISP248(iuout,STRING,100)
      else
        CALL EDISP(iuout,STRING)
      endif
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      LNM=max(1,lnblnk(STRING))
      if(IOS.eq.2)then
        write(outs,*) 'EGETWXML: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      else
        write(outs,*) 'EGETWXML: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      endif
      return

      END


C ******************** EGETWIXML ********************
C Gets first word after position K from the STRING of
C characters and converts it into an integer IV, tests it against
C the minimum MN and the maximum MX and provides a warning
C message if ACT='W', a failure message if ACT='F' and does
C no range checking if ACT='-'. Words may be separated by blanks,
C commas, <, or tab: WORD,WORD,WORD or WORD WORD WORD or WORD, WORD, WORD
C are all valid.

      SUBROUTINE EGETWIXML(STRING,K,IV,MN,MX,ACT,MSG,IER)
#include "espriou.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER*(*) STRING, MSG
      CHARACTER ACT*1,STR1*10,STR2*10,WORD*20,loutstr*248,outs*124
      logical unixok

C Pick up line and lenght for error messages.
      ils=max(1,lnblnk(STRING))
      if(ils.gt.105)ils=105

      IER=0
      WORD=' '

C Use alternative call that also checks for <.
      CALL EGETWXML(STRING,K,WORD,'-','integer',IER)
      IF(IER.NE.0)RETURN
      read(WORD,*,IOSTAT=IOS,ERR=1002)iv

C If range checking disabled, jump out of routine.
      IF(ACT.EQ.'-')RETURN

C Make up reporting string.
      CALL INTSTR(IV,STR1,IW1,IER)

C Check IV against minimum and respond based on ACT.
      IF(IV.LT.MN)THEN
        CALL INTSTR(MN,STR2,IW2,IER)
        IF(ACT.EQ.'W')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(4a)',IOSTAT=IOS,ERR=1)
     &        'WARNING: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(loutstr,'(a,a)',IOSTAT=IOS,ERR=1)' WARNING in: ',
     &      STRING(1:ils)
          endif
          CALL EDISP248(iuout,loutstr,100)
          write(loutstr,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &      ' value (',STR1(1:IW1),') < normal minimum ',STR2(1:IW2),'!'
          CALL EDISP248(iuout,loutstr,100)
        ELSEIF(ACT.EQ.'F')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(4a)',IOSTAT=IOS,ERR=1)
     &        'FAILURE: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(loutstr,'(a,a)',IOSTAT=IOS,ERR=1)' FAILURE in: ',
     &        STRING(1:ils)
          endif
          CALL EDISP248(iuout,loutstr,100)
          write(loutstr,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &       ' value (',STR1(1:IW1),') < allowable minimum ',
     &       STR2(1:IW2),'!'
          CALL EDISP248(iuout,loutstr,100)
          IER=1
          RETURN
        ENDIF
      ELSEIF(IV.GT.MX)THEN
        CALL INTSTR(MX,STR2,IW2,IER)
        IF(ACT.EQ.'W')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(4a)',IOSTAT=IOS,ERR=1)
     &        'WARNING: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(loutstr,'(a,a)',IOSTAT=IOS,ERR=1)' WARNING in: ',
     &        STRING(1:ils)
          endif
          CALL EDISP248(iuout,loutstr,100)
          write(loutstr,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &      ' value (',STR1(1:IW1),') > normal maximum ',STR2(1:IW2),'!'
          CALL EDISP248(iuout,loutstr,100)
        ELSEIF(ACT.EQ.'F')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(loutstr,'(4a)',IOSTAT=IOS,ERR=1)
     &        'FAILURE: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(loutstr,'(a,a)',IOSTAT=IOS,ERR=1)' FAILURE in: ',
     &        STRING(1:ils)
          endif
          CALL EDISP248(iuout,loutstr,100)
          write(loutstr,'(7a)',IOSTAT=IOS,ERR=1) ' the ',MSG(1:LNM),
     &       ' value (',STR1(1:IW1),') > allowable maximum ',
     &       STR2(1:IW2),'!'
          CALL EDISP248(iuout,loutstr,100)
          IER=1
          RETURN
        ENDIF
      ENDIF

  100 RETURN

 1002 if(currentfile(1:2).ne.'  ')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        WRITE(loutstr,'(6a)',IOSTAT=IOS,ERR=1) 'Failed in ',
     &    currentfile(1:LN),': conversion of ',
     &    MSG(1:LNM),' value in ',WORD
      else
        LNM=max(1,lnblnk(MSG))
        WRITE(loutstr,1003,IOSTAT=IOS,ERR=1)MSG(1:LNM),WORD
 1003   FORMAT(' Failed: conversion of ',A,' value in ',A)
      endif
      CALL EDISP248(iuout,loutstr,100)
      IER=2
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      LNM=max(1,lnblnk(STRING))
      if(IOS.eq.2)then
        write(outs,*) 'EGETWIXML: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      else
        write(outs,*) 'EGETWIXML: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      endif
      return

      END


C ******************** EGETWRXML ********************
C Gets first word after position K from the STRING of
C characters and converts it into a real number RV, tests it against
C the minimum RMN and the maximum RMX and provides a warning
C message if RACT='W', a failure message if RACT='F' and does
C no range checking if RACT='-'. Words may be separated by blanks,
C commas, <, or tab: WORD,WORD,WORD or WORD WORD WORD or WORD, WORD, WORD
C are all valid.

      SUBROUTINE EGETWRXML(STRING,K,RV,RMN,RMX,RACT,MSG,IER)
#include "espriou.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

C LOUTSTR is for messages to the user and should be long enough
c to prevent truncation of messge contents.
      CHARACTER*(*) STRING, MSG
      CHARACTER RACT*1,STR1*16,STR2*16,WORD*20,LOUTSTR*248,outs*124
      logical unixok

C Pick up line and length for error messages.
      ils=max(1,lnblnk(STRING))
      if(ils.gt.230)ils=230

      IER=0
      WORD=' '

C Use alternative call that also checks for <.
      CALL EGETWXML(STRING,K,WORD,'-','real',IER)
      IF(IER.NE.0) goto 1002
      read(WORD,*,ERR=1002)rv

C If range checking disabled, jump out of routine.
      IF(RACT.EQ.'-')RETURN

C Make up reporting string.
      CALL REL16STR(RV,STR1,IW1,IER)

C Check RV against minimum and respond based on RACT.
      IF(RV.LT.RMN)THEN
        CALL REL16STR(RMN,STR2,IW2,IER)
        IF(RACT.EQ.'W')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(LOUTSTR,'(4a)',IOSTAT=IOS,ERR=1)
     &        'WARNING: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(LOUTSTR,'(a,a)',IOSTAT=IOS,ERR=1)' WARNING in: ',
     &        STRING(1:ils)
          endif
          call edisp248(iuout,LOUTSTR,100)
          write(LOUTSTR,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &      ' value (',STR1(1:IW1),') < normal minimum ',STR2(1:IW2),'!'
          call edisp248(iuout,LOUTSTR,100)
        ELSEIF(RACT.EQ.'F')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(LOUTSTR,'(4a)',IOSTAT=IOS,ERR=1)
     &        'FAILURE: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(LOUTSTR,'(a,a)',IOSTAT=IOS,ERR=1)' FAILURE in: ',
     &      STRING(1:ils)
          endif
          call edisp248(iuout,LOUTSTR,100)
          write(LOUTSTR,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &       ' value (',STR1(1:IW1),') < allowable minumum ',
     &       STR2(1:IW2),'!'
          call edisp248(iuout,loutstr,100)
          IER=1
          RETURN
        ENDIF
      ELSEIF(RV.GT.RMX)THEN
        CALL REL16STR(RMX,STR2,IW2,IER)
        IF(RACT.EQ.'W')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(LOUTSTR,'(4a)',IOSTAT=IOS,ERR=1)
     &        'WARNING: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(LOUTSTR,'(a,a)',IOSTAT=IOS,ERR=1)' WARNING in: ',
     &      STRING(1:ils)
          endif
          call edisp248(iuout,LOUTSTR,100)
          write(LOUTSTR,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &       ' value (',STR1(1:IW1),') > normal  maximum ',
     &       STR2(1:IW2),'!'
          call edisp248(iuout,LOUTSTR,100)
        ELSEIF(RACT.EQ.'F')THEN
          call edisp(iuout,' ')
          LN=max(1,lnblnk(currentfile))
          LNM=max(1,lnblnk(MSG))
          if(currentfile(1:2).ne.'  ')then
            WRITE(LOUTSTR,'(4a)',IOSTAT=IOS,ERR=1)
     &        'FAILURE: in ',currentfile(1:LN),': ',STRING(1:ils)
          else
            write(LOUTSTR,'(a,a)',IOSTAT=IOS,ERR=1)' FAILURE in: ',
     &      STRING(1:ils)
          endif
          call edisp248(iuout,LOUTSTR,100)
          write(LOUTSTR,'(7a)',IOSTAT=IOS,ERR=1)' the ',MSG(1:LNM),
     &       ' value (',STR1(1:IW1),') > allowable maximum ',
     &       STR2(1:IW2),'!'
          call edisp248(iuout,loutstr,100)
          IER=1
          RETURN
        ENDIF
      ENDIF

  100 RETURN

 1002 if(currentfile(1:2).ne.'  ')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        WRITE(LOUTSTR,'(6a)',IOSTAT=IOS,ERR=1) 'Failed in ',
     &    currentfile(1:LN),': conversion of ',
     &    MSG(1:LNM),' value in ',WORD
      else
        LNM=max(1,lnblnk(MSG))
        WRITE(LOUTSTR,1003,IOSTAT=IOS,ERR=1)MSG(1:LNM),WORD
 1003   FORMAT(' Failed: conversion of ',A,' value in ',A)
      endif
      CALL EDISP248(iuout,LOUTSTR,100)
      IER=2
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      LNM=max(1,lnblnk(STRING))
      if(IOS.eq.2)then
        write(outs,*) 'EGETWRXML: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      else
        write(outs,*) 'EGETWRXML: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      endif
      return

      END

C ******************** EGETEQDQXML ********************
C Gets tag & first quoted PHRASE after position K 
C from the STRING of characters. Phrases can contain spaces
C and commas but not tabs.
C Provides a warning message if ACT='W', a failure message if ACT='F' and
C no message if ACT='-'.  The string returned has the quotes removed.
C Note it uses an internal string buffer which assumes that the
C phrase is less than 248 characters long.
C It is dealing with this pattern IDType="WMO"

      SUBROUTINE EGETEQDQXML(STRING,K,TAG,PHRASE,ACT,MSG,ier)
#include "espriou.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER*(*) TAG, PHRASE, STRING, MSG
      CHARACTER ACT*1,A*1,loutstr*124,outs*124
      character dq*1,sqleft*1,sqright*1
      logical unixok,isphrase

      dq = char(34)  ! double quote
      sqleft = char(96)  ! single quote left
      sqright = char(39)  ! single quote right

C LS is th maximum length of STRING, L the current position,
C LW the maximum length of PHRASE.
      ier=0
      TAG=' '; PHRASE=' '
      LS=LEN(STRING)
      LW=LEN(PHRASE)
      LT=LEN(TAG)
      L=0; isphrase=.false.

C Start by skipping blanks, commas and tabs before the PHRASE.
   10 K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(A.EQ.' '.OR.A.EQ.','.OR.(ICHAR(A).EQ.9)) GOTO 10

C Copy TAG or PHRASE from STRING, character by character until 
C a double quote or single quote is found (a quoted phrase 
C could include a comma, but not a tab)
   20 L=L+1
      IF(L.GE.LW) GOTO 100   ! at end of available phrase chars
      if(isphrase)then
        if(A.eq.dq.or.A.eq.sqleft.or.A.eq.sqright)then
          goto 100  ! we have terminating quote
        else
          PHRASE(L:L)=A
        endif
      else
        TAG(L:L)=A
      endif
      K=K+1
      IF(K.GT.LS) GOTO 999
      A=STRING(K:K)
      IF(ICHAR(A).EQ.9) GOTO 100
      if(A.eq."=")then
        isphrase=.true.  ! mark end of tag and advance
        K=K+1            ! increment K counter
        A=STRING(K:K)
        L=0              ! reset L counter
        if(A.eq.dq.or.A.eq.sqleft.or.A.eq.sqright)then
          K=K+1          ! increment K counter
          A=STRING(K:K)
          goto 20        ! loop back and deal with it
        endif
      endif
      if(A.eq.dq.or.A.eq.sqleft.or.A.eq.sqright)then

C Found terminating quote, so process.
        GOTO 100
      endif
      GOTO 20

  100 return

  999 if(ACT.EQ.'-')then
        RETURN
      elseif(ACT.EQ.'W')then
        ier=1
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' (phrase) in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'WARNING: Past end of line for ',
     &      MSG(1:LNM),' (phrase) in...'
        endif
      elseif(ACT.EQ.'F')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: in ',currentfile(1:LN),
     &      ' past end of line for ',MSG(1:LNM),' (phrase) in...'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: Past end of line for ',
     &      MSG(1:LNM),' (phrase) in...'
        endif
      endif
      CALL EDISP248(iuout,loutstr,100)
      if(lnblnk(STRING).gt.123)then
        CALL EDISP248(iuout,STRING,100)
      else
        CALL EDISP(iuout,STRING)
      endif
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      LNM=max(1,lnblnk(STRING))
      if(IOS.eq.2)then
        write(outs,*) 
     &    'EGETEQDQXML: permission error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      else
        write(outs,*) 'EGETEQDQXML: error writing warning re: '
        call edisp(iuout,outs)
        call edisp(iuout,STRING(1:LNM))
      endif
      return

      END


C ******************** EGETRMXML ********************
C Returns the remainder of a text string TSTR after position k
C in RSTR until a '<' is noticed.

      SUBROUTINE EGETRMXML(TSTR,K,RSTR,ACT,MSG,IER)
#include "espriou.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

      CHARACTER*(*) TSTR,RSTR,MSG
      character ACT*1,A*1,loutstr*248
      logical unixok

C LS is th maximum length of STRING, L the current position,
C LR the maximum length of RSTR.
      RSTR=' '
      LS=LEN(TSTR)
      LR=LEN(RSTR)
      L=0

C Start by skipping blanks and tabs before TSTR.
   10 K=K+1
      IF(K.GT.LS) GOTO 999
      A=TSTR(K:K)
      IF(A.EQ.' '.OR.A.EQ.','.OR.(ICHAR(A).EQ.9)) GOTO 10

C Copy RSTR from TSTR character by character until either end
C of RSTR or TSTR or a '<' found..
   20 L=L+1
      IF(L.GT.LR)goto 100
      IF(A.EQ.'<') GOTO 100  ! start of next XML tag
      RSTR(L:L)=A
      K=K+1
      IF(K.GT.LS) goto 100
      A=TSTR(K:K)
      GOTO 20

  100 RETURN

  999 if(ACT.EQ.'-')then
        RETURN
      elseif(ACT.EQ.'W')then
        ier=1
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1) 'WARNING in ',
     &      currentfile(1:LN),': no chars found for ',
     &      MSG(1:LNM),' string in:'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &     'WARNING: no characters found for ',MSG(1:LNM),' string in:'
        endif
      elseif(ACT.EQ.'F')then
        LN=max(1,lnblnk(currentfile))
        LNM=max(1,lnblnk(MSG))
        if(currentfile(1:2).ne.'  ')then
          WRITE(loutstr,'(5a)',IOSTAT=IOS,ERR=1) 'FAILURE in ',
     &      currentfile(1:LN),': no chars found for ',
     &      MSG(1:LNM),' string in:'
        else
          WRITE(loutstr,'(3a)',IOSTAT=IOS,ERR=1)
     &      'FAILURE: no characters found for ',MSG(1:LNM),' string in:'
        endif
      endif
      CALL EDISP248(iuout,loutstr,100)
      if(lnblnk(TSTR).gt.123)then
        CALL EDISP248(iuout,TSTR,100)
      else
        CALL EDISP(iuout,TSTR)
      endif
      GOTO 100

   1  call isunix(unixok)
      if(.NOT.unixok) return  ! if DOS return because of lack of I/O channel
      if(IOS.eq.2)then
        write(loutstr,*) 'EGETRM: permission error writing warning ',
     &    RSTR
      else
        write(loutstr,*) 'EGETRM: error writing warning re: ',RSTR
      endif
      call edisp248(iuout,loutstr,100)
      return

      END
      

C ******************** isadll ********************
C Find if module is being used as a dll (silent running).
      subroutine isadll(yes)
      logical yes

C NOTE: Edit as required for machine environment, normally
C will return false.
      yes = .false.
      return
      end

C ******************** isunix ********************
C Determine if machine is unix.
C NOTE: uses compiler variable -DMINGW to signal .false.

      subroutine isunix(yes)
      logical yes

C [Unix or Linux or Cygwin gets true].
      yes = .true.
#ifdef MINGW
      yes = .false.
#endif
      return
      end

C ******************** usrhome ********************
C Find user's home directory.

      subroutine usrhome(upath)
      COMMON/OUTIN/IUOUT,IUIN,iEOUT
      character*(*) upath
      character homedrive*24,homepath*48,outs*124
      logical unixok

C Find if Unix or NT.
      call isunix(unixok)
      if(unixok)then
        call getenv('HOME',upath)
      else
        call getenv ('HOMEDRIVE',homedrive)
        write(outs,*) '  Homedrive - ',homedrive
        call edisp(IUOUT,outs)
        call getenv ('HOMEPATH',homepath)
        write(outs,*) '  Homepath - ',homepath
        call edisp(IUOUT,outs)
        if(homedrive(1:1).ne.' '.and.homepath(1:1).ne.' ')then
          write(upath,'(2a)') homedrive(1:lnblnk(homedrive)),
     &      homepath(1:lnblnk(homepath))
        else
          upath = 'c:/esru'
        endif
      endif
      return
      end

C ******************** usrname ********************
C Find user name.

      subroutine usrname(uname)
      character*(*) uname
      logical unixok

C Find if Unix or NT.
      call isunix(unixok)
      if(unixok)then
        call getenv('USER',uname)
      else
        uname = 'user'
      endif
      return
      end

C ******************** usrdir ********************
C Find current folder.

      subroutine usrdir(upwd)
      common/OUTIN/IUOUT,IUIN,IEOUT
      character*(*) upwd
      character local*124   ! getcwd needs a fixed char array
      integer i,jlen,kblnk

C Modern compilers GCC F90 provide subroutine.
      call getcwd(local,i)
      jlen= LEN(upwd)        ! defined length of upwd
      kblnk= lnblnk(local)   ! actual length of local

C Ensure that string buffer does not overflow.
      if(kblnk.lt.jlen)then
        write(upwd,'(a)') local(1:kblnk)
      else
        write(upwd,'(a)') local(1:jlen)
      endif

C Normal return value is zero, otherwise provide warning.
      if(i.ne.0)then
        call edisp(iuout,'usrdir: problem recovering current folder')
      endif
      return
      end

C ******************** clearfolderlist ********************
C Called from getfileslist_() in esru_util.c to clear the listfold
C common block.

      subroutine clearfolderlist()
#include "espriou.h"
      integer nboflistf  ! how many folders or file names passed
      integer listfoldertype   ! zero if unused one if folder two if file
      integer lenlistfolder    ! width of each string
      character*72 listfolder  ! array of folder or file names
      common/listfold/nboflistf,listfoldertype(MFFOLD),
     &                lenlistfolder(MFFOLD),listfolder(MFFOLD)

C Clear the list.
      nboflistf=0
      do 42 ilist=1,MFFOLD
        listfoldertype(ilist)=0
        listfolder(ilist)='  '
        lenlistfolder(ilist)=0
  42  continue
      return
      end

C ******************** addfolderlist ********************
C Called from getfileslist_() in esru_util.c to pass back a
C file or folder name in the string folderf into a fortran
C data structure for other code to use.

      subroutine addfolderlist(type,folderf)
#include "espriou.h"

      integer lnblnk  ! function definition
      
      integer nboflistf  ! how many folders or file names passed
      integer listfoldertype   ! zero if unused one if folder two if file
      integer lenlistfolder    ! width of each string
      character*72 listfolder  ! array of folder or file names
      common/listfold/nboflistf,listfoldertype(MFFOLD),
     &                lenlistfolder(MFFOLD),listfolder(MFFOLD)

C Passed variables.
      character folderf*(*)   ! file name null terminated within calling code
      character type*(*)      ! type is D for directory and L for file

C Local variables
      CHARACTER WORD*96
      integer iln
      
C Add to the list.
      nboflistf=nboflistf+1
      call c2fstr(folderf,WORD)
      iln=lnblnk(WORD)

      listfolder(nboflistf)=WORD(1:iln)
      lenlistfolder(nboflistf)=iln
      if(type(1:1).eq.'D')then
        listfoldertype(nboflistf)= 1
      elseif(type(1:1).eq.'L')then
        listfoldertype(nboflistf)= 2
      else
        listfoldertype(nboflistf)= 0
      endif
      return
      end

C ******************** esppid ********************
C Find current process number.

      subroutine esppid(ipid)
      integer getpid
      logical unixok

C Find if Unix or NT.
      call isunix(unixok)
      if(unixok)then
        ipid = getpid()
      else
        ipid = 999
      endif
      return
      end

C ******************** tstamp ********************
C Generates a timestamped message.
C act (char*1) = '-' Current time, = 'm' Message plus time
C  = '>' append message to journal

      subroutine tstamp(act,msg)
      common/OUTIN/IUOUT,IUIN,IEOUT
      common/journopt/journio,iuj,journcmd,jfile

      character*(*) msg
      character dstmp*24,act*1,outs*124,journcmd*20,jfile*72
      character outs248*248  ! to handle longer messages
      logical there

      call dstamp(dstmp)
      if(act.eq.'-')then
        write(outs,'(A,A)')' Current time is : ',dstmp
        call edisp(iuout,outs)
      elseif(act.eq.'m')then
        LN=max(1,lnblnk(msg))
        write(outs,'(3A)')msg(1:LN),' @ ',dstmp
        call edisp(iuout,outs)
      elseif(act.eq.'>')then

C If there is a journal file append the current message to it and close.
        LN=max(1,lnblnk(msg))
        write(outs248,'(3A)')msg(1:LN),' @ ',dstmp
        ltf=max(1,LNBLNK(jfile))
        if(journio.eq.1.and.
     &    (jfile(1:2).ne.'  '.and.jfile(1:4).ne.'UNKN'))then
          there=.false.
          INQUIRE(FILE=jfile(1:ltf),EXIST=there)
          if(there)then
            close(iuj)
            open(iuj,file=jfile(1:ltf),position='APPEND',
     &        status='UNKNOWN',err=1)
            LN=max(1,lnblnk(outs248))
            write(iuj,'(a)',iostat=ios,err=2)outs248(1:LN)
            close(iuj)
          endif
        endif
      endif

      return

   1  if(IOS.eq.2)then
        call edisp(iuout,
     &  'Permission issue while opening journal file, continuing.')
      else
        call edisp(iuout,'Error opening journal file, continuing.')
      endif
      return
   2  if(IOS.eq.2)then
        call edisp(iuout,
     &  'Permission issue while updating journal file, continuing.')
      else
        call edisp(iuout,'Error updating journal file, continuing.')
      endif
      close(iuj)
      return
      end

C ******************** ectime ********************
C Returns a fixed length character*24 based on the
C the passed value of ictime.

      character*24 FUNCTION ECTIME(ICTIME)
      integer ICTIME
#ifdef GCC4
      call CTIME(ICTIME,ECTIME)
#else
      character*24 CTIME
      ECTIME = CTIME(ICTIME)
#endif
      return
      end

C ******************** to_session ********************
C Opens and appends a timestamped message into the session log.
C It closes the session file before exiting.

      subroutine to_session(msg)
      common/OUTIN/IUOUT,IUIN,IEOUT

C Attributes of the session log file.
      logical ieopened     ! Has session file been started.
      integer iecount      ! Does it hold error messages.
      character iefile*72  ! The name of the session file.
      common/logs/ieopened,iecount,iefile

      character*(*) msg
      character dstmp*24,act*1,outs*124
      character outs248*248  ! to handle longer messages
      logical there

      call dstamp(dstmp)

C Append the current message to session log and close.
      LN=max(1,lnblnk(msg))
      write(outs248,'(3A)')msg(1:LN),' @ ',dstmp
      ltf=max(1,LNBLNK(iefile))
      if(ieopened.and.
     &  (iefile(1:2).ne.'  '.and.iefile(1:4).ne.'UNKN'))then
        there=.false.
        INQUIRE(FILE=iefile(1:ltf),EXIST=there)
        if(there)then
          close(IEOUT)
          open(IEOUT,file=iefile(1:ltf),position='APPEND',
     &      status='UNKNOWN',err=1)
          LN=max(1,lnblnk(outs248))
          write(IEOUT,'(a)',iostat=ios,err=2)outs248(1:LN)
          iecount=iecount+1
          close(IEOUT)
        endif
      endif

      return

   1  if(IOS.eq.2)then
        call edisp(iuout,
     &  'Permission issue while opening session file, continuing.')
      else
        call edisp(iuout,'Error opening session file, continuing.')
      endif
      return
   2  if(IOS.eq.2)then
        call edisp(iuout,
     &  'Permission issue while updating session file, continuing.')
      else
        call edisp(iuout,'Error updating session file, continuing.')
      endif
      close(iuj)
      return
      end

C ******************** dstamp ********************
C Get date stamp in the form: Fri Jan 23 09:34:31 1998.
C Used to isolate code from system details.

      subroutine dstamp(date_str)
      common/OUTIN/IUOUT,IUIN,IEOUT

      integer time
      integer lnblnk
      integer lns
      character date_str*24
      character ectime*24

C Unix date function. If date_str blank assign a timestamp.
      ictime=time()
      write(date_str,'(A)',iostat=ios,err=1) ectime(ictime)
      lns=lnblnk(date_str)
      if(lns.gt.20)then
        continue
      else
        date_str='Mon Feb 11 16:42:00 2013'
      endif
      return

C If an error detected assign a timestamp.
   1  if(IOS.eq.2)then
        call edisp(iuout,'dstamp: permission error getting time.')
        date_str='Mon Feb 11 16:42:00 2013'
      else
        call edisp(iuout,'dstamp: error getting time, using fixed date')
        date_str='Mon Feb 11 16:42:00 2013'
      endif
      return
      end

C ******************** comparedate ********************
C Compares two date strings (generated by call to dstamp).
C act is requested action '?' is ??
C dif is positive if datea is more current than dateb.

      subroutine comparedate(datea,dateb,act,idif)
      common/OUTIN/IUOUT,IUIN,IEOUT
      character datea*24,dateb*24,act*1,datetest*24
      character word*24

C Setup arrays of indices to hold the day of the week (1-7), the month
C (1-12), the day of the month (1-31), the hour of the day (1-24), the
C minute of the hour (1-60) and the year.
      dimension imon(2),idom(2),ihrod(2),imohr(2),iyears(2)
      dimension isohr(2)

C Note datea and dateb are expeced to be in the form of:
C Fri Oct 17 11:13:16 2003.
C Scan each date in turn.
      do 42 i=1,2
        if(i.eq.1)datetest=datea
        if(i.eq.2)datetest=dateb
        K=0
        CALL EGETW(datetest,K,WORD,'W','day of week',IFLAG)
        CALL EGETW(datetest,K,WORD,'W','month name',IFLAG)
        if(word(1:3).eq.'Jan')then
          imon(i)=1
        elseif(word(1:3).eq.'Feb')then
          imon(i)=2
        elseif(word(1:3).eq.'Mar')then
          imon(i)=3
        elseif(word(1:3).eq.'Apr')then
          imon(i)=4
        elseif(word(1:3).eq.'May')then
          imon(i)=5
        elseif(word(1:3).eq.'Jun')then
          imon(i)=6
        elseif(word(1:3).eq.'Jul')then
          imon(i)=7
        elseif(word(1:3).eq.'Aug')then
          imon(i)=8
        elseif(word(1:3).eq.'Sep')then
          imon(i)=9
        elseif(word(1:3).eq.'Oct')then
          imon(i)=10
        elseif(word(1:3).eq.'Nov')then
          imon(i)=11
        elseif(word(1:3).eq.'Dec')then
          imon(i)=12
        endif
        CALL EGETWI(datetest,K,idom(i),1,31,'W','day of month',IER)

C Parse time phrase in the form 11:13:16
        CALL EGETW(datetest,K,WORD,'W','time phrase',IFLAG)
        read(WORD(1:2),*,IOSTAT=IOS,ERR=1002)ihrod(i)
        read(WORD(4:5),*,IOSTAT=IOS,ERR=1003)imohr(i)
        read(WORD(7:8),*,IOSTAT=IOS,ERR=1004)isohr(i)
        CALL EGETWI(datetest,K,iyears(i),1900,2051,'-','year',IER)
  42  continue

C First compare the years. If the same compare the months.
      diffhours = 0.0
      if(iyears(1).eq.iyears(2))then
        if(imon(1).eq.imon(2))then

C Year and month are the same, compare the day of the month.
          if(idom(1).eq.idom(2))then

C Year, month, day-of-month are the same, compare the hour of the day.
            if(ihrod(1).eq.ihrod(2))then

C Compare the minute of the hour.
              if(imohr(1).eq.imohr(2))then

C Compare the second of the minute.
                if(isohr(1).eq.isohr(2))then
                  idif = 0
                elseif(isohr(1).gt.isohr(2))then
                  idif = 1
                elseif(isohr(1).lt.isohr(2))then
                  idif = -1
                endif
              elseif(imohr(1).gt.imohr(2))then
                idif = 1
              elseif(imohr(1).lt.imohr(2))then
                idif = -1
              endif
            elseif(ihrod(1).gt.ihrod(2))then
              idif = 1
            elseif(ihrod(1).lt.ihrod(2))then
              idif = -1
            endif
          elseif(idom(1).gt.idom(2))then
            idif = 1
          elseif(idom(1).lt.idom(2))then
            idif = -1
          endif
        elseif(imon(1).gt.imon(2))then
          idif = 1
        elseif(imon(1).lt.imon(2))then
          idif = -1
        endif
      elseif(iyears(1).gt.iyears(2))then
        idif = 1
      elseif(iyears(1).lt.iyears(2))then
        idif = -1
      endif
      return

 1002 if(IOS.eq.2)then
        call edisp(iuout,
     &    'comparedate: permission error extracting hour.')
      else
        call edisp(iuout,'comparedate: error extracting hour.')
      endif
      return
 1003 if(IOS.eq.2)then
        call edisp(iuout,
     &    'comparedate: permission error extracting minute.')
      else
        call edisp(iuout,'comparedate: error extracting minute.')
      endif
      return
 1004 if(IOS.eq.2)then
        call edisp(iuout,
     &    'comparedate: permission error extracting second.')
      else
        call edisp(iuout,'comparedate: error extracting second.')
      endif
      return
      end

C ******************** getsecs ********************
C Get computer clock seconds. Used to isolate fortran code from 
C system specifics; edit for machine type.

      subroutine getsecs(ictime)
      integer time

      ictime=time()
      return
      end

C ******************** runit ********************
C Execute a command string, in text (runs in a new xterm) or graphics
C mode.  Assumes that command string terminates in a '&' if the
C user wishes to run in background mode. For minGW xterm is not used.
C As runit is sometimes called with verbose command strings ensure
C the local buffer (tmp & ltmp) can handle 1K characters.
C WARNING - possible OS dependency:
C runit invokes the command. system() is a C function which
C works well on MSYS and Linux while execute_command_line is
C a gfortran facility which works on MSYS2. If runit is not 
C working then try switching bweteen these options.

C *** UNIX ONLY ***
C tg: '-' call the command as-is.
C     'text' or 'script' call the command in an xterm.
C     'silent' suppress stdout (redirect to /dev/null).
C     '[file_name]' redirect stdout to "[file_name].out" and
C                   redirect stderr to "[file_name].err"
C                   maximum 8 characters

      subroutine runit(cmd,tg)
#include "building.h"
#include "model.h"

      common/OUTIN/IUOUT,IUIN,IEOUT
      character*(*) cmd,tg
      character tmp*1200,ltmp*1200,tmode*8,message*248
      character dq*1
      integer system
      logical unixok
      integer i

      dq = char(34)  ! double quote

C Find if Unix or NT.
      call isunix(unixok)
      if(unixok)then
        LN=max(1,lnblnk(tg))
        LN=min0(LN,8)
        write (tmode,'(a)') tg(1:LN)
        if(tmode(1:4).eq.'text'.or.tmode(1:6).eq.'script')then  ! need a terminal to run in
          LN=max(1,lnblnk(cmd))
          if(found_xterm)then         ! if there is an xterm on computer
            write(tmp,'(a,a)') 'xterm -e ',cmd(1:LN)
          else                        ! use fallback
            call execute_command_line (cmd, exitstat=i)
          endif
C          i = system(tmp)
         call execute_command_line (tmp, exitstat=i)
        elseif(tmode(1:6).eq.'silent')then  ! diverts chatter to /dev/null
          LN=max(1,lnblnk(cmd))
          write(tmp,'(a,a)') cmd(1:LN),' > /dev/null'
C          i = system(tmp)
         call execute_command_line (tmp, exitstat=i)
        elseif(tmode(1:1).eq.'-')then  ! just make the system call
C          i = system(cmd)
          call execute_command_line (cmd, exitstat=i)
        else          
          LN2=max(1,lnblnk(cmd))       ! record command to a file
          write(tmp,'(6a)') cmd(1:LN2),' >',tmode(1:LN),'.out 2>',
     &      tmode(1:LN),'.err'
C          i = system(tmp)
         call execute_command_line (tmp, exitstat=i)
        endif
        if(i.gt.0.and.i.le.126)then
          LN=max(1,lnblnk(cmd))
          write(tmp,'(a,i3,2a)') 'status ',i,' ',cmd(1:LN)
          call edisp(ieout,'requested task returned an error state.')
          call edisp248(ieout,tmp,100)
        elseif(i.eq.127)then
          LN=max(1,lnblnk(cmd))
          write(tmp,'(a,i3,2a)') 'status ',i,' ',cmd(1:LN)
          call edisp248(ieout,tmp,100)
        endif
        return
      else

C Non-linux only uses tg if passed as 'bg'. It then
C attempts to mimic starting a background task.
        write (tmode,'(a)') tg(1:lnblnk(tg))
        LN=lnblnk(cmd)
        if (cmd(LN:LN).eq."&") then
          write (tmp,'(4a)') dq,'start ',cmd(1:LN-1),dq
C          write (cmd,'(a)') tmp(1:lnblnk(tmp))
          call edisp248(iuout,tmp,90)
        else
          if(tmode.eq.'bg')then  ! background
            write (tmp,'(7a)') dq,'start /B ',dq,dq,' ',
     &        cmd(1:LN),dq
C            write (cmd,'(a)') tmp(1:lnblnk(tmp))
          else
            write (tmp,'(3a)') dq,cmd(1:LN),dq
C            write (cmd,'(a)') tmp(1:lnblnk(tmp))
          endif
          call edisp248(iuout,tmp,90)
          call edisp(iuout,' ')
          call forceflush()
        endif

C        i = system(cmd)
        call execute_command_line (tmp, exitstat=i)
        if(i.gt.0.and.i.le.126)then
C         call edisp(iuout,' task terminated with an error condition')
          LN=lnblnk(cmd)
          write(ltmp,'(a,i3,2a)') 'status ',i,' ',cmd(1:LN)
          call edisp248(iuout,ltmp,90)
        elseif(i.eq.127)then
          call edisp(iuout,'task requested was unknown.')
          LN=lnblnk(cmd)
          write(ltmp,'(a,i3,2a)') 'status ',i,' ',cmd(1:LN)
          call edisp248(iuout,ltmp,90)
        endif

C An older method of asking OS to do some work.
C        call cissue(cmd,ifail)
        return
      endif
      end

C ******************** runitv ********************
C Execute a command string, in text (runs in a new xterm) or graphics
C mode with a status indicator which is returned to the calling code.
C  Assumes that command string terminates in a '&' if the
C user wishes to run in background mode. For minGW xterm is not used.
C As runit is sometimes called with verbose command strings ensure
C the local buffer (tmp & ltmp) can handle 1K characters.
C WARNING - possible OS dependency:
C runit invokes the command. system() is a C function which
C works well on MSYS and Linux while execute_command_line is
C a gfortran facility which works on MSYS2. If runit is not 
C working then try switching bweteen these options.

      subroutine runitv(cmd,tg,i)
      common/OUTIN/IUOUT,IUIN,IEOUT
      character*(*) cmd,tg
      integer i
      character tmp*1200,ltmp*1200,tmode*8,message*248
      character dq*1
      integer system
      logical unixok

      dq = char(34)  ! double quote

C Find if Unix or NT.
      call isunix(unixok)
      if(unixok)then
        LN=max(1,lnblnk(tg))
        LN=min0(LN,4)
        write (tmode,'(a)') tg(1:LN)
        if(tmode(1:4).eq.'text')then
          LN=max(1,lnblnk(cmd))
          write(tmp,'(a,a)') 'xterm -e ',cmd(1:LN)
C          i = system(tmp)
         call execute_command_line (tmp, exitstat=i)
        else
C          i = system(cmd)
          call execute_command_line (cmd, exitstat=i)
        endif
        if(i.gt.0.and.i.le.126)then
C          call edisp(iuout,' task terminated with an error condition')
          LN=max(1,lnblnk(cmd))
          write(tmp,'(a,i3,2a)') 'status ',i,' ',cmd(1:LN)
C          call edisp248(iuout,tmp,100)
        elseif(i.eq.127)then
          call edisp(iuout,' task requested was unknown.')
          LN=max(1,lnblnk(cmd))
          write(tmp,'(a,i3,2a)') 'status ',i,' ',cmd(1:LN)
          call edisp248(iuout,tmp,100)
        endif
        return
      else

C Non-linux only uses tg if passed as 'bg'. It then
C attempts to mimic starting a background task.
        write (tmode,'(a)') tg(1:lnblnk(tg))
        LN=lnblnk(cmd)
        if (cmd(LN:LN).eq."&") then
          write (tmp,'(4a)') dq,'start ',cmd(1:LN-1),dq
C          write (cmd,'(a)') tmp(1:lnblnk(tmp))
          call edisp248(iuout,tmp,90)
        else
          if(tmode.eq.'bg')then  ! background
            write (tmp,'(7a)') dq,'start /B ',dq,dq,' ',
     &        cmd(1:LN),dq
C            write (cmd,'(a)') tmp(1:lnblnk(tmp))
          else
            write (tmp,'(3a)') dq,cmd(1:LN),dq
C            write (cmd,'(a)') tmp(1:lnblnk(tmp))
          endif
          call edisp248(iuout,tmp,90)
          call edisp(iuout,' ')
          call forceflush()
        endif

C        i = system(cmd)
        call execute_command_line (tmp, exitstat=i)
        if(i.gt.0.and.i.le.126)then
C         call edisp(iuout,'task terminated with an error condition')
          LN=lnblnk(cmd)
          write(ltmp,'(a,i3,2a)') 'status ',i,' ',cmd(1:LN)
          call edisp248(iuout,ltmp,90)
        elseif(i.eq.127)then
          call edisp(iuout,'task requested was unknown.')
          LN=lnblnk(cmd)
          write(ltmp,'(a,i3,2a)') 'status ',i,' ',cmd(1:LN)
          call edisp248(iuout,ltmp,90)
        endif

C An even older method of asking OS to do some work.
C        call cissue(cmd,ifail)
        return
      endif
      end

C ******************** isinstalled ********************
C Returns status=.TRUE. if application 'name' found.
C Test if application [name] is in the user's path by
C issuing a 'which' command and piping the result to a
C temporary file, which is then read. If there are
C characters in the file /tmp/foundit then we assume
C the application exists. Only works for Linux & OSX.

      subroutine isinstalled(name,status)
      integer lnblnk  ! function definition
      common/FILEP/IFIL
      common/OUTIN/IUOUT,IUIN,IEOUT

C Parameters.
      character name*32
      logical status

C Local variables.
      logical unixok,XST
      character doit*124,hold32*32,hold32a*32,outs*124
      character tmode*8
      integer iunit,ltf,length
      integer is   ! status

      doit=' '; status=.false.

C Check if Unix-based or DOS based.
      call isunix(unixok)
      if(unixok)then
        IUNIT=IFIL+1
        IF(IUNIT.NE.0)CALL ERPFREE(IUNIT,ISTAT)
        tmode='-'
        hold32='/tmp/foundit'; ltf=lnblnk(hold32); length=0
        write(doit,'(3a)')'which ',name(1:lnblnk(name)),
     &    ' >/tmp/foundit'

C Use version of 'runit' that returns an error status and acts
C on it. If 'which' did not find anything the return will be non-zero.
        call runitv(doit,tmode,is)
        if(is.gt.0.and.is.le.126)then
          write(outs,*) 'Cannot find ',hold32a(1:lnblnk(hold32a)),' ',
     &      name(1:lnblnk(name))
          call edisp(iuout,' ')
          call edisp(iuout,outs)
          status=.false.
          CALL ERPFREE(IUNIT,ISTAT)
          return
        endif
        doit='ls -l /tmp/foundit >/dev/null'          
        call runit(doit,tmode)
        INQUIRE (FILE=hold32(1:ltf),EXIST=XST)
        IF (XST) THEN
          OPEN (IUNIT,FILE=hold32(1:ltf),ACCESS='SEQUENTIAL',
     &          STATUS='OLD',IOSTAT=ISTAT)
          CALL ERPFREE(IUNIT,ISTAT)  ! flush the buffer
          OPEN (IUNIT,FILE=hold32(1:ltf),ACCESS='SEQUENTIAL',
     &          STATUS='OLD',IOSTAT=ISTAT)
          READ(IUNIT,'(a)',IOSTAT=IOS)hold32a
          CALL ERPFREE(IUNIT,ISTAT)
          length=lnblnk(hold32a)
          if(length.gt.1)then
            status=.true.
          else
            write(outs,*) 'Cannot find ',hold32a(1:lnblnk(hold32a)),
     &        ' ',name(1:lnblnk(name))
            call edisp(iuout,' ')
            call edisp(iuout,outs)
            status=.false.
          endif
          CALL EFDELET(IUNIT,ISTAT) ! clean up the temporary file
          return
        else
          call usrmsg('Problem trying to open ',hold32,'W')
          status=.false.
          return
        endif
      endif
      end  ! of isinstalled

C ***************** iEGetArrW *****************
C Read all words in a 248 character string, and populate a character
C array with each word. Words are separated by blanks, commas, or tab: WORD,WORD,WORD
C or WORD WORD WORD or WORD, WORD, WORD  are all valid.

      integer function iEGetArrW(cString,cWORDS)
      implicit none

      character*248 cString, cWords(124)
      character*1 cChar
      logical bWord
      integer iStrLoc, iWordLoc, iEnd, iWordCount, iEr
      parameter (iEnd=248)

C Empty word array
      do iWordCount = 1, 124
        cWords(iWordCount) = ' '
      enddo

C Reset word counter
      iWordCount = 0

C Loop through string
      do iStrLoc = 1, iEnd

C Get current character
        cChar = cString(iStrLoc:iStrLoc)

C If string is a word separation character, move on
        if ( cChar .eq. ' ' .or.
     &       cChar .eq. ',' .or.
     &       iChar(cChar) .eq. 9 ) then

           bWord = .false.
        else

C is this a new word?
          if ( .not. bWord ) then

            bWord = .true.
            iWordCount = iWordCount + 1
            iWordLoc = 1

          endif

C Copy current character into word buffer
          cWords(iWordCount)(iWordLoc:iWordLoc) = cChar

          iWordLoc = iWordLoc + 1

        endif

      enddo

      iEGetArrW = iWordCount

      return
      end

C ***************** iEGetArrW_2500 *****************
C iRead all words in a 2500 character string, and populate a character
C array with each word. Words are separated by blanks, commas, or tab: WORD,WORD,WORD
C or WORD WORD WORD or WORD, WORD, WORD  are all valid.

      integer function iEGetArrW_2500(cString,cWORDS)
      implicit none

      character*2500 cString 
      character*248  cWords(124)
      character*1 cChar
      logical bWord
      integer iStrLoc, iWordLoc, iEnd, iWordCount, iEr
      parameter (iEnd=2500)

C Empty word array.
      do iWordCount = 1, 124
        cWords(iWordCount) = ' '
      enddo

C Reset word counter.
      iWordCount = 0

C Loop through string.
      do iStrLoc = 1, iEnd

C Get current character.
        cChar = cString(iStrLoc:iStrLoc)

C If string is a word separation character, move on.
        if ( cChar .eq. ' ' .or.
     &       cChar .eq. ',' .or.
     &       iChar(cChar) .eq. 9 ) then

           bWord = .false.
        else

C is this a new word?
          if ( .not. bWord ) then

            bWord = .true.
            iWordCount = iWordCount + 1
            iWordLoc = 1

          endif

C Copy current character into word buffer.
          cWords(iWordCount)(iWordLoc:iWordLoc) = cChar

          iWordLoc = iWordLoc + 1

        endif

      enddo

      iEGetArrW_2500 = iWordCount

      return
      end

C ******************** sitell2s ********************
C Take latitude (clat) and longitude difference (clong) and return
C a descriptive string (descr).

      subroutine sitell2s(clat,clong,descr)
      character descr*16

      IF(CLAT.LT.0.0)goto 11
      IF(CLONG.LT.0.0)goto 12
      WRITE(descr,'(F7.1,a,F7.1,a)')CLAT,'N',CLONG,'E'
      RETURN

   12 ACLONG=ABS(CLONG)
      WRITE(descr,'(F7.1,a,F7.1,a)')CLAT,'N',ACLONG,'W'
      RETURN

   11 ACLAT=ABS(CLAT)
      IF(CLONG.LT.0.0)goto 13
      WRITE(descr,'(F7.1,a,F7.1,a)')ACLAT,'S',CLONG,'E'
      RETURN

   13 ACLONG=ABS(CLONG)
      WRITE(descr,'(F7.1,a,F7.1,a)')ACLAT,'S',ACLONG,'W'
      RETURN

      end

C ***************** SIGFIG *****************
C Returns number to required significant figures.
C R - real number
C NSIG - number of significant figureds required.
C RNO - real number to NSIG figures
C STR - string version of RNO
C LSTR - length of STR

       subroutine sigfig(R,NSIG,RNO,STR,LSTR)

       character*12 STR, TMP, FMT

       STR='  '

C Need to check if number is greater or less than one (i.e. should
C zeros be counted.
       if (abs(R).gt.1.) then

C Compare number against required number if sig figs.
         if (abs(R).gt.(10.**NSIG)) then

C Need to turn last NZD digits to zeros.
           NZD=0
 100       if (abs(R)/(10.**NZD).gt.(10.**NSIG)) then
             NZD=NZD+1
             goto 100
           endif
           RNO=real(NINT(R/10.**NZD)*10.**NZD)
           if (NZD.gt.6.or.RNO.gt.1.0e12) then
             write (FMT,'(a,i2,a)') '(g12.',NSIG,')'
           else
             write (FMT,'(a)') '(f12.0)'
           endif
           write (STR,FMT) RNO
         else

C Need to preserve NDP decimal places.
           NDP=0
 110       if (abs(R)*10.**NDP.lt.(10.**NSIG)) then
             NDP=NDP+1
             goto 110
           endif
           NDP=NDP-1
           RNO=real(NINT(R*10.**NDP))/10.**NDP
           if(NDP.lt.0) NDP=0
           write (FMT,'(a,i2,a)') '(f12.',NDP,')'
           write (STR,FMT) RNO
         endif
       else

C Need to preserve NSIG decimal places.
         RNO=real(NINT(R*10.**NSIG))/10.**NSIG
         write (FMT,'(a,i2,a)') '(f12.',NSIG,')'
         write (STR,FMT) RNO
       endif

C Strip leading blanks from text string.
       K=1
       call EGETRM(STR,K,TMP,'-','generating sig fig string ',IER)
       write (STR,'(a)') TMP
       LSTR=max(1,lnblnk(STR))

       return
       end

C ***************** SIpre *****************
C Returns suitable SI prefix for number supplied. It is
C assumed that the value is supplied in the standard SI unit,
C for example, flux in W not kW.
C
C R - real number
C NSIG - number of significant figureds required.
C RNO - real number to NSIG figures, including prefix
C STR - string version of RNO
C LSTR - length of STR
C PRE - prefix
C SYM - symbol

       subroutine SIpre(R,NSIG,RNO,STR,LSTR,PRE,SYM)
       character*12 STR
       character SYM*1, SYMH(6)*1, SYML(6)*1
       character PRE*5, PREH(6)*5, PREL(6)*5

       data SYMH/'k','M','G','T','P','E'/
       data PREH/'kilo ','mega ','giga ','tera ','peta ','exa  '/
       data SYML/'m','u','n','p','f','a'/
       data PREL/'milli','micro','nano ','pico ','femto','atto '/

C Set default output.
       RNO=R
       STR=' '
       SYM=' '
       PRE=' '

C Need to check if big or small number.
       if (R.gt.1.) then
         IP=0
 100     if (R.gt.(1000.**IP)) then
           IP=IP+1
           goto 100
         endif
         IP=IP-1
         if (IP.gt.0.and.IP.lt.7) then
           RX=R/(1000.**IP)
           call sigfig(RX,NSIG,RNO,STR,LSTR)
           PRE=PREH(IP)
           SYM=SYMH(IP)
         else
           call sigfig(R,NSIG,RNO,STR,LSTR)
         endif
       else
         IP=0
 110     if (R.lt.(0.001**IP)) then
           IP=IP+1
           goto 110
         endif
         if (IP.gt.0.and.IP.lt.7) then
           RX=R*(1000.**IP)
           call sigfig(RX,NSIG,RNO,STR,LSTR)
           PRE=PREL(IP)
           SYM=SYML(IP)
         else
           call sigfig(R,NSIG,RNO,STR,LSTR)
         endif

       endif

       return
       end

C ***************** PRONAM *****************
C Returns the characters of a string after the last
C occurance of '/' or '\'.

      SUBROUTINE PRONAM(longstr,last)
      CHARACTER*(*) LONGSTR,LAST

      ILEN=LNBLNK(LONGSTR)
      DO 90 I=1,ILEN
        IF(LONGSTR(I:I).EQ.'/'.OR.LONGSTR(I:I).EQ.'\\')LOCUR=I
 90   CONTINUE
      LAST=LONGSTR(LOCUR:ILEN)
      RETURN
      END

C ***************** DNOTZERO *****************
C Returns a non-zero value with the same sign.

      DOUBLE PRECISION FUNCTION DNOTZERO(A)

      double precision A,small
      SMALL=1E-35

      if (abs(A).lt.(SMALL)) then

C Value is approx zero, return small with correct sign.
        DNOTZERO=SIGN(SMALL,A)

      else
        DNOTZERO=A
      endif

      RETURN
      END

C ***************** ANOTZERO *****************
C Returns a non-zero value with the same sign.

      REAL FUNCTION ANOTZERO(A)

      SMALL=1E-10

      if (abs(A).lt.(SMALL)) then

C Value is approx zero, return small with correct sign.
        ANOTZERO=SIGN(SMALL,A)

      else
        ANOTZERO=A
      endif

      RETURN
      END

C ***************** ASKTIM *****************
C Ask for month, day and time. Returns IMO (month), IDO (day
C of month), IJDAY (day of year for output), TIME (real representation),
C and IT (timestep). IFDAY is a toggle provided in setres.f to control
C the display and input of periods - 0 = year day, 1 or 2 = day of month.

      SUBROUTINE ASKTIM(IFDAY,NTS,IMO,IDO,IJDAY,TIME,IT,IER)
#include "help.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT

      character HOLD*24
      DIMENSION ID(12)
      DATA ID/31,28,31,30,31,30,31,31,30,31,30,31/

      IER=0
      H(1)='The output time defines the day, month and decimal'
      H(2)='hour at which the output is requested. This last'
      H(3)='value must be no earlier than 0.00 hours or later'
      H(4)='than 24.00 hours.'
      H(5)=' '
      IF(IFDAY.EQ.0)THEN
        CALL EDAY(IDO,IMO,IJDAY)
        write(HOLD,'(I6,F6.1)') IJDAY,TIME
        H(6)='Example: 6th March at 09h30 is returned as 64  9.5'
      else
        write(HOLD,'(I6,I4,F6.1)') IDO,IMO,TIME
        H(6)='Example: 6th March at 09h30 is given as 6 3 9.5'
      endif

  281 IF(IFDAY.EQ.0)THEN
        CALL EASKS(HOLD,' ','Day-of-year & time?',
     &     24,' 1  7.0 ','doy and time',IER,6)
        K=0
        CALL EGETWI(HOLD,K,IJDAY,1,365,'F','day of year',IER)
        CALL EGETWR(HOLD,K,TIME,0.0,24.0,'F','time',IER)
        if(IER.ne.0)goto 281
        CALL EDAYR(IJDAY,IDO,IMO)
      ELSE
 283    CALL EASKS(HOLD,' ','Day-of-month, month & time?',
     &     24,' 1  1  7.0 ','doy, month time',IER,6)
        K=0
        CALL EGETWI(HOLD,K,IDO,1,31,'F','day of month',IER)
        CALL EGETWI(HOLD,K,IMO,1,12,'F','month',IER)
        CALL EGETWR(HOLD,K,TIME,0.0,24.0,'F','time',IER)
        if(IER.ne.0)goto 283
        CALL EDAY(IDO,IMO,IJDAY)
      ENDIF

C Check range.
      IF(IDO.GT.ID(IMO))THEN
        call edisp(iuout,'Day past end of month!. Reenter.')
        goto 281
      ENDIF
      CALL EDAY(IDO,IMO,IJDAY)

C Convert time to time-step number.
      XX=TIME+(1.0/(FLOAT(NTS)*2.0))
      IT=INT(XX)*NTS
      IF(IT.EQ.0)IT=1

      RETURN
      END

c ******************** SOLAIR ********************
c Compute sol-air temperature.

      FUNCTION SOLAIR(T,QF,QD)

C Temporary measure.
      SOLAIR=T
      RETURN
      END

C ******************** LISTAS ********************
C General read of an ascii file.

      SUBROUTINE LISTAS(iunit,LFIL,IER)
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      character*(*) LFIL
      character outstr*124
      logical :: XST
      logical :: unixok
      integer ltf   ! position of last character in the string.

C List out an ASCII file, if file name begins with "/" open
C it directly.
      call isunix(unixok)
      if(LFIL(1:1).eq.'/')then
        ltf=max(1,LNBLNK(LFIL))
        INQUIRE (FILE=LFIL(1:ltf),EXIST=XST)
        IF (XST) THEN
          LN=max(1,lnblnk(LFIL))
          OPEN (IUNIT,FILE=LFIL(1:LN),ACCESS='SEQUENTIAL',
     &              STATUS='OLD',IOSTAT=ISTAT)
        else
          call usrmsg('LISTAS: problem opening file!',LFIL,'W')
          return
        endif
      else
        if(.NOT.unixok)then   ! For W10 use FPOPEN
          CALL ERPFREE(IUNIT,ISTAT)
          call FPOPEN(IUNIT,ISTAT,1,0,LFIL)
          if(ISTAT.ge.0)XST=.true.
          if(ISTAT.eq.-301)XST=.false.
        else
          CALL EFOPSEQ(IUNIT,LFIL,1,IER)
        endif
      endif
      IF(IER.LT.0)THEN
        call usrmsg(' Problem detected while trying to open',LFIL,'W')
        return
      ENDIF

C Read ASCII file.
    7 READ(IUNIT,10,IOSTAT=IOS,END=102)OUTSTR
   10 FORMAT(A124)
      CALL  EDISP(IUOUT,OUTSTR)
      goto 7

C End of file encountered, close the file and return to menu.
  102 if(IOS.eq.2)then
        CALL USRMSG(' ','LISTAS: end of file problem!','-')
      else
        CALL USRMSG(' ','LISTAS: end of file reached!','-')
      endif
      CALL ERPFREE(IUNIT,ISTAT)
      return
      end

C ******************** SDELIM ********************
C Replaces blanks in a string A with alternative delimiter and
C returns in B. If the last character in the string is the alternative
C delimiter then it is replaceed with a blank.

      SUBROUTINE SDELIM(A,B,delm,IW)
      CHARACTER*(*) A,B
      CHARACTER C*1,CL*1,delm*1

C Depending on the replacement separator, convert all existing
C ' ', ',' and 'tabs' to the new separator. Loop through filled
C part of string A plus one character.
      LS=LEN(A)
      LSN=max(1,LNBLNK(A))
      lsmn=MIN0(LS,LSN+1)
      B=' '
      K=0
      DO 99 I=1,lsmn
        C=A(I:I)
        if(K.eq.0)then

C Check for initial blanks, commas or tabs.
          if(C.eq.' '.or.C.EQ.','.OR.ICHAR(C).EQ.9)then
            goto 99
          else
            K=K+1
            B(K:K)=C
            CL=C
          endif
        else

C If a separator found, convert it unless the previous character
C was a separator in which case it can be skipped.
          if(C.eq.' '.or.C.EQ.','.OR.ICHAR(C).EQ.9)then
            if(CL.eq.' '.or.CL.EQ.','.OR.ICHAR(CL).EQ.9)then
              goto 99
            else
              K=K+1
              if(delm.eq.'T')then
                B(K:K)=CHAR(9)
              elseif(delm.eq.'S')then
                B(K:K)=' '
              elseif(delm.eq.'C')then
                B(K:K)=','
              elseif(delm.eq.'N')then
                K=K-1
              endif
            endif
            CL=C
            goto 99
          else
            K=K+1
            B(K:K)=A(I:I)
            CL=A(I:I)
          endif
        endif
 99   CONTINUE

C If the last character is a separator replace it with a blank
C (so trailing commas and tabs are not written out).
      lastb=max(1,lnblnk(b))
      C=B(lastb:lastb)
      if(delm.eq.'T'.and.ICHAR(C).EQ.9)then
        B(lastb:lastb)=' '
      elseif(delm.eq.'S')then
        continue
      elseif(delm.eq.'C'.and.C.eq.',')then
        B(lastb:lastb)=' '
      endif

      END

C ********************* EDDISP *********************
C Generic routine which displays lines of text passed to it
C in a format depending on the terminal type and the currently set
C delimiter.

      SUBROUTINE EDDISP(ITRU,MSG)
      common/exporttg/xfile,tg,delim
      COMMON/EXPORTI/ixopen,ixunit,ixpunit
      CHARACTER*(*) MSG
      CHARACTER WWMSG*124
      character xfile*144,tg*1,delim*1,dg*1

C If delimiter set to alternative then process text before edisp call.
      if(delim.eq.'-')then
        call edisp(itru,MSG)
      else

C If using X delimeter (tagged data) then set the delimeter to a comma.
C Tagging is handled when the initial string is created.
        dg=delim
        if (delim.eq.'X') dg='C'
        call SDELIM(MSG,WWMSG,dg,IW)
        call edisp(itru,WWMSG)
      endif
      return
      END

C ********************* EDDISP248 *********************
C EGeneric routine which displays long lines of text passed
C to it in a format depending on the terminal type and the currently set
C delimiter.

      SUBROUTINE EDDISP248(ITRU,MSG,iwid)
      common/exporttg/xfile,tg,delim
      COMMON/EXPORTI/ixopen,ixunit,ixpunit
      CHARACTER*(*) MSG
      CHARACTER WWMSG*248
      character xfile*144,tg*1,delim*1,dg*1
      integer iwid

C If delimiter set to alternative then process text before edisp call.
      if(delim.eq.'-')then
        call edisp248(itru,MSG,iwid)
      else

C If using X delimeter (tagged data) then set the delimeter to a comma.
C Tagging is handled when the initial string is created.
        dg=delim
        if (delim.eq.'X') dg='C'
        call SDELIM(MSG,WWMSG,dg,IW)
        call edisp248(itru,WWMSG,iwid)
      endif
      return
      END

C ********************* EDISP248 *********************
C Displays a 248 char block of text passed to it
C in a format depending on the terminal type. If it will not
C fit on one line, subsequent lines are used and breaks are
C set based on nearest width to iwid.

      SUBROUTINE EDISP248(ITRU,MSG,iwid)
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      CHARACTER*(*) MSG
      CHARACTER outs*144      ! the text buffer for printing one line
      character mstr*72
      integer lenmsg,lenalts  ! length of the passed msg

C If width for word-warp is not reasonable, return.
      if(iwid.lt.24.or.iwid.gt.144)then
        write(mstr,'(a,i3,a)') 'edisp248 word warp width ',iwid,
     &    ' is overly short or long.' 
        call edisp(iuout,mstr)
        return
      endif

C Remember length of MSG.
      lenmsg=0
      lenmsg=max0(1,lnblnk(MSG))

C Find blanks near iwid points along the text block. It is
C assumed that iwid is ~72, but can be 48-144. In the case of
C long words burried in the block of text we might get cases
C where ipb is the same as ipa. Use call to inextblnk to find
C the next blank after ipos. If the next blank is farther along
C the string than we would normally warp to this indicates we
C have a long phrase. If is is before we would normally warp
C then it might be because we have reaced the end of the string.
      iwidhalf=iwid/2 
      ipa=iprevblnk(MSG,iwid)     ! find blank before iwid
      ipanext=inextblnk(MSG,iwid) ! find blank after iwid
      ipb=iprevblnk(MSG,ipa+iwid) ! find blank nearest next iwid point
      if(ipa.eq.ipb)then
        if(ipanext.ge.ipa+iwid)then
          ipb=ipanext             ! found a long phrase
        elseif(ipanext.eq.lenmsg)then
          ipb=lenmsg              ! at end of MSG
        else
          ipb=iprevblnk(MSG,ipa+iwid+iwidhalf)
          if(ipa.eq.ipb)then

C If ipb is still the same test from double the iwid.
C This should catch phrases that are approximately
C double the width of iwid (like file names with explicit paths).
            ipb=iprevblnk(MSG,ipa+iwid+iwid)
          endif
        endif
      endif 

C Repeat logic for potential third line. First ensure we
C are not testing beyond the length of the string.
      if(ipb+iwid.ge.lenmsg)then
        ipc=lenmsg
      else
        ipc=iprevblnk(MSG,ipb+iwid)
        ipcnext=inextblnk(MSG,ipb+iwid)
        if(ipb.eq.ipc)then
          if(ipcnext.ge.ipb+iwid)then
            ipc=ipcnext
          elseif(ipcnext.eq.lenmsg)then
            ipc=lenmsg
          else
            ipc=iprevblnk(MSG,ipb+iwid+iwidhalf)
            if(ipb.eq.ipc)then
              ipc=iprevblnk(MSG,ipb+iwid+iwid)
            endif
          endif
        endif
      endif

C Repeat logic for potential fourth line. First ensure we
C are not testing beyond the length of the string.
      if(ipc+iwid.ge.lenmsg)then
        ipd=lenmsg
      else
        ipd=iprevblnk(MSG,ipc+iwid)
        ipdnext=inextblnk(MSG,ipc+iwid)
        if(ipc.eq.ipd)then
          if(ipdnext.ge.ipc+iwid)then
            ipd=ipdnext
          elseif(ipdnext.eq.lenmsg)then
            ipd=lenmsg
          else
            ipd=iprevblnk(MSG,ipc+iwid+iwidhalf)
            if(ipc.eq.ipd)then
              ipd=iprevblnk(MSG,ipc+iwid+iwid)
            endif
          endif
        endif
      endif

C Repeat logic for potential fifth line. First ensure we
C are not testing beyond the length of the string.
      if(ipd+iwid.ge.lenmsg)then
        ipe=lenmsg
      else
        ipe=iprevblnk(MSG,ipd+iwid)
        ipenext=inextblnk(MSG,ipd+iwid)
        if(ipd.eq.ipe)then
          if(ipenext.ge.ipc+iwid)then
            ipe=ipenext
          elseif(ipenext.eq.lenmsg)then
            ipe=lenmsg
          else
            ipe=iprevblnk(MSG,ipd+iwid+iwidhalf)
            if(ipd.eq.ipe)then
              ipe=iprevblnk(MSG,ipd+iwid+iwid)
            endif
          endif
        endif
      endif

C And the 6th line.
      if(ipe+iwid.ge.lenmsg)then
        ipf=lenmsg
      else
        ipf=iprevblnk(MSG,ipe+iwid)
      endif

C Debug.
      lenalts=0
      lenalts=max0(lenmsg,ipa,ipb,ipc,ipd,ipe,ipf)

C Debug.
C      write(6,*) 'blanks @ ',ipa,ipb,ipc,ipd,ipe,ipf,
C     &  'lenmsg lenalts ',lenmsg,lenalts
C      write(6,*) 'n blanks @ ',ipanext,ipcnext,ipdnext,ipenext

C Process first block of text, and if subsequent blocks are
C non-blank, do them as well. If iwid is less than 72 there
C will be more lines.
      outs=' '
      write(outs,'(a)') MSG(1:ipa)
      call edisp(itru,outs)
      if(ipb.gt.ipa)then

C Trap phrases longer than the outs buffer.
        if(ipb-(ipa+1).gt.143)then
          ipbb = ipa+143
        else
          ipbb = ipb
        endif
        outs=' '
        write(outs,'(a)') MSG(ipa+1:ipbb)
        call edisp(itru,outs)
      endif
      if(ipc.gt.ipb)then
        if(ipc-(ipb+1).gt.143)then
          ipcc = ipb+143
        else
          ipcc = ipc
        endif
        outs=' '
        write(outs,'(a)') MSG(ipb+1:ipcc)
        call edisp(itru,outs)
      endif
      if(ipd.gt.ipc)then
        if(ipd-(ipc+1).gt.143)then
          ipdd = ipc+143
        else
          ipdd = ipd
        endif
        outs=' '
        write(outs,'(a)') MSG(ipc+1:ipdd)
        call edisp(itru,outs)
      endif
      if(ipe.gt.ipd)then
        if(ipe-(ipd+1).gt.143)then
          ipee = ipd+143
        else
          ipee = ipe
        endif
        outs=' '
        write(outs,'(a)') MSG(ipd+1:ipee)
        call edisp(itru,outs)
      endif
      if(ipf.gt.ipe)then
        outs=' '
        write(outs,'(a)') MSG(ipe+1:ipf)
        call edisp(itru,outs)
      endif

      RETURN
      END

C ********************* clrtextbuf *********************
C Clears the text buffer common blocks.

      subroutine clrtextbuf()
      common/textbuf/dispbuf(500)
      common/textbufl/indexbuf,lnbuf(500)
      character dispbuf*144

      indexbuf=0
      do 42 i=1,500
        dispbuf(i)='  '
        lnbuf(i)=1
  42  continue
      return
      end

C ********************* UPDVIEW *********************
C Called from C code in esru_x.c or esp-r.c with values to 
C update commonblocks GFONT, VIEWPX and SPAD. Set REFRESH=true.
C Set MODIFYVIEW=true to force any wire-frame images to be redrawn.

      subroutine updview(ifsc,itfsc,imfsc,ilc,irc,itc,ibc,iwc,ihc,lttyc)
#include "building.h"
#include "prj3dv.h"

      integer ifs,itfs,imfs
      COMMON/GFONT/IFS,ITFS,IMFS
      integer menuchw,igl,igr,igt,igb,igw,igwh
      COMMON/VIEWPX/menuchw,igl,igr,igt,igb,igw,igwh
      COMMON/SPAD/MMOD,LIMIT,LIMTTY
      COMMON/FRESH/refrsh
      logical refrsh

      IFS=ifsc
      ITFS=itfsc
      IMFS=imfsc
      igl=ilc
      igr=irc
      igt=itc
      igb=ibc
      igw=iwc
      igwh=ihc
      LIMTTY=lttyc
      LIMIT=1
      refrsh=.TRUE.
      MODIFYVIEW=.TRUE.

      return
      end

C ********************* WIREPK *********************
C Called from C code in esru_x.c with current
C number of zones which have been selected for display.
C Compilation of X11 assumes this will be available in
C each modules code somewhere.  If not needed then provide a dummy.

      subroutine wirepk(inpk)
#include "building.h"

C izgfoc is the graphic focus zone, nzg is the number of zones selected
C and nznog array of selected zone indices.
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/gzonpik/izgfoc,nzg,nznog(mcom)
      if(inpk.ne.nzg)then
        call edisp(iuout,'Mismatch between inpk and nzg.')
      endif
      CALL EVSET('B',IER)
      if(ier.ne.0)then
        call edisp(iuout,'Problem after image button pick.')
      endif

      return
      end

C ******************** EPROMPT ********************
C Does nothing, for compatibility only.

      SUBROUTINE EPROMPT
      RETURN
      END


C ********************* iCountWords ********************
C Checks a character string (A), returning the number of
C data items (IW) separated by ' ' tab or ','.  To prevent 
C overwriting the passed string , deal with a copy.

      integer function iCountWords(cString)
      implicit none
      character*(*) cString
      character*1   cChar
      integer iLastChar, iCurChar
      logical bLastCharWasSpace
      integer lnblnk
      integer iChar

C Length of string.     
      iLastChar = lnblnk(cString)

C Search for words.
      bLastCharWasSpace = .true.
      iCountWords       = 0

      SearchForWords: do iCurChar=1, iLastChar

        cChar = cString(iCurChar:iCurChar)

        IsNonBlank: if ( cChar /= ' '      .and.
     &                   cChar /= ','      .and.
     &                   iChar(cChar) /= 9        ) then

C If character is non-blank and last character
C was blank, comma or tab, increment word count.
          IsNewWord: if ( bLastCharWasSpace ) then
            iCountWords = iCountWords + 1
          endif IsNewWord
          bLastCharWasSpace = .false.

        else
          bLastCharWasSpace = .true. 
        endif IsNonBlank
      enddo SearchForWords

      return
      end function iCountWords

C ******************** CHITMS, LCHITMS, CHITMS400 ********************
C Legacy interfaces to iCountWords. They return
C the number of words (IW) for a string A.

      subroutine CHITMS(a,iw)
      implicit none
      integer iw
      character*(*) a
      integer iCountWords

      iw = iCountWords(a)

      return
      end

      subroutine LCHITMS(a,iw)
      implicit none
      integer iw
      character*(*) a
      integer iCountWords

      iw = iCountWords(a)

      return
      end

      subroutine CHITMS400(a,iw)
      implicit none
      integer iw
      character*(*) a
      integer iCountWords

      iw = iCountWords(a)

      return
      end
      
C ********************* bStringsMatch *********************
C Function that checks if the non-blank portion of two
C two strings of arbitrary length match. 

      logical function bStringsMatch(cStringA, cStringB)
      implicit none 
      character*(*) cStringA
      character*(*) cStringB
      
      integer lnblnk
      integer iLengthA, iLengthB
      
      iLengthA = lnblnk(cStringA)
      iLengthB = lnblnk(cStringB)

      if ( iLengthA /= iLengthB ) then 
        bStringsMatch = .false. 
      elseif ( cStringA(1:iLengthA) /= cStringB(1:iLengthB) ) then 
        bStringsMatch = .false.
      else
        bStringsMatch = .true. 
      endif  

      return 
      end function bStringsMatch

C ********************* iCtoI *********************
C Convert a string into an integer.
    
      integer function  iCtoI ( cBuffer, bError )
      implicit none 
      character*(*) cBuffer
      logical bError 
      integer iMiscError

      read (cBuffer,*, IOSTAT=iMiscError) iCtoI
      if ( iMiscError /= 0 ) then
        bError = .true. 
      else
        bError = .false. 
      endif 
      return 
      end 

C ********************* fCtoI_err *********************
C Interface to iCtoI with an in-built warning message.
   
      integer function iCtoI_err ( cBuffer, cContext, bError )
      implicit none

      common/outin/iuout,iuin,ieout
      integer iuout,iuin,ieout    
      integer lnblnk
           
      character*(*) cBuffer, cContext
      character*124 cMsg
      character*80 cTemp
      logical bError 

      integer iCtoI
      
C Convert string.      
      iCtoI_err = iCtoI ( cBuffer, bError )

C Error message.       
      if ( bError ) then
        write (cTemp, '(A)' ) cContext(1:min( 80, lnblnk( cContext ) ) ) 
        write (cMsg,'(A,A,A,A,A)')
     &       'Error converting ',
     &       cTemp(1:lnblnk(cTemp)),
     &       ' (',
     &       cBuffer(1:lnblnk(cBuffer)),
     &       ') to integer.'
        call edisp248(iUout, cMsg,80)

      endif
      return
      end 
      
C ********************* fCtoR *********************
C Convert a string into a float.
      
      real function  fCtoR ( cBuffer, bError )
      implicit none 
      character*(*) cBuffer
      logical bError 
      integer iMiscError 
      
      read (cBuffer,*, IOSTAT=iMiscError) fCtoR
      
      if ( iMiscError /= 0 ) then
        bError = .true. 
      else
        bError = .false. 
      endif 
      return 
      end

C ********************* fCtoR_err *********************
C Interface to fCtoR with a built-in warning message
   
      real function fCtoR_err ( cBuffer, cContext, bError )
      implicit none

      common/outin/iuout,iuin,ieout
      integer iUout,iuin,ieout      ! channels for writing messages to screen
      integer lnblnk
           
      character*(*) cBuffer, cContext
      character*248 cMsg
      character*80 cTemp
      logical bError 

      real fCtoR
      
C Convert string.     
      fCtoR_err = fCtoR ( cBuffer, bError )

C Error report.    
      if ( bError ) then
        write (cTemp, '(A)' ) cContext(1:min( 80, lnblnk( cContext ) ) )    
        write (cMsg,'(A,A,A,A,A)')
     &       'Error converting ',
     &       cTemp(1:lnblnk(cTemp)),
     &       ' (',
     &       cBuffer(1:lnblnk(cBuffer)),
     &       ') to real.'
        call edisp248(iUout, cMsg,80)

      endif
      return
      end

C ********************* ISEOF *********************
C Called with an ISTAT value from a READ statement
C performed on a binary (random access) file, it returns a logical that
C signals whether the EOF has been reached. This is present because
C ISTAT codes are compiler-dependent, so it is convenient to centralise
C this check in case of future changes. This function could do with
C improvement in future, perhaps making use of #ifdef statements on a
C set of variables that define which compiler has been used.

      LOGICAL FUNCTION ISEOF(ISTAT)

      if ( ISTAT.LT.0 .or. ISTAT.EQ.5002) then    ! EOF or non-existing record number
        ISEOF=.TRUE.
      else
        ISEOF=.FALSE.
      endif

      RETURN
      END
