C This file is part of the ESP-r system.
C Copyright Energy Systems Research Unit, University of
C Strathclyde, Glasgow Scotland, 2001.

C ESP-r is free software.  You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation
C (version 2 orlater).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.

C You should have received a copy of the GNU General Public
C License along with ESP-r. If not, write to the Free
C Software Foundation, Inc., 59 Temple Place, Suite 330,
C Boston, MA 02111-1307 USA.

C This file contains the following routines:
C      MZSDAT
C      CMP01S   1 node (ISV=21) AC  converging 2-leg junction
C      CMP02S   1 node (ISV=21) AC  humidifier adiabatic
C      CMP03S   1 node (ISV=21) AC  fan
C      CMP04S   1 node (ISV=21) AC  cooling coil & flux control
C      CMP05S   1 node (ISV=21) AC  heating coil & flux control
C      CMP06S   1 node (ISV=21) AC  duct & condensation ?
C      CMP07S   1 node (ISV=21) AC  damper & flow ratio control
C      CMP08S   1 node (ISV=21) AC  flow diversion damper
C      CMP09S   1 node (ISV=21) AC  counterflow cooling coil &
C                                   water mass flow rate control
C                                   (Q calculated by TRNSYS type32)
C      CMP10S   1 node (ISV=21) AC  counterflow cooling coil &
C                                   water mass flow rate control
C      CMP11S   1 node (ISV=21) AC  counterflow heating coil &
C                                   water mass flow rate control
C      CMP12S   2 node (ISV=21) AC  plate heat exchanger
C      CMP14S   2 node (ISV=21) Cooling tower model
C      CMP15S   1 node (ISV=21) AC  fan mass flow rate control
C      CMP20S   1 node (ISV=20) WCH boiler
C      CMP21S   2 node (ISV=20) WCH radiator
C      CMP22S   1 node (ISV=20) WCH pipe
C      CMP23S   1 node (ISV=20) WCH converging 2-leg junction
C      CMP24S   1 node (ISV=20) WCH pump
C      CMP25S   2 node (ISV=20) WCH boiler with on/off control
C      CMP26S   2 node (ISV=20) WCH boiler with aquastat control
C      CMP27S   8 node (ISV=20) WCH radiator
C      CMP28S   1 node (ISV=29) Oil-filled electric panel radiator
C      CMP29S   1 node (ISV=20) WCH flow control valve
C      CMP30S   1 node (ISV=20) Water cooler with flux control.
C      CMP31S   2 node (ISV=20) WCH calorifier.
C      CMP32S   2 node (ISV=20) WCH generic liquid/liquid heat exchanger
C      CMP33S   2 node (ISV=20) WCH generic gas/liquid heat exchanger
C      CMP34S   3 node (ISV>19) WCH heated water storage
C      CMP35S   2 node (ISV>19) WCH basic radiator
C      CMP36S   2 node (ISV>19) WCH reversible heat pump
C      CMP37S   1 node (ISV>19) WCH air cooled or evaporative heat 
C                                   rejection device (chiller)
C      CMP38S   1 node (ISV=21) WCH converging multi-leg junction 
C      CMP39S   1 node (ISV>19) WCH pump mass flow rate control
C      CMP40S   3 node (ISV>19) AC  counterflow cooling coil fed
C                                   by WCH system
C      CMP41S   3 node (ISV>19) AC  counterflow heating coil fed
C      CMP42S   2 node (ISV=20) WCH modulating boiler
C      CMP43S   2 node (ISV=20) AC  cooling coil fed
C                                   by WCH system
C      CMP44S   2 node (ISV=20) WCH exponent model radiator
C      CMP45S   1 node (ISV=20) WCH air source heat pump
C      CMP46S   2 node (ISV=20) WCH advanced modulating boiler model
C      CMP47S   1 node (ISV=20) WCH ground source heat pump (EXPERIMENTAL)
C      CMP48S   2 node (ISV=20) WCH ground source heat pump (EXPERIMENTAL)
C      CMP49S   2 node (ISV=20) WCH inverter controlled ground source heat pump (EXPERIMENTAL)
C      CMP50S   1 node (ISV=29) WCH thermostatic valve
C      CMP51S   1 node (ISV=29)     mechanical room thermostat
C      CMP52S   1 node (ISV=29) WCH thermostatic valve (sensor only)
C PP-based components
C      CMP53S   4 node (ISV>19) AC  heat transfer tube with transport delay at water side
C      CMP54S   1 node (ISV=20) Water/steam flow multiplier
C      CMP55S   1 node (ISV=21) AC  air flow multiplier
C      CMP56S   5 node (ISV>19) AC  fan with submerged motor
C      CMP57S   4 node (ISV>19) WCH water pipe with transport delay
C      CMP58S   4 node (ISV>19) AC  air duct with transport delay
C      CMP60S   4 node (ISV>19) AC  air electric heater
C      CMP64S   1 node (ISV=20) Water/steam flow converger
C      CMP65S   3 node (ISV=20) Water/steam flow diverger
C      CMP66S   3 node (ISV=21) AC  moist air flow diverger
C      CMP68S   1 node (ISV=21) AC  moist air flow converger
C PP-based components end
C      CMP67S   3 node (ISV>19) AC  heat transfer tube
C      CMP70S   1 node (ISV=20) Flat plate solar collector (type1)
C      CMP71S   1 node (ISV=20) Solar collector "segment"
C      CMP72S   1 node (ISV=20) Water storage tank layer
C      CMP73S   3 node (ISV=20) WCH 3-port valve
C      CMP74S   1 node (ISV=20) WCH 3-port valve w/ automatic control
C                               valve (EXPERIMENTAL)
C      CMP79S   1 node (ISV=20) WCH stochastic hot water draw
C      CMP90S   2 node (ISV>19)     air & water temperature source
C      CMP91S   2 node (ISV>19)     imaginary building-like plant load
C                                   acting on heat and mass balances
C      CMP92S   1 node (ISV=21) AC  fictitious air boundary component
C      CMP93S  10 node (ISV>20) Heat exchanger.
C      CMP94S   3 node (ISV=20) Heat exchanger "segment"
C      CMP98S   1 node (ISV=20) CHP engine component
C      CMP99S   1 node (ISV=20) 3 node small scale a CHP unit
C      FC_static_template  3 node (ISV=21) fuel cell
C      FC_tank_fuel_static_temp  3 node (ISV=20) gas-fired water tank for fuel cell
C      FC_tank_elec_static_temp  1 node (ISV=20) electrically heated water tank for fuel cell
C      AIMS_tank1_fuel_static_temp  3 node (ISV=20) gas-fired water tank for first phase of AIMS work
C      TAC_1_static_template  3 node (ISV=20) first phase thermally activated cooling component
C      TAC_coldtank_static_temp  1 node (ISV=20) cold water storage tank for TAC
C      SOFC_A42_static 7-node (ISV=20) IEA/ECBCS Annex 42 SOFC-cogeneration device
C      simple_solar_coll_static 1 node (ISV=20) simple solar collector 
C      NCHE_static_template 2 node (ISV=20) natural convection heat exchanger (NCHE)
C      stratified_tank_1HX_static 2 node (ISV=20) stratified tank with one immersed coil
C      stratified_tank_2HX_static 3 node (ISV=20) stratified tank with two immersed coil
C
C      cmp134s 1 node (ISV=20) Heating or Cooling Load
C      cmp135s 3 node (ISV=20) Shell Boiler / Water Heater
C      cmp136s 4 node (ISV=20) Buried DH Water Pipe w/ adjacent pipe
C      cmp137s 1 node (ISV=20) Massless Converging Junction
C      cmp138s 3 node (ISV=20) Water 3-port diverter with flow balancing
C      cmp139s 3 node (ISV=20) Flow Branch Water Diverter Junction - Relative Flow 
C      cmp140s 3 node (ISV=20) Water 3-port Diverter Valve - Relative Flow
C      cmp141s 2 node (ISV=20) Massless Diverter Valve
C      cmp144s 1 node (ISV=21) AC cooling coil with flux determined by evaporator inlet conditions

C ******************** MZSDAT ********************

C MZSDAT calls for each plant component considered, a routine CMP??S
C which establishes for a ?? type component, data derivable from the
C static manufacturers data already read from the database and saves
C it in the static data common block. If appropriate, the user
C specified number of controlled variables is checked
C A CMP??S routine also check whether the number of connections to
C that particular ?? type component is correct and whether the
C connected nodes are of a type as expected by the corresponding
C CMP??C coefficient generator routine

      SUBROUTINE MZSDAT
      USE TCC, ONLY:TCC_static, HCC, ACC, InitializeCosimDataTypes
      USE DLL_Functions, ONLY:SetArraySizes
#include "plant.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)

C Number of building zones and constructions.
      integer ncomp,ncon
      common/c1/ncomp,ncon

C Declare local variables to track the number of coupling components to TRNSYS
C that are utilized in the user's model.
      INTEGER :: HCC_S, HCC_R, ACC_S, ACC_R
      INTEGER, DIMENSION(5) :: Sizes
      HCC_S = 1
      HCC_R = 1
      ACC_S = 1
      ACC_R = 1
      
C Initialize the variables that are used in communication with TRNSYS when
C co-simulation is active.      
      call InitializeCosimDataTypes


C Component by component, call routine to deal with relevant type
      DO 10 IPCOMP=1,NPCOMP
      ITYPE=NPCDAT(IPCOMP,4)
      IF(ITYPE.EQ.1) THEN
         CALL CMP01S(IPCOMP)
      ELSE IF(ITYPE.EQ.2) THEN
         CALL CMP02S(IPCOMP)
      ELSE IF(ITYPE.EQ.3) THEN
         CALL CMP03S(IPCOMP)
      ELSE IF(ITYPE.EQ.4) THEN
         CALL CMP04S(IPCOMP)
      ELSE IF(ITYPE.EQ.5) THEN
         CALL CMP05S(IPCOMP)
      ELSE IF(ITYPE.EQ.6) THEN
         CALL CMP06S(IPCOMP)
      ELSE IF(ITYPE.EQ.7) THEN
         CALL CMP07S(IPCOMP)
      ELSE IF(ITYPE.EQ.8) THEN
         CALL CMP08S(IPCOMP)
      ELSE IF(ITYPE.EQ.9) THEN
         CALL CMP09S(IPCOMP)
      ELSE IF(ITYPE.EQ.10) THEN
         CALL CMP10S(IPCOMP)
      ELSE IF(ITYPE.EQ.11) THEN
         CALL CMP11S(IPCOMP)
      ELSE IF(ITYPE.EQ.12) THEN
         CALL CMP12S(IPCOMP)
      ELSE IF(ITYPE.EQ.13) THEN
         CALL CMP13S(IPCOMP)
      ELSE IF(ITYPE.EQ.14) THEN
         CALL CMP14S(IPCOMP)
      ELSE IF(ITYPE.EQ.15) THEN
         CALL CMP15S(IPCOMP)
      ELSE IF(ITYPE.EQ.20) THEN
         CALL CMP20S(IPCOMP)
      ELSE IF(ITYPE.EQ.21) THEN
         CALL CMP21S(IPCOMP)
      ELSE IF(ITYPE.EQ.22) THEN
         CALL CMP22S(IPCOMP)
      ELSE IF(ITYPE.EQ.23) THEN
         CALL CMP23S(IPCOMP)
      ELSE IF(ITYPE.EQ.24) THEN
         CALL CMP24S(IPCOMP)
      ELSE IF(ITYPE.EQ.25) THEN
         CALL CMP25S(IPCOMP)
      ELSE IF(ITYPE.EQ.26) THEN
         CALL CMP26S(IPCOMP)
      ELSE IF(ITYPE.EQ.27) THEN
         CALL CMP27S(IPCOMP)
      ELSE IF(ITYPE.EQ.28) THEN
         CALL CMP28S(IPCOMP)
      ELSE IF(ITYPE.EQ.29) THEN
         CALL CMP29S(IPCOMP)
      ELSE IF(ITYPE.EQ.30) THEN
         CALL CMP30S(IPCOMP)
      ELSE IF(ITYPE.EQ.31) THEN
         CALL CMP31S(IPCOMP)
      ELSE IF(ITYPE.EQ.32) THEN
         CALL CMP32S(IPCOMP)
      ELSE IF(ITYPE.EQ.33) THEN
         CALL CMP33S(IPCOMP)
      ELSE IF(ITYPE.EQ.34) THEN
         CALL CMP34S(IPCOMP)
      ELSE IF(ITYPE.EQ.35) THEN
         CALL CMP35S(IPCOMP)
      ELSE IF(ITYPE.EQ.36) THEN
         CALL CMP36S(IPCOMP)
      ELSE IF(ITYPE.EQ.37) THEN
         CALL CMP37S(IPCOMP)
      ELSE IF(ITYPE.EQ.38) THEN
         CALL CMP38S(IPCOMP)
      ELSE IF(ITYPE.EQ.39) THEN
         CALL CMP39S(IPCOMP)
      ELSE IF(ITYPE.EQ.40) THEN
         CALL CMP40S(IPCOMP)
      ELSE IF(ITYPE.EQ.41) THEN
         CALL CMP41S(IPCOMP)
      ELSE IF(ITYPE.EQ.42) THEN
         CALL CMP42S(IPCOMP)
      ELSE IF(ITYPE.EQ.43) THEN
         CALL CMP43S(IPCOMP)
      ELSE IF(ITYPE.EQ.44) THEN
         CALL CMP44S(IPCOMP)
      ELSE IF(ITYPE.EQ.45) THEN
         CALL CMP45S(IPCOMP)
      ELSE IF(ITYPE.EQ.46) THEN
         CALL CMP46S(IPCOMP)
      ELSE IF(ITYPE.EQ.47) THEN
         CALL CMP47S(IPCOMP)
      ELSE IF(ITYPE.EQ.48) THEN
         CALL CMP48S(IPCOMP)
      ELSE IF(ITYPE.EQ.49) THEN
         CALL CMP49S(IPCOMP)
      ELSE IF(ITYPE.EQ.50) THEN
         CALL CMP50S(IPCOMP)
      ELSE IF(ITYPE.EQ.51) THEN
         CALL CMP51S(IPCOMP)
      ELSE IF(ITYPE.EQ.52) THEN
         CALL CMP52S(IPCOMP)
      ELSE IF(ITYPE.EQ.53) THEN
         CALL CMP53S(IPCOMP)
      ELSE IF(ITYPE.EQ.54) THEN
         CALL CMP54S(IPCOMP)
      ELSE IF(ITYPE.EQ.55) THEN
         CALL CMP55S(IPCOMP)
      ELSE IF(ITYPE.EQ.56) THEN
         CALL CMP56S(IPCOMP)
      ELSE IF(ITYPE.EQ.57) THEN
         CALL CMP57S(IPCOMP)
      ELSE IF(ITYPE.EQ.58) THEN
         CALL CMP58S(IPCOMP)
      ELSE IF(ITYPE.EQ.60) THEN
         CALL CMP60S(IPCOMP)
      ELSE IF(ITYPE.EQ.64) THEN
         CALL CMP64S(IPCOMP)
      ELSE IF(ITYPE.EQ.65) THEN
         CALL CMP65S(IPCOMP)
      ELSE IF(ITYPE.EQ.66) THEN
         CALL CMP66S(IPCOMP)
      ELSE IF(ITYPE.EQ.68) THEN
         CALL CMP68S(IPCOMP)
      ELSE IF(ITYPE.EQ.67) THEN
         CALL CMP67S(IPCOMP)
      ELSE IF(ITYPE.EQ.70) THEN
         CALL CMP70S(IPCOMP)
      ELSE IF(ITYPE.EQ.71) THEN
         CALL CMP71S(IPCOMP)
      ELSE IF(ITYPE.EQ.72) THEN
         CALL CMP72S(IPCOMP)
      ELSE IF(ITYPE.EQ.73) THEN
         CALL CMP73S(IPCOMP)
      ELSE IF(ITYPE.EQ.74) THEN
         CALL CMP74S(IPCOMP)
      ELSE IF(ITYPE.EQ.79) THEN
         CALL CMP79S(IPCOMP)
      ELSE IF(ITYPE.EQ.80) THEN
         CALL FC_static_template(IPCOMP)
      ELSE IF(ITYPE.EQ.81) THEN
         CALL FC_tank_fuel_static_temp(IPCOMP)
      ELSE IF(ITYPE.EQ.82) THEN
         CALL FC_tank_elec_static_temp(IPCOMP)
      ELSE IF(ITYPE.EQ.83) THEN
         CALL AIMS_tank1_fuel_static_temp(IPCOMP)
      ELSE IF(ITYPE.EQ.84) THEN
         CALL TAC_1_static_template(IPCOMP)
      ELSE IF(ITYPE.EQ.85) THEN
         CALL TAC_coldtank_static_temp(IPCOMP)
      ELSE IF(ITYPE.EQ.86) THEN
         CALL PEM_static_template(IPCOMP)
      ELSE IF(ITYPE.EQ.90) THEN
         CALL CMP90S(IPCOMP)
      ELSE IF(ITYPE.EQ.91) THEN
         CALL CMP91S(IPCOMP)
      ELSE IF(ITYPE.EQ.92) THEN
         CALL CMP92S(IPCOMP)
      ELSE IF(ITYPE.EQ.93) THEN
         CALL CMP93S(IPCOMP)
      ELSE IF(ITYPE.EQ.94) THEN
         CALL CMP94S(IPCOMP)
      ELSE IF(ITYPE.EQ.95) THEN
C radiant heating/cooling system
         CALL RadFloor_StaticTemp(IPCOMP)
      ELSE IF(ITYPE.EQ.98) THEN
         CALL CMP98S(IPCOMP)
      ELSE IF(ITYPE.EQ.99) THEN
         CALL CMP99S(IPCOMP)
      ELSE IF(ITYPE.EQ. 87
     &  .OR. ITYPE .EQ. 89
     &  .OR. ITYPE .EQ. 100
     &  .OR. ITYPE .EQ. 101
     &  .OR. ITYPE .EQ. 102
     &  .OR. ITYPE .EQ. 103
     &  .OR. ITYPE .EQ. 105
     &  .OR. ITYPE .EQ. 106
     &  .OR. ITYPE .EQ. 107 
     &  .OR. ITYPE .EQ. 108
     &  .OR. ITYPE .EQ. 109 
     &  .OR. ITYPE .EQ. 110 ) THEN
C Generic syntax checker for coefficient generator
         CALL HVAC_model_static_temp(IPCOMP)
      ELSE IF ( ( iType .eq. 88 ) .or.
     &          ( iType .EQ. 125 ) ) then
C Hydrogen compressor static template
         call H2_comp_static_template (iPComp)
      ELSE IF( ITYPE .EQ. 104 ) THEN
C 3 node electrolyser_model
         CALL H2_electrolyser_static_temp(IPCOMP)
      ELSE IF(ITYPE.EQ.111) THEN
         CALL SOFC_A42_static(IPCOMP)
      ELSE IF(ITYPE.EQ.112) THEN
C 3-node residential SOFC
         CALL FC_static_template(IPCOMP)
C Gas-fired storage tank with 3 connections
      ELSE IF(ITYPE.EQ.113) THEN
         CALL ADS_tank_fuel_static_temp(IPCOMP)
C Electric storage tank with 3 connections
      ELSE IF(ITYPE.EQ.114) THEN
         CALL ADS_tank_elec_static_temp(IPCOMP)
C Adsorption storage unit
      ELSE IF(ITYPE.EQ.115) THEN
         CALL ADS_stg_static_template(IPCOMP)
C Pressurized WCH pipe
      ELSE IF(ITYPE.EQ.116) THEN
         CALL Press_WCH_pipe_static_temp(IPCOMP)
C Pressurized WCH pump
      ELSE IF(ITYPE.EQ.117) THEN
         CALL Press_WCH_pump_static_temp(IPCOMP)
C 1 node simple solar collector model
      ELSE IF(ITYPE.EQ.118) THEN
         CALL simple_solar_coll_static(IPCOMP)
C X node generic TRNSYS type
      ELSE IF(ITYPE.EQ.119) THEN
         CALL trnsys_static_temp(IPCOMP)
      ELSE IF(ITYPE.EQ.120) THEN
C 2-node storage tank with immersed coil
         CALL tank_intank_hx_static(IPCOMP)
      ELSE IF(ITYPE.EQ.121) THEN
C 1-node mains 24-hour flow rate and 12-hour temperature profiles
         CALL mains_temp_draw_profile_static(IPCOMP) 
C 1-node mains 24-hour flow rate profile
      ELSE IF(ITYPE.EQ.122) THEN
         CALL mains_draw_daily_profile_static(IPCOMP)
C 2-node stratified storage tank model
      ELSE IF(ITYPE.EQ.123) THEN
         CALL stratified_tank_static(IPCOMP)
C 2-node NCHE (natural convection heat exchanger) model
      ELSE IF(ITYPE.EQ.124) THEN
         CALL NCHE_static_template(IPCOMP)
C 2-node stratified storage tank model with 1 immersed heat exchanger
      ELSE IF(ITYPE.EQ.126) THEN
         CALL stratified_tank_1HX_static(IPCOMP)
C 3-node stratified storage tank model with 2 immersed heat exchangers
      ELSE IF(ITYPE.EQ.127) THEN
         CALL stratified_tank_2HX_static(IPCOMP)
C Hydronic sending coupling component (HCC-S) for co-simulation with TRNSYS.
      ELSE IF (ITYPE .EQ. 130) THEN
         CALL TCC_static(IPCOMP, HCC_S, HCC, .true.)
         HCC_S = HCC_S+1
C Hydronic receiving coupling component (HCC-R) for co-simulation with TRNSYS.
      ELSE IF (ITYPE .EQ. 131) THEN
         CALL TCC_static(IPCOMP, HCC_R, HCC, .false.)
         HCC_R = HCC_R+1
C Air-based sending coupling component (HCC-S) for co-simulation with TRNSYS.
      ELSE IF (ITYPE .EQ. 132) THEN
         CALL TCC_static(IPCOMP, ACC_S, ACC, .true.)
         ACC_S = ACC_S+1
C Air-based receiving coupling component (HCC-R) for co-simulation with TRNSYS.
      ELSE IF (ITYPE .EQ. 133) THEN
         CALL TCC_static(IPCOMP, ACC_R, ACC, .false.)
         ACC_R = ACC_R+1

      ELSE IF(ITYPE.EQ.134) THEN
         CALL CMP134S(IPCOMP)
      ELSE IF(ITYPE.EQ.135) THEN
         CALL CMP135S(IPCOMP)
      ELSE IF(ITYPE.EQ.136) THEN
         CALL CMP136S(IPCOMP)
      ELSE IF(ITYPE.EQ.137) THEN
         CALL CMP137S(IPCOMP)
      ELSE IF(ITYPE.EQ.138) THEN
         CALL CMP138S(IPCOMP)
      ELSE IF(ITYPE.EQ.139) THEN
         CALL CMP139S(IPCOMP)
      ELSE IF(ITYPE.EQ.140) THEN
         CALL CMP140S(IPCOMP)
      ELSE IF(ITYPE.EQ.141) THEN
         CALL CMP141S(IPCOMP)
C 2-node stratified storage tank model
      ELSE IF(ITYPE.EQ.142) THEN
         CALL stratified_tank_byp_static(IPCOMP)
      ELSE IF(ITYPE.EQ.143) THEN
         CALL CMP143S(IPCOMP)
      ELSE IF(ITYPE.EQ.144) THEN
         CALL CMP144S(IPCOMP)
      ELSE IF(ITYPE.EQ.145) THEN
         CALL CMP145S(IPCOMP)
      ELSE
        WRITE(IUOUT,*) ' No static template for component ',IPCOMP,ITYPE
        STOP ' MZSDAT: unresolvable error on component type'
      END IF
   10 CONTINUE

C When co-simulation with TRNSYS is active, the Harmonizer needs to know how many of
C each type of coupling there is with TRNSYS. It also needs to know the number of building
C zones in the model.
      Sizes(1) = HCC_S-1
      Sizes(2) = HCC_R-1
      Sizes(3) = ACC_S-1
      Sizes(4) = ACC_R-1
      Sizes(5) = nComp
      call SetArraySizes(Sizes)

      RETURN
      END

C ******************** CMP01S ********************
C Establishes for a plant component with plant db code 10 ie.
C 1 node (ISV=21) AC  converging 2-leg junction
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP01S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  converging 2-leg junction'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP01S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 2 connections only, both to "moist air"
      NCONS=2
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP01S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP01S: unresolvable error'

      END

C ******************** CMP02S ********************
C CMP02S establishes for a plant component with plant db code 20 ie.
C 1 node (ISV=21) AC  humidifier
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP02S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,7)
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  humidifier adiabatic'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=4
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP02S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP02S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP02S: unresolvable error'

      END

C ******************** CMP03S ********************
C Establishes for a plant component with plant db code 30 ie.
C 1 node (ISV=21) AC  fan
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP03S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,6)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  fan'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=3
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP03S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP03S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP03S: unresolvable error'

      END

C ******************** CMP04S ********************
C Establishes for a plant component with plant db code 40 ie.
C 1 node (ISV=21) AC  cooling coil & flux control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP04S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Negate CDATA(?,1) as this component is a cooling coil
C and CDATA(?,1) relates to the maximum cooling energy.
      CDATA(IPCOMP,1)=-CDATA(IPCOMP,1)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  cooling coil and',
     &                ' water mass flow rate control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP04S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP04S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP04S: unresolvable error'

      END

C ******************** CMP05S ********************
C Establishes for a plant component with plant db code 50 ie.
C 1 node (ISV=21) AC  heating coil & flux control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP05S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  heating coil & water',
     &                ' mass flow rate control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP05S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP05S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP05S: unresolvable error'

      END

C ******************** CMP06S ********************

C CMP06S establishes for a plant component with plant db code 20 ie.
C 1 node (ISV=21) AC  duct & condensation ?
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP06S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  duct & condensation ?'
         NITMS=6
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP06S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE


      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP06S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '   should be ',NCONS,' connection(s) to air'
      WRITE(IUOUT,*) '   found ',IPCON,'.'
      STOP ' CMP06S: unresolvable error'

      END

C ******************** CMP07S ********************
C Establishes for a plant component with plant db code 70 ie.
C 1 node (ISV=21) AC  damper & flow ratio control
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      subroutine cmp07s(ipcomp)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  damper & flow ratio control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' cmp07s warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

C Return to calling module
      RETURN
C Error handling
  990 WRITE(IUOUT,*) ' cmp07s: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' cmp07s: unresolvable error'
      END

C ******************** CMP08S ********************
C Establishes for a plant component with plant db code 80 ie.
C 1 node (ISV=21) AC flow diversion damper
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      subroutine cmp08s(ipcomp)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  flow diversion damper'
         NITMS=4
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' cmp08s warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

C Return to calling module
      RETURN
C Error handling
  990 WRITE(IUOUT,*) ' cmp08s: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' cmp08s: unresolvable error'
      END
      
C ******************** CMP09S ********************
C Establishes for a plant component with plant db code 90 ie.
C 1 node (ISV=21) AC  cooling coil & water mass flow rate control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP09S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,8)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  cooling coil and',
     &                ' water mass flow rate control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=5
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP09S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP09S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP09S: unresolvable error'
      END

C ******************** CMP10S ********************
C CMP10S establishes for a plant component with plant db code 100 ie.
C 1 node (ISV=21) AC  cooling coil & water mass flow rate control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP10S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,9)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  cooling coil and',
     &                ' water mass flow rate control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=6
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP10S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP10S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP10S: unresolvable error'

      END

C ******************** CMP11S ********************
C Establishes for a plant component with plant db code 110 ie.
C 1 node (ISV=21) AC  heating coil & water mass flow rate control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP11S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,9)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  heating coil & water',
     &                ' mass flow rate control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=6
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP11S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP11S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP11S: unresolvable error'

      END

C ******************** CMP12S ********************
C Establishes for a plant component with plant db code 120 ie.
C 2 node (ISV=21) AC  plate heat exchanger
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP12S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=21) AC  plate heat exchanger'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=2
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP12S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 2 connections only, one per node, and both to air
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IPCON2=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.21.OR.ICONTP(IPCON2).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP12S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie ',NCONPN,' per node, and both to air'
      STOP ' CMP12S: unresolvable error'

      END

C ******************** CMP13S ********************
C Establishes for a plant component with plant db code 120 ie.
C 2 node (ISV=21) AC  basic heat exchanger model with moisture recovery
C the model is based on a simple effectiveness calculation
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP13S(IPCOMP)
#include "plant.h"
#include "building.h"
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=21) AC basic heat exchanger'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=2
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP13S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 2 connections only, one per node, and both to air
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IPCON2=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.21.OR.ICONTP(IPCON2).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP13S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie ',NCONPN,' per node, and both to air'
      STOP ' CMP13S: unresolvable erro r'
      END

C ******************** CMP14S ********************
C Establishes for a plant component with plant db code 120 ie.
C 2 node (ISV>20) Cooling tower model
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP14S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=21) AC  plate heat exchanger'
         NITMS=6
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP14S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 2 connections only, one per node, and both to air
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IPCON2=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.21.OR.ICONTP(IPCON2).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP14S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie ',NCONPN,' per node, one to air'
      WRITE(IUOUT,*) '         and one to water'
      STOP ' CMP14S: unresolvable error'

      END

C ******************** CMP15S ********************
C Establishes for a plant component with plant db code 150 ie.
C 1 node (ISV=21) AC  fan with mass flow control
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP15S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,6)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  fan mass flow control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=3
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP15S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP15S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP03S: unresolvable error'

      END

C ******************** CMP20S ********************
C Establishes for a plant component with plant db code 200 ie.
C 1 node (ISV=20) WCH boiler & flux control
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP20S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH boiler & flux control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP20S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP20S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP20S: unresolvable error'

      END

C ******************** CMP21S ********************
C Establishes for a plant component with plant db code 210 ie.
C 2 node (ISV=20) WCH radiator
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP21S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      NBDAT=7+int(ADATA(IPCOMP,9))*2
      DO 5 IBDAT=1,NBDAT
         BDATA(IPCOMP,IBDAT)=ADATA(IPCOMP,IBDAT+2)
    5 CONTINUE

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH radiator'
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=NBDAT
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP21S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP21S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP21S: unresolvable error'

      END

C ******************** CMP22S ********************
C Establishes for a plant component with plant db code 220 ie.
C 1 node (ISV=20) WCH pipe
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP22S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH pipe'
         NITMS=6
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP22S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP22S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP22S: unresolvable error'

      END

C ******************** CMP23S ********************
C Establishes for a plant component with plant db code 230 ie.
C 1 node (ISV=20) WCH converging 2-leg junction
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP23S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH converging 2-leg junction'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP23S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 2 connections only, both to water
      NCONS=2
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP23S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP23S: unresolvable error'

      END

C ******************** CMP24S ********************
C Establishes for a plant component with plant db code 240 ie.
C 1 node (ISV=20) WCH pump
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP24S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,6)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH pump'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=3
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP24S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP24S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP24S: unresolvable error'

      END

C ******************** CMP25S ********************
C Establishes for a plant component with plant db code 250 ie.
C C 2 node (ISV=20) WCH boiler with on/off control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP25S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,3)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,7)=ADATA(IPCOMP,9)
      BDATA(IPCOMP,8)=ADATA(IPCOMP,10)
      BDATA(IPCOMP,9)=ADATA(IPCOMP,11)
      BDATA(IPCOMP,10)=ADATA(IPCOMP,12)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH boiler with on/off control '
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=10
         WRITE(ITU,*) ' BDATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP25S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP25S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP25S: unresolvable error'

      END
C
C ******************** CMP26S ********************
C Establishes for a plant component with plant db code 260 ie.
C C 2 node (ISV=20) WCH boiler with aquastat control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP26S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,3)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,7)=ADATA(IPCOMP,9)
      BDATA(IPCOMP,8)=ADATA(IPCOMP,10)
      BDATA(IPCOMP,9)=ADATA(IPCOMP,11)
      BDATA(IPCOMP,10)=ADATA(IPCOMP,12)
      BDATA(IPCOMP,11)=ADATA(IPCOMP,13)
      BDATA(IPCOMP,12)=ADATA(IPCOMP,14)
      BDATA(IPCOMP,13)=ADATA(IPCOMP,15)
      BDATA(IPCOMP,14)=ADATA(IPCOMP,16)
      BDATA(IPCOMP,15)=ADATA(IPCOMP,17)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH boiler & aquastat control '
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=15
         WRITE(ITU,*) ' BDATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=2
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=2
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP26S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP26S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP26S: unresolvable error'

      END

C ******************** CMP27S ********************
C Establishes for a plant component with plant db code 270 ie.
C 8 node (ISV=20) WCH radiator
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP27S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Establish static data derivable from the data read from database
      NBDAT=7+int(ADATA(IPCOMP,9))*2
      DO 5 IBDAT=1,NBDAT
         BDATA(IPCOMP,IBDAT)=ADATA(IPCOMP,IBDAT+2)
    5 CONTINUE

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 8 node (ISV=20) WCH radiator'
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=NBDAT
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP27S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP27S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP27S: unresolvable error'

      END

C ******************** CMP28S ********************
C Establishes for a plant component with plant db code 280 ie.
C 1 node (ISV=29) Oil-filled electric panel radiator
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP28S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      NBDAT=6+int(ADATA(IPCOMP,9))*2
      DO 5 IBDAT=1,NBDAT
         BDATA(IPCOMP,IBDAT)=ADATA(IPCOMP,IBDAT+2)
    5 CONTINUE

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=29) Oilo-filled electric radiator'
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=NBDAT
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP28S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has no connection.
      NCONS=0
      IPCON=ICONDX(IPCOMP,1,1)
      IF(IPCON.NE.NCONS) GOTO 990
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP28S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP28S: unresolvable error'
      END

C ******************** CMP29S ********************
C Establishes for a plant component with plant db code 280 ie.
C 1 node (ISV=20) WCH flow control valve
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      subroutine cmp29s(ipcomp)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH valve & flow ratio control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' cmp29s warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "water"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

C Return to calling module
      RETURN
C Error handling
  990 WRITE(IUOUT,*) ' cmp29s: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' cmp29s: unresolvable error'
      END

C ******************** CMP30S ********************
C Establishes for a plant component with plant db code 300 ie.
C 1 node (ISV=20) water cooler & flux control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP30S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Negate CDATA(?,1) as this component is a water cooler
C and CDATA(?,1) relates to the maximum cooling energy.
      CDATA(IPCOMP,1)=-CDATA(IPCOMP,1)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) Water cooler and',
     &                ' flux control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP30S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

C Return to calling module.
      RETURN

C Error handling.
  990 WRITE(IUOUT,*) ' CMP30S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP30S: unresolvable error'

      END
C ******************** CMP31S ********************
C Establishes for a plant component with plant db code 310 ie.
C C 2 node (ISV=20) WCH calorifier
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP31S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,8)


C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH calorifier '
         NITMS=4
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=4
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP31S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 2 connections only, to water
      NCONS=2
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,IPCONC,1)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP31S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP31S: unresolvable error'

      END

C ******************** CMP32S ********************
C Establishes for a plant component with plant db code 320 ie.
C 2 node (ISV>20) WCH generic liquid/liquid heat exchanger
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

C node 1 represents water              (ISV=20)
C node 2 represents water              (ISV=20)

      SUBROUTINE CMP32S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,6)
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH  liquid/liquid heat  ',
     &                ' exchangerfed by WCH system'
         NITMS=5
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' BDATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP32S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 2 connections only, one per node, and both to water
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IPCON2=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.20.OR.ICONTP(IPCON2).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      return

C Error handling
  990 WRITE(IUOUT,*) ' CMP32S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie  1st connection to water'
      WRITE(IUOUT,*) '             2nd connection to water'
      STOP ' CMP32S: unresolvable error'


      END
C
C ******************** CMP33S ********************
C Establishes for a plant component with plant db code 330 ie.
C 2 node (ISV>20) WCH generic gas/liquid heat exchanger
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

C node 1 represents gas                (ISV=21)
C node 2 represents water              (ISV=20)

      SUBROUTINE CMP33S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,6)
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH  liquid/liquid heat  ',
     &                ' exchangerfed by WCH system'
         NITMS=5
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' BDATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP33S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 2 connections only, one per node, and both to water
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IPCON2=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.21.OR.ICONTP(IPCON2).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      return

C Error handling
  990 WRITE(IUOUT,*) ' CMP33S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie  1st connection to gas'
      WRITE(IUOUT,*) '             2nd connection to water'
      STOP ' CMP33S: unresolvable error'


      END
C
C ******************** CMP34S ********************
C Establishes for a plant component with plant db code 340 ie.
C 3 node (ISV>20) WCH
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
C node 1 represents water              (ISV=20) (2 connections)
C node 2 represents gas                (ISV=21) (1 connection)
C node 3 represents gas                (ISV=21)

      SUBROUTINE CMP34S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      INTEGER ADAT

C Establish static data derivable from the data read from database
      NDAT=13
      ADAT=3
      DO 10 IDAT=1,NDAT-ADAT
        BDATA(IPCOMP,IDAT)=ADATA(IPCOMP,IDAT+ADAT)
  10  CONTINUE

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 3 node (ISV>20) WCH  heated water storage  '
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=10
         WRITE(ITU,*) ' BDATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP34S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has at least two connections.
      NCONPN=1
      DO 20 IPCONC=1,MPCONC
        IPCON1=ICONDX(IPCOMP,1,IPCONC)
        IPCON2=ICONDX(IPCOMP,2,IPCONC)
        IF(IPCONC.LE.NCONPN) THEN
          IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
             GOTO 990
          ELSE IF(ICONTP(IPCON1).NE.20.OR.ICONTP(IPCON2).NE.21) THEN
             GOTO 990
          END IF
        ELSE
          IF(IPCONC.EQ.2.AND.IPCON2.NE.0) THEN
            GOTO 990
          ELSEIF(IPCONC.EQ.3.AND.IPCON1.NE.0)THEN
            GOTO 990
          ENDIF
        END IF

   20 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP34S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be 2 or 3 connection(s)'
      WRITE(IUOUT,*) '         ie  1st connection to water node'
      WRITE(IUOUT,*) '             2nd connection to water node'
      WRITE(IUOUT,*) '             3rd connection to combustion node'
      STOP ' CMP34S: unresolvable error'

      END
C
C ******************** CMP35S ********************
C Establishes for a plant component with plant db code 350 ie.
C 2 node (ISV=20) WCH basic radiator
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP35S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      NBDAT=5+int(ADATA(IPCOMP,9))*2
      DO 5 IBDAT=1,NBDAT
         BDATA(IPCOMP,IBDAT)=ADATA(IPCOMP,IBDAT+2)
    5 CONTINUE

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH basic radiator'
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=NBDAT
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP35S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN
      
C Error handling
  990 WRITE(IUOUT,*) ' CMP35S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP35S: unresolvable error'

      END

C ******************** CMP36S ********************
C Establishes for a plant component with plant db code 360 ie.
C 2 node (ISV>19) WCH reversible heat pump 
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

C node 1 part of component in contact with water loop (ISV=20)
C node 2 part of component in moist air stream (ISV=21)

      SUBROUTINE CMP36S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)


C Establish static data derivable from the data read from database
      DO 5 J=1,8
        BDATA(IPCOMP,j)=ADATA(IPCOMP,3+j)
5     CONTINUE

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH basic chiller unit '
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=8
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP36S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 2 connections only, one per node, to water (node1) and
C air (node 2)
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IPCON2=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.20.OR.ICONTP(IPCON2).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      return

C Error handling
  990 WRITE(IUOUT,*) ' CMP36S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie  1st connection to water'
      WRITE(IUOUT,*) '             2nd connection to air'
      STOP ' CMP36S: unresolvable error'


      END


C ******************** CMP37S ********************
C Establishes for a plant component with plant db code 370 ie.
C 1 node (ISV=20) WCH air or evaporatively cooled heat rejection device
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP37S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)  

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH heat rejection device'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP37S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP37S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMPS: unresolvable error'

      END


C ******************** CMP38S ********************
C Establishes for a plant component with plant db code 380 ie.
C 1 node (ISV=20) WCH converging multi-leg junction
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP38S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)


C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4) 

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH multi-leg junction'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP38S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has N connections, all to water
      NCONS=NINT(BDATA(IPCOMP,1))
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP38S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP38S: unresolvable error'

      END

     
C ******************** CMP39S ********************
C Establishes for a plant component with plant db code 390 ie.
C 1 node (ISV=20) WCH pump with mass flow control
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP39S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,6)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH pump with mass flow control'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=3
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP39S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP39S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP39S: unresolvable error'

      END


C ******************** CMP40S ********************
C Establishes for a plant component with plant db code 400 ie.
C 3 node (ISV>19) AC  cooling coil fed by WCH system
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

C node 1 represents solid material (ISV=29)
C node 2 represents air            (ISV=21)
C node 3 represents water          (ISV=20)

      SUBROUTINE CMP40S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,9)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,10)
      BDATA(IPCOMP,7)=ADATA(IPCOMP,11)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 3 node (ISV>19) AC  counterflow cooling coil',
     &                ' fed by WCH system'
         NITMS=4
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=7
         WRITE(ITU,*) ' BDATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP40S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1st connection to "moist air"
C                     2nd connection to water
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,2,IPCONC)
      IPCON2=ICONDX(IPCOMP,3,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.21.OR.ICONTP(IPCON2).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP40S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie  1st connection to air'
      WRITE(IUOUT,*) '             2nd connection to water'
      STOP ' CMP40S: unresolvable error'

      END
C
C ******************** CMP41S ********************
C Establishes for a plant component with plant db code 410 ie.
C 3 node (ISV>19) AC  heating coil fed by WCH system
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

C node 1 represents solid material (ISV=29)
C node 2 represents air            (ISV=21)
C node 3 represents water          (ISV=20)

      SUBROUTINE CMP41S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,9)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 3 node (ISV>19) AC  counterflow heating coil',
     &                ' fed by WCH system'
         NITMS=4
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=5
         WRITE(ITU,*) ' BDATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP41S warning: user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 1st connection to "moist air"
C                     2nd connection to water
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,2,IPCONC)
      IPCON2=ICONDX(IPCOMP,3,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.21.OR.ICONTP(IPCON2).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP41S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie  1st connection to air'
      WRITE(IUOUT,*) '             2nd connection to water'
      STOP ' CMP41S: unresolvable error'

      END

C ******************** CMP42S ********************
C Establishes for a plant component with plant db code 420 ie.
C C 2 node (ISV=20) modulating WCH boiler with on/off control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP42S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,3)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,7)=ADATA(IPCOMP,9)
      BDATA(IPCOMP,8)=ADATA(IPCOMP,10)
      BDATA(IPCOMP,9)=ADATA(IPCOMP,11)
      BDATA(IPCOMP,10)=ADATA(IPCOMP,12)
      BDATA(IPCOMP,11)=ADATA(IPCOMP,13)
      BDATA(IPCOMP,12)=ADATA(IPCOMP,14)
      BDATA(IPCOMP,13)=ADATA(IPCOMP,15)
      BDATA(IPCOMP,14)=ADATA(IPCOMP,16)
      BDATA(IPCOMP,15)=ADATA(IPCOMP,17)
      BDATA(IPCOMP,16)=ADATA(IPCOMP,18)
      BDATA(IPCOMP,17)=ADATA(IPCOMP,19)
      BDATA(IPCOMP,18)=ADATA(IPCOMP,20)
      BDATA(IPCOMP,19)=ADATA(IPCOMP,21)
      BDATA(IPCOMP,20)=ADATA(IPCOMP,22)
      BDATA(IPCOMP,21)=ADATA(IPCOMP,23)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) modulating WCH boiler '
         WRITE(ITU,*) ' with on/off control '
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=10
         WRITE(ITU,*) ' BDATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP42S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP42S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP42S: unresolvable error'

      END

      
C ******************** CMP43S ********************
C Establishes for a plant component with plant db code 430 ie.
C 2 node (ISV>20) AC  cooling coil fed by WCH system
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

C node 1 represents solid&air material (ISV=21)
C node 2 represents water              (ISV=20)

      SUBROUTINE CMP43S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,9)
      BDATA(IPCOMP,7)=ADATA(IPCOMP,10)
      BDATA(IPCOMP,8)=ADATA(IPCOMP,11)
      BDATA(IPCOMP,9)=ADATA(IPCOMP,12)
      BDATA(IPCOMP,10)=ADATA(IPCOMP,13)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV>20) AC  coling coil',
     &                ' fed by WCH system'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=10
         WRITE(ITU,*) ' BDATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP41S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1st connection to "moist air"
C                     2nd connection to water
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IPCON2=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.21.OR.ICONTP(IPCON2).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP43S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie  1st connection to air'
      WRITE(IUOUT,*) '             2nd connection to water'
      STOP ' CMP43S: unresolvable error'

      END

C ******************** CMP44S ********************
C Establishes for a plant component with plant db code 440 ie.
C 2 node (ISV=20) WCH basic radiator
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP44S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      NBDAT=11
      DO 5 IBDAT=1,NBDAT
         BDATA(IPCOMP,IBDAT)=ADATA(IPCOMP,IBDAT+2)
    5 CONTINUE

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH basic radiator'
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=NBDAT
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP44S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP44S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP44S: unresolvable error'
C
      END

C ******************** CMP45S ********************
C CMP45S establishes for a plant component with plant db code 450 ie.
C 1 node (ISV=20) WCH air-source heat pump feeding a hydronic heating system
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine.
C
      SUBROUTINE CMP45S(IPCOMP)

      implicit none

#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      COMMON/ASHPvar1/CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,
     &CompressPf,PumpRating,PumpPf,RatedFlow,
     &FanRating,FanPf,CtlRating,CtlPf,ToutMax,TinMax,DefrostT,
     &DefrostTime,Defrostb0,Defrostb1,DefrostLockout,DefrostMinTime,
     &DefrostMaxTime,NomRetT,NomRetTDeadB,TempCompS,TempCompE,
     &TempCompc0,DefrostDur,DefrostLockDur

      COMMON/ASHPvar2/COPModel,CompModel,DefrostCalc,AmbientTempComp,
     &DeviceONOFFp,DeviceONOFF

      COMMON/ASHPvar3/CallforHeat,InDeadB,DefrostLock,InDefrost

      REAL SMALL 
      PARAMETER (SMALL=1.0E-20)

      INTEGER COPModel,CompModel,DefrostCalc,AmbientTempComp,
     &DeviceONOFFp,DeviceONOFF

      REAL CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,CompressPf,
     &PumpRating,PumpPf,RatedFlow,FanRating,FanPf,CtlRating,CtlPf,
     &ToutMax,TinMax,DefrostTime,DefrostT,Defrostb0,Defrostb1,
     &DefrostLockout,DefrostMinTime,DefrostMaxTime,NomRetT,NomRetTDeadB,
     &TempCompS,TempCompE,TempCompc0,DefrostDur,DefrostLockDur

      LOGICAL CallforHeat,InDeadB,DefrostLock,InDefrost,ConError

      INTEGER ITC,ICNT,ITCF,ITRACE,IZNTRC,ITU,NPCOMP,NCI,
     &ICONTP,ICONDX,J,NCITM,NCONS,IPCONC,IPCOMP,IPCON,IX1,NITMS
      
      REAL CDATA,ADATA,BDATA,CONVAR

      INTEGER IDAT

      CHARACTER OUTS*248
      
C To be edited to include increased data entry nos. 

C Establish static data derivable from the data read from database
      DO IDAT=4,36
        BDATA(IPCOMP,IDAT-3)=ADATA(IPCOMP,IDAT)
      ENDDO

C Initialise parameters      
      InDeadB=.false.
      CallforHeat=.false.
      DefrostLock=.false.
      InDefrost=.false.
      DefrostDur=0.0
      DefrostLockDur=0.0
      DeviceONOFF=0
      DeviceONOFFp=0

C Assign variables to be used in coeff generator. 
C ADATA: 
      CompMass=ADATA(IPCOMP,1)   ! Mass of component (solids+liquid) (kg)
      AveSpHt=ADATA(IPCOMP,2)    ! Mass weighted average specific heat node (J/kgK)
      UAMod=ADATA(IPCOMP,3)      ! UA modulus for component (W/K)

C BDATA: 
C COP model [1- fixed 2 - modified Carnot efficiency; 3- cubic polynomial]
      COPModel=NINT(BDATA(IPCOMP,1))
      COPa0=BDATA(IPCOMP,2)      ! COP empirical coef a0 (or fixed COP, or Eff. modifier)
      COPa1=BDATA(IPCOMP,3)      ! COP empirical coefficient a1 [Ti]
      COPa2=BDATA(IPCOMP,4)      ! COP empirical coefficient a2 [Ti^2]
      COPa3=BDATA(IPCOMP,5)      ! COP empirical coefficient a3 [Ti^3]

C Compressor model (W) 1- fixed, 2- 1000.*(C0+(C1*Ta)+(C2*Td)), 3- exponential.
      CompModel=NINT(BDATA(IPCOMP,6))  ! Compressor model (W) 
      Compa0=BDATA(IPCOMP,7)     ! Comp empirical coef a0 (or fixed COP, or Eff. modifier)
      Compa1=BDATA(IPCOMP,8)     ! Comp empirical coefficient a1 
      Compa2=BDATA(IPCOMP,9)     ! Comp empirical coefficient a2
      Compa3=BDATA(IPCOMP,10)    ! Comp empirical coefficient a3 
C ****
      CompressPf=BDATA(IPCOMP,11)! Compressor pf (-)
      IF(CompressPf.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in ASHP: comp pf is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

      PumpRating=BDATA(IPCOMP,12) ! Pump rating (W)
      PumpPf=BDATA(IPCOMP,13)     ! Pump pf (-)
      RatedFlow=BDATA(IPCOMP,14)  ! Flowrate at rated pump power (l/s)
      IF(RatedFlow.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in ASHP: rated flow is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

      FanRating=BDATA(IPCOMP,15)  ! Fan rating (W)
      FanPf=BDATA(IPCOMP,16)      ! Fan power factor (-)
      IF(FanPf.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in ASHP: fan pf is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

      CtlRating=BDATA(IPCOMP,17)  ! Controller power (W)
      CtlPf=BDATA(IPCOMP,18)      ! Controller power factor (W)
      IF(CtlPf.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in ASHP: ctl pf is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

      ToutMax=BDATA(IPCOMP,19)     ! To max (degC) seems not to be used.
      TinMax=BDATA(IPCOMP,20)      ! Ti max (degC)
      DefrostT=BDATA(IPCOMP,21)    ! Defrost cycle trigger ambient temp (degC)

C Defrost cycle time calc (0-no defrost 1-user def 2-f(RH))
      DefrostCalc=NINT(BDATA(IPCOMP,22))

C Defrost cycle  calc coefficient b1 or fixed defrost time (-)
      Defrostb0=BDATA(IPCOMP,23)

      Defrostb1=BDATA(IPCOMP,24)      ! Defrost cycle  calc coefficient b2 (-)
      DefrostLockout=BDATA(IPCOMP,25) ! Defrost cycle  lockout time (mins)
      DefrostMinTime=BDATA(IPCOMP,26) ! Min defrost time (mins)
      DefrostMaxTime=BDATA(IPCOMP,27) ! Max defrost time (mins)

C AmbientTemp compensation on/off.
      AmbientTempComp=NINT(BDATA(IPCOMP,28))

      NomRetT=BDATA(IPCOMP,29)        ! Nominal water return temperature (Deg C)
      IF(NomRetT.GE.TinMax)THEN
        WRITE(OUTS,'(a)')'Error in ASHP Tnom => Tin max'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

      NomRetTDeadB=BDATA(IPCOMP,30)   ! Nominal water return deadband (Deg C)
      IF((NomRetT+0.5*(NomRetTDeadB)).GE.TinMax)THEN
        WRITE(OUTS,'(a)')'Error in ASHP Tnom + TdeadB < Tin max'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

      TempCompS=BDATA(IPCOMP,31)  ! Ambient temperature for temp compensation start [C]
      TempCompE=BDATA(IPCOMP,32)  ! Ambient temperature for temp compensation end [C]
      TempCompc0=BDATA(IPCOMP,33) ! Temp compensation gradient (-)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH heat rejection device'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=33
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         ENDIF
      ENDIF
C
C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP45S warning: user specified wrong number',
     &                ' of controlled variables'

C Check for a single connection to water.
      ConError=.false.
      NCONS=1
      IPCONC=1
      DO WHILE(IPCONC.LE.MPCONC.AND..NOT.ConError)
        IPCON=ICONDX(IPCOMP,1,IPCONC)
        IF(IPCONC.LE.NCONS) THEN
          IF(IPCON.EQ.0) THEN
            ConError=.true.
          ELSE IF(ICONTP(IPCON).NE.20) THEN
            ConError=.true.
          END IF
        ELSE IF(IPCON.NE.0) THEN
          ConError=.true.
        END IF
        IPCONC=IPCONC+1
      ENDDO

C Error handling
      IF(ConError)THEN
        WRITE(IUOUT,*) 'CMP45S: connection error for component ',IPCOMP
        WRITE(IUOUT,*) '        should be ',NCONS,' water connection(s)'
        STOP 'CMP45S: unresolvable error'
      ENDIF
C
      END

C ******************** CMP46S ********************

C CMP46S establishes for a plant component with plant db code 420 ie.
C C 2 node (ISV=20) modulating WCH boiler with on/off control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE CMP46S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,3)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,7)=ADATA(IPCOMP,9)
      BDATA(IPCOMP,8)=ADATA(IPCOMP,10)
      BDATA(IPCOMP,9)=ADATA(IPCOMP,11)
      BDATA(IPCOMP,10)=ADATA(IPCOMP,12)
      BDATA(IPCOMP,11)=ADATA(IPCOMP,13)
      BDATA(IPCOMP,12)=ADATA(IPCOMP,14)
      BDATA(IPCOMP,13)=ADATA(IPCOMP,15)
      BDATA(IPCOMP,14)=ADATA(IPCOMP,16)
      BDATA(IPCOMP,15)=ADATA(IPCOMP,17)
      BDATA(IPCOMP,16)=ADATA(IPCOMP,18)
      BDATA(IPCOMP,17)=ADATA(IPCOMP,19)
      BDATA(IPCOMP,18)=ADATA(IPCOMP,20)
      BDATA(IPCOMP,19)=ADATA(IPCOMP,21)
      BDATA(IPCOMP,20)=ADATA(IPCOMP,22)
      BDATA(IPCOMP,21)=ADATA(IPCOMP,23)
      BDATA(IPCOMP,22)=ADATA(IPCOMP,24)
      BDATA(IPCOMP,23)=ADATA(IPCOMP,25)
      BDATA(IPCOMP,24)=ADATA(IPCOMP,26)
      BDATA(IPCOMP,25)=ADATA(IPCOMP,27)
      BDATA(IPCOMP,26)=ADATA(IPCOMP,28)
      BDATA(IPCOMP,27)=ADATA(IPCOMP,29)
      BDATA(IPCOMP,28)=ADATA(IPCOMP,30)
      BDATA(IPCOMP,29)=ADATA(IPCOMP,31)
      BDATA(IPCOMP,30)=ADATA(IPCOMP,32)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) advanced modulating WCH boiler '
         WRITE(ITU,*) ' model with on/off control '
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=10
         WRITE(ITU,*) ' BDATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP46S warning: user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
C
C
      RETURN
C
C Error handling
  990 WRITE(IUOUT,*) ' CMP46S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' CMP46S: unresolvable error'
C
      END
C

C ******************** CMP47S ********************
C CMP47S establishes for a plant component with plant db code 470 ie.
C 1 node (ISV=20) WCH ground-source heat pump feeding a hydronic heating system
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine.
C
      SUBROUTINE CMP47S(IPCOMP)

      implicit none

#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      COMMON/GSHPvar1/CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &  COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,
     &  CompressPf,PumpRating,PumpPf,RatedFlow,
     &  CtlRating,CtlPf,ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,
     &  NomRetTSH,NomRetTDHW,NomRetTDeadB,NomRetTDeadBDHW,
     &  TempCompS,TempCompE,TempCompc0

      COMMON/GSHPvar2/COPModel,CompModel,AmbientTempComp,
     &                DeviceONOFFp,DeviceONOFF,GroundSourceModel,
     &                GroundSourceCoeff

      COMMON/GSHPvar3/CallforHeat,InDeadB

      COMMON/GSHPvar4/INSO,ITSBO,bSwitched,BLKOUT,NTSBO,bBLKOUT
      INTEGER INSO,ITSBO,NTSBO
      REAL BLKOUT
      LOGICAL bSwitched,bBLKOUT

      REAL SMALL
      PARAMETER (SMALL=1.0E-20)

      INTEGER COPModel,CompModel,AmbientTempComp,
     &DeviceONOFFp,DeviceONOFF,GroundSourceModel,GroundSourceCoeff

      REAL CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,CompressPf,
     &PumpRating,PumpPf,RatedFlow,CtlRating,CtlPf,
     &ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,NomRetTSH,NomRetTDHW,
     &NomRetTDeadB,NomRetTDeadBDHW,TempCompS,TempCompE,TempCompc0

      LOGICAL CallforHeat,InDeadB,ConError

      INTEGER ITC,ICNT,ITCF,ITRACE,IZNTRC,ITU,NPCOMP,NCI,
     &ICONTP,ICONDX,J,NCITM,NCONS,IPCONC,IPCOMP,IPCON,IX1,NITMS
      
      REAL CDATA,ADATA,BDATA,CONVAR

      INTEGER IDAT

      CHARACTER OUTS*248
      
C To be edited to include increased data entry nos. 

C Establish static data derivable from the data read from database
      DO IDAT=4,33
        BDATA(IPCOMP,IDAT-3)=ADATA(IPCOMP,IDAT)
      ENDDO

C Initialise parameters      
      InDeadB=.false.
      CallforHeat=.false.

C Assign variables to be used in coeff generator. 
C ADATA: 
C Mass of component (solids+liquid) (kg)
      CompMass=ADATA(IPCOMP,1)
C Mass weighted average specific heat node (J/kgK)
      AveSpHt=ADATA(IPCOMP,2)
C UA modulus for component (W/K)
      UAMod=ADATA(IPCOMP,3)

C BDATA: 
C COP model [1- fixed 2 - modified Carnot efficiency; 3- cubic polynomial]
      COPModel=NINT(BDATA(IPCOMP,1))

C COP empirical coefficient a0 (or fixed COP, or Eff. modifier)
      COPa0=BDATA(IPCOMP,2)

C COP empirical coefficient a1 [Ti]
      COPa1=BDATA(IPCOMP,3)

C COP empirical coefficient a2 [Ti^2]
      COPa2=BDATA(IPCOMP,4)

C COP empirical coefficient a3 [Ti^3]
      COPa3=BDATA(IPCOMP,5)

C Compressor model (W) 
      CompModel=NINT(BDATA(IPCOMP,6))

C Comp empirical coefficient a0 (or fixed COP, or Eff. modifier)
      Compa0=BDATA(IPCOMP,7)

C Comp empirical coefficient a1 
      Compa1=BDATA(IPCOMP,8)

C Comp empirical coefficient a2 
      Compa2=BDATA(IPCOMP,9)

C Comp empirical coefficient a3 
      Compa3=BDATA(IPCOMP,10)
C ****
C Compressor pf (-)
      CompressPf=BDATA(IPCOMP,11)
      IF(CompressPf.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in GSHP: comp pf is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      
      ENDIF

C Pump rating (W)
      PumpRating=BDATA(IPCOMP,12)

C Pump pf (-)
      PumpPf=BDATA(IPCOMP,13)

C Flowrate at rated pump power (l/s)
      RatedFlow=BDATA(IPCOMP,14)
      IF(RatedFlow.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in GSHP: rated flow is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Controller power (W)
      CtlRating=BDATA(IPCOMP,15)

C Controller power factor (W)
      CtlPf=BDATA(IPCOMP,16)
      IF(CtlPf.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in GSHP: ctl pf is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C To max (degC)
      ToutMaxSH=BDATA(IPCOMP,17)

C Ti max (degC)
      TinMaxSH=BDATA(IPCOMP,18)

C AmbientTemp compensation on/off
      AmbientTempComp=NINT(BDATA(IPCOMP,19))

C Nominal water return temperature (Deg C)
      NomRetTSH=BDATA(IPCOMP,20)
      IF(NomRetTSH.GE.TinMaxSH)THEN
        WRITE(OUTS,'(a)')'Error in GSHP Tnom >= Tin max SH'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Nominal water return deadband (Deg C)
      NomRetTDeadB=BDATA(IPCOMP,21)
      IF((NomRetTSH+0.5*(NomRetTDeadB)).GE.TinMaxSH)THEN
        WRITE(OUTS,'(a)')'Error in GSHP Tnom + TdeadB SH >= Tin max'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Ambient temperature for temp compensation start [Deg C]
      TempCompS=BDATA(IPCOMP,22)

C Ambient temperature for temp compensation end [Deg C]
      TempCompE=BDATA(IPCOMP,23)

C Temp compensation gradient (-)
      TempCompc0=BDATA(IPCOMP,24)

C Is a lock out time defined?
      BLKOUT=NINT(BDATA(IPCOMP,25))*60

      if (BLKOUT.gt.0.) then
        bBLKOUT=.true.
      else
        bBLKOUT=.false.
      endif

      bSwitched=.false.

C GroundSourceModel [1 - standard profile 2 - user profile 3 - f(depth)]
      GroundSourceModel=NINT(BDATA(IPCOMP,26))

C GroundSourceCoeff [model 1 & 2 - profile number; model 3 - depth in m]
      GroundSourceCoeff=BDATA(IPCOMP,27)

C Nominal water return temperature DHW (Deg C)
      NomRetTDHW=BDATA(IPCOMP,28)

C Tout max DHW (degC)
      ToutMaxDHW=BDATA(IPCOMP,29)

C Tin max DHW (degC)
      TinMaxDHW=BDATA(IPCOMP,30)

C Check nominal return temperature.
      IF(NomRetTDHW.GE.TinMaxDHW)THEN
        WRITE(OUTS,'(a)')'Error in GSHP Tnom DHW >= Tin max DHW'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Check nominal water return deadband DHW (Deg C)
      IF((NomRetTDHW+0.5*(NomRetTDeadB)).GE.TinMaxDHW)THEN
        WRITE(OUTS,'(a)')
     &         'Error in GSHP Tnom DHW + 0.5 TdeadB >= Tin max DHW'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
        WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
        WRITE(ITU,*) ' 1 node (ISV=20) ground source heat pump '
        NITMS=3
        WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
        NITMS=30
        WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
        NITMS=3
        WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
        IF(ITU.EQ.IUOUT) THEN
           IX1=(IPCOMP/5)*5
           IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
        ENDIF
      ENDIF

C Check user specified number of controlled variables
      NCITM=3
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP47S warning: user specified wrong number',
     &                ' of controlled variables'

C Check for a single connection to water.
      ConError=.false.
      NCONS=1
      IPCONC=1
      DO WHILE(IPCONC.LE.MPCONC.AND..NOT.ConError)
        IPCON=ICONDX(IPCOMP,1,IPCONC)
        IF(IPCONC.LE.NCONS) THEN
          IF(IPCON.EQ.0) THEN
            ConError=.true.
          ELSE IF(ICONTP(IPCON).NE.20) THEN
            ConError=.true.
          END IF
        ELSE IF(IPCON.NE.0) THEN
          ConError=.true.
        END IF
        IPCONC=IPCONC+1
      ENDDO

C Error handling
      IF(ConError)THEN
        WRITE(IUOUT,*) 'CMP47S: connection error for component ',IPCOMP
        WRITE(IUOUT,*) '        should be ',NCONS,' water connection(s)'
        STOP 'CMP47S: unresolvable error'
      ENDIF
C
      END

C ******************** CMP48S ********************
C CMP48S establishes for a plant component with plant db code 480 ie.
C 2 node (ISV=20) WCH ground-source heat pump feeding a hydronic heating system
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine.
C
      SUBROUTINE CMP48S(IPCOMP)

      implicit none

#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      COMMON/GSHPvar1/CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &  COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,
     &  CompressPf,PumpRating,PumpPf,RatedFlow,
     &  CtlRating,CtlPf,ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,
     &  NomRetTSH,NomRetTDHW,NomRetTDeadB,NomRetTDeadBDHW,
     &  TempCompS,TempCompE,TempCompc0

      COMMON/GSHPvar2/COPModel,CompModel,AmbientTempComp,
     &                DeviceONOFFp,DeviceONOFF,GroundSourceModel,
     &                GroundSourceCoeff

      COMMON/GSHPvar3/CallforHeat,InDeadB

      COMMON/GSHPvar4/INSO,ITSBO,bSwitched,BLKOUT,NTSBO,bBLKOUT
      INTEGER INSO,ITSBO,NTSBO
      REAL BLKOUT
      LOGICAL bSwitched,bBLKOUT

      REAL SMALL
      PARAMETER (SMALL=1.0E-20)

      INTEGER COPModel,CompModel,AmbientTempComp,
     &DeviceONOFFp,DeviceONOFF,GroundSourceModel,GroundSourceCoeff

      REAL CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,CompressPf,
     &PumpRating,PumpPf,RatedFlow,CtlRating,CtlPf,
     &ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,NomRetTSH,NomRetTDHW,
     &NomRetTDeadB,NomRetTDeadBDHW,TempCompS,TempCompE,TempCompc0

      LOGICAL CallforHeat,InDeadB,ConError

      INTEGER ITC,ICNT,ITCF,ITRACE,IZNTRC,ITU,NPCOMP,NCI,
     &  ICONTP,ICONDX,J,NCITM,NCONS,IPCONC,IPCOMP,IPCON1,IX1,NITMS,
     &  NCONPN,IPCON2
      
      REAL CDATA,ADATA,BDATA,CONVAR

      INTEGER IDAT

      CHARACTER OUTS*248
      
C To be edited to include increased data entry nos. 

C Establish static data derivable from the data read from database
      DO IDAT=4,34
        BDATA(IPCOMP,IDAT-3)=ADATA(IPCOMP,IDAT)
      ENDDO

C Initialise parameters      
      InDeadB=.false.
      CallforHeat=.false.

C Assign variables to be used in coeff generator. 
C ADATA: 
C Mass of component (solids+liquid) (kg)
      CompMass=ADATA(IPCOMP,1)
C Mass weighted average specific heat node (J/kgK)
      AveSpHt=ADATA(IPCOMP,2)
C UA modulus for component (W/K)
      UAMod=ADATA(IPCOMP,3)

C BDATA: 
C COP model [1- fixed 2 - modified Carnot efficiency; 3- cubic polynomial]
      COPModel=NINT(BDATA(IPCOMP,1))

C COP empirical coefficient a0 (or fixed COP, or Eff. modifier)
      COPa0=BDATA(IPCOMP,2)

C COP empirical coefficient a1 [Ti]
      COPa1=BDATA(IPCOMP,3)

C COP empirical coefficient a2 [Ti^2]
      COPa2=BDATA(IPCOMP,4)

C COP empirical coefficient a3 [Ti^3]
      COPa3=BDATA(IPCOMP,5)

C Compressor model (W) 
      CompModel=NINT(BDATA(IPCOMP,6))

C Comp empirical coefficient a0 (or fixed COP, or Eff. modifier)
      Compa0=BDATA(IPCOMP,7)

C Comp empirical coefficient a1 
      Compa1=BDATA(IPCOMP,8)

C Comp empirical coefficient a2 
      Compa2=BDATA(IPCOMP,9)

C Comp empirical coefficient a3 
      Compa3=BDATA(IPCOMP,10)
C ****
C Compressor pf (-)
      CompressPf=BDATA(IPCOMP,11)
      IF(CompressPf.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in 2n-GSHP: comp pf is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      
      ENDIF

C Pump rating (W)
      PumpRating=BDATA(IPCOMP,12)

C Pump pf (-)
      PumpPf=BDATA(IPCOMP,13)

C Flowrate at rated pump power (l/s)
      RatedFlow=BDATA(IPCOMP,14)
      IF(RatedFlow.LT.SMALL)THEN
      WRITE(OUTS,'(a)')'Error in 2n-GSHP: rated flow is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Controller power (W)
      CtlRating=BDATA(IPCOMP,15)

C Controller power factor (W)
      CtlPf=BDATA(IPCOMP,16)
      IF(CtlPf.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in 2n-GSHP: ctl pf is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Tout max SH (degC)
      ToutMaxSH=BDATA(IPCOMP,17)

C Tin max SH (degC)
      TinMaxSH=BDATA(IPCOMP,18)

C AmbientTemp compensation on/off
      AmbientTempComp=NINT(BDATA(IPCOMP,19))

C Nominal water return temperature for SH (Deg C)
      NomRetTSH=BDATA(IPCOMP,20)
      IF(NomRetTSH.GE.TinMaxSH)THEN
        WRITE(OUTS,'(a)')'Error in 2n-GSHP Tnom => Tin max for SH'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Nominal water return deadband (Deg C)
      NomRetTDeadB=BDATA(IPCOMP,21)
      IF((NomRetTSH+0.5*(NomRetTDeadB)).GE.TinMaxSH)THEN
        WRITE(OUTS,'(a)')
     &     'Error in 2n-GSHP Tnom + TdeadB >= Tin max for SH'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Ambient temperature for temp compensation start [Deg C]
      TempCompS=BDATA(IPCOMP,22)

C Ambient temperature for temp compensation end [Deg C]
      TempCompE=BDATA(IPCOMP,23)

C Temp compensation gradient (-)
      TempCompc0=BDATA(IPCOMP,24)

C Is a lock out time defined?
      BLKOUT=NINT(BDATA(IPCOMP,25))*60

      if (BLKOUT.gt.0.) then
        bBLKOUT=.true.
      else
        bBLKOUT=.false.
      endif

      bSwitched=.false.

C Nominal water return temperature DHW (Deg C)
      NomRetTDHW=BDATA(IPCOMP,29)

C Tout max DHW (degC)
      ToutMaxDHW=BDATA(IPCOMP,30)

C Tin max DHW (degC)
      TinMaxDHW=BDATA(IPCOMP,31)

C Check nominal water return temperature DHW.
      IF(NomRetTDHW.GE.TinMaxDHW)THEN
        WRITE(OUTS,'(a)')
     &     'Error in 2n-GSHP Tnom DHW => Tin max DHW'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Nominal water return deadband DHW (Deg C)
      IF((NomRetTDHW+0.5*(NomRetTDeadB)).GE.TinMaxDHW)THEN
        WRITE(OUTS,'(a)')
     &     'Error in 2n-GSHP Tnom DHW + 0.5 TdeadB >= Tin max DHW'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV=20) ground source heat pump '
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=31
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=3
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         ENDIF
      ENDIF

C Check user specified number of controlled variables
      NCITM=3
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP48S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 2 connections only, one per node,
C both to water
      ConError=.false.
      NCONS=2
      NCONPN=1
      DO 15 IPCONC=1,MPCONC
        IPCON1=ICONDX(IPCOMP,1,IPCONC)
        IPCON2=ICONDX(IPCOMP,2,IPCONC)
        IF(IPCONC.LE.NCONPN) THEN
          IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            ConError=.true.
          ELSE IF(ICONTP(IPCON1).NE.20.OR.ICONTP(IPCON2).NE.20) THEN
            ConError=.true.
          END IF
        ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
          ConError=.true.
        END IF
   15 CONTINUE

C Error handling
      IF(ConError)THEN
        WRITE(IUOUT,*) 'CMP48S: connection error for component ',IPCOMP
        WRITE(IUOUT,*) '        should be ',NCONS,
     &                                     ' water connection(s)'
        STOP 'CMP48S: unresolvable error'
      ENDIF
C
      END


C ******************** CMP49S ********************
C CMP49S establishes for a plant component with plant db code 490 ie.
C 1 node (ISV=20) WCH inverter controlled ground-source heat pump
C feeding a hydronic heating system whether the specified number of
C controlled variables is OK, and also whether the number of connections
C to this component is correct and whether the connected nodes are of
C a type as expected by the corresponding coefficient generator routine.
      SUBROUTINE CMP49S(IPCOMP)

      implicit none

#include "plant.h"
#include "building.h"

      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      common/pcnam/pcname(mpcom)       ! Plant component names
      character*15 pcname

      COMMON/GSHPvar1/CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &  COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,
     &  CompressPf,PumpRating,PumpPf,RatedFlow,
     &  CtlRating,CtlPf,ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,
     &  NomRetTSH,NomRetTDHW,NomRetTDeadB,NomRetTDeadBDHW,
     &  TempCompS,TempCompE,TempCompc0

      COMMON/GSHPvar2/COPModel,CompModel,AmbientTempComp,
     &                DeviceONOFFp,DeviceONOFF,GroundSourceModel,
     &                GroundSourceCoeff

      COMMON/GSHPvar3/CallforHeat,InDeadB

      COMMON/GSHPvar4/INSO,ITSBO,bSwitched,BLKOUT,NTSBO,bBLKOUT
      INTEGER INSO,ITSBO,NTSBO
      REAL BLKOUT
      LOGICAL bSwitched,bBLKOUT

      COMMON/GSHPvar5/CTLmode,HP_0(6),HP_100(6),
     &                CP_0(6),CP_100(6),ToutDHWlow,NomRetTDHWlow,
     &                RatedFlowDHW

      REAL HP_0,HP_100,CP_0,CP_100,ToutDHWlow,NomRetTDHWlow,
     &  RatedFlowDHW
      INTEGER CTLmode

      REAL SMALL
      PARAMETER (SMALL=1.0E-20)

      INTEGER COPModel,CompModel,AmbientTempComp,
     &DeviceONOFFp,DeviceONOFF,GroundSourceModel,GroundSourceCoeff

      REAL CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,CompressPf,
     &PumpRating,PumpPf,RatedFlow,CtlRating,CtlPf,
     &ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,NomRetTSH,NomRetTDHW,
     &NomRetTDeadB,NomRetTDeadBDHW,TempCompS,TempCompE,TempCompc0

      LOGICAL CallforHeat,InDeadB,ConError

C----------------
C icGSHP data file commons
C----------------
C Heat pump performance data / spline function data
      common/icgshpspline/
     &          tground(MPCOM,MPCDAT),
     &          tfeed(MPCOM,MPCDAT),
     &          heat0(MPCOM,MPCDAT,MPCDAT),
     &          heat100(MPCOM,MPCDAT,MPCDAT),
     &          cool0(MPCOM,MPCDAT,MPCDAT),
     &          cool100(MPCOM,MPCDAT,MPCDAT),
     &          nx(MPCOM),ny(MPCOM),
     &          heat0_2d(MPCOM,MPCDAT,MPCDAT),
     &          heat100_2d(MPCOM,MPCDAT,MPCDAT),
     &          cool0_2d(MPCOM,MPCDAT,MPCDAT),
     &          cool100_2d(MPCOM,MPCDAT,MPCDAT),
     &          fnamHPCoefdat(MPCOM)
      real tground    ! Ground source temperature data values, degC
      real tfeed      ! Feed temperature values, degC
      real heat0      ! HP heating power at lowest converter frequency, kW
      real heat100    ! HP heating power at highest converter frequency, kW
      real cool0      ! HP cooling power at lowest converter frequency, kW
      real cool100    ! HP cooling power at highest converter frequency, kW
      real heat0_2d   ! HP heating power 0 % second-derivatives
      real heat100_2d ! HP heating power 100 % second-derivatives
      real cool0_2d   ! HP cooling power 0 % second-derivatives
      real cool100_2d ! HP cooling power 100 % second-derivatives

      integer nx,ny              ! number of data points in x and y directions

      character fnamHPCoefdat*72 ! Data file name for current icGSHP entry

C----------------
C Local variables
C----------------
      INTEGER ITC,ICNT,ITCF,ITRACE,IZNTRC,ITU,NPCOMP,NCI,
     &ICONTP,ICONDX,J,NCITM,NCONS,IPCONC,IPCOMP,IPCON,IX1,NITMS

      REAL CDATA,ADATA,BDATA,CONVAR

      INTEGER IDAT,IDT,JJ,ix,iy

      CHARACTER OUTS*248

C Establish static data derivable from the data read from database
      DO IDAT=4,51
        BDATA(IPCOMP,IDAT-3)=ADATA(IPCOMP,IDAT)
      ENDDO

C Initialise parameters
      InDeadB=.false.
      CallforHeat=.false.

C Assign variables to be used in coeff generator.
C ADATA:
C Mass of component (solids+liquid) (kg)
      CompMass=ADATA(IPCOMP,1)
C Mass weighted average specific heat node (J/kgK)
      AveSpHt=ADATA(IPCOMP,2)
C UA modulus for component (W/K)
      UAMod=ADATA(IPCOMP,3)

C BDATA:
C Control mode [1 - fx  set in GSHP; 2 - fx via CDATA]
      CTLmode=NINT(BDATA(IPCOMP,1))

C COP model [1- fixed 2 - HP/(HP-CP)]
      COPModel=NINT(BDATA(IPCOMP,2))

C Set IDT to number of previous parameters for easier adaptation of
C the reading of following parameters.
      IDT=2

      if ((COPmodel.eq.1).or.(COPmodel.eq.2)) then
C       Coefficients for 0 % and 100 % compressor settings, heating and
C       cooling power.
        do 15 JJ=1,6
          IDT=IDT+1
          HP_0(JJ)   = BDATA(IPCOMP,IDT)
          HP_100(JJ) = BDATA(IPCOMP,IDT+6)
          CP_0(JJ)   = BDATA(IPCOMP,IDT+12)
          CP_100(JJ) = BDATA(IPCOMP,IDT+18)
  15    continue

C       Set IDT to last polynom data item (BDATA item 26).
        IDT=IDT+18

      ELSEIF((COPmodel.eq.3).or.(COPmodel.eq.5)) then
C       A file <plantcomponentname> with the extension .hpd is expected
C       to exist in the same subdirectory where the plant definition
C       file resides.
        write(fnamHPCoefdat(ipcomp),'(3a)')
     &     '../nets/',pcname(ipcomp)(1:lnblnk(pcname(ipcomp))),'.hpd'

cx        write(*,*)'HPDatfile= ',
cx     &          fnamHPCoefdat(ipcomp)(1:lnblnk(fnamHPCoefdat(ipcomp)))

C       Read data file and fill commons for performance data.
        call ReadHPCoefDataFile(IPCOMP)

C       Calculate 2nd derivatives for spline functions.
C       Heating data
        call splie2(tground(IPCOMP,:),
     &              tfeed(IPCOMP,:),
     &              heat0(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &              nx(IPCOMP), ny(IPCOMP),
     &              heat0_2d(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)))

        call splie2(tground(IPCOMP,:),
     &              tfeed(IPCOMP,:),
     &              heat100(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &              nx(IPCOMP), ny(IPCOMP),
     &              heat100_2d(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)))

C       Cooling data
        call splie2(tground(IPCOMP,:),
     &              tfeed(IPCOMP,:),
     &              cool0(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &              nx(IPCOMP), ny(IPCOMP),
     &              cool0_2d(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)))

        call splie2(tground(IPCOMP,:),
     &              tfeed(IPCOMP,:),
     &              cool100(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &              nx(IPCOMP), ny(IPCOMP),
     &              cool100_2d(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)))

C Output 2nd derivative matrices for checking purposes.
        write(IUOUT,*)' '
        write(IUOUT,*)
     &  '***************** 2nd Derivative Matrices ******************'
        write(IUOUT,*)' '
        write(IUOUT,*)'2nd d. heating power at 0 %:'
        write(IUOUT,'(a,7(f7.3,1x))')'        ',
     &                               (tfeed(IPCOMP,iy),iy=1,7)
        DO 40 ix=1,nx(ipcomp)
            write(IUOUT,'(f7.3,1x,7(f7.3,1x))')tground(IPCOMP,ix),
     &                             (heat0_2d(IPCOMP,ix,iy),iy=1,7)
 40     CONTINUE

        write(IUOUT,*)' '
        write(IUOUT,*)'2nd d. heating power at 100 %:'
        write(IUOUT,'(a,7(f7.3,1x))')'        ',
     &                               (tfeed(IPCOMP,iy),iy=1,7)
        DO 50 ix=1,nx(ipcomp)
            write(IUOUT,'(f7.3,1x,7(f7.3,1x))')tground(IPCOMP,ix),
     &                             (heat100_2d(IPCOMP,ix,iy),iy=1,7)
 50     CONTINUE

        write(IUOUT,*)' '
        write(IUOUT,*)'2nd d. cooling power at 0 %:'
        write(IUOUT,'(a,7(f7.3,1x))')'        ',
     &                               (tfeed(IPCOMP,iy),iy=1,7)
        DO 60 ix=1,nx(ipcomp)
            write(IUOUT,'(f7.3,1x,7(f7.3,1x))')tground(IPCOMP,ix),
     &                               (cool0_2d(IPCOMP,ix,iy),iy=1,7)
 60     CONTINUE

        write(IUOUT,*)' '
        write(IUOUT,*)'2nd d. cooling power at 100 %:'
        write(IUOUT,'(a,7(f7.3,1x))')'        ',
     &                               (tfeed(IPCOMP,iy),iy=1,7)
        DO 70 ix=1,nx(ipcomp)
            write(IUOUT,'(f7.3,1x,7(f7.3,1x))')tground(IPCOMP,ix),
     &                               (cool100_2d(IPCOMP,ix,iy),iy=1,7)
 70     CONTINUE
        write(IUOUT,*)' '
        write(IUOUT,*)
     &  '************************************************************'
        write(IUOUT,*)' '

C Preprocessing complete. Leave binary file open for use during time-step simulation.
        call usrmsg(' Preprocessing icGSHP data (splie2()) done.',
     &            ' ','-')
        write(IUOUT,*)' '

C       Set IDT to last polynom data item (BDATA item 26).
        IDT=26

      ELSEIF(COPmodel.eq.4) then
C       A file <plantcomponentname> with the extension .hpd is expected
C       to exist in the same subdirectory where the plant definition
C       file resides.
        write(fnamHPCoefdat(ipcomp),'(3a)')
     &     '../nets/',pcname(ipcomp)(1:lnblnk(pcname(ipcomp))),'.hpd'

C       Read data file and fill commons for performance data.
        call ReadHPCoefDataFile(IPCOMP)

C       Set IDT to last polynom data item (BDATA item 26).
        IDT=26
      ENDIF

C Pump rating (W)
      IDT=IDT+1
      PumpRating=BDATA(IPCOMP,IDT)

C Pump pf (-)
      IDT=IDT+1
      PumpPf=BDATA(IPCOMP,IDT)

C Flowrate at rated pump power SH (l/s)
      IDT=IDT+1
      RatedFlow=BDATA(IPCOMP,IDT)
      IF(RatedFlow.LT.SMALL)THEN
        WRITE(OUTS,'(a)')
     &             'Error in icGSHP: rated flow is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Flowrate DHW (l/s)
      IDT=IDT+1
      RatedFlowDHW=BDATA(IPCOMP,IDT)
      IF(RatedFlowDHW.LT.SMALL)THEN
        WRITE(OUTS,'(a)')
     &             'Error in icGSHP: rated DHW flow is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Controller power (W)
      IDT=IDT+1 ! 30 (33)
      CtlRating=BDATA(IPCOMP,IDT)

C Controller power factor (W)
      IDT=IDT+1
      CtlPf=BDATA(IPCOMP,IDT)
      IF(CtlPf.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in icGSHP: ctl pf is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Compressor pf (-)
      IDT=IDT+1
      CompressPf=BDATA(IPCOMP,IDT)
      IF(CompressPf.LT.SMALL)THEN
        WRITE(OUTS,'(a)')'Error in icGSHP: comp pf is close to zero!'
        CALL EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Tout SH (degC)
      IDT=IDT+1
      ToutMaxSH=BDATA(IPCOMP,IDT)

C Nominal water return temperature SH (Deg C)
      IDT=IDT+1
      NomRetTSH=BDATA(IPCOMP,IDT)

      IF(NomRetTSH.GT.ToutMaxSH)THEN
        WRITE(OUTS,'(a)')
     &          'Error in icGSHP nom ret T SH > Tout for SH'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Nominal return deadband SH (Deg C)
      IDT=IDT+1 ! 35 (38)
      NomRetTDeadB=BDATA(IPCOMP,IDT)

C Check nominal water return deadband SH (Deg C)
      IF((NomRetTSH+0.5*(NomRetTDeadB)).GT.ToutMaxSH)THEN
        WRITE(OUTS,'(a)')
     &     'Error in icGSHP nom ret T SH + 0.5 TdeadB > Tout SH'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C AmbientTemp compensation on/off
      IDT=IDT+1
      AmbientTempComp=NINT(BDATA(IPCOMP,IDT))

C Low amb. air temperature for temp compensation [Deg C]
      IDT=IDT+1
      TempCompS=BDATA(IPCOMP,IDT)

C High amb. air temperature for temp compensation [Deg C]
      IDT=IDT+1
      TempCompE=BDATA(IPCOMP,IDT)

C Temp compensation gradient (-)
      IDT=IDT+1
      TempCompc0=BDATA(IPCOMP,IDT)

C =================================
C Domestic hot water (DHW) settings
C Tout DHW high (degC)
      IDT=IDT+1 ! 40 (43)
      ToutMaxDHW=BDATA(IPCOMP,IDT)

C Nominal water return temperature DHW (Deg C)
      IDT=IDT+1
      NomRetTDHW=BDATA(IPCOMP,IDT)

      IF(NomRetTDHW.GT.ToutMaxDHW)THEN
        WRITE(OUTS,'(a)')'Error in icGSHP nom ret T DHW > Tout DHW'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Tout DHW low (degC)
      IDT=IDT+1
      ToutDHWlow=BDATA(IPCOMP,IDT)

C Nominal water return temperature DHW (Deg C)
      IDT=IDT+1
      NomRetTDHWlow=BDATA(IPCOMP,IDT)

      IF(NomRetTDHWlow.GT.ToutDHWlow)THEN
        WRITE(OUTS,'(a)')
     &         'Error in icGSHP nom ret T DHW > Tout DHW (low)'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF
C Nominal water return deadband DHW (Deg C)
      IDT=IDT+1
      NomRetTDeadBDHW=BDATA(IPCOMP,IDT)

C Check nominal water return deadband DHW, high (Deg C)
      IF((NomRetTDHW+0.5*(NomRetTDeadBDHW)).GT.ToutMaxDHW)THEN
        WRITE(OUTS,'(a)')
     &      'Error in icGSHP nom ret T DHW + 0.5 TdeadB > Tin DHW'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Check nominal water return deadband DHW, low (Deg C)
      IF((NomRetTDHWlow+0.5*(NomRetTDeadBDHW)).GT.ToutDHWlow)THEN
        WRITE(OUTS,'(a)')
     &     'Error in icGSHP nom ret T DHW + 0.5 TdeadB > Tin DHW (low)'
        call EDISP(IUOUT,OUTS)
        STOP
      ENDIF

C Is a lock out time defined (value is interpreted as minutes)?
      IDT=IDT+1 ! 45 (48)
      BLKOUT=NINT(BDATA(IPCOMP,IDT))*60

      if (BLKOUT.gt.0.) then
        bBLKOUT=.true.
      else
        bBLKOUT=.false.
      endif

      bSwitched=.false.

C GroundSourceModel [1- fixed 2 - HP/(HP-CP)]
      IDT=IDT+1
      GroundSourceModel=NINT(BDATA(IPCOMP,IDT))

C GroundSourceCoeff [model 1 & 2 - profile number; model 3 - depth in m]
      IDT=IDT+1 ! 47 (50)
      GroundSourceCoeff=NINT(BDATA(IPCOMP,IDT))

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
        WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
        WRITE(ITU,*) ' 1 node (ISV=20) inverter controlled GSHP '
        NITMS=3
        WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
        NITMS=30
        WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
        NITMS=3
        WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
        IF(ITU.EQ.IUOUT) THEN
           IX1=(IPCOMP/5)*5
           IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
        ENDIF
      ENDIF

C Check user specified number of controlled variables
      NCITM=3
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP49S warning: wrong number of controlled',
     &                ' variables specified: ',NCI(IPCOMP),
     &                ' instead of ',NCITM,'.'

C Check for a single connection to water.
      ConError=.false.
      NCONS=1
      IPCONC=1
      DO WHILE(IPCONC.LE.MPCONC.AND..NOT.ConError)
        IPCON=ICONDX(IPCOMP,1,IPCONC)
        IF(IPCONC.LE.NCONS) THEN
          IF(IPCON.EQ.0) THEN
            ConError=.true.
          ELSE IF(ICONTP(IPCON).NE.20) THEN
            ConError=.true.
          END IF
        ELSE IF(IPCON.NE.0) THEN
          ConError=.true.
        END IF
        IPCONC=IPCONC+1
      ENDDO

C Error handling
      IF(ConError)THEN
        WRITE(IUOUT,*) 'CMP49S: connection error for component ',IPCOMP
        WRITE(IUOUT,*) '        should be ',NCONS,' water connection(s)'
        STOP 'CMP49S: unresolvable error'
      ENDIF
C
      END

C ******************** CMP50S ********************
C
C CMP50S establishes for a plant component with plant db code 500 ie.
C 1 node (ISV=29) WCH thermostatic valve
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE CMP50S(IPCOMP)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,3)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,8)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=29) WCH thermostatic valve'
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=6
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP50S warning: user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 2 connections
      NCONS=2
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
C
C
      RETURN
C
C Error handling
  990 WRITE(IUOUT,*) ' CMP50S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      STOP ' CMP50S: unresolvable error'
C
      END
C
C ******************** CMP51S ********************
C
C CMP51S establishes for a plant component with plant db code 510 ie.
C 1 node (ISV=29)     mechanical room thermostat
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE CMP51S(IPCOMP)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,3)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,8)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=29) mechanical room thermostat'
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=6
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP51S warning: user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 0 connections
      NCONS=0
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
C
C
      RETURN
C
C Error handling
  990 WRITE(IUOUT,*) ' CMP51S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      STOP ' CMP51S: unresolvable error'
C
      END

C ******************** CMP52S ********************
C
C CMP52S establishes for a plant component with plant db code 520 ie.
C 1 node (ISV=29) WCH thermostatic valve (sensor only i.e. 0 connections)
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE CMP52S(IPCOMP)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,3)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,5)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,6)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,7)=ADATA(IPCOMP,9)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=29) WCH thermostatic valve'
         NITMS=2
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=6
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP52S warning: user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 0 connections
      NCONS=0
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
C
C
      RETURN
C
C Error handling
  990 WRITE(IUOUT,*) ' CMP52S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      STOP ' CMP52S: unresolvable error'
C
      END
C
C ******************** CMP67S ********************
C
C CMP67S establishes for a plant component with plant db code 670 ie.
C 3 node (ISV>19) AC  heat transfer tube fed by WCH system
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
C node 1 represents solid material (ISV=29)
C node 2 represents air            (ISV=21)
C node 3 represents water          (ISV=20)
C
      SUBROUTINE CMP67S(IPCOMP)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,8)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 3 node (ISV>19) AC  heat transfer tube',
     &                ' fed by WCH system'
         NITMS=4
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=4
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP67S warning: user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 1st connection to "moist air"
C                     2nd connection to water
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,2,IPCONC)
      IPCON2=ICONDX(IPCOMP,3,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.21.OR.ICONTP(IPCON2).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
C
C
      RETURN
C
C Error handling
  990 WRITE(IUOUT,*) ' CMP67S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie  1st connection to air'
      WRITE(IUOUT,*) '             2nd connection to water'
      STOP ' CMP67S: unresolvable error'
C
      END
C
C ******************** CMP70S ********************
C
C CMP70S establishes for a plant component with plant db code 700 ie.
C 1 node (ISV=20) Flat plate solar collector (type1).
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE CMP70S(IPCOMP)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1) =ADATA(IPCOMP,4)
      BDATA(IPCOMP,2) =ADATA(IPCOMP,5)
      BDATA(IPCOMP,3) =ADATA(IPCOMP,6)
      BDATA(IPCOMP,4) =ADATA(IPCOMP,7)
      BDATA(IPCOMP,5) =ADATA(IPCOMP,8)
      BDATA(IPCOMP,6) =ADATA(IPCOMP,9)
      BDATA(IPCOMP,7) =ADATA(IPCOMP,10)
      BDATA(IPCOMP,8) =ADATA(IPCOMP,11)
      BDATA(IPCOMP,9) =ADATA(IPCOMP,12)
      BDATA(IPCOMP,10)=ADATA(IPCOMP,13)
      BDATA(IPCOMP,11)=ADATA(IPCOMP,14)
      BDATA(IPCOMP,12)=ADATA(IPCOMP,15)
      BDATA(IPCOMP,13)=ADATA(IPCOMP,16)
      BDATA(IPCOMP,14)=ADATA(IPCOMP,17)
      BDATA(IPCOMP,15)=ADATA(IPCOMP,18)
      BDATA(IPCOMP,16)=ADATA(IPCOMP,19)
      BDATA(IPCOMP,17)=ADATA(IPCOMP,20)
      BDATA(IPCOMP,18)=ADATA(IPCOMP,21)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) Flat-plate solar collector'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=18
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP70S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connections
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP70S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      STOP ' CMP70S: unresolvable error'
      END

C ******************** CMP71S ********************
C
C CMP71S establishes for a plant component with plant db code 710 ie.
C 1 node (ISV=20) solar collector "segment".
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE CMP71S(IPCOMP)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)


C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) Flat-plate solar collector'
         NITMS=22
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP71S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connections
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP71S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      STOP ' CMP71S: unresolvable error'
      END


C ******************** CMP72S ********************
C
C CMP72S establishes for plant component with plant db code 720 i.e
C 1 node (ISV=20) water storage tank with converging inlets
C layer. Checks are made on the number controlled variables
C the number of connections and whether the connected nodes are of the
C correct type as expected by the plant solver.
C

      SUBROUTINE CMP72S(IPCOMP)

#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node water storage tank layer.'
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C Check component has 2 connections to water
      NCONS=2
      NOCON=0
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
C Allow this as there can be 2 or 1 connections.
           NOCON=NOCON+1
C Check that there is at least ONE connection,
C If there is then NOCON should never be 2.
           IF(NOCON.EQ.2) THEN
              GOTO 990
           ENDIF
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
C
C
      RETURN
C
C Error handling
  990 WRITE(IUOUT,*) ' CMP72S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be 1 or  2  connection(s)'
      WRITE(IUOUT,*) '         to water.'
      STOP ' CMP72S: unresolvable error'
C
      END

C ******************** CMP73S ********************
C
C CMP73S establishes for plant component with plant db code 730 i.e
C 3 node (ISV=20) 3-port valve. Checks are made on the number controlled 
C variables the number of connections and whether the connected nodes are 
C of the correct type as expected by the plant solver.
C

      SUBROUTINE CMP73S(IPCOMP)

#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 3 node 3-port valve.'
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check component has 1 connections to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
           GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
C
C
      RETURN
C
C Error handling
  990 WRITE(IUOUT,*) ' CMP73S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be 1 connection(s)'
      WRITE(IUOUT,*) '         to water.'
      STOP ' CMP73S: unresolvable error'
C
      END

C ******************** CMP74S ********************
C
C CMP74S establishes for plant component with plant db code 740 i.e
C 3 node (ISV=20) 3-port valve including automatic flow control.
C Checks are made on the number controlled variables the number of
C connections and whether the connected nodes are of the correct
C type as expected by the plant solver.
C

      SUBROUTINE CMP74S(IPCOMP)

#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 3 node 3-port valve w/ automatic flow cnt.'
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check component has 1 connections to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
           GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN
C Error handling
  990 WRITE(IUOUT,*) ' CMP74S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be 1 connection(s)'
      WRITE(IUOUT,*) '         to water.'
      STOP ' CMP74S: unresolvable error'
      END

C ******************** CMP79S ********************
C
C CMP79S establishes for a plant component with plant db code 790 ie.
C 1 node (ISV=20) Stochastic DHW flow calculation.
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE CMP79S(IPCOMP)

      implicit none
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)


      COMMON/DHWStoch1/DHWsubmin,CalCount,DHWCalcInc,
     &    GroundSourceModel,GroundSourceCoeff

      Common/DHWStoch3R/NomDraw,SeasDrawFluc,DrawPerS(6),NomDrawFrac(6),
     &NomTypeFrac(6),DrawV(6),DrawVstd(6),DrawWdvar(6),DrawWevar(6)
      Common/DHWStoch3I/SeasPhas,NoHolPer,HolPerS(3),HolPerE(3),
     &NoDrawPer,NDrawType,NomDrawDur(6)

      Common/DHWStoch4/CurDrawDur(6),ElapseDrawDur(6),DHWDraw(6)
      Common/DHWStochL/inDraw(6)


      logical conError,InDraw,DHWsubmin 

      real NomDraw,SeasDrawFluc,DrawPerS,NomDrawFrac,NomTypeFrac,
     &DrawV,DrawVstd,NomDrawDur,DrawWdvar,DrawWevar,CurDrawDur,
     &ElapseDrawDur,DHWDraw
      integer SeasPhas,NoHolPer,HolPerS,HolPerE,NoDrawPer,NDrawType,
     &CalCount,DHWCalcInc,GroundSourceModel,GroundSourceCoeff
      integer IDT,IDp,NSTAT,NBDAT1,NBDAT2,NBDAT3,NBDAT

      integer ITC,ICNT,ITCF,ITRACE,IZNTRC,ITU
      integer ICONTP,ICONDX
      integer IPCOMP,IPCON,IPCONC,NCITM,NCONS,NITMS
      integer IX1,IDW,IHP,J,NCI,NPCOMP,DHWDrawSeed,Iseed
      real ADATA,BDATA,CDATA,CONVAR
     

C Establish static data derivable from the data read from database
      NSTAT=3
      NBDAT1=1+NSTAT+nint(ADATA(IPCOMP,NSTAT+1))*2
      NBDAT2=1+NBDAT1+nint(ADATA(IPCOMP,NBDAT1+1))*2
      NBDAT3=1+NBDAT2+nint(ADATA(IPCOMP,NBDAT2+1))*6
      NBDAT=NBDAT3+1+2

C Initiate calculation counter for sub 60s timesteps
      calcount=1

C Trace
c      write(87,*)'DHW: static set up'
c      write(87,*)'NSTAT, NBDAT1, NBDAT2, NBDAT3', 
c     &NSTAT,NBDAT1,NBDAT2,NBDAT3,NBDAT

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) stochastic DHW draw'
         NITMS=NBDAT
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)

         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      NomDraw=ADATA(IPCOMP,1)
      IF(NomDraw.LT.1E-6) 
     &WRITE(IUOUT,*) 'CMP79S WARNING. DHW: nominal draw is zero!'
      SeasDrawFluc=ADATA(IPCOMP,2)
      SeasPhas=nint(ADATA(IPCOMP,3))

      NoHolPer=nint(ADATA(IPCOMP,4))
      IDT=5
      if(NoHolPer.GT.0)THEN
        DO 10 IHp=1,NoHolPer
          HolPerS(IHp)=nint(ADATA(IPCOMP,IDT))
          IDT=IDT+1
          HolPerE(IHp)=nint(ADATA(IPCOMP,IDT))
          IDT=IDT+1
 10     CONTINUE 
      endif

      NoDrawPer=nint(ADATA(IPCOMP,IDT))
      IDT=IDT+1   
      IF(NoDrawPer.GT.0)THEN 
        DO 15 IDp=1,NoDrawPer
          DrawPerS(IDp)=ADATA(IPCOMP,IDT)
          IF(IDp.GT.1)THEN
            if(DrawPerS(IDp).LE.DrawPerS(IDp-1))THEN
              WRITE(IUOUT,*) 'CMP79S ERROR. DHW: non-concurrent draw'
              WRITE(IUOUT,*) 'period starts defined.Check stochastic' 
              WRITE(IUOUT,*) 'DHW draw component data!' 
              STOP
            endif
          ELSE
            if(abs(DrawPerS(IDp)-0.0).GT.1.0E-6)then
              WRITE(IUOUT,*) 'CMP79S ERROR. DHW: draw start time'
              WRITE(IUOUT,*) 'should be 0.0. Check stochastic'
              WRITE(IUOUT,*) 'DHW draw component data!' 
            endif
          ENDIF
          IDT=IDT+1
          NomDrawFrac(IDp)=ADATA(IPCOMP,IDT)
          IDT=IDT+1
 15     CONTINUE
      ELSE
         WRITE(IUOUT,*) 'CMP79S WARNING. DHW: no draw periods defined '
         WRITE(IUOUT,*) 'check stochastic DHW draw component data!' 
      ENDIF   
   
      NDrawType=nint(ADATA(IPCOMP,IDT))
      IDT=IDT+1 
      IF(NDrawType.GT.0)THEN
      DO 20 IDw=1,NDrawType
        indraw(IDw)=.false.
        CurDrawDur(IDw)=0.0
        DHWDraw(IDw)=0.0
        NomTypeFrac(IDw)=ADATA(IPCOMP,IDT)
        IDT=IDT+1        
        DrawV(IDw)=ADATA(IPCOMP,IDT)
        IDT=IDT+1              
        DrawVstd(IDw)=ADATA(IPCOMP,IDT)
        IDT=IDT+1        
        NomDrawDur(IDw)=ADATA(IPCOMP,IDT)
        IDT=IDT+1         
        DrawWdvar(IDw)=ADATA(IPCOMP,IDT)
        IDT=IDT+1          
        DrawWevar(IDw)=ADATA(IPCOMP,IDT)
        IDT=IDT+1          
 20   CONTINUE
      ELSE
         WRITE(IUOUT,*) 'CMP79S WARNING. DHW: no draw types defined '
         WRITE(IUOUT,*) 'check stochastic DHW draw component data!'
      ENDIF

      DHWDrawSeed=NINT(ADATA(IPCOMP,IDT))

C Seed the random number generator
      ISEED=DHWDrawSeed
      CALL SRAND(ISEED)

C GroundSourceModel [1- fixed 2 - HP/(HP-CP)]
      IDT=IDT+1
      GroundSourceModel=NINT(ADATA(IPCOMP,IDT))

C GroundSourceCoeff [model 1 & 2 - profile number; model 3 - depth in m]
      IDT=IDT+1
      GroundSourceCoeff=NINT(ADATA(IPCOMP,IDT))

C
C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(IUOUT,*) ' CMP79S warning: user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 1 connection only, to water
      ConError=.false.
      NCONS=1
      IPCONC=1
      DO WHILE(IPCONC.LE.MPCONC.AND..NOT.ConError)
        IPCON=ICONDX(IPCOMP,1,IPCONC)
        IF(IPCONC.LE.NCONS) THEN
          IF(IPCON.EQ.0) THEN
            ConError=.true.
          ELSE IF(ICONTP(IPCON).NE.20) THEN
            ConError=.true.
          END IF
        ELSE IF(IPCON.NE.0) THEN
          ConError=.true.
        END IF
        IPCONC=IPCONC+1
      ENDDO

C Error handling
      IF(ConError)THEN
        WRITE(IUOUT,*) 'CMP79S: connection error for component ',IPCOMP
        WRITE(IUOUT,*) '        should be ',NCONS,' water connection(s)'
        STOP 'CMP79S: unresolvable error'
      ENDIF

      RETURN

      END

C ******************** CMP90S ********************
C
C CMP90S establishes for a plant component with plant db code 900 ie.
C 2 node (ISV>19)     air & water temperature source
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE CMP90S(IPCOMP)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV>19) air & water temperature source'
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=2
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP90S warning: user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 1st connection to "moist air"
C                     2nd connection to water
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IPCON2=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.21.OR.ICONTP(IPCON2).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
C
C
      RETURN
C
C Error handling
  990 WRITE(IUOUT,*) ' CMP90S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie  1st connection to air'
      WRITE(IUOUT,*) '             2nd connection to water'
      STOP ' CMP90S: unresolvable error'
C
      END
C
C ******************** CMP91S ********************
C
C CMP91S establishes for a plant component with plant db code 80 ie.
C 2 node (ISV>19) imaginary building-like plant load
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine.
c
c node 1 is external wall
c node 2 is zone air
C
      subroutine cmp91s(ipcomp)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,8)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 2 node (ISV>19) imaginary building-like plant',
     &                ' load'
         NITMS=5
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=4
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP91S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(IPCONC.EQ.1.AND.ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP91S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be 1 connection to air'
      STOP ' CMP91S: unresolvable error'
      END

C ******************** CMP92S ********************
C
C CMP92S establishes for a plant component with plant db code 920 ie.
C 1 node (ISV=21) AC  fictitious boundary component.
C There are no static manufacturers data for this component as it
C acts to establish mass flow rate at a boundary connection.
C Checks are performed to see whether the number of connections to this
C component is correct and whether the connected nodes are of a type as
C expected by the corresponding coefficient generator routine.
C
      SUBROUTINE CMP92S(IPCOMP)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) fictitious boundary component'
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP92S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP92S: unresolvable error'
      END

C ******************** CMP93S ********************
C
C CMP93S establishes for a plant component with plant db code 930 ie.
C 10 node (ISV>20) heat exchanger component.
C
C Nodes 1 to 5 are cold fluid nodes.
C Nodes 6 to 10 are solid nodes.
C Connection to this component is via nodes 1 and 5.
      SUBROUTINE CMP93S(IPCOMP)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV>20) Heat exchanger component'
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,7)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,8)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,9)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 10 node (ISV>21) heat exchanger'
         NITMS=5
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=4
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP93S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP93S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP93S: unresolvable error'
      END

C***************** CMP94S ********************
C
C CMP94S establishes for a plant component with plant db code 940 ie.
C 3 node (ISV=20) heat exchanger segment whether the correct number
C of control variables have been specified, whether the number of
C connections is correct and whether the connections are to the
C correct fluid type.
       SUBROUTINE CMP94S(IPCOMP)
#include "plant.h"
#include "building.h"
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 3 node (ISV=20) heat exchanger segment'
         NITMS=13
         WRITE (ITU,*)' ADATA ',(ADATA(IPCOMP,J), J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check component has 1st connection to water
C                     2nd connection to water
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IPCON2=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.20.OR.ICONTP(IPCON2).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP94S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie  1st connection to water'
      WRITE(IUOUT,*) '             2nd connection to water'
      STOP ' CMP94S: unresolvable error'
      END

C***************** CMP98S ********************
C
C CMP98S establishes for a plant component with plant db code 980 ie.
C 3 node (ISV>19) CHP engine model whether the correct number
C of control variables have been specified, whether the number of
C connections is correct and whether the connections are to the
C correct fluid type.
       SUBROUTINE CMP98S(IPCOMP)
#include "plant.h"
#include "building.h"
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) CHP engine component'
         NITMS=10
         WRITE (ITU,*)' ADATA ',(ADATA(IPCOMP,J), J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Fill bdata array
      NITMS=9
      DO 5 I=1,NITMS-3
          BDATA(ipcomp,I)=ADATA(ipcomp,I+3)
  5   CONTINUE

C Check component has 1st connection to water.
C
      NCONS=1
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP98S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s)'
      WRITE(IUOUT,*) '         ie connection to water'
      WRITE(IUOUT,*) '             '
      STOP ' CMP98S: unresolvable error'
      END
C ******************** CMP99S ********************
C
C CMP99S establishes for a plant component with plant db code 990 ie.
C a 3 node CHP engine unit model:
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE CMP99S(IPCOMP)
#include "plant.h"
#include "building.h"
C
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Fill the BDATA array
      DO 5 I=1,18
         BDATA(IPCOMP,I)=ADATA(IPCOMP,5+I)
  5   CONTINUE

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 3 node (ISV=21) Small scale CHP gas engine unit'
         NITMS=5
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=18
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP99S warning: user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 1st connection to water
C                     2nd connection to air
      NCONS=2
      NCONPN=1
      DO 10 IPCONC=1,MPCONC
      IPCON1=ICONDX(IPCOMP,1,IPCONC)
      IPCON2=ICONDX(IPCOMP,2,IPCONC)
      IF(IPCONC.LE.NCONPN) THEN
         IF(IPCON1.EQ.0.OR.IPCON2.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON1).NE.20.OR.ICONTP(IPCON2).NE.21) THEN
            write(*,*) ICONTP(IPCON1), ICONTP(IPCON2)
            GOTO 990
         END IF
      ELSE IF(IPCON1.NE.0.OR.IPCON2.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP99S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' 1st to water 2nd to'
      WRITE(IUOUT,*) '         ambient.'
      STOP ' CMP99S: unresolvable error'
      END

C ******************** CMP144S ********************
C Establishes for a plant component with plant db code 144 ie.
C 1 node (ISV=21) AC  cooling coil & ON OFF control
C data derivable from the static manufacturers data already read from
C the database and saves it in the static data common block, and also
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP144S(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      COMMON/C13PS/NPMCOE,NPNOD,NPMTYP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      COMMON/ATPRES/PATMOS
      COMMON/WBULBO/IOPT
      
      REAL :: fRatio

C First check that plant simulation is TYPE 3 (Energy+2 phase).
      IF(NPMTYP .NE. 3)THEN
         WRITE(IUOUT,*) ' CMP144S: Plant simulation type ',NPMTYP,
     &    ' is not supported for this component.'
         WRITE(IUOUT,*) ' Must be Type 3 (Energy + 2 phase).'
         STOP ' CMP144S: unresolvable error'
      ENDIF

C Establish static data derivable from the data read from database.
      NITMS=23
      DO 5 I=1,NITMS-3
          BDATA(IPCOMP,I)=ADATA(IPCOMP,I+3)
  5   CONTINUE

C Validate the static data.
      IF(BDATA(IPCOMP,1)<= 0.) THEN
          WRITE(IUOUT,*) ' CMP144S: Capacity ',BDATA(IPCOMP,1),
     &                '. Must be greater than zero.'
          STOP ' CMP144S: unresolvable error'
      ELSEIF(BDATA(IPCOMP,2)<= 0. .OR.BDATA(IPCOMP,2)> 1.)THEN
          WRITE(IUOUT,*) ' CMP144S: SHR ',BDATA(IPCOMP,2),
     &                ' is outside valid range of 0 to 1.'
          STOP ' CMP144S: unresolvable error'
      ELSEIF(BDATA(IPCOMP,3)<= 0.)THEN
          WRITE(IUOUT,*) ' CMP144S: COP ',BDATA(IPCOMP,3),
     &                ' cannot be less than or equal to zero.'
          STOP ' CMP144S: unresolvable error'
      ELSEIF(BDATA(IPCOMP,4)<= 0.)THEN
          WRITE(IUOUT,*) ' CMP144S: Rated flow ',BDATA(IPCOMP,4),
     &                ' cannot be less than or equal to zero.'
          STOP ' CMP144S: unresolvable error'
      ENDIF
C Check the validity of the curves
      CALL validateCMP144Sinput(IPCOMP)
C Check the rated flow versus capacity ratio
      fRatio=BDATA(IPCOMP,4)/BDATA(IPCOMP,1)
      IF(fRatio<0.00004027 .OR. fRatio>0.00006041) ! From Coil:Cooling:DX:SingleSpeed documentation
     &    WRITE(IUOUT,*) ' CMP144S warning: flow to capacity ',
     &            'ratio outside typical range. ', fRatio

C Determine the rated bypass factor and coil geometry parameter
      CALL getBypassFactorCMP144(IPCOMP)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=21) AC  cooling coil &',
     &                ' empirical performance'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=20
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP144S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to "moist air"
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.21) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP144S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP144S: unresolvable error'

      END
      
      SUBROUTINE validateCMP144Sinput(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/OUTIN/IUOUT,IUIN,IEOUT

      INTEGER, INTENT(IN) :: IPCOMP
      REAL :: fFactor
      LOGICAL :: CLOSEA
      REAL, PARAMETER :: twb = 19.4, tdb = 35.0 ! AHRI test conditions

C Check Total cooling capacity modifier curve
      fFactor=BDATA(IPCOMP,5)+(BDATA(IPCOMP,6)*twb)+
     &   (BDATA(IPCOMP,7)*(twb**2.0))+(BDATA(IPCOMP,8)*tdb)+
     &   (BDATA(IPCOMP,9)*(tdb**2.0))+(BDATA(IPCOMP,10)*1.)+
     &   (BDATA(IPCOMP,11)*1.)+(BDATA(IPCOMP,12)*twb*tdb)
      CALL ECLOSE(fFactor,1.,0.05,CLOSEA)
      IF(.NOT.CLOSEA)
     &   WRITE(IUOUT,*) ' CMP144S warning: Total cooling capacity',
     &       ' modifier does not equal 1 at test conditions. ',fFactor

C Check Energy input ratio modifier curve
      fFactor=BDATA(IPCOMP,13)+(BDATA(IPCOMP,14)*twb)+
     &   (BDATA(IPCOMP,15)*(twb**2.0))+(BDATA(IPCOMP,16)*tdb)+
     &   (BDATA(IPCOMP,17)*(tdb**2.0))+(BDATA(IPCOMP,18)*1.)+
     &   (BDATA(IPCOMP,19)*1.)+(BDATA(IPCOMP,20)*twb*tdb)
      CALL ECLOSE(fFactor,1.,0.05,CLOSEA)
      IF(.NOT.CLOSEA)
     &   WRITE(IUOUT,*) ' CMP144S warning: Energy input ratio',
     &       ' modifier does not equal 1 at test conditions. ',fFactor
      
      END


      SUBROUTINE getBypassFactorCMP144(IPCOMP)
#include "plant.h"
#include "building.h"

      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/OUTIN/IUOUT,IUIN,IEOUT

      INTEGER, INTENT(IN) :: IPCOMP
      REAL :: fSlopeRated
      REAL :: fSHRrated, fQtotRated, fQsensRated, fQlatRated
      REAL :: fFlowRated, fCpIn, fCpOut, fEnthIn, fEnthOut
      REAL :: fEnthADP
      REAL :: fTdbOut   ! Coil outlet drybulb temperature [oC]
      REAL :: fRHOut    ! Coil outlet relative humidity
      REAL :: fWOut     ! humidity ratio of outlet air [kg_w/kg_a]
      REAL :: fWadp     ! humidity ratio at apparatus dewpoint [kg_w/kg_a]
      REAL :: fTadp     ! Drybulb temperature at apparatus dewpoint [oC]
      REAL :: fBFrated     ! Bypass factor at rated conditions
      REAL :: fAo   ! Coil geometry parameter
      REAL :: fHTinWout   ! Coil geometry parameter
C     FUNCTION DECLARATIONS
      REAL :: HUMRT1, ENTHP2, PCRH2

C     ITERATION VARIABLES
      REAL :: fFuncOne, fFuncTwo, fTOne, fTTwo, fWOne, fWTwo, fErrApp
      INTEGER :: iIter
      
      REAL, PARAMETER :: fTwbIn = 19.4, fTdbIn = 26.7, fWIn = 0.01112 ! AHRI/ASHRAE test conditions
      REAL, PARAMETER :: fSpecVolIn = 0.849, fHeatVap = 2.47E6 ! AHRI/ASHRAE test conditions
      REAL, PARAMETER :: fDelta=10E-6 ! Perturbation value for modified secant method
      REAL, PARAMETER :: fErrStop=0.000005 ! Error stopping criteria
      INTEGER, PARAMETER :: iItermax=100 ! Maximum number of iterations
      
      ! Load the rated data
      fQtotRated = BDATA(IPCOMP,1) ! Pull in the rated total cooling capacity [W] from BDATA
      fSHRrated = BDATA(IPCOMP,2) ! Pull in the SHR rated value from BDATA
      fFlowRated = BDATA(IPCOMP,4) ! Pull in the rated flow rate value [m3/s] from BDATA
      fQsensRated = fQtotRated * fSHRrated ! Sensible capacity at rated conditions [W]
      fQlatRated = fQtotRated - fQsensRated ! Latent capacity at rated conditions [W]

      ! Determine the humidity ratio at the outlet using latent cooling
      ! See ASHRAE 37-2009
      fWOut = fWIn-(fQlatRated*fSpecVolIn)/(fFlowRated*fHeatVap)

      ! Determine outlet temperature
      fCpIn = 1005.+(1805.*fWIn)  ! Inlet air specific heat [J/kg_da K]
      fCpOut = 1005.+(1805.*fWOut)  ! Inlet air specific heat [J/kg_da K]
      fTdbOut = ((fCpIn*fTdbIn)-((fSpecVolIn*fQsensRated)/fFlowRated))/
     &              fCpOut
      ! Determine the outlet relative humidity
      fRHOut = PCRH2(fTdbOut,fWOut,1013.25)
      If(fRHOut>100.) Then
         WRITE(IUOUT,*) ' getBypassFactorCMP144: ',
     &                'Calculated outlet relative humidity above 100.'
         WRITE(IUOUT,*) ' Combination of rated flow rate, total ',
     &                'cooling, and SHR yields exit conditions '
         WRITE(IUOUT,*) ' above saturation curve. Possible fix is to ',
     &                'reduce rated total, increase rated'
         WRITE(IUOUT,*)'flow, or reduce SHR.'
         fEnthOut = ENTHP2(fTdbIn,fWIn)*1000.
     &         -((fSpecVolIn*fQtotRated)/fFlowRated)
         fTsatOut = TSATH0((fEnthOut/1000.),1013.25)
         If(fTdbOut<fTsatOut)Then
            fTdbOut = fTsatOut+0.1
            fWOut = ((fEnthOut/1000.)+0.026-
     &               (fTdbOut*1.006))/(2501.+
     &               (1.805*fTdbOut))
            fHTinWout = ENTHP2(fTdbIn,fWOut)*1000.
            WRITE(IUOUT,*) ' getBypassFactorCMP144: adjusting SHR: ', 
     &                     fSHRrated
            fSHRrated = (fHTinWout-fEnthOut)/((fSpecVolIn*fQtotRated)/
     &                   fFlowRated)
            WRITE(IUOUT,*) ' new rated SHR: ', fSHRrated
            BDATA(IPCOMP,2) = fSHRrated
         EndIf
      EndIf
      ! Compute the slope rated
      fSlopeRated = (fWIn-fWOut)/(fTdbIn-fTdbOut)

      ! Search along the saturation curve to find the ADP using the Modified
      ! Secant Method
      fTOne = fTdbOut ! Take the outlet drybulb as the initial guess
      iIter = 1
      fErrApp = fErrStop+1.
      findADP: DO while (fErrApp>fErrStop .AND. iIter < iItermax)
         ! Point one function value
         fWOne = HUMRT1(fTOne,100.,1013.25,2) ! PATMOS not initialized, use standard pressure
         fFuncOne = fSlopeRated-((fWIn-fWOne)/(fTdbIn-fTOne))
         
         ! Point two function value
         fTTwo = fTOne+(fDelta*fTOne)
         fWTwo = HUMRT1(fTTwo,100.,1013.25,2)
         fFuncTwo = fSlopeRated-((fWIn-fWTwo)/(fTdbIn-fTTwo))
         
         ! Update guess for Tadp
         fTadp = fTOne-((fDelta*fTOne*fFuncOne)/(fFuncTwo-fFuncOne))
         ! Check to see if temperature is rational
         IF(fTadp <= -273) THEN
            WRITE(IUOUT,*) ' getBypassFactorCMP144: ',
     &                'Bypass iterated to below absolute zero.'
          STOP ' CMP135S: unresolvable error'
         ELSEIF(fTadp>50.) THEN
            WRITE(IUOUT,*) ' getBypassFactorCMP144: ',
     &                'Bypass iterated to above 50 oC.'
          STOP ' CMP135S: unresolvable error'
         ENDIF
         ! Get the approximate error
         fErrApp = abs((fTadp-fTOne)/fTadp)
         iIter = iIter+1
         fTOne = fTadp
         IF(iIter == iItermax) THEN
            WRITE(IUOUT,*) ' getBypassFactorCMP144: ',
     &                'Max iterations exceed.'
            WRITE(IUOUT,*) ' The rated SHR:',fSHRrated, 
     &                ' may be causing this. The line defined by the ',
     &                'coil rated inlet air conditions and rated ',
     &                'SHR must intersect the saturation curve.'
            WRITE(IUOUT,*) ' If the rated SHR is too low then the ', 
     &                'intersection may not occur and the bypass ',
     &                'calc will not converge.'
            STOP ' CMP135S: unresolvable error'
         ENDIF
      ENDDO findADP
      
      ! Retrieve Enthaplies at inlet, outlet, and ADP
      fEnthIn = ENTHP2(fTdbIn,fWIn) ! [kJ/kg]
      fEnthOut = ENTHP2(fTdbOut,fWOut)
      fWadp = HUMRT1(fTadp,100.,1013.25,2)
      fEnthADP = ENTHP2(fTadp,fWadp)
      
      ! Compute the bypass factor at rated conditions
      fBFrated = (fEnthOut-fEnthADP)/(fEnthIn-fEnthADP)
      IF(fBFrated<0) Then
         WRITE(IUOUT,*) ' getBypassFactorCMP144: bypass factor ',
     &         fBFrated, ' is less than zero.'
         fBFrated=0. 
         STOP ' CMP135S: unresolvable error'
      ELSEIF(fBFrated>1.) Then
         fBFrated = 1.
      ENDIF
      
      ! Compute the coil geometry parameter
      IF(fBFrated > 0.) THEN
         fAo = -1*(fFlowRated/fSpecVolIn)*LOG(fBFrated)
      ELSE
         fAo = 0.
      ENDIF
      BDATA(IPCOMP,21) = fAo ! Store at end of BDATA for later retrieval

      end subroutine getBypassFactorCMP144

C ******************** CMP145S ********************
C Establishes for a plant component with plant db code 1360 i.e.
C 1 node (ISV>21)     moist air flow source
C whether the specified number of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine

      SUBROUTINE CMP145S(IPCOMP)
#include "plant.h"
#include "building.h"
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Trace output.
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV>21) moist air flow source'
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

C Check user specified number of controlled variables.
      NCITM=3
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' CMP145S warning: user specified wrong number',
     &                ' of controlled variables'

C Check component has 1 connection only, to 'moist air'.
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCON.GT.0)THEN
        IF(IPCONC.LE.NCONS) THEN
            IF(IPCON.EQ.0) THEN
                GOTO 990
            ELSE IF(ICONTP(IPCON).NE.21) THEN
                GOTO 990
            END IF
        ELSE IF(IPCON.NE.0) THEN
            GOTO 990
        END IF
      ENDIF
   10 CONTINUE

      RETURN

C Error handling
  990 WRITE(IUOUT,*) ' CMP145S: connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' connection(s) to air'
      STOP ' CMP145S: unresolvable error'

      END
