C This file is part of the ESP-r system.
C Copyright Energy Systems Research Unit, University of
C Strathclyde, Glasgow Scotland, 2001-.

C ESP-r is free software.  You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation
C (version 2 orlater).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.


C This file contains the following subroutines. 
C  CMP20C   1 node (ISV=20) WCH boiler
C  CMP21C   2 node (ISV=20) WCH radiator
C  CMP22C   1 node (ISV=20) WCH pipe
C  CMP23C   1 node (ISV=20) WCH converging 2-leg junction
C  CMP24C   1 node (ISV=20) WCH pump
C  CMP25C   2 node (ISV=20) WCH boiler with on/off control
C  CMP26C   2 node (ISV=20) WCH boiler with aquastat control
C  CMP27C   8 node (ISV=20) WCH radiator
C  CMP28C   1 node (ISV=9)  Oil-filled electric panel radiator
C  CMP29C   1 node (ISV=20) WCH flow control valve
C  CMP31C   2 node (ISV=20) WCH calorifier
C  CMP32C   2 node (ISV=20) WCH generic liquid/liquid heat exchanger
C  CMP33C   2 node (ISV>20) WCH generic gas/liquid heat exchanger
C  CMP34C   3 node (ISV>20) WCH storage water heater
C  CMP35C   2 node (ISV=20) WCH basic radiator
C  CMP36C   2 node (ISV=20) WCH basic chiller or heat pump
C  CMP37C   1 node (ISV=20) WCH water/air heat rejector
C  CMP38C   1 node (ISV=20) WCH converging multi-leg junction
C                           (up to 10 connections)
C  CMP39C   1 node (ISV=20) WCH pump (mass flow control)
C  CMP42C   2 node (ISV=20) WCH modulating boiler
C  CMP44C   2 node (ISV=20) WCH exponent model radiator
C  CMP45C   1 node (ISV=20) Air source heat pump connection to WCH
C  CMP46C   2 node (ISV=20) WCH advanced modulating boiler model
C  CMP47C   1 node (ISV=20) Ground source heat pump (EXPERIMENTAL)
C  CMP48C   2 node (ISV=20) Ground source heat pump (EXPERIMENTAL)
C  CMP49C   1 node (ISV=20) Inverter control ground source heat pump (EXPERIMENTAL)
C  CMP50C   1 node (ISV=29) WCH thermostatic radiator valve
C  CMP51C   1 node (ISV=29)     mechanical room thermostat
C  CMP52C   1 node (ISV=29) WCH thermostatic radiator valve (sensor only)
C  CMP73C   3 node (ISV=20) WCH 3-port valve
C  CMP74C   3 node (ISV=20) WCH 3-port valve w/ automatic flow control
C                               valve (EXPERIMENTAL)
C  CMP79C   1 node (ISV=20) WCH stochastic hot water draw
C  CMP90C   2 node (ISV>19)     air & water temperature source
C  CMP91C   2 node (ISV>19)     imaginary building-like plant load
C                               acting on heat and mass balances
C  TYPE6    TRNSYS type component for WCH boiler with aquastat control
C
C
C ******************** CMP20C ********************
C Generates for plant component IPCOMP with plant db code 200 ie.
C 1 node (ISV=20) WCH boiler & flux control
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 UA modulus (W/K)
C     BDATA: none
C     CDATA: 1 Heating duty (W)

      SUBROUTINE CMP20C(IPCOMP,COUT,ISTATS)
       use h3kmodule
#include "plant.h"
#include "building.h"

      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PTIME/PTIMEP,PTIMEF
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      common/pcnam/pcname(mpcom)       ! Plant component names
      character*15 pcname

      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom) !GF
      REAL QDATA,PCAOUT ! GF
      integer napdat

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      character outs*124

      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP20C'

C Check control data for relevant balance type
      IF(ISTATS.EQ.1.AND.CDATA(IPCOMP,1).LT.0.) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP20C: invalid control data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP20C: unresolvable error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss)
         UA=ADATA(IPCOMP,3)
         call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
         IF(closea) UA=0.

C Then initialize added heat Q
         Q=CDATA(IPCOMP,1)

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(C1+UA))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self- and cross-coupling coefficients
         COUT(1)=ALPHA*(-C1-UA)-CM/TIMSEC
         COUT(2)=ALPHA*C1
C and then present-time coefficient (ie. right hand side)
         COUT(3)=((1.-ALPHA)*(PCRP(ICON1)+PUAP(INOD1))
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &             -ALPHA*UA*PCNTMF(IPCOMP)
     &             -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)
     &             -ALPHA*Q-(1.-ALPHA)*PCQP(INOD1)

C Store "environment" variables future values
         PUAF(INOD1)=UA
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD1)=Q

C Establish "containment loss" data
         QDATA(IPCOMP)=UA*(alpha*(csvf(inod1,1)-pcntmf(ipcomp))+
     &                (1.-alpha)*(csvp(inod1,1)-pcntmp(ipcomp)))
         call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iConvective)
         call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iRadiant)

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-PCONDR(ICON1)
         COUT(3)=0.

C 2nd phase mass (ie. vapour) balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH boiler & flux control'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1)
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP20C'

C------------------------------------------------------
C XML output
C------------------------------------------------------
       call AddToReport(rvPltQAddedH%Identifier,
     &         Q,
     &         pcname(IPCOMP)(1:lnblnk(pcname(IPCOMP))))

      RETURN
      END

C ******************** CMP21C ********************

C CMP21C generates for plant component IPCOMP with plant db code 210 ie.
C 2 node (ISV=20) WCH radiator
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Radiator exponent (-)
C            2 Nominal heat emission of radiator (W)
C            3 Nominal supply temperature (C)
C            4 Nominal exit temperature (C)
C            5 Nominal environment temperature (C)
C            6 Index of coupled building zone (-)
C            7 Number of walls used for defining Te (-)
C            8 Index of 1st wall for defining Te (-)
C            9 Weighting factor for 1st wall when defining Te (-)
C           10 Index of 2nd wall for defining Te (-)
C           11 Weighting factor for 2nd wall when defining Te (-)
C           12 etc.
C     CDATA: none

C     PCDATF/P
C            1 Inter-node fluid heat capacity rate (W/K)

      SUBROUTINE CMP21C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      COMMON/C6/INDCFG
      COMMON/FVALA/TFA(MCOM),QFA(MCOM)
      COMMON/FVALS/TFS(MCOM,MS),QFS(MCOM)

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      logical closea
      logical bNoFlow

      real fQ_node1_frac
      real fQ_node2_frac

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP21C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Initialize the nominal log mean temperature difference DTLM0
         TS0=BDATA(IPCOMP,3)
         TX0=BDATA(IPCOMP,4)
         TE0=BDATA(IPCOMP,5)
         call eclose(((TS0-TE0)/(TX0-TE0)),1.00,0.001,closea)
         IF(ABS(TS0-TE0).LT.SMALL.OR.ABS(TX0-TE0).LT.SMALL
     &      .OR.(TS0-TE0)/(TX0-TE0).LT.SMALL
     &      .OR.closea) THEN
C     &      .OR.(TS0-TE0)/(TX0-TE0).EQ.1.0) THEN
           DTLM0=(TS0+TX0)/2.-TE0
         ELSE
           DTLM0=(TS0-TX0)/ALOG((TS0-TE0)/(TX0-TE0))
         END IF

C Then evaluate the current environment temperature
         IF(INDCFG.EQ.2.OR.NINT(BDATA(IPCOMP,6)).EQ.0) THEN
            call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
            IF(closea) THEN
               TE=BDATA(IPCOMP,5)
            ELSE
               TE=PCNTMF(IPCOMP)
            END IF
         ELSE
            IZ=INT(BDATA(IPCOMP,6))
            SUMT=TFA(IZ)
            NW=int(BDATA(IPCOMP,7))
            SUMW=1.
            IF(NW.GE.1.) THEN
               DO 10 IW=1,NW
                  W=BDATA(IPCOMP,7+IW*2)
                  SUMW=SUMW+W
                  SUMT=SUMT+W*TFS(IZ,NINT(BDATA(IPCOMP,6+IW*2)))
   10          CONTINUE
            END IF
            TE=SUMT/SUMW
         END IF
C And the current log mean temperature difference DTLM based on
C current node 1 temp., so mark node 1 temperature for iteration
         ICSV(INOD1,1)=1
         CSVI(INOD1,1)=CSVF(INOD1,1)
         TS=CSVF(INOD1,1)
         TX=CSVF(INOD2,1)
         call eclose(((TS-TE)/(TX-TE)),1.00,0.001,closea)
         IF(ABS(TS-TE).LT.SMALL.OR.ABS(TX-TE).LT.SMALL
     &      .OR.(TS-TE)/(TX-TE).LT.SMALL
     &      .OR.closea) THEN
C     &      .OR.(TS-TE)/(TX-TE).EQ.1.0) THEN
           DTLM=(TS+TX)/2.-TE
         ELSE
           DTLM=(TS-TX)/ALOG((TS-TE)/(TX-TE))
         END IF

C Then calculate radiator heat emission Q
         DTR=AMAX1(SMALL,(DTLM/DTLM0))
         Q=BDATA(IPCOMP,2)*DTR**BDATA(IPCOMP,1)

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)/2.
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))
C Inter-node fluid heat capacity rate (W/K)
         PCDATF(IPCOMP,1)=CSVF(INOD1,2)*SHTFLD(3,CSVF(INOD1,1))
C Determine if there's fluid flow in the raditor
         call eclose  ( PCONDR(ICON1)*CONVAR(ICON1,2),
     &                  0.0, 1.0E-05, bNoFlow )

C Modification: 2007-11-19 (aferguso)
C Hensen's original model appended all of the heat emission to the
C second node's heat balance. Hensen (1991) writes:
C
C  " The heat transfer from the radiator to the environment
C    is assumed to take place between the two nodes, and will
C    therefore appear only in the energy balance of the second
C    node (hence, this model still needs refinements because
C    it suffers from the same problem as indicated before for
C    the boiler models: ie no heat loss from the first node
C    when the water flow rate approaches zero). "
C
C Since the LMTD method uses the first node temperature to compute
C compute heat transfer from the radiator, this defect is more severe
C than Hensen suggests. When there is no flow to the radiator, the
C model will:
C
C   - incorrectly compute the resulting heat transfer, and
C
C   - predict outlet temperatures well below the environment
C     temperature (and often below zero).
C
C To correct this behavior, we'll modify the node heat balance
C equations depending on whether there's flow through the
C radiator.
C
C   - If there's flow through the radiator, we'll use the
C     same energy balance equations that Hensen originally
C     proposed (that is, all of the heat transfer is appended
C     to the second node energy balance).
C
C   - Otherwise, we'll divide the computed heat transfer between
C     the first and second nodes. This will cause both nodes to
C     approach the the environmental temperaute as they cool.
C
C
C Under no-flow conditions, the heat transfer will be apportioned
C according to the temperature differences between the nodes
C and the enviroment:
C
C                           [node 1 temp ] - [env. temp]
C    [node 1 %] = ------------------------------------------------
C                 [node 1 temp ] + [node 2 temp] - 2 * [env. temp]
C
C                           [node 2 temp ] - [env. temp]
C    [node 2 %] = ------------------------------------------------
C                 [node 1 temp ] + [node 2 temp] - 2 * [env. temp]
C
C We must also ensure the resulting heat transfer does not cool
C the nodes below the enviroment temperature --- that is:
C
C
C                       [heat capacity] * ([node 1 temp] - [env. temp])
C   [Q] * [node 1 %] <= -----------------------------------------------
C                                  [time-step duration]
C
C                       [heat capacity] * ([node 2 temp] - [env. temp])
C   [Q] * [node 2 %] <= -----------------------------------------------
C                                  [time-step duration]
C
C This modification ignores the central issue --- the model's LMTD
C basis is not valid under unsteady conditions. But the model's heat
C transfer predictions when water is flowing through the radiator are
C unaffected, and with these changes it does provide sensible predictions
C under no-flow conditions.
C
C Reference:
C
C Hensen J L M (1991) 'On the thermal interaction of building
C     structure and heating and ventilating system', PhD Thesis,
C     Eindhoven University of Technology.

C Set default values (just to be on the safe side)
        fQ_node1_frac = 0.0
        fQ_node2_frac = 1.0

        if ( bNoFlow ) then

          fQ_node1_frac = (TS-TE) / ( TS + TX - 2.0 * TE)
          !check for the case where TS=TE=TX
          call eclose(TS,TE,0.001,closea)
          IF(closea) fQ_node1_frac = 0.

          fQ_node2_frac = (TX-TE) / ( TS + TX - 2.0 * TE)
          !check for the case where TS=TE=TX
          call eclose(TX,TE,0.001,closea)
          IF(closea) fQ_node2_frac = 0.

C Make sure Resulting heat transfer does not cool nodes below environment
C temperature
          if ( TS - Q * fQ_node1_frac * TIMSEC / CM < TE ) then

            fQ_node1_frac = CM * (TS-TE) / TIMSEC / Q

          endif

C Make sure Resulting heat transfer does not cool nodes below environment
C temperature
          if ( TX - Q * fQ_node2_frac * TIMSEC / CM < TE ) then

            fQ_node2_frac = CM * (TX-TE) / TIMSEC / Q

          endif


        else

          fQ_node1_frac = 0.0
          fQ_node2_frac = 1.0

        endif



C Calculate current component time-constant TC
         TC(IPCOMP)=AMAX1(
     &        CM/AMAX1(SMALL,(C1)),
     &        CM/AMAX1(SMALL,(PCDATF(IPCOMP,1)+
     &            Q*(fQ_node1_frac + fQ_node2_frac)
     &            /AMAX1(SMALL,AMAX1(SMALL,(TS-TE))))))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self-coupling coefficients,
C  node 1
         COUT(1)=ALPHA*(-C1)-CM/TIMSEC
C  node 2
         COUT(2)=ALPHA*PCDATF(IPCOMP,1)
         COUT(3)=ALPHA*(-PCDATF(IPCOMP,1))-CM/TIMSEC
C then matrix equation cross-coupling coefficient,
         COUT(4)=ALPHA*C1
C and then present-time coefficients (ie. right hand sides). Scale
C future-time-row Q value by respecitve node fraction.
         COUT(5)=((1.-ALPHA)*PCRP(ICON1)-CM/TIMSEC)*CSVP(INOD1,1)
     &           +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &           +fQ_node1_frac* ALPHA*Q +(1.-ALPHA)*PCQP(INOD1)
         COUT(6)=(1.-ALPHA)*(-PCDATP(IPCOMP,1))*CSVP(INOD1,1)
     &           +((1.-ALPHA)*PCDATP(IPCOMP,1)-CM/TIMSEC)*CSVP(INOD2,1)
     &           +fQ_node2_frac* ALPHA*Q+(1.-ALPHA)*PCQP(INOD2)


C Store "environment" variables future values. Scale Q value by
C node fractions.
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD1)=Q * fQ_node1_frac
         PCQF(INOD2)=Q * fQ_node2_frac

C Establish "containment loss" data. Scale Q value by sum
C of node fractions.
         QDATA(IPCOMP)=ALPHA*Q* ( fQ_node1_frac + fQ_node2_frac )
     &       +(1.-ALPHA)*PCQP(INOD2)

C Establish additional output variables
         PCAOUT(IPCOMP,1)=TE
         PCAOUT(IPCOMP,2)=DTLM
         PCAOUT(IPCOMP,3)=Q

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-1.
         COUT(3)=1.
         COUT(4)=-PCONDR(ICON1)
         COUT(5)=0.
         COUT(6)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=1.
         COUT(4)=0.
         COUT(5)=0.
         COUT(6)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH radiator'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' TS     = ',TS,' (C)'
            WRITE(ITU,*) ' TX     = ',TX,' (C)'
            WRITE(ITU,*) ' TE     = ',TE,' (C)'
            WRITE(ITU,*) ' DTLM   = ',DTLM,' (K)'
            WRITE(ITU,*) ' Q      = ',
     &            Q*(fQ_node1_frac + fQ_node2_frac ),' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP21C'

      RETURN
      END

C ******************** CMP22C ********************

C CMP22C generates for plant component IPCOMP with plant db code 220 ie.
C 1 node (ISV=20) WCH pipe
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 UA modulus from wall to environment (W/K)
C            4 Hydraulic diameter of pipe (m)
C            5 Length of pipe (m)
C            6 Cross sectional face area (m^2)
C     BDATA: none
C     CDATA: none

      SUBROUTINE CMP22C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      logical closea,closeb,closec

      PI = 4.0 * ATAN(1.0)

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP22C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss)
         UA=ADATA(IPCOMP,3)
         call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
         IF(closea) THEN
            UA=0.
         ELSE
            DH=ADATA(IPCOMP,4)
            PL=ADATA(IPCOMP,5)
            TW=CONVAR(ICON1,1)
            WMFR=PCONDR(ICON1)*CONVAR(ICON1,2)

C minimum allowed pipe diameter 
            call eclose(DH,0.00,0.0005,closeb)

C Calculate internal heat transfer coefficient using Dittus-Boelter
C equation
            IF(.NOT.closeb) THEN
              VW=WMFR/(RHOFLD(3,TW)*ADATA(IPCOMP,6))
              HW=1400.*(1.+0.015*TW)*VW**0.8*DH**(-0.2)

C Apply limiting value to heat transfer coefficient (which is for
C laminar flow). 
              IF(HW.lt.(2.1/DH))HW=2.1/DH

C Modify UA to account for internal pipe heat transfer
              UA=1./(1./(HW*PI*DH*PL)+1./UA)

C Apply limiting value to mass flow rate. If mass flow rate is less than
C limiting value then it can be safely assumed that the pipe loses heat
C fairly quickly and a high value of UA will be calculated below
              call eclose(WMFR,0.00,0.001,closec)
              if(closec)WMFR=0.001 
              WMCP=WMFR*SHTFLD(3,CONVAR(ICON1,1))

C Pipe heat loss is calculated using exit temperature therefore UA is
C artificially modified to account for the temperature distribution over
C the pipe length. For low flow rates this will substantially increase
C UA because average pipe temperature is much higher than at exit.
              UA=WMCP*(EXP(UA/WMCP)-1.)
            ENDIF
         END IF

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Calculate current component time-constant TC and determine rate of 
C change of inlet conditions (RCC). ALPHA set to fully implicit if
C RCC changes by a factor of two from previous timestep and a small
C timestep is used to avoid discretisation errors.
         TC(IPCOMP)=CM/AMAX1(SMALL,(C1+UA))
         RCC = (C1+UA) / AMAX1(SMALL,(PCRP(ICON1)+PUAP(INOD1)))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(RCC.GT.2.OR.RCC.LT.0.5)THEN
               IF(TIMSEC.LT.300)THEN 
                 ALPHA=1.
               ELSE
                 ALPHA=RATIMP
               END IF
            ELSE IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

         call eclose(C1,0.00,0.0001,closec)
         if(closec)ALPHA=1.

C Establish matrix equation self- and cross-coupling coefficients
         COUT(1)=ALPHA*(-C1-UA)-CM/TIMSEC
         COUT(2)=ALPHA*C1

C and then present-time coefficient (ie. right hand side)
         COUT(3)=((1.-ALPHA)*(PCRP(ICON1)+PUAP(INOD1))
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &             -ALPHA*UA*PCNTMF(IPCOMP)
     &             -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)

C Store "environment" variables future values
         PUAF(INOD1)=UA
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1

C Establish "containment loss" data
        QDATA(IPCOMP)=UA*(alpha*(csvf(inod1,1)-pcntmf(ipcomp))+
     &                (1.-alpha)*(csvp(inod1,1)-pcntmp(ipcomp)))
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iConvective)
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iRadiant)

C For embedded pipes:
        call store_plt_gain ( IPCOMP, QDATA(IPCOMP), iConductive)

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-PCONDR(ICON1)
         COUT(3)=0.

C 2nd phase mass (ie. vapour) balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH pipe'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP22C'

      RETURN
      END

C ******************** CMP23C ********************

C CMP23C generates for plant component IPCOMP with plant db code 230 ie.
C 1 node (ISV=20) WCH converging 2-leg junction
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 UA modulus (W/K)
C     BDATA: none
C     CDATA: none

      SUBROUTINE CMP23C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP23C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      ICON2=ICONDX(IPCOMP,1,2)
      INOD1=NPCDAT(IPCOMP,9)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss)
        UA=ADATA(IPCOMP,3)
        call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
        IF(closea) UA=0.0

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
        CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
        C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))
        C2=PCONDR(ICON2)*CONVAR(ICON2,2)*SHTFLD(3,CONVAR(ICON2,1))

C Calculate current component time-constant TC and determine rate of 
C change of inlet conditions (RCC). ALPHA set to fully implicit if
C RCC changes by a factor of two from previous timestep and a small
C timestep is used to avoid discretisation errors.
        TC(IPCOMP)=CM/AMAX1(SMALL,(C1+C2+UA))
        RCC = AMAX1(SMALL,(C1+C2+UA)) / 
     &        AMAX1(SMALL,(PCRP(ICON1)+PCRP(ICON2)+PUAP(INOD1)))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
          ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
          ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(RCC.GT.2.OR.RCC.LT.0.5)THEN
            IF(TIMSEC.LT.300)THEN 
              ALPHA=1.
            ELSE
              ALPHA=RATIMP
            END IF
          ELSE IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
            ALPHA=1.
          ELSE
            ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
          CM=0.
          ALPHA=1.
        END IF

        call eclose(C1,0.00,0.0001,closea)
        if(closea)ALPHA=1.

C Establish matrix equation self- and cross-coupling coefficients
        COUT(1)=ALPHA*(-C1-C2-UA)-CM/TIMSEC
        COUT(2)=ALPHA*C1
        COUT(3)=ALPHA*C2
C and then present-time coefficient (ie. right hand side)
        COUT(4)=((1.-ALPHA)*(PCRP(ICON1)+PCRP(ICON2)+PUAP(INOD1))
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &             +(1.-ALPHA)*(-PCRP(ICON2))*PCTP(ICON2)
     &             -ALPHA*UA*PCNTMF(IPCOMP)
     &             -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)

C Store "environment" variables future values
        PUAF(INOD1)=UA
        PCTF(ICON1)=CONVAR(ICON1,1)
        PCTF(ICON2)=CONVAR(ICON2,1)
        PCRF(ICON1)=C1
        PCRF(ICON2)=C2

C Establish "containment loss" data
        QDATA(IPCOMP)=UA*(alpha*(csvf(inod1,1)-pcntmf(ipcomp))+
     &                (1.-alpha)*(csvp(inod1,1)-pcntmp(ipcomp)))
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iConvective)
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iRadiant)

C For embedded two leg junction:
        call store_plt_gain ( IPCOMP, QDATA(IPCOMP), iConductive)

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
        COUT(1)=1.
        COUT(2)=-PCONDR(ICON1)
        COUT(3)=-PCONDR(ICON2)
        COUT(4)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
         COUT(4)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH converging 2-leg junction'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1,ICON2
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' C2     = ',C2,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=4
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP23C'

      RETURN
      END

C ******************** CMP24C ********************

C CMP24C generates for plant component IPCOMP with plant db code 240 ie.
C 1 node (ISV=20) WCH pump
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 UA modulus (W/K)
C     BDATA: 1 Rated absorbed power (W)
C            2 Rated volume flow rate (m^3/s)
C            3 Overall efficiency (-)
C     CDATA: 1 Volume flow rate (m^3/s)

      SUBROUTINE CMP24C(IPCOMP,COUT,ISTATS)
       use h3kmodule

#include "plant.h"
#include "building.h"
#include "net_flow.h"
#include "net_flow_data.h"
#include "OffsiteUtilitiesPublic.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PTIME/PTIMEP,PTIMEF
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      COMMON/FFN/IFLWN,ICFFS(MPCON)
      COMMON/MFLRES/FLW1(MCNN),FLW2(MCNN),PRES(MNOD),
     &              RESID(MNOD),SAFLW(MNOD)

      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      common/pcnam/pcname(mpcom)       ! Plant component names
      character*15 pcname

C Electrical details for specified plant components
      common/pcelflg/ipcelf(mpcom)
      common/elpcp/NPEL,PFP(mpcom),IPFP(mpcom),PWRP(mpcom),
     &BVOLTP(mpcom),IPHP(mpcom)

      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      DOUBLE PRECISION FLW1,FLW2,PRES,RESID,SAFLW
      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      character outs*124
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP24C'

C Check control data for relevant balance type
      IF(ISTATS.EQ.2.AND.CDATA(IPCOMP,1).LT.0.) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP24C: invalid control data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP24C: unresolvable error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

      ICSV(INOD1,1)=1
      CSVI(INOD1,1)=CSVF(INOD1,1)
      ICSV(INOD1,2)=1
      CSVI(INOD1,2)=CSVF(INOD1,2)
      ICSV(INOD1,3)=1
      CSVI(INOD1,3)=CSVF(INOD1,3)

C Total mass flow rate into pump
      TOTAL_MASS_FR = PCONDR(ICON1) *
     &                (CONVAR(ICON1,2) + CONVAR(ICON1,3))

C Mass fraction of glycol in incoming flow
      GLYCOL_MASS_FRACTION =
     &            100. * CONVAR(ICON1,3) / (TOTAL_MASS_FR + SMALL)

C Find volume fraction of glycol in mixture of incoming flow
      GLYCOL_VOL_FRACTION =
     &GLYCOL_VOL_FRACTION_FIND(GLYCOL_MASS_FRACTION,CONVAR(ICON1,1))

C Boiling temperature of glycol mixture
      TBOIL = GLYCOL_WATER_TBOIL(GLYCOL_VOL_FRACTION)

C Freezing temperature of glycol-water mixture
      TFREEZE = GLYCOL_WATER_TFREEZE(GLYCOL_VOL_FRACTION)

C Set incoming flow temperature using upper and lower bounds
      TEMP_INFLOW = MAX(TFREEZE,MIN(TBOIL,CONVAR(ICON1,1)))

C Set specific heat of incoming flow
      GLY_WATER_CP = GLYCOL_WATER_CP(GLYCOL_VOL_FRACTION,TEMP_INFLOW)

C Set temperature of pump using lower and upper bounds
      TEMP_PUMP = MAX(TFREEZE,MIN(TBOIL,CSVF(INOD1,1)))

C Density of fluid based on exit pump temperature
      GLY_WATER_DEN = GLYCOL_WATER_DENSITY(GLYCOL_VOL_FRACTION,
     &                 TEMP_PUMP)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss)
         UA=ADATA(IPCOMP,3)
         call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
         IF(closea) UA=0.

C Establish absorbed power E based on current water flow rate which might
C have been calculated by mfs
C In case of mfs E is based on flow rate, pressure *rise*, and efficiency
         IF(IFLWN.NE.0.AND.ICFFS(ICON1).NE.0) THEN
            ICNN=ICFFS(ICON1)
            E=real((FLW1(ICNN)+FLW2(ICNN))
     &        *(PRES(NODNE(ICNN))-PRES(NODPS(ICNN)))
     &        /(BDATA(IPCOMP,3)*GLY_WATER_DEN))
         ELSE
            E=((CSVF(INOD1,2) + CSVF(INOD1,3))
     &         /(GLY_WATER_DEN*BDATA(IPCOMP,2)))**3
     &         *BDATA(IPCOMP,1)
         END IF

         PWRP(IPCOMP)=-ABS(E)
         IEMODEL=1
         CALL EMACH(IPCOMP,IEMODEL,PWRP(IPCOMP),PQ,PA)
         PWRQ=PQ

C Pass power consumption to Site Utilities facility
         fSUFuelEnergyUse( iElectricity, iUseUncatagorized) = ABS(E)
         call StorePltCompEnergyUse ( IPCOMP, fSUFuelEnergyUse )

C Report power consumption
         call AddToReport(rvPltWCHPumpEInput%Identifier,
     &         ABS(E),
     &         pcname(ipcomp)(1:iPltNameLen(ipcomp)))


C Now Q is made up of all inefficiencies
         Q=(1.-BDATA(IPCOMP,3))*E

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
         C1=TOTAL_MASS_FR*GLY_WATER_CP

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(C1+UA))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self- and cross-coupling coefficients
         COUT(1)=ALPHA*(-C1-UA)-CM/TIMSEC
         COUT(2)=ALPHA*C1
C and then present-time coefficient (ie. right hand side)
         COUT(3)=((1.-ALPHA)*(PCRP(ICON1)+PUAP(INOD1))
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &             -ALPHA*UA*PCNTMF(IPCOMP)
     &             -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)
     &             -ALPHA*Q-(1.-ALPHA)*PCQP(INOD1)

C Store "environment" variables future values
         PUAF(INOD1)=UA
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD1)=Q

C Save plant additional output data.
         napdat(ipcomp)=2
         pcaout(ipcomp,1)=pwrp(ipcomp)
         pcaout(ipcomp,2)=pwrq

C Establish "containment loss" data
         QDATA(IPCOMP)=UA*(alpha*(csvf(inod1,1)-pcntmf(ipcomp))+
     &                (1.-alpha)*(csvp(inod1,1)-pcntmp(ipcomp)))
         call store_plt_gain (IPCOMP,0.5*QDATA(IPCOMP),iConvective)
         call store_plt_gain (IPCOMP,0.5*QDATA(IPCOMP),iRadiant)

C 1st phase mass (ie. water) balance coefficients
C Note that if fluid mass flow solver active, source of mass zeroised
      ELSE IF(ISTATS.EQ.2) THEN
        COUT(1)=1.
        IF(IFLWN.EQ.0.or.ICFFS(ICON1).eq.0) THEN
          COUT(2)=0.
          COUT(3)=CDATA(IPCOMP,1)*GLY_WATER_DEN*
     &            (1.-GLYCOL_MASS_FRACTION/100.)
        ELSE
          COUT(2)=-PCONDR(ICON1)
          COUT(3)=0.
        END IF

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
        COUT(1)=1.
        IF(IFLWN.EQ.0.or.ICFFS(ICON1).eq.0) THEN
          COUT(2)=0.
          COUT(3)=CDATA(IPCOMP,1)*GLY_WATER_DEN*
     &    GLYCOL_MASS_FRACTION/100.
        ELSE
          COUT(2)=-PCONDR(ICON1)
          COUT(3)=0.
        END IF
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH pump'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
            WRITE(ITU,*) ' Q      = ',Q,' (W)'
            WRITE(ITU,*) ' E      = ',E,' (W)'
            WRITE(ITU,*) ' E,cum. = ',E*TIMSEC,' (J)'
         ELSE IF(ISTATS.EQ.2) THEN
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1)
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP24C'

      RETURN
      END

C ******************** CMP25C ********************

C CMP25C generates for plant component IPCOMP with plant db code 250 ie.
C 2 node (ISV=20) WCH boiler with on/off control
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Full load gas firing rate when boiler on (m^3/s)
C            2 Stand-by mode gas consumption relative to 1 (-)
C            3 Gas heating value at STP (J/m^3)
C            4 Full load water sided efficiency at Tc (-)
C            5 Tangent of efficiency curve for Tj < Tc (1/K)
C            6 Tangent of efficiency curve for Tj > Tc (1/K)
C            7 Stand-by loss at Tj = Te relative to 1 (-)
C            8 Tangent of stand-by loss curve (1/K)
C            9 Normalized start-stop loss (s)
C           10 Upper boiler temperature limit (C)
C     CDATA: 1 ON/OFF control signal (-)

C     PCDATF/P
C            1 Holds boiler actual ON/OFF state (-)
C            2 Inter-node fluid heat capacity rate (W/K)

      SUBROUTINE CMP25C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PTIME/PTIMEP,PTIMEF
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      PARAMETER (SMALL=1.0E-20)
      REAL      COUT(MPCOE)
      character outs*124

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP25C'

C Check control data for relevant balance type
      IF(ISTATS.EQ.1.AND.
     &   (CDATA(IPCOMP,1).LT.0..OR.CDATA(IPCOMP,1).GT.1.)) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP25C: invalid control data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP25C: unresolvable error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First calculate boiler efficiency ETA based on current node 1 temp.
C so mark node 1 temperature for iteration
         ICSV(INOD1,1)=1
         CSVI(INOD1,1)=CSVF(INOD1,1)
         IF(CSVF(INOD1,1).LT.50.) THEN
            TGALPH=BDATA(IPCOMP,5)
         ELSE
            TGALPH=BDATA(IPCOMP,6)
         END IF
         ETA=BDATA(IPCOMP,4)-TGALPH*(50.-CSVF(INOD1,1))

C Establish whether boiler is on or off (ie 1 or 0)
         IONOFF=int(CDATA(IPCOMP,1))

C Reset to off in case maximum temperature exceeded
         IF(CSVF(INOD2,1).GE.BDATA(IPCOMP,10)) IONOFF=0

C Then calculate heat input into the water PHIW for ON or OFF
         IF(IONOFF.EQ.1) THEN
            PHISB=0.
            FGAS=BDATA(IPCOMP,1)

C Adjust efficiency for start-stop losses if the boiler was off
C during the previous time-step
            IF(NINT(PCDATP(IPCOMP,1)).EQ.0) THEN
               ETA=ETA*(TIMSEC-BDATA(IPCOMP,9))/TIMSEC
            END IF
            PHIW=ETA*FGAS*BDATA(IPCOMP,3)
         ELSE
            FGAS=BDATA(IPCOMP,2)*BDATA(IPCOMP,1)
            PHIW=ETA*FGAS*BDATA(IPCOMP,3)
            PHISB=(BDATA(IPCOMP,7)
     &            +BDATA(IPCOMP,8)*(CSVF(INOD2,1)-PCNTMF(IPCOMP)))
     &            *BDATA(IPCOMP,1)*BDATA(IPCOMP,3)
         END IF

C Then calculate net heat input Q
         Q=PHIW-PHISB

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)/2.
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Boiler actual ON/OFF state PCDATF(IPCOMP,1) (-)
         PCDATF(IPCOMP,1)=IONOFF

C Inter-node fluid heat capacity rate (W/K)
         PCDATF(IPCOMP,2)=CSVF(INOD1,2)*SHTFLD(3,CSVF(INOD1,1))

C Calculate current component time-constant TC and determine rate of 
C change of inlet conditions (RCC). ALPHA set to fully implicit if
C RCC changes by a factor of two from previous timestep and a small
C timestep is used to avoid discretisation errors.
         TC(IPCOMP)=AMAX1(
     &        CM/AMAX1(SMALL,(C1)),
     &        CM/AMAX1(SMALL,(PCDATF(IPCOMP,2)
     &        +PHISB/(CSVF(INOD1,1)-PCNTMF(IPCOMP)))))
         RCC = AMAX1(SMALL,(C1)) / AMAX1(SMALL,(PCRP(ICON1)))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(RCC.GT.2.OR.RCC.LT.0.5)THEN
              IF(TIMSEC.LE.300) THEN 
                 ALPHA=1.
              ELSE
                 ALPHA=RATIMP
              END IF  
            ELSE IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self-coupling coefficients,
         COUT(1)=ALPHA*(-C1)-CM/TIMSEC  ! node 1
         COUT(2)=ALPHA*PCDATF(IPCOMP,2) ! node 2
         COUT(3)=ALPHA*(-PCDATF(IPCOMP,2))-CM/TIMSEC
         COUT(4)=ALPHA*C1     ! then matrix equation cross-coupling coefficient,
C and then present-time coefficients (ie. right hand sides)
         COUT(5)=((1.-ALPHA)*PCRP(ICON1)-CM/TIMSEC)*CSVP(INOD1,1)
     &           +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
         COUT(6)=(1.-ALPHA)*(-PCDATP(IPCOMP,2))*CSVP(INOD1,1)
     &           +((1.-ALPHA)*PCDATP(IPCOMP,2)-CM/TIMSEC)*CSVP(INOD2,1)
     &           -ALPHA*Q-(1.-ALPHA)*PCQP(INOD2)

C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD2)=Q

C Establish "containment loss" data
         QDATA(IPCOMP)=0.

C Establish additional output variables
         napdat(ipcomp)=5
         PCAOUT(IPCOMP,1)=IONOFF
         PCAOUT(IPCOMP,2)=FGAS
         PCAOUT(IPCOMP,3)=ETA
         PCAOUT(IPCOMP,4)=PHIW
         PCAOUT(IPCOMP,5)=PHISB

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-1.
         COUT(3)=1.
         COUT(4)=-PCONDR(ICON1)
         COUT(5)=0.
         COUT(6)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=1.
         COUT(4)=0.
         COUT(5)=0.
         COUT(6)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH boiler & on/off control'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1)
            WRITE(ITU,*) ' IONOFF = ',IONOFF,' (-)'
            WRITE(ITU,*) ' FGAS   = ',FGAS,' (m^3/s)'
            WRITE(ITU,*) ' ETA    = ',ETA,' (-)'
            WRITE(ITU,*) ' PHIW   = ',PHIW,' (W)'
            WRITE(ITU,*) ' PHISB  = ',PHISB,' (W)'
            WRITE(ITU,*) ' GASCO  = ',FGAS*TIMSEC,' (m^3)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP25C'

      RETURN
      END

C ******************** CMP26C ********************

C CMP26C generates for plant component IPCOMP with plant db code 260 ie.
C 2 node (ISV=20) WCH boiler with aquastat control
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 fuel mass flow rate (kg/s)
C            2 volumetric ratio CO2 in flue gases during operation (-)
C            3 heat exchange coefficient water/flue gases in
C              nominal conditions (W/K)
C            4 sensitivity coefficient for 3 (-)
C            5 sensitivity coefficient for 3 (-)
C            6 heat loss coefficient to the environment if OFF (W/K)
C            7 heat loss increase to the environment if ON (W/K)
C            8 weighting factor for defining mean water temperature (-)
C            9 fuel nominal mass flow rate (kg/s)
C           10 water nominal mass flow rate (kg/s)
C           11 nominal ratio of CO2 in flue gases (-)
C           12 coefficient for defining specific heat flue gases (J/kgK)
C           13 coefficient for defining specific heat flue gases (J/kgK)
C           14 fuel specific heat (J/kgK)
C           15 fuel heating value (J/kg)
C     CDATA: 1 aquastat set point (C)
C            2 ON/OFF control signal (-)

C     PCDATF/P
C            1 Inter-node fluid heat capacity rate (W/K)

      SUBROUTINE CMP26C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"
#include "net_flow.h"
#include "tdf2.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/BTIME/BTIMEP,BTIMEF
      COMMON/PTIME/PTIMEP,PTIMEF
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      DIMENSION XIN(5),PAR(18),OUT(11),INFO(10)
      DIMENSION VAL(MBITS+2)

      PARAMETER (SMALL=1.0E-20)
      REAL      COUT(MPCOE)
      character outs*124

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP26C'

C Check control data for relevant balance type
      IF(ISTATS.EQ.1.AND.
     &   (CDATA(IPCOMP,1).LE.0..OR.
     &    CDATA(IPCOMP,2).LT.0..OR.CDATA(IPCOMP,2).GT.1.)) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP26C: invalid control data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP26C: unresolvable error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Set up interface parameters and variables for TRNSYS type
         TWSU=CSVF(INOD1,1)
         WMFR=CSVF(INOD1,2)
         CPWT=SHTFLD(3,TWSU)
         XIN(1)=TWSU
         XIN(2)=WMFR
         XIN(3)=PCNTMF(IPCOMP)
         XIN(4)=CDATA(IPCOMP,1)

C If supply water temperature is to be read from a temporal file then set it here. 
         IF(ISUPPLWT(IPCOMP).gt.0)THEN
           itdi=ISUPPLWT(IPCOMP)
           IFOC=itdi
           CALL RCTDFB(0,btimef,VAL,ISD,IFOC,IER)
           CDATA(IPCOMP,1)=VAL(ISD)

C Remove low supply temp if there are instances in temporal file (reset to 5.0).
           IF(CDATA(IPCOMP,1).lt.5.0)CDATA(IPCOMP,1)=5.0
         ENDIF

         XIN(5)=CDATA(IPCOMP,2)
         DO 10 I=1,13
            PAR(I)=BDATA(IPCOMP,I)
   10    CONTINUE
         PAR(14)=CPWT
         PAR(15)=BDATA(IPCOMP,14)
         PAR(16)=BDATA(IPCOMP,15)
C Now activate TRNSYS type representing static boiler model
C which will start from the current node 1 temperature
C so mark node 1 temperature for iteration
         ICSV(INOD1,1)=1
         CSVI(INOD1,1)=CSVF(INOD1,1)
         CALL TYPE6(TIME,XIN,OUT,T,DTDT,PAR,INFO)

C Then calculate net heat input Q
         Q=WMFR*CPWT*(OUT(1)-CSVF(INOD1,1))

C For the case of using measured supply temperatures for the boiler (e.g as in IEA Annex 71), where
C boiler performance is irrelevant, set the output to the required measured temperature.
         IF(ISUPPLWT(IPCOMP).gt.0)THEN
           out(1)=CDATA(IPCOMP,1)
           Q=WMFR*CPWT*(OUT(1)-CSVF(INOD1,1))
         ENDIF

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)/2.
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Inter-node fluid heat capacity rate (W/K)
         PCDATF(IPCOMP,1)=WMFR*CPWT

C Calculate current component time-constant TC
         TC(IPCOMP)=AMAX1(
     &        CM/AMAX1(SMALL,C1),
     &        CM/AMAX1(SMALL,PCDATF(IPCOMP,1)))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self-coupling coefficients,
C  node 1
         COUT(1)=ALPHA*(-C1)-CM/TIMSEC
C  node 2
         COUT(2)=ALPHA*PCDATF(IPCOMP,1)
         COUT(3)=ALPHA*(-PCDATF(IPCOMP,1))-CM/TIMSEC
C then matrix equation cross-coupling coefficient,
         COUT(4)=ALPHA*C1
C and then present-time coefficients (ie. right hand sides)
         COUT(5)=((1.-ALPHA)*PCRP(ICON1)-CM/TIMSEC)*CSVP(INOD1,1)
     &           +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
         COUT(6)=(1.-ALPHA)*(-PCDATP(IPCOMP,1))*CSVP(INOD1,1)
     &           +((1.-ALPHA)*PCDATP(IPCOMP,1)-CM/TIMSEC)*CSVP(INOD2,1)
     &           -ALPHA*Q-(1.-ALPHA)*PCQP(INOD2)

C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD2)=Q

C Establish "containment loss" data
         QDATA(IPCOMP)=0.

C Establish additional output variables
         napdat(ipcomp)=12
         DO 12 IOUT=1,11
            PCAOUT(IPCOMP,IOUT)=OUT(IOUT)
   12    CONTINUE
         PCAOUT(IPCOMP,12)=Q

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-1.
         COUT(3)=1.
         COUT(4)=-PCONDR(ICON1)
         COUT(5)=0.
         COUT(6)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=1.
         COUT(4)=0.
         COUT(5)=0.
         COUT(6)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH boiler & aquastat control'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
            WRITE(ITU,*) ' CDATA1 = ',CDATA(IPCOMP,1)
            WRITE(ITU,*) ' CDATA2 = ',CDATA(IPCOMP,2)
            WRITE(ITU,*) ' OUT,1  = ',OUT(1)
            WRITE(ITU,*) ' OUT,2  = ',OUT(2)
            WRITE(ITU,*) ' OUT,3  = ',OUT(3)
            WRITE(ITU,*) ' OUT,4  = ',OUT(4)
            WRITE(ITU,*) ' OUT,5  = ',OUT(5)
            WRITE(ITU,*) ' OUT,6  = ',OUT(6)
            WRITE(ITU,*) ' OUT,7  = ',OUT(7)
            WRITE(ITU,*) ' OUT,8  = ',OUT(8)
            WRITE(ITU,*) ' OUT,9  = ',OUT(9)
            WRITE(ITU,*) ' OUT,10 = ',OUT(10)
            WRITE(ITU,*) ' OUT,11 = ',OUT(11)
            WRITE(ITU,*) ' Q      = ',Q,' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP26C'

      RETURN
      END

C ******************** CMP27C ********************

C CMP27C generates for plant component IPCOMP with plant db code 270 ie.
C 8 node (ISV=20) WCH radiator
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Radiator exponent (-)
C            2 Nominal heat emission of radiator (W)
C            3 Nominal supply temperature (C)
C            4 Nominal exit temperature (C)
C            5 Nominal environment temperature (C)
C            6 Index of coupled building zone (-)
C            7 Number of walls used for defining Te (-)
C            8 Index of 1st wall for defining Te (-)
C            9 Weighting factor for 1st wall when defining Te (-)
C           10 Index of 2nd wall for defining Te (-)
C           11 Weighting factor for 2nd wall when defining Te (-)
C           12 etc.
C     CDATA: none

C     PCDATF/P
C            1 Inter-node fluid heat capacity rate (W/K)
C            2 Total radiator heat emission (W)

      SUBROUTINE CMP27C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      COMMON/C6/INDCFG
      COMMON/FVALA/TFA(MCOM),QFA(MCOM)
      COMMON/FVALS/TFS(MCOM,MS),QFS(MCOM)

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      logical closea,close,closec

      DIMENSION INOD(8),Q(8)

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP27C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD(1)=NPCDAT(IPCOMP,9)
      DO 10 I=2,8
         INOD(I)=INOD(I-1)+1
   10 CONTINUE

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Initialize the nominal log mean temperature difference DTLM0
         TS0=BDATA(IPCOMP,3)
         TX0=BDATA(IPCOMP,4)
         TE0=BDATA(IPCOMP,5)
         TE0TMP=(TS0-TE0)/(TX0-TE0)
         call eclose(TE0TMP,1.00,0.0001,close)
         IF(ABS(TS0-TE0).LT.SMALL.OR.ABS(TX0-TE0).LT.SMALL
     &      .OR.(TS0-TE0)/(TX0-TE0).LT.SMALL.OR.close) then
           DTLM0=0.   
         ELSE
           DTLM0=(TS0-TX0)/ALOG((TS0-TE0)/(TX0-TE0))
         END IF
         DTAM0=((TS0+TX0)/2.)-TE0

C Then evaluate the current environment temperature
         IF(INDCFG.EQ.2.OR.NINT(BDATA(IPCOMP,6)).EQ.0) THEN
            call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
            IF(closea) THEN
               TE=BDATA(IPCOMP,5)
            ELSE
               TE=PCNTMF(IPCOMP)
            END IF
         ELSE
            IZ=INT(BDATA(IPCOMP,6))
            SUMT=TFA(IZ)
            NW=INT(BDATA(IPCOMP,7))
            SUMW=1.
            IF(NW.GE.1.) THEN
               DO 12 IW=1,NW
                  W=BDATA(IPCOMP,7+IW*2)
                  SUMW=SUMW+W
                  SUMT=SUMT+W*TFS(IZ,NINT(BDATA(IPCOMP,6+IW*2)))
   12          CONTINUE
            END IF
            TE=SUMT/SUMW
         END IF
C As the current log mean temperature difference DTLM will be based
C on it, mark current node 1 temperature for iteration
         ICSV(INOD(1),1)=1
         CSVI(INOD(1),1)=CSVF(INOD(1),1)

C Now calculate the heat emission at each successive radiator part
C and the total heat emission PCDATF(IPCOMP,2)
         PCDATF(IPCOMP,2)=0.
         IF(CONVAR(ICON1,2).le.0.0001)then
           DO 15 IP=1,8
               TX=CSVF(INOD(IP),1)
               DTR=AMAX1(SMALL,((TX-TE)/DTAM0))
               Q(IP)=BDATA(IPCOMP,2)/8.*DTR**BDATA(IPCOMP,1)
               PCDATF(IPCOMP,2)=PCDATF(IPCOMP,2)+Q(IP)
   15      CONTINUE
         ELSE
           DO 14 IP=2,8
             TS=CSVF(INOD(IP-1),1)
             TX=CSVF(INOD(IP),1)
             if(DTLM0.eq.0)THEN
               DTAM=((TS+TX)/2.)-TE
               DTR=AMAX1(SMALL,(DTAM/DTAM0))
               Q(IP)=BDATA(IPCOMP,2)/7.*DTR**BDATA(IPCOMP,1)
               PCDATF(IPCOMP,2)=PCDATF(IPCOMP,2)+Q(IP)
             else
               TXTMP=(tx-te)
               call eclose(TXTMP,0.00,0.0001,close)
               if(.NOT.close) then
                 TSTMP=(TS-TE)/(TX-TE)
                 call eclose(TSTMP,1.00,0.0001,close)
                 IF(ABS(TS-TE).LT.SMALL.OR.ABS(TX-TE).LT.SMALL
     &             .OR.(TS-TE)/(TX-TE).LT.SMALL.OR.close)then
                    DTLM=0.
                 ELSE
                    DTLM=(TS-TX)/ALOG((TS-TE)/(TX-TE))
                 END IF
               endif
               DTR=AMAX1(SMALL,(DTLM/DTLM0))
               Q(IP)=BDATA(IPCOMP,2)/7.*DTR**BDATA(IPCOMP,1)
               PCDATF(IPCOMP,2)=PCDATF(IPCOMP,2)+Q(IP)
             end if
   14      CONTINUE
         END IF

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)/8.
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))
C Inter-node fluid heat capacity rate (W/K)
         PCDATF(IPCOMP,1)=CSVF(INOD(1),2)*SHTFLD(3,CSVF(INOD(1),1))

C Calculate current component time-constant TC, based on node 1 only
         TC(IPCOMP)=CM/AMAX1(SMALL,C1)
         RCC = AMAX1(SMALL,C1) / PCRP(ICON1)

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(C1.lt.0.00001) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(RCC.GT.2.OR.RCC.LT.0.5)THEN
              IF(TIMSEC.LE.300) THEN 
                ALPHA=1.
            ELSE
                 ALPHA=RATIMP
              END IF  
            ELSE IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

         call eclose(C1,0.00,0.0001,closec)
         if(closec)ALPHA=1.

C Establish matrix equation self-coupling coefficients,
C  node 1
         COUT(1)=ALPHA*(-C1)-CM/TIMSEC
C  nodes 2 to 8
         COEF1=ALPHA*PCDATF(IPCOMP,1)
         COEF2=ALPHA*(-PCDATF(IPCOMP,1))-CM/TIMSEC
         DO 20 IP=2,8
            COUT(2*IP-2)=COEF1
            COUT(2*IP-1)=COEF2
   20    CONTINUE
C then matrix equation cross-coupling coefficient,
         COUT(16)=ALPHA*C1
C and then present-time coefficients (ie. right hand sides)
C  node 1
         IF(CONVAR(ICON1,2).gt.0.0001)then
           COUT(17)=((1.-ALPHA)*PCRP(ICON1)-CM/TIMSEC)*CSVP(INOD(1),1)
     &            +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
         END IF
C  nodes 2 to 8
         COEF1=(1.-ALPHA)*(-PCDATP(IPCOMP,1))
         COEF2=(1.-ALPHA)*PCDATP(IPCOMP,1)-CM/TIMSEC
         IF(CONVAR(ICON1,2).gt.0.0001)then
           DO 22 IP=2,8
             COUT(16+IP)=COEF1*CSVP(INOD(IP-1),1)
     &                 +COEF2*CSVP(INOD(IP),1)
     &                 +ALPHA*Q(IP)+(1.-ALPHA)*PCQP(INOD(IP))
   22      CONTINUE
         ELSE
           DO 23 IP=1,8
             COUT(16+IP)=COEF1*CSVP(INOD(IP-1),1)
     &                 +COEF2*CSVP(INOD(IP),1)
     &                 +ALPHA*Q(IP)+(1.-ALPHA)*PCQP(INOD(IP))
   23      CONTINUE
         END IF

C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         DO 24 IP=1,8
            PCQF(INOD(IP))=Q(IP)
   24    CONTINUE

C Establish "containment loss" data
         QDATA(IPCOMP)=ALPHA*PCDATF(IPCOMP,2)+
     &                 (1.-ALPHA)*PCDATP(IPCOMP,2)

C Establish additional output variables
         napdat(ipcomp)=9
         PCAOUT(IPCOMP,1)=TE
         PCAOUT(IPCOMP,2)=DTLM
         DO 26 IP=2,8
            PCAOUT(IPCOMP,1+IP)=Q(IP)
   26    CONTINUE

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         DO 30 IP=2,8
            COUT(2*IP-2)=-1.
            COUT(2*IP-1)=1.
   30    CONTINUE
         COUT(16)=-PCONDR(ICON1)
         COUT(17)=0.
         DO 32 IP=2,8
            COUT(16+IP)=0.
   32    CONTINUE
C
C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         DO 40 IP=2,8
            COUT(2*IP-2)=0.
            COUT(2*IP-1)=1.
   40    CONTINUE
         COUT(16)=0.
         COUT(17)=0.
         DO 42 IP=2,8
            COUT(16+IP)=0.
   42    CONTINUE
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 8 node (ISV=20) WCH radiator'
         WRITE(ITU,*) ' Matrix node(s) ',(INOD(I),I=1,8)
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' Q(I)   = ',(Q(I),I=1,8),' (W)'
            WRITE(ITU,*) ' Qtot   = ',PCDATF(IPCOMP,2),' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=24
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP27C'

      RETURN
      END

C ******************** CMP28C ********************

C CMP28C generates for plant component IPCOMP with plant db code 280 ie.
C 1 node (ISV=29) Oil-filled electric panel radiator
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Radiator exponent (-)
C            2 Nominal heat emission of radiator (W)
C            3 Nominal radiator temperature (C)
C            4 Nominal environment temperature (C)
C            5 Index of coupled building zone (-)
C            6 Number of walls used for defining Te (-)
C            7 Index of 1st wall for defining Te (-)
C            8 Weighting factor for 1st wall when defining Te (-)
C            9 Index of 2nd wall for defining Te (-)
C           10 Weighting factor for 2nd wall when defining Te (-)
C           11 etc.
C     CDATA: Supplied electric energy (W).
C
C     PCDATF(IPCOMP,1): Supplied electrical energy (W).

      SUBROUTINE CMP28C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

C Electrical details for specified plant components
      common/elpcp/NPEL,PFP(mpcom),IPFP(mpcom),PWRP(mpcom),
     &BVOLTP(mpcom),IPHP(mpcom)

      COMMON/C6/INDCFG
      COMMON/FVALA/TFA(MCOM),QFA(MCOM)
      COMMON/FVALS/TFS(MCOM,MS),QFS(MCOM)

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP21C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      INOD1=NPCDAT(IPCOMP,9)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Initialize the nominal log mean temperature difference DTLM0
         TR0=BDATA(IPCOMP,3)
         TE0=BDATA(IPCOMP,4)
         DTLM0=TR0-TE0

C Then evaluate the current environment temperature
         IF(INDCFG.EQ.2.OR.NINT(BDATA(IPCOMP,5)).EQ.0) THEN
            call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
            IF(closea) THEN
               TE=BDATA(IPCOMP,4)
            ELSE
               TE=PCNTMF(IPCOMP)
            END IF
         ELSE
            IZ=INT(BDATA(IPCOMP,5))
            SUMT=TFA(IZ)
            NW=int(BDATA(IPCOMP,6))
            SUMW=1.
            IF(NW.GE.1.) THEN
               DO 10 IW=1,NW
                  W=BDATA(IPCOMP,6+IW*2)
                  SUMW=SUMW+W
                  SUMT=SUMT+W*TFS(IZ,NINT(BDATA(IPCOMP,5+IW*2)))
   10          CONTINUE
            END IF
            TE=SUMT/SUMW
         END IF

C And the current log mean temperature difference DTLM based on
C current node 1 temp., so mark node 1 temperature for iteration
         ICSV(INOD1,1)=1
         CSVI(INOD1,1)=CSVF(INOD1,1)
         TR=CSVF(INOD1,1)
         DTLM=TR-TE

C Then calculate radiator heat emission Q
         DTR=AMAX1(SMALL,(DTLM/DTLM0))
         QR=BDATA(IPCOMP,2)*DTR**BDATA(IPCOMP,1)

C Establish required electrical energy.
         QE=CDATA(IPCOMP,1)

C Establish heat capacity of component mass CM (J/K).
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/
     &         AMAX1(SMALL,(QR/AMAX1(SMALL,AMAX1(SMALL,(TR-TE)))))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self-coupling coefficients,
C  node 1
         COUT(1)=-CM/TIMSEC

C and then present-time coefficients (ie. right hand sides)
         COUT(2)=(-CM/TIMSEC)*CSVP(INOD1,1)
     &           +ALPHA*QR+(1.-ALPHA)*PCQP(INOD1)
     &           -ALPHA*QE-(1.-ALPHA)*PCDATF(IPCOMP,1)

C Store "environment" variables future values
         PCQF(INOD1)=QR
         PCDATF(IPCOMP,1)=QE

         PWRP(IPCOMP)=-ABS(QE)
         IEMODEL=1
         CALL EMACH(IPCOMP,IEMODEL,PWRP(IPCOMP),PQ,PA)
         PWRQ=PQ
C Establish "containment loss" data
         QDATA(IPCOMP)=ALPHA*QR+(1.-ALPHA)*PCQP(INOD1)

C Establish additional output variables
         napdat(ipcomp)=6
         PCAOUT(IPCOMP,1)=TE
         PCAOUT(IPCOMP,2)=DTLM
         PCAOUT(IPCOMP,3)=QR
         PCAOUT(IPCOMP,4)=TR
         pcaout(ipcomp,5)=qe
         pcaout(ipcomp,6)=pwrq

C 1st phase mass (ie. "water") balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
      endif

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=9) Oil-filled electric radiator'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  (NONE)'
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' TR     = ',TR,' (C)'
            WRITE(ITU,*) ' TE     = ',TE,' (C)'
            WRITE(ITU,*) ' DTLM   = ',DTLM,' (K)'
            WRITE(ITU,*) ' QR      = ',QR,' (W)'
            WRITE(ITU,*) ' QE      = ',QE,' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=2
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP28C'
      RETURN
      END

c ******************** CMP29C ********************

c 'cmp29c' generates for plant component IPCOMP with plant db code 290 ie.
c 1 node (ISV=20) WCH  valve.
c matrix equation coefficients COUT (in order: self-coupling, cross-
c coupling, and present-time coefficients) for energy balance (ISTATS=1),
c 1st phase mass balance (ISTATS=2)
c     adata: 1 Component total mass (kg)
c            2 Mass weighted average specific heat (J/kgK)
c            3 UA modulus (W/K)
c     bdata: none
c     cdata: mass flow fraction entering damper.

      subroutine cmp29c(ipcomp,cout,istats)
#include "plant.h"
#include "building.h"

      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/pctime/timsec
      common/pctc/tc(mpcom)

      common/pcequ/impexp,ratimp

      common/c9/npcomp,nci(mpcom),cdata(mpcom,mmiscd)
      common/c10/npcon,ipc1(mpcon),ipn1(mpcon),ipct(mpcon),
     &           ipc2(mpcon),ipn2(mpcon),pcondr(mpcon),pconsd(mpcon,2)
      common/c12ps/npcdat(mpcom,9),ipofs1(mcoefg),ipofs2(mcoefg,mpvar)
      common/pdbdt/adata(mpcom,madata),bdata(mpcom,mbdata)
      common/pcval/csvf(mpnode,mpvar),csvp(mpnode,mpvar)
      common/pcvar/pctf(mpcon),pcrf(mpcon),puaf(mpnode),pcqf(mpnode),
     &             pcntmf(mpcom),
     &             pctp(mpcon),pcrp(mpcon),puap(mpnode),pcqp(mpnode),
     &             pcntmp(mpcom)
      common/pcond/convar(mpcon,mconvr),icontp(mpcon),
     &             icondx(mpcom,mnodec,mpconc)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      parameter (small=1.0e-15)
      real      cout(mpcoe)
      logical closea

c Trace output
      if(itc.gt.0.and.nsinc.ge.itc.and.nsinc.le.itcf.and.
     &   ITRACE(37).ne.0) write(itu,*) ' Entering subroutine CMP29C'

c Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      icon1=icondx(ipcomp,1,1)
      inod1=npcdat(ipcomp,9)

c Initialise mass flow fraction entering damper.
         fm=cdata(ipcomp,1)

c Check for bad data.
c Note that very small values of 'fm' sometimes causes unpredictable
c results. For now always assume a min value of 0.1 or 10%.
         if(fm.gt.1.0) fm=1.0
         if(fm.lt.1E-06) fm=1E-06

c Generate coefficients for energy balance equation
      if(istats.eq.1) then

c First initialize UA modulus (for calculation of containment heat loss)
         ua=adata(ipcomp,3)
         call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
         IF(closea) UA=0.



c Establish heat capacity of component mass CM (J/K) and
c fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         cm=adata(ipcomp,1)*adata(ipcomp,2)
         c1=pcondr(icon1)*convar(icon1,2)*shtfld(1,convar(icon1,1))


c Calculate current component time-constant TC
         tc(ipcomp)=cm/amax1(small,(c1+ua))

c Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         if(impexp.eq.1) then
            alpha=1.
         else if(impexp.eq.2) then
            alpha=ratimp
         else if(impexp.eq.3) then
            if(timsec.gt.0.63*tc(ipcomp)) then
               alpha=1.
            else
               alpha=ratimp
            end if
         else if(impexp.eq.4) then
            cm=0.
            alpha=1.
         end if

c Establish matrix equation self- and cross-coupling coefficients
         cout(1)=alpha*(-c1-ua)-cm/timsec
         cout(2)=alpha*c1

c and then present-time coefficient (ie. right hand side)
         cout(3)=((1.-alpha)*(pcrp(icon1)+puap(inod1))
     &              -cm/timsec)*csvp(inod1,1)
     &             +(1.-alpha)*(-pcrp(icon1))*pctp(icon1)
     &             -alpha*ua*pcntmf(ipcomp)
     &             -(1.-alpha)*puap(inod1)*pcntmp(ipcomp)

c Store "environment" variables future values
         puaf(inod1)=ua
         pctf(icon1)=convar(icon1,1)
         pcrf(icon1)=c1

c 1st phase mass (ie. water) balance coefficients
      else if(istats.eq.2) then
         cout(1)=1.
         cout(2)=-pcondr(icon1)*fm
         cout(3)=0.

c 2nd phase mass (ie. none) balance coefficients
      else if(istats.eq.3) then
         cout(1)=1.
         cout(2)=0
         cout(3)=0.
      end if

C Save additional output.
      napdat(ipcomp)=1
      pcaout(ipcomp,1)=fm

c Trace output
      if(itc.gt.0.and.nsinc.ge.itc.and.nsinc.le.itcf.and.
     &   itrace(37).ne.0) then
         write(itu,*) ' Component      ',ipcomp,':'
         write(itu,*) ' 1 node (ISV=20) WCH valve'
         write(itu,*) ' Matrix node(s) ',inod1
         write(itu,*) ' Connection(s)  ',icon1
         if(istats.eq.1) then
            write(itu,*) ' CM     = ',cm,' (J/K)'
            write(itu,*) ' C1     = ',c1,' (W/K)'
            write(itu,*) ' TC     = ',TC(IPCOMP),' (s)'
            write(itu,*) ' ALPHA  = ',alpha,' (-)'
            write(itu,*) ' UA     = ',ua,' (W/K)'
            write(itu,*) ' PCNTMF = ',pcntmf(ipcomp),' (C)'
            write(itu,*) ' CDATA  = ',cdata(ipcomp,1)
         end if
         write(itu,*) ' Matrix coefficients for ISTATS = ',istats
         nitms=3
         write(itu,*) (cout(i),i=1,nitms)
         if(itu.eq.iuout) then
            ix1=(ipcomp/4)*4
            if(ix1.eq.ipcomp.or.ipcomp.eq.npcomp) call epagew
         end if
      end if

      if(itc.gt.0.and.nsinc.ge.itc.and.nsinc.le.itcf.and.
     &   ITRACE(37).ne.0) write(itu,*) ' Leaving subroutine CMP29C'

      return
      end

C ******************** CMP31C ********************

C CMP31C generates for plant component IPCOMP with plant db code 310 ie.
C 2 node (ISV=20) WCH calorifier with electric emersion heater
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 UA modulus
C            4 Mass of water encapsulated in tubes (kg)
C     BDATA: 1 Coil internal heat transfer surface area (m^2)
C            2 Coil internal heat transfer coefficient (W/m^2K)
C            3 Coil external heat transfer surface area (m^2)
C            4 Coil external heat transfer coefficient (W/m^2k)

      SUBROUTINE CMP31C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PTIME/PTIMEP,PTIMEF
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      character outs*124
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP31C'

C Check control data for relevant balance type
      IF(ISTATS.EQ.1.AND.
     &   (CDATA(IPCOMP,1).LT.0..OR.CDATA(IPCOMP,1).GT.1.)) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP31C: invalid control data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP31C: unresolvable error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      ICON2=ICONDX(IPCOMP,2,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

        ua=adata(ipcomp,3)
        call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
        IF(closea) UA=0.

C Determine flux transfer between tubes and tank.
        RI=1./(BDATA(IPCOMP,1)*BDATA(IPCOMP,2))
        RO=1./(BDATA(IPCOMP,3)*BDATA(IPCOMP,4))
        H=1./(RI+RO)

C Heat flux based on temperatures so mark 1st and 2nd nodal temps
C and flows for iteration.
        ICSV(INOD1,1)=1
        CSVI(INOD1,1)=CSVF(INOD1,1)
        ICSV(INOD2,1)=1
        CSVI(INOD2,1)=CSVF(INOD2,1)
        ICSV(INOD1,2)=1
        CSVI(INOD1,2)=CSVF(INOD1,2)
        ICSV(INOD2,2)=1
        CSVI(INOD2,2)=CSVF(INOD2,2)

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
         CW=SHTFLD(3,CONVAR(ICON2,1))*ADATA(IPCOMP,4)
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))
         C2=PCONDR(ICON2)*CONVAR(ICON2,2)*SHTFLD(3,CONVAR(ICON2,1))


C Calculate current component time-constant TC
         CMAX=AMAX1(CM,CW)
         TC(IPCOMP)=CMAX/AMAX1(SMALL,(C1+C2))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            CW=0.
            ALPHA=1.
         END IF

C Establish matrix equation self-coupling coefficients,
         COUT(1)=ALPHA*(-C1-UA-H)-CM/TIMSEC
         COUT(2)=ALPHA*H
         COUT(3)=ALPHA*H
         COUT(4)=ALPHA*(-C2-H)-CW/TIMSEC

C then matrix equation cross-coupling coefficients,
         COUT(5)=ALPHA*C1
         COUT(6)=ALPHA*C2

C and then present-time coefficients (ie. right hand sides)
         COUT(7)=((1.-ALPHA)*(PCRP(ICON1)+UA+H)-CM/TIMSEC)
     &           *CSVP(INOD1,1)
     &           -(1.-ALPHA)*H*CSVP(INOD2,1)
     &           +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &           -(1.-ALPHA)*UA*PCNTMP(IPCOMP)
     &           -ALPHA*UA*PCNTMF(IPCOMP)

         COUT(8)=((1.-ALPHA)*(PCRP(ICON2)+H)-CW/TIMSEC)*CSVP(INOD2,1)
     &           -(1.-ALPHA)*H*CSVP(INOD1,1)
     &           -(1.-ALPHA)*PCRP(ICON2)*PCTP(ICON2)

C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCTF(ICON2)=CONVAR(ICON2,1)
         PCRF(ICON1)=C1
         PCRF(ICON2)=C2

C Establish "containment loss" data
        presentTemp=0.5*(csvp(inod1,1)+csvp(inod2,1))
        futureTemp =0.5*(csvf(inod1,1)+csvf(inod2,1))
        QDATA(IPCOMP)=UA*(alpha*(futureTemp-pcntmp(ipcomp))+
     &                (1.-alpha)*(presentTemp-pcntmp(ipcomp)))

C Establish "containment loss" data
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iConvective)
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iRadiant)

C Establish additional output variables
         QCOIL=H*(CSVF(INOD2,1)-CSVF(INOD1,1))
         napdat(ipcomp)=1
         PCAOUT(IPCOMP,1)=QCOIL

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
         COUT(4)=1.
         COUT(5)=-PCONDR(ICON1)
         COUT(6)=-PCONDR(ICON2)
         COUT(7)=0.
         COUT(8)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
         COUT(4)=1.
         COUT(5)=0.
         COUT(6)=0.
         COUT(7)=0.
         COUT(8)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH calorifier'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1,ICON2
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' C2     = ',C2,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1)
            WRITE(ITU,*) ' QCOIL  = ',QCOIL,' (W)'
            WRITE(ITU,*) ' HTC  = ',H,' (W/K)'
            WRITE(ITU,*) ' TDIFF  = ',
     &(CSVF(INOD1,1)-CSVF(INOD2,1)),' (W/K)'

         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP31C'

      RETURN
      END

C ******************** CMP32C ********************
C CMP40C generates for plant component IPCOMP with plant db code 320 ie.
C 2 node (ISV=20) WCH generic liquid-liquid heat exchanger.
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)

C     ADATA: 1 Mass of node 1 (solids+liquid) (kg)
C            2 Mass of node 2 (solids+liquid) (kg)
C            3 Overall heat transfer coefficient (W/m^2K)
C            4 Heat transfer surface area (m^2)
C            5 UA modulus for component (W/K)

C     BDATA: 1 Flow arrangement index (1-7)

C     CDATA: NONE

C The heat exchanger model uses the NTU method to determine the exchanger
C effectiveness. Several different configurations of heat exchanger are
C dealt with in the one model;
C 1 - Parallel flow
C 2 - Counter flow
C 3 - Shell and tube
C 4 - Cross flow with both fluids mixed
C 5 - Cross flow with both fluids unmixed
C 6 - Cross flow with Cmin unmixed
C 7 - Cross flow with Cmax unmixed
C
C Important assumptions made in the model are that the exchanger mass is small
C compared to the mass of encapsulated fluids, it is also assumed that the
C interface thermal resistance is negligible compared to the film resistances.
C Fouling resistances are neglected. Overall UA value is assumed constant.

C REF:OPSYS - simulation of thermal systems P169.
C
C
      SUBROUTINE CMP32C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      PARAMETER (SMALL=1.0E-20)
      REAL      COUT(MPCOE),K,NTU

      LOGICAL CLOSE
C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP32C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      ICON2=ICONDX(IPCOMP,2,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C Heat flux based on temperatures so mark 1st and 2nd nodal temps
C and flows for iteration.
      ICSV(INOD1,1)=1
      CSVI(INOD1,1)=CSVF(INOD1,1)
      ICSV(INOD2,1)=1
      CSVI(INOD2,1)=CSVF(INOD2,1)
      ICSV(INOD1,2)=1
      CSVI(INOD1,2)=CSVF(INOD1,2)
      ICSV(INOD2,2)=1
      CSVI(INOD2,2)=CSVF(INOD2,2)
      ICSV(INOD1,3)=1
      CSVI(INOD1,3)=CSVF(INOD1,3)
      ICSV(INOD2,3)=1
      CSVI(INOD2,3)=CSVF(INOD2,3)

C Total mass flow rate into node 1
      TOTAL_MASS_FR1 = PCONDR(ICON1) *
     &                (CONVAR(ICON1,2) + CONVAR(ICON1,3))

C Total mass flow rate into node 2
      TOTAL_MASS_FR2 = PCONDR(ICON2) *
     &                (CONVAR(ICON2,2) + CONVAR(ICON2,3))

C Mass fraction of glycol in incoming flow to node 1
      GLYCOL_MASS_FRACTION1 =
     &            100 * CONVAR(ICON1,3) / (TOTAL_MASS_FR1 + SMALL)

C Mass fraction of glycol in incoming flow to node 2
      GLYCOL_MASS_FRACTION2 =
     &            100 * CONVAR(ICON2,3) / (TOTAL_MASS_FR2 + SMALL)

C Find volume fraction of glycol in mixture of incoming flow for node 1
      GLYCOL_VOL_FRACTION1 =
     &GLYCOL_VOL_FRACTION_FIND(GLYCOL_MASS_FRACTION1,CONVAR(ICON1,1))

C Find volume fraction of glycol in mixture of incoming flow for node 2
      GLYCOL_VOL_FRACTION2 =
     &GLYCOL_VOL_FRACTION_FIND(GLYCOL_MASS_FRACTION2,CONVAR(ICON2,1))

C Boiling temperature of glycol mixture for node 1
      TBOIL1 = GLYCOL_WATER_TBOIL(GLYCOL_VOL_FRACTION1)

C Boiling temperature of glycol mixture for node 2
      TBOIL2 = GLYCOL_WATER_TBOIL(GLYCOL_VOL_FRACTION2)

C Freezing temperature of glycol-water mixture for node 1
      TFREEZE1 = GLYCOL_WATER_TFREEZE(GLYCOL_VOL_FRACTION1)

C Freezing temperature of glycol-water mixture for node 2
      TFREEZE2 = GLYCOL_WATER_TFREEZE(GLYCOL_VOL_FRACTION2)

C Set incoming flow temperature using upper and lower bounds for node 1
      TEMP_INFLOW1 = MAX(TFREEZE1,MIN(TBOIL1,CONVAR(ICON1,1)))

C Set incoming flow temperature using upper and lower bounds for node 2
      TEMP_INFLOW2 = MAX(TFREEZE2,MIN(TBOIL2,CONVAR(ICON2,1)))

C Set specific heat of incoming flow for node 1
      GLY_WATER_CP1 = GLYCOL_WATER_CP(GLYCOL_VOL_FRACTION1,TEMP_INFLOW1)

C Set specific heat of incoming flow for node 2
      GLY_WATER_CP2 = GLYCOL_WATER_CP(GLYCOL_VOL_FRACTION2,TEMP_INFLOW2)

C Set temperature of node 1
      TEMP_NODE1 = MAX(TFREEZE1,MIN(TBOIL1,CSVF(INOD1,1)))

C Set temperature of node 2
      TEMP_NODE2 = MAX(TFREEZE2,MIN(TBOIL2,CSVF(INOD2,1)))

C Set specific heat of node 1
      GLY_WATER_CP_NODE1 =
     &          GLYCOL_WATER_CP(GLYCOL_VOL_FRACTION1,TEMP_NODE1)

C Set specific heat of node 2
      GLY_WATER_CP_NODE2 =
     &          GLYCOL_WATER_CP(GLYCOL_VOL_FRACTION2,TEMP_NODE2)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Establish fluid thermal capacity rates
        CW1=GLY_WATER_CP_NODE1*ADATA(IPCOMP,1)
        CW2=GLY_WATER_CP_NODE2*ADATA(IPCOMP,2)
        C1=TOTAL_MASS_FR1*GLY_WATER_CP1
        C2=TOTAL_MASS_FR2*GLY_WATER_CP2

C Establish the overall K value
        K=ADATA(IPCOMP,3)*ADATA(IPCOMP,4)

C Establish heat loss modulus
        UA=ADATA(IPCOMP,5)
        CALL ECLOSE(PCNTMF(IPCOMP),99.,0.0001,CLOSE)
        IF(CLOSE) UA=0.0

C Check for zero flow.
C If zero flow found then set transfered flux to zero and skip.
        IF(C1.LT.SMALL.OR.C2.LT.SMALL) THEN
          HFLX=0.0
          GOTO 777
        ENDIF

C Find Cmin and Cmax
        CMIN=AMIN1(C1,C2)
        CMAX=AMAX1(C1,C2)

C Determine the NTU's and the thermal capacities ratio TCRAT
        NTU=K/CMIN
        TCRAT=CMIN/CMAX

C Determine the flow regime operational within the heat exchanger.
        IFLWR=INT(BDATA(IPCOMP,1))

C Establish the Effectiveness of the exchanger.
         IF(IFLWR.EQ.1) THEN

C Parallel Flow.
           EFF=(1.-EXP(-NTU*(1.+TCRAT)))/(1.+TCRAT)
        ELSEIF(IFLWR.EQ.2) THEN
C Counter Flow.
         A1=(1.-TCRAT*(EXP(-NTU*(1.-TCRAT))))
         IF(A1.LT.SMALL) A1=SMALL
         EFF=(1.-EXP(-NTU*(1.-TCRAT)))/A1
     &
        ELSEIF(IFLWR.EQ.3) THEN

C Shell and Tube.
          A1=1.+EXP(-NTU*((1.+TCRAT**2)**0.5))
          A2=1.-EXP(-NTU*((1.+TCRAT**2)**0.5))
          EFF=2./(1.+TCRAT+((1.+TCRAT**2)**0.5)*(A1/A2))
        ELSEIF(IFLWR.EQ.4) THEN

C Cross FLow with both fluids mixed.
          A1=NTU/(1.-EXP(-NTU))
          A2=TCRAT*NTU/(1.-EXP(-TCRAT*NTU))
          EFF=NTU/(A1+A2-1.)
        ELSEIF (IFLWR.EQ.5) THEN

C Cross flow with both fluids unmixed.
          A1=((1.-EXP(-TCRAT*NTU**1.22))/TCRAT*NTU**0.22)
          EFF=1.-EXP(-A1)
        ELSEIF(IFLWR.EQ.6) THEN

C Cross flow with CMIN=unmixed.
          A1=(1.-EXP(-NTU))
          EFF=(1.- EXP(-TCRAT*A1))/TCRAT
        ELSEIF(IFLWR.EQ.7) THEN
C Cross flow with CMAX=unmixed.

          A1=(1.-EXP(-TCRAT*NTU))/TCRAT
          EFF=1.-EXP(-A1)
        ENDIF

C Establish the heat flux transferred between the nodes.
        HFLX=EFF*CMIN*(CONVAR(ICON2,1)-CONVAR(ICON1,1))
  777   CONTINUE
        TDIF=AMAX1(CSVF(INOD1,1)-CSVF(INOD2,1),SMALL)
C Calculate current component time-constant TC
        TC(IPCOMP)=AMAX1(
     &CW1/AMAX1(SMALL,(C1+ABS(HFLX)/TDIF)),
     &CW2/AMAX1(SMALL,(C2+ABS(HFLX)/TDIF)))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
           ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
           ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
             ALPHA=1.
          ELSE
             ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
            CW1=0.
            CW2=0.
            ALPHA=1.
        END IF

C Establish matrix equation self- and cross-coupling coefficients.
        COUT(1)=ALPHA*(-C1-UA/2.)-CW1/TIMSEC
        COUT(2)=ALPHA*(-C2-UA/2.)-CW2/TIMSEC
C Matrix cross coupling coefficients.
        COUT(3)=ALPHA*C1
        COUT(4)=ALPHA*C2
C Establish the present and known coefficient i.e. RHS
        COUT(5)=((1.-ALPHA)*(PCRP(ICON1)+UA/2.)-CW1/TIMSEC)
     &          *CSVP(INOD1,1)
     &          -(1.-ALPHA)*PCRP(ICON1)*PCTP(ICON1)
     &          -(1.-ALPHA)*(UA/2.)*PCNTMP(IPCOMP)
     &          -ALPHA*(UA/2.)*PCNTMF(IPCOMP)
     &          -ALPHA*HFLX
     &          -(1.-ALPHA)*PCQP(INOD1)
         COUT(6)=((1.-ALPHA)*(PCRP(ICON2)+UA/2.)-CW1/TIMSEC)
     &           *CSVP(INOD2,1)
     &          -(1.-ALPHA)*PCRP(ICON2)*PCTP(ICON2)
     &          -(1.-ALPHA)*(UA/2.)*PCNTMP(IPCOMP)
     &          -ALPHA*(UA/2.)*PCNTMF(IPCOMP)
     &          -ALPHA*(-HFLX)
     &          -(1.-ALPHA)*(-PCQP(INOD2))
C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCTF(ICON2)=CONVAR(ICON2,1)
         PCRF(ICON1)=C1
         PCRF(ICON2)=C2
         PCQF(INOD1)=HFLX
         PCQF(INOD2)=HFLX
C Addition output for results analysis.
         NAPDAT(IPCOMP)=3
         PCAOUT(IPCOMP,1)=HFLX
         PCAOUT(IPCOMP,2)=NTU
         PCAOUT(IPCOMP,3)=EFF

C Establish "containment loss" data
        QDATA(IPCOMP)=UA*(alpha*(csvf(inod1,1)-pcntmf(ipcomp))+
     &                (1.-alpha)*(csvp(inod1,1)-pcntmp(ipcomp)))
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iConvective)
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iRadiant)

C 1st phase mass (ie. "water") balance coefficients

      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=-PCONDR(ICON1)
         COUT(4)=-PCONDR(ICON2)
         COUT(5)=0.
         COUT(6)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=-PCONDR(ICON1)
         COUT(4)=-PCONDR(ICON2)
         COUT(5)=0.
         COUT(6)=0.
      END IF


C Trace.
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=29) WCH generic fluid-fluid HE'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1,ICON2
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CW1     = ',CW1,' (J/K)'
            WRITE(ITU,*) ' CW2     = ',CW2,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/Ks)'
            WRITE(ITU,*) ' C2     = ',C2,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' EFF     = ',EFF,' (-)'
            WRITE(ITU,*) ' NTU     = ',NTU,' (-)'
            WRITE(ITU,*) ' HFLX     = ',HFLX,' (W)'
            WRITE(ITU,*) ' IFLWR     = ',IFLWR,' (-)'
            WRITE(ITU,*) ' TCRAT     = ',TCRAT,' (-)'
            WRITE(ITU,*) ' TDIFF  = ',
     &(CONVAR(ICON2,1)-CONVAR(ICON1,1)),' (W/K)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP32C'

      RETURN
      END

C ******************** CMP33C ********************
C CMP33C generates for plant component IPCOMP with plant db code 320 ie.
C 2 node (ISV=20) WCH generic gas-liquid heat exchanger. This heat exchanger
C is intended for use with the CMP99C CHP unit model, where the input
C for the gas side is a hight temp exhaust gas (T>400 C). The properties
C of the exhaust gas are assumed to be those of high temperature air.
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)

C Node 1 - Gas node.
C Node 2 - Fluid node.

C     ADATA: 1 Mass of node 1 (solids+liquid) (kg)
C            2 Mass of node 2 (solids+liquid) (kg)
C            3 Overall heat transfer coefficient (W/m^2K)
C            4 Heat transfer surface area (m^2)
C            5 UA modulus for component (W/K)

C     BDATA: 1 Flow arrangement index (1-7)

C     CDATA: NONE

C The heat exchanger model uses the NTU method to determine the exchanger
C effectiveness. Several different configurations of heat exchanger are
C dealt with in the one model;
C 1 - Parallel flow
C 2 - Counter flow
C 3 - Shell and tube
C 4 - Cross flow with both fluids mixed
C 5 - Cross flow with both fluids unmixed
C 6 - Cross flow with Cmin unmixed
C 7 - Cross flow with Cmax unmixed
C
C Important assumptions made in the model are that the exchanger mass is small
C compared to the mass of encapsulated fluids, it is also assumed that the
C interface thermal resistance is negligible compared to the film resistances.
C Fouling resistances are neglected. Overall UA value is assumed constant.

C REF:OPSYS - simulation of thermal systems P169.
C
C
      SUBROUTINE CMP33C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      PARAMETER (SMALL=1.0E-20)
      REAL      COUT(MPCOE),K,NTU

      LOGICAL CLOSE
C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP33C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      ICON2=ICONDX(IPCOMP,2,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C Heat flux based on temperatures so mark 1st and 2nd nodal temps
C and flows for iteration.
      ICSV(INOD1,1)=1
      CSVI(INOD1,1)=CSVF(INOD1,1)
      ICSV(INOD2,1)=1
      CSVI(INOD2,1)=CSVF(INOD2,1)
      ICSV(INOD1,2)=1
      CSVI(INOD1,2)=CSVF(INOD1,2)
      ICSV(INOD2,2)=1
      CSVI(INOD2,2)=CSVF(INOD2,2)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Establish fluid thermal capacity rates. Assume the specific
C heat of the combustion products is the following value.
C Cp=1.17kJ/kgK Assuming 200% theoretical (stochiometric) air.
        CW1=1170.0*ADATA(IPCOMP,1)
        CW2=SHTFLD(3,CSVF(INOD2,1))*ADATA(IPCOMP,2)
        C1=PCONDR(ICON1)*CONVAR(ICON1,2)*1170.0
        C2=PCONDR(ICON2)*CONVAR(ICON2,2)*SHTFLD(3,CONVAR(ICON2,1))

C Establish the overall K value
        K=ADATA(IPCOMP,3)*ADATA(IPCOMP,4)

C Establish heat loss modulus
        UA=ADATA(IPCOMP,5)
        CALL ECLOSE(PCNTMF(IPCOMP),99.,0.0001,CLOSE)
        IF(CLOSE) UA=0.0

C Check for zero flow.
C If zero flow found then set transfered flux to zero and skip.
        IF(C1.LT.0.0001.OR.C2.LT.0.0001) THEN
          HFLX=0.0
          GOTO 777
        ENDIF

C Find Cmin and Cmax
        CMIN=AMIN1(C1,C2)
        CMAX=AMAX1(C1,C2)

C Determine the NTU's and the thermal capacities ratio TCRAT
        NTU=K/CMIN
        TCRAT=CMIN/CMAX

C Determine the flow regime operational within the heat exchanger.
        IFLWR=INT(BDATA(IPCOMP,1))

C Establish the Effectiveness of the exchanger.
         IF(IFLWR.EQ.1) THEN

C Parallel Flow.
           EFF=(1.-EXP(-NTU*(1.+TCRAT)))/(1.+TCRAT)
        ELSEIF(IFLWR.EQ.2) THEN
C Counter Flow.
         A1=(1.-TCRAT*(EXP(-NTU*(1.-TCRAT))))
         IF(A1.LT.SMALL) A1=SMALL
         EFF=(1.-EXP(-NTU*(1.-TCRAT)))/A1
     &
        ELSEIF(IFLWR.EQ.3) THEN

C Shell and Tube.
          A1=1.+EXP(-NTU*((1.+TCRAT**2)**0.5))
          A2=1.-EXP(-NTU*((1.+TCRAT**2)**0.5))
          EFF=2./(1.+TCRAT+((1.+TCRAT**2)**0.5)*(A1/A2))
        ELSEIF(IFLWR.EQ.4) THEN

C Cross FLow with both fluids mixed.
          A1=NTU/(1.-EXP(-NTU))
          A2=TCRAT*NTU/(1.-EXP(-TCRAT*NTU))
          EFF=NTU/(A1+A2-1.)
        ELSEIF (IFLWR.EQ.5) THEN

C Cross flow with both fluids unmixed.
          A1=((1.-EXP(-TCRAT*NTU**1.22))/TCRAT*NTU**0.22)
          EFF=1.-EXP(-A1)
        ELSEIF(IFLWR.EQ.6) THEN

C Cross flow with CMIN=unmixed.
          A1=(1.-EXP(-NTU))
          EFF=(1.- EXP(-TCRAT*A1))/TCRAT
        ELSEIF(IFLWR.EQ.7) THEN
C Cross flow with CMAX=unmixed.

          A1=(1.-EXP(-TCRAT*NTU))/TCRAT
          EFF=1.-EXP(-A1)
        ENDIF

C Establish the heat flux transferred between the nodes.
        HFLX=EFF*CMIN*(CONVAR(ICON2,1)-CONVAR(ICON1,1))
  777   CONTINUE
        TDIF=AMAX1(CSVF(INOD1,1)-CSVF(INOD2,1),SMALL)
C Calculate current component time-constant TC
        TC(IPCOMP)=AMAX1(
     &CW1/AMAX1(SMALL,(C1+ABS(HFLX)/TDIF)),
     &CW2/AMAX1(SMALL,(C2+ABS(HFLX)/TDIF)))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
           ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
           ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
             ALPHA=1.
          ELSE
             ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
            CW1=0.
            CW2=0.
            ALPHA=1.
        END IF

C Establish matrix equation self- and cross-coupling coefficients.
        COUT(1)=ALPHA*(-C1-UA/2.)-CW1/TIMSEC
        COUT(2)=ALPHA*(-C2-UA/2.)-CW2/TIMSEC
C Matrix cross coupling coefficients.
        COUT(3)=ALPHA*C1
        COUT(4)=ALPHA*C2
C Establish the present and known coefficient i.e. RHS
        COUT(5)=((1.-ALPHA)*(PCRP(ICON1)+UA/2.)-CW1/TIMSEC)
     &          *CSVP(INOD1,1)
     &          -(1.-ALPHA)*PCRP(ICON1)*PCTP(ICON1)
     &          -(1.-ALPHA)*(UA/2.)*PCNTMP(IPCOMP)
     &          -ALPHA*(UA/2.)*PCNTMF(IPCOMP)
     &          -ALPHA*HFLX
     &          -(1.-ALPHA)*PCQP(INOD1)
         COUT(6)=((1.-ALPHA)*(PCRP(ICON2)+UA/2.)-CW1/TIMSEC)
     &           *CSVP(INOD2,1)
     &          -(1.-ALPHA)*PCRP(ICON2)*PCTP(ICON2)
     &          -(1.-ALPHA)*(UA/2.)*PCNTMP(IPCOMP)
     &          -ALPHA*(UA/2.)*PCNTMF(IPCOMP)
     &          -ALPHA*(-HFLX)
     &          -(1.-ALPHA)*(-PCQP(INOD2))
C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCTF(ICON2)=CONVAR(ICON2,1)
         PCRF(ICON1)=C1
         PCRF(ICON2)=C2
         PCQF(INOD1)=HFLX
         PCQF(INOD2)=HFLX
C Addition output for results analysis.
         NAPDAT(IPCOMP)=3
         PCAOUT(IPCOMP,1)=HFLX
         PCAOUT(IPCOMP,2)=NTU
         PCAOUT(IPCOMP,3)=EFF
C 1st phase mass (ie. "water") balance coefficients

      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=-PCONDR(ICON1)
         COUT(4)=-PCONDR(ICON2)
         COUT(5)=0.
         COUT(6)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=0.
         COUT(4)=0.
         COUT(5)=0.
         COUT(6)=0.
      END IF


C Trace.
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV>20) WCH generic gas-fluid HE'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1,ICON2
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CW1     = ',CW1,' (J/K)'
            WRITE(ITU,*) ' CW2     = ',CW2,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/Ks)'
            WRITE(ITU,*) ' C2     = ',C2,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' EFF     = ',EFF,' (-)'
            WRITE(ITU,*) ' NTU     = ',NTU,' (-)'
            WRITE(ITU,*) ' HFLX     = ',HFLX,' (W)'
            WRITE(ITU,*) ' IFLWR     = ',IFLWR,' (-)'
            WRITE(ITU,*) ' TCRAT     = ',TCRAT,' (-)'
            WRITE(ITU,*) ' TDIFF  = ',
     &(CONVAR(ICON2,1)-CONVAR(ICON1,1)),' (W/K)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP33C'

      RETURN
      END
C ******************** CMP34C ********************
C CMP34C generates for plant component IPCOMP with plant db code 340 ie.
C 3 node (ISV>20) WCH storage water heater. Developed from the model described
C in "Combo/AIMS model description" by Michael Parent;
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 Component UA modulus (for heat loss to environment) (W/K)
C
C Component data.
C     BDATA:
C            1 Burner capacity when ON (W)
C            2 Burner capacity when OFF (standby) (W)
C            3 UAx value for heat transfer to water (W/K)
C
C Fuel data.
C            4 Fuel heating value @ nominal AF ratio (J/kg fuel)
C            5 Stoichiometric air/fuel mass ratio (W/K)
C            6 Excess air (%)
C            7 Fuel specific heat capacity (J/kgK)
C            8 Efficiency at full load (ON) (K)
C
C Combustion products.
C            9 Specific heat of combustion gases with T<537.8C (J/kgK)
C           10 Mass ratio of water in combustion gas per unit mass of fuel (kg/kg)
C
C     CDATA: 1 ON/OFF signal
      SUBROUTINE CMP34C(IPCOMP,COUT,ISTATS)
C
#include "plant.h"
#include "building.h"
#include "site.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/PCTIME/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      COMMON/LASTCTL/CTLDATA(MPCOM,MMISCD)


      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE),MSC,MFCP,LMTD
      LOGICAL CLOSE,CLOSEA,CLOSEB,CLOSEC
      CHARACTER*124 OUTS

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP01C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD

C 2 connections 1,2 to the water storage tank (for DHW and heating system)
      ICON1=ICONDX(IPCOMP,1,1)
      ICON2=ICONDX(IPCOMP,1,2)

C 1 connection 3 to the combustion chamber node (for combustion air)
      ICON3=ICONDX(IPCOMP,2,1)

C 3 - nodes 1) water+casing 2) combustion node 2) flue (outlet) node
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1
      INOD3=NPCDAT(IPCOMP,9)+2

C Mark the nodal temperatures (flue outlet node) for iteration.
      ICSV(INOD3,1)=1
      CSVI(INOD3,1)=CSVF(INOD3,1)

C Generate coefficients for the energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss),
C if there is no containment then zero the UA value.
         UA=ADATA(IPCOMP,3)
         CALL ECLOSE(PCNTMF(IPCOMP),-99.00,0.001,CLOSEA)
         IF(CLOSEA) UA=0.

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))
         C2=PCONDR(ICON2)*CONVAR(ICON2,2)*SHTFLD(3,CONVAR(ICON2,1))

C The flow rate supplied to the component from a fan is the
C combustion air and should = stochiometric air fuel ratio * excess air/100 * fuel flow
         CV=PCONDR(ICON3)*CONVAR(ICON3,3)*SHTFLD(2,CONVAR(ICON3,1))
         CA=PCONDR(ICON3)*CONVAR(ICON3,2)*SHTFLD(1,CONVAR(ICON3,1))

C Burner output at 'ON'
         BURNHI=BDATA(IPCOMP,1)

C Burner output at 'OFF' of 'STANDBY'
         BURNLO=BDATA(IPCOMP,2)

C Heat transfer conductance to water.
         UAx=BDATA(IPCOMP,3)

C Nominal Fuel heating value @ EXCESAIR (J/kg)
         FUELHV=BDATA(IPCOMP,4)

C Stochiometric air fuel mass ratio (kg/kg)
         AIRFLR=BDATA(IPCOMP,5)

C Nominal excess air (%)
         EXSAIR=BDATA(IPCOMP,6)

C Fuel specific heat
         CPFUEL=BDATA(IPCOMP,7)

C Efficiency at fully load (ON)
         EFFX=BDATA(IPCOMP,8)/100.

C Specific heat of combustion products @ TCOMB
         CPLOW=BDATA(IPCOMP,9)

C Mass ratio of H2O in exhaust gas
         EXGH2O=BDATA(IPCOMP,10)

C Calculate the fuel flow rate depending on the ON/OFF control signal.
         CALL ECLOSE(CDATA(IPCOMP,2),0.0,0.0001,CLOSE)
         IF(FUELHV.GT.0.)THEN
           IF(CLOSE)THEN
             FFR=BURNLO/FUELHV
           ELSE
C If fan is not on the previous time step step up burner output to trigger
C the fan.
             IF(CTLDATA(IPCOMP,1).GT.0.01)THEN
               FFR=BURNHI/FUELHV
             ELSE
               FFR=4*BURNLO/FUELHV
             ENDIF
           ENDIF
         ELSE
           WRITE(IUOUT,*)'Error in CMP34C fuel heating value is zero.'
           STOP
         ENDIF
         CTLDATA(IPCOMP,1)=CDATA(IPCOMP,2)

C Check for zero flow and issue warrning
         IF(CONVAR(ICON3,2).LT.SMALL)THEN
           FFR=0.0
           WRITE(OUTS,*)'WARNING: CMP34C no combustion air flow.'
           CALL EDISP(IUOUT,OUTS)
           WRITE(OUTS,*)'fuel supply shut off. '
           CALL EDISP(IUOUT,OUTS)
         ENDIF

C Dry Air Cp
         CPA=SHTFLD(1,CONVAR(ICON3,2))

C Vapour Cp
         CPV=SHTFLD(2,CONVAR(ICON3,3))

C Moisture content
         IF(CONVAR(ICON3,2).GT.0.)THEN
           MSC=CONVAR(ICON3,3)/CONVAR(ICON3,2)
         ELSE

C Check for zero flow.
           MSC=0.0
         ENDIF

C hfg of air at temperature of the inlet
         HFGIN=CNDWAT(CONVAR(ICON3,1))

C Calculate the enthalpy of the inlet fuel and air streams.
         HAIR=CPA*CONVAR(ICON3,1)+MSC*(HFGIN+CPV*CONVAR(ICON3,1))
         HIN=FFR*(FUELHV+CPFUEL*CONVAR(ICON3,1))
     &   +PCONDR(ICON3)*CONVAR(ICON3,2)*HAIR


C Calculate the specific heat of the combustion gases
         CPG=(0.28399*(EXSAIR/100.)**(-0.08244))*4.184*1000.

C Calculate the combustion product flow rate (kg) (the flow rate between nodes
C 2 and 3) air flow + moisture flow + fuel flow.
         MFCP=PCONDR(ICON3)*(CONVAR(ICON3,2)+CONVAR(ICON3,3))+FFR
         FW=FFR*EXGH2O/(PCONDR(ICON3)*(CONVAR(ICON3,2)))
C Calculate the dew point temperature of the combustion gases
         TDEW=19.456*LOG(FW*100)+2.7339

C Calculate the temperature differences  dT1 and dT2
         dT2=CSVF(INOD3,1)-CSVF(INOD1,1)
         IF(dT2.LT.0.0)dT2=0.0
         dT1=CSVF(INOD2,1)-CSVF(INOD1,1)
         IF(dT1.LT.0.0)dT1=0.0
         CLOSE=.FALSE.
         CALL ECLOSE(dT1,dT2,0.01,CLOSEA)
         CALL ECLOSE(dT1,0.00,0.01,CLOSEB)
         CALL ECLOSE(dT2,0.00,0.01,CLOSEC)

C Calculate the temperature drop along the heat exchanger
C using iteration.
         IF(CLOSEA)THEN
           LMTD=(dT1+dT2)*0.5
         ELSEIF(CLOSEB)THEN
           LMTD=0.0
         ELSEIF(CLOSEC)THEN
           LMTD=dT1/2.
         ELSE
           LMTD=(dT2-dT1)/(ALOG(dT2/dT1))
         ENDIF

         IF(LMTD.LT.0.)LMTD=0.0

C Check that the transferred heat flux does not exceed the full load
C efficiency.
         HFLX=UAx*LMTD
         IF(HFLX.GT.EFFX*HIN)HFLX=EFFX*HIN

C Also assign new value to UAx
         IF(LMTD.GT.0.001)THEN
           UAx=EFFX*HIN/LMTD
         ELSE
           UAx=0.0
         ENDIF
         HFLX=UAx*LMTD

C If the calculated outlet temperature < Tdew then reduce the moisture
C content of the outlet air and reduce the sensible heat removed from the
C gas stream
         IF(TDEW.GT.CSVF(INOD3,1))THEN

C Calculate the "equivalent" heat transfer in the condensing region
           Qx=MFCP*CPG*(TDEW-CSVF(INOD3,1))

C Calculate the inlet and outlet enthalpy streams to the condensing region
           Hi=ENTHP2(TDEW,MSC)*1000.
           Ho=Hi-(Qx/MFCP)

C Calculate the outlet temp assuming RH=100%
           Tox=TSATH0(Ho/1000.,patmos)

C Calculate the sensible heat transfer in the condensing region
           Qs=MFCP*CPG*(TDEW-Tox)
           HFLXs=HFLX-(Qx-Qs)
C Calculate the amount of condensation
           CMx=(Qx-Qs)/CNDWAT(CSVF(INOD1,1))
         ELSE
           HFLXs=HFLX
         ENDIF

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(UA+HFLX))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self-and cross-coupling coefficients
C matrix order as follows
C   A  B  C | i  j  k   RHS
C   1  0  0 | 5  6  0    11
C   0  2  0 | 0  0  7  = 12
C   0  3  4 | 0  0  0    13

C Solid node A
         COUT(1)=ALPHA*(-C1-C2-UA)-CM/TIMSEC

C Combustion node B
         COUT(2)=-MFCP*CPG

C Outlet node C
         COUT(3)=(MFCP*CPG)
         COUT(4)=-(MFCP*CPG)

C Cross coupling coefficients
C Node A
         COUT(5)=ALPHA*C1
         COUT(6)=ALPHA*C2

C Node B.
         COUT(7)=(MFCP*CPG)

C and then present-time coefficient (ie. right hand side)
         COUT(8)=((1.-ALPHA)*(PCRP(ICON1)+PCRP(ICON2)+PUAP(INOD1))
     &           -CM/TIMSEC)*CSVP(INOD1,1)
     &           -(1.-ALPHA)*(PCRP(ICON1))*PCTP(ICON1)
     &           -(1.-ALPHA)*(PCRP(ICON2))*PCTP(ICON2)
     &           -ALPHA*UA*PCNTMF(IPCOMP)
     &           -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)
     &           -(1.-ALPHA)*PCQP(INOD1)
     &           -(ALPHA)*HFLX
         COUT(9)=-HIN
     &            +(PCONDR(ICON3)*CONVAR(ICON3,3)+(FFR*EXGH2O))*HFGIN
         COUT(10)=HFLXs

C Store "environment" variables future values
         PUAF(INOD1)=UA
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCTF(ICON2)=CONVAR(ICON2,1)
         PCRF(ICON1)=C1
         PCRF(ICON2)=C2
         PCQF(INOD1)=HFLX

C 1st phase mass (ie. dry air) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=1.
         COUT(4)=-1.
         COUT(5)=-PCONDR(ICON1)
         COUT(6)=-PCONDR(ICON2)
         COUT(7)=-PCONDR(ICON3)
         COUT(8)=0.
         COUT(9)=0.
         COUT(10)=0.

C 2nd phase mass (ie. vapour) balance coefficients.
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=1.
         COUT(4)=-1.
         COUT(5)=0.
         COUT(6)=0.
         COUT(7)=0.
         COUT(8)=0.
         COUT(9)=FFR*EXGH2O-CMx
         COUT(10)=0.
      END IF

C Establish additional output variables
      NAPDAT(IPCOMP)=3
C On/off signal
      PCAOUT(IPCOMP,1)=CDATA(IPCOMP,2)
C Fuel flow rate
      PCAOUT(IPCOMP,2)=FFR
C Total Heating flux
      PCAOUT(IPCOMP,3)=HFLX
C Condensation
      PCAOUT(IPCOMP,4)=CMx

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 3 node (ISV>21) WCH Storage water heater'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1,ICON2
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' C2     = ',C2,' (W/K)'
            WRITE(ITU,*) ' CV     = ',CV,' (W/K)'
            WRITE(ITU,*) ' CA     = ',CA,' (W/K)'
            WRITE(ITU,*) ' UAx   =  ',UAx,' (W/K)'
            WRITE(ITU,*) ' MSC    = ',MSC,' (kg/kg) da'
            WRITE(ITU,*) ' HFG    = ',HFGIN,' (J/kg) da'
            WRITE(ITU,*) ' FFR    = ',FFR,' (kg/s)'
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1),' (-)'
            WRITE(ITU,*) ' QH     = ',HFLX, ' (W) '
            WRITE(ITU,*) ' QF     = ',FFR*FUELHV, ' (W) '
            WRITE(ITU,*) ' CPG    = ',CPG,' (J/kgK) '
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=10
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) then
        WRITE(ITU,*) ' Leaving subroutine CMP34C trace'
      ENDIF

      RETURN
      END

C ******************** CMP35C ********************
C CMP35C generates for plant component IPCOMP with plant db code 350 ie.
C 2 node (ISV=20) WCH basic radiator model
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Total radiator surface area (m2)
C            2 Surface heat transfer coefficient (W/m2K)
C            3 Nominal environment temperature (C)
C            4 Index of coupled building zone (0-n)
C            5 Number of walls used for defining Te (0-n)
C            Then for each wall ....
C            6 Index of 1st wall for defining Te (-)
C            7 Weighting factor for 1st wall when defining Te (-)
C            8 Index of 2nd wall for defining Te (-)
C            9 Weighting factor for 2nd wall when defining Te (-)
C           10 etc.
C
C If no zone or wall surface is specified then the nominal environment
C temperature value is used
C
C     CDATA: none

C     PCDATF/P
C            1 Inter-node fluid heat capacity rate (W/K)

      SUBROUTINE CMP35C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

      COMMON/C6/INDCFG
      COMMON/FVALA/TFA(MCOM),QFA(MCOM)
      COMMON/FVALS/TFS(MCOM,MS),QFS(MCOM)

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP35C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Initialize hA value for the radiator
         rad_area=BDATA(IPCOMP,1)
         ht_coeff=BDATA(IPCOMP,2)

         T_env_nom=BDATA(IPCOMP,3)

C Set default for TE
         TE=T_env_nom

C Then evaluate the current environment temperature

C If this is a plant only simulation or if no zone has been defined then
C set the environmental temperature to the nominal value specified in the
C data.
         IF(INDCFG.EQ.2.OR.NINT(BDATA(IPCOMP,4)).EQ.0) THEN
            call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
            IF(closea) THEN
               TE=T_env_nom
            ELSE
               TE=PCNTMF(IPCOMP)
            ENDIF

C Otherwise get the zone/surface temperatures.
         ELSE
            IZ=INT(BDATA(IPCOMP,4))
            SUMT=TFA(IZ)
            NW=int(BDATA(IPCOMP,5))
            SUMW=1.
            IF(NW.GE.1.) THEN
               DO 10 IW=1,NW
                  W=BDATA(IPCOMP,5+IW*2)
                  SUMW=SUMW+W
                  SUMT=SUMT+W*TFS(IZ,NINT(BDATA(IPCOMP,4+IW*2)))
   10          CONTINUE
            END IF
            TE=SUMT/SUMW
         END IF

C Mark the nodal temperatures for iteration.
         ICSV(INOD1,1)=1
         CSVI(INOD1,1)=CSVF(INOD1,1)
         ICSV(INOD2,1)=1
         CSVI(INOD2,1)=CSVF(INOD2,1)
         TS=CSVF(INOD1,1)
         TX=CSVF(INOD2,1)


C Then calculate radiator heat emission Q
           Q1=0.5*rad_area*ht_coeff*(TS-TE)
           Q2=0.5*rad_area*ht_coeff*(TX-TE)
           Q=Q1+Q2


C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)/2.
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Inter-node fluid heat capacity rate (W/K)
         PCDATF(IPCOMP,1)=CSVF(INOD1,2)*SHTFLD(3,CSVF(INOD1,1))

C Calculate current component time-constant TC
         TC(IPCOMP)=AMAX1(
     &        CM/AMAX1(SMALL,(C1)),
     &        CM/AMAX1(SMALL,(PCDATF(IPCOMP,1)+
     &        Q/AMAX1(SMALL,AMAX1(SMALL,((0.5*(TS+TX))-TE))))))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self-coupling coefficients,
C  node 1
         COUT(1)=ALPHA*(-C1)-CM/TIMSEC
C  node 2
         COUT(2)=ALPHA*PCDATF(IPCOMP,1)
         COUT(3)=ALPHA*(-PCDATF(IPCOMP,1))-CM/TIMSEC
C then matrix equation cross-coupling coefficient,
         COUT(4)=ALPHA*C1
C and then present-time coefficients (ie. right hand sides)
         COUT(5)=((1.-ALPHA)*PCRP(ICON1)-CM/TIMSEC)*CSVP(INOD1,1)
     &           +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &           +ALPHA*Q1+(1.-ALPHA)*PCQP(INOD1)
         COUT(6)=(1.-ALPHA)*(-PCDATP(IPCOMP,1))*CSVP(INOD1,1)
     &           +((1.-ALPHA)*PCDATP(IPCOMP,1)-CM/TIMSEC)*CSVP(INOD2,1)
     &           +ALPHA*Q2+(1.-ALPHA)*PCQP(INOD2)

C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD1)=Q1
         PCQF(INOD2)=Q2

C Establish "containment loss" data
         QDATA(IPCOMP)=ALPHA*Q+(1.-ALPHA)*(PCQP(INOD2)+PCQP(INOD1))

C Establish additional output variables
         PCAOUT(IPCOMP,1)=Q
         PCAOUT(IPCOMP,2)=0.5*(TS+TX)
         PCAOUT(IPCOMP,3)=TE

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-1.
         COUT(3)=1.
         COUT(4)=-PCONDR(ICON1)
         COUT(5)=0.
         COUT(6)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=1.
         COUT(4)=0.
         COUT(5)=0.
         COUT(6)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH basic radiator'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' T_rad_mean = ',0.5*(TS+TX),' (C)'
            WRITE(ITU,*) ' TE     = ',TE,' (C)'
            WRITE(ITU,*) ' ht_coeff   = ',ht_coeff,' (W/m2K)'
            WRITE(ITU,*) ' rad_area   = ',rad_area,' (m2)'
            WRITE(ITU,*) ' Q      = ',Q,' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP35C'

      RETURN
      END

C ******************** CMP36C ********************
C CMP36C generates for plant component IPCOMP with plant db code 360 ie.
C 2 node (ISV=19,21) WCH basic water loop reversible heat pump.
C
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)

C     ADATA: 1 Mass of component (solids+liquid) (kg)
C            2 Mass weighted average specific heat node (J/kgK)
C            3 UA modulus for component (W/K)

C     BDATA: 1 COP heating (if < 0 use quadratic f(Twater))
C            2 coef. h0
C            3 coef. h1
C            4 coef. h2
C            5 COP cooling (if < 0 use quadratic f(Tair))
C            6 coef. c0
C            7 coef. c1
C            8 coef. c2

C     CDATA: 1 cooling or heating duty (W)


C Node 1 is the element of the component connected to the water loop. Node 2 is the
C part of the conponent connected to conditioned air.
C
C
      SUBROUTINE CMP36C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat


C Electrical details for specified plant components
      common/elpcp/NPEL,PFP(mpcom),IPFP(mpcom),PWRP(mpcom),
     &BVOLTP(mpcom),IPHP(mpcom)

C Reversible heat pump common
      common/rev_HP_logic/rev_heat_pump(MPCOM)

      logical rev_heat_pump


      PARAMETER (SMALL=1.0E-20)
      REAL      COUT(MPCOE)

      LOGICAL CLOSE
C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP36C'

C Set reversible flag to true
      rev_heat_pump(IPCOMP)= .true.

      write(84,*) 'rev_heat_pump ',ipcomp,rev_heat_pump(IPCOMP)

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      ICON2=ICONDX(IPCOMP,2,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C COP based on inlet temperatures so mark 1st and 2nd nodal temps
C and flows for iteration.
      ICSV(INOD1,1)=1
      CSVI(INOD1,1)=CSVF(INOD1,1)
      ICSV(INOD2,1)=1
      CSVI(INOD2,1)=CSVF(INOD2,1)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Establish the nodal thermal capacities.
        CM1=0.5*ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
        CM2=CM1

C node 1 water
        C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C node 2 moist air
        C2=PCONDR(ICON2)*(CONVAR(ICON2,2)*SHTFLD(1,CONVAR(ICON2,1))
     &     +CONVAR(ICON2,3)*SHTFLD(2,CONVAR(ICON2,1)))

C Establish the COP for heating mode
        COPh=BDATA(IPCOMP,1)

C Establish hot and cold temperatures
        T_h=convar(icon1,1)
        T_c=convar(icon2,1)

C Establish the COP for heating mode if calculated using a quadratic
        if(COPh.lt.0.)then
          h2=BDATA(IPCOMP,2)
          h1=BDATA(IPCOMP,3)
          h0=BDATA(IPCOMP,4)
          COPh=h2*T_h**2+h1*T_h+h0
        endif

C Establish the COP for cooling mode
        COPc=BDATA(IPCOMP,5)

C Establish the COP for heating mode if calculated using a quadratic
        if(COPc.lt.0.)then
          c2=BDATA(IPCOMP,6)
          c1=BDATA(IPCOMP,7)
          c0=BDATA(IPCOMP,8)
          T_c=CONVAR(ICON2,1)
          COPc=c2*T_c**2+c1*T_c+c0
        endif

C Establish the heat pump duty cooling or heating
C (heat to or from the air stream)
        heat_pump_duty=CDATA(IPCOMP,1)

C Establish heat loss modulus
        UA=ADATA(IPCOMP,5)
        CALL ECLOSE(PCNTMF(IPCOMP),99.,0.0001,CLOSE)
        IF(CLOSE) UA=0.0

C Set default state (off)
        w_compressor=0.
        Q_water=0.
        Q_air=0.

C If zero flow found then set transfered flux to zero and skip.
        IF(C1.LT.SMALL.OR.C2.LT.SMALL) THEN
          heat_pump_duty=0.
          Q_water=0.
          Q_air=0.
          w_compressor=0.
        ELSE

C Establish if heating or cooling
          if(heat_pump_duty.ge.0.)then
            if(COPh.gt.small) w_compressor=heat_pump_duty/COPh

C heating mode, heat removed from water + compressor work added to
C air node.
            Q_air=heat_pump_duty+w_compressor
            Q_water=heat_pump_duty
          else

C cooling mode, heat removed from air + compressor work added to
C water node.
            if(COPc.gt.small) w_compressor=-heat_pump_duty/COPc
            Q_water=heat_pump_duty-w_compressor
            Q_air=heat_pump_duty

C Calculate effect of possible condensation here.
C << to be added >>
C end condensation calc.

          endif
        ENDIF

C Calculate the electrical demand of the components
        PWRP(IPCOMP)=w_compressor
        IEMODEL=1
        CALL EMACH(IPCOMP,IEMODEL,PWRP(IPCOMP),PQ,PA)

C Calculate current component time-constant TC
        TC(IPCOMP)=AMAX1(
     &CM1/AMAX1(SMALL,(C1+UA)),
     &CM2/AMAX1(SMALL,(C2+UA)))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
           ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
           ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
             ALPHA=1.
          ELSE
             ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
            CM1=0.
            CM2=0.
            ALPHA=1.
        END IF

C Establish matrix equation self- and cross-coupling coefficients.
        COUT(1)=ALPHA*(-C1)-CM1/TIMSEC
        COUT(2)=ALPHA*(-C2-UA)-CM2/TIMSEC
C Matrix cross coupling coefficients.
        COUT(3)=ALPHA*C1
        COUT(4)=ALPHA*C2
C Establish the present and known coefficient i.e. RHS
        COUT(5)=((1.-ALPHA)*(PCRP(ICON1))-CM1/TIMSEC)
     &          *CSVP(INOD1,1)
     &          -(1.-ALPHA)*PCRP(ICON1)*PCTP(ICON1)
     &          +ALPHA*Q_water
     &          +(1.-ALPHA)*PCQP(INOD1)
         COUT(6)=((1.-ALPHA)*(PCRP(ICON2)+UA)-CM1/TIMSEC)
     &           *CSVP(INOD2,1)
     &          -(1.-ALPHA)*PCRP(ICON2)*PCTP(ICON2)
     &          -(1.-ALPHA)*(UA)*PCNTMP(IPCOMP)
     &          -ALPHA*(UA)*PCNTMF(IPCOMP)
     &          -ALPHA*Q_air
     &          -(1.-ALPHA)*(PCQP(INOD2))

C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCTF(ICON2)=CONVAR(ICON2,1)
         PCRF(ICON1)=C1
         PCRF(ICON2)=C2
         PCQF(INOD1)=Q_water
         PCQF(INOD2)=Q_air

C Addition output for results analysis.
         NAPDAT(IPCOMP)=6
         PCAOUT(IPCOMP,1)=heat_pump_duty
         PCAOUT(IPCOMP,2)=Q_water
         PCAOUT(IPCOMP,3)=Q_air
         PCAOUT(IPCOMP,4)=w_compressor
         PCAOUT(IPCOMP,5)=COPh
         PCAOUT(IPCOMP,6)=COPc

C 1st phase mass (ie. "water") balance coefficients

      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=-PCONDR(ICON1)
         COUT(4)=-PCONDR(ICON2)
         COUT(5)=0.
         COUT(6)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=0.
         COUT(4)=-PCONDR(ICON2)
         COUT(5)=0.
         COUT(6)=0.
      END IF


C Trace.
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=29) WCH reversible heat pump'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1,ICON2
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM1     = ',CM1,' (J/K)'
            WRITE(ITU,*) ' CM2     = ',CM2,' (J/K)'
            WRITE(ITU,*) ' C1      = ',C1,' (W/Ks)'
            WRITE(ITU,*) ' C2      = ',C2,' (W/K)'
            WRITE(ITU,*) ' TC      = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA   = ',ALPHA,' (-)'
            WRITE(ITU,*) ' T_h     = ',convar(icon1,1),' (C)'
            WRITE(ITU,*) ' T_c     = ',convar(icon2,1),' (C)'
            WRITE(ITU,*) ' COPh    = ',COPh,' (-)'
            WRITE(ITU,*) ' COPc    = ',COPc,' (-)'
            WRITE(ITU,*) ' q_air   = ',Q_air,' (W)'
            WRITE(ITU,*) ' q_water = ',Q_water,' (W)'
            WRITE(ITU,*) ' w_comp  = ',w_compressor,' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP36C'

      RETURN
      END

C ******************** CMP37C ********************
C
C CMP37C generates for plant component IPCOMP with plant db code 370 ie.
C 1 node (ISV=20) WCH air cooled or evaporative heat rejector unit
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 UA modulus (W/K)
C     BDATA: 1 Pump or fan parasitic load (W/W of cooling)
C            2 Parasitic load coefficients <<to be added>>
C     CDATA: 1 Cooling duty (W)
C
      SUBROUTINE CMP37C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PTIME/PTIMEP,PTIMEF
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat


C Electrical details for specified plant components
      common/elpcp/NPEL,PFP(mpcom),IPFP(mpcom),PWRP(mpcom),
     &BVOLTP(mpcom),IPHP(mpcom)

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      character outs*124
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP37C'

C Check control data for relevant balance type
      IF(ISTATS.EQ.1.AND.CDATA(IPCOMP,1).GT.0.) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP37C: invalid control data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP37C: unresolvable error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss)
         UA=ADATA(IPCOMP,3)
         call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
         IF(closea) UA=0.

C Then initialize heat rejected  Q
         Q=CDATA(IPCOMP,1)

C Calculate the parasitic loads here.
         E_parasitic=Q*BDATA(IPCOMP,1)


C Polynomial calc of E_parasitic to go here.
C For example.
C        E_parasitic=Q^2*a2+Q*a1+a0 + TE^2*b2+TE*b2+b0
C        TE=ambient air temp.

         PWRP(IPCOMP)=E_parasitic
         IEMODEL=1
         CALL EMACH(IPCOMP,IEMODEL,PWRP(IPCOMP),PQ,PA)

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(C1+UA))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self- and cross-coupling coefficients
         COUT(1)=ALPHA*(-C1-UA)-CM/TIMSEC
         COUT(2)=ALPHA*C1
C and then present-time coefficient (ie. right hand side)
         COUT(3)=((1.-ALPHA)*(PCRP(ICON1)+PUAP(INOD1))
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &             -ALPHA*UA*PCNTMF(IPCOMP)
     &             -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)
     &             -ALPHA*Q-(1.-ALPHA)*PCQP(INOD1)

C Store "environment" variables future values
         PUAF(INOD1)=UA
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD1)=Q

C Addition output for results analysis.
         NAPDAT(IPCOMP)=2
         PCAOUT(IPCOMP,1)=Q
         PCAOUT(IPCOMP,2)=E_parasitic

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-PCONDR(ICON1)
         COUT(3)=0.

C 2nd phase mass (ie. vapour) balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) air cooled or evaporative heat'
         WRITE(ITU,*) ' 1 node (ISV=20) rejection device'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1)
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP37C'


      RETURN
      END

C ******************** CMP38C ********************
C CMP38C generates for plant component IPCOMP with plant db code 380 ie.
C 1 node (ISV=20) WCH converging mutli-leg junction (manifold) with up
C to 10 connections.
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 UA modulus (W/K)
C     BDATA: 1 Number of connections (-)
C     CDATA: none

      SUBROUTINE CMP38C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE),Coeff(10),PCRP_tot,Coeff_tot,CoeffT_tot
      INTEGER   ICON(10)
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP38C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD

      INOD1=NPCDAT(IPCOMP,9)
      NCON=NINT(BDATA(IPCOMP,1))

      DO 5 I=1,10
        ICON(I)=0
        Coeff(I)=0.0
5     CONTINUE

C Set up connection data for up to 10 connections
      PCRP_tot=0.0
      DO 10 I=1,NCON
        ICON(I)=ICONDX(IPCOMP,1,I)
        if(icon(I).ne.0)then
          PCRP_tot=PCRP_tot+PCRP(ICON(I))
        endif
10    CONTINUE


C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss)
        UA=ADATA(IPCOMP,3)
        call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
        IF(closea) UA=0.0

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
        CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
        Coeff_tot=0.0
        DO 20 J=1,NCON
          if(icon(J).ne.0)then
            Coeff(J)=PCONDR(ICON(J))*CONVAR(ICON(J),2)*
     &        SHTFLD(3,CONVAR(ICON(J),1))
            Coeff_tot=Coeff_tot+Coeff(J)
          endif
20      CONTINUE

C Calculate current component time-constant TC
        TC(IPCOMP)=CM/AMAX1(SMALL,(Coeff_tot+UA))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
          ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
          ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
            ALPHA=1.
          ELSE
            ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
          CM=0.
          ALPHA=1.
        END IF

C Establish matrix equation self- and cross-coupling coefficients
        COUT(1)=ALPHA*(-Coeff_tot-UA)-CM/TIMSEC
        DO 30 K=1,10
          COUT(K+1)=ALPHA*Coeff(K)
30      CONTINUE

C and then present-time coefficient (ie. right hand side)
        CoeffT_tot=0.0
        PCRP_tot=0.0
        DO 35 L=1,10
          if(icon(L).ne.0)then
            CoeffT_tot=CoeffT_tot+(-PCRP(ICON(L)))*PCTP(ICON(L))
            PCRP_tot=PCRP_tot+PCRP(ICON(L))
          endif
35      CONTINUE
        COUT(12)=((1.-ALPHA)*(PCRP_tot+PUAP(INOD1))
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             +(1.-ALPHA)*CoeffT_tot
     &             -ALPHA*UA*PCNTMF(IPCOMP)
     &             -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)

C Store "environment" variables future values
        PUAF(INOD1)=UA

        DO 40 I=1,10
          if(icon(I).ne.0)then
            PCTF(ICON(I))=CONVAR(ICON(I),1)
            PCRF(ICON(I))=Coeff(I)
          else
            continue
          endif
40      CONTINUE

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
        COUT(1)=1.
        DO 50 J=1,10
          if(icon(j).ne.0)then
            COUT(J+1)=-PCONDR(ICON(J))
          else
            COUT(J+1)=0.0
          endif
50      CONTINUE
        COUT(12)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         DO 60 K=1,10
           COUT(K+1)=0.
60       CONTINUE
         COUT(12)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH multi-leg junction'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',(ICON(J),J=1,NCON)
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1-10     = ',(Coeff(K),K=1,10),' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=12
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP38C'

      RETURN
      END


C ******************** CMP39C ********************

C CMP39C generates for plant component IPCOMP with plant db code 390 ie.
C 1 node (ISV=20) WCH pump with mass flow control
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 UA modulus (W/K)
C     BDATA: 1 Rated absorbed power (W)
C            2 Rated mass flow rate (kg/s)
C            3 Overall efficiency (-)
C     CDATA: 1 Mass flow rate (kg/s)

      SUBROUTINE CMP39C(IPCOMP,COUT,ISTATS)

#include "plant.h"
#include "building.h"
#include "net_flow.h"
#include "net_flow_data.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PTIME/PTIMEP,PTIMEF
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      COMMON/FFN/IFLWN,ICFFS(MPCON)
      COMMON/MFLRES/FLW1(MCNN),FLW2(MCNN),PRES(MNOD),
     &              RESID(MNOD),SAFLW(MNOD)

      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER napdat

C Electrical details for specified plant components
      common/elpcp/NPEL,PFP(mpcom),IPFP(mpcom),PWRP(mpcom),
     &BVOLTP(mpcom),IPHP(mpcom)

      DOUBLE PRECISION FLW1,FLW2,PRES,RESID,SAFLW
      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      character outs*124
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP39C'

C Check control data for relevant balance type
      IF(ISTATS.EQ.2.AND.CDATA(IPCOMP,1).LT.0.) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP39C: invalid control data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP39C: unresolvable error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss)
         UA=ADATA(IPCOMP,3)
         call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
         IF(closea) UA=0.

C Establish absorbed power E based on current water flow rate which might
C have been calculated by mfs
C In case of mfs E is based on flow rate, pressure *rise*, and efficiency
         IF(IFLWN.NE.0.AND.ICFFS(ICON1).NE.0) THEN
            ICNN=ICFFS(ICON1)
            E=real((FLW1(ICNN)+FLW2(ICNN))
     &        *(PRES(NODNE(ICNN))-PRES(NODPS(ICNN)))
     &        /(BDATA(IPCOMP,3)*RHOFLD(3,CSVF(INOD1,1))))
         ELSE
            E=((CSVF(INOD1,2)
     &         /BDATA(IPCOMP,2))**3)*BDATA(IPCOMP,1)
         END IF

         PWRP(IPCOMP)=-ABS(E)
         IEMODEL=1
         CALL EMACH(IPCOMP,IEMODEL,PWRP(IPCOMP),PQ,PA)
         PWRQ=PQ

C Now Q is made up of all inefficiencies
         Q=(1.-BDATA(IPCOMP,3))*E

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(C1+UA))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self- and cross-coupling coefficients
         COUT(1)=ALPHA*(-C1-UA)-CM/TIMSEC
         COUT(2)=ALPHA*C1
C and then present-time coefficient (ie. right hand side)
         COUT(3)=((1.-ALPHA)*(PCRP(ICON1)+PUAP(INOD1))
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &             -ALPHA*UA*PCNTMF(IPCOMP)
     &             -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)
     &             -ALPHA*Q-(1.-ALPHA)*PCQP(INOD1)

C Store "environment" variables future values
         PUAF(INOD1)=UA
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD1)=Q

C Save plant additional output data.
         napdat(ipcomp)=2
         pcaout(ipcomp,1)=pwrp(ipcomp)
         pcaout(ipcomp,2)=pwrq

C GF New Start
C Establish "containment loss" data
         QDATA(IPCOMP)=UA*(alpha*(csvf(inod1,1)-pcntmf(ipcomp))+
     &                (1.-alpha)*(csvp(inod1,1)-pcntmp(ipcomp)))
         call store_plt_gain (IPCOMP,0.5*QDATA(IPCOMP),iConvective)
         call store_plt_gain (IPCOMP,0.5*QDATA(IPCOMP),iRadiant)
C GF New End

C 1st phase mass (ie. water) balance coefficients
C Note that if fluid mass flow solver active, source of mass zeroised
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         IF(IFLWN.EQ.0.or.ICFFS(ICON1).eq.0) THEN
            COUT(2)=0.
            COUT(3)=CDATA(IPCOMP,1)
         ELSE
            COUT(2)=-PCONDR(ICON1)
            COUT(3)=0.
         END IF

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH pump (mass flow control)'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
            WRITE(ITU,*) ' Q      = ',Q,' (W)'
            WRITE(ITU,*) ' E      = ',E,' (W)'
            WRITE(ITU,*) ' E,cum. = ',E*TIMSEC,' (J)'
         ELSE IF(ISTATS.EQ.2) THEN
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1)
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP39C'

      RETURN
      END

C ******************** CMP42C ********************

C CMP42C generates for plant component IPCOMP with plant db code 420 ie.
C 2 node (ISV=20) WCH boiler with modulation
C This boiler is exactly similar to the WCH 2 node boiler with ON/OFF
C control except that full load gas firing rate BDATA(?,1) is modified
C at run time depending upon return water temperature.
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Full load gas firing rate when boiler on (m^3/s)
C            2 Dump load fraction
C            3 Gas heating value at STP (J/m^3)
C            4 Boiler lock out time (min)
C            5 Boiler heat loss to surroundings UA (W/K)
C            6 DHW draw schedule control loop no. if combiboiler (0 otherwise)
C            7 DHW calorifier component number (0 otherwise)
C            8 Constant cold water feed temperature for DHW for combiboiler (degC)
C            9 Constant supply temperature for DHW for combiboiler(degC)
C           10 Upper boiler temperature limit (C)
C           11 Lower limit of modulating range (%), 0 for nonmodulating
C           12 Lower limit total differential (%), 0 for nonmodulating
C        13-21 Coefficients a1 to a9 for efficiency equation
C     CDATA: 1 ON/OFF control signal (-)

C     PCDATF/P
C            1 Holds boiler actual ON/OFF state (-)
C            2 Inter-node fluid heat capacity rate (W/K)

      SUBROUTINE CMP42C(IPCOMP,COUT,ISTATS)
      IMPLICIT NONE

#include "plant.h"
#include "building.h"
#include "control.h"
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      INTEGER ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      INTEGER ITCF,ITRACE,IZNTRC,ITU
      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      INTEGER IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      REAL TIMSEC
      COMMON/PTIME/PTIMEP,PTIMEF
      REAL PTIMEP,PTIMEF
      COMMON/PCTC/TC(MPCOM)
      REAL TC
      COMMON/PCEQU/IMPEXP,RATIMP
      INTEGER IMPEXP
      REAL RATIMP
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)
      INTEGER MAXITP,ITRCLP,ICSV
      REAL PERREL,PERTMP,PERFLX,PERMFL,CSVI
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      INTEGER NPCOMP,NCI
      REAL CDATA
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      INTEGER NPCON,IPC1,IPN1,IPCT,IPC2,IPN2
      REAL PCONDR,PCONSD
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      INTEGER NPCDAT,IPOFS1,IPOFS2
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      REAL ADATA,BDATA
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      REAL CSVF,CSVP
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      REAL PCTF,PCRF,PUAF,PCQF,PCNTMF,PCTP,PCRP,PUAP,PCQP,PCNTMP
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      REAL CONVAR
      INTEGER ICONTP,ICONDX
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)
      REAL PCDATF,PCDATP
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER NAPDAT

C Does not match size of LOCAL1 in CMP46C!
      COMMON/LOCAL1/ISET(MCF),PHIMAX
      INTEGER ISET
      REAL PHIMAX
      COMMON/LOKOUT/INSO,ITSBO
      INTEGER INSO,ITSBO
      COMMON/PCLOP8/LASTOUT(MCF)
      INTEGER LASTOUT
      REAL SMALL
      PARAMETER (SMALL=1.0E-20)
      REAL      COUT(MPCOE),TOTMAS,FLGFR,UA,TF,GFR,EFF_ON,GMAX,
     & RUBTL,A1,A2,A3,A4,A5,A6,A7,A8,A9,CUTIN,CUTOUT,FEEDTM,SUPPTM,RI,
     & RO,HK,PHIW,A,TR,ZMFR,QL,TRD,G,Q,PHI,QLD,BLKOUT,FGAS,PHISB,CM,
     & ALPHA,C1,SHTFLD
      character outs*124
      INTEGER ICON1,INOD1,INOD2,IONOFF,ICOMBI,ICCOMP,ICOMBIMOD,ICCN1,
     & IPC,NTSBO,IX1,I,NITMS,IPCON,ISTATS,IPCOMP
      integer istat

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP42C'

      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C Check control data for relevant balance type
      IF(ISTATS.EQ.1.AND.
     &   (CDATA(IPCOMP,1).LT.0..OR.CDATA(IPCOMP,1).GT.1.)) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP42C: invalid control data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP42C: unresolvable error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF

C Scale down parameters if dump load is to be supplied.
      TOTMAS=ADATA(IPCOMP,1)*(1.-BDATA(IPCOMP,2))
      FLGFR=BDATA(IPCOMP,1)*(1.-BDATA(IPCOMP,2))
      UA=BDATA(IPCOMP,5)*(1.-BDATA(IPCOMP,2))

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      IF(ISTATS.EQ.1) THEN
      TF=BDATA(IPCOMP,10)
      IF(ISET(IPCOMP).EQ.0)THEN
        GFR=1.0
        EFF_ON=1.
        PHIMAX=FLGFR*BDATA(IPCOMP,3)*EFF_ON
        GMAX=FLGFR*BDATA(IPCOMP,3)
        ISET(IPCOMP)=1
        RUBTL=BDATA(IPCOMP,10)
      ENDIF

C Establish whether boiler is on or off (ie 1 or 0)
      IONOFF=Nint(CDATA(IPCOMP,1))

C Set up coefficients for efficiency equation
      A1=BDATA(IPCOMP,13)
      A2=BDATA(IPCOMP,14)
      A3=BDATA(IPCOMP,15)
      A4=BDATA(IPCOMP,16)
      A5=BDATA(IPCOMP,17)
      A6=BDATA(IPCOMP,18)
      A7=BDATA(IPCOMP,19)
      A8=BDATA(IPCOMP,20)
      A9=BDATA(IPCOMP,21)

C Get gas firing modulation cut-in and cut-out
      CUTIN=(BDATA(IPCOMP,11)+BDATA(IPCOMP,12)*0.5)*0.01
      CUTOUT=(BDATA(IPCOMP,11)-BDATA(IPCOMP,12)*0.5)*0.01

C Determine if combi-boiler or not
      ICOMBI=NINT(BDATA(IPCOMP,6))
      ICCOMP=NINT(BDATA(IPCOMP,7))
      FEEDTM=BDATA(IPCOMP,8)
      SUPPTM=BDATA(IPCOMP,9)
      ICOMBIMOD=0

C If boiler is combi and is supplying DHW determine firing rate
      IF(ICOMBI.NE.0)THEN
        IF(LASTOUT(ICOMBI).NE.0)THEN
          ICOMBIMOD=1
          ICCN1=ICONDX(ICCOMP,1,1)
          RI=1./(BDATA(ICCOMP,1)*BDATA(ICCOMP,2))
          RO=1./(BDATA(ICCOMP,3)*BDATA(ICCOMP,4))
          HK=1./(RI+RO)
          PHIW=(PCONDR(ICCN1)*CONVAR(ICCN1,2)*SHTFLD(3,CONVAR(ICCN1,1)))
     &         *(SUPPTM-FEEDTM)
          A=0.5*HK/
     &      (PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1)))
          TR=((1.-A)*CSVF(INOD2,1)+A*(SUPPTM+FEEDTM))/(1.+A)
          IF(TR.LT.FEEDTM)TR=FEEDTM
          IF(TR.GT.CSVF(INOD2,1))TR=CSVF(INOD2,1)
          ZMFR=PCONDR(ICON1)*CONVAR(ICON1,2)
          QL=1.
          TRD=TR
          IF(TR.LT.30.)TRD=30.
          IF(TR.GT.70.)TRD=70.
          EFF_ON=(A1*TRD+A2*(TRD**2.)+A3*(TRD**3.)+A4*(TRD**4.)
     &         +A5*QL+A6*TRD*QL+A7*(TRD**2.)*QL+A8+A9*(QL**2.))/100.
          PHIMAX=FLGFR*BDATA(IPCOMP,3)*EFF_ON
          QL=PHIW/PHIMAX
          IF(QL.GT.1.)THEN
            QL=1.
            PHIW=PHIMAX
          ENDIF
          EFF_ON=(A1*TRD+A2*(TRD**2.)+A3*(TRD**3.)+A4*(TRD**4.)
     &         +A5*QL+A6*TRD*QL+A7*(TRD**2.)*QL+A8+A9*(QL**2.))/100.
          PHIMAX=FLGFR*BDATA(IPCOMP,3)*EFF_ON
          G=PHIW/EFF_ON
          GFR=G/(FLGFR*BDATA(IPCOMP,3))

C Consider decreasing modulation lower limit to below 30% otherwise
C low volumes of water draw do not switch boiler on
          IF(GFR.LT.CUTIN)GFR=0.
          IF(GFR.GT.1.)GFR=1.
          Q=PHIW
        ENDIF
      ENDIF

C Set boiler on if there is demand for hot water from a combi boiler
      IF(ICOMBIMOD.EQ.1)IONOFF=1
      IF(ICOMBIMOD.EQ.1)GOTO 33

C TR=Return water temperature, QL=Load
C TF=Flow water (temperature design value of boiler) to be controlled in
C ZMFR=Water mass flow rate though boiler
C G=heat supplied by gas (gas firing rate*calorifc value)
C GFR=gas firing rate
C Get connection which sends mass to boiler, assuming only one
C component sends mass to the boiler
      DO 1515 IPC=1,NPCON
        IF(IPC1(IPC).EQ.IPCOMP)IPCON=IPC
 1515 CONTINUE
      TR=CONVAR(IPCON,1)
      IF(TR.GT.TF)TR=TF
      ZMFR=PCONDR(ICON1)*CONVAR(ICON1,2)
      QL=1.
      TRD=TR
      IF(TR.LT.30.)TRD=30.
      IF(TR.GT.70.)TRD=70.
      EFF_ON=(A1*TRD+A2*(TRD**2.)+A3*(TRD**3.)+A4*(TRD**4.)
     &         +A5*QL+A6*TRD*QL+A7*(TRD**2.)*QL+A8+A9*(QL**2.))/100.
      PHI=ZMFR*SHTFLD(3,CONVAR(ICON1,1))*(TF-TR)
      IF(PHI.GT.PHIMAX)PHI=PHIMAX
      QL=PHI/PHIMAX
      IF (NINT(BDATA(IPCOMP,12)).NE.0) THEN

C Condensing boiler
        EFF_ON=(A1*TRD+A2*(TRD**2.)+A3*(TRD**3.)+A4*(TRD**4.)
     &         +A5*QL+A6*TRD*QL+A7*(TRD**2.)*QL+A8+A9*(QL**2.))/100.
      ELSE

C non condensing - non modulating (QLD=1)
        QLD=1
        EFF_ON=(A1*TRD+A2*(TRD**2.)+A3*(TRD**3.)+A4*(TRD**4.)
     &         +A5*QLD+A6*TRD*QLD+A7*(TRD**2.)*QLD+A8+A9*(QL**2.))/100.
      ENDIF
      G=PHI/EFF_ON
      GFR=G/GMAX

C Reset to off in case maximum temperature exceeded
      IF(IONOFF.EQ.1)THEN
        IF(GFR.GT.1.)THEN
          GFR=1.
        ELSEIF(GFR.LT.CUTOUT)THEN
          GFR=0.
          IONOFF=0
        ENDIF
      ELSE
        IF(GFR.GT.1.)THEN
          GFR=1.
        ELSEIF(GFR.LT.CUTIN)THEN
          GFR=0.
        ENDIF
      ENDIF

C Lock out time (in seconds), If boiler temperature reaches upper
C boiler temperature limit it is shut down for this amount of time
C INSO = boiler on/off (1/0) flag for lockout
C ITSBO = present no. of timesteps boiler is off due to lockout
C NTSBO = total no. of timesteps boiler is off due to lockout
 33     BLKOUT=BDATA(IPCOMP,4)*60.
        NTSBO=NINT(BLKOUT/TIMSEC)
        IF(CSVF(INOD2,1).GT.RUBTL)THEN
          INSO=1
          ITSBO=0
        ENDIF
        IF(ITSBO.LE.0)ITSBO=0
        IF(INSO.EQ.1)THEN
          ITSBO=ITSBO+1
          IF(ITSBO.GE.NTSBO)INSO=0
          Q=0.0
          IONOFF=0
          FGAS=0.
        ENDIF
        IF(INSO.EQ.1)GOTO 333

C First calculate boiler efficiency ETA based on current node 1 temp.
C so mark node 1 temperature for iteration
        ICSV(INOD1,1)=1
        CSVI(INOD1,1)=CSVF(INOD1,1)

C Then calculate heat input into the water PHIW if boiler is on (assume
C that boiler comes on regardless of control loop if there is demand for
C hot water if it is a combiboiler)
        PHIW=0.0
        FGAS=0.0
        IF(IONOFF.EQ.1.OR.ICOMBIMOD.EQ.1) THEN
          PHISB=0.
          FGAS=FLGFR
          IF (NINT(BDATA(IPCOMP,12)).NE.0) THEN
            FGAS=GMAX*GFR/BDATA(IPCOMP,3)
          ENDIF
          PHIW=EFF_ON*FGAS*BDATA(IPCOMP,3)
        ENDIF

C Then calculate net heat input Q
        Q=PHIW

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
 333    CM=TOTMAS*ADATA(IPCOMP,2)/2.
        C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Boiler actual ON/OFF state PCDATF(IPCOMP,1) (-)
        PCDATF(IPCOMP,1)=IONOFF

C Inter-node fluid heat capacity rate (W/K)
        PCDATF(IPCOMP,2)=CSVF(INOD1,2)*SHTFLD(3,CSVF(INOD1,1))

C Calculate current component time-constant TC
        TC(IPCOMP)=AMAX1(
     &        CM/AMAX1(SMALL,(C1)),CM/AMAX1(SMALL,(PCDATF(IPCOMP,2))))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
          ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
          ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
            ALPHA=1.
          ELSE
            ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
          CM=0.
          ALPHA=1.
        ENDIF

C Establish matrix equation self-coupling coefficients,
C  node 1
        COUT(1)=ALPHA*(-C1-UA)-CM/TIMSEC

C  node 2
        COUT(2)=ALPHA*(PCDATF(IPCOMP,2)+UA)
        COUT(3)=ALPHA*(-PCDATF(IPCOMP,2)-UA)-CM/TIMSEC

C then matrix equation cross-coupling coefficient,
        COUT(4)=ALPHA*C1

C and then present-time coefficients (ie. right hand sides)
        COUT(5)=((1.-ALPHA)*(PCRP(ICON1)+UA)-CM/TIMSEC)*CSVP(INOD1,1)
     &           +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &           -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)
     &           -ALPHA*UA*PCNTMF(IPCOMP)-(1.-ALPHA)*UA*PCNTMP(IPCOMP)

        COUT(6)=(1.-ALPHA)*(-PCDATP(IPCOMP,2)-UA)*CSVP(INOD1,1)
     &           +((1.-ALPHA)*(UA+PCDATP(IPCOMP,2))
     &           -CM/TIMSEC)*CSVP(INOD2,1)-ALPHA*Q-
     &           (1.-ALPHA)*PCQP(INOD2)

C Store "environment" variables future values
        PCTF(ICON1)=CONVAR(ICON1,1)
        PCRF(ICON1)=C1
        PCQF(INOD2)=Q

C Establish "containment loss" data
        QDATA(IPCOMP)=0.

C If boiler is off set its efficiency to zero
        IF(IONOFF.EQ.0)EFF_ON=0.

C Establish additional output variables
C FGAS*1000 is to give resolution on output
c this will translate from m3/s to l/s??
        napdat(ipcomp)=5
        PCAOUT(IPCOMP,1)=IONOFF
        PCAOUT(IPCOMP,2)=FGAS*1000.
        PCAOUT(IPCOMP,3)=TF
        PCAOUT(IPCOMP,4)=PHIW
        PCAOUT(IPCOMP,5)=EFF_ON

C 1st phase mass (ie. water) balance coefficients
      ENDIF
      IF(ISTATS.EQ.2) THEN
        COUT(1)=1.
        COUT(2)=-1.
        COUT(3)=1.
        COUT(4)=-PCONDR(ICON1)
        COUT(5)=0.
        COUT(6)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
        COUT(1)=1.
        COUT(2)=0.
        COUT(3)=1.
        COUT(4)=0.
        COUT(5)=0.
        COUT(6)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH boiler & modulation'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1)
            WRITE(ITU,*) ' IONOFF = ',IONOFF,' (-)'
            WRITE(ITU,*) ' FGAS   = ',FGAS,' (m^3/s)'
            WRITE(ITU,*) ' PHIW   = ',PHIW,' (W)'
            WRITE(ITU,*) ' PHISB  = ',PHISB,' (W)'
            WRITE(ITU,*) ' GASCO  = ',FGAS*TIMSEC,' (m^3)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP42C'

      RETURN
      END

C ******************** CMP44C ********************
C CMP44C generates for plant component IPCOMP with plant db code 440 ie.
C 2 node (ISV=20) WCH exponent radiator model
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Nominal heat supply rate (W)
C            2 Nominal supply water temperature (C)
C            3 Nominal environment temperature (C)
C            4 Nominal exit water temperature (C)
C            5 Radiator Exponent (-)
C            6 Index of 1st wall for defining Te (-)
C            7 Weighting factor for 1st wall when defining Te (-)
C            8 Index of 2nd wall for defining Te (-)
C            9 Weighting factor for 2nd wall when defining Te (-)
C           10 Index of coupled building zone (0-n)
C           11 Number of walls used for defining Te (0-n)
C
C If no zone or wall surface is specified then the nominal environment
C temperature value is used
C
C     CDATA: none

C     PCDATF/P
C            1 Inter-node fluid heat capacity rate (W/K)

      SUBROUTINE CMP44C(IPCOMP,COUT,ISTATS)
      IMPLICIT NONE
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      INTEGER ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      INTEGER ITCF,ITRACE,IZNTRC,ITU
      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      INTEGER IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      REAL TIMSEC
      COMMON/PCTC/TC(MPCOM)
      REAL TC
      COMMON/PCEQU/IMPEXP,RATIMP
      INTEGER IMPEXP
      REAL RATIMP
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)
      INTEGER MAXITP,ITRCLP,ICSV
      REAL PERREL,PERTMP,PERFLX,PERMFL,CSVI
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      INTEGER NPCOMP,NCI
      REAL CDATA
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      INTEGER NPCON,IPC1,IPN1,IPCT,IPC2,IPN2
      REAL PCONDR,PCONSD
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      INTEGER NPCDAT,IPOFS1,IPOFS2
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      REAL ADATA,BDATA
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      REAL CSVF,CSVP
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      REAL PCTF,PCRF,PUAF,PCQF,PCNTMF,PCTP,PCRP,PUAP,PCQP,PCNTMP
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      REAL CONVAR
      INTEGER ICONTP,ICONDX
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)
      REAL PCDATF,PCDATP
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER NAPDAT
      COMMON/C6/INDCFG
      INTEGER INDCFG
      COMMON/FVALA/TFA(MCOM),QFA(MCOM)
      REAL TFA,QFA
      COMMON/FVALS/TFS(MCOM,MS),QFS(MCOM)
      REAL TFS,QFS
      REAL SMALL
      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE),PHI_N,TS_N,TE_N,TX_N,RADEX,XKVALUE,TE,SUMT,
     & SUMW,W,TS,TX,T_rad_mean,Q1,Q2,Q,CM,C1,ALPHA,SHTFLD
      logical closea
      INTEGER ICON1,INOD1,INOD2,IZ,NW,IW,IX1,I,NITMS,ISTATS,IPCOMP

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP44C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Initialize parameters value for the radiator
         PHI_N=BDATA(IPCOMP,1)
         TS_N=BDATA(IPCOMP,2)
         TE_N=BDATA(IPCOMP,3)
         TX_N=BDATA(IPCOMP,4)
         RADEX=BDATA(IPCOMP,5)
         XKVALUE=PHI_N*((0.5*(TS_N+TX_N)-TE_N)**(-1.*RADEX))

C Then evaluate the current environment temperature
C If this is a plant only simulation or if no zone has been defined then
C set the environmental temperature to the nominal value specified in the
C data.
         IF(INDCFG.EQ.2.OR.NINT(BDATA(IPCOMP,10)).EQ.0) THEN
            call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
            IF(closea) THEN
               TE=TE_N
            ELSE
               TE=PCNTMF(IPCOMP)
            ENDIF

C Otherwise get the zone/surface temperatures.
         ELSE
            IZ=INT(BDATA(IPCOMP,10))
            SUMT=TFA(IZ)

C            NW=int(BDATA(IPCOMP,11))
C Just consider two maximum of two walls currently
            NW=2

            SUMW=1.
            IF(NW.GE.1.) THEN
               DO 10 IW=1,NW
                  W=BDATA(IPCOMP,5+IW*2)
                  SUMW=SUMW+W  ! next line needs check for zero array.
                  SUMT=SUMT+W*TFS(IZ,NINT(BDATA(IPCOMP,4+IW*2)))
   10          CONTINUE
            END IF
            TE=SUMT/SUMW
         END IF

C Mark the nodal temperatures for iteration.
         ICSV(INOD1,1)=1
         CSVI(INOD1,1)=CSVF(INOD1,1)
         ICSV(INOD2,1)=1
         CSVI(INOD2,1)=CSVF(INOD2,1)
         TS=CSVF(INOD1,1)
         TX=CSVF(INOD2,1)
         T_rad_mean=0.5*(TS+TX)

C Then calculate radiator heat emission Q
           IF(TS.GT.TE)THEN
             Q1=0.5*XKVALUE*(TS-TE)**RADEX
           ELSE
             Q1=0.0
           ENDIF
           IF(TX.GT.TE)THEN
             Q2=0.5*XKVALUE*(TX-TE)**RADEX
           ELSE
             Q2=0.
           ENDIF
           Q=Q1+Q2

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)/2.
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Inter-node fluid heat capacity rate (W/K)
         PCDATF(IPCOMP,1)=CSVF(INOD1,2)*SHTFLD(3,CSVF(INOD1,1))

C Calculate current component time-constant TC
         TC(IPCOMP)=AMAX1(
     &        CM/AMAX1(SMALL,(C1)),
     &        CM/AMAX1(SMALL,(PCDATF(IPCOMP,1)+
     &        Q/AMAX1(SMALL,AMAX1(SMALL,(T_rad_mean-TE))))))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self-coupling coefficients,
C  node 1
         COUT(1)=ALPHA*(-C1)-CM/TIMSEC
C  node 2
         COUT(2)=ALPHA*PCDATF(IPCOMP,1)
         COUT(3)=ALPHA*(-PCDATF(IPCOMP,1))-CM/TIMSEC
C then matrix equation cross-coupling coefficient,
         COUT(4)=ALPHA*C1
C and then present-time coefficients (ie. right hand sides)
         COUT(5)=((1.-ALPHA)*PCRP(ICON1)-CM/TIMSEC)*CSVP(INOD1,1)
     &           +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &           +ALPHA*Q1+(1.-ALPHA)*PCQP(INOD1)
         COUT(6)=(1.-ALPHA)*(-PCDATP(IPCOMP,1))*CSVP(INOD1,1)
     &           +((1.-ALPHA)*PCDATP(IPCOMP,1)-CM/TIMSEC)*CSVP(INOD2,1)
     &           +ALPHA*Q2+(1.-ALPHA)*PCQP(INOD2)

C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD1)=Q1
         PCQF(INOD2)=Q2

C Establish "containment loss" data
         QDATA(IPCOMP)=ALPHA*Q+(1.-ALPHA)*(PCQP(INOD2)+PCQP(INOD1))

C Establish additional output variables
         NAPDAT(IPCOMP)=3
         PCAOUT(IPCOMP,1)=Q
         PCAOUT(IPCOMP,2)=0.5*(TS+TX)
         PCAOUT(IPCOMP,3)=TE

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-1.
         COUT(3)=1.
         COUT(4)=-PCONDR(ICON1)
         COUT(5)=0.
         COUT(6)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=1.
         COUT(4)=0.
         COUT(5)=0.
         COUT(6)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH basic radiator'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' T_rad_mean = ',T_rad_mean,' (C)'
            WRITE(ITU,*) ' TE     = ',TE,' (C)'
            WRITE(ITU,*) ' Q      = ',Q,' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP44C'

      RETURN
      END

C ******************** CMP45C ********************
C CMP45C generates for plant component IPCOMP with plant db code 450 ie.
C 1 node (ISV=20) WCH air-source heat pump feeding a hydronic heating system.
C the matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)

C     ADATA: 1 Mass of component (solids+liquid) (kg)
C            2 Mass weighted average specific heat node (J/kgK)
C            3 UA modulus for component (W/K)

C     BDATA: 1 COP model [1 - fixed COP 2 - modified Carnot efficiency; 3- f(dT) 4 - polynomial]
C            2 Model coef a0
C            3 Model coef a1
C            4 Model coef a2
C            5 Model coef a3
C            6 Device power draw (kW) model [1 - fixed 2- polynomial 3-exponential b0*e^(b1)]
C            7 Model coef b0
C            8 Model coef b1
C            9 Model coef b2
C           10 Model coef b3
C           11 Compressor pf (-)
C           12 Pump rating (W)
C           13 Pump pf (-)
C           14 Flowrate at rated pump power (l/s)
C           15 Fan power (W)
C           16 Fan pf (-)
C           17 Controller power (W)
C           18 Controller pf (-)
C           19 Tout max (degC)
C           20 Tin max (degC)
C           21 Defrost cycle trigger ambient temp (degC)
C           22 Defrost cycle time calc (1-user def 2-f(RH))
C           23 Defrost cycle  calc coefficient b1/fixed defrost cycle  time  (-)
C           24 Defrost cycle  calc coefficient b2 (-)
C           25 Defrost cycle  lockout time (mins)
C           26 Min defrost time (mins)
C           26 Max defrost time (mins)
C           27 Temp compensation on/off (-)
C           28 Nominal water return temperature (Deg C)
C           29 Nominal water return deadband (Deg C)
C           30 Ambient temperature for temp compensation start [Deg C]
C           31 Ambient temperature for temp compensation end [Deg C]
C           32 Temp compensation gradient [deg C return/deg C ambient] (degCr/degCa)


C     CDATA: 1 Call for heat [0 or 1] (-)

C Node 1 represents the condenser heat exchanger and couples to the hydronic heating circuit.
C The device has an internal pump and so an explicit pump model is not needed in system
C models containing this component. This particular device model is suitable for ON/OFF type control
C and will require modification if PID-type control is to be applied. The model internally controls the
C water outlet temperature and ambient temperature compensation can be applied [linear model]

      SUBROUTINE CMP45C(IPCOMP,COUT,ISTATS)
       use h3kmodule
      implicit none
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/ITERINDEX/ITERNU !plant iteration number
      real BTIMEP,BTIMEF
      COMMON/BTIME/BTIMEP,BTIMEF
      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)

      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      common/pcnam/pcname(mpcom)

      COMMON/CLIMIP/QFPP,QFFP,TPP,TFP,QDPP,QDFP,VPP,VFP,DPP,DFP,HPP,HFP

      COMMON/ASHPvar1/CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,
     &CompressPf,PumpRating,PumpPf,RatedFlow,
     &FanRating,FanPf,CtlRating,CtlPf,ToutMax,TinMax,DefrostT,
     &DefrostTime,Defrostb0,Defrostb1,DefrostLockout,DefrostMinTime,
     &DefrostMaxTime,NomRetT,NomRetTDeadB,TempCompS,TempCompE,
     &TempCompc0,DefrostDur,DefrostLockDur

      COMMON/ASHPvar2/COPModel,CompModel,DefrostCalc,AmbientTempComp,
     &DeviceONOFFp,DeviceONOFF

      COMMON/ASHPvar3/CallforHeat,InDeadB,DefrostLock,InDefrost

C Electrical details for specified plant components
      common/pcelflg/ipcelf(mpcom)
      common/elpcp/NPEL,PFP(mpcom),IPFP(mpcom),PWRP(mpcom),
     &BVOLTP(mpcom),IPHP(mpcom)

      REAL SMALL
      PARAMETER (SMALL=1.0E-20)

      REAL PFP,PWRP,BVOLTP,PWRQ,PA,PQ

      INTEGER ipcelf,NPEL,IPFP,IPHP,IEMODEL

      LOGICAL CallforHeat,InDeadB,DefrostLock,InDefrost

      INTEGER COPModel,CompModel,DefrostCalc,AmbientTempComp,
     &DeviceONOFFp,DeviceONOFF

      REAL CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,CompressPf,
     &PumpRating,PumpPf,RatedFlow,FanRating,FanPf,CtlRating,CtlPf,
     &ToutMax,TinMax,DefrostTime,DefrostT,Defrostb0,Defrostb1,
     &DefrostLockout,DefrostMinTime,DefrostMaxTime,NomRetT,NomRetTDeadB,
     &TempCompS,TempCompE,TempCompc0,DefrostDur,DefrostLockDur

      REAL DefrostStat,CompPower,CompressPower,COP,CtlPower,DeviceFlow,
     &FanPower,HeatOutput,PumpPower,ReturnSP,ReturnSPH,ReturnSPL,RHamb,
     &TotApparentPower,TotReacPower,TotRealPower

      INTEGER lnblnk
      REAL SHTFLD

      INTEGER ITC,ICNT,ITCF,ITRACE,IZNTRC,ITU,ITERNU,idynow,
     &IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,IMPEXP
      INTEGER NPCDAT,IPOFS1,IPOFS2,ICON1,INOD1,napdat,
     &IPCOMP,ISTATS,I,ICONDX,ICONTP,IX1,NCI,NITMS,NPCOMP

      REAL TIMSEC,TC,RATIMP,CSVF,CSVP,PCTF,PCRF,PUAF,PCQF,PCNTMF,
     &PCTP,PCRP,PUAP,PCQP,PCNTMP,CONVAR,QFPP,QFFP,TPP,TFP,QDPP,QDFP,VPP
      REAL VFP,DPP,DFP,HPP,HFP,Tamb,ReturnT,Td,Ta,TL,TH,ALPHA,CM,C1,
     &TmpDev

      REAL COUT(MPCOE),QDATA,PCAOUT,CDATA

      LOGICAL CLOSE,CLOSEA

      CHARACTER OUTS*248,PCNAME*15
      character clkstr*36  ! timestamp string

      INTEGER IPCOMP_LEN   ! lenght of ipcomp
      integer lnclk        ! length of timestapm

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP45C'

C **NB** Variable Assignment and initialisation in CMP45S

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

      call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
      IF(closea) UAMod=0.

      Tamb=TFP                        ! Read ambient temperature.
      IF(CDATA(IPCOMP,1).GT.0.0) THEN ! Set initial operation mode based on CDATA.
        CallForHeat=.true.
        PumpPower=PumpRating          ! Pump circulates while there is a call for heat.
        DeviceFlow=RatedFlow
      ELSE
        CallforHeat=.false.
        DeviceFlow=0.
        PumpPower=0.
      ENDIF


C Calculate the return water temperature set point if temperature compensation is active.
      ReturnSP=NomRetT !default return temperature
      ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
      ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)
      IF(ReturnSP.GT.TinMax) ReturnSP=TinMax

      IF(AmbientTempComp.GT.0)THEN

        IF(Tamb.GE.TempCompS.AND.Tamb.LT.TempCompE)THEN
          ReturnSP=((Tamb-TempCompS)*TempCompc0)+NomRetT
          ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
          ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)

C Case where Tamb > Temp compensation saturation point
        ELSEIF(Tamb.GE.TempCompE)THEN
          ReturnSP=((TempCompE-TempCompS)*TempCompc0)+NomRetT
          ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
          ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)
        ENDIF
        IF(ReturnSP.GT.TinMax)THEN
          ReturnSP=TinMax
          ReturnSPH=ReturnSP
        ENDIF
      ENDIF

C Determine whether machine should be on or off based on return water temperature.
      ReturnT=CONVAR(ICON1,1)
      DeviceONOFFp=DeviceONOFF
      IF(CallforHeat)THEN

C Check to see if the device is in the dead band.
        IF(ReturnT.GE.ReturnSPL.AND.ReturnT.LE.ReturnSPH)THEN
          InDeadB=.true.
        ELSE
          InDeadB=.false.
        ENDIF

C Evaluate the device status.
        IF(ReturnT.GT.ReturnSPH)THEN
          DeviceONOFF=0
        ELSEIF(ReturnT.LT.ReturnSPL)THEN
          DeviceONOFF=1
        ELSEIF(DeviceONOFFp.EQ.1.AND.InDeadB)THEN
          DeviceONOFF=1
        ELSEIF(DeviceONOFFp.EQ.0.AND.InDeadB)THEN
          DeviceONOFF=0
        ENDIF
      ELSE
        DeviceONOFF=0
      ENDIF

C Establish the device COP
      IF(DeviceONOFF.GT.0)THEN

C Determine the defrost status of the device (do only once per timestep so check
C iteration number.
        IF(DefrostCalc.GT.0)THEN

          IF(.NOT.InDefrost)THEN
C Update the defrost time.
            IF(DefrostCalc.eq.1)THEN
              DefrostTime=Defrostb0
            ELSE
              RHamb=HFP
C Calculate defrost time to the nearest minute.
              DefrostTime=float(nint(Defrostb0+Defrostb1*RHamb))
            ENDIF

C Impose limits on defrost time
            IF(DefrostTime.GT.DefrostMaxTime)
     &DefrostTime=DefrostMaxTime
            IF(DefrostTime.LT.DefrostMinTime)
     &DefrostTime=DefrostMinTime

C Issue warning about timestep length if it is longer than the Defrost time.
            IF(DefrostTime.LT.TIMSEC/60.)THEN
              WRITE(OUTS,'(a,f7.4,a,f7.4,a)')
     &'ASHP Warning: time step ',TIMSEC/60.,' > defrost cycle time',
     &DefrostTime,' reduce the time step to avoid problems.'
              CALL EDISP(IUOUT,OUTS)
            ENDIF
          ENDIF

C Set defrost state.
          IF(Tamb.LE.DefrostT)THEN
            IF(.NOT.InDefrost)THEN
              IF(.NOT.DefrostLock)THEN
                InDefrost=.TRUE.
                DefrostDur=0.0
              ENDIF
            ENDIF
          ELSE
            InDefrost=.FALSE.
            DefrostLock=.FALSE.
          ENDIF

C Increment timers
          IF(InDefrost.and.ITERNU.eq.1)
     &DefrostDur=DefrostDur+0.5*timsec/60.

          IF(DefrostLock.and.ITERNU.eq.1)
     &DefrostLockDur=DefrostLockDur+0.5*timsec/60.

C Update state for next ts
          IF(DefrostDur.GT.DefrostTime)THEN
            InDefrost=.false.
            DefrostDur=0.0
            DefrostLock=.true.
            DefrostLockDur=0.0
          ENDIF

          IF(DefrostLockDur.GT.DefrostLockout)THEN
            DefrostLock=.false.
            DefrostLockDur=0.0
          ENDIF

        ENDIF !end of defrost code.
C --------------------------

C Calculate the TOTAL device power draw (W). Debug lines below
C write for fort.45 in support of performance curve visualisation.
        IF(CompModel.eq.1)THEN
          CompPower=1000.*(Compa0) !fixed power consumption
        ELSEIF(CompModel.eq.2)THEN
          Td=ReturnT
          Ta=Tamb
          CompPower=1000.*(Compa0+(Compa1*Ta)+(Compa2*Td))
C          write(45,'(3a,8F11.4)') 'CMP45C ',clkstr(1:lnclk),
C     &      ' comp',CompPower,
C     &      Compa0,Compa1,Compa2,TD,TA,(Compa1*Ta),Compa2*Td
        ELSEIF(CompModel.eq.3)THEN
          Td=ReturnT
          Ta=Tamb
          CompPower=1000.*Compa0*EXP(Compa1*(Td-Ta)) !exponential power consumption.
C          write(45,'(3a,7F11.4)') 'CMP45C ',clkstr(1:lnclk),
C     &      ' expo comp',CompPower,
C     &      Compa0,Compa1,TD,TA,(Td-Ta),Compa1*(Td-Ta)
        ENDIF
        if(CompPower.LT.0.0)CompPower=0.0

        FanPower=FanRating

C Calculate the COP based on the user-specified method. 
        CALL DAYCLKSTR(IDYP,BTIMEF,clkstr)
        lnclk=lnblnk(clkstr)
        IF(COPModel.EQ.1)THEN
          COP=COPa0 !fixed COP
        ELSEIF(COPModel.EQ.2)THEN
          CarEffMod = COPa0   ! Other code blocks suggest this is the modifier slot.
          TL=Tamb+273.15
          TH=ReturnT+273.15
          COP=CarEffMod*((1-(TL/TH))**(-1)) !modified carnot COP
C          write(45,'(3a,5F10.4,F10.2)') 'CMP45C ',clkstr(1:lnclk),
C     &     ' carnot COP amb retrn',
C     &      COP,CarEffMod,TL,TH,CompPower,COP*CompPower
        ELSEIF(COPModel.EQ.3)THEN !quadratic based on Td-Ta
          TL=Tamb
          TH=ReturnT
          COP=COPa0+(COPa1*(TH-TL))+(COPa2*(TH-TL)**2.)
C          write(45,'(3a,7F10.4,F10.2)') 'CMP45C ',clkstr(1:lnclk),
C     &     ' quad COP',COP,COPa0,
C     &      COPa1,COPa2,TL,TH,CompPower,COP*CompPower
        ELSEIF(COPModel.EQ.4)THEN
          COP=COPa0+COPa1*Tamb+COPa2*Tamb**2.+COPa3*Tamb**3. !cubic polymonial COP based on ambient T
C          write(45,'(3a,7F10.4,F10.2)') 'CMP45C ',clkstr(1:lnclk),
C     &      ' cubic COP',COP,COPa0,
C     &      COPa1,COPa2,COPa3,Tamb,CompPower,COP*CompPower
        ELSE
          WRITE(OUTS,'(a)')'Error in ASHP, COP model #'
          CALL EDISP(IUOUT,OUTS)
        ENDIF

C Set limits on calculated values of COP
        IF(COP.LT.0.0) COP=0.0
        IF(COP.GT.12.0) COP=12.0

        IF(InDefrost)THEN
          HeatOutput=0.0
        ELSE
          HeatOutput=COP*CompPower
          IF(HeatOutput.LT.0) THEN
            write(*,*) "Error-5920: heat output < 0! ",HeatOutput, COP,
     &CompPower
            STOP
          ENDIF
        ENDIF
      ELSE

C Device is 'off' set parameters accordingly (assume controller is still operating).
        CompPower=0.
        HeatOutput=0.
        FanPower=0.
        COP=0.
        InDefrost=.false.
        DefrostLock=.false.
      ENDIF

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Establish the nodal thermal capacities.
        CM=CompMass*AveSpHt

C node 1 water
        C1=DeviceFlow*SHTFLD(3,CONVAR(ICON1,1))

C Establish heat loss modulus ((see line 5711 ... !!))
cx        CALL ECLOSE(PCNTMF(IPCOMP),99.,0.0001,CLOSE)
cx        IF(CLOSE) UAMod=0.0

C Calculate the electrical demand of the device assumung the controller is ON.
        CtlPower=CtlRating
        TotRealPower=CompPower+CtlPower

C Save the real power draw for use in an electrical power flow simulation. 
         PWRP(IPCOMP)=-ABS(TotRealPower)
         IEMODEL=1
         CALL EMACH(IPCOMP,IEMODEL,PWRP(IPCOMP),PQ,PA)
         PWRQ=PQ


c Calculate the compressor power draw.
        CompressPower=CompPower-(PumpPower+FanPower+CtlPower)
        IF(CompressPower.LT.0.0)CompressPower=0.0
C        TotRealPower=CompPower+PumpPower+CtlPower+FanPower

        TotReacPower=(((CompressPower/CompressPf)**2)
     &-(CompressPower**2))**0.5
     &+(((PumpPower/PumpPf)**2)-(PumpPower**2))**0.5
     &+(((CtlPower/CtlPf)**2)-(CtlPower**2))**0.5
     &+(((FanPower/FanPf)**2)-(FanPower**2))**0.5

        TotApparentPower=((TotRealPower**2)+(TotReacPower**2))**0.5

C Calculate current component time-constant TC
        TC(IPCOMP)=CM/AMAX1(SMALL,(C1+UAMod))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
           ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
           ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
             ALPHA=1.
          ELSE
             ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
        END IF

C Establish matrix equation self- and cross-coupling coefficients.
        COUT(1)=ALPHA*(-C1-UAMod)-CM/TIMSEC
C Matrix cross coupling coefficients.
        COUT(2)=ALPHA*C1
C Establish the present and known coefficient i.e. RHS
        COUT(3)=((1.-ALPHA)*(PCRP(ICON1)+PUAP(INOD1))-CM/TIMSEC)
     &          *CSVP(INOD1,1)
     &          +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &          -ALPHA*UAMod*PCNTMF(IPCOMP)
     &          -ALPHA*HeatOutput
     &          -(1.-ALPHA)*PCQP(INOD1)
     &          -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)

C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD1)=HeatOutput
         PUAF(INOD1)=UAMod

C Addition output for results analysis.
         NAPDAT(IPCOMP)=9
         PCAOUT(IPCOMP,1)=HeatOutput
         PCAOUT(IPCOMP,2)=COP
         PCAOUT(IPCOMP,3)=Tamb
         PCAOUT(IPCOMP,4)=DeviceONOFF
         PCAOUT(IPCOMP,5)=ReturnSP
         PCAOUT(IPCOMP,6)=TotRealPower
         PCAOUT(IPCOMP,7)=TotReacPower
         PCAOUT(IPCOMP,8)=TotApparentPower
         if(Indefrost)then
           PCAOUT(IPCOMP,9)=1.0
         elseif(DefrostLock)then
           PCAOUT(IPCOMP,9)=-1.0
         else
           PCAOUT(IPCOMP,9)=0.0
         endif
C---------------------------------------------------------------------------------
C Make select results available in XML and CVS output.
C---------------------------------------------------------------------------------
        IPCOMP_LEN = lnblnk(pcname(IPCOMP))

        call AddToReport(rvPltHOut%Identifier,
     &         HeatOutput,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltCOP%Identifier,
     &         COP,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltTambient%Identifier,
     &         Tamb,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        tmpdev=float(DeviceONOFF)
        call AddToReport(rvPltDeviceONOFF%Identifier,
     &         tmpdev,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltReturnTSP%Identifier,
     &         ReturnSP,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltAmbientHeat%Identifier,
     &         -HeatOutput+CompPower*CompressPf,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

       call AddToReport(rvPltRealPow%Identifier,
     &         TotRealPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltReacPow%Identifier,
     &         TotReacPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

       call AddToReport(rvPltApparPow%Identifier,
     &         TotApparentPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

       DefrostStat=PCAOUT(IPCOMP,9)
       call AddToReport(rvPltDefrostStat%Identifier,
     &         DefrostStat,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))


C 1st phase mass (ie. "water") balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=DeviceFlow

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
      END IF


C Trace.
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) Air Source Heat Pump'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM      = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1      = ',C1,' (W/Ks)'
            WRITE(ITU,*) ' Flow      = ',DeviceFlow,' (l/s)'
            WRITE(ITU,*) ' TC      = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA   = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UAMod   = ',UAMod,' (W/K)'
            WRITE(ITU,*) ' Tamb    = ',Tamb,' (C)'
            WRITE(ITU,*) ' ReturnT = ',ReturnT,' (C)'
            WRITE(ITU,*) ' ReturnSP        = ',ReturnSP,' (C)'
            WRITE(ITU,*) ' ReturnSPH       = ',ReturnSPH,' (C)'
            WRITE(ITU,*) ' ReturnSPL       = ',ReturnSPL,' (C)'
            WRITE(ITU,*) ' InDeadB         = ',InDeadB,' (-)'
            WRITE(ITU,*) ' CallforHeat     = ',CallforHeat,' (-)'
            WRITE(ITU,*) ' DeviceONOFF     = ',DeviceONOFF,' (-)'
            WRITE(ITU,*) ' DeviceONOFFp    = ',DeviceONOFFp,' (-)'
            WRITE(ITU,*) ' COP             = ',COP,' (-)'
            WRITE(ITU,*) ' AmbientTempComp = ',AmbientTempComp,' (C)'
            WRITE(ITU,*) ' TempCompS       = ',TempCompS,' (C)'
            WRITE(ITU,*) ' TempCompE       = ',TempCompE,' (C)'
            WRITE(ITU,*) ' DefrostCalc     = ',DefrostCalc,' (-)'
            WRITE(ITU,*) ' AmbientRH       = ',RHamb,' (%)'
            WRITE(ITU,*) ' DefrostTime     = ',DefrostTime,' (mins)'
            WRITE(ITU,*) ' DefrostDur      = ',DefrostDur,' (-)'
            WRITE(ITU,*) ' InDefrost       = ',Indefrost,' (-)'
            WRITE(ITU,*) ' DefrostLock     = ',DefrostLock,' (-)'
            WRITE(ITU,*) ' DefrostLockout  = ',DefrostLockout,' (mins)'
            WRITE(ITU,*) ' DefrostLockDur  = ',DefrostLockDur,' (mins)'
            WRITE(ITU,*) ' HeatOutput   = ',HeatOutput,' (W)'
            WRITE(ITU,*) ' CompPower    = ',CompPower,' (W)'
            WRITE(ITU,*) ' FanPower     = ',FanPower,' (W)'
            WRITE(ITU,*) ' PumpPower    = ',PumpPower,' (W)'
            WRITE(ITU,*) ' CtlPower     = ',CtlPower,' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP45C'

      RETURN
      END

C ******************** CMP46C ********************

C CMP46C generates for plant component IPCOMP with plant db code 420 ie.
C 2 node (ISV=20) WCH boiler with modulation
C This boiler is similar to the WCH 2 node boiler CMP42
C except the efficiency is a function of outdoor air, fuel burned and
C operating conditions
C Air is taken to be 79% Nitrogen and 21% Oxygen

C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Full load firing rate kW
C            2 Dump load fraction
C            3 BS 4947 / EN 437 gas specification (e.g. for G20 enter 20)
C            4 Boiler lock out time (min)
C            5 Boiler heat loss to surroundings UA (W/K)
C            6 DHW draw schedule control loop no. if combiboiler (0 otherwise)
C            7 DHW calorifier component number (0 otherwise)
C            8 Constant cold water feed temperature for DHW for combiboiler (degC)
C            9 Constant supply temperature for DHW for combiboiler(degC)
C           10 Upper boiler temperature limit (C)
C           11 Lower limit of modulating range (%), 0 for nonmodulating
C           12 Lower limit total differential (%), 0 for nonmodulating
C        13-20 Coefficients of percentage CO2 and delta temperature to flue equations: 
C              perCO2=A+B*FR+C*FR^2+D*FR^3 where FR is firing rate in kW, deltaTFlue has similar equation
C           21 Heat loss coefficient at startup purges
C           22 First pre-purge time (seconds)
C           23 First pre-purge fan rate as % of maximum
C           24 Secoond pre-purge time  (seconds)
C           25 Second pre-purge fan rate as % of maximum
C           26 Ignition stabilisation time (seconds)
C           27 Firing rate during ignition stabilisation as % of maximum 
C           28 Anticycle time (seconds)
C           29 Firing rate during anticycling as % of maximum
C           30 Maximum Ramp up gradient (typically supply water is restricted to temp rise of 3K/min)
C     CDATA: 1 ON/OFF control signal (-)

C     PCDATF/P
C            1 Holds boiler actual ON/OFF state (-)
C            2 Inter-node fluid heat capacity rate (W/K)
C
C Various variables used in the subroutine:
C perCO2     = percentage of CO2 in the flue
C fRate      = firing rate (kW)
C dTFlue     = delta Temp flue gas and return water 
C fnCarbon   = fractional number of Carbon atoms in fuel
C fnHydrogen = fractional number of Hydrogen molecules in fuel
C fnNitrogen = fractional number of Nitrogen molecules in fuel
C pCH4,pH2,pN2,pC3H8,pC4H10,pC2H6 = % or each chemical in fuel
C fnCO2,fnH2O,fnN2 = fractional number of each molecule in fuel exhaust
C                    (excluding N2 from air)
C R_H2O,R_Air = Universal gas constant for each compound/mixture
C L_H2O       = latent heat of vapourisation of water
C volFlue     = volume of flue gas
C TR          = Return water temperature
C TFLOW       = Flow water temperature
C ZMFR        = Water mass flow rate though boiler
C G           = heat supplied by gas (gas firing rate*calorifc value)
C INSO        = boiler flag for lockout (if = 1 then in lockout time)
C ITSBO       = present no. of timesteps boiler is off due to lockout
C NTSBO       = total no. of timesteps boiler is to be off due to lockout

      SUBROUTINE CMP46C(IPCOMP,COUT,ISTATS)
      IMPLICIT NONE

#include "plant.h"
#include "building.h"
#include "control.h"
#include "chemical_properties.h"

      INTEGER IUOUT,IUIN,IEOUT
      COMMON/OUTIN/IUOUT,IUIN,IEOUT

      COMMON/TC/ITC,ICNT
      INTEGER ITC,ICNT

      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      INTEGER ITCF,ITRACE,IZNTRC,ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      INTEGER IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow

      COMMON/Pctime/TIMSEC
      REAL TIMSEC

      COMMON/PTIME/PTIMEP,PTIMEF
      REAL PTIMEP,PTIMEF

      COMMON/PCTC/TC(MPCOM)
      REAL TC

      COMMON/PCEQU/IMPEXP,RATIMP
      INTEGER IMPEXP
      REAL RATIMP

      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)
      INTEGER MAXITP,ITRCLP,ICSV
      REAL PERREL,PERTMP,PERFLX,PERMFL,CSVI

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      INTEGER NPCOMP,NCI
      REAL CDATA

      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      INTEGER NPCON,IPC1,IPN1,IPCT,IPC2,IPN2
      REAL PCONDR,PCONSD

      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      INTEGER NPCDAT,IPOFS1,IPOFS2

      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      REAL ADATA,BDATA

      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      REAL CSVF,CSVP

      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      REAL PCTF,PCRF,PUAF,PCQF,PCNTMF,PCTP,PCRP,PUAP,PCQP,PCNTMP

      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      REAL CONVAR
      INTEGER ICONTP,ICONDX

      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)
      REAL PCDATF,PCDATP

      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      REAL QDATA,PCAOUT
      INTEGER NAPDAT

      COMMON/LOCAL1/ISET(MCF),PHIMAX
      INTEGER ISET
      REAL PHIMAX

      COMMON/LOKOUT/INSO,ITSBO
      INTEGER INSO,ITSBO

      COMMON/PCLOP8/LASTOUT(MCF)
      INTEGER LASTOUT

      COMMON/FN_CHEM/fnCO2,fnH2O,fnN2,fnO2,fnN2O2,iFuel,IPCMP,
     &A1,A2,A3,A4,A5,A6,A7,A8
      REAL fnCO2,fnH2O,fnN2,fnO2,fnN2O2,A1,A2,A3,A4,A5,A6,A7,A8
      INTEGER IPCMP

      COMMON/pstctr/nsincp,nsncpr
      INTEGER nsincp, nsncpr

      COMMON/CLIMI/QFP,QFF,TP,TF,QDP,QDF,VP,VF,DP,DF,HP,HF
      REAL QFP,QFF,TP,TF,QDP,QDF,VP,VF,DP,DF,HP,HF

      REAL SMALL
      PARAMETER (SMALL=1.0E-20)
      REAL      COUT(MPCOE),TOTMAS,UA,TFLOW,
     & RUBTL,CUTIN,CUTOUT,FEEDTM,SUPPTM,RI,
     & RO,HK,A,TR,ZMFR,Q,PHI,BLKOUT,FGAS,CM,
     & ALPHA,C1,SHTFLD,UBTL,futureTemp,presentTemp
      character outs*124
      INTEGER ICON1,INOD1,INOD2,IONOFF,ICOMBI,ICCOMP,ICOMBIMOD,ICCN1,
     & IPC,NTSBO,IX1,I,NITMS,IPCON,ISTATS,IPCOMP,IPSND,icon2,idebug
      integer iFuel,ii,iRunOnce,ICL,iii,noTStepsOn,nsincp_local
      integer debug_counter
      
      real tempAmb,  
     &     effGross,            timeCompOn,
     &     time1stPrePurge,     time2ndPrePurge,   timeIgnStab,
     &     timeAntiCyc,         fanRate1stPP,      fanRate2ndPP,
     &     fRatePerIgnStab,     fRatePerAntiCyc,   tempRampUp,
     &     tempRampUpMax,       initTemp,
     &     volFlowrateFuel,     heatOutFrac,BOIL_EFF

      real fFuel_composition ( iCompound_count )
      real fEval_element_mols_in_mixture
      real maxFanFlowrate,airden
      logical bInStartup
      integer istat

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP46C'

C Check control data for relevant balance type
      IF(ISTATS.EQ.1.AND.
     &   (CDATA(IPCOMP,1).LT.0..OR.CDATA(IPCOMP,1).GT.1.)) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP46C: invalid control data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP46C: unresolvable error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF

C Following code to be run once only
      if(nsinc.eq.1)iRunOnce=0
      if(iRunOnce.eq.0)then
        nsincp_local=0
        iRunOnce=1 !save iRunOnce
        initTemp=20.
        iFuel=nint(BDATA(IPCOMP,3))
        IPCMP=IPCOMP
        ICON1=ICONDX(IPCOMP,1,1)
        INOD1=NPCDAT(IPCOMP,9)
        INOD2=NPCDAT(IPCOMP,9)+1
        debug_counter=0

C Set up real upper boiler temperature limit this will not be modified
        RUBTL=BDATA(IPCOMP,10)

C Set up coefficients for efficiency equations
        A1=BDATA(IPCOMP,13)
        A2=BDATA(IPCOMP,14)
        A3=BDATA(IPCOMP,15)
        A4=BDATA(IPCOMP,16)
        A5=BDATA(IPCOMP,17)
        A6=BDATA(IPCOMP,18)
        A7=BDATA(IPCOMP,19)
        A8=BDATA(IPCOMP,20)

C Set up startup characteristics
        time1stPrePurge   =BDATA(IPCOMP,22)
        time2ndPrePurge   =BDATA(IPCOMP,24)
        timeIgnStab       =BDATA(IPCOMP,26)
        timeAntiCyc       =BDATA(IPCOMP,28)
        fanRate1stPP      =BDATA(IPCOMP,23)
        fanRate2ndPP      =BDATA(IPCOMP,25)
        fRatePerIgnStab   =BDATA(IPCOMP,27)
        fRatePerAntiCyc   =BDATA(IPCOMP,29)
        tempRampUpMax     =BDATA(IPCOMP,30)

C Get composition of fuel and flue
        do ii = 1, iCompound_count
          fFuel_composition (ii) = fuel2ChemProp(iFuel,ii)
        enddo
        fnCO2=fEval_element_mols_in_mixture (
     &      fFuel_Composition, iElement_Carbon ) !x in combustion eqn
        fnH2O=fEval_element_mols_in_mixture (
     &      fFuel_Composition, iElement_Hydrogen )/2. ! y/2 in combustion eqn
        fnN2 =fEval_element_mols_in_mixture (
     &      fFuel_Composition, iElement_Nitrogen )/2.
        fnO2  = fnCO2+fnH2O/2.      ! O2 for combustion
        fnN2O2= fnN2 + fnO2*79./21. ! N2 and O2 for combustion

C Determine loop number of first control loop that controls this boiler
        do 325 iii=1,ncl
          IF(IPAN(Iii,1).eq.-1)THEN
            IF(IPAN(Iii,2).EQ.IPCOMP)ICL=iii
          ENDIF
 325    continue

C Calculate maximum volume flow rate of air by first calculating volume
C flow rate of fuel assuming stoichiometric mass of oxygen supplied
        volFlowrateFuel=1000.*BDATA(IPCOMP,1)/grossCalorific(iFuel)
        maxFanFlowrate=(fnCO2+0.5*fnH2O)*(100./21.)*volFlowrateFuel/100.

C Scale down parameters if dump load is to be supplied.
        TOTMAS=ADATA(IPCOMP,1)*(1.-BDATA(IPCOMP,2))
        UA=BDATA(IPCOMP,5)*(1.-BDATA(IPCOMP,2))
        PHIMAX=1000.*BDATA(IPCOMP,1)*(1.-BDATA(IPCOMP,2))

C Get gas firing modulation cut-in and cut-out
        CUTIN=(BDATA(IPCOMP,11)+BDATA(IPCOMP,12)*0.5)*0.01
        CUTOUT=(BDATA(IPCOMP,11)-BDATA(IPCOMP,12)*0.5)*0.01

C Determine if combi-boiler or not
        ICOMBI=NINT(BDATA(IPCOMP,6))
        IF(ICOMBI.NE.0)THEN
          ICCOMP=NINT(BDATA(IPCOMP,7))
          FEEDTM=BDATA(IPCOMP,8)
          SUPPTM=BDATA(IPCOMP,9)
        ENDIF

C Lock out time (in seconds), 
        BLKOUT=BDATA(IPCOMP,4)*60.       
        NTSBO=NINT(BLKOUT/TIMSEC)

C Get connection which sends mass to boiler, assuming only one 
C component sends mass to the boiler
      DO 1515 IPC=1,NPCON
        IF(IPC1(IPC).EQ.IPCOMP)IPCON=IPC ! This connection sends mass to boiler (node 1)
        IF(IPC2(IPC).EQ.IPCOMP)IPSND=IPC ! Boiler (node 2) sends mass to this connection
 1515 CONTINUE
      ICON2=IPSND

C End code to be run once only
      endif

C Write debugging information
      if(bInStartup())then
        idebug=0
      else
        idebug=0
      endif
      if(idebug.eq.1)then
        if(mod(debug_counter,40).eq.0)write(96,*)
     &    '  plant time    boiler state     tsteps on nsincp',
     &    '  Q  (heat)       T_flow         eff_gross        T_return',
     &    '        CDATA'
        debug_counter=debug_counter+1
      endif  

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      IF(ISTATS.EQ.1) THEN

C Set up upper boiler temperature limit (this can be modified by
C controllers like the PI room controller (OpenTHERM), OTC  (outside
C temperature compensation) etc.)
        UBTL=BDATA(IPCOMP,10)

C %CO2 and flue delta Temperature is currently worked out as a 
C function of percentage of firing rate to maximum (full load) firing 
C rate determined from experimental observations (using dry flue gas)
C Calculate full load %CO2 and dTFlue
      TFLOW=CSVF(INOD2,1)
      TR=CONVAR(IPCON,1)
      IF(TR.GT.TFLOW)TR=TFLOW
      effGross=BOIL_EFF(TR,1.)
      PHI=0.

C Establish whether boiler is on or off (ie 1 or 0)
      IONOFF=NINT(CDATA(IPCOMP,1))

C If boiler is combi and is supplying DHW determine firing rate 
      ICOMBIMOD=0
      IF(ICOMBI.NE.0)THEN
        IF(LASTOUT(ICOMBI).NE.0)THEN
          ICOMBIMOD=1
          ICCN1=ICONDX(ICCOMP,1,1)
          RI=1./(BDATA(ICCOMP,1)*BDATA(ICCOMP,2))
          RO=1./(BDATA(ICCOMP,3)*BDATA(ICCOMP,4))
          HK=1./(RI+RO)
          PHI=(PCONDR(ICCN1)*CONVAR(ICCN1,2)*SHTFLD(3,CONVAR(ICCN1,1)))
     &         *(SUPPTM-FEEDTM)
          A=0.5*HK/
     &      (PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1)))
          TR=((1.-A)*TFLOW+A*(SUPPTM+FEEDTM))/(1.+A)
          IF(TR.LT.FEEDTM)TR=FEEDTM
          IF(PHI.LT.0.)PHI=0.
          IF(PHI.GT.PHIMAX)PHI=PHIMAX
        ENDIF
      ENDIF

C Set boiler on if there is demand for hot water from a combi boiler
      IF(ICOMBIMOD.EQ.1)IONOFF=1
      IF(ICOMBIMOD.EQ.1)GOTO 33

C Get PHI for normal operation (i.e. not in combi mode)
      ZMFR=PCONDR(ICON1)*CONVAR(ICON1,2)
      PHI=ZMFR*SHTFLD(3,CONVAR(ICON1,1))*(UBTL-TR)
      IF(PHI.LT.0.)PHI=0. ! Shut boiler if UBTL is less than return water temperature
      IF(PHI.GT.PHIMAX)PHI=PHIMAX
      IF (NINT(BDATA(IPCOMP,12)).EQ.0)PHI=PHIMAX ! non-modulating boiler (i.e. on/off)

C Calculate load or heat out fraction
 33   heatOutFrac=PHI/PHIMAX

C Reset to off in case GFR is less than CUTOUT or CUTIN as appropriate
      IF(IONOFF.EQ.1)THEN
        IF(heatOutFrac.GT.1.)heatOutFrac=1.
        IF(heatOutFrac.LT.CUTOUT)THEN
          IF((UBTL-TR).LT.10.)THEN
            heatOutFrac=0.
            IONOFF=0
            if(idebug.eq.1)write(96,*)'######### CUTOUT #########'
          ENDIF
        ENDIF
      ELSE
        IF(heatOutFrac.LT.CUTIN)heatOutFrac=0.
      ENDIF

C Update efficiency based on current heatOutFrac
      effGross=BOIL_EFF(TR,heatOutFrac)
      FGAS=PHI/(grossCalorific(iFuel)*effGross)

C If boiler temperature reaches real upper boiler temperature limit 
C it is shut down for amount of time = lock out time
C Note that the check is made if temperature is 5degC more than the
C real UBTL to prevent lockout during ignition stabilisation which
C usually is at a high firing rate (nominally 70%). This has now been
C modified by skipping ignition stabilisation period and going
C straight into anti-cycling which resolves the issue somewhat
      IF(TFLOW.GT.(RUBTL+5.))THEN
        if(idebug.eq.1)write(96,*)ptimef,'hit UBTL'
        INSO=1
        ITSBO=0
      ENDIF
      IF(ITSBO.LE.0)ITSBO=0
      IF(INSO.EQ.1)THEN
        ITSBO=ITSBO+1
        IF(ITSBO.GE.NTSBO)INSO=0
        FGAS=0.
        PHI=0.
        Q=0.0
        noTStepsOn=0
        IONOFF=0
        if(idebug.eq.1)
     &write(96,*)ptimef,'Lockout ',noTStepsOn,nsincp,Q,tflow,
     &effGross,TR,CDATA(IPCOMP,1)
      ENDIF
      IF(INSO.EQ.1)GOTO 333

C First calculate boiler efficiency ETA based on current node 1 temp.
C so mark node 1 temperature for iteration
      ICSV(INOD1,1)=1
      CSVI(INOD1,1)=CSVF(INOD1,1)
 
C Determine startup characteristics if applicable
C For uncontrolled boiler timing is not calculated
      IF(IONOFF.EQ.1.OR.ICOMBIMOD.EQ.1)THEN
        IF (icl.eq.0)then ! i.e. boiler not controlled
          Q=PHI
        else

C Calculate time boiler has been on
          if(nsincp_local.lt.nsincp)then
            noTStepsOn=noTStepsOn+1
            nsincp_local=nsincp
          endif
          timeCompOn=noTStepsOn*TIMSEC  ! Time boiler is ON (sec)

C If in pre-purge time boiler no heat or gas is supplied only fan is on
          tempAmb=TF
          if(timeCompOn.le.time1stPrePurge)then

C Q is the heat lost due to air purging through boiler =m*Cp*dTemp
            Q=-1.*(maxFanFlowrate*fanRate1stPP/100.) *
     &      AIRDEN(tempAmb)*1006.*(TFLOW-tempAmb)*BDATA(IPCOMP,21)
            PHI=Q
            FGAS=0.
            if(idebug.eq.1)
     &      write(96,*)ptimef,'Prepur1 ',noTStepsOn,nsincp,Q,tflow,
     &      effGross,TR,CDATA(IPCOMP,1)
          elseif(timeCompOn.le.(time1stPrePurge+time2ndPrePurge))then
            Q=-1.*(maxFanFlowrate*fanRate2ndPP/100.) *
     &      AIRDEN(tempAmb)*1006.*(TFLOW-tempAmb)*BDATA(IPCOMP,21)
            phi=Q
            FGAS=0.
            if(idebug.eq.1)
     &      write(96,*)ptimef,'Prepur2 ',noTStepsOn,nsincp,Q,tflow,
     &      effGross,TR,CDATA(IPCOMP,1)

C No startup routine for non-modulating (on-off) boiler
          elseif(NINT(BDATA(IPCOMP,12)).EQ.0)then
            Q=PHI
            if(idebug.eq.1)
     &      write(96,*)ptimef,'NonModB ',noTStepsOn,nsincp,Q,tflow,
     &      effGross,TR,CDATA(IPCOMP,1)

C If in ignition stabilisation time boiler runs at a fixed load 
          elseif(timeCompOn.le.
     &    (time1stPrePurge+time2ndPrePurge+timeIgnStab))then

C If flow temperature is close to real UBTL then skip remainder of
C ignition stabilisation and go into anti-cycling 
C set firing rate to anti-cycle rate for this time step as well
            if(TFlow.GT.(RUBTL-4.0))then
              heatOutFrac=fRatePerAntiCyc/100.
              timeCompOn=time1stPrePurge+time2ndPrePurge+timeIgnStab
              if(idebug.eq.1)
     &        write(96,*)ptimef,'ModIStab',noTStepsOn,nsincp,Q,tflow,
     &        effGross,TR,CDATA(IPCOMP,1)
            else
              heatOutFrac=fRatePerIgnStab/100.
              if(idebug.eq.1)
     &        write(96,*)ptimef,'IgnStab ',noTStepsOn,nsincp,Q,tflow,
     &        effGross,TR,CDATA(IPCOMP,1)
            endif            
            PHI=heatOutFrac*PHIMAX
            IF(PHI.LT.0.)PHI=0.
            IF(PHI.GT.PHIMAX)PHI=PHIMAX
            IF(heatOutFrac.GT.1.)THEN
              heatOutFrac=1.
            ELSEIF(heatOutFrac.LT.CUTOUT)THEN
              heatOutFrac=0.
              IONOFF=0
            ENDIF
            effGross=BOIL_EFF(TR,heatOutFrac)
            FGAS=PHI/(grossCalorific(iFuel)*effGross)
            Q=PHI

C If in anti-cycle time boiler runs at a fixed load
          elseif(timeCompOn.le.
     &    (time1stPrePurge+time2ndPrePurge+timeIgnStab+timeAntiCyc))then
            heatOutFrac=fRatePerAntiCyc/100.
            PHI=heatOutFrac*PHIMAX
            IF(PHI.LT.0.)PHI=0.
            IF(PHI.GT.PHIMAX)PHI=PHIMAX
            IF(heatOutFrac.GT.1.)THEN
              heatOutFrac=1.
            ELSEIF(heatOutFrac.LT.CUTOUT)THEN
              heatOutFrac=0.
              IONOFF=0
            ENDIF
            effGross=BOIL_EFF(TR,heatOutFrac)
            FGAS=PHI/(grossCalorific(iFuel)*effGross)
            Q=PHI
            if(idebug.eq.1)
     &      write(96,*)ptimef,'AntiCyc ',noTStepsOn,nsincp,Q,tflow,
     &      effGross,TR,CDATA(IPCOMP,1)

C If in normal operation then restrict heating to ramp up gradient
          else
            tempRampUp=TR-initTemp
            initTemp=TR ! Init for next timestep
            if(tempRampUp.gt.(tempRampUpMax/60.*TIMSEC))then
              PHI=PHI*(tempRampUpMax/60.*TIMSEC)/tempRampUp
              IF(PHI.LT.0.)PHI=0.
              IF(PHI.GT.PHIMAX)PHI=PHIMAX
              heatOutFrac=PHI/PHIMAX
              IF(heatOutFrac.GT.1.)THEN
                heatOutFrac=1.
              ELSEIF(heatOutFrac.LT.CUTOUT)THEN
                heatOutFrac=0.
                IONOFF=0
              ENDIF
              effGross=BOIL_EFF(TR,heatOutFrac)
              FGAS=PHI/(grossCalorific(iFuel)*effGross)
              Q=PHI
              if(idebug.eq.1)write(96,*)ptimef,'NormRam ',noTStepsOn,
     &        nsincp,Q,tflow,effGross,TR,CDATA(IPCOMP,1)
            else
              Q=PHI
              if(idebug.eq.1)write(96,*)ptimef,'NormNRa ',noTStepsOn,
     &        nsincp,Q,tflow,effGross,TR,CDATA(IPCOMP,1)
            endif
          endif
        endif

C Set boiler state to OFF
      ELSE
        if(idebug.eq.1)
     &  write(96,*)ptimef,'OFF     ',noTStepsOn,nsincp,Q,tflow,
     &  effGross,TR,CDATA(IPCOMP,1)
        FGAS=0.
        PHI=0.
        Q=0.0
        noTStepsOn=0
        IONOFF=0
      ENDIF

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
 333    CM=TOTMAS*ADATA(IPCOMP,2)/2.
        C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Boiler actual ON/OFF state PCDATF(IPCOMP,1) (-)
        PCDATF(IPCOMP,1)=IONOFF

C Inter-node fluid heat capacity rate (W/K)
        PCDATF(IPCOMP,2)=CSVF(INOD1,2)*SHTFLD(3,CSVF(INOD1,1))

C Calculate current component time-constant TC
        TC(IPCOMP)=AMAX1(
     &        CM/AMAX1(SMALL,(C1)),CM/AMAX1(SMALL,(PCDATF(IPCOMP,2))))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
          ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
          ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
            ALPHA=1.
          ELSE
            ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
          CM=0.
          ALPHA=1.
        ENDIF

C Establish matrix equation self-coupling coefficients,
C  node 1
        COUT(1)=ALPHA*(-C1-UA)-CM/TIMSEC

C  node 2
        COUT(2)=ALPHA*(PCDATF(IPCOMP,2)+UA)
        COUT(3)=ALPHA*(-PCDATF(IPCOMP,2)-UA)-CM/TIMSEC

C then matrix equation cross-coupling coefficient,
        COUT(4)=ALPHA*C1

C and then present-time coefficients (ie. right hand sides)
c        COUT(5)=((1.-ALPHA)*(PCRP(ICON1)+UA)-CM/TIMSEC)*CSVP(INOD1,1)
c     &           +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
c     &           -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)
c     &           -ALPHA*UA*PCNTMF(IPCOMP)-(1.-ALPHA)*UA*PCNTMP(IPCOMP)
c
c        COUT(6)=(1.-ALPHA)*(-PCDATP(IPCOMP,2)-UA)*CSVP(INOD1,1)
c     &          +((1.-ALPHA)*(UA+PCDATP(IPCOMP,2))
c     &          -CM/TIMSEC)*CSVP(INOD2,1)-ALPHA*Q-
c     &            (1.-ALPHA)*PCQP(INOD2)

C Set present-time coefficients to be mean of both nodes because due to
C pump overrun both nodes will be at similar temperature and will lose
C heat in approximately the same manner
        COUT(5)=((1.-ALPHA)*(0.5*PCRP(ICON1)+0.5*PCRP(ICON2)+UA)-
     &           CM/TIMSEC)*(0.5*CSVP(INOD1,1)+0.5*CSVP(INOD2,1))
     &           +(1.-ALPHA)*(-0.5*PCRP(ICON1)-0.5*PCRP(ICON2))*
     &           (0.5*PCTP(ICON1)+0.5*PCTP(ICON2))
     &           -(1.-ALPHA)*(0.5*PUAP(INOD1)+0.5*PUAP(INOD2))*
     &            PCNTMP(IPCOMP)
     &           -ALPHA*UA*PCNTMF(IPCOMP)-(1.-ALPHA)*UA*PCNTMP(IPCOMP)

        COUT(6)=(1.-ALPHA)*(-PCDATP(IPCOMP,2)-UA)*
     &           (0.5*CSVP(INOD1,1)+0.5*CSVP(INOD2,1))
     &           +((1.-ALPHA)*(UA+PCDATP(IPCOMP,2))
     &           -CM/TIMSEC)*(0.5*CSVP(INOD2,1)+0.5*CSVP(INOD1,1))-
     &           ALPHA*Q-(1.-ALPHA)*(0.5*PCQP(INOD2)+0.5*PCQP(INOD1))

C Store "environment" variables future values
        PCTF(ICON1)=CONVAR(ICON1,1)
        PCRF(ICON1)=C1
        PCQF(INOD2)=Q

C Establish "containment loss" data
        presentTemp=0.5*(csvp(inod1,1)+csvp(inod2,1))
        futureTemp =0.5*(csvf(inod1,1)+csvf(inod2,1))
        QDATA(IPCOMP)=UA*(alpha*(futureTemp-pcntmf(ipcomp))+
     &                (1.-alpha)*(presentTemp-pcntmp(ipcomp)))

C If boiler is off set its efficiency to zero
        IF(IONOFF.EQ.0)effGross=0.

C Establish additional output variables
        napdat(ipcomp)=5

C Boiler ON/OFF signal
        PCAOUT(IPCOMP,1)=IONOFF

C FGAS*1000 is to give resolution on output
c this will translate from m3/s to l/s??
        PCAOUT(IPCOMP,2)=FGAS*1000.

C Upper boiler temperature limit
        PCAOUT(IPCOMP,3)=UBTL

C Heat input to water
        PCAOUT(IPCOMP,4)=PHI

C Efficiency
        PCAOUT(IPCOMP,5)=effGross

C 1st phase mass (ie. water) balance coefficients
      ENDIF
      IF(ISTATS.EQ.2) THEN
        COUT(1)=1.
        COUT(2)=-1.
        COUT(3)=1.
        COUT(4)=-PCONDR(ICON1)
        COUT(5)=0.
        COUT(6)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
        COUT(1)=1.
        COUT(2)=0.
        COUT(3)=1.
        COUT(4)=0.
        COUT(5)=0.
        COUT(6)=0.
      END IF

        !write(96,*)ptimef," boiler ",Q
C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=20) WCH boiler & modulation'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1)
            WRITE(ITU,*) ' IONOFF = ',IONOFF,' (-)'
            WRITE(ITU,*) ' FGAS   = ',FGAS,' (m^3/s)'
            WRITE(ITU,*) ' GASCO  = ',FGAS*TIMSEC,' (m^3)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP46C'

      RETURN
      END

C ******************** BOIL_EFF ********************
C Function to calculate WCH boiler efficiency at any return water
C temperature (T_ret) and part load (F_rate)
      FUNCTION BOIL_EFF(T_ret,F_rate)
c      implicit none
#include "plant.h"
#include "chemical_properties.h"
#include "building.h"
#include "site.h"

      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      REAL ADATA,BDATA
      COMMON/CLIMI/QFP,QFF,TP,TF,QDP,QDF,VP,VF,DP,DF,HP,HF
      REAL QFP,QFF,TP,TF,QDP,QDF,VP,VF,DP,DF,HP,HF
      COMMON/FN_CHEM/fnCO2,fnH2O,fnN2,fnO2,fnN2O2,iFuel,IPCMP,
     &A1,A2,A3,A4,A5,A6,A7,A8
      REAL fnCO2,fnH2O,fnN2,fnO2,fnN2O2,A1,A2,A3,A4,A5,A6,A7,A8,L_H2O,
     &massWaterCond
      INTEGER IPCMP
      INTEGER iFuel
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER IUOUT,IUIN,IEOUT
      logical closer

C Init
      IPCOMP=IPCMP
      R_H2O = fR_universal / fEval_Compound_MM(iWater_l)
      iFuel=nint(BDATA(IPCOMP,3))

C %CO2 and flue delta Temperature is currently worked out as a 
C function of percentage of firing rate to maximum firing rate
C determined from experimental observations (using dry flue gas)
C Calculate full load %CO2 and dTFlue
      fRate=F_rate*100.
      perCO2= a1 + a2*fRate + a3*fRate**2 + a4*fRate**3
      dTFlue= a5 + a6*fRate + a7*fRate**2 + a8*fRate**3
      tempFlue=T_ret+dTFlue
      tempAmb=TF
      L_H2O = EVAP01(tempFlue)

C Determine excess air factor (it is worked out as a ratio and does not
C require the actual volume of flue gas or air intake)
      call eclose(fRate,100.,0.0001,closer)

      volFlue=100.*fnCO2/perCO2
      volXsAir=volFlue-fnN2-(79./21.)*(fnCO2+0.5*fnH2O)-fnCO2
      xsAirFactor=(fnCO2+0.5*fnH2O+0.21*volXsAir)/(fnCO2+0.5*fnH2O)
      if(xsAirFactor.lt.1.0)
     &  call edisp(iuout,
     &  'WARNING: CMP46 Excess air factor less than unity')

C Get net boiler efficiency (non-condensing) using Siegert's equation
      effSiegert=P(iFuel)-
     &(Qgas(iFuel)/perCO2+R(iFuel))*(tempFlue-tempAmb)

C Can get saturated vapour pressure using Teten's formula.
C      sattPressWater=610.78*exp(17.2694*tempFlue/(tempFlue+238.3))
C Get saturated vapour pressure of water at tempFlue 
      sattPressWater=Psat01(tempFlue)

C Get %CO2 but adjusted for water vapour as well
      perCO2Adj=100.*perCO2/(fnH2O*perCO2/fnCO2+100.)

C Calculate partial pressure of water vapour after combustion
      partPressWater=(fnH2O/fnCO2)*(perCO2Adj/100.)*atmpres

C Calculate density of water vapour
      densityWaterVap=atmpres/(R_H2O*(tempAmb+273.16))

C Calculate mass of water condensed from differnce in partial pressures
      massWaterCond=(fnH2O/100.)*densityWaterVap*
     &              ((partPressWater-sattPressWater)/partPressWater)

C Calculate gross efficiency
      effGross=(effSiegert*r_netCalorific(iFuel)+L_H2O*massWaterCond)/
     &          grossCalorific(iFuel)/100.
      BOIL_EFF=effGross
      RETURN
      END

C ******************** CMP47C ********************
C CMP47C generates for plant component IPCOMP with plant db code 470 ie.
C 1 node (ISV=20) WCH ground-source heat pump feeding a hydronic heating system.
C Ground source temperature is taken from climate file data for desired depth.
C Therefore, no ground temperature "degradation" considered (see CMP48C)!
C the matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)

C     ADATA: 1 Mass of component (solids+liquid) (kg)
C            2 Mass weighted average specific heat node (J/kgK)
C            3 UA modulus for component (W/K)

C     BDATA: 1 COP model [1 - fixed COP 2 - modified Carnot efficiency; 3- f(dT) 4 - polynomial 5 - linear]
C            2 Model coef a0 [model 5: offset SH]
C            3 Model coef a1 [model 5: slope SH]
C            4 Model coef a2 [model 5: offset DHW]
C            5 Model coef a3 [model 5: slope DHW]
C            6 Device power draw (kW) model [1 - fixed 2- polynomial 3-exponential b0*e^(b1)]
C            7 Model coef b0
C            8 Model coef b1
C            9 Model coef b2
C           10 Model coef b3
C           11 Compressor pf (-)
C           12 Pump rating (W)
C           13 Pump pf (-)
C           14 Flowrate at rated pump power (l/s)
C           15 Controller power (W)
C           16 Controller pf (-)
C           17 Tout max (degC)
C           18 Tin max (degC)
C           19 Ambient temp compensation on/off (-)
C           20 Nominal water return temperature (Deg C)
C           21 Nominal water return deadband (Deg C)
C           22 Amb. air Temperature for temp compensation start [Deg C]
C           23 Amb. air Temperature for temp compensation end [Deg C]
C           24 Temp compensation gradient [deg C return/deg C ground source] (degCr/degCa)
C           25 Heat pump lock out time (min)
C           26 Ground source model [1 - standard profile 2 - user profile 3 - f(depth)]
C           27 Model coef g0 [1 & 2 - profile number 3 - depth in m]
C           28 Nominal water return temperature for DHW
C           29 Tout max DHW (degC)
C           30 Tin max DHW (degC)
C
C     CDATA: 1 ON/OFF signal for space heat (-)
C            2 ON/OFF signal for DHW (-)
C            3 Preference signal 0 space heat, 1 DHW

C Node 1 represents the condenser heat exchanger and couples to the hydronic heating circuit.
C The device has an internal pump and so an explicit pump model is not needed in system
C models containing this component. This particular device model is suitable for ON/OFF type control
C and will require modification if PID-type control is to be applied. The model internally controls the
C water outlet temperature and ambient temperature compensation can be applied [linear model]

      SUBROUTINE CMP47C(IPCOMP,COUT,ISTATS)
      use h3kmodule
      implicit none
#include "plant.h"
#include "building.h"
#include "site.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/ITERINDEX/ITERNU !plant iteration number

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      common/pstctr/nsincp,nsncpr
      integer nsincp          ! number of plant side time increments
      integer nsncpr          ! number of ??

      COMMON/PCEQU/IMPEXP,RATIMP

C     Variables for iteration in plant loop (?)
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)
      INTEGER MAXITP,ICSV,itrclp
      REAL    PERREL,PERTMP,PERFLX,PERMFL,CSVI

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)

      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      INTEGER NPCON,IPC1,IPN1,IPCT,IPC2,IPN2
      REAL PCONDR,PCONSD

      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      REAL ADATA,BDATA

      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      common/pcnam/pcname(mpcom)

C Description for additional output? ((see esrures/moplnt, line 1183))
cx      common/adopdesc/adopdsc(mpcom,MPCRES)
cx      CHARACTER adopdsc*30,


      COMMON/CLIMIP/QFPP,QFFP,TPP,TFP,QDPP,QDFP,VPP,VFP,DPP,DFP,HPP,HFP

      COMMON/GSHPvar1/CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &  COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,
     &  CompressPf,PumpRating,PumpPf,RatedFlow,
     &  CtlRating,CtlPf,ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,
     &  NomRetTSH,NomRetTDHW,NomRetTDeadB,NomRetTDeadBDHW,
     &  TempCompS,TempCompE,TempCompc0

      COMMON/GSHPvar2/COPModel,CompModel,AmbientTempComp,
     &                DeviceONOFFp,DeviceONOFF,GroundSourceModel,
     &                GroundSourceCoeff

      COMMON/GSHPvar3/CallforHeat,InDeadB

      COMMON/GSHPvar4/INSO,ITSBO,bSwitched,BLKOUT,NTSBO,bBLKOUT
      INTEGER INSO,ITSBO,NTSBO
      REAL BLKOUT
      LOGICAL bSwitched,bBLKOUT

C Electrical details for specified plant components
      common/pcelflg/ipcelf(mpcom)
      common/elpcp/NPEL,PFP(mpcom),IPFP(mpcom),PWRP(mpcom),
     &BVOLTP(mpcom),IPHP(mpcom)

      common/forlock/NSINCP_lock
      integer NSINCP_lock

      REAL SMALL
      PARAMETER (SMALL=1.0E-20)

      REAL PFP,PWRP,BVOLTP,PWRQ,PA,PQ

      INTEGER ipcelf,NPEL,IPFP,IPHP,IEMODEL

      LOGICAL CallforHeat,InDeadB

      INTEGER COPModel,CompModel,AmbientTempComp,
     &DeviceONOFFp,DeviceONOFF,GroundSourceModel,GroundSourceCoeff

      REAL CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,CompressPf,
     &PumpRating,PumpPf,RatedFlow,CtlRating,CtlPf,
     &ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,NomRetTSH,NomRetTDHW,
     &NomRetTDeadB,NomRetTDeadBDHW,TempCompS,TempCompE,TempCompc0

      REAL CompPower,CompressPower,COP,CtlPower,DeviceFlow,
     &HeatOutput,PumpPower,ReturnSP,ReturnSPH,ReturnSPL,RHamb,
     &TotApparentPower,TotReacPower,TotRealPower

      INTEGER lnblnk
      REAL SHTFLD

      INTEGER ITC,ICNT,ITCF,ITRACE,IZNTRC,ITU,ITERNU,idynow,
     &IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,IMPEXP

      INTEGER NPCDAT,IPOFS1,IPOFS2,ICON1,INOD1,napdat,
     &IPCOMP,ISTATS,I,ICONDX,ICONTP,IX1,NCI,NITMS,NPCOMP,
     &INOD2,ICON2

      REAL TIMSEC,TC,RATIMP,CSVF,CSVP,PCTF,PCRF,PUAF,PCQF,PCNTMF,
     &PCTP,PCRP,PUAP,PCQP,PCNTMP,CONVAR,QFPP,QFFP,TPP,TFP,QDPP,QDFP,VPP

      REAL VFP,DPP,DFP,HPP,HFP,Tamb,ReturnT,Td,Ta,TL,TH,ALPHA,CM,
     &      C1,TmpDev,futureTemp,presentTemp,Tgrnd,GroundHeatInput

      REAL COUT(MPCOE),QDATA,PCAOUT,CDATA

      REAL TG1,TG2,GRNDSourceDepth,LimFac,CfH_out,ToutMax
      INTEGER IM,ID,curDoM,curMonth,NDMonth

      DIMENSION NDMonth(12)
      DATA NDMonth/31,28,31,30,31,30,31,31,30,31,30,31/

      LOGICAL CLOSE,CLOSEA,bSHrqd,bDHWrqd,bSHpref

      CHARACTER OUTS*248,PCNAME*15

      INTEGER IPCOMP_LEN !lenght of ipcomp name

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP47C'

C ************
C Variable Assignment and initialisation mainly in CMP47S (pcomps.F)

      if (NSINC.eq.1) then
C NTSBO = total no. of timesteps HP is off due to lockout
        NTSBO=NINT(BLKOUT/TIMSEC)
      endif

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

C COP based on inlet temperatures so mark 1st nodal temp
C for iteration.
cx      ICSV(INOD1,1)=1
cx      CSVI(INOD1,1)=CSVF(INOD1,1)

      UAMod=ADATA(ipcomp,3)
      call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
      IF(closea) UAMod=0.

C Read ambient temperature (for temperature compensation of WCH return temp)
      Tamb=TFP

C Set initial operation mode based on CDATA.
      call eclose(CDATA(IPCOMP,1),1.0,0.001,bSHrqd)
      call eclose(CDATA(IPCOMP,2),1.0,0.001,bDHWrqd)
      call eclose(CDATA(IPCOMP,3),0.0,0.001,bSHpref)

C Check which mode is actually to be used for the case when both demands
C are set to .true.
      if ((bSHrqd.and.bDHWrqd)) then
C       Both are .true.
        if (bSHpref) then
C         Re-set DHW to false, as SH is preferred !
          bDHWrqd=.false.
        else
C         Re-set SH to false, as DHW is preferred !
          bSHrqd=.false.
        endif
      endif

      if ((bSHrqd.or.bDHWrqd)) then
C       Either space heat or dhw needs heat
        CallForHeat=.true.

C Pump circulates while there is a call for heat.
        PumpPower=PumpRating
        DeviceFlow=RatedFlow
      ELSE
        CallforHeat=.false.
C       Only turn circulation pump off when no call for heat!
C       Careful: DeviceFlow is used in ISTAT.eq.2 !
        PumpPower=0.
        DeviceFlow=0.0
      ENDIF

C Set return setpoints based on which function requires heat
      if (bDHWrqd) then
        ToutMax=ToutMaxDHW
        ReturnSP=NomRetTDHW !default return temperature DHW mode
        ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
        ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)

        IF(ReturnSP.GT.TinMaxDHW) ReturnSP=TinMaxDHW

C       Ambient temperature compensation not used for DHW mode!

      else ! "default"
        ToutMax=ToutMaxSH
        ReturnSP=NomRetTSH   !default return temperature SH mode
        ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
        ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)

        IF(ReturnSP.GT.TinMaxSH) ReturnSP=TinMaxSH

        IF(AmbientTempComp.GT.0)THEN
C Calculate the return water temperature set point if temperature compensation is active.

          IF(Tamb.GE.TempCompS.AND.Tamb.LT.TempCompE)THEN
            ReturnSP=((Tamb-TempCompS)*TempCompc0)+NomRetTSH
            ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
            ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)

C Case where Tamb > Temp compensation saturation point
          ELSEIF(Tamb.GE.TempCompE)THEN
            ReturnSP=((TempCompE-TempCompS)*TempCompc0)+NomRetTSH
            ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
            ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)
          ENDIF

          IF(ReturnSPH.GT.TinMaxSH) ReturnSPH=TinMaxSH

          IF(ReturnSP.GT.TinMaxSH)THEN
            ReturnSP=TinMaxSH
cx            ReturnSPH=ReturnSP
          ENDIF
        ENDIF
      endif ! space heat rq'd

C     Set ground source temperature based on model chosen.
      if (GroundSourceModel.eq.1 .or. GroundSourceModel.eq.2) then

C     Standard ground temperature profile from grdtmp(12,mgrdp), see esru_blk.F,
C     or user defined ground temperature profile from ugrdtp(12,mgrdp).

        if (GroundSourceModel.eq.2) then
          if (NGRDP.le.0) then
            write(*,*) "Error GSHP: Option 2 requires user defined",
     &             " ground temperature profile! "
            STOP
          elseif (GroundSourceCoeff.gt.NGRDP) then
            write(*,*) "Error GSHP: Ground temperature profile #",
     &             GroundSourceCoeff," not available! "
            STOP
          endif
        endif
C       Calculate the current ground temperature by interpolating the
C       monthly ground temperatures. ((code from CMP79C(), lines 8976 ff.))

C       Get current month temperature
        CALL EDAYR(IDYP,ID,IM)
        curMonth=IM
        curDoM=ID

        if (GroundSourceModel.eq.1) then
          TG1=grdtmp(curMonth,int(GroundSourceCoeff))
        else
          TG1=ugrdtp(curMonth,int(GroundSourceCoeff))
        endif

        IF(curMonth.EQ.12)THEN
C         Get next month's temperature (assume loop round to January).
          curMonth=1
        ELSE
C         Get next month's temperature
          curMonth=curMonth+1
        ENDIF

        if (GroundSourceModel.eq.1) then
          TG2=grdtmp(curMonth,int(GroundSourceCoeff))
        else
          TG2=ugrdtp(curMonth,int(GroundSourceCoeff))
        endif

C       Calculate today's ground temperature by linear interpolation
        Tgrnd=((float(curDoM)/float(NDMonth(curMonth)))
     &                                              *(TG2-TG1))+TG1

cx          write(*,*)"Tgrnd=",Tgrnd,",  Tamb=",Tamb

      elseif (GroundSourceModel.eq.3) then
C       Calculate ground source temperature in desired depth GroundSourceCoeff
cx        CALL GTCALC(GroundSourceCoeff,'-',IER)
cx        Tgrnd=??
        write(*,*) "Error GSHP: Option 3 for ground temperature",
     &               " not available to date! "
        STOP
      else
C       Ground source temperature option invalid
        write(*,*) "Error GSHP: Option for ground temperature",
     &               " invalid! "
        STOP
      endif

C Determine whether machine should be on or off based on return
C water temperature.
      ReturnT=CONVAR(ICON1,1)

C Check if device switched off last TS
      IF ((DeviceONOFFp.eq.1).and.(DeviceONOFF.eq.0)) THEN
        bSwitched=.true.
      ENDIF

      DeviceONOFFp=DeviceONOFF

      IF(CallforHeat)THEN

C Check to see if the device is in the dead band.
        IF(ReturnT.GE.ReturnSPL.AND.ReturnT.LE.ReturnSPH)THEN
          InDeadB=.true.
        ELSE
          InDeadB=.false.
        ENDIF

C Evaluate the device status.
        IF(ReturnT.GT.ReturnSPH)THEN
          DeviceONOFF=0
        ELSEIF(ReturnT.LT.ReturnSPL)THEN
          DeviceONOFF=1
        ELSEIF(DeviceONOFFp.EQ.1.AND.InDeadB)THEN
          DeviceONOFF=1
        ELSEIF(DeviceONOFFp.EQ.0.AND.InDeadB)THEN
          DeviceONOFF=0
        ENDIF
      ELSE
        DeviceONOFF=0
      ENDIF ! call for heat

C Finally, check if device is in lockout time, i.e. last "off" was
C less than lockouttime ago ... ((code from CMP42C(), line 5102 ff.))
C Lock out time (in seconds), Heat pump should not be turned on/off
C with high frequency. When HP is turned off, wait this amount of time
C before turning on again.
C INSO = HP on/off (1/0) flag for lockout
C ITSBO = present no. of timesteps HP is off due to lockout
C NTSBO = total no. of timesteps HP is off due to lockout

      IF (bSwitched) THEN
        if (bBLKOUT) then
          NSINCP_lock=NSINCP
          INSO=1
          ITSBO=0
          bSwitched=.false. ! only set once after signal
        endif
      ENDIF

      IF (INSO.EQ.1) THEN
        IF (ITSBO.GT.NTSBO) INSO=0
        if (NSINCP_lock.lt.NSINCP) then
          ITSBO=ITSBO+1
          NSINCP_lock=NSINCP
        endif
        DeviceONOFF=0
      ENDIF
C == end lockout

      IF (DeviceONOFF.GT.0) THEN

C Calculate the TOTAL device power draw (W)
        IF(CompModel.eq.1)THEN
          CompPower=1000.*(Compa0) !fixed power consumption
        ELSEIF(CompModel.eq.2)THEN
          Ta=Tgrnd
          Td=ToutMax
          CompPower=1000.*(Compa0+(Compa1*Ta)+(Compa2*Td))
        ELSEIF(CompModel.eq.3)THEN
cx          Td=min(ReturnT,ToutMax)
          Ta=Tgrnd
          Td=ToutMax
          CompPower=1000.*(Compa0*EXP(Compa1*(Td-Ta)))
cx     &                 +Compa2*(ToutMax-(BDATA(IPCOMP,17))))!adapted exponential power consumption.
        ENDIF
        if(CompPower.LT.0.0)CompPower=0.0

C Calculate the COP based on the user-specified method.
        IF(COPModel.EQ.1)THEN
          COP=COPa0 !fixed COP
        ELSEIF(COPModel.EQ.2)THEN
          TL=Tgrnd+273.15
          TH=ToutMax+273.15
          COP=CarEffMod*((1-(TL/TH))**(-1)) !modified carnot COP
        ELSEIF(COPModel.EQ.3)THEN !quadratic based on Td-Tgrnd
          TL=Tgrnd
          TH=ToutMax
cx          TH=ReturnT
          COP=min(COPa0+(COPa1*(TH-TL))+(COPa2*(TH-TL)**2.),
     &             COPa0+(COPa1*(ToutMax-TL))
     &                             +(COPa2*(ToutMax-TL)**2.))
        ELSEIF(COPModel.EQ.4)THEN
          COP=COPa0+COPa1*Tgrnd+COPa2*Tgrnd**2.+COPa3*Tgrnd**3. !cubic polymonial COP based on ground T
        ELSEIF(COPModel.EQ.5)THEN
          TL=Tgrnd
          TH=ToutMax
          if (bSHrqd) then
            COP=COPa0+COPa1*(TH-TL)
          else
            COP=COPa2+COPa3*(TH-TL)
          endif
        ELSE
          WRITE(OUTS,'(a)')'Error in 1-node GSHP, COP model # '
          CALL EDISP(IUOUT,OUTS)
        ENDIF

C Set limits on calculated values of COP
        IF(COP.LT.0.0) COP=0.0
        if (bSHrqd) then
C         This probably should either not be necessary or a input variable!
          IF(COP.GT.9.0) COP=9.0
        elseif (bDHWrqd) then
C         This probably should either not be necessary or a input variable!
          IF(COP.GT.3.5) COP=3.5
        endif

cx "after line 7840 ... " ((2node- GSHP, which file version??))
C Limit output temperature to Toutmax ((suggestion by Graeme))
        IF (CSVF(INOD1,1).GE.ToutMax) then
cx            LimFac=((ToutMax-CONVAR(ICON1,1))
cx     &          / (CSVF(INOD1,1)-CONVAR(ICON1,1)))
cx            CompPower = LimFac*CompPower
        END IF
C===
        HeatOutput=COP*CompPower
        GroundHeatInput=-HeatOutput+CompPower*CompressPf

        IF(HeatOutput.LT.0) THEN
          write(*,*) "Error-7374: heat output < 0! ",
     &                 HeatOutput, COP, CompPower
          STOP
        ENDIF

        IF(GroundHeatInput.GT.0) THEN
          write(*,*) "Error-7380: ground heat input > 0! ",
     &                 GroundHeatInput, HeatOutput, COP, CompPower
          STOP
        ENDIF

      ELSE
C       Device is 'off' (DeviceONOFF.eq.0), i.e. the heat pump is off,
C       however the circulation pump remains on. Set parameters
C       accordingly (assume controller is still operating).

        CompPower=0.
        HeatOutput=0.
        GroundHeatInput=0.
        COP=0.
      ENDIF

C Generate coefficients for energy balance equation
      IF (ISTATS.EQ.1) THEN

C Establish the nodal thermal capacities (see CMP36C).
        CM=CompMass*AveSpHt

C Node 1 water
        C1=DeviceFlow*SHTFLD(3,CONVAR(ICON1,1))

C Set heat loss to zero if no containment defined
        CALL ECLOSE(PCNTMF(IPCOMP),99.,0.0001,CLOSE)
        IF(CLOSE) UAMod=0.0

C Calculate the electrical demand of the device assumung the controller is ON.
        CtlPower=CtlRating
        TotRealPower=CompPower+CtlPower+PumpPower

C Save the real power draw for use in an electrical power flow simulation. 
        PWRP(IPCOMP)=-ABS(TotRealPower)
        IEMODEL=1
        CALL EMACH(IPCOMP,IEMODEL,PWRP(IPCOMP),PQ,PA)
        PWRQ=PQ

c Calculate the compressor power draw.
        CompressPower=CompPower-(PumpPower+CtlPower)
        IF(CompressPower.LT.0.0)CompressPower=0.0

        TotReacPower=( ((CompressPower/CompressPf)**2)
     &                                        -(CompressPower**2) )**0.5
     &              +( ((PumpPower/PumpPf)**2)-(PumpPower**2) )**0.5
     &              +( ((CtlPower/CtlPf)**2)  -(CtlPower**2) )**0.5

        TotApparentPower=((TotRealPower**2)+(TotReacPower**2))**0.5

C Calculate current component time-constant TC
        TC(IPCOMP)=CM/AMAX1(SMALL,(C1+UAMod))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
           ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
           ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
             ALPHA=1.
          ELSE
             ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
        END IF

C Establish matrix equation self- and cross-coupling coefficients.
        COUT(1)=ALPHA*(-C1-UAMod)-CM/TIMSEC
C Matrix cross coupling coefficients.
        COUT(2)=ALPHA*C1
C Establish the present and known coefficient i.e. RHS
        COUT(3)=((1.-ALPHA)*(PCRP(ICON1)+PUAP(INOD1))-CM/TIMSEC)
     &          *CSVP(INOD1,1)
     &          +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &          -ALPHA*UAMod*PCNTMF(IPCOMP)
     &          -ALPHA*HeatOutput
     &          -(1.-ALPHA)*PCQP(INOD1)
     &          -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)

C Establish "containment loss" data
        presentTemp=csvp(inod1,1)
        futureTemp =csvf(inod1,1)
        QDATA(IPCOMP)=UAMod*(alpha*(futureTemp-pcntmp(ipcomp))+
     &                (1.-alpha)*(presentTemp-pcntmp(ipcomp)))

        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iConvective)
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iRadiant)

C Store "environment" variables future values
        PCTF(ICON1)=CONVAR(ICON1,1)
        PCRF(ICON1)=C1
        PCQF(INOD1)=HeatOutput

C Addition output for results analysis.
        NAPDAT(IPCOMP)=9
        PCAOUT(IPCOMP,1)=HeatOutput
        PCAOUT(IPCOMP,2)=COP
        PCAOUT(IPCOMP,3)=Tgrnd
        if (DeviceONOFF.eq.1) then
          if (bDHWrqd) then
           PCAOUT(IPCOMP,4)=2.
          else
           PCAOUT(IPCOMP,4)=1.
          endif
        else
          PCAOUT(IPCOMP,4)=0.
        endif
        PCAOUT(IPCOMP,5)=ReturnSP
        PCAOUT(IPCOMP,6)=TotRealPower
        PCAOUT(IPCOMP,7)=TotReacPower
        PCAOUT(IPCOMP,8)=TotApparentPower
        PCAOUT(IPCOMP,9)=GroundHeatInput

C---------------------------------------------------------------------------------
C Make select results available in XML and CVS output.
C---------------------------------------------------------------------------------
        IPCOMP_LEN = lnblnk(pcname(IPCOMP))

        if (bSHrqd) then
          CfH_out = 1.
        elseif (bDHWrqd) then
          CfH_out = 2.
        else
          CfH_out = 0.
        endif

        call AddToReport(rvPltCallForHeat%Identifier,
     &         CfH_out,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltHOut%Identifier,
     &         HeatOutput,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltCOP%Identifier,
     &         COP,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltTambient%Identifier,
     &         Tgrnd,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        tmpdev=float(DeviceONOFF)
        call AddToReport(rvPltDeviceONOFF%Identifier,
     &         tmpdev,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltReturnTSP%Identifier,
     &         ReturnSP,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltAmbientHeat%Identifier,
     &         GroundHeatInput,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltRealPow%Identifier,
     &         TotRealPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltReacPow%Identifier,
     &         TotReacPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltApparPow%Identifier,
     &         TotApparentPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))


C 1st phase mass (ie. "water") balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=DeviceFlow

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
      END IF

C Trace.
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) Ground Source Heat Pump'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM      = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1      = ',C1,' (W/Ks)'
            WRITE(ITU,*) ' Flow      = ',DeviceFlow,' (l/s)'
            WRITE(ITU,*) ' TC      = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA   = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UAMod   = ',UAMod,' (W/K)'
            WRITE(ITU,*) ' Tground = ',Tgrnd,' (C)'
            WRITE(ITU,*) ' tBlkout = ',BDATA(IPCOMP,26),' (min)'
            WRITE(ITU,*) ' ReturnT = ',ReturnT,' (C)'
            WRITE(ITU,*) ' ReturnSP        = ',ReturnSP,' (C)'
            WRITE(ITU,*) ' ReturnSPH       = ',ReturnSPH,' (C)'
            WRITE(ITU,*) ' ReturnSPL       = ',ReturnSPL,' (C)'
            WRITE(ITU,*) ' InDeadB         = ',InDeadB,' (-)'
            WRITE(ITU,*) ' CallforHeat     = ',CallforHeat,' (-)'
            WRITE(ITU,*) ' DeviceONOFF     = ',DeviceONOFF,' (-)'
            WRITE(ITU,*) ' DeviceONOFFp    = ',DeviceONOFFp,' (-)'
            WRITE(ITU,*) ' COP             = ',COP,' (-)'
            WRITE(ITU,*) ' AmbientTempComp = ',AmbientTempComp,' (C)'
            WRITE(ITU,*) ' TempCompS       = ',TempCompS,' (C)'
            WRITE(ITU,*) ' TempCompE       = ',TempCompE,' (C)'
            WRITE(ITU,*) ' HeatOutput      = ',HeatOutput,' (W)'
            WRITE(ITU,*) ' GroundHeatInput = ',GroundHeatInput,' (W)'
            WRITE(ITU,*) ' CompPower       = ',CompPower,' (W)'
            WRITE(ITU,*) ' PumpPower       = ',PumpPower,' (W)'
            WRITE(ITU,*) ' CtlPower        = ',CtlPower,' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP47C'

      RETURN
      END


C ******************** CMP48C ********************
C CMP48C generates for plant component IPCOMP with plant db code 480 ie.
C 2 node (ISV=20) WCH ground-source heat pump feeding a hydronic heating system.
C the matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)

C     ADATA: 1 Mass of component (solids+liquid) (kg)
C            2 Mass weighted average specific heat node (J/kgK)
C            3 UA modulus for component (W/K)

C     BDATA: 1 COP model [1 - fixed COP 2 - modified Carnot efficiency; 3- f(dT) 4 - polynomial]
C            2 Model coef a0
C            3 Model coef a1
C            4 Model coef a2
C            5 Model coef a3
C            6 Device power draw (kW) model [1 - fixed 2- polynomial 3-exponential b0*e^(b1)]
C            7 Model coef b0
C            8 Model coef b1
C            9 Model coef b2
C           10 Model coef b3
C           11 Compressor pf (-)
C           12 Pump rating (W)
C           13 Pump pf (-)
C           14 Flowrate at rated pump power (l/s)
C           15 Controller power (W)
C           16 Controller pf (-)
C           17 Tout max SH (degC)
C           18 Tin max SH (degC)
C           19 Ambient temp compensation on/off (-)
C           20 Nominal water return temperature for SH (Deg C)
C           21 Nominal water return deadband (Deg C)
C           22 Ambient temperature for temp compensation start [Deg C]
C           23 Ambient temperature for temp compensation end [Deg C]
C           24 Temp compensation gradient [deg C return/deg C ambient] (degCr/degCa)
C           25 Heat pump lock out time (min)
C           26 BrinePump rating (W)
C           27 BrinePump pf (-)
C           28 BrineFlowrate at rated brine pump power (l/s)
C           29 Nominal water return temperature for DHW
C           30 Tout max DHW (degC)
C           31 Tin max DHW (degC)
C
C     CDATA: 1 ON/OFF signal for space heat (-)
C            2 ON/OFF signal for DHW (-)
C            3 Preference signal 0 space heat, 1 DHW

C Node 1 represents the condenser heat exchanger and couples to the hydronic heating circuit.
C Node 2 represents the evaporator heat exchanger and couples to a (ground) heat source.
C The device has an internal pump and so an explicit pump model is not needed in system
C models containing this component. This particular device model is suitable for ON/OFF type control
C and will require modification if PID-type control is to be applied. The model internally controls the
C water outlet temperature and ground source temperature compensation can be applied [linear model]

      SUBROUTINE CMP48C(IPCOMP,COUT,ISTATS)
      use h3kmodule
      implicit none
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/ITERINDEX/ITERNU !plant iteration number

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      common/pstctr/nsincp,nsncpr
      integer nsincp          ! number of plant side time increments
      integer nsncpr          ! number of ??

      COMMON/PCEQU/IMPEXP,RATIMP

C     Variables for iteration in plant loop (?)
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)
      INTEGER MAXITP,ICSV,itrclp
      REAL    PERREL,PERTMP,PERFLX,PERMFL,CSVI

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)

      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      INTEGER NPCON,IPC1,IPN1,IPCT,IPC2,IPN2
      REAL PCONDR,PCONSD

      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      REAL ADATA,BDATA

      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      common/pcnam/pcname(mpcom)

      COMMON/CLIMIP/QFPP,QFFP,TPP,TFP,QDPP,QDFP,VPP,VFP,DPP,DFP,HPP,HFP

      COMMON/GSHPvar1/CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &  COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,
     &  CompressPf,PumpRating,PumpPf,RatedFlow,
     &  CtlRating,CtlPf,ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,
     &  NomRetTSH,NomRetTDHW,NomRetTDeadB,NomRetTDeadBDHW,
     &  TempCompS,TempCompE,TempCompc0

      COMMON/GSHPvar2/COPModel,CompModel,AmbientTempComp,
     &                DeviceONOFFp,DeviceONOFF,GroundSourceModel,
     &                GroundSourceCoeff

      COMMON/GSHPvar3/CallforHeat,InDeadB

      COMMON/GSHPvar4/INSO,ITSBO,bSwitched,BLKOUT,NTSBO,bBLKOUT
      INTEGER INSO,ITSBO,NTSBO
      REAL BLKOUT
      LOGICAL bSwitched,bBLKOUT

C Electrical details for specified plant components
      common/pcelflg/ipcelf(mpcom)
      common/elpcp/NPEL,PFP(mpcom),IPFP(mpcom),PWRP(mpcom),
     &BVOLTP(mpcom),IPHP(mpcom)

      common/forlock/NSINCP_lock
      integer NSINCP_lock

      REAL SMALL
      PARAMETER (SMALL=1.0E-20)

      REAL PFP,PWRP,BVOLTP,PWRQ,PA,PQ

      INTEGER ipcelf,NPEL,IPFP,IPHP,IEMODEL

      LOGICAL CallforHeat,InDeadB

      INTEGER COPModel,CompModel,AmbientTempComp,
     &DeviceONOFFp,DeviceONOFF,GroundSourceModel,GroundSourceCoeff

      REAL CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,CompressPf,
     &PumpRating,PumpPf,RatedFlow,CtlRating,CtlPf,
     &ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,NomRetTSH,NomRetTDHW,
     &NomRetTDeadB,NomRetTDeadBDHW,TempCompS,TempCompE,TempCompc0

      REAL CompPower,CompressPower,COP,CtlPower,DeviceFlow,
     &HeatOutput,PumpPower,ReturnSP,ReturnSPH,ReturnSPL,RHamb,
     &TotApparentPower,TotReacPower,TotRealPower

      INTEGER lnblnk
      REAL SHTFLD

      INTEGER ITC,ICNT,ITCF,ITRACE,IZNTRC,ITU,ITERNU,idynow,
     &IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,IMPEXP

      INTEGER NPCDAT,IPOFS1,IPOFS2,ICON1,INOD1,napdat,
     &IPCOMP,ISTATS,I,ICONDX,ICONTP,IX1,NCI,NITMS,NPCOMP,
     &INOD2,ICON2

      REAL TIMSEC,TC,RATIMP,CSVF,CSVP,PCTF,PCRF,PUAF,PCQF,PCNTMF,
     &PCTP,PCRP,PUAP,PCQP,PCNTMP,CONVAR,QFPP,QFFP,TPP,TFP,QDPP,QDFP,VPP

      REAL VFP,DPP,DFP,HPP,HFP,Tamb,ReturnT,Td,Ta,TL,TH,ALPHA,CM1,CM2,
     &      C1,TmpDev,C2,futureTemp,presentTemp,Tgrnd,GroundHeatInput

      REAL COUT(MPCOE),QDATA,PCAOUT,CDATA

      REAL BrinePumpRating,BrineRatedFlow,BrinePumpPower,
     &      BrinePumpPf,BrineFlow,LimFac,CfH_out,ToutMax

      LOGICAL CLOSE,CLOSEA,bSHrqd,bDHWrqd,bSHpref

      CHARACTER OUTS*248,PCNAME*15

      INTEGER IPCOMP_LEN !lenght of ipcomp

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP48C'

C ************
C Variable Assignment and initialisation mainly in CMP48S (pcomps.F)

      if (NSINC.eq.1) then
C NTSBO = total no. of timesteps HP is off due to lockout
        NTSBO=NINT(BLKOUT/TIMSEC)
      endif

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      ICON2=ICONDX(IPCOMP,2,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C COP based on inlet temperatures so mark 1st and 2nd nodal temps
C for iteration.
cx      ICSV(INOD1,1)=1
cx      CSVI(INOD1,1)=CSVF(INOD1,1)
cx      ICSV(INOD2,1)=1
cx      CSVI(INOD2,1)=CSVF(INOD2,1)

C Initialize variables that are not (yet?) covered in CMP48S()
      BrinePumpRating=BDATA(IPCOMP,26)
      BrinePumpPf=BDATA(IPCOMP,27)
      BrineRatedFlow=BDATA(IPCOMP,28)

      call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
      IF(closea) UAMod=0.

C Read ambient temperature (for temperature compensation of WCH return temp)
      Tamb=TFP

C Set initial operation mode based on CDATA.
      call eclose(CDATA(IPCOMP,1),1.0,0.001,bSHrqd)
      call eclose(CDATA(IPCOMP,2),1.0,0.001,bDHWrqd)
      call eclose(CDATA(IPCOMP,3),0.0,0.001,bSHpref)

C Check which mode is actually to be used for the case when both demands
C are set to .true.
      if ((bSHrqd.and.bDHWrqd)) then
C       Both are .true.
        if (bSHpref) then
C         Re-set DHW to false, as SH is preferred !
          bDHWrqd=.false.
        else
C         Re-set SH to false, as DHW is preferred !
          bSHrqd=.false.
        endif
      endif

      if ((bSHrqd.or.bDHWrqd)) then
C       Either space heat or dhw needs heat
        CallForHeat=.true.

C Pump circulates while there is a call for heat.
        PumpPower=PumpRating
        DeviceFlow=RatedFlow
        BrinePumpPower=BrinePumpRating
        BrineFlow=BrineRatedFlow
      ELSE
        CallforHeat=.false.
        PumpPower=0.
        DeviceFlow=0.0
        BrinePumpPower=0.
        BrineFlow=0.0
      ENDIF
        BrineFlow=BrineRatedFlow

C Set return setpoints based on which function requires heat
      if (bDHWrqd) then
        ToutMax=ToutMaxDHW
        ReturnSP=NomRetTDHW !default return temperature DHW mode
        ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
        ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)

        IF(ReturnSP.GT.TinMaxDHW) ReturnSP=TinMaxDHW

C       Ambient temperature compensation not used for DHW mode!

      else ! "default"
        ToutMax=ToutMaxSH
        ReturnSP=NomRetTSH !default return temperature
        ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
        ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)

        IF(ReturnSP.GT.TinMaxSH) ReturnSP=TinMaxSH

        IF(AmbientTempComp.GT.0)THEN
C Calculate the return water temperature set point if temperature compensation is active.

          IF(Tamb.GE.TempCompS.AND.Tamb.LT.TempCompE)THEN
            ReturnSP=((Tamb-TempCompS)*TempCompc0)+NomRetTSH
            ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
            ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)

C Case where Tamb > Temp compensation end point
          ELSEIF(Tamb.GE.TempCompE)THEN
            ReturnSP=((TempCompE-TempCompS)*TempCompc0)+NomRetTSH
            ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
            ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)
          ENDIF

          IF(ReturnSPH.GT.TinMaxSH) ReturnSPH=TinMaxSH

          IF(ReturnSP.GT.TinMaxSH)THEN
            ReturnSP=TinMaxSH
cx            ReturnSPH=ReturnSP
          ENDIF
        ENDIF
      endif ! space heat rq'd

C Get ground source temperature
      Tgrnd=convar(icon2,1)

C Determine whether machine should be on or off based on return water temperature.
      ReturnT=CONVAR(ICON1,1)

C Check if device switched off last TS
      IF ((DeviceONOFFp.eq.1).and.(DeviceONOFF.eq.0)) THEN
        bSwitched=.true.
      ENDIF

      DeviceONOFFp=DeviceONOFF

      IF(CallforHeat)THEN

C Check to see if the device is in the dead band.
        IF(ReturnT.GE.ReturnSPL.AND.ReturnT.LE.ReturnSPH)THEN
          InDeadB=.true.
        ELSE
          InDeadB=.false.
        ENDIF

C Evaluate the device status.
        IF(ReturnT.GT.ReturnSPH)THEN
          DeviceONOFF=0
        ELSEIF(ReturnT.LT.ReturnSPL)THEN
          DeviceONOFF=1
        ELSEIF(DeviceONOFFp.EQ.1.AND.InDeadB)THEN
          DeviceONOFF=1
        ELSEIF(DeviceONOFFp.EQ.0.AND.InDeadB)THEN
          DeviceONOFF=0
        ENDIF
      ELSE
        DeviceONOFF=0
      ENDIF ! call for heat

C Finally, check if device is in lockout time, i.e. last "off" was
C less than lockouttime ago ... ((code from CMP42C(), line 5102 ff.))
C Lock out time (in seconds), Heat pump should not be turned on/off
C with high frequency. When HP is turned off, wait this amount of time
C before turning on again.
C NSINCP = plant time step increment counter
C INSO   = HP on/off (1/0) flag for lockout
C ITSBO  = present no. of timesteps HP is off due to lockout
C NTSBO  = total no. of timesteps HP is off due to lockout

      IF (bSwitched) THEN
        if (bBLKOUT) then
          NSINCP_lock=NSINCP
          INSO=1
          ITSBO=0
          bSwitched=.false. ! only set once after signal
        endif
      ENDIF

      IF (INSO.EQ.1) THEN
        IF (ITSBO.GT.NTSBO) INSO=0
        if (NSINCP_lock.lt.NSINCP) then
          ITSBO=ITSBO+1
          NSINCP_lock=NSINCP
        endif
        DeviceONOFF=0
      ENDIF
C == end lockout

      IF (DeviceONOFF.GT.0) THEN

C Calculate the TOTAL device power draw (W)
        IF(CompModel.eq.1)THEN
          CompPower=1000.*(Compa0) !fixed power consumption
        ELSEIF(CompModel.eq.2)THEN
          Td=ToutMax
          Ta=Tgrnd
          CompPower=1000.*(Compa0+(Compa1*Ta)+(Compa2*Td))
        ELSEIF(CompModel.eq.3)THEN
          Td=ToutMax
          Ta=Tgrnd
          CompPower=1000.*(Compa0*EXP(Compa1*(Td-Ta)))
cx     &                 +Compa2*(ToutMax-(BDATA(IPCOMP,17))))!adapted exponential power consumption.
        ENDIF

        if(CompPower.LT.0.0) CompPower=0.0

C Calculate the COP based on the user-specified method.
        IF(COPModel.EQ.1)THEN
          COP=COPa0 !fixed COP
        ELSEIF(COPModel.EQ.2)THEN
          TL=Tgrnd+273.15
          TH=ToutMax+273.15
          COP=CarEffMod*((1-(TL/TH))**(-1)) !modified carnot COP
        ELSEIF(COPModel.EQ.3)THEN !quadratic based on Td-Ta
          TL=Tgrnd
cx          TH=ReturnT
          TH=ToutMax
          COP=min(COPa0+(COPa1*(TH-TL))+(COPa2*(TH-TL)**2.),
     &               COPa0+(COPa1*(ToutMax-TL))
     &                               +(COPa2*(ToutMax-TL)**2.))
        ELSEIF(COPModel.EQ.4)THEN
          COP=COPa0+COPa1*Tgrnd+COPa2*Tgrnd**2.+COPa3*Tgrnd**3. !cubic polymonial COP based on ground T
        ELSE
          WRITE(OUTS,'(a)')'Error in 2node-GSHP, COP model # '
          CALL EDISP(IUOUT,OUTS)
        ENDIF

C Set limits on calculated values of COP
        IF(COP.LT.0.0) COP=0.0
        if (bSHrqd) then
C         This probably should either not be necessary or a input variable!
          IF(COP.GT.9.0) COP=9.0
        elseif (bDHWrqd) then
C         This probably should either not be necessary or a input variable!
          IF(COP.GT.3.5) COP=3.5
        endif

C Limit output temperature to Toutmax ((suggestion by Graeme))
        IF(CSVF(INOD1,1).GE.ToutMax)then
cx            LimFac=((Toutmax-CONVAR(ICON1,1))
cx     &          /(CSVF(INOD1,1)-CONVAR(ICON1,1)))
cx            CompPower = LimFac*CompPower
        END IF
C===
        HeatOutput=COP*CompPower
        GroundHeatInput=-HeatOutput+CompPower*CompressPf

        IF(HeatOutput.LT.0) THEN
          write(*,*) "Error-7979: heat output < 0! ",HeatOutput, COP,
     &                                                     CompPower
          STOP
        ENDIF

        IF(GroundHeatInput.GT.0) THEN
          write(*,*) "Error-7985: ground heat input > 0! ",
     &                   GroundHeatInput,HeatOutput, COP, CompPower
          STOP
        ENDIF

      ELSE
C Device is 'off', i.e. heat pump is off, circulation pump is still on
C but brine pump is off.
C Set parameters accordingly (assume controller is still operating).
        CompPower=0.
        HeatOutput=0.
        GroundHeatInput=0.
        BrinePumpPower=0.
        BrineFlow=0.0
        COP=0.
      ENDIF

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Establish the nodal thermal capacities (see CMP36C).
        CM1=0.5*CompMass*AveSpHt
        CM2=CM1

C Node 1 water
        C1=DeviceFlow*SHTFLD(3,CONVAR(ICON1,1))

C Node 2 water
cx        C2=PCONDR(ICON2)*CONVAR(ICON2,2)*SHTFLD(3,CONVAR(ICON2,1))
        C2=BrineFlow*SHTFLD(3,CONVAR(ICON2,1))

C Set heat loss to zero if no containment defined
        CALL ECLOSE(PCNTMF(IPCOMP),99.,0.0001,CLOSE)
        IF(CLOSE) UAMod=0.0

C Calculate the electrical demand of the device assumung the controller is ON.
        CtlPower=CtlRating
        TotRealPower=CompPower+CtlPower+PumpPower+BrinePumpPower

C Save the real power draw for use in an electrical power flow simulation. 
        PWRP(IPCOMP)=-ABS(TotRealPower)
        IEMODEL=1
        CALL EMACH(IPCOMP,IEMODEL,PWRP(IPCOMP),PQ,PA)
        PWRQ=PQ

c Calculate the compressor power draw.
        CompressPower=CompPower-(PumpPower+CtlPower+BrinePumpPower)
        IF(CompressPower.LT.0.0)CompressPower=0.0

        TotReacPower=( ((CompressPower/CompressPf)**2)
     &                                        -(CompressPower**2) )**0.5
     &              +( ((PumpPower/PumpPf)**2)-(PumpPower**2) )**0.5
     &              +( ((CtlPower/CtlPf)**2)  -(CtlPower**2) )**0.5
     &              +( ((BrinePumpPower/BrinePumpPf)**2)
     &                                       -(BrinePumpPower**2) )**0.5

        TotApparentPower=((TotRealPower**2)+(TotReacPower**2))**0.5

C Calculate current component time-constant TC
        TC(IPCOMP)=AMAX1(
     &              CM1/AMAX1(SMALL,(C1+UAMod)),
     &              CM2/AMAX1(SMALL,(C2+UAMod)))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
           ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
           ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
             ALPHA=1.
          ELSE
             ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
            CM1=0.
            CM2=0.
            ALPHA=1.
        END IF

C==== see CMP36C() line 4135 ff. above ... ====
C Establish matrix equation self- and cross-coupling coefficients.
        COUT(1)=ALPHA*(-C1)-CM1/TIMSEC
        COUT(2)=ALPHA*(-C2-UAMod)-CM2/TIMSEC
C Matrix cross coupling coefficients.
        COUT(3)=ALPHA*C1
        COUT(4)=ALPHA*C2
C Establish the present and known coefficient i.e. RHS
        COUT(5)=((1.-ALPHA)*(PCRP(ICON1))-CM1/TIMSEC)
     &          *CSVP(INOD1,1)
     &          -(1.-ALPHA)*PCRP(ICON1)*PCTP(ICON1)
     &          -ALPHA*HeatOutput
     &          -(1.-ALPHA)*PCQP(INOD1)

         COUT(6)=((1.-ALPHA)*(PCRP(ICON2)+UAMod)-CM2/TIMSEC)
     &           *CSVP(INOD2,1)
     &          -(1.-ALPHA)*PCRP(ICON2)*PCTP(ICON2)
     &          -(1.-ALPHA)*UAMod*PCNTMP(IPCOMP)
     &          -ALPHA*UAMod*PCNTMF(IPCOMP)
     &          -ALPHA*GroundHeatInput
     &          -(1.-ALPHA)*(PCQP(INOD2))
C===================

C Establish "containment loss" data
        presentTemp=0.5*(csvp(inod1,1)+csvp(inod2,1))
        futureTemp =0.5*(csvf(inod1,1)+csvf(inod2,1))
        QDATA(IPCOMP)=UAMod*(alpha*(futureTemp-pcntmp(ipcomp))+
     &                (1.-alpha)*(presentTemp-pcntmp(ipcomp)))

        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iConvective)
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iRadiant)


C Store "environment" variables future values
        PCTF(ICON1)=CONVAR(ICON1,1)
        PCTF(ICON2)=CONVAR(ICON2,1)
        PCRF(ICON1)=C1
        PCRF(ICON2)=C2
        PCQF(INOD1)=HeatOutput
        PCQF(INOD2)=GroundHeatInput

C Addition output for results analysis.
        NAPDAT(IPCOMP)=9
        PCAOUT(IPCOMP,1)=HeatOutput
        PCAOUT(IPCOMP,2)=COP
        PCAOUT(IPCOMP,3)=Tgrnd
        if (DeviceONOFF.eq.1) then
          if (bDHWrqd) then
           PCAOUT(IPCOMP,4)=2.
          else
           PCAOUT(IPCOMP,4)=1.
          endif
        else
          PCAOUT(IPCOMP,4)=0.
        endif
        PCAOUT(IPCOMP,5)=ReturnSP
        PCAOUT(IPCOMP,6)=TotRealPower
        PCAOUT(IPCOMP,7)=TotReacPower
        PCAOUT(IPCOMP,8)=TotApparentPower
        PCAOUT(IPCOMP,9)=GroundHeatInput

C---------------------------------------------------------------------------------
C Make select results available in XML and CVS output.
C---------------------------------------------------------------------------------
        IPCOMP_LEN = lnblnk(pcname(IPCOMP))

        if (bSHrqd) then
          CfH_out = 1.
        elseif (bDHWrqd) then
          CfH_out = 2.
        else
          CfH_out = 0.
        endif

        call AddToReport(rvPltCallForHeat%Identifier,
     &         CfH_out,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltHOut%Identifier,
     &         HeatOutput,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltCOP%Identifier,
     &         COP,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltTambient%Identifier,
     &         Tgrnd,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        tmpdev=float(DeviceONOFF)
        call AddToReport(rvPltDeviceONOFF%Identifier,
     &         tmpdev,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltReturnTSP%Identifier,
     &         ReturnSP,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltAmbientHeat%Identifier,
     &         GroundHeatInput,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltRealPow%Identifier,
     &         TotRealPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltReacPow%Identifier,
     &         TotReacPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltApparPow%Identifier,
     &         TotApparentPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))


C COUT(3/4) should be -1 (or PCONDR(ICONx) which is the same thing in this case) to
C give you the mass balance Flow*C(1)+Flow_In*C(3)=0 with Flow=Flow_In.

C If you want a fixed flow you could set C(1/2)=1, C(3/4)=0, C(5)=DeviceFlow, C(6)=BrineFlow.

C 1st phase mass (ie. "water") balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=0.
         COUT(4)=0.
         COUT(5)=DeviceFlow
         COUT(6)=BrineFlow

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=0.
         COUT(4)=0. !-PCONDR(ICON2)
         COUT(5)=0.
         COUT(6)=0.
      END IF

C Trace.
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV=20) Ground Source Heat Pump'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,',',INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1,',',ICON2
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM1     = ',CM1,' (J/K)'
            WRITE(ITU,*) ' CM2     = ',CM2,' (J/K)'
            WRITE(ITU,*) ' C1      = ',C1,' (W/Ks)'
            WRITE(ITU,*) ' C2      = ',C2,' (W/Ks)'
            WRITE(ITU,*) ' Flow      = ',DeviceFlow,' (l/s)'
            WRITE(ITU,*) ' TC      = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA   = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UAMod   = ',UAMod,' (W/K)'
            WRITE(ITU,*) ' Tground = ',Tgrnd,' (C)'
            WRITE(ITU,*) ' ReturnT = ',ReturnT,' (C)'
            WRITE(ITU,*) ' ReturnSP        = ',ReturnSP,' (C)'
            WRITE(ITU,*) ' ReturnSPH       = ',ReturnSPH,' (C)'
            WRITE(ITU,*) ' ReturnSPL       = ',ReturnSPL,' (C)'
            WRITE(ITU,*) ' InDeadB         = ',InDeadB,' (-)'
            WRITE(ITU,*) ' CallforHeat     = ',CallforHeat,' (-)'
            WRITE(ITU,*) ' DeviceONOFF     = ',DeviceONOFF,' (-)'
            WRITE(ITU,*) ' DeviceONOFFp    = ',DeviceONOFFp,' (-)'
            WRITE(ITU,*) ' COP             = ',COP,' (-)'
            WRITE(ITU,*) ' AmbientTempComp = ',AmbientTempComp,' (C)'
            WRITE(ITU,*) ' TempCompS       = ',TempCompS,' (C)'
            WRITE(ITU,*) ' TempCompE       = ',TempCompE,' (C)'
            WRITE(ITU,*) ' HeatOutput      = ',HeatOutput,' (W)'
            WRITE(ITU,*) ' GroundHeatInput = ',GroundHeatInput,' (W)'
            WRITE(ITU,*) ' CompPower       = ',CompPower,' (W)'
            WRITE(ITU,*) ' PumpPower       = ',PumpPower,' (W)'
            WRITE(ITU,*) ' CtlPower        = ',CtlPower,' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP48C'

      RETURN
      END


C *************************** CMP49C ***************************
C CMP49C generates for plant component IPCOMP with plant db code 490 ie.
C 1 node (ISV=20) WCH inverter control ground-source heat pump (icGSHP)
C feeding a hydronic heating system. Ground source temperature is taken
C from climate file data for desired depth, therefore, no ground temperature
C "degradation" is considered.
C The matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3) are
C generated.

C     ADATA: 1 Mass of component (solids+liquid) (kg)
C            2 Mass weighted average specific heat node (J/kgK)
C            3 UA modulus for component (W/K)

C     BDATA: 1 Control mode [1 - fx set in GSHP; 2 - fx via CDATA]
C            2 COP model [1- simple; 2 - 3d-pol. fit; 3 - spline int; 4 - pol. int]
C            3 -  8 HP at 0 %, coef HP_0(1) ... HP_0(6)
C            9 - 14 HP at 100 %, coef HP_100(1) ... HP_100(6)
C           15 - 20 CP at 0 %, coef CP_0(1) ... CP_0(6)
C           21 - 26 CP at 100 %, coef CP_100(1) ... CP_100(6)
C           27 Pump rating (W)
C           28 Pump pf (-)
C           29 Flowrate at rated pump power SH (l/s)
C           30 Flowrate DHW (l/s)
C           31 Controller power (W)
C           32 Controller pf (-)
C           33 Compressor pf (-)
C           34 Tout max SH (degC)
C           35 Tin max SH (degC)
C           36 Nominal water return temperature for SH (Deg C)
C           37 Tout max DHW (degC)
C           38 Tin max DHW (degC)
C           39 Nominal water return temperature for DHW
C           40 Nominal water return deadband (SH and DHW, Deg C)
C           41 Ambient air temp compensation on/off (-)
C           42 Low amb. air temperature for temp compensation [Deg C]
C           43 High amb. air temperature for temp compensation [Deg C]
C           44 Temp compensation gradient [deg C return/deg C amb. air temp] (degCr/degCa)
C           45 Heat pump lock out time (min)
C           46 Ground source model [1 - std. profile 2 - user profile 3 - f(depth)]
C           47 Model coef g0 [1 & 2 - profile number 3 - depth in m]
C
C     CDATA: 1 Relative frequency signal for space heat (-)
C            2 Relative frequency signal for DHW (-)
C            3 Preference signal 0 space heat, 1 DHW low, 2 DHW high

C Node 1 represents the condenser heat exchanger and couples to the hydronic
C heating circuit. The device has an internal pump and so an explicit pump
C model is not needed in system models containing this component.
C
C Four "COP" models are available.
C
C Model 1, "simple", is basically for testing. With this mode, the icGSHP
C can be used with very limited input data as follows.
C   HP_0(1)     0 % heating power for nominal SH temperature (kW)
C   HP_0(2)   100 % heating power for nominal SH temperature (kW)
C   HP_0(3)     0 % cooling power for nominal SH temperature (kW)
C   HP_0(4)   100 % cooling power for nominal SH temperature (kW)
C   HP_0(5)     0 % heating power for nominal DHW temperature (kW)
C   HP_0(6)   100 % heating power for nominal DHW temperature (kW)
C   HP_100(1)   0 % cooling power for nominal DHW temperature (kW)
C   HP_100(2) 100 % cooling power for nominal DHW temperature (kW)
C Note that both inverter control modes can be used with this simplified model,
C however, source temperature dependency is ignored and thus the HP only
C very crudely approximated.
C
C Model 2, "3d-polynomial fit" requires 24 coefficients which describe the
C available heating power and the ambient heat extraction (aka. cooling
C power) in (kW) at lowest and highest compressor frequencies via 3d
C polynomials of the form
C   P = C[1] + C[2]*x + C[3]*x^2 + C[4]*y + C[5]*x*y + C[6]*y^2
C where
C  the coefficients C[i] are given as BDATA(3) through BDATA(26)
C  x  denotes the evaporator source temperature (ground temperaturen) and
C  y  denotes Tout max (i.e. the feed temperature for SH or DHW).
C
C Models 3 + 4, spline and polynomial interpolations of power output
C respectively. These modes expect a data file <<componentname>>.hpd
C in the same subdirectory as the .pln file itself.
C The data file must contain following data in the given structure:
C performance data file, containing a header line and three columns of
C data. NX and NY are the number of data points for tgrnd and tfeed,
C respectively.
C
C Format:
C NX, NY
C Tgrnd1,  Tfeed1,  HPheat0-1, HPheat100-1, HPcool0-1, HPcool100-1
C Tgrnd1,  Tfeed2,  HPheat0-2, HPheat100-2, HPcool0-2, HPcool100-2
C ...
C Tgrnd1,  TfeedNY, HPheat0-N, HPheat100-N, HPcool0-N, HPcool100-N
C Tgrnd2,  Tfeed1,  HPheat0-1, HPheat100-1, HPcool0-1, HPcool100-1
C ...
C ...
C TgrndNX, TfeedNY, HPheat0-N, HPheat100-N, HPcool0-N, HPcool100-N
C
C This particular device model is suitable for ON/OFF type control in control
C mode 1 and if PI-type control in control mode 2. The model internally
C controls the compressor frequency in mode 1. Ambient temperature compensation
C for the space heating return temperature can be applied [linear model].
C
C Control parameter number three (3) is the "preference signal":
C   0 space heat, 1 DHW low, 2 DHW high

      SUBROUTINE CMP49C(IPCOMP,COUT,ISTATS)
      use h3kmodule
      implicit none
#include "plant.h"
#include "building.h"
#include "site.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/ITERINDEX/ITERNU !plant iteration number

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      common/pstctr/nsincp,nsncpr
      integer nsincp          ! number of plant side time increments
      integer nsncpr          ! number of ??

      COMMON/PCEQU/IMPEXP,RATIMP

C     Variables for iteration in plant loop (?)
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)
      INTEGER MAXITP,ICSV,itrclp
      REAL    PERREL,PERTMP,PERFLX,PERMFL,CSVI

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)

      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      INTEGER NPCON,IPC1,IPN1,IPCT,IPC2,IPN2
      REAL PCONDR,PCONSD

      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      REAL ADATA,BDATA

      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      common/pcnam/pcname(mpcom)

C Description for additional output? ((see esrures/moplnt, line 1183))
cx      common/adopdesc/adopdsc(mpcom,MPCRES)
cx      CHARACTER adopdsc*30,


      COMMON/CLIMIP/QFPP,QFFP,TPP,TFP,QDPP,QDFP,VPP,VFP,DPP,DFP,HPP,HFP

      COMMON/GSHPvar1/CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &  COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,
     &  CompressPf,PumpRating,PumpPf,RatedFlow,
     &  CtlRating,CtlPf,ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,
     &  NomRetTSH,NomRetTDHW,NomRetTDeadB,NomRetTDeadBDHW,
     &  TempCompS,TempCompE,TempCompc0

      COMMON/GSHPvar2/COPModel,CompModel,AmbientTempComp,
     &                DeviceONOFFp,DeviceONOFF,GroundSourceModel,
     &                GroundSourceCoeff

      COMMON/GSHPvar3/CallforHeat,InDeadB

      COMMON/GSHPvar4/INSO,ITSBO,bSwitched,BLKOUT,NTSBO,bBLKOUT
      INTEGER INSO,ITSBO,NTSBO
      REAL BLKOUT
      LOGICAL bSwitched,bBLKOUT

      COMMON/GSHPvar5/CTLmode,HP_0(6),HP_100(6),
     &                CP_0(6),CP_100(6),ToutDHWlow,NomRetTDHWlow,
     &                RatedFlowDHW

      REAL HP_0,HP_100,CP_0,CP_100,ToutDHWlow,NomRetTDHWlow,
     &  RatedFlowDHW
      INTEGER CTLmode

C----------------
C icGSHP data file commons
C----------------
C Heat pump performance data / spline function data
      common/icgshpspline/
     &          tground(MPCOM,MPCDAT),
     &          tfeed(MPCOM,MPCDAT),
     &          heat0(MPCOM,MPCDAT,MPCDAT),
     &          heat100(MPCOM,MPCDAT,MPCDAT),
     &          cool0(MPCOM,MPCDAT,MPCDAT),
     &          cool100(MPCOM,MPCDAT,MPCDAT),
     &          nx(MPCOM),ny(MPCOM),
     &          heat0_2d(MPCOM,MPCDAT,MPCDAT),
     &          heat100_2d(MPCOM,MPCDAT,MPCDAT),
     &          cool0_2d(MPCOM,MPCDAT,MPCDAT),
     &          cool100_2d(MPCOM,MPCDAT,MPCDAT),
     &          fnamHPCoefdat(MPCOM)
      real tground    ! Ground source temperature data values, degC
      real tfeed      ! Feed temperature values, degC
      real heat0      ! HP heating power at lowest converter frequency, kW
      real heat100    ! HP heating power at highest converter frequency, kW
      real cool0      ! HP cooling power at lowest converter frequency, kW
      real cool100    ! HP cooling power at highest converter frequency, kW
      real heat0_2d   ! HP heating power 0 % second-derivatives
      real heat100_2d ! HP heating power 100 % second-derivatives
      real cool0_2d   ! HP cooling power 0 % second-derivatives
      real cool100_2d ! HP cooling power 100 % second-derivatives

      integer nx,ny              ! number of data points in x and y directions

      character fnamHPCoefdat*72 ! Data file name for current icGSHP entry

C Electrical details for specified plant components
      common/pcelflg/ipcelf(mpcom)
      common/elpcp/NPEL,PFP(mpcom),IPFP(mpcom),PWRP(mpcom),
     &BVOLTP(mpcom),IPHP(mpcom)

      common/forlock/NSINCP_lock
      integer NSINCP_lock

      REAL SMALL
      PARAMETER (SMALL=1.0E-20)

      REAL PFP,PWRP,BVOLTP,PWRQ,PA,PQ

      INTEGER ipcelf,NPEL,IPFP,IPHP,IEMODEL

      LOGICAL CallforHeat,InDeadB

      INTEGER COPModel,CompModel,AmbientTempComp,
     &DeviceONOFFp,DeviceONOFF,GroundSourceModel,GroundSourceCoeff

      REAL CompMass,AveSpHt,UAMod,CarEffMod,COPa0,COPa1,
     &COPa2,COPa3,Compa0,Compa1,Compa2,Compa3,CompressPf,
     &PumpRating,PumpPf,RatedFlow,CtlRating,CtlPf,
     &ToutMaxSH,TinMaxSH,ToutMaxDHW,TinMaxDHW,NomRetTSH,NomRetTDHW,
     &NomRetTDeadB,NomRetTDeadBDHW,TempCompS,TempCompE,TempCompc0

      REAL CompPower,CompressPower,COP,CtlPower,DeviceFlow,
     &HeatOutput,PumpPower,ReturnSP,ReturnSPH,ReturnSPL,RHamb,
     &TotApparentPower,TotReacPower,TotRealPower

      INTEGER lnblnk
      REAL SHTFLD

      INTEGER ITC,ICNT,ITCF,ITRACE,IZNTRC,ITU,ITERNU,
     &IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow,IMPEXP

      INTEGER NPCDAT,IPOFS1,IPOFS2,ICON1,INOD1,napdat,
     &IPCOMP,ISTATS,I,ICONDX,ICONTP,IX1,NCI,NITMS,NPCOMP,
     &INOD2,ICON2

      REAL TIMSEC,TC,RATIMP,CSVF,CSVP,PCTF,PCRF,PUAF,PCQF,PCNTMF,
     &PCTP,PCRP,PUAP,PCQP,PCNTMP,CONVAR,QFPP,QFFP,TPP,TFP,QDPP,QDFP,VPP

      REAL VFP,DPP,DFP,HPP,HFP,Tamb,ReturnT,Td,Ta,TL,TH,ALPHA,CM,
     &      C1,TmpDev,futureTemp,presentTemp,Tgrnd,GroundHeatInput

      REAL COUT(MPCOE),QDATA,PCAOUT,CDATA

      REAL TG1,TG2,LimFac,CfH_out,ToutMax,
     &      CP,HP,CP0,CP100,HP0,HP100,RelCompFreq

      REAL dHP0, dCP0, dHP100, dCP100

      INTEGER IM,ID,curDoM,curMonth,NDMonth

      DIMENSION NDMonth(12)
      DATA NDMonth/31,28,31,30,31,30,31,31,30,31,30,31/

      LOGICAL CLOSE,CLOSEA,bSHrqd,bDHWrqd,bSHpref

      CHARACTER OUTS*248,PCNAME*15

      INTEGER IPCOMP_LEN !lenght of ipcomp name

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP49C'

C ************
C Variable Assignment and initialisation mainly in CMP49S (pcomps.F)

      if (NSINC.eq.1) then
C NTSBO = total no. of timesteps HP is off due to lockout
        NTSBO=NINT(BLKOUT/TIMSEC)
      endif

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

C Check for containment and adapt component heat loss coefficient.
      call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
      IF(closea) UAMod=0.

C Read ambient temperature (for temperature compensation of WCH return temp)
      Tamb=TFP

C Set initial operation mode switches based on CDATA.
      bSHrqd  =.false.
      bDHWrqd =.false.
      if (CDATA(IPCOMP,1).ge.0.005) then
         bSHrqd=.true.
      endif
      if (CDATA(IPCOMP,2).ge.0.005) then
         bDHWrqd=.true.
      endif
      call eclose(CDATA(IPCOMP,3),0.0,0.001,bSHpref)

C Check which mode is actually to be used for the case when both demands
C are set to .true.
      if ((bSHrqd.and.bDHWrqd)) then
C       Both are .true.
        if (bSHpref) then
C         Re-set DHW to false, as SH is preferred !
          bDHWrqd=.false.
        else
C         Re-set SH to false, as DHW is preferred !
          bSHrqd=.false.
        endif
      endif

      if ((bSHrqd.or.bDHWrqd)) then
C       Either space heat or dhw needs heat
        CallForHeat=.true.

C Pump circulates while there is a call for heat.
cx <<extend with brine pump? >>
        PumpPower=PumpRating/PumpPf
        if (bSHrqd) then
          DeviceFlow=RatedFlow
        else
          DeviceFlow=RatedFlowDHW
        endif
      ELSE
        CallforHeat=.false.
C       Only turn circulation pump off when no call for heat!
C       Careful: DeviceFlow is used in ISTAT.eq.2 !
        PumpPower=0.
        DeviceFlow=0.0
      ENDIF

C Set return setpoints based on which function requires heat
      if (bDHWrqd) then
        if (NINT(CDATA(IPCOMP,3)).eq.2) then
C         DHW in high temperature mode.
          ToutMax=ToutMaxDHW
          ReturnSP=NomRetTDHW !default return temperature DHW mode
          ReturnSPL=ReturnSP-0.5*(NomRetTDeadBDHW)
          ReturnSPH=ReturnSP+0.5*(NomRetTDeadBDHW)

        elseif (NINT(CDATA(IPCOMP,3)).eq.1) then
C         DHW in low temperature mode.
          ToutMax=ToutDHWlow
          ReturnSP=NomRetTDHWlow !default return temperature DHW mode
          ReturnSPL=ReturnSP-0.5*(NomRetTDeadBDHW)
          ReturnSPH=ReturnSP+0.5*(NomRetTDeadBDHW)

        else
C         Issue warning.
        endif

C       Check against max. temperature.
        IF(ReturnSP.GT.ToutMax) ReturnSP=ToutMax

        if (CTLmode.eq.1) then
C       Ambient temperature compensation not used for DHW mode.
          RelCompFreq=1.0

C       Set compressor frequency for DHW to 100 % or to CDATA
C       depending on control mode.
        elseif (CTLmode.eq.2) then
C         Use control setpoint as compressor frequency (Hz). Make sure
C         that relative compressor frequency is not out of bounds.
          if (CDATA(IPCOMP,2).gt.0.05) then
            if (CDATA(IPCOMP,2).le.1.0) then
C             Set compressor frequency value to CDATA.
              RelCompFreq=CDATA(IPCOMP,2)
            else
              RelCompFreq=1.0
            endif
          else
            RelCompFreq=0.0
          endif

        else
          write(*,*)"Error icGSHP: CTLmode for DHW not in valid range!"
          STOP
        endif

      else
C       Space heat mode.
        ToutMax=ToutMaxSH
        ReturnSP=NomRetTSH   !default return temperature SH mode
        ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
        ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)

        IF(ReturnSP.GT.ToutMaxSH) ReturnSP=ToutMaxSH

        IF(AmbientTempComp.GT.0)THEN
C       Calculate the return water temperature set point if
C       temperature compensation is active.

          IF(Tamb.GE.TempCompS.AND.Tamb.LT.TempCompE)THEN
            ReturnSP=((Tamb-TempCompS)*TempCompc0)+NomRetTSH
            ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
            ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)

C         Case where Tamb > Temp compensation end point.
          ELSEIF(Tamb.GE.TempCompE)THEN
            ReturnSP=((TempCompE-TempCompS)*TempCompc0)+NomRetTSH
            ReturnSPL=ReturnSP-0.5*(NomRetTDeadB)
            ReturnSPH=ReturnSP+0.5*(NomRetTDeadB)
          ENDIF

          IF(ReturnSPH.GT.ToutMaxSH) ReturnSPH=ToutMaxSH

          IF(ReturnSP.GT.ToutMaxSH)THEN
            ReturnSP=ToutMaxSH ! i.e. ReturnSP = ReturnSPH
          ENDIF
        ENDIF

C       Set compressor frequency for SH depending on return setpoint
C       temperature or to CDATA depending on control mode.
        if (CTLmode.eq.1) then
C         Calculate relative compressor frequency (-) from ambient
C         air temperature and temperature compensation data.
          if ((TempCompE-TempCompS).gt.0.0) then
            RelCompFreq=(TempCompE-Tamb)/
     &                              (TempCompE-TempCompS)
          endif
C         Make sure that relative compressor frequency is not out OB.
          if (RelCompFreq.GT.1.0) RelCompFreq=1.0
          if (RelCompFreq.LT.0.0) RelCompFreq=0.0

        elseif (CTLmode.eq.2) then
C         Use control setpoint as compressor frequency (Hz). Make sure
C         that relative compressor frequency is not out of bounds.
          if (CDATA(IPCOMP,1).gt.0.05) then
            if (CDATA(IPCOMP,1).le.1.0) then
C             Set compressor frequency value to CDATA.
              RelCompFreq=CDATA(IPCOMP,1)
            else
              RelCompFreq=1.0
            endif
          else
            RelCompFreq=0.0
          endif

        else
          write(*,*)"Error icGSHP: CTLmode for SH not in valid range!"
          STOP
        endif

      endif ! space heat rq'd

C     Set ground source temperature based on model chosen.
      if (GroundSourceModel.eq.1 .or. GroundSourceModel.eq.2) then

C     Standard ground temperature profile from grdtmp(12,mgrdp), see esru_blk.F,
C     or user defined ground temperature profile from ugrdtp(12,mgrdp).

        if (GroundSourceModel.eq.2) then
          if (NGRDP.le.0) then
            write(*,*) "Error icGSHP: Option 2 requires user defined",
     &             " ground temperature profile! "
            STOP
          elseif (GroundSourceCoeff.gt.NGRDP) then
            write(*,*) "Error icGSHP: Ground temperature profile #",
     &             GroundSourceCoeff," not available! "
            STOP
          endif
        endif
C       Calculate the current ground temperature by interpolating the
C       monthly ground temperatures. ((code from CMP79C(), lines 8976 ff.))

C       Get current month temperature
        CALL EDAYR(IDYP,ID,IM)
        curMonth=IM
        curDoM=ID

        if (GroundSourceModel.eq.1) then
          TG1=grdtmp(curMonth,int(GroundSourceCoeff))
        else
          TG1=ugrdtp(curMonth,int(GroundSourceCoeff))
        endif

        IF(curMonth.EQ.12)THEN
C         Get next month's temperature (assume loop round to January).
          curMonth=1
        ELSE
C         Get next month's temperature
          curMonth=curMonth+1
        ENDIF

        if (GroundSourceModel.eq.1) then
          TG2=grdtmp(curMonth,int(GroundSourceCoeff))
        else
          TG2=ugrdtp(curMonth,int(GroundSourceCoeff))
        endif

C       Calculate today's ground temperature by linear interpolation
        Tgrnd=((float(curDoM)/float(NDMonth(curMonth)))
     &                                              *(TG2-TG1))+TG1

cx        write(*,*)"Tgrnd=",Tgrnd,",  Tamb=",Tamb

      elseif (GroundSourceModel.eq.3) then
C     Calculate ground source temperature in desired depth GroundSourceCoeff
cx      CALL GTCALC(GroundSourceCoeff,'-',IER)
cx      Tgrnd=??
        write(*,*) "Error icGSHP: Option 3 for ground temperature",
     &               " not available to date! "
        STOP
      else
C     Ground source temperature option invalid
        write(*,*) "Error icGSHP: Option for ground temperature",
     &               " invalid! "
        STOP
      endif

C Determine whether machine should be on or off based on return
C water temperature.
      ReturnT=CONVAR(ICON1,1)

C Check if device switched off last TS
      IF ((DeviceONOFFp.eq.1).and.(DeviceONOFF.eq.0)) THEN
        bSwitched=.true.
      ENDIF

      DeviceONOFFp=DeviceONOFF

      IF(CallforHeat)THEN

C Check to see if the device is in the dead band.
        IF(ReturnT.GE.ReturnSPL.AND.ReturnT.LE.ReturnSPH)THEN
          InDeadB=.true.
        ELSE
          InDeadB=.false.
        ENDIF

C Evaluate the device status.
        IF(ReturnT.GT.ReturnSPH)THEN
          DeviceONOFF=0
        ELSEIF(ReturnT.LT.ReturnSPL)THEN
          DeviceONOFF=1
        ELSEIF(DeviceONOFFp.EQ.1.AND.InDeadB)THEN
          DeviceONOFF=1
        ELSEIF(DeviceONOFFp.EQ.0.AND.InDeadB)THEN
          DeviceONOFF=0
        ENDIF
      ELSE
        DeviceONOFF=0
      ENDIF ! call for heat

C == start lockout code
C Finally, check if device is in lockout time, i.e. last "off" was
C less than lockouttime ago ... ((code from CMP42C(), line 5102 ff.))
C Lock out time (in seconds), Heat pump should not be turned on/off
C with high frequency. When HP is turned off, wait this amount of time
C before turning on again.
C INSO = HP on/off (1/0) flag for lockout
C ITSBO = present no. of timesteps HP is off due to lockout
C NTSBO = total no. of timesteps HP is off due to lockout
      IF (bSwitched) THEN
        if (bBLKOUT) then
          NSINCP_lock=NSINCP
          INSO=1
          ITSBO=0
          bSwitched=.false. ! only set once after signal
        endif
      ENDIF

      IF (INSO.EQ.1) THEN
        IF (ITSBO.GT.NTSBO) INSO=0
        if (NSINCP_lock.lt.NSINCP) then
          ITSBO=ITSBO+1
          NSINCP_lock=NSINCP
        endif
        DeviceONOFF=0
      ENDIF
C == end lockout

      IF (DeviceONOFF.GT.0) THEN

C Calculate the TOTAL device power draw (W)
        IF(COPmodel.eq.1)THEN
          if (bSHrqd) then
            HP0   = HP_0(1)
            HP100 = HP_0(2)
            CP0   = HP_0(3)
            CP100 = HP_0(4)
          elseif (bDHWrqd) then
            HP0   = HP_0(5)
            HP100 = HP_0(6)
            CP0   = HP_100(1)
            CP100 = HP_100(2)
          else
C         Something wrong!
            WRITE(OUTS,'(a)')'Error in 1-node icGSHP, SH or DHW? '
            CALL EDISP(IUOUT,OUTS)
          endif
C         Heating and cooling power at current compressor frequency.
          HP=RelCompFreq*(HP100-HP0)+HP0
          CP=RelCompFreq*(CP100-CP0)+CP0

        ELSEIF(COPmodel.eq.2) then
          Ta=Tgrnd    ! "x"
          Td=ToutMax  ! "y"
C         Heating and cooling power at lowest compressor frequency.
          HP0 = HP_0(1) + HP_0(2)*Ta + HP_0(3)*Ta*Ta
     &           + HP_0(4)*Td + HP_0(5)*Ta*Td + HP_0(6)*Td*Td
          CP0 = CP_0(1) + CP_0(2)*Ta + CP_0(3)*Ta*Ta
     &           + CP_0(4)*Td + CP_0(5)*Ta*Td + CP_0(6)*Td*Td

C         Heating and cooling power at highest compressor frequency.
          HP100 = HP_100(1) + HP_100(2)*Ta + HP_100(3)*Ta*Ta
     &            + HP_100(4)*Td + HP_100(5)*Ta*Td + HP_100(6)*Td*Td
          CP100 = CP_100(1) + CP_100(2)*Ta + CP_100(3)*Ta*Ta
     &            + CP_100(4)*Td + CP_100(5)*Ta*Td + CP_100(6)*Td*Td

C         Heating and cooling power at current compressor frequency.
          HP=RelCompFreq*(HP100-HP0)+HP0
          CP=RelCompFreq*(CP100-CP0)+CP0

        ELSEIF((COPmodel.eq.3).or.(COPmodel.eq.5)) then
          Ta=Tgrnd    ! "x"
          Td=ToutMax  ! "y"
C         Heating and cooling power at lowest compressor frequency.
          call splin2(tground(IPCOMP,:),
     &                tfeed(IPCOMP,:),
     &                heat0(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                heat0_2d(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                nx(IPCOMP), ny(IPCOMP),
     &                Ta, Td, HP0)

          call splin2(tground(IPCOMP,:),
     &                tfeed(IPCOMP,:),
     &                cool0(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                cool0_2d(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                nx(IPCOMP), ny(IPCOMP),
     &                Ta, Td, CP0)

C         Heating and cooling power at highest compressor frequency.
          call splin2(tground(IPCOMP,:),
     &                tfeed(IPCOMP,:),
     &                heat100(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                heat100_2d(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                nx(IPCOMP), ny(IPCOMP),
     &                Ta, Td, HP100)

          call splin2(tground(IPCOMP,:),
     &                tfeed(IPCOMP,:),
     &                cool100(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                cool100_2d(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                nx(IPCOMP), ny(IPCOMP),
     &                Ta, Td, CP100)

C         Heating and cooling power at current compressor frequency.
          HP=RelCompFreq*(HP100-HP0)+HP0
          if (COPmodel.eq.3) then
            CP=RelCompFreq*(CP100-CP0)+CP0
          else
C           COPmodel 5 is a "hidden mode" which tweaks the approximate
C           COP-behaviour as function of the relative inverter frequency
C           for a specific heat pump model (cta Optiheat Inverta Economy
C           Compact, OH I 9ec). SH mode, only.
            if (bSHrqd) then
              CP=(RelCompFreq*(CP100-CP0)+CP0)*
     &          (1.0 + (-0.0015119 + 0.00022779 * (100.*RelCompFreq)
     &           + (-1.2154e-5) * (100.*RelCompFreq)**2))
            else
              CP=RelCompFreq*(CP100-CP0)+CP0
            endif
          endif

cx          write(*,'(8(a,f7.3))')'Mode 3: Ta=',Ta,', Td=',Td,
cx     &              ' HP0=',HP0,', HP100=',HP100,
cx     &              ', CP0=',CP0,', CP100=',CP100,
cx     &              ', **HP=',HP,', **CP=',CP

        ELSEIF(COPmodel.eq.4) then
          Ta=Tgrnd    ! "x"
          Td=ToutMax  ! "y"
C         Heating and cooling power at lowest compressor frequency.
          call polin2(tground(IPCOMP,:),
     &                tfeed(IPCOMP,:),
     &                heat0(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                nx(IPCOMP), ny(IPCOMP),
     &                Ta, Td, HP0, dHP0)

          call polin2(tground(IPCOMP,:),
     &                tfeed(IPCOMP,:),
     &                cool0(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                nx(IPCOMP), ny(IPCOMP),
     &                Ta, Td, CP0, dCP0)

C         Heating and cooling power at highest compressor frequency.
          call polin2(tground(IPCOMP,:),
     &                tfeed(IPCOMP,:),
     &                heat100(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                nx(IPCOMP), ny(IPCOMP),
     &                Ta, Td, HP100, dHP100)

          call polin2(tground(IPCOMP,:),
     &                tfeed(IPCOMP,:),
     &                cool100(IPCOMP,1:nx(IPCOMP),1:ny(IPCOMP)),
     &                nx(IPCOMP), ny(IPCOMP),
     &                Ta, Td, CP100, dCP100)

C         Heating and cooling power at current compressor frequency.
          HP=RelCompFreq*(HP100-HP0)+HP0
          CP=RelCompFreq*(CP100-CP0)+CP0

cx          write(*,'(8(a,f7.3))')'Mode 4: Ta=',Ta,', Td=',Td,
cx     &              ' HP0=',HP0,', HP100=',HP100,
cx     &              ', CP0=',CP0,', CP100=',CP100,
cx     &              ', **HP=',HP,', **CP=',CP

        ELSE
          WRITE(OUTS,'(a)')'Error in 1-node icGSHP, COP model # '
          CALL EDISP(IUOUT,OUTS)
        ENDIF

C       Electrical compressor power (taking efficiency into account).
        CompPower=(HP-CP)/CompressPf
        if(CompPower.LT.0.0) then
          CompPower=0.0
          COP=0.0
        else
C         COP from heating power and electrical compressor power.
          COP=HP/CompPower
        endif

C Set limits on calculated values of COP
        IF(COP.LT.0.0) COP=0.0
        if (bSHrqd) then
C         This probably should either not be necessary or a input variable!
          IF(COP.GT.9.0) COP=9.0
        elseif (bDHWrqd) then
C         This probably should either not be necessary or a input variable!
          IF(COP.GT.3.5) COP=4.0
        endif

C Final heat output and ground heat extraction in (W)
        HeatOutput=1000.*HP
        GroundHeatInput=-1000.*CP

        IF(HeatOutput.LT.0) THEN
          write(*,*) "Error-8374: heat output < 0! ",
     &                   HeatOutput, COP, CompPower
          STOP
        ENDIF

        IF(GroundHeatInput.GT.0) THEN
          write(*,*) "Error-8380: ground heat input > 0! ",
     &                 GroundHeatInput, HeatOutput, COP, CompPower
          STOP
        ENDIF

      ELSE
C     Device is 'off' (DeviceONOFF.eq.0), i.e. the heat pump is off,
C     <<however the circulation pump remains on ??>>. Set parameters
C     accordingly (assume controller is still operating).
        PumpPower=0.
cx        DeviceFlow=0.000001

        CompPower=0.
        HeatOutput=0.
        GroundHeatInput=0.
        COP=0.
      ENDIF

C Generate coefficients for energy balance equation
      IF (ISTATS.EQ.1) THEN

C Establish the nodal thermal capacities (see CMP36C).
        CM=CompMass*AveSpHt

C Node 1 water
        C1=AMAX1(SMALL,DeviceFlow)*SHTFLD(3,CONVAR(ICON1,1))

C Calculate the electrical demand of the device assumung the controller is ON.
        CtlPower=CtlRating
        TotRealPower=1000.*CompPower+CtlPower+PumpPower ! [W]

C Save the real power draw for use in an electrical power flow simulation.
        PWRP(IPCOMP)=-ABS(TotRealPower)
        IEMODEL=1
        CALL EMACH(IPCOMP,IEMODEL,PWRP(IPCOMP),PQ,PA)
        PWRQ=PQ

c Calculate the compressor power draw.
        CompressPower=1000.*CompPower*CompressPf ! [W] -(PumpPower+CtlPower)
        IF(CompressPower.LT.0.0)CompressPower=0.0

        TotReacPower=( ((CompressPower/CompressPf)**2)
     &                                       -(CompressPower**2) )**0.5
     &              +( ((PumpPower/PumpPf)**2)-(PumpPower**2) )**0.5
     &              +( ((CtlPower/CtlPf)**2) -(CtlPower**2) )**0.5

        TotApparentPower=((TotRealPower**2)+(TotReacPower**2))**0.5

C Calculate current component time-constant TC
        TC(IPCOMP)=CM/AMAX1(SMALL,(C1+UAMod))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
        IF(IMPEXP.EQ.1) THEN
           ALPHA=1.
        ELSE IF(IMPEXP.EQ.2) THEN
           ALPHA=RATIMP
        ELSE IF(IMPEXP.EQ.3) THEN
          IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
             ALPHA=1.
          ELSE
             ALPHA=RATIMP
          END IF
        ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
        END IF

C Establish matrix equation self- and cross-coupling coefficients.
        COUT(1)=ALPHA*(-C1-UAMod)-CM/TIMSEC
C Matrix cross coupling coefficients.
        COUT(2)=ALPHA*C1
C Establish the present and known coefficient i.e. RHS
        COUT(3)=((1.-ALPHA)*(PCRP(ICON1)+PUAP(INOD1))-CM/TIMSEC)
     &          *CSVP(INOD1,1)
     &          +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &          -ALPHA*UAMod*PCNTMF(IPCOMP)
     &          -ALPHA*HeatOutput
     &          -(1.-ALPHA)*PCQP(INOD1)
     &          -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)

C Establish "containment loss" data
        presentTemp=csvp(inod1,1)
        futureTemp =csvf(inod1,1)
        QDATA(IPCOMP)=UAMod*(alpha*(futureTemp-pcntmp(ipcomp))+
     &                (1.-alpha)*(presentTemp-pcntmp(ipcomp)))

        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iConvective)
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iRadiant)

C Store "environment" variables future values
        PCTF(ICON1)=CONVAR(ICON1,1)
        PCRF(ICON1)=C1
        PCQF(INOD1)=HeatOutput

C Addition output for results analysis.
        NAPDAT(IPCOMP)=9
        PCAOUT(IPCOMP,1)=HeatOutput
        PCAOUT(IPCOMP,2)=COP
        PCAOUT(IPCOMP,3)=Tgrnd
        if (DeviceONOFF.eq.1) then
          if (bDHWrqd) then
           PCAOUT(IPCOMP,4)=2.
          else
           PCAOUT(IPCOMP,4)=1.
          endif
        else
          PCAOUT(IPCOMP,4)=0.
        endif
        PCAOUT(IPCOMP,5)=ReturnSP
        PCAOUT(IPCOMP,6)=1000.*CompPower
        PCAOUT(IPCOMP,7)=CtlPower+PumpPower
        PCAOUT(IPCOMP,8)=TotApparentPower
        PCAOUT(IPCOMP,9)=GroundHeatInput

C---------------------------------------------------------------------------------
C Make select results available in XML and CVS output.
C---------------------------------------------------------------------------------
        IPCOMP_LEN = lnblnk(pcname(IPCOMP))

        if (bSHrqd) then
          CfH_out = 1.
        elseif (bDHWrqd) then
          CfH_out = 2.
        else
          CfH_out = 0.
        endif

        call AddToReport(rvPltCallForHeat%Identifier,
     &         CfH_out,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltRelCompFreq%Identifier,
     &         RelCompFreq,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltHOut%Identifier,
     &         HeatOutput,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltCOP%Identifier,
     &         COP,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltTambient%Identifier,
     &         Tgrnd,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        tmpdev=float(DeviceONOFF)
        call AddToReport(rvPltDeviceONOFF%Identifier,
     &         tmpdev,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltReturnTSP%Identifier,
     &         ReturnSP,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltAmbientHeat%Identifier,
     &         GroundHeatInput,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltRealPow%Identifier,
     &         TotRealPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltReacPow%Identifier,
     &         TotReacPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))

        call AddToReport(rvPltApparPow%Identifier,
     &         TotApparentPower,
     &         pcname(IPCOMP)(1:IPCOMP_LEN))


C 1st phase mass (ie. "water") balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=DeviceFlow

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
      END IF

C Trace.
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) Inverter Control GSHP'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM      = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1      = ',C1,' (W/Ks)'
            WRITE(ITU,*) ' Flow      = ',DeviceFlow,' (l/s)'
            WRITE(ITU,*) ' TC      = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA   = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UAMod   = ',UAMod,' (W/K)'
            WRITE(ITU,*) ' Tground = ',Tgrnd,' (C)'
            WRITE(ITU,*) ' tBlkout = ',BDATA(IPCOMP,26),' (min)'
            WRITE(ITU,*) ' ReturnT = ',ReturnT,' (C)'
            WRITE(ITU,*) ' ReturnSP        = ',ReturnSP,' (C)'
            WRITE(ITU,*) ' ReturnSPH       = ',ReturnSPH,' (C)'
            WRITE(ITU,*) ' ReturnSPL       = ',ReturnSPL,' (C)'
            WRITE(ITU,*) ' InDeadB         = ',InDeadB,' (-)'
            WRITE(ITU,*) ' CallforHeat     = ',CallforHeat,' (-)'
            WRITE(ITU,*) ' DeviceONOFF     = ',DeviceONOFF,' (-)'
            WRITE(ITU,*) ' DeviceONOFFp    = ',DeviceONOFFp,' (-)'
            WRITE(ITU,*) ' COP             = ',COP,' (-)'
            WRITE(ITU,*) ' AmbientTempComp = ',AmbientTempComp,' (C)'
            WRITE(ITU,*) ' TempCompS       = ',TempCompS,' (C)'
            WRITE(ITU,*) ' TempCompE       = ',TempCompE,' (C)'
            WRITE(ITU,*) ' HeatOutput      = ',HeatOutput,' (W)'
            WRITE(ITU,*) ' GroundHeatInput = ',GroundHeatInput,' (W)'
            WRITE(ITU,*) ' CompPower       = ',CompPower,' (W)'
            WRITE(ITU,*) ' PumpPower       = ',PumpPower,' (W)'
            WRITE(ITU,*) ' CtlPower        = ',CtlPower,' (W)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP49C'

      RETURN
      END


C ******************** CMP50C ********************

C CMP50C generates for plant component IPCOMP with plant db code 500 ie.
C 1 node (ISV=29) WCH thermostatic radiator valve
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Index of coupled building zone (-)
C            2 Index of coupled wall in that zone (-)
C            3 Thermal conductance between water and sensor (W/K)
C            4 Equiv. convective conductance to air (W/K)
C            5 Equiv. radiative conductance to wall (W/K)
C            6 Equiv. radiative conductance to radiator (W/K)
C     CDATA: none

C     PCDATF/P
C            1 Air temperature of coupled building zone (C)
C            2 Surface temperature of coupled wall in 1 (C)

      SUBROUTINE CMP50C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)

      COMMON/C6/INDCFG
      COMMON/FVALA/TFA(MCOM),QFA(MCOM)
      COMMON/FVALS/TFS(MCOM,MS),QFS(MCOM)

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP50C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      ICON2=ICONDX(IPCOMP,1,2)
      INOD1=NPCDAT(IPCOMP,9)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First set-up "surroundings"
         IF(INDCFG.EQ.2.OR.NINT(BDATA(IPCOMP,6)).EQ.0) THEN
           call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
           IF(closea) THEN
               TAIR=20.
            ELSE
               TAIR=PCNTMF(IPCOMP)
            END IF
            TWAL=TAIR
         ELSE
            IZ=NINT(BDATA(IPCOMP,1))
            TAIR=TFA(IZ)
            TWAL=TFS(IZ,NINT(BDATA(IPCOMP,2)))
         END IF
         HWAT=BDATA(IPCOMP,3)
         HAIR=BDATA(IPCOMP,4)
         HWAL=BDATA(IPCOMP,5)
         HRAD=BDATA(IPCOMP,6)

C Establish heat capacity of component mass CM (J/K)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(HWAT+HAIR+HWAL+HRAD))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self- and cross-coupling coefficients
         COUT(1)=ALPHA*(-HWAT-HAIR-HWAL-HRAD)-CM/TIMSEC
         COUT(2)=ALPHA*HWAT
         COUT(3)=ALPHA*HRAD
C and then present-time coefficient (ie. right hand side)
         COUT(4)=((1.-ALPHA)*(HWAT+HAIR+HWAL+HRAD)
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             +(1.-ALPHA)*(-HWAT)*PCTP(ICON1)
     &             +(1.-ALPHA)*(-HRAD)*PCTP(ICON2)
     &             -ALPHA*HAIR*TAIR
     &             -(1.-ALPHA)*HAIR*PCDATP(IPCOMP,1)
     &             -ALPHA*HWAL*TWAL
     &             -(1.-ALPHA)*HWAL*PCDATP(IPCOMP,2)

C Store "environment" variables future values
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCTF(ICON2)=CONVAR(ICON2,1)
         PCDATF(IPCOMP,1)=TAIR
         PCDATF(IPCOMP,2)=TWAL

C 1st phase mass (ie. "water") balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
         COUT(4)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
         COUT(4)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=29) WCH thermostatic valve'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1,ICON2
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' Tair   = ',TAIR,' (C)'
            WRITE(ITU,*) ' Twall  = ',TWAL,' (C)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP50C'

      RETURN
      END

C ******************** CMP51C ********************

C CMP51C generates for plant component IPCOMP with plant db code 510 ie.
C 1 node (ISV=29)     mechanical room thermostat
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Index of coupled building zone (-)
C            2 Index of "viewed" wall in that zone (-)
C            3 Index of wall on which device is mounted (-)
C            4 Equiv. convective conductance to air (W/K)
C            5 Equiv. radiative conductance to wall 2 (W/K)
C            6 Equiv. thermal conductance to wall 3 (W/K)
C     CDATA: 1 acceleration heating (W)

C     PCDATF/P
C            1 Air temperature of coupled building zone (C)
C            2 Surface temperature of "viewed" wall in 1 (C)
C            3 Surface temperature of mount wall in 1 (C)

      SUBROUTINE CMP51C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)

      COMMON/C6/INDCFG
      COMMON/FVALA/TFA(MCOM),QFA(MCOM)
      COMMON/FVALS/TFS(MCOM,MS),QFS(MCOM)

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP51C'

C Initialize pointers to node(s) INOD
      INOD1=NPCDAT(IPCOMP,9)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First set-up "surroundings"
         IF(INDCFG.EQ.2.OR.NINT(BDATA(IPCOMP,1)).EQ.0) THEN
            call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
            IF(closea) THEN
               TAIR=20.
            ELSE
               TAIR=PCNTMF(IPCOMP)
            END IF
            TWL1=TAIR
            TWL2=TAIR
         ELSE
            IZ=NINT(BDATA(IPCOMP,1))
            TAIR=TFA(IZ)
            TWL1=TFS(IZ,NINT(BDATA(IPCOMP,2)))
            TWL2=TFS(IZ,NINT(BDATA(IPCOMP,3)))
         END IF
         HAIR=BDATA(IPCOMP,4)
         HWL1=BDATA(IPCOMP,5)
         HWL2=BDATA(IPCOMP,6)

C Then initialize acceleration heat Q
         Q=CDATA(IPCOMP,1)

C Establish heat capacity of component mass CM (J/K)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(HAIR+HWL1+HWL2))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self-coupling coefficient
         COUT(1)=ALPHA*(-HAIR-HWL1-HWL2)-CM/TIMSEC
C and then present-time coefficient (ie. right hand side)
         COUT(2)=((1.-ALPHA)*(HAIR+HWL1+HWL2)
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             -ALPHA*HAIR*TAIR
     &             -(1.-ALPHA)*HAIR*PCDATP(IPCOMP,1)
     &             -ALPHA*HWL1*TWL1
     &             -(1.-ALPHA)*HWL1*PCDATP(IPCOMP,2)
     &             -ALPHA*HWL2*TWL2
     &             -(1.-ALPHA)*HWL2*PCDATP(IPCOMP,3)
     &             -ALPHA*Q-(1.-ALPHA)*PCQP(INOD1)

C Store "environment" variables future values
         PCQF(INOD1)=Q
         PCDATF(IPCOMP,1)=TAIR
         PCDATF(IPCOMP,2)=TWL1
         PCDATF(IPCOMP,3)=TWL2

C 1st phase mass (ie. "water") balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=29) mechanical room thermostat'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' Tair   = ',TAIR,' (C)'
            WRITE(ITU,*) ' Twall,1= ',TWL1,' (C)'
            WRITE(ITU,*) ' Twall,2= ',TWL2,' (C)'
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1)
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=2
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP51C'

      RETURN
      END

C ******************** CMP52C ********************

C CMP52C generates for plant component IPCOMP with plant db code 520 ie.
C 1 node (ISV=29) WCH thermostatic radiator valve (sensor only)
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C     BDATA: 1 Index of coupled building zone (-)
C            2 Index of coupled wall in that zone (-)
C            3 Thermal conductance between water and sensor (W/K)
C            4 Equiv. convective conductance to air (W/K)
C            5 Equiv. radiative conductance to wall (W/K)
C            6 Equiv. radiative conductance to radiator (W/K)
C            7 Index of radiator connected to TRV (-)
C     CDATA: none

C     PCDATF/P
C            1 Air temperature of coupled building zone (C)
C            2 Surface temperature of coupled wall in 1 (C)

      SUBROUTINE CMP52C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)

      COMMON/C6/INDCFG
      COMMON/FVALA/TFA(MCOM),QFA(MCOM)
      COMMON/FVALS/TFS(MCOM,MS),QFS(MCOM)

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      logical closea

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP52C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      JPCOMP=NINT(BDATA(IPCOMP,7))
      JNOD1=NPCDAT(JPCOMP,9)
      ICON1=ICONDX(JPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First set-up "surroundings"
         IF(INDCFG.EQ.2.OR.NINT(BDATA(IPCOMP,7)).EQ.0) THEN
           call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
           IF(closea) THEN
               TAIR=20.
            ELSE
               TAIR=PCNTMF(IPCOMP)
            END IF
            TWAL=TAIR
         ELSE
            IZ=INT(BDATA(IPCOMP,1))
            TAIR=TFA(IZ)
            TWAL=TFS(IZ,NINT(BDATA(IPCOMP,2)))
         END IF
         HWAT=BDATA(IPCOMP,3)
         HAIR=BDATA(IPCOMP,4)
         HWAL=BDATA(IPCOMP,5)
         HRAD=BDATA(IPCOMP,6)

C Establish heat capacity of component mass CM (J/K)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(HWAT+HAIR+HWAL+HRAD))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self- and cross-coupling coefficients
         COUT(1)=ALPHA*(-HWAT-HAIR-HWAL-HRAD)-CM/TIMSEC

C and then present-time coefficient (ie. right hand side)
         COUT(2)=((1.-ALPHA)*(HAIR+HWAL+HWAT+HRAD)
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             -ALPHA*HWAT*CSVP(JNOD1,1)
     &             -(1.-ALPHA)*(HWAT)*CSVP(JNOD1,1)
     &             -ALPHA*HRAD*CSVP(JNOD1,1)
     &             -(1.-ALPHA)*(HRAD)*CSVP(JNOD1,1)
     &             -ALPHA*HAIR*TAIR
     &             -(1.-ALPHA)*HAIR*PCDATP(IPCOMP,1)
     &             -ALPHA*HWAL*TWAL
     &             -(1.-ALPHA)*HWAL*PCDATP(IPCOMP,2)

C Store "environment" variables future values
         PCDATF(IPCOMP,1)=TAIR
         PCDATF(IPCOMP,2)=TWAL

C 1st phase mass (ie. "water") balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
         COUT(4)=0.

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
         COUT(4)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=29) WCH thermostatic valve'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' Tair   = ',TAIR,' (C)'
            WRITE(ITU,*) ' Twall  = ',TWAL,' (C)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP52C'

      RETURN
      END

C ******************** CMP73C ********************
C CMP73C generates for plant component IPCOMP with plant db code 730 ie.
C 3 node (ISV>20) WCH 3-port valve. This model supercedes CMP29C, which is
C now depreciated;
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 Component UA modulus (for heat loss to environment) (W/K)
C
C     CDATA: 1 Valve position (0-1)
C 
C This valve model has three nodes: 
C 1 - inlet
C 2 - primary outlet
C 3 - secondary outlet
C The input flow rate m_1 is distributed between the two nodes as follows: 
C node 2 m_2 = a x m_1
C node 3 m_3 = (1-a) x m_1

      SUBROUTINE CMP73C(IPCOMP,COUT,ISTATS)

       use h3kmodule
      implicit none
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/PCTIME/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)

      REAL UA,CM,C1,V_POS,SMALL
     
      INTEGER IMPEXP,IPCOMP

      integer itc,icnt,ITCF,ITRACE,IZNTRC,ITU,IHRP,IHRF,idynow,
     &IDYP,IDYF,IDWP,IDWF,NSINC,ITS,NPCOMP,ICONTP,ISTATS
      integer ICONDX,ICON1,INOD1,INOD2,INOD3,IPN1,IPN2,IPC1,IPC2,NPCON,
     &IPCT,NPCDAT,NAPDAT,NCI,IPOFS1,IPOFS2,IX1,NITMS,I

      real RATIMP,TC,TIMSEC,PCONDR,PCONSD,ADATA,BDATA,CDATA,CSVF,CSVP
      real PCTF,PCRF,PUAF,PCQF,PCNTMF,PCTP,PCRP,PUAP,PCQP,PCNTMP,CONVAR
      real PCDATF,PCDATP,ALPHA,QDATA,PCAOUT,SHTFLD


      PARAMETER (SMALL=1.0E-15)
      REAL COUT(MPCOE)
      LOGICAL CLOSEA


C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP73C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD

C 1 connection
      ICON1=ICONDX(IPCOMP,1,1)

C 3 - nodes 1) water inlet 2) primary outlet m_out1=(a X m_in) 2) secondary outlet m_out2=(1-a) x m_in
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1
      INOD3=NPCDAT(IPCOMP,9)+2

C Check the specified valve position
      V_POS=CDATA(IPCOMP,1)
      IF(V_POS.LT.0.OR.V_POS.GT.1.)THEN
        V_POS=0.
        WRITE(IUOUT,*)
     &' WARNING:CMP73C a 3-port valve control setting <0 or >1'
        WRITE(IUOUT,*)
     &' has been detected. Resetting to valve position to zero.'
      ENDIF

C Generate coefficients for the energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss),
C if there is no containment then zero the UA value.
         UA=ADATA(IPCOMP,3)
         CALL ECLOSE(PCNTMF(IPCOMP),-99.00,0.001,CLOSEA)
         IF(CLOSEA) UA=0.

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Store inter-node fluid capacity rates (W/K).
C Node 2
         PCDATF(IPCOMP,2)=C1*V_POS

C Node 3
         PCDATF(IPCOMP,3)=C1*(1.-V_POS)

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(UA+C1))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

cx        call eclose(C1,0.00,0.0001,closea)
cx        if(closea)alpha=1.

C Establish matrix equation self-and cross-coupling coefficients
C matrix order as follows
C   A  B  C | X     RHS
C   1  0  0 | 6      7
C   2  3  0 | 0    = 8
C   4  0  5 | 0      9

C Self-coupling coefficients.
C Inlet node 1
         COUT(1)=ALPHA*(-C1)-CM/(3.*TIMSEC) 

C Primary outlet 2
         COUT(2)=ALPHA*PCDATF(IPCOMP,2)
         COUT(3)=ALPHA*(-PCDATF(IPCOMP,2)-(UA/2.))-CM/(3.*TIMSEC)

C Secondary outlet node 3
         COUT(4)=ALPHA*PCDATF(IPCOMP,3)
         COUT(5)=ALPHA*(-PCDATF(IPCOMP,3)-(UA/2.))-CM/(3.*TIMSEC)

C Cross coupling coefficient
C Node 1
         COUT(6)=ALPHA*C1 


C 'Known coefficients' present-time coefficient (ie. right hand side
C Node 1.
         COUT(7)=(((1.-ALPHA)*(PCRP(ICON1)))
     &-CM/(3.*TIMSEC))*CSVP(INOD1,1)
     &-(1.-ALPHA)*(PCRP(ICON1))*PCTP(ICON1)


C Node 2. 
         COUT(8)=(((1.-ALPHA)*(PCDATP(IPCOMP,2)+(UA/2.)))
     &           -CM/(3.*TIMSEC))*CSVP(INOD2,1)
     &           -(1.-ALPHA)*(PCDATP(IPCOMP,2))*CSVP(INOD1,1)
     &           -(ALPHA)*(UA/2.)*PCNTMF(IPCOMP)
     &           -(1.-ALPHA)*PUAP(INOD2)*PCNTMP(IPCOMP)

C Node 3.
         COUT(9)=(((1.-ALPHA)*(PCDATP(IPCOMP,3)+(UA/2.)))
     &           -CM/(3.*TIMSEC))*CSVP(INOD3,1)
     &           -(1.-ALPHA)*(PCDATP(IPCOMP,3))*CSVP(INOD1,1)
     &           -(ALPHA)*(UA/2.)*PCNTMF(IPCOMP)
     &           -(1.-ALPHA)*PUAP(INOD3)*PCNTMP(IPCOMP)

C Store "environment" variables future values.
         PUAF(INOD2)=UA/2.
         PUAF(INOD3)=UA/2.
         
C Establish "containment loss" data
        QDATA(IPCOMP)=UA*(alpha*(csvf(inod1,1)-pcntmf(ipcomp))+
     &                (1.-alpha)*(csvp(inod1,1)-pcntmp(ipcomp)))
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iConvective)
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iRadiant)

C 1st phase mass (ie. dry air) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-V_POS
         COUT(3)=1.
         COUT(4)=-(1.-V_POS)
         COUT(5)=1.
         COUT(6)=-PCONDR(ICON1)
         COUT(7)=0.
         COUT(8)=0.
         COUT(9)=0.
         COUT(10)=0.

C 2nd phase mass (ie. vapour) balance coefficients.
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=1.
         COUT(4)=0.
         COUT(5)=1.
         COUT(6)=0.
         COUT(7)=0.
         COUT(8)=0.
         COUT(9)=0.
         COUT(10)=0.
      END IF

C Establish additional output variables
      NAPDAT(IPCOMP)=1

C Valve position
      PCAOUT(IPCOMP,1)=CDATA(IPCOMP,1)

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 3 node (ISV>21) WCH 3-port valve'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2,INOD3
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' V_POS  = ',V_POS,' (0-1)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=9
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) then
        WRITE(ITU,*) ' Leaving subroutine CMP73C trace'
      ENDIF

      RETURN
      END

C ******************** CMP74C ********************
C CMP74C generates for plant component IPCOMP with plant db code 740 ie.
C 3 node (ISV>20) WCH 3-port valve with integrated automatic flow
C control valve matrix equation coefficients COUT (in order:
C self-coupling, cross-coupling, and present-time coefficients) for
C energy balance (ISTATS=1), 1st phase mass balance (ISTATS=2),
C or 2nd phase mass (ISTATS=3).
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 Component UA modulus (for heat loss to environment) (W/K)
C            4 Maximum flow setpoint (l/s)
C
C     BDATA: None
C
C     CDATA: 1 Valve position (0-1)
C
C This valve model has three nodes:
C   1 - inlet
C   2 - primary outlet
C   3 - secondary outlet
C Based on the valve position CDATA and the maximum flow
C setpoint ADATA(4) the flow distribution factor a is established
C according to
C   a = min(CDATA, ADATA(4)/m_1)
C
C The input flow rate m_1 is distributed between the two nodes according
C to the distribution factor a as follows:
C   node 2 m_2 = a x m_1
C   node 3 m_3 = (1-a) x m_1
C
      SUBROUTINE CMP74C(IPCOMP,COUT,ISTATS)
      use h3kmodule
      implicit none
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/PCTIME/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)
      COMMON/PCDAT/PCDATF(MPCOM,MPCDAT),PCDATP(MPCOM,MPCDAT)

      REAL UA,CM,C1,V_POS,SMALL,WMFR

      INTEGER IMPEXP,IPCOMP

      integer itc,icnt,ITCF,ITRACE,IZNTRC,ITU,IHRP,IHRF,idynow,
     &IDYP,IDYF,IDWP,IDWF,NSINC,ITS,NPCOMP,ICONTP,ISTATS
      integer ICONDX,ICON1,INOD1,INOD2,INOD3,IPN1,IPN2,IPC1,IPC2,NPCON,
     &IPCT,NPCDAT,NAPDAT,NCI,IPOFS1,IPOFS2,IX1,NITMS,I

      real RATIMP,TC,TIMSEC,PCONDR,PCONSD,ADATA,BDATA,CDATA,CSVF,CSVP
      real PCTF,PCRF,PUAF,PCQF,PCNTMF,PCTP,PCRP,PUAP,PCQP,PCNTMP,CONVAR
      real PCDATF,PCDATP,ALPHA,QDATA,PCAOUT,SHTFLD

      PARAMETER (SMALL=1.0E-15)
      REAL COUT(MPCOE)
      LOGICAL CLOSEA

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP74C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD

C 1 connection
      ICON1=ICONDX(IPCOMP,1,1)

C 3 - nodes 1) water inlet 2) primary outlet m_out1=(a X m_in) 2)
C     secondary outlet m_out2=(1-a) x m_in
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1
      INOD3=NPCDAT(IPCOMP,9)+2

C Check the specified valve position
      WMFR=PCONDR(ICON1)*CONVAR(ICON1,2)
      V_POS=MIN(CDATA(IPCOMP,1),ADATA(IPCOMP,4)/WMFR)
      IF (V_POS.LT.0) THEN
        V_POS=0.
        WRITE(IUOUT,*)
     &   ' WARNING:CMP74C a 3-port valve control setting <0 has'
        WRITE(IUOUT,*)
     &   ' been detected. Resetting valve position to zero.'
      elseif (V_POS.GT.1.) THEN
        V_POS=1.
        WRITE(IUOUT,*)
     &   ' WARNING:CMP74C a 3-port valve control setting >1 has'
        WRITE(IUOUT,*)
     &   ' been detected. Resetting valve position to one.'
      ENDIF

C Generate coefficients for the energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss),
C if there is no containment then zero the UA value.
         UA=ADATA(IPCOMP,3)
         CALL ECLOSE(PCNTMF(IPCOMP),-99.00,0.001,CLOSEA)
         IF(CLOSEA) UA=0.

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)
         C1=PCONDR(ICON1)*CONVAR(ICON1,2)*SHTFLD(3,CONVAR(ICON1,1))

C Store inter-node fluid capacity rates (W/K).
C Node 2
         PCDATF(IPCOMP,2)=C1*V_POS

C Node 3
         PCDATF(IPCOMP,3)=C1*(1.-V_POS)

C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(UA+C1))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

cx        call eclose(C1,0.00,0.0001,closea)
cx        if(closea)alpha=1.

C Establish matrix equation self-and cross-coupling coefficients
C matrix order as follows
C   A  B  C | X     RHS
C   1  0  0 | 6      7
C   2  3  0 | 0    = 8
C   4  0  5 | 0      9

C Self-coupling coefficients.
C Inlet node 1
         COUT(1)=ALPHA*(-C1)-CM/(3.*TIMSEC)

C Primary outlet 2
         COUT(2)=ALPHA*PCDATF(IPCOMP,2)
         COUT(3)=ALPHA*(-PCDATF(IPCOMP,2)-(UA/2.))-CM/(3.*TIMSEC)

C Secondary outlet node 3
         COUT(4)=ALPHA*PCDATF(IPCOMP,3)
         COUT(5)=ALPHA*(-PCDATF(IPCOMP,3)-(UA/2.))-CM/(3.*TIMSEC)

C Cross coupling coefficient
C Node 1
         COUT(6)=ALPHA*C1


C 'Known coefficients' present-time coefficient (ie. right hand side
C Node 1.
         COUT(7)=(((1.-ALPHA)*(PCRP(ICON1)))
     &-CM/(3.*TIMSEC))*CSVP(INOD1,1)
     &-(1.-ALPHA)*(PCRP(ICON1))*PCTP(ICON1)


C Node 2.
         COUT(8)=(((1.-ALPHA)*(PCDATP(IPCOMP,2)+(UA/2.)))
     &           -CM/(3.*TIMSEC))*CSVP(INOD2,1)
     &           -(1.-ALPHA)*(PCDATP(IPCOMP,2))*CSVP(INOD1,1)
     &           -(ALPHA)*(UA/2.)*PCNTMF(IPCOMP)
     &           -(1.-ALPHA)*PUAP(INOD2)*PCNTMP(IPCOMP)

C Node 3.
         COUT(9)=(((1.-ALPHA)*(PCDATP(IPCOMP,3)+(UA/2.)))
     &           -CM/(3.*TIMSEC))*CSVP(INOD3,1)
     &           -(1.-ALPHA)*(PCDATP(IPCOMP,3))*CSVP(INOD1,1)
     &           -(ALPHA)*(UA/2.)*PCNTMF(IPCOMP)
     &           -(1.-ALPHA)*PUAP(INOD3)*PCNTMP(IPCOMP)

C Store "environment" variables future values.
         PUAF(INOD2)=UA/2.
         PUAF(INOD3)=UA/2.

C Establish "containment loss" data
        QDATA(IPCOMP)=UA*(alpha*(csvf(inod1,1)-pcntmf(ipcomp))+
     &                (1.-alpha)*(csvp(inod1,1)-pcntmp(ipcomp)))
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iConvective)
        call store_plt_gain ( IPCOMP, 0.5*QDATA(IPCOMP), iRadiant)

C 1st phase mass (ie. dry air) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-V_POS
         COUT(3)=1.
         COUT(4)=-(1.-V_POS)
         COUT(5)=1.
         COUT(6)=-PCONDR(ICON1)
         COUT(7)=0.
         COUT(8)=0.
         COUT(9)=0.
         COUT(10)=0.

C 2nd phase mass (ie. vapour) balance coefficients.
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=1.
         COUT(4)=0.
         COUT(5)=1.
         COUT(6)=0.
         COUT(7)=0.
         COUT(8)=0.
         COUT(9)=0.
         COUT(10)=0.
      END IF

C Establish additional output variables
      NAPDAT(IPCOMP)=1

C Valve position
      PCAOUT(IPCOMP,1)=CDATA(IPCOMP,1)

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 3 node (ISV>21) WCH 3-port valve w/ flow cnt'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2,INOD3
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' V_POS  = ',V_POS,' (0-1)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=9
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) then
        WRITE(ITU,*) ' Leaving subroutine CMP74C trace'
      ENDIF

      RETURN
      END

C --------------------------------- CMP79C -----------------------------------
C CMP79C (DHWdraw) calculates an instantaneous hot water draw based on a
C probabilistic model. For each simulation time step the model calculates
C a hot water draw V in l/s. This flow is then passed into the model coefficient
C generator. The model is (loosely) based on the work of Jordan and Vajen in IEA SHC Annex26.
C
C The draw calculation is based on the user supplying a nominal daily draw (litres).
C The user also needs to define the typical % of the total draw consumed in up to
C 6 concurent user-defined periods adding up to 24h.
C
C The user can define up to 6 distinct draw types; these can be individual
C loads such as a shower or grouped loads of simular types. These are characterised by
C 1) a flowrate when in use (l/min)
C 2) a nominal flow duration (min)
C 3) a flow duration standard dev.
C
C For each draw types the user needs to define the typical % of the total daily
C draw attributable to the draw type along with information on any change in draw
C use between weekdays and weekends (e.g. applies to use of baths)
C
C Data on holidays, seasonal variation of the total daily draw also need to be defined.
C During holiday periods the total draw is set to zero.
C
C
C Data input and main variable descriptions (up to 60 items):
C NomDraw               Total Average Nominal HW Draw (l per day)
C SeasDrawFluc          Seasonal draw fluctuation (%)
C SeasPhas              Seasonal phase shift (days)
C
C NoHolPer              # of Holiday Periods (up to 3)
C For Each Holiday Period:
C HolPerS(MHolPer)      Start DOY
C HolPerE(MHolPer)      End DOY
C
C NoDrawPer             # of Concurrent Draw Periods in Day (up to 6)
C For each draw period i
C DrawPerS(MDrawPer)    Start of draw period (hours)
C NomDrawFrac(MDrawPer) Nominal. % of total Nominal. draw taken in period i
C
C
C NDrawType             Number of distinct draw types (up to 6)
C NomTypeFrac[i]        % of total Nominal draw attributable to type i
C DrawV[i]              Draw i flow rate (l/min)
C DrawVstd[i]           Draw i flow rate std dev (l/min)
C NomDrawDur[i]         Draw i nominal flow duration (min)
C DrawWdvar[i]          Draw i weekday probability modifier (-)
C DrawWevar[i]          Draw i weekend probability modifier (-)
C DHWDrawSeed           Random number generator seed (-)
C GroundSourceModel     Ground source model [1 - std. profile 2 - user profile 3 - f(dep
C GroundSourceCoeff     Model coef g0 [1 & 2 - profile number 3 - depth in m]

C ----------------------------------------------------------------------
      SUBROUTINE CMP79C(IPCOMP,COUT,ISTATS)
      use h3kmodule
      implicit none
#include "plant.h"
#include "building.h"
#include "site.h"


      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      integer iuout,iuin,ieout
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PTIME/PTIMEP,PTIMEF

      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)

      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      common/pcnam/pcname(mpcom)       ! Plant component names
      COMMON/ITERINDEX/ITERNU !plant iteration number

C Ground temperature
      common/C25/igttype,GTMP(12)

C Commons for DWHStochastic calculation.
      COMMON/DHWStoch1/DHWsubmin,CalCount,DHWCalcInc,
     &    GroundSourceModel,GroundSourceCoeff
      COMMON/DHWStoch2/DHWdrawV,DHWDrawVp

      Common/DHWStoch3R/NomDraw,SeasDrawFluc,DrawPerS(6),NomDrawFrac(6),
     &NomTypeFrac(6),DrawV(6),DrawVstd(6),DrawWdvar(6),DrawWevar(6)
      Common/DHWStoch3I/SeasPhas,NoHolPer,HolPerS(3),HolPerE(3),
     &NoDrawPer,NDrawType,NomDrawDur(6)

      Common/DHWStoch4/CurDrawDur(6),ElapseDrawDur(6),DHWDraw(6)

      Common/DHWStochL/inDraw(6)

      Common/DHWTrakType/DHWDrawTypeTot(6),DHWDrawTypeTotV(6),
     &DHWDrawTypeTotVp(6)


      DIMENSION NDMonth(12)
      DATA NDMonth/31,28,31,30,31,30,31,31,30,31,30,31/


      REAL COUT(MPCOE)
      character outs*124
      Integer iNameLength,iAddDatNumLen ! temporary variable length

      character*15 pcname
      character*2 char_tmp              ! used for reporting

      logical DHWsubmin,DHWCalc,Weekend,indraw,DisPos


      integer itc,icnt,ITCF,ITRACE,IZNTRC,ITU,IHRP,IHRF,IDYP,
     &IDYF,IDWP,IDWF,NSINC,ITS,NPCOMP,NCI,I,IHol,ITERNU,idynow,
     &NPCDAT,IPOFS1,IPOFS2,ICONTP,ICONDX,NDMonth,igttype,ICON1

      integer IPCOMP,IX1,NAPDAT

      integer INOD1,curMonth,IM,curDoM,ID,ISTATS,iMinS,CalCount,
     &iPer,inPer,NITMS

      integer DHWCalcInc,iDHWinc,GroundSourceModel,GroundSourceCoeff

      real DHWdrawV,DHWDrawVp,FracMin,TSperDHWCalc,DHWDrawTot,DrawV,
     &DHWDraw,DrawPVal,DrawProb,DrawWevar,DrawWdvar,NomDrawFrac,
     &NomDrawDur,Tincr,NomTypeFrac,ActV

      integer IDHWType,HolPerS,HolPerE,NDrawType,NoDrawPer,NoHolPer,
     &SeasPhas

      REAL DHWDrawTypeTot,DHWDrawTypeTotV,DHWDrawTypeTotVp

      real TIMSEC,PTIMEP,PTIMEF,CDATA,PCAOUT,QDATA,
     &CONVAR,GTMP,TG2,TG1,CurrGTMP,PI,SeasDrawFluc,RHOFLD


      real DOY,NomDrawDay,NomDraw,xHol,xWE,DrawPerS,PerDur,
     &ElapseDrawDur,DrawVstd,CurDrawDur,DenS,Rand_No
      integer istat


      PI=22./7.

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP79C'

C Check control data for relevant balance type
      IF(ISTATS.EQ.1.AND.CDATA(IPCOMP,1).LT.0.) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP79C: invalid control data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP79C: unresolvable error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C     Set ground source temperature based on model chosen.
        if (GroundSourceModel.eq.1 .or. GroundSourceModel.eq.2) then

C     Standard ground temperature profile from grdtmp(12,mgrdp), see esru_blk.F,
C     or user defined ground temperature profile from ugrdtp(12,mgrdp).

          if (GroundSourceModel.eq.2) then
            if (NGRDP.le.0) then
              write(*,*) "Error DHWdraw: Option 2 requires user",
     &               " defined ground temperature profile! "
              STOP
            elseif (GroundSourceCoeff.gt.NGRDP) then
              write(*,*) "Error DWHdraw: Ground temperature profile #",
     &               GroundSourceCoeff," not available! "
              STOP
            endif
          endif
C Calculate the current ground temperatures by interpolating the monthly ground
C temperatures.
          CALL EDAYR(IDYP,ID,IM)
          curMonth=IM
          curDoM=ID

          if (GroundSourceModel.eq.1) then
cx            TG1=grdtmp(curMonth,int(GroundSourceCoeff))
            TG1=GTMP(curMonth)
          else
            TG1=ugrdtp(curMonth,int(GroundSourceCoeff))
          endif

          IF(curMonth.EQ.12)THEN
C           Get next month's temperature (assume loop round to January).
            curMonth=1
          ELSE
C           Get next month's temperature
            curMonth=curMonth+1
          ENDIF

          if (GroundSourceModel.eq.1) then
cx            TG2=grdtmp(curMonth,int(GroundSourceCoeff))
            TG2=GTMP(curMonth)
          else
            TG2=ugrdtp(curMonth,int(GroundSourceCoeff))
          endif

C         Calculate today's ground temperature by linear interpolation
          currGTMP=((float(curDoM)/float(NDMonth(curMonth)))
     &                                              *(TG2-TG1))+TG1

        elseif (GroundSourceModel.eq.3) then
C        Calculate ground source temperature in desired depth GroundSourceCoeff
cx         CALL GTCALC(GroundSourceCoeff,'-',IER)
cx         Tgrnd=??
          write(*,*) "Error DHWdraw: Option 3 for ground temperature",
     &                  " not available to date! "
          STOP
        else
C        Ground source temperature option invalid
          write(*,*) "Error DHWdraw: Option for ground temperature",
     &                  " invalid! "
          STOP
        endif

C Establish matrix equation self- and cross-coupling coefficients

C The return flow from this component (if needed) is mains cold water and so the
C 'exit' temperature is set to the interpolated ground temperature.
         COUT(1)=1.
         COUT(2)=0.
C and then present-time coefficient (ie. right hand side)
         COUT(3)=currGTMP


C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN

C Calculate the DHWdraw (l) at this simulation timestep.

C Determine the calculation frequency if timestep < 1 min or the number of draw time increments
C to be calculated if timestep > 1 min.
        IF(TIMSEC.LT.60.)THEN
          DHWSubmin=.true.
          iMinS=NINT(60./TIMSEC)
          FracMin=abs(TIMSEC*float(iMinS)-60.)
          DHWCalcInc=1
          TSperDHWCalc=iMinS
          if(FracMin.gt.1E-2)then
            WRITE(IUOUT,*)' WARNING:CMP79C [DHW] requires the plant'
            WRITE(IUOUT,*)' time step (s) to be an integer multiple or'
            WRITE(IUOUT,*)' divisor of 60. Adjust the plant time   '
            WRITE(IUOUT,*)' step accordingly to avoid errors!'
          endif
        ELSE
          DHWSubmin=.false.
          iMinS=NINT(TIMSEC/60.)
          FracMin=abs(TIMSEC-(60.*float(iMinS)))
          DHWCalcInc=iMinS
          if(FracMin.gt.1E-2)then
            WRITE(IUOUT,*)' WARNING:CMP79C [DHW] requires the plant'
            WRITE(IUOUT,*)' time step (s) to be an integer multiple or'
            WRITE(IUOUT,*)' divisor of 60. Adjust the plant time      '
            WRITE(IUOUT,*)' step accordingly to avoid errors!'
          endif
        ENDIF


C Calculate the draw ONLY at the first calc increment if sim timestep < 1 min ,
C and in the first iteration of a plant time step.
        IF(ITERNU.EQ.1)THEN
          DHWCalc=.true.
          if(DHWsubmin.AND.CalCount.EQ.1)then
            DHWCalc=.true.
            CalCount=CalCount+1
          elseif(DHWsubmin.AND.CalCount.GT.1)then
            DHWCalc=.false.
            CalCount=CalCount+1
            if(CalCount.GT.TSperDHWCalc)then
              CalCount=1
            endif
          endif
        ELSE
          DHWCalc=.FALSE.
        ENDIF

C Calculate the current DHWDraw over the current time step. If time step > 1 min then
C the routine will loop N times and will calculate the equivalent average draw during the time
C step.
        if(DHWcalc)then

C Calculate general info required for calculation
C Day of year [needed for average flow modifier]
          DOY=float(iDYP)

C Calculate nominal draw for today, accounting for seasonal variablity
          NomDrawDay=NomDraw*
     &(1.+(SeasDrawFluc/100.)*(cos(2*PI*(1/365.)
     &*DOY-(SeasPhas*(2*PI/365.)))))

          if(NomDrawDay.LT.0.0)then
             NomDrawDay=0.0
             WRITE(IUOUT,*)' WARNING:CMP79C [DHW] a nominal draw < 0.0'
             WRITE(IUOUT,*)' has been calculated. Resetting to zero'
          endif


C Check for weekend [flow modifier]
          if(IDWP.GT.5)then
            Weekend=.true.
          else
            Weekend=.false.
          endif

C Check for holiday [flow modifier]
          xHol=1.0
          do 5 iHol=1,noHolPer
            if(iDYP.GE.HolPerS(iHol).AND.iDYP.LT.HolPerE(iHol))then
              xHol=0.0
            endif
   5      continue


C Work out the current draw period
          do 7 iPer=1,NoDrawPer-1
            if(PTIMEP.GE.DrawPerS(iPer).AND.
     &      PTIMEP.LT.DrawPerS(iPer+1))then
              inPer=iPer
              PerDur=(DrawPerS(inPer+1)-DrawPerS(inPer))*60.
            endif
  7       continue

C Check if in final draw period
          if(PTIMEP.GE.DrawPerS(NoDrawPer))then
            inPer=NoDrawPer
            PerDur=(24.0-DrawPerS(NoDrawPer))*60.
          endif


          DHWDrawTot=0.0
          do 10 iDHWinc=1,DHWCalcInc !increment loop if simulation TS > 1 min
            do 101 iDHWType=1,NDrawType !Types loop

C Check to see if the component is in the process of a multi-timestep draw
              if(indraw(iDHWType))then
                ElapseDrawDur(iDHWType)=ElapseDrawDur(iDHWType)+1.0
                if(ElapseDrawDur(iDHWType).GT.CurDrawDur(iDHWType))then
                  DHWDraw(iDHWType)=0.0
                  indraw(iDHWType)=.false.
                  ElapseDrawDur(iDHWType)=0.0
                  CurDrawDur(iDHWType)=0.0
                else
                  indraw(iDHWType)=.true.
                endif
              else
C Not currently in a draw, check to see if a draw is made and calculate the draw characteristics
C if required.
                DHWDraw(iDHWType)=0.0

                if(weekend)then
                  xWE=DrawWevar(iDHWType)
                else
                  xWE=DrawWdvar(iDHWType)
                endif

C Calculate the draw probability (all times in minutes)
                DrawProb=((NomDrawFrac(inPer)/100.)*
     &          ((NomTypeFrac(iDHWType)/100.)*NomDrawDay*xHol*xWE))/
     &          (DrawV(iDHWType)*PerDur*NomDrawDur(iDHWType))

                 DrawPVal=Rand_No()

                if(DrawPVal.LE.DrawProb)then

C Set the draw to the nominal value.
                  DHWDraw(iDHWType)=DrawV(iDHWType)
                  Tincr=0.1

C Calculate the actual draw flow rate based on a normal distribution about the mean
C As flow cannot be zero force a lognormal distribution if the range goes below zero
                  DisPos=.true.
                  call StdDisVal(DrawV(iDHWType),
     &DrawVstd(iDHWType),Tincr,ActV,DisPos)
                  DHWDraw(iDHWType)=ActV

c Calculate the time duration of the current draw and set the inDraw varaible if the duration > 1min.
C This allows draw to persist over multiple time steps.
                  ElapseDrawDur(iDHWType)=0.0
                  CurDrawDur(iDHWType)=NomDrawDur(iDHWType)
                  if(CurDrawDur(iDHWType).gt.1.0)then
                    ElapseDrawDur(iDHWType)=ElapseDrawDur(iDHWType)+1.0
                    inDraw(iDHWType)=.true.
                  else
                    inDraw(iDHWType)=.false.
                  endif
                else
                  DHWDraw(iDHWType)=0.0
                endif

              endif

C Add to the calculated total draw to the totals for tracking draw from each type.
              DHWDrawTypeTot(iDHWType)=DHWDrawTypeTot(iDHWType)
     &+DHWDraw(iDHWType)

              DHWDrawTot=DHWDrawTot+DHWDraw(iDHWType)

  101       continue
  10      continue

C Overwrite the previous time step DHW draw with the average (convert l/min -> l/s) of the calculated
C draw over the plant simulation time step
          DHWDrawV=DHWDrawTot/(60.*float(DHWCalcInc))
          DHWDrawVp=DHWDrawV
          DHWDrawTot=0.0
          DO 105 iDHWType=1,NDrawType
            DHWDrawTypeTotV(iDHWType)=
     &DHWDrawTypeTot(iDHWType)/(60.*float(DHWCalcInc))
            DHWDrawTypeTotVp(iDHWType)=DHWDrawTypeTotV(iDHWType)
            DHWDrawTypeTot(iDHWType)=0.0
  105     Continue

        else

C Don't need to calculate draw as plant simulation TS is < 1 min,
C use previously calculated value instead.
          DHWDrawV=DHWDrawVp

          DO 107 iDHWType=1,NDrawType
            DHWDrawTypeTotV(iDHWType)=DHWDrawTypeTotVp(iDHWType)
  107     Continue

        endif !end on DHW calc check



C Fluid density
       DenS=RHOFLD(3,CONVAR(ICON1,1))

       COUT(1)=1.
       COUT(2)=0.
       COUT(3)=DHWDrawV*(DenS/1000.)

C Store individual type flows as addtitional output.
        NAPDAT(IPCOMP)=6
        DO 109 iDHWType=1,NDrawType
          PCAOUT(IPCOMP,iDHWType)=
     &                DHWDrawTypeTotV(iDHWType)*(DenS/1000.)
  109   CONTINUE

C------------------------------------------------------
C XML output
C------------------------------------------------------
C.....Get component name's length
        iNameLength = lnblnk(pcname(IPCOMP))

C.....Format string as: 'plant/NAME/misc_data/DHW_draw_stoch'
        Call AddToReport(rvPltDHWDrawStoch%Identifier,
     &         DHWDrawV*(DenS/1000.),
     &         pcname(IPCOMP)(1:iNameLength))

        DO 111 iDHWType=1,NDrawType
             if (iDHWType<10) then  ! First nine draw types.
               iAddDatNumLen = 1
               write(char_tmp,'(I1)')iDHWType
             else           ! The max. num. of types is currently only 6
               iAddDatNumLen = 2
               write(char_tmp,'(I2)')iDHWType
             endif

C.....Format string as: 'plant/NAME/misc_data/DHW_draw_stoch/type_#'
            Call AddToReport(rvPltDHWDrawStochTp%Identifier,
     &               DHWDrawTypeTotV(iDHWType)*(DenS/1000.),
     &               pcname(IPCOMP)(1:iNameLength),
     &               char_tmp(1:iAddDatNumLen))
  111   Continue

c      write(87,*)'DHW: mass flow rate '
c      write(87,*)'kg/s | rho ', DHWDrawV*(DenS/1000.), DenS

C 2nd phase mass (ie. vapour) balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
      END IF



C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH Stochastic DHW draw'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' GTMP       = ',currGTMP,' (C)'
            WRITE(ITU,*) ' DHWCalc    = ',DHWCalc,' (-)'
            WRITE(ITU,*) ' DHWSubmin  = ',DHWSubmin,' (-)'
            WRITE(ITU,*) ' DHWCalcInc = ',DHWCalcInc,' (-)'
            WRITE(ITU,*) ' DOY        = ',DOY,' (-)'
            WRITE(ITU,*) ' NomDraw    = ',NomDraw,' (l)'
            WRITE(ITU,*) ' NomDrawDay = ',NomDrawDay,' (l)'
            WRITE(ITU,*) ' DoW        = ',IDWP,' (-)'
            WRITE(ITU,*) ' Weekend    = ',Weekend,' (-)'
            WRITE(ITU,*) ' Holiday    = ',xHol,' (-)'
            WRITE(ITU,*) ' # Periods  = ',NoDrawPer,'(-)'
            WRITE(ITU,*) ' # Types    = ',NDrawType,'(-)'
            WRITE(ITU,*) ' Period     = ',inPer,'(-)'
            WRITE(ITU,*) ' DHWDraw    = ',inPer,'(l)'
            WRITE(ITU,*) ' DHWDrawV   = ',DHWDrawV,'(l/s)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP79C'

      RETURN
      END



C ******************** CMP90C ********************

C CMP90C generates for plant component IPCOMP with plant db code 900 ie.
C 2 node (ISV>19)     air & water temperature source
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: none
C     BDATA: none
C     CDATA: 1 air temperature (C)
C            2 water temperature (C)

      SUBROUTINE CMP90C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      REAL      COUT(MPCOE)

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP90C'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      ICON2=ICONDX(IPCOMP,2,1)
      INOD1=NPCDAT(IPCOMP,9)
      INOD2=NPCDAT(IPCOMP,9)+1

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C Establish matrix equation self-coupling coefficients,
         COUT(1)=1.
         COUT(2)=1.
C then matrix equation cross-coupling coefficients,
         COUT(3)=0.
         COUT(4)=0.
C and then present-time coefficients (ie. right hand sides)
         COUT(5)=CDATA(IPCOMP,1)
         COUT(6)=CDATA(IPCOMP,2)

C 1st phase mass (ie. dry air) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=-PCONDR(ICON1)
         COUT(4)=-PCONDR(ICON2)
         COUT(5)=0.
         COUT(6)=0.

C 2nd phase mass (ie. vapour) balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=1.
         COUT(3)=-PCONDR(ICON1)
         COUT(4)=-PCONDR(ICON2)
         COUT(5)=0.
         COUT(6)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 2 node (ISV>19) air & water temperature source'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1,INOD2
         WRITE(ITU,*) ' Connection(s)  ',ICON1,ICON2
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CDATA1 = ',CDATA(IPCOMP,1)
            WRITE(ITU,*) ' CDATA2 = ',CDATA(IPCOMP,2)
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=6
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP90C'

      RETURN
      END

c ******************** CMP91C ********************

C CMP91C generates for plant component IPCOMP with plant db code 910 ie.
c 2 node (ISV>19) imaginary building-like plant load
c matrix equation coefficients COUT (in order: self-coupling, cross-
c coupling, and present-time coefficients) for energy balance (ISTATS=1),
c 1th phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
c     adata: 1 Total mass (kg)
c            2 Mass weighted average specific heat (J/kgK)
c            3 Wall U value (W/m^2K)
c            4 Total surface area of walls (m^2)
c            5 zone space volume (m^3)
c     bdata: 1 Inside heat transfer coefficient (W/m^2K)
c            2 Outside heat transfer coefficient (W/m^2K)
c            3 Air changes per hour
c     cdata: 1 Heat gain load (W)

      subroutine cmp91c(ipcomp,cout,istats)
#include "plant.h"
#include "building.h"

      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu

      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/pctime/timsec
      common/pctc/tc(mpcom)

      common/pcequ/impexp,ratimp

      common/c9/npcomp,nci(mpcom),cdata(mpcom,mmiscd)
      common/c10/npcon,ipc1(mpcon),ipn1(mpcon),ipct(mpcon),
     &           ipc2(mpcon),ipn2(mpcon),pcondr(mpcon),PCONSD(MPCON,2)
      common/c12ps/npcdat(mpcom,9),ipofs1(mcoefg),ipofs2(mcoefg,mpvar)
      common/pdbdt/adata(mpcom,madata),bdata(mpcom,mbdata)
      common/pcval/csvf(mpnode,mpvar),csvp(mpnode,mpvar)
      common/pcvar/pctf(mpcon),pcrf(mpcon),puaf(mpnode),pcqf(mpnode),
     &             pcntmf(mpcom),
     &             pctp(mpcon),pcrp(mpcon),puap(mpnode),pcqp(mpnode),
     &             pcntmp(mpcom)
      common/pcond/convar(mpcon,mconvr),icontp(mpcon),
     &             icondx(mpcom,mnodec,mpconc)

      COMMON/CLIMIP/QFPP,QFFP,TPP,TFP,QDPP,QDFP,VPP,VFP,DPP,DFP,HPP,HFP
      COMMON/CLMPHG/HEXTPP,HEXTFP,GEXTPP,GEXTFP,TWBPP,TWBFP

      parameter (small=1.0e-15)
      real      cout(mpcoe)
      logical closea

c Trace output
      if(itc.gt.0.and.nsinc.ge.itc.and.nsinc.le.itcf.and.
     &   ITRACE(37).ne.0) write(itu,*) ' Entering subroutine CMP91C'

c Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      icon1=icondx(ipcomp,2,1)
      inod1=npcdat(ipcomp,9)
      inod2=npcdat(ipcomp,9)+1

c Establish air changes per hour (acph) in zone.
      acph=bdata(ipcomp,3)

c Evaluate air leakage flow rate and air specific heat capcity
c based on whether air going in (acph +ve) or out (acph -ve).
      if (acph.gt.0.0) then
         amdotl=rhofld(1,tfp)*acph*adata(ipcomp,5)/3600.
         cpa=shtfld(1,tfp)
         hmdr=gextfp
      else if(acph.lt.0.0) then
         amdotl=rhofld(1,csvf(inod2,1))*acph*adata(ipcomp,5)/3600.
         cpa=shtfld(1,csvf(inod2,1))
         call eclose(csvf(inod2,2),0.00,0.0001,closea)
         if(closea) then
            hmdr=gextfp
         else
            hmdr=csvf(inod2,3)/csvf(inod2,2)
         endif
      endif

c Generate coefficients for energy balance equation
      if(istats.eq.1) then

c Establish heat capacity of component mass CM (J/K) and
c fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         cm=adata(ipcomp,1)*adata(ipcomp,2)
         c1=pcondr(icon1)*convar(icon1,2)*shtfld(1,convar(icon1,1))+
     &      pcondr(icon1)*convar(icon1,3)*shtfld(2,convar(icon1,1))

c Establish thermal resistances for inside and outside.
         U=adata(ipcomp,3)
         hi=bdata(ipcomp,1)
         ho=bdata(ipcomp,2)
         ri=adata(ipcomp,4)/(1./hi+1./(2.*U))
         ru=adata(ipcomp,4)/(1./ho+1./(2.*U))
         call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
         IF(closea) ru=0.0

c Calculate heat gain in zone.
         qg=cdata(ipcomp,1)


c Calculate current component time-constant TC
         tc(ipcomp)=cm/amax1(small,(ri+ru))

c Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         if(impexp.eq.1) then
            alpha=1.
         else if(impexp.eq.2) then
            alpha=ratimp
         else if(impexp.eq.3) then
            if(timsec.gt.0.63*tc(ipcomp)) then
               alpha=1.
            else
               alpha=ratimp
            end if
         else if(impexp.eq.4) then
            cm=0.
            alpha=1.
         end if

c Establish matrix equation self- and cross-coupling coefficients
c Do wall first.
         cout(1)=alpha*(-ri-ru)-cm/timsec
         cout(2)=alpha*ri

c then air.
         cout(3)=ri
         cout(4)=-c1-ri-amdotl*cpa

c Cross coupling.
         cout(5)=c1

c and then present-time coefficient (ie. right hand side)
         cout(6)=((alpha-1.)*(-ri-ru)
     &              -cm/timsec)*csvp(inod1,1)
     &             +((alpha-1.)*ri*csvp(inod2,1))
     &             -(alpha*ru*pcntmf(ipcomp))
     &             +((alpha-1.)*ru*pcntmp(ipcomp))
         cout(7)=-qg-amdotl*cpa*pcntmf(ipcomp)

c 1th phase mass (ie. dry air) balance coefficients
      else if(istats.eq.2) then
         cout(1)=1.
         cout(2)=0.
         cout(3)=0.
         cout(4)=1.
         cout(5)=-pcondr(icon1)
         cout(6)=0.
         cout(7)=amdotl

c 2nd phase mass (ie. vapour) balance coefficients
      else if(istats.eq.3) then
         cout(1)=1.
         cout(2)=0.
         cout(3)=0.
         cout(4)=1.
         cout(5)=-pcondr(icon1)
         cout(6)=0.
         cout(7)=amdotl*hmdr
      end if

c Trace output
      if(itc.gt.0.and.nsinc.ge.itc.and.nsinc.le.itcf.and.
     &   itrace(37).ne.0) then
         write(itu,*) ' Component      ',ipcomp,':'
         write(itu,*) ' 2 node (ISV>19) Building zone'
         write(itu,*) ' Matrix node(s) ',inod1,inod2
         write(itu,*) ' Connection(s)  ',icon1
         if(istats.eq.1) then
            write(itu,*) ' CM     = ',cm,' (J/K)'
            write(itu,*) ' C1     = ',c1,' (W/K)'
            write(itu,*) ' TC     = ',TC(IPCOMP),' (s)'
            write(itu,*) ' ALPHA  = ',alpha,' (-)'
            write(itu,*) ' U      = ',u,' (W/K)'
            write(itu,*) ' qg     = ',qg,' (W)'
            write(itu,*) ' PCNTMF = ',pcntmf(ipcomp),' (C)'
            write(itu,*) ' ri     = ',ri,' (W/m^2K)'
            write(itu,*) ' ru     = ',ru,' (W/m^2K)'
         end if
         write(itu,*) ' Matrix coefficients for ISTATS = ',istats
         nitms=7
         write(itu,*) (cout(i),i=1,nitms)
         if(itu.eq.iuout) then
            ix1=(ipcomp/4)*4
            if(ix1.eq.ipcomp.or.ipcomp.eq.npcomp) call epagew
         end if
      end if

      if(itc.gt.0.and.nsinc.ge.itc.and.nsinc.le.itcf.and.
     &   ITRACE(37).ne.0) write(itu,*) ' Leaving subroutine CMP91C'

      return
      end

C ***********************************************************************
C ***************************** ReadHPCoefDataFile **********************
C Created by: Achim Geissler
C Initial Creation Date: February 11, 2020
C Copyright FHNW
C
C This subroutine is based on ebld/spmatl.F::ReadDSCDataFile. It reads
C the user-specified inverter controlled ground source heat pump
C performance data file, containing a header line and three columns of
C data. NX and NY are the number of data points for tgrnd and tfeed,
C respectively.
C
C Format:
C NX, NY
C Tgrnd1,  Tfeed1,  HPheat0-1, HPheat100-1, HPcool0-1, HPcool100-1
C Tgrnd1,  Tfeed2,  HPheat0-2, HPheat100-2, HPcool0-2, HPcool100-2
C ...
C Tgrnd1,  TfeedNY, HPheat0-N, HPheat100-N, HPcool0-N, HPcool100-N
C Tgrnd2,  Tfeed1,  HPheat0-1, HPheat100-1, HPcool0-1, HPcool100-1
C ...
C ...
C TgrndNX, TfeedNY, HPheat0-N, HPheat100-N, HPcool0-N, HPcool100-N
C
C This subroutine is executed once per simulation, prior to the time-step
C calculations.
C -----------------------------------------------------------------------
      SUBROUTINE ReadHPCoefDataFile(ipcomp)
      IMPLICIT NONE

#include "building.h"
#include "plant.h"

      integer ipcomp

C---------------------------------------------------------------------------------
C ESP-r commons
C---------------------------------------------------------------------------------
      COMMON/FILEP/IFIL
      INTEGER  IFIL

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER  IUOUT,IUIN,IEOUT

C----------------
C icGSHP data file commons
C----------------
C Heat pump performance data / spline function data
      common/icgshpspline/
     &          tground(MPCOM,MPCDAT),
     &          tfeed(MPCOM,MPCDAT),
     &          heat0(MPCOM,MPCDAT,MPCDAT),
     &          heat100(MPCOM,MPCDAT,MPCDAT),
     &          cool0(MPCOM,MPCDAT,MPCDAT),
     &          cool100(MPCOM,MPCDAT,MPCDAT),
     &          nx(MPCOM),ny(MPCOM),
     &          heat0_2d(MPCOM,MPCDAT,MPCDAT),
     &          heat100_2d(MPCOM,MPCDAT,MPCDAT),
     &          cool0_2d(MPCOM,MPCDAT,MPCDAT),
     &          cool100_2d(MPCOM,MPCDAT,MPCDAT),
     &          fnamHPCoefdat(MPCOM)
      real tground    ! Ground source temperature data values, degC
      real tfeed      ! Feed temperature values, degC
      real heat0      ! HP heating power at lowest converter frequency, kW
      real heat100    ! HP heating power at highest converter frequency, kW
      real cool0      ! HP cooling power at lowest converter frequency, kW
      real cool100    ! HP cooling power at highest converter frequency, kW
      real heat0_2d   ! HP heating power 0 % second-derivatives
      real heat100_2d ! HP heating power 100 % second-derivatives
      real cool0_2d   ! HP cooling power 0 % second-derivatives
      real cool100_2d ! HP cooling power 100 % second-derivatives

      integer nx,ny              ! number of data points in x and y directions

      character fnamHPCoefdat*72 ! Data file name for current icGSHP entry

C---------------------------------------------------------------------------------
C Declare local variables.
C---------------------------------------------------------------------------------
      INTEGER ND,K
      integer iFIL_ascii  ! temporary local file number

      CHARACTER OUTSTR*124

      LOGICAL fclerror

      INTEGER ISTAT,IER,I
      INTEGER ix, iy

      integer lnblnk  ! function definition

C---------------------------------------------------------------------------------
C Read the header information from the user-specified data file (.hpd)
C and perform some checks to ensure that the correct number of data
C are specified. Note that this does not check the validity of the data,
C but rather ensures that the correct number of data items are specified.
C The existence of the .hpd file was confirmed when the .cfg file was read.
C ((then why check below??))
C `fclerror' is a flag indicating whether there were errors reading the .hpd file.
C---------------------------------------------------------------------------------
C-----Notify user that data file is being processed (this can take some time).
      call usrmsg(' Processing icGSHP data input file...',
     &            ' ','-')
C-----Open the .hpd file.
      fclerror = .false.
      iFIL_ascii = IFIL+38  ! Should be a safe unit number to use.
      CALL ERPFREE(iFIL_ascii,ISTAT)
      CALL EFOPSEQ(iFIL_ascii,fnamHPCoefdat(ipcomp),1,IER)
      IF(IER /= 0)THEN
        WRITE(IUOUT,*) ' Error opening icGSHP data input file ',
     &    fnamHPCoefdat(ipcomp),', plant node ',
     &    ipcomp,'.'
        fclerror = .true.
      ENDIF

C-----Read the number of data points.
      CALL STRIPC(iFIL_ascii,OUTSTR,99,ND,1,' icGSHP data input',IER)
      IF(IER /= 0)THEN
        WRITE(IUOUT,*) ' Error reading icGSHP data input file ',
     &    fnamHPCoefdat(ipcomp)(1:lnblnk(fnamHPCoefdat(ipcomp))),
     &    ', plant node ',ipcomp,'.'
        fclerror = .true.
      ENDIF
C     << check ND? >>
      K=0
      CALL EGETWI(OUTSTR,K,nx(ipcomp),6,MPCDAT,'-',
     &                       ' Number of icGSHP x data points ',IER)
      IF(IER /= 0)THEN
        WRITE(IUOUT,*)
     &    ' Error reading number of icGSHP tgrnd data points; file ',
     &    fnamHPCoefdat(ipcomp)(1:lnblnk(fnamHPCoefdat(ipcomp))),
     &    ', plant node ',ipcomp,'.'
        fclerror = .true.
      ENDIF
      CALL EGETWI(OUTSTR,K,ny(ipcomp),6,MPCDAT,'-',
     &                       ' Number of icGSHP y data points ',IER)
      IF(IER /= 0)THEN
        WRITE(IUOUT,*)
     &    ' Error reading number of icGSHP tfeed data points; file ',
     &    fnamHPCoefdat(ipcomp)(1:lnblnk(fnamHPCoefdat(ipcomp))),
     &    ', plant node ',ipcomp,'.'
        fclerror = .true.
      ENDIF

C-----Error handling on reading of .fcl file.
      IF(fclerror)THEN
        STOP ' Fatal error in icGSHP input file.'
      ENDIF

C---------------------------------------------------------------------------------
C     Read the data.
C---------------------------------------------------------------------------------

      DO 20 ix=1,nx(ipcomp)
        DO 30 iy=1,ny(ipcomp)
C---------Read the data from the ASCII file.
          CALL STRIPC(iFIL_ascii,OUTSTR,99,ND,1,'the raw data line',IER)
          IF(IER /= 0)THEN
            WRITE(IUOUT,'(3a,i3,a,i3,a)')
     &        ' Error reading icGSHP data from file ',
     &        fnamHPCoefdat(ipcomp)(1:lnblnk(fnamHPCoefdat(ipcomp))),
     &        ', plant node ',ipcomp,', data line ',ix*iy,'.'
          ENDIF
C         << check ND? >>
          K=0
          CALL EGETWR(OUTSTR,K,tground(ipcomp,ix),0.,0.,'-','tground',
     &                                                            IER)
          CALL EGETWR(OUTSTR,K,tfeed(ipcomp,iy),0.,0.,'-','tfeed',IER)
          CALL EGETWR(OUTSTR,K,heat0(ipcomp,ix,iy),0.,0.,'-','h0',IER)
          CALL EGETWR(OUTSTR,K,heat100(ipcomp,ix,iy),0.,0.,'-','h100',
     &                                                            IER)
          CALL EGETWR(OUTSTR,K,cool0(ipcomp,ix,iy),0.,0.,'-','c0',IER)
          CALL EGETWR(OUTSTR,K,cool100(ipcomp,ix,iy),0.,0.,'-','c100',
     &                                                            IER)

 30     CONTINUE
 20   CONTINUE

C Output data matrices for checking purposes.
      write(IUOUT,*)' '
      write(IUOUT,*)
     &  '********************** Data Matrices ***********************'
      write(IUOUT,*)' '
      write(IUOUT,*)'Heating power at 0 %:'
      write(IUOUT,'(a,7(f7.3,1x))')'        ',(tfeed(IPCOMP,iy),iy=1,7)
      DO 40 ix=1,nx(ipcomp)
          write(IUOUT,'(f7.3,1x,7(f7.3,1x))')tground(IPCOMP,ix),
     &                                 (heat0(IPCOMP,ix,iy),iy=1,7)
 40   CONTINUE

      write(IUOUT,*)' '
      write(IUOUT,*)'Heating power at 100 %:'
      write(IUOUT,'(a,7(f7.3,1x))')'        ',(tfeed(IPCOMP,iy),iy=1,7)
      DO 50 ix=1,nx(ipcomp)
          write(IUOUT,'(f7.3,1x,7(f7.3,1x))')tground(IPCOMP,ix),
     &                                 (heat100(IPCOMP,ix,iy),iy=1,7)
 50   CONTINUE

      write(IUOUT,*)' '
      write(IUOUT,*)'Cooling power at 0 %:'
      write(IUOUT,'(a,7(f7.3,1x))')'        ',(tfeed(IPCOMP,iy),iy=1,7)
      DO 60 ix=1,nx(ipcomp)
          write(IUOUT,'(f7.3,1x,7(f7.3,1x))')tground(IPCOMP,ix),
     &                                 (cool0(IPCOMP,ix,iy),iy=1,7)
 60   CONTINUE

      write(IUOUT,*)' '
      write(IUOUT,*)'Cooling power at 100 %:'
      write(IUOUT,'(a,7(f7.3,1x))')'        ',(tfeed(IPCOMP,iy),iy=1,7)
      DO 70 ix=1,nx(ipcomp)
          write(IUOUT,'(f7.3,1x,7(f7.3,1x))')tground(IPCOMP,ix),
     &                                 (cool100(IPCOMP,ix,iy),iy=1,7)
 70   CONTINUE
      write(IUOUT,*)' '
      write(IUOUT,*)
     &  '************************************************************'
      write(IUOUT,*)' '

C Processing complete. Leave binary file open for use during time-step simulation.
      call usrmsg(' Processing icGSHP data input file... done.',
     &            ' ','-')

      RETURN
      END

C************************************************************************
C     N U M E R I C A L    R E C I P E S
C************************************************************************
C
C Routines splie2() and splin2() from Numerical Recipes in Fortran 77.
C
C========================================================================
      SUBROUTINE splie2(x1a,x2a,ya,m,n,y2a)
C     USES spline()
#include "plant.h"
C
C Given an m by n tabulated function ya(1:m,1:n), and tabulated independent variables
C x2a(1:n), this routine constructs one-dimensional natural cubic splines of the rows of ya
C and returns the second-derivatives in the array y2a(1:m,1:n). (The array x1a is included
C in the argument list merely for consistency with routine splin2.)

C Subroutine parameters
      INTEGER:: m,n
      REAL:: x1a(m),x2a(n),y2a(m,n),ya(m,n)

C------------------------------------------------------------------------
C ESP-r commons
C------------------------------------------------------------------------
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER  IUOUT,IUIN,IEOUT

C------------------------------------------------------------------------
C Local variables
C------------------------------------------------------------------------
      INTEGER:: j,k,ix,iy
      REAL:: y2tmp(MPCDAT),ytmp(MPCDAT)

      REAL:: dd(7,7)

      do 13 j=1,m
        do 11 k=1,n
          ytmp(k)=ya(j,k)
 11     continue

        call spline(x2a,ytmp,n,1.e30,1.e30,y2tmp) ! Values 1x10^30 signal a natural spline.

        do 12 k=1,n
          y2a(j,k)=y2tmp(k)
 12     continue

 13   continue

      return
      END

C========================================================================
      SUBROUTINE splin2(x1a,x2a,ya,y2a,m,n,x1,x2,y)
C     USES spline(),splint()
#include "plant.h"
C
C Given x1a, x2a, ya, m, n as described in splie2 and y2a as produced by that routine;
C and given a desired interpolating point x1,x2; this routine returns an interpolated function
C value y by bicubic spline interpolation.

C Subroutine parameters
      INTEGER:: m,n
      REAL:: x1,x2,y,x1a(m),x2a(n),y2a(m,n),ya(m,n)

C------------------------------------------------------------------------
C Local variables
C------------------------------------------------------------------------

      INTEGER:: j,k
      REAL:: y2tmp(MPCDAT),ytmp(MPCDAT),yytmp(MPCDAT)

      do 12 j=1,m
C       Perform m evaluations of the row splines constructed by splie2,
C       using the onedimensional spline evaluator splint.
        do 11 k=1,n
          ytmp(k)=ya(j,k)
          y2tmp(k)=y2a(j,k)
 11     continue

        call splint(x2a,ytmp,y2tmp,n,x2,yytmp(j))

 12   continue

      call spline(x1a,yytmp,m,1.e30,1.e30,y2tmp) ! Construct the one-dimensional column spline
      call splint(x1a,yytmp,y2tmp,m,x1,y)        ! and evaluate it.

      return
      END

C========================================================================
      SUBROUTINE polint(xa,ya,n,x,y,dy)
#include "plant.h"
C Given arrays xa and ya, each of length n, and given a value x, this routine returns a
C value y, and an error estimate dy. If P(x) is the polynomial of degree N − 1 such that
C P(xai) = yai; i = 1; : : : ;n, then the returned value y = P(x).

C Subroutine parameters
      INTEGER n
      REAL dy,x,y,xa(n),ya(n)

C------------------------------------------------------------------------
C ESP-r commons
C------------------------------------------------------------------------
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER  IUOUT,IUIN,IEOUT

C------------------------------------------------------------------------
C Local variables
C------------------------------------------------------------------------
      REAL den,dif,dift,ho,hp,w,c(MPCDAT),d(MPCDAT)
      INTEGER i,m,ns

      ns=1
      dif=abs(x-xa(1))

      do 11 i=1,n ! Here we find the index ns of the closest table entry,
        dift=abs(x-xa(i))
        if (dift.lt.dif) then
          ns=i
          dif=dift
        endif
        c(i)=ya(i) ! and initialize the tableau of c's and d's.
        d(i)=ya(i)
 11   continue

      y=ya(ns) ! This is the initial approximation to y.
      ns=ns-1

      do 13 m=1,n-1   ! For each column of the tableau,
        do 12 i=1,n-m ! we loop over the current c's and d's and update them.
          ho=xa(i)-x
          hp=xa(i+m)-x
          w=c(i+1)-d(i)
          den=ho-hp
          if (den.eq.0.) then ! pause 'failure in polint'
C           This error can occur only if two input xa's are (to within
C           roundoff) identical.
            WRITE(IUOUT,'(a,f10.4,a)')' *** Failure in polint!'
          endif
          den=w/den
          d(i)=hp*den ! Here the c's and d's are updated.
          c(i)=ho*den
 12     continue

        if (2*ns.lt.n-m)then
C         After each column in the tableau is completed, we decide which
C         correction, c or d, we want to add to our accumulating value
C         of y, i.e., which path to take through the tableau -- forking
C         up or down. We do this in such a way as to take the most
C         "straight line" route through the tableau to its apex, updating
C         ns accordingly to keep track of where we are. This route keeps
C         the partial approximations centered (insofar as possible) on
C         the target x. The last dy added is thus the error indication.
          dy=c(ns+1)
        else
          dy=d(ns)
          ns=ns-1
        endif

        y=y+dy

 13   continue

      return
      END

C========================================================================
      SUBROUTINE polin2(x1a,x2a,ya,m,n,x1,x2,y,dy)
C     USES polint
#include "plant.h"
C Given arrays x1a(1:m) and x2a(1:n) of independent variables, and an
C m by n array of function values ya(1:m,1:n), tabulated at the grid
C points defined by x1a and x2a; and given values x1 and x2 of the
C independent variables; this routine returns an interpolated function
C value y, and an accuracy indication dy (based only on the interpolation
C in the x1 direction, however).

C Subroutine parameters
      INTEGER m,n
      REAL dy,x1,x2,y,x1a(m),x2a(n),ya(m,n)

C------------------------------------------------------------------------
C Local variables
C------------------------------------------------------------------------
      INTEGER j,k
      REAL ymtmp(MPCDAT),yntmp(MPCDAT)

      do 12 j=1,m   ! Loop over rows.
        do 11 k=1,n ! Copy the row into temporary storage.
          yntmp(k)=ya(j,k)
 11     continue

        call polint(x2a,yntmp,n,x2,ymtmp(j),dy) ! Interpolate answer into temporary storage.

 12   continue

      call polint(x1a,ymtmp,m,x1,y,dy) ! Do the final interpolation.

      return
      END

C ******************** CMP145C ********************
C Generates for plant component IPCOMP with plant db code 146 ie.
C 1 node (ISV=21) AC moist air flow source
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: none
C     BDATA: none
C     CDATA: 1 Drybulb temperature (C)
C            2 Dry air mass flow rate (kg/s)
C            3 Water vapour mass flow rate (kg/s)

      SUBROUTINE CMP145C(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/PTIME/PTIMEP,PTIMEF

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)

      COMMON/ATPRES/PATMOS
      COMMON/WBULBO/IOPT
      
      LOGICAL bConnected       ! Flag indicating incomming connection exists

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      character outs*124
      REAL GS ! Humidity ratio (kg_v/kg_da)
      REAL fVapFlow ! Water vapour flow rate (kg_v/s)
      REAL PCRH2,HUMR

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine CMP145C'

C Check control data
      IF(ISTATS.eq.2 .and. CDATA(IPCOMP,2).LT.0.) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' CMP145C: dry air mass flow less than zero',
     &             ' for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP145C: unresolvable error.')
         call epwait
         call epagend
         STOP
      ELSEIF(ISTATS.eq.3 .and. CDATA(IPCOMP,3).LT.0.) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*)' CMP145C: water vapour mass flow less than zero',
     &             ' for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' CMP145C: unresolvable error.')
         call epwait
         call epagend
         STOP
      END IF    

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)

C Sort out the connections
      if ( ICON1 > 0 ) then
C........Connection exists!
         bConnected = .true.         
      else
C........Connection doesn't exist!
         bConnected = .false.
      endif 

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN
C Establish matrix equation self- and cross-coupling coefficients
         COUT(1)=1.
         IF(bConnected) THEN
            COUT(2)=0.
            COUT(3)=CDATA(IPCOMP,1)
         ELSE
            COUT(2)=CDATA(IPCOMP,1)
         ENDIF

C 1st phase mass (ie. dry air) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         IF(bConnected) THEN
            COUT(2)=0.
            COUT(3)=CDATA(IPCOMP,2)
         ELSE
            COUT(2)=CDATA(IPCOMP,2)
         ENDIF

C 2nd phase mass (ie. vapour) balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1) = 1.
         fVapFlow=0.
         ! Check relative humidity if there is flow
         IF(CDATA(IPCOMP,3)>0. .AND. CDATA(IPCOMP,2)>0.)THEN
            fVapFlow=CDATA(IPCOMP,3)
            GS=CDATA(IPCOMP,3)/CDATA(IPCOMP,2)
            IF(PCRH2(CDATA(IPCOMP,1),GS,PATMOS).GT.100.)THEN
                WRITE(IUOUT,*) ' CMP145C warning: Relative humidity', 
     &         ' exceeds 100 percent. Resetting vapour flow.'
                GS=HUMR(CDATA(IPCOMP,1),100.,PATMOS)
                fVapFlow=GS*CDATA(IPCOMP,2)
            ENDIF
         ENDIF
         
         IF(bConnected)THEN
            COUT(2)=0.
            COUT(3)=fVapFlow
         ELSE
            COUT(2)=fVapFlow
         ENDIF
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=21) moist air source'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' Temp  = ',CDATA(IPCOMP,1), '(C)'
         ELSE IF(ISTATS.EQ.2) THEN
            WRITE(ITU,*) ' Dry air flow  = ',CDATA(IPCOMP,2), '(kg/s)'
         ELSE IF(ISTATS.EQ.3) THEN
            WRITE(ITU,*)' Water vap. flow  = ',CDATA(IPCOMP,3), '(kg/s)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine CMP145C'

      RETURN
      END
