C This file is part of the ESP-r system.
C Copyright CANMET Energy Technology Centre
C Natural Resources Canada, Government of Canada
C 2004. Please Contact Ian Beausoliel-Morrison for details
C concerning licensing.

C ESP-r is free software.  You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation
C (version 2 or later).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.

C This file contains the following routines related to the
C hot-water storage tank component models created for
C connecting fuel cell and adsorption storage systems.

C ADS_tank_fuel_static_temp    Static template for fuel-fired water tank.
C ADS_tank_fuel_coeff_gen      Coefficient generator for fuel-fired water tank.
C ADS_tank_elec_static_temp    Static template for electrically heated water tank.
C ADS_tank_elec_coeff_gen      Coefficient generator for electrically heated water tank.
C Press_WCH_pipe_static_temp   Static template for pressurized WCH pipe
C Press_WCH_pipe_coeff_gen     Coefficient generator for pressurized WCH pipe
C Press_WCH_pump_static_temp   Static template for pressurized WCH pump
C Press_WCH_pump_coeff_gen     Coefficient generator for pressurized WCH pump

C *********************************************************************************************
C References:
C
C    Beausoleil-Morrison I. (2001a), Design of Fuel Cell Component Model for
C    FCT project, CETC internal report, 16 pages, revised November 29, 2001.
C
C    Beausoleil-Morrison I. (2001b), Design of Storage Tank Component Models for
C    FCT project, CETC internal report, 10 pages, February 15, 2002.
C
C    Hensen J.L.M. (1991), On the Thermal Interaction of Building Structure and
C    Heating and Ventilation System, PhD Thesis, Technical University of Eindhoven,
C    The Netherlands.
C
C    Kelly N. (2001), Development of a Heated Water Storage Plant Component
C    Model, ESRU Technical Report 150/01, University of Strathclyde,
C    12 pages, March 19, 2001.

C    Mottillo M. (2006), Investigation of an Adsorption System for the Seasonal
C    Storage of Heat Applied to Residential Buildings, M.A.Sc. dissertation,
C    Concordia University, Montreal, Canada.
C *********************************************************************************************


C *********************************************************************************************
C ****************************** ADS_tank_fuel_static_temp *************************************
C Created by: Maria Mottillo
C Adapted from FC_tank_fuel_static_temp developed by Ian Beausoleil-Morrison
C Initial Creation Date: October 20, 2004

C This subroutine is the static template for a gas-fired water tank that is fed by
C either a residential fuel cell system or an adsorption storage unit.
C The original gas-fired water tank component model from which this
C model was adapted allowed only 2 connections whereas this component model allows 3.
C The component model was developed for connection to the residential fuel cell
C component model and the adsorption storage component model but could be used in other
C plant arrangements as well.
C The model supports the following configurations :
C   1) a combo tank that supplies a fan-coil for space heat + domestic hot water (DHW)
C   2) a tank that supplies a fan-coil for space heat
C   3) a tank that supplies DHW

C This subroutine checks whether the component is properly used in the user-defined
C plant network, it assigns user-input data to module variables, and it performs
C time-invariant calculations.

C The tank, including its combustion system, is represented with three nodes:
C   Node 1 represents the water and the tank casing.
C   Node 2 represents the combustion chamber.
C   Node 3 represents the flue gases.

C INPUTS:
C    IPCOMP     index number of the component under consideration
C    NPCOMP     number of plant components in user-defined plant network
C    NCI(i)     holds the number of possible plant control variables for plant component `i'
C    ADATA(i,j) holds `j'th data item for component `i' read from .pln file
C               (derived from plant components database)
C    ICONTP(i)  holds the state variable index (ISV) of the sending node for connection `i'.
C    IUOUT      output channel to feedback to user
C    ITC        trace output index (>0 gives a trace output)
C    ITU        output channel for trace output
C    ITRACE(35) flag indicating one of the plant traces (1= active)
C    molW_N2    molecular weight of fuel constituent, kg/kmol (in MODULE FC_FLUIDS)
C    molW_CO2   ditto
C    molW_CH4   ditto
C    molW_C2H6  ditto
C    molW_C3H8  ditto
C    HHV_CH4    higher heating value of fuel constituent, MJ/kg (in MODULE FC_tanks)
C    HHV_C2H6   ditto
C    HHV_C3H8   ditto
C    FC_DHW_CSA_fraction(j)    Nominal DHW draw (L/hour) for hour `j' based on CSA schedule
C                              (in MODULE FC_tanks)

C OUTPUTS:
C    tank_mass                 mass of node 1, kg (in MODULE FC_tanks)
C    tank_Cp                   specific heat of node 1, J/kgK (in MODULE FC_tanks)
C    tank_UA                   heat loss coefficient between node 1 and surroundings,
C                              W/K (in MODULE FC_tanks)
C    tank_burner_ON            burner heat output when on, W (in MODULE FC_tanks)
C    tank_burner_OFF           burner heat output when off, W (in MODULE FC_tanks)
C    tank_efficiency           combustion efficiency, fraction (in MODULE FC_tanks)
C    tank_excess_air           excess air ratio, fraction (in MODULE FC_tanks)
C    tank_fuel_molefrac_N2     fuel molar fractions (in MODULE FC_tanks)
C    tank_fuel_molefrac_CO2    ditto
C    tank_fuel_molefrac_CH4    ditto
C    tank_fuel_molefrac_C2H6   ditto
C    tank_fuel_molefrac_C3H8   ditto
C    tank_fuel_molefrac_total  ditto
C    molW_fuel_tank            molecular weight of fuel, kg/kmol (in MODULE FC_tanks)
C    HHV_fuel_tank             HHV of fuel, MJ/kmol (in MODULE FC_tanks)
C    tank_DHW_draw             flag indicating whether tank supplies DHW (in MODULE FC_tanks)
C    tank_space_heat           flag indicating whether tank is connected to a
C                              fan-coil (in MODULE FC_tanks)
C                              that supplies space heating (in MODULE FC_tanks)
C    FC_DHW_nominal_Lperday    Nominal DHW draw (L/day) based on occupancy (in MODULE FC_tanks)
C    FC_DHW_CSA_Lperday        Nominal DHW draw (L/day) based on CSA schedule
C                              (in MODULE FC_tanks)
C    FC_DHW_CSA_fraction(j)  fraction of total daily water draw that occurs during
C                            hour `j' (from MODULE FC_tanks)
C -------------------------------------------------------------------------------------------

      SUBROUTINE ADS_tank_fuel_static_temp(IPCOMP)
      IMPLICIT NONE
#include "plant.h"
#include "building.h"
#include "SOFC.h"
#include "cogen_tanks.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER IUOUT,IUIN,IEOUT

      COMMON/TC/ITC,ICNT
      INTEGER ITC,ICNT

      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      INTEGER ITCF,ITRACE,IZNTRC,ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      INTEGER NPCOMP,NCI
      REAL CDATA

      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      REAL ADATA,BDATA

      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     & ICONDX(MPCOM,MNODEC,MPCONC)

      REAL CONVAR
      INTEGER ICONTP,ICONDX

      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),
     & IPOFS2(MCOEFG,MPVAR)

      INTEGER NPCDAT,IPOFS1,IPOFS2

C---------------------------------------------------------------------------------
C Declare local variables.
C---------------------------------------------------------------------------------
      INTEGER NumADATA,Itemp,N_expect,IPCONC
      LOGICAL mistake,CLOSE
      INTEGER IPCOMP,J

C---------------------------------------------------------------------------------
C Initialize variables. These were formerly initialized in MODULEs prior to the
C conversion of the code to enable compilation with g77.
C---------------------------------------------------------------------------------

C---------------------------------------------------------------------------------
C DHW nominal water draws.
C---------------------------------------------------------------------------------
C      DATA ( FC_DHW_CSA_Lperhour(jhour),jhour=1,24 )
C     &                           /   9., 7., 0., 0., 0.,
C     &                               0., 0., 8.,18.,15.,
C     &                              16.,16.,14.,13.,12.,
C     &                              10., 9.,10.,12.,16.,
C     &                              17.,14.,12.,11. /



C---------------------------------------------------------------------------------
C The gas-fired water tank with 3 connections cannot be used concurrently with
C the gas-fired water tank with 2 connections or the electric water tank (2
C or 3 connections) because they share common variables through MODULE FC_tanks.
C If there is a need in the future to allow multiple storage tanks, this
C restriction can be overcome by creating separate MODULEs for the four component
C models.
C Variables used:
C    NPCDAT(i,4)  pointer to static template and coefficient generator for component `i'
C                 (in common/C12PS): =82 for electric component model (2 connections),
C                                    =81 for gas-fired component model (2 connections),
C                                    =112 for electric component model (3 connections).
C    NPCOMP       number of plant components in user-defined plant network
C---------------------------------------------------------------------------------
      DO J=1,NPCOMP
        IF( (NPCDAT(J,4).eq.81) .OR. (NPCDAT(J,4).eq.114) .OR.
     &      (NPCDAT(J,4).eq.82) )THEN
          WRITE(IUOUT,*) ' ADS_tank_fuel_static_temp: the gas'
          WRITE(IUOUT,*) ' water tank with 3 connections cannot'
          WRITE(IUOUT,*) ' be used concurrently with the electric'
          WRITE(IUOUT,*) ' tank (3 or 2 connections) or with the'
          WRITE(IUOUT,*) ' gas-fired tank with 2 connections.'
          STOP ' ADS_tank_fuel_static_temp: unresolvable error'
        END IF
      END DO



C---------------------------------------------------------------------------------
C Write out ADATA if there is a trace output. Note that there is no BDATA
C used with this component.
C---------------------------------------------------------------------------------
      IF(ITC.gt.0 .AND. ITRACE(35).ne.0) THEN
        WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
        WRITE(ITU,*) ' 3 node gas-fired space-heating water tank  '
        NumADATA = 14  ! Number of ADATA items.
        WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NumADATA)
        IF(ITU.eq.IUOUT) THEN  ! trace output going to screen, not file
          itemp=(IPCOMP/5)*5
          IF(itemp.eq.IPCOMP .OR. IPCOMP.eq.NPCOMP) call epagew ! write 5 lines at a time.
        END IF
      END IF


C---------------------------------------------------------------------------------
C Ensure that user has specified the correct number of control variables in
C .pln file. NCI(IPCOMP) holds the number of possible plant control variables
C as specified in the .pln file. There should be a single control variable.
C---------------------------------------------------------------------------------
      N_expect = 1
      IF(NCI(IPCOMP) .ne. N_expect) THEN
        WRITE(ITU,*) ' ADS_tank_fuel_static_temp warning: ',
     &               ' incorrect num of ctl variables specified.'
      ENDIF


C---------------------------------------------------------------------------------
C Check that each node in the component has the correct number of connections
C to other components.
C Variables used:
C    MPCONC           the maximum allowable connections to a node (from plant.h).
C    ICONDX(i,j,k)    the connection number for the k'th connection to the j'th node
C                     of component i. It is used as a pointer.
C    `mistake'        a flag indicating whether there are connection errors:
C                     .true. means there are errors.
C---------------------------------------------------------------------------------
      mistake = .false.
C-----There should be three connections to node 1.
      IF( ICONDX(IPCOMP,1,1) .eq. 0 ) mistake=.true.
      IF( ICONDX(IPCOMP,1,2) .eq. 0 ) mistake=.true.
      IF( ICONDX(IPCOMP,1,3) .eq. 0 ) mistake=.true.
      DO IPCONC=4,MPCONC
        IF( ICONDX(IPCOMP,1,IPCONC) .ne. 0 ) mistake=.true.
      END DO
C-----There should be no connections to node 2.
      DO IPCONC=1,MPCONC
        IF( ICONDX(IPCOMP,2,IPCONC) .ne. 0 ) mistake=.true.
      END DO
C-----There should be no connections to node 3.
      DO IPCONC=1,MPCONC
        IF( ICONDX(IPCOMP,3,IPCONC) .ne. 0 ) mistake=.true.
      END DO
C-----Write error message if the number of connections to the nodes are incorrect.
      IF(mistake)THEN
        WRITE(IUOUT,*) ' ADS_tank_fuel_static_temp: incorrect'
        WRITE(IUOUT,*) ' number of connections for component ',IPCOMP
        STOP ' ADS_tank_fuel_static_temp: unresolvable error'
      END IF


C---------------------------------------------------------------------------------
C Check that the connections to node 1 are of the correct type. The three connections
C to node 1 should be of type ISV=20 so that the tank component can be used in
C networks in which both first and second phase mass balances are formed.
C Variables used:
C    ICONTP(i)  holds the state variable index (ISV) of the sending node for connection `i'.
C    ISV        defines nodal fluid type & coefficient generator model capabilities:
C               ISV=0,10,20 node represents water + ....
C               ISV=1,11,21 node represents dry air + ....
C               ISV=9,19,29 node represents some solid material only
C               0 <=ISV<10  model suitable for energy balance only
C               10<=ISV<20  model suitable for energy + single phase mass balance
C               20<=ISV<30  model suitable for energy + two phase mass balances
C---------------------------------------------------------------------------------
      mistake = .false.
      IF( ICONTP( ICONDX(IPCOMP,1,1) ) .ne. 20 ) mistake=.true.
      IF( ICONTP( ICONDX(IPCOMP,1,2) ) .ne. 20 ) mistake=.true.
      IF( ICONTP( ICONDX(IPCOMP,1,3) ) .ne. 20 ) mistake=.true.
      IF(mistake)THEN
        WRITE(IUOUT,*) ' ADS_tank_fuel_static_temp: incorrect'
        WRITE(IUOUT,*) ' connection type to node 1 for comp ',IPCOMP
        STOP ' ADS_tank_fuel_static_temp: unresolvable error'
      END IF


C---------------------------------------------------------------------------------
C Assign user-specified mass (kg) and specific heat (J/kgK) of node 1
C (water in tank + casing).
C---------------------------------------------------------------------------------
      tank_mass = ADATA(IPCOMP,1)
      tank_Cp   = ADATA(IPCOMP,2)


C---------------------------------------------------------------------------------
C Assign user-specified heat loss coefficient (UA-value) between node 1 and
C surroundings (W/K).
C---------------------------------------------------------------------------------
      tank_UA = ADATA(IPCOMP,3)


C---------------------------------------------------------------------------------
C Assign user-specified burner capacity (W) for on and off (i.e. pilot) states.
C---------------------------------------------------------------------------------
      tank_burner_ON  = ADATA(IPCOMP,4)
      tank_burner_OFF = ADATA(IPCOMP,5)


C---------------------------------------------------------------------------------
C Assign user-specified combustion + flue efficiency (fraction). This is the
C percentage of the fuel's chemical energy content that is converted to the
C water's thermal energy. As such, this includes both the effects of combustion
C inefficiencies and the effects of the heat exchanger between the flue gases
C and the water. This is assumed to be a constant and thus not a function of tank
C temperature or cyclying frequency. This treatment could be refined in the future.
C---------------------------------------------------------------------------------
      tank_efficiency = ADATA(IPCOMP,6)/100.


C---------------------------------------------------------------------------------
C Assign user-specified excess air ratio (fraction).
C---------------------------------------------------------------------------------
      tank_excess_air = ADATA(IPCOMP,7)/100.


C---------------------------------------------------------------------------------
C Assign user-specified molar fractions of fuel constituents.
C---------------------------------------------------------------------------------
      tank_fuel_molefrac_N2    = ADATA(IPCOMP,8)
      tank_fuel_molefrac_CO2   = ADATA(IPCOMP,9)
      tank_fuel_molefrac_CH4   = ADATA(IPCOMP,10)
      tank_fuel_molefrac_C2H6  = ADATA(IPCOMP,11)
      tank_fuel_molefrac_C3H8  = ADATA(IPCOMP,12)


C---------------------------------------------------------------------------------
C Assign variable indicating whether there is a DHW as well as a space-heating
C draw from the storage tank. The DHW draw is not treated as a `connection'
C but rather is handled within the coefficient generator as a right-hand side
C term, this to simplify the plant networks.
C---------------------------------------------------------------------------------
      tank_DHW_draw = ADATA(IPCOMP,13)


C---------------------------------------------------------------------------------
C Assign variable indicating whether the tank is connected to a fan-coil that
C supplies space heat.  The tank may be configured to supply DHW, space-heat,
C or both.
C---------------------------------------------------------------------------------
      tank_space_heat = ADATA(IPCOMP,14)


C---------------------------------------------------------------------------------
C Determine the DHW nominal water draw. This equation is taken from the HOT3000
C DHW model, which took it directly from the HOT2000 bin model.
C---------------------------------------------------------------------------------
      FC_DHW_nominal_Lperday = 85. + ( 35. * float(FC_NumOfOccupants) )


C---------------------------------------------------------------------------------
C Determine the fraction of the daily DHW water draw that is drawn at each hour of
C the day. The actual water draws will be determined using these hourly profiles,
C the total DHW draw over the day, and the time-varying tank temperature.
C---------------------------------------------------------------------------------
      FC_DHW_CSA_Lperday = 0.
      DO j=1,24
        FC_DHW_CSA_Lperday = FC_DHW_CSA_Lperday
     &                     + FC_DHW_CSA_Lperhour(j)
      END DO
      DO j=1,24
        FC_DHW_CSA_fraction(j) =
     &                       FC_DHW_CSA_Lperhour(j) / FC_DHW_CSA_Lperday
      END DO


C---------------------------------------------------------------------------------
C Check to ensure that the molar fractions of the fuel constituents sum to unity.
C---------------------------------------------------------------------------------
      tank_fuel_molefrac_total = tank_fuel_molefrac_N2
     &                         + tank_fuel_molefrac_CO2
     &                         + tank_fuel_molefrac_CH4
     &                         + tank_fuel_molefrac_C2H6
     &                         + tank_fuel_molefrac_C3H8
      CALL ECLOSE(tank_fuel_molefrac_total,1.0,0.001,CLOSE)
      IF(.not.CLOSE)THEN
        WRITE(IUOUT,*) ' ADS_tank_fuel_static_temp:'
        WRITE(IUOUT,*) ' molar fractions of fuel'
        WRITE(IUOUT,*) ' do not sum to unity.'
        STOP ' ADS_tank_fuel_static_temp: unresolvable error'
      END IF


C---------------------------------------------------------------------------------
C Calculate the molecular weight of the fuel (kg/kmol).
C---------------------------------------------------------------------------------
      molW_fuel_tank = tank_fuel_molefrac_CH4   * molW_CH4
     &               + tank_fuel_molefrac_C2H6  * molW_C2H6
     &               + tank_fuel_molefrac_C3H8  * molW_C3H8
     &               + tank_fuel_molefrac_N2    * molW_N2
     &               + tank_fuel_molefrac_CO2   * molW_CO2


C---------------------------------------------------------------------------------
C Calculate the energy content of the fuel, HHV (Beausoleil-Morrison 2001b).
C Units of MJ/kmol.
C---------------------------------------------------------------------------------
      HHV_fuel_tank = tank_fuel_molefrac_CH4   * molW_CH4   * HHV_CH4
     &              + tank_fuel_molefrac_C2H6  * molW_C2H6  * HHV_C2H6
     &              + tank_fuel_molefrac_C3H8  * molW_C3H8  * HHV_C3H8


C---------------------------------------------------------------------------------
C Set local variables to point to the nodes and to the connections.
C Variables used:
C    ICONDX(i,j,k)  the connection number for the k'th connection to the j'th node
C                   of component i. It is used as a pointer.
C    NPCDAT(i,9)    row and column number defining location of component `i'
C                   sub-matrix template in the overall plant network matrix. It is
C                   used to identify the location of the current component's nodes
C                   within the global plant matrix.
C    node1          global matrix node number for water tank component node 1
C    node2          global matrix node number for water tank component node 2
C    node3          global matrix node number for water tank component node 3
C    con1a          number of first connection to node 1 (to fuel cell, fan coil or adsorber)
C    con1b          number of second connection to node 1 (to fuel cell, fan coil or adsorber)
C    con1c          numer of third connection to node 1 (to fuel cell, fan coil or adsorber)
C---------------------------------------------------------------------------------
      node1 = NPCDAT(IPCOMP,9)
      node2 = NPCDAT(IPCOMP,9)+1
      node3 = NPCDAT(IPCOMP,9)+2
      con1a = ICONDX(IPCOMP,1,1)
      con1b = ICONDX(IPCOMP,1,2)
      con1c = ICONDX(IPCOMP,1,3)

C---------------------------------------------------------------------------------
C Completion of time-invariant operations for component.
C---------------------------------------------------------------------------------
      RETURN
      END



C *********************************************************************************************
C ******************************** ADS_tank_fuel_coeff_gen *************************************
C Created by: Maria Mottillo
C Adapted from FC_tank_fuel_coeff_gen created by Ian Beausoleil-Morrison
C Initial Creation Date: October 20, 2004

C This subroutine is the coefficient generator for the gas-fired storage tank component
C model. The original gas-fired water tank component model from which this
C model was adapted allowed only 2 connections whereas this component model allows 3.
C The component model was developed for connection to the residential fuel cell
C component model and the adsorber storage component model but could be used in other
C plant arrangements as well.
C The model supports the following configurations :
C   1) a combo tank that supplies a fan-coil for space heat + domestic hot water (DHW)
C   2) a tank that supplies a fan-coil for space heat
C   3) a tank that supplies DHW

C The DHW loads are treated as `connectionless'.  The DHW water draws and make-up water
C temperature are calculated within the coefficient generator.  These appear on the RHS
C of the energy balance for the water node.  This is done to minimize the number of
C components and connections and to allow for time-varying draws and make-up water
C temperatures.

C The tank is configured to be connected to a fuel cell. A second connection to the
C water node ia also required.  A flag can be set to indicate whether the tank supplies
C a fan-coil unit for space heating.  If the tank does supply space heat, then this
C second connection is to the fan-coil system.  If the tank does not supply space heat,
C then a second connection is required to satisfy the matrix requirements.  However,
C this connection is a `dummy'.  This coefficient generator nullifies the impact of
C this second connection.  A constant temperature supply is recommended as the inlet
C for this dummy connection: a pipe is recommended for the output. A third connection
C to a water node is also required. The third connection is configured to be connected
C to the adsorber of an adsorption storage unit.

C This subroutine performs the time-step calculations and establishes coefficients for the
C sub-matrices that define the energy, 1st phase mass flow, and 2nd phase mass
C flow balances on the storage tank's nodes. A fully explicit form of the energy
C balances is formed for nodes 2 and 3 since transients in the combustion chamber and flue
C are ignored. A weighted implicit-explicit energy balance is formed for node 1.

C The tank, including its combustion system, is represented with three nodes:
C   Node 1 represents the water and the tank casing.
C   Node 2 represents the combustion chamber.
C   Node 3 represents the flue gases.

C INPUTS:
C    tank_mass                 mass of node 1, kg (in MODULE FC_tanks)
C    tank_Cp                   specific heat of node 1, J/kgK (in MODULE FC_tanks)
C    tank_UA                   heat loss coefficient between node 1 and surroundings,
C                              W/K (in MODULE FC_tanks)
C    tank_burner_ON            burner heat output when on, W (in MODULE FC_tanks)
C    tank_burner_OFF           burner heat output when off, W (in MODULE FC_tanks)
C    tank_efficiency           combustion efficiency, fraction (in MODULE FC_tanks)
C    tank_excess_air           excess air ratio, fraction (in MODULE FC_tanks)
C    tank_fuel_molefrac_N2     fuel molar fractions (in MODULE FC_tanks)
C    tank_fuel_molefrac_CO2    ditto
C    tank_fuel_molefrac_CH4    ditto
C    tank_fuel_molefrac_C2H6   ditto
C    tank_fuel_molefrac_C3H8   ditto
C    tank_fuel_molefrac_total  ditto
C    molW_fuel_tank            molecular weight of fuel, kg/kmol (in MODULE FC_tanks)
C    HHV_fuel_tank             HHV of fuel, MJ/kmol (in MODULE FC_tanks)
C    tank_DHW_draw             flag indicating whether tank supplies DHW (in MODULE FC_tanks)
C    tank_space_heat           flag indicating whether tank is connected to a fan-coil
C                              that supplies space heating (in MODULE FC_tanks)
C    T_refCp                   reference temperature used to calculate heat capacities, oC
C                              (in MODULE FC_FLUIDS)
C    air_molefrac_N2           mole fraction of gas in air (in MODULE FC_FLUIDS)
C    air_molefrac_O2           ditto
C    air_molefrac_CO2          ditto
C    air_molefrac_H2O          ditto
C    air_molefrac_Ar           ditto
C    molW_air                  molecular weight of air, kg/kmol (in MODULE FC_FLUIDS)
C    PCNTMP(i)                 present time-row temperature of room containing
C                              component `i' (common/PCVAR)
C    PCNTMF(i)                 future time-row temperature of room containing
C                              component `i' (common/PCVAR)
C    IMPEXP                    identifies implicit/explicit handling of plant
C                              equations (common/PCEQU)
C    RATIMP                    user-specified implicit weighting fraction (common/PCEQU)
C    TIMSEC                    time-step in explicit plant domain (seconds) (common/PCTIME)
C    CDATA(i,j)                control signal for component `i', node `j' (common/C9)

C OUTPUTS:
C    COUT(i)             coefficient for ISTATS balance for `i'th matrix position
C    NAPDAT(i)           number of "additional" output for component `i'
C    PCAOUT(i,j)         additional output `j' for component `i'
C    TC(i)               time constant (seconds) of plant component `i'
C -------------------------------------------------------------------------------------------

      SUBROUTINE ADS_tank_fuel_coeff_gen(IPCOMP,COUT,ISTATS)
      use h3kmodule
      IMPLICIT NONE
#include "plant.h"
#include "building.h"
#include "SOFC.h"
#include "cogen_tanks.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER IUOUT,IUIN,IEOUT

      COMMON/TC/ITC,ICNT
      INTEGER ITC,ICNT

      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      INTEGER ITCF,ITRACE,IZNTRC,ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      INTEGER IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow

      COMMON/PITER/
     & MAXITP,
     & PERREL,
     & PERTMP,
     & PERFLX,
     & PERMFL,
     & itrclp,
     & ICSV(MPNODE,MPVAR),
     & CSVI(MPNODE,MPVAR)

      INTEGER MAXITP
      REAL PERREL
      REAL PERTMP
      REAL PERFLX
      REAL PERMFL
      INTEGER itrclp
      INTEGER ICSV
      REAL CSVI


      COMMON/C9/
     & NPCOMP,
     & NCI(MPCOM),
     & CDATA(MPCOM,MMISCD)

      INTEGER NPCOMP
      INTEGER NCI
      REAL CDATA


      COMMON/C10/
     & NPCON,
     & IPC1(MPCON),
     & IPN1(MPCON),
     & IPCT(MPCON),
     & IPC2(MPCON),
     & IPN2(MPCON),
     & PCONDR(MPCON),
     & PCONSD(MPCON,2)

      INTEGER NPCON
      INTEGER IPC1
      INTEGER IPN1
      INTEGER IPCT
      INTEGER IPC2
      INTEGER IPN2
      REAL PCONDR
      REAL PCONSD


      COMMON/PCVAL/
     & CSVF(MPNODE,MPVAR),
     & CSVP(MPNODE,MPVAR)

      REAL CSVF
      REAL CSVP


      COMMON/PCVAR/
     & PCTF(MPCON),
     & PCRF(MPCON),
     & PUAF(MPNODE),
     & PCQF(MPNODE),
     & PCNTMF(MPCOM),
     & PCTP(MPCON),
     & PCRP(MPCON),
     & PUAP(MPNODE),
     & PCQP(MPNODE),
     & PCNTMP(MPCOM)

      REAL PCTF
      REAL PCRF
      REAL PUAF
      REAL PCQF
      REAL PCNTMF
      REAL PCTP
      REAL PCRP
      REAL PUAP
      REAL PCQP
      REAL PCNTMP

      COMMON/PCOND/CONVAR(MPCON,MCONVR),
     & ICONTP(MPCON),
     & ICONDX(MPCOM,MNODEC,MPCONC)

      REAL CONVAR
      INTEGER ICONTP
      INTEGER ICONDX

      COMMON/PCRES/
     & QDATA(MPCOM),
     & PCAOUT(MPCOM,MPCRES),
     & napdat(mpcom)

      REAL QDATA
      REAL PCAOUT
      INTEGER napdat

      COMMON/PCTC/
     &TC(MPCOM)

      REAL TC

      COMMON/PCEQU/
     & IMPEXP,
     & RATIMP

      INTEGER IMPEXP
      REAL RATIMP

      COMMON/PCTIME/
     & TIMSEC

      REAL TIMSEC

      common/pcnam/pcname(mpcom)       ! Plant component names
      character*15 pcname

      COMMON/Stgdata/iCharge_signal,iDischarge_signal
      INTEGER iCharge_signal    ! charging signal for adsorption storage unit (on/off)
      INTEGER iDischarge_signal ! discharge signal for adsorption storage unit (on/off)

C---------------------------------------------------------------------------------
C Declare local variables. These variables are defined at their first use.
C---------------------------------------------------------------------------------
      LOGICAL CLOSE
      REAL COUT(MPCOE) ! plant matrix coefficients (passed in calling statement)
      REAL tank_UA_future,tank_UA_present
      REAL T_node1_present,T_node1_future
      REAL mdotCp_1a_future,mdotCp_1b_future,mdotCp_1c_future,
     &     mdotCp_1a_present,mdotCp_1b_present,mdotCp_1c_present
      REAL T_1a_present,T_1b_present,T_1c_present
      REAL Rinv,alpha
      REAL part1,part2,part3,part4,part5,part6,part7,part8,part9
      REAL T_fuel_tank,T_air_tank
      REAL m_dot_CH4_kmol_t,m_dot_C2H6_kmol_t,m_dot_C3H8_kmol_t
      REAL m_dot_O2_for_CH4_kmol_t,m_dot_O2_for_C2H6_kmol_t,
     &     m_dot_O2_for_C3H8_kmol_t,m_dot_O2_stoich_t,m_dot_air_stoich,
     &     m_dot_air_kmol_t,Vdot_air_STP_t
      REAL H_fuel,H_air
      REAL m_dot_N2_exh_kmol_t,m_dot_Ar_exh_kmol_t,m_dot_O2_exh_kmol_t,
     &     m_dot_CO2_exh_kmol_t,m_dot_H20_exh_kmol_t
      REAL m_dot_exh_kmol_t,Vdot_exh_STP_t,tank_exh_molefrac_N2,
     &     tank_exh_molefrac_Ar,tank_exh_molefrac_O2,
     &     tank_exh_molefrac_CO2,tank_exh_molefrac_H2O
      REAL molW_exh_t
      REAL T_comb,H_comb,T_exh_tank,H_exh_tanka,Cp_exh_tank,Cp_comb

      REAL  H3K_Connect_property   !- function used to return temperature of most recently
                                   !- calculated temperature (oC)

      INTEGER IPCOMP
      INTEGER ISTATS
      INTEGER ITEMP

      REAL SHTFLD
      REAL RHOFLD
      REAL H_gas
      INTEGER LNBLNK
      INTEGER I

      INTEGER iNameLength           ! temporary variables's length

      real  check        ! for debugging


C---------------------------------------------------------------------------------
C Start trace if trace output requested.
C---------------------------------------------------------------------------------
      IF(ITC.gt.0 .AND. NSINC.ge.ITC .AND.NSINC.le.ITCF .AND.
     &   ITRACE(37).ne.0)THEN
        WRITE(ITU,*) ' Entering sub ADS_tank_fuel_coeff_gen'
      END IF

C---------------------------------------------------------------------------------
C Mark the temperature of the water node for iteration.
C Variables used:
C     ICSV(i,j)  flag indicating that node `i' is marked for iteration for state
C                variable `j'; j=1 for temperature, j=2 for 1st phase mass flow,
C                j=3 for 2nd phase mass flow.
C     CSVI(i,j)  initial value for judging whether iteration required. Same
C                indices as ICSV.
C     CSVF(i,j)  future time-row solution variable for plant. Same indices as ICSV.
C---------------------------------------------------------------------------------
      ICSV(node1,1) = 1
      CSVI(node1,1) = CSVF(node1,1)


C---------------------------------------------------------------------------------
C Generate coefficients for the appropriate equation set. This subroutine was
C called to either generate coefficients for energy balances (ISTATS=1), 1st
C phase mass balances (ISTATS=2), or 2nd phase mass balances (ISTATS=3).
C Start with the energy balances.
C---------------------------------------------------------------------------------

C*********************************************************************************
C Beginning of energy balance section ********************************************
C*********************************************************************************
      IF(ISTATS.eq.1) THEN


         if ( nsinc .eq. 6 ) then
            check = 1.
         endif

C---------------------------------------------------------------------------------
C Determine the `present' and `future' temperature of the room containing the
C water tank. This is used to calculate the heat lost from the hot tank to the
C room (Beausoleil-Morrison 2001b, eq 4).
C Variables used:
C    PCNTMP(i)  present time-row temperature of room containing component `i'.
C    PCNTMF(i)  future time-row temperature of room containing component `i';
C               =-99 if no containment defined.
C---------------------------------------------------------------------------------
        CALL ECLOSE(PCNTMF(IPCOMP),-99.0,0.001,CLOSE)
        IF(CLOSE)THEN
          WRITE(IUOUT,*) ' ADS_tank_fuel_coeff_gen: '
          WRITE(IUOUT,*) ' the storage tank must be contained within '
          WRITE(IUOUT,*) ' a room.'
          STOP ' ADS_tank_fuel_coeff_gen: unresolvable error'
        ELSE
          Troom_present = PCNTMP(IPCOMP)
          Troom_future  = PCNTMF(IPCOMP)
        END IF


C---------------------------------------------------------------------------------
C Establish the heat loss coefficient (UA-value, W/K) between node 1 and the
C surroundings for the present and future time-rows. A fixed UA-value is used here.
C However, this code is structured to easily enable a time-varying UA-value in the
C future. PUAP and PUAF are standard ESP-r variables holding the present and future
C time-row values of the UA-value. PUAF is mapped to PUAP by subroutine MZNASS
C following the solution of the energy matrix for the current time-step.
C Variables used:
C    PUAP(i)  present time-row UA-value
C    PUAF(i)  future time-row UA-value
C---------------------------------------------------------------------------------
C-------Determine UA.
        tank_UA_future  = tank_UA  ! `tank_UA' from dbase (could be replaced by a correlation)
C-------Save future value.
        PUAF(node1)     = tank_UA_future
C-------Set present value.
        tank_UA_present = PUAP(node1)

C---------------------------------------------------------------------------------
C     Determine most recently calculated mass flow:
C     CONVAR(i,2) water mass flow rate (kg/s) at sending node for connection `i'
C     PCONDR(i)   ratio of flow rate leaving sending node `i' that reaches
C                receiving node.
C---------------------------------------------------------------------------------

C crude way but can't get pump controls to properly get this value.

         if ( iCharge_signal .eq. 1 ) then
              mdot_conn1a_future = 0.
         else
              mdot_conn1a_future = PCONDR(con1a) *
     &                             CONVAR(con1a,2)  ! (kg/s)
         endif

         mdot_conn1b_future = PCONDR(con1b) * CONVAR(con1b,2)  ! (kg/s)

         if ( iDischarge_signal .eq. 1 ) then
              mdot_conn1c_future = PCONDR(con1c) *
     &                             CONVAR(con1c,2)  ! (kg/s)
         else
              mdot_conn1c_future = 0.
         endif


C---------------------------------------------------------------------------------
C Determine the products of mass flow and heat capacity (W/K) for the three
c connections to node 1, for the present and future time-rows. These are the heat
C capacity rates (W/K) of the the water entering the storage tank from the fuel
C cell, from the fan-coil return and from the adsorber. PCRP and PCRF are standard
C ESP-r variables holding the present and future time-row values of the
C mass flow * heat capacity.
C PCRF is mapped to PCRP by subroutine MZNASS following the solution of the energy
C matrix for the current time-step.
C Variables used:
C    CONVAR(i,1) temperature (oC) of sending node for connection `i'
C    CONVAR(i,2) water mass flow rate (kg/s) at sending node for connection `i'
C    PCONDR(i)   ratio of flow rate leaving sending node `i' that reaches
C                receiving node.
C    SHTFLD(3,T) function that returns heat capacity (J/kgK) of water (index 3)
C                at temperature `T'.
C    PCRP(i)     present time-row mass flow * heat capacity for connection `i'
C    PCRF(i)     future time-row mass flow * heat capacity for connection `i'
C---------------------------------------------------------------------------------
         mdotCp_1a_future = mdot_conn1a_future
     &                    * SHTFLD( 3,CONVAR(con1a,1) )
         mdotCp_1c_future = mdot_conn1c_future
     &                    * SHTFLD( 3,CONVAR(con1c,1) )
C--------If space-heat flag set, then second connection is real: otherwise make flow zero.
         IF( tank_space_heat.eq.tank_yes_heat )THEN
           mdotCp_1b_future = mdot_conn1b_future
     &                      * SHTFLD( 3,CONVAR(con1b,1) )
         ELSE
           mdotCp_1b_future = 0.
         END IF


C--------Save future values.
         PCRF(con1a) = mdotCp_1a_future
         PCRF(con1c) = mdotCp_1c_future
         IF( tank_space_heat.eq.tank_yes_heat )THEN
           PCRF(con1b) = mdotCp_1b_future
         END IF
C--------Set present values.
         mdotCp_1a_present = PCRP(con1a)
         mdotCp_1c_present = PCRP(con1c)
         IF( tank_space_heat.eq.tank_yes_heat )THEN
           mdotCp_1b_present = PCRP(con1b)
         ELSE
           mdotCp_1b_present = 0.
         END IF


C...Save future values

         PCTF(con1a) = H3K_Connect_property(con1a,1)
         PCTF(con1c) = H3K_Connect_property(con1c,1)
         PCTF(con1b) = H3K_Connect_property(con1b,1)

C---------------------------------------------------------------------------------
C Establish the present time-row temperatures of the water flowing into the tank
C at the three connections.  PCTP and PCTF are standard ESP-r variables holding the
C present and future time-row values of the temperature of the fluid flowing at
C the connection. PCTF is mapped to PCTP by subroutine MZNASS following the
C solution of the energy matrix for the current time-step.
C Variables used:
C    CONVAR(i,1) temperature (oC) of sending node for connection `i'
C    PCTP(i)     present temperature of sending node for connection `i'
C    PCTF(i)     future temperature of sending node for connection `i'
C---------------------------------------------------------------------------------
         T_1a_present = PCTP(con1a)
         T_1c_present = PCTP(con1c)
C--------If space-heat flag set, then second connection is real: otherwise make temp zero.
         IF( tank_space_heat.eq.tank_yes_heat )THEN
           T_1b_present = PCTP(con1b)
         ELSE
           T_1b_present = 0.
         END IF


C---------------------------------------------------------------------------------
C Establish the present and future time-row temperatures of node 1.
C Variables used:
C     CSVP(i,j)  present time-row solution variable for node `i' and variable `j';
C                j=1 for temperature
C     CSVF(i,j)  future time-row solution variable (same indices as with CSVP)
C---------------------------------------------------------------------------------
         T_node1_present = CSVP(node1,1)
         T_node1_future  = CSVF(node1,1)


C---------------------------------------------------------------------------------
C Check the temperature of the tank and activiate the fictitious heat dump if
C necessary to maintain the temperature within a reasonable range. This should
C prevent the occurence of boiling and will allow the examination of scenarios for
C dealing with excessive energy build-up in the system.
C---------------------------------------------------------------------------------

C Note: The temperature setpoints for the heat dump are specified in cogen_tanks.h.

         CALL FC_dump_excess_energy(node1)


C---------------------------------------------------------------------------------
C Check the temperature of the tank to ensure that the water is not going to
C boil. This may occur is the fuel cell consistently adds more energy to the
C tank than is drawn off to meet the space heating and/or DHW loads. This should
C never occur with addition of call above to `FC_dump_excess_energy', but left
C here anyway for safety.
C---------------------------------------------------------------------------------

C Note: Message issued if water in tank > 95C.

         CALL FC_check_for_boil(node1)


C---------------------------------------------------------------------------------
C Determine the products of mass flow and heat capacity (W/K) for the DHW draw
C on the tank, for the present and future time-rows. Also determine the temperature
C of the DHW make-up water, for the present and future time-rows.
C Variables used:
C    RHOFLD(3,T)   function that returns density (kg/m3) of water (index 3) at
C                  temperaturere `T'.
C    SHTFLD(3,T)   function that returns heat capacity (J/kgK) of water (index 3)
C                  at temperature `T'.
C---------------------------------------------------------------------------------
        CALL FC_DHW_load(T_node1_future)

        mdotCp_DHW_present = DHW_draw_present
     &                     * RHOFLD(3,T_makeup_present)
     &                     * SHTFLD( 3,T_makeup_present )
        mdotCp_DHW_future  = DHW_draw_future
     &                     * RHOFLD(3,T_makeup_future)
     &                     * SHTFLD( 3,T_makeup_future )


C---------------------------------------------------------------------------------
C Calculate the fuel flow rate (Beausoleil-Morrison 2001b, eq 9). The burner will
C either be on or off, depending on the control signal. CDATA=0 signals that the
C burner should be off; CDATA=1 signals that it should be on. Only the combustion
C chamber (node 2) is controlled and should be made to respond to the water
C temperature (node 1).
C Variables used:
C    CDATA(i,j)   control signal for component `i's node `j'.
C---------------------------------------------------------------------------------
C-------Determine whether burner is on or off this time-step.
        CALL ECLOSE( CDATA(IPCOMP,2),0.,0.0001,CLOSE )
        IF(CLOSE)THEN  ! Burner is in "off" state (there may be a pilot light)
          q_burner = tank_burner_OFF
        ELSE
          q_burner = tank_burner_ON
        ENDIF
C-------Determine the fuel flow rate in kmol/s (Beausoleil-Morrison 2001b eq 9).
        m_dot_fuel_kmol_t = q_burner / HHV_fuel_tank / 1000000.
C-------Determine fuel flow rate in kg/s and standard L/min (slpm). Note that
C-------a kmol occupies 22 414 L at STP.
        m_dot_fuel_kg_t = m_dot_fuel_kmol_t * molW_fuel_tank
        Vdot_fuel_STP_t = m_dot_fuel_kmol_t * 22414. * 60.


C---------------------------------------------------------------------------------
C Calculate the energy liberated from the fuel's combustion which is transferred
C to the water (node 1) using Beausoleil-Morrison 2001b eq 2 (W). PCQP and PCQF
C are standard ESP-r variables holding the present and future time-row values of
C nodal injections (the energy transferred from the combustion to the water in
C this case). PCQF is mapped to PCQP by subroutine MZNASS following the solution
C of the energy matrix for the current time-step.
C Variables used:
C    PCQP(i)     present energy transferred to water from combustion for node `i'
C    PCQF(i)     future energy transferred to water from combustion for node `i'
C---------------------------------------------------------------------------------
        q_capture_future = tank_efficiency * HHV_fuel_tank
     &                   * m_dot_fuel_kmol_t * 1000000.
C-------Save future value.
        PCQF(node1) = q_capture_future
C-------Set present value.
        q_capture_present = PCQP(node1)


C---------------------------------------------------------------------------------
C Calculate the enthalpy (kJ/kmol) of the fuel entering the combustion chamber
C (node 2) using Beausoleil-Morrison 2001a eq 11. Assume that the fuel enters
C at a temperature equal to that of the room containing the tank. The function
C H_gas returns the relative enthalpy of each fuel constituent in kJ/kmol.
C Variables used:
C    PCNTMF(i)  future time-row temperature of room containing component `i';
C               =-99 if no containment defined.
C---------------------------------------------------------------------------------
        T_fuel_tank = PCNTMF(IPCOMP) ! Already confirmed above that containment defined.
        H_fuel = tank_fuel_molefrac_N2   * H_gas(T_fuel_tank,N2,IUOUT)
     &         + tank_fuel_molefrac_CO2  * H_gas(T_fuel_tank,CO2,IUOUT)
     &         + tank_fuel_molefrac_CH4  * H_gas(T_fuel_tank,CH4,IUOUT)
     &         + tank_fuel_molefrac_C2H6 * H_gas(T_fuel_tank,C2H6,IUOUT)
     &         + tank_fuel_molefrac_C3H8 * H_gas(T_fuel_tank,C3H8,IUOUT)


C---------------------------------------------------------------------------------
C Calculate the air flow rate to node 2 (combustion chamber).
C First, determine the molar flow rates (kmol/s) of the three fuel constituents
C that react with O2. Then determine the O2 flow rate required to
C stoichiometrically react with these fuel constituents (Beausoleil-Morrison
C 2001b eq 11). Determine the stoichiometric air flow requirement, then
C finally the actual air flow rate (Beausoleil-Morrison 2001b eq 10). Note that
C this approach assumes that the hydrocarbons fully react in the combustion
C chamber.
C---------------------------------------------------------------------------------
C-------Flow rates of hydrocarbons (kmol/s).
        m_dot_CH4_kmol_t  = tank_fuel_molefrac_CH4  * m_dot_fuel_kmol_t
        m_dot_C2H6_kmol_t = tank_fuel_molefrac_C2H6 * m_dot_fuel_kmol_t
        m_dot_C3H8_kmol_t = tank_fuel_molefrac_C3H8 * m_dot_fuel_kmol_t
C-------Stoichiometric O2 flow rate required for each fuel constituent (kmol/s).
        m_dot_O2_for_CH4_kmol_t  = 2.    * m_dot_CH4_kmol_t
        m_dot_O2_for_C2H6_kmol_t = 7./2. * m_dot_C2H6_kmol_t
        m_dot_O2_for_C3H8_kmol_t = 5.    * m_dot_C3H8_kmol_t
        m_dot_O2_stoich_t = m_dot_O2_for_CH4_kmol_t
     &                    + m_dot_O2_for_C2H6_kmol_t
     &                    + m_dot_O2_for_C3H8_kmol_t
C-------Stoichiometric air flow rate (kmol/s).
        m_dot_air_stoich = m_dot_O2_stoich_t / air_molefrac_O2
C-------Air flow rate to node 2 (kmol/s, kg/s, and slpm).
        m_dot_air_kmol_t = m_dot_air_stoich * tank_excess_air
        m_dot_air_kg_t   = m_dot_air_kmol_t * molW_air     ! molW_air calculated in FC_static_template
        Vdot_air_STP_t   = m_dot_air_kmol_t * 22414. * 60.


C---------------------------------------------------------------------------------
C Calculate the enthalpy (kJ/kmol) of the air entering node 2 (combustion chamber)
C (Beausoleil-Morrison 2001a eq 11). Assume that the air enters at a temperature
C equal to that of the room containing the water tank. The function H_gas returns
C the relative enthalpy of each fuel constituent in kJ/kmol.
C Variables used:
C    PCNTMF(i)  future time-row temperature of room containing component `i';
C               =-99 if no containment defined.
C---------------------------------------------------------------------------------
        T_air_tank = PCNTMF(IPCOMP) ! Already confirmed above that containment defined.
        H_air = air_molefrac_N2   * H_gas(T_air_tank,N2,IUOUT)
     &         + air_molefrac_O2  * H_gas(T_air_tank,O2,IUOUT)
     &         + air_molefrac_CO2 * H_gas(T_air_tank,CO2,IUOUT)
     &         + air_molefrac_H2O * H_gas(T_air_tank,H2O,IUOUT)
     &         + air_molefrac_Ar  * H_gas(T_air_tank,Ar,IUOUT)


C---------------------------------------------------------------------------------
C Determine the composition and flow rate of the exhaust gases. Assume that
C the combustion process is complete. Based on the chemical reactions occuring
C within the combustion chamber (Beausoleil-Morrison 2001b eq 11), calculate
C the molar flow rate of each constituent. As reactions are assumed to
C be complete, there are no hydrocarbons in the exhaust.
C---------------------------------------------------------------------------------
C-------N2 (Beausoleil-Morrison 2001a eq 18).
        m_dot_N2_exh_kmol_t = m_dot_air_kmol_t  * air_molefrac_N2
     &                    + m_dot_fuel_kmol_t * tank_fuel_molefrac_N2
C-------Ar (Beausoleil-Morrison 2001a eq 19)
        m_dot_Ar_exh_kmol_t = m_dot_air_kmol_t  * air_molefrac_Ar
C-------O2 (Beausoleil-Morrison 2001a eq 20)
        m_dot_O2_exh_kmol_t =
     &                 m_dot_air_kmol_t * air_molefrac_O2
     &               - 2.    * m_dot_fuel_kmol_t
     &                       * tank_fuel_molefrac_CH4
     &               - 7./2. * m_dot_fuel_kmol_t
     &                       * tank_fuel_molefrac_C2H6
     &               - 5.    * m_dot_fuel_kmol_t
     &                       * tank_fuel_molefrac_C3H8
C-------CO2 (Beausoleil-Morrison 2001a eq 21)
        m_dot_CO2_exh_kmol_t =
     &                    m_dot_air_kmol_t  * air_molefrac_CO2
     &                  + m_dot_fuel_kmol_t * tank_fuel_molefrac_CO2
     &                  + 1. * m_dot_fuel_kmol_t
     &                       * tank_fuel_molefrac_CH4
     &                  + 2. * m_dot_fuel_kmol_t
     &                       * tank_fuel_molefrac_C2H6
     &                  + 3. * m_dot_fuel_kmol_t
     &                       * tank_fuel_molefrac_C3H8
C-------H2O (Beausoleil-Morrison 2001a eq 22)
        m_dot_H20_exh_kmol_t =
     &                    m_dot_air_kmol_t * air_molefrac_H2O
     &                  + 2. * m_dot_fuel_kmol_t
     &                       * tank_fuel_molefrac_CH4
     &                  + 3. * m_dot_fuel_kmol_t
     &                       * tank_fuel_molefrac_C2H6
     &                  + 4. * m_dot_fuel_kmol_t
     &                       * tank_fuel_molefrac_C3H8


C---------------------------------------------------------------------------------
C Calculate the flow rate of all exhaust gases in terms of kmol/s `m_dot_exh_kmol_t'
C and standard Litres/min `Vdot_exh_STP_t'. Then determine the mole fractions of
C the exhaust gas constituents (Beausoleil-Morrison 2001a eq 23).
C---------------------------------------------------------------------------------
        m_dot_exh_kmol_t = m_dot_N2_exh_kmol_t
     &                 + m_dot_Ar_exh_kmol_t
     &                 + m_dot_O2_exh_kmol_t
     &                 + m_dot_CO2_exh_kmol_t
     &                 + m_dot_H20_exh_kmol_t
        Vdot_exh_STP_t = m_dot_exh_kmol_t * 22414. * 60.
C-------Are there any exhaust gases?
        IF( m_dot_exh_kmol_t > 0. )THEN
C---------There is exhaust, so calculate molar fractions.
          tank_exh_molefrac_N2  = m_dot_N2_exh_kmol_t / m_dot_exh_kmol_t
          tank_exh_molefrac_Ar  = m_dot_Ar_exh_kmol_t / m_dot_exh_kmol_t
          tank_exh_molefrac_O2  = m_dot_O2_exh_kmol_t / m_dot_exh_kmol_t
          tank_exh_molefrac_CO2 = m_dot_CO2_exh_kmol_t/ m_dot_exh_kmol_t
          tank_exh_molefrac_H2O = m_dot_H20_exh_kmol_t/ m_dot_exh_kmol_t
        ELSE
C---------No exhaust: set molar fractions to zero (avoids div by 0).
          tank_exh_molefrac_N2  = 0.
          tank_exh_molefrac_Ar  = 0.
          tank_exh_molefrac_O2  = 0.
          tank_exh_molefrac_CO2 = 0.
          tank_exh_molefrac_H2O = 0.
        END IF


C---------------------------------------------------------------------------------
C Determine the molecular weight of the exhaust gases (Beausoleil-Morrison
C 2001a eq 24).
C---------------------------------------------------------------------------------
        molW_exh_t = tank_exh_molefrac_N2  * molW_N2
     &           + tank_exh_molefrac_Ar  * molW_Ar
     &           + tank_exh_molefrac_O2  * molW_O2
     &           + tank_exh_molefrac_CO2 * molW_CO2
     &           + tank_exh_molefrac_H2O * molW_H2O


C---------------------------------------------------------------------------------
C Determine the flow rate of all exhaust gases in terms of kg/s.
C---------------------------------------------------------------------------------
        m_dot_exh_kg_t = m_dot_exh_kmol_t * molW_exh_t


C---------------------------------------------------------------------------------
C Calculate the enthalpy (kJ/kmol) of the combustion gases leaving node 2 and
C entering node 3 (Beausoleil-Morrison 2001a eq 25). The function H_gas returns the
C relative enthalpy of each fuel constituent in kJ/kmol. Evaluate the enthalpies
C at the temperature solved the previous time-step.
C The coefficients established for node 2 the previous iteration may have led
C to an unrealistic CSVF solution. This is particularly prone to happen on the
C very first plant simulation increment. In order to prevent the H_comb calculation
C from blowing up, place a range check on T_comb to force it within a physically
C realistic range.
C Variables used:
C     CSVF(i,1)  future time-row temperature for node `i'.
C---------------------------------------------------------------------------------
        T_comb = CSVF(node2,1)
        IF( T_comb<100. .or. T_comb>2000.) T_comb=400.
        H_comb = tank_exh_molefrac_N2  * H_gas(T_comb,N2,IUOUT)
     &         + tank_exh_molefrac_Ar  * H_gas(T_comb,Ar,IUOUT)
     &         + tank_exh_molefrac_O2  * H_gas(T_comb,O2,IUOUT)
     &         + tank_exh_molefrac_CO2 * H_gas(T_comb,CO2,IUOUT)
     &         + tank_exh_molefrac_H2O * H_gas(T_comb,H2O,IUOUT)


C---------------------------------------------------------------------------------
C Calculate the enthalpy (kJ/kmol) of the exhaust gas mixture at the reference
C temperature `T_refCp' (see Beausoleil-Morrison eq 26a and 26b). The function
C H_gas returns the relative enthalpy of each fuel constituent in kJ/kmol.
C---------------------------------------------------------------------------------
        Href_tank = tank_exh_molefrac_N2  * H_gas(T_refCp_tank,N2,IUOUT)
     &          + tank_exh_molefrac_Ar   * H_gas(T_refCp_tank,Ar,IUOUT)
     &          + tank_exh_molefrac_O2   * H_gas(T_refCp_tank,O2,IUOUT)
     &          + tank_exh_molefrac_CO2  * H_gas(T_refCp_tank,CO2,IUOUT)
     &          + tank_exh_molefrac_H2O  * H_gas(T_refCp_tank,H2O,IUOUT)


C---------------------------------------------------------------------------------
C Linearise the enthalpy of the node 2 gases so that temperature can appear in
C the energy balance for nodes 2 and 3 (Beausoleil-Morrison 2001a eq 26a). The
C Cp calculated here is used to linearise the enthalpy term. The difference
C between the exhaust gas temperature at the previous time-step and `T_refCp_tank'
C is used to linearise the enthalpy. Degrees Celcius are used since ESP-r's plant
C matrix solver works in these units.
C The units of this heat capacity are J/(kmol*oC)
C---------------------------------------------------------------------------------
        Cp_comb = (H_comb-Href_tank)*1000. / (T_comb - T_refCp_tank)


C---------------------------------------------------------------------------------
C Calculate the enthalpy (kJ/kmol) of the exhaust gases leaving node 3
C (Beausoleil-Morrison 2001a eq 25). The function H_gas returns the
C relative enthalpy of each fuel constituent in kJ/kmol. Evaluate the enthalpies
C at the temperature solved the previous time-step.
C The coefficients established for node 3 the previous iteration may have led
C to an unrealistic CSVF solution. This is particularly prone to happen on the
C very first plant simulation increment. In order to prevent the H_exh_tanka calculation
C from blowing up, place a range check on T_exh_tank to force it within a physically
C realistic range.
C Variables used:
C     CSVF(i,1)  future time-row temperature for node `i'.
C---------------------------------------------------------------------------------
        T_exh_tank = CSVF(node3,1)
        IF( T_exh_tank.lt.100. .or. T_exh_tank.gt.2000.) T_exh_tank=400.
        H_exh_tanka = tank_exh_molefrac_N2  * H_gas(T_exh_tank,N2,IUOUT)
     &        + tank_exh_molefrac_Ar  * H_gas(T_exh_tank,Ar,IUOUT)
     &        + tank_exh_molefrac_O2  * H_gas(T_exh_tank,O2,IUOUT)
     &        + tank_exh_molefrac_CO2 * H_gas(T_exh_tank,CO2,IUOUT)
     &        + tank_exh_molefrac_H2O * H_gas(T_exh_tank,H2O,IUOUT)


C---------------------------------------------------------------------------------
C Linearise the enthalpy of the node 3 gases so that temperature can appear in
C the energy balance for node 3 (Beausoleil-Morrison 2001a eq 26a). The Cp
C calculated here is used to linearise the enthalpy term. The difference between
C the exhaust gas temperature at the previous time-step and `T_refCp_tank'.
C Degrees Celcius are used since ESP-r's plant matrix solver works in these units.
C The units of this heat capacity are J/(kmol*oC)
C---------------------------------------------------------------------------------
        Cp_exh_tank = (H_exh_tanka-Href_tank)*1000. /
     &           (T_exh_tank - T_refCp_tank)


C---------------------------------------------------------------------------------
C Calculate component's time-constant for the current time-step. Refer to
C Hensen (1991) pages 5.9 and 5.10.  The environmental losses from the tank and
C the water streams flowing into the tank are the three heat transfers that
C affect the transient storage. Therefore, treat these as parallel paths for
C the purposes of calculating the time constant, as described in eq 5.5 of
C Hensen (1991).
C Variables used:
C     tank_mass    mass of node 1, kg
C     tank_Cp      specific heat of node 1, J/kgK
C     TC(i)        time constant (seconds) of plant component `i'
C---------------------------------------------------------------------------------
        Rinv = tank_UA_future + mdotCp_1a_future + mdotCp_1b_future +
     &              mdotCp_1c_future
        CALL ECLOSE(Rinv,0.,0.001,CLOSE)
        IF(CLOSE)THEN
          WRITE(IUOUT,*) ' ADS_tank_fuel_coeff_gen: '
          WRITE(IUOUT,*) ' impossible time constant. '
          STOP ' ADS_tank_fuel_coeff_gen: unresolvable error'
        END IF
        TC(IPCOMP) = tank_mass*tank_Cp / Rinv


C---------------------------------------------------------------------------------
C Determine implicit/explicit weighting for the energy balance on node 1. `alpha'
C is the implicit weighting factor used to form the energy balance this time-step.
C It is the Greek "alpha" in Beausoleil-Morrison (2001b) equation 7. The approach
C used here to determine `alpha' is used with most plant components, as discussed
C by Hensen (1991).
C Variables used:
C    IMPEXP   identifies handling of plant equations: 1= fully implicit case;
C             2= mixed explicit/implicit; 3= 1 or 2 depending on component time
C             constant and time-step (this is the default treatment);
C             4= steady state.
C    RATIMP   user-specified implicit weighting fraction
C    TIMSEC   the time-step in the explicit plant domain (seconds)
C
C Although for IMPEXP = 3 the model calculates the required alpha, it is not
C recommended to use different values for alpha during a single simulation run,
C due to the likely introduction of errors in the simulation results.
C The user is therefore warned in case the model decides to change alpha from the
C default value to fully implicit (alpha = 1) due to a too small a time constant.
C---------------------------------------------------------------------------------
C-------Fully implicit.
        IF( IMPEXP.eq.1 ) THEN
          alpha = 1.
C-------Implicit/explicit with user-specified (fixed) weighting.
        ELSE IF( IMPEXP .eq. 2 ) THEN
          alpha = RATIMP
C-------General case: implicit/explicit with calculated weighting.
        ELSE IF( IMPEXP .eq. 3 ) THEN
          IF( TIMSEC.gt.0.63*TC(IPCOMP) ) THEN
            alpha = 1.
            WRITE(IUOUT,*) ' ADS_tank_fuel_coeff_gen: '
            WRITE(IUOUT,*) ' The time constant of the tank is too '
            WRITE(IUOUT,*) ' small compared to the current simulation '
            WRITE(IUOUT,*) ' time step.'
            WRITE(IUOUT,*) ' Choose a smaller time step to prevent '
            WRITE(IUOUT,*) ' erroneous results.'
          ELSE
            alpha = RATIMP
          END IF
C-------Steady-state.
        ELSE IF( IMPEXP.eq.4 ) THEN
          tank_mass = 0.  ! make node 1 massless
          alpha = 1.
        END IF


C---------------------------------------------------------------------------------
C Preparation for forming matrix coefficients for energy balances is now complete.
C Establish matrix equation self-coupling, cross-coupling, and RHS coefficients.
C Node coefficient transport for COUT

C       <--self-->|<cross>
C  node   1  2  3 |  fc fan-coil adsorber   RHS
C       ---------------------------------------
C         1  0  0 |   5    6    7         8
C         0  2  0 |   0    0    0      =  9
C         0  3  4 |   0    0    0         10
C---------------------------------------------------------------------------------
C-------Node 1 energy balance given by Beausoleil-Morrison (2001b) eq 7.
        COUT(1) = tank_mass * tank_Cp / TIMSEC       ! self-coupling to itself (W/K)
     &          + alpha * tank_UA_future
     &          + alpha * mdotCp_1a_future
     &          + alpha * mdotCp_1b_future
     &          + alpha * mdotCp_1c_future
     &          + alpha * mdotCp_DHW_future
        COUT(5) = -1. * alpha * mdotCp_1a_future     ! cross-coupling (W/K)
        COUT(6) = -1. * alpha * mdotCp_1b_future     ! cross-coupling (W/K)
        COUT(7) = -1. * alpha * mdotCp_1c_future     ! cross-coupling (W/k)
        part1 = tank_mass * tank_Cp / TIMSEC         ! (W/K)
     &        - (1.-alpha) * tank_UA_present
     &        - (1.-alpha) * mdotCp_1a_present
     &        - (1.-alpha) * mdotCp_1b_present
     &        - (1.-alpha) * mdotCp_1c_present
     &        - (1.-alpha) * mdotCp_DHW_present
        part2 = (1.-alpha) * mdotCp_1a_present       ! (W)
        part3 = (1.-alpha) * mdotCp_1b_present       ! (W)
        part9 = (1.-alpha) * mdotCp_1c_present       ! (W)
        part6 = (1.-alpha) * mdotCp_DHW_present      ! (W)
        part7 = alpha * mdotCp_DHW_future            ! (W)
        part4 = (1.-alpha) * q_capture_present       ! (W)
     &        + alpha * q_capture_future
        part5 = (1.-alpha) * tank_UA_present * Troom_present
     &        + alpha * tank_UA_future * Troom_future
        part8 = -1.*(1.-alpha) * q_heatdump_present  ! (W)
     &          -1*alpha * q_heatdump_future
        COUT(8) = part1 * T_node1_present            ! RHS (W)
     &          + part2 * T_1a_present
     &          + part3 * T_1b_present
     &          + part9 * T_1c_present
     &          + part6 * T_makeup_present
     &          + part7 * T_makeup_future
     &          + part4
     &          + part5
     &          + part8

C-------Node 2 energy balance. Treatment depends on whether there is combustion.
C-------Are there any exhaust gases (indicates that there is combustion)?
        IF( m_dot_exh_kmol_t .gt. 0. )THEN
C---------There is exhaust: calculate coefficients using energy balance given by
C---------Beausoleil-Morrison (2001b) eq 12.
          COUT(2) = m_dot_exh_kmol_t * Cp_comb               ! self-coupling to itself (W/K)
          COUT(9) = m_dot_fuel_kmol_t * H_fuel * 1000.     ! RHS (W)
     &            + m_dot_air_kmol_t  * H_air  * 1000.
     &            + q_burner
     &            + m_dot_exh_kmol_t * Cp_comb * T_refCp_tank
     &            - m_dot_exh_kmol_t * Href_tank * 1000.
        ELSE
C---------No exhaust: make combustion temperature equal room temperature.
          COUT(2) = 1.
          COUT(9) = Troom_future
        END IF

C-------Node 3 energy balance. Treatment depends on whether there is combustion.
C-------Are there any exhaust gases (indicates that there is combustion)?
        IF( m_dot_exh_kmol_t .gt. 0. )THEN
C---------There is exhaust: calculate coefficients using energy balance given by
C---------Beausoleil-Morrison (2001b) eq 14.
          COUT(3) = -1. * m_dot_exh_kmol_t * Cp_comb           ! self-coupling to node 2 (W/K)
          COUT(4) = m_dot_exh_kmol_t * Cp_exh_tank                  ! self-coupling to itself (W/K)
          COUT(10) = m_dot_exh_kmol_t * Cp_exh_tank  * T_refCp_tank  ! RHS (W)
     &            - m_dot_exh_kmol_t * Cp_comb * T_refCp_tank
     &            - q_capture_future
        ELSE
C---------No exhaust: make exhaust temperature equal room temperature.
          COUT(3) = 0.
          COUT(4) = 1.
          COUT(10) = Troom_future
        END IF


C*********************************************************************************
C End of energy balance section / Beginning of 1st phase mass balances ***********
C*********************************************************************************
      ELSE IF(ISTATS.eq.2) THEN
C-------Node 1: total flow in equals total flow out.
        COUT(1) = 1.
        COUT(5) = -1. * PCONDR(con1a) ! (dimensionless)
C-------If space-heat flag set, then second connection is real: otherwise make flow zero.
        IF( tank_space_heat.eq.tank_yes_heat )THEN
          COUT(6) = -1. * PCONDR(con1b)
        ELSE
          COUT(6) = 0.
        END IF
        COUT(7) = -1. * PCONDR(con1c)
        COUT(8) = 0.

        if ( iCharge_signal .eq. 1 ) then
           COUT(5) = 0.
           COUT(7) = 0.
        endif


        if ( iDischarge_signal .eq. 1 ) then
           COUT(5) = 0.
           COUT(7) = -1. * PCONDR(con1c)
        endif

C-------Node 2: no balance required so make flow zero.
        COUT(2) = 1.
        COUT(9) = 0.
C-------Node 3: flow in equals flow out.
        COUT(3) = -1.
        COUT(4) = 1.
        COUT(10) = 0.


C*********************************************************************************
C End of energy 1st phase mass balances / Beginning of 2nd phase mass balances ***
C*********************************************************************************
      ELSE IF(ISTATS.eq.3) THEN
C-------Node 1: no balance required so make flow zero.
        COUT(1) = 1.
        COUT(5) = 0.
        COUT(6) = 0.
        COUT(7) = 0.
        COUT(8) = 0.
C-------Node 2: no balance required so make flow zero.
        COUT(2) = 1.
        COUT(9) = 0.
C-------Node 3: no balance required so make flow zero.
        COUT(3) = 0.
        COUT(4) = 1.
        COUT(10) = 0.
      END IF

C*********************************************************************************
C End of energy and mass balances ************************************************
C*********************************************************************************


C---------------------------------------------------------------------------------
C Save the `additional' output variables for outputting on time-step basis.
C---------------------------------------------------------------------------------
      NAPDAT(IPCOMP)    = 8                    ! number of additional outputs
      PCAOUT(IPCOMP,1)  = q_burner             ! burner output (W)
      PCAOUT(IPCOMP,2)  = q_capture_future     ! burner energy captured in water (W)
      PCAOUT(IPCOMP,3)  = m_dot_fuel_kg_t        ! fuel consumption (kg/s)
      PCAOUT(IPCOMP,4)  = Vdot_fuel_STP_t        ! fuel consumption (slpm)
      PCAOUT(IPCOMP,5)  = m_dot_air_kg_t         ! air flow (kg/s)
      PCAOUT(IPCOMP,6)  = Vdot_air_STP_t         ! air flow (slpm)
      PCAOUT(IPCOMP,7)  = T_comb               ! temperature of combustion gases (oC)
      PCAOUT(IPCOMP,8)  = T_exh_tank                ! temperature of exhaust gases (oC)


C---------------------------------------------------------------------------------
C Complete trace if trace output requested.
C---------------------------------------------------------------------------------
      IF(ITC.gt.0 .AND. NSINC.ge.ITC .AND.NSINC.le.ITCF .AND.
     &   ITRACE(37).ne.0)THEN
        WRITE(ITU,*) ' Component      ',IPCOMP,':'
        WRITE(ITU,*) ' 3 node gas-fired storage tank for fuel cell ',
     &                 'and adsorption storage.'
        WRITE(ITU,*) ' Matrix node(s) ',node1,node2,node3

        WRITE(ITU,*) ' Connection(s)  ',con1a,con1b,con1c
        IF(ISTATS.eq.1) THEN
           WRITE(ITU,*) ' CDATA             = ',CDATA(IPCOMP,2),' (-)'
           WRITE(ITU,*) ' Water Temp (p)    = ',T_node1_present,
     &          ' (oC)'
           WRITE(ITU,*) ' Water Temp (f)    = ',T_node1_future,' (oC)'
           WRITE(ITU,*) ' Conn 1. Temp (p)  = ',T_1a_present,' (oC)'
           WRITE(ITU,*) ' Conn 2. Temp (p)  = ',T_1b_present,' (oC)'
           WRITE(ITU,*) ' Conn 3. Temp (p)  = ',T_1c_present,' (oC)'
           WRITE(ITU,*) ' q_burner          = ',q_burner,' (W)'
           WRITE(ITU,*) ' q_capture_future  = ',q_capture_future,' (W)'
           WRITE(ITU,*) ' m_dot_fuel_kg_t     = ',m_dot_fuel_kg_t,
     &                                          ' (kg/s)'
           WRITE(ITU,*) ' m_dot_air_kg_t      = ',m_dot_air_kg_t,
     &                                          ' (kg/s)'
           WRITE(ITU,*) ' m_dot_exh_kg_t      = ',m_dot_exh_kg_t,
     &                                          ' (kg/s)'
           WRITE(ITU,*) ' T_comb            = ',T_comb,' (oC)'
           WRITE(ITU,*) ' T_exh_tank             = ',T_exh_tank,' (oC)'
           WRITE(ITU,*) ' alpha             = ',alpha,' (-)'
        END IF
        WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
        WRITE(ITU,*) (COUT(I),I=1,10)
        IF(ITU.eq.IUOUT) THEN  ! trace output going to screen, not file
          itemp=(IPCOMP/4)*4
          IF(itemp.eq.IPCOMP .OR. IPCOMP.eq.NPCOMP) call epagew ! write 4 lines at a time.
        END IF
        WRITE(ITU,*) ' Leaving sub ADS_tank_fuel_coeff_gen'
      END IF


C------------------------------------------------------------------------------------
C XML output.
C------------------------------------------------------------------------------------

      if ( istats .eq. 1 ) then


C.....Get component name's length
      iNameLength = lnblnk(pcname(IPCOMP))

       call AddToReport(rvPltBurnerOut%Identifier,
     &         q_burner,
     &         pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltBurnerWater%Identifier,
     &         q_capture_future,
     &         pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltFuelConsKGS%Identifier,
     &         m_dot_fuel_kg_t,
     &         pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltFuelConsSLPM%Identifier,
     &         Vdot_fuel_STP_t,
     &         pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltAirFlowKGS%Identifier,
     &         m_dot_air_kg_t,
     &         pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltAirFlowSLPM%Identifier,
     &         Vdot_air_STP_t,
     &         pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltTempComb%Identifier,
     &         T_comb,
     &         pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltTempExh%Identifier,
     &         T_exh_tank,
     &         pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltHeatDumpFuture%Identifier,
     &         q_heatdump_future,
     &         pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltDHWDraw%Identifier,
     &         DHW_draw_future,
     &         pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltDHWMakeupTemp%Identifier,
     &         T_makeup_future,
     &         pcname(IPCOMP)(1:iNameLength))

       endif




C---------------------------------------------------------------------------------
C Store results to calculate monthly averages and (optionally) to produce an
C ASCII file with time-step output.
C
C  - Note: report "present" (ie. last timestep values of all state variables)
C          since they are not available for the current tiemstep yet
C---------------------------------------------------------------------------------
C      IF(ISTATS.eq.1) CALL H3KSTORE_FuelCell_Fossil_Tank(
C     &     Vdot_fuel_STP_t,     ! fuel supply
C     &     q_capture_future,    ! burner -> water
C     &     T_node1_present,     ! tank temperature  (state variable)
C     &     T_1a_present,        ! temp of water at 1st connection (state variable)
C     &     mdot_conn1a_present, ! flow rate of water at 1st connection (state variable)
C     &     T_1b_present,         ! temp of water at 2nd connection (state variable)
C     &     mdot_conn1b_present, ! flow rate of water at 2nd connection (state variable)
C     &     T_makeup_future,     ! temp of DHW make-up water
C     &     DHW_draw_future,     ! flow rate of DHW draw
C     &     tank_UA_future,      ! UA of tank
C     &     Troom_future,        ! Predicted future temp of Room (from building domain)
C     $     q_heatdump_future    ! rate of energy dumped
C     &                        )



      RETURN
      END






C *********************************************************************************************
C ****************************** ADS_tank_elec_static_temp *************************************
C Created by: Maria Mottillo
C Adapted from 'FC_tank_elec_static_temp' developed by Ian Beausoleil-Morrison
C Initial Creation Date: October 21, 2004

C This subroutine is the static template for an electrically heated water tank
C that is fed by either a residential fuel cell system or an adsorption storage unit.
C The original electrically heated water tank component model from which this
C model was adapted allowed only 2 connections whereas this component model allows 3.
C The component model was developed for connection to the residential fuel cell
C component model and the adsorption storage  component model but could be used
C in other plant arrangements as well.
C The model supports the following configurations :
C   1) a combo tank that supplies a fan-coil for space heat + domestic hot water (DHW)
C   2) a tank that supplies a fan-coil for space heat
C   3) a tank that supplies DHW


C This subroutine checks whether the component is properly used in the user-defined
C plant network, it assigns user-input data to module variables, and it performs
C time-invariant calculations.

C The tank, including its electric resistance heater, is represented with a single node:

C INPUTS:
C    IPCOMP       index number of the component under consideration
C    NPCOMP       number of plant components in user-defined plant network
C    NCI(i)       holds the number of possible plant control variables for plant component `i'
C    ADATA(i,j)   holds `j'th data item for component `i' read from .pln file
C                 (derived from plant components database)
C    ICONTP(i)    holds the state variable index (ISV) of the sending node for connection `i'.
C    IUOUT        output channel to feedback to user
C    ITC          trace output index (>0 gives a trace output)
C    ITU          output channel for trace output
C    ITRACE(35)   flag indicating one of the plant traces (1= active)
C    FC_DHW_CSA_fraction(j)    Nominal DHW draw (L/hour) for hour `j' based on CSA schedule
C                              (in MODULE FC_tanks)
C    NPCDAT(i,4)  pointer to static template and coefficient generator for component `i'
C                 (in common/C12PS)

C OUTPUTS:
C    tank_mass                 mass of node 1, kg (in MODULE FC_tanks)
C    tank_Cp                   specific heat of node 1, J/kgK (in MODULE FC_tanks)
C    tank_UA                   heat loss coefficient between node 1 and surroundings,
C                              W/K (in MODULE FC_tanks)
C    element_capacity_ON       resistance heater heat output when on, W (in MODULE FC_tanks)
C    element_capacity_OFF      resistance heater heat output when off, W (in MODULE FC_tanks)
C    tank_DHW_draw             flag indicating whether tank supplies DHW (in MODULE FC_tanks)
C    tank_space_heat           flag indicating whether tank is connected to a fan-coil
C                              that supplies space heating (in MODULE FC_tanks)
C    FC_DHW_nominal_Lperday    Nominal DHW draw (L/day) based on occupancy (in MODULE FC_tanks)
C    FC_DHW_CSA_Lperday        Nominal DHW draw (L/day) based on CSA schedule
C                              (in MODULE FC_tanks)
C -------------------------------------------------------------------------------------------

      SUBROUTINE ADS_tank_elec_static_temp(IPCOMP)
      IMPLICIT NONE
#include "plant.h"
#include "building.h"
#include "SOFC.h"
#include "cogen_tanks.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER IUOUT,IUIN,IEOUT

      COMMON/TC/ITC,ICNT
      INTEGER ITC,ICNT

      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      INTEGER ITCF,ITRACE,IZNTRC,ITU

      COMMON/C9/
     & NPCOMP,
     & NCI(MPCOM),
     & CDATA(MPCOM,MMISCD)

      INTEGER NPCOMP
      INTEGER NCI
      REAL CDATA


      COMMON/PDBDT/
     & ADATA(MPCOM,MADATA),
     & BDATA(MPCOM,MBDATA)

      REAL ADATA
      REAL BDATA

      COMMON/PCOND/
     & CONVAR(MPCON,MCONVR),
     & ICONTP(MPCON),
     & ICONDX(MPCOM,MNODEC,MPCONC)

      REAL CONVAR
      INTEGER ICONTP
      INTEGER ICONDX


      COMMON/C12PS/
     & NPCDAT(MPCOM,9),
     & IPOFS1(MCOEFG),
     & IPOFS2(MCOEFG,MPVAR)

      INTEGER NPCDAT
      INTEGER IPOFS1
      INTEGER IPOFS2

C---------------------------------------------------------------------------------
C Declare local variables.
C---------------------------------------------------------------------------------
      INTEGER NumADATA,Itemp,N_expect,IPCONC
      LOGICAL mistake

      INTEGER IPCOMP,I,J


C---------------------------------------------------------------------------------
C The electrically heated and gas-fired water tanks cannot be used concurrently
C because they share common variables through MODULE FC_tanks. If there is a need
C in the future to allow multiple storage tanks, this restriction can be overcome
C by creating separate MODULEs for the two component models.
C Variables used:
C    NPCDAT(i,4)  pointer to static template and coefficient generator for component `i'
C                 (in common/C12PS): =81 for gas-fired component model (2 connections),
C                                    =111 for gas-fired component model (3 connections),
C                                    =82 for electric component model (2 connections).
C    NPCOMP       number of plant components in user-defined plant network
C---------------------------------------------------------------------------------
      DO I=1,NPCOMP
        IF( (NPCDAT(I,4) .eq. 81) .OR. (NPCDAT(I,4) .eq. 111) .OR.
     &      (NPCDAT(I,4) .eq. 82) )THEN
          WRITE(IUOUT,*) ' ADS_tank_elec_static_temp: the elec'
          WRITE(IUOUT,*) ' water tank with 3 connections cannot'
          WRITE(IUOUT,*) ' be used concurrently with the gas-fired'
          WRITE(IUOUT,*) ' tank (3 or 2 connections) or with the'
          WRITE(IUOUT,*) ' elec water tank with 2 connections.'
          STOP ' ADS_tank_elec_static_temp: unresolvable error'
        END IF
      END DO


C---------------------------------------------------------------------------------
C Write out ADATA if there is a trace output. Note that there is no BDATA
C used with this component.
C---------------------------------------------------------------------------------
      IF(ITC.gt.0 .AND. ITRACE(35).ne.0) THEN
        WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
        WRITE(ITU,*) ' 1 node elec heated water tank  '
        NumADATA = 7  ! Number of ADATA items.
        WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NumADATA)
        IF(ITU.eq.IUOUT) THEN  ! trace output going to screen, not file
          itemp=(IPCOMP/5)*5
          IF(itemp.eq.IPCOMP .OR. IPCOMP.eq.NPCOMP) call epagew ! write 5 lines at a time.
        END IF
      END IF


C---------------------------------------------------------------------------------
C Ensure that user has specified the correct number of control variables in
C .pln file. NCI(IPCOMP) holds the number of possible plant control variables
C as specified in the .pln file. There should be a single control variable.
C---------------------------------------------------------------------------------
      N_expect = 1
      IF(NCI(IPCOMP) .ne. N_expect) THEN
        WRITE(ITU,*) ' ADS_tank_elec_static_temp warning: ',
     &               ' incorrect num of ctl variables specified.'
      ENDIF



C---------------------------------------------------------------------------------
C Check that the node in the component has the correct number of connections
C to other components.
C Variables used:
C    MPCONC           the maximum allowable connections to a node (from plant.h).
C    ICONDX(i,j,k)    the connection number for the k'th connection to the j'th node
C                     of component i. It is used as a pointer.
C    `mistake'        a flag indicating whether there are connection errors:
C                     .true. means there are errors.
C---------------------------------------------------------------------------------
      mistake = .false.
C-----There should be three connections to node 1.
      IF( ICONDX(IPCOMP,1,1) .eq. 0 ) mistake=.true.
      IF( ICONDX(IPCOMP,1,2) .eq. 0 ) mistake=.true.
      IF( ICONDX(IPCOMP,1,3) .eq. 0 ) mistake=.true.
      DO IPCONC=4,MPCONC
        IF( ICONDX(IPCOMP,1,IPCONC) .ne. 0 ) mistake=.true.
      END DO
C-----Write error message if the number of connections to the node is incorrect.
      IF(mistake)THEN
        WRITE(IUOUT,*) ' ADS_tank_elec_static_temp: incorrect'
        WRITE(IUOUT,*) ' number of connections for component ',IPCOMP
        STOP ' ADS_tank_elec_static_temp: unresolvable error'
      END IF


C---------------------------------------------------------------------------------
C Check that the connections to node 1 are of the correct type. The three connections
C to node 1 should be of type ISV=20 so that the tank component can be used in
C networks in which both first and second phase mass balances are formed.
C Variables used:
C    ICONTP(i)  holds the state variable index (ISV) of the sending node for connection `i'.
C    ISV        defines nodal fluid type & coefficient generator model capabilities:
C               ISV=0,10,20 node represents water + ....
C               ISV=1,11,21 node represents dry air + ....
C               ISV=9,19,29 node represents some solid material only
C               0 <=ISV<10  model suitable for energy balance only
C               10<=ISV<20  model suitable for energy + single phase mass balance
C               20<=ISV<30  model suitable for energy + two phase mass balances
C---------------------------------------------------------------------------------
      mistake = .false.
      IF( ICONTP( ICONDX(IPCOMP,1,1) ) .ne. 20 ) mistake=.true.
      IF( ICONTP( ICONDX(IPCOMP,1,2) ) .ne. 20 ) mistake=.true.
      IF( ICONTP( ICONDX(IPCOMP,1,3) ) .ne. 20 ) mistake=.true.
      IF(mistake)THEN
        WRITE(IUOUT,*) ' ADS_tank_elec_static_temp: incorrect'
        WRITE(IUOUT,*) ' connection type to node 1 for comp ',IPCOMP
        STOP ' ADS_tank_elec_static_temp: unresolvable error'
      END IF


C---------------------------------------------------------------------------------
C Assign user-specified mass (kg) and specific heat (J/kgK) of node 1
C (water in tank + casing).
C---------------------------------------------------------------------------------
      tank_mass = ADATA(IPCOMP,1)
      tank_Cp   = ADATA(IPCOMP,2)


C---------------------------------------------------------------------------------
C Assign user-specified heat loss coefficient (UA-value) between node 1 and
C surroundings (W/K).
C---------------------------------------------------------------------------------
      tank_UA = ADATA(IPCOMP,3)


C---------------------------------------------------------------------------------
C Assign user-specified resistance heater element capacity (W) for on and off
C states. Off-state will normally have zero heating capacity, but the user can
C alter this.
C---------------------------------------------------------------------------------
      element_capacity_ON  = ADATA(IPCOMP,4)
      element_capacity_OFF = ADATA(IPCOMP,5)


C---------------------------------------------------------------------------------
C Assign variable indicating whether there is a DHW as well as a space-heating
C draw from the storage tank. The DHW draw is not treated as a `connection'
C but rather is handled within the coefficient generator as a right-hand side
C term, this to simplify the plant networks.
C---------------------------------------------------------------------------------
      tank_DHW_draw = ADATA(IPCOMP,6)


C---------------------------------------------------------------------------------
C Assign variable indicating whether the tank is connected to a fan-coil that
C supplies space heat.  The tank may be configured to supply DHW, space-heat,
C or both.
C---------------------------------------------------------------------------------
      tank_space_heat = ADATA(IPCOMP,7)


C---------------------------------------------------------------------------------
C Determine the DHW nominal water draw. This equation is taken from the HOT3000
C DHW model, which took it directly from the HOT2000 bin model.
C---------------------------------------------------------------------------------
      FC_DHW_nominal_Lperday = 85. + ( 35. * float(FC_NumOfOccupants) )


C---------------------------------------------------------------------------------
C Determine the fraction of the daily DHW water draw that is drawn at each hour of
C the day. The actual water draws will be determined using these hourly profiles,
C the total DHW draw over the day, and the time-varying tank temperature.
C---------------------------------------------------------------------------------
      FC_DHW_CSA_Lperday = 0.
      DO j=1,24
        FC_DHW_CSA_Lperday = FC_DHW_CSA_Lperday
     &                     + FC_DHW_CSA_Lperhour(j)
      END DO
      DO j=1,24
        FC_DHW_CSA_fraction(j) =
     &                       FC_DHW_CSA_Lperhour(j) / FC_DHW_CSA_Lperday
      END DO


C---------------------------------------------------------------------------------
C Completion of time-invariant operations for component.
C---------------------------------------------------------------------------------
      RETURN
      END



C *********************************************************************************************
C ******************************** ADS_tank_elec_coeff_gen *************************************
C Created by: Maria Mottillo
C Adapted from FC_tank_elec_coeff_gen developed by Ian Beausoleil-Morrison
C Initial Creation Date: October 21, 2004

C This subroutine is the coefficient generator for an electrically heated water
C tank that is fed by either a residential fuel cell system or an adsorption
C storage unit. The original electrically heated water tank component
C model from which this model was adapted allowed only 2 connections whereas this
C component model allows 3. The component model was developed for connection to
C the residential fuel cell component model and the adsorption storage  component
C model but could be used in other plant arrangements as well.
C The model supports the following configurations :
C   1) a combo tank that supplies a fan-coil for space heat + domestic hot water (DHW)
C   2) a tank that supplies a fan-coil for space heat
C   3) a tank that supplies DHW

C The DHW loads are treated as `connectionless'.  The DHW water draws and make-up water
C temperature are calculated within the coefficient generator.  These appear on the RHS
C of the energy balance for the water node.  This is done to minimize the number of
C components and connections and to allow for time-varying draws and make-up water
C temperatures.

C The tank is configured to be connected to a fuel cell. A second connection to the
C water node is also required.  A flag can be set to indicate whether the tank supplies
C a fan-coil unit for space heating.  If the tank does supply space heat, then this
C second connection is to the fan-coil system.  If the tank does not supply space heat,
C then a second connection is required to satisfy the matrix requirements.  However,
C this connection is a `dummy'.  This coefficient generator nullifies the impact of
C this second connection.  A constant temperature supply is recommended as the inlet
C for this dummy connection: a pipe is recommended for the output. A third connection
C to a water node is also required. The third connection is configured for an
C adsorption storage unit.

C This subroutine performs the time-step calculations and establishes coefficients for the
C sub-matrices that define the energy, 1st phase mass flow, and 2nd phase mass
C flow balances on the storage tank's node. A weighted implicit-explicit energy balance
C is formed for the single node.

C The tank, including the electric resistance heating element, is represented with
C a single node.

C INPUTS:
C    tank_mass                 mass of node 1, kg (in MODULE FC_tanks)
C    tank_Cp                   specific heat of node 1, J/kgK (in MODULE FC_tanks)
C    tank_UA                   heat loss coefficient between node 1 and surroundings,
C                              W/K (in MODULE FC_tanks)
C    element_capacity_ON       resistance heater heat output when on, W (in MODULE FC_tanks)
C    element_capacity_OFF      resistance heater heat output when off, W (in MODULE FC_tanks)
C    tank_DHW_draw             flag indicating whether tank supplies DHW (in MODULE FC_tanks)
C    tank_space_heat           flag indicating whether tank is connected to a fan-coil
C                              that supplies space heating (in MODULE FC_tanks)
C    PCNTMP(i)                 present time-row temperature of room containing
C                              component `i' (common/PCVAR)
C    PCNTMF(i)                 future time-row temperature of room containing
C                              component `i' (common/PCVAR)
C    IMPEXP                    identifies implicit/explicit handling of plant
C                              equations (common/PCEQU)
C    RATIMP                    user-specified implicit weighting fraction (common/PCEQU)
C    TIMSEC                    time-step in explicit plant domain (seconds) (common/PCTIME)
C    CDATA(i,j)   control signal for component `i', node `j' (common/C9)

C OUTPUTS:
C    COUT(i)             coefficient for ISTATS balance for `i'th matrix position
C    NAPDAT(i)           number of "additional" output for component `i'
C    PCAOUT(i,j)         additional output `j' for component `i'
C    TC(i)               time constant (seconds) of plant component `i'
C -------------------------------------------------------------------------------------------

      SUBROUTINE ADS_tank_elec_coeff_gen(IPCOMP,COUT,ISTATS)
      use h3kmodule
      IMPLICIT NONE
#include "plant.h"
#include "building.h"
#include "SOFC.h"
#include "cogen_tanks.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER IUOUT,IUIN,IEOUT

      COMMON/TC/ITC,ICNT
      INTEGER ITC,ICNT

      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      INTEGER ITCF,ITRACE,IZNTRC,ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      INTEGER IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow

      COMMON/PITER/
     & MAXITP,
     & PERREL,
     & PERTMP,
     & PERFLX,
     & PERMFL,
     & itrclp,
     & ICSV(MPNODE,MPVAR),
     & CSVI(MPNODE,MPVAR)

      INTEGER MAXITP
      REAL PERREL
      REAL PERTMP
      REAL PERFLX
      REAL PERMFL
      INTEGER itrclp
      INTEGER ICSV
      REAL CSVI

      COMMON/C9/
     & NPCOMP,
     & NCI(MPCOM),
     & CDATA(MPCOM,MMISCD)

      INTEGER NPCOMP
      INTEGER NCI
      REAL CDATA

      COMMON/C10/
     & NPCON,
     & IPC1(MPCON),
     & IPN1(MPCON),
     & IPCT(MPCON),
     & IPC2(MPCON),
     & IPN2(MPCON),
     & PCONDR(MPCON),
     & PCONSD(MPCON,2)

      INTEGER NPCON
      INTEGER IPC1
      INTEGER IPN1
      INTEGER IPCT
      INTEGER IPC2
      INTEGER IPN2
      REAL PCONDR
      REAL PCONSD

      COMMON/C12PS/
     & NPCDAT(MPCOM,9),
     & IPOFS1(MCOEFG),
     & IPOFS2(MCOEFG,MPVAR)

      INTEGER NPCDAT
      INTEGER IPOFS1
      INTEGER IPOFS2


      COMMON/PCVAL/
     & CSVF(MPNODE,MPVAR),
     & CSVP(MPNODE,MPVAR)

      REAL CSVF
      REAL CSVP


      COMMON/PCVAR/
     & PCTF(MPCON),
     & PCRF(MPCON),
     & PUAF(MPNODE),
     & PCQF(MPNODE),
     & PCNTMF(MPCOM),
     & PCTP(MPCON),
     & PCRP(MPCON),
     & PUAP(MPNODE),
     & PCQP(MPNODE),
     & PCNTMP(MPCOM)

      REAL PCTF
      REAL PCRF
      REAL PUAF
      REAL PCQF
      REAL PCNTMF
      REAL PCTP
      REAL PCRP
      REAL PUAP
      REAL PCQP
      REAL PCNTMP

      COMMON/PCOND/
     & CONVAR(MPCON,MCONVR),
     & ICONTP(MPCON),
     & ICONDX(MPCOM,MNODEC,MPCONC)

      REAL CONVAR
      INTEGER ICONTP
      INTEGER ICONDX


      COMMON/PCRES/
     & QDATA(MPCOM),
     & PCAOUT(MPCOM,MPCRES),
     & napdat(mpcom)

      REAL QDATA
      REAL PCAOUT
      INTEGER napdat

      COMMON/PCTC/
     &TC(MPCOM)

      REAL TC

      COMMON/PCEQU/
     & IMPEXP,
     & RATIMP

      INTEGER IMPEXP
      REAL RATIMP

      COMMON/PCTIME/
     & TIMSEC

      REAL TIMSEC

      common/pcnam/pcname(mpcom)       ! Plant component names
      character*15 pcname

C---------------------------------------------------------------------------------
C Declare local variables. These variables are defined at their first use.
C---------------------------------------------------------------------------------
      LOGICAL CLOSE
      REAL COUT(MPCOE) ! plant matrix coefficients (passed in calling statement)
      REAL tank_UA_future,tank_UA_present
      REAL T_node1_present,T_node1_future
      REAL mdotCp_1a_future,mdotCp_1b_future,mdotCp_1c_future,
     &     mdotCp_1a_present,mdotCp_1b_present,mdotCp_1c_present
      REAL T_1a_present,T_1b_present,T_1c_present
      REAL Rinv,alpha

      REAL part1,part2,part3,part4,part5,part6,part7,part8,part9

      REAL  H3K_Connect_property   !- function used to return temperature of most recently
                                   !- calculated temperature (oC)
      INTEGER IPCOMP
      INTEGER ISTATS
      INTEGER ITEMP
      REAL SHTFLD
      REAL RHOFLD
      INTEGER I

      INTEGER lnblnk                ! implicit function returning the length of a string

      INTEGER iNameLength           ! temporary variable length

C---------------------------------------------------------------------------------
C Start trace if trace output requested.
C---------------------------------------------------------------------------------
      IF(ITC.gt.0 .AND. NSINC.ge.ITC .AND.NSINC.le.ITCF .AND.
     &   ITRACE(37).ne.0)THEN
        WRITE(ITU,*) ' Entering sub ADS_tank_fuel_coeff_gen'
      END IF


C---------------------------------------------------------------------------------
C Set local variables to point to the nodes and to the connections.
C Variables used:
C    ICONDX(i,j,k)  the connection number for the k'th connection to the j'th node
C                   of component i. It is used as a pointer.
C    NPCDAT(i,9)    row and column number defining location of component `i'
C                   sub-matrix template in the overall plant network matrix. It is
C                   used to identify the location of the current component's nodes
C                   within the global plant matrix.
C    node1          global matrix node number for single node
C    con1a          number of first connection to node 1 (to fuel cell, adsorber or fan coil)
C    con1b          number of second connection to node 1 (to fuel cell, adsorber or fan coil)
C    con1c          number of third connection to node 1 (to fuel cell, adsorber or fan coil)
C---------------------------------------------------------------------------------
      node1 = NPCDAT(IPCOMP,9)
      con1a = ICONDX(IPCOMP,1,1)
      con1b = ICONDX(IPCOMP,1,2)
      con1c = ICONDX(IPCOMP,1,3)


C---------------------------------------------------------------------------------
C Mark the temperature of node 1 for iteration.
C Variables used:
C     ICSV(i,j)  flag indicating that node `i' is marked for iteration for state
C                variable `j'; j=1 for temperature, j=2 for 1st phase mass flow,
C                j=3 for 2nd phase mass flow.
C     CSVI(i,j)  initial value for judging whether iteration required. Same
C                indices as ICSV.
C     CSVF(i,j)  future time-row solution variable for plant. Same indices as ICSV.
C---------------------------------------------------------------------------------
      ICSV(node1,1) = 1
      CSVI(node1,1) = CSVF(node1,1)


C---------------------------------------------------------------------------------
C Generate coefficients for the appropriate equation set. This subroutine was
C called to either generate coefficients for energy balances (ISTATS=1), 1st
C phase mass balances (ISTATS=2), or 2nd phase mass balances (ISTATS=3).
C Start with the energy balances.
C---------------------------------------------------------------------------------

C*********************************************************************************
C Beginning of energy balance section ********************************************
C*********************************************************************************
      IF(ISTATS.eq.1) THEN


C---------------------------------------------------------------------------------
C Determine the `present' and `future' temperature of the room containing the
C water tank. This is used to calculate the heat lost from the hot tank to the
C room (Beausoleil-Morrison 2001b, eq 4).
C Variables used:
C    PCNTMP(i)  present time-row temperature of room containing component `i'.
C    PCNTMF(i)  future time-row temperature of room containing component `i';
C               =-99 if no containment defined.
C---------------------------------------------------------------------------------
        CALL ECLOSE(PCNTMF(IPCOMP),-99.0,0.001,CLOSE)
        IF(CLOSE)THEN
          WRITE(IUOUT,*) ' ADS_tank_elec_coeff_gen: '
          WRITE(IUOUT,*) ' the storage tank must be contained within '
          WRITE(IUOUT,*) ' a room.'
          STOP ' ADS_tank_tank_coeff_gen: unresolvable error'
        ELSE
          Troom_present = PCNTMP(IPCOMP)
          Troom_future  = PCNTMF(IPCOMP)
        END IF


C---------------------------------------------------------------------------------
C Establish the heat loss coefficient (UA-value, W/K) between node 1 and the
C surroundings for the present and future time-rows. A fixed UA-value is used here.
C However, this code is structured to easily enable a time-varying UA-value in the
C future. PUAP and PUAF are standard ESP-r variables holding the present and future
C time-row values of the UA-value. PUAF is mapped to PUAP by subroutine MZNASS
C following the solution of the energy matrix for the current time-step.
C Variables used:
C    PUAP(i)  present time-row UA-value
C    PUAF(i)  future time-row UA-value
C---------------------------------------------------------------------------------
C-------Determine UA.
        tank_UA_future  = tank_UA  ! `tank_UA' from dbase (could be replaced by a correlation)
C-------Save future value.
        PUAF(node1)     = tank_UA_future
C-------Set present value.
        tank_UA_present = PUAP(node1)


C---------------------------------------------------------------------------------
C     Determine most recently calculated mass flow:
C     CONVAR(i,2) water mass flow rate (kg/s) at sending node for connection `i'
C     PCONDR(i)   ratio of flow rate leaving sending node `i' that reaches
C                receiving node.
C---------------------------------------------------------------------------------
         mdot_conn1a_future = PCONDR(con1a) * CONVAR(con1a,2)  ! (kg/s)
         mdot_conn1b_future = PCONDR(con1b) * CONVAR(con1b,2)  ! (kg/s)
         mdot_conn1c_future = PCONDR(con1c) * CONVAR(con1c,2)  ! (kg/s)

C---------------------------------------------------------------------------------
C Determine the products of mass flow and heat capacity (W/K) for the three
c connections to node 1, for the present and future time-rows. These are the heat
C capacity rates (W/K) of the the water entering the storage tank from the fuel
C cell, from the fan-coil return and from the adsorber. PCRP and PCRF are standard
C ESP-r variables holding the present and future time-row values of the
C mass flow * heat capacity. PCRF is mapped to PCRP by subroutine MZNASS following
C the solution of the energy matrix for the current time-step.
C Variables used:
C    CONVAR(i,1) temperature (oC) of sending node for connection `i'
C    CONVAR(i,2) water mass flow rate (kg/s) at sending node for connection `i'
C    PCONDR(i)   ratio of flow rate leaving sending node `i' that reaches
C                receiving node.
C    SHTFLD(3,T) function that returns heat capacity (J/kgK) of water (index 3)
C                at temperature `T'.
C    PCRP(i)     present time-row mass flow * heat capacity for connection `i'
C    PCRF(i)     future time-row mass flow * heat capacity for connection `i'
C---------------------------------------------------------------------------------
         mdotCp_1a_future = mdot_conn1a_future
     &                    * SHTFLD( 3,CONVAR(con1a,1) )
         mdotCp_1c_future = mdot_conn1c_future
     &                    * SHTFLD( 3,CONVAR(con1c,1) )
C--------If space-heat flag set, then second connection is real: otherwise make flow zero.
         IF( tank_space_heat.eq.tank_yes_heat )THEN
           mdotCp_1b_future = mdot_conn1b_future
     &                      * SHTFLD( 3,CONVAR(con1b,1) )
         ELSE
           mdotCp_1b_future = 0.
         END IF
C--------Save future values.
         PCRF(con1a) = mdotCp_1a_future
         PCRF(con1c) = mdotCp_1c_future
         IF( tank_space_heat==tank_yes_heat )THEN
           PCRF(con1b) = mdotCp_1b_future
         END IF
C--------Set present values.
         mdotCp_1a_present = PCRP(con1a)
         mdotCp_1c_present = PCRP(con1c)
         IF( tank_space_heat.eq.tank_yes_heat )THEN
           mdotCp_1b_present = PCRP(con1b)
         ELSE
           mdotCp_1b_present = 0.
         END IF

C...Save future values

         PCTF(con1a) = H3K_Connect_property(con1a,1)
         PCTF(con1c) = H3K_Connect_property(con1c,1)
         PCTF(con1b) = H3K_Connect_property(con1b,1)


C---------------------------------------------------------------------------------
C Establish the present time-row temperatures of the water flowing into the tank
C at the three connections.  PCTP and PCTF are standard ESP-r variables holding the
C present and future time-row values of the temperature of the fluid flowing at
C the connection. PCTF is mapped to PCTP by subroutine MZNASS following the
C solution of the energy matrix for the current time-step.
C Variables used:
C    CONVAR(i,1) temperature (oC) of sending node for connection `i'
C    PCTP(i)     present temperature of sending node for connection `i'
C    PCTF(i)     future temperature of sending node for connection `i'
C---------------------------------------------------------------------------------
         T_1a_present = PCTP(con1a)
         T_1c_present = PCTP(con1c)
C--------If space-heat flag set, then second connection is real: otherwise make temp zero.
         IF( tank_space_heat.eq.tank_yes_heat )THEN
           T_1b_present = PCTP(con1b)
         ELSE
           T_1b_present = 0.
         END IF


C---------------------------------------------------------------------------------
C Establish the present time-row temperature of node 1.
C Variables used:
C     CSVP(i,j)  present time-row solution variable for node `i' and variable `j';
C                j=1 for temperature
C     CSVF(i,j)  future time-row solution variable (same indices as with CSVP)
C---------------------------------------------------------------------------------
         T_node1_present = CSVP(node1,1)
         T_node1_future  = CSVF(node1,1)


C---------------------------------------------------------------------------------
C Check the temperature of the tank and activiate the fictitious heat dump if
C necessary to maintain the temperature within a reasonable range. This should
C prevent the occurence of boiling and will allow the examination of scenarios for
C dealing with excessive energy build-up in the system.
C---------------------------------------------------------------------------------

C Note: heat dump setpoint temperatures are specified in cogen_tanks.h

         CALL FC_dump_excess_energy(node1)



C---------------------------------------------------------------------------------
C Check the temperature of the tank to ensure that the water is not going to
C boil. This may occur is the fuel cell consistently adds more energy to the
C tank than is drawn off to meet the space heating and/or DHW loads. This should
C never occur with addition of call above to `FC_dump_excess_energy', but left
C here anyway for safety.
C---------------------------------------------------------------------------------

C Note: Error message if temperature > 95C

         CALL FC_check_for_boil(node1)


C---------------------------------------------------------------------------------
C Determine the products of mass flow and heat capacity (W/K) for the DHW draw
C on the tank, for the present and future time-rows. Also determine the temperature
C of the DHW make-up water, for the present and future time-rows.
C Variables used:
C    RHOFLD(3,T)   function that returns density (kg/m3) of water (index 3) at
C                  temperaturere `T'.
C    SHTFLD(3,T)   function that returns heat capacity (J/kgK) of water (index 3)
C                  at temperature `T'.
C
C    Modification: Check to see if the tank
C---------------------------------------------------------------------------------
        CALL FC_DHW_load(T_node1_future)
        mdotCp_DHW_present = DHW_draw_present
     &                     * RHOFLD(3,T_makeup_present)
     &                     * SHTFLD( 3,T_makeup_present )
        mdotCp_DHW_future  = DHW_draw_future
     &                     * RHOFLD(3,T_makeup_future)
     &                     * SHTFLD( 3,T_makeup_future )


C---------------------------------------------------------------------------------
C Calculate the energy added by the electric resistance heating element. The
C heating element will either be on or off, depending on the control signal.
C CDATA=0 signals that the heating element should be off; CDATA=1 signals that it
C should be on.
C    CDATA(i,j)   control signal for component `i's node `j'.
C---------------------------------------------------------------------------------
C-------Determine whether the heating element is on or off this time-step.
        CALL ECLOSE( CDATA(IPCOMP,1),0.,0.0001,CLOSE )
        IF(CLOSE)THEN  ! Element is in "off" state (there may be heat output still)
          q_element = element_capacity_OFF
        ELSE
          q_element = element_capacity_ON
        ENDIF


C---------------------------------------------------------------------------------
C All of the energy liberated from the heating element is transferred to the water
C PCQP and PCQF are standard ESP-r variables holding the present and future
C time-row values of nodal injections (the energy transferred from the combustion
C to the water in this case). PCQF is mapped to PCQP by subroutine MZNASS
C following the solution of the energy matrix for the current time-step.
C Variables used:
C    PCQP(i)     present energy transferred to water from combustion for node `i'
C    PCQF(i)     future energy transferred to water from combustion for node `i'
C---------------------------------------------------------------------------------
        q_capture_future = q_element
C-------Save future value.
        PCQF(node1) = q_capture_future
C-------Set present value.
        q_capture_present = PCQP(node1)


C---------------------------------------------------------------------------------
C Calculate component's time-constant for the current time-step. Refer to
C Hensen (1991) pages 5.9 and 5.10.  The environmental losses from the tank and
C the water streams flowing into the tank are the three heat transfers that
C affect the transient storage. Therefore, treat these as parallel paths for
C the purposes of calculating the time constant, as described in eq 5.5 of
C Hensen (1991).
C Variables used:
C     tank_mass    mass of node 1, kg
C     tank_Cp      specific heat of node 1, J/kgK
C     TC(i)        time constant (seconds) of plant component `i'
C---------------------------------------------------------------------------------
        Rinv = tank_UA_future + mdotCp_1a_future + mdotCp_1b_future +
     &         mdotCp_1c_future
        CALL ECLOSE(Rinv,0.,0.001,CLOSE)
        IF(CLOSE)THEN
          WRITE(IUOUT,*) ' ADS_tank_elec_coeff_gen: '
          WRITE(IUOUT,*) ' impossible time constant. '
          STOP ' ADS_tank_elec_coeff_gen: unresolvable error'
        END IF
        TC(IPCOMP) = tank_mass*tank_Cp / Rinv


C---------------------------------------------------------------------------------
C Determine implicit/explicit weighting for the energy balance on node 1. `alpha'
C is the implicit weighting factor used to form the energy balance this time-step.
C It is the Greek "alpha" in Beausoleil-Morrison (2001b) equation 7. The approach
C used here to determine `alpha' is used with most plant components, as discussed
C by Hensen (1991).
C Variables used:
C    IMPEXP   identifies handling of plant equations: 1= fully implicit case;
C             2= mixed explicit/implicit; 3= 1 or 2 depending on component time
C             constant and time-step (this is the default treatment);
C             4= steady state.
C    RATIMP   user-specified implicit weighting fraction
C    TIMSEC   the time-step in the explicit plant domain (seconds)
C---------------------------------------------------------------------------------
C-------Fully implicit.
        IF( IMPEXP.eq.1 ) THEN
          alpha = 1.
C-------Implicit/explicit with user-specified (fixed) weighting.
        ELSE IF( IMPEXP .eq. 2 ) THEN
          alpha = RATIMP
C-------General case: implicit/explicit with calculated weighting.
        ELSE IF( IMPEXP .eq. 3 ) THEN
          IF( TIMSEC.gt.0.63*TC(IPCOMP) ) THEN
            alpha = 1.
          ELSE
            alpha = RATIMP
          END IF
C-------Steady-state.
        ELSE IF( IMPEXP.eq.4 ) THEN
          tank_mass = 0.  ! make node 1 massless
          alpha = 1.
        END IF


C---------------------------------------------------------------------------------
C Preparation for forming matrix coefficients for energy balances is now complete.
C Establish matrix equation self-coupling, cross-coupling, and RHS coefficients.
C Node coefficient transport for COUT

C       <-self->| <---cross-->
C  node   1     |  fc fan-coil adsorber   RHS
C       -------------------------------------
C         1     |   2     3     4         5
C---------------------------------------------------------------------------------

        COUT(1) = tank_mass * tank_Cp / TIMSEC       ! self-coupling to itself (W/K)
     &          + alpha * tank_UA_future
     &          + alpha * mdotCp_1a_future
     &          + alpha * mdotCp_1b_future
     &          + alpha * mdotCp_1c_future
     &          + alpha * mdotCp_DHW_future
        COUT(2) = -1. * alpha * mdotCp_1a_future     ! cross-coupling (W/K)
        COUT(3) = -1. * alpha * mdotCp_1b_future     ! cross-coupling (W/K)
        COUT(4) = -1. * alpha * mdotCp_1c_future     ! cross-coupling (W/K)
        part1 = tank_mass * tank_Cp / TIMSEC         ! (W/K)
     &        - (1.-alpha) * tank_UA_present
     &        - (1.-alpha) * mdotCp_1a_present
     &        - (1.-alpha) * mdotCp_1b_present
     &        - (1.-alpha) * mdotCp_1c_present
     &        - (1.-alpha) * mdotCp_DHW_present
        part2 = (1.-alpha) * mdotCp_1a_present       ! (W)
        part3 = (1.-alpha) * mdotCp_1b_present       ! (W)
        part9 = (1.-alpha) * mdotCp_1c_present
        part6 = (1.-alpha) * mdotCp_DHW_present      ! (W)
        part7 = alpha * mdotCp_DHW_future            ! (W)
        part4 = (1.-alpha) * q_capture_present       ! (W)
     &        + alpha * q_capture_future
        part5 = (1.-alpha) * tank_UA_present * Troom_present
     &        + alpha * tank_UA_future * Troom_future
        part8 = -1.*(1.-alpha) * q_heatdump_present  ! (W)
     &          -1*alpha * q_heatdump_future
        COUT(5) = part1 * T_node1_present            ! RHS (W)
     &          + part2 * T_1a_present
     &          + part3 * T_1b_present
     &          + part9 * T_1c_present
     &          + part6 * T_makeup_present
     &          + part7 * T_makeup_future
     &          + part4
     &          + part5
     &          + part8


C*********************************************************************************
C End of energy balance section / Beginning of 1st phase mass balances ***********
C*********************************************************************************
      ELSE IF(ISTATS.eq.2) THEN
C-------Node 1: total flow in equals total flow out.
        COUT(1) = 1.
        COUT(2) = -1. * PCONDR(con1a) ! (dimensionless)
C-------If space-heat flag set, then second connection is real: otherwise make flow zero.
        IF( tank_space_heat.eq.tank_yes_heat )THEN
          COUT(3) = -1. * PCONDR(con1b)
        ELSE
          COUT(3) = 0.
        END IF
        COUT(4) = -1. * PCONDR(con1c)
        COUT(5) = 0.


C*********************************************************************************
C End of energy 1st phase mass balances / Beginning of 2nd phase mass balances ***
C*********************************************************************************
      ELSE IF(ISTATS.eq.3) THEN
C-------Node 1: no balance required so make flow zero.
        COUT(1) = 1.
        COUT(2) = 0.
        COUT(3) = 0.
        COUT(4) = 0.
        COUT(5) = 0.

      END IF

C*********************************************************************************
C End of energy and mass balances ************************************************
C*********************************************************************************

C Record the electrical draw (W) of the electric resistance heater for use by the
C fuel cell controller.
      FCctl_elec_water_tank = q_element



C---------------------------------------------------------------------------------
C Save the `additional' output variables for outputting on time-step basis.
C---------------------------------------------------------------------------------
      NAPDAT(IPCOMP)    = 1                    ! number of additional outputs
      PCAOUT(IPCOMP,1)  = q_element            ! output of heating element (W)


C---------------------------------------------------------------------------------
C Complete trace if trace output requested.
C---------------------------------------------------------------------------------
      IF(ITC.gt.0 .AND. NSINC.ge.ITC .AND.NSINC.le.ITCF .AND.
     &   ITRACE(37).ne.0)THEN
        WRITE(ITU,*) ' Component      ',IPCOMP,':'
        WRITE(ITU,*) ' 1 node elec heated storage tank for fuel cell'
        WRITE(ITU,*) ' Matrix node(s) ',node1
        WRITE(ITU,*) ' Connection(s)  ',con1a,con1b,con1c
        IF(ISTATS.eq.1) THEN
           WRITE(ITU,*) ' CDATA         = ',CDATA(IPCOMP,1),' (-)'
           WRITE(ITU,*) ' q_element        = ',q_element,' (W)'
           WRITE(ITU,*) ' q_capture_future  = ',q_capture_future,' (W)'
           WRITE(ITU,*) ' alpha        = ',alpha,' (-)'
        END IF
        WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
        WRITE(ITU,*) (COUT(I),I=1,5)
        IF(ITU.eq.IUOUT) THEN  ! trace output going to screen, not file
          itemp=(IPCOMP/4)*4
          IF(itemp.eq.IPCOMP .OR. IPCOMP.eq.NPCOMP) call epagew ! write 4 lines at a time.
        END IF
        WRITE(ITU,*) ' Leaving sub ADS_tank_elec_coeff_gen'
      END IF

C----------------------------------------------------------------------------------
C XML output
C----------------------------------------------------------------------------------


C.....Get component name
      iNameLength = lnblnk(pcname(IPCOMP))

       call AddToReport(rvPltHeatingOut%Identifier,
     &        q_element,
     &        pcname(IPCOMP)(1:iNameLength))

       call AddToReport(rvPltHeatDumpFuture%Identifier,
     &        q_heatdump_future,
     &        pcname(IPCOMP)(1:iNameLength))

      RETURN
      END

C ******************** Press_WCH_pipe_static_temp ********************
C
C This subroutine is a modified version of CMP22S - the static
C template for the 1-node WCH pipe plant component (plant db code 220).
C The modified version of the WCH pipe plant component model allows
C the input of pressure in order to take into account properties
C of pressurized water.
C This static template subroutine whether the specified number
C of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE Press_WCH_pipe_static_temp(IPCOMP)
#include "plant.h"
#include "building.h"
C
      common/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
C
      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)
C
C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH pipe'
         NITMS=7
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=0
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' Press_WCH_pipe_static_temp warning: ',
     &                ' user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE
C
C
      RETURN
C
C Error handling
  990 WRITE(IUOUT,*) ' Press_WCH_pipe_static_temp: ',
     &               ' connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' Press_WCH_pipe_static_temp: unresolvable error'
C
      END


C ******************** Press_WCH_pipe_coeff_gen ********************

C This subroutine is a modified version of CMP22C - the coefficient
C generator subroutine for the 1-node WCH pipe plant component (plant db code 220).
C The modified version of the WCH pipe plant component model allows
C the input of pressure in order to take into account properties
C of pressurized water. The modified version of the WCH pipe plant component
C was used to model the adsorption storage unit and fuel cell plant network
C investigated for Maria Mottillo's thesis work.
C This coefficient generator subroutine determines the
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 UA modulus from wall to environment (W/K)
C            4 Hydraulic diameter of pipe (m)
C            5 Length of pipe (m)
C            6 Cross-sectional face area (m^2)
C            7 Pressure (kPa)
C     BDATA: none
C     CDATA: none

      SUBROUTINE Press_WCH_pipe_coeff_gen(IPCOMP,COUT,ISTATS)
#include "plant.h"
#include "building.h"

      common/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      logical closea

      REAL      Press         !-pressure, Pa
      REAL      Temp          !-temperature, K
      REAL      H20LiquidDensity    !-function that calculates density of water (kg/m^3)
                                    !-for a given temperature (K) and pressure (Pa)
      REAL      H20LiquidCp         !-function that calculates specific heat (J/kgK)
                                    !-for a given temperature (K) and pressure (Pa)

      PI = 4.0 * ATAN(1.0)

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine ',
     &                                 ' Press_WCH_pipe_coeff_gen'

C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)


C Obtain operating pressure
      Press = ADATA(IPCOMP,7) * 1000.          !-convert input from kPa to Pa

C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN


C First initialize UA modulus (for calculation of containment heat loss)
         UA=ADATA(IPCOMP,3)
         call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
         IF(closea) THEN
            UA=0.
         ELSE
            DH=ADATA(IPCOMP,4)
            PL=ADATA(IPCOMP,5)
            TW=CONVAR(ICON1,1)
            WMFR=PCONDR(ICON1)*CONVAR(ICON1,2)


            Temp = TW + 273.15                       !-convert from oC to K

C revised calculation of VW
            VW = WMFR / ( H20LiquidDensity(Temp,Press) *
     &                    ADATA(IPCOMP,6) )



            HW=1400.*(1.+0.015*TW)*VW**0.8*DH**(-0.2)
            UA=1./(1./(HW*PI*DH*PL)+1./UA)


C revised calculation of WMCP
            Temp = CONVAR(ICON1,1) + 273.15
            WMCP = WMFR * H20LiquidCp(Temp,Press)


            UA=WMCP*(EXP(UA/WMCP)-1.)
         END IF

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)


C revised calculation of C1
         Temp = CONVAR(ICON1,1) + 273.15
         C1 = PCONDR(ICON1) * CONVAR(ICON1,2) *
     &        H20LiquidCp(Temp,Press)


C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(C1+UA))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self- and cross-coupling coefficients
         COUT(1)=ALPHA*(-C1-UA)-CM/TIMSEC
         COUT(2)=ALPHA*C1
C and then present-time coefficient (ie. right hand side)
         COUT(3)=((1.-ALPHA)*(PCRP(ICON1)+PUAP(INOD1))
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &             -ALPHA*UA*PCNTMF(IPCOMP)
     &             -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)

C Store "environment" variables future values
         PUAF(INOD1)=UA
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1

C 1st phase mass (ie. water) balance coefficients
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         COUT(2)=-PCONDR(ICON1)
         COUT(3)=0.

C 2nd phase mass (ie. vapour) balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH pipe'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine ',
     &                                 ' Press_WCH_pipe_coeff_gen'



      RETURN
      END


C ******************** Press_WCH_pump_static_temp ********************
C
C This subroutine is a modified version of CMP24S - the static template
C subroutine for the 1-node WCH pump plant component (plant db code 240).
C The modified version of the WCH pump plant component model allows
C the input of pressure in order to take into account properties
C of pressurized water. The modified version of the WCH pump plant component
C was used to model the adsorption storage unit and fuel cell plant network
C investigated for Maria Mottillo's thesis work.

C The static template subroutine verifies whether the specified number
C of controlled variables is OK, and also
C whether the number of connections to this component is correct
C and whether the connected nodes are of a type as expected by the
C corresponding coefficient generator routine
C
      SUBROUTINE Press_WCH_pump_static_temp(IPCOMP)
#include "plant.h"
#include "building.h"

      common/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

C Establish static data derivable from the data read from database
      BDATA(IPCOMP,1)=ADATA(IPCOMP,4)
      BDATA(IPCOMP,2)=ADATA(IPCOMP,5)
      BDATA(IPCOMP,3)=ADATA(IPCOMP,6)
      BDATA(IPCOMP,4)=ADATA(IPCOMP,7)

C Trace output
      IF(ITC.GT.0.AND.ITRACE(35).NE.0) THEN
         WRITE(ITU,*) ' Component ',IPCOMP,' pre-simulation data for a:'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH pump'
         NITMS=3
         WRITE(ITU,*) ' ADATA ',(ADATA(IPCOMP,J),J=1,NITMS)
         NITMS=4
         WRITE(ITU,*) ' BDATA ',(BDATA(IPCOMP,J),J=1,NITMS)
         NITMS=1
         WRITE(ITU,*) ' CDATA ',(CDATA(IPCOMP,J),J=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/5)*5
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF
C
C Check user specified number of controlled variables
      NCITM=1
      IF(NCI(IPCOMP).NE.NCITM)
     &   WRITE(ITU,*) ' Press_WCH_pump_static_temp warning: ',
     &                ' user specified wrong number',
     &                ' of controlled variables'
C
C Check component has 1 connection only, to water
      NCONS=1
      DO 10 IPCONC=1,MPCONC
      IPCON=ICONDX(IPCOMP,1,IPCONC)
      IF(IPCONC.LE.NCONS) THEN
         IF(IPCON.EQ.0) THEN
            GOTO 990
         ELSE IF(ICONTP(IPCON).NE.20) THEN
            GOTO 990
         END IF
      ELSE IF(IPCON.NE.0) THEN
         GOTO 990
      END IF
   10 CONTINUE

      RETURN


C Error handling
  990 WRITE(IUOUT,*) ' Press_WCH_pump_static_temp_warning: ',
     &               ' connection error for component ',IPCOMP
      WRITE(IUOUT,*) '         should be ',NCONS,' water connection(s)'
      STOP ' Press_WCH_pump_static_temp_warning: unresolvable error'


      END



C ******************** Press_WCH_pump_coeff_gen ********************

C This subroutine is a modified version of CMP24C - the coefficient generator
C subroutine for the 1-node WCH pump plant component (plant db code 240).
C The modified version of the WCH pump plant component model allows
C the input of pressure in order to take into account properties
C of pressurized water. The modified version of the WCH pump plant component
C was used to model the adsorption storage unit and fuel cell plant network
C investigated for Maria Mottillo's thesis work.

C The coefficient generator subroutine generates the
C matrix equation coefficients COUT (in order: self-coupling, cross-
C coupling, and present-time coefficients) for energy balance (ISTATS=1),
C 1st phase mass balance (ISTATS=2), or 2nd phase mass (ISTATS=3)
C     ADATA: 1 Component total mass (kg)
C            2 Mass weighted average specific heat (J/kgK)
C            3 UA modulus (W/K)
C     BDATA: 1 Rated absorbed power (W)
C            2 Rated volume flow rate (m^3/s)
C            3 Overall efficiency (-)
C            4 Pressure (kPa)
C     CDATA: 1 Volume flow rate (m^3/s)

      SUBROUTINE Press_WCH_pump_coeff_gen(IPCOMP,COUT,ISTATS)
      use h3kmodule
#include "plant.h"
#include "building.h"
#include "net_flow.h"
#include "net_flow_data.h"

      common/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/Pctime/TIMSEC
      COMMON/PTIME/PTIMEP,PTIMEF
      COMMON/PCTC/TC(MPCOM)

      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/C9/NPCOMP,NCI(MPCOM),CDATA(MPCOM,MMISCD)
      COMMON/C10/NPCON,IPC1(MPCON),IPN1(MPCON),IPCT(MPCON),
     &           IPC2(MPCON),IPN2(MPCON),PCONDR(MPCON),PCONSD(MPCON,2)
      COMMON/C12PS/NPCDAT(MPCOM,9),IPOFS1(MCOEFG),IPOFS2(MCOEFG,MPVAR)
      COMMON/PDBDT/ADATA(MPCOM,MADATA),BDATA(MPCOM,MBDATA)
      COMMON/PCVAL/CSVF(MPNODE,MPVAR),CSVP(MPNODE,MPVAR)
      COMMON/PCVAR/PCTF(MPCON),PCRF(MPCON),PUAF(MPNODE),PCQF(MPNODE),
     &             PCNTMF(MPCOM),
     &             PCTP(MPCON),PCRP(MPCON),PUAP(MPNODE),PCQP(MPNODE),
     &             PCNTMP(MPCOM)
      COMMON/PCOND/CONVAR(MPCON,MCONVR),ICONTP(MPCON),
     &             ICONDX(MPCOM,MNODEC,MPCONC)

      COMMON/FFN/IFLWN,ICFFS(MPCON)
      COMMON/MFLRES/FLW1(MCNN),FLW2(MCNN),PRES(MNOD),
     &              RESID(MNOD),SAFLW(MNOD)

      COMMON/PCRES/QDATA(MPCOM),PCAOUT(MPCOM,MPCRES),napdat(mpcom)

C Electrical details for specified plant components
      common/pcelflg/ipcelf(mpcom)
      common/elpcp/NPEL,PFP(mpcom),IPFP(mpcom),PWRP(mpcom),
     &BVOLTP(mpcom),IPHP(mpcom)

      common/pcnam/pcname(mpcom)       ! Plant component names
      character*15 pcname

      DOUBLE PRECISION FLW1,FLW2,PRES,RESID,SAFLW
      PARAMETER (SMALL=1.0E-15)
      REAL      COUT(MPCOE)
      character outs*124

      INTEGER   iNameLength        ! temporary variabele name length

      logical closea

      REAL      Temp               !-temperature, K
      REAL      H20LiquidDensity   !-function calculates density of water (kg/m^3)
                                   !-for a given temperature (K) and pressure (Pa)
      REAL      H20LiquidCp        !-function calculations specific heat of water (J/kgK)
                                   !-for a given temperature (K) and pressure (Pa)
      REAL      fPowerDraw         !-power draw of pump (W)



C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Entering subroutine ',
     &                                 ' Press_WCH_pump_coeff_gen'

C Check control data for relevant balance type
      IF(ISTATS.EQ.2.AND.CDATA(IPCOMP,1).LT.0.) THEN
         CALL DAYCLK(IDYP,PTIMEF,IUOUT)
         WRITE(outs,*) ' Press_WCH_pump_coeff_gen: invalid control',
     &                 ' data for component ',
     &                  IPCOMP,' : ',CDATA(IPCOMP,1)
         call edisp(iuout,outs)
         call edisp(iuout,' Press_WCH_pump_coeff_gen: error.')
         close(ieout)
         CALL ERPFREE(ieout,ISTAT)
         call epwait
         call epagend
         STOP
      END IF


C Initialize pointers to inter-connection(s) ICON, and node(s) INOD
      ICON1=ICONDX(IPCOMP,1,1)
      INOD1=NPCDAT(IPCOMP,9)


      Press = BDATA(IPCOMP,4) * 1000.     !-convert from kPa to Pa


C Generate coefficients for energy balance equation
      IF(ISTATS.EQ.1) THEN

C First initialize UA modulus (for calculation of containment heat loss)
         UA=ADATA(IPCOMP,3)
         call eclose(PCNTMF(IPCOMP),-99.00,0.001,closea)
         IF(closea) UA=0.

C Establish absorbed power E based on current water flow rate which might
C have been calculated by mfs
C In case of mfs E is based on flow rate, pressure *rise*, and efficiency

         IF(IFLWN.NE.0.AND.ICFFS(ICON1).NE.0) THEN
            ICNN=ICFFS(ICON1)

            Temp = CSVF(INOD1,1) + 273.15

            E=real((FLW1(ICNN)+FLW2(ICNN))
     &        *(PRES(NODNE(ICNN))-PRES(NODPS(ICNN)))
     &        /(BDATA(IPCOMP,3) * H20LiquidDensity(Temp,Press)))


         ELSE

C revised calculation of E
           Temp = CSVF(INOD1,1) + 273.15


           E = (CSVF(INOD1,2)
     &         /(H20LiquidDensity(Temp,Press)*BDATA(IPCOMP,2)))**3
     &         *BDATA(IPCOMP,1)


         END IF

         PWRP(IPCOMP)=-ABS(E)
         IEMODEL=1
         CALL EMACH(IPCOMP,IEMODEL,PWRP(IPCOMP),PQ,PA)
         PWRQ=PQ

C Now Q is made up of all inefficiencies
         Q=(1.-BDATA(IPCOMP,3))*E

C Establish heat capacity of component mass CM (J/K) and
C fluid heat capacity rate(s) C (W/K), ie. SUM(mass flow * specific heat)
         CM=ADATA(IPCOMP,1)*ADATA(IPCOMP,2)


C revised calculation of C1
         Temp = CONVAR(ICON1,1) + 273.15
         C1 = PCONDR(ICON1)*CONVAR(ICON1,2)*H20LiquidCp(Temp,Press)


C Calculate current component time-constant TC
         TC(IPCOMP)=CM/AMAX1(SMALL,(C1+UA))

C Set up implicit/explicit weighting factor ALPHA (1 = fully implicit)
         IF(IMPEXP.EQ.1) THEN
            ALPHA=1.
         ELSE IF(IMPEXP.EQ.2) THEN
            ALPHA=RATIMP
         ELSE IF(IMPEXP.EQ.3) THEN
            IF(TIMSEC.GT.0.63*TC(IPCOMP)) THEN
               ALPHA=1.
            ELSE
               ALPHA=RATIMP
            END IF
         ELSE IF(IMPEXP.EQ.4) THEN
            CM=0.
            ALPHA=1.
         END IF

C Establish matrix equation self- and cross-coupling coefficients
         COUT(1)=ALPHA*(-C1-UA)-CM/TIMSEC
         COUT(2)=ALPHA*C1
C and then present-time coefficient (ie. right hand side)
         COUT(3)=((1.-ALPHA)*(PCRP(ICON1)+PUAP(INOD1))
     &              -CM/TIMSEC)*CSVP(INOD1,1)
     &             +(1.-ALPHA)*(-PCRP(ICON1))*PCTP(ICON1)
     &             -ALPHA*UA*PCNTMF(IPCOMP)
     &             -(1.-ALPHA)*PUAP(INOD1)*PCNTMP(IPCOMP)
     &             -ALPHA*Q-(1.-ALPHA)*PCQP(INOD1)

C Store "environment" variables future values
         PUAF(INOD1)=UA
         PCTF(ICON1)=CONVAR(ICON1,1)
         PCRF(ICON1)=C1
         PCQF(INOD1)=Q

C Save plant additional output data.
          napdat(ipcomp)=2
          pcaout(ipcomp,1)=pwrp(ipcomp)
          pcaout(ipcomp,2)=pwrq

C calculate power draw (assume equal to rated power when on)
            call eclose(CDATA(IPCOMP,1),0.00,0.000001,closea)
            if ( .not. closea ) then
               fPowerDraw = BDATA(IPCOMP,1)          ! W
            else
               fPowerDraw = 0.
            endif


C 1st phase mass (ie. water) balance coefficients
C Note that if fluid mass flow solver active, source of mass zeroised
      ELSE IF(ISTATS.EQ.2) THEN
         COUT(1)=1.
         IF(IFLWN.EQ.0.OR.ICFFS(ICON1).EQ.0) THEN
            COUT(2)=0.

C revised calculation of COUT(3)
            Temp = CSVF(INOD1,1) + 273.15

            COUT(3) = CDATA(IPCOMP,1)*H20LiquidDensity(Temp,Press)


         ELSE
            COUT(2)=-PCONDR(ICON1)
            COUT(3)=0.

         END IF

C 2nd phase mass (ie. "vapour") balance coefficients
      ELSE IF(ISTATS.EQ.3) THEN
         COUT(1)=1.
         COUT(2)=0.
         COUT(3)=0.
      END IF

C Trace output
      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) THEN
         WRITE(ITU,*) ' Component      ',IPCOMP,':'
         WRITE(ITU,*) ' 1 node (ISV=20) WCH pump'
         WRITE(ITU,*) ' Matrix node(s) ',INOD1
         WRITE(ITU,*) ' Connection(s)  ',ICON1
         IF(ISTATS.EQ.1) THEN
            WRITE(ITU,*) ' CM     = ',CM,' (J/K)'
            WRITE(ITU,*) ' C1     = ',C1,' (W/K)'
            WRITE(ITU,*) ' TC     = ',TC(IPCOMP),' (s)'
            WRITE(ITU,*) ' ALPHA  = ',ALPHA,' (-)'
            WRITE(ITU,*) ' UA     = ',UA,' (W/K)'
            WRITE(ITU,*) ' PCNTMF = ',PCNTMF(IPCOMP),' (C)'
            WRITE(ITU,*) ' Q      = ',Q,' (W)'
            WRITE(ITU,*) ' E      = ',E,' (W)'
            WRITE(ITU,*) ' E,cum. = ',E*TIMSEC,' (J)'
         ELSE IF(ISTATS.EQ.2) THEN
            WRITE(ITU,*) ' CDATA  = ',CDATA(IPCOMP,1)
         END IF
         WRITE(ITU,*) ' Matrix coefficients for ISTATS = ',ISTATS
         NITMS=3
         WRITE(ITU,*) (COUT(I),I=1,NITMS)
         IF(ITU.EQ.IUOUT) THEN
            IX1=(IPCOMP/4)*4
            IF(IX1.EQ.IPCOMP.OR.IPCOMP.EQ.NPCOMP) call epagew
         END IF
      END IF

      IF(ITC.GT.0.AND.NSINC.GE.ITC.AND.NSINC.LE.ITCF.AND.
     &   ITRACE(37).NE.0) WRITE(ITU,*) ' Leaving subroutine ',
     &                                 ' Press_WCH_pump_coeff_gen'


C---------------------------------------------------------
C XML output
C--------------------------------------------------------

C.....Get component name's length
      iNameLength = lnblnk(pcname(IPCOMP))

      call AddToReport(rvPltPowDraw%Identifier,
     &      fPowerDraw,
     &      pcname(IPCOMP)(1:iNameLength))

      RETURN
      END

