C This file is part of the ESP-r system.
C Copyright Energy Systems Research Unit, University of
C Strathclyde, Glasgow Scotland, 2001.

C ESP-r is free software.  You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation 
C (version 2 orlater).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.


C This version includes some tests of diffuse shading
C from 3D grid points.

C This file comprises the following subroutines.
C INSCON - readies the insolation computation and calls INSOLC.
C findtmc- now located in esru_misc.F
C IGEOMS - establishes the surface grid.
C INWSHD - reads the zone transitional shading file to extract
C          insolation source shading information.
C INSOLC - controls the insolation computation.
C INCAI  - computes areas and angles.
C INSORT - sets up mesh points for transparent surfaces.
C TRANSP - transforms a point on a surface to 3D coordinate space.

C ******************** INSCON ********************
C Controls the internal surface insolation calculation:
C   icomp  - the zone index;
C   icstat - computation status (1 if data exists);
C   act    - 's' for silent mode, '-' for user interaction mode.

      subroutine inscon(icomp,icstat,act)
#include "building.h"
#include "geometry.h"
#include "help.h"
      
      integer lnblnk  ! function definition

C   ntmc & nwins: the number of insolation sources (i.e. external
C                 and transparent surfaces); the latter is held
C                 in common. CFCs count as transparent surfaces for
C                 shading analysis
C   itmc & isc:   toggles for each surface, if 1 then surface is an
C                 insolation source. The latter is held
C                 in common.
      common/outin/iuout,iuin,ieout
      common/filep/ifil
      COMMON/SPAD/MMOD,LIMIT,LIMTTY
      common/shad0/ISIcalc,icalcD,icalcM
      character ltrns*72
      integer multic,mons,monf
      common/mtfile/ltrns,multic,mons,monf
      common/shdfil/ishadf
      integer mon,isc,iyd
      common/contr/mon,isc(ms),iyd
      common/ract/paction

      dimension month(12),iavdec(12)
      character paction*16
      character month*3,outs*124,tbase*72,act*1
      logical ok,xst
      logical newgeo  ! Used to test for new/old geometry file.
      integer nboftmc ! equivalent of ntmc returned from findtmc
      integer ier
      integer irpt    ! level of graphic reporting and delay

C Day of average solar declination in each month.
      data iavdec/17,15,16,15,15,11,17,16,16,16,15,11/
      data month/'Jan','Feb','Mar','Apr','May','Jun','Jul','Aug',
     &            'Sep','Oct','Nov','Dec'/

      helpinsub='inscon'  ! set for subroutine
      newgeo=.false.  ! Assume older format geometry.
      iunit=ifil
      
C Fill the itmc and isc arrays.
  4   call findtmc(icomp,act,nboftmc)
      ntmc=nboftmc

C If no insolation sources return from inscon. Do not pause
C for warning if update_silent or useupdate_silent.
      if(ntmc.eq.0)then
        if(paction(1:13).eq.'update_silent'.or.
     &     paction(1:16).eq.'useupdate_silent')then
         return
        else
          call usrmsg('No insolation source surfaces found!',' ','W')
          return
        endif
      endif

C Make up help text for following dialogs.
      helptopic='inscon_actions'
      call gethelptext(helpinsub,helptopic,nbhelp)
    
C Check that at least one transparent surface has been chosen or gleaned
C from the zone geometry file.
      icont=0
      do 70 i=1,nzsur(icomp)
         if(isc(i).eq.1)icont=1
   70 continue
      if(icont.eq.0)then
         if(newgeo)then
            call edisp(iuout,' ')
            call edisp(iuout,
     &      'No insolation sources noted in geometry file - skipping!')
            return
         else
            if(act.eq.'s')then
               call edisp(iuout,'No insolation sources - skipping!')
               return
            else
               call easkok('No surfaces specified!','Re-specify?',
     &           ok,nbhelp)
     
C Use a 2nd call to findtmc to allow the user to select different surfaces.
               if(ok)goto 4
               call edisp(iuout,'Insolation computation abandoned!')
               return
            endif
         endif
      endif

C Establish calculation period.
      if(ISIcalc.eq.1)then
         mons=icalcM
         monf=icalcM
         mon=icalcM
      elseif(act.eq.'s')then
         mons=1
         monf=12
         mon=mons
      else
         if(mons.gt.12) mons=1  ! if mons was clobbered reset
         call easki(mons,' ','Start month number?',
     &     1,'F',12,'F',1,'start month',ier,nbhelp)
         if(ier.ne.0)return
         if(monf.gt.12) monf=12  ! if monf was clobbered reset
         call easki(monf,' ','End month number?',
     &     mons,'F',12,'F',1,'end month',ier,nbhelp)
         if(ier.ne.0)return
         mon=mons
      endif

C If in graphic mode ask about delay in graphic display of calcs.
      if(mmod.eq.8)then
        if(act.eq.'s')then
          irpt=0  ! silent so do not bother with display
        else
          irpt=0
          CALL EASKMBOX(' ','Insolation display options:',
     &      'none','quick','slower','slowest',
     &      ' ',' ',' ',' ',irpt,nbhelp)
          irpt=irpt-1
        endif
      else
        irpt=0
      endif

C For testing ask method.
      icmethod=1   ! use legacy for now

C Convert start month day of average solar declination to a year
C day number.
  81  iday=iavdec(mon)
      if(ISIcalc.eq.1)iday=icalcD
      call eday(iday,mon,iyd)

C Attempt to open the relevant zone transitional shading file 
C for this month. If unknown call igeoms to set up grid.
      write(ltrns,'(a)')zname(icomp)(1:lnzname(icomp))
      write(tbase,'(a,a3)')ltrns(:lnblnk(ltrns)),month(mon)
      call erpfree(iunit,istat)
      call findfil(tbase,xst)

C Allow for number of surfaces within the zone transitional shading
C file.
      irecw=ms+5
      if(xst)then
         ier=0
         call efopran(iunit,tbase,irecw,1,ier)

C Since this file exists, there must have been a matching
C obstruction file and therefore the data in common /GS6/
C can be used.
         ishadf=1
      else
         ishadf=0
         if(mon.eq.mons)call igeoms(icomp)
      endif

C Commence computation for each specified month and allow user to
C update the zone shading & insolation db.
      write(outs,'(a3,a)')month(mon),' calculation commenced.'
      if(paction(1:13).ne.'update_silent')then
         call usrmsg(outs,' ','-')
      endif

      call insolc(icomp,irpt)
      icstat=1
      call isave(icomp,ier)

C Next month.
      if(ISIcalc.eq.1)return  ! do only once if in embedded mode

      mon=mon+1
      if(mon.le.monf)goto 81

      return
      end
      
C ******************** IGEOMS ********************
C Establishes the source surface grid if a zone transitional shading
C file does not exist.

      subroutine igeoms(izone)
#include "building.h"
#include "geometry.h"

C Parameter.
      integer izone  ! index of zone

      common/tmc/itmc(ms),nwins

C Set default surface grid.
      if(nox(izone).eq.0) nox(izone)=20
      if(noz(izone).eq.0) noz(izone)=20
      nwins=0
      do i=1,nzsur(izone)
        if(itmc(i).ne.0)nwins=nwins+1
      enddo  ! of i
      return
      end

C ******************** INSOLC ********************
C Calculates intra-zone solar insolation. Called from inscon.
C insst()       - 0 if the entire surface is shaded, -1 if the sun
C                 is not up; otherwise the index of the surface being
C                 insolated.
C pinsst(i,j,k) - irradiance proportion to kth internal insolated
C                 surface received from insolation source i at hour j.
C N.B The proportions correspond to the situation after source shading has
C been imposed with igins() holding information for each source surface
C grid point as follows. 
C        0 - undefined; 
C     -100 - point is shaded;
C     -200 - point is outwith the source surface;
C       k  - indicates that point projects to the kth internal surface
C            in the misur list.

      subroutine insolc(icomp,irpt)
#include "building.h"
#include "geometry.h"
#include "prj3dv.h"

C Parameters.
      integer icomp  ! focus zone
      integer irpt   ! level of graphic reporting 0=none 1=quick 2=slower 3=slow

      common/tracech/icout
      common/tc/itc,icnt
      COMMON/SPAD/MMOD,LIMIT,LIMTTY
      integer izstocn
      common/c24/izstocn(mcom,ms)
      COMMON/gzonpik/izgfoc,nzg,nznog(mcom)
      common/sunhr/isunhr
      common/tshd/ioshd(mox,moz)
      common/grid11/xop(mox,moz),zop(mox,moz)
      common/shad1/isunup
      common/cai/caii(ms),caie(ms)
      integer mon,isc,iyd
      common/contr/mon,isc(ms),iyd

C Point (xyz) projection as set in trnsf3i.
      common/proji/xp,yp,zp
      common/inswin/igins(mox,moz)
      common/pinsol/ins(ms),pins(ms),pinw(ms),pcshad
      common/stins/insst(mgt,24,misur),pinsst(mgt,24,misur),
     &             shadst(mgt,24)
      common/ract/paction
      common/sangl1/sazi1,salt1
      common/tmc/itmc(ms),nwins
      common/icflag/ic
      common/shdinf/noshad
      common/prec8/slat,slon

C Mesh for source surface as points in 3D.
      real x3op,y3op,z3op
      common/grid31/x3op(mox,moz),y3op(mox,moz),z3op(mox,moz)

C 3D Points of intersection (for the current hour for each of
C the source grid points.
      real x3intrs,y3intrs,z3intrs
      common/grid32/x3intrs(mox,moz),y3intrs(mox,moz),z3intrs(mox,moz)

C ical = 0 indicates that no insolation calculations have been
C performed and is set to 1 when they have.
      common/data1/ical,idifc,init

      dimension xf(mtv),yf(mtv),zf(mtv)
      dimension ydist(mox,moz)   ! between source mesh and intersection
      character outs*124,paction*16
      character*3 month(12)
      real xxop,zzop  ! to pass to point1
      logical inside  ! for insurbox

C For 3D distance tests.
      DIMENSION XX(mv),YY(mv),ZZ(mv),VP(3),EQN(4)
      real XXuv,YYuv,ZZuv  ! 1 unit inwards
      logical tok,ok

      data month/'Jan','Feb','Mar','Apr','May','Jun','Jul','Aug',
     &            'Sep','Oct','Nov','Dec'/

C Set initial values.
      ical=0
      ic=0
      kind=0
      indcom=0  ! counter for completion
      isunhr=0  ! first sun-up hour
      ksu=0     ! number of sun-up hours

C If trace is on and verbose set tok, if not so verbose set ok.
      tok=.false.; ok=.false.
      if(icout.eq.33.and.itc.eq.2)tok=.true.
      if(icout.eq.33.and.itc.eq.1)ok=.true.

C Determine number of sun-up hours.
      do i=1,24
        stime=float(i)
        call eazalts(stime,iyd,slat,slon,isunup,sazi,salt)
        if(isunup.eq.1)ksu=ksu+1
      enddo  ! of i

C Find bounding boxes of surfaces.
      call findsurbox()

C Estimate simulation extent (source-hours) to support the
C progress report.         
      kt=nwins*ksu

C Loop through surfaces and for each insolation source perform
C calculations.
      iwin=0
      do 20 i=1,nzsur(icomp)

C Get the current connection for surface isrc
        isrc=i
        iccisrc=izstocn(icomp,isrc)
        if(itmc(isrc).eq.0)goto 20
        iwin=iwin+1
        modes=1

C 'noshad' = 1 indicates that there is no shading information for
C              surface i in the zone transitional shading file.
        noshad=0
        if(paction(1:13).ne.'update_silent')then
          call usrmsg(' ',' ','-')
        endif

C Partially derive transformation equations by computing time
C independent part of equation coefficients - equivalent to
C translation plus three-dimensional plane rotation.

C Kyunghun suggests move these two calls to within the hour
C loop just before incai.
C       call trnsf1i(icomp,isrc)

C Grid the insolation source surface.
C       call mesh(icomp,isrc)

C Proceed hour-by-hour.
        do 30 j=1,24

C Clear the ydist array to quite far away.
          do m=1,nox(icomp)
            do n=1,noz(icomp)
              ydist(m,n)=1000.0   ! set a long way away
              x3intrs(m,n)=0.0; y3intrs(m,n)=0.0; z3intrs(m,n)=0.0
            enddo ! of n
          enddo   ! of m

          ihr=j
          stime=float(ihr)

C Determine the solar azimuth and altitude angles relative to the
C original coordinate system.
          call eazalts(stime,iyd,slat,slon,isunup,sazi1,salt1)

C For this azimuth and altitude get unit vector and then negate
C to create a unit vector pointing inwards along the sun path.
          call  AZ2UV(sazi1,salt1,vdx,vdy,vdz)
          if(tok) write(icout,*)'insolc: sazi1,salt1,vdx,vdy,vdz ',
     &      sazi1,salt1,vdx,vdy,vdz
          uvdx=-1.0*vdx; uvdy=-1.0*vdy; uvdz=-1.0*vdz

C Abandon calculation for this hour if before sunrise or after sunset.
          if(isunup.eq.0)goto 1

C Refresh the view of the zone prior to display of the intersection points.
          if(mmod.eq.8.and.irpt.gt.0)then
            MODIFYVIEW=.true.
            modlen=.true.
            itsnm=1
            CALL INLNST(1)
            nzg=1; nznog(1)=ICOMP; izgfoc=ICOMP
            CALL redraw(IER)
            write(outs,'(3a,i2,a,i3,2a)') 'Direct insolation: ',
     &        month(mon),' @ ',j,':00, from surface ',isrc,' ',
     &        sname(icomp,isrc)
            call usrmsg(' ',outs,'-')
          endif

C Record first sun-up hour.
          if(isunhr.eq.0)isunhr=j

C Suggested change of position of trnsf1i and mesh call above.
          call trnsf1i(icomp,isrc)
          call mesh(icomp,isrc)        ! generate 2D mesh points
          call mesh3d(icomp,isrc,'-')  ! generate 3D mesh points

C Determine angles for all external and internal surfaces.
          call incai(icomp,ihr)

C Initiallise igins, the array of source grid points holding
C the insolated surface numbers.
          do m=1,nox(icomp)
            do n=1,noz(icomp)
              xxw=x3op(m,n); yyw=y3op(m,n); zzw=z3op(m,n)
              call insurbox(xxw,yyw,zzw,icomp,isrc,inside)

C If inside the bounding box determine if current grid point 
C is within source surface via winding method.
              if(inside)then
                xxop=xop(m,n); zzop=zop(m,n)  ! to pass to ipippa
                call ipippa(icomp,isrc,xxop,zzop,m,n,ipip)
                if(ipip.eq.0)then
                  igins(m,n)=-200     ! outwith source surface
                else
                  igins(m,n)=0        ! not yet defined
                  ydist(m,n)=1000.0   ! a long way
                  xxw=x3op(m,n); yyw=y3op(m,n); zzw=z3op(m,n)
                  if(mmod.eq.8.and.irpt.gt.0)then
                    call wiresymbol(xxw,yyw,zzw,0,32)
                  endif
                endif
              endif
            enddo   ! of n
          enddo     ! of m

C Take zone transitional shading file information into account.
C modes=1 indicates that call is for first sun-up hour; 2 otherwise.
          call inwshd(icomp,isrc,ihr,modes)
          modes=2

C If caie is positive then source surface can be considered.
          if(caie(isrc).le.0.0)then
            do m=1,nox(icomp)
              do n=1,noz(icomp)

C Any grid point which has not already been found to be outwith
C the source surface reset to assume it is shaded < why? >
                if(igins(m,n).ne.-200)igins(m,n)=-100
              enddo  ! of n
            enddo    ! of m
            goto 2
          endif

C Examine each internal face of surface for insolation.
          do 80 l=1,nzsur(icomp)
            intsur=l

C Exclude a surface that cannot see the source.
            if(intsur.eq.isrc.or.caii(l).le.0.0)goto 80

C Derive the transformation equations for current internal surface.
            call trnsf1i(icomp,intsur)

C Compute new solar angles relative to internal surface coordinate
C system.
            call trnsf2i(ihr)

C Set up source surface global coordinates.
            do inv=1,isznver(icomp,isrc)
              iwhich=iszjvn(icomp,isrc,inv)
              xf(inv)=szcoords(icomp,iwhich,1)
              yf(inv)=szcoords(icomp,iwhich,2)
              zf(inv)=szcoords(icomp,iwhich,3)
            enddo  ! of ivn

C Initiate source surface grid point-by-point analysis.
            ipass=0
            do 100 m=1,nox(icomp)
              do 110 n=1,noz(icomp)

C Add unit vector along the sun path to the current mesh cell.
                XXuv=x3op(m,n)+uvdx; YYuv=y3op(m,n)+uvdy
                ZZuv=z3op(m,n)+uvdz

                if(igins(m,n).eq.-200)goto 110  ! outwith source surf
                if(ioshd(m,n).eq.1)then
                  igins(m,n)=-100              ! point is shaded
                  goto 110                     ! jump to next n
                endif

C Convert source surface grid point coordinates.
                call transp(iccisrc,xf,yf,zf,xop(m,n),zop(m,n),
     &            xgp,ygp,zgp)

                call trnsf3i(ihr,icomp,intsur,ipass,xgp,ygp,zgp)
                ipass=1
                if(yp.lt.0.0)goto 110

C Use vecpln to project mesh unit vector to the possible receiving 
C surface, use crowxyz to check real distance between mesh point 
C and the intersection point. And update igins if nearer. Remember
C the points of intersection for display after all mesh points have
C been processed.
                icc=izstocn(icomp,intsur)
                EQN(1)=sureqn(icomp,intsur,1)
                EQN(2)=sureqn(icomp,intsur,2)
                EQN(3)=sureqn(icomp,intsur,3)
                EQN(4)=sureqn(icomp,intsur,4)
                CALL VECPLN(x3op(m,n),y3op(m,n),z3op(m,n),
     &            XXuv,YYuv,ZZuv,EQN, x3,y3,z3, IERR)

C Is this intersection within the bounding box of the receiving surface?
C If so do point containment test via winding method.
                call insurbox(x3,y3,z3,icomp,intsur,inside)
                if(inside)then
                  call ipippa(icomp,intsur,xp,zp,m,n,ipip)
C                  write(6,*) 'ipippa for ',icomp,intsur,xp,zp,ipip
                  if(ipip.eq.0)then
                    goto 110  ! not actually within the polygon
                  endif
                else
                  goto 110  ! not within BB
                endif

C If point (x3,y3,z3) is within the receiving surface check for
C the distance from the source grid point.
                chkdis=crowxyz(x3op(m,n),y3op(m,n),z3op(m,n),x3,y3,z3)
                x3intrs(m,n)=x3; y3intrs(m,n)=y3; z3intrs(m,n)=z3

                if(chkdis.lt.ydist(m,n))then
                  if(igins(m,n).eq.0)then
C Debug of initial points of intersection (uncomment if wanted in trace).
C                    if(tok)then
C                      write(icout,'(a,4i5,a,i5,a,f7.3,a,6f7.3)') 
C     &                  'Initial hit at hour grid m n',ihr,
C     &                  m,n,igins(m,n),' -> ',intsur,' & dist ',
C     &                  chkdis,' ',x3op(m,n),y3op(m,n),z3op(m,n),
C     &                  x3,y3,z3
C                    endif
C                     if(mmod.eq.8) call wiresymbol(x3,y3,z3,0,32)
                  else
                    if(igins(m,n).ne.intsur)then
                      if(tok)then
                        write(icout,
     &                    '(a,4i5,a,i5,a,f6.3,a,f7.3,a,7f7.3)') 
     &                    'Found closer at hour grid m n',ihr,
     &                     m,n,igins(m,n),' -> ',intsur,' dist ',
     &                     ydist(m,n),' -> ',chkdis,' 3D ',
     &                     x3op(m,n),y3op(m,n),z3op(m,n),x3,y3,z3
                      endif
C                      if(mmod.eq.8)call wiresymbol(x3,y3,z3,0,32)
                    endif
                  endif

                  igins(m,n)=intsur
                  ydist(m,n)=chkdis
                endif
  110         continue  ! of loop n

C Debug ydist after the n loop.
C             call dumpnoxnozr(icomp,ihr,ydist)
C             call pauses(1)
  100       continue     ! of loop m
   80     continue        ! of surfaces in zone

C Display the resolved intersection points with slight delay.
          if(mmod.eq.8.and.irpt.gt.0)then
            do m=1,nox(icomp)
              do n=1,noz(icomp)
                x3=x3intrs(m,n); y3=y3intrs(m,n); z3=z3intrs(m,n)
                call wiresymbol(x3,y3,z3,1,32)
                if(irpt.eq.2)call pausems(4)   ! medium pause
                if(irpt.eq.3)call pausems(6)   ! longer pause
              enddo ! of n
              if(irpt.eq.1)call pausems(3)     ! slight pause for column
            enddo   ! of m
          endif

C Debug ydist after the m,n loop.
          if(ok.or.tok)then
            call dumpnoxnozr(icomp,ihr,ydist)
          endif
          if(irpt.eq.1)call pausems(500) ! slight pause
          if(irpt.eq.2)call pauses(2)    ! medium pause
          if(irpt.eq.3)call pauses(5)    ! longer pause

C Write out the current contents of igins.
          if(ok.or.tok)then
            call dumpigins(icomp,ihr)
          endif

C Now that all of the grid cells have been evaluated, sort 
C the insolation passing through each insolation source surface.
    2     call insort(icomp,isrc,ihr)

C Store for each source and hour.
          do m=1,misur
            insst(iwin,ihr,m)=ins(m)
            pinsst(iwin,ihr,m)=pins(m)
          enddo   ! of m
          shadst(iwin,ihr)=pcshad
          kind=kind+1
          calcom=float(kind)/float(kt)
          write(outs,'(a,a3,a)')'`',month(mon),
     &      '` insolation calculation commenced:'
          if(calcom.ge.0.95)then
            if(indcom.eq.4)goto 30
            indcom=indcom+1
            if(paction(1:13).ne.'update_silent')then
              call usrmsg(outs,' complete.','-')
            endif
          elseif(calcom.ge.0.75)then
            if(indcom.eq.3)goto 30
            indcom=indcom+1
            if(paction(1:13).ne.'update_silent')then
              call usrmsg(outs,' 75% complete.','-')
            endif
          elseif(calcom.ge.0.50)then
            if(indcom.eq.2)goto 30
            indcom=indcom+1
            if(paction(1:13).ne.'update_silent')then
              call usrmsg(outs,' 50% complete.','-')
            endif
          elseif(calcom.ge.0.25)then
            if(indcom.eq.1)goto 30
            indcom=indcom+1
            if(paction(1:13).ne.'update_silent')then
              call usrmsg(outs,' 25% complete.','-')
            endif
          endif
          goto 30

C Sun not up.
    1     do m=1,misur
            insst(iwin,ihr,m)=-1
            pinsst(iwin,ihr,m)=0.0
          enddo  ! of m
   30   continue  ! end of hour loop
   20 continue     ! end of source surface loop
      ical=1
      return
      end  ! of insolc

C ******************** inwshd ********************
C Reads the transitional shading file if one exists and extracts
C surface shading information.

      subroutine inwshd(izone,isur,ihr,modes)
#include "building.h"
#include "geometry.h"

C Parameters passed:
      integer izone  ! the current zone
      integer isur   ! ??
      integer ihr    ! current hour
      common/outin/iuout,iuin,ieout
      common/filep/ifil
      common/tshd/ioshd(mox,moz)
      common/sunhr/isunhr
      common/shdfil/ishadf
      common/shdinf/noshad
      common/recshd/irectx

      dimension irecs(ms)
      character outs*124

      iushd=ifil

      if(ishadf.eq.0.or.noshad.eq.1)goto 1
      goto (2,3),modes

C Transfer here for the first hour every day when sun is up.
    2 irec=1
      read(iushd,rec=irec,iostat=istat,err=1000)
     &    (irecs(i),i=1,nzsur(izone))
      irecn=irecs(isur)
      if(irecn.eq.0)then
          noshad=1
          goto 1
      endif

C Skip previous records written prior to sun-rise.
      irectx=irecn+2+(isunhr-1)

C Transfer here at all other sun-up hours.
    3 irec=irectx
      read(iushd,rec=irec,iostat=istat,err=1000)isunup

C Sun not up must be an error!
      if(isunup.eq.0)goto 1000

      irec=irec+1
      read(iushd,rec=irec,iostat=istat,err=1000)ians
      irec=irec+1
      if(ians.eq.-2)goto 1
      if(ians.eq.-1)goto 5

C Read surface shading information.
      do 10 i=1,nox(izone)
         read(iushd,rec=irec,iostat=istat,err=1000)
     &     (ioshd(i,j),j=1,noz(izone))
         irec=irec+1
   10 continue
      goto 4

C Surface is not shaded.
    1 do 20 i=1,nox(izone)
         do 30 j=1,noz(izone)
            ioshd(i,j)=0
   30    continue
   20 continue
      goto 4

C Surface is fully shaded.
    5 do 40 i=1,nox(izone)
         do 50 j=1,noz(izone)
            ioshd(i,j)=1
   50    continue
   40 continue

C Increment irec to skip record holding pso & psof
    4 irectx=irec+1
      return

 1000 write(outs,1001)irec,ihr,isur
 1001 format('INWSHD: Zone Transitional Shading file error, record',i6,
     &       ', hour ',i3,' source ',i3)
      call edisp(iuout,outs)
      return
      end

C ******************** INCAI ********************
C Much of what is calculated in incai is available elsewhere.
c There are several opportunities to use other common blocks.

      subroutine incai(icomp,ihour)
#include "building.h"
#include "geometry.h"

C Parameters.
      integer icomp  ! the focus zone
      integer ihour  ! the focus hour

      common/traceout/icout
      common/tc/itc,icnt
      common/cai/caii(ms),caie(ms)
      common/sangl1/sazi1,salt1
      integer izstocn
      common/c24/izstocn(mcom,ms)

      character outs*124
      logical ok,tok
      character ZSDES*28,ZSDESC*20,ZSDESS*16
      integer lnzsn,icc

C If trace is on set tok.
      tok=.false.
      if(icout.eq.33)tok=.true.

      pi = 4.0 * atan(1.0)
      r=pi/180.

C  Cosine of the angle of incidence and surface-solar azimuth angle.
      do i=1,nzsur(icomp)
        icc=izstocn(icomp,i)
        if(salt1.gt.0.0)then
          psazi=abs(spazi(icomp,i)-sazi1)
          x1=cos(salt1*r)
          x2=sin((90.-spelv(icomp,i))*r)
          x3=cos(psazi*r)
          x4=sin(salt1*r)
          x5=cos((90.-spelv(icomp,i))*r)
          caie(i)=x1*x2*x3+x4*x5
          paz=spazi(icomp,i)+180.0
          if(spazi(icomp,i).gt.180.0)paz=spazi(icomp,i)-180.0
          psazi=abs(paz-sazi1)
          x3=cos(psazi*r)
          x1=cos(salt1*r)
          x2=sin((90.+spelv(icomp,i))*r)
          x3=cos(psazi*r)
          x4=sin(salt1*r)
          x5=cos((90.+spelv(icomp,i))*r)
          caii(i)=x1*x2*x3+x4*x5
        else
          caii(i)=0.0
          caie(i)=0.0
        endif
      enddo   ! of i

C Trace output, check also that user has asked for detailed trace.
      ok=.false.
      if(itc.eq.2)ok=.true.
      if(tok.and.ok)then
        call edisp(icout,' ')
        write(outs,'(a,i3,a,i3)')'*** INCAI: Hour',ihour,' zone',icomp
        call edisp(icout,outs)
        call edisp(icout,
     &  ' Surface    Internal    External    Azimuth    Elevation')

        do 50 i=1,nzsur(icomp)
          icc=izstocn(icomp,i)
          call zsid(icomp,i,ZSDES,ZSDESC,ZSDESS)
          lnzsn=lnblnk(ZSDES)
          write(outs,'(a,f6.4,6x,f6.4,7x,f5.1,6x,f5.1)')
     &     ZSDES(1:lnzsn),caii(i),caie(i),spazi(icomp,i),spelv(icomp,i)
          call edisp(icout,outs)
   50   continue
      endif
      return
      end

C ******************** INSORT ********************
C Parameters passed are: ii, the current surface being processed in inscon
C at hour jj.

      subroutine insort(icomp,ii,jj)
#include "building.h"
#include "geometry.h"

      common/tracech/icout
      common/tc/itc,icnt
      common/inswin/igins(mox,moz)
      common/pinsol/ins(ms),pins(ms),pinw(ms),pcshad
      common/outin/iuout,iuin,ieout
      integer izstocn
      common/c24/izstocn(mcom,ms)

      dimension icounts(ms+4),icountw(ms+4),icountt(ms+4)
      character outs*124,louts*244
      logical tok
      character ZSDES*28,ZSDESC*20,ZSDESS*16
      integer lnzsn

C If trace requested enable writing.
      tok=.false.
      if(icout.eq.33)tok=.true.
      if(tok)then
        call zsid(icomp,ii,ZSDES,ZSDESC,ZSDESS)
        lnzsn=lnblnk(ZSDES)
        call edisp(icout,' ')
        write(icout,'(3a,i3)') ' Subroutine insort ',
     &    ZSDES(1:lnzsn),' hour ',jj
      endif

C Initialise.
      ihit=0
      pinrl=0.0
      pintot=0.0
      do i=1,(ms+4)
        icounts(i)=0
        icountw(i)=0
        icountt(i)=0
      enddo  ! of i

C Count the number of grid points that project onto each surface.
      do i=1,nox(icomp)
        do j=1,noz(icomp)
          if(igins(i,j).eq.-200)then
            icountt(nzsur(icomp)+3)=icountt(nzsur(icomp)+3)+1
          elseif(igins(i,j).eq.-100)then
            icountt(nzsur(icomp)+1)=icountt(nzsur(icomp)+1)+1
          elseif(igins(i,j).ge.1.and.igins(i,j).le.nzsur(icomp))then
            icounts(igins(i,j))=icounts(igins(i,j))+1
            icountt(igins(i,j))=icountt(igins(i,j))+1
            ihit=ihit+1
          elseif(igins(i,j).le.-1.and.igins(i,j).ge.
     &          (-1*nzsur(icomp)))then
            icountw(abs(igins(i,j)))=icountw(abs(igins(i,j)))+1
            icountt(abs(igins(i,j)))=icountt(abs(igins(i,j)))+1
            ihit=ihit+1
          else
            icountt(nzsur(icomp)+2)=icountt(nzsur(icomp)+2)+1
          endif
        enddo ! of j
      enddo   ! of i

C Initialise insolated surfaces.
      do i=1,nzsur(icomp)
        ins(i)=i
      enddo  ! of i

C Bubble sort to give highest count to ins(1) etc.
C Sort based on total hits on surface (i.e. icountt).
      do i=1,nzsur(icomp)-1
        do j=1,nzsur(icomp)-1 
          if(icountt(j).lt.icountt(j+1))then
             itemp=icountt(j)
             icountt(j)=icountt(j+1)
             icountt(j+1)=itemp
             itemp=icounts(j)
             icounts(j)=icounts(j+1)
             icounts(j+1)=itemp
             itemp=icountw(j)
             icountw(j)=icountw(j+1)
             icountw(j+1)=itemp
             itemp=ins(j)
             ins(j)=ins(j+1)
             ins(j+1)=itemp
          endif
        enddo  ! of j
      enddo    ! of i

C Trace output.
      if(tok)then
        call zsid(icomp,ii,ZSDES,ZSDESC,ZSDESS)
        lnzsn=lnblnk(ZSDES)
        call edisp(icout,' ')
        write(louts,'(a,i5,a)') ' total hits ',
     &    ihit,' then icounts icountw icountt ins for each:'
        call edisp(icout,louts)
        itrunc=1
        ipos=1
        do while (itrunc.ne.0)
          call ailist(ipos,nzsur(icomp)+3,icounts,MS+4,'C',louts,
     &      loutln,itrunc)
          write(icout,'(1x,a)',IOSTAT=ios,ERR=147) louts(1:loutln)
          ipos=itrunc+1
        end do
        itrunc=1
        ipos=1
        do while (itrunc.ne.0)
          call ailist(ipos,nzsur(icomp)+3,icountw,MS+4,'C',louts,
     &      loutln,itrunc)
          write(icout,'(1x,a)',IOSTAT=ios,ERR=147) louts(1:loutln)
          ipos=itrunc+1
        end do
        itrunc=1
        ipos=1
        do while (itrunc.ne.0)
          call ailist(ipos,nzsur(icomp)+3,icountt,MS+4,'C',louts,
     &      loutln,itrunc)
          write(icout,'(1x,a)',IOSTAT=ios,ERR=147) louts(1:loutln)
          ipos=itrunc+1
        end do
        itrunc=1
        ipos=1
        do while (itrunc.ne.0)
          call ailist(ipos,nzsur(icomp),ins,MS,'C',louts,loutln,itrunc)
          write(icout,'(1x,a)',IOSTAT=ios,ERR=147) louts(1:loutln)
          ipos=itrunc+1
        end do
      endif
  147 continue

C Set insolated planes to zero if there are no hits, i.e. total shading.
      if(ihit.eq.0)then
        do i=1,nzsur(icomp)
          ins(i)=0
          pins(i)=0.0
          pinw(i)=0.0
        enddo  ! of i
        pcshad=1.0
        return
      endif   

C Calculate surface insolation proportion of total zone insolation
C (i.e. all insolated surfaces add to 1).
      do i=1,nzsur(icomp)
        pins(i)=real(icounts(i))/(real(ihit))
        pinw(i)=real(icountw(i))/(real(ihit))
      enddo  ! of i

C pcshad is the proportion of grid points shaded.
      pcshad=real(icountt(nzsur(icomp)+1))/
     &           (real(ihit+icountt(nzsur(icomp)+1)))

C Re-allocate proportions for surfaces greater than the maximum 
C number of insolated surfaces ('misur').
      do i=(misur+1),nzsur(icomp)
         pinrl=pinrl+pins(i)
         pinrl=pinrl+pinw(i)
      enddo  ! of i
      if(pinrl.gt.0.0)then
         do i=1,misur
            pins(i)=pins(i)+(pins(i)*pinrl/(1.0-pinrl))
            pinw(i)=pinw(i)+(pinw(i)*pinrl/(1.0-pinrl))
         enddo  ! of i
      endif

C Checks.
      do i=1,misur
         pintot=pintot+pins(i)+pinw(i)
      enddo  ! of i
      if(pintot.lt.0.99.or.pintot.gt.1.01)then
         icn=izstocn(icomp,ii)
         write(outs,'(a,f5.3,3a,i2)') 'Warning: solar split is ',
     &                pintot,' in ',sname(icomp,ii),' at hour ',jj
         call edisp(iuout,outs)
      endif

C If trace report pins and pinw.
      if(tok)then
        write(icout,'(a)') 'for each surface: pins'
        itrunc=1
        ipos=1
        do while (itrunc.ne.0)
          call arlist(ipos,nzsur(icomp),pins,MS,'C',louts,loutln,itrunc)
          write(icout,'(1x,a)',IOSTAT=ios,ERR=146) louts(1:loutln)
          ipos=itrunc+1
        end do
        write(icout,'(a)') 'for each surface: pinw'
        itrunc=1
        ipos=1
        do while (itrunc.ne.0)
          call arlist(ipos,nzsur(icomp),pinw,MS,'C',louts,loutln,itrunc)
          write(icout,'(1x,a)',IOSTAT=ios,ERR=146) louts(1:loutln)
          ipos=itrunc+1
        end do
      endif
  146 continue
      return
      end

C ******************** Dumpigins ********************
C Prints a noz (row) x nox (column) array as a
C compact matrix (looking from the outside into the room). Called
C if trace has been set.

      subroutine dumpigins(icomp,ihr)
#include "building.h"
#include "geometry.h"

      common/tracech/icout
      common/tc/itc,icnt
      common/inswin/igins(mox,moz)
      character outs*124,louts*244
      integer ilist ! to hold one row
      dimension ilist(mox)
      write(icout,'(a,i3,a,i2,a)') 'IGINS: ',icomp,' hour ',ihr,
     &  ' row  & column data....'
      do m=nox(icomp),1,-1
        do loop=1,noz(icomp)
C          ilist(loop)=igins(m,loop)  ! rotated 90 deg
          ilist(loop)=igins(loop,m)   ! as people see it
        enddo  ! of loop
        itrunc=1
        ipos=1
        do while (itrunc.ne.0)
          call ailist(ipos,noz(icomp),ilist,mox,'C',louts,
     &      loutln,itrunc)
          write(icout,'(i3,2a)',IOSTAT=ios,ERR=147) m,' ',
     &      louts(1:loutln)
          ipos=itrunc+1
        end do
      enddo    ! of m
  147 continue
      return
      end

C ******************** Dumpnoznozr ********************
C prints a noz (row) x nox (column) real array as a
C compact matrix (looking from the outside into the room). Array is
C assumed to be mox x moz. Called if trace has been set.

      subroutine dumpnoxnozr(icomp,ihr,vals)
#include "building.h"
#include "geometry.h"

C Parameters.
      integer icomp           ! the focus zone
      integer ihr             ! the focus hour
      real vals               ! noz  x nox array
      dimension vals(mox,moz)

      common/tracech/icout
      common/tc/itc,icnt
      character outs*124,louts*244
      real rlist ! to hold one row
      dimension rlist(mox)
      write(icout,'(a,i2,a)') 'dumpnoxnoz: hour ',ihr,
     &  ' row  & column data....'
      do m=nox(icomp),1,-1
        do loop=1,noz(icomp)
          rlist(loop)=vals(loop,m)   ! as people see it
        enddo  ! of loop
        itrunc=1
        ipos=1
        do while (itrunc.ne.0)
          call arlist(ipos,noz(icomp),rlist,mox,'C',louts,
     &      loutln,itrunc)
          write(icout,'(i3,2a)',IOSTAT=ios,ERR=147) m,' ',
     &      louts(1:loutln)
          ipos=itrunc+1
        end do
      enddo    ! of m
  147 continue
      return
      end

C ******************** Dumpnoznozi ********************
C Prints a noz (row) x nox (column) int array as a
C compact matrix (looking from the outside into the room). Array is
C assumed to be mox x moz. Called if trace has been set.

      subroutine dumpnoxnozi(icomp,ihr,ivals)
#include "building.h"
#include "geometry.h"

C Parameters.
      integer icomp           ! the focus zone
      integer ihr             ! the focus hour
      integer ivals           ! noz  x nox array
      dimension ivals(mox,moz)

      common/tracech/icout
      common/tc/itc,icnt
      character outs*124,louts*244
      integer ilist ! to hold one row
      dimension ilist(mox)
      write(icout,'(a,i2,a)') 'dumpnoxnoz: hour ',ihr,
     &  ' row  & column data....'
      do m=nox(icomp),1,-1
        do loop=1,noz(icomp)
          ilist(loop)=ivals(loop,m)   ! as people see it
        enddo  ! of loop
        itrunc=1
        ipos=1
        do while (itrunc.ne.0)
          call ailist(ipos,noz(icomp),ilist,mox,'C',louts,
     &      loutln,itrunc)
          write(icout,'(i3,2a)',IOSTAT=ios,ERR=147) m,' ',
     &      louts(1:loutln)
          ipos=itrunc+1
        end do
      enddo    ! of m
  147 continue
      return
      end

C ******************** TRANSP ********************
C Transforms a grid point defined relative to a given
C surface back to global co-ordinates.
C 1 - Recover equation of surface (eqn(4)) and centre of 
C     gravity (vp(3)) via pleqn().
C 2 - Set up eye point normal to plane at centre of gravity
C     (ep(3)) and use eyemat() to establish matrix and 
C     reverse matrix that transform surface to normal view.
C 3 - Use orttrn() to find co-ordinates of origin (first
C     surface vertex) when transformed to normal view and
C     apply reverse transformation to give vertices in
C     global co-ordinates.

      subroutine  transp(icc,xx,yy,zz,dx,dz,xxw,yyw,zzw)
#include "building.h"
#include "geometry.h"

C Parameters.
      integer icc   ! the model surface connection index
      real xx,yy,zz    ! coordinates passed
      real dx,dz    ! grid location
      real xxw,yyw,zzw  ! transformed coordinate
      real xxx,yyy,zzzz ! for use in last orttrn
      COMMON/C3/IC1(MCON),IE1(MCON),ICT(MCON),IC2(MCON),IE2(MCON)

      dimension xx(mtv),yy(mtv),zz(mtv)
      dimension tmat(4,4),rmat(4,4)
      dimension vp(3),ep(3)

C Recover eye point and view point for eyemat.
      iz=ic1(icc); is=IE1(icc)
      vp(1)=SURCOG(iz,is,1)
      vp(2)=SURCOG(iz,is,2)
      vp(3)=SURCOG(iz,is,3)
      ep(1)=SURVN(iz,is,1)
      ep(2)=SURVN(iz,is,2)
      ep(3)=SURVN(iz,is,3)

      call eyemat(ep,vp,1.0,tmat,rmat)

C Transform first 2 surface vertices.
      call orttrn(xx(1),yy(1),zz(1),tmat,x1,y1,zzz,ierr)
      call orttrn(xx(2),yy(2),zz(2),tmat,x2,y2,zzz,ierr)
      alpha=atan2((y2-y1),(x2-x1))
      beta=atan2(dz,dx)
      d=sqrt(dx*dx+dz*dz)
      xxw=x1+d*cos(alpha+beta)
      yyw=y1+d*sin(alpha+beta)

C Take each grid point and apply transformation.
      call orttrn(xxw,yyw,zzz,rmat,xxx,yyy,zzzz,ierr)
      xxw=xxx
      yyw=yyy
      zzw=zzzz
      return
      end

C ******************** ang3dvtx ********************
C Determine angle A between two lines given 3 vertices as in:
C                            * 1
C                           A \
C                     3 *------* 2
C With the sign of the dot product also returned as either
C 1.0 or -1.0

      SUBROUTINE ang3dvtx(x1,y1,z1,x2,y2,z2,x3,y3,z3,ang,sign)
      dimension vd21(3),vd32(3)
      PI = 4.0 * ATAN(1.0)
      RAD = PI/180.
      vdx21=x1-x2
      vdy21=y1-y2
      vdz21=z1-z2
      vdx32=x3-x2
      vdy32=y3-y2
      vdz32=z3-z2
      CROW21 = CROWXYZ(x1,y1,z1,x2,y2,z2)
      if (abs(crow21).lt.0.001) then
        return
      endif
      CROW32 = CROWXYZ(x2,y2,z2,x3,y3,z3)
      if (abs(crow32).lt.0.001) then
        return
      endif

C Make vectors into unit vectors.
      vd21(1)=vdx21/crow21
      vd21(2)=vdy21/crow21
      vd21(3)=vdz21/crow21
      vd32(1)=vdx32/crow32
      vd32(2)=vdy32/crow32
      vd32(3)=vdz32/crow32

C Get dot product. Note ACOS always returns positive
C so additional work needed to identify angle > 180.0
      call dot3(vd21,vd32,prod)
      ang=ACOS(prod)/RAD
      if(prod.gt.0.0)then
        sign=1.0
      else
        sign= -1.0
      endif
      return
      end
