C This file is part of the ESP-r system.
C Copyright Energy Systems Research Unit, University of
C Strathclyde, Glasgow Scotland, 2001.

C ESP-r is free software.  You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation 
C (version 2 orlater).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.

C This file contains routines which execute Radiance commands.
C  autorad   - executes the Radiance commands for a particular command 
C              line act
C  simscene  - simulates daylight coefficients and coupling (if called
C              from bps) or interactively.
C  settime   - user choice of standard times for sky generation
C  touchrif  - generates an options file based on information 
C              generated by rad -t after removing current octree.
C  defview   - generates a new default view.
C  genimage  - generates image(s) to screen or file.
C  dispimage - displays an image file to screen.
C  getglr    - recovers view information and runs glare calculation.
C  getdf     - runs daylight factor calculation.
C  getlux    - runs grid of illuminance calculation.

C ********************** autorad **********************
C autorad options:
C  'Create' - depending on scene purpose create Radiance model
C  else simulate the scene by calling simscene. Note: there are
C  thus two passess for daylight coefficients, the first 'create'
C  to setup the scene, and if called from bps to generate the
C  coefficients (can take a very long time).

      SUBROUTINE autorad
#include "building.h"
#include "model.h"
#include "site.h"
#include "e2r_common.h"
#include "geometry.h"
#include "esprdbfile.h"
#include "material.h"
#include "help.h"
      
      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/FILEP/IFIL
      integer ncomp,ncon
      COMMON/C1/NCOMP,NCON
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

      logical both_esp_wave      ! If true manage pair of scenes.
      logical focus_espg         ! If true editing in context of ESP-r geometry
      integer ipairedscene       ! Index of paired scene.
      COMMON/ESPWAVE/both_esp_wave,focus_espg,ipairedscene(16)

      character outs*124,tfile*72,smode*1,twave*72
      character rifwave*72,ltmpsw*72
      character doitl*256

      logical fict,XST,OK,unixok,WFOK
      logical dowave

      helpinsub='radcmds'  ! set for subroutine

      ITA2=IFIL+7
      ITA3=IFIL+10

C Create a default scene based on current information - ask 
C for additional info if necessary.
      if (cmdact(1:6).ne.'Create') then

C Request was for Calculation rather than Creation so use
C simscene to drive the calculation process. In the case
C of Coupling then simscene will be looking for the file
C passed via -actf in the command line.
        smode='-'
        call simscene(smode)

C Having finished that process, quit.
        close(ieout)
        CALL ERPFREE(ieout,ISTAT)
        CALL EPAGEND
        STOP
      endif

C The command line requested that the model be created
C so, for each scene purpose generate typical sky file.
C Process daylight coefficients seperately as sky file is artificial.
      if (SCENEPURP(ISCENE)(1:8).eq.'Day_coef') then

C Search for existing 'sky' file. (If found delete)
        write(tfile,'(2a)')runpath(1:lnblnk(runpath)),
     &                     rskyfil(1:lnblnk(rskyfil))
        INQUIRE (FILE=tfile,EXIST=XST)
        if(XST)then
          call FPOPEN(ITA3,ISTAT,1,0,tfile)
          call EFDELET(ITA3,ISTAT)
        endif
        call FPOPEN(ITA3,ISTAT,1,2,tfile)
        write(ITA3,'(a)') '# sky patch: 0'
        write(ITA3,'(a)') '  '
        write(ITA3,'(a)') 'void  light  patch_light'
        write(ITA3,'(a)') '0 '
        write(ITA3,'(a)') '0 '
        write(ITA3,'(a)') '3  1000000  1000000  1000000'
        write(ITA3,'(a)') '  '
        write(ITA3,'(a)')'patch_light source patch'
        write(ITA3,'(a)')'0'
        write(ITA3,'(a)')'0'
        write(ITA3,'(a)') '4  0.  0.  1.  13.39'
        call ERPFREE(ITA3,ISTAT)
        skydone=.true.
      else

C Generate sky command use a sun for all skys except for daylight factors
C and night.
C Day_lux is similar to Day_fact except it returns lux values rather than
C daylight factors for a CIE sky. Purpose Illum uses the current sky.
        if (SCENEPURP(ISCENE)(1:8).eq.'External'.or.
     &      SCENEPURP(ISCENE)(1:8).eq.'Internal'.or.
     &      SCENEPURP(ISCENE)(1:5).eq.'Glare'.or.
     &      SCENEPURP(ISCENE)(1:5).eq.'Illum') then
          call settime(IRM,IRD,XRT)
          write(outs,'(a,2i3,f6.2,a,f5.2,2(a,f7.1),a)')'gensky',IRM,IRD,
     &      XRT,' +s -g ',rgrfl,' -a ',sitelat,' -o ',-sitelongdif,
     &      ' -m 0.0'
        elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
          write(outs,'(a,f5.2,2(a,f7.1),a)') 'gensky 6 20 12.0 -c -g ',
     &      rgrfl,' -a ',sitelat,' -o ',-sitelongdif,' -m 0.0'
        elseif (SCENEPURP(ISCENE)(1:7).eq.'Day_lux') then
          write(outs,'(a,f5.2,2(a,f7.1),a)') 'gensky 6 20 12.0 -c -g ',
     &      rgrfl,' -a ',sitelat,' -o ',-sitelongdif,' -m 0.0'
        elseif (SCENEPURP(ISCENE)(1:8).eq.'Coupling') then

C For Coupling and Create we need to initially generate a sky file based on
C calling gendaylit -ang 60 0 -L QDIRILL QDIFILL
C However this results in Warning : skyclearness or skybrightness out of range
C from gendaylit. Instead try -W 840 135 equivalant to CIE clear sky.
C          write(outs,'(a,f5.2)')
C     &      'gendaylit -ang 60 0 -L 805 100 -g ',rgrfl
          write(outs,'(a,f5.2)')
     &      'gendaylit -ang 60 0 -W 840 135 -g ',rgrfl

        elseif (SCENEPURP(ISCENE)(1:9).eq.'Night_ext') then

C Ignore the sun when calling gensky as per Radiance book.
          write(outs,'(a)') 'gensky 6 15 12.0 -s '
        endif

C Now call Radiance with command stored in variable 'outs'.
        call wrtsky(outs)
        call RRSKY(ITA2,ga,IER)
        if (SCENEPURP(ISCENE)(1:8).eq.'External') then
          call RADPAR('av',ga)
        elseif (SCENEPURP(ISCENE)(1:8).eq.'Internal') then
          xga=ga/10.
          call RADPAR('av',xga)
        elseif (SCENEPURP(ISCENE)(1:9).eq.'Night_ext') then
          xga=0.0005   ! as per Radiance book
          call RADPAR('av',xga)
        endif
      endif

C Create inside and outside Radiance 'rad' files.
      call module_opendb(ier)

C Scan for matching MLC for surfaces.
      do 30 ICOMP=1,NCOMP
        call georead(IFIL+1,LGEOM(ICOMP),ICOMP,1,iuout,ier)

C Find the index of the MLC which matches each surface.
        DO 9994 ijj=1,NZSUR(icomp)
          icn1=izstocn(icomp,ijj)
          if(icn1.gt.0)then
            smlcindex(icomp,ijj)=0  ! assume no matching MLC          
            lnssmlc=lnblnk(SMLCN(icomp,ijj))
            do 5 ii=1,nmlc
              if(SMLCN(icomp,ijj)(1:lnssmlc).eq.
     &           mlcname(ii)(1:lnmlcname(ii)))then
                smlcindex(icomp,ijj)=ii   ! remember MLC index  
              endif
  5         continue
          endif
 9994   continue
 30   continue
      ftr=0.92
      fict=.false.

C Ask how to treat adobe surface treatment rough (1) medium (2) fine (3)
      if(iadobe.eq.0)then
        iadobe=1
        SCENEMOTL(ISCENE)='rough'
        call edisp(iuout,'Currently assumes rough adobe treatment.')
      elseif(iadobe.eq.1)then
        call edisp(iuout,'Currently assumes rough adobe treatment.')
      elseif(iadobe.eq.2)then
        call edisp(iuout,'Currently assumes medium adobe treatment.')
      elseif(iadobe.eq.3)then
        call edisp(iuout,'Currently assumes fine adobe treatment.')
      endif
      CALL EASKI(iadobe,' ',
     &  'Texture on surfaces 1=rough 2=medium 3=fine',
     &   1,'F',3,'W',2,'adobe level',IER,nbhelp)
      if(iadobe.eq.1)then
        SCENEMOTL(ISCENE)='rough'
      elseif(iadobe.eq.2)then
        SCENEMOTL(ISCENE)='medium'
      elseif(iadobe.eq.3)then
        SCENEMOTL(ISCENE)='fine'
      endif

C If also producing a Wavefront variant ask for the Wavefront
C file. If currently focused on ESP-r then use ISCENE+1.
      if(both_esp_wave.and.focus_espg)then
        if(SCENEWAVE(ISCENE+1)(1:7).eq.'UNKNOWN')then
          CALL EASKOK(' ','Supply name of Wavefront object file?',
     &      WFOK,nbhelp)
          if(WFOK)then
            call edisp(iuout,' ')
            call edisp(iuout,
     &'Place the Wavefront obj and mtl files in the model rad folder.')
 642        helptopic='e2r_RIF_file_name'
            call gethelptext(helpinsub,helptopic,nbhelp)
            write(ltmpsw,'(a)') 'xx.obj'
            CALL EASKS(ltmpsw,'Wavefront object file?',
     &        '  ',72,'xxx.obj','wavefront file',IER,nbhelp)
            if(ltmpsw.eq.' ')goto 642
            write(SCENEWAVE(ISCENE+1),'(a)') ltmpsw(1:lnblnk(ltmpsw))
          endif
        endif
      elseif(.NOT.focus_espg)then  ! Wavefront focus for scene.
        if(SCENEWAVE(ISCENE)(1:7).eq.'UNKNOWN')then
          CALL EASKOK(' ','Supply name of Wavefront object file?',
     &      WFOK,nbhelp)
          if(WFOK)then
            call edisp(iuout,' ')
            call edisp(iuout,
     &'Place the Wavefront obj and mtl files in the model rad folder.')
 643        helptopic='e2r_RIF_file_name'
            call gethelptext(helpinsub,helptopic,nbhelp)
            write(ltmpsw,'(a)') 'xx.obj'
            CALL EASKS(ltmpsw,'Wavefront object file?',
     &        '  ',72,'xxx.obj','wavefront file',IER,nbhelp)
            if(ltmpsw.eq.' ')goto 643
            write(SCENEWAVE(ISCENE),'(a)') ltmpsw(1:lnblnk(ltmpsw))
          endif
        endif
      endif

C Create the associated Radiance files for form and composition.
C Note this will include most of the files needed for Wavefront
C use whether or not the user has requested this.
      call mkriofil(ftr,fict,'s',IER)
 
C Check for surfaces with SUSE=FIXTURE. If so create a separate
C file to hold the xform commands.
      if(nbofies.gt.0)then
        call mkxform('i',IER)
C << to be done >>
      endif

      IFC=3
      call mkrif(ita2,RIFNAME(ISCENE),IFC,0,'-',ier)

C If Wavefront variant needed setup rifwave and sky file names.
      dowave=.false.
      if(focus_espg.and.both_esp_wave)then
        dowave=.true.
      elseif(.NOT.focus_espg.and.both_esp_wave)then
        dowave=.true.
      elseif(focus_espg.and.(.NOT.both_esp_wave))then
        dowave=.false.
      elseif(.NOT.focus_espg.and.(.NOT.both_esp_wave))then
        dowave=.false.
      endif
      if(dowave)then
        if (SCENEPURP(ISCENE)(1:8).eq.'External')then
          rifwave='obj_ex.rif' 
          rskyfil='obj_ex.sky'
        elseif(SCENEPURP(ISCENE)(1:8).eq.'Internal')then
          rifwave='obj_in.rif'
          rskyfil='obj_in.sky'
        elseif(SCENEPURP(ISCENE)(1:5).eq.'Glare')then
          rifwave='obj_gl.rif'
          rskyfil='obj_gl.sky'
        elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
          rifwave='obj_df.rif'
          rskyfil='obj_df.sky'
        else
          rifwave='obj.rif'
          rskyfil='obj.sky'
        endif
        call mkrif(ITA2,rifwave,IFC,0,'w',IER)
      endif
      if (NBSRIF(ISCENE).eq.1) then
        call mkrif(ita2,LBSRIF(ISCENE),IFC,1,'-',ier)
      endif

C If current focus uses Wavefront geometry and obj file known 
c ask user about generating the rtm file.
      iother=ipairedscene(ISCENE)
      if(SCENEGSRC(ISCENE)(1:9).eq.'Wavefront')then
        if(SCENEWAVE(ISCENE)(1:7).eq.'UNKNOWN')then
          continue
        else
          lnobj=lnblnk(SCENEWAVE(ISCENE))
          call edisp(iuout,' ')
          call edisp(iuout,'The manual command is:')
          if (SCENEPURP(ISCENE)(1:8).eq.'External')then
            write(outs,'(3a)') 'obj2mesh -a obj_ex.mat ',
     &        SCENEWAVE(ISCENE)(1:lnobj),' obj_ex.rtm'
          elseif(SCENEPURP(ISCENE)(1:8).eq.'Internal')then
            write(outs,'(3a)') 'obj2mesh -a obj_in.mat ',
     &        SCENEWAVE(ISCENE)(1:lnobj),' obj_in.rtm'
          elseif(SCENEPURP(ISCENE)(1:5).eq.'Glare')then
            write(outs,'(3a)') 'obj2mesh -a obj_gl.mat ',
     &        SCENEWAVE(ISCENE)(1:lnobj),' obj_gl.rtm'
          elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
            write(outs,'(3a)') 'obj2mesh -a obj_df.mat ',
     &        SCENEWAVE(ISCENE)(1:lnobj),' obj_df.rtm'
          endif
          call edisp(iuout,outs)
          call EASKOK('Convert the Wavefront obj file into a rtm',
     &    'file via obj2mesh?',OK,nbhelp)
          if(OK)then
            call isunix(unixok)
            if(unixok)then
              write(doitl,'(4a)') 'cd ',runpath(1:lnrp),
     &          '; ',outs(1:lnblnk(outs))
C              write(6,*) doitl(1:lnblnk(doitl))
              call runit(doitl,'-')
            else
              call usrmsg('e2r cannot yet invoke obj2mesh',
     &          'on non-linux computers.','W')
            endif
          endif
        endif
      endif

C Or if paired with a Wavefront variant complete the mesh file.
      if(iother.gt.0)then
        if(SCENEGSRC(iother)(1:9).eq.'Wavefront')then
          if(SCENEWAVE(iother)(1:7).eq.'UNKNOWN')then
            continue
          else

C If the user included the runpath in the name of the object file
C this needs to be removed (if the command is invoked in the rad folder.
            call isunix(unixok)
            if(unixok)then
              lnrp=lnblnk(runpath)
              if(SCENEWAVE(iother)(1:lnrp).eq.runpath(1:lnrp))then
                write(twave,'(a)')  
     &          SCENEWAVE(iother)(lnrp+1:lnblnk(SCENEWAVE(iother)))
              else
                write(twave,'(a)')  
     &          SCENEWAVE(iother)(1:lnblnk(SCENEWAVE(iother)))
              endif
            endif
            lnobj=lnblnk(twave)
            call edisp(iuout,' ')
            call edisp(iuout,
     &        'The manual command (within the rad folder) is:')
            if (SCENEPURP(iother)(1:8).eq.'External')then
              write(outs,'(3a)') 'obj2mesh -a obj_ex.mat ',
     &          twave(1:lnobj),' obj_ex.rtm'
            elseif(SCENEPURP(iother)(1:8).eq.'Internal')then
              write(outs,'(3a)') 'obj2mesh -a obj_in.mat ',
     &          twave(1:lnobj),' obj_in.rtm'
            elseif(SCENEPURP(iother)(1:5).eq.'Glare')then
              write(outs,'(3a)') 'obj2mesh -a obj_gl.mat ',
     &          twave(1:lnobj),' obj_gl.rtm'
            elseif(SCENEPURP(iother)(1:8).eq.'Day_fact') then
              write(outs,'(3a)') 'obj2mesh -a obj_df.mat ',
     &          twave(1:lnobj),' obj_df.rtm'
            endif
            call edisp(iuout,outs)
            call EASKOK('Convert the Wavefront obj file into a rtm',
     &        'file via obj2mesh?',OK,nbhelp)
            if(OK)then
              call isunix(unixok)
              if(unixok)then
                write(doitl,'(4a)') 'cd ',runpath(1:lnrp),
     &            '; ',outs(1:lnblnk(outs))
C                write(6,*) doitl(1:lnblnk(doitl))
                call runit(doitl,'-')
              else
                call usrmsg('e2r cannot yet invoke obj2mesh',
     &            'on non-linux computers.','W')
               endif
            endif
          endif
        endif
      endif

C Update rcf file and Radiance configuration file.
      ICHK=3
      call RADCFGOUT(ICHK)

C Generate new octree.
      call touchrif(RIFNAME(ISCENE))
      if (NBSRIF(ISCENE).gt.0) then
        call touchrif(LBSRIF(ISCENE))
      endif

C If scene purpose is coupling, daylight coefficients 
C then return from the create pass.
      if (SCENEPURP(ISCENE)(1:8).eq.'Coupling') then
        return
      elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_coef') then
        return
      endif

C Continue and run radiance simulation?
      helptopic='e2r_proceed_with_sim'
      call gethelptext(helpinsub,helptopic,nbhelp)
      if (SCENEPURP(ISCENE)(1:5).eq.'Glare') then
        CALL USRMSG(
     &  'Glare scene description is now complete. You will need to',
     &  'redefine the eye point directives before simulating.','W')
        ok=.false.
      else
        if (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
          continue
        else
          if(indxvew.eq.0)then
            call usrmsg('The scene is complete except for view points.',
     &        'Please define one or more and then try rendering.','W')
            return
          endif
        endif

        call EASKOK('Scene description complete.',
     &           'Proceed with simulation?',OK,nbhelp)
      endif
      if (OK) then
        if (SCENEPURP(ISCENE)(1:8).eq.'External'.or.
     &      SCENEPURP(ISCENE)(1:8).eq.'Internal'.or.
     &      SCENEPURP(ISCENE)(1:9).eq.'Night_ext') then

C Generate default view information and then the image.
          call defview
          call genimage(1)
        elseif (SCENEPURP(ISCENE)(1:5).eq.'Glare') then

C Generate the glare image assuming the user has set viewpoints.
C          call defview
          call genimage(2)
          call getglr
          call dispimage(2)
        elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then

C Calculate the daylight factors and then diplay them via dfgridlbl.
          call getdf
          call DFGRIDLBL('d')
        elseif (SCENEPURP(ISCENE)(1:7).eq.'Day_lux'.or.
     &          SCENEPURP(ISCENE)(1:5).eq.'Illum') then

C Calculate the illuminance at grid points and then diplay via dfgridlbl.
          call getlux
          call DFGRIDLBL('l')
        endif
      endif

      return
      end

C ********************** simscene **********************
C Simulates daylight coefficients and coupling (called 
C from bps with mode '-' or interactively with mode 'i').

C << adapt data in file from bps to include directives for adjusting
C << the intensity of Radiance lighting fixtures.

      SUBROUTINE simscene(mode)
#include "building.h"
#include "model.h"
#include "e2r_common.h"
      
      integer lnblnk  ! function definition
      character mode*1 ! if 'i' then interactive

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/FILEP/IFIL
      integer itc,icout  ! trace verbosity and output channel
      common/trace/itc,icout

C Radiance based commons are explained in e2r_common.h.
      character simfile*72
      COMMON/fromsim/simfile

      character outs*124,doit*124,doitl*256,tfile*72
      character QT*1,WORD*24
      character*72 LTMPFL,LTMPFA,rn

C There are 8 patch zones in the sky dome.
      dimension npz(8),pzalt(8),pzazim(8),solid(8)

      LOGICAL XST,unixok

      data npz/30,60,84,108,126,138,144,145/
      data pzalt/6.,18.,30.,42.,54.,66.,78.,90./
      data pzazim/12.,12.,15.,15.,20.,30.,60.,0./
      data solid/13.39,13.39,13.39,13.39,13.39,13.39,13.39,13.39/

      QT=CHAR(39)
      ITA1=IFIL+6
      ITA2=IFIL+7
      ITA3=IFIL+10
      call isunix(unixok)

C Simulate scene as defined by aim argument.
      if (SCENEPURP(ISCENE)(1:8).eq.'Coupling') then

C Read from transfer file defined as the argument for -actf from
C the users home folder.
        if(unixok)then
          write(doit,'(6a)') 'cp ',upath(1:lnblnk(upath)),'/',
     &               simfile(1:lnblnk(simfile)),'  ',runpath(1:lnrp)
        else
          write(doit,'(6a)') 'copy /y ',upath(1:lnblnk(upath)),'/',
     &               simfile(1:lnblnk(simfile)),'  ',runpath(1:lnrp)
        endif
        call runit(doit,'-')
        ier=0
        write (LTMPFL,'(2a)') runpath(1:lnrp),simfile(1:lnblnk(simfile))
        call FPOPEN(ITA1,ISTAT,1,1,LTMPFL)
        if (ier.ne.0) then 
          call usrmsg('Cannot open transfer file:',simfile,'F')
          return
        endif
        CALL STRIPC(ITA1,OUTS,99,ND,0,'transfer file',IER)
        if (ier.ne.0) then 
          call usrmsg('Cannot read transfer file:',simfile,'F')
          return
        endif
        K=0
        call EGETWR(OUTS,K,RSALT,0.,90.,'F','Solar alt',IER)
        call EGETWR(OUTS,K,RSAZI,-180.,180.,'W','Sol azi',IER)
        call EGETWR(OUTS,K,QDIR,0.,1250.,'W','Dir solar',IER)
        call EGETWR(OUTS,K,QDIF,0.,1250.,'W','Diff solar',IER)
        call EGETWI(OUTS,K,IBLNDS,0,1,'W','Blinds state',IER)

C If ND >  5 then read the next token as YES or NO. If yes then implied
C request for manual edting of the iesfil.
        IF(ND.gt.5)THEN
          CALL EGETW(OUTS,K,WORD,'W','EDIT',IFLAG)
          if(WORD.eq.'YES')then
            write (tfile,'(a,a)') runpath(1:lnrp),
     &        iesfil(1:lnblnk(iesfil))
            call vifile(ITA2,tfile,'e',ier)
          endif
        endif

C << Update the cgc file need to include direct reference? >>
C << if there are read subsequent lines read them for lighting controls >>

C Check solar position and radiation.
        QDIRH=QDIR*esind(RSALT/57.296)

C Create sky command and sky.
        if (RSALT.gt.5.0) then
          write(outs,'(a,2f7.1,a,2f7.1,a,f5.2)')'gendaylit -ang ',RSALT,
     &      RSAZI,' -W ',QDIR,QDIF,' -g ',rgrfl
        else
          if (QDIR.gt.0.0) then
            write(outs,'(a,2f7.1,2(a,f7.1),a,f5.2)')'gensky -ang ',
     &        RSALT,RSAZI,' -B ',QDIF,' -R ',QDIRH,' -g ',rgrfl
          else
            write(outs,'(a,2f7.1,a,f7.1,a,f5.2)')'gensky -ang ',RSALT,
     &        RSAZI,' -c -B ',QDIF,' -g ',rgrfl
          endif
        endif
        call wrtsky(outs)

C Recreate octree, setting rtrace options, and run rtrace.
        if (IBLNDS.ne.0.and.NBSRIF(ISCENE).eq.0) then

C Error can't simulate switched blinds.
          call usrmsg('Cannot simulate switched blinds.','  ','F')
        endif
        if (IBLNDS.eq.0) then
          rn=RIFNAME(ISCENE)
        else
          rn=LBSRIF(ISCENE)
        endif

C Rebuild.
        call touchrif(rn)

C Run rtrace.
        write(doitl,'(4a,1x,9a)') 'cd ',runpath(1:lnrp),
     &    '; rtrace -ov -I -h @',
     &    optnfil(1:lnblnk(optnfil)),octfil(1:lnblnk(octfil)),
     &     ' < ',LDFGRID(ISCENE)(1:lnblnk(LDFGRID(ISCENE))),
     &     ' | rcalc -e ',QT,
     &     '$1=179*(0.265*$1+0.670*$2+0.065*$3)',QT,' >> ',
     &    simfile(1:lnblnk(simfile))

C Debug/trace.
        if(itc.eq.0)then
          continue
        else     
          write(icout,*) doitl(1:lnblnk(doitl))
        endif
        call runit(doitl,'-')

C Copy transfer file back to users homedir (bps is expecting it to be there).
        call isunix(unixok)
        if(unixok)then
          write(doit,'(5a)') 'cp ',runpath(1:lnrp),
     &      simfile(1:lnblnk(simfile)),'  ',upath(1:lnblnk(upath))
        else
          write(doit,'(5a)') 'copy /y ',runpath(1:lnrp),
     &      simfile(1:lnblnk(simfile)),'  ',upath(1:lnblnk(upath))
        endif
        call runit(doit,'-')

C Daylight coefficients.
      elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_coef') then

C Do for all blind states.
        IBLND=0
 1234   do 123 IPTCH=1,145

C Search for existing 'sky' file. (If found delete)
          write(tfile,'(a,a)')runpath(1:lnrp),
     &                        rskyfil(1:lnblnk(rskyfil))
          INQUIRE (FILE=tfile,EXIST=XST)
          if(XST)then
            call FPOPEN(ITA3,ISTAT,1,0,tfile)
            call EFDELET(ITA3,ISTAT)
          endif
          call FPOPEN(ITA3,ISTAT,1,2,tfile)
          write(ITA3,'(a,i4)') '# sky patch: ',IPTCH
          write(ITA3,'(a)') '  '
          write(ITA3,'(a)') 'void  light  patch_light'
          write(ITA3,'(a)') '0 '
          write(ITA3,'(a)') '0 '
          write(ITA3,'(a)') '3  1000000  1000000  1000000'
          write(ITA3,'(a)') '  '

C Calculate patch direction vector (location).
          if(IPTCH.ge.1.and.IPTCH.le.30) nzone = 1
          if(IPTCH.ge.31.and.IPTCH.le.60) nzone = 2
          if(IPTCH.ge.61.and.IPTCH.le.84) nzone = 3
          if(IPTCH.ge.85.and.IPTCH.le.108) nzone = 4
          if(IPTCH.ge.109.and.IPTCH.le.126) nzone = 5
          if(IPTCH.ge.127.and.IPTCH.le.138) nzone = 6
          if(IPTCH.ge.139.and.IPTCH.le.144) nzone = 7
          if(IPTCH.eq.145) nzone = 8
          if(nzone.eq.8) then
            xp = 0.0
            yp = 0.0
            zp = 1.0
          elseif(nzone.eq.1) then
            aazim = pzazim(nzone)*(IPTCH - 1)
            xp = ecosd(aazim)*ecosd(pzalt(nzone))
            yp = esind(aazim)*ecosd(pzalt(nzone))
            zp = esind(pzalt(nzone))
          else
            aazim = pzazim(nzone)*((IPTCH - npz(nzone-1)) - 1)
            xp = ecosd(aazim)*ecosd(pzalt(nzone))
            yp = esind(aazim)*ecosd(pzalt(nzone))
            zp = esind(pzalt(nzone))
          endif

C Debug
C          write(6,*) 'IPTCH',' Xp',' Yp',' Zp',' azim',' alt'
C          write(6,*) IPTCH,' ',xp,' ',yp,' ',zp,' ',aazim,' ',pzalt   
C End Debug

          write(ITA3,'(a)') 'patch_light source patch'
          write(ITA3,'(a)') '0'
          write(ITA3,'(a)') '0'
          write(ITA3,'(a,4f10.6)') '4 ',xp,yp,zp,solid(nzone)
          call ERPFREE(ITA3,ISTAT)
          skydone=.true.

C Generate new octree.
          if (IBLND.eq.0) then
            rn=RIFNAME(ISCENE)
          else
            rn=LBSRIF(ISCENE)
          endif

C Recreate octree and get current file names.
          call touchrif(rn)
          write(outs,'(2a,i3)')' Daylight coefficient: ',
     &                         'Run oconv for patch ',IPTCH
          call edisp(iuout,outs)

C Check for output file only for first patch.
          if (IPTCH.eq.1) then

C Create temporary file (format: .PID.cmd). And write a header
C to that file.
            call esppid(ipid)
            write(outs,'(a,i7,a)') runpath(1:lnrp),ipid,'.dcf'
            call st2file(outs,LTMPFA)
            XST=.FALSE.
            INQUIRE (FILE=LTMPFA(1:lnblnk(LTMPFA)),EXIST=XST)
            if(XST)then
              call FPOPEN(ITA2,ISTAT,1,0,LTMPFA)
              call EFDELET(ITA2,ISTAT)
            endif
            call FPOPEN(ITA2,ISTAT,1,2,LTMPFA)
            write(ITA2,'(a)')'# Sensor illuminance for daylight coefs.'
            write(ITA2,'(a)')'# Temporary data file.'
            call ERPFREE(ITA2,ISTAT)
          endif

C Run rtrace. Because this takes a long time provide feedback.
          write(outs,98)' Daylight coefficient: ',
     &                  'Run rtrace for patch ',IPTCH
 98       format(2a,i3)

          write(doitl,'(4a,1x,9a)') 'cd ',runpath(1:lnrp),
     &      '; rtrace -ov -I -h @',
     &      optnfil(1:lnblnk(optnfil)),octfil(1:lnblnk(octfil)),
     &      ' < ',LDFGRID(ISCENE)(1:lnblnk(LDFGRID(ISCENE))),
     &      ' | rcalc -e ',QT,
     &      '$1=(0.265*$1+0.670*$2+0.065*$3)/(1000000*2*PI/145.)',
     &                               QT,' >>',LTMPFA(1:lnblnk(LTMPFA))

C Debug/trace.
          if(itc.eq.0)then
            continue
          else     
            write(icout,*) doitl(1:lnblnk(doitl))
          endif
          call runit(doitl,'-')

C Display progress...
          if (IPTCH.eq.35) then
            call edisp(IUOUT,'Calculation 25% complete.')
          elseif (IPTCH.eq.70) then
            call edisp(IUOUT,'Calculation 50% complete.')
          elseif (IPTCH.eq.105) then
            call edisp(IUOUT,'Calculation 75% complete.')
          endif
 123    continue

C Read in daylight coefficients and save to cgc file.
        ISET=IBLND+1
        call RDCFTMP(LTMPFA,ISET)

C Remove temporary files if not interactive.
        if(mode.eq.'i')then
          continue
        else
          XST=.FALSE.
          INQUIRE (FILE=LTMPFA(1:lnblnk(LTMPFA)),EXIST=XST)
          if(XST)then
            call FPOPEN(ITA2,ISTAT,1,0,LTMPFA)
            call EFDELET(ITA2,ISTAT)
          endif
        endif
         
        call edisp(IUOUT,'Daylight coefficient set complete')

C Do other blind states.
        if (NBSRIF(ISCENE).gt.0) then
          IBLND=IBLND+1
          if (IBLND.le.NBSRIF(ISCENE)) then
            call edisp(IUOUT,'Calculating next blind state.')
            goto 1234
          endif
        endif
      endif

      return
      end


C ********************** settime **********************
C User choices for standard times for sky generation.

      SUBROUTINE settime(IRM,IRD,XRT)
#include "building.h"
#include "e2r_common.h"
#include "help.h"

      helpinsub='radcmds'  ! set for subroutine

C Ask winter am/pm spring am/pm summer am/pm
      helptopic='standard_times_of_day'
      call gethelptext(helpinsub,helptopic,nbhelp)
      ino=0
      idno=1
      call MENUATOL('Select day & time for scene.',
     &    'Select one from list:',
     &    'a winter morning','b winter noon','c winter afternoon',
     &    'd spring morning','e spring noon','f spring afternoon',
     &    'g summer morning','h summer noon','i summer afternoon',' ',
     &    ' ',' ',ino,idno,nbhelp)
      IWDT=INO
      if(IWDT.eq.1)then
        irdoy=15
        rtime=10.0
      elseif(IWDT.eq.2)then
        irdoy=15
        rtime=12.0
      elseif(IWDT.eq.3)then
        irdoy=15
        rtime=15.0
      elseif(IWDT.eq.4)then
        irdoy=74
        rtime=9.5
      elseif(IWDT.eq.5)then
        irdoy=74
        rtime=12.0
      elseif(IWDT.eq.6)then
        irdoy=74
        rtime=15.5
      elseif(IWDT.eq.7)then
        irdoy=160
        rtime=9.0
      elseif(IWDT.eq.8)then
        irdoy=160
        rtime=12.0
      elseif(IWDT.eq.9)then
        irdoy=160
        rtime=16.0
      endif
      CALL EDAYR(irdoy,IRD,IRM)
      XRT=rtime

      return
      end

C ********************** touchrif **********************
C Generates an options file based on information generated
C by a 'rad -t' call after removing current octree.

      SUBROUTINE touchrif(RIFFILE)
#include "building.h"
#include "e2r_common.h"
      
      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/FILEP/IFIL
      integer itc,icout  ! trace verbosity and output channel
      common/trace/itc,icout

      character doit*124,RIFFILE*72

C Read current rif file.
      ITA2=IFIL+7
      ITA3=IFIL+10
      call RRIF(ITA2,ITA3,RIFFILE,'s',IERR)
      
C Remove octree to force rebuild as radiance sometimes gets this wrong.
      write(doit,'(4a)') 'cd ',runpath(1:lnrp),
     &                   '; rm -f ',octfil(1:lnblnk(octfil))
      call runit(doit,'-')

C Run rad in 'touch up' mode.
      write(doit,'(4a)') 'cd ',runpath(1:lnrp),
     &                   '; rad -t ',RIFFILE(1:lnblnk(RIFFILE))

C Debug/trace.
      if(itc.eq.0)then
        continue
      else     
        write(icout,*) doit(1:lnblnk(doit))
      endif
      call runit(doit,'-')
      
      return
      end

C ********************** defview **********************
C Generate a new default view.

      SUBROUTINE defview
#include "building.h"
#include "prj3dv.h"
#include "e2r_common.h"

      COMMON/FILEP/IFIL

      COMMON/RAY5OB/OXMN,OYMN,OZMN,OXMX,OYMX,OZMX

      common/initv/initvt,EYEMI(3),VIEWMI(3),ANGI
      logical both_esp_wave      ! If true manage pair of scenes.
      logical focus_espg         ! If true editing in context of ESP-r geometry
      integer ipairedscene       ! Index of paired scene.
      COMMON/ESPWAVE/both_esp_wave,focus_espg,ipairedscene(16)

      real dvazim  ! default calculated viewing angle azimuth
      real dvelev  ! default calculated viewing angle elevation
      real dvangv,dvangh ! vertical and horizontal default viewing angles
      integer ier  ! for use with call to mkrif
      integer ita2 ! file unit for use with mkrif
      logical short ! briefer format
      logical okfore,okaft ! for near zero cuts
      logical dowave
      character rifwave*72

      ITA2=IFIL+7

C External view point.
      if (SCENEPURP(ISCENE)(1:8).eq.'External'.or.
     &    SCENEPURP(ISCENE)(1:9).eq.'Night_ext') then

C Create a default view, set file name, set bounds for slightly
C smaller than the ground disk. Save off zone bounds before
C adding in the ground disk.
        OXMN=XMN; OYMN=YMN; OZMN=ZMN
        OXMX=XMX; OYMX=YMX; OZMX=ZMX
        XD=(XMX-XMN)**2 + (YMX-YMN)**2
        DD=SQRT(XD)*1.5
        cx=XMN+((XMX-XMN)/2.)
        cy=YMN+((YMX-YMN)/2.)
        XMN=cx-DD; XMX=cx+DD
        YMN=cy-DD; YMX=cy+DD
        ZMN=ZMN; ZMX=DD
        rvpx=-100.0; rvpy=-100.0; rvpz=100.
        if(initvt.eq.1)then
          rvpx=EYEMI(1)
          rvpy=EYEMI(2)
          rvpz=EYEMI(3)
        endif

C Given the eye point and the view point return azimuth & elevation
C and set the default viewing angles to 1.2 factor.
        call ln2az(rvpx,rvpy,rvpz,VIEWM(1),VIEWM(2),VIEWM(3),
     &    dvazim,dvelev)
        call ang3vtx(OXMN,OYMN,OZMN,EYEM(1),EYEM(2),EYEM(3),OXMX,
     &    OYMX,OZMX,eang)
        dvangv=eang*1.2
        dvangh=eang*1.2
        if(rvpx.lt.99.0.and.rvpy.lt.99.0.and.rvpz.lt.99.0)then
          short=.true.
        else
          short=.false.
        endif
        call eclose(cutfor,0.0,0.1,okfore)
        call eclose(cutaft,0.0,0.1,okaft)
 
        call AZ2UV(dvazim,dvelev,vdx,vdy,vdz)
        if(short)then
          if(okfore.and.okaft)then
            write(vewcmds(1),'(a,3F7.1,a,3F7.3,a,3F5.2,2(a,F5.1),a)') 
     &       '-vtv -vp',rvpx,rvpy,rvpz,' -vd',vdx,vdy,vdz,
     &       ' -vu ',vux,vuy,vuz,' -vh ',dvangh,' -vv ',dvangv,
     &       ' -vs 0 -vl 0 -vo 0 -va 0'
          else
            write(vewcmds(1),'(a,3F7.1,a,3F7.3,a,3F5.2,4(a,F5.1))') 
     &       '-vtv -vp',rvpx,rvpy,rvpz,' -vd',vdx,vdy,vdz,
     &       ' -vu ',vux,vuy,vuz,' -vh ',dvangh,' -vv ',dvangv,
     &       ' -vs 0 -vl 0 -vo ',cutfor,' -va ',cutaft
          endif
        else
          write(vewcmds(1),'(a,3F8.2,a,3F7.3,a,3F5.2,4(a,F5.1))') 
     &       '-vtv -vp',rvpx,rvpy,rvpz,' -vd',vdx,vdy,vdz,
     &       ' -vu ',vux,vuy,vuz,' -vh ',dvangh,' -vv ',dvangv,
     &       ' -vs 0 -vl 0 -vo ',cutfor,' -va ',cutaft
        endif
        rvewsh(1) = 'def'
        indxvew=1
        ipckvew=1

      elseif (SCENEPURP(ISCENE)(1:8).eq.'Internal') then

C Default internal view at max X average Y and Z, looking in -x dir.
        indxvew=0
        ipckvew=-1
      else
        indxvew=0
        ipckvew=-1
      endif

C If Wavefront variant needed setup rifwave file name.
      dowave=.false.
      if(focus_espg.and.both_esp_wave)then
        dowave=.true.
      elseif(.NOT.focus_espg.and.both_esp_wave)then
        dowave=.true.
      elseif(focus_espg.and.(.NOT.both_esp_wave))then
        dowave=.false.
      elseif(.NOT.focus_espg.and.(.NOT.both_esp_wave))then
        dowave=.false.
      endif

C Update rif file. No use yet made of ier.
      IFC=3
      IER=0
      call MKRIF(ITA2,RIFNAME(ISCENE),IFC,0,'-',IER)
      if(dowave)then
        if (SCENEPURP(ISCENE)(1:8).eq.'External')then
          rifwave='obj_ex.rif'
        elseif(SCENEPURP(ISCENE)(1:8).eq.'Internal')then
          rifwave='obj_in.rif'
        elseif(SCENEPURP(ISCENE)(1:5).eq.'Glare')then
          rifwave='obj_gl.rif'
        elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
          rifwave='obj_df.rif'
        else
          rifwave='obj.rif'
        endif
        call mkrif(ITA2,rifwave,IFC,0,'w',IER)
      endif
      if (NBSRIF(ISCENE).eq.1) then
        call mkrif(ita2,LBSRIF(ISCENE),IFC,1,'-',ier)
      endif

      return
      end


C ********************** genimage **********************
C Manage generation of a rendering to screen or file.
C ITP =1 - image only; =2 - glare info to display on image,
C ITP =3 falsecolor

C << ?? possible addition illuminance and/or false colour image >>

      SUBROUTINE genimage(ITP)
#include "building.h"
#include "site.h"
#include "e2r_common.h"
#include "help.h"
      
      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/FILEP/IFIL
      integer itc,icout  ! trace verbosity and output channel
      common/trace/itc,icout

C Radiance processor cores.
      integer radcores
      common/radcor/radcores

      logical both_esp_wave      ! If true manage pair of scenes.
      logical focus_espg         ! If true editing in context of ESP-r geometry
      integer ipairedscene       ! Index of paired scene.
      COMMON/ESPWAVE/both_esp_wave,focus_espg,ipairedscene(16)

      character tpic*72,tpicd*72,tgif*72,tgifd*72
      character ttif*72,ttifd*72,tjpg*72,tjpgd*72
      character pf*72,outs*124,ltmp*72
      character doit*256
      character rifwave*72
      integer IWT,IWP   ! for radio buttons

      logical XST,anim,unixok,dowave
      CHARACTER*3 NMTHNM(12)

      DATA NMTHNM/'Jan','Feb','Mar','Apr','May','Jun','Jul',
     &'Aug','Sep','Oct','Nov','Dec'/

      helpinsub='radcmds'  ! set for subroutine
      ITA2=IFIL+7
      anim=.false.

      if (ITP.eq.3) then

C Falsecolor image.
        call getfalse(ier)
        return
      elseif (ITP.eq.1) then

C Ask if an animation required.
        helptopic='single_time_or_sequence'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL EASKMBOX(' ',' Image(s):','single time instance',
     &    'sequence',' ',' ',' ',' ',' ',' ',IWDA,nbhelp)
        if(iwda.eq.2)then
          anim = .true.
          IWDQ=2
        else
          if(mono.gt.0)then
            helptopic='render_to_screen_or_file'
            call gethelptext(helpinsub,helptopic,nbhelp)
            CALL EASKMBOX(' ','Image rendering to:','screen','file',
     &        ' ',' ',' ',' ',' ',' ',IWDQ,nbhelp)
          else
            IWDQ=2
          endif
        endif
      else

C If glare put it to file.
        IWDQ=2
      endif

      if(IWDQ.eq.1)then
      
C If in unix then use interactive rendering. If Native Windows do
C not try and display. Parallel via -N number.
        call isunix(unixok)
        if (ipckvew.eq.-1) then 
          if(unixok)then
            if(radcores.eq.1)then
              write(doit,'(5a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e -o x11 ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            else
              write(doit,'(3a,i2,3a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e -o x11 -N',radcores,' ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            endif
          else
            if(radcores.eq.1)then
              write(doit,'(5a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            else
              write(doit,'(3a,i2,3a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e -N',radcores,' ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            endif
          endif
        elseif (ipckvew.eq.0) then 
          if(unixok)then
            if(radcores.eq.1)then
              write(doit,'(5a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e -o x11 ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            else
              write(doit,'(3a,i2,3a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e -o x11 -N',radcores,' ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            endif
          else
            if(radcores.eq.1)then
              write(doit,'(5a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            else
              write(doit,'(3a,i2,3a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e -N',radcores,' ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            endif
          endif
        else
          if(unixok)then
            if(radcores.eq.1)then
              write(doit,'(7a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e -o x11 -v ',
     &          rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),' ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            else
              write(doit,'(3a,i2,5a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e -o x11 -N',radcores,' -v ',
     &          rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),' ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            endif
          else
            if(radcores.eq.1)then
              write(doit,'(7a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e -v ',
     &          rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),' ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            else
              write(doit,'(3a,i2,5a)') 'cd ',runpath(1:lnrp),
     &          '; rad -e -N',radcores,' -v ',
     &          rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),' ',
     &          RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE))),' & '
            endif
          endif
        endif

C Debug/trace.
        if(itc.eq.0)then
          continue
        else     
          write(icout,*) doit(1:lnblnk(doit))
        endif

        call runit(doit,'-')

C Display helpful messages while the image is rendering.
        call edisp(iuout,'Helpful rview commands:  ')
        call edisp(iuout,' All commands can be shortened, to use ')
        call edisp(iuout,' just type in the `rview` window.')
        call edisp(iuout,'  ')
        call edisp(iuout,' aim      - adjust the view direction.')
        call edisp(iuout,' pivot    - pivot eyepoint position.')
        call edisp(iuout,' trace    - info about point @ mouse.')
        call edisp(iuout,' exposure - adjust exposure:')
        call edisp(iuout,'           if `e +1` increase 1 f stop.')
        call edisp(iuout,'           if `e 1` then exposure averaged.')
        call edisp(iuout,' last     - restore last view.  ')
        call edisp(iuout,' last [file] - restore view from file.')
        call edisp(iuout,' view file_name - save current view > file.')
        call edisp(iuout,' view [return] - edit view parameters.')
        call edisp(iuout,' write    - save current image to file.')
        call edisp(iuout,' quit     - give up preview. ')
        call edisp(iuout,'  ')
        call edisp(iuout,'For more info see the radiance manual. ')
      else
        if(anim)then

C Confirm end time and increment. ATIMST is the start time (real) and
C ITST is the start time (int).
          IWF=1  ! assume gif for animation
          call edisp(iuout,' ')
          call edisp(iuout,' Date & time at start of animation...')
          iardoy=irdoy
          CALL EDAYR(iardoy,IADO,IAMO)
          ATIMEST=4.0
          CALL ASKTIM(2,1,IAMO,IADO,iardoy,ATIMEST,ITST,IER)

C Ask for end time of the animation.
          helptopic='end_time_of_animation'
          call gethelptext(helpinsub,helptopic,nbhelp)
          ATIMEFN = ATIMEST + 1.0
          CALL EASKR(ATIMEFN,' ','End time of the animation?',
     &      ATIMEST,'W',24.0,'W',0.2,'end time of anim',IER,nbhelp)

C Figure out the increment.
          IWT=1
          helptopic='single_time_or_sequence'
          call gethelptext(helpinsub,helptopic,nbhelp)
          CALL EASKMBOX('Interval between views:',' ','one min',
     &      'five min','ten min','15 min','20 min','30 min','cancel',
     &      ' ',IWT,nbhelp)
          if(IWT.EQ.1)then
            DT=0.016666
          elseif(IWT.EQ.2)then
            DT=0.083333
          elseif(IWT.EQ.3)then
            DT=0.166666
          elseif(IWT.EQ.4)then
            DT=0.25
          elseif(IWT.EQ.5)then
            DT=0.33333
          elseif(IWT.EQ.6)then
            DT=0.5
          elseif(IWT.EQ.7)then
            return
          endif

C Up the image quality if low.
          if(imgqua(1:3).eq.'Low')imgqua = 'Medium'

C Size of the images (set IFP to double so can filter).
          IWP=1
          CALL EASKMBOX('Image frame size (pixels):',' ','320',
     &      '400','600','700','800','1000','cancel',' ',IWP,nbhelp)
          if(IWP.EQ.1)then
            ipicx=320
          elseif(IWP.EQ.2)then
            ipicx=400
          elseif(IWP.EQ.3)then
            ipicx=600
          elseif(IWP.EQ.4)then
            ipicx=700
          elseif(IWP.EQ.5)then
            ipicx=800
          elseif(IWP.EQ.6)then
            ipicx=1000
          elseif(IWP.EQ.7)then
            return
          endif

C Gif or other conversion.
          CALL EASKMBOX('Output format',' : ','gif (default)',
     &      'tiff','jpeg',' ',' ',' ',' ',' ',IWF,nbhelp)

C Loop from start and increment until at or past the finish time.
          idone=1
          CURT=ATIMEST
          do while (idone.ne.0)
            write(outs,'(a,2i3,f7.3,a,f5.2,2(a,f7.1),a)')'gensky',IAMO,
     &        IADO,CURT,' +s -g ',rgrfl,' -a ',sitelat,' -o ',
     &        -sitelongdif,' -m 0.0'
            CALL wrtsky(outs)

C We will need hdr & gif file for this time. Write out and then use
C sdelim with a N to remove spaces. If before 10h00 include a zero.
            if(CURT.lt.10.0)then
              write(tpic,'(a,2i2.2,a,f6.3,a)') runpath(1:lnrp),
     &          IAMO,IADO,'0',CURT,'.hdr'
              call sdelim(tpic,tpicd,'N',IW)
              write(tgif,'(a,2i2.2,a,f6.3,a)') runpath(1:lnrp),
     &          IAMO,IADO,'0',CURT,'.gif'
              call sdelim(tgif,tgifd,'N',IW)
              write(ttif,'(a,2i2.2,a,f6.3,a)') runpath(1:lnrp),
     &          IAMO,IADO,'0',CURT,'.tif'
              call sdelim(ttif,ttifd,'N',IW)
              write(tjpg,'(a,2i2.2,a,f6.3,a)') runpath(1:lnrp),
     &          IAMO,IADO,'0',CURT,'.jpg'
              call sdelim(tjpg,tjpgd,'N',IW)
            else
              write(tpic,'(a,2i2.2,f6.3,a)') runpath(1:lnrp),
     &          IAMO,IADO,CURT,'.hdr'
              call sdelim(tpic,tpicd,'N',IW)
              write(tgif,'(a,2i2.2,f6.3,a)') runpath(1:lnrp),
     &          IAMO,IADO,CURT,'.gif'
              call sdelim(tgif,tgifd,'N',IW)
              write(ttif,'(a,2i2.2,f6.3,a)') runpath(1:lnrp),
     &          IAMO,IADO,CURT,'.tif'
              call sdelim(ttif,ttifd,'N',IW)
              write(tjpg,'(a,2i2.2,f6.3,a)') runpath(1:lnrp),
     &          IAMO,IADO,CURT,'.jpg'
              call sdelim(tjpg,tjpgd,'N',IW)
            endif

C If animation use a standard name for hdr file (which is going to
C be converted into a timestamp named gif file. Any existing hdr
C file should be removed.
            if (ipckvew.gt.0) then
              write(pf,'(5a)') runpath(1:lnrp),
     &          picfil(1:lnblnk(picfil)),'_',
     &          rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'.hdr'
            else
              write(pf,'(3a)') runpath(1:lnrp),
     &          picfil(1:lnblnk(picfil)),'_X.hdr'
            endif
            CALL ERPFREE(ITA2,ISTAT)
            INQUIRE (FILE=pf,EXIST=XST)
            if(XST)then
              call ERPFREE(ITA2,ISTAT)
              call FPOPEN(ITA2,ISTAT,1,1,pf)
              call EFDELET(ITA2,ISTAT)
            endif

C If Wavefront variant needed setup rifwave file name.
            dowave=.false.
            if(focus_espg.and.both_esp_wave)then
              dowave=.true.
            elseif(.NOT.focus_espg.and.both_esp_wave)then
              dowave=.true.
            elseif(focus_espg.and.(.NOT.both_esp_wave))then
              dowave=.false.
            elseif(.NOT.focus_espg.and.(.NOT.both_esp_wave))then
              dowave=.false.
            endif

C Save changes to RIF file
            IFC=3
            call MKRIF(ITA2,RIFNAME(ISCENE),IFC,0,'-',IER)
            if(dowave)then
              if (SCENEPURP(ISCENE)(1:8).eq.'External')then
                rifwave='obj_ex.rif'
              elseif(SCENEPURP(ISCENE)(1:8).eq.'Internal')then
                rifwave='obj_in.rif'
              elseif(SCENEPURP(ISCENE)(1:5).eq.'Glare')then
                rifwave='obj_gl.rif'
              elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
                rifwave='obj_df.rif'
              else
                rifwave='obj.rif'
              endif
              call mkrif(ITA2,rifwave,IFC,0,'w',IER)
            endif
            if (NBSRIF(ISCENE).eq.1) then
              call mkrif(ita2,LBSRIF(ISCENE),IFC,1,'-',ier)
            endif

C call rad to create image(s) based on current setting of ipckvew.
            if (ipckvew.eq.-1) then 
              if(radcores.eq.1)then
                write(doit,'(4a)') 'cd ',runpath(1:lnrp),
     &            '; rad -e ',RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
              else
                write(doit,'(3a,i2,2a)') 'cd ',runpath(1:lnrp),
     &            '; rad -e -N',radcores,' ',
     &            RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
              endif
            elseif (ipckvew.eq.0) then 
              if(radcores.eq.1)then
                write(doit,'(4a)') 'cd ',runpath(1:lnrp),
     &            '; rad -e ',RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
              else
                write(doit,'(3a,i2,2a)') 'cd ',runpath(1:lnrp),
     &            '; rad -e -N',radcores,' ',
     &            RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
              endif
            else
              if(radcores.eq.1)then
                write(doit,'(6a)') 'cd ',runpath(1:lnrp),
     &            '; rad -e -v ',
     &            rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),
     &            ' ',RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
              else
                write(doit,'(3a,i2,4a)') 'cd ',runpath(1:lnrp),
     &            '; rad -e -N',radcores,' -v ',
     &            rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),' ',
     &            RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
              endif
            endif

C Debug/trace.
            if(itc.eq.0)then
              continue
            else     
              write(icout,*) doit(1:lnblnk(doit))
            endif

            write(outs,'(2a,i3,F6.2)') 'Currently working on...',
     &        NMTHNM(IAMO),IADO,CURT
            call edisp(iuout,outs)
            call runit(doit,'-')

C Generate a label and put into tpic.
            write(doit,'(2a,i3,F6.2,4a)') 'psign -h 18 ',NMTHNM(IAMO),
     &        IADO,CURT,' | pcompos ',pf(1:lnblnk(pf)),' 0 0 - 2 2 > ',
     &        tpicd(1:lnblnk(tpicd))


C Debug/trace.
            if(itc.eq.0)then
              continue
            else     
              write(icout,*) doit(1:lnblnk(doit))
            endif
            call runit(doit,'-')

C Generate command to convert to a gif.
            if(IWF.eq.1)then
              write(doit,'(4a)') 'ra_gif ',tpicd(1:lnblnk(tpicd)),' ',
     &          tgifd(1:lnblnk(tgifd))
            elseif(IWF.eq.2)then
              write(doit,'(4a)') 'ra_tiff ',tpicd(1:lnblnk(tpicd)),' ',
     &          ttifd(1:lnblnk(ttifd))
            elseif(IWF.eq.3)then

C Using the Imagemagic utility convert to go from hdr to jpg.
C Includes -gamma 2.2 to prevent darkening of the jpg. 
C An alternative command line option for a 20% brightening and
C -20% contrast reduction to look better.
C              write(doit,'(4a)') 'convert -brightness-contrast 20x-20 ',
C     &          tpicd(1:lnblnk(tpicd)),' ',tjpgd(1:lnblnk(tjpgd))
C              write(doit,'(4a)') 'convert ',tpicd(1:lnblnk(tpicd)),' ',
C     &          tjpgd(1:lnblnk(tjpgd))
              write(doit,'(4a)') 'convert -gamma 2.2 ',
     &          tpicd(1:lnblnk(tpicd)),' ',tjpgd(1:lnblnk(tjpgd))
            endif

C Debug/trace.
            if(itc.eq.0)then
              continue
            else     
              write(icout,*) doit(1:lnblnk(doit))
            endif
            call runit(doit,'-')

C Increment the current time.
            CURT=CURT+DT
            if(CURT.gt.ATIMEFN)idone=0
          end do
        else

C See if there is an existing hdr file by this name.
          if (ipckvew.gt.0) then
            write(pf,'(5a)') runpath(1:lnrp),
     &        picfil(1:lnblnk(picfil)),'_',
     &        rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'.hdr'
          else
            write(pf,'(3a)') runpath(1:lnrp),
     &        picfil(1:lnblnk(picfil)),'_X.hdr'
          endif
          CALL ERPFREE(ITA2,ISTAT)
          INQUIRE (FILE=pf,EXIST=XST)
          if(XST)then
            helptopic='overwrite_existing_pic'
            call gethelptext(helpinsub,helptopic,nbhelp)
            write(outs,'(a,a)') 'There is an existing picture file ',
     &        pf(1:lnblnk(pf))
            CALL EASKMBOX(outs,'Do you want to: ','overwrite',
     &        'choose another name','cancel',' ',' ',' ',' ',' ',
     &        IWR,nbhelp)
            if(IWR.eq.3)then
              return
            elseif(IWR.eq.2)then
 1111         ltmp = picfil
              CALL EASKS(ltmp,
     &          'Picture root name (modify slightly)?',
     &          ' ',72,'xx','picture root file name',IER,nbhelp)
              if(ltmp(1:2).ne.'  ') then
                picfil = ltmp
              else
                goto 1111
              endif
              dowave=.false.
              if(focus_espg.and.both_esp_wave)then
                dowave=.true.
              elseif(.NOT.focus_espg.and.both_esp_wave)then
                dowave=.true.
              elseif(focus_espg.and.(.NOT.both_esp_wave))then
                dowave=.false.
              elseif(.NOT.focus_espg.and.(.NOT.both_esp_wave))then
                dowave=.false.
              endif
              IFC=3
              call MKRIF(ITA2,RIFNAME(ISCENE),IFC,0,'-',IER)
              if(dowave)then
                if (SCENEPURP(ISCENE)(1:8).eq.'External')then
                  rifwave='obj_ex.rif'
                elseif(SCENEPURP(ISCENE)(1:8).eq.'Internal')then
                  rifwave='obj_in.rif'
                elseif(SCENEPURP(ISCENE)(1:5).eq.'Glare')then
                  rifwave='obj_gl.rif'
                elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
                  rifwave='obj_df.rif'
                else
                  rifwave='obj.rif'
                endif
                call mkrif(ITA2,rifwave,IFC,0,'w',IER)
              endif
              if (NBSRIF(ISCENE).eq.1) then
                call mkrif(ita2,LBSRIF(ISCENE),IFC,1,'-',ier)
              endif
            elseif(IWR.eq.1)then

C Set file name including path and open.
              call ERPFREE(ITA2,ISTAT)
              call FPOPEN(ITA2,ISTAT,1,1,pf)
              call EFDELET(ITA2,ISTAT)
            endif
          endif
        endif
        if (ipckvew.eq.-1) then 
          if(radcores.eq.1)then
            write(doit,'(4a)') 'cd ',runpath(1:lnrp),
     &        '; rad -e ',RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
          else
            write(doit,'(3a,i2,2a)') 'cd ',runpath(1:lnrp),
     &        '; rad -e -N',radcores,' ',
     &        RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
          endif
        elseif (ipckvew.eq.0) then 
          if(radcores.eq.1)then
            write(doit,'(4a)') 'cd ',runpath(1:lnrp),
     &        '; rad -e ',RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
          else
            write(doit,'(3a,i2,2a)') 'cd ',runpath(1:lnrp),
     &        '; rad -e -N',radcores,' ',
     &        RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
          endif
        else
          if(radcores.eq.1)then
            write(doit,'(6a)') 'cd ',runpath(1:lnrp),
     &        '; rad -e -v ',
     &        rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),
     &        ' ',RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
          else
            write(doit,'(3a,i2,4a)') 'cd ',runpath(1:lnrp),
     &        '; rad -e -N',radcores,' -v ',
     &        rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),' ',
     &        RIFNAME(ISCENE)(1:lnblnk(RIFNAME(ISCENE)))
          endif
        endif

C Debug/trace.
        if(itc.eq.0)then
          continue
        else     
          write(icout,*) doit(1:lnblnk(doit))
        endif
        call edisp(iuout,'Look in the text window for progress.')

C Note the time.
        call runit(doit,'-')
        if (ITP.eq.1)call dispimage(1)
      endif

      return
      end

C ********************** dispimage **********************
C Display an image file to screen.
C ITP =1 - image only; =2 - glare info to display on image.
C Assumes that the path to the image is given.

      SUBROUTINE dispimage(ITP)
#include "building.h"
#include "e2r_common.h"
      
      integer lnblnk  ! function definition

      COMMON/FILEP/IFIL
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/SPAD/MMOD,LIMIT,LIMTTY
      integer menuchw,igl,igr,igt,igb,igw,igwh
      common/VIEWPX/menuchw,igl,igr,igt,igb,igw,igwh

      character doit*124,pf*72,gf*72,ugrf*72,ltmp*72
      character outs*124,outstr*124
      
      real UGRV
      
      logical XST,unixok

C Generate picture and glare file names.
      if (ipckvew.gt.0) then
        write(pf,'(4a)') picfil(1:lnblnk(picfil)),'_',
     &    rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'.hdr'
        write(gf,'(4a)') picfil(1:lnblnk(picfil)),'_',
     &    rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'.glr'
      else
        write(pf,'(2a)') picfil(1:lnblnk(picfil)),'_X.hdr'
        write(gf,'(2a)') picfil(1:lnblnk(picfil)),'_X.glr'
      endif

      if (ITP.eq.1) then
        call isunix(unixok)
        if(unixok)then
          call edisp(iuout,' Starting ximage with the commands:')

C -m forces a dither to monochrome, -b is supposed to force greyscale
C but does not.
          if(mono.eq.0)then
            write(doit,'(5a)') 'cd ',runpath(1:lnrp),
     &       '; ximage -m ',pf(1:lnblnk(pf)),' & '
          elseif(mono.eq.1)then
            write(doit,'(5a)') 'cd ',runpath(1:lnrp),
     &       '; ximage -b ',pf(1:lnblnk(pf)),' & '
          elseif(mono.eq.2)then
            write(doit,'(5a)') 'cd ',runpath(1:lnrp),
     &       '; ximage ',pf(1:lnblnk(pf)),' & '
          endif
          call runit(doit,'-')
        else
          call edisp(iuout,' ximage not available on Windows.')
        endif
      elseif (ITP.eq.2) then
      
C First display glare sources on image.
        write(doit,'(7a)') 'cd ',runpath(1:lnrp),
     &       '; xglaresrc ',pf(1:lnblnk(pf)),' ',gf(1:lnblnk(gf)),' & '
        call runit(doit,'-')

C Calculate UGR from glare file and display to screen.
        write (doit,'(7a)') 'cd ',runpath(1:lnrp),
     &      '; glarendx -t ugr -h ',gf(1:lnblnk(gf)),' > ',
     &      gf(1:lnblnk(gf)),'.ugr'
        call runit(doit,'-')
        
C Read file then write to text feedback.
        write (ugrf,'(2a)') gf(1:lnblnk(gf)),'.ugr'
        write (ltmp,'(2a)') 
     &             runpath(1:lnrp),ugrf(1:lnblnk(ugrf))
        ITA1 = IFIL+6
        call ERPFREE(ITA1,ISTAT)
        call FINDFIL(ltmp,XST)
        if(XST)then
          call edisp (iuout,'Angle  UGR')
          XMIN=0.0; XMAX=0.0; YMIN=0.0; YMAX=1.0
          call FPOPEN(ITA1,ISTAT,1,1,ltmp)
          CALL STRIPC(ITA1,OUTSTR,0,ND,0,'UGR results',IFER)
          K=0
          CALL EGETWR(OUTSTR,K,XTMP,0.,0.,'-','angle',IER)
          CALL EGETWR(OUTSTR,K,XTMP,0.,0.,'-','UGR value',IER)
          UGRV=XTMP
          write (outs,'(5a,f6.2)')' UGR for scene ',
     &      SCENERT(ISCENE)(1:lnblnk(SCENERT(ISCENE))),' view ',
     &      rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),': ',UGRV
          call edisp (iuout,outs)
          call ERPFREE(ITA1,ISTAT)
        endif
      else
        return
      endif

      return
      end

C ********************** getglr **********************
C Recover view information and run a glare calculation.
C Suggested values for rendering options are shown on the web:
C radsite.lbl.gov/radiance/refer/Notes/rpict_options.html

      SUBROUTINE getglr
#include "building.h"
#include "e2r_common.h"
      
      integer lnblnk  ! function definition
      integer iCountWords

      COMMON/FILEP/IFIL
      integer itc,icout  ! trace verbosity and output channel
      common/trace/itc,icout

      character rendopt*200
      character word*12,doit*300,ltmp*200,gf*72,pf*72,glst*4

      ITA2 = IFIL+7
      ITA3 = IFIL+10

      IER=0
      call RRIF(ita2,ita3,RIFNAME(ISCENE),'s',IER)

C Check if view info is in a file or kept in the rif file and set 
C glare angle (set minimum angle to 90 degrees).
      glang=90.
      if (vewcmds(ipckvew)(1:3).eq.'-vf') then
        K=4
        CALL EGETW(vewcmds(ipckvew),K,ltmp,'-','view file',IFLAG) 
        call RRVEW(ITA2,ltmp,IER)
        if ((angh/2.).gt.glang) glang=angh/2.
        if ((angv/2.).gt.glang) glang=angv/2.
      else
        K=0
        ITMS = iCountWords(vewcmds(ipckvew))
        do 2112 IX=1,(ITMS-1)
          CALL EGETW(vewcmds(ipckvew),K,word,'-','view data',IFLAG)
          if (word(1:3).eq.'-vh'.or.word(1:3).eq.'-vv') then
            CALL EGETWR(vewcmds(ipckvew),K,ang,0.0,360.0,'W',
     &                                          'view angle',IFLAG)
            if ((ang/2.).gt.glang) glang=ang/2.
          endif
 2112   continue
      endif

C Round glang up to next multiple of ten degrees.
      test=(glang/10.0)-float(INT(glang/10.0))
      if (abs(test).ge.0.001) then
        glang=float(INT(glang/10.)+1)*10.
      endif

C Create strings of maximum angle and glare file name (based on pic name).
      write (word,'(i4)') INT(glang)
      K=0
      CALL EGETW(word,K,glst,'-','max glr angle',IFLAG)
      
C Read opt file so as rendering parameters can be set for rtrace.
      xval=-1.
      call RADPAR('dp',xval)
      write (ltmp,'(a,i4)') '-dp ',int(xval)
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('ds',xval)
      write (ltmp,'(2a,f4.2,1x)') 
     &       rendopt(1:lnblnk(rendopt)),' -ds ',xval
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('dj',xval)
      if (xval.gt.0.) then
        write (ltmp,'(2a,f4.2,1x)') 
     &    rendopt(1:lnblnk(rendopt)),' -dj ',xval
        write (rendopt,'(a)') ltmp
      endif
      xval=-1.
      call RADPAR('dt',xval)
      write (ltmp,'(2a,f4.2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -dt ',xval
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('dc',xval)
      write (ltmp,'(2a,f4.2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -dc ',xval
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('dr',xval)
      write (ltmp,'(2a,i2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -dr ',int(xval)
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('st',xval)
      write (ltmp,'(2a,f4.2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -st ',xval
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('ab',xval)
      write (ltmp,'(2a,i2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -ab ',int(xval)
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('aa',xval)
      write (ltmp,'(2a,f4.2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -aa ',xval
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('ad',xval)
      write (ltmp,'(2a,i4,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -ad ',int(xval)
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('as',xval)
      write (ltmp,'(2a,i4,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -as ',int(xval)
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('ar',xval)
      write (ltmp,'(2a,i4,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -ar ',int(xval)
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('av',xval)
      if(xval.lt.0.1) xval = 0.1
      write (ltmp,'(2a,3(1x,f4.1),1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -av',xval,xval,xval
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('lr',xval)
      write (ltmp,'(2a,i2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -lr ',int(xval)
      write (rendopt,'(a)') ltmp
      xval=-1.
      call RADPAR('lw',xval)
      write (ltmp,'(2a,f5.3,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -lw ',xval
      write (rendopt,'(a)') ltmp

C Run findglare to identify glare sources.
      if (ipckvew.gt.0) then
        write(pf,'(4a)') picfil(1:lnblnk(picfil)),'_',
     &    rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'.hdr'
        write(gf,'(4a)') picfil(1:lnblnk(picfil)),'_',
     &    rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'.glr'
      else
        write(pf,'(2a)') picfil(1:lnblnk(picfil)),'_X.hdr'
        write(gf,'(2a)') picfil(1:lnblnk(picfil)),'_X.glr'
      endif
      write(doit,'(12a)') 'cd ',runpath(1:lnblnk(runpath)),
     &  '; findglare -v -ga 10-60:10 -p ',
     &   pf(1:lnblnk(pf)),' ',rendopt(1:lnblnk(rendopt)),' ',
     &   octfil(1:lnblnk(octfil)),' > ',gf(1:lnblnk(gf))

C Debug/trace.
      if(itc.eq.0)then
        continue
      else     
        write(icout,*) doit(1:lnblnk(doit))
      endif
      call runit(doit,'-')
      
C Now display glare sources on image.
      write(doit,'(7a)') 'cd ',runpath(1:lnrp),
     &  '; xglaresrc ',pf(1:lnblnk(pf)),' ',gf(1:lnblnk(gf)),' &'
      call runit(doit,'-')
      call usrmsg(
     &  'The glare sources are shown via circles & intensity values.',
     &  'Click when done. More glare info via view visu results','W')

C Use glarendx to generate Guth VCP data.
      write(doit,'(5a)') 'cd ',runpath(1:lnrp),
     &  '; glarendx -t guth_vcp ',gf(1:lnblnk(gf)),' &'
      call runit(doit,'-')
      call sleep(1)

C And re-run it dumping values to a file.
      write (doit,'(7a)') 'cd ',runpath(1:lnrp),
     &  '; glarendx -t guth_vcp ',gf(1:lnblnk(gf)),' > ',
     &  gf(1:lnblnk(gf)),'.vcp'
      call runit(doit,'-')
      call sleep(1)

      return
      end

C ********************** getdf **********************
C Invoke daylight factor calculation and then displays them.

      SUBROUTINE getdf
#include "building.h"
#include "e2r_common.h"
#include "espriou.h"
#include "help.h"
      
      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/FILEP/IFIL
      integer itc,icout  ! trace verbosity and output channel
      common/trace/itc,icout

      logical both_esp_wave      ! If true manage pair of scenes.
      logical focus_espg         ! If true editing in context of ESP-r geometry
      integer ipairedscene       ! Index of paired scene.
      COMMON/ESPWAVE/both_esp_wave,focus_espg,ipairedscene(16)

      CHARACTER df*72,outs*124,ltmp*72,doit*256
      character QT*1,OUTSTR*124
      character tfile*72,outsl*160
      character rifwave*72

      logical XST,RECALC,dowave

      helpinsub='radcmds'  ! set for subroutine

      ITA1 = IFIL+6
      ITA2 = IFIL+7
      QT=CHAR(39)

C Check for daylight factor points - if none then try and read grid file.
      if (NDFP.eq.0) then
        call ERPFREE(ITA1,ISTAT)
        write(tfile,'(a,a)')runpath(1:lnrp),
     &    LDFGRID(ISCENE)(1:lnblnk(LDFGRID(ISCENE)))
        write(currentfile,'(a)') tfile(1:lnblnk(tfile))
        call FPOPEN(ITA1,ISTAT,1,1,tfile)
        if(ISTAT.eq.0)then
  42      CALL STRIPC(ITA1,OUTSTR,0,ND,0,'grid points',IFER)
          if (IFER.eq.0) then
            NDFP=NDFP+1
            K=0
            CALL EGETWR(OUTSTR,K,XTMP,0.,0.,'-','grid x',IER)
            DFPTS(NDFP,1)=XTMP
            CALL EGETWR(OUTSTR,K,XTMP,0.,0.,'-','grid y',IER)
            DFPTS(NDFP,2)=XTMP
            CALL EGETWR(OUTSTR,K,XTMP,0.,0.,'-','grid z',IER)
            DFPTS(NDFP,3)=XTMP
            CALL EGETWR(OUTSTR,K,DFDIR(1),0.,0.,'-','vec x',IER)
            CALL EGETWR(OUTSTR,K,DFDIR(2),0.,0.,'-','vec y',IER)
            CALL EGETWR(OUTSTR,K,DFDIR(3),0.,0.,'-','vec z',IER)
            goto 42
          endif
          CALL DFGRIDLBL('g')
        endif
        call ERPFREE(ITA1,ISTAT)
      endif
      if (NDFP.eq.0) then
        CALL USRMSG('Could not read grid points file.',
     &     'Cannot simulate without this information.  Returning.','W')
        return
      endif

C Ask for convergence limit.
      if(CONV(ISCENE).lt.0.001) CONV(ISCENE)=0.25
      helptopic='e2r_df_convergence'
      call gethelptext(helpinsub,helptopic,nbhelp)
      CALL EASKR(CONV(ISCENE),'Daylight factor convergence criteria',
     &  'Max total diff between iterations) ',0.00,
     &  'F',5.0,'W',0.25,'df convergence criteria',IER,nbhelp)

C Reset DFVALS to 0.0
      do 5 I=1,500
        DFVALS(I)=0.0
 5    continue
      ICONV=0

C Check octree up to date.
      call touchrif(RIFNAME(ISCENE))

C If Wavefront variant needed setup rifwave file name.
      dowave=.false.
      if(focus_espg.and.both_esp_wave)then
        dowave=.true.
      elseif(.NOT.focus_espg.and.both_esp_wave)then
        dowave=.true.
      elseif(focus_espg.and.(.NOT.both_esp_wave))then
        dowave=.false.
      elseif(.NOT.focus_espg.and.(.NOT.both_esp_wave))then
        dowave=.false.
      endif
      
C Check rendering parameters.  If the number of ambient bounces is 
C greater than 1 then the simulation has probably been run before. 
C Ask if the parameters should be reset.
      XAB=-1.
      call RADPAR('ab',XAB)
      if (XAB.gt.1.5) then
        helptopic='e2r_confusion'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL EASKMBOX(
     &    'The rendering parameters would appear to be set',
     &    'from a previous simulation.  Do you want to: ','use them',
     &    'start from beginning (recommended)',
     &    ' ',' ',' ',' ',' ',' ',IWR,nbhelp)
        if (IWR.eq.2) then
          write (outsl,'(2a)') crenrp(1:lnblnk(crenrp)),
     &                         ' -ab 1 -ad 1024 -as 512 -av 0 0 0'
          write (crenrp,'(a)') outsl
          IFC=3
          call MKRIF(ITA2,RIFNAME(ISCENE),IFC,0,'-',IER)
          if(dowave)then
            if (SCENEPURP(ISCENE)(1:8).eq.'External')then
              rifwave='obj_ex.rif'
            elseif(SCENEPURP(ISCENE)(1:8).eq.'Internal')then
              rifwave='obj_in.rif'
            elseif(SCENEPURP(ISCENE)(1:5).eq.'Glare')then
              rifwave='obj_gl.rif'
            elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
              rifwave='obj_df.rif'
            else
              rifwave='obj.rif'
            endif
            call mkrif(ITA2,rifwave,IFC,0,'w',IER)
          endif
          call touchrif(RIFNAME(ISCENE))
        endif
      endif

C Daylight factors output file.
      write (df,'(2a)') SCENERT(ISCENE)(1:lnblnk(SCENERT(ISCENE))),'.df'
      call ERPFREE(ITA1,ISTAT)
 1111 write (ltmp,'(2a)') runpath(1:lnrp),df(1:lnblnk(df))
      call FINDFIL(ltmp,XST)
      helptopic='e2r_file_overwrite'
      call gethelptext(helpinsub,helptopic,nbhelp)
      if(XST)then
        write(outs,'(2a)')'There is an existing file ',df(1:lnblnk(df))
        CALL EASKMBOX(outs,'Do you want to: ','overwrite',
     &    'choose another name',' ',' ',' ',' ',' ',' ',IWR,0)
        if (IWR.eq.2) then
          ltmp = df
 1221     CALL EASKS(ltmp,
     &      'Daylight factors results file name (modify slightly)?',
     &      ' ',72,'xx.df','df resfile name',IER,nbhelp)
          if(ltmp.ne.'  '.and.ltmp.ne.df) then
            df = ltmp
          else
            goto 1221
          endif
          goto 1111
        else
          write(ltmp,'(2a)')runpath(1:lnrp),df(1:lnblnk(df))
          call FPOPEN(ITA1,ISTAT,1,1,ltmp)
          CALL EFDELET(ITA1,ISTAT)
        endif
      endif

C Open sky file and read ground ambient level.
      call RRSKY(ITA1,ga,IER)

C Call rtrace and calculate daylight factors. This is the point where
C the logic jumps back to if additional resolution is required.
 99   write(doit,'(4a,1x,6a,f5.2,4a)') 'cd ',runpath(1:lnrp),
     &  '; rtrace -opv -I -h @',optnfil(1:lnblnk(optnfil)),
     &  octfil(1:lnblnk(octfil)),
     &  ' < ',LDFGRID(ISCENE)(1:lnblnk(LDFGRID(ISCENE))),
     &  ' | rcalc -e ',QT,
     &  '$1=$1;$2=$2;$3=$3;$4=((0.265*$4+0.670*$5+0.065*$6)*100)/(PI*',
     &  ga,')',QT,' > ',df(1:lnblnk(df))

C Debug/trace.
      if(itc.eq.0)then
        continue
      else     
        write(icout,*) doit(1:lnblnk(doit))
      endif
      call runit(doit,'-')

C Now need to read in daylight factors and store in memory.
      DFmaxdiff=0.
      write (ltmp,'(2a)') runpath(1:lnrp),df(1:lnblnk(df))
      write(currentfile,'(a)') tfile(1:lnblnk(tfile))
      call FPOPEN(ITA1,ISTAT,1,1,ltmp)
      do 10 I=1,NDFP
        call STRIPC(ITA1,OUTSTR,4,ND,0,'DF results',IER)
        K=0
        call EGETWR(OUTSTR,K,X,0.0,0.0,'-','X coord',IER)
        call EGETWR(OUTSTR,K,Y,0.0,0.0,'-','Y coord',IER)
        call EGETWR(OUTSTR,K,Z,0.0,0.0,'-','Z coord',IER)
        call EGETWR(OUTSTR,K,DFval,0.0,100.0,'W','DF',IER)
        do 20 J=1,NDFP
          if (abs(DFPTS(J,1)-X).lt.0.001) then
            if (abs(DFPTS(J,2)-Y).lt.0.001) then
              if (abs(DFPTS(J,3)-Z).lt.0.001) then

C Found the DF for the current sensor.
                if (abs(DFVALS(J)-DFval).gt.DFmaxdiff) then
                  DFmaxdiff=abs(DFVALS(J)-DFval)
                endif
                DFVALS(J)=DFval
              endif
            endif
          endif
 20     continue
 10   continue

C Debug.
C      write(6,*) DFmaxdiff

C Display current results and ambient rendering parameters.
      call DFGRIDLBL('d')
      XAB=-1.0; XAD=-1.0; XAS=-1.
      call RADPAR('ab',XAB)
      call RADPAR('ad',XAD)
      call RADPAR('as',XAS)
      write (outs,'(a,i2,2(a,i4))') 'Ambient parameter settings: -ab ',
     &              nint(XAB),' -ad ',nint(XAD),' -as ',nint(XAS)
      call edisp(iuout,outs)

C Check for convergance.
      RECALC=.false.

C Check value of ad.
      if (ICONV.eq.0) then
        YAD=-1.0
        call RADPAR('ad',YAD)
        if (DFmaxdiff.gt.CONV(ISCENE)) then

C Increase value and recalculate.
          if (YAD.lt.4000.0) then
            YAD=YAD*2.
            call RADPAR('ad',YAD)
            RECALC=.true.
          else
            DFmaxdiff=9999.9
            ICONV=1
          endif
        else

C Decrease value (last change had no effect) and increase as.
          if (YAD.gt.1025.0) then
            YAD=YAD/2.
            call RADPAR('ad',YAD)
            DFmaxdiff=9999.9
            ICONV=1
          endif
        endif
      endif

C Check value of as.
      if (ICONV.eq.1) then
        YAS=-1.0
        call RADPAR('as',YAS)
        if (DFmaxdiff.gt.CONV(ISCENE)) then

C Increase value and recalculate.
          if (YAS.lt.((YAD/2.0)-1.0)) then
            YAS=YAS*2.
            call RADPAR('as',YAS)
            RECALC=.true.
          else
            DFmaxdiff=9999.9
            ICONV=2
          endif
        else

C Decrease value (last change had no effect) and increase ab.
          if (YAS.gt.513.0) then
            YAS=YAS/2.
            call RADPAR('as',YAS)
            DFmaxdiff=9999.9
            ICONV=2
          endif
        endif
      endif

C Check value of ab.
      if (ICONV.eq.2) then
        YAB=-1.0
        call RADPAR('ab',YAB)
        if (DFmaxdiff.gt.CONV(ISCENE)) then

C Increase value and recalculate.
          if (YAB.lt.5.0) then
            YAB=YAB+1.0
            call RADPAR('ab',YAB)
            RECALC=.true.
          else
            DFmaxdiff=9999.9
            ICONV=3
          endif
        else

C Decrease value (last change had no effect) and increase ??.
          if (YAB.gt.2.0) then
            YAB=YAB-1.0
            call RADPAR('ab',YAB)
            DFmaxdiff=9999.9
            ICONV=3
          endif
        endif
      endif

C Save RIF file and update options file.
      if (RECALC) then
        call edisp(iuout,'Recalculating with different resolution...')
        IFC=3
        call MKRIF(ITA2,RIFNAME(ISCENE),IFC,0,'-',IER)
        if(dowave)then
          if (SCENEPURP(ISCENE)(1:8).eq.'External')then
            rifwave='obj_ex.rif'
          elseif(SCENEPURP(ISCENE)(1:8).eq.'Internal')then
            rifwave='obj_in.rif'
          elseif(SCENEPURP(ISCENE)(1:5).eq.'Glare')then
            rifwave='obj_gl.rif'
          elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
            rifwave='obj_df.rif'
          else
            rifwave='obj.rif'
          endif
          call mkrif(ITA2,rifwave,IFC,0,'w',IER)
        endif
        call touchrif(RIFNAME(ISCENE))
        if (NBSRIF(ISCENE).eq.1) then
          call mkrif(ita2,LBSRIF(ISCENE),IFC,1,'-',ier)
          call touchrif(LBSRIF(ISCENE))
        endif

C Move results file and recalculate DFs. Note this move creates
C a catch-22 for viewing the DF later on.
        call ERPFREE(ITA1,ISTAT)
        write(doit,'(7a)') 'cd ',runpath(1:lnrp),'; mv ',
     &        df(1:lnblnk(df)),' ',df(1:lnblnk(df)),'-'
        call runit(doit,'-')
        goto 99
      endif
      call ERPFREE(ITA1,ISTAT)

      return
      end

C ********************** getfalse **********************
C Create a standard image via rpict, an illuminance
C image via rpict -i calculation and then a falsecolor command
C with Lux legend.

      SUBROUTINE getfalse(ier)
#include "building.h"
#include "e2r_common.h"
#include "help.h"
      
      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/FILEP/IFIL
      integer itc,icout  ! trace verbosity and output channel
      common/trace/itc,icout

      CHARACTER ltmp*96,doit*400
      character rendopt*200,buffer*200
      character ilfile*72,ulfile*72 ! for illuminance
      character ihfile*72,uhfile*72 ! for luminance
      character luxfile*72  ! for false color
      integer maxlux        ! max lux coutour to plot
      integer numcontours   ! number of contours

      helpinsub='radcmds'  ! set for subroutine
      ITA1 = IFIL+6

C Set file name including path and open.
      if(ipckvew.eq.0)then
        call edisp(iuout,'No view selected, returning.')
        return
      endif

C Setup help.
      helptopic='e2r_falsecolour'
      call gethelptext(helpinsub,helptopic,nbhelp)

      write(uhfile,'(4a)') picfil(1:lnblnk(picfil)),'_',
     &  rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'.unf'
      write(ihfile,'(4a)') picfil(1:lnblnk(picfil)),'_',
     &  rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'.hdr'
      write(ulfile,'(4a)') picfil(1:lnblnk(picfil)),'_',
     &  rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'_i.unf'
      write(ilfile,'(4a)') picfil(1:lnblnk(picfil)),'_',
     &  rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'_i.hdr'
      write(luxfile,'(4a)') picfil(1:lnblnk(picfil)),'_',
     &  rvewsh(ipckvew)(1:lnblnk(rvewsh(ipckvew))),'_lux.hdr'

C Ask max lux and how many steps e.g. 30 and 15 results in
C 2 lux intervals.
      maxlux=200
      CALL EASKI(maxlux,'Lux range from zero to X. Confirm the',
     &  'maximum Lux level contour:',
     &   1,'F',3000,'W',20,'max lux range',IER,nbhelp) 
      numcontours=20
      CALL EASKI(numcontours,'Number of contours to plot. (if max 20',
     &  'and 10 contours then 2 lux increment:',
     &   1,'F',300,'W',10,'lux contours',IER,nbhelp) 

C Check octree up to date.
      call edisp(iuout,'Radiance feedback is displayed in xterm.')
      call touchrif(RIFNAME(ISCENE))

C Update rendering parameters from opt file (as done for glare).
C << Make this into a subroutine? >>
      xval=-1.
      call RADPAR('dp',xval)
      write (buffer,'(a,i4)') '-dp ',int(xval)
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('ds',xval)
      write (buffer,'(2a,f4.2,1x)') 
     &       rendopt(1:lnblnk(rendopt)),' -ds ',xval
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('dj',xval)
      if (xval.gt.0.) then
        write (buffer,'(2a,f4.2,1x)') 
     &    rendopt(1:lnblnk(rendopt)),' -dj ',xval
        write (rendopt,'(a)') buffer
      endif
      xval=-1.
      call RADPAR('dt',xval)
      write (buffer,'(2a,f4.2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -dt ',xval
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('dc',xval)
      write (buffer,'(2a,f4.2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -dc ',xval
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('dr',xval)
      write (buffer,'(2a,i2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -dr ',int(xval)
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('st',xval)
      write (buffer,'(2a,f4.2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -st ',xval
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('ab',xval)
      write (buffer,'(2a,i2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -ab ',int(xval)
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('aa',xval)
      write (buffer,'(2a,f4.2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -aa ',xval
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('ad',xval)
      write (buffer,'(2a,i4,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -ad ',int(xval)
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('as',xval)
      write (buffer,'(2a,i4,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -as ',int(xval)
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('ar',xval)
      write (buffer,'(2a,i4,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -ar ',int(xval)
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('av',xval)
      write (buffer,'(2a,3(1x,f4.1),1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -av',xval,xval,xval
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('lr',xval)
      write (buffer,'(2a,i2,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -lr ',int(xval)
      write (rendopt,'(a)') buffer
      xval=-1.
      call RADPAR('lw',xval)
      write (buffer,'(2a,f5.3,1x)') 
     &  rendopt(1:lnblnk(rendopt)),' -lw ',xval
      write (rendopt,'(a)') buffer

C Debug.
C      write(6,*) 'render options ',lnblnk(rendopt)
C      write(6,*) rendopt(1:lnblnk(rendopt))
C      write(6,*) vewcmds(ipckvew)

C Compose standard image command (this is typically what the
C rad -e xx.rif would have created). If pfilt is needed un-comment
C the next section and change ihfile to uhfile.
      write(doit,'(5a,i5,a,i5,6a)') 'cd ',runpath(1:lnrp),
     &  '; rpict -t 60 ',
     &  vewcmds(ipckvew)(1:lnblnk(vewcmds(ipckvew))),' -x',ipicx,
     &  ' -y',ipicx,' -ps 3 -pt 0.08 ',rendopt(1:lnblnk(rendopt)),
     &  ' ',octfil(1:lnblnk(octfil)),' > ',ihfile(1:lnblnk(ihfile))

C Debug/trace.
      if(itc.eq.0)then
        continue
      else     
        write(icout,*) doit(1:lnblnk(doit))
      endif
      call edisp(iuout,'Generating standard picture...')
      call runit(doit,'-')

C Compose pfilt command as rad -e xx.rif would have issued.
C This halves the size of the image - for what reason?
C      write(doit,'(6a)') 'cd ',runpath(1:lnrp),
C     &  '; pfilt -r .6 -x /2 -y /2 ',uhfile(1:lnblnk(uhfile)),' > ',
C     &  ihfile(1:lnblnk(ihfile))
C      if(itc.eq.0)then
C        continue
C      else     
C        write(icout,*) doit(1:lnblnk(doit))
C      endif
C      call edisp(iuout,'Filtering standard picture...')
C      call runit(doit,'-')
       
C Compose illuminance command as above except with added -i.
      write(doit,'(5a,i5,a,i5,6a)') 'cd ',runpath(1:lnrp),
     &  '; rpict -i -t 60 ',
     &  vewcmds(ipckvew)(1:lnblnk(vewcmds(ipckvew))),' -x',ipicx,
     &  ' -y',ipicx,' -ps 3 -pt 0.08 ',rendopt(1:lnblnk(rendopt)),
     &  ' ',octfil(1:lnblnk(octfil)),' > ',ilfile(1:lnblnk(ilfile))
      if(itc.eq.0)then
        continue
      else     
        write(icout,*) doit(1:lnblnk(doit))
      endif
      call edisp(iuout,'Generating illuminance picture...')
      call runit(doit,'-')

C Compose pfilt command (assumes medium quality).
C      write(doit,'(6a)') 'cd ',runpath(1:lnrp),
C     &  '; pfilt -r .6 -x /2 -y /2 ',ulfile(1:lnblnk(ulfile)),' > ',
C     &  ilfile(1:lnblnk(ilfile))
C      if(itc.eq.0)then
C        continue
C      else     
C        write(icout,*) doit(1:lnblnk(doit))
C      endif
C      call edisp(iuout,'Filtering illuminance picture...')
C      call runit(doit,'-')
      
C Compose falsecolor command.
      write(doit,'(7a,i5,a,i5,2a)') 'cd ',runpath(1:lnrp),
     &  '; falsecolor -i ',ilfile(1:lnblnk(ilfile)),' -p ',
     &  ihfile(1:lnblnk(ihfile)),' -cl -s ',maxlux,' -n ',
     &  numcontours,' -l Lux > ',luxfile(1:lnblnk(luxfile))
      if(itc.eq.0)then
        continue
      else     
        write(icout,*) doit(1:lnblnk(doit))
      endif
      call edisp(iuout,'Creating false color picture...')
      call runit(doit,'-')

C Display the falsecolor image.
      call edisp(iuout,'View of false color picture...')
      write(doit,'(5a)') 'cd ',runpath(1:lnrp),
     &  '; ximage ',luxfile(1:lnblnk(luxfile)),' & '
      call runit(doit,'-')

C Remove the *.unf files (if the pfilt code is uncommented).
C      write(ltmp,'(2a)')runpath(1:lnrp),uhfile(1:lnblnk(uhfile))
C      call FPOPEN(ITA1,ISTAT,1,1,ltmp)
C      CALL EFDELET(ITA1,ISTAT)
C      write(ltmp,'(2a)')runpath(1:lnrp),ulfile(1:lnblnk(ulfile))
C      call FPOPEN(ITA1,ISTAT,1,1,ltmp)
C      CALL EFDELET(ITA1,ISTAT)

      return
      end

C ********************** getlux **********************
C Invoke illuminance calculation and then display using
C logic similar to getdf.

      SUBROUTINE getlux
#include "building.h"
#include "e2r_common.h"
#include "help.h"
      
      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/FILEP/IFIL
      integer itc,icout  ! trace verbosity and output channel
      common/trace/itc,icout

      logical both_esp_wave      ! If true manage pair of scenes.
      logical focus_espg         ! If true editing in context of ESP-r geometry
      integer ipairedscene       ! Index of paired scene.
      COMMON/ESPWAVE/both_esp_wave,focus_espg,ipairedscene(16)

      CHARACTER df*72,outs*124,ltmp*72,doit*256
      character QT*1,OUTSTR*124
      character tfile*72,outsl*160
      character rifwave*72

      logical XST,RECALC,dowave

      helpinsub='radcmds'  ! set for subroutine

      ITA1 = IFIL+6
      ITA2 = IFIL+7
      QT=CHAR(39)

C If Wavefront variant needed setup rifwave file name.
      dowave=.false.
      if(focus_espg.and.both_esp_wave)then
        dowave=.true.
      elseif(.NOT.focus_espg.and.both_esp_wave)then
        dowave=.true.
      elseif(focus_espg.and.(.NOT.both_esp_wave))then
        dowave=.false.
      elseif(.NOT.focus_espg.and.(.NOT.both_esp_wave))then
        dowave=.false.
      endif

C Check for illuminance points - if none then try and read grid file.
      if (NDFP.eq.0) then
        call ERPFREE(ITA1,ISTAT)
        write(tfile,'(a,a)')runpath(1:lnrp),
     &    LDFGRID(ISCENE)(1:lnblnk(LDFGRID(ISCENE)))
        call FPOPEN(ITA1,ISTAT,1,1,tfile)
        if(ISTAT.eq.0)then
  42      CALL STRIPC(ITA1,OUTSTR,0,ND,0,'grid points',IFER)
          if (IFER.eq.0) then
            NDFP=NDFP+1
            K=0
            CALL EGETWR(OUTSTR,K,XTMP,0.,0.,'-','grid x',IER)
            DFPTS(NDFP,1)=XTMP
            CALL EGETWR(OUTSTR,K,XTMP,0.,0.,'-','grid y',IER)
            DFPTS(NDFP,2)=XTMP
            CALL EGETWR(OUTSTR,K,XTMP,0.,0.,'-','grid z',IER)
            DFPTS(NDFP,3)=XTMP
            CALL EGETWR(OUTSTR,K,DFDIR(1),0.,0.,'-','vec x',IER)
            CALL EGETWR(OUTSTR,K,DFDIR(2),0.,0.,'-','vec y',IER)
            CALL EGETWR(OUTSTR,K,DFDIR(3),0.,0.,'-','vec z',IER)
            goto 42
          endif
          CALL DFGRIDLBL('g')
        endif
        call ERPFREE(ITA1,ISTAT)
      endif
      if (NDFP.eq.0) then
        CALL USRMSG('Could not read grid points file.',
     &     'Cannot simulate without this information.  Returning.','W')
        return
      endif

C Ask for convergence limit.
      if(CONV(ISCENE).lt.0.01) CONV(ISCENE)=0.25
      helptopic='e2r_lux_convergence'
      call gethelptext(helpinsub,helptopic,nbhelp)
      CALL EASKR(CONV(ISCENE),'illuminance convergence criteria',
     &  'Max total diff between iterations) ',1.00,
     &  'F',100.0,'W',10.0,'lux convergence criteria',IER,nbhelp)

C Reset DFVALS (holding lux values) to 0.0
      do 5 I=1,500
        DFVALS(I)=0.0
 5    continue
      ICONV=0

C Check octree up to date.
      call edisp(iuout,'Radiance feedback is displayed in xterm.')
      call touchrif(RIFNAME(ISCENE))
      
C Check rendering parameters.  If the number of ambient bounces is 
C greater than 1 then the simulation has probably been run before. 
C Ask if the parameters should be reset.
      XAB=-1.
      call RADPAR('ab',XAB)
      if (XAB.gt.1.5) then
        helptopic='e2r_confusion'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL EASKMBOX(
     &    'The rendering parameters would appear to be set',
     &    'from a previous simulation.  Do you want to: ','use them',
     &    'start from beginning (recommended)',
     &    ' ',' ',' ',' ',' ',' ',IWR,nbhelp)
        if (IWR.eq.2) then
          write (outsl,'(2a)') crenrp(1:lnblnk(crenrp)),
     &                         ' -ab 1 -ad 1024 -as 512 -av 0 0 0'
          write (crenrp,'(a)') outsl
          IFC=3
          call MKRIF(ITA2,RIFNAME(ISCENE),IFC,0,'-',IER)
          if(dowave)then
            if (SCENEPURP(ISCENE)(1:8).eq.'External')then
              rifwave='obj_ex.rif'
            elseif(SCENEPURP(ISCENE)(1:8).eq.'Internal')then
              rifwave='obj_in.rif'
            elseif(SCENEPURP(ISCENE)(1:5).eq.'Glare')then
              rifwave='obj_gl.rif'
            elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
              rifwave='obj_df.rif'
            else
              rifwave='obj.rif'
            endif
            call mkrif(ITA2,rifwave,IFC,0,'w',IER)
          endif
          call touchrif(RIFNAME(ISCENE))
        endif
      endif

C Illuminance output file.
      write (df,'(2a)') SCENERT(ISCENE)(1:lnblnk(SCENERT(ISCENE))),
     &  '.lux'
      call ERPFREE(ITA1,ISTAT)
 1111 write (ltmp,'(2a)') runpath(1:lnrp),df(1:lnblnk(df))
      call FINDFIL(ltmp,XST)
      helptopic='e2r_file_overwrite'
      call gethelptext(helpinsub,helptopic,nbhelp)
      if(XST)then
        write(outs,'(2a)')'There is an existing file ',df(1:lnblnk(df))
        CALL EASKMBOX(outs,'Do you want to: ','overwrite',
     &    'choose another name',' ',' ',' ',' ',' ',' ',IWR,0)
        if (IWR.eq.2) then
          ltmp = df
 1221     CALL EASKS(ltmp,
     &      'Illuminance results file name (modify slightly)?',
     &      ' ',72,'xx.lux','lux resfile name',IER,nbhelp)
          if(ltmp.ne.'  '.and.ltmp.ne.df) then
            df = ltmp
          else
            goto 1221
          endif
          goto 1111
        else
          write(ltmp,'(2a)')runpath(1:lnrp),df(1:lnblnk(df))
          call FPOPEN(ITA1,ISTAT,1,1,ltmp)
          CALL EFDELET(ITA1,ISTAT)
        endif
      endif

C Open sky file and read ground ambient level.
      call RRSKY(ITA1,ga,IER)

C Call rtrace and calculate illuminance. This is the point where
C the logic jumps back to if additional resolution is required.
 99   write(doit,'(4a,1x,9a)') 'cd ',runpath(1:lnrp),
     &  '; rtrace -opv -I -h @',optnfil(1:lnblnk(optnfil)),
     &  octfil(1:lnblnk(octfil)),
     &  ' < ',LDFGRID(ISCENE)(1:lnblnk(LDFGRID(ISCENE))),
     &  ' | rcalc -e ',QT,
     &  '$1=$1;$2=$2;$3=$3;$4=179*(0.265*$4+0.670*$5+0.065*$6)',
     &  QT,' > ',df(1:lnblnk(df))

C Debug/trace.
      if(itc.eq.0)then
        continue
      else     
        write(icout,*) doit(1:lnblnk(doit))
      endif
      call runit(doit,'-')

C Now need to read in lux and store in memory.
      DFmaxdiff=0.
      write (ltmp,'(2a)') runpath(1:lnrp),df(1:lnblnk(df))
      call FPOPEN(ITA1,ISTAT,1,1,ltmp)
      do 10 I=1,NDFP
        call STRIPC(ITA1,OUTSTR,4,ND,0,'lux results',IER)
        K=0
        call EGETWR(OUTSTR,K,X,0.0,0.0,'-','X coord',IER)
        call EGETWR(OUTSTR,K,Y,0.0,0.0,'-','Y coord',IER)
        call EGETWR(OUTSTR,K,Z,0.0,0.0,'-','Z coord',IER)
        call EGETWR(OUTSTR,K,DFval,0.0,20000.0,'W','lux',IER)
        do 20 J=1,NDFP
          if (abs(DFPTS(J,1)-X).lt.0.001) then
            if (abs(DFPTS(J,2)-Y).lt.0.001) then
              if (abs(DFPTS(J,3)-Z).lt.0.001) then

C Found the DF for the current sensor.
                if (abs(DFVALS(J)-DFval).gt.DFmaxdiff) then
                  DFmaxdiff=abs(DFVALS(J)-DFval)
                endif
                DFVALS(J)=DFval
              endif
            endif
          endif
 20     continue
 10   continue

C Debug.
      write(6,*) 'DF maximum difference is ',DFmaxdiff

C Display current results and ambient rendering parameters.
      call DFGRIDLBL('l')
      XAB=-1.0; XAD=-1.0; XAS=-1.
      call RADPAR('ab',XAB)
      call RADPAR('ad',XAD)
      call RADPAR('as',XAS)
      write (outs,'(a,i2,2(a,i4))') 'Ambient parameter settings: -ab ',
     &              nint(XAB),' -ad ',nint(XAD),' -as ',nint(XAS)
      call edisp(iuout,outs)

C Check for convergance.
      RECALC=.false.

C Check value of ad.
      if (ICONV.eq.0) then
        YAD=-1.0
        call RADPAR('ad',YAD)
        if (DFmaxdiff.gt.CONV(ISCENE)) then

C Increase value and recalculate.
          if (YAD.lt.4000.0) then
            YAD=YAD*2.
            call RADPAR('ad',YAD)
            RECALC=.true.
          else
            DFmaxdiff=9999.9
            ICONV=1
          endif
        else

C Decrease value (last change had no effect) and increase as.
          if (YAD.gt.1025.0) then
            YAD=YAD/2.
            call RADPAR('ad',YAD)
            DFmaxdiff=9999.9
            ICONV=1
          endif
        endif
      endif

C Check value of as.
      if (ICONV.eq.1) then
        YAS=-1.0
        call RADPAR('as',YAS)
        if (DFmaxdiff.gt.CONV(ISCENE)) then

C Increase value and recalculate.
          if (YAS.lt.((YAD/2.0)-1.0)) then
            YAS=YAS*2.
            call RADPAR('as',YAS)
            RECALC=.true.
          else
            DFmaxdiff=9999.9
            ICONV=2
          endif
        else

C Decrease value (last change had no effect) and increase ab.
          if (YAS.gt.513.0) then
            YAS=YAS/2.
            call RADPAR('as',YAS)
            DFmaxdiff=9999.9
            ICONV=2
          endif
        endif
      endif

C Check value of ab.
      if (ICONV.eq.2) then
        YAB=-1.0
        call RADPAR('ab',YAB)
        if (DFmaxdiff.gt.CONV(ISCENE)) then

C Increase value and recalculate.
          if (YAB.lt.5.0) then
            YAB=YAB+1.0
            call RADPAR('ab',YAB)
            RECALC=.true.
          else
            DFmaxdiff=9999.9
            ICONV=3
          endif
        else

C Decrease value (last change had no effect) and increase ??.
          if (YAB.gt.2.0) then
            YAB=YAB-1.0
            call RADPAR('ab',YAB)
            DFmaxdiff=9999.9
            ICONV=3
          endif
        endif
      endif

C Save RIF file and update options file.
      if (RECALC) then
        call edisp(iuout,'Recalculating with different resolution...')
        call edisp(iuout,'feedback is displayed in xterm.')
        IFC=3
        call MKRIF(ITA2,RIFNAME(ISCENE),IFC,0,'-',IER)
        if(dowave)then
          if (SCENEPURP(ISCENE)(1:8).eq.'External')then
            rifwave='obj_ex.rif'
          elseif(SCENEPURP(ISCENE)(1:8).eq.'Internal')then
            rifwave='obj_in.rif'
          elseif(SCENEPURP(ISCENE)(1:5).eq.'Glare')then
            rifwave='obj_gl.rif'
          elseif (SCENEPURP(ISCENE)(1:8).eq.'Day_fact') then
            rifwave='obj_df.rif'
          else
            rifwave='obj.rif'
          endif
          call mkrif(ITA2,rifwave,IFC,0,'w',IER)
        endif
        call touchrif(RIFNAME(ISCENE))
        if (NBSRIF(ISCENE).eq.1) then
          call mkrif(ita2,LBSRIF(ISCENE),IFC,1,'-',ier)
          call touchrif(LBSRIF(ISCENE))
        endif

C Move results file and recalculate lux. Note this move creates
C a catch-22 for viewing the lux later on.
        call ERPFREE(ITA1,ISTAT)
        write(doit,'(7a)') 'cd ',runpath(1:lnrp),'; mv ',
     &        df(1:lnblnk(df)),' ',df(1:lnblnk(df)),'-'
        call runit(doit,'-')
        goto 99
      endif
      call ERPFREE(ITA1,ISTAT)

      return
      end
