C This file is part of the ESP-r system.
C Copyright Energy Systems Research Unit, University of
C Strathclyde, Glasgow Scotland, 2001-.

C ESP-r is free software.  You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation 
C (version 2 orlater).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.


C This file contains the following routines:
C  SIMCON - Executes the required simulations.
C  MZSIML - Sets the final simulation parameters prior to simulation. 
C  SIMTOG - Set simulation toggles
C  MSTTOG - Set moisture flow toggles.
C  STMTOG - Set building construction structured mesh toggles.
C  CFDTOG - Set CFD coupling toggles.
C  MZSPER - Definition of simulation period.
C  MZSIMH - Lists the current simulator parameters (dates, save option etc.)
C  MZTRAC - Define quantities to be traced.
C  MZINFO - Displays values of the currently set simulation parameters.


C ******************** SIMCON ********************
C Sets the final parameters before executing a simulation. MZNUMA is 
C called the required number of times, saving results after each
C simulation. 

      subroutine SIMCON
      
      USE AIM2_InputData, ONLY:iAIM2
      USE AIM2, ONLY: AIM2_SETVARS
      
      IMPLICIT NONE

C      USE H3KSTORE_FC_DATA,ONLY: iCurrent_FCT_OUTPUT_FILE
#include "building.h"
#include "geometry.h"
#include "uncertainty.h"
#include "hvac_parameters.h"
#include "h3kstore_parameters.h"
#include "h3kstore_common.h"
#include "SOFC.h"
#include "UserSimulationToggles.h"
#include "plant.h"
#include "FMI.h"
#include "agent.h"
#include "gremlin.h"
#include "help.h"

      common/filep/ifil
      INTEGER :: ifil
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER :: iuout,iuin,ieout

      COMMON/PBYDAT/PBYFIL,NSCVP,IPS
      INTEGER :: NSCVP,IPS

      COMMON/PERS/ISD1,ISM1,ISD2,ISM2,ISDS,ISDF,NTSTEP
      INTEGER :: ISD1,ISM1,ISD2,ISM2,ISDS,ISDF,NTSTEP

      COMMON/SOLOFF/ISOLAR
      INTEGER ISOLAR
      
      COMMON/GR1D04/GAM,RGAM
      REAL GAM, RGAM

C Plant matrix solver convergence criteria.      
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

      integer MAXITP, itrclp, ICSV
      real PERREL, PERTMP, PERFLX, PERMFL, CSVI
     
C Tolerance for iteration of hydrogen flow matrix.
      common/PITER_H2/fH2_iteration_tolerance
      real fH2_iteration_tolerance

      COMMON/SAVE/ISAVE
      INTEGER :: isave
      COMMON/SAVE1/LSAVE
      INTEGER :: lsave

      COMMON/C6/INDCFG
      INTEGER :: indcfg

C Uncertainty.
      COMMON/UA5/IMET,ISIM,NRSIM
      INTEGER :: imet,isim,nrsim

C Mechanical ventilation system activator.
      Integer iMech
      Character *72 LMech
      Common /mechvdef/ iMech, LMech

C Simulator parameter.
      COMMON/SPFL/spfileok,perok,cfdperok,tstepok,saveok,autook,exitok,
     &  startupok
      LOGICAL spfileok,perok,cfdperok,tstepok,saveok,autook,exitok,
     &  startupok

C Toggle for extreme silent running (endless) mode.
      logical endless,issilient
      common/endlessmode/endless,issilient

      logical PBYFIL,OK

      character outs*124
      
      INTEGER :: i,ier,irep,isimn1,isimn2,istat,iualf,iunit
      INTEGER :: nstep
      REAL :: tstep
      integer J

      helpinsub='simcon'  ! set for cfiles

      OK=.FALSE.

C Default values for IMET and NRSIM. 
      IMET=0
      NRSIM=1

C Set simulation control.
      IER=0
      call MZCSTR(IER)
      if (IER.eq.1) then
C If simulator is in silent running mode, set flag to terminate run      
         if ( autook ) exitok = .true.
         return
      endif

C Read functional mock-up interface specification file.
      if (is_FMU) then
        IUNIT=IFIL+1
        CALL ERPFREE(IUNIT,IER)
        CALL EFOPSEQ(IUNIT,FMISFL,1,IER)
        call FMI_RD(IUNIT,IER)
        CALL ERPFREE(IUNIT,IER)
        if (IER.eq.1) then
C If simulator is in silent running mode, set flag to terminate run
           if ( autook ) exitok = .true.
           return
        endif  
      endif
      
C Read agent specification file and open log file.
      if (IS_AGT) then
        IUNIT=IFIL+1
        CALL ERPFREE(IUNIT,IER)
        CALL EFOPSEQ(IUNIT,AGTFIL,1,IER)
        call AGT_RD(IUNIT,IER)
        CALL ERPFREE(IUNIT,IER)
        if (IER.eq.1) then
          if ( autook ) exitok = .true.
          return
        endif
        call AGT_INITLOG(IER)
        if (IER.ne.0) then
          if ( autook ) exitok = .true.
          return
        endif
      endif
      
C Read gremlin specification file and open log file.
      if (IS_GRM) then
        IUNIT=IFIL+1
        CALL ERPFREE(IUNIT,IER)
        CALL EFOPSEQ(IUNIT,GRMFIL,1,IER)
        call GRM_RD(IUNIT,IER)
        CALL ERPFREE(IUNIT,IER)
        if (IER.eq.1) then
          if ( autook ) exitok = .true.
          return
        endif
        call GRM_INITLOG(IER)
        if (IER.ne.0) then
          if ( autook ) exitok = .true.
          return
        endif
      endif

C Calculate the BASESIMP heat loss factors.
      CALL BSFCCTL

C Calculate the time-invariant variables for the AIM-2 infiltration model.
      IF( iAIM2==1 ) THEN
        CALL AIM2_SETVARS
      ENDIF

c Initialisation for Mechanical ventilation systems
      If ( iMech .eq. 1 ) Call MECH_VENT_INIT ( )

C Are uncertainties defined? If senrns returns with ier=2
C then user has selected just to run the base case.
      if (LUALF(1:7).ne.'UNKNOWN') then
        IUALF=IFIL+1
        call readual(IUALF)
        call SENRNS(IER)
        if (IER.eq.1) return
        if (IER.eq.2)then
          continue  ! run only the base case
        endif
      endif

C Reimpose any user-specified simulation toggles as needed.
      if ( bUserGAM ) GAM = fUserGAM        ! equation implicitness
      if ( bUserSolar ) ISOLAR = iUserSolar ! solar processing   
      if ( bUserPltSolvConfig ) then   
        MAXITP = iUserMAXITP    
        itrclp = iUserITRCLP    
        PERREL = fUserPERREL    
        PERTMP = fUserPERTMP    
        PERFLX = fUserPERFLX    
        PERMFL = fUserPERMFL    
        fH2_iteration_tolerance  = fUserH2iterTol 
      endif  
      
C Run the necessary number of simulations.
      do 100, I=1,NRSIM 
        ISIM=I

C Write info to sensitivity trace file.
        if (NRSIM.gt.1) then
          write (36,*) '**'
          write (36,*) '** Edits for changes in set:',ISIM
          write (36,*) '**'
        endif

C Reset the data if ISIM > 1 and change the required data. 
        if (ISIM.gt.1) then

C Re-scan the model description files silently.
          IREP=1
          write (36,*) '** Edits for changes in set:',ISIM,NTSTEP
          call MZINPT(IREP,IER)
          TSTEP=1.0/float(NTSTEP)
          CALL MZCOE2(TSTEP)
          write (36,*) '** Edits for changes in set:',ISIM,NTSTEP
          call MZCSTR(IER)
          write (36,*) '** Edits for changes in set:',ISIM,NTSTEP
        endif
        
C MZNUMA is effectively a simulation clock.
        CALL MZNUMA

C Close ascii plant-side boundary data file if last simulation 
C otherwise rewind for use in next simulation.
        if(pbyfil)then
          if (ISIM.eq.NRSIM) then 
            iunit=ifil+9
            call erpfree(iunit,istat)
          else
            iunit=ifil+9
            rewind(iunit)
          endif
        endif
        LSAVE=ISAVE

C Confirm save with user - only if not in autoexec mode as
C defined in the simulator perameter file.
c Send Output to file if save level = 5
        IF (ISAVE.eq.5 .and.
     &        ( indcfg == 1 .or. indcfg == 3 ) ) then
          call h3kmzoutp()
        ENDIF

c Close FC timstep output if open.      
        IF (FC_timeStep_output == FC_timeStep_output_yes) then
          close(unit=iCurrent_FCT_OUTPUT_FILE)
        ENDIF

        if(autook) then
          ok=.true.
          exitok=.true.
        elseif (LUALF(1:7).NE.'UNKNOWN') then
          ok=.true.
        else
          helptopic='say_yes_to_complete'
          call gethelptext(helpinsub,helptopic,nbhelp)
          CALL EASKOK(' ','Save results?',OK,nbhelp)
        endif
        IF(.NOT.OK)then
          return
        endif
        IF(LSAVE.EQ.0)then

C Message to recover save option 0 data from 'OUTPUT'.
          if(endless.or.issilient)then

C << place holder for menu option o. >>
            continue
          else
            call edisp(iuout,'Save option 0 data cannot be saved.')
            call edisp(iuout,'Results recovered via menu command o.')
          endif
          return
        endif

C No results file for save level 5.
        if (LSAVE.EQ.5)return

        call edisp(iuout,' ')
        CALL MZSAVE(ISIMN1,ISIMN2)
        if(INDCFG.eq.1)then
          write(outs,'(A,I3)')'Saved building result-set',ISIMN1
          call edisp(iuout,outs)
        elseif(INDCFG.eq.2)then
          write(outs,'(A,I3)')'Saved plant result-set',ISIMN2
          call edisp(iuout,outs)
        elseif(INDCFG.eq.3)then
          write(outs,1870)ISIMN1
 1870     format('Saved building result-set',I3,' and')
          if(endless.or.issilient)then
            continue
          else
            call edisp(iuout,outs)
          endif
          write(outs,1871)ISIMN2
 1871     format('plant result-set',I3)
          if(endless.or.issilient)then
            continue
          else
            call edisp(iuout,outs)
          endif
        endif
        if (ISIM.le.1.or.ISIM.eq.NRSIM) then
          write(outs,1872)ISD1,ISM1,ISD2,ISM2
 1872     format(3x,'for period day',I3,' month',I3,
     &              ' to day',I3,' month',I3)
          if(endless.or.issilient)then
            continue
          else
            call edisp(iuout,outs)
          endif
        endif
 100  continue

      return
      end

C ******************** MZSIML ********************
C The main energy simulation controlling routine
C performing the following functions.

C On each entry a routine is called which assigns the user's
C solution library and transfers zone description
C information to library header records.

C On first entry only the simulation period is requested.
C This period is subsequently assumed in force unless
C explicitly changed.

C After each simulation request, a routine is called which
C allows definition of the simulation control parameters.

C The '>1SAVE' command MUST be chosen after each simulation
C if the simulation results are to be saved in the solution
C library for subsequent review.

C The '<1DEL' command deletes the last simulation result
C set saved in the solutions library.

C The '0SAVE' command does not write to the results library.
C Simulation results retrieved from 'OUTPUT'.
C The '1SAVE', '2SAVE' and '3SAVE' commands cause minimum,
C moderate and all simulation results to be transferred to
C the results library; where:

C     'minimum'  includes zone air temperatures and plant
C                parameters only.
C     'moderate' includes (in addition to the minimum
C                transfer) all other energy flows plus surface
C                temperatures.
C     'all'      includes (in addition to the minimum and
C                moderate transfers) internal fabric
C                temperatures.
C Save option 4 is a special save for surface energy balances.
C It requires the same disk space as save option 3 and uses
C space normally used by the inter surface node temperatures.

C The default is '2SAVE'.

C The 'INFO' command outputs the current simulation period
C and information on the saved result set.

      SUBROUTINE MZSIML
#include "building.h"
#include "model.h"
#include "geometry.h"
#include "plant.h"
#include "net_flow.h"
#include "monitor.h"
#include "cfd.h"
#include "UserSimulationToggles.h"
#include "help.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT

      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      COMMON/MFTRA/IMFTU

      COMMON/PERS/ISD1,ISM1,ISD2,ISM2,ISDS,ISDF,NTSTEP

      COMMON/SAVE/ISAVE
      COMMON/C6/INDCFG
      COMMON/BEFORM/IBEQNF
      COMMON/PCEQU/IMPEXP,RATIMP
      COMMON/SOLOFF/ISOLAR
      COMMON/SKYFLG/ISKYFL
      COMMON/PMSLV/IPMSLV

      COMMON/FFN/IFLWN,ICFFS(MPCON)
      COMMON/MFSINC/MFBSNC
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

C Tolerance for iteration of hydrogen flow matrix.
      common/PITER_H2/fH2_iteration_tolerance
      real fH2_iteration_tolerance

      COMMON/TS/ITSFLG,NTSCPD,ITSSCH(MSCH,5),itsmus
      COMMON/CLMFLG/ICLMFL
      COMMON/VERSOL/ipkzon(mzs),ipksf(mzs),ivsolfil,vsol,VTSOL,vsnam
      COMMON/PWRFLG/IPWRSIM
      common /bmontr/mnzn,mzone(mcom),montrb,yminb(2),ymaxb(2),
     &               ibplot,irecpb(MDPH),mbstv
      common /montrv/mnpc,mcomp(mpcom),mcnod(mpcom),mstate,monitr,
     &               ymin,ymax
      COMMON/SPFL/spfileok,perok,cfdperok,tstepok,saveok,autook,exitok,
     &  startupok
      LOGICAL     spfileok,perok,cfdperok,tstepok,saveok,autook,exitok,
     &  startupok
      
      integer ncomp,ncon
      COMMON/C1/NCOMP,NCON
      common/ndcfd/ncfdnd,icfdnd(MNZ),NCONF

      LOGICAL VTSOL,SAVE4,SAVE4OK

      COMMON/zfluid/znotair(mcom),zfldK,zfldD,zfldC,zfldA,
     &              zSWAp(mcom),zSWAf(mcom)
      real zfldK,zfldD,zfldC,zfldA,zSWAp,zSWAf

C Toggle for extreme silent running (endless) mode.
      logical endless,issilient
      common/endlessmode/endless,issilient

      LOGICAL znotair

      CHARACTER*72 vsnam
      CHARACTER*29 ITEM(18)
      character outs*124
      integer NITMS,INO  ! max items and current menu item
      integer J

      helpinsub='simcon'  ! set for cfiles

C Intialise timestep controller flag. By default, timestep controller
C is disabled. Also initialise number of TSC operations to zero.
C Initialise power simulation flag.
      ITSFLG=0
      NTSCPD=0

C Intialise weather data interpolation flag. By default, interpolation
C is enabled.
      ICLMFL=0

C Intialise default value of zone(s) temperature monitor switch.
      montrb=0
      ibplot=0

C Intialise default value of plant state variable monitor switch.
      monitr=0

C Default for zone nodal equation generation is implicit or mixed,
C based on time-step evaluation.
      IBEQNF=0

C Default for plant equation generation is implicit or mixed,
C based on time-step evaluation; if mixed then impl/expl = 0.5
      IMPEXP=3
      RATIMP=.5

C Default solar treatment, solar ON, Perez model.
      ISOLAR=0
      iUserSolar = iSolar
      ISKYFL=3

C Default mass flow simulation iteration parameters are set in MFLWSU
C Default fluid (air) flow simulation is once per zone timestep
C (MFBSNC=1). This is over-ruled when plant fluid flow simulation
C is active, in which case fluid flow simulation will be performed
C every plant time-step.
      MFBSNC=1
      if(IFLWN.eq.1)MFBSNC=IFLWN

C Initialise plant side matrix solver. By default, the Gaussian approach.
      IPMSLV=1

C Initialise user-defined plant side convergence params to default values .  
      UsrPltConfig: if ( bUserPltSolvConfig ) then 
      
C Use specified plant solution parameters.
        MAXITP  = iUserMAXITP   
        ITRCLP  = iUserITRCLP   
        PERREL  = fUserPERREL   
        PERTMP  = fUserPERTMP   
        PERFLX  = fUserPERFLX   
        PERMFL  = fUserPERMFL   
        fH2_iteration_tolerance = fUserH2iterTol
      
      else 
      
C Save computed values for user report. 
      endif UsrPltConfig

C Initialise reading of vertical solar data.
      VTSOL=.false.

C Couple zone/plant & mass flow calculation trace output channels.
      IMFTU=ITU

C Assign user-specified library.
      CALL MZLIBF

C Set periods if not already done.
      IF ( .not. perok ) THEN
        CALL MZSPER
      ENDIF
      call askaboutcfd
      
C Display menu of alternatives.
      if(.not.saveok)ISAVE=2

C Set up the menu text.
   92 CONTINUE
      INO=-4

      ITEM(1)=   'a results library            '
      ITEM(2)=   'b simulation period          '
      ITEM(3)=   '  ....................       '
      if(ISAVE.eq.0)then
        ITEM(4)= '* save >> 0 (summary table)  '
      elseif(ISAVE.eq.1)then
        ITEM(4)= '* save >> 1 (temps & flux)   '
      elseif(ISAVE.eq.2)then
        ITEM(4)= '* Save >> 2 (1+ surf temps)  '
      elseif(ISAVE.eq.3)then
        ITEM(4)= '* save >> 3 (2+ node temps)  '
      elseif(ISAVE.eq.4)then
        ITEM(4)= '* save >> 4 (2+ energy bal)  '
c save level 5 
      elseif(ISAVE.eq.5)then
        ITEM(4)= '* save >> 5 (H3K no res file)'
      elseif(ISAVE.eq.6)then
        ITEM(4)= '* save >> 6 (csv summary)    '
      endif
      ITEM(5)=   '  ....................       '
      if(ISAVE.eq.0)then
        ITEM(6)= '  monitor (not available)    '
      else
        ITEM(6)= 'm monitor progress           '
      endif
      ITEM(7)=   's commence simulation        '
      ITEM(8)=   't time-step control          '
      ITEM(9)=   '                             '
      ITEM(10)=  '< delete last result set     '
      ITEM(11)=  '  ....................       '
      ITEM(12)=  'g simulation options         '
      ITEM(13)=  'i view simulation parameters '
      ITEM(14)=  'o view save level 0 result   '
      ITEM(15)=  '  ....................       '
      ITEM(16)=  '? help                       '
      ITEM(17)=  '- exit menu                  '
      NITMS=17

C Help text for this menu.
  91  helptopic='main_sim_menu'
      call gethelptext(helpinsub,helptopic,nbhelp)

C Do not display the menu if in autook mode as defined in the 
C simulator parameter file.
      if (autook) then
        if (exitok) then
          ino=NITMS
        else
          ino=7   ! sensitive to menu layout
        endif
      else
        CALL EMENU('Simulation control',ITEM,NITMS,INO)
      endif

C Results database name.
      if(ino.eq.1)then
        CALL MZLIBF

C Simulation period.
      elseif(ino.eq.2)then
        CALL MZSPER

C Results save level.
      elseif(ino.eq.4)then
         ISAVE=ISAVE+1
         if(ISAVE.gt.6)ISAVE=0  ! Reset to 0 if toggle >6.

C Graphic monitor setup.
      elseif(ino.eq.6)then
        if(indcfg.eq.1) then
          montrb=1
          call minptb
        elseif(indcfg.eq.2) then
          monitr=1
          call minput
        elseif(indcfg.eq.3) then
          montrb=0
          monitr=0
          CALL EASKMBOX(' ','Progress monitor options:',
     &      'temperature','heat flux',
     &      ' ',' ',' ',' ',' ',' ',IDIM,nbhelp)
          if(idim.eq.1)then
            montrb=1
            call minptb
          elseif(idim.eq.2) then
            monitr=1
            call minput
          endif
        endif

C Commence simulation.
      elseif(ino.eq.7)then

C If a surface convection control file exists for any zone then
C force a level 4 save.
        SAVE4=.false.
        do 42 ICOMP=1,NCOMP
          if(IHC(ICOMP).EQ.1.and.ISAVE.lt.4)SAVE4=.true.
   42   continue
        if (.not.autook) then
          if(SAVE4.and.ISAVE.ne.4)then
            helptopic='includes_non_standard'
            call gethelptext(helpinsub,helptopic,nbhelp)
            CALL EASKOK('Zone convection file detected.',
     &                  'Change save level to 4?',SAVE4OK,nbhelp)
           if(SAVE4OK)ISAVE=4
          endif
        endif

C If CFD is active in any zone then force a save level 4.
        do 43 ICOMP=1,NCOMP
          if(NCONF.gt.0) ISAVE=4
   43   continue
        call SIMCON
        if(isave.eq.6)then
          call MZOUT6  ! Output from 6 save
        elseif(isave.eq.0)then
          if(endless.or.issilient)then
            continue
          else
            call edisp(iuout,' ')
            call edisp(iuout,'Get save level 0 output via option `o`.')
          endif
        endif

C Timestep control.
      elseif(ino.eq.8)then
        CALL TSCINP

C Delete last result set.
      elseif(ino.eq.10)then
        IB=0
        IP=0
        IF(INDCFG.EQ.1)IB=1
        IF(INDCFG.EQ.2)IP=1
        if(INDCFG.EQ.3)then
          CALL EASKMBOX(' ','Delete last result set for:',
     &      'building','plant',' ',' ',' ',' ',' ',' ',IW,0)
          IBP=IW
          if(IBP.eq.1)IB=1
          if(IBP.eq.2)IP=1
        endif

        CALL MZDEL(IB,IP,ISIMLB,ISIMLP)
        IF(IP.EQ.1)then
          IF(ISIMLP.LT.0)GOTO 92
          write(outs,68)ISIMLP
   68     format('Number of plant result sets remaining = ',I3)
          call edisp(iuout,' ')
          call edisp(iuout,outs)
        endif
        IF(ISIMLB.LT.0)then

C Results file has zero result sets.
          call edisp(iuout,' ')
          call edisp(iuout,'No result sets in library!')
          goto 92
        endif
        write(outs,23)ISIMLB
   23   format('Number of building result sets remaining = ',I3)
        call edisp(iuout,' ')
        call edisp(iuout,outs)

C Simulation toggles.
      elseif(ino.eq.12)then
C        helptopic='sim_toggle_warning'
C        call gethelptext(helpinsub,helptopic,nbhelp)
C        CALL EASKI(IANS,' ',' Password: ',
C     &           0,'-',0,'-',0,'Password',IER,nbhelp)
C        IF(IANS.NE.101)GOTO 92
        call SIMTOG

C Simulation parameters.
      elseif(ino.eq.13)then
        CALL MZINFO(IUOUT)
 
C Output from save level 0.
      elseif(ino.eq.14)then
        CALL MZOUTP

C Help.
      elseif(ino.eq.16)then
        helptopic='main_sim_menu'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL PHELPD('sim control',nbhelp,'simul_cmds  ',0,0,IER)

C Exit.
      elseif(ino.eq.17)then
        return

C Unknown.
      else
        ino= -1
        goto 92
      endif
      goto 92

      END

C ******************** SIMTOG ********************
C Set simulation control toggles.

      subroutine simtog
#include "building.h"
#include "geometry.h"
#include "plant.h"
#include "power.h"
#include "net_flow.h"
#include "UserSimulationToggles.h"
#include "help.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT

      COMMON/AVRAGE/IDAVER
      common/hcthry/ihct,icorexhct
      COMMON/PCEQU/IMPEXP,RATIMP

      COMMON/SKYFLG/ISKYFL

      COMMON/AFN/IAIRN,LAPROB,ICAAS(MCOM)
      INTEGER :: iairn,icaas
      COMMON/FFN/IFLWN,ICFFS(MPCON)
      COMMON/PITER/MAXITP,PERREL,PERTMP,PERFLX,PERMFL,itrclp,
     &             ICSV(MPNODE,MPVAR),CSVI(MPNODE,MPVAR)

C Tolerance for iteration of hydrogen flow matrix.
      common/PITER_H2/fH2_iteration_tolerance
      real fH2_iteration_tolerance

C Some of these variables may have been set in the mass flow
C network file.
      COMMON/MFSINC/MFBSNC
      COMMON/MFCALC/IPSMOD,MSLVTP
      COMMON/MFLITR/MAXITF,FERREL,FERMFL,PMAX,STEFFR,MFTRAC,ITER,IOK

      COMMON/CLMFLG/ICLMFL
      COMMON/PMSLV/IPMSLV

      common/spmfxst/ispmxist,spflnam 
      
      integer ncomp,ncon
      COMMON/C1/NCOMP,NCON

      INTEGER :: ispmxist


      COMMON/GRSD100/IndxSt
      COMMON/MOIST01/MSTROK,MSTRZN(MCOM)
      LOGICAL MSTROK,MSTRZN
      COMMON/VERSOL/ipkzon(mzs),ipksf(mzs),ivsolfil,vsol,VTSOL,vsnam
      common/filep/ifil
      COMMON/Vld30/Tinitial
      REAL :: Tinitial

C Fluid properties to use if a zone is not air filled:
C  zfldK - conductivity;
C  zfldD - density;
C  zfldC - specific heat capacity;
C  zflsA - total shortwave absorptivity;
C  zSWA - shortwave absorption (W).
      COMMON/zfluid/znotair(mcom),zfldK,zfldD,zfldC,zfldA,
     &              zSWAp(mcom),zSWAf(mcom)
      real zfldK,zfldD,zfldC,zfldA,zSWAp,zSWAf
      LOGICAL znotair

      common/ACTDOM/CFDOK
      logical CFDOK

C Treatment of solar radiation data in weather file.
C ESP-r operates on hourly-centered data. That is, the amount of solar
C irradiance at the recorded hour is the instantaneous irradiance at the
C hour. In Canadian Weather for Energy Calculations (CWEC) files, solar
C radiation is integrated over the previous hour. To account for 
C the half hour shift, a flag has been implemented to indicate
C whether solar radiation data in the weather file is hour-centered
C (default) or half-hour centered.
C The flag can be set in the .cfg file or within bps in the 
C 'simulation toggles' menu.
      common/CWEC_SOL/iSlr_half_hr_flg
      integer iSlr_half_hr_flg     ! if 0 then radiation data in weather file
                                   ! is hour-centered (default); if 1 half-hour centered.

      common/EVCOOL/iflgevapC(mcom) ! if 1, makes the indoor wet bulb temp = outdoor wet
                                    ! bulb temperature (more precisely: the indoor air
                                    ! humidity ratio)
      integer iflgevapC    
      
      dimension SALT(7),IVALS(5),IVAL(MCOM),HCALT(15),IHCVAL(15)


C Used to define the sky temperature calculation method.
      COMMON/EXLONGWV/iExt_longwave_flg
      integer iExt_longwave_flg    ! 0 = Martin & Berdahl (1984);
                                   ! 1 = Clarke ($7.5.4 JC book, 2nd edition, 2001);
                                   ! 2 = Cole (?);
                                   ! 3 = Czeplak & Kasten (1987);               
                                   ! 4 = Swinbank (clear - uses dry bulb T);                  
                                   ! 5 = Swinbank (clear - uses wet bulb T);       
                                                                                                                                                    
      CHARACTER*72 LAPROB,vsnam,lfil
      CHARACTER*40 ITEMG(35)
      character SALT*51,spflnam*72,outs*124
      character HCALT*42

      logical VTSOL
      integer NITMS,INO  ! max items and current menu item

      helpinsub='simcon'  ! set for cfiles

C Make up strings for the current set of toggles.
  92  ino= -4
      if(IDAVER.eq.0)then
        ITEMG(1)=   'a timestep averaging >> ON              '
      elseif(IDAVER.eq.1)then
        ITEMG(1)=   'a timestep averaging >> OFF             '
      endif
      WRITE(ITEMG(2),'(A,F5.2)')
     &              'b bldng eqns implicitness degree: ',fUserGAM
      if(IMPEXP.eq.1)then
        ITEMG(3)=   'c plant equations    >> implicit        '
      elseif(IMPEXP.eq.2)then
        ITEMG(3)=   'c plant equations    >> mixed impl/expl '
      elseif(IMPEXP.eq.3)then
        ITEMG(3)=   'c plant equations    >> time-const depnd'
      elseif(IMPEXP.eq.4)then
        ITEMG(3)=   'c plant equations    >> steady state    '
      endif
      if(iUserSolar.eq.0)then
        ITEMG(4)=   'd solar calculations >> processing on   '
      elseif(iUserSolar.eq.1)then
        ITEMG(4)=   'd solar calculations >> processing off  '
      endif
      if(ISKYFL.eq.-1)then
        ITEMG(5)=   'e sky radiation method >> Isotropic     '
      elseif(ISKYFL.eq.0)then
        ITEMG(5)=   'e sky radiation method >> Klucher       '
      elseif(ISKYFL.eq.1)then
        ITEMG(5)=   'e sky radiation method >> Muneer        '
      elseif(ISKYFL.eq.2)then
        ITEMG(5)=   'e sky radiation method >> Perez 1987    '
      elseif(ISKYFL.eq.3)then
        ITEMG(5)=   'e sky radiation method >> Perez 1990    '
      endif
      write(ITEMG(6),'(a32,F7.2)')'f building initial temperature= ',
     &                          Tinitial
      ITEMG(7)=     '  ________________________________      '
      ITEMG(8)=     'g mass flow parameters                  '
      if(IPSMOD.eq.1)then
        ITEMG(9)=   'h mass flow stack press >> sending node'
      elseif(IPSMOD.eq.2)then
        ITEMG(9)=   'h mass flow stack press >> avg of nodes'
      elseif(IPSMOD.eq.0)then
        ITEMG(9)=   'h mass flow stack press >> NA          '
      endif
      if(MSLVTP.eq.1)then
        ITEMG(10)=  'i mass flow solution >> Gaussian        '
      elseif(MSLVTP.eq.2)then
        ITEMG(10)=  'i mass flow solution >> LU decomposition'
      elseif(MSLVTP.eq.0)then
        ITEMG(10)=  'i mass flow solution >> NA              '
      endif
      if(MFTRAC.eq.-1)then
        ITEMG(11)=  'j mass flow trace >> silent             '
      elseif(MFTRAC.eq.0)then
        ITEMG(11)=  'j mass flow trace >> iterations only    '
      elseif(MFTRAC.eq.1)then
        ITEMG(11)=  'j mass flow trace >> data on worst node '
      elseif(MFTRAC.eq.2)then
        ITEMG(11)=  'j mass flow trace >> data on all nodes  '
      elseif(MFTRAC.eq.3)then
        ITEMG(11)=  'j mass flow trace >> all nodes + solver '
      endif
      if(MFBSNC.eq.0)then
        ITEMG(12)=  'k mass flow timestep >> 1 per hour      '
      elseif(MFBSNC.eq.1)then
        ITEMG(12)=  'k mass flow timestep >> 1 per zone_step '
      endif
      ITEMG(13)=    '  ________________________________      '
      if(CFDOK)then
        ITEMG(14)=  'l CFD options                  '
      else
        ITEMG(14)=  'l CFD options (NA)             '
      endif
      if(ICLMFL.eq.0) then
        ITEMG(15)=  'm weather interpolation >> ON           '
      elseif(ICLMFL.eq.1) then
        ITEMG(15)=  'm weather interpolation >> OFF          '
      endif
      ITEMG(16)=    'n plant solution method:                '
      ITEMG(17)=    '  ________________________________      '
      if(ispmxist.eq.0) then
        ITEMG(18)=  'o special materials >> OFF              '
      elseif(ispmxist.gt.0)then
        ITEMG(18)=  'o special materials >> ON               '
      endif
      IF(IENTXIST.GT.0) then
        ITEMG(19)=  'p power flow parameters                 '
      ELSE
        ITEMG(19)=  'p power flow parameters (NA)            '
      ENDIF
      ITEMG(20)=    '  ________________________________      '
      if(MSTROK)then
        ITEMG(21)=  'r moisture flow solution parameters     '
      else
        ITEMG(21)=  'r moisture flow solution parameters (NA)'
      endif
      if(IndxSt.GT.0)then
        ITEMG(22)=  'v structured mesh soln parameters     '
      else
        ITEMG(22)=  'v structured mesh soln parameters (NA)'
      endif
      ITEMG(23)=    's convection methods (inside)           '
      ITEMG(24)=    't convection methods (external)         '
      ITEMG(25)=    '  ________________________________      '
      ITEMG(26)=    'u non air filled zones                  '
      ITEMG(27)=    '  ________________________________      '
      ITEMG(28)=    'v vertical solar radiation              '
      ITEMG(29)=    '  ________________________________      '
      if ( iSlr_half_hr_flg .eq. 0 ) then
        ITEMG(30)=  'w solar radiation data >> hour centered '
      else
        ITEMG(30)=  'w solar radiation data >> half-hour cent'
      endif
      if(iExt_longwave_flg.eq.0)then
        ITEMG(31)=  'x ext. longwave rad >> Martin & Berdahl '
      elseif(iExt_longwave_flg.eq.1)then
        ITEMG(31)=  'x ext. longwave rad >> Clarke           '
      elseif(iExt_longwave_flg.eq.2)then
        ITEMG(31)=  'x ext. longwave rad >> Cole             '
      elseif(iExt_longwave_flg.eq.3)then
        ITEMG(31)=  'x ext. longwave rad >> Czeplak & Kasten '
      elseif(iExt_longwave_flg.eq.4)then
        ITEMG(31)=  'x ext. longwave rad >>  Swinbank-clear  '
      elseif(iExt_longwave_flg.eq.5)then
        ITEMG(31)=  'x ext. longwave rad >>  Swinbank-bldng  '
      endif
      ITEMG(32)=    'y indoor TWB = outdoor TWB '    
      ITEMG(33)=    '  ________________________________      '      
      ITEMG(34)=    '? help                                  '
      ITEMG(35)=    '- exit menu                             '
      NITMS=35

C Help text for this menu.
      helptopic='sim_toggles_menu'
      call gethelptext(helpinsub,helptopic,nbhelp)

      CALL EMENU('Simulation options',ITEMG,NITMS,INO)

      if(INO.eq.1)then

C Averaging on/off.
  126   CALL EASKI(IDAVER,' ','Results averaging 0) On or 1) Off? ',
     &    0,'F',1,'F',0,'Results averaging iq 14',IER,nbhelp)
        IF(IER.NE.0)GOTO 126
      elseif(INO.eq.2)then

C Zone side equations implicitness degree.

        CALL EASKR(fUserGAM,' ',
     &        'Building equations implicitness degree?',
     &         0.,'F',1.,'F',0.5,' implicitness degree',IER,nbhelp)

        bUserGAM = .true. 
      elseif(INO.eq.3)then

C Plant components state-space equation types.
  131   call edisp(iuout,' ')
        call edisp(iuout,'Plant equation type from ')
        call edisp(iuout,'  implicit                          (1)')
        call edisp(iuout,'  mixed                             (2)')
        call edisp(iuout,'  time constant dependent (default) (3)')
        call edisp(iuout,'  steady state                      (4)')
        call edisp(iuout,' ')

        CALL EASKI(IMPEXP,' ','Plant equation type (1-4)?',
     &         1,'F',4,'F',3,'plant equation type iq 15',IER,nbhelp)
        IF(IER.NE.0)GOTO 131

        IF(IMPEXP.EQ.2.OR.IMPEXP.EQ.3) THEN
          CALL EASKR(RATIMP,' ','Implicit portion of ratio?',
     &       0.0,'F',1.0,'F',0.5,'implicit explicit ratio',IER,nbhelp)
        ENDIF

      elseif(INO.eq.4)then

C Solar ON/OFF option.

  191   CALL EASKI(iUserSolar,' ',
     &    'Solar processing 0) On or 1) Off?',
     &    0,'F',1,'F',0,'Solar processing is 23',IER,nbhelp)
  
        bUserSolar = .true.
        IF(IER.NE.0)GOTO 191

      elseif(INO.eq.5)then

C Set required sky model: Klucher (0) or Muneer (1) etc.
  301   CALL EASKI(ISKYFL,'Sky model?',
     &' -1)Isotropic, 0)Klucher, 1)Muneer, 2)Perez 1987, 3)Perez 1990',
     &    -1,'F',3,'F',3,'Sky model',IER,nbhelp)
        IF(IER.NE.0)goto 301
      elseif(INO.eq.6)then

C Building initial temperature.
        CALL EASKR(Tinitial,' ','Building initial temperature?',
     &         0.,'-',1.,'-',15.0,' temperature ',IER,nbhelp)
      elseif(INO.eq.8)then

C Alter mass flow solver iteration parameters, but
C always synchronize corresponding plant solution iteration
C parameters except for MAXITP / MAXITF.
        IF(IAIRN.EQ.0.AND.IFLWN.EQ.0) THEN
          call edisp(iuout,
     &      'No fluid flow file active, menu selection invalid.')
        else
          CALL MFMODE(1)
          PERREL=FERREL
          PERMFL=FERMFL
        endif
      elseif(INO.eq.9)then

C Use different stack pressure assumptions.?
        IF(IAIRN.EQ.0.AND.IFLWN.EQ.0) THEN
          call edisp(iuout,
     &      'No fluid flow file active, menu selection invalid.')
        else
          CALL MFMODE(2)
        endif
      elseif(INO.eq.10)then

C Use a different matrix solver?
        IF(IAIRN.EQ.0.AND.IFLWN.EQ.0) THEN
          call edisp(iuout,
     &      'No fluid flow file active, menu selection invalid.')
        else
          CALL MFMODE(3)
        endif
      elseif(INO.eq.11)then

C Mass flow trace.
        IF(IAIRN.EQ.0.AND.IFLWN.EQ.0) THEN
          call edisp(iuout,
     &      'No fluid flow file active, menu selection invalid.')
        else
          CALL MFMODE(4)
        endif
      elseif(INO.eq.12)then

C Reset fluid flow calculation & zone time-step synchronisation
C This is over-ruled in case of plant fluid simulation
        IF(IAIRN.EQ.0.AND.IFLWN.EQ.0) THEN
          call edisp(iuout,
     &     'No fluid flow file active, menu selection invalid.')
        ELSE IF(IFLWN.EQ.1) THEN
          call edisp(iuout,
     &     'Plant fluid flow simulation active; fluid flow')
          call edisp(iuout,
     &     'simulation & zone time-step synchronisation')
          call edisp(iuout,'cannot be reset!')
        ELSE
          CALL EASKMBOX(' ','Fluid flow simulation:',
     &      'per hour','per zone time-step',
     &      ' ',' ',' ',' ',' ',' ',IW,nbhelp)
          MFBSNC=IW-1
        ENDIF
      elseif(INO.eq.14)then
        if(CFDOK)then
          call CFDTOG
        else
          call edisp(iuout,'No CFD domain defined for current model.')
        endif
      elseif(INO.eq.15)then
        helptopic='toggle_clm_interpolation'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL EASKMBOX(' ','Weather interpolation:','on','off',
     &    ' ',' ',' ',' ',' ',' ',IW,nbhelp)
        ICLMFL=IW-1
      elseif(INO.eq.16)then

C Plant solution technique.
  308   helptopic='toggle_plant_solver'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL EASKI(IPMSLV,' ','Plant solver type?',
     &    1,'F',3,'F',2,'solver type',IER,nbhelp)
        IF(IER.NE.0) GOTO 308
        
        bUserPltSolvConfig = .true. 

  309   CALL EASKI(MAXITP,' ','Maximum number of iterations?',
     &    1,'F',0,'-',1,'no iter',IER,nbhelp)

        CALL EASKI(iUserITRCLP,
     &    ' ','No. of iterations for control loops?',
     &    1,'F',0,'-',1,'no iter',IER,nbhelp)

        CALL EASKR(fUserPERREL,' ','Relative error (-)?',
     &    0.0001,'F',0.,'-',1.,'rel err',IER,nbhelp)

        CALL EASKR(fUserPERTMP,' ','Absolute temperature error (K)?',
     &    0.0001,'F',0.,'-',1.,'abs err temp',IER,nbhelp)

        CALL EASKR(fUserPERFLX,' ','Absolute heat flux error (W)?',
     &    0.0001,'F',0.,'-',1.,'abs err flux',IER,nbhelp)

        CALL EASKR(fUserPERMFL,
     &    ' ','Absolute mass flow error (kg/s)?',
     &    0.0001,'F',0.,'-',1.,'abs err flux',IER,nbhelp)

        CALL EASKR(fUserH2iterTol,
     &    ' ','Absolute water mass flow error (kg/s)?',
     &    5.0E-6,'F',0.,'-',1.,'abs err flux',IER,nbhelp)

        IF(IER.NE.0) GOTO 309

C Update parameters.        
        MAXITP = iUserMAXITP    
        itrclp = iUserITRCLP    
        PERREL = fUserPERREL    
        PERTMP = fUserPERTMP    
        PERFLX = fUserPERFLX    
        PERMFL = fUserPERMFL    
        fH2_iteration_tolerance = fUserH2iterTol          
        
C Perhaps plant solution iteration parameters are altered, so always
C synchronize corresponding mass flow solver iteration parameters
C except for MAXITP / MAXITF
        FERREL=PERREL
        FERMFL=PERMFL
      elseif(INO.eq.18) then

C Special materials.
        helptopic='toggle_special_materials'
        call gethelptext(helpinsub,helptopic,nbhelp)
        call EASKMBOX(' ','Special facade materials:','off','on',
     &    ' ',' ',' ',' ',' ',' ',IW,nbhelp)
        ispmxist=IW-1

C Identify special materials file & call routine to read it.
        if(ispmxist.gt.0)then
          call spminit
          call easks(lfil,' ','Special materials output?',
     &      72,'fort.55','special materials output',IER,nbhelp)
          call efopseq(55,lfil,3,IER)
        endif

C Set power flow simulation parameters.
      elseif(INO.eq.19)then
        IF(IENTXIST.GT.0) THEN
          CALL ENETPARAM
        ELSE
          CALL EDISP(IUOUT,'No electrical network is associated with ')
          CALL EDISP(IUOUT,'model so you cannot specify electrical ')
          CALL EDISP(IUOUT,'solution parameters.                     ')
        ENDIF
      elseif(INO.eq.21.and.MSTROK)then

C Moisture flow solution parameters.
        call msttog
      elseif(INO.eq.22.AND.IndxSt.GT.0)then

C Building structured mesh relaxation factor.
        CALL STMTOG
      elseif(INO.eq.23)then

C Internal surface hc theory selection. Display as a set of text
C alternatives and allow one (only) to be selected.
        helptopic='toggle_int_hc_methods'
        call gethelptext(helpinsub,helptopic,nbhelp)

        SALT(1)='Alamdari and Hammond (default) '
        SALT(2)='Khalifa & Marshall, radiator located under window '
        SALT(3)='Khalifa & Marshall, no radiator under window  '
        SALT(4)='Halcrow, time invariant low    '
        SALT(5)='Halcrow, time invariant high   '
        SALT(6)='CIBSE guide, time invariant    '
        SALT(7)='CEN simplified calc method     '
        IX=1
        CALL EPICKS(IX,IVALS,' ','Internal surface convection model:',
     &         51,7,SALT,'hc model',IER,nbhelp)
        IF(IX.EQ.0)goto 92
        ihct=IVALS(1)
      elseif(INO.eq.24)then

C External hc correlations:
C  icorhc = 1 then use the default method,
C  icorhc = 2 then use MoWiTT model (Yazdanian & Klems 1994)
C  icorhc = 3 Aya Hagishima correlation ( ),
C  icorhc = 4 Ya Liu (roof wind speed) ( ),
C  icorhc = 5 Ya Lui (raw wind speed) ( ),
C  icorhc = 6 Loveday (roof wind speed) ( ),
C  icorhc = 7 Loveday (raw wind speed) ( ),
C  icorhc = 8 CIBSE guide ( ),
C  icorhc = 9 ASHRAE Task Group ( ),
C  icorhc = 10 Sturrock ( ),
C  icorhc = 11 Keith Nicol ( ),
C  icorhc = 12 S.E .G.Jayamaha ( ).
        helptopic='toggle_ext_hc_methods'
        call gethelptext(helpinsub,helptopic,nbhelp)
        icorhc=icorexhct
        HCALT(1)='the default ESP-r method'
        HCALT(2)='MoWiTT low rise hc '
        HCALT(3)='Aya Hagishima hc '
        HCALT(4)='Ya Liu (roof wind speed) hc'
        HCALT(5)='Ya Lui (raw wind speed) hc'
        HCALT(6)='Ya Lui (weather station wind speed) hc'
        HCALT(7)='Loveday (roof wind speed) hc'
        HCALT(8)='Loveday (raw wind speed) hc'
        HCALT(9)='CIBSE guide hc'
        HCALT(10)='Loveday (mixed) hc'
        HCALT(11)='British Standard'
        HCALT(12)='ASHRAE Task Group hc'
        HCALT(13)='Sturrock hc'
        HCALT(14)='Keith Nicol hc'
        HCALT(15)='S.E.G. Jayamaha hc'
        IX=1
        CALL EPICKS(IX,IHCVAL,' ','External surface convection model:',
     &         42,15,HCALT,'ext hc model',IER,nbhelp)
        IF(IX.EQ.0)goto 92
        icorexhct=IHCVAL(1)
C        CALL EASKMBOX(' ','External hc correlation:','default',
C     &   'MoWiTT low rise','cancel',' ',' ',' ',' ',' ',ichc,2)
C        if(ichc.eq.1)then
C          icorexhct = 1
C        elseif(ichc.eq.2)then
C          icorexhct = 2
C        else
C          continue
C        endif
      elseif(INO.eq.26)then

C Select air (default) or non air filled zones.
        helptopic='list_water_filled_zones'
        call gethelptext(helpinsub,helptopic,nbhelp)
        INPIC=NCOMP
        CALL EPICKS(INPIC,IVAL,' ',
     &    ' Select zones that are not air filled:',
     &    12,NCOMP,zname,' zone list',IER,nbhelp)
        if(inpic.eq.0)goto 92
        do 142 ixx=1,inpic
          if(ival(ixx).ne.0)then
            znotair(IVAL(ixx))=.true.
          endif
 142    continue
        CALL EASKR(zfldK,' ','Conductivity?',
     &     0.01,'F',500.0,'F',0.6,'Conductivity',IER,nbhelp)
        CALL EASKR(zfldD,' ','Density?',
     &     0.5,'F',1500.0,'F',998.2,'Density',IER,nbhelp)
        CALL EASKR(zfldC,' ','Heat capacity?',
     &     100.0,'F',6000.0,'F',4190.0,'Capacity',IER,nbhelp)
        CALL EASKR(zfldA,' ','Absorptivity?',
     &     0.0,'F',1.0,'F',0.0,'Total absorptivity',IER,nbhelp)

      elseif(INO.eq.28)then

C Read in global vertical radiation from file.
C << The 143 loop is problematic - mzs is a large number
C << consider a different approach
        helptopic='zones_for_vert_solar'
        call gethelptext(helpinsub,helptopic,nbhelp)
        vtsol=.true.
        do 143 iiii=1,mzs
          inpic=1
          call epicks(inpic,IVAL,' ',
     &      'Which zone has vertical radiation?',
     &      12,ncomp,zname,' zone list',IER,nbhelp)
          if(inpic.eq.0)goto 144
          ipkzon(iiii)=IVAL(1)
          call EASKSUR(ipkzon(iiii),ipksf(iiii),' ',
     &      'Surface to associate with radiation?',' ',IER)
  143   continue
  144   if (inpic.eq.0.and.iiii.eq.1) then
           vtsol=.false.
           goto 92
        endif
        ivsolfil=ifil+31
        call erpfree(ivsolfil,istat)
        call easks(vsnam,' ','Name of vertical solar file?',
     &    72,' ','vertical solar file',IER,nbhelp)
        if(IER.NE.0) then
          write(outs,'(a,a)') 'Could not find file ',vsnam
          call edisp(iuout,outs)      
          vtsol=.false.
          goto 92
        endif
        call efopseq(ivsolfil,vsnam,1,IER)
        if(IER.NE.0) then
          write(outs,'(a,a)') 'Error opening file ',vsnam
          call edisp(iuout,outs)
          vtsol=.false.
          goto 92
        endif

      elseif(INO.eq.30)then

C Determine if solar radiation data is to be treated as half-hour
C centered.
        helptopic='solar_data_timing'
        call gethelptext(helpinsub,helptopic,nbhelp)

        call easki(iSlr_half_hr_flg,'Solar data 0) hour-centered,',
     &    '1) half-hour centered?',0,'F',1,'F',0,'solar data',
     &    IER,nbhelp)

      elseif(INO.eq.31)then

C Determine how sky temperatures are calculated.
        helptopic='sky_temperature_options'
        call gethelptext(helpinsub,helptopic,nbhelp)

        call easki(iExt_longwave_flg,
     &    'External longwave calculation 0) Martin & Berdahl,',
     &    '1) Clarke, 2) Cole, 3) Czeplak & Kasten, 4) Swinbank-clear, 
     &     5) Swinbank-bldng?',
     &    0,'F',5,'F',0,'ext lngwave method',IER,nbhelp)  

      elseif(INO.eq.32)then

C Make indoor wet bulb temperature = outdoor wet bulb (for direct evaporative cooling).
        helptopic='wet_bulb_indoor_outdoor'
        call gethelptext(helpinsub,helptopic,nbhelp)

        INPIC=NCOMP
        CALL EPICKS(INPIC,IVAL,' ',
     &    ' Select evaporative cooling zones:',
     &    12,NCOMP,zname,' zone list',IER,nbhelp)
        if(inpic.eq.0)goto 92
        do 88 ixx=1,inpic
          if(ival(ixx).ne.0)then
            iflgevapC(IVAL(ixx))=1 
          endif
 88    continue

      elseif(INO.eq.(NITMS-1))then

C Help.
        helptopic='sim_toggles_menu'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL PHELPD(' sim toggles',nbhelp,'simul_toggl ',0,0,IER)
      elseif(INO.eq.NITMS)then
        return
      else
        ino = -1
        goto 92
      endif
      ino = -4
      goto 92

      end

C ******************** MSTTOG ********************
C Set moisture flow toggles.

      subroutine msttog
#include "help.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/MOIST11/MITRM,RITRMP,RITRMF
      COMMON/MOIST12/GAMm,RGAMm
      COMMON/MOIST13/RLXln,RRLXln,RLXlnF,RRLXlnF,RLXlnT,RRLXlnT
      COMMON/MOIST14/DTfalse,TRMfls,TRMflsp1

      CHARACTER ITEMG(11)*40 
      integer NITMS,INO  ! max items and current menu item

      helpinsub='simcon'  ! set for cfiles

  10  INO=-4
      WRITE(ITEMG(1),'(A,F5.2)')
     &              'a implicitness degree        :',GAMm
      WRITE(ITEMG(2),'(A,F5.2)')
     &              'b pressure linear relaxation :',RLXln
      WRITE(ITEMG(3),'(A,F5.2)')
     &              'c fluid linear relaxation    :',RLXlnF
      WRITE(ITEMG(4),'(A,F5.2)')
     &              'd temp. linear relaxation    :',RLXlnT
      WRITE(ITEMG(5),'(A,F7.1)')
     &              'e false time step relaxation :',DTfalse
      WRITE(ITEMG(6),'(A,I5)')
     &              'f maximum iterations         :',MITRm
      WRITE(ITEMG(7),'(A,F5.2)')
     &              'g pressure stopping criterion:',RITRMp
      WRITE(ITEMG(8),'(A,F8.5)')
     &              'h fluid stopping criterion   :',RITRMf
      ITEMG(9)=     '  ________________________________      '
      ITEMG(10)=    '? help                                  '
      ITEMG(11)=    '- exit menu                             '
      NITMS=11

C Help text for this menu.
      helptopic='moisture_solution_par'
      call gethelptext(helpinsub,helptopic,nbhelp)

      CALL EMENU('Moisture flow solution parameters',ITEMG,NITMS,INO)
      if(INO.eq.1)then

C Moisture flow equations implicitness degree.
        CALL EASKR(GAMm,' ','Implicitness degree?',
     &    0.,'F',1.,'F',0.5,'implicitness degree',IER,nbhelp)
      elseif(INO.eq.2)then

C Moisture flow equations pressure linear relaxation factor.
        CALL EASKR(RLXln,' ','Pressure linear relaxation factor?',
     &    0.,'F',2.,'F',0.1,'relaxation factor',IER,nbhelp)
      elseif(INO.eq.3)then

C Moisture flow equations fluid linear relaxation factor.
        CALL EASKR(RLXlnF,' ','Fluid linear relaxation factor?',
     &    0.,'F',2.,'F',0.1,'relaxation factor',IER,nbhelp)
      elseif(INO.eq.4)then

C Temperature linear relaxation factor.
        CALL EASKR(RLXlnT,' ','temp. linear relaxation factor?',
     &    0.,'F',2.,'F',0.1,'relaxation factor',IER,nbhelp)
      elseif(INO.eq.5)then

C Moisture flow equations false time step relaxation factor.
        CALL EASKR(DTfalse,' ','False time step?',
     &    0.,'F',3600.,'-',3600.,'false time step',IER,nbhelp)
      elseif(INO.eq.6)then

C Maximum iterations moisture flow equations.
        CALL EASKI(MITRm,' ','Maximum iterations?',
     &    2,'F',1000,'W',200,'maximum iterations',IER,nbhelp)
      elseif(INO.eq.7)then

C Pressure stopping criterion moisture flow equations.
        CALL EASKR(RITRMp,' ','Pressure stopping criterion?',
     &    0.01,'W',10.,'W',0.1,'pressure stopping criterion',
     &    IER,nbhelp)
      elseif(INO.eq.8)then

C Fluid stopping criterion moisture flow equations.
        CALL EASKR(RITRMf,' ','Fluid stopping criterion?',
     &    0.00001,'W',1.0,'W',0.001,'fluid stopping criterion',
     &    IER,nbhelp)
      elseif(INO.eq.(NITMS-1))then

C Help.
        helptopic='moisture_solution_par'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL PHELPD('moisture toggles',nbhelp,'-',0,0,IER)
      elseif(INO.eq.NITMS)then

C Return to simulation toggles menu.
        return
      endif

C Redisplay the moisture flow solution parameters menu.
      goto 10
      end

C ******************** STMTOG ********************
C Set building construction structured mesh toggles.

      subroutine STMTOG
#include "help.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/GRSD120/RLnrSD,RRLnrSD
      COMMON/GRSD121/InSSlv,MSItrn,TSCrSM,QSCrSM
      COMMON/GRSD122/InSvST,ISTemp

      CHARACTER ITEMG(10)*40 
      integer NITMS,INO  ! max items and current menu item

      helpinsub='simcon'  ! set for cfiles

   10 INO=-4
      IF(InSSlv.EQ.1)THEN
        ITEMG(1)=   'a structured mesh solver >> direct'
        ITEMG(2)=   'b linear relaxation factor : (N/A)'
        ITEMG(3)=   'c maximum iterations       : (N/A)'
        ITEMG(4)=   'd temp. stopping criterion : (N/A)'
        ITEMG(5)=   'e plant stopping criterion : (N/A)'
      ELSE
        ITEMG(1)=   'a structured mesh solver >> iterative'
        WRITE(ITEMG(2),'(A,F5.2)')
     &              'b linear relaxation factor :',RLnrSD
        WRITE(ITEMG(3),'(A,I5)')
     &              'c maximum iterations       :',MSItrn
        WRITE(ITEMG(4),'(A,F8.5)')
     &              'd temp. stopping criterion :',TSCrSM
        WRITE(ITEMG(5),'(A,F8.5)')
     &              'e plant stopping criterion :',QSCrSM
      ENDIF
      ITEMG(6)=     '  ________________________________      '
      IF(InSvST.EQ.0)THEN
        ITEMG(7)=     's save mesh temperatures >> OFF '
      ELSE
        ITEMG(7)=     's save mesh temperatures >> dump3d'
      ENDIF
      ITEMG(8)=     '  ________________________________      '
      ITEMG(9)=     '? help                                  '
      ITEMG(10)=    '- exit menu                             '
      NITMS=10

C Help text for this menu.
      helptopic='structured_mesh_options'
      call gethelptext(helpinsub,helptopic,nbhelp)

      CALL EMENU('structured mesh solution parameters',ITEMG,NITMS,INO)

C Solver index.
      if(INO.eq.1)then
        IF(InSSlv.EQ.1)THEN
          InSSlv=2
        ELSE
          InSSlv=1
        ENDIF
      elseif(INO.eq.2.AND.InSSlv.EQ.2)then

C linear relaxation factor.
        CALL EASKR(RLnrSD,' ','Structured mesh relaxation factor?',
     &    0.,'F',2.,'F',1.0,'relaxation factor',IER,nbhelp)
      elseif(INO.eq.3.AND.InSSlv.EQ.2)then

C Maximum iterations.
        CALL EASKI(MSItrn,' ','Maximum iterations?',
     &    2,'F',1000,'W',200,'maximum iterations',IER,nbhelp)
      elseif(INO.eq.4.AND.InSSlv.EQ.2)then

C Temperature stopping criterion.
        CALL EASKR(TSCrSM,' ','Temperature stopping criterion?',
     &    0.000001,'W',1.,'W',0.00001,'temp. stopping criterion',
     &    IER,nbhelp)
      elseif(INO.eq.5.AND.InSSlv.EQ.2)then

C Plant power stopping criterion.
        CALL EASKR(TSCrSM,' ','Plant power stopping criterion?',
     &    0.00001,'W',10.,'W',0.1,'plant stopping criterion',IER,nbhelp)
      elseif(INO.eq.7)then

C Save structured mesh temperature.
        if(InSvST.EQ.0)then
          InSvST=1
        else
          InSvST=0
        endif
      elseif(INO.eq.(NITMS-1))then

C Help.
        helptopic='structured_mesh_options'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL PHELPD(' structured mesh toggles',3,'-',0,0,IER)
      elseif(INO.eq.NITMS)then

C Return to simulation toggles menu.
        return
      endif

C Redisplay the moisture flow solution parameters menu.
      goto 10
      end

C ******************** MZSPER ********************
C Allows user definition of the simulation
C period (in terms of start and finish day and month
C numbers) and the computational time step (measured
C as a fraction of and including one hour).

C Common block variables are:

C ISD1 & ISM1  - start day and month numbers.
C ISD2 & ISM2  - finish day and month numbers.
C ISDS & ISDF  - start and finish year day numbers.
C NTSTEP       - number of computational time-steps
C                within each hour.
C TIMSEC       - length of time increment in seconds

      SUBROUTINE MZSPER
#include "building.h"
#include "MultiYear_simulations.h"
#include "help.h"

      COMMON/SET1/IYEAR,IBDOY,IEDOY,IFDAY,IFTIME
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/PERS/ISD1,ISM1,ISD2,ISM2,ISDS,ISDF,NTSTEP
      COMMON/PCTIME/TIMSEC
      COMMON/PCTSTP/NTSTPP
      COMMON/C6/INDCFG
      COMMON/MOIST14/DTfalse,TRMfls,TRMflsp1
      COMMON/PREC7/ITCNST
      
C Active domain common.
      common/ACTDOM/CFDOK
      logical CFDOK

C Once per hour storage of results, averaged over that hour.
      common/rdcgen/irdact

      COMMON/SPFL/spfileok,perok,cfdperok,tstepok,saveok,autook,exitok,
     &  startupok
      LOGICAL spfileok,perok,cfdperok,tstepok,saveok,autook,exitok,
     &  startupok
      common/spfldat/nsset,isset,isstup,isbnstep,ispnstep,issave,isavgh
      INTEGER :: nsset,isset,isstup,isbnstep,ispnstep,issave,isavgh

      character outs*124

      logical hri,defok,OK

      helpinsub='simcon'  ! set for cfiles

C Request simulation period in user defined format of d-o-y
C or day and month, return julian days and convert to month
C and day.
  14  if(.NOT.perok) then

C If period has not been defined, assign defaults        
        isds=9
        isdf=15
        if ( bMY_sim_enabled ) then
           iMY_start_year = 2000
           iMY_end_year   = 2000
        endif     
      else

C If period has been defined get start & end days
        CALL EDAY(isd1,ism1,isds)
        CALL EDAY(isd2,ism2,isdf)

C Adjust end date for multi-year simulation
        if ( bMY_sim_enabled ) then           
          isdf = isdf + 365 * ( iMY_end_year - iMY_start_year )         
        endif
        
      endif
    8 if(.not.autook) then
        if ( bMY_sim_enabled ) then
C Multi-year simulation prompt.   
          call eAskPerYear('Simulation period',isds,isdf,
     &                   iMY_start_year,iMY_end_year,ifday,ier)
        else
C Single year simulation prompt.
          call eAskPer('Simulation period',isds,isdf,ifday,ier)
        endif ! <- matches if ( bMY_sim_enabled )...
        IF(IER.NE.0) GOTO 14
      endif ! <- matches if (.not. autook)     
      call edayr(isds,isd1,ism1)
      call edayr(ISDF - 365*(iMY_end_year - iMY_start_year), ISD2,ISM2)

C Check start day before finish day.
      IF(ISDS.LE.ISDF)goto 16
      CALL edisp(iuout,'Start/ finish dates out of order!')
      if ( .not. autook ) goto 8

C Define startup period.
 16   if ( .not.autook ) then
         helptopic='startup_period_dialog'
         call gethelptext(helpinsub,helptopic,nbhelp)
         CALL EASKI(ITCNST,' ','Number of start-up days?',
     &     0,'F',364,'F',1,'startup period',IER,nbhelp)
      endif

C Define computational time-step.
      if (.not.autook) then
        if(.not.tstepok) NTSTEP=1
        IF(INDCFG.EQ.2)goto 31
        helptopic='timestep_freq_dialog'
        call gethelptext(helpinsub,helptopic,nbhelp)
        mtsphr=60
        call easki(ntstep,' ','Building time-steps/hour?',
     &    1,'F',mtsphr,'F',1,'time-steps/hour?',ier,nbhelp)
      endif

C MZCOE2 called to ensure that coefficient values are
C updated according to the selected time step.
      TSTEP=1./FLOAT(NTSTEP)
      CALL MZCOE2(TSTEP)
      DTfalse=TSTEP*3600.0

   31 IF (INDCFG.ne.1) then

C Define plant time-step. Plant can operate at a different
C frequency to the zones. It assumes the zones constant
C during the plant simulation.
        RMIN=60.0/NTSTEP
        write(outs,22)RMIN
   22   format('Zone time-step is',F6.2,' minutes.')
        call edisp(iuout,' ')
        call edisp(iuout,outs)

C If not in automatic mode and plant timestep is known then
        if (.not.autook) then
          if(.not.tstepok)NTSTPP=4
          helptopic='timestep_freq_dialog'
          call gethelptext(helpinsub,helptopic,nbhelp)

C Minimum allowable time-step within specified period is 0.01 seconds.
          call easki(NTSTPP,' ',
     &      'Plant time-steps/building time-step?',1,'F',
     &      360000,'F',4,'plnt time-steps/building time-step',
     &      ier,nbhelp)

C Calculate the length of the time increment in seconds.
          TIMSEC=3600.0/(FLOAT(NTSTEP)*FLOAT(NTSTPP))
        endif
      endif

C Set, once per hour storage of results, averaged over that hour.
      if(ntstep.gt.1.or.(indcfg.ne.1.and.ntstep*ntstpp.gt.1)) then
        if (.not.autook) then
          helptopic='timestep_avg_dialog'
          call gethelptext(helpinsub,helptopic,nbhelp)
          defok=.false.
          hri=.false.
          call easkok(' ','Hourly results integration?',hri,
     &      nbhelp)
          if(hri)then
            irdact=1
          else
            irdact=0
          endif
        else

C Check against user preference in simulation parameters.
          if(isavgh.eq.1)then
            irdact=1
          else
            irdact=0
          endif
        endif
      else
        irdact=0
      endif

      RETURN
      END

C ******************** askaboutcfd ********************
C Confirms simulation periods for CFD assessments.
C If there is a simulation parameter set with attributes about
C the cfd assessments some of common block CFSEUP will have
C been filled in.

      subroutine askaboutcfd
#include "building.h"
#include "cfd.h"
#include "help.h"

      COMMON/SET1/IYEAR,IBDOY,IEDOY,IFDAY,IFTIME
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/PERS/ISD1,ISM1,ISD2,ISM2,ISDS,ISDF,NTSTEP
      common/C1/NCOMP,NCON
      COMMON/SPAD/MMOD,LIMIT,LIMTTY
      
C Active domain common.
      common/ACTDOM/CFDOK
      logical CFDOK

C CFD commons.
      common/cfdfil/LCFD(MCOM),IFCFD(MCOM)
      character LCFD*72
      integer IFCFD
      real CFTIMS,CFTIMF    ! start and finish times for CFD assessment
      integer ICFDYS,ICFDYF ! julian day-of-year at start and end of CFD
      integer ICFAUX  ! helps to keep track of IFCFD()
      integer ICFVIEW ! toggle for preview via dfv during assessment
      integer ICFVIEWMINT,ICFVIEWMAXT  ! range of temperatures for dfv
      logical DFVinvoked
      COMMON/CFSEUP/CFTIMS,CFTIMF,ICFDYS,ICFDYF,ICFAUX(MCOM),ICFVIEW,
     &  ICFVIEWMINT,ICFVIEWMAXT,DFVinvoked
      common/ndcfd/ncfdnd,icfdnd(MNZ),NCONF

C cfdperok - has period been defined by preset.
      common/SPFL/spfileok,perok,cfdperok,tstepok,saveok,autook,exitok,
     &  startupok
      LOGICAL     spfileok,perok,cfdperok,tstepok,saveok,autook,exitok,
     &  startupok

      character outs*124
      logical OK
      integer preRF_DorU

      helpinsub='simcon'  ! set for cfiles
      helptopic='cfd_setup_dialogs'
      call gethelptext(helpinsub,helptopic,nbhelp)

C CFD simulation period.
      if (CFDOK) then

C If period has been defined by preset, skip user interaction, and just
C do the initialisation.

        if (.not.cfdperok) then
          CALL EASKOK('Do you want to activate the CFD domain',
     &                'during the simulation?',OK,nbhelp)
          IF (.not.OK) then
            CFDOK=.false.
            RETURN
          endif
        endif

C Disable CFD calculations during the startup period.
        do IC=1,NCOMP
          IFCFD(IC)=-IFCFD(IC)
        enddo

C Currently this information is not included in cfg file
C so must ask, unless in script mode.
        if (MMOD.eq.-6) then
          ICFVIEW=0
        else
          CALL EASKOK('Display CFD predictions as the',
     &                'simulation progresses?',OK,nbhelp)
          IF(OK)then
            ICFVIEW=1  ! call dfv module when convergence reached
            if(ICFVIEWMAXT.eq.0) ICFVIEWMINT=10; ICFVIEWMAXT=30
            CALL EASKI(ICFVIEWMINT,' ',
     &        'CFD temperature display minimum value?',
     &        0,'F',100,'-',1,'CFD minimum display T',IER,nbhelp)
            CALL EASKI(ICFVIEWMAXT,' ',
     &        'CFD temperature display maximum value?',
     &        0,'F',100,'-',1,'CFD max display T',IER,nbhelp)
          ELSE
            ICFVIEW=0
            if(ICFVIEWMAXT.eq.0) ICFVIEWMINT=10; ICFVIEWMAXT=30
          ENDIF
        endif

C Period has been defined get start & end days, but since isds and isdf
C would have been defined in a separate call to MZPER make sure these
C values have been set if ICFDYS & ICFDYF are still zero.
        CALL EDAY(isd1,ism1,isds)
        CALL EDAY(isd2,ism2,isdf)
        if(ICFDYS.eq.0.and.ICFDYF.eq.0)then
          ICFDYS=ISDS
          ICFDYF=ISDF
          CFTIMS=1.00
          CFTIMF=24.99
        endif

        if (.not.cfdperok) then
          IFDAY=1
  40      call EASKPER('CFD active period',
     &      ICFDYS,ICFDYF,IFDAY,IER)
          if (ICFDYF.lt.ICFDYS) then
            call edisp(iuout,' ')
            call edisp(iuout,'Start/ finish days out of order!')
            goto 40
          endif
          if (ICFDYS.lt.ISDS) then
            call edisp(iuout,' ')
            call edisp(iuout,
     &        'Start day is before simulation start day!')
            goto 40
          endif
          if (ICFDYF.gt.ISDF) then
            call edisp(iuout,' ')
            call edisp(iuout,
     &        'Finish day is after simulation finish day!')
            goto 40
          endif

C Ask about start and stop times.
  50      write(outs,'(a)')'For the first day CFD is active specify'
          CALL EASKR(CFTIMS,outs,'Starting hour?',
     &      1.0,'F',24.99,'F',1.0,'Decimal fraction',IER,nbhelp)
          write(outs,'(a)')'For the last day CFD is active specify'
          CALL EASKR(CFTIMF,outs,'Finishing hour?',
     &      1.0,'F',24.99,'F',24.99,'Decimal fraction',IER,nbhelp)
          if ((ICFDYF.eq.ICFDYS).and.(CFTIMF.lt.CFTIMS)) then
            call edisp(iuout,' ')
            call edisp(iuout,'Start/ finish hours out of order.')
            goto 50
          endif
        endif

        DO 30 IC=1,NCOMP
          ICFAUX(IC)=IFCFD(IC)
          IFCFD(IC)=0
 30     CONTINUE
      endif
      return
      end


C ******************** MZSIMH ********************
C Outputs simulation period and wait time information
C prior to each simulation commencing.

      SUBROUTINE MZSIMH(ICON)
#include "building.h"
#include "model.h"
#include "geometry.h"
#include "esprdbfile.h"
#include "control.h"
#include "UserSimulationToggles.h"
#include "help.h"
     
      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/SHOUT/ICOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      
      integer ncomp,ncon
      COMMON/C1/NCOMP,NCON
      COMMON/PREC7/ITCNST

      COMMON/C6/INDCFG

      COMMON/PERS/ISD1,ISM1,ISD2,ISM2,ISDS,ISDF,NTSTEP
      COMMON/PCTIME/TIMSEC
      COMMON/PCTSTP/NTSTPP
      COMMON/SIMTIM/IHRP,IHRF,IDYP,IDYF,IDWP,IDWF,NSINC,ITS,idynow
      COMMON/SET1/IYEAR,IBDOY,IEDOY,IFDAY,IFTIME

      COMMON/SAVE/ISAVE
      COMMON/LIBREC/IRECPL
      COMMON/LIBRCP/IRCPLP
      common/reclen/nzrl,nprl,nerl

      COMMON/PREC9/NCONST(MCOM),NELTS(MCOM,MS),NGAPS(MCOM,MS),
     &             NPGAP(MCOM,MS,MGP)

      COMMON/CCTLNM/CTLDOC,LCTLF
      COMMON/AVRAGE/IDAVER
      COMMON/BEFORM/IBEQNF
      COMMON/PCEQU/IMPEXP,RATIMP
      COMMON/SOLOFF/ISOLAR

      COMMON/WARN/IWARN

C TSC data.
      COMMON/TS/ITSFLG,NTSCPD,ITSSCH(MSCH,5),itsmus

      COMMON/VTHP30/ILTHPS,ILTHPZ(MCOM)
      COMMON/VTHP31/INTHPS,INTHPZ(MCOM)
      LOGICAL INTHPS,INTHPZ
      COMMON/GR1D04/GAM,RGAM
      COMMON/GR1D06/IGR1D
      LOGICAL IGR1D
      COMMON/GR3D100/BLDG3D,ZONE3D(MCOM)
      LOGICAL :: BLDG3D,ZONE3D
      COMMON/GRND100/GRND3D
      LOGICAL GRND3D
      COMMON/MOIST01/MSTROK,MSTRZN(MCOM)
      LOGICAL MSTROK,MSTRZN
      COMMON/GRSD100/IndxSt
      common/glbctl/global,glbrwd

C Results libraries.
      COMMON/RESLIB/RFILE,PFILE,MSTRFILE,LAFRES,CFRESF
      character RFILE*72,PFILE*72,MSTRFILE*72,LAFRES*72,CFRESF*72

C Once per hour storage of results, averaged over that hour.
      common/rdcgen/irdact

C Uncertainty.
      COMMON/UA5/IMET,ISIM,NRSIM
      INTEGER :: imet,isim,nrsim

C Simulator parameters.
      COMMON/SPFL/spfileok,perok,cfdperok,tstepok,saveok,autook,exitok,
     &  startupok
      LOGICAL     spfileok,perok,cfdperok,tstepok,saveok,autook,exitok,
     &  startupok

C Toggle for extreme silent running (endless) mode.
      logical endless,issilient
      common/endlessmode/endless,issilient

      CHARACTER ctldoc*248
      CHARACTER LCTLF*72
      character PERST1*14,PERST2*44,PERST3*44
      character outs*124,louts*248

      LOGICAL OK
      LOGICAL ILTHPS,ILTHPZ,global,glbrwd

      helpinsub='simcon'  ! set for cfiles
      helptopic='sim_starting_report'
      call gethelptext(helpinsub,helptopic,nbhelp)

      ISAV1=ISAVE+1

      IDAYS=ISDF-ISDS+1

C If multiple simulations then skip reporting (its the same as 
C the first time).
      IF (ISIM.gt.1) goto 6

C Calculate number of words required by solution file.
C The record length of the results file is nzrl words.
c Added a 5th computed goto for save level 5.
      goto (41,8,9,11,11,41),ISAV1
    8 if(irdact.ne.0)then   ! save 1
        NWORD=(IDAYS*irdact*NCOMP*24*1+IRECPL)*(nzrl)
      else
        NWORD=(IDAYS*NTSTEP*NCOMP*24*1+IRECPL)*(nzrl)
      endif
      goto 12
    9 if(irdact.ne.0)then   ! save 2
        NWORD=(IDAYS*irdact*NCOMP*24*3+IRECPL)*(nzrl)
      else
        NWORD=(IDAYS*NTSTEP*NCOMP*24*3+IRECPL)*(nzrl)
      endif
      goto 12
   11 NREC=0                ! save 3 or 4.
      DO 10 I=1,NCOMP
        NREC=NREC+3         ! the initial 3 records
        DO 20 J=1,NCONST(I)
          NREC=NREC+1       ! one more per surace in zone
   20   CONTINUE
   10 CONTINUE

C Use the actual record width (nzrl).
      if(irdact.ne.0)then
        NWORD=(IDAYS*irdact*NREC*24+IRECPL)*(nzrl)
      else
        NWORD=(IDAYS*NTSTEP*NREC*24+IRECPL)*(nzrl)
      endif

C There are 4 bytes in a word therefore the number
C of kilobytes required by the solution file is:
   12 BYTE=ABS(FLOAT(NWORD)*4.0/1024.0)

C Determine the number of zone days simulated.
   41 NZTS=(NCOMP*(IDAYS+ITCNST))*NTSTEP*24

C Write out appropriate message.
      if(endless.or.issilient)then
        continue
      else
        call edisp(icout,' ')
        write(louts,'(a,a)')'Model configuration file: ',
     &                      LCFGF(1:lnblnk(LCFGF))
        call edisp248(icout,louts,100)
        write(louts,'(a,a)')'Model description: ',
     &                      modeltitle(:lnblnk(modeltitle))
        call edisp248(icout,louts,100)
        write(louts,'(a,a)')'Weather file: ',
     &                      LCLIM(1:lnblnk(LCLIM))
        call edisp248(icout,louts,100)
        if(NCF.eq.0.and.NCL.eq.0.and.NCC.eq.0)then
          call edisp(icout,'Control: free-floating')
        else
          write(louts,'(a,a)')'Control file: ',
     &                      LCTLF(1:lnblnk(LCTLF))
          call edisp248(icout,louts,100)
        endif
        write(louts,'(a,a)')'Building results file: ',
     &                      RFILE(1:lnblnk(RFILE))
        call edisp248(icout,louts,100)
        if(isave.eq.0)then
          call edisp(icout,'Save option: 0 (No results file)')
        elseif(isave.eq.1)then
          call edisp(icout,'Save option: 1 (Minimum)')
        elseif(isave.eq.2)then
          call edisp(icout,'Save option: 2 (Moderate)')
        elseif(isave.eq.3)then
          call edisp(icout,'Save option: 3 (2 + node temperatures)')
        elseif(isave.eq.4)then
          call edisp(icout,'Save option: 4 (2 + energy balance)')
        elseif(isave.eq.5)then
          call edisp(icout,'Save option: 5 (for Hot 3000)')
        elseif(isave.eq.6)then
          call edisp(icout,'Save option: 6 (No results file)')
        endif
        write(outs,'(a,I6)') 'Prior warnings: ',IWARN
        call edisp(icout,outs)
      endif

      NPTS=NTSTEP*NTSTPP
      TMIN=60.0/NTSTEP
      TMINP=TIMSEC/60.
C      write(outs,'(a,I6,a)')'Simulation period: ',IDAYS,' day(s).'
C      if(endless.or.issilient)then
C        continue
C      else
C        call edisp(icout,' ')
C        call edisp(icout,outs)
C      endif

C Derive string for simulation period.
      call edisp(icout,' ')
      CALL EPERSTR(IYEAR,ISDS,1,ISDF,24,1,1,0,PERST1,PERST2,PERST3,IER)
      if(endless.or.issilient)then
        continue
      else
        write(outs,'(2A)')'Simulation ',PERST2
      endif
      call edisp(icout,outs)
      write(outs,'(a,I6,a)')'Start-up period: ',ITCNST,' day(s).'
      if(endless.or.issilient)then
        continue
      else
        call edisp(icout,outs)
      endif

      IF(INDCFG.EQ.2)goto 131
      if(irdact.eq.0)then
        write(outs,'(a,F5.2,a)')'Building time steps: ',TMIN,
     &                                           ' minutes.'
      else
        write(outs,'(a,F5.2,a)')'Building time steps: ',TMIN,
     &                                           ' minutes.'
      endif 
      if(endless.or.issilient)then
        continue
      else
        call edisp(icout,outs)
      endif
      write(outs,'(a,I6,a)') 'Number of zones: ',NCOMP,'.'
      if(endless.or.issilient)then
        continue
      else
        call edisp(icout,outs)
      endif
C      write(outs,'(a,I7,a)') 'Zone-time increments:',NZTS,'.'
C      if(endless.or.issilient)then
C        continue
C      else
C        call edisp(icout,outs)
C      endif

      IF(INDCFG.EQ.1)goto 133
  131 write(outs,'(a,F6.2,a)')'Plant time-steps: ',TMINP,' minutes.'
      call edisp(icout,outs)
  133 IF(INDCFG.EQ.2)goto 135
      IF(ISAVE.GT.0)goto 42
      if(endless.or.issilient)then
        continue
      else
        call edisp(icout,'Building results file size: not applicable')
      endif
      goto 135
   42 IF(BYTE.GT.2000.0)goto 14
      write(outs,'(a,F8.1,a)')'Building results file size: ',BYTE,
     &                        ' Kbytes.'
      if(isave.eq.1.or.isave.eq.2.or.isave.eq.3.or.isave.eq.4)then
        if(endless)then
          continue
        else
          call edisp(icout,outs)
        endif
      endif
      goto 135
   14 BYTE=BYTE/1024.0
      write(outs,'(a,F8.1,a)')'Building results file size: ',BYTE,
     &                        ' Mbytes.'
      if(isave.eq.1.or.isave.eq.2.or.isave.eq.3.or.isave.eq.4)then
        if(endless)then
          continue
        else
          call edisp(icout,outs)
        endif
      endif

  135 IF(INDCFG.EQ.1)goto 136
      NREC=3
      if(irdact.ne.0)then
        NWORD=(IDAYS*irdact*NREC*24+IRCPLP)*(MS+5)
      else
        NWORD=(IDAYS*NPTS*NREC*24+IRCPLP)*(MS+5)
      endif
      BYTE=FLOAT(NWORD)*4.0/1024.0
      IF(BYTE.GT.2000.0)goto 137
      write(outs,'(a,F7.1,a)')'Plant results file size: ',BYTE,
     &                        ' Kbytes.'
      call edisp(icout,outs)
      goto 136
  137 BYTE=BYTE/1024.0
      write(outs,'(a,F7.1,a)')'Plant results file size: ',BYTE,
     &                        ' Mbytes.'
      call edisp(icout,outs)

C Time-step controller status.
  136 IF(ITSFLG.EQ.0) THEN
        if(endless.or.issilient)then
          continue
        else
          continue
C          call edisp(icout, 'Time-step controller not active')
        endif
      ELSE
        if(endless.or.issilient)then
          continue
        else
          call edisp(icout, 'Time-step controller active.')
        endif
      ENDIF

C If appropriate, output special features invoked.
      IMESS=0
      IF(IBEQNF.NE.0)IMESS=1
      IF(IMPEXP.NE.3.OR.RATIMP.NE.0.5)IMESS=1
      IF(ISOLAR.NE.0)IMESS=1
      IF(IDAVER.NE.0)IMESS=1
      IF(ILTHPS)IMESS=1
      IF(INTHPS)IMESS=1
      IF(IGR1D)IMESS=1
      IF(ABS(GAM-0.5).GT.1.E-4)IMESS=1
      IF(BLDG3D)IMESS=1
      IF(GRND3D)IMESS=1
      IF(MSTROK)IMESS=1
      IF(IndxSt.GT.0)IMESS=1
      IF(bUserPltSolvConfig) IMESS = 1
      IF(IMESS.EQ.1)THEN
        call edisp(icout,' ')
        call edisp(icout,'Special features invoked:')
      ENDIF

C Zone equations implicitness degree.
      IF(ABS(GAM-0.5).GT.1.E-4)THEN
        WRITE(OUTS,'(A,F5.2)')
     &' Zone equations implicitness degree: ',GAM
        CALL EDISP(IUOUT,OUTS)
      ENDIF

C Linear thermal conductivity.
      IF(ILTHPS)
     &CALL EDISP(IUOUT,'Linear thermal conductivity assigned.')

C Non-linear thermophysical properties assigned.
      IF(INTHPS)
     &CALL EDISP(IUOUT,
     &    'Non-linear thermophysical properties assigned.')

C Adaptive 1D gridding performed.
      IF(IGR1D)CALL EDISP(IUOUT,'Adaptive 1D gridding invoked.')
      
C 3D conduction modelling invoked.
      IF(BLDG3D)CALL EDISP(IUOUT,'3D conduction modelling invoked.')

C 3D ground modelling invoked.
      IF(GRND3D)CALL EDISP(IUOUT,'3D ground modelling invoked.')

C Moisture transfer invoked.
      IF(MSTROK)CALL EDISP(IUOUT,'Moisture transfer invoked.')

C Structured mesh modelling invoked.
      IF(IndxSt.GT.0)CALL EDISP(IUOUT,
     &                     'Structured mesh modelling invoked.')

C Default for plant equation generation is implicit
C or mixed, based on time-step evaluation.
      IF(IMPEXP.NE.3.OR.RATIMP.NE.0.5) THEN
        call edisp(icout,' ')
        call edisp(icout,'Plant equation type:')
        IF(IMPEXP.EQ.1)call edisp(icout,' 1 - implicit')
        IF(IMPEXP.EQ.2)call edisp(icout,' 2 - mixed')
        IF(IMPEXP.EQ.3)call edisp(icout,' 3 - timestep dependent')
        IF(IMPEXP.EQ.4)call edisp(icout,' 4 - steady state')
        IF(RATIMP.NE.0.5)then
          write(outs,*) ' impl/expl = ',RATIMP
          call edisp(icout,outs)
        endif
      ENDIF

       UserPltConfigSettings: if ( bUserPltSolvConfig ) then 
        
        WRITE(OUTS,'(A,I5)')
     &  'Plant - max iterations: ',iUserMAXITP
        CALL EDISP(IUOUT,OUTS)
        
        WRITE(OUTS,'(A,I5)')
     &  '        max control iterations: ', iUserITRCLP
        CALL EDISP(IUOUT,OUTS)

        WRITE(OUTS,'(A,F5.2)')
     &  '        max relative error: ',fUserPERREL
        CALL EDISP(IUOUT,OUTS)  
        
        WRITE(OUTS,'(A,F5.2)')
     &  '        max temperature error (C): ',fUserPERTMP
        CALL EDISP(IUOUT,OUTS)  

        WRITE(OUTS,'(A,F5.2)')
     &  '        max mass flow error (kg/s): ',fUserPERMFL
        CALL EDISP(IUOUT,OUTS)          

        WRITE(OUTS,'(A,F5.2)')
     &  '        max heat flux error (W): ',fUserPERFLX
        CALL EDISP(IUOUT,OUTS)          

        WRITE(OUTS,'(A,E12.6)')
     &  '        max H2 flow error (kg/s): ',fUserH2iterTol
        CALL EDISP(IUOUT,OUTS)   

      endif UserPltConfigSettings
      
C Default - solar on.
      IF(ISOLAR.NE.0)call edisp(icout,'Solar processing off')

C Default - results are averaged.
      IF(IDAVER.NE.0)call edisp(icout,'Results averaging off')

C Option to terminate - ignore if in autoexec mode as defined in 
C the simulator parameter file.
      if (autook.or.(ISIM.gt.1)) then
        ok=.true.
      else
        CALL EASKOK(' ','Continue with simulation?',OK,nbhelp)
      endif
      IF(OK)goto 6
      ICON=0
      goto 7

C Commence simulation.
    6 call edisp(icout,' ')
      if (ISIM.gt.1) then
        write(outs,'(a,i4,a,i4,a)')'Simulation ',ISIM,' of ',NRSIM,
     &    ' has now commenced.'
        call edisp(icout,outs)
      else
        if(endless)then
          call edisp(icout,'Simulation proceeding.')
        else
          call edisp(icout,'Simulation commenced.')
        endif
      endif
      if(global)then
         call edisp(icout,' ')
         call edisp(icout,'Global control strategy operating.')
      endif
      ICON=1

C If trace is active then write to the file the header 
C information above to assist in identification.
      if(ITC.gt.0.and.ITU.ne.iuout)then
        call mzinfo(ITU)
      endif
    7 RETURN
      END


C ******************** MZTRAC ********************
C Allows the user to select areas of bps for trace output.

C Common block variables are:
C ITC     - trace output index consisting of some +ve integer
C           value where:
C                      ITC=0 gives no trace output
C                      ITC>0 gives trace output commencing during
C                            the ITC'th time-increment
C ITCF    - is the ITCF'th time increment for the
C           trace output to end
C ITRACE  - trace call index for appropriate subroutine.
C ICNT    - counts trace output for page numbering purposes

      SUBROUTINE MZTRAC
#include "building.h"
#include "model.h"
#include "geometry.h"
#include "help.h"
      
      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      COMMON/PERS/ISD1,ISM1,ISD2,ISM2,ISDS,ISDF,NTSTEP
      
      integer ncomp,ncon
      COMMON/C1/NCOMP,NCON
      COMMON/C6/INDCFG
      COMMON/PREC7/ITCNST
      COMMON/PREC9/NCONST(MCOM),NELTS(MCOM,MS),NGAPS(MCOM,MS),
     &             NPGAP(MCOM,MS,MGP)
      COMMON/PREC13/C(MCOM,MS,MN,2),QC(MCOM,MS,MN)
      COMMON/GR1D01/NNDS,NNDZ(MCOM),NNDC(MCOM,MS),NNDL(MCOM,MS,ME)

C Common for libsv3 trace.
      COMMON/enbal/nsurf,insurf(MS)

      dimension IVALS(MCOM)
      CHARACTER*29 ITEM(35)
      character outs*124,dtrace*72,ltrace*72
      logical OK
      integer NITEMS,INO  ! max items and current menu item

      helpinsub='simcon'  ! set for cfiles
      helptopic='trace_setup_dialogs'
      call gethelptext(helpinsub,helptopic,nbhelp)

      ITEM(1) ='a electrical calculations    '
      ITEM(2) ='b temporal file items & assoc'
      ITEM(3) ='c view factors & long-wave   '
      ITEM(4) ='d zone shading & insolation  '
      ITEM(5) ='e heat transfer coeff & calc '
      ITEM(6) ='f zone RH & condensation     '
      ITEM(7) ='g infil & vent coefficients  '
      ITEM(8) ='h zone casual gains          '
      ITEM(9) ='i zone miscellaneous data    '
      ITEM(10)='j zone matrix coef setup/mods'
      ITEM(11)='l solar radiation summary    '
      ITEM(12)='n air flow calculations      '
      ITEM(13)='o zone matrix setup          '
      ITEM(14)='p matrix forward red         '
      ITEM(15)='q matrix surface adjust      '
      ITEM(16)='r matrix forward red compl   '
      ITEM(17)='s zone equation solution     '
      ITEM(18)='t zone & surface flux balance'
      ITEM(19)='u surface adjacent conditions'
      ITEM(20)='v plant matrix template      '
      ITEM(21)='w plant component static     '
      ITEM(22)='x plant component adj info   '
      ITEM(23)='y plant coef generator       '
      ITEM(24)='z plant matrix solution      '
      ITEM(25)='1 plant matrix setup         '
      ITEM(26)='2 all plant trace items      '
      ITEM(27)='3 control functions          '
      ITEM(28)='4 time-step controller       '
      ITEM(29)='5 network flow results       '
      ITEM(30)='6 Not applicable             '
      ITEM(31)='7 CFD-MFS iteration          '
      ITEM(32)='  ___________________________'
      ITEM(33)='/ clear trace selections     '
      ITEM(34)='? help                       '
      ITEM(35)='- exit menu                  '

C Initilise trace output facility.
      ITC=0
      ITCF=0
      ITU=IUOUT
      DO 10 I=1,MTRACE
        ITRACE(I)=0
   10 CONTINUE
      DO 20 I=1,MCOM
        IZNTRC(I)=0
   20 CONTINUE

      INPIC=NCOMP
      CALL EPICKS(INPIC,IVALS,' ','Zones to include?',
     &  12,NCOMP,zname,' zone trace list',IER,nbhelp)
      IZT=INPIC

      DO 40 I=1,IZT
        IX=IVALS(I)
        IZNTRC(IX)=1
   40 CONTINUE

C Case for plant only.
      if(indcfg.eq.2) goto 9989      

      CALL EASKOK(' ','Select trace topics:',OK,nbhelp)
      IF(OK)goto 9989
      DO 9990 I=1,MTRACE
        ITRACE(I)=1
 9990 CONTINUE
      GOTO 9988

 9989 NITEMS=35

C Help text for this menu.
   4  helptopic='trace_setup_dialogs'
      call gethelptext(helpinsub,helptopic,nbhelp)

      INO=-2
      CALL EMENU('Selective trace',ITEM,NITEMS,INO)
      IF(INO.LE.0)then
        ino=-1
        goto 4
      elseif(INO.EQ.NITEMS-3)then
        ino=-1
        goto 4
      elseif(INO.eq.NITEMS)then

C Jump to process the selected list.
        goto 9988
      elseif(INO.EQ.NITEMS-1)then

C Help for trace facility.
        helptopic='trace_setup_dialogs'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL PHELPD(' simulation trace',10,'-',0,0,IER)
      endif

C Shift the menu index back to get trace index.
      if(ino.eq.1)then
        II=2
        call edisp(iuout,'Selected electical calculation trace.')
      elseif(ino.eq.2)then
        II=3
        call edisp(iuout,'Selected temporal items trace.')
      elseif(ino.eq.3)then
        II=4
        call edisp(iuout,'Selected view factor & longwave trace.')
      elseif(ino.eq.4)then
        II=5
        call edisp(iuout,
     &    'Selected shading & insolation trace.You must re-scan')
        call edisp(iuout,'the model cfg file to get this report.')
      elseif(ino.eq.5)then
        II=6
        call edisp(iuout,'Selected heat transfer coef trace.')
      elseif(ino.eq.6)then
        II=10
        call edisp(iuout,'Selected zone RH & condensation trace.')
      elseif(ino.eq.7)then
        II=11
        call edisp(iuout,'Selected infiltration & ventilation trace.')
      elseif(ino.eq.8)then
        II=12
        call edisp(iuout,'Selected zone casual gains trace.')
      elseif(ino.eq.9)then
        II=14
        call edisp(iuout,'Selected zone miscellaneous trace.')
      elseif(ino.eq.10)then
        II=15
        call edisp(iuout,'Selected zone matrix modification trace.')
      elseif(ino.eq.11)then
        II=19
        call edisp(iuout,'Selected solar radiation trace.')
      elseif(ino.eq.12)then
        II=25
        call edisp(iuout,'Selected air flow calculations trace.')
      elseif(ino.eq.13)then
        II=26
        call edisp(iuout,'Selected zone matrix setup trace.')
      elseif(ino.eq.14)then
        II=27
        call edisp(iuout,'Selected matrix forward reduction trace.')
      elseif(ino.eq.15)then
        II=28
        call edisp(iuout,'Selected matrix forward adjustment trace.')
      elseif(ino.eq.16)then
        II=29
        call edisp(iuout,'Selected matrix forward red compl trace.')
      elseif(ino.eq.17)then
        II=30
        call edisp(iuout,'Selected zone equation solution trace.')
      elseif(ino.eq.18)then
        II=31

C Trace output of library save.
        if(IZT.eq.0)then
          call edisp(iuout,'no zones yet for library save trace...')
          goto 4
        endif
        call edisp(iuout,'Selected zone & surface flux balance trace.')

C Ask for surfaces to be traced in the surface balances.
        CALL EASKI(NSURF,' ',
     &    'No. of surfaces for energy balance trace?',
     &    1,'F',0,'-',1,'no of surf for trace',IER,nbhelp)
        DO 703 I=1,nsurf
          write(outs,8011)I
 8011     format('Which surface for energy balance number ',I2,'?')
          CALL EASKI(IVAL,' ',outs,
     &      1,'F',0,'-',1,'surf for trace',IER,nbhelp)
          INSURF(I)=IVAL
  703   CONTINUE
      elseif(ino.eq.19)then
        II=32
        call edisp(iuout,'Selected surface adjacent conditions trace.')
      elseif(ino.eq.20)then
        II=34
        call edisp(iuout,'Selected plant matrix template trace.')
      elseif(ino.eq.21)then
        II=35
        call edisp(iuout,'Selected plant component staic trace.')
      elseif(ino.eq.22)then
        II=36
        call edisp(iuout,'Selected plant component adj info trace.')
      elseif(ino.eq.23)then
        II=37
        call edisp(iuout,'Selected plant coefficient generator trace.')
      elseif(ino.eq.24)then
        II=38
        call edisp(iuout,'Selected plant matrix solution trace.')
      elseif(ino.eq.25)then
        II=39
        call edisp(iuout,'Selected plant matrix setup trace.')
      elseif(ino.eq.26)then

C If all plant turn on related traces as well.
        II=40
        call edisp(iuout,'Selected all plant trace.')
        ITRACE(34)=1
        ITRACE(35)=1
        ITRACE(36)=1
        ITRACE(37)=1
        ITRACE(38)=1
        ITRACE(39)=1
        ITRACE(40)=1
        goto 4
      elseif(ino.eq.27)then
        II=41
        call edisp(iuout,'Selected control functions trace.')
      elseif(ino.eq.28)then
        II=42
        call edisp(iuout,'Selected timestep controller trace.')
      elseif(ino.eq.29)then
        II=7
        ITRACE(7)=1
        call edisp(iuout,'Selected network flow ascii trace.')
        goto 4
C      elseif(ino.eq.30)then
C        II=43
C        call edisp(iuout,'Selected CFD cell velocities trace.')
      elseif(ino.eq.31) then
        II=44
        call edisp(iuout,'Selected CFD-MFS iteration trace.')
      elseif(ino.eq.NITEMS-2)then

C Clear all selections.
        call edisp(iuout,'Selected CLEAR ALL trace.')
        DO 50 I=1,MTRACE
          ITRACE(I)=0
   50   CONTINUE
        GOTO 4
      endif

C Set trace index for the topic ON.
      ITRACE(II)=1
      GOTO 4

C Good place to remind user of the number of timesteps
C in the pre-simulation period.
 9988 write(outs,'(a,i5,a)')'The presimulation period is ',
     &  ITCNST*24*NTSTEP,' timesteps.'
      call edisp(iuout,outs)
      IDAYS=ISDF-ISDS+1+ITCNST
      write(outs,'(a,i5,a)')'The last timestep of the simulation is ',
     &  IDAYS*24*NTSTEP,' timesteps.'
      call edisp(iuout,outs)
      IVAL=ITC
      CALL EASKI(IVAL,'Start trace after how many',
     &  '(zone-side) time increments?',
     &  1,'F',1,'-',1,'trace start',IER,nbhelp)
      ITC=IVAL

      IVAL=ITC+1
      CALL EASKI(IVAL,'Finish trace after how many',
     &  '(zone-side) time increments?',
     &  ITC,'F',0,'-',5,'trace finish',IER,nbhelp)
      ITCF=IVAL

      CALL EASKMBOX(' ','Write trace to:',
     &  'terminal','file',' ',' ',' ',' ',' ',' ',IW,nbhelp)
      if(IW.eq.1)then
        ITU=IUOUT
      else
        dtrace = 'fort.33'
        ltrace = ' '
        write(ltrace,'(a,a3)') cfgroot(1:lnblnk(cfgroot)),'.33'
  44    CALL EASKS(ltrace,' ','Trace file name ?',
     &    72,dtrace,'scratch file',IER,nbhelp)
        ITU=33
        call fpopen(itu,istat,1,3,ltrace)
      endif

      RETURN
      END


C ******************** mzinfo ********************
C Displays the values of the currently set simulation
C parameters. If itru=iuout then goes to the display, 
C if itru=33 then works as a header to the trace.

      subroutine mzinfo(itru)
#include "building.h"
#include "model.h"
#include "esprdbfile.h"
#include "control.h"
      
      integer lnblnk  ! function definition

      common/pers/isd1,ism1,isd2,ism2,isds,isdf,ntstep
      common/save/isave

      integer ncomp,ncon
      common/c1/ncomp,ncon
      COMMON/RESLIB/RFILE,PFILE,MSTRFILE,LAFRES,CFRESF
      character RFILE*72,PFILE*72,MSTRFILE*72,LAFRES*72,
     &  CFRESF*72

      common/cctlnm/ctldoc,lctlf
      integer icascf
      common/cctl/icascf(mcom)

      character lctlf*72
      character ctldoc*248,outs*124,louts*248

      dimension io(mcom)

      call edisp(itru,' ')
      write(outs,'(A,A)')'Results file: ',RFILE
      call edisp(itru,outs)
      write(louts,'(A,A)')'Weather file: ',LCLIM
      call edisp248(itru,louts,100)
      write(outs,'(A,A)')'Configuration file: ',LCFGF
      call edisp(itru,outs)
      write(outs,'(A,A)')'Configuration descr: ',modeltitle
      call edisp(itru,outs)
      call edisp(itru,' ')
      IF(NCF.EQ.0.AND.NCL.EQ.0)then
        call edisp(itru,'Control strategy: free-floating  ')
      else
        write(outs,'(A,A)')'Control file name: ',
     &    LCTLF(1:lnblnk(LCTLF))
        call edisp(itru,outs)
      endif
      call edisp(itru,' ')

      call edisp(itru,'Simulation parameters:')

      write(outs,2)isd1,ism1
    2 format('Period from day',i3,' of month',i3)
      call edisp(itru,outs)
      write(outs,93)isd2,ism2,ntstep
   93 format(10x,'to day',i3,' of month',i3,' (',
     &i3,'time-step(s)/hr).')
      call edisp(itru,outs)

      if(isave.eq.1)then
        call edisp(itru,'Save option 1 (minimum).')
      elseif(isave.eq.2)then
        call edisp(itru,'Save option 2 (moderate).')
      elseif(isave.eq.3)then
        call edisp(itru,'Save option 3 (maximum).')
      elseif(isave.eq.4)then
        call edisp(itru,'Save option 4 (energy balance).')
      elseif(isave.eq.5)then
        call edisp(itru,'Save option 5 (Hot3000).')
      endif

      call edisp(itru,'Control description:')
      call edisp248(itru,ctldoc,72)

      write(outs,94)ncf
   94 format('Control functions = ',i2)
      call edisp(itru,outs)

      write(outs,95)ncl
   95 format('Control loops = ',i2)
      call edisp(itru,outs)

      if(ncf.eq.0)goto 12

      call edisp(itru,' Control    Associated ')
      call edisp(itru,' function   zone ')
      do 10 i=1,ncf
        n=0
        ic=0
        do 20 j=1,ncomp
          if(icascf(j).eq.i)goto 14
          goto 20
   14     n=n+1
          ic=ic+1
          io(ic)=j
   20   continue
   10 continue
      if(n.eq.0)then 
        call edisp(itru,'         None')
      else
        if(n.gt.24)then   ! If more than 24 associated write 2 lines.
          write(outs,16)i,(io(j),j=1,24)
          call edisp(itru,outs)
          write(outs,16)i,(io(j),j=25,n)
          call edisp(itru,outs)
        else
          write(outs,16)i,(io(j),j=1,n)
          call edisp(itru,outs)
        endif
   16   format(i5,7x,24(i2,','))
      endif

   12 return
      end


C ******************** CFDTOG ********************
C Set CFD coupling toggles.

      subroutine CFDTOG
#include "building.h"
#include "cfd.h"
#include "help.h"

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/CFDTOGS/ACCON,CFD2BLD,CFD2MFS
      COMMON/GGDH/GENB1(ntcelx,ntcely,ntcelz),GGDH
      COMMON/RFNGRD/DOFLT,DORFN,IFNDWT(MNZ),NRFN,IRFNCLS(MNZ,MRFN,3),
     &  NRFND,IRFND(MNZ,MRFNT,3)
      logical DOFLT,DORFN

      integer ACCON,CFD2BLD,CFD2MFS
      logical GGDH

      CHARACTER ITEMG(10)*40 
      integer NITMS,INO  ! max items and current menu item

      helpinsub='simcon'  ! set for cfiles
      
 10   INO=-4
      ITEMG(1)=    ' ------------ CFD OPTIONS ------------- '
      if(ACCON.eq.1)then
        ITEMG(2)=  'a adaptive turbulence model >> On       '
      elseif(ACCON.eq.0)then
        ITEMG(2)=  'a adaptive turbulence model >> Off      '
      else
        call edisp(iuout,'Unrecognized value of ACCON')
        return
      endif
      if(ACCON.eq.1)then
        if(CFD2BLD.eq.1)then
          ITEMG(3)='b allow CFD to set air-surf. flux >> No '
        elseif(CFD2BLD.eq.2)then
          ITEMG(3)='b allow CFD to set air-surf. flux >> Yes'
        else
          call edisp(iuout,'Unrecognized value of CFD2BLD')
          return
        endif
      else
        ITEMG(3)=  'b allow CFD to set air-surf. flux >> NA '
      endif
      if(CFD2MFS.eq.1)then
        ITEMG(4)=  'c allow CFD-MFS iteration >> No         '
      elseif(CFD2MFS.eq.2)then
        ITEMG(4)=  'c allow CFD-MFS iteration >> Yes        '
      else
        call edisp(iuout,'Unrecognized value of CFD2MFS')
        return
      endif
      if (GGDH) then
        ITEMG(5)=  'd general gradient diffusion hyp. >> On '
      else
        ITEMG(5)=  'd general gradient diffusion hyp. >> Off'
      endif
      if (DOFLT) then
        ITEMG(6)=  'e residual flatline detection >> On     '
      else
        ITEMG(6)=  'e residual flatline detection >> Off    '
      endif
      if (DORFN) then
        ITEMG(7)=  'e automatic grid refinement >> On       '
      else
        ITEMG(7)=  'e automatic grid refinement >> Off      '
      endif
      ITEMG(8)=    '  ________________________________      '
      ITEMG(9)=    '? help                                  '
      ITEMG(10)=   '- exit menu                             '
      NITMS=10

C Help text for this menu.
      helptopic='cfd_coupling_toggles'
      call gethelptext(helpinsub,helptopic,nbhelp)

      CALL EMENU('CFD coupling toggles',ITEMG,NITMS,INO)

      if(INO.eq.2)then
        IF(ACCON.eq.1)then
C Turn ACC off.
          ACCON=0
        else
C Turn ACC on.
          ACCON=1
        endif
      elseif((INO.eq.3).and.(ACCON.eq.1))then
        if(CFD2BLD.eq.1)then
C 2-way CFD-BLD conflation.
          CFD2BLD=2
        else
C 1-way CFD-BLD conflation.
          CFD2BLD=1
        endif
      elseif(INO.eq.4)then
        if(CFD2MFS.eq.2)then
C 1-way CFD-MFS conflation.
          CFD2MFS=1
        else
C 2-way CFD-MFS conflation.
          CFD2MFS=2
        endif
      elseif(INO.eq.5)then
        if(.not.GGDH)then
C GGDH on.
          GGDH=.TRUE.
        else
C GGDH off.
          GGDH=.FALSE.
        endif
      elseif(INO.eq.6)then
        if(.not.DOFLT)then
C Residual flatline detection on.
          DOFLT=.TRUE.
        else
C Residual flatline detection off.
          DOFLT=.FALSE.
        endif
      elseif(INO.eq.7)then
        if(.not.DORFN)then
C Grid refinement on.
          DORFN=.TRUE.
        else
C Grid refinement off.
          DORFN=.FALSE.
        endif
      elseif(INO.eq.(NITMS-1))then
C Help.
        helptopic='cfd_coupling_toggles'
        call gethelptext(helpinsub,helptopic,nbhelp)
        CALL PHELPD('CFD coupling toggles',nbhelp,'-',0,0,IER)
      elseif(INO.eq.NITMS)then
C Return to simulation toggles menu.
        return
      endif

C Redisplay menu.
      goto 10
      end
 
