C This file is part of the ESP-r system.
C Copyright Energy Systems Research Unit, University of
C Strathclyde, Glasgow Scotland, 2001.

C ESP-r is free software. You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation
C (Version 2 or later).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.

C ***************** SPMMOD *****************
C This subroutine calls the relevant subroutine for each special
C material (SM) identified in SPMINIT. SMs are components
C that are considered to be an integral part of the building fabric.
C Two types are available: renewable energy components for the
C production of heat and/or power; and active components such as
C phase change materials and thermo-, photo- and electro-chromic
C glazing. The following SMs are available.

C Renewable energy components:
C SPMCMP1  - crystalline silicon solar cell model with power output
C            determined as a function of absorbed solar flux and cell
C            temperature;
C SPMCMP2  - constant efficiency solar cell model with power output
C            determined as a function of absorbed solar flux and a
C            user-specifed efficiency;
C SPMCMP3  - ducted wind turbine model;
C SPMCMP4  - solar thermal collector model;
C SPMCMP5  - a one diode PV model.
C SPMCMP6  - free-standing wind turbine model;

C Active material components:
C SPMCMP51 - thermo-chromic glazing;
C SPMCMP52 - saturated surface with evaporation;
C SPMCMP53 - phase change material model 1;
C SPMCMP54 - phase change material model 2;
C SPMCMP55 - phase change material model 3;
C SPMCMP56 - phase change material model 4 (with sub-cooling);
C SPMCMP57 - phase change material model 5 (with sub-cooling).

      subroutine spmmod(icomp)
#include "building.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)

      character spflnam*72

C For each node with a special material, identify the type as held
C in ispmtyp(mspmnod,1). If the affected zone matches the current
C zone then call the relevant subroutine.
      if(ispmxist.gt.0)then
         do i=1,nspmnod
            ispmnod=i
            ispmtype=ispmtyp(i,1)
            if((ispmloc(i,1)).eq.icomp)then
               if(ispmtype.eq.1)then
                  call spmcmp1(icomp,ispmnod)
               elseif(ispmtype.eq.2)then
                  call spmcmp2(icomp,ispmnod)
               elseif(ispmtype.eq.3)then
                  call spmcmp3(icomp,ispmnod)
               elseif(ispmtype.eq.4)then
                  call spmcmp4(icomp,ispmnod)
               elseif(ispmtype.eq.5)then
                  call spmcmp5(icomp,ispmnod)
               elseif(ispmtype.eq.6)then
                  call spmcmp6(icomp,ispmnod)
               elseif(ispmtype.eq.51)then
                  call spmcmp51(icomp,ispmnod)
               elseif(ispmtype.eq.52)then
                  call spmcmp52(icomp,ispmnod)
               elseif(ispmtype.eq.53)then
                  call spmcmp53(icomp,ispmnod)
               elseif(ispmtype.eq.54)then
                  call spmcmp54(icomp,ispmnod)
               elseif(ispmtype.eq.55)then
                  call spmcmp55(icomp,ispmnod)
               elseif(ispmtype.eq.56)then
                  call spmcmp56(icomp,ispmnod)
               elseif(ispmtype.eq.57)then
                  call spmcmp57(icomp,ispmnod)
               endif
            endif
         enddo
      endif
      return
      end

C ***************** SPMCMP1 *****************
C A crystalline silicon solar cell model (after Nick Kelly) with
C efficiency determined as a function of cell temperature. Defining
C data as read from spmdat:
C    1 - open circuit voltage (volts);
C    2 - short circuit current (amps);
C    3 - voltage at maximum power point;
C    4 - current at maximum power point;
C    5 - reference insolation level (W/m^2);
C    6 - reference temperature (K);
C    7 - number of series connected cells (not panels)(-);
C    8 - number of parallel connected branches (-);
C    9 - number of panels in surface (-);
C   10 - empirical value used in calculation of Io;
C   11 - load type (0-maximum power point,
C                   1-fixed resistance,
C                   2-fixed voltage);
C   12 - fixed voltage (volts) or resistance (ohms) value;
C   13 - shading effects:
C          0 - as 1, retained for backward compatibility (default);
C          1 - proportional power loss based on calculated incident
C              radiation, which includes full shading effects if
C              obstructions are defined;
C          2 - total power loss, solar flux set to zero to give zero
C              power output;
C          3 - direct radiation power loss, the incident flux is set
C              to the diffuse component only (the power output of all
C              cells falls to that of the shaded cells).
C   14 - miscellaneous loss factor, which acts to derates the power
C        produced by the panel. Such losses may be attributed to
C        uncertainty in the manufacturer's rating, panel ageing,
C        maintenance, snow cover, blocked diodes and wiring etc.

C Reference:
C Kelly N (1998), 'Towards a Design Environment for Building-Integrated
C Energy Systems: The Integration of Electrical Power Flow Modelling
C with Building Simulation', PhD Thesis, ESRU, University of Strathclyde,
C Glasgow, UK.

      subroutine spmcmp1(icomp,ispmnod)
#include "building.h"
#include "geometry.h"
#include "plant.h"
#include "power.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)
      common/fvalc/tfc(mcom,ms,mn),qfc(mcom)
      common/coe32/qsoli(ms,2),qsole(ms,2)
      common/coe32j/qtmca(ms,mn,2)
      common/pvpnl/tpnl,qpnl,pvpwr
      common/pers/isd1,ism1,isd2,ism2,isds,isdf,ntstep
      common/adjc/ie(mcom,ms),atp(mcom,ms),atf(mcom,ms),
     &            arp(mcom,ms),arf(mcom,ms)
      common/solext/exrad(ms),exdir(ms),exdif(ms),exshad(ms),
     &              exshadf(ms)

      real isc,imp,k,io,il,imod,iiter
      character spmlabel*16,outs*124,spflnam*72,strspm*16,unitspm*16,
     &          txtspm*72

C Explicit definitions.
      REAL VOC,VMP,QREF,TREF,SCELL,PBRCH,EMPV,VALF,
     &E,DF1,DF2,DF,VMP1,VMP2,VMP3,VMP4,
     &VMPMOD,VITER,A1,A2,fMiscLossFactor

      logical bNumsAreClose
      parameter (fSmall=1.0E-05)

C Open circuit voltage.
      voc=spmdat(ispmnod,1)

C Short circuit current.
      isc=spmdat(ispmnod,2)

C Voltage at maximum power point.
      vmp=spmdat(ispmnod,3)

C Current at maximum power point.
      imp=spmdat(ispmnod,4)

C Reference insolation and temperature.
      qref=spmdat(ispmnod,5)
      tref=spmdat(ispmnod,6)

C Number of series connected cells.
      scell=spmdat(ispmnod,7)

C Number of parallel connected branches.
      pbrch=spmdat(ispmnod,8)

C Number of panels in surface.
      npnnls=nint(spmdat(ispmnod,9))

C Empirical value used in calculation of diode current.
      empv=spmdat(ispmnod,10)
      if(empv.lt.1E-10)empv=10.

C Load type.
      ilodt=nint(spmdat(ispmnod,11))

C Fixed voltage or resistance as appropriate.
      valf=spmdat(ispmnod,12)

C Shading treatment.
      ishad=nint(spmdat(ispmnod,13))

C Miscellaneous loss factor, which acts to derates the power
C produced by the PV panel. Such losses may be attributed to
C uncertainty in the manufacturer's rating, panel ageing,
C maintenance issues, snow covering, blocked diodes and wiring, etc.
      fMiscLossFactor=spmdat(ispmnod,14)

C Physical constants: electrical charge and Boltzman constant.
      e=1.60E-19
      k=1.38E-23

C Panel temperature assigned from the associated construction node
C temperature.
      tpnl=tfc(ispmloc(ispmnod,1),ispmloc(ispmnod,2),
     &                                      ispmloc(ispmnod,3))+273.0

C Determine the total solar flux, qpnl, incident on the panel as a
C function of the requested treatment of shading as defined by ishad.

C No shading case.
      qpnl=exrad(ispmloc(ispmnod,2))

C Shading case.
      if((exshad(ispmloc(ispmnod,2)).gt.0.0).or.
     &   (exshadf(ispmloc(ispmnod,2)).gt.0.0))then
         if(ishad.eq.0.or.ishad.eq.1)then
            qpnl=exrad(ispmloc(ispmnod,2))
         elseif(ishad.eq.2)then
            qpnl=0.0
         elseif(ishad.eq.3)then
            qpnl=exdif(ispmloc(ispmnod,2))
         endif
      endif

C Diode factor.
      df1=e*(vmp-voc)/(k*tref*scell)
      df2=log((isc-imp)/isc)
      df=df1/df2

C Diffusion current.
      io=(isc/pbrch)/(exp(((e*voc/scell)/(k*df*tref)))-1.)*
     &                                        (2**((tpnl-tref)/empv))

C Light generated current.
      if(qpnl.lt.0.0)qpnl=1E-10
      il=(qpnl/qref)*(isc/pbrch)

C Maximum power point tracking, use iteration to determine the maximum
C output voltage, vmpmod.
      if(ilodt.eq.0)then
         viter=0.4
         iter=1
    2    vmp1=(k*tpnl*df/e)
         vmp2=((il/io)+1.)
         vmp3=1.+(e*viter)/(k*tpnl*df)
         vmp4=log(vmp2/vmp3)
         vmpmod=vmp1*vmp4
         iter=iter+1
         if(abs(viter-vmpmod).gt.0.05.and.iter.lt.101)then
            viter=abs(vmpmod)
            goto 2
         elseif(iter.ge.101)then
            call edisp(iuout,' ')
            write(outs,*)
     &           'SPMCMP1 fatal error: panel voltage solution failed!'
            call edisp(iuout,outs)
            stop
         endif

C Panel power output.
         a1=vmpmod*io*(exp((e*vmpmod)/(k*tpnl*df))-1.)
         a2=vmpmod*il
         pvpwr=(a1-a2)*scell*pbrch*npnnls*(1.-fMiscLossFactor)
         if(pvpwr.gt.0.0)then
            pvpwr=0.0
         else
            pvpwr=-pvpwr
         endif
         vpv=vmpmod*scell  !- panel voltage

C Fixed resistance load, use iteration to calculate panel current.
      elseif(ilodt.eq.1)then
         valf=valf/scell
         iiter=0.999*il
         iter=1
    3    a1=(df*k*tpnl)/(e*valf)
         a2=(1.+(il-iiter)/io)
         if(iiter.gt.il)a2=(1.+(iiter-il)/io)
         imod=a1*log(a2)
         iter=iter+1
         if(abs(iiter-imod).gt.0.05.and.iter.lt.101)then
            if(iter.gt.50)then
               iiter=iiter+(imod-iiter)*0.25
            else
               iiter=(imod+iiter)/2.
            endif
            goto 3
         elseif(iter.ge.101)then
           call edisp(iuout,' ')
           write(outs,*)
     &           'SPMCMP1 fatal error: panel current solution failed!'
           call edisp(iuout,outs)
           stop
         endif

C Check that iteration has converged.
         a1=io*(exp((e*imod*valf)/(k*tpnl*df))-1.)
         a2=il

C Check the magnitudes of the diode and light generated currents; if
C id > il set the power output to zero.
         pvpwr=(a1-a2)**2*valf*scelL*pbrch*npnnls*(1.- fMiscLossFactor)
         if(a1.gt.a2)then
            pvpwr=0.0
         else
            pvpwr=pvpwr
         endif
         vpv=(pvpwr*valf)**0.5

C Fixed voltage type load (e.g. a battery), calculate panel power.
      elseif(ilodt.eq.2)then
         valf=valf/scell
         pvpwr=valf*(io*(exp(e*valf/(k*tpnl*df))-1.)-il)
         a1=io*(exp(e*valf/(k*tpnl*df))-1.)
         a2=il
         pvpwr=pvpwr*scell*pbrch*npnnls*(1.- fMiscLossFactor)
         if(pvpwr.gt.0.0)then
            pvpwr=0.0
         else
            pvpwr=-pvpwr
         endif
         vpv=valf*scell  !- panel voltage
      endif

C Results output.
      sar=sna(ispmloc(ispmnod,1),ispmloc(ispmnod,2))
      ndatspm(ispmnod)=2
      dataspm(ispmnod,1)=pvpwr
      strspm(ispmnod,1)='Power (W)'
      call eclose(qpnl,0.,fSmall,bNumsAreClose)
      if(bNumsAreClose)then
         dataspm(ispmnod,2)=0.
      else
         dataspm(ispmnod,2)=((pvpwr/sar)/qpnl)*100.
      endif
      strspm(ispmnod,2)='Efficiency (%)'

C As a temporary measure, use zone-related variables, zspmf1 and zspmf2,
C for results library storage. Output will be sum of special materials
C per zone.
      if (ispmnod.eq.1) then 
         do i=1,mcom
         zspmf1(i)=0.0
         zspmf2(i)=0.0
         enddo
      endif
      zspmf1(ispmloc(ispmnod,1))=
     &  zspmf1(ispmloc(ispmnod,1))+dataspm(ispmnod,1)
      zspmf2(ispmloc(ispmnod,1))=
     &  zspmf2(ispmloc(ispmnod,1))+dataspm(ispmnod,2)

C Save panel power output (for load flow simulation) and the current
C panel voltage. (This could later be changed to a power converter
C component that can vary the values of P and Q as generated by the
C panel.)
      pspm(ispmnod)=pvpwr
      vspm(ispmnod)=cmplx(vpv,0.0)
      qspm(ispmnod)=0.0

C Convert panel power output from W to W/m^2.
      pvpwrsq=pvpwr/sna(ispmloc(ispmnod,1),ispmloc(ispmnod,2))

C Save original nodal solar absorptions for trace output.
      qsole1=qsole(ispmloc(ispmnod,2),2)
      qtmca1=qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)

C Alter the nodal solar absorption according to whether node is
C transparent (0) or opaque (1).
      if(ispmtyp(ispmnod,2).gt.0)then

C Opaque layer.
         if(ie(ispmloc(ispmnod,1),ispmloc(ispmnod,2)).eq.0)then
            qsole(ispmloc(ispmnod,2),2)=qsole(ispmloc(ispmnod,2),2)
     &                                                       -pvpwrsq
         else
            qsoli(ispmloc(ispmnod,2),2)=qsoli(ispmloc(ispmnod,2),2)
     &                                                       -pvpwrsq
         endif
      else

C Transparent layer, modify the solar flux absorption of the middle
C node and the two boundary nodes.
         qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)=
     &    qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)-(pvpwrsq/2.)
         qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)+1,2)=
     &    qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)+1,2)-(pvpwrsq/4.)
         qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)-1,2)=
     &    qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)-1,2)-(pvpwrsq/4.)
      endif

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP1 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,2x,i4,2x,i4,2x,i4)')
     &        'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &         ispmloc(ispmnod,2),ispmloc(ispmnod,3)
      call edisp(itu,outs)
      write(outs,'(a,i3)') 'Number of panels in surface: ',npnnls
      call edisp(itu,outs)
      if(ispmtyp(ispmnod,2).gt.0)then
         write(outs,'(a,1x,f10.2)') 'Original flux absorption: ',qsole1
         call edisp(itu,outs)
         write(outs,'(a,1x,f10.2)') 'Modified flux absorption: ',
     &                                    qsole(ispmloc(ispmnod,2),2)
         call edisp(itu,outs)
      else
         write(outs,'(a,1x,f10.2)') 'Original flux absorption: ',qtmca1
         call edisp(itu,outs)
         write(outs,'(a,1x,f10.2)') 'Modified flux absorption: ',
     &                 qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)
         call edisp(itu,outs)
      endif
      write(outs,'(a,1x,f10.2)') 'Panel power output: ', pvpwr
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Panel temperature: ',tpnl
      call edisp(itu,outs)
  999 return
      end

C ***************** SPMCMP2 *****************
C A crystalline silicon solar cell model (after Nick Kelly), with
C electrical power output based on a user-specified efficiency.
C Defining data items read from spmdat:
C    1 - nominal panel efficiency (%).

      subroutine spmcmp2(icomp,ispmnod)
#include "building.h"
#include "geometry.h"
#include "plant.h"
#include "power.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)
      common/fvalc/tfc(mcom,ms,mn),qfc(mcom)
      common/coe32j/qtmca(ms,mn,2)
      common/coe32/qsoli(ms,2),qsole(ms,2)
      common/pvpnl/tpnl,qpnl,pvpwr
      common/adjc/ie(mcom,ms),atp(mcom,ms),atf(mcom,ms),
     &arp(mcom,ms),arf(mcom,ms)
      common/pers/isd1,ism1,isd2,ism2,isds,isdf,ntstep
      common/solext/exrad(ms),exdir(ms),exdif(ms),exshad(ms),
     &                                                    exshadf(ms)

      real neff
      character spmlabel*16, outs*124,spflnam*72,strspm*16,unitspm*16,
     &          txtspm*72

C Nominal efficiency.
      neff=spmdat(ispmnod,1)/100.

C Solar radiation absorbed by panel.
      qpnl=exrad(ispmloc(ispmnod,2))

C Power generated by panel.
      pvpwr=qpnl*neff*sna(ispmloc(ispmnod,1),ispmloc(ispmnod,2))

C Results output.
      ndatspm(ispmnod)=2
      dataspm(ispmnod,1)=pvpwr
      strspm(ispmnod,1)='Power (W)'
      dataspm(ispmnod,2)=(pvpwr/qpnl)*100.
      strspm(ispmnod,2)='Efficiency (%)'

C As a temporary measure, use zone-related variables, zspmf1 and zspmf2,
C for results library storage. Output will be sum of special materials
C per zone.
      if (ispmnod.eq.1) then 
        do i=1,mcom
          zspmf1(i)=0.0
          zspmf2(i)=0.0
        enddo
      endif
      zspmf1(ispmloc(ispmnod,1))=
     &  zspmf1(ispmloc(ispmnod,1))+dataspm(ispmnod,1)
      zspmf2(ispmloc(ispmnod,1))=
     &  zspmf2(ispmloc(ispmnod,1))+dataspm(ispmnod,2)

C Save panel power output (for load flow simulation) and the current
C panel voltage. (This could later be changed to a power converter
C component that can vary the values of P and Q as generated by the
C panel.)
      vspm(ispmnod)=cmplx(220.,0.0)
      pspm(ispmnod)=pvpwr
      qspm(ispmnod)=0.0

C Convert the panel power output from W to W/m^2.
       pvpwrsq=pvpwr/sna(ispmloc(ispmnod,1),ispmloc(ispmnod,2))

C Save original nodal absorptions for trace output.
      qsole1=qsole(ispmloc(ispmnod,2),2)
      qtmca1=qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)

C Alter the nodal absorption according to whether node is
C transparent (0) or opaque (1).

C Opaque layer.
      if(ispmtyp(ispmnod,2).gt.0)then
         if(ie(ispmloc(ispmnod,1),ispmloc(ispmnod,2)).eq.0)then
            qsole(ispmloc(ispmnod,2),2)=qsole(ispmloc(ispmnod,2),2)
     &                                                       -pvpwrsq
         else
            qsoli(ispmloc(ispmnod,2),2)=qsoli(ispmloc(ispmnod,2),2)
     &                                                       -pvpwrsq
         endif
      else

C Transparent layer: modify the solar flux absorption of the middle
C node and the two boundary nodes.
         qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)=
     &    qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)-(pvpwrsq/2.)
         qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)+1,2)=
     &    qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)+1,2)-(pvpwrsq/4.)
         qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)-1,2)=
     &    qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)-1,2)-(pvpwrsq/4.)
      endif

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP2 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,2x,i4,2x,i4,2x,i4)')
     &        'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &         ispmloc(ispmnod,2),ispmloc(ispmnod,3)
      call edisp(itu,outs)
C      write(outs,'(a,i3)') 'Number of panels in surface: ',npnnls
      call edisp(itu,outs)
      if(ispmtyp(ispmnod,2).gt.0)then
         write(outs,'(a,1x,f10.2)') 'Original flux absorption: ',qsole1
         call edisp(itu,outs)
         write(outs,'(a,1x,f10.2)') 'Modified flux absorption: ',
     &                                    qsole(ispmloc(ispmnod,2),2)
         call edisp(itu,outs)
      else
         write(outs,'(a,1x,f10.2)') 'Original flux absorption: ',qtmca1
         call edisp(itu,outs)
         write(outs,'(a,1x,f10.2)') 'Modified flux absorption: ',
     &                 qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)
         call edisp(itu,outs)
      endif
      write(outs,'(a,1x,f10.2)') 'Panel power output: ', pvpwr
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Panel temperature: ',tpnl
      call edisp(itu,outs)
  999 return
      end

C ***************** SPMCMP3 *****************
C A model that calculates the electrical power output from a ducted
C wind turbine (after Nick Kelly and Andy Grant). The model includes
C a statistical manipulation of the mean wind velocity in order to
C estimate the power output. Defining data as read from spmdat:
C    1 - zone containing outlet (-)
C    2 - surface containing outlet (-)
C    3 - outlet ourface type (-)
C    4 - inlet azimuth angle (degrees)
C    5 - outlet azimuth angle (degrees)
C    6 - height of turbine inlet (m)
C    7 - turbine cross sectional area (m^2)
C    8 - duct velocity coefficient (-)
C    9 - cut-in wind speed (m/s)
C   10 - number of turbines in the surface (-)
C   11 - location (1-urban, 2-suburban, 3-rural)
C   12 - reference height (m)
C   13 - wind speed statistics (on/off)
C   14 - wind speed/turbulence profile (on/off)
C   15 - default turbulence intensity (-)

      subroutine spmcmp3(icomp,ispmnod)
#include "building.h"
#include "plant.h"
#include "net_flow.h"
#include "esprdbfile.h"
#include "power.h"

C N.B. All parameters comply with the Fortran implicit naming
C convention except where explicitly redefined.
      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/filep/ifil
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/pers/isd1,ism1,isd2,ism2,isds,isdf,ntstep
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)
      common/ffn/iflwn,icffs(mpcon)
      common/mflwpr/npre,fpre(mpos,mprd)
      common/afn/iairn,laprob,icaas(mcom)
      common/mflclm/dryb,qdif,qdnr,irvh,wdir,wspd,wred
      common/climi/qfp,qff,tp,tf,qdp,qdf,vp,vf,dp,df,hp,hf

      real minu,maxu,minv,maxv,inc
      logical unixok

      character spmlabel*16,outs*124,spflnam*72,laprob*72,strspm*16,
     &          unitspm*16,txtspm*72,lworking*144,fs*1

C Set folder separator (fs) to \ or / as required.
      call isunix(unixok)
      if(unixok)then
        fs = char(47)
      else
        fs = char(92)
      endif

C Component data.
C      inzone=ispmloc(ispmnod,1)
C      insurf=ispmloc(ispmnod,2)
      insurft=ispmloc(ispmnod,3)
C      iozone=nint(spmdat(ispmnod,1))
C      iosurf=nint(spmdat(ispmnod,2))
      iosurft=nint(spmdat(ispmnod,3))
      aziin=spmdat(ispmnod,4)
      aziou=spmdat(ispmnod,5)
      hdwti=spmdat(ispmnod,6)
      tarea=spmdat(ispmnod,7)
      dvcoeff=spmdat(ispmnod,8)
      ciwspd=spmdat(ispmnod,9)
      nturb=nint(spmdat(ispmnod,10))
      iloc=nint(spmdat(ispmnod,11))
      refh=spmdat(ispmnod,12)
      iwstat=nint(spmdat(ispmnod,13))
      iprof=nint(spmdat(ispmnod,14))
      turbi=spmdat(ispmnod,15)

      pi=4.0*atan(1.0)

C Current weather data.
      widir=df
      drybt=tf
      wispd=vf
      if(wispd.lt.1E-3)wispd=1E-3

C If no mass flow file exists then open wind pressure coefficients
C database, read coefficient sets and close.
      if(iairn.eq.0.and.iflwn.eq.0.and.nsinc.lt.2)then
         iunit=ifil+1
         if(ipathapres.eq.0.or.ipathapres.eq.1)then
            call efopseq(iunit,lapres,1,ier)
         elseif(ipathapres.eq.2)then
            lndbp=lnblnk(standarddbpath)
            write(lworking,'(3a)') standarddbpath(1:lndbp),fs,
     &      lapres(1:lnblnk(lapres))
            call efopseq(iunit,lworking,1,ier)
         endif
         if(ier.gt.0)then
            call edisp(iuout,
     &         'SPMCMP3 fatal error: when opening pressure coeff. db!')
            stop
         endif
         read(iunit,*,iostat=istat)npre
         do j=1,npre
            read(iunit,*,iostat=istat)(fpre(i,j),i=1,mpos)
         enddo
         call erpfree(iunit,istat)
      endif

C Calculate the wind velocity prevailing at the height of the ducted
C wind turbine. This information will be available if an air flow
C network exists. If profiles or speed reduction are not specified
C then use default values.
      if(wred.gt.1E-6.and.iprof.lt.1)then
         wispd=wispd*wred
      elseif(iprof.gt.1)then

C Urban location.
         if(iloc.eq.1)then
            pwr=0.32
            rough=5.

C Suburban location
         elseif(iloc.eq.2)then
            pwr=0.23
            rough=0.8

C Rural location
         elseif(iloc.eq.3)then
            pwr=0.14
            rough=0.05
         endif

C Set the new wind speed and turbulent intensity from profiles. If
C height is zero, set to a small value to avoid numerical problems.
         if(hdwti.lt.1E-6)hdwti=1E-6
         if(refh.lt.1E-6)refh=1E-6
         wispd=wispd*(hdwti/refh)**pwr
         turbi=(rough*log(30./refh))/log(hdwti/refh)
      endif

      if(iwstat.gt.0)then

C Wind velocity components.
         compu=wispd*sin((widir*pi)/180.)
         compv=wispd*cos((widir*pi)/180.)

C Standard deviation, based on turbulent intensity.
         stdev=turbi*wispd

C Wind velocity ranges.
         minu=compu-3*stdev
         maxu=compu+3*stdev
         minv=compv-3*stdev
         maxv=compv+3*stdev

C Calculate the increment.
         inc=6*stdev/10.
         icount=1
         pwrtot=0.

C Set starting value of U velocity component.
         valu=minu

C Set up trace probability counters.
         tpru=0
C         tprc=0

C Loop to generate power output.
         do 66 i=1,10

C Current U velocity component probability density.
            pdu=(1/(stdev*(2*pi)**0.5))*
     &                       exp(-(((valu-compu)**2.)/(2*stdev**2.)))
            tpru=tpru+pdu*inc
            tprv=0

C Set starting value of V velocity component.
            valv=minv

            do 77 j=1,10

C Current V velocity component probability density.
               pdv=(1/(stdev*(2*pi)**0.5))*
     &                       exp(-(((valv-compv)**2.)/(2*stdev**2.)))
               tprv=tprv+pdv*inc

C Combined probability.
               prc=(inc*pdu)*(inc*pdv)

C Duration of current wind speed and direction.
               tsl=3600./float(ntstep)
               tdur=prc*tsl

C New wind speed and direction.
               wispdm=(valu**2.+valv**2.)**0.5
               if(abs(valv).lt.1E-6)valv=1E-6
               widirm=atan(valu/valv)*(180./pi)
               if(valv.ge.0)then
                  if(valu.ge.0.)then
                     widirm=widirm
                  else
                     widirm=360.+widirm
                  endif
               else
                  if(valu.ge.0.)then
                     widirm=180.+widirm
                  else
                     widirm=180.+widirm
                  endif
               endif

C Check that the wind is strong enough for the turbine to cut in.
               if(wispdm.ge.ciwspd)then

C Check that the component of the wind speed incident on the turbine
C is greater than the cut-in wind speed. If not, set the wind speed
C to zero for zero power output.
                  angi=widirm-aziin
                  if(abs(angi).ge.180.)angi=360.-abs(angi)
                  if(abs(angi).ge.90.)then
                     wispdm=0.
                  else
                     gamma=abs(angi)
                     vpd=wispdm*cos((gamma*pi)/180.)
                     if(vpd.lt.ciwspd)then
                        wispdm=0.
                     else
                        continue
                     endif
                  endif
               else
                  wispdm=0.
               endif

C Calculate the differential pressure coefficient and set the wind
C direction index (1-16 corresponds to intervals with mid-points
C at 0, 22.5, ..., 337.5 degrees relative to the surface azimuth,
C i.e. index 1 = wind normal to surface).
               iwdr1=nint((widirm-aziin)/22.5)+1
               if(iwdr1.lt.1)iwdr1=iwdr1+16
               if(iwdr1.gt.16)iwdr1=iwdr1-16
               iwdr2=nint((widirm-aziou)/22.5)+1
               if(iwdr2.lt.1)iwdr2=iwdr2+16
               if(iwdr2.gt.16)iwdr2=iwdr2-16

C Calculate wind pressure and include stack pressure based on
C P = 0 and height = 0.
               cd1=fpre(iwdr1,insurft)
               cd2=fpre(iwdr2,iosurft)
               delta=cd1-cd2

C Power output from P={Cv/[3.3^(1/2)]}.rho.A.d^(2/3).V^3
               ro=densit(1,drybt)
               pwr=(dvcoeff/(3*3**(0.5)))*ro*tarea*
     &                             (delta**(0.667))*(wispdm**3)*nturb
               icount=icount+1
               valv=valv+inc
               pwrtot=pwrtot+pwr*tdur
   77       continue
            valu=valu+inc
   66    continue

C Write results for probability-based power output.
         pspm(ispmnod)=pwrtot/tsl

      else
         if(wispd.ge.ciwspd)then

C Check that the component of the wind speed incident on the turbine
C is greater than the cut-in wind speed. If not, set to zero for
C zero power output.
            angi=widir-aziin
            if(abs(angi).ge.180.)angi=360.-abs(angi)
            if(abs(angi).ge.90.)then
               wispd=0.
            else
               gamma=abs(angi)
               vpd=wispd*cos((gamma*pi)/180.)
               if(vpd.lt.ciwspd)then
                  wispd=0.
               else
                  continue
               endif
            endif
         else
            wispd=0.
         endif

C Calculate the differential pressure coefficient and set the wind
C direction index (1-16 corresponds to intervals with mid-points
C at 0, 22.5, ..., 337.5 degrees relative to the surface azimuth
C (i.e. index 1 = wind normal to surface)
         iwdr1=nint((widir-aziin)/22.5)+1
         if(iwdr1.lt.1)iwdr1=iwdr1+16
         if(iwdr1.gt.16)iwdr1=iwdr1-16
         iwdr2=nint((widir-aziou)/22.5)+1
         if(iwdr2.lt.1)iwdr2=iwdr2+16
         if(iwdr2.gt.16)iwdr2=iwdr2-16

C Calculate wind pressure and include stack pressure based on
C P = 0 for height=0.
         cd1=fpre(iwdr1,insurft)
         cd2=fpre(iwdr2,iosurft)
         delta=cd1-cd2

C Power output from P={Cv/[3.3^(1/2)]}.rho.A.d^(2/3).V^3
         ro=densit(1,drybt)
         pspm(ispmnod)=(dvcoeff/(3*3**(0.5)))*ro*tarea*
     &                              (delta**(0.667))*(wispd**3)*nturb
      endif

C Results output.
      ndatspm(ispmnod)=1
      dataspm(ispmnod,1)=pspm(ispmnod)
      strspm(ispmnod,1)='Power (W)'

C As a temporary measure, use zone-related variable, zspmf1, for
C results library storage. Output is limited to one special material
C per zone.
       zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP3 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Pressure at inlet: ',cd1
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Pressure at outlet: ',cd2
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Pressure difference: ',delta
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Power output: ',pspm(ispmnod)
      call edisp(itu,outs)
  999 return
      end

C  ***************** SPMCMP4 *****************
C A model to calculate the heat flux recovered from a flat-plate
C solar collector. Defining data as read from spmdat:
C    1 - water flow rate (l/s);
C    2 - water inlet temperature (degC);
C    3 - number of tubes (-);
C    4 - tube diameter (m);
C    5 - tube length (m).

C The model has been subjected to rudimentary verification.

      subroutine spmcmp4(icomp,ispmnod)
#include "building.h"
#include "geometry.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      common/spvst/spmvstr(mspmnod,5)
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)
      common/adjc/ie(mcom,ms),atp(mcom,ms),atf(mcom,ms),arp(mcom,ms),
     &            arf(mcom,ms)
      common/fvalc/tfc(mcom,ms,mn),qfc(mcom)
      common/coe32/qsoli(ms,2),qsole(ms,2)
      common/coe32j/qtmca(ms,mn,2)
      common/pvpnl/tpnl,qpnl,pvpwr
      common/pers/isd1,ism1,isd2,ism2,isds,isdf,ntstep

      real mu
      character spmlabel*16, outs*124,spflnam*72,strspm*16,unitspm*16,
     &          txtspm*72

      pi=4.0*atan(1.0)

C Volume flow rate.
      vfr=spmdat(ispmnod,1)/1000.

C Inlet temperature.
      tin=spmdat(ispmnod,2)

C Outlet temperature.
      tout=spmvstr(ispmnod,1)
      if(abs(tout).lt.1E-6)tout=tin

C Number of tubes.
      ntb=nint(spmdat(ispmnod,3))

C Tube diameter.
      tdia=spmdat(ispmnod,4)

C Tube length.
      tlen=spmdat(ispmnod,5)

C Internal surface area.
      area=pi*tdia*tlen*ntb

C Temperature of the backplate.
      tcol=tfc(ispmloc(ispmnod,1),ispmloc(ispmnod,2),
     &                                           ispmloc(ispmnod,3))

C Iteration counter.
      iter=1
   20 continue

C Average fluid temperature.
      dt1=tcol-tin
      dt2=tcol-tout
      if(dt1.le.0..or.abs(dt1-dt2).lt.1E-6)then
         tave=tout+tin/2.
      else
         if(dt2.le.0.)then
            dt2=1E-6
         endif
         tave=(dt1-dt2)/log(dt1/dt2)
      endif
      if(tave.le.0.0)tave=0.0
      if(tave.ge.100.)tave=100.

C Fluid density.
      den=1001.1-0.082*tave-0.0036*tave**2.

C Velocity through tubes.
      vel=vfr/(ntb*pi*(tdia**2)/4.)

C Viscosity.
      visc=0.17444E-02-.15885E-03*tave**0.50

C Conductivity.
      cond=((1.12*tave)+569.)/1000.

C Specific heat capacity.
      sht=4244.-(22.65*sqrt(tave))+(1.95*tave)

C Reynold's Number.
      reyno=den*vel*tdia/visc

C Prandtl Number.
      prandt=visc*sht/cond

C Heat transfer coefficient for either laminar or fully developed
C turbulent flow.
       if(reyno.gt.2200.)then
          ffr=(0.79*alog(reyno)-1.64)**(-2.)
          fr=ffr/8.
          hin=(cond/tdia)*(fr*reyno*prandt)/
     &                    (1.07+12.7*(fr**0.5)*((prandt**0.667)-1.0))
       else
          hin=cond*4.4/tdia
       endif

       rt=(den*vfr*sht)/hin*area
       touti=2.*(tcol-rt*(tout-tin))-tin

       if(abs(touti-tout).gt.0.01)then
          if(iter.lt.101)then
             iter=iter+1
             tout=(toutI+tout)/2.
             goto 20
          else
             call edisp(iuout,' ')
             call edisp(iuout,
     &              'SPMCMP4 fatal error: iteration limit exceeded!')
             stop
          endif
       endif

C Save the outlet temperature.
       spmvstr(ispmnod,1)=tout

C Heat flux.
       qtherm=vfr*den*sht*(touti-tin)
       if(qtherm.lt.1E-6)qtherm=0.0

C Results output.
       ndatspm(ispmnod)=1
       dataspm(ispmnod,1)=qtherm
       strspm(ispmnod,1)='Heat output (W)'
       dataspm(ispmnod,2)=touti
       strspm(ispmnod,2)='Outlet temp. (C)'

C As a temporary measure, use zone-related variables, zspmf1 and zspmf2,
C for results library storage. Output is limited to one special material
C per zone.
       zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)
       zspmf2(ispmloc(ispmnod,1))=dataspm(ispmnod,2)

C Save original nodal absorptions for trace output.
       qsole1=qsole(ispmloc(ispmnod,2),2)
       qsoli1=qsoli(ispmloc(ispmnod,2),2)
       qtmca1=qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)

C Convert the panel power output from W to W/m^2.
       qthermqf=qtherm/sna(ispmloc(ispmnod,1),ispmloc(ispmnod,2))
       qthermqp=spmvstr(ispmnod,2)

C Save the power output.
       spmvstr(ispmnod,2)=qthermqf

C Modify thermal output to be an average of future and present values.
       qthermq=(qthermqf+qthermqp)*0.5

C Alter the nodal absorption according to whether node is an external
C (ie=1) or internal (ie=0) surface, and opaque (1) or transparent (0).
       if(ispmtyp(ispmnod,2).gt.0)then
          if(ie(ispmloc(ispmnod,1),ispmloc(ispmnod,2)).eq.0)then
             qsole(ispmloc(ispmnod,2),2)=qsole(ispmloc(ispmnod,2),2)
     &                                                       -qthermq
          else
             qsoli(ispmloc(ispmnod,2),2)=qsoli(ispmloc(ispmnod,2),2)
     &                                                       -qthermq
         endif

C For a transparent layer, modify the solar flux absorption of middle
C node as well as the two boundary nodes.
       else
          qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)=
     &    qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)-(qthermq/2.)
          qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)+1,2)=
     &    qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)+1,2)-(qthermq/4.)
          qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)-1,2)=
     &    qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)-1,2)-(qthermq/4.)
       endif

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP4 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,2x,i4,2x,i4,2x,i4)')
     &        'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &                          ispmloc(ispmnod,2),ispmloc(ispmnod,3)
      call edisp(itu,outs)
      if(ispmtyp(ispmnod,2).gt.0)then
         write(outs,'(a,1x,f10.2)') 'Original flux absorption: ',qsole1
         call edisp(itu,outs)
         write(outs,'(a,1x,f10.2)') 'Modified flux absorption: ',
     &                                    qsole(ispmloc(ispmnod,2),2)
         call edisp(itu,outs)
         write(outs,'(a,1x,f10.2)') 'Original flux absorption: ',qsoli1
         call edisp(itu,outs)
         write(outs,'(a,1x,f10.2)') 'Modified flux absorption: ',
     &                                    qsoli(ispmloc(ispmnod,2),2)
         call edisp(itu,outs)
      else
         write(outs,'(a,1x,f10.2)') 'Original flux absorption: ',qtmca1
         call edisp(itu,outs)
         write(outs,'(a,1x,f10.2)') 'Modified flux absorption: ',
     &                 qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)
         call edisp(itu,outs)
      endif
      write(outs,'(a,1x,f10.2)') 'Thermal output: ',qtherm
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Nodal temperature: ',tcol
      call edisp(itu,outs)
  999 return
      end

C ***************** SPMCMP5 *****************
C A one diode PV model (after Maria Mottillo). The SPMCMP1 model
C is modified to match the Watsun-PV model following recommendations
C by Thevenard. The modifications relate to the method used to
C calculate the output power. Unlike the SPMCMP1 model, this model
C does not cater for PV modules with a fixed resistive load.

C References:
C Thevenard D (2004), 'Literature and Source Code Review of ESP-r's
C Exisiting Photovoltaic Models', NRCan Internal Report 23229-049028/001/SQ.

C Thevenard D (2005), 'Review and Recommendations for Improving the
C Modelling of Building-Integrated Photovoltaic Systems', Proc.
C Building Simulation 2005, pp1221-1228, Montreal, Canada.

C Thevenard D (2007), 'Continued Validation of ESP-r's Equivalent
C One-Diode PV Models', NRCan Internal Report.

C Defining data as read from array spmdat:
C    1 - open circuit voltage at reference conditions (volts);
C    2 - short circuit current at reference conditions (amps);
C    3 - voltage at maximum power point and reference conditions (volts);
C    4 - current at maximum power point and reference conditions (amps);
C    5 - reference insolation (W/m^2);
C    6 - reference temperature (K);
C    7 - coefficient 'alpha' of short circuit current (/K);
C    8 - coefficient 'gamma' of open circuit voltage (/K);
C    9 - coefficient of logarithm of irradiance for open circuit voltage (-);
C   10 - number of series connected cells (NOT modules) (-);
C   11 - number of parallel connected branches (-);
C   12 - number of modules in surface (-);
C   13 - load type (0 - maximum power point,;
C                   1 - fixed voltage);
C   14 - fixed voltage if load type = 1 (volts);
C   15 - shading effects
C          0 - as 1, retained for backward compatibility (default);
C          1 - proportional power loss based on calculated incident
C              radiation, which includes full shading effects if
C              obstructions are defined;
C          2 - total power loss, solar flux set to zero to give zero
C              power output;
C          3 - direct radiation power loss, the incident flux is set
C              to the diffuse component only (the power output of all
C              cells falls to that of the shaded cells).
C   16 - miscellaneous loss factor used to derate the panel power
C        to account for uncertainty in the manufacturer's rating,
C        ageing, soiling, snow cover,blocking diodes and wiring.
C   17 - 21 - optional reference absorption at PV cell layer

      subroutine spmcmp5(icomp,ispmnod)
      use h3kmodule
      implicit none

#include "building.h"
#include "geometry.h"
#include "plant.h"
#include "power.h"

      integer icomp     !- zone number
      integer ispmnod   !- special material node index
      integer lnblnk    !- implicit function returning length of a string

C I/O.
      common/outin/iuout,iuin,ieout
      integer iuout     !- write unit number
      integer iuin      !- read unit number
      integer ieout     !- error unit number

C Special materials.
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)
      integer   ndatspm     !- number of data items for special material
      real      dataspm     !- special component results data
      character strspm*16   !- name string for the data item
      character unitspm*16  !- unit for data item
      character txtspm*72   !- description of data item

      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      real zspmf1
      real zspmf2

      common/spmfxst/ispmxist,spflnam
      integer   ispmxist
      character spflnam*72

      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      integer nspmnod,ispmloc,ispmtyp,nnodat
      real    spmdat

      common/spmatlbl/spmlabel(mspmnod)
      character spmlabel*16

      common/tc/itc,icnt
      integer itc    !- trace output index
      integer icnt   !- trace output counter

      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      integer itcf    !- building side increment to end trace output
      integer itrace  !- trace call index for subroutine
      integer izntrc  !- zone trace index
      integer itu     !- trace output unit number

      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      integer ihrp    !- hour number of present time step
      integer ihrf    !- hour number of future time step
      integer idyp    !- day number of present time step
      integer idyf    !- day number of future time step
      integer idwp    !- day of the week of present day
      integer idwf    !- day of the week of future day
      integer nsinc   !- number of building side time increments
                      !  since start of simulation
      integer its,idynow !- current building time-step within current hour

      common/fvalc/tfc(mcom,ms,mn),qfc(mcom)
      real    tfc   !- construction node temperature at future time-row
      real    qfc   !- energy injected at construction node at future time-row

      common/coe32j/qtmca(ms,mn,2)
      real    qtmca    !- shortwave absorption at each node of a transparent mlc

      common/coe32/qsoli(ms,2),qsole(ms,2)
      real    qsoli    !- solar energy absorbed by each internal opaque
                       !  surface after adjustments by window shading and
                       !  taking into account the directional property of
                       !  the direct beam and multiple diffuse reflections
      real    qsole    !- solar energy absorbed by each external opaque
                       !  surface after adjustment by surface shading
                       !  and taking into account solar/building geometry

      common/pvpnl/tpnl,qpnl,pvpwr
      real    tpnl,qpnl,pvpwr

      common/pers/isd1,ism1,isd2,ism2,isds,isdf,ntstep
      integer isd1     !- start day number
      integer ism1     !- start month number
      integer isd2     !- finish day number
      integer ism2     !- finish month number
      integer isds     !- start year day number
      integer isdf     !- finish year day number
      integer ntstep   !- number of building time steps within each hour

      common/adjc/ie(mcom,ms),atp(mcom,ms),atf(mcom,ms),
     &arp(mcom,ms),arf(mcom,ms)
      integer ie       !- surface defining index
      real    atp      !- present time adjacent temperature
      real    atf      !- future time adjacent temperature
      real    arp      !- present time adjacent incident radiation
      real    arf      !- future time adjacent incident radiation

      common/solext/exrad(ms),exdir(ms),exdif(ms),exshad(ms),
     &                                                    exshadf(ms)
      real    exrad    !- total solar radiation incident on surface
                       !  per unit area (includes shading effects)
      real    exdir    !- direct solar radiation incident on surface
                       !  per unit area (includes shading effects)
      real    exdif    !- diffuse solar radiation incident on surface
                       !  per unit area (included shading effects)
      real    exshad   !- direct shaded portion expressed as a factor
                       !  b/w 0 and 1 of surface
      real    exshadf  !- diffuse shaded portion expressed as a factor
                       !  b/w 0 and 1 of surface

      common/prectc/itmcfl(mcom,ms),tmct(mcom,mtmc,5),
     &       tmca(mcom,mtmc,me,5),tmcref(mcom,mtmc),tvtr(mcom,mtmc)
      integer itmcfl   !- index indicating whether surface is transparent
      real    tmct     !- direct solar transmittance at 5 representative
                       !  angles for TMC system
      real    tmca     !- absorptances for each glazing element of
                       !  TMC system at 5 representative angles
      real    tmcref   !- reflectance of TMC system (at 55 degrees)
      real    tvtr     !- visible transmittance for the TMC (for daylighting)

      common/prec9/nconst(mcom),nelts(mcom,ms),ngaps(mcom,ms),
     &             npgap(mcom,ms,mgp)
      integer nconst   !- number of surfaces per zone
      integer nelts    !- number of layers within construction
      integer ngaps    !- number of air gaps within construction
      integer npgap    !- location of air gap

      common/pvang/fInc_angle(ms)
      real fInc_angle  !- angle of incidence for surface

C Local variables.
      real    fVocRef  !- open circuit voltage at reference condition
      real    fIscRef  !- short circuit current at reference condition
      real    fVoc     !- open circuit voltage for cell
      real    fIsc     !- short circuit current for cell
      real    fVmpRef  !- module voltage at maximum power point and reference condition
      real    fImpRef  !- module current at maximum power point and reference condition
      real    fVmp     !- cell voltage at maximum power point
      real    fImp     !- cell current at maximum power point
      real    fQref    !- reference insolation (W/m^2)
      real    fTref    !- reference temperature (K)

      integer iSCell   !- number of cells connected in series
      integer iNPBrch  !- number of branches connected in parallel
      integer iNPnnls  !- number of modules in surface

      real    fAlpha   !- empirical coefficient for short circuit current calculation (/K)
      real    fBeta    !- empirical coefficient for open circuit voltage calculation (-)
      real    fGamma   !- empirical coefficient for open circuit voltage calculation (/K)

      integer iLodT    !- load type for the panel:
                       !         0 - maximum power point tracking (default)
                       !         1 - fixed voltage

      integer iMaxPPTrack,iFixedVolt        !- named constants
      parameter(iMaxPPTrack=0,iFixedVolt=1)

      real    fValf    !- module fixed voltage if load type = 1
                       !  or fixed resistance if load type = 2

      integer iShad    !- shading treatment index (see above)

      integer iIgnore,iProportion,iTotalPLoss,iPwrShd    !- named constants
      parameter(iIgnore=0,iProportion=1,iTotalPLoss=2,iPwrShd=3)

      real    fE       !- physical constant, electrical charge (J)
      parameter(fE=1.60E-19)

      real    fK       !- physical constant, Boltzmann constant (J/K)
      parameter(fK=1.38E-23)

      real    fQPnl        !- solar incident on the module (W/m^2)
      real    fTPnl        !- temperature of solar module (K)
      real    fLambda      !- empirical coefficient used to calculate current (/V)
      real    fRatioIoIsc  !- ratio of diode reverse saturation current
                           !  to short-circuit current
      parameter(fRatioIoIsc=1.0E-09)           !- model assumption

      real    fRseries     !- internal series resistance (ohms)
      real    fPVPower     !- power generated by modules (W)
      real    fPVPwrPerArea  !- generated power per unit area (W/m^2)
      real    fPVVoltage   !- array voltage
      real    fPVCurrent   !- array current
      real    fIcell       !- cell current
      real    fVcell       !- cell voltage
      real    fSurfArea    !- module surface area
      real    fIBound1     !- boundary values required for iterative solution
      real    fIBound2     !- of current, I (amps)
      real    F, FMID      !- required for iterative solution of I (f(I) = 0)
      real    fDeltaI      !- change in I, variable used in iterative solution
      real    fIMID        !- mid-point value used in iterative solution of I
      integer i,j          !- counters
      integer maxit        !- maximum iterations allowed in iterative solution of I
      parameter (maxit=100)
      integer iter         !- iteration counter for iterative solution of I
      logical bfound       !- boolean used in iterative solution of I

      character outs*124   !- user message outputs

      logical close        !- boolean used to compare two values

      real    qsole1       !- original value of external surface solar absorption
      real    qtmca1       !- original value of shortwave absorption at transparent surface

      real    fMiscLossFactor   !- miscellaneous loss factor used to derate the power
                                !  generated by the module (-)

      integer iTMC_index        !- index of TMC type for surface containing modules

      real    fTrans_0      !- transmittance at zero degrees (normal incidence)
      real    fTrans_60     !- transmittance at 60 degrees (used for diffuse rad.)
      real    fTrans_IncAng !- transmittance at angle of incidence

      real    fIncidence_angle  !- angle of incidence

      real    fQinc   !- irradiance of module with no account taken of
                      !  reflectance of module front cover (W/m^2)

      logical bAngIncidence_found  !- boolean used for interpolation

      real    X1, X2, Y1, Y2    !- variables used for interpolation calculation

      integer iNum_layers   !- number of layers in PV module surface
      integer iPV_layer     !- layer number with PV

      real    fTrans_Ads(5) !- transmission at 5 angles (0,40,55,70,80 degrees)
                            !  for PV module surface calculated as the sum of
                            !  the transmission and absorption for all layers 
                            !  in the surface at each individual angle

      real    fTransCorr(5) ! Correction factor for deviant transmission of
                            ! cover glass as compared to reference cover

      real    fAdsRef(5)    ! Reference Absorption of PV layer with reference
                            ! cover glass

      real    fSmall
      parameter (fSmall=1.0E-06)

C Parameters for h3kreports reporting.
      integer iNameLen(mspmnod)
      character*72 cName(mspmnod)
      logical bInitalized(mspmnod)
      data bInitalized / mspmnod * .false. /

C Store name of special material string in array.
      if(.not.bInitalized(ispmnod))then
         cName(ispmnod)=spmlabel(ispmnod)
         iNameLen(ispmnod)=lnblnk(spmlabel(ispmnod))
         bInitalized(ispmnod)=.true.
      endif

C Module open circuit voltage at reference condition.
      fVocRef=spmdat(ispmnod,1)

C Module short circuit current at reference condition.
      fIscRef=spmdat(ispmnod,2)

C Module voltage at maximum power point and reference condition.
      fVmpRef=spmdat(ispmnod,3)

C Module current at maximum power point and reference condition.
      fImpRef=spmdat(ispmnod,4)

C Reference insolation and temperature.
      fQref=spmdat(ispmnod,5)
      fTref=spmdat(ispmnod,6)

C Empirical temperature coefficient used in calculation of short circuit
C current. Calculations assume value gt. 0.
      fAlpha=ABS(spmdat(ispmnod,7))

C Empirical temperature coefficients used in calculation of open circuit
C voltage. Calculations assume values lt. 0
      fGamma=-1.*ABS(spmdat(ispmnod,8))
      fBeta=-1.*ABS(spmdat(ispmnod,9))

C Number of series connected cells (NOT modules).
      iSCell=spmdat(ispmnod,10)

C Number of parallel connected branches.
      iNPBrch=spmdat(ispmnod,11)

C Number of modules in surface.
      iNPnnls=nint(spmdat(ispmnod,12))

C Load type for the panel.
      iLodT=nint(spmdat(ispmnod,13))

C Fixed voltage for the entire module (used if iLodT = 1).
      fValf=spmdat(ispmnod,14)

C Shading treatment.
      iShad=nint(spmdat(ispmnod,15))

C Miscellaneous loss factor.
      fMiscLossFactor=spmdat(ispmnod,16)

C fTPNL is the temperature of the PV panel, assigned from the
C prevailing nodal temperature.
      fTPnl=tfc(ispmloc(ispmnod,1),ispmloc(ispmnod,2),
     &                                      ispmloc(ispmnod,3))+273.0

C Angle of incidence.
      fIncidence_angle=fInc_angle(ispmloc(ispmnod,2))

C Reset variable used for interpolation.
      bAngIncidence_found=.false.

C Obtain transparent surface index related to PV module location.
      iTMC_index=ITMCFL(ispmloc(ISPMNOD,1),ispmloc(ISPMNOD,2))

C Check that surface associated with the PV module is transparent;
C if not, issue an error message.
      if(iTMC_index.eq.0)then
        call edisp(iuout,' ')
        write(outs,*)'SMCMP5 fatal error: PV surface must be a TMC! ',
     &    '  Location: Zone ',ispmloc(ISPMNOD,1),', Surface ',
     &    ispmloc(ISPMNOD,2)
        call edisp(iuout,outs)
        stop
      endif

C ag@2015 Extend logic to take only layers external of and
C         including the PV node layer into account (more robust
C         in regard to layer layout of PV surfaces).
C Number of layers in PV module surface.
C Node number ispmloc(ispmnod,3) is PV location
      if (mod(ispmloc(ispmnod,3),2).gt.0) then
        call edisp(iuout,' ')
        write(outs,*)'SMCMP5 fatal error: PV node nr. must be even!'
        call edisp(iuout,outs)
        stop
      endif
cx       iNum_layers=nelts(ispmloc(ispmnod,1),ispmloc(ispmnod,2))
      iNum_layers=ispmloc(ispmnod,3)/2
      iPV_layer=iNum_layers

C Specification data for PV modules are generally based on a set-up
C according to
C   3 - 4 mm low iron (ideally w/ anti-reflective coating)
C   resin with cells embedded
C   backing
C
C Especially in facades, the external build-up is typically more
C like
C   44.2 or more laminated glazing (8.76 mmm)
C
C Also, the external glazing can be coloured, structured or both.
C To take these typically radiation reducing differences into account,
C a simplified reference-method is applied. For this it is necessary
C to use Window-output for the module glass layout as desired
C separated from IGDB #11126 by 0.001 mm air layer to create the
C TMC optical data
C
C Asign optional reference data for absorption at PV cell layer and
C calculate correction factor for transmission of cover glass
      if (nnodat(ispmnod).eq.21) then
        do j=1,5
          fAdsRef(j)=spmdat(ispmnod,j+16)
          if(tmca(ispmloc(ispmnod,1),iTMC_index,iPV_layer,j).gt.0.)then
            if (fAdsRef(j).gt.0.) then
              fTransCorr(j)=
     &           tmca(ispmloc(ispmnod,1),iTMC_index,iPV_layer,j)/
     &                                                     fAdsRef(j)
            else
              fTransCorr(j)=1.0
              call edisp(iuout,' ')
              write(outs,'(a,i1,a)')
     &           'SMCMP5 error: fTransCorr(',j,') not gt. 0.0!'
              call edisp(iuout,outs)
            endif ! fAdsRef gt. 0?
          else
            fTransCorr(j)=1.0
            call edisp(iuout,' ')
            write(outs,'(a,i2,a,i2,a,i2,a,i2,a)')
     &         'SMCMP5: tmca(',ispmloc(ispmnod,1),',',iTMC_index,
     &         ',',iPV_layer,',',j,') not gt. 0.0!'
            call edisp(iuout,outs)
          endif ! tmca gt. 0?
        enddo
      else
        do j=1,5
          fTransCorr(j)=1.0
        enddo
      endif

C Initialize array.
      do j=1,5
        fTrans_Ads(j)=0.
      enddo

C Determine transmission-absorptance factors of the PV module
C surface at 5 incidence angles.
C ag@2015 Extend logic to include transmission of relevant layers (actual
C         PV modules often have residual transmission!).
C TMCA    - absorptances for each glazing layer at 5 representative
C           incidence angles each (standard values)
C TMCT    - direct solar transmittance at 5 representative
C           incidence angles for TMC system (standard values).
      do j=1,5
C       econstr.F, line 920 ff.
        fTrans_Ads(j)=tmct(ispmloc(ispmnod,1),iTMC_index,j)
        do i=1,iNum_layers
           fTrans_Ads(j)=fTrans_Ads(j)+
     &                     tmca(ispmloc(ispmnod,1),iTMC_index,i,j)
        enddo
      enddo

C Get transmission-absorptance of PV module surface at 0 degrees,
C at 60 degres and at angle of incidence.
      fTrans_0=fTrans_Ads(1)*fTransCorr(1)

C Linearly interpolate to find absorptance at 60 degrees.
      fTrans_60=fTrans_Ads(3)*fTransCorr(3)+
     &             (60.0-55.0)*
     &             (fTrans_Ads(4)*fTransCorr(4)
     &                   -fTrans_Ads(3)*fTransCorr(3))/
     &             (70.0-55.0)

C Linearly interpolate to find transmittance at angle of incidence.
      if(fIncidence_angle.ge.0.0.and.
     &         fIncidence_angle.lt.40.0)then
         bAngIncidence_found=.true.
         X1=0.0
         X2=40.0
         Y1=fTrans_Ads(1)*fTransCorr(1)
         Y2=fTrans_Ads(2)*fTransCorr(2)
      elseif(fIncidence_angle.ge.40.0.and.
     &         fIncidence_angle.lt.55.0)then
         bAngIncidence_found=.true.
         X1=40.0
         X2=55.0
         Y1=fTrans_Ads(2)*fTransCorr(2)
         Y2=fTrans_Ads(3)*fTransCorr(3)
      elseif(fIncidence_angle.ge.55.0.and.
     &         fIncidence_angle.lt.70.0)then
         bAngIncidence_found=.true.
         X1=55.0
         X2=70.0
         Y1=fTrans_Ads(3)*fTransCorr(3)
         Y2=fTrans_Ads(4)*fTransCorr(4)
      elseif(fIncidence_angle.ge.70.0.and.
     &         fIncidence_angle.lt.80.0)then
         bAngIncidence_found=.true.
         X1=70.0
         X2=80.0
         Y1=fTrans_Ads(4)*fTransCorr(4)
         Y2=fTrans_Ads(5)*fTransCorr(5)
      endif

      if(bAngIncidence_found)then
         fTrans_IncAng=Y1+(fIncidence_angle-X1)*(Y2-Y1)/(X2-X1)
      else

C Assume equal to incidence angle at 80degC.
         fTrans_IncAng=fTrans_Ads(5)*fTransCorr(5)
      endif

C Obtain module irradiance without taking into account reflectance
C of front cover. This variable is used for reporting and debugging
C purposes only.
      fQInc=exrad(ispmloc(ispmnod,2))

C Impose specified shading treatment.
      if(exshad(ispmloc(ispmnod,2)).gt.1E-6)then

C Factor fTrans_IncAng/fTrans_0 below corrects irradiance for reflectance.
C External shading case.
        if(iShad.eq.iIgnore.or.iShad.eq.iProportion)then
           fQPnl=(exdir(ispmloc(ispmnod,2))*fTrans_IncAng+
     &              exdif(ispmloc(ispmnod,2))*fTrans_60)/
     &              fTrans_0
        elseif(iShad.eq.iTotalPLoss)then
           fQPnl=0.0
        else ! iPwrShd (iShad = 3)
           fQPnl=exdif(ispmloc(ispmnod,2))*fTrans_60/
     &             fTrans_0
        endif
      else

C No external shading present.
        fQPnl=(exdir(ispmloc(ispmnod,2))*fTrans_IncAng+
     &              exdif(ispmloc(ispmnod,2))*fTrans_60)/
     &              fTrans_0
      endif

C Calculate cell short circuit current.
      fIsc=fIscRef/iNPBrch*(fQPnl/fQref)*(1.-fAlpha*(fTPnl-fTref))

C Calculate cell open circuit voltage.
      call eclose(fQPnl,0.0,0.001,close)
      if(.not.close)then
         fVoc=fVocRef/iScell*(1.+fGamma*(fTPnl-fTref))*
     &          MAX(0.,(1.+fBeta*log(fQPnl/fQref)))
      else
         fVoc=0.
      endif

      if(iLodT.eq.iMaxPPTrack)then

C Assuming that the maximum power point of the module is tracked by
C a converter, calculate the maximum output voltage. Note that
C the Watsun-PV model assumes that the maximum power point voltage
C has the same dependancy on cell temperature and incident irradiance
C as the open circuit voltage.
          fVmp=fVmpRef/float(iScell)*   !- cell value
     &            (1.+fGamma*(fTPnl-fTref))*
     &            max(0.,(1.+fBeta*log((fQPnl+fSmall)/fQref)))

C Calculate the maximum module power.
          fPVPower=fImpRef*fVmpRef*fIsc*fVoc/
     &               (fIscRef*fVocRef)*
     &               iNPBrch*iSCell*iNPnnls*(1.-fMiscLossFactor)

C Calculate the module voltage.
          fPVVoltage=fVmp*iSCell

      elseif(iLodT.eq.iFixedVolt)then

C Panel output with a fixed voltage type load, e.g. a battery.

C Empirical coefficient used in the calculation of current.
          fLambda=1./fVoc*log(1./fRatioIoIsc)

C Current at maximum power point. Note that the Watsun-PV
C model assumes that the maximum power point current has the same
C dependancy on cell temperature and incident irradiance as the
C short circuit current.
          fImp=fImpRef/iNPBrch*(fQPnl/fQref)*   !- cell value
     &              (1.-fAlpha*(fTPnl-fTref))

C Calculate voltage at maximum power point.
          fVmp=fVmpRef/iScell*                  !- cell value
     &              (1.+fGamma*(fTPnl-fTref))*
     &              max(0.,(1.+fBeta*log(fQPnl/fQref)))

C Calculate the internal series resistance.
          fRseries=(1./fLambda*log(1./fRatioIoIsc*
     &              (1.-fImp/fIsc))-fVmp)/fImp

C Determine the cell voltage. fValf = fixed voltage value provided
C as model input.
          fVcell=fValf/iScell

C Calculate the cell current. This is done iteratively using the
C bisection method. The cell current will be between 0 and the
C short-circuit current, therefore these are the bounds for iteration.
          fIBound1=0.
          fIBound2=fIsc
          FMID=fIBound2-fIsc*(1.-fRatioIoIsc*exp(fLambda*
     &                              (fVcell+fIBound2*fRseries)))
          F=fIBound1-fIsc*(1.-fRatioIoIsc*exp(fLambda*
     &                              (fVcell+fIBound1*fRseries)))

          if(FMID.eq.0.)then
             fIcell=fIBound2
          elseif(F.eq.0.)then
             fIcell=fIBound1
          endif

          if((F*FMID).gt.0.)then
             continue        !- place error message here
          endif
          if(F.lt.0.)then
             fIcell=fIBound1
             fDeltaI=fIBound2-fIBound1
          else
             fIcell=fIBound2
             fDeltaI=fIBound1-fIBound2
          endif

          bfound=.false.
          iter=0
          do while (.not.bfound)

C Check that the maximum number of iterations is not exceeded.
             if(iter.gt.maxit)then
                call edisp(iuout,' ')
                call edisp(iuout,
     &              'SPMCMP5 fatal error: iteration limit exceeded!')
                stop
             endif

             fDeltaI=fDeltaI*0.5
             fIMID=fIcell+fDeltaI
             FMID=fIMID-fIsc*(1.-fRatioIoIsc*exp(fLambda*
     &                                  (fVcell+fIMID*fRseries)))
             if(FMID.le.0.)fIcell=fIMID
             if((ABS(fDeltaI).lt.0.001).or.(FMID.eq.0.))then
                bfound=.true.
             endif
             iter=iter+1
          enddo

C Current of entire module.
          fPVcurrent=fIcell*iNPBrch

C PV module voltage.
          fPVVoltage=fValf

C Power output.
          fPVPower=fPVVoltage*fPVCurrent*iNPnnls*
     &               (1.-fMiscLossFactor)
      endif

C Results output.
      fSurfArea=sna(ispmloc(ispmnod,1),ispmloc(ispmnod,2))
      ndatspm(ispmnod)=2
      dataspm(ispmnod,1)=fPVPower
      strspm(ispmnod,1)='Power (W)'
      if(fQInc.ne.0.)then
         dataspm(ispmnod,2)=((fPVPower/fSurfArea)/fQInc)*100.
      else
         dataspm(ispmnod,2) = 0.
      endif
      strspm(ispmnod,2)='Efficiency (%)'

C As a temporary measure, use zone-related variables, zspmf1 and zspmf2,
C for results library storage. Output is limited to one special material
C per zone.
      zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)
      zspmf2(ispmloc(ispmnod,1))=dataspm(ispmnod,2)

C Save panel power output (for load flow simulation) and the current
C panel voltage. (This could later be changed to a power converter
C component that can vary the values of P and Q as generated by the
C panel.)
      pspm(ispmnod)=fPVPower
      vspm(ispmnod)=cmplx(fPVVoltage,0.0)
      qspm(ispmnod)=0.0

C Convert the panel power output from W to W/m^2.
      fPVPwrPerArea = fPVPower/fSurfArea

C Save original nodal absorptions for trace output.
      qsole1=qsole(ispmloc(ispmnod,2),2)
      qtmca1=qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)

C Alter the nodal absorption according to whether the node is
C opaque (1) or transparent (0).
      if(ispmtyp(ispmnod,2).gt.0)then
        if(ie(ispmloc(ispmnod,1),ispmloc(ispmnod,2)).eq.0)then
          qsole(ispmloc(ispmnod,2),2)=qsole(ispmloc(ispmnod,2),2)
     &                                -fPVPwrPerArea
        else
          qsoli(ispmloc(ispmnod,2),2)=qsoli(ispmloc(ispmnod,2),2)
     &                                -fPVPwrPerArea
        endif
      else

C For a transparent layer, modify absorption of the middle node
C and the two boundary nodes.
        qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)=
     &          qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)-
     &          (fPVPwrPerArea/2.)
        qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)+1,2)=
     &          qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)+1,2)-
     &          (fPVPwrPerArea/4.)
        qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)-1,2)=
     &          qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3)-1,2)-
     &          (fPVPwrPerArea/4.)
      endif

C XML output.
      Call AddToReport(rvBldSPMatlEffIrr%Identifier,
     &        fQPnl,
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

      Call AddToReport(rvBldSPMatlEff%Identifier,
     &        dataspm(ispmnod,2),
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

      Call AddToReport(rvBldSPMatlIncAngl%Identifier,
     &        fIncidence_angle,
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

      Call AddToReport(rvBldSPMatlTrnsAngl%Identifier,
     &        fTrans_IncAng,
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

      Call AddToReport(rvBldSPMatlTrns0%Identifier,
     &        fTrans_0,
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

      Call AddToReport(rvBldSPMatlTrns60%Identifier,
     &        fTrans_60,
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

      Call AddToReport(rvBldSPMatlTtlIncAr%Identifier,
     &        fQInc,
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

      Call AddToReport(rvBldSPMatlTtlIncTtl%Identifier,
     &        fQInc * fSurfArea,
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

      Call AddToReport(rvBldSPMatlDrtIncAr%Identifier,
     &        exdir(ispmloc(ispmnod,2)),
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

      Call AddToReport(rvBldSPMatlDffIncAr%Identifier,
     &        exdif(ispmloc(ispmnod,2)),
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

      Call AddToReport(rvBldSPMatlPVPow%Identifier,
     &        fPVPower,
     &        cName(ispmnod)(1:iNameLen(ispmnod)))

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP5 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,2x,i4,2x,i4,2x,i4)')
     &    'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &                          ispmloc(ispmnod,2),ispmloc(ispmnod,3)
      call edisp(itu,outs)
      write(outs,'(a,i3)') 'Number. of modules in surface:',iNPnnls
      call edisp(itu,outs)
      if(ispmtyp(ispmnod,2).gt.0)then
         write(outs,'(a,1x,f10.2)') 'Original flux absorption: ',qsole1
         call edisp(itu,outs)
         write(outs,'(a,1x,f10.2)') 'Modified flux absorption: ',
     &                                    qsole(ispmloc(ispmnod,2),2)
         call edisp(itu,outs)
      else
         write(outs,'(a,1x,f10.2)') 'Original flux absorption: ',qtmca1
         call edisp(itu,outs)
         write(outs,'(a,1x,f10.2)') 'Modified flux absorption: ',
     &                 qtmca(ispmloc(ispmnod,2),ispmloc(ispmnod,3),2)
         call edisp(itu,outs)
      endif
      write(outs,'(a,1x,f10.2)') 'Power output: ', fPVPower
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Panel temperature: ', fTPnl
      call edisp(itu,outs)
  999 return
      end

C ***************** SPMCMP6 *****************
C A model that calculates the electrical power output from a free standing
C wind turbine (after Nick Kelly and Andy Grant). The model includes
C a statistical manipulation of the mean wind velocity in order to
C estimate the power output. Defining data as read from spmdat:
C    1 - height of turbine axis (m)
C    2 - turbine cross sectional area (m^2)
C    3 - power velocity coefficient (-)
C    4 - cut-in wind speed (m/s)
C    5 - rated wind speed (m/s)
C    6 - number of turbines (-)
C    7 - location (1-urban, 2-suburban, 3-rural)
C    8 - reference height (m)
C    9 - wind speed statistics (on/off)
C   10 - wind speed/turbulence profile (on/off)
C   11 - default turbulence intensity (-)

      subroutine spmcmp6(icomp,ispmnod)
#include "building.h"
#include "plant.h"
#include "net_flow.h"
C #include "esprdbfile.h"
#include "power.h"

C N.B. All parameters comply with the Fortran implicit naming
C convention except where explicitly redefined.
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/pers/isd1,ism1,isd2,ism2,isds,isdf,ntstep
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)
      common/mflclm/dryb,qdif,qdnr,irvh,wdir,wspd,wred
      common/climi/qfp,qff,tp,tf,qdp,qdf,vp,vf,dp,df,hp,hf

      real minu,minv,inc
      real hdwti,tarea,pvcoeff,ciwspd,ratewspd,refh,turbi
      integer nturb,iloc,iwstat,iprof

      character spmlabel*16,outs*124,spflnam*72,strspm*16,
     &          unitspm*16,txtspm*72


C Component data.
      hdwti=spmdat(ispmnod,1)  ! height of turbine axis (m)
      tarea=spmdat(ispmnod,2)  ! turbine cross sectional area (m^2)
      pvcoeff=spmdat(ispmnod,3) ! power velocity coefficient (-)
      ciwspd=spmdat(ispmnod,4)  ! cut-in wind speed (m/s)
      ratewspd=spmdat(ispmnod,5) ! rated wind speed (m/s)
      nturb=nint(spmdat(ispmnod,6))  ! number of turbines in the surface (-)
      iloc=nint(spmdat(ispmnod,7))   ! location (1-urban, 2-suburban, 3-rural)
      refh=spmdat(ispmnod,8)         ! reference height (m)
      iwstat=nint(spmdat(ispmnod,9)) ! wind speed statistics (on/off)
      iprof=nint(spmdat(ispmnod,10))  ! wind speed/turbulence profile (on/off)
      turbi=spmdat(ispmnod,11)        ! default turbulence intensity (-)

      pi=4.0*atan(1.0)

C Current weather data.
      widir=df
      drybt=tf
      wispd=vf
      if(wispd.lt.1E-3)wispd=1E-3

C Calculate the wind velocity prevailing at the height of the ducted
C wind turbine. This information will be available if an air flow
C network exists. If profiles or speed reduction are not specified
C then use default values.
      if(wred.gt.1E-6.and.iprof.lt.1)then
         wispd=wispd*wred
      elseif(iprof.gt.1)then

C Urban location.
         if(iloc.eq.1)then
            pwr=0.32
            rough=5.

C Suburban location
         elseif(iloc.eq.2)then
            pwr=0.23
            rough=0.8

C Rural location
         elseif(iloc.eq.3)then
            pwr=0.14
            rough=0.05
         endif

C Set the new wind speed and turbulent intensity from profiles. If
C height is zero, set to a small value to avoid numerical problems.
         if(hdwti.lt.1E-6)hdwti=1E-6
         if(refh.lt.1E-6)refh=1E-6
         wispd=wispd*(hdwti/refh)**pwr
         turbi=(rough*log(30./refh))/log(hdwti/refh)
      endif

      if(iwstat.gt.0)then

C Wind velocity components.
         compu=wispd*sin((widir*pi)/180.)
         compv=wispd*cos((widir*pi)/180.)

C Standard deviation, based on turbulent intensity.
         stdev=turbi*wispd

C Wind velocity ranges.
         minu=compu-3*stdev
         minv=compv-3*stdev

C Calculate the increment.
         inc=6*stdev/10.
         icount=1
         pwrtot=0.

C Set starting value of U velocity component.
         valu=minu

C Set up trace probability counters.
         tpru=0
C         tprc=0

C Loop to generate power output.
         do 66 i=1,10

C Current U velocity component probability density.
            pdu=(1/(stdev*(2*pi)**0.5))*
     &                       exp(-(((valu-compu)**2.)/(2*stdev**2.)))
            tpru=tpru+pdu*inc
            tprv=0

C Set starting value of V velocity component.
            valv=minv

            do 77 j=1,10

C Current V velocity component probability density.
               pdv=(1/(stdev*(2*pi)**0.5))*
     &                       exp(-(((valv-compv)**2.)/(2*stdev**2.)))
               tprv=tprv+pdv*inc

C Combined probability.
               prc=(inc*pdu)*(inc*pdv)

C Duration of current wind speed and direction.
               tsl=3600./float(ntstep)
               tdur=prc*tsl

C New wind speed and direction.
               wispdm=(valu**2.+valv**2.)**0.5
               if(abs(valv).lt.1E-6)valv=1E-6
               widirm=atan(valu/valv)*(180./pi)
               if(valv.ge.0)then
                  if(valu.ge.0.)then
                     widirm=widirm
                  else
                     widirm=360.+widirm
                  endif
               else
                  if(valu.ge.0.)then
                     widirm=180.+widirm
                  else
                     widirm=180.+widirm
                  endif
               endif

C Check that the wind is strong enough for the turbine to cut in.
               if(wispdm.ge.ciwspd)then
                 continue
               else
                 wispdm=0. ! If not, set the wind speed to zero
               endif

               ro=densit(1,drybt)  ! get air density
               if(wispdm.lt.ratewspd)then

C Power output below rated from P=Cp(1/2).rho.A.V^3
                 pwr=(pvcoeff*(0.5))*ro*tarea*(wispdm**3)*nturb
               else

C Power output above rated from P=Cp(1/2).rho.A.Vr^3
                 pwr=(pvcoeff*(0.5))*ro*tarea*(ratewspd**3)*nturb
               endif

               icount=icount+1
               valv=valv+inc
               pwrtot=pwrtot+pwr*tdur
   77       continue
            valu=valu+inc
   66    continue

C Write results for probability-based power output.
         pspm(ispmnod)=pwrtot/tsl

      else

C Check that the wind is strong enough for the turbine to cut in.
         if(wispd.ge.ciwspd)then
           continue
         else
           wispd=0. ! If not, set the wind speed to zero
         endif

         ro=densit(1,drybt)  ! get air density
         if(wispd.lt.ratewspd)then

C Power output below rated from P=Cp(1/2).rho.A.V^3
           pspm(ispmnod)=(pvcoeff*(0.5))*ro*tarea*(wispd**3)*nturb
         else

C Power output above rated from P=Cp(1/2).rho.A.Vr^3
           pspm(ispmnod)=(pvcoeff*(0.5))*ro*tarea*(ratewspd**3)*nturb
         endif

      endif

C Results output.
      ndatspm(ispmnod)=1
      dataspm(ispmnod,1)=pspm(ispmnod)
      strspm(ispmnod,1)='Power (W)'

C As a temporary measure, use zone-related variable, zspmf1, for
C results library storage. Output is limited to one special material
C per zone.
       zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP6 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      if(iwstat.gt.0)then
        write(outs,'(a,1x,f10.2)') 'Wind speed via probability: ',
     &    wispdm
      else
        write(outs,'(a,1x,f10.2)') 'Wind speed: ',wispd
      endif
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Pressure coefficient: ',pvcoeff
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Air density: ',ro
      call edisp(itu,outs)
      write(outs,'(a,1x,f10.2)') 'Power output: ',pspm(ispmnod)
      call edisp(itu,outs)
  999 return
      end

C ***************** SPMCMP51 *****************
C A model of the change in shortwave flux transmission and absorption
C due to the changing tint of a thermo-chromic glass. The model
C assumes a linear variation of transmission over the working
C temperature range (Tmin-Tmax). Outwith this range, the material
C is either clear (<Tmin) or at maximum tint (>Tmax). While the
C transmission and absorption both vary with temperature, the
C reflectivity remains constant.Transmission is therefore a
C discontinuous function of the temperature of the associated node.
C Note that the transmission charactersitics of all layers after
C the switched layer are affected. Defining data as read from spmdat:
C    1 - temperature at minimum transmission, Tmint (degC);
C    2 - temperature at maximum transmission, Tmaxt (degC);
C    3 - minimum transmission as a % of max transmission;
C    4 - associated layer number in TMC.

      subroutine spmcmp51(icomp,ispmnod)
#include "building.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      common/fvalc/tfc(mcom,ms,mn),qfc(mcom)

C TMC and construction data.
      common/prectc/itmcfl(mcom,ms),tmct(mcom,mtmc,5),
     &         tmca(mcom,mtmc,me,5),tmcref(mcom,mtmc),tvtr(mcom,mtmc)
      common/prec9/nconst(mcom),nelts(mcom,ms),ngaps(mcom,ms),
     &             npgap(mcom,ms,mgp)

C Stored TMC information.
      common/tmcstor/tran0(mspmnod,5),abs0(mspmnod,ms,5),istore

      character spmlabel*16,outs*124,spflnam*72,strspm*16,unitspm*16,
     &          txtspm*72

      small=1.0E-5

C Temperature at maximum transmission.
      tmaxt=spmdat(ispmnod,1)

C Temperature at minimum transmission.
      tmint=spmdat(ispmnod,2)
      if(tmint.lt.tmaxt)then
         call edisp(iuout,' ')
         write(outs,'(a)')'SPMCMP51 fatal error: temperature at'
         call edisp(iuout,outs)
         write(outs,'(a)')'minimum transmission is lower than'
         call edisp(iuout,outs)
         write(outs,'(a)')'at maximum transmission!'
         call edisp(iuout,outs)
         stop
      endif

C Minimum transmission as a % of transmission at clear state.
      perc=spmdat(ispmnod,3)

C Location layer for thermo-chromic material.
      ilyr=int(spmdat(ispmnod,4))

C Number of layers in the TMC.
      nlyr=nelts(icomp,ispmloc(ispmnod,2))

C Node location and temperature.
      isur=ispmloc(ispmnod,2)
      inod=ispmloc(ispmnod,3)
      tnod=tfc(icomp,isur,inod)

C Store the original clear state optical properties for the affected
C layer(s).
      if(istore.eq.0)then

C Sytem transmission.
         do iang0=1,5
            tran0(ispmnod,iang0)=tmct(icomp,itmcfl(icomp,isur),iang0)
         enddo

C Layer absorption.
         do layer=1,nlyr
            do iang=1,5
               abs0(ispmnod,layer,iang)=tmca(icomp,itmcfl(icomp,isur),
     &                                                    layer,iang)
            enddo
         enddo
         istore=1
      endif

C System is unaltered.
      if(tnod.lt.tmaxt)then
        tint=1.0

C Alter system transmission and layer absorption.
      elseif(tnod.gt.tmaxt.and.tnod.lt.tmint) then

C Calculate transmission percentage at current nodal temperature.
         diff=abs(tmaxt-tmint)
         if(diff.le.small)then

C User has set tmint=tmaxt.
            tint=(100.+perc)/200.
            goto 33
         endif

         tint=((tnod-tmint)/(tmaxt-tmint))*(100.-perc)+perc
         tint=tint/100.
   33    continue

C > tmint.
      elseif(tnod.ge.tmint)then
         tint=perc/100.
      endif

C Results output.
      ndatspm(ispmnod)=1
      dataspm(ispmnod,1)=tint
      strspm(ispmnod,1)='Transmission (%)'

C As a temporary measure, use zone-related variable, zspmf1, for
C results library storage. Output is limited to one special material
C per zone.
      zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)

C Alter the values of transmission for the affected layer as a
C function of the tint value.
      do iang2=1,5
         tmct(icomp,itmcfl(icomp,isur),iang2)=tran0(ispmnod,iang2)*tint

C Adjust the layer absorption as a function of this transmission
C change.
         tmca(icomp,itmcfl(icomp,isur),ilyr,iang2)=
     &       tran0(ispmnod,iang2)*(1.- tint)+abs0(ispmnod,ilyr,iang2)
      enddo

C Reduce relative absorption for all layers after the tinted layer.
      if(ilyr.lt.nlyr)then
         do layer2=ilyr+1,nlyr
            do iang3=1,5
               tmca(icomp,itmcfl(icomp,isur),layer2,iang3)=
     &                                abs0(ispmnod,layer2,iang3)*tint
            enddo
         enddo
      endif

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP51 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,2x,i4,2x,i4,2x,i4)')
     &  'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &                          ispmloc(ispmnod,2),ispmloc(ispmnod,3)
      call edisp(itu,outs)
      call edisp(itu,' ')
      write(outs,'(a)') 'Original transmissions:'
      call edisp(itu,outs)
      write(outs,'(5(2x,f7.4))') (tran0(ispmnod,idat),idat=1,5)
      call edisp(itu,outs)
      write(outs,'(a)') 'Original absorptions:'
      call edisp(itu,outs)
      write(outs,'(5(2x,f7.4))') (abs0(ispmnod,ilyr,idat),idat=1,5)
      call edisp(itu,outs)
      call edisp(itu,' ')
      write(outs,'(a)') 'Altered transmissions:'
      call edisp(itu,outs)
      write(outs,'(5(2x,F7.4))') (tmct(icomp,itmcfl(icomp,isur),idat),
     &                                                      idat=1,5)
      call edisp(itu,outs)
      write(outs,'(a)') 'Altered absorptions:'
      call edisp(itu,outs)
      write(outs,'(5(2x,f7.4))') (tmca(icomp,itmcfl(icomp,isur),
     &                                         ilyr,idat),idat=1,5)
      call edisp(itu,outs)
  999 return
      end

C ***************** SPMCMP52 *****************
C A model of saturated surface evaporation and heat loss. The
C evaporation (kg/s) is calculated from the Lewis relationship:
C        Ev=hc.Asurf.(Wsurf-Wzone)/Cp (kg/s)
C where,
C       hc = convective heat transfer coefficient (W/m^2.degC)
C       Asurf = surface area (m^2)
C       Wsurf = saturated moisture content at Tsurf (kg/kg da)
C       Tsurf = surface temperature (degC)
C       Wzone = moisture content of zone air (kg/kg da)
C       Cp = specific heat capacity of air (J/kgK).

      subroutine spmcmp52(icomp,ispmnod)
#include "building.h"
#include "site.h"
#include "geometry.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      common/evapsur/vapsur(mcom)
      common/fvala/tfa(mcom),qfa(mcom)
      common/fvals/tfs(mcom,ms),qfs(mcom)
      common/concoe/hcip(mcom,ms),hcif(mcom,ms),hcop(mcom,ms),
     &              hcof(mcom,ms)
      common/fvalg/gfa(mcom)

C Generic nodal flux common allowing a positive or negative flux
C to be applied to a node; genflxp is updated in mzls5.
      common/genflux/genflxf(mcom,ms,mn),genflxp(mcom,ms,mn)

      character spmlabel*16,spflnam*72,outs*124,strspm*16,unitspm*16,
     &          txtspm*72

C surface location.
      if(icomp.ne.ispmloc(ispmnod,1))then
         write(outs,*)'SPMAT52 fatal error: incorrect zone assignment!'
         call edisp(iuout,outs)
         stop
      endif
      isurf=ispmloc(ispmnod,2)
      inode=ispmloc(ispmnod,3)

      td=tfa(icomp)
      gz=gfa(icomp)
      zrh=pcrh2(td,gz,patmos)
      asurf=sna(icomp,isurf)
      tsurf=tfs(icomp,isurf)
      alpha=hcif(icomp,isurf)


C Because there may be more than one evaporative surface, Ev is added
C to the current value of vapsur(icomp).
      wzone=gfa(icomp)
      cp=sphtc2(td,wzone)*1000
      saturate=100.0
      wsurf=humrt1(tsurf,saturate,PATMOS,0)

      ev=alpha*asurf*(wsurf-wzone)/cp
      vapsur(icomp)=vapsur(icomp)+ev

C Determine the heat drawn from the surface during evaporation.

C Enthalpy of water vapour at tsurf (approximate value is 2550E+03).
      hvap=cndwat1(tsurf)

C Enthalpy of saturation at tsurf.
      hsurf=shth2O(tsurf)*1000.

C Surface evaporative loss (W/m^2).
      genflxf(icomp,isurf,inode)=-ev*(hvap-hsurf)/sna(icomp,isurf)

C Results output.
      ndatspm(ispmnod)=2
      dataspm(ispmnod,1)=genflxf(icomp,isurf,inode)*sna(icomp,isurf)
      strspm(ispmnod,1)='Evaporation (W)'
      dataspm(ispmnod,2)=ev*1000.
      strspm(ispmnod,2)='Evaporation(g/s)'

C As a temporary measure, use zone-related variables, zspmf1 and zspmf2,
C for results library storage. Output is limited to one special material
C per zone.
      zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)
      zspmf2(ispmloc(ispmnod,1))=dataspm(ispmnod,2)

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP52 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,2x,i4,2x,i4,2x,i4)')
     &        'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &                          ispmloc(ispmnod,2),ispmloc(ispmnod,3)
      call edisp(itu,outs)
      call edisp(itu,' ')
      write(outs,'(a)') 'Evaporation from surface (kg):'
      call edisp(itu,outs)
      write(outs,'(f7.4)') ev
      call edisp(itu,outs)
      write(outs,'(a)') 'Evaporative heat loss from surface (W/m^2):'
      call edisp(itu,outs)
      write(outs,'(F10.4)') genflxf(icomp,isurf,inode)
      call edisp(itu,outs)
      write(outs,'(a)') 'Evaporative heat loss from surface (W):'
      call edisp(itu,outs)
      write(outs,'(F10.4)')genflxf(icomp,isurf,inode)*sna(icomp,isurf)
      call edisp(itu,outs)
  999 return
      end

C ***************** SPMCMP53 *****************
C A model of phase change material based on the apparent heat capacity
C method. The  thermal properties of a construction layer defined as
C a phase change material (PCM) are adjusted to represent the latent
C heat stored within the material during melting and released during
C solidification. The phase transition is represented by an apparent
C heat capacity (appsht), which is equal to the sum of specific (shtsol)
C and latent (lht) heat capacity, with the latter established as a linear
C function of temperature (tfcm) in the phase change temperature range
C (tsoli-tmelt): lht=shtliqa*tfcm+shtliqb; where shtliqa is given as 0,
C lht = shtliqb.

C Defining data as read from spmdat:
C    1 - melting temperature, tmelt (degC);
C    2 - solidification temperature, tsoli (degC);
C    3 - conductivity in solid phase, consol (W/m.degC);
C    4 - conductivity in liquid phase, conliq (W/m.degC);
C    5 - specific heat in all phases, shtsol (J/kg.degC);
C    6 - latent heat coefficient, shtliqa (J/kg.degC^2);
C    7 - latent heat coefficient, shtliqb (J/kg.degC).

      subroutine spmcmp53(icomp,ispmnod)
#include "building.h"
#include "geometry.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/fvalc/tfc(mcom,ms,mn),qfc(mcom)
      common/fvals/tfs(mcom,ms),qfs(mcom)
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)

C thrmli(icomp,isurf,ielem,1) = conductivity
C thrmli(icomp,isurf,ielem,2) = density
C thrmli(icomp,isurf,ielem,3) = specific heat
C thrmli(icomp,isurf,ielem,4) = thickness
      common/vthp14/thrmli(mcom,ms,me,7)
      common/prec9/nconst(mcom),nelts(mcom,ms),ngaps(mcom,ms),
     &             npgap(mcom,ms,mgp)
      common/prec13/c(mcom,ms,mn,2),qc(mcom,ms,mn)
      common/pers/isd1,ism1,isd2,ism2,isds,isdf,ntstep
      common/vthp15/vcp(ms,mn,2),qcp(ms,mn)
      common/vthp16/vcf(ms,mn,2),qcf(ms,mn)
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)

C Special materials.
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)

C Phase change material.
      common/pcm02/spcm(mcom,ms,mn),spcmtf,spcmtp
      common/pcm03/pcmfac(mcom,ms,mn),pcmshts(mcom,ms,mn)

      character spmlabel*16,spflnam*72,outs*124,strspm*16,unitspm*16,
     &          txtspm*72
      logical close
      real val1,val2,val3,val4,val5,val6,val7
     &                   val8,val9,val10,val11  ! to pass as parameters

C PCM layer pointers.
      if(icomp.ne.ispmloc(ispmnod,1))then
         write(outs,*)'SPMAT53 fatal error: incorrect zone assignment!'
         call edisp(iuout,outs)
         stop
      endif
      isurf=ispmloc(ispmnod,2)
      inode=ispmloc(ispmnod,3)
      ielem=nint(real(inode)/2.)

C Melting and solidification temperatures (degC).
      tmelt=spmdat(ispmnod,1)
      tsoli=spmdat(ispmnod,2)

C Conductivity in solid and liquid phase (W/m.degC).
      consol=spmdat(ispmnod,3)
      conliq=spmdat(ispmnod,4)

C Specific heat capacity (J/kg.degC).
      shtsol=spmdat(ispmnod,5)

C Latent heat coefficients (J/kg.degC^2 & J/kg.degC respectively).
      shtliqa=spmdat(ispmnod,6)
      shtliqb=spmdat(ispmnod,7)

C Initialisations for xml output.
      Rtot=0.0
      THKsum=0.0
      sumSHTTHK=0.0

C Process the 3 nodes of the PCM layer.
      spcmtf=0.0
      do 10 iphnl=1,3
         if(iphnl.eq.1)then
            nloc=inode-1
            efthk=thrmli(icomp,isurf,ielem,4)*0.25
         elseif(iphnl.eq.2)then
            nloc=inode
            efthk=thrmli(icomp,isurf,ielem,4)*0.5
         elseif(iphnl.eq.3)then
            nloc=inode+1
            efthk=thrmli(icomp,isurf,ielem,4)*0.25
         endif
         THKsum=THKsum+efthk      !- total PCM thickness

C Establish temperature of the PCM material node from tfc if the node
C is intra-construction and tfs if located at the internal surface.
         if(iphnl.eq.3.and.ielem.eq.nelts(icomp,isurf))then
            tfcm=tfs(icomp,isurf)
         else
            tfcm=tfc(icomp,isurf,nloc)
         endif

C If phase change is underway, calculate the new value of
C conductivity (W/m.degC):
C  appcon = consol          for tfcm < tmelt (solid phase);
C         = conliq          for tfcm > tsoli (liquid phase);
C         = consol+conliq/2 for tmelt < tfcm < tsoli (mixed phase);
C ... and apparent heat capacity (J/kg.degC):
c  appsht = shtsol + lht
C   where lht = 0                      for tsoli < tfcm < tmelt (solid or liquid phase);
C             = shtliqa*tfcm + shtliqb for tmelt < tfcm < tsoli (mixed phase).
C         if(tfcm.lt.tmelt)then
C            appcon=consol
C            appsht=shtsol
C         elseif(tfcm.gt.tsoli)then
C            appcon=conliq
C            appsht=shtsol
C         else
C            appcon=(consol+conliq)/2.
C            appsht=shtsol+(shtliqa*tfcm+shtliqb)
C        endif

         imethod=1
         val1=tsoli
         val2=tmelt
         val3=shtsol
         val4=consol
         val5=conliq
         val6=shtliqa
         val7=shtliqb
         val8=0.0
         val9=0.0
         val10=0.0
         val11=0.0
         call maxlatent(imethod,tfcm,tpcm,val1,val2,val3,val4,
     &           val5,val6,val7,val8,val9,val10,
     &            val11,appcon,appsht,spcmf)


C Parameters for xml output.
         Rtot=Rtot+efthk/appcon
         sumSHTTHK=sumSHTTHK+appsht

C Establish present value of factp and pcmshtp as last used value
C of factf and pcmsht respectively.
         factp=pcmfac(icomp,isurf,nloc)
         pcmshtp=pcmshts(icomp,isurf,nloc)
         if(nsinc.eq.1)then
            factp=0
         endif

C Calculate new factf; this is used below to remove original values
C of PCM layer conductivity and specific heat as used to establish the
C self- and cross- coupling coefficients of the nodal energy balance
C equations, and substitute the new values as established above. An
C adjusted factf is required where the PCM node separates the PCM layer
C from a non-PCM layer.
         pcmcon=appcon/consol
         pcmsht=appsht/shtsol
         factf=pcmcon/pcmsht

C Modify present and future value of 'c' and 'qc' coefficients (i.e.
C coeficients of nodal energy balance equations.
         x1=1./pcmshtp
         if(nsinc.eq.1)then
            x1=0
         endif
         x2=1./pcmsht
         if(iphnl.eq.1.and.ielem.eq.1)then                  !- outermost node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x1
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factp
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x2
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factf
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.2)then                             !- centre node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factp
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factp
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factf
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factf
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.3.and.ielem.eq.nelts(icomp,isurf))then !- internal surface node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factp
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x1
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factf
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x2
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.1.or.iphnl.eq.3)then               !- interface node

C Redefine modification factors to take account of weighted node
C thermal capacity.
            if(iphnl.eq.1)then
               iouter=ielem
               iinner=ielem-1
            else
               iinner=ielem
               iouter=ielem+1
            endif

C Original averaged capacity before adjustment.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               thrmli(icomp,isurf,iouter,3)*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)

C Cater for an adjacent air gap.
            ians1=0
            ians2=0
            if(ngaps(icomp,isurf).ne.0)then
               if(iphnl.eq.1)then
                  do igap=1,ngaps(icomp,isurf)
                     if(iinner.eq.npgap(icomp,isurf,igap))then
                        capinner=1.3*1005.5*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
                        ians1=1
                     endif
                  enddo
               elseif(iphnl.eq.3)then
                  do igap=1,ngaps(icomp,isurf)
                     if(iouter.eq.npgap(icomp,isurf,igap))then
                        capouter=1.3*1005.5*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
                        ians2=1
                     endif
                  enddo
               endif
            endif
            cap=capouter+capinner

C Average capacity utilising new PCM apparent specific heat.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               appsht*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
            if(iphnl.eq.1.and.ians1.eq.1)then
               capinner=1.3*1005.5*(thrmli(icomp,isurf,iinner,4)/2.)
            elseif(iphnl.eq.3.and.ians2.eq.1)then
               capouter=1.3*1005.5*(thrmli(icomp,isurf,iouter,4)/2.)
            endif
            capPCM=capouter+capinner

C capPCM value at previous time step.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               pcmshtp*shtsol*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
            if(iphnl.eq.1.and.ians1.eq.1)then
               capinner=1.3*1005.5*(thrmli(icomp,isurf,iinner,4)/2.)
            elseif(iphnl.eq.3.and.ians2.eq.1)then
               capouter=1.3*1005.5*(thrmli(icomp,isurf,iouter,4)/2.)
            endif
            capPCMp=capouter+capinner

            x1=cap/capPCMp
            x2=cap/capPCM
            xfactp=factp*pcmshtp*x1
            xfactf=pcmcon*x2
            if(iphnl.eq.3)then
               vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*xfactp
               vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x1
               qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
               vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*xfactf
               vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x2
               qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
            elseif(iphnl.eq.1.and.ians1.eq.1)then
               vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x1
               vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*xfactp
               qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
               vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x2
               vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*xfactf
               qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
            endif
         endif

C Save factf and pcmsht for use as factp and pcmshtp at next
C time step.
         pcmfac(icomp,isurf,nloc)=factf
         pcmshts(icomp,isurf,nloc)=pcmsht

C Maximum latent heat (J/kg).
C         shtliq=(shtliqa*tsoli+shtliqb)

C Latent heat stored (J) in part of the PCM layer represented by
C node nloc.
C         if(tfcm.lt.tmelt)then
C            spcmf=0.0
C         elseif(tfcm.gt.tsoli)then
C            call eclose(shtliqa,0.00,0.001,close)
C            if(close)then
C               spcmf=shtliq*(tsoli-tmelt)*
C     &             thrmli(icomp,isurf,ielem,2)*sna(icomp,isurf)*efthk
C            else
C               spcmf=0.5*shtliq*(tsoli-tmelt)*
C     &             thrmli(icomp,isurf,ielem,2)*sna(icomp,isurf)*efthk
C            endif
C         else
C            call eclose(shtliqa,0.00,0.001,close)
C            if(close)then
C               spcmf=(appsht-shtsol)*(tfcm-tmelt)*
C     &             thrmli(icomp,isurf,ielem,2)*sna(icomp,isurf)*efthk
C            else
C               spcmf=0.5*(appsht-shtsol)*(tfcm-tmelt)*
C     &             thrmli(icomp,isurf,ielem,2)*sna(icomp,isurf)*efthk
C            endif
C         endif

C Save node-related heat stored.
C         spcm(icomp,isurf,nloc)=spcmf

   10 continue

C Save total PCM heat stored.
C      spcmtf=spcm(icomp,isurf,inode-1)+spcm(icomp,isurf,inode)+
C     &                                      spcm(icomp,isurf,inode+1)


C Results output.
      ndatspm(ispmnod)=4
      if(nsinc.le.1)then
         strspm(ispmnod,1)='/AppCON'
         unitspm(ispmnod,1)='(W/(m.degC))'
         txtspm(ispmnod,1)='Average conductivity of PCM layer'

         strspm(ispmnod,2)='/SHCtot'
         unitspm(ispmnod,2)='(J/kg)'
         txtspm(ispmnod,2)='Total heat stored in PCM'

         strspm(ispmnod,3)='/SHCarea'
         unitspm(ispmnod,3)='(J/m^2)'
         txtspm(ispmnod,3)='Heat stored per PCM area'

         strspm(ispmnod,4)='/PCMFAC'
         unitspm(ispmnod,4)='(-)'
         txtspm(ispmnod,4)='PCMFAC of middle node'
      endif

      if(Rtot.gt.0)dataspm(ispmnod,1)=THKsum/Rtot
      if(THKsum.gt.0)dataspm(ispmnod,2)=sumSHTTHK
      dataspm(ispmnod,3)=(spcmtp+spcmtf)/2.0
      spcmtp=spcmtf
      dataspm(ispmnod,4)=pcmfac(icomp,isurf,inode)

C As a temporary measure, use zone-related variable, zspmf1, for
C for results library storage. Output is limited to one special
C material per zone.
      zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)

C XML output.
      call PCM_to_h3k(icomp,inode,isurf,ispmnod)

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP53 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,5(2x,i4))')
     &        'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &                       ispmloc(ispmnod,2),inode-1,inode,inode+1
      call edisp(itu,outs)
      call edisp(itu,' ')
      do iphnl=1,3
         if(iphnl.eq.1)then
            write(outs,'(a,i3)') 'Outermost node (',inode-1,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode-1),' J'
            call edisp(itu,outs)
         elseif(iphnl.eq.2)then
            write(outs,'(a,i3)') 'Middle node (',inode,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode),' J'
            call edisp(itu,outs)
         else
            write(outs,'(a,i3)') 'Intermost node (',inode+1,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode+1),' J'
            call edisp(itu,outs)
         endif
      enddo

  999 return
      end

C ***************** maxlatent
C Return the maximum latent heat based on different equations.
C variable returned is shtliq
      subroutine maxlatent(imethod,tfcm,tpcm,val1,val2,val3,val4,
     &           val5,val6,val7,val8,val9,val10,
     &            val11,appcon,appsht,spcmf)

      integer imethod  ! method to use
      real tfcm        ! PCM node temperature
      real tpcm        ! PCM node temperature
      real val1,val2,val3,val4,val5,val6     ! passed parameters
      real val7,val8,val9,val10,val11        ! passed parameters
      real appcon      ! calculated conductivity (W/m.degC)
      real appsht      ! calculated apparent heat capacity (J/kg.degC)
      real spcmf       ! calculated stored latent heat (J/kg)
C Variables for all methods
      real tsoli       ! solification temperature (degC)
      real tmelt       ! melting temperature (degC)
      real shtsol      ! sensible heat capacity in solid phase (J/kg.degC)
      real consol      ! conductivity in solid phase (W/m.degC)
      real conliq      ! conductivity in liquid phase (W/m.degC)
      real shtliq      ! maximum stored latetn heat (J/kg)

C Local variables.
C Method 1
      real shtliqa     ! a coeficient of linear eqn (J/kg.degC^2)
      real shtliqb     ! b coefficient (J/kg.degC)
C Method 2
      real sht2        ! sensible heat capacity in liquid phase
      real shtl        ! latent heat of the PCM (J/kg.degC).
      real shta        ! parameter a of latent heat of the PCM
      real shtb        ! parameter b of latent heat of the PCM
      real c1          ! parameter c1 of latent heat of the PCM
C Method 3
      real shtc        ! coefficient c for apparent heat equation
      real shtd        ! coefficient d for apparent heat equation
      real shte        ! coefficient e for apparent heat equation
      real shtf        ! coefficient f for apparent heat equation


C Latent heat stored (J/kg).
      if(imethod.eq.1)then ! use logic of SPMCMP53

        tsoli  =val1
        tmelt  =val2
        shtsol =val3
        consol =val4
        conliq =val5
        shtliqa=val6
        shtliqb=val7
        shtliq=(0.5*shtliqa*(tsoli+tmelt)+shtliqb)*(tsoli-tmelt)
        if(tfcm.lt.tmelt)then
          appcon=consol
          appsht=shtsol
          spcmf=0.0
        elseif(tfcm.gt.tsoli)then
          appcon=conliq
          appsht=shtsol
          spcmf=shtliq
        else
          appcon=(consol+conliq)/2.0
          appsht=shtsol+(shtliqa*tfcm+shtliqb)
          spcmf=(0.5*shtliqa*(tfcm+tmelt)+shtliqb)*(tfcm-tmelt)
        endif

      elseif(imethod.eq.2)then ! use logic of SPMCMP54
        tsoli =val1
        tmelt =val2
        shtsol=val3
        consol=val4
        conliq=val5
        sht2  =val6
        shtl  =val7
        shta=-shtb*tsoli+1.00002397*log(shtl)-0.271249489
        shtb=0.76224
        c1=exp(shta)
        shtliq=(c1/shtb)*(exp(shtb*tsoli)-exp(shtb*tmelt))/1000
        if(tfcm.lt.tmelt)then
          appcon=consol
          appsht=shtsol
          spcmf=0.0
        elseif(tfcm.gt.tsoli)then
          appcon=conliq
          appsht=sht2
          spcmf=shtliq
        else
          appcon=(spcmf/shtliq)*conliq+((shtliq-spcmf)/shtliq)*consol
          appsht=c1*exp(shtb*tfcm)+shtsol
          spcmf=((c1/shtb)*exp(shtb*tfcm)-
     &                                (c1/shtb)*exp(shtb*tmelt))/1000
        endif
      
      elseif(imethod.eq.3)then ! use logic of SPMCMP55
        tsoli =val1
        tmelt =val2
        shtsol=val3
        consol=val4
        conliq=val5
        sht2  =val6
        shta  =val7
        shtb  =val8
        shtc  =val9
        shtd  =val10
        shte  =val11
C Time invariant coefficients for integration 
C of c(T)=(a+cT+eT**2)/(1+bT+dT**2) equation.
        c1=2*shta*(shtd**2)+(shtb**2)*shte-shtb*shtc*shtd-2*shtd*shte
        c2=2*shtd/sqrt((-(shtb**2)+4*shtd))
        c3=shtb/(sqrt(-shtb**2+4*shtd))
        c4=(shtd**2)*(sqrt(-shtb**2+4*shtd))
        c5=(shtc*shtd-shtb*shte)
        c6=shtb
        c7=shtd
        c8=2*(shtd**2)
        c9=shte/shtd

C Maximum value of latent heat, shtliq (J/kg), i.e. integration
C of c(t) equation from tmelt to tsoli.
        shtliq1=((c1*atan(c2*tsoli+c3))/c4)-((c1*atan(c2*tmelt+c3))/c4)
        shtliq2=((c5*log(1.0E+0+c6*tsoli+c7*(tsoli**2.0)))/c8)-
     &        ((c5*log(1.0E+0+c6*tmelt+c7*(tmelt**2.0)))/c8)
        shtliq3=c9*(tsoli-tmelt)
        shtliq=(shtliq1+shtliq2+shtliq3)/1.0E+03
        if(tfcm.lt.tmelt)then
          appcon=consol
          appsht=shtsol
          spcmf=0.0
        elseif(tfcm.gt.tsoli)then
          appcon=conliq
          appsht=sht2
          spcmf=shtliq
        else
          appcon=(spcmf/shtliq)*conliq+((shtliq-spcmf)/shtliq)*consol
          appsht=((shta+shtc*tfcm+shte*(tfcm**2))
     &                      /(1.0E0+shtb*tfcm+shtd*(tfcm**2)))+shtsol
          spcmf1=(c1*atan(c2*tfcm+c3))/c4-(c1*atan(c2*tmelt+c3))/c4
          spcmf2=(c5*log(1.0E+0+c6*tfcm+c7*(tfcm**2.0)))/c8-
     &                 (c5*log(1.0E+0+c6*tmelt+c7*(tmelt**2.0)))/c8
          spcmf3=c9*tfcm-c9*tmelt
          spcmf4=spcmf1+spcmf2+spcmf3
          spcmf=spcmf4/1.0E+03
        endif

      elseif(imethod.eq.4)then ! use logic of SPMCMP57
        tsoli  =val1
        tmelt  =val2
        shtsol =val3
        consol =val4
        conliq =val5
        shtliqa=val6
        shtliqb=val7
        tsoli2= val8
        tmelt2= val9
        shtliqc=val10
        shtliqd=val11

        if(tfcm.gt.tpcm)then ! heating
          shtliq=(0.5*shtliqa*(tsoli+tmelt)+shtliqb)*(tsoli-tmelt)
          if(tfcm.lt.tmelt)then
            appcon=consol
            appsht=shtsol
            spcmf=0.0000000001
          elseif(tfcm.gt.tsoli)then
            appcon=conliq
            appsht=shtsol
            spcmf=shtliq
          else 
            appcon=(spcmf/shtliq)*conliq+((shtliq-spcmf)/shtliq)*consol
            appsht=shtsol+(shtliqa*tfcm+shtliqb)
            spcmf=(0.5*shtliqa*(tfcm+tmelt)+shtliqb)*(tfcm-tmelt)
          endif
        elseif(tfcm.lt.tpcm)then ! cooling
          shtliq=(0.5*shtliqc*(tsoli2+tmelt2)+shtliqd)*
     &                                  (tsoli2-tmelt2)
          if(tfcm.lt.tmelt2)then
            appcon=consol
            appsht=shtsol
            spcmf=0.0000000001
          elseif(tfcm.gt.tsoli2)then
            appcon=conliq
            appsht=shtsol
            spcmf=shtliq
          else
            appcon=(spcmf/shtliq)*conliq+((shtliq-spcmf)/shtliq)*consol
            appsht=shtsol+(shtliqc*tfcm+shtliqd)
            spcmf=(0.5*shtliqc*(tfcm+tmelt2)+shtliqd)*(tfcm-tmelt2)
          endif
        else                  ! constant temperature
          if(tfcm.lt.tmelt2)then                
            appcon=consol
            appsht=shtsol
            spcmf=0.0000000001
          elseif(tfcm.gt.tsoli)then
            appcon=conliq
            appsht=shtsol
            spcmf=shtliq
          else 
            if(tfcm.lt.tpcm)then
             appcon=(spcmf/shtliq)*conliq+((shtliq-spcmf)/shtliq)*consol
             appsht=shtsol+(shtliqc*tfcm+shtliqd)
             spcmf=(0.5*shtliqc*(tfcm+tmelt2)+shtliqd)*(tfcm-tmelt2)
            else
             shtliq=(0.5*shtliqa*(tsoli+tmelt)+shtliqb)*(tsoli-tmelt)
             appcon=(spcmf/shtliq)*conliq+((shtliq-spcmf)/shtliq)*consol
             appsht=shtsol+(shtliqa*tfcm+shtliqb)
             spcmf=(0.5*shtliqa*(tfcm+tmelt)+shtliqb)*(tfcm-tmelt)
            endif
          endif
        endif
      endif
      return   ! finished
      end


C ***************** SPMCMP54 *****************
C A model of phase change material taken from the PhD thesis of
C Sabine Hoffmann, University of Weimar, September 2006, pp139-140.
C The functions are of the type 'exponential'. The model parameters
C are defined in file databases/mscomp.db1.

      subroutine spmcmp54(icomp,ispmnod)
#include "building.h"
#include "geometry.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      common/fvalc/tfc(mcom,ms,mn),qfc(mcom)
      common/fvals/tfs(mcom,ms),qfs(mcom)
      common/vthp14/thrmli(mcom,ms,me,7)
      common/prec9/nconst(mcom),nelts(mcom,ms),ngaps(mcom,ms),
     &             npgap(mcom,ms,mgp)
      common/prec13/c(mcom,ms,mn,2),qc(mcom,ms,mn)
      common/pers/isd1,ism1,isd2,ism2,isds,isdf,ntstep
      common/vthp15/vcp(ms,mn,2),qcp(ms,mn)
      common/vthp16/vcf(ms,mn,2),qcf(ms,mn)
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)

C Special materials.
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)

C Phase change material.
      common/pcm02/spcm(mcom,ms,mn),spcmtf,spcmtp
      common/pcm03/pcmfac(mcom,ms,mn),pcmshts(mcom,ms,mn)

      character spflnam*72,spmlabel*16,outs*124,strspm*16,unitspm*16,
     &          txtspm*72

C PCM layer pointers.
      if(icomp.ne.ispmloc(ispmnod,1))then
         write(outs,*)'SPMAT54 fatal error: incorrect zone assignment!'
         call edisp(iuout,outs)
         stop
      endif
      isurf=ispmloc(ispmnod,2)
      inode=ispmloc(ispmnod,3)
      ielem=nint(real(inode)/2.)

C Melting and solidification temperatures (degC).
      tmelt=spmdat(ispmnod,1)
      tsoli=spmdat(ispmnod,2)

C Conductivity in solid and liquid state (W/m.degC).
      consol=spmdat(ispmnod,3)
      conliq=spmdat(ispmnod,4)

C Specific heat capacity in solid and liquid state (J/kg.degC).
      c1=exp(shta)
      shtsol=spmdat(ispmnod,5)
      sht2=spmdat(ispmnod,6)

C Latent heat of the PCM (J/kg.degC).
      shtl=spmdat(ispmnod,7)
C      shtb=0.76224
C      shta=-shtb*tsoli+1.00002397*log(shtl)-0.271249489
C      c1=exp(shta)

C Maximum value of latent heat, shtliq (J/kg), i.e. integration
C of c(t) equation from tmelt to tsoli.
C      shtliq=(c1/shtb)*(exp(shtb*tsoli)-exp(shtb*tmelt))/1000

C Initialisations for xml output.
      Rtot = 0.0
      THKsum = 0.0
      sumSHTTHK = 0.0

C Process the 3 nodes of the PCM layer.
      spcmtf=0.0
      do 10 iphnl=1,3
         if(iphnl.eq.1)then
            nloc=inode-1
            efthk=thrmli(icomp,isurf,ielem,4)*0.25
         elseif(iphnl.eq.2)then
            nloc=inode
            efthk=thrmli(icomp,isurf,ielem,4)*0.5
         elseif(iphnl.eq.3)then
            nloc=inode+1
            efthk=thrmli(icomp,isurf,ielem,4)*0.25
         endif

C Total thickness of PCM.
         THKsum=THKsum+efthk

C Establish temperature of the PCM material node from tfc if the node
C is intra-construction and tfs if located at the internal surface.
         if(iphnl.eq.3.and.ielem.eq.nelts(icomp,isurf))then
            tfcm=tfs(icomp,isurf)
         else
            tfcm=tfc(icomp,isurf,nloc)
         endif

C If phase change is underway, calculate the new value of
C conductivity (W/m.degC):
C  appcon = consol          for tfcm < tmelt (solid phase);
C         = conliq          for tfcm > tsoli (liquid phase);
C         = (spcmc/shtliq)*conliq+[(shtliq-spcmc)/shtliq]*consol
C           where spcmc=[(c1/shtb)*exp(shtb*tfcm)-(c1/shtb)*exp(shtb*tmelt)]/1000
C                shtliq=(c1/shtb)*[exp(shtb*tsoli)-exp(shtb*tmelt)]/1000
C                           for tmelt < tfcm < tsoli (mixed phase);
C ... and apparent heat capacity (J/kg.degC):
C  appsht = shtsol          for tfcm < tmelt (solid phase);
C         = sht2            for tfcm > tsoli (liquid phase);
C         = c1*exp(shtb*tfcm)+shtsol
C                           for tmelt < tfcm < tsoli (mixed phase).
C         if(tfcm.lt.tmelt)then
C            appcon=consol
C            appsht=shtsol
C         elseif(tfcm.gt.tsoli)then
C            appcon=conliq
C            appsht=sht2
C         else
C            spcmc=spcm(icomp,isurf,nloc)
C            appcon=(spcmc/shtliq)*conliq+((shtliq-spcmc)/shtliq)*consol
C            appsht=c1*exp(shtb*tfcm)+shtsol
C         endif

         imethod=2
         val1=tsoli
         val2=tmelt
         val3=shtsol
         val4=consol
         val5=conliq
         val6=sht2
         val7=shtl
         val8=0.0
         val9=0.0
         val10=0.0
         val11=0.0
         call maxlatent(imethod,tfcm,tpcm,val1,val2,val3,val4,
     &           val5,val6,val7,val8,val9,val10,
     &            val11,appcon,appsht,spcmf)

C Parameters for xml output.
         Rtot=Rtot+efthk/appcon
         sumSHTTHK=sumSHTTHK+appsht

C Establish present value of factp and pcmshtp as last used value
C of factf and pcmsht respectively.
         factp=pcmfac(icomp,isurf,nloc)
         pcmshtp=pcmshts(icomp,isurf,nloc)
         if(nsinc.eq.1)then
            factp=0
         endif

C Calculate new factf; this is used below to remove original values
C of PCM layer conductivity and specific heat as used to establish the
C self- and cross- coupling coefficients of the nodal energy balance
C equations, and substitute the new values as established above.
         pcmcon=appcon/consol
         pcmsht=appsht/shtsol
         factf=pcmcon/pcmsht

C Modify present and future value of 'c' and 'qc' coefficients (i.e.
C coeficients of nodal energy balance equations.
         x1=1./pcmshtp
         if(nsinc.eq.1)then
            x1=0
         endif
         x2=1./pcmsht
         if(iphnl.eq.1.and.ielem.eq.1)then                 !- outermost node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x1
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factp
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x2
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factf
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.2)then                            !- centre node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factp
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factp
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factf
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factf
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.3.and.ielem.eq.nelts(icomp,isurf))then  !- internal surface node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factp
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x1
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factf
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x2
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.1.or.iphnl.eq.3)then              !- interface node

C Redefine modification factors to take account of weighted node
C thermal capacity.
            if(iphnl.eq.1)then
               iouter=ielem
               iinner=ielem-1
            else
               iinner=ielem
               iouter=ielem+1
            endif

C Original averaged capacity before adjustment.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               thrmli(icomp,isurf,iouter,3)*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,ielem-1,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)

C Cater for an adjacent air gap.
            ians1=0
            ians2=0
            if(ngaps(icomp,isurf).ne.0)then
               if(iphnl.eq.1)then
                  do igap=1,ngaps(icomp,isurf)
                     if(iinner.eq.npgap(icomp,isurf,igap))then
                        capinner=1.3*1005.5*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
                        ians1=1
                     endif
                  enddo
               elseif(iphnl.eq.3)then
                  do igap=1,ngaps(icomp,isurf)
                     if(iouter.eq.npgap(icomp,isurf,igap))then
                        capouter=1.3*1005.5*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
                        ians2=1
                     endif
                  enddo
               endif
            endif
            cap=capouter+capinner

C Average capacity utilising new PCM apparent specific heat.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               appsht*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
            if(iphnl.eq.1.and.ians1.eq.1)then
               capinner=1.3*1005.5*(thrmli(icomp,isurf,iinner,4)/2.)
            elseif(iphnl.eq.3.and.ians2.eq.1)then
               capouter=1.3*1005.5*(thrmli(icomp,isurf,iouter,4)/2.)
            endif
            capPCM=capouter+capinner

C capPCM value at previous time step.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               pcmshtp*shtsol*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
            if(iphnl.eq.1.and.ians1.eq.1)then
               capinner=1.3*1005.5*(thrmli(icomp,isurf,iinner,4)/2.)
            elseif(iphnl.eq.3.and.ians2.eq.1)then
               capouter=1.3*1005.5*(thrmli(icomp,isurf,iouter,4)/2.)
            endif
            capPCMp=capouter+capinner

            x1=cap/capPCMp
            x2=cap/capPCM
            xfactp=factp*pcmshtp*x1
            xfactf=pcmcon*x2
            if(iphnl.eq.3)then
               vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*xfactp
               vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x1
               qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
               vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*xfactf
               vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x2
               qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
            elseif(iphnl.eq.1.and.ians1.eq.1)then
               vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x1
               vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*xfactp
               qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
               vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x2
               vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*xfactf
               qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
            endif
         endif

C Save factf and pcmsht for use as factp and pcmshtp at next
C time step.
         pcmfac(icomp,isurf,nloc)=factf
         pcmshts(icomp,isurf,nloc)=pcmsht

C Calculate the value of latent heat stored (J) in part of the PCM
C layer represented by node nloc.
C         if(tfcm.lt.tmelt)then
C            spcmf=0.0
C         elseif(tfcm.gt.tsoli)then
C            spcmf=shtliq
C         else
C            spcmf=((c1/shtb)*exp(shtb*tfcm)-
C     &                                (c1/shtb)*exp(shtb*tmelt))/1000
C         endif

C Save node-related heat stored.
C         spcm(icomp,isurf,nloc)=spcmf
10    continue

C Save total PCM heat stored.
C      spcmtf=spcm(icomp,isurf,inode-1)+spcm(icomp,isurf,inode)+
C     &                                      spcm(icomp,isurf,inode+1)

C Results output.
      ndatspm(ispmnod)=4
      if(nsinc.le.1)then
         strspm(ispmnod,1)='/AppCON'
         unitspm(ispmnod,1)='(W/(m.degC))'
         txtspm(ispmnod,1)='Average conductivity of PCM layer'

         strspm(ispmnod,2)='/SHCtot'
         unitspm(ispmnod,2)='(J/kg)'
         txtspm(ispmnod,2)='Total heat stored in PCM'

         strspm(ispmnod,3)='/SHCarea'
         unitspm(ispmnod,3)='(J/m^2)'
         txtspm(ispmnod,3)='Heat stored per PCM area'

         strspm(ispmnod,4)='/PCMFAC'
         unitspm(ispmnod,4)='(-)'
         txtspm(ispmnod,4)='PCMFAC of middle node'
      endif

      if(Rtot.gt.0)dataspm(ispmnod,1)=THKsum/Rtot
      if(THKsum.gt.0)dataspm(ispmnod,2)=sumSHTTHK
      dataspm(ispmnod,3)=((spcmtp+spcmtf)/2.0)
      spcmtp=spcmtf
      dataspm(ispmnod,4)=pcmfac(icomp,isurf,inode)

C As a temporary measure, use zone-related variable, zspmf1, for
C results library storage. Output is limited to one special material
C per zone.
      zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)

C XML output.
      call PCM_to_h3k(icomp,inode,isurf,ispmnod)

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.EQ.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP54 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,5(2x,i4))')
     &        'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &                       ispmloc(ispmnod,2),inode-1,inode,inode+1
      call edisp(itu,outs)
      call edisp(itu,' ')
      do iphnl=1,3
         if(iphnl.eq.1)then
            write(outs,'(a,i3)') 'Outermost node (',inode-1,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode-1),' J'
            call edisp(itu,outs)
         elseif(iphnl.eq.2)then
            write(outs,'(a,i3)') 'Middle node (',inode,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode),' J'
            call edisp(itu,outs)
         else
            write(outs,'(a,i3)') 'Intermost node (',inode+1,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode+1),' J'
            call edisp(itu,outs)
         endif
      enddo
  999 return
      end

C ***************** SPMCMP55 *****************
C A model of phase change material taken from the PhD thesis of
C Sabine Hoffmann, University of Weimar, September 2006, pp139-140.
C The functions used are of type 'rational'. The model parameters
C are defined in file databases/mscomp.db1

      subroutine spmcmp55(icomp,ispmnod)
#include "building.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      common/fvalc/tfc(mcom,ms,mn),qfc(mcom)
      common/fvals/tfs(mcom,ms),qfs(mcom)
      common/vthp14/thrmli(mcom,ms,me,7)
      common/prec9/nconst(mcom),nelts(mcom,ms),ngaps(mcom,ms),
     &             npgap(mcom,ms,mgp)
      common/prec13/c(mcom,ms,mn,2),qc(mcom,ms,mn)
      common/vthp15/vcp(ms,mn,2),qcp(ms,mn)
      common/vthp16/vcf(ms,mn,2),qcf(ms,mn)
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)

C Special materials.
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)

C Phase change material.
      common/pcm02/spcm(mcom,ms,mn),spcmtf,spcmtp
      common/pcm03/pcmfac(mcom,ms,mn),pcmshts(mcom,ms,mn)

      character spflnam*72,spmlabel*16,outs*124,strspm*16,unitspm*16,
     &          txtspm*72

C PCM layer pointers.
      if(icomp.ne.ispmloc(ispmnod,1))then
         write(outs,*)'SPMAT55 fatal error: incorrect zone assignment!'
         call edisp(iuout,outs)
         stop
      endif
      isurf=ispmloc(ispmnod,2)
      inode=ispmloc(ispmnod,3)
      ielem=nint(real(inode)/2.)

C Melting and solidification temperatures (degC).
      tmelt=spmdat(ispmnod,1)
      tsoli=spmdat(ispmnod,2)

C Conductivity in solid and liquid state (W/m.degC).
      consol=spmdat(ispmnod,3)
      conliq=spmdat(ispmnod,4)

C Specific heat capacity in solid and liquid state (J/kg.degC)
      shtsol=spmdat(ispmnod,5)
      sht2=spmdat(ispmnod,6)

C Coefficients for apparent heat equation: c(T)=(a+cT+eT**2)/(1+bT+dT**2)
      shta=spmdat(ispmnod,7)
      shtb=spmdat(ispmnod,8)
      shtc=spmdat(ispmnod,9)
      shtd=spmdat(ispmnod,10)
      shte=spmdat(ispmnod,11)

C Time invariant coefficients for integration of c(T) equation.
C      c1=2*shta*(shtd**2)+(shtb**2)*shte-shtb*shtc*shtd-2*shtd*shte
C      c2=2*shtd/sqrt((-(shtb**2)+4*shtd))
C      c3=shtb/(sqrt(-shtb**2+4*shtd))
C      c4=(shtd**2)*(sqrt(-shtb**2+4*shtd))
C      c5=(shtc*shtd-shtb*shte)
C      c6=shtb
C      c7=shtd
C      c8=2*(shtd**2)
C      c9=shte/shtd

C Maximum value of latent heat, shtliq (J/kg), i.e. integration
C of c(t) equation from tmelt to tsoli.
C      shtliq1=((c1*atan(c2*tsoli+c3))/c4)-((c1*atan(c2*tmelt+c3))/c4)
C      shtliq2=((c5*log(1.0E+0+c6*tsoli+c7*(tsoli**2.0)))/c8)-
C     &        ((c5*log(1.0E+0+c6*tmelt+c7*(tmelt**2.0)))/c8)
C      shtliq3=c9*(tsoli-tmelt)
C      shtliq=(shtliq1+shtliq2+shtliq3)/1.0E+03

C Initialisations for xml output.
      Rtot = 0.0
      THKsum = 0.0
      sumSHTTHK = 0.0

C Process the 3 nodes of the PCM layer.
      spcmtf=0.0
      do 10 iphnl=1,3
         if(iphnl.eq.1)then
            nloc=inode-1
            efthk=thrmli(icomp,isurf,ielem,4)*0.25
         elseif(iphnl.eq.2)then
            nloc=inode
            efthk=thrmli(icomp,isurf,ielem,4)*0.5
         elseif(iphnl.eq.3)then
            nloc=inode+1
            efthk=thrmli(icomp,isurf,ielem,4)*0.25
         endif

C Total thickness of PCM.
         THKsum=THKsum+efthk

C Establish temperature of the PCM material node from tfc if the node
C is intra-construction and tfs if located at the internal surface.
         if(iphnl.eq.3.and.ielem.eq.nelts(icomp,isurf))then
            tfcm=tfs(icomp,isurf)
         else
            tfcm=tfc(icomp,isurf,nloc)
         endif

C Current value of conductivity and specific heat:
C         if(tfcm.lt.tmelt)then
C            appcon=consol
C            appsht=shtsol
C         elseif(tfcm.gt.tsoli)then
C            appcon=conliq
C            appsht=sht2
C         else
C            spcmc=spcm(icomp,isurf,nloc)
C            appcon=(spcmc/shtliq)*conliq+((shtliq-spcmc)/shtliq)*consol
C            appsht=((shta+shtc*tfcm+shte*(tfcm**2))
C     &                      /(1.0E0+shtb*tfcm+shtd*(tfcm**2)))+shtsol
C         endif

         imethod=3
         val1=tsoli
         val2=tmelt
         val3=shtsol
         val4=consol
         val5=conliq
         val6=sht2
         val7=shta
         val8=shtb
         val9=shtc
         val10=shtd
         val11=shte
         call maxlatent(imethod,tfcm,tpcm,val1,val2,val3,val4,
     &           val5,val6,val7,val8,val9,val10,
     &            val11,appcon,appsht,spcmf)

C Parameters for xml output.
         Rtot=Rtot+efthk/appcon
         sumSHTTHK=sumSHTTHK+appsht

C Establish present value of factp and pcmshtp as last used value
C of factf and pcmsht respectively.
         factp=pcmfac(icomp,isurf,nloc)
         pcmshtp=pcmshts(icomp,isurf,nloc)
         if(nsinc.eq.1)then
            factp=0
         endif

C Calculate new factf; this is used below to remove original values
C of PCM layer conductivity and specific heat as used to establish the
C self- and cross- coupling coefficients of the nodal energy balance
C equations, and substitute the new values as established above. An
C adjusted factf is required where the PCM node separates the PCM layer
C from a non-PCM layer.
         pcmcon=appcon/consol
         pcmsht=appsht/shtsol
         factf=pcmcon/pcmsht

C Modify present and future value of 'c' and 'qc' coefficients (i.e.
C coeficients of nodal energy balance equations.
         x1=1./pcmshtp
         if(nsinc.eq.1)then
            x1=0
         endif
         x2=1./pcmsht
         if(iphnl.eq.1.and.ielem.eq.1)then                !- outermost node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x1
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factp
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x2
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factf
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.2)then                           !- centre node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factp
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factp
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factf
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factf
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.3.and.ielem.eq.nelts(icomp,isurf))then !- internal surface node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factp
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x1
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factf
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x2
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.1.or.iphnl.eq.3)then             !- interface node

C Redefine modification factors to take account of weighted node
C thermal capacity.
            if(iphnl.eq.1)then
               iouter=ielem
               iinner=ielem-1
            else
               iinner=ielem
               iouter=ielem+1
            endif

C Original averaged capacity before adjustment.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               thrmli(icomp,isurf,iouter,3)*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,ielem-1,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)

C Cater for an adjacent air gap.
            ians1=0
            ians2=0
            if(ngaps(icomp,isurf).ne.0)then
               if(iphnl.eq.1)then
                  do igap=1,ngaps(icomp,isurf)
                     if(iinner.eq.npgap(icomp,isurf,igap))then
                        capinner=1.3*1005.5*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
                        ians1=1
                     endif
                  enddo
               elseif(iphnl.eq.3)then
                  do igap=1,ngaps(icomp,isurf)
                     if(iouter.eq.npgap(icomp,isurf,igap))then
                        capouter=1.3*1005.5*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
                        ians2=1
                     endif
                  enddo
               endif
            endif
            cap=capouter+capinner

C Average capacity utilising new PCM apparent specific heat.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               appsht*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
            if(iphnl.eq.1.and.ians1.eq.1)then
               capinner=1.3*1005.5*(thrmli(icomp,isurf,iinner,4)/2.)
            elseif(iphnl.eq.3.and.ians2.eq.1)then
               capouter=1.3*1005.5*(thrmli(icomp,isurf,iouter,4)/2.)
            endif
            capPCM=capouter+capinner

C capPCM value at previous time step.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               pcmshtp*shtsol*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
            if(iphnl.eq.1.and.ians1.eq.1)then
               capinner=1.3*1005.5*(thrmli(icomp,isurf,iinner,4)/2.)
            elseif(iphnl.eq.3.and.ians2.eq.1)then
               capouter=1.3*1005.5*(thrmli(icomp,isurf,iouter,4)/2.)
            endif
            capPCMp=capouter+capinner

            x1=cap/capPCMp
            x2=cap/capPCM
            xfactp=factp*pcmshtp*x1
            xfactf=pcmcon*x2
            if(iphnl.eq.3)then
               vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*xfactp
               vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x1
               qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
               vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*xfactf
               vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x2
               qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
            elseif(iphnl.eq.1.and.ians1.eq.1)then
               vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x1
               vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*xfactp
               qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
               vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x2
               vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*xfactf
               qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
            endif
         endif

C Save factf and pcmsht for use as factp and pcmshtp at next
C time step.
         pcmfac(icomp,isurf,nloc)=factf
         pcmshts(icomp,isurf,nloc)=pcmsht

C Calculate the value of latent heat stored (J) in part of the PCM
C layer represented by node nloc.
C         if(tfcm.lt.tmelt)then
C            spcmf=0.0
C         elseif(tfcm.gt.tsoli)then
C            spcmf=shtliq
C         else
C            spcmf1=(c1*atan(c2*tfcm+c3))/c4-(c1*atan(c2*tmelt+c3))/c4
C            spcmf2=(c5*log(1.0E+0+c6*tfcm+c7*(tfcm**2.0)))/c8-
C     &                 (c5*log(1.0E+0+c6*tmelt+c7*(tmelt**2.0)))/c8
C            spcmf3=c9*tfcm-c9*tmelt
C            spcmf4=spcmf1+spcmf2+spcmf3
C            spcmf=spcmf4/1.0E+03
C         endif

C Save node-related heat stored.
C         spcm(icomp,isurf,nloc)=spcmf
10    continue

C Save total PCM heat stored.
C      spcmtf=spcm(icomp,isurf,inode-1)+spcm(icomp,isurf,inode)+
C     &                                      spcm(icomp,isurf,inode+1)

C Results output.
      ndatspm(ispmnod)=4
      if(nsinc.le.1)then
         strspm(ispmnod,1)='/AppCON'
         unitspm(ispmnod,1)='(W/(m.degC))'
         txtspm(ispmnod,1)='Average conductivity of PCM layer'

         strspm(ispmnod,2)='/SHCtot'
         unitspm(ispmnod,2)='(J/kg)'
         txtspm(ispmnod,2)='Tot. heat stored in PCM'

         strspm(ispmnod,3)='/SHCarea'
         unitspm(ispmnod,3)='(J/m2)'
         txtspm(ispmnod,3)='Heat stored per PCM area'

         strspm(ispmnod,4)='/PCMFAC'
         unitspm(ispmnod,4)='(-)'
         txtspm(ispmnod,4)='PCMFAC of middle node'
      endif

      if(Rtot.gt.0)dataspm(ispmnod,1)=THKsum/Rtot
      if(THKsum.gt.0)dataspm(ispmnod,2)=sumSHTTHK
      dataspm(ispmnod,3)=(spcmtp+spcmtf)/2.0
      spcmtp=spcmtf
      dataspm(ispmnod,4)=pcmfac(icomp,isurf,inode)

C As a temporary measure, use zone-related variables, zspmf1, for
C results library storage. Output is limited to one special material
C per zone.
      zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)

C XML output.
      call PCM_to_h3k(icomp,inode,isurf,ispmnod)

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP55 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,5(2x,i4))')
     &        'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &                       ispmloc(ispmnod,2),inode-1,inode,inode+1
      call edisp(itu,outs)
      call edisp(itu,' ')
      do iphnl=1,3
         if(iphnl.eq.1)then
            write(outs,'(a,i3)') 'Outermost node (',inode-1,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode-1),' J'
            call edisp(itu,outs)
         elseif(iphnl.eq.2)then
            write(outs,'(a,i3)') 'Middle node (',inode,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode),' J'
            call edisp(itu,outs)
         else
            write(outs,'(a,i3)') 'Intermost node (',inode+1,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode+1),' J'
            call edisp(itu,outs)
         endif
      enddo
  999 return
      end

C ***************** SPMCMP56 *****************
C Phase change material model: extended approach allowing sub-cooling.
C Created by Achim Geissler, July 30 2008. Parameters are defined in
C file databases/mscomp.db1. Calls:
C    LatentHeat(k,T1,T2) - calculate integral of latent heat between
C                          T1 and T2 for k=1 'standard' range and
C                          k=0 sub-cooled PCM;
C    PCMStore()          - calculate change in stored latent heat,
C                          apparent conduction and apparent specific
C                          heat capacity.
C
C 170331: Extension added to allow for arbitrary PCM curves. Use spline function
C for these. Penalty in run-time.

      subroutine spmcmp56(icomp,ispmnod)
#include "building.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/pvalc/tpc(mcom,ms,mn),qpc(mcom)
      common/pvals/tps(mcom,ms),qps(mcom)
      common/fvalc/tfc(mcom,ms,mn),qfc(mcom)
      common/fvals/tfs(mcom,ms),qfs(mcom)
      common/prec9/nconst(mcom),nelts(mcom,ms),ngaps(mcom,ms),
     &             npgap(mcom,ms,mgp)
      common/prec13/c(mcom,ms,mn,2),qc(mcom,ms,mn)
      common/vthp14/thrmli(mcom,ms,me,7)
      common/vthp15/vcp(ms,mn,2),qcp(ms,mn)
      common/vthp16/vcf(ms,mn,2),qcf(ms,mn)
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)

C Special materials.
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)

C Phase change materials.
      common/pcm02/spcm(mcom,ms,mn),spcmtf,spcmtp
      common/pcm03/pcmfac(mcom,ms,mn),pcmshts(mcom,ms,mn)
      common/pcm05/appconp(mcom,ms,mn),appshcp(mcom,ms,mn),
     &                                                 pk(mcom,ms,mn)
      integer pk   !- Present value of switch 'k' (-).

      common/pcmparm/tsoli,tmelt,sub,Lges,consol,conliq,shcsol,shcliq,
     &               a1,b1,c1,d1,e1,f1,a2,b2,c2,d2,e2,f2,LHSimple
      real Lges         !- Total latent heat capacity of PCM (J/kg).
      logical LHSimple  !- .true. is previous simple model, .false. is
                        !  extended model.

C DSC data / spline function data
      common/pcmspline/temp1d(mspmnod,MSPMSPLM),
     &          heat1d(mspmnod,MSPMSPLM),
     &          cool1d(mspmnod,MSPMSPLM),heat2d(mspmnod,MSPMSPLM),
     &          cool2d(mspmnod,MSPMSPLM),nxy(mspmnod),
     &          bUseSpline(mspmnod),fnamDSCdat(mspmnod)
      real temp1d   ! DSC temperature data values, degC
      real heat1d   ! DSC heating enthalpy values, J/(kg K)
      real cool1d   ! DSC cooling enthalpy values, J/(kg K)
      real heat2d   ! 2nd derivatives at spline x values
      real cool2d
      integer nxy             ! number of data sets
      logical bUseSpline      ! .true. means DSC data file available, use
                              ! cubic spline function for apparent specific
                              ! heat function.
      character fnamDSCdat*72 ! DSC data file name for current spm entry

C Local variables.
      real yp1,ypn            ! values for start and end points for spline
      real LayFrac(3)
      integer presk
      integer db(3)     !- Debug variable, stores 'case' of PCM heating/cooling.

C Data.
      data LayFrac /0.25, 0.5, 0.25/  !- PCM split into 3 layers.

      character spflnam*72,spmlabel*16,outs*124,strspm*16,unitspm*16,
     &          txtspm*72

C PCM location.
      if(icomp.ne.ispmloc(ispmnod,1))then
         write(outs,*)'SPMAT56 fatal error: incorrect zone assignment!'
         call edisp(iuout,outs)
         stop
      endif
      isurf=ispmloc(ispmnod,2)
      inode=ispmloc(ispmnod,3)
      ielem=nint(real(inode)/2.)

C Melting and solidification temperatures (degC), and sub-cooling
C temperature difference (degC).
      tmelt=spmdat(ispmnod,1)
      tsoli=spmdat(ispmnod,2)
      sub=spmdat(ispmnod,3)

C Conductivity in solid and liquid phase (W/m.K).
      consol=spmdat(ispmnod,4)
      conliq=spmdat(ispmnod,5)

C Specific heat capacity in solid and liquid phase (J/kg.K)
      shcsol=spmdat(ispmnod,6)
      shcliq=spmdat(ispmnod,7)

C Total latent heat capacity of PCM (J/kg).
      Lges=spmdat(ispmnod,8)

      if (spmdat(ispmnod,9).eq.0) then
         LHSimple=.true.
      elseif (spmdat(ispmnod,9).eq.1) then
         LHSimple=.false.
      else
C       Use spline, read DSC data from file
        LHSimple=.true.

      endif

      if (.not.LHSimple) then
C       Coefficients for non-subcooling (a1-f1) and sub-cooling (a2-f2)
C       integral of latent heat - lh(t1,t2)=exp(a)/b*(exp(b*T1)-exp(b*t2))
C        - and latent heat function from c(T)=(a+cT+eT**2)/(1+bT+dT**2)
C       extended to (t**f)*c(t) (see thesis by Sabine Hoffmann).
        a1=spmdat(ispmnod,10)
        b1=spmdat(ispmnod,11)
        c1=spmdat(ispmnod,12)
        d1=spmdat(ispmnod,13)
        e1=spmdat(ispmnod,14)
        f1=spmdat(ispmnod,15)
        a2=spmdat(ispmnod,16)
        b2=spmdat(ispmnod,17)
        c2=spmdat(ispmnod,18)
        d2=spmdat(ispmnod,19)
        e2=spmdat(ispmnod,20)
        f2=spmdat(ispmnod,21)
      endif

C Initialisations.
      if(nsinc.le.1)then
         do iphnl=1,3
            nloc=inode-2+iphnl
            pk(icomp,isurf,nloc)=1
            spcm(icomp,isurf,nloc)=0.0
            appconp(icomp,isurf,nloc)=consol
            appshcp(icomp,isurf,nloc)=shcsol
         enddo
         if (bUseSpline(ispmnod)) then
C          Read DSC file and set up cubic spline data
           call ReadDSCDataFile(ispmnod)
           yp1=0.
           ypn=0.
C          Heating curve
           call spline(temp1d(ispmnod,:), heat1d(ispmnod,:),
     &                 nxy(ispmnod), yp1, ypn, heat2d(ispmnod,:))

C          Cooling curve
           call spline(temp1d(ispmnod,:), cool1d(ispmnod,:),
     &                 nxy(ispmnod), yp1, ypn, cool2d(ispmnod,:))

         endif
      endif

C Initialisations for xml output.
      Rtot = 0.0
      aveAppSHC= 0.0
      sumStoredLH = 0.0

C Process the 3 nodes of the PCM layer.
      do 10 iphnl=1,3
         nloc=inode-2+iphnl
         efthk=thrmli(icomp,isurf,ielem,4)*LayFrac(iphnl)

C Establish temperature of the PCM material node from tfc if the node
C is intra-construction and tfs if located at the internal surface.
         if(iphnl.eq.3.and.ielem.eq.nelts(icomp,isurf))then
            tfcm=tfs(icomp,isurf)
            tpcm=tps(icomp,isurf)
         else
            tfcm=tfc(icomp,isurf,nloc)
            tpcm=tpc(icomp,isurf,nloc)
         endif

C Retrieve stored latent heat, apparent conductivity and apparent
C specific heat from previous time step.
         storedLH=spcm(icomp,isurf,nloc)
         appcon=appconp(icomp,isurf,nloc)
         appshc=appshcp(icomp,isurf,nloc)
         presk=pk(icomp,isurf,nloc)

C Calculated change in stored latent heat, apparent conduction and
C apparent specific heat capacity based on present and future
C temperatures and state, 'k', of PCM.
         call PCMStore(ispmnod,tpcm,tfcm,presk,storedLH,appcon,appshc,
     &                                                     db(iphnl))

C Parameters for xml output.
         Rtot=Rtot+efthk/appcon
         aveAppSHC=aveAppSHC+appshc*LayFrac(iphnl)
         sumStoredLH=sumStoredLH+storedLH*LayFrac(iphnl)

C Establish present value of factp and pcmshtp as last used value
C of factf and pcmsht respectively.
         factp=pcmfac(icomp,isurf,nloc)
         pcmshtp=pcmshts(icomp,isurf,nloc)
         if(nsinc.eq.1)then
            factp=0.
            pcmshtp=1.
         endif

C Calculate new factf; this is used below to remove original values
C of PCM layer conductivity and specific heat as used to establish the
C self- and cross- coupling coefficients of the nodal energy balance
C equations, and substitute the new values as established above. An
C adjusted factf is required where the PCM node separates the PCM layer
C from a non-PCM layer.
         pcmcon=appcon/consol
         pcmsht=appshc/shcsol
         factf=pcmcon/pcmsht

C Modify present and future value of 'c' and 'qc' coefficients (i.e.
C coeficients of nodal energy balance equations.
         x1=1./pcmshtp
         if(nsinc.eq.1)then
            x1=0.
         endif
         x2=1./pcmsht
         if(iphnl.eq.1.and.ielem.eq.1)then              !- outermost node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x1
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factp
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x2
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factf
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.2)then                         !- centre node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factp
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factp
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factf
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factf
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.3.and.ielem.eq.nelts(icomp,isurf))then !- internal surface node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factp
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x1
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factf
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x2
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.1.or.iphnl.eq.3)then           !- interface node

C Redefine modification factors to take account of weighted node
C thermal capacity.
            if(iphnl.eq.1)then
               iouter=ielem
               iinner=ielem-1
            else
               iinner=ielem
               iouter=ielem+1
            endif

C Original averaged capacity before adjustment.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               thrmli(icomp,isurf,iouter,3)*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)

C Cater for an adjacent air gap.
            ians1=0
            ians2=0
            if(ngaps(icomp,isurf).ne.0)then
               if(iphnl.eq.1)then
                  do igap=1,ngaps(icomp,isurf)
                     if(iinner.eq.npgap(icomp,isurf,igap))then
                        capinner=1.3*1005.5*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
                        ians1=1
                     endif
                  enddo
               elseif(iphnl.eq.3)then
                  do igap=1,ngaps(icomp,isurf)
                     if(iouter.eq.npgap(icomp,isurf,igap))then
                        capouter=1.3*1005.5*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
                        ians2=1
                     endif
                  enddo
               endif
            endif
            cap=capouter+capinner

C Average capacity utilising new PCM apparent specific heat.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               appshc*  ! appsht in t53, but should be correct, here!
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
            if(iphnl.eq.1.and.ians1.eq.1)then
               capinner=1.3*1005.5*(thrmli(icomp,isurf,iinner,4)/2.)
            elseif(iphnl.eq.3.and.ians2.eq.1)then
               capouter=1.3*1005.5*(thrmli(icomp,isurf,iouter,4)/2.)
            endif
            capPCM=capouter+capinner

C capPCM value at previous time step.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               pcmshtp*shcsol*  !!? shtsol in t53
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
            if(iphnl.eq.1.and.ians1.eq.1)then
               capinner=1.3*1005.5*(thrmli(icomp,isurf,iinner,4)/2.)
            elseif(iphnl.eq.3.and.ians2.eq.1)then
               capouter=1.3*1005.5*(thrmli(icomp,isurf,iouter,4)/2.)
            endif
            capPCMp=capouter+capinner

            x1=cap/capPCMp
            x2=cap/capPCM
            xfactp=factp*pcmshtp*x1
            xfactf=pcmcon*x2
            if(iphnl.eq.3)then
               vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*xfactp
               vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x1
               qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
               vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*xfactf
               vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x2
               qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
            elseif(iphnl.eq.1.and.ians1.eq.1)then
               vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x1
               vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*xfactp
               qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
               vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x2
               vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*xfactf
               qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
            endif
         endif

C Save factf and pcmsht for use as factp and pcmshtp at next
C time step.
         pcmfac(icomp,isurf,nloc)=factf
         pcmshts(icomp,isurf,nloc)=pcmsht

C Save node-related heat stored.
         spcm(icomp,isurf,nloc)=storedLH

C Save future time row values as present values for next time step.
         appconp(icomp,isurf,nloc)=appcon
         appshcp(icomp,isurf,nloc)=appshc
         pk(icomp,isurf,nloc)=presk

10    continue ! next layer node

C Results output.
      ndatspm(ispmnod)=12
      if(nsinc.le.1)then
         strspm(ispmnod,1)='/AppCON'
         unitspm(ispmnod,1)='(W/(m.K))'
         txtspm(ispmnod,1)='Apparent conductivity of PCM layer'

         strspm(ispmnod,2)='/APPSHC'
         unitspm(ispmnod,2)='(J/(kg.K))'
         txtspm(ispmnod,2)='Apparent spec. heat capacity of PCM layer'

         strspm(ispmnod,3)='/StoredLH'
         unitspm(ispmnod,3)='(J/kg)'
         txtspm(ispmnod,3)='Latent heat stored in PCM'

         strspm(ispmnod,4)='/SHCarea'
         unitspm(ispmnod,4)='(J/m2)'
         txtspm(ispmnod,4)='Latent heat stored per PCM area'

         strspm(ispmnod,5)='/PCMFAC'
         unitspm(ispmnod,5)='(-)'
         txtspm(ispmnod,5)='PCMFAC of middle node'

         strspm(ispmnod,6)='/DT_F-P'
         unitspm(ispmnod,6)='(K)'
         txtspm(ispmnod,6)=
     &                 'Temperature difference (T_Future - T_Present)'

         strspm(ispmnod,7)='/lflag_out'
         unitspm(ispmnod,7)='(-)'
         txtspm(ispmnod,7)='Liquification flag for outer subnode'

         strspm(ispmnod,8)='/lflag_middle'
         unitspm(ispmnod,8)='(-)'
         txtspm(ispmnod,8)='Liquification flag for middle subnode'

         strspm(ispmnod,9)='/lflag_in'
         unitspm(ispmnod,9)='(-)'
         txtspm(ispmnod,9)='Liquification flag for inner subnode'

         strspm(ispmnod,10)='/db_out'
         unitspm(ispmnod,10)='(-)'
         txtspm(ispmnod,10)='Case / debug flag for outer subnode'

         strspm(ispmnod,11)='/db_middle'
         unitspm(ispmnod,11)='(-)'
         txtspm(ispmnod,11)='Case / debug flag for middle subnode'

         strspm(ispmnod,12)='/db_in'
         unitspm(ispmnod,12)='(-)'
         txtspm(ispmnod,12)='Case / debug flag for inner subnode'
      endif

      if (Rtot.gt.0) then
        dataspm(ispmnod,1)=thrmli(icomp,isurf,ielem,4)/Rtot
      else
        dataspm(ispmnod,1)=consol
      endif
      dataspm(ispmnod,2)=aveAPPSHC
      dataspm(ispmnod,3)=sumStoredLH
      dataspm(ispmnod,4)=sumStoredLH*thrmli(icomp,isurf,ielem,2)*
     &                                    thrmli(icomp,isurf,ielem,4)
cx      dataspm(ispmnod,5)=pcmfac(icomp,isurf,inode)
      dataspm(ispmnod,5)=sumStoredLH/Lges
      dataspm(ispmnod,6)=(tfcm-tpcm)

      dataspm(ispmnod,7)=pk(icomp,isurf,inode-1)
      dataspm(ispmnod,8)=pk(icomp,isurf,inode)
      dataspm(ispmnod,9)=pk(icomp,isurf,inode+1)

      dataspm(ispmnod,10)=db(1)
      dataspm(ispmnod,11)=db(2)
      dataspm(ispmnod,12)=db(3)

C As a temporary measure, use zone-related variables, zspmf1, for
C results library storage. Output is limited to one special material
C per zone.
      zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)

C XML output.
      call PCM_to_h3k(icomp,inode,isurf,ispmnod)

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from spmcmp56 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,5(2x,i4))')
     &        'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &                       ispmloc(ispmnod,2),inode-1,inode,inode+1
      call edisp(itu,outs)
      call edisp(itu,' ')
      do iphnl=1,3
         if(iphnl.eq.1)then
            write(outs,'(a,i3)') 'Outermost node (',inode-1,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode-1),' J'
            call edisp(itu,outs)
         elseif(iphnl.eq.2)then
            write(outs,'(a,i3)') 'Middle node (',inode,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode),' J'
            call edisp(itu,outs)
         else
            write(outs,'(a,i3)') 'Intermost node (',inode+1,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode+1),' J'
            call edisp(itu,outs)
         endif
      enddo
  999 return
      end

C ***************** LatentHeat *****************
C Created by Achim Geissler, July 11 2008.
C This function models PCM materials, which show significant sub-cooling
C properties, i.e. materials that, once melted, must be sub-cooled by
C a few degrees below the conversion temperature at which they are
C completely liquified. The function is called by SPMCPM56 to calculate
C the integral of the stored/released latent heat between the present
C temperature, T1, and the future temperature, T2:
C          k = normal (1) or sub-cooled (0) range;
C          LHSimple = simple (.true.) or extended function (.false.).
C          bUseSpline .true. => use spline function for heating/cooling
C                                  enthalpy curves

C The returned value is LH, the stored/released latent heat (J/kg);
C thereby, an energy balance is formulated from the viewpoint of the
C surrounding zone:
C          LH > 0 when T1 > T2, i.e. heat released to zone, PCM cools in TS;
C          LH < 0 when T1 < T2, i.e. heat stored in PCM, PCM heats up in TS.

C The base equations are taken from the PhD thesis of Sabine Hoffmann,
C University of Weimar, September 2006 ('Advanced' function extended
C by factor T^f).

      real function LatentHeat(isp,k,T1,T2)
      implicit none

#include "building.h"

      real T1,T2
      integer isp,k

      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      integer ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow

C Phase change material parameters.
      common/pcmparm/tsoli,tmelt,sub,Lges,consol,conliq,shcsol,shcliq,
     &                   a1,b1,c1,d1,e1,f1,a2,b2,c2,d2,e2,f2,LHSimple
      real tsoli              !- solidification temperature (degC)
      real tmelt              !- melting temperature (degC)
      real sub                !- sub-cooling temperature difference (degC)
      real Lges               !- total latent heat capacity of PCM (J/kg)
      real consol,conliq      !- conductivity of solid and liquid phase (W/(m.K))
      real shcsol,shcliq      !- specific heat of solid and liquid phase (J/(kg.K)
      real a1,b1,c1,d1,e1,f1  !- equation parameters for non-subcooling (a1-f1)
      real a2,b2,c2,d2,e2,f2  !  and subcooling (a2-f2) states
      logical LHSimple        !- .true. is simple model, .false. is extended model

C DSC data / spline function data
      common/pcmspline/temp1d(mspmnod,MSPMSPLM),
     &          heat1d(mspmnod,MSPMSPLM),
     &          cool1d(mspmnod,MSPMSPLM),heat2d(mspmnod,MSPMSPLM),
     &          cool2d(mspmnod,MSPMSPLM),nxy(mspmnod),
     &          bUseSpline(mspmnod),fnamDSCdat(mspmnod)
      real temp1d   ! DSC temperature data values, degC
      real heat1d   ! DSC heating enthalpy values, J/(kg K)
      real cool1d   ! DSC cooling enthalpy values, J/(kg K)
      real heat2d   ! 2nd derivatives at spline x values
      real cool2d
      integer nxy             ! number of data sets
      logical bUseSpline      ! .true. means DSC data file available, use
                              ! cubic spline function for apparent specific
                              ! heat function.
      character fnamDSCdat*72 ! DSC data file name for current spm entry


C Local variables.
      real aa,bb,ff,LH_T1,LH_T2,Tlow,Thigh,resultsign

cx          integer jj
cx          real LH_1,LH_0,T11,T22,vf1low,vf1high,vf0low,vf0high
cx          real v_func

C Which function?
      if(k.eq.0)then
        if (.not.bUseSpline(isp)) then
           if(LHSimple)then  !- simple function
              aa=a2
              bb=b2
           else              !- extended function
              ff=f2          !- aa to ee are set in function v_func()
           endif
        endif
      else                   !- function defaults to 'normal' mode
        if (.not.bUseSpline(isp)) then
           if(LHSimple)then  !- simple function
              aa=a1
              bb=b1
           else              !- extended function
              ff=f1          !- aa to ee are set in function v_func()
           endif
        endif
      endif

      if (bUseSpline(isp)) then
        if (k.eq.0) then
C         Use s_trap() to calculate cooling curve integral between
C         T1 and T2 ...
          call s_trap(isp,k,T1,T2,LatentHeat)
C DEBUG
cx          write(88,*)'nsinc,k,T1,T2,LH_int: ',
cx     &                           nsinc,k,T1,T2,LatentHeat

        else ! k=1, "normal" (heating) curve
C         Use s_trap() to calculate heating curve integral between
C         T1 and T2 ...
          call s_trap(isp,k,T1,T2,LatentHeat)
C DEBUG
cx          write(88,*)'nsinc,k,T1,T2,LH_int: ',
cx     &                           nsinc,k,T1,T2,LatentHeat

        endif

      else ! not spline
        if(LHSimple)then     !- simple function
          LatentHeat=exp(aa)/bb*(exp(bb*T1)-exp(bb*T2))
        else

C         Call integral function with T2 > T1, only,
C         LatentHeat returned is > 0

          Tlow=min(T1,T2)
          Thigh=max(T1,T2)
          call f56_trap(isp,k,Tlow,Thigh,LatentHeat)

C DEBUG
cx          if (nsinc.eq.3) then
cx            Tlow=14
cx            Thigh=15
cx            write(96,'(a,6E17.8)')'1: ',a1,b1,c1,d1,e1,f1
cx            write(96,'(a,6E17.8)')'2: ',a2,b2,c2,d2,e2,f2
cx            write(96,'(a)')'Tlow,Thigh,v_func(1,Tlow),v_func(1,Tlow),',
cx     &           'v_func(0,Tlow),,v_func(0,Tlow),LH_1,LH_0'
cx            do 10 jj=1,12
cx              T11=Tlow+jj
cx              T22=Thigh+jj
cx              k=1
cx              call  f56_trap(isp,k,T11,T22,LH_1)
cx              vf1low=v_func(k,T11)
cx              vf1high=v_func(k,T22)
cx              k=0
cx              call  f56_trap(isp,k,T11,T22,LH_0)
cx              vf0low=v_func(k,T11)
cx              vf0high=v_func(k,T22)
cx              write(96,'(f5.2,a,f5.2,6(a,f9.4))')
cx     &          T11,',',T22,',',
cx     &          vf1low,',',vf1high,',',vf0low,',',vf0high,',',
cx     &          LH_1,',',LH_0
cx  10        continue
cx          endif
C *** END DEBUG

        endif
      endif

C     LH > 0 when T1 > T2, i.e. heat released to zone;
C     LH < 0 when T1 < T2, i.e. heat stored in PCM.
      if (T1.gt.T2) resultsign=1.0
      if (T1.lt.T2) resultsign=-1.0

      LatentHeat=resultsign*ABS(LatentHeat)*1000. ! J/kg

C More than |Lges| is not possible.
cx      LatentHeat=sign(1.0,LatentHeat)*min(Lges,abs(LatentHeat))

      return
      end

C ***************** v_func *****************
C Extended latent heat function
C
C        a + c T + e T^2
C   LH = ------------------ T^f   [J/g]
C        1 + b T + d T^2
C
C Function parameter definition is via fit from DCS data sets. For
C "clearer" ((to be verified)) parameter definition, the DSC curve
C is offset by 0.5 ( shcsol + shcliq). C

      real function v_func(k,T)
      implicit none

      integer k
      real T

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER  IUOUT,IUIN,IEOUT

      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      integer ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow

C Phase change material parameters.
      common/pcmparm/tsoli,tmelt,sub,Lges,consol,conliq,shcsol,shcliq,
     &                    a1,b1,c1,d1,e1,f1,a2,b2,c2,d2,e2,f2,LHSimple
      real tsoli              !- solidification temperature (degC)
      real tmelt              !- melting temperature (degC)
      real sub                !- sub-cooling temperature difference (degC)
      real Lges               !- total latent heat capacity of PCM (J/kg)
      real consol,conliq      !- conductivity of solid and liquid phase (W/m.K)
      real shcsol,shcliq      !- specific heat of solid and liquid phase (J/kg.K)
      real a1,b1,c1,d1,e1,f1  !- equation parameters for non-subcooling (a1-f1)
      real a2,b2,c2,d2,e2,f2  !  and subcooling (a2-f2) states
      logical LHSimple        !- .true. is simple model, .false. is extended model

C Local variables.
      REAL aa,bb,cc,dd,ee,ff,CC1,CC2

C Coefficients for extended function.
      if(k.eq.0)then    !- subcooling
         aa=a2
         bb=b2
         cc=c2
         dd=d2
         ee=e2
         ff=f2
      else              !- non-subcooling
         aa=a1
         bb=b1
         cc=c1
         dd=d1
         ee=e1
         ff=f1
      endif

        CC1=aa+cc*T+ee*T**2
        CC2=1.+bb*T+dd*T**2

C << Check for valid temperature range (?)

        v_func=T**ff*CC1/CC2 ! J/g

C DEBUG
cx      WRITE(98,'(a,i7,a,i1,a,f8.4,a,f10.4)')
cx     & ' *** nsinc: ',nsinc,', v_func(',k,',',T,') = ',v_func

      return
      end

C ***************** PCMSTORE *****************
C This routine keeps track of the latent heat stored/released by a
C PCM. The routine is called once each simulation time step and for
C each PCM node.

      subroutine PCMStore(isp,TnS,TnS1,k,storedLH,AppCon,AppSHC,Debug)
      implicit none

C Parameters.
      real TnS
      real TnS1
      integer isp,k   !- spm index, sub-cooling switch, value of preceding time
                      !  step passed (typically initialised to 1)
      real storedLH   !- stored latent heat, value of preceeding time
                      !  step passed (typically initialised to 0)
      real AppCon     !- apparent thermal conductivity
      real AppSHC     !- apparent specific heat capacity
      integer Debug   !- output of 'case' as defined below

C Time common.
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      integer ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow

C Phase change material parameters.
      common/pcmparm/tsoli,tmelt,sub,Lges,consol,conliq,shcsol,shcliq,
     &                    a1,b1,c1,d1,e1,f1,a2,b2,c2,d2,e2,f2,LHSimple
      real tsoli              !- solidification temperature (degC)
      real tmelt              !- melting temperature (degC)
      real sub                !- sub-cooling temperature difference (degC)
      real Lges               !- total latent heat capacity of PCM (J/kg)
      real consol,conliq      !- conductivity of solid and liquid phase (W/m.K)
      real shcsol,shcliq      !- specific heat of solid and liquid phase (J/(kg.K)
      real a1,b1,c1,d1,e1,f1  !- equation parameters for non-subcooling (a1-f1)
      real a2,b2,c2,d2,e2,f2  !  and subcooling (a2-f2) states
      logical LHSimple        !- .true. is simple model, .false. is extended model

C Local variables.
      real tsoli2       !- sub-cooling solidification temperature (degC)
      real tmelt2       !- sub-cooling melting temperature (degC)
      real storeThisTS  !- latent heat stored in PCM (J/kg)
      real LatentHeat
      logical bNoTempDiff

C Set up sub-cooling temperatures.
      tsoli2=tsoli-sub
      tmelt2=tmelt ! -sub << add 2nd "sub" parameter??

C *****************
C Calculate the heat stored according to the temperature change
C between present (TnS) and future (TnS1) temperatures. Calculate also
C the apparent heat capacity and thermal conductivity based on the
C percentage of liquidification of the PCM.

C The following cases are checked/differentiated:
C      heating - cases 1, 2, 3, 4, 5, 6 and  7;
C      cooling - cases 1, 2, 3, 5, 6, 7 and -8.

C
C         ^ C (J/(kg.K)
C         |                       _____
C         |                      /     \
C         | k=0                 /       \
C         |                    /    7    \
C         |                   /           \
C         |                  /             \ 6   5   tsoli   (5)
C         |      (1)     -8 /      tmelt    \---<-->---/----<-->- C_liq
C   C_sol |-----<-->-------/----------\   tsoli2      / 4
C         |             tmelt2   1   2 \             /
C         |                     -1  -2  \           /
C         |                              \    3    /
C         | k=1                           \  -3   /
C         |                                \     /
C         |                                 \___/
C         |-------------------------------------------------> T (degC)
C *****************

cx    use eclose logic here to "dampen" calcs? ... do what if TnS approx. equal TnS1?
cx    storedLH += time step dependant?? 17/2/22
      call eclose(TnS,TnS1,1.0E-03,bNoTempDiff)

      if (bNoTempDiff) then
         !- Set AppCon and AppSHC according to k
        if (k.eq.1) then
          AppCon=consol
          AppSHC=shcsol
        else ! k = 0
          AppCon=conliq
          AppSHC=shcliq
        endif
        Debug=0
        storeThisTS=0.0

C ********* HEATING (PCM temperature increase) *********
      elseif(TnS.lt.TnS1)then

         if(k.eq.1)then

C PCM is "solid", behaves according to non-subcooling properties.
            if(TnS1.le.tmelt)then

C PCM is solid (case 1), StoredLH = 0.
               AppCon=consol
               AppSHC=shcsol
               Debug=1
            elseif(TnS.le.tmelt.and.TnS1.gt.tmelt)then

C PCM begins to melt (case 2).
               storeThisTS=LatentHeat(isp,k,tmelt,TnS1) !- storeThisTS < 0
               storedLH=0.0E0-storeThisTS
               if(storedLH.lt.0.0E+0)storedLH=0.0E+0    !- storedLH >= 0
               if(storedLH.gt.Lges)storedLH=Lges        !- storedLH <= Lges
               AppCon=(1.0E0-storedLH/Lges)*consol+
     &                                       storedLH/Lges*conliq
               AppSHC=shcsol+storeThisTS/(tmelt-TnS1)   !- to be verified
               Debug=2
            elseif(TnS.gt.tmelt.and.TnS1.le.tsoli)then

C PCM is partially liquid, latent heat is stored in the normal
C range (k=1, case 3).
               storeThisTS=LatentHeat(isp,k,TnS,TnS1) !- storeThisTS < 0
               storedLH=storedLH-storeThisTS
               if(storedLH.lt.0.0E+0)storedLH=0.0E+0  !- storedLH >= 0
               if(storedLH.gt.Lges)storedLH=Lges      !- storedLH <= Lges
               AppCon=(1.0E0-storedLH/Lges)*consol+
     &                                        storedLH/Lges*conliq
               AppSHC=shcsol+storeThisTS/(TnS-TnS1)   !- to be verified
               Debug=3
            elseif(TnS.gt.tmelt.and.TnS1.gt.tsoli)then

C Remainder of solid PCM is completely melted and also heated as
C liquid (case 4).
cx << after timestep, appcon=conliq and appshc=shcliq, no? 17/2/22
               storeThisTS=LatentHeat(isp,k,TnS,tsoli) !- storeThisTS < 0
               storedLH=storedLH-storeThisTS
               if(storedLH.lt.0.0E+0)storedLH=0.0E+0   !- storedLH >= 0
               if(storedLH.gt.Lges)storedLH=Lges       !- storedLH <= Lges
               AppCon=consol
               AppSHC=shcsol+storeThisTS/(TnS-tsoli)   !- to be verified

C Set new k.
               k=0
               Debug=4
            endif     !- temperature bracketing for k = 1
         else         !- k=0

C PCM is/was liquid and thus behaves according to sub-cooling properties.
            if(TnS.gt.tsoli2)then

C PCM is completely liquified and heated further (case 5).
               storedLH=Lges
               AppCon=conliq
               AppSHC=shcliq
               Debug=5
            elseif(TnS1.gt.tsoli2)then

C PCM is sub-cooled partial liquid and is completely
C (re-)melted (case 6).
               storeThisTS=LatentHeat(isp,k,TnS,tsoli2) !- storeThisTS < 0
               storedLH=storedLH-storeThisTS
               if(storedLH.lt.0.0E+0)storedLH=0.0E+0    !- storedLH >= 0
               if(storedLH.gt.Lges)storedLH=Lges        !- storedLH <= Lges
               AppCon=conliq
               AppSHC=shcliq+storeThisTS/(TnS-tsoli2)   !- to be verified
               Debug=6
            else ! if(TnS1.le.tsoli2)then

C PCM is sub-cooled partial liquid and not completely
C melted (k=0, case 7).
               storeThisTS=LatentHeat(isp,k,TnS,TnS1) !- storeThisTS < 0
               storedLH=storedLH-storeThisTS
               if(storedLH.lt.0.0E+0)storedLH=0.0E+0  !- storedLH >= 0
               if(storedLH.gt.Lges)storedLH=Lges      !- storedLH <= Lges
               AppCon=(1.0E0-storedLH/Lges)*consol+
     &                                        storedLH/Lges*conliq
               AppSHC=shcliq+storeThisTS/(TnS-TnS1)   !- to be verified
               Debug=7
            endif         !- temperature bracketing for k=0
         endif ! k = 0 for temperature rise

C ********* COOLING *********
      elseif(TnS.gt.TnS1)then

C PCM temperature decrease.
         if(k.eq.1)then

C PCM behaves according to non-subcooling properties.
            if(TnS.le.tmelt)then

C PCM is solid (case -1), StoredLH = 0.
               AppCon=consol
               AppSHC=shcsol
               Debug=-1
            elseif(TnS.gt.tmelt.and.TnS1.le.tmelt)then

C PCM non-subcooled solidification is completed (case -2).
               storeThisTS=LatentHeat(isp,k,TnS,TnS1) !- storeThisTS > 0
               storedLH=storedLH-storeThisTS
               if(storedLH.lt.0.0E+0)storedLH=0.0E+0  !- storedLH >= 0
               if(storedLH.gt.Lges)storedLH=Lges      !- storedLH <= Lges
               AppCon=(1.0E0-storedLH/Lges)*consol+storedLH/Lges*conliq
               AppSHC=shcsol+storedLH/(TnS-tmelt)
               storedLH=0.0E0     !- now all PCM is solid
               Debug=-2
            elseif(TnS.le.tsoli.and.TnS1.ge.tmelt) then

C PCM is partially liquid and is now cooled (non-subcooling, case -3).
               storeThisTS=LatentHeat(isp,k,TnS,TnS1) !- storeThisTS > 0
               storedLH=storedLH-storeThisTS
               if(storedLH.lt.0.0E+0)storedLH=0.0E+0  !- storedLH >= 0
               if(storedLH.gt.Lges)storedLH=Lges      !- storedLH <= Lges
               AppCon=(1.0E0-storedLH/Lges)*consol+storedLH/Lges*conliq
               AppSHC=shcsol+storeThisTS/(TnS-TnS1)   ! to be verified
               Debug=-3
            endif     !- temperature bracketing for k=1
         else         !- k=0

C PCM behaves according to subcooling properties.
            if(TnS1.ge.tsoli2)then

C PCM is completely melted and does not reach sub-cooled
C solidification temperature (case -5).
               storedLH=Lges    !- crude
               AppCon=conliq
               AppSHC=shcliq
               Debug=-5
             elseif(TnS.gt.tsoli2.and.TnS1.lt.tsoli2)then

C PCM begins to solidify in sub-cooled mode (k=0, case -6).
               storeThisTS=LatentHeat(isp,k,tsoli2,TnS1) !- storeThisTS > 0
               storedLH=Lges-storeThisTS
               if(storedLH.lt.0.0E+0)storedLH=0.0E+0     !- storedLH >= 0
               if(storedLH.gt.Lges)storedLH=Lges         !- storedLH <= Lges
               AppCon=(1.0E0-storedLH/Lges)*consol+storedLH/Lges*conliq
               AppSHC=shcsol+storeThisTS/(tsoli2-TnS1)   !- to be verified
               Debug=-6
            elseif(TnS.le.tsoli2.and.TnS1.gt.tmelt2)then

C PCM is subcooled partial liquid and not completely
C solidified (k=0, case -7).
               storeThisTS=LatentHeat(isp,k,TnS,TnS1) !- storeThisTS > 0
               storedLH=storedLH-storeThisTS
               if(storedLH.lt.0.0E+0)storedLH=0.0E+0  !- storedLH >= 0
               if(storedLH.gt.Lges)storedLH=Lges      !- storedLH <= Lges
               AppCon=(1.0E0-storedLH/Lges)*consol+storedLH/Lges*conliq
               AppSHC=shcsol+storeThisTS/(TnS-TnS1)   !- to be verified
               Debug=-7
             elseif(TnS.gt.tmelt2.and.TnS1.le.tmelt2)then

C PCM completely solidifies (k=0, case -8).
               storeThisTS=LatentHeat(isp,k,TnS,tmelt2) !- storeThisTS > 0
               storedLH=0.0E0
               AppCon=consol
               AppSHC=shcsol+storeThisTS/(TnS-tmelt2)   !- to be verified

C Set new k.
               k=1
               Debug=-8
            endif      !- temperature bracketing for k=0
         endif
      endif ! heating / cooling check

      return
      end

C ***************** PCM_to_h3k *****************
C This routine transfers PCM domain data to the H3K reporting
C facilities. Created by Achim Geissler, June 17 2008.

      subroutine PCM_to_h3k(iZone,iNode,iSurf,ispmnod)
      use h3kmodule
      implicit none

#include "building.h"
#include "geometry.h"

C Parameters.
      integer iZone,iNode,iSurf,ispmnod

C Commons.
      common/spmatlbl/spmlabel(mspmnod)
      character spmlabel*16
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)
      integer ndatspm       !- number of results items for each special material
      real dataspm          !- results data for special material
      character strspm*16
      character unitspm*16  !- unit for data item
      character txtspm*72   !- description of data item

C Local variables.
      character*12 cZone_Chars, cSurf_Chars
      character*34 cZoneSurfNode
      character*2  cNode_Chars
      integer ndat,idat

C References.
      integer lnblnk

C.....Switch for zone name output here
      if (ReportBoolConfig("use_zonenames")) then
        write (cZone_Chars,'(A)') zname(iZone)(1:lnzname(iZone))
      else
C.......Pad zone index to 'XXX'
        write (cZone_Chars,'(A,I3.3)') 'zone_',iZone
      endif ! use_zonenames

C.....Switch for surface name output here
      if (ReportBoolConfig ("use_surfacenames")) then
        write (cSurf_Chars,'(A)')
     &    sname(iZone,iSurf)(1:lnblnk(sname(iZone,iSurf)))
      else
C.......Pad surface index to 'XXX'
        write (cSurf_Chars,'(A,I3.3)') 'surface_',iSurf
      endif ! use_surfacenames

      write(cNode_Chars,'(i2.2)') inode

      write(cZoneSurfNode,'(a,a,a,a,a)')
     &    cZone_Chars(1:lnblnk(cZone_Chars)),'/',
     &    cSurf_Chars(1:lnblnk(cSurf_Chars)),
     &    '/nd',cNode_Chars(1:lnblnk(cNode_Chars))

      ndat=min0(mspmres,ndatspm(ispmnod)) !- maximum of mspmres entries possible

C Output available PCM layer data.
      do idat=1,ndat
         Call AddToReport(rvBuiSpm%Identifier,
     &      dataspm(ispmnod,idat),
     &      spmlabel(ispmnod)(1:lnblnk(spmlabel(ispmnod))),
     &      cZoneSurfNode(1:lnblnk(cZoneSurfNode)) // '/'
     &      // strspm(ispmnod,idat)(2:lnblnk(strspm(ispmnod,idat))))
C                                   ^ the strings "strspm" have a leading '/' ...

cx << bug in the "detailed report" functions with more than one "*"???
cx    The "[]" information in out.dictionary is not written!
cx    ag@2017/02/15

         Call AddToReportDetails(rvBuiSpm%Identifier,
     &      spmlabel(ispmnod)(1:lnblnk(spmlabel(ispmnod))),
     &      cZoneSurfNode(1:lnblnk(cZoneSurfNode)) // '/'
     &      // strspm(ispmnod,idat)(2:lnblnk(strspm(ispmnod,idat))),
     &      'units',
     &      unitspm(ispmnod,idat)(1:lnblnk(unitspm(ispmnod,idat))),
     &      txtspm(ispmnod,idat)(1:lnblnk(txtspm(ispmnod,idat))))

      enddo

      return
      end

C ***************** SPMCMP57 *****************
C A model of phase change material based on the apparent heat capacity
C method. The  thermal properties of a construction layer defined as
C a phase change material (PCM) are adjusted to represent the latent
C heat stored within the material during melting and released during
C solidification. The phase transition is represented by an apparent
C heat capacity (appsht), which is equal to the sum of specific (shtsol)
C and latent (lht) heat capacity, with the latter established as a linear
C function of temperature (tfcm) in the phase change temperature range
C (tsoli-tmelt): lht=shtliqa*tfcm+shtliqb, when temperature is rising 
C and for cooling (tsoli2-tmelt2): lht=shtliqc*tfcm+shtliqd; 

C Defining data as read from spmdat:
C    1 - melting temperature (heating), tmelt (degC);
C    2 - solidification temperature (heating), tsoli (degC);
C    3 - conductivity in solid phase, consol (W/m.degC);
C    4 - conductivity in liquid phase, conliq (W/m.degC);
C    5 - specific heat in all phases, shtsol (J/kg.degC);
C    6 - latent heat coefficient, shtliqa (J/kg.degC^2);
C    7 - latent heat coefficient, shtliqb (J/kg.degC);
C    8 - melting temperature (cooling), tmelt2 (degC);
C    9 - solidification temperature (cooling), tsoli2 (degC);
C    10 - latent heat coefficient, shtliqa (J/kg.degC^2);
C    11 - latent heat coefficient, shtliqb (J/kg.degC).
C 
C Possibilities for calculation of a,b,c,d coefficients:
C CASE 1
C apparent heat capacity - appsht(tcmf) constant value in phase change
C temperature range 
C laten heat stored - spcmf(tcmf) linear function 
C then a and b calculated as:
C a=0 
c b=shtliq/(tsoli-tmelt) (shtliq - entire latent heat of phase change)
C
C appsht (J/(kg.degC)
C         ^ 
C         |                       
C         |    ________       
C         |    .       .
C         |    .       .
C         |    .       .
C   shtsol|____.       .__________
C         |                   
C         |-------------------------> tcmf (degC) 
C            tmelt   tsoli
C
C   spcmf (J/kg)
C         ^ 
C         |           .            
C         |          /.     
C         |         / .   
C         |        /  . 
C         |       /   . 
C         |      /    .     
C         |     /     .  
C         |____/      .__________
C         |                   
C         |-------------------------> tcmf (degC) 
C            tmelt   tsoli
C CASE 2
C apparent heat capacity - appsht(tcmf) increasing linear function 
C laten heat stored - spcmf(tcmf) parabolic function opens up
C then a and b calculated as:
C a=2*shtliq/(tsoli-tmelt)**2 (shtliq - entire latent heat of phase change)
c b=-a*tmelt
C
C appsht (J/(kg.degC)
C         ^ 
C         |           .            
C         |          /.     
C         |         / .   
C         |        /  . 
C         |       /   . 
C         |      /    .     
C         |     /     .  
C   shtsol|____/      .__________
C         |                   
C         |-------------------------> tcmf (degC) 
C            tmelt   tsoli
C
C   spcmf (J/kg)
C         ^ 
C         |                      .    
C         |                     .
C         |                    .   
C         |                   .    
C         |                 .    
C         |               .         
C         |            .       
C         |____.    .    
C         |------------------------> tcmf (degC)
C            tmelt              tsoli
C
C CASE 3
C apparent heat capacity - appsht(tcmf) decreasing linear function 
C laten heat stored - spcmf(tcmf) parabolic function opens down
C then a and b calculated as:
C a=-2*shtliq/(tsoli-tmelt)**2 (shtliq - entire latent heat of phase change)
c b=a*tsoli
C 
C appsht (J/(kg.degC)
C         ^ 
C         |    .            
C         |    .\
C         |    . \
C         |    .  \
C         |    .   \
C         |    .    \
C         |    .     \
C   shtsol|____.      \____
C         |                   
C         |-------------------------> tcmf (degC)
C            tmelt   tsoli
C
C   spcmf (J/kg)
C         ^ 
C         |                  .   .    
C         |              .     
C         |           .   
C         |         .   
C         |       .    
C         |      .         
C         |     .       
C         |____.      
C         |------------------------> tcmf (degC)
C            tmelt              tsoli

      subroutine spmcmp57(icomp,ispmnod)
#include "building.h"
#include "geometry.h"

C N.B. All parameters comply with the Fortran implicit naming convention
C except where explicitly redefined.
      common/outin/iuout,iuin,ieout
      common/tc/itc,icnt
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/pvalc/tpc(mcom,ms,mn),qpc(mcom)
      common/pvals/tps(mcom,ms),qps(mcom)
      common/fvalc/tfc(mcom,ms,mn),qfc(mcom)
      common/fvals/tfs(mcom,ms),qfs(mcom)
      common/zonspmf/zspmf1(mcom),zspmf2(mcom)

C thrmli(icomp,isurf,ielem,1) = conductivity
C thrmli(icomp,isurf,ielem,2) = density
C thrmli(icomp,isurf,ielem,3) = specific heat
C thrmli(icomp,isurf,ielem,4) = thickness
      common/vthp14/thrmli(mcom,ms,me,7)
      common/prec9/nconst(mcom),nelts(mcom,ms),ngaps(mcom,ms),
     &             npgap(mcom,ms,mgp)
      common/prec13/c(mcom,ms,mn,2),qc(mcom,ms,mn)
      common/pers/isd1,ism1,isd2,ism2,isds,isdf,ntstep
      common/vthp15/vcp(ms,mn,2),qcp(ms,mn)
      common/vthp16/vcf(ms,mn,2),qcf(ms,mn)
      common/resspm/ndatspm(mspmnod),dataspm(mspmnod,mspmres),
     &         strspm(mspmnod,mspmres),unitspm(mspmnod,mspmres),
     &         txtspm(mspmnod,mspmres)

C Special materials.
      common/spmfxst/ispmxist,spflnam
      common/spmatl/nspmnod,ispmloc(mspmnod,3),ispmtyp(mspmnod,2),
     &              nnodat(mspmnod),spmdat(mspmnod,mspmdat)
      common/spmatlbl/spmlabel(mspmnod)

C Phase change material.
      common/pcm02/spcm(mcom,ms,mn),spcmtf,spcmtp
      common/pcm03/pcmfac(mcom,ms,mn),pcmshts(mcom,ms,mn)

      character spmlabel*16,spflnam*72,outs*124,strspm*16,unitspm*16,
     &          txtspm*72
      logical close
      real val1,val2,val3,val4,val5,val6,val7,val8,val9,val10,val11  ! to pass as parameters

C PCM layer pointers.
      if(icomp.ne.ispmloc(ispmnod,1))then
         write(outs,*)'SPMAT57 fatal error: incorrect zone assignment!'
         call edisp(iuout,outs)
         stop
      endif
      isurf=ispmloc(ispmnod,2)
      inode=ispmloc(ispmnod,3)
      ielem=nint(real(inode)/2.)

C Melting and solidification temperatures (degC).
      tmelt=spmdat(ispmnod,1)
      tsoli=spmdat(ispmnod,2)
      tmelt2=spmdat(ispmnod,8)
      tsoli2=spmdat(ispmnod,9)

C Conductivity in solid and liquid phase (W/m.degC).
      consol=spmdat(ispmnod,3)
      conliq=spmdat(ispmnod,4)

C Specific heat capacity (J/kg.degC).
      shtsol=spmdat(ispmnod,5)

C Latent heat coefficients (J/kg.degC^2 & J/kg.degC respectively).
      shtliqa=spmdat(ispmnod,6)
      shtliqb=spmdat(ispmnod,7)
      shtliqc=spmdat(ispmnod,10)
      shtliqd=spmdat(ispmnod,11)

C Initialisations for xml output.
      Rtot=0.0
      THKsum=0.0
      sumSHTTHK=0.0

C Process the 3 nodes of the PCM layer.
      spcmtf=0.0
      do 10 iphnl=1,3
         if(iphnl.eq.1)then
            nloc=inode-1
            efthk=thrmli(icomp,isurf,ielem,4)*0.25
         elseif(iphnl.eq.2)then
            nloc=inode
            efthk=thrmli(icomp,isurf,ielem,4)*0.5
         elseif(iphnl.eq.3)then
            nloc=inode+1
            efthk=thrmli(icomp,isurf,ielem,4)*0.25
         endif
         THKsum=THKsum+efthk    !- total PCM thickness

C Establish temperature of the PCM material node from tfc if the node
C is intra-construction and tfs if located at the internal surface.
         if(iphnl.eq.3.and.ielem.eq.nelts(icomp,isurf))then
            tfcm=tfs(icomp,isurf)
            tpcm=tps(icomp,isurf)
         else
            tfcm=tfc(icomp,isurf,nloc)
            tpcm=tpc(icomp,isurf,nloc)
         endif

         imethod=4
         val1=tsoli
         val2=tmelt
         val3=shtsol
         val4=consol
         val5=conliq
         val6=shtliqa
         val7=shtliqb
         val8=tsoli2
         val9=tmelt2
         val10=shtliqc
         val11=shtliqd
         call maxlatent(imethod,tfcm,tpcm,val1,val2,val3,val4,
     &           val5,val6,val7,val8,val9,val10,
     &            val11,appcon,appsht,spcmf)


C Parameters for xml output.
         Rtot=Rtot+efthk/appcon
         sumSHTTHK=sumSHTTHK+appsht

C Establish present value of factp and pcmshtp as last used value
C of factf and pcmsht respectively.
         factp=pcmfac(icomp,isurf,nloc)
         pcmshtp=pcmshts(icomp,isurf,nloc)
         if(nsinc.eq.1)then
            factp=0
         endif

C Calculate new factf; this is used below to remove original values
C of PCM layer conductivity and specific heat as used to establish the
C self- and cross- coupling coefficients of the nodal energy balance
C equations, and substitute the new values as established above. An
C adjusted factf is required where the PCM node separates the PCM layer
C from a non-PCM layer.
         pcmcon=appcon/consol
         pcmsht=appsht/shtsol
         factf=pcmcon/pcmsht

C Modify present and future value of 'c' and 'qc' coefficients (i.e.
C coeficients of nodal energy balance equations.
         x1=1./pcmshtp
         if(nsinc.eq.1)then
            x1=0
         endif
         x2=1./pcmsht
         if(iphnl.eq.1.and.ielem.eq.1)then      !- outermost node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x1
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factp
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x2
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factf
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.2)then                 !- centre node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factp
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factp
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factf
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*factf
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.3.and.ielem.eq.nelts(icomp,isurf))then  !- internal surface node
            vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factp
            vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x1
            qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
            vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*factf
            vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x2
            qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
         elseif(iphnl.eq.1.or.iphnl.eq.3)then    !- interface node

C Redefine modification factors to take account of weighted node
C thermal capacity.
            if(iphnl.eq.1)then
               iouter=ielem
               iinner=ielem-1
            else
               iinner=ielem
               iouter=ielem+1
            endif

C Original averaged capacity before adjustment.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               thrmli(icomp,isurf,iouter,3)*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)

C Cater for an adjacent air gap.
            ians1=0
            ians2=0
            if(ngaps(icomp,isurf).ne.0)then
               if(iphnl.eq.1)then
                  do igap=1,ngaps(icomp,isurf)
                     if(iinner.eq.npgap(icomp,isurf,igap))then
                        capinner=1.3*1005.5*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
                        ians1=1
                     endif
                  enddo
               elseif(iphnl.eq.3)then
                  do igap=1,ngaps(icomp,isurf)
                     if(iouter.eq.npgap(icomp,isurf,igap))then
                        capouter=1.3*1005.5*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
                        ians2=1
                     endif
                  enddo
               endif
            endif
            cap=capouter+capinner

C Average capacity utilising new PCM apparent specific heat.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               appsht*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
            if(iphnl.eq.1.and.ians1.eq.1)then
               capinner=1.3*1005.5*(thrmli(icomp,isurf,iinner,4)/2.)
            elseif(iphnl.eq.3.and.ians2.eq.1)then
               capouter=1.3*1005.5*(thrmli(icomp,isurf,iouter,4)/2.)
            endif
            capPCM=capouter+capinner

C capPCM value at previous time step.
            capouter=thrmli(icomp,isurf,iouter,2)*
     &                               pcmshtp*shtsol*
     &                               (thrmli(icomp,isurf,iouter,4)/2.)
            capinner=thrmli(icomp,isurf,iinner,2)*
     &                               thrmli(icomp,isurf,iinner,3)*
     &                               (thrmli(icomp,isurf,iinner,4)/2.)
            if(iphnl.eq.1.and.ians1.eq.1)then
               capinner=1.3*1005.5*(thrmli(icomp,isurf,iinner,4)/2.)
            elseif(iphnl.eq.3.and.ians2.eq.1)then
               capouter=1.3*1005.5*(thrmli(icomp,isurf,iouter,4)/2.)
            endif
            capPCMp=capouter+capinner

            x1=cap/capPCMp
            x2=cap/capPCM
            xfactp=factp*pcmshtp*x1
            xfactf=pcmcon*x2
            if(iphnl.eq.3)then
               vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*xfactp
               vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x1
               qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
               vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*xfactf
               vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*x2
               qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
            elseif(iphnl.eq.1.and.ians1.eq.1)then
               vcp(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x1
               vcp(isurf,nloc,2)=c(icomp,isurf,nloc,2)*xfactp
               qcp(isurf,nloc)=qc(icomp,isurf,nloc)*x1
               vcf(isurf,nloc,1)=c(icomp,isurf,nloc,1)*x2
               vcf(isurf,nloc,2)=c(icomp,isurf,nloc,2)*xfactf
               qcf(isurf,nloc)=qc(icomp,isurf,nloc)*x2
            endif
         endif

C Save factf and pcmsht for use as factp and pcmshtp at next
C time step.
         pcmfac(icomp,isurf,nloc)=factf
         pcmshts(icomp,isurf,nloc)=pcmsht

   10 continue

C Results output.
      ndatspm(ispmnod)=4
C     if(nsinc.le.1)then
         strspm(ispmnod,1)='/AppCON'
         unitspm(ispmnod,1)='(W/(m.degC))'
         txtspm(ispmnod,1)='Average conductivity of PCM layer'

         strspm(ispmnod,2)='/SHCtot'
         unitspm(ispmnod,2)='(J/kg)'
         txtspm(ispmnod,2)='Total heat stored in PCM'

         strspm(ispmnod,3)='/SHCarea'
         unitspm(ispmnod,3)='(J/m^2)'
         txtspm(ispmnod,3)='Heat stored per PCM area'

         strspm(ispmnod,4)='/PCMFAC'
         unitspm(ispmnod,4)='(-)'
         txtspm(ispmnod,4)='PCMFAC of middle node'
C      endif


      if(Rtot.gt.0)dataspm(ispmnod,1)=THKsum/Rtot
      if(THKsum.gt.0)dataspm(ispmnod,2)=sumSHTTHK
      dataspm(ispmnod,3)=(spcmtp+spcmtf)/2.0
      spcmtp=spcmtf
      dataspm(ispmnod,4)=pcmfac(icomp,isurf,inode)

C As a temporary measure, use zone-related variable, zspmf1, for
C for results library storage. Output is limited to one special
C material per zone.
      zspmf1(ispmloc(ispmnod,1))=dataspm(ispmnod,1)

C XML output.
      call PCM_to_h3k(icomp,inode,isurf,ispmnod)

C Trace output.
      if(itc.le.0.or.nsinc.lt.itc.or.ispmxist.eq.0)goto 999
      if(itrace(19).eq.0.or.nsinc.gt.itcf.or.
     &   izntrc(icomp).ne.1) goto 999
      write(outs,'(a,a)') 'Trace output from SPMCMP53 for ',
     &                                              spmlabel(ispmnod)
      call edisp(itu,outs)
      write(outs,'(a,5(2x,i4))')
     &        'Affected zone, surface and node: ',ispmloc(ispmnod,1),
     &                       ispmloc(ispmnod,2),inode-1,inode,inode+1
      call edisp(itu,outs)
            write(outs,'(a,i3)') 'Outermost node (',inode-1,')'
      call edisp(itu,' ')
      do iphnl=1,3
         if(iphnl.eq.1)then
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode-1),' J'
            call edisp(itu,outs)
         elseif(iphnl.eq.2)then
            write(outs,'(a,i3)') 'Middle node (',inode,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode),' J'
            call edisp(itu,outs)
         else
            write(outs,'(a,i3)') 'Intermost node (',inode+1,')'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Temperature ',tfcm,' deg.C'
            call edisp(itu,outs)
            write(outs,'(a,1x,f10.2)') 'Heat stored (future time row)',
     &                                  spcm(icomp,isurf,inode+1),' J'
            call edisp(itu,outs)
         endif
      enddo

  999 return
      end

C ******************************* ReadDSCDataFile ***********************
C Created by: Achim Geissler
C Initial Creation Date: March 31, 2017
C Copyright FHNW
C
C This subroutine is based on cetc/FC_components.F::FC_eloads_convert. 
C It reads the user-specified DSC data file, containing a header line and
C three columns of data. N is the number of measurement data points,
C DT the temperature interval of DSC measurement and MUNIT the mass
C conversion factor so that enthalpy is in J/(kg K)
C Format:
C N, DT, MUNIT
C T1, DHheat1, DHcool1
C T2, DHheat2, DHcool2
C ...
C TN, DHheatN, DHcoolN
C
C Typically,
C
C This subroutine is executed once per simulation, prior to the time-step
C calculations.
C -----------------------------------------------------------------------
      SUBROUTINE ReadDSCDataFile(ispmnod)
      IMPLICIT NONE

#include "building.h"

      integer ispmnod

C---------------------------------------------------------------------------------
C ESP-r commons
C---------------------------------------------------------------------------------
      COMMON/FILEP/IFIL
      INTEGER  IFIL

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER  IUOUT,IUIN,IEOUT

C----------------
C SPM commons
C----------------
C DSC data / spline function data
      common/pcmspline/temp1d(mspmnod,MSPMSPLM),
     &          heat1d(mspmnod,MSPMSPLM),
     &          cool1d(mspmnod,MSPMSPLM),heat2d(mspmnod,MSPMSPLM),
     &          cool2d(mspmnod,MSPMSPLM),nxy(mspmnod),
     &          bUseSpline(mspmnod),fnamDSCdat(mspmnod)
      real temp1d   ! DSC temperature data values, degC
      real heat1d   ! DSC heating enthalpy values, J/(kg K)
      real cool1d   ! DSC cooling enthalpy values, J/(kg K)
      real heat2d   ! 2nd derivatives at spline x values
      real cool2d
      integer nxy             ! number of data sets
      logical bUseSpline      ! .true. means DSC data file available, use
                              ! cubic spline function for apparent specific
                              ! heat function.
      character fnamDSCdat*72 ! DSC data file name for current spm entry

C---------------------------------------------------------------------------------
C Declare local variables.
C---------------------------------------------------------------------------------
      INTEGER ND,K,num_recs,num_col,num_incbin,inc,incbi,ire
      integer iFIL_ascii  ! temporary local file number
      
      CHARACTER OUTSTR*124

      LOGICAL fclerror

      INTEGER ISTAT,IER,I
      INTEGER irec, incbin

      real dataDT          ! Temperature interval of DSC data points
      real MUNIT           ! Mass unit conversion factor (1. or 1000.)

      integer lnblnk  ! function definition

C---------------------------------------------------------------------------------
C Read the header information from the user-specified data file (.dsc)
C and perform some checks to ensure that the correct number of data
C are specified. Note that this does not check the validity of the data,
C but rather ensures that the correct number of data items are specified.
C The existence of the .dsc file was confirmed when the .cfg file was read.
C ((then why check below??))
C `fclerror' is a flag indicating whether there were errors reading the .fcl file.
C---------------------------------------------------------------------------------
C-----Notify user that data file is being processed (this can take some time).
      call usrmsg(' Processing DSC input file...',
     &            ' ','-')
C-----Open the .dsc file.
      fclerror = .false.
      iFIL_ascii = IFIL+28  ! Should be a safe unit number to use.
      CALL ERPFREE(iFIL_ascii,ISTAT)
      CALL EFOPSEQ(iFIL_ascii,fnamDSCdat(ispmnod),1,IER)
      IF(IER /= 0)THEN
        WRITE(IUOUT,*) ' Error opening DSC type input file ',
     &    fnamDSCdat(ispmnod),', spm node ',
     &    ispmnod,'.'
        fclerror = .true.
      ENDIF

C-----Read the number of data points and temperature interval of data.
      CALL STRIPC(iFIL_ascii,OUTSTR,99,ND,1,' DSC input type',IER)
      IF(IER /= 0)THEN
        WRITE(IUOUT,*) ' Error reading DSC type input file ',
     &    fnamDSCdat(ispmnod),', spm node ',
     &    ispmnod,'.'
        fclerror = .true.
      ENDIF
C     << check ND? >>
      K=0
      CALL EGETWI(OUTSTR,K,nxy(ispmnod),10,MSPMSPLM,'-',
     &                               ' Number of DSC data points ',IER)
      IF(IER /= 0)THEN
        WRITE(IUOUT,*) ' Error reading number of DSC data points; ',
     &    ' file ',fnamDSCdat(ispmnod),', spm node ',
     &    ispmnod,'.'
        fclerror = .true.
      ENDIF
      CALL EGETWR(OUTSTR,K,dataDT,0.,0.,'-',
     &                 ' Temperature interval of DSC data points ',IER)
      IF(IER /= 0)THEN
        WRITE(IUOUT,*) ' Error reading temperature interval of DSC ',
     &    'data points; file ',fnamDSCdat(ispmnod),', spm node ',
     &    ispmnod,'.'
        fclerror = .true.
      ENDIF

      IF (dataDT.le.0) THEN
        WRITE(IUOUT,*) ' Temperature interval of DSC ',
     &    'file ',fnamDSCdat(ispmnod),', spm node ',
     &    ispmnod,' value bad ... setting to unity.'
        dataDT=1.0
      ENDIF

      CALL EGETWR(OUTSTR,K,MUNIT,0.,0.,'-',
     &                 ' Mass unit conversion of DSC data points ',IER)
      IF(IER /= 0)THEN
        WRITE(IUOUT,*) ' Error reading mass unit conversion of DSC ',
     &    'data points; file ',fnamDSCdat(ispmnod),', spm node ',
     &    ispmnod,'.'
        fclerror = .true.
      ENDIF


C-----Error handling on reading of .fcl file.
      IF(fclerror)THEN
        STOP ' Error in DSC input file.'
      ENDIF

C---------------------------------------------------------------------------------
C     Read the data triplet at each temperature increment. Normalize
C     enthalpy values "heat" and "cool" to 1 K (i.e. divide by dataDT)
C     and if neccessary correct unit (e.g. multiply by 0.001 kg/g if
C     data values in J/(kg K))
C---------------------------------------------------------------------------------

      DO irec=1,nxy(ispmnod)
C-------Read the data from the ASCII file.
        CALL STRIPC(iFIL_ascii,OUTSTR,99,ND,1,'the raw data line',IER)
        IF(IER /= 0)THEN
          WRITE(IUOUT,'(4a,i3,a,i3,a)') ' Error reading DSC data from ',
     &      'file ',fnamDSCdat(ispmnod)(1:lnblnk(fnamDSCdat(ispmnod))),
     &      ', spm node ',ispmnod,', data line ',irec,'.'
        ENDIF
C       << check ND? >>
        K=0
        CALL EGETWR(OUTSTR,K,temp1d(ispmnod,irec),0.,0.,'-','t1d',IER)
        CALL EGETWR(OUTSTR,K,heat1d(ispmnod,irec),0.,0.,'-','h1d',IER)
        CALL EGETWR(OUTSTR,K,cool1d(ispmnod,irec),0.,0.,'-','c1d',IER)

C       Normalize enthalpy values to J/(g K)
        heat1d(ispmnod,irec)=MUNIT*heat1d(ispmnod,irec)/dataDT
        cool1d(ispmnod,irec)=MUNIT*cool1d(ispmnod,irec)/dataDT

      END DO

C Processing complete. Leave binary file open for use during time-step simulation.
      call usrmsg(' Processing DSC input file... done.',
     &            ' ','-')

      RETURN
      END

C************************************************************************
C     N U M E R I C A L    R E C I P E S
C************************************************************************
C
C Routines spline() and splint() from Numerical Recipes.
C
      SUBROUTINE spline(x, y, n, yp1, ypn, y2)
!     use nrtype
!
! Given arrays x(1:n) and y(1:n) containing a tabulated function, i.e.
! y(i)=f(x(i)), with x(1)<x(2)<...<x(n), and given values yp1 and ypn for
! the first derivative of the interpolating function at points 1 and n,
! respectively, this routine returns an array y2(1:n) of length n which
! contains the second derivatives of the interpolating function at the
! tabulated points x(i).  If yp1 and/or ypn are equal to 1.e30 or larger,
! the routine is signaled to set the corresponding boundary condition for
! a natural spline with zero second derivative on that boundary.
! Parameter: nmax is the largest anticipiated value of n
! (adopted from Numerical Recipes in FORTRAN 77)
!
cx      INTEGER, PARAMETER :: DP=KIND(1.0D0)
      INTEGER:: n
      INTEGER, PARAMETER:: nmax=500
      REAL:: yp1, ypn, x(n), y(n), y2(n)
      INTEGER:: i, k
      REAL:: p, qn, sig, un, u(nmax)

      if (yp1.gt..99e30) then
        y2(1)=0.
        u(1)=0.
      else
        y2(1)=-0.5
        u(1)=(3./(x(2)-x(1)))*((y(2)-y(1))/(x(2)-x(1))-yp1)
      endif

      do i=2, n-1
        sig=(x(i)-x(i-1))/(x(i+1)-x(i-1))
        p=sig*y2(i-1)+2.
        y2(i)=(sig-1.)/p
        u(i)=(6.*((y(i+1)-y(i))/(x(i+1)-x(i))-(y(i)-y(i-1))
     &         /(x(i)-x(i-1)))/(x(i+1)-x(i-1))-sig*u(i-1))/p
      enddo

      if (ypn.gt..99e30) then
        qn=0.
        un=0.
      else
        qn=0.5
        un=(3./(x(n)-x(n-1)))*(ypn-(y(n)-y(n-1))/(x(n)-x(n-1)))
      endif

      y2(n)=(un-qn*u(n-1))/(qn*y2(n-1)+1.)

      do k=n-1, 1, -1
        y2(k)=y2(k)*y2(k+1)+u(k)
      enddo

      return
      END

C========================================================================
      SUBROUTINE splint(xa, ya, y2a, n, x, y)
C     USE nrtype
C
C Given the arrays xa(1:n) and ya(1:n) of length n, which tabulate a
C function (with the xa(i) in order), and given the array y2a(1:n), which
C is the output from the subroutine spline, and given a value of x, this
C routine returns a cubic spline interpolated value y.
C (adopted from Numerical Recipes in FORTRAN 77)


C------------------------------------------------------------------------
C ESP-r commons
C------------------------------------------------------------------------
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER  IUOUT,IUIN,IEOUT

cx      INTEGER, PARAMETER :: DP = KIND(1.0D0)
      INTEGER:: n
      REAL:: x, y, xa(n), y2a(n), ya(n)
      INTEGER:: k, khi, klo
      REAL:: a, b, h

C     << improve performance by remembering last klo / khi??
      klo=1
      khi=n
  1   if (khi-klo.gt.1) then
        k=(khi+klo)/2
        if (xa(k).gt.x) then
           khi=k
        else
           klo=k
        endif
        goto 1
      endif

      h=xa(khi)-xa(klo)
      if (h.eq.0.) then
C       pause 'bad xa input in splint'
          WRITE(IUOUT,'(2a)') ' Error in splint: ',
     &      '"bad xa".'
        return
      endif

      a=(xa(khi)-x)/h
      b=(x-xa(klo))/h
      y=a*ya(klo)+b*ya(khi)
     &           +((a**3-a)*y2a(klo)+(b**3-b)*y2a(khi))*(h**2)/6.

      return
      END

C======================================================================
C Based on Numerical Recipes for Fortran 77.
C
C Returns as LH the integral of the spline of the PCM function
C from T1 to T2.
C
C The parameters EPS can be set to the desired fractional accuracy and
C JMAX so that 2^(JMAX-1) is the maximum allowed number of steps.
C
C Integration is performed by the trapezoidal rule.
C
      SUBROUTINE s_trap(isp,k,T1,T2,LH)
      implicit none

#include "building.h"

C Subroutine parameters
      INTEGER isp,k
      REAL T1,T2,LH

C------------------------------------------------------------------------
C ESP-r commons
C------------------------------------------------------------------------
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER  IUOUT,IUIN,IEOUT

C DSC data / spline function data
      common/pcmspline/temp1d(mspmnod,MSPMSPLM),
     &          heat1d(mspmnod,MSPMSPLM),
     &          cool1d(mspmnod,MSPMSPLM),heat2d(mspmnod,MSPMSPLM),
     &          cool2d(mspmnod,MSPMSPLM),nxy(mspmnod),
     &          bUseSpline(mspmnod),fnamDSCdat(mspmnod)
      real temp1d   ! DSC temperature data values, degC
      real heat1d   ! DSC heating enthalpy values, J/(kg K)
      real cool1d   ! DSC cooling enthalpy values, J/(kg K)
      real heat2d   ! 2nd derivatives at spline x values
      real cool2d
      integer nxy             ! number of data sets
      logical bUseSpline      ! .true. means DSC data file available, use
                              ! cubic spline function for apparent specific
                              ! heat function.
      character fnamDSCdat*72 ! DSC data file name for current spm entry

C------------------------------------------------------------------------
C Local variables
C------------------------------------------------------------------------
      INTEGER JMAX
      REAL EPS

      PARAMETER (EPS=1.e-3, JMAX=9) ! Default values are 1.e-6, 20
C     USES trapzd
      INTEGER n,j,it

      REAL oldLH,del,sum,tnm,x
      real func_T1,func_T2,func_x

C------------------------------------------------------------------------

      oldLH=-1.e30

C If k=0, we are cooling, if k=1, we are heating.
      if (k.eq.0) then
        call splint(temp1d(isp,:), cool1d(isp,:), cool2d(isp,:),
     &                 nxy(isp), T1, func_T1)
        call splint(temp1d(isp,:), cool1d(isp,:), cool2d(isp,:),
     &                 nxy(isp), T2, func_T2)
      else ! heating
        call splint(temp1d(isp,:), heat1d(isp,:), heat2d(isp,:),
     &                 nxy(isp), T1, func_T1)
        call splint(temp1d(isp,:), heat1d(isp,:), heat2d(isp,:),
     &                 nxy(isp), T2, func_T2)
      endif

      do 20 n=1,JMAX

C       Calculate n-th stage of refinement.
        if (n.eq.1) then
          LH=0.5*(T2-T1)*(func_T1+func_T2)
        else
          it=2**(n-2)
          tnm=it
          del=(T2-T1)/tnm
          x=T1+0.5*del
          sum=0.

          do 30 j=1,it
            if (k.eq.0) then ! cooling
              call splint(temp1d(isp,:), cool1d(isp,:), cool2d(isp,:),
     &                 nxy(isp), x, func_x)
            else ! heating
              call splint(temp1d(isp,:), heat1d(isp,:), heat2d(isp,:),
     &                 nxy(isp), x, func_x)
            endif

            sum=sum+func_x
            x=x+del
 30       continue

          LH=0.5*(LH+(T2-T1)*sum/tnm)

        endif

C       Force minimum of four iterations ...
        if (j.gt.5) then
          if (abs(LH-oldLH).lt.EPS*abs(oldLH).or.
     &          LH.eq.0..and.oldLH.eq.0.) return
        endif

        oldLH=LH

20    continue

C      pause 'too many steps in qtrap'
      WRITE(IUOUT,'(a,f10.4,a)')
     & ' *** Too many steps in s_trap! LH=',LH,', setting LH=0.!'

C     Set LH to zero
      LH=0.

      return

      END  


C======================================================================
C Based on Numerical Recipes for Fortran 77.
C
C Returns as LH the integral of the the PCM function
C
C
C
C from T1 to T2.
C
C The offset for function parameter estimation must be added
C to the function value here again.
C
C   LH + 0.5 ( shcsol + shcliq ) T
C
C The parameters EPS can be set to the desired fractional accuracy and
C JMAX so that 2^(JMAX-1) is the maximum allowed number of steps.
C
C Integration is performed by the trapezoidal rule.
C
      SUBROUTINE f56_trap(isp,k,T1,T2,LH)
      implicit none

#include "building.h"

C Subroutine parameters
      INTEGER isp,k
      REAL T1,T2,LH

C------------------------------------------------------------------------
C ESP-r commons
C------------------------------------------------------------------------
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      INTEGER  IUOUT,IUIN,IEOUT

C Phase change material parameters.
      common/pcmparm/tsoli,tmelt,sub,Lges,consol,conliq,shcsol,shcliq,
     &                    a1,b1,c1,d1,e1,f1,a2,b2,c2,d2,e2,f2,LHSimple
      real tsoli              !- solidification temperature (degC)
      real tmelt              !- melting temperature (degC)
      real sub                !- sub-cooling temperature difference (degC)
      real Lges               !- total latent heat capacity of PCM (J/kg)
      real consol,conliq      !- conductivity of solid and liquid phase (W/m.K)
      real shcsol,shcliq      !- specific heat of solid and liquid phase (J/kg.K)
      real a1,b1,c1,d1,e1,f1  !- equation parameters for non-subcooling (a1-f1)
      real a2,b2,c2,d2,e2,f2  !  and subcooling (a2-f2) states
      logical LHSimple        !- .true. is simple model, .false. is extended model

C------------------------------------------------------------------------
C Local variables
C------------------------------------------------------------------------
      INTEGER JMAX
      REAL EPS

      PARAMETER (EPS=1.e-3, JMAX=20) ! Default values are 1.e-6, 20

      INTEGER n,j,it

      REAL oldLH,del,sum,tnm,x
      REAL shs,shl

      REAL v_func

C------------------------------------------------------------------------

      oldLH=-1.e30

      shs=shcsol/1000.  ! here we are using J/g !!
      shl=shcliq/1000.  !

      do 20 n=1,JMAX

C       Calculate n-th stage of refinement.
        if (n.eq.1) then
C         If k=0, we are cooling, if k=1, we are heating.
          LH=0.5*(T2-T1)*(v_func(k,T1)+v_func(k,T2))
     &                                +0.5*(shs+shl)*(T2-T1)
        else
          it=2**(n-2)
          tnm=it
          del=(T2-T1)/tnm
          x=T1+0.5*del
          sum=0.

          do 30 j=1,it
C           v_func() not defined for negative temperature!
            if(x.gt.0) sum=sum+v_func(k,x)
            x=x+del
 30       continue

          LH=0.5*( LH+(T2-T1)*sum/tnm + 0.5*(shs+shl)*(T2-T1) )

        endif

C       Force minimum of four iterations ...
        if (j.gt.5) then
          if (abs(LH-oldLH).lt.EPS*abs(oldLH).or.
     &          LH.eq.0..and.oldLH.eq.0.) return
        endif

        oldLH=LH

20    continue

C      pause 'too many steps in qtrap'
      WRITE(IUOUT,'(a,f10.4,a)')
     & ' *** Too many steps in f56_trap! LH=',LH,', setting LH=0.!'

C     Set LH to zero
      LH=0.

      return

      END  
