C This file is part of the ESP-r system.
C Copyright Energy Systems Research Unit, University of
C Strathclyde, Glasgow Scotland, 2001-2013.

C ESP-r is free software.  You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation 
C (version 2 or later).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.


C This file contains the following routines:
C MZCASI   - total convective and radiant casual gains.
C INTLUM   - timestep determination of controlled casual gains.
C CALCILUM - illuminance at daylight sensors.
C DWINVT   - visible transmission of a TMC window.
C HUNTA    - determine if current hour has scheduled casual gains to control.
C VRTILM   - illum (Ev) on a unobstructed vert surface.

C ******************** MZCASI ********************************
C MZCASI sets the present and future time-row total convective and
C radiant casual gains.  Any control action (e.g. occupant behaviour, 
C lighting)is applied here.

C Returned variables are:
C QCASRT - summation of the present and future time-row total radiant
C          injections.
C QCASCT - summation of the present and future time-row total convective
C          injections.
C QCASR  - radiant gain for present (1) and future (2) time step.
C QCASC  - convective gain for present (1) and future (2) time step.
C ZPWRR  - zone real power consumption total.
C ZPWRQ  - zone reactive power total
C QR     - real power for present (1) and future (2) time step.
C QP     - reactive power for present (1) and future (2) time step.
C QLAT   - latent load for present (1) and future (2) time step.

      subroutine mzcasi(icomp)
      USE TCC, ONLY:COSIM, COSIM_DATA
#include "building.h"
#include "site.h"
#include "model.h"
#include "geometry.h"
#include "schedule.h"
#include "net_flow.h"
#include "tdf2.h"
#include "sbem.h"
#include "plant.h"
#include "power.h"
#include "dhw_parameters.h"
#include "dhw_common.h"
#include "OffsiteUtilitiesPublic.h"
#include "h3k_report_data.h"
#include "FMI.h"
#include "cfd.h"

      common/filep/ifil
      common/tc/itc,icnt
      common/trc/itrc
      common/trace/itcf,itrace(mtrace),izntrc(mcom),itu
      common/outin/iuout,iuin,ieout
      common/caleni/nbdaytype,nbcaldays(MDTY),icalender(365)
      INTEGER NBDAYTYPE,NBCALDAYS,ICALENDER,IDTY
      COMMON/SUNPOS/SAZI,SALT,ISUNUP

      common/btime/btimep,btimef

C Zone lighting and power commons.
      COMMON/ELECFLG/IELF(MCOM)

C Zone casual gain control data.
      COMMON/CGCIN2N/NCGTC(MCOM,MDTY),NLITZ(MCOM),IDFST(MCOM,MLCOM),
     & CGX(MCOM,MLCOM,MDF),CGY(MCOM,MLCOM,MDF),CGH(MCOM,MLCOM,MDF),
     & UX(MCOM,MLCOM,MDF),UY(MCOM,MLCOM,MDF),UH(MCOM,MLCOM,MDF),
     & SETPT(MCOM,MLCOM),SYSPER(MCOM,MLCOM),SOFFLL(MCOM,MLCOM),
     & IOFFDT(MCOM,MLCOM),SMLOUT(MCOM,MLCOM),SMEOUT(MCOM,MLCOM)

      COMMON/CGCIN3/ICGCS(MCOM,MDTY),ICGCF(MCOM,MDTY),
     & ICGCFL(MCOM,MLCOM),SPELEC(MCOM,MLCOM),SLOPEM(MCOM,MLCOM)

C Switch off delay time memory common block.
      COMMON/SOFDT/IDTM(MCOM,MLCOM)

      COMMON/PERS/ISD1,ISM1,ISD2,ISM2,ISDS,ISDF,NTSTEP
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/prec7/itcnst

C Common ncmtmstps is used as a flag to start counting simulation timesteps
C and is used for code related to UK NCM method.
      common/CountTimesteps/i_countsteps(mcom),numbStartTimesteps,
     &numTotTimstps
      integer i_countsteps
      integer numbStartTimesteps  !total number of start-up timesteps
      integer numTotTimstps !total number of timesteps including the start-up days
     
      COMMON/COE34/QCASRT(MCOM),QCASCT
      COMMON/COE34D/QCASR(2),QCASC(2)
      COMMON/COE34S/QCCS(MCOM),QCRS(MCOM)
      COMMON/COE34L/QLAT(MCOM)
      COMMON/GRSD130/QCASRS(MCOM)

      COMMON/BCLCHP/StoreCap,znElecCasGain(MCOM),kounter
      real StoreCap,znElecCasGain
      integer kounter

C Common for occupant/lighting/equipment/other sensible convective, sensible
C radiant and latent W for future and present in each zone.
      real ctlperocupc,ctlperocupr,ctlperocupl    ! occupant after control applied
      real ctlperlightc,ctlperlightr,ctlperlightl ! lighting after control applied
      real ctlperequipc,ctlperequipr,ctlperequipl ! equipment after control applied
      real ctlperotherc,ctlperotherr,ctlperotherl ! other after control applied
      common/applyctl/ctlperocupc(MCOM,2),ctlperocupr(MCOM,2),
     &  ctlperocupl(MCOM,2),ctlperlightc(MCOM,2),ctlperlightr(MCOM,2),
     &  ctlperlightl(MCOM,2),ctlperequipc(MCOM,2),ctlperequipr(MCOM,2),
     &  ctlperequipl(MCOM,2),ctlperotherc(MCOM,2),ctlperotherr(MCOM,2),
     &  ctlperotherl(MCOM,2) 

C Common for HOT3000 electrical-network -> casual-gain coupling
C E_to_G_coupling - boolian indicating coupling exists.
C Coupling_gain_info(i,j,k,l) - data array for gain k on daytype j for zone i. 
C    l = 1 -> start hour (0->24).
C    l = 2 -> end hour (0->24).
C    l = 3 -> sensible load (fraction of non-hvac elec load.
C    l = 4 -> latent load (fraction of non-hvac elec load.
C    l = 5 -> radiative feaction (0->1).
C    l = 6 -> convective fraction (0->1).
C    l = 7 -> casual gain slot.
      COMMON/CasCou/iNumb_Gain_coupling(MCOM,MDTY),
     &     Coupling_gain_info(MCOM,MDTY,MC,7)

C Data associated with HOT3000 central and zonal base loads.
      common / H3K_BL_data / iH3K_Base_Loads(MCOM),
     &                       iBCD_column(MCOM,MDTY,MC),
     &                       fBL_fraction(MCOM,MDTY,MC),
     &                       nGain_types(MCOM,MDTY),
     &                       iGain_type(MCOM,MDTY,MC)

      integer iH3K_Base_Loads, iBCD_column, nGain_types, iGain_type
      real fBL_fraction

C Function determining if HOT3000 features are enabled.
      logical bH3KExtentionsActive
      
C Store casual/radiative gain injections on a zone-by-zone basis. Note: the
C latent gain variable is a place-holder for storing this data but is not
C presently used.
      COMMON/CasDat/Cas_Rad_dat(MCOM),Cas_Con_dat(MCOM),
     &     Cas_Lat_dat(MCOM)
 
C Store electrical power of lights and equipment, 
C set up as casual gains on a zone-by-zone basis.
      COMMON/CasPowerDat/Cas_Lights(MCOM),Cas_Equip(MCOM),
     &     Cas_Lights2(MCOM),Cas_Equip2(MCOM)

C Casual gain data.
C NCGPER - number of periods for each gain (zone, day type, gain type).
C TCGS   - start time of gain (zone, day type, gain type, period).
C CGSENR - sensible gain radiant portion (zone, day type, gain type, period).
C CGSENC - sensible gain convective portion (zone, day type, gain type, period).
C CGLAT  - latent gain (zone, day type, gain type, period).
C CGCTL  - gain control fraction (zone, present/future, gain type).
C ICGCTL - pointer to controlled gain type.
      common/CASGNS/NCGPER(MCOM,MDTY,MGTY),TCGS(MCOM,MDTY,MGTY,MGPER),
     &        CGSENC(MCOM,MDTY,MGTY,MGPER),CGSENR(MCOM,MDTY,MGTY,MGPER),
     &        CGLAT(MCOM,MDTY,MGTY,MGPER),CGCTL(MCOM,2,MGTY)
      COMMON/LIGHTS/ICGCTL(MCOM)
      common/CASGNSE/ELQ(MCOM,MDTY,MGTY,MGPER,MPHAS),
     &               ELR(MCOM,MDTY,MGTY,MGPER,MPHAS)

C Instruction to pause assessment to allow rescan of a zone operation file.
      integer ipausecas   ! zero no pause, non-zero pause 
      common/pausecas/ipausecas(MCOM)

C Variables used in trace.
      common/CASGNTRC/QP(MCOM,2,MC,MPHAS),RP(MCOM,2,MC,MPHAS)

C Adaptive comfort.
      COMMON/ADPCOM/PREVSTATWIN,PREVSTATFAN,PREVSTATLIGHT,TOTPWR(MCOM),
     &              TRMSTAT

C Dynamic person model.
      COMMON/CASGNSO/META1(MCOM,MDTY,MGTY,MGPER),
     &         NBMEN(MCOM,MDTY,MGTY,MGPER),NBWOM(MCOM,MDTY,MGTY,MGPER),
     &         NBCHI(MCOM,MDTY,MGTY,MGPER),CONCC(MCOM,MDTY,MGTY,MGPER),
     &         RADCC(MCOM,MDTY,MGTY,MGPER),CLOTH(MCOM,MDTY,MGTY,MGPER)
      REAL META1,NBMEN,NBWOM,NBCHI
      COMMON/PVALA/TPA(MCOM),QPA(MCOM)
      common/occload/sklat(MCOM),skcon(MCOM),skrad(MCOM),ttcr(MCOM),
     &     ttsk(MCOM)
      real headflux1,trunkflux1,larmflux1,rarmflux1,llegflux1
      real rlegflux1,H2Oexp1,H2Oswt1,Qoccsens1,Qocclat1
C       real COflux,Tsko_av1,Tco_av1,Tsk_av1
      COMMON/ocflux1/headflux1(MNZ),trunkflux1(MNZ),larmflux1(MNZ),
     &  rarmflux1(MNZ),llegflux1(MNZ),rlegflux1(MNZ),H2Oexp1(MNZ),
     &  H2Oswt1(MNZ),Qoccsens1(MCOM),Qocclat1(MCOM)
C      &  COflux,Tsko_av1,Tco_av1,Tsk_av1
      common/FVALG/GFA(MCOM)
      COMMON/PVALG/GPA(MCOM)
      COMMON/MOIST53/Ppair(MCOM),Fpair(MCOM),Spair(MCOM)
      real tavhead1,tavtrunk1,tavarml1,tavarmr1,tavlegl1,tavlegr1
      real hrhead,hrtrunk,hrarml,hrarmr,hrlegl,hrlegr,HMETA2
      integer cfdcheck,msegflag
      COMMON/segtemp1/tavhead1(MNZ),tavtrunk1(MNZ),tavarml1(MNZ),
     &  tavarmr1(MNZ),tavlegl1(MNZ),tavlegr1(MNZ),hrhead(MNZ),
     &  hrtrunk(MNZ),hrarml(MNZ),hrarmr(MNZ),hrlegl(MNZ),hrlegr(MNZ),
     &  cfdcheck(MNZ),msegflag(MNZ),HMETA2(MNZ)
      common/ndcfd/ncfdnd,icfdnd(MNZ),NCONF    
      REAL rlegflux,H2Oexp,H2Oswt
      REAL TPA, QPA,TAIR,TMRT
      REAL SGAIN,LGAIN,TOP,TOCN,CLOTHO,tatsix

C CGSECo, CGSERo and CGLATo - total convective, radiative and latent loads from
C dynamic occupant model. (CGSENC, CGSENR and CGLAT previously defined in common/CASGNS/.)
C< META and METAO - metabolic rate of occupant in met and watts respectively.
      REAL METAo,CGSECo,CGSERo,CGLATo,CGSENC,CGSENR,CGLAT
      REAL HMETA1,RHOCC
      common/clim/idif(mt),itmp(mt),idnr(mt),ivel(mt),idir(mt),
     &            ihum(mt),idiff,itmpf,idnrf,ivelf,idirf,ihumf

C Activity TDF data.
C QTOTOCCUP used as a shortcut for the total occupant gains
C that are taken from the activities temporal data
      REAL QTOTOCCUP,QOCCCASC,QOCCCASR,QOCCLATN,QTOTEQUIP,QEQUIPCASC
      REAL QEQUIPCASR,QEQUIPLATN,QTOTLIGHT,QLIGHTCASC,QLIGHTCASR
      REAL QLIGHTLATN
      INTEGER ILIGHT_IN_SBEMDB

C Data to calculate NCM lighting energy for results output.
      common/ncmlightEnergy/LGHT_ncm_Energy(MCOM),
     &      fmonthLGHT_ncm_Energy(12,MCOM)
      real LGHT_ncm_Energy,fmonthLGHT_ncm_Energy

C Count for the number of occupied hours above 27 C operative temperature as
C used for UK NCM purposes (i.e. timestep overheating).
      common/ncmHrOverheat/ihroverheats(MCOM),ihroccup(MCOM)
      integer ihroverheats,ihroccup

C Count for the number of timesteps that systems are in operation.
C Used to calculate an NCM auxiliary energy (post-processed with save level 6).
C Excludes start-up timesteps.
      common/ncmOperHrSystem/iOperTimestep(MCOM),
     &       iMonthOperTimestep(12,MCOM)      
      integer iOperTimestep,iMonthOperTimestep      

      REAL QCASLT  ! to hold the summation of present + future latent gains
      integer nbhelp

      DIMENSION QLATN(2),VAL(MBITS+2)

      CHARACTER outs*124, DSTR*10
      character*10 wkd1, wkd2
      logical traceok,closea
      
C Declation of heat gain from h3k models.
C     REAL H3K_HeatGainsFromModels(ICOMP)
      REAL H3KGains
C     Number of electric-load->gain couplings per zone,
      INTEGER iNumb_Gain_coupling
C     Data fro each coupling
      REAL Coupling_gain_info
C     HOT3000 storage arrays
      REAL Cas_Rad_dat, Cas_Con_dat, Cas_Lat_dat
C     Function returning gains from ideal hot water tank
      real fIdealDHWGain
      REAL W_Heat_nonHVAC ! heat for the particular casual gain
      Real W_NG_nonHVAC   ! natural gas for the particular casual gain
    
      REAL Total_elec_CON_gain          ! total electrical convective casual gain (type 5)
      REAL Total_elec_RAD_gain          ! total electrical radiative casual gain (type 5)
      REAL Cas_Lights, Cas_Equip

      REAL Total_NG     ! total natural gas for the zone from the casual gains

      Integer iGainSlot ! recovered from Coupling_gain_info(icomp,iDay,ii,7)
      
C Variables for access to the boundary condition data.
      Real fConvert_current_TS_to_DAY ! return day
      Real fGet_BC_data_by_name       ! return the data for the timestep
      Integer iInterp_method          ! 1 = step-wise; 2 = linear
      Character cContext*124          ! string to print a BCD failure
      Character cBCD_col*248

C HOT3000 base loads.
      REAL W_BL_heat_gain                  ! heat for the particular base load casual gain
      REAL Total_BL_Convective_gain        ! total base loads convective casual gain 
      REAL Total_BL_Radiative_gain         ! total base loads radiative casual gain
      REAL Total_BL_electricity_Appliances ! total electric load from base load appliances      
      REAL Total_BL_electricity_Lights     ! total electric load from base load lights
      REAL Total_BL_electricity_Other      ! total electric load from base load other
      
C Flag indicating that function Elec_Net_Load_Calc(?) should 
C return the occupant-driven electrical load (excluding HVAC equipment).
      INTEGER occupant_load 
      PARAMETER (occupant_load = 3)

C Named constants for timerows.
      integer iFuture
      parameter (iFuture=2) 
      common/wkdtyp/idwe1,idwe2,wkd1,wkd2
      integer imonth_iter !looping variable for months 1 - 12
      integer idayn  
      logical closer
      logical OK
      logical bInStartup  ! logical function returning true if in startup period

C Occupancy, for agents.
      common/occup/isocc(MCOM)
      logical isocc

C Trace output?
      traceok = .true.
      IF(ITC.LE.0.OR.NSINC.LT.ITC)traceok = .false.
      IF(ITRACE(12).EQ.0.OR.NSINC.GT.ITCF.OR.
     &   IZNTRC(ICOMP).NE.1)traceok = .false.
      if(traceok)then
        call edisp(itu,' ')
        write(outs,9995)ICNT,ICOMP
 9995   format(' Subroutine MZCASI Trace output',I4,5X,'Zone',I3)
        call edisp(itu,outs)
        ICNT=ICNT+1
        CALL DAYCLK(IDYP,BTIMEF,ITU)
      endif

C This loop executes 6720 times per timestep since
C MC=24*MCT*MDTY and MPHAS=4 as defined in building.h.
C Code activated only when trace output is on.
      if(traceok) then 
        DO 4 IPHS=1,MPHAS
          DO 5 ICAS=1,MC
            RP(ICOMP,1,ICAS,IPHS)=RP(ICOMP,2,ICAS,IPHS)
            QP(ICOMP,1,ICAS,IPHS)=QP(ICOMP,2,ICAS,IPHS)
            RP(ICOMP,2,ICAS,IPHS)=0.
            QP(ICOMP,2,ICAS,IPHS)=0.
  5       CONTINUE
  4     CONTINUE
      endif

C Zero zone power loads and update present control fractions.
      DO 9 ITY=1,MGTY
        PEZON(ICOMP,ITY)=0.
        QEZON(ICOMP,ITY)=0.
        CGCTL(ICOMP,1,ITY)=CGCTL(ICOMP,2,ITY)
        CGCTL(ICOMP,2,ITY)=1.0
  9   CONTINUE

C Update value for dynamic people model.
      SGAIN=0
      LGAIN=0

C Update present values.
      QCASC(1)=QCCS(ICOMP)
      QCASR(1)=QCRS(ICOMP)
      QLATN(1)=QLAT(ICOMP)
      QCASC(2)=0.
      QCASR(2)=0.
      QLATN(2)=0.

C Recover definitions for current day type and timeF.
      if (IHRF.eq.1) then 
        ID=IDWF
      else
        ID=IDWP
      endif
      if (btimef.gt.24.) then
        btimef=btimef-24.
        ID=IDWF
      endif
      DSTR='       '
      if(ID.EQ.IDWE1)then
        DSTR=wkd1
        IDAY=2
      elseif(ID.EQ.IDWE2)then
        DSTR=wkd2
        IDAY=3
      else
        DSTR='weekdays'
        IDAY=1
      endif
      IF(NBDAYTYPE.gt.3)THEN ! non-standard calendar
        IDAY=IDYP
        IF(IHRF.EQ.1)IDAY=IDYP+1
        IF(IDAY.GT.365)IDAY=IDAY-365
        IDTY=ICALENDER(IDAY)

C use IDAY in place of IDTY to keep to nomenclature in this subroutine
        IDAY=IDTY
      ENDIF
 
C Set controlled gain type.
      icgctl(icomp)=ncgtc(icomp,IDAY)
 
C Determine future time row casual gain multiplier, CGCTL(2,ICGCTL(ICOMP)).
      if (ICGC(ICOMP).EQ.1) then
        if (NCGTC(ICOMP,IDAY).NE.0) then

C Set switch-off delay time memory to 0 before start of the
C new day control period.
          IT=int(btimef)
          if (IT.EQ.ICGCS(ICOMP,IDAY)) then
            do 20 N=1,NLITZ(ICOMP)
              IDTM(ICOMP,N)=0
20          continue
          endif

          IPOINT=0

C Check time as in 933 below so for multiple timesteps per hour.
          if (btimef.GT.float(ICGCS(ICOMP,IDAY)).AND.
     &        btimef.LE.float(ICGCF(ICOMP,IDAY))) IPOINT=1
          if (IPOINT.NE.0) then
            call INTLUM(ICOMP,FRAC)  ! get the fraction of control
            CGCTL(ICOMP,2,ICGCTL(ICOMP))=FRAC
          endif
        endif
      endif

C Update SHOCC-calculated equipment loads.
      call updateequipment(icomp)

C Use data from one of three sources.
C << NB. tdf and time step file should have electrical data added to the 
C definition. >>

      if(ICASUAL(icomp).ne.0)then

C Data in temporal database (aggregate convective, radiant and
C latent casual, i.e. 3 columns). Also instantiate the future value
C of equipment casual gain so that it can be used within MZLS3.
C Equipment is an arbitrary choice for this aggregate data.
        IFOC=ICASUAL(icomp)
        CALL RCTDFB(itrc,btimef,VAL,ISD,IFOC,IER)
        QCASC(2)=VAL(ISD)
        ctlperocupc(icomp,2)=0.0
        ctlperlightc(icomp,2)=0.0
        ctlperequipc(icomp,2)=VAL(ISD)
        ctlperotherc(icomp,2)=0.0

        QCASR(2)=VAL(ISD+1)
        ctlperocupr(icomp,2)=0.0
        ctlperlightr(icomp,2)=0.0
        ctlperequipr(icomp,2)=VAL(ISD+1)
        ctlperotherr(icomp,2)=0.0

        QLATN(2)=VAL(ISD+2)
        ctlperocupl(icomp,2)=0.0
        ctlperlightl(icomp,2)=0.0
        ctlperequipl(icomp,2)=VAL(ISD+2)
        ctlperotherl(icomp,2)=0.0
        if(traceok)then
          write(outs,*)'casual gains from temporal db=',
     &          VAL(ISD),VAL(ISD+1),VAL(ISD+2)
          call edisp(itu,outs)
        endif
      elseif(ICASUAL3(icomp).ne.0)then

C Data in temporal database (convective, radiant, latent for 3 separate
C slots, i.e. 9 columns of gains). Also instantiate the future value of
C each casual gain so that it can be used within MZLS3.
        IFOC=ICASUAL3(icomp)
        CALL RCTDFB(itrc,btimef,VAL,ISD,IFOC,IER)
        QCASC(2)=VAL(ISD)+VAL(ISD+3)+VAL(ISD+6)
        ctlperocupc(icomp,2)=VAL(ISD)
        ctlperlightc(icomp,2)=VAL(ISD+3)
        ctlperequipc(icomp,2)=VAL(ISD+6)
        ctlperotherc(icomp,2)=0.0

        QCASR(2)=VAL(ISD+1)+VAL(ISD+4)+VAL(ISD+7)
        ctlperocupr(icomp,2)=VAL(ISD+1)
        ctlperlightr(icomp,2)=VAL(ISD+4)
        ctlperequipr(icomp,2)=VAL(ISD+7)
        ctlperotherr(icomp,2)=0.0

        QLATN(2)=VAL(ISD+2)+VAL(ISD+5)+VAL(ISD+8)
        ctlperocupl(icomp,2)=VAL(ISD+2)
        ctlperlightl(icomp,2)=VAL(ISD+5)
        ctlperequipl(icomp,2)=VAL(ISD+8)
        ctlperotherl(icomp,2)=0.0

C Set occupancy flag for agents.
        isocc(ICOMP)=.false.
        if ((ctlperocupc(icomp,2)+
     &       ctlperocupr(icomp,2)+
     &       ctlperocupl(icomp,2)).gt.0.) then
          isocc(ICOMP)=.true.
        endif

        if(traceok)then
          write(outs,*)'separate casual gains from temporal db=',
     &        VAL(ISD),VAL(ISD+1),VAL(ISD+2),VAL(ISD+3),VAL(ISD+4),
     &        VAL(ISD+5),VAL(ISD+6),VAL(ISD+7),VAL(ISD+8)
          call edisp(itu,outs)
        endif
      elseif(IACTIV(icomp).ne.0)then

C Activity data for UK NCM calculations are held in a temporal file.
C The calculations need to change in case the order of the 
C activity data type changes (in the tdf file).
        IFOC=IACTIV(icomp)
        CALL RCTDFB(itrc,btimef,VAL,ISD,IFOC,IER)

C Total occupancy gains: occupancy density*metabolic rate*fraction*Zone floor area.
        QTOTOCCUP=TMAR(IFOC,1)*ITMAR(IFOC,2)*VAL(ISD)*ZBASEA(ICOMP)

C Sensible casual gains from occupants (using the percentage of 
C latent occupant gains here).
        QOCCCASC=(QTOTOCCUP-((QTOTOCCUP*ITMAR(IFOC,3))/100.0))/2.0

C Convective and radiative occupant gains are the same at the moment.
        QOCCCASR=QOCCCASC

C Using again the percentage of latent gains for the occupants' latent heat gains.
        QOCCLATN=((QTOTOCCUP*ITMAR(IFOC,3))/100.0)

C For UK NCM calculations, check for overheating.
        if(isbem.eq.2)then

C Count and increment timesteps to store overheating and lighting
C results that do not include start-up days.
          i_countsteps(icomp)=i_countsteps(icomp)+1        

C Initialise counters for occupied and overheating hours.
          if(i_countsteps(icomp).eq.1)then
            ihroverheats(icomp)=0
            ihroccup(icomp)=0
            iOperTimestep(ICOMP)=0 
          endif
   
C Calculate total number of start-up timesteps.
          numbStartTimesteps=itcnst*NTSTEP*24
          if(i_countsteps(icomp).gt.numbStartTimesteps)then
            call RegOveheat(icomp)  ! timestep overheating
          endif
        endif

C Total equipment gains: equipment gains in w/m2*fraction*Zone floor area.
        QTOTEQUIP=TMAR(IFOC,4)*VAL(ISD+2)*ZBASEA(ICOMP)

C Get the casual sensible gains from equipment (using the percentage of 
C latent equipment gains).
        QEQUIPCASC=(QTOTEQUIP-
     &  ((QTOTEQUIP*(real(ITMAR(IFOC,5))))/100.0))/2.

C Convective and radiative equipment gains are the same at the moment (50%).
        QEQUIPCASR=QEQUIPCASC
        QEQUIPLATN=QTOTEQUIP*(real(ITMAR(IFOC,5)))/100.0

C If UK notional or reference model then assign the value of 3.75 or 5.2 W/m2 per 100 lux (sbem.db1). 
C If UK typical model then assign the value of 4.5 or 6.2 W/m2 per 100 lux (sbem.db1).
C Number of lux and fraction of opearation is still taken from the activity database (tdf).
C << Additional logic in the future should separate office/storage/industrial spaces
C for notional building that take 3.75 W/m2 per 100 lux from "other spaces 
C that take 5.2 W/m2 per 100 lux. >>
C Display lighting, TMAR(IFOC,7), is added for 24hrs every day. 

C Establish solar data.
        CALL MZSINT(ICOMP,QDIR,QDIF)
        CALL LUMEFF(QDIF,QDIR,SALT,IDYP,skyeff,suneff)
        E_ext=QDIF*skyeff + QDIR *suneff*sin(salt*PI/180.0)

C Convert units to klux (DF are in %) following SBEM manual units.
        E_ext=E_ext/1000.
        E_design=lighting_lux(theactivityindex(ICOMP))

C Initialise.
          FDF1=1. ; FDR1=1. ; FDF2=1. ; FDR2=1. ; FO=1. ; FD=1.
     
C Manual switching.
        IF((E_ext*DFFront(Icomp)*10.).GT.E_design)THEN
          FDF1=0.5
        ELSE
          FDF1=1.0
        ENDIF
        IF((E_ext*DFRear(Icomp)*10.).GT.E_design)THEN
          FDR1=0.5
        ELSE
          FDR1=1.0
        ENDIF
        FD1=0.5*(FDF1+FDR1)

C Do not apply local manual switching if floor area is more than 30 m^2.
        IF(ZBASEA(ICOMP).ge.30.0)FD1=1.0

C Apply local manual switching if activity is not one of the following:
C corridors or other circulation areas, dry sports/fitness, ice rinks,
C changing rooms, swimming pools, sales areas, baggage reclaim areas,
C security check areas, eating/drinking areas, halls, lecture theatres,
C cold stores, display areas, A&E, industrial process areas, warehouse 
C storage or performance areas (stages) for which FD=1.
        IF(theactivityindex(ICOMP).GT.0)THEN   ! trap zero array index
          IF(loc_man_sw(theactivityindex(ICOMP)).eq.1)FD1=1.0
        ENDIF

C Apply local manual switching to notional, reference and typical models.
        IF(INOTI.EQ.1.OR.INOTI.EQ.2)THEN  ! notional and reference models
          QTOTLIGHT=fNotionalLighting(theactivityindex(ICOMP))*
     &     (real(ITMAR(IFOC,6))/100.0)*VAL(ISD+1)*ZBASEA(ICOMP)*FD1+
     &     TMAR(IFOC,7)*ZBASEA(ICOMP)
        ELSEIF(INOTI.EQ.3)THEN            ! typical model
          QTOTLIGHT=fTypicalLighting(theactivityindex(ICOMP))*
     &     (real(ITMAR(IFOC,6))/100.0)*VAL(ISD+1)*ZBASEA(ICOMP)*FD1+
     &      TMAR(IFOC,7)*ZBASEA(ICOMP)
        ELSE

C Arrive here if INOTI is 4 (stripped UKNCM model) or INOTI is 0 (not a UKNCM model).
C Apply UKNCM lighting control to the stripped model.

C Photoelectric switching.
          IF(Ipe_control(ICOMP).EQ.1)THEN
            IF((E_ext*DFFront(Icomp)*10.).GT.E_design)THEN
              FDF2=0.0
            ELSEIF((E_ext*DFFront(Icomp)*10.).GT.(0.5*E_design))THEN
              FDF2=0.5
            ELSE
              FDF2=1.0
            ENDIF
            IF((E_ext*DFRear(Icomp)*10.).GT.E_design)THEN
              FDR2=0.0
            ELSEIF((E_ext*DFRear(Icomp)*10.).GT.(0.5*E_design))THEN
              FDR2=0.5
            ELSE
              FDR2=1.0
            ENDIF

C Photoelectric dimming.
          ELSEIF(Ipe_control(ICOMP).EQ.2)THEN
            IF((E_ext*DFFront(Icomp)*10.).GT.E_design)THEN
              FDF2=0.0
            ELSE
              call eclose(E_design,0.0,0.0001,closer)
              IF(.NOT.CLOSER)FDF2=1.-E_ext*DFFront(Icomp)*10./E_design
              IF(FDF2.LT.0.0)FDF2=0.0
            ENDIF
            IF((E_ext*DFRear(Icomp)*10.).GT.E_design)THEN
              FDR2=0.0
            ELSE
              call eclose(E_design,0.0,0.0001,closer)
              IF(.NOT.CLOSER)FDR2=1.-E_ext*DFRear(Icomp)*10./E_design
              IF(FDR2.LT.0.0)FDR2=0.0
            ENDIF
          ENDIF

C Photosensor front only.
          if(Idaylightzoning(Icomp).eq.1)then
            FD2=0.5*(FDF2+1.0)
          elseif(Idaylightzoning(Icomp).eq.2)then

C Photosensor front and back.
            FD2=0.5*(FDF2+FDR2)
          endif

C Calculate daylight correction factor, FD.
C Manual switching.
          IF(Ilightcontrol(ICOMP).EQ.1)THEN
            FD=FD1

C Photoelectric control.
          ELSEIF(Ilightcontrol(ICOMP).EQ.2)THEN         
            FD=FD2

C Both photoelectric and manual.
          ELSEIF(Ilightcontrol(ICOMP).EQ.3)THEN
            FD=min(FD1,FD2)

C No control.
          ELSE
            FD=1.
          ENDIF

C Range error trap.
          IF(FD.LE.1.0.and.FD.GE.0.0)then
            continue
          ELSE
            FD=1.
          ENDIF

C Calculate occupancy sensing correction factor, FO.
          IF(IOcc_sensing(ICOMP).EQ.1)THEN
            FO=0.95
          ELSEIF(IOcc_sensing(ICOMP).EQ.2)THEN
            FO=0.95
          ELSEIF(IOcc_sensing(ICOMP).EQ.3)THEN
            FO=0.90
          ELSEIF(IOcc_sensing(ICOMP).EQ.4)THEN
            FO=0.90
          ELSEIF(IOcc_sensing(ICOMP).EQ.5)THEN 
            FO=0.82
          ELSE
            FO=1.0
          ENDIF

C Calculate parasitic power, PP.
          PP=PE_sensor_PP(ICOMP)+Occ_sensor_PP(ICOMP)

C Need type of light and then a value taken from sbem.db1 will be combined with a TDF static data.       
          IF(ILIGHTUSER(ICOMP).EQ.0)THEN
            ILIGHT_IN_SBEMDB=ILITYP(ICOMP)

C Multiply lux levels (static tdf) with fraction schedule and 
C a value (in W/m^2/100lux) taken from NCM file for the specific type of lights.
C In PDRL(ILIGHT_IN_SBEMDB,2), 2 means commercial buildings (the only option
C at preset). << Additional logic required: 1 for industrial buildings as in
C the NCM method.
C Display lighting, TMAR(IFOC,7), is added for 24hrs every day.
            if(ILIGHT_IN_SBEMDB.eq.0)then
              QTOTLIGHT=0.
            else            
              QTOTLIGHT=
     &        PDRL(ILIGHT_IN_SBEMDB,2)*(real(ITMAR(IFOC,6))/100.0)*
     &        VAL(ISD+1)*ZBASEA(ICOMP)*FD*FO+
     &        (TMAR(IFOC,7)+PP)*ZBASEA(ICOMP)
            endif
          ELSEIF(ILIGHTUSER(ICOMP).EQ.1)THEN

C User specfied gains from lights (NCM purposes) in W/m^2 per 100 lux
C Gains=W/m^2per100lux*fraction lights on*area*(required lux level/100lux)+
C DisplaylightingW/m^2*Area
            QTOTLIGHT=LIGHTWATTAGE(ICOMP)*VAL(ISD+1)*ZBASEA(ICOMP)*
     &         (real(ITMAR(IFOC,6)/100.0))*FO*FD+
     &         (PP+TMAR(IFOC,7))*ZBASEA(ICOMP)
          ENDIF
        ENDIF
 
C Convective and radiative lighting gains are the same at the moment (50%).
        QLIGHTCASC=QTOTLIGHT/2.
        QLIGHTCASR=QLIGHTCASC
              
C For lighting heat gains assume latent gains are zero.
        QLIGHTLATN=0
        QCASC(2)=QOCCCASC+QEQUIPCASC+QLIGHTCASC
        QCASR(2)=QOCCCASR+QEQUIPCASR+QLIGHTCASR
        QLATN(2)=QOCCLATN+QEQUIPLATN+QLIGHTLATN

C Instantiate the future value of each casual gain so that
C it can be used within MZLS3.
C << ? for 4th other slot. >>
        ctlperocupc(icomp,2)=QOCCCASC
        ctlperlightc(icomp,2)=QLIGHTCASC
        ctlperequipc(icomp,2)=QEQUIPCASC
        ctlperotherc(icomp,2)=0.0
        ctlperocupr(icomp,2)=QOCCCASR
        ctlperlightr(icomp,2)=QLIGHTCASR
        ctlperequipr(icomp,2)=QEQUIPCASR
        ctlperotherr(icomp,2)=0.0
        ctlperocupl(icomp,2)=QOCCLATN
        ctlperlightl(icomp,2)=QLIGHTLATN
        ctlperequipl(icomp,2)=QEQUIPLATN
        ctlperotherl(icomp,2)=0.0

C Using LGHT_ncm_Energy(icomp) for the results of the UK's NCM method.
C This will be in Wh units. It will also need to be divided with NTSTEP
C (number of timesteps) to use it in results.
        if(isbem.eq.2)then

C Initialise LGHT_ncm_Energy(icomp).
          if(i_countsteps(icomp).eq.1)then
            LGHT_ncm_Energy(icomp)=0.
            do 333 imonth_iter=1,12
              fmonthLGHT_ncm_Energy(imonth_iter,icomp)=0.
 333        continue
          endif          

C Total number of start-up timesteps.
C           numbStartTimesteps=itcnst*NTSTEP*24
          if(i_countsteps(icomp).gt.numbStartTimesteps)then

C tmpLGHT_ncm_Energy(icomp) is a temporary array to pass the lighting energy
C gains/consumption back to LGHT_ncm_Energy(icomp) and allow re-initialiasation
C for the next simulation.
            LGHT_ncm_Energy(icomp)=LGHT_ncm_Energy(icomp)+
     &        ctlperlightc(icomp,2)+ctlperlightr(icomp,2)+
     &        ctlperlightl(icomp,2)    

C Month number.
            call edayr(idyp,idayn,imonth_iter)
            fmonthLGHT_ncm_Energy(imonth_iter,icomp)=
     &          fmonthLGHT_ncm_Energy(imonth_iter,icomp)+ 
     &          ctlperlightc(icomp,2)+ctlperlightr(icomp,2)+
     &          ctlperlightl(icomp,2)       
          endif

C At the last timestep, re-initialise i_countsteps(icomp) to 0.
C First find total number of simulation period timesteps including start-up period.
          numTotTimstps=(((isdf-isds)+1)*24*NTSTEP)+numbStartTimesteps
          if(i_countsteps(icomp).eq.numTotTimstps)then
            i_countsteps(icomp)=0
          endif           
        endif
        
        if(traceok)then
          write(outs,*)'TOTAL OCCUPANT gains from temporal db=',
     &      QTOTOCCUP,QOCCCASC,QOCCCASR,QOCCLATN
          call edisp(itu,outs)
          write(outs,*)'TOTAL LIGHTING gains from temporal db=',
     &      QTOTLIGHT,QLIGHTCASC,QLIGHTCASR,QLIGHTLATN
          call edisp(itu,outs)
          write(outs,*)'TOTAL EQUIPMENT gains from temporal db=',
     &      QTOTEQUIP,QEQUIPCASC,QEQUIPCASR,QEQUIPLATN
          call edisp(itu,outs)
        endif

      else

C Data in zone operations file. For each casual gain type, set gain for current period.
C Check if ipausecas is non-zero for this zone and simulation is after the startup period.
C If so, consult user and if they agree call eroper and MZCASG for this 
C zone to refresh common CASGNS. << Note that this does not yet alter infiltration or
C ventilation in the zone. >>
        if(ipausecas(icomp).eq.1)then
          if (bInStartup())then
            continue
          else
            CALL DAYCLK(IDYP,BTIMEF,iuout)
            write(outs,'(4a)') 'For ',zname(icomp),
     &        ' rescan operation file: ',LPROJ(icomp)(1:36)
            nbhelp=2
            CALL EASKOK(OUTS,'Confirm?',OK,nbhelp)
            if(ok)then
              iunit=ifil+1
              iitrc=1
              CALL EROPER(IITRC,ITU,IUNIT,icomp,IER)
              CALL MZCASG(icomp)
            endif
          endif
        endif

C Process HOT3000 base loads. Note that latent gains are presently not
C supported. << However, if they were to be implemented, code similar to below
C need to be added to moistr.F, subsys.F and drylink.F in the ebld folder.
C Casual gain types for use with HOT3000 central and zonal base loads:
C  1. BL-App (appliances linked with BCD file).
C  2. BL-Lights (lights linked with BCD file).
C  3. BL-OElec (other electrical linked with BCD file).
C  4. BL-Occ (occupants linked with BCD file).
C Time period is always 0-24 for HOT3000 base loads.
       IF(bH3KExtentionsActive().and.iH3K_Base_Loads(icomp).eq.1)then
         Total_BL_Convective_gain        = 0. ! (W) 
         Total_BL_Radiative_gain         = 0. ! (W)
         Total_BL_electricity_Lights     = 0. ! (W)
         Total_BL_electricity_Appliances = 0. ! (W)
         Total_BL_electricity_Other      = 0. ! (W) 

         do 100 IGN=1,nGain_types(icomp,iday)
           W_BL_heat_gain          = 0.       ! Zero out heat gain power for this next gain loop
           iColumn   = iBCD_column(icomp,iday,IGN)

C BCD interpolation method.
           iInterp_method = 1 ! 1 = step-wise; 2 = linear

           write (cContext,*) "Determining BL profile for gain type ",
     &          iGain_Type(icomp,iday,IGN)

           if(iGain_Type(icomp,iday,IGN).eq.1)then      ! BL-App
             W_BL_heat_gain = fGet_BC_data_by_col(      ! heat gain only
     &        iColumn,
     &        fConvert_current_TS_to_DAY(),
     &        iInterp_method,
     &        cContext)
     &        * fBL_fraction(icomp,iday,IGN)            ! multiply by BL fraction (= 1 for zonal base loads, 0.0 < fBL_fraction < 1.0  for central base loads) 

             Total_BL_electricity_Appliances = W_BL_heat_gain

           elseif(iGain_Type(icomp,iday,IGN).eq.2)then  ! BL-Lights
             W_BL_heat_gain = fGet_BC_data_by_col(      ! heat gain only
     &        iColumn,
     &        fConvert_current_TS_to_DAY(),
     &        iInterp_method,
     &        cContext)
     &        * fBL_fraction(icomp,iday,IGN)            ! multiply by BL fraction (= 1 for zonal base loads, 0.0 < fBL_fraction < 1.0  for central base loads) 

             Total_BL_electricity_Lights = W_BL_heat_gain

           elseif(iGain_Type(icomp,iday,IGN).eq.3)then  ! BL-OElec
             W_BL_heat_gain = fGet_BC_data_by_col(      ! heat gain only
     &        iColumn,
     &        fConvert_current_TS_to_DAY(),
     &        iInterp_method,
     &        cContext)
     &        * fBL_fraction(icomp,iday,IGN)            ! multiply by BL fraction (= 1 for zonal base loads, 0.0 < fBL_fraction < 1.0  for central base loads) 

             Total_BL_electricity_Other = W_BL_heat_gain

           elseif(iGain_Type(icomp,iday,IGN).eq.4)then  ! BL-Occ
             W_BL_heat_gain = fGet_BC_data_by_col(      ! heat gain only
     &        iColumn,
     &        fConvert_current_TS_to_DAY(),
     &        iInterp_method,
     &        cContext)
     &        * fBL_fraction(icomp,iday,IGN)            ! multiply by BL fraction (= 1 for zonal base loads, 0.0 < fBL_fraction < 1.0  for central base loads) 
     
           end if

C HOT3000 base loads convective gains (W).
           Total_BL_Convective_gain = Total_BL_Convective_gain 
     &             + CGSENC(icomp,iday,iGain_Type(icomp,iday,IGN),1) ! sensible load fraction * convective load fraction
     &             * W_BL_heat_gain                                  ! heat gain for this time step

C HOT3000 base loads radiative gains (W).
           Total_BL_Radiative_gain = Total_BL_Radiative_gain 
     &             + CGSENR(icomp,iday,iGain_Type(icomp,iday,IGN),1) ! sensible load fraction * radiative load fraction
     &             * W_BL_heat_gain                                  ! heat gain for this time step

  100    continue       ! number of casual gains loop

C Add radiant and convective gains from HOT3000 base loads to QCASC and QCASR, which
C are used to calculate zone heat injections.
         QCASC(2)=QCASC(2) + Total_BL_Convective_gain
         QCASR(2)=QCASR(2) + Total_BL_Radiative_gain

C Report electricity consumption for each type of casual gain
C for the zone to SiteUtilities.

C Appliances.
        fSUFuelEnergyUse(iElectricity, iUseEquipment) = 
     &          Total_BL_electricity_Appliances

C Lights.
        fSUFuelEnergyUse(iElectricity, iUseLights) = 
     &          Total_BL_electricity_Lights

C Other.
        fSUFuelEnergyUse(iElectricity, iUseOther) = 
     &          Total_BL_electricity_Other
     
        call StoreH3KBaseLoadsEnergyUse(icomp, fSUFuelEnergyUse)

       ELSE

C ESP-r default treatment, i.e. data from zone operations file.
C For each casual gain type, set gain for current period.
        do 10 IGN=1,MGTY

C First initialise SHOCC commons to -1, i.e. assume no SHOCC-calulated loads.
C The 'fetchloads' subroutine will override '-1' if SHOCC calculates loads.
          shcon=-1.0; shrad=-1.0; shlat=-1.
                    
          call fetchloads(icomp,lodlabel(icomp,ign),
     &      len(lodlabel(icomp,ign)),shcon,shrad,shlat)

C Determine the current period.
          IPER=NCGPER(ICOMP,IDAY,IGN)
          if (IPER.gt.0) then
            do I=1,NCGPER(ICOMP,IDAY,IGN)
              if (BTIMEF.le.TCGS(ICOMP,IDAY,IGN,I+1).and.
     &            BTIMEF.gt.TCGS(ICOMP,IDAY,IGN,I)) then
                IPER=I
              endif
            enddo

C Calculate the sensible and latent load from occupants in case of
C dynamic people.
            if (caskeytype(ICOMP,IGN)(1:14).eq.'dynamic_people'
     &          .and. lodatr1(ICOMP,IGN).GT.0) then

              INDX=ICOMP*(-1)   ! negate the zone index for use in mzmixt
              CALL MZMIXT(INDX,TMRT,TMIX)
              TAIR=TPA(ICOMP)
              TOP=0.5*TMRT+0.5*TAIR 
C              call MZVAPC(ICOMP,ZRH,CNDS)   ! not used
  
              RHOCC=PCRH2(TPA(ICOMP),GPA(ICOMP),PATMOS)/100.0
              if (RHOC.GE.100) RHOCC=0.99
              HMETA1=META1(ICOMP,IDAY,IGN,IPER)

C Metabolic rate (METAo) is converted to watts.
C 1 Met = 58 W/m^2 and the mean surface area of the human body is
C approximately 1.8 m^2.
              METAo=META1(ICOMP,IDAY,IGN,IPER)*58*1.8
              CONCCO=CONCC(ICOMP,IDAY,IGN,IPER)
              RADCCO=RADCC(ICOMP,IDAY,IGN,IPER)

C Dynamic clothing prediction based on t out at 6 am and Tmix.
              if (CLOTH(ICOMP,IDAY,IGN,IPER).lt.0.) then
                tatsix=itmp(6)/10.
                CLOTHO=10**(0.2134-0.0063*tatsix-0.0165*TMIX)
                if(CLOTHO.le.0.4)CLOTHO=0.4
                if(CLOTHO.GE.1.1)CLOTHO=1.1
              else
                CLOTHO=CLOTH(ICOMP,IDAY,IGN,IPER)
              endif

              if (NSINC.LE.2)msegflag=0
              TOCN=NBMEN(ICOMP,IDAY,IGN,IPER)+
     &          NBWOM(ICOMP,IDAY,IGN,IPER)+NBCHI(ICOMP,IDAY,IGN,IPER)
     
              if(TOCN.GT.0.1) then

C Polynomial equation to calculate sensible and latent heat load
C from occupant where Top and METAo are the operative zone temperature
C and metabolic rate of occupant respectively. This model was correlated
C from published experimental data. The relationship between the body’s
C
C                      M = E ± R ± C ± S 
C where,
C       M = metabolic rate;
C       E = rate of heat loss by evaporation, respiration and elimination;
C       R = radiation rate;
C       C = conduction and convection rate;
C       S = body heat storage rate.
                if(lodatr1(ICOMP,IGN).eq.1)then
                  SGAIN=198.42617-3.80901*TOP-0.05419*TOP**2-0.42472*
     &              METAo+0.00171*METAo**2+0.01287*TOP*METAo-
     &              0.00004*TOP*METAo**2+0.00002*METAo*TOP**2-
     &              0.00000038516*METAo**2*TOP**2
                  skcon(ICOMP) = SGAIN ! set for CFD
                  LGAIN=227.89011-14.95202*TOP+0.24884*TOP**2-
     &              2.56734 *METAo+0.00715*METAo**2+
     &              0.132*TOP*METAo-0.00017*TOP*METAo**2-
     &              0.00107*METAo*TOP**2

C Polynomial equation from Energy Plus (function of Temperature and metabolic rate).
C If E should be always positive, this polynomial is inefficient, for example when
C temperature goes below 14C at Met = 100 W (ASHRAE).
                elseif(lodatr1(ICOMP,IGN).eq.2)then
                  SGAIN=6.461927+0.946892*METAo+0.0000255737*METAo**2+
     &              7.139322*TOP-0.0627909*TOP*METAo+
     &              0.0000589172*TOP*METAo**2-
     &              0.19855*TOP**2+0.000940018*TOP**2*METAo-
     &              0.00000149532*TOP**2*METAo**2
                  skcon(ICOMP) = SGAIN ! set for CFD
                  LGAIN=METAo-SGAIN

C Two node model.
                elseif(lodatr1(ICOMP,IGN).eq.3)THEN
                  call DynamicOcc(ICOMP,METAo,CLOTHO)
                  SGAIN=skcon(ICOMP)
                  LGAIN=sklat(ICOMP)

C Segmented person model.
                elseif(lodatr1(ICOMP,IGN).eq.4)then
                  call segmentedocc(ICOMP,METAo,TOP,RHOCC,CLOTHO)
                  SGAIN=Qoccsens1(ICOMP)
                  skcon(ICOMP)=SGAIN ! set in case CFD BC is whole_body
                  LGAIN=Qocclat1(ICOMP)
                endif

C CGSECo CGSERo and CGLATo are the total convective radiative and
C latent load from occupants taking into consideration number of
C people and age/gender.
                CGSECo=(SGAIN*NBMEN(ICOMP,IDAY,IGN,IPER)+
     &            0.85*SGAIN*NBWOM(ICOMP,IDAY,IGN,IPER)+
     &            0.75*SGAIN*NBCHI(ICOMP,IDAY,IGN,IPER))*
     &            CONCCO
                CGSERo=(SGAIN*NBMEN(ICOMP,IDAY,IGN,IPER)+
     &            0.85*SGAIN*NBWOM(ICOMP,IDAY,IGN,IPER)+
     &            0.75*SGAIN*NBCHI(ICOMP,IDAY,IGN,IPER))*
     &            RADCCO
                CGLATo=LGAIN*NBMEN(ICOMP,IDAY,IGN,IPER)+
     &            0.85*LGAIN*NBWOM(ICOMP,IDAY,IGN,IPER)+
     &            0.75*LGAIN*NBCHI(ICOMP,IDAY,IGN,IPER)

                CGSENC(ICOMP,IDAY,IGN,IPER)=CGSECo
                CGSENR(ICOMP,IDAY,IGN,IPER)=CGSERo
                CGLAT(ICOMP,IDAY,IGN,IPER)=CGLATo
              else
                CGSENC(ICOMP,IDAY,IGN,IPER)=0.0
                CGSENR(ICOMP,IDAY,IGN,IPER)=0.0
                CGLAT(ICOMP,IDAY,IGN,IPER)=0.0
              endif
            endif

C Set future casual gain values if there are periods defined.
            if(shcon.lt.0.) then

C FMI: If occupancy, lighting or equipment control is active, multiply CGCTL
C by value from FMU.
              if (IGN.eq.1.and.FMUDOCTL(ICOMP,7)) then
                rtmp=FMUCTL(ICOMP,7)
              elseif (IGN.eq.2.and.FMUDOCTL(ICOMP,2)) then
                rtmp=FMUCTL(ICOMP,2)
              elseif (IGN.eq.3.and.FMUDOCTL(ICOMP,4)) then
                rtmp=FMUCTL(ICOMP,4)
              else
                rtmp=1.0
              endif
              CGCTL(ICOMP,2,IGN)=CGCTL(ICOMP,2,IGN)*rtmp

C Set occupancy flag for agents.
              if (IGN.eq.1) then
                isocc(ICOMP)=.false.
                if ((CGSENC(ICOMP,IDAY,IGN,IPER)+
     &               CGSENR(ICOMP,IDAY,IGN,IPER)+
     &               CGLAT(ICOMP,IDAY,IGN,IPER))*
     &              CGCTL(ICOMP,2,IGN).gt.0.) then
                  isocc(ICOMP)=.true.
                endif
              endif

              QCASC(2)=QCASC(2)+CGSENC(ICOMP,IDAY,IGN,IPER)*
     &                 CGCTL(ICOMP,2,IGN)
              QCASR(2)=QCASR(2)+CGSENR(ICOMP,IDAY,IGN,IPER)*
     &                 CGCTL(ICOMP,2,IGN)
              QLATN(2)=QLATN(2)+CGLAT(ICOMP,IDAY,IGN,IPER)*
     &                 CGCTL(ICOMP,2,IGN)
              if(IGN.gt.1)then 
                znElecCasGain(ICOMP)=QCASC(2)+QCASR(2)
              endif
            else

C Instead of using fracf, add-on shcon, shrad and shlat for SHOCC.
              QCASC(2)=QCASC(2)+shcon
              QCASR(2)=QCASR(2)+shrad
              QLATN(2)=QLATN(2)+shlat

C Quick fix: cgctl variable is used within eletrical network
C so if SHOCC load then cgctl equals shcon/CGCSENC ratio.
              CGCTL(ICOMP,2,IGN)=shcon/CGSENC(ICOMP,IDAY,IGN,IPER)
            endif

C Debug.
            if(traceok)then
              write(itu,*) 'match ',DSTR,BTIMEF,IPER,IGN,IDAY,ICOMP
              write(itu,*) 'Time    Zone   Period   GainSen'
              write(itu,*) BTIMEF,ICOMP,IPER,CGSENC(ICOMP,IDAY,IGN,IPER)
              write(itu,*) QCASC,QCASR,QLATN,' ctl:',CGCTL(ICOMP,1,1),
     &          CGCTL(ICOMP,2,1),CGCTL(ICOMP,1,2),CGCTL(ICOMP,2,2)  
            endif

C Set future electrical loads.
            if (BTIMEF.le.TCGS(ICOMP,IDAY,IGN,IPER+1).and.
     &          BTIMEF.gt.TCGS(ICOMP,IDAY,IGN,IPER)) then
              IF(IELF(ICOMP).GT.0) THEN

C Loop through power consumption for each phase.
                DO 75 IPH=1,MPHAS
                  PEZON(ICOMP,IGN)=PEZON(ICOMP,IGN)-
     &              ELQ(ICOMP,IDAY,IGN,IPER,IPH)*CGCTL(ICOMP,2,IGN)
                  QEZON(ICOMP,IGN)=QEZON(ICOMP,IGN)-
     &              ELR(ICOMP,IDAY,IGN,IPER,IPH)*CGCTL(ICOMP,2,IGN)

C Keep track of electrical loads for trace.
                  RP(ICOMP,2,IGN,IPH)=RP(ICOMP,2,IGN,IPH)+
     &              ELQ(ICOMP,IDAY,IGN,IPER,IPH)*CGCTL(ICOMP,2,IGN)
                  QP(ICOMP,2,IGN,IPH)=QP(ICOMP,2,IGN,IPH)+
     &              ELR(ICOMP,IDAY,IGN,IPER,IPH)*CGCTL(ICOMP,2,IGN)
  75            CONTINUE
              ENDIF 
            endif

            Cas_Lights2(icomp) = CGCTL(ICOMP,2,2)*
     &          (CGSENC(ICOMP,IDAY,2,IPER)+CGSENR(ICOMP,IDAY,2,IPER))
            Cas_Equip2(icomp)  = CGCTL(ICOMP,2,3)*
     &          (CGSENC(ICOMP,IDAY,3,IPER)+CGSENR(ICOMP,IDAY,3,IPER))

          endif
 10     continue   ! end of gain type loop
      
C Store real power consumption for lights and equipment.
        Cas_Lights(icomp) = -1. * pezon(icomp,2)    ! lights are casual gain type 2
        Cas_Equip(icomp) = -1. * pezon(icomp,3)     ! equipment is casual gain type 3
      

       ENDIF    ! HOT3000 Base Loads conditional block
      endif     ! UK's NCM method conditional block

C If adaptive comfort algorithm active, add sensible heat gains when on.
      call eclose(PREVSTATFAN,1.00,0.01,closea)
      IF(CLOSEA)THEN

        QCASC(2)=QCASC(2)+TOTPWR(ICOMP)

      ENDIF

C BEGIN NRCan casual-gain -> electrical coupling code.
C
C NOTE: electric-load -> latent couplings are presently not supported. 
C However, if they were to be implemented, code similar to that below
C would have to be added to moistr.F, subsys.F and drylink.F.

C For the present zone, loop through all gains associated
C with the casual-gain->electrical load coupling

C Common for HOT3000 electrical-network -> casual-gain coupling
C Symopsys: E_to_G_coupling: boolean indicating coupling exists
C Coupling_gain_info(i,j,k,l): array containing data for gain k on daytype j for zone i. 
C    l = 1 -> start hour (0->24)
C    l = 2 -> end hour (0->24)
C    l = 3 -> sensible load (fraction of non-hvac elec load)
C    l = 4 -> latent load (fraction of non-hvac elec load)
C    l = 5 -> radiative feaction (0->1)
C    l = 6 -> convective fraction (0->1)
C    l = 7 -> casual gain slot (1-7)
C
C Initialise total electric load-related gains.
      Total_NG = 0. ! (W)
      Total_elec_CON_gain = 0. ! (W) 
      Total_elec_RAD_gain = 0. ! (W)
      DO ii = 1, iNumb_Gain_coupling(icomp,iDay)

C Check if the future building time is between the start hour and            
C the end hour of the gain.
        start_hour = Coupling_gain_info(icomp,iDay,ii,1) ! start hour 0-24)
        stop_hour  = Coupling_gain_info(icomp,iDay,ii,2) ! end hour (0-24)
        if ( ( btimef .gt. start_hour ) 
     &          .and. ( btimef .le. stop_hour ) )then

C Gain applies to future time. Get total non-hvac electric load (W_elec_nonHVAC, W),
C set the BCD interpolation method and generate a context message.
          iInterp_method = 1 ! 1 = step-wise; 2 = linear
          write (cContext,*) "Determining the AL profile for type ",
     &          Coupling_gain_info(icomp,iDay,ii,7)
          iGainSlot = nint(Coupling_gain_info(icomp,iDay,ii,7))

C Test for case of non_HVAC_electrical loads and then for tagged appliance
C loads that may have been encountered in the zone operations file. Use calls
C to fGet_BC_data_by_name to recover data in boundary conditions file.
          if(caskeytype(icomp,iGainSlot)(1:13).eq.'net_utilities')then
            W_electric_nonHVAC =  Elec_Net_Load_Calc(occupant_load) ! W
          elseif(caskeytype(icomp,iGainSlot)(1:15).eq.
     &           'ALOtherElectric')then
            cBCD_col = "ALOtherElectric"
            W_heat_nonHVAC = fGet_BC_data_by_name( ! heat gain only (no NG)
     &        cBCD_col,
     &        fConvert_current_TS_to_DAY(),
     &        iInterp_method,
     &        cContext)
          elseif(caskeytype(icomp,iGainSlot)(1:9).eq.'ALStoveNG')then
            cBCD_col = "ALStoveNG"
            W_heat_nonHVAC = fGet_BC_data_by_name( ! heat gain (from elec or NG)
     &        cBCD_col,
     &        fConvert_current_TS_to_DAY(),
     &        iInterp_method,
     &        cContext)
                             ! for a NG stove, set the NG equal to the heat gain
            W_NG_nonHVAC = W_heat_nonHVAC
          elseif(caskeytype(icomp,iGainSlot)(1:15).eq.
     &           'ALStoveElectric')then
            cBCD_col = "ALStoveElectric"
            W_heat_nonHVAC = fGet_BC_data_by_name( ! heat gain (from elec or NG)
     &        cBCD_col,
     &        fConvert_current_TS_to_DAY(),
     &        iInterp_method,
     &        cContext)
          elseif(caskeytype(icomp,iGainSlot)(1:7).eq.'ALDryer')then
            cBCD_col = "AL-Dryer"
            W_NG_nonHVAC = fGet_BC_data_by_name(  ! natural gas only
     &        cBCD_col,
     &        fConvert_current_TS_to_DAY(),
     &        iInterp_method,
     &        cContext)
          endif

C Get data for gain coupling.
          Sens_load_fraction = 
     &             Coupling_gain_info(icomp,iDay,ii,3) ! (0->1)
          radiant_load_fraction =  
     &             Coupling_gain_info(icomp,iDay,ii,5) ! (0->1)
          convective_load_fraction =  
     &             Coupling_gain_info(icomp,iDay,ii,6) ! (0->1)        

C Convective casual gains (W)
          Total_elec_CON_gain = Total_elec_CON_gain 
     &             + sens_load_fraction 
     &             * convective_load_fraction 
     &             * W_electric_nonHVAC

C Radiative casual gains (W)
          Total_elec_RAD_gain = Total_elec_RAD_gain 
     &             + sens_load_fraction 
     &             * radiant_load_fraction 
     &             * W_electric_nonHVAC

C Accumulate NG consumption (W)
          Total_NG = Total_NG 
     &             + sens_load_fraction 
     &             * W_NG_nonHVAC 

        endif          
      ENDDO

C NOTE: add radiant & convective gains from occupant-driven
C electric loads to ESP-r common blocks QCASR and QCASC.
C These data are used to calculate the heat injection
C into the building.
C
C Also append convective and radiant gains associated with
C heat losses from plant components. Gains from ideal plant 
C components are assumed to be convective. 
      QCASC(2)= QCASC(2) + Total_elec_CON_gain
     &                   + fPltZone_Convective_Gain(iComp,iFuture) ! (W)
     &                   + fIdealDHWGain(iComp)
     
      QCASR(2)= QCASR(2) + Total_elec_RAD_gain
     &                   + fPltZone_Radiant_Gain(iComp,iFuture)    ! (W)

C When co-simulation with TRNSYS is active, parasitic thermal losses from TRNSYS
C components (i.e. types) can be injected as casual gains into ESP-r's building
C zone. This is accomplished when the TRNSYS user connects a type's output
C representing the thermal loss to a Type 130 input. Within Type 130 this
C heat loss is then associated with an ESP-r building zone. If any such thermal
C losses have been represented then append them here.
C In the current implementation, it is assumed that all such thermal losses
C from TRNSYS appear as convective gains within ESP-r's building zone.
      IF ( COSIM ) THEN
        QCASC(2) = QCASC(2) +
     &              COSIM_DATA%ESPrZonesData(icomp)%AirPointCasualGains
      END IF

C Store future row casual and radiant gains.
      Cas_Con_dat(icomp) = QCASC(2)  ! W
      Cas_Rad_dat(icomp) = QCASR(2)  ! W

C Remember future values for next timestep.
      QCCS(ICOMP)=QCASC(2)
      QCRS(ICOMP)=QCASR(2)
      QLAT(ICOMP)=QLATN(2)
    
C If first timestep, set previous values to current.
      if(NSINC.eq.1)then
        QCASC(1)=QCASC(2)
        QCASR(1)=QCASR(2)
        QLATN(1)=QLATN(2)
      endif

C Sum present and future time rows and convert radiant casual 
C gains to W/m^2.
      QCASCT=QCASC(1)+QCASC(2)
      QCASRT(icomp)=(QCASR(1)+QCASR(2))/zonetotsurfacearea(ICOMP)
      QCASR(1)=QCASR(1)/zonetotsurfacearea(ICOMP)
      QCASR(2)=QCASR(2)/zonetotsurfacearea(ICOMP)
      QCASRS(ICOMP)=QCASR(2)
      QCASLT=QLATN(1)+QLATN(2)

C Trace output?
      if(.NOT.traceok)goto 999

C Output computed results.
      call edisp(itu,' ')
      write(outs,'(A,A)')' Casual gain summary for ',DSTR
      call edisp(itu,outs)
      call edisp(itu,' ')
      call edisp(itu,' Control fraction for each type:')
      write(outs,'(a,10f7.4)') 'Present: ',(CGCTL(icomp,1,I),I=1,MGTY)
      call edisp(itu,outs)
      write(outs,'(a,10f7.4)') 'Future : ',(CGCTL(icomp,2,I),I=1,MGTY)
      call edisp(itu,outs)
      call edisp(itu,' ')
      write(outs,'(15X,A)')
     & '|      Radiant       |    Convection      |    Latent         '
      call edisp(itu,outs)
      write(outs,'(15X,A)')
     & '|   Present   Future |  Present   Future  |  Present    Future'
      call edisp(itu,outs)

C Display radiant contribution as Watts.
      RWP=QCASR(1)*zonetotsurfacearea(ICOMP)
      RWF=QCASR(2)*zonetotsurfacearea(ICOMP)
      RW=QCASRT(icomp)*zonetotsurfacearea(ICOMP)
      write (outs,9994) RWP,RWF,QCASC(1),QCASC(2),QLATN(1),QLATN(2)
 9994 format(' Total gain    |',2F10.2,'|',2F10.2,'|',2F10.2)
      call edisp(itu,outs)
      write (outs,9996) RW,QCASCT,QCASLT
 9996 format(' Summation     |   ',F10.2,'W      |   ',F10.2,
     &                                   'W      |   ',F10.2)
      call edisp(itu,outs)

C Output the electrical results for the zone.
      if(ielf(icomp).gt.0) then
        call edisp(itu,' ')
        write(outs,'(a,1x,i3,1x,i3)') ' Zone: ',icomp,ielf(icomp)
        call edisp(itu,outs)
        write(outs,'(A,A)')' Electrical load summary for ',DSTR
        call edisp(itu,outs)
        call edisp(itu,'Each Phase  Real Power         Reactive Power')
        call edisp(itu,'          |Present|Future|    |Present|Future|')
        DO 777 K=1,MGTY
          write(outs,'(6(2F7.0,1x))') 
     &              (RP(ICOMP,1,K,J),RP(ICOMP,2,K,J),
     &               QP(ICOMP,1,K,J),QP(ICOMP,2,K,J),J=1,MPHAS)
          call edisp(itu,outs) 
          
          
  777   CONTINUE    
      endif

  999 RETURN
      END

C ******************** INTLUM ********************
C Calculate the switched level of casual gains (normally associated
C with artificial lighting) when controlled on the basis of available
C natural light.
C  SAZI   - solar azimuth.
C  SALT   - solar altitude.
C  QDIR   - direct normal irradiance.
C  QDIF   - diffuse horizontal irradiance.
C  SETPT  - set point illumination.
C  ICGCFL - control law flag (-1,0,1,2,3,4 and 31, 32 as below).
C  NEWT   - total number of external (default) windows in the zone.
C  NDF    - number of light sensors (daylight factors) for each external window.
C  DFDAT  - daylight factors (for a CIE standard overcast sky).
C  GAZI & GELV - window edge azimuthal and elevational angles.
C  FRAC   - fraction of casual gain (returned value).

      SUBROUTINE INTLUM(ICOMP,FRAC)
#include "building.h"
#include "model.h"
#include "geometry.h"
      
      integer lnblnk  ! function definition

      COMMON/BTIME/BTIMEP,BTIMEF
      common/filep/ifil
      COMMON/TC/ITC,ICNT
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      COMMON/SUNPOS/SAZI,SALT,ISUNUP
C      COMMON/CLIMI/QFP,QFF,TP,TF,QDP,QDF,VP,VF,DP,DF,HP,HF

C Zone casual gain control data.
      COMMON/CGCIN2N/NCGTC(MCOM,MDTY),NLITZ(MCOM),IDFST(MCOM,MLCOM),
     & CGX(MCOM,MLCOM,MDF),CGY(MCOM,MLCOM,MDF),CGH(MCOM,MLCOM,MDF),
     & UX(MCOM,MLCOM,MDF),UY(MCOM,MLCOM,MDF),UH(MCOM,MLCOM,MDF),
     & SETPT(MCOM,MLCOM),SYSPER(MCOM,MLCOM),SOFFLL(MCOM,MLCOM),
     & IOFFDT(MCOM,MLCOM),SMLOUT(MCOM,MLCOM),SMEOUT(MCOM,MLCOM)
      COMMON/CGCIN3/ICGCS(MCOM,MDTY),ICGCF(MCOM,MDTY),
     & ICGCFL(MCOM,MLCOM),SPELEC(MCOM,MLCOM),SLOPEM(MCOM,MLCOM)

C Daylight factor data (see dayfac.F).
      COMMON/DAYF/NDF(MCOM,MLCOM),DFDAT(MCOM,MLCOM,MGT,MDF),
     & AZIJL(MCOM,MGT),SUMDF(MCOM,MLCOM,MDF),NUDWIN(MCOM)

      COMMON/ADJC2/DIRT(MCOM,MS),DIFT(MCOM,MS),AIRT(MCOM,MS)

C TMC data.
      COMMON/PRECTC/ITMCFL(MCOM,MS),TMCT(MCOM,MTMC,5),
     &       TMCA(MCOM,MTMC,ME,5),TMCREF(MCOM,MTMC),TVTR(MCOM,MTMC)

C TMC blind control flag.
      COMMON/TMCB4/IBOFOT(MCOM,MS),NBPONT(MCOM,MS),ECRAT(MCOM,MS)

C Construction information.
      COMMON/PREC9/NCONST(MCOM),NELTS(MCOM,MS),NGAPS(MCOM,MS),
     &             NPGAP(MCOM,MS,MGP)
      
C Hunt algorithms.
      COMMON/HUNT/FIRSTL(MCOM),RANDN(MCOM)

C Switch off delay time memory.
      COMMON/SOFDT/IDTM(MCOM,MLCOM)

      DIMENSION selum(MLCOM,MDF),alt(8)
      DIMENSION zelum(MLCOM)
      DIMENSION ZFRAC(MLCOM)
      character outs*124

      logical close

C Definition of sun altitude for daylight coefficient method.
      data alt/6.,18.,30.,42.,54.,66.,78.,90./

      iunit=ifil+1

C Trace output?
      IFT=1
      IF(ITC.LE.0.OR.NSINC.LT.ITC)IFT=0
      IF(ITRACE(12).EQ.0.OR.NSINC.GT.ITCF.OR.IZNTRC(ICOMP).NE.1)IFT=0
      IF(IFT.EQ.1)THEN
        write(outs,9998)ICNT,ICOMP,zname(icomp)
 9998   format(' Subroutine INTLUM  Trace output',I4,' Zone',I3,1x,a)
        call edisp(itu,outs)
        ICNT=ICNT+1
        CALL DAYCLK(IDYP,BTIMEF,ITU)
        write(outs,'(A,A)')' Light|Sens|sun |via   |daylight',
     &        '|visib|sun/dir|sky/dif|sensed|set  |ext vert'
        call edisp(itu,outs)
        write(outs,'(A,A)')'  zone|num.|seen|window|factor% ',
     &        '|trnsm|contrib|contrib|lux   |point|ilum(Ev)'
        call edisp(itu,outs)
      ENDIF

C Assume casual gains untouched and establish solar data for correct
C time-row. 
      CALL MZSINT(ICOMP,QDIR,QDIF)

C For conversion from radians to degrees.
      PI = 4.0 * ATAN(1.0)
      TODEG=180/PI
      RAD=PI/180.

C Transform direct normal to horizontal irradiance as only horizontal
C sensor allowed in methods 1 and 2.
      dir=QDIR*sin(SALT*rad)

C Calculate external horizontal unobstructed diffuse and direct
C illuminance.
      call LUMEFF(QDIF,QDIR,SALT,IDYP,skyeff,suneff)
      esun=dir*suneff
      esky=QDIF*skyeff

C Loop through lighting zones.
      FRAC=0.
      DO 9 N=1,NLITZ(ICOMP)
        ZFRAC(N)=0.
      CALL CALCILUM(icomp,N,selum,zelum,alt,skyeff,suneff,esun,esky,IFT)

C Find lighting output fraction (FRAC) under current control.
C ICGCFL = -1 casual gain ON.
C        =  0 casual gain OFF.
C        =  1 ON/OFF control.
C        =  2 step-down control.
C        =  3 proportional control.
C        =  4 Hunt probability function.

C Special lighting control parameters, which are currently set
C to default values but later will be user specified:
C SOFFLL - switch-off light reference level as fraction of lux
C          setpoint (usual range 1.2-2.5, i.e. 120%-250%).
C SMEOUT - constant control gear + minimu lamp dimming or
C          stepping wattage fraction of total lights casual load.
C SMLOUT - minimum dimming light output fraction (usually 0.1, i.e. 10%)
C SOFFRL - switch-off light reference level (Lux).
C IOFFDT - switch-of delay time (usual range 5- 0 minutes), here number
C          of simulation time steps.
C DFRAC  - fraction of luminous output of the artificial lighting.
C ZFRAC  - fraction of electric power consumption of the artificial
C          lighting.

C Definition of switchi-off light reference level (Lux).
      SOFFRL=SOFFLL(ICOMP,N)*SETPT(ICOMP,N)

C Generally the setpoints can be set at the same lux
C level to follow Lynes' method or different setpoints can be used
C for different tasks. The grid option (1) uses individually
C placed sensors supporting sensor differentiation or averaging.
      IF(ICGCFL(ICOMP,N) .eq. -1)THEN
        ZFRAC(N)=SYSPER(ICOMP,N)
        IF(IFT.EQ.1)then
          WRITE(outs,801)N,ZFRAC(N)
  801     FORMAT(' Light zone ',I1,' always on!    ZFRAC=',F5.2)
          call edisp(itu,outs)
        endif
        
C Update corresponding SHOCC lighting.
        call updatelighting(icomp,n,zelum(n),btimep,btimef)

      ELSEIF(ICGCFL(ICOMP,N) .eq. 0)THEN
        ZFRAC(N)=0.
        IF(IFT.EQ.1)then
          WRITE(outs,802)N,ZFRAC(N)
  802     FORMAT(' Light zone ',I1,' always off!   ZFRAC=',F5.2)
          call edisp(itu,outs)
        endif
        
C Update corresponding SHOCC lighting.
        call updatelighting(icomp,n,zelum(n),btimep,btimef)

C Case of On/Off with chosen lighting systems and sensors.
      ELSEIF(ICGCFL(ICOMP,N).EQ.1)THEN
          ZFRAC(N)=0.0
          zell=zelum(N)+SETPT(ICOMP,N)
        IF(zell.GE.SOFFRL.AND.IDTM(ICOMP,N).GE.IOFFDT(ICOMP,N))THEN
          ZFRAC(N)=0.0
          IDTM(ICOMP,N)=IDTM(ICOMP,N)+1
        ELSEIF(zell.GE.SOFFRL.AND.
     &         IDTM(ICOMP,N).LT.IOFFDT(ICOMP,N))THEN
          ZFRAC(N)=SYSPER(ICOMP,N)
          IDTM(ICOMP,N)=IDTM(ICOMP,N)+1
        ELSE
          ZFRAC(N)=SYSPER(ICOMP,N)
          IDTM(ICOMP,N)=0
        ENDIF
        IF(IFT.EQ.1)then
          IF(IDFST(ICOMP,N).EQ.1)WRITE(outs,803)N,ZFRAC(N)
          IF(IDFST(ICOMP,N).EQ.2)WRITE(outs,804)N,ZFRAC(N)
          IF(IDFST(ICOMP,N).EQ.3)WRITE(outs,805)N,ZFRAC(N)
          IF(IDFST(ICOMP,N).EQ.4)WRITE(outs,806)N,ZFRAC(N)
          IF(IDFST(ICOMP,N).EQ.5)WRITE(outs,807)N,ZFRAC(N)
  803     FORMAT(' Light zone ',I1,' ON/OFF computed df, ZFRAC=',F5.2)
  804     FORMAT(' Light zone ',I1,' ON/OFF user def df, ZFRAC=',F5.2)
  805     FORMAT(' Light zone ',I1,' ON/OFF ext sensor, ZFRAC=',F5.2)
  806     FORMAT(' Light zone ',I1,' ON/OFF coupling, ZFRAC=',F5.2)
  807     FORMAT(' Light zone ',I1,' ON/OFF daylight coef, ZFRAC=',F5.2)
          call edisp(itu,outs)
        endif

C If in coupling mode, write out the controlled fraction so that at the
C next timestep the the user can edit the IES xforms file to match.
        IF(IDFST(ICOMP,N).EQ.4)then
          write(outs,'(a,F9.2)') 
     &      'COUPLING ADJUSTMENT next timestep after:',btimef
          call edisp(iuout,outs)
          call edisp(6,outs)
          WRITE(outs,'(a,i3,a,i1,a,F5.2)') 'In zone ',icomp,
     &      ' & light zone ',N,'edit xform for fixture faction to ',
     &      ZFRAC(N)
          call edisp(iuout,outs)
          call edisp(6,outs)
        ENDIF

C Update corresponding SHOCC lighting.
        call updatelighting(icomp,n,zelum(n),btimep,btimef)

C Case of Step-Up/Down with chosen sensors/lighting systems.
      ELSEIF(ICGCFL(ICOMP,N) .EQ. 2)THEN
        ZFRAC(N)=0.
        zell=zelum(N)+0.5*SETPT(ICOMP,N)
        IF(zell.GE.SOFFRL.AND.IDTM(ICOMP,N).GE.IOFFDT(ICOMP,N)) THEN
          ZFRAC(N)=0.0
          IDTM(ICOMP,N)=IDTM(ICOMP,N)+1
        ELSEIF(zell.GE.SOFFRL.AND.IDTM(ICOMP,N).LT.IOFFDT(ICOMP,N)) THEN
          ZFRAC(N)=SMEOUT(ICOMP,N)*SYSPER(ICOMP,N)
          IDTM(ICOMP,N)=IDTM(ICOMP,N)+1
        ELSEIF(zell.LT.SOFFRL.AND.zell.GE.SETPT(ICOMP,N))THEN
          ZFRAC(N)=SMEOUT(ICOMP,N)*SYSPER(ICOMP,N)
          IDTM(ICOMP,N)=0
        ELSEIF(zell.LT.SETPT(ICOMP,N))THEN
          ZFRAC(N)=SYSPER(ICOMP,N)
          IDTM(ICOMP,N)=0
        ENDIF
        IF(IFT.EQ.1)then
          IF(IDFST(ICOMP,N).EQ.1)WRITE(outs,808)N,ZFRAC(N)
          IF(IDFST(ICOMP,N).EQ.2)WRITE(outs,809)N,ZFRAC(N)
          IF(IDFST(ICOMP,N).EQ.3)WRITE(outs,810)N,ZFRAC(N)
          IF(IDFST(ICOMP,N).EQ.4)WRITE(outs,811)N,ZFRAC(N)
          IF(IDFST(ICOMP,N).EQ.5)WRITE(outs,812)N,ZFRAC(N)        
  808   FORMAT(' Light zone ',I1,' STEP u/d ESP-r calc df, ZFRAC=',F5.2)
  809   FORMAT(' Light zone ',I1,' STEP u/d User def df, ZFRAC=',F5.2)
  810   FORMAT(' Light zone ',I1,' STEP u/d Ext sensor, ZFRAC=',F5.2)
  811   FORMAT(' Light zone ',I1,' STEP u/d Coupling, ZFRAC=',F5.2)
  812   FORMAT(' Light zone ',I1,' STEP u/d Daylight coef, ZFRAC=',F5.2)
          call edisp(itu,outs)
        endif

C Update corresponding SHOCC lighting.
        call updatelighting(icomp,n,zelum(n),btimep,btimef)

C Case of proproportional dimming control with chosen sensors/lighting
C systems.
      ELSEIF(ICGCFL(ICOMP,N).EQ.3)THEN
          ZFRAC(N)=0.0
          DFRAC = 0.0
          zell=zelum(N)+SMLOUT(ICOMP,N)*SETPT(ICOMP,N)
        IF(zell.GE.SOFFRL.AND.IDTM(ICOMP,N).GE.IOFFDT(ICOMP,N))THEN
          ZFRAC(N)=0.0
          DFRAC = 0.0
          IDTM(ICOMP,N)=IDTM(ICOMP,N)+1
        ELSEIF(zell.GE.SOFFRL.AND.IDTM(ICOMP,N).LT.IOFFDT(ICOMP,N))THEN
          ZFRAC(N)=SMEOUT(ICOMP,N)*SYSPER(ICOMP,N)
          DFRAC = SMLOUT(ICOMP,N)
          IDTM(ICOMP,N)=IDTM(ICOMP,N)+1
        ELSEIF(zell.LT.SOFFRL.AND.zell.GE.SETPT(ICOMP,N))THEN
          ZFRAC(N)=SMEOUT(ICOMP,N)*SYSPER(ICOMP,N)
          DFRAC = SMLOUT(ICOMP,N)
          IDTM(ICOMP,N)=0
        ELSEIF(zell.LT.SETPT(ICOMP,N).AND.zell.GE.
     &        (1.-SMLOUT(ICOMP,N))*SETPT(ICOMP,N))THEN
          ZFRAC(N)=SMEOUT(ICOMP,N)*SYSPER(ICOMP,N)
          DFRAC = SMLOUT(ICOMP,N)
          IDTM(ICOMP,N)=0
        ELSE
         XXX=((1.-SMEOUT(ICOMP,N))*SMLOUT(ICOMP,N))/(1.-SMLOUT(ICOMP,N))
         YYY=SMEOUT(ICOMP,N)-XXX
         ZFRAC(N)=(((1.-SMEOUT(ICOMP,N))/(1.-SMLOUT(ICOMP,N)))*
     &   ((SETPT(ICOMP,N)-zelum(N))/SETPT(ICOMP,N))+YYY)*SYSPER(ICOMP,N)
         DFRAC = (SETPT(ICOMP,N)-zelum(N))/SETPT(ICOMP,N)
         IDTM(ICOMP,N)=0
        ENDIF
        IF(IFT.EQ.1)then
          IF(IDFST(ICOMP,N).EQ.1)WRITE(outs,813)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.2)WRITE(outs,814)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.3)WRITE(outs,815)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.4)WRITE(outs,816)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.5)WRITE(outs,817)N,ZFRAC(N),DFRAC  
  813     FORMAT(' Light zone ',I1,' Prop ESP-r calc df, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  814     FORMAT(' Light zone ',I1,' Prop User def df, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  815     FORMAT(' Light zone ',I1,' Prop Ext sensor, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  816     FORMAT(' Light zone ',I1,' Prop Coupling, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  817     FORMAT(' Light zone ',I1,' Prop Daylight coef, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
          call edisp(itu,outs)
        ENDIF

C Update corresponding SHOCC lighting.
        call updatelighting(icomp,n,zelum(n),btimep,btimef)

C Case of explicit proproportional dimming control with integral
C reset control algorithm.
C SPELEC  - sensor signal (lux) under artificial lighting
C           night time calibration;
C zelum  - sensor signal (lux) under daylight (time dependent).
C 
C For detailed explanation of control algorithms see:
C   Rubinstein F, Ward G and Verderberg R (1989) Improving the Performance of
C   Photo-Electrically Controled Lighting Systems, J. IES.
      ELSEIF(ICGCFL(ICOMP,N).EQ.31)THEN

C Redefine SOFFRL for explicit control.
          SOFFRL=SOFFLL(ICOMP,N)*SPELEC(ICOMP,N)

          ZZZ1 = (1.-SMLOUT(ICOMP,N))*SOFFRL
          YYY1 = (1.-SMLOUT(ICOMP,N))*SPELEC(ICOMP,N)

          ZFRAC(N)=0.0
          DFRAC = 0.0
          zell=zelum(N)
        IF(zell.GE.ZZZ1.AND.IDTM(ICOMP,N).GE.IOFFDT(ICOMP,N))THEN
          ZFRAC(N)=0.0
          DFRAC = 0.0
          IDTM(ICOMP,N)=IDTM(ICOMP,N)+1
        ELSEIF(zell.GE.ZZZ1.AND.IDTM(ICOMP,N).LT.IOFFDT(ICOMP,N))THEN
          ZFRAC(N)=SMEOUT(ICOMP,N)*SYSPER(ICOMP,N)
          DFRAC = SMLOUT(ICOMP,N)
          IDTM(ICOMP,N)=IDTM(ICOMP,N)+1
        ELSEIF(zell.LT.ZZZ1.AND.zell.GT.YYY1)THEN
          ZFRAC(N)=SMEOUT(ICOMP,N)*SYSPER(ICOMP,N)
          DFRAC = SMLOUT(ICOMP,N)
          IDTM(ICOMP,N)=0
        ELSE
          DFRAC=(1.- zelum(N)/SPELEC(ICOMP,N))
          if(DFRAC.lt.SMLOUT(ICOMP,N))DFRAC = SMLOUT(ICOMP,N)
         XXX=((1.-SMEOUT(ICOMP,N))*SMLOUT(ICOMP,N))/(1.-SMLOUT(ICOMP,N))
         YYY=SMEOUT(ICOMP,N)-XXX
         ZFRAC(N)=(((1.-SMEOUT(ICOMP,N))/(1.-SMLOUT(ICOMP,N)))*DFRAC
     &   + YYY)*SYSPER(ICOMP,N)
          IDTM(ICOMP,N)=0
        ENDIF
        IF(IFT.EQ.1)then
          IF(IDFST(ICOMP,N).EQ.1)WRITE(outs,913)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.2)WRITE(outs,914)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.3)WRITE(outs,915)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.4)WRITE(outs,916)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.5)WRITE(outs,917)N,ZFRAC(N),DFRAC  
  913     FORMAT(' Light zone ',I1,' Prop ESP-r calc df, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  914     FORMAT(' Light zone ',I1,' Prop User def df, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  915     FORMAT(' Light zone ',I1,' Prop Ext sensor, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  916     FORMAT(' Light zone ',I1,' Prop Coupling, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  917     FORMAT(' Light zone ',I1,' Prop Daylight coef, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
          call edisp(itu,outs)
        ENDIF

C Update corresponding SHOCC lighting.
        call updatelighting(icomp,n,zelum(n),btimep,btimef)

C Case of explicit proproportional dimming control with closed-loop
C algorithm.

C SPELEC - sensor signal (lux) under artificial lighting
C          night time calibration.
C zelum  - sensor signal (lux) under daylight (time dependend).
C SLOPEM - linear control response slope (-).
      ELSEIF(ICGCFL(ICOMP,N).EQ.32)THEN

C Redefine SOFFRL for explicit control.
          SOFFRL=SOFFLL(ICOMP,N)*SPELEC(ICOMP,N)

          ZZ1 = SMLOUT(ICOMP,N)*(1. - SLOPEM(ICOMP,N)*SOFFRL)
          ZZ2 = (ZZ1 - 1.)/SLOPEM(ICOMP,N) + SOFFRL

          YY1 = SMLOUT(ICOMP,N)*(1. - SLOPEM(ICOMP,N)*SPELEC(ICOMP,N))
          YY2 = (ZZ1 - 1.)/SLOPEM(ICOMP,N) + SPELEC(ICOMP,N)

          ZFRAC(N)=0.0
          DFRAC = 0.0
          zell=zelum(N)
        IF(zell.GT.ZZ2.AND.IDTM(ICOMP,N).GE.IOFFDT(ICOMP,N))THEN
          ZFRAC(N)=0.0
          DFRAC = 0.0
          IDTM(ICOMP,N)=IDTM(ICOMP,N)+1
        ELSEIF(zell.GT.ZZ2.AND.IDTM(ICOMP,N).LT.IOFFDT(ICOMP,N))THEN
          ZFRAC(N)=SMEOUT(ICOMP,N)*SYSPER(ICOMP,N)
          DFRAC = SMLOUT(ICOMP,N)
          IDTM(ICOMP,N)=IDTM(ICOMP,N)+1
        ELSEIF(zell.LT.ZZ2.AND.zell.GT.YY2)THEN
          ZFRAC(N)=SMEOUT(ICOMP,N)*SYSPER(ICOMP,N)
          DFRAC = SMLOUT(ICOMP,N)
          IDTM(ICOMP,N)=0
        ELSE
          XX1 =  1.+ SLOPEM(ICOMP,N)*(zelum(N)-SPELEC(ICOMP,N))
          XX2 =  1. - SLOPEM(ICOMP,N)*SPELEC(ICOMP,N)
          DFRAC =  XX1/XX2
          if(DFRAC.lt.SMLOUT(ICOMP,N))DFRAC=SMLOUT(ICOMP,N)
         XXX=((1.-SMEOUT(ICOMP,N))*SMLOUT(ICOMP,N))/(1.-SMLOUT(ICOMP,N))
         YYY=SMEOUT(ICOMP,N)-XXX
         ZFRAC(N)=(((1.-SMEOUT(ICOMP,N))/(1.-SMLOUT(ICOMP,N)))*DFRAC
     &   + YYY)*SYSPER(ICOMP,N)
          IDTM(ICOMP,N)=0
        ENDIF
        IF(IFT.EQ.1)then
          IF(IDFST(ICOMP,N).EQ.1)WRITE(outs,713)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.2)WRITE(outs,714)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.3)WRITE(outs,715)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.4)WRITE(outs,708)N,ZFRAC(N),DFRAC
          IF(IDFST(ICOMP,N).EQ.5)WRITE(outs,709)N,ZFRAC(N),DFRAC  
  713     FORMAT(' Light zone ',I1,' Prop ESP-r calc df, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  714     FORMAT(' Light zone ',I1,' Prop User def df, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  715     FORMAT(' Light zone ',I1,' Prop Ext sensor, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  708     FORMAT(' Light zone ',I1,' Prop Coupling, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
  709     FORMAT(' Light zone ',I1,' Prop Daylight coef, ZFRAC=',F5.2,
     &    ' DFRAC=',F5.2)
          call edisp(itu,outs)
        ENDIF

C Update corresponding SHOCC lighting.
        call updatelighting(icomp,n,zelum(n),btimep,btimef)

C Hunt's manual switching.
C Assume that the on/off periods set in the operation file for the
C matching casual gain represent occupancy times. If there are two
C periods (e.g. 8h00 to 12h00 and 14h00 to 18h00) then the probability
C will be first calculated at 8h00 and then recalculated at 14h00.
C FIRSTL() is the illumination when the probability was last calculated.
C With multiple sensors, use the ratio of the daylight factor
C for the brightest and dimmest sensors to judge uniformity.
C If the ratio is outside the range of 0.33<r<3.0 then occupants
C will tend to judge the spatial character of the lighting as
C high contrast and will keep the lights on. If only one sensor,
C assume all is ok.
      ELSEIF(ICGCFL(ICOMP,N).EQ.4)THEN
        iuni=0
        IMAXX=1
        IMINN=1
        SMAXX=0.0
        SMINN=1000.0
        DO 886 I=1,NDF(ICOMP,N)
          IF(SUMDF(ICOMP,N,I).GT.SMAXX)THEN
            SMAXX=SUMDF(ICOMP,N,I)
            IMAXX=I
          ENDIF
          IF(SUMDF(ICOMP,N,I).LE.SMINN)THEN
            SMINN=SUMDF(ICOMP,N,I)
            IMINN=I
          ENDIF
  886   CONTINUE
        ratdf=1.0
        IF(SUMDF(ICOMP,N,IMINN).GT.1E-10)THEN
          ratdf=SUMDF(ICOMP,N,IMAXX)/SUMDF(ICOMP,N,IMINN)
        ENDIF
        IF((ratdf.LE.0.33).OR.(ratdf.GT.3.))iuni=1
        elum1=selum(N,IMINN)

C Read in the current zone operation file so that common/P3N will
C be correct for access by HUNTA. HUNTA returns ACTIVE=1. if there
C is a matching casual gain to control at the current timestep.
        IUNITA=IFIL+22
        CALL EROPER(IFT,ITU,IUNITA,ICOMP,IER)
        CALL HUNTA(ICOMP,IHRF,ACTIVE)

C If first hour of the day, calculate a new random number for
C the current zone.
C Following code works on a Sun workstation:
        IF(IHRF.EQ.1)THEN
          seed=(idyp*24+ihrp)+icomp
          RANDN(ICOMP)=RAND(INT(seed))
        ENDIF

C If ACTIVE = 0. reset the probability, FRAC and FIRSTL to 0. to
C force later calculation of probability.
        call eclose(ACTIVE,0.00,0.01,close)
        IF(close)THEN
          prob=0.
          ZFRAC(N)=0.
          FIRSTL(ICOMP)=0.

C Generate description for use in the trace.
          IF(IFT.EQ.1)then
            WRITE(outs,901)N,ZFRAC(N)
  901      FORMAT(' Light zone ',I1,' Hunt: no occup! ZFRAC=(',F5.2,')')
            call edisp(itu,outs)
          endif
        ELSE

C Check limits, if elum1 low lighting must be turned on
C (reset Firstl), if high lighting must be turned off (do
C not reset FIRSTL).
          IF(elum1.LT.16.0)THEN
            ZFRAC(N)=SYSPER(ICOMP,N)
            prob=1.
            FIRSTL(ICOMP)=elum1
            IF(IFT.EQ.1)then
               WRITE(outs,902)N,ZFRAC(N),elum1
  902          FORMAT(' Light zone ',I1,' Hunt: ZFRAC=',F4.2,
     &             ' elum low(',F6.0,'): force lighting on')
               call edisp(itu,outs)
            endif
          ELSEIF(elum1.GT.1510.0)THEN
            ZFRAC(N)=0.0
            prob=0.0
            IF(IFT.EQ.1)then
                WRITE(outs,903)N,ZFRAC(N),elum1
  903          FORMAT(' Light zone ',I1,' Hunt: ZFRAC=',F4.2,
     &         ' elum high(',F6.0,'): force lighting off')
               call edisp(itu,outs)
            endif
          ELSEIF(iuni.EQ.1)THEN
            ZFRAC(N)=SYSPER(ICOMP,N)
            prob=1.0
            IF(IFT.EQ.1)then
               WRITE(outs,908)N,ZFRAC(N),elum1
  908          FORMAT(' Light zone ',I1,' Hunt: Z FRAC=',F4.2,
     &         ' elum(',F6.0,'): uniformity fails, lighting on')
               call edisp(itu,outs)
            endif
          ELSE

C If FIRSTL <> 0. then prob previously calculated. If elum > FIRSTL
C keep switched on, if less then test again.
            call eclose(FIRSTL(ICOMP),0.00,0.001,close)
            IF(FIRSTL(ICOMP).GT.0.)THEN
              IF(elum1.GT.FIRSTL(ICOMP))THEN
                ZFRAC(N)=SYSPER(ICOMP,N)
                prob=1.
                IF(IFT.EQ.1)then
                  WRITE(outs,904)N,ZFRAC(N),elum1,FIRSTL(ICOMP)
  904             FORMAT(' Light zone ',I1,' Hunt: ZFRAC=',F4.2,
     &            ' elum(',F6.0,')>inital(',F6.0,
     &            '): keep lighting switched on')
                  call edisp(itu,outs)
                endif
              ELSEIF(elum1.LE.FIRSTL(ICOMP))THEN

C Test again using previous random seed.
                Xel=log10(elum1)
                prob=-0.0175+(1.0361/(1.+exp(4.0835*(Xel-1.8223))))
                if(prob.lt.0.)prob=0.
                if(prob.gt.1.)prob=1.
                IF(prob.gt.RANDN(ICOMP))THEN
                  ZFRAC(N)=SYSPER(ICOMP,N)
                ELSE
                  ZFRAC(N)=0.
                ENDIF
                IF(IFT.EQ.1)then
                  WRITE(outs,905)N,ZFRAC(N),elum1,FIRSTL(ICOMP),prob,
     &                          RANDN(ICOMP)
  905             FORMAT(' Light zone ',I1,' Hunt: ZFRAC=',F4.2,
     &            ' elum(',F6.0,')<inital(',F6.0,
     &            ') test again: prob(',F4.2,') r(',F4.2,')')
                  call edisp(itu,outs)
                endif
              ENDIF
            ELSEIF(close)THEN

C Occupants arriving, calculate probability and save FIRSTL.
              Xel=log10(elum1)
              prob=-0.0175+(1.0361/(1.+exp(4.0835*(Xel-1.8223))))
              if(prob.lt.0.)prob=0.
              if(prob.gt.1.)prob=1.
              IF(prob.gt.RANDN(ICOMP))THEN
                ZFRAC(N)=SYSPER(ICOMP,N)
                FIRSTL(ICOMP)=elum1
              ELSE
                ZFRAC(N)=0.
              ENDIF
                IF(IFT.EQ.1)then
                  WRITE(outs,906)N,ZFRAC(N),elum1,prob,RANDN(ICOMP)
  906             FORMAT(' Light zone ',I1,' Hunt: ZFRAC=',F4.2,
     &            ' Occup arrival first ','elum(',F6.0,
     &            ') prob(',F4.2,') r(',F4.2,')')
                  call edisp(itu,outs)
                endif
            ENDIF
          ENDIF
        ENDIF

C Update corresponding SHOCC lighting.
        call updatelighting(icomp,n,zelum(n),btimep,btimef)

      ENDIF
      FRAC=FRAC+ZFRAC(N)
    9 CONTINUE

C Remove transfer file.
      call EFDELET(IUNIT,ISTAT)

C Trace output.
      IF(IFT.EQ.1)then
        write(outs,910)FRAC
  910   format('Fraction of lighting "ON" FRAC=',F4.2)
        call edisp(itu,outs)
      endif

C Trace output?
      IF(ITC.LE.0.OR.NSINC.LT.ITC)goto 999
      IF(ITRACE(12).EQ.0.OR.NSINC.GT.ITCF.OR.
     &   IZNTRC(ICOMP).NE.1)goto 999

C Trace solar incident on interior transparent surfaces if DIFT
C is non-zero.
      NC=NCONST(ICOMP)
      DO 52 I=1,NC
        IF(ITMCFL(ICOMP,I).GT.0.and.DIFT(ICOMP,I).gt.0.0)THEN        
          write(outs,'(a,i3,a,f5.1)') 'equiv source illum on surface ',
     &      I,'=',DIFT(ICOMP,I)
          call edisp(itu,outs)
          write(outs,911)I,ICOMP,IBOFOT(ICOMP,I)
 911      format('the TMC surface ',I3,' on zone',I3,' has IBOFOT=',I1)  
          call edisp(itu,outs)
        ENDIF 
 52   CONTINUE  

C Description for irradiation/illumination data.
      write(outs,99971)QDIR,QDIF
99971 format(' Ext: Dir.nor.rad. W/m^2(',F7.1,') Dif.hor.rad.(',
     &       F7.1,')')
      call edisp(itu,outs)
      write(outs,99972)QDIR*suneff,esky
99972 format('    : sun light lux(',F7.1,') sky light lux(',F7.1,')')
      call edisp(itu,outs)
      write(outs,99973)suneff,skyeff
99973 format('    : sun/dir efficacy (',F7.2,') sky/dif efficacy (',
     &       F7.2,')')
      call edisp(itu,outs)
      write(outs,99974)SAZI,SALT
99974 format('    : solar azi deg(',F6.1,') solar alt deg(',F6.1,')')
      call edisp(itu,outs)

  999 return
      end

C ******************* CALCILUM ***************************
C Returns the average illuminance of lighting zone N for casual
C gain control.

      SUBROUTINE CALCILUM(icomp,N,selum,zelum,alt,skyeff,suneff,esun,
     &                    esky,IFT)
#include "building.h"
#include "model.h"
#include "geometry.h"

      common/filep/ifil
      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/SHOUT/ICOUT
      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      COMMON/PERS/ISD1,ISM1,ISD2,ISM2,ISDS,ISDF,NTSTEP
      COMMON/SUNPOS/SAZI,SALT,ISUNUP
      COMMON/TRACE/ITCF,ITRACE(MTRACE),IZNTRC(MCOM),ITU

C Zone casual gain control data.
      COMMON/CGCIN2N/NCGTC(MCOM,MDTY),NLITZ(MCOM),IDFST(MCOM,MLCOM),
     & CGX(MCOM,MLCOM,MDF),CGY(MCOM,MLCOM,MDF),CGH(MCOM,MLCOM,MDF),
     & UX(MCOM,MLCOM,MDF),UY(MCOM,MLCOM,MDF),UH(MCOM,MLCOM,MDF),
     & SETPT(MCOM,MLCOM),SYSPER(MCOM,MLCOM),SOFFLL(MCOM,MLCOM),
     & IOFFDT(MCOM,MLCOM),SMLOUT(MCOM,MLCOM),SMEOUT(MCOM,MLCOM)
      COMMON/CGCIN3/ICGCS(MCOM,MDTY),ICGCF(MCOM,MDTY),
     & ICGCFL(MCOM,MLCOM),SPELEC(MCOM,MLCOM),SLOPEM(MCOM,MLCOM)

C Daylight coefficient common block data.
      COMMON/DCOEF/NDCFP,NDCFS(MDCFP),DCOEF(MDCFP,MDCFS,MDCFV),
     & IDCFID(MCOM,MLCOM,MDF)

C Daylight factor data (see dayfac.f).
      COMMON/DAYF/NDF(MCOM,MLCOM),DFDAT(MCOM,MLCOM,MGT,MDF),
     & AZIJL(MCOM,MGT),SUMDF(MCOM,MLCOM,MDF),NUDWIN(MCOM)

      COMMON/ADJC2/DIRT(MCOM,MS),DIFT(MCOM,MS),AIRT(MCOM,MS)
      COMMON/D7/T1,T2,P1,P2

C TMC data.
      COMMON/PRECTC/ITMCFL(MCOM,MS),TMCT(MCOM,MTMC,5),
     &       TMCA(MCOM,MTMC,ME,5),TMCREF(MCOM,MTMC),TVTR(MCOM,MTMC)

C TMC blind control flag.
      COMMON/TMCB4/IBOFOT(MCOM,MS),NBPONT(MCOM,MS),ECRAT(MCOM,MS)

C Construction information.
      COMMON/PREC9/NCONST(MCOM),NELTS(MCOM,MS),NGAPS(MCOM,MS),
     &             NPGAP(MCOM,MS,MGP)

      common/sus/issny

C Illuminance database.
      common/ILDB/NDBENT,DBILLA(200,5),DBILLS(200,MLCOM,MDF)

      DIMENSION selum(MLCOM,MDF),alt(8),SENTMP(MLCOM,MDF)
      DIMENSION QDFTMP(MLCOM,MDF),Yb(MLCOM,MDF)
      DIMENSION Aa(MLCOM,MDF),Ab(MLCOM,MDF),Ac(MLCOM,MDF),Ya(MLCOM,MDF)
      DIMENSION zelum(MLCOM)
      DIMENSION XPL(MV),YPL(MV),ZPL(MV)
      CHARACTER*25 CSTR
      character outs*124,doit*124,outstr*124
      character*72 LTMPFL,LTMPFLA,OUTCMD
      character e2redit*8

      logical close,closea,closeb,closec,closed,closee,closef,closeg
      logical closeh,closei,instart,neglum,XST,bInStartup,nog
      REAL PI,TODEG,RAD

      iunit=ifil+1
      e2redit='NONE'
      PI = 4.0 * ATAN(1.0)
      TODEG=180/PI
      RAD=PI/180.0

C If pre-processor or user-defined daylight factor then read geometry data.
      nog=.false.
      do  1 iu=1,NLITZ(icomp)
        IF((IDFST(ICOMP,iu).EQ.1).or.(IDFST(ICOMP,iu).eq.2))nog=.true.
  1   continue
      if(nog)then
        call georead(iunit,lgeom(icomp),icomp,0,itu,ier)
      endif

      IF(IDFST(ICOMP,N).EQ.3)THEN

C If external sensor & surface (overload CGX) is exterior
C calculate lux. If CGY (overloaded)=1. then calculate
C vertical unobstructed illum otherwise use horizontal illum.
        J=INT(CGX(ICOMP,N,1))
        IF(J.EQ.0)J=1
        CALL SURADJ(ICOMP,J,IES,T,IZC,ISC,ICN,CSTR)
        IF(IES.EQ.0)THEN
          call eclose(CGY(ICOMP,N,1),1.00,0.01,close)
          IF(close)THEN

C Establish solar data.
            CALL MZSINT(ICOMP,QDIR,QDIF)
            CALL VRTILM(SPAZI(ICOMP,J),QDIR,QDIF,SRADDO,SKYDIF)
            zelum(N)=SRADDO*suneff+SKYDIF*skyeff
            if(IFT.EQ.1)then
              write(outs,105)N,CGY(ICOMP,N,1),J,esun,esky,zelum(N),
     &              SETPT(ICOMP,N)
  105         format(I6,F5.1,' N/A ','  N/A  ',I5,'   N/A   ','  N/A ',
     &                       F8.1,F8.1,F7.1,F6.0,'   N/A   ')
              call edisp(itu,outs)
            endif
          ELSE
            zelum(N)=esun+esky
            if(IFT.EQ.1)then
              write(outs,106)N,CGY(ICOMP,N,1),J,esun,esky,zelum(N),
     &              SETPT(ICOMP,N)
  106         format(I6,F5.1,' N/A ','  N/A  ',I5,'   N/A   ','  N/A ',
     &                       F8.1,F8.1,F7.1,F6.0,'   N/A   ')
              call edisp(itu,outs)
            endif
          ENDIF
        ELSE
          Ev=0.
          IF(IFT.EQ.1)call edisp(itu,
     &      '  not external surface, sensor not allowed!')
        ENDIF
        
      ELSEIF(IDFST(ICOMP,N).EQ.5)THEN

C Daylight coefficient method. Test if there is any diffuse 
C irradiation and if solar altitude is positive. If not, set
C photocell values to zero.
        call eclose(QDIF,0.00,0.01,closea)
        if(SALT.LT.0.) closea=.true.
        if(closea) then
          suma=0.
          DO 41 I=1,NDF(ICOMP,N)
            selum(N,I)=0.

C Trace output.
            if(IFT.EQ.1)then
              write(outs,111)N,I,selum(N,I),SETPT(ICOMP,N)
              call edisp(itu,outs)
            endif
   41     CONTINUE
        else

C Calculate reference horizontal illuminance.
          call REFILL(QDIF,QDIR,SALT,SAZI,IDYP,refell)

C Loop through all TMCs in the zone and determine blind and
C shuter control status. 
C If IBOFOT()=1 then blind/shutter is ON and use second stage
C daylight coefficients or if IBOFOT()=2 then linear interpolate
C optical properties, e.g. electrochromic glazing.
          NC=NCONST(ICOMP)
          DO 51 I=1,NC
            IF(ITMCFL(ICOMP,I).GT.0)THEN
              IF(IBOFOT(ICOMP,I).EQ.0)RATIO=1.0
              IF(IBOFOT(ICOMP,I).EQ.1)RATIO=0.0
              IF(IBOFOT(ICOMP,I).EQ.2)RATIO=ECRAT(ICOMP,I)
            ENDIF
51        CONTINUE

C Loop all sensors for this lighting zone.
          suma=0.
          DO 40 J=1,NDF(ICOMP,N)

C Find daylight coefficient set associated with current sensor. 
            IDCFP=IDCFID(ICOMP,N,J)

C Define starting angle for test of sky element comprising sun.
            ang0=4.0*PI

C Loop through all 145 sky patches and calculate delta illuminance.
            sill = 0.0
            do 30 npatch = 1, 145
              dill=0.0

C Calculate sky patch reference luminance.
              call PATLUM(QDIF,QDIR,SALT,SAZI,IDYP,npatch,plv,ang)

C Find sky element with minimum angle between sun position and element.
C This identifies the sky element number for the calculation of the direct
C illuminance contribution.
              if(ang.lt.ang0)then
                nelem = npatch
                ang0 = ang
              endif

C Calculate luminance (cd/m^2) for given sky patch.
              plum=plv*esky/refell

C If inconsistency in irradiance data (plum < 0) set it to 0.
              if(plum.le.0.0)then
                if((npatch.ge.1).and.(npatch.le.30))then
                  nzone = 1
                elseif((npatch.ge.31).and.(npatch.le.60))then
                  nzone = 2
                elseif((npatch.ge.61).and.(npatch.le.84))then
                  nzone = 3
                elseif((npatch.ge.85).and.(npatch.le.108))then
                  nzone = 4
                elseif((npatch.ge.109).and.(npatch.le.126))then
                  nzone = 5
                elseif((npatch.ge.127).and.(npatch.le.138))then
                  nzone = 6
                elseif((npatch.ge.139).and.(npatch.le.144))then
                  nzone = 7
                elseif(npatch.eq.145)then
                  nzone = 8
                endif
                neglum = .true.

C Use aproximate solution assuming uniform sky luminance distribution
C to estimate sky patch luminance.
                plum = esky/(2.0*PI*sin(alt(nzone)*rad))
              else
                neglum = .false.
              endif

C Calculate delta illuminance at sensor from given sky patch.
C Test for blind or linear interpolation and use appropriate
C daylight coefficient.
              CALL ECLOSE(RATIO,1.00,0.0001,closea)
              IF(CLOSEA)THEN
                DMIN=0.0
                DMAX=DCOEF(IDCFP,1,npatch)
              ELSE
                DMIN=DCOEF(IDCFP,2,npatch)
                DMAX=DCOEF(IDCFP,1,npatch)
              ENDIF

              DD1=DMIN + (DMAX - DMIN)*RATIO 
              dill=DD1*plum*(2.0*PI/145.0)
              sill = sill + dill

30          continue

C Warn about negative luminance.
            if(neglum)then
              write (outs,'(a)') 'Sky patch luminance < = 0 occured!'
              call edisp(iuout,outs)
              write (outs,'(a,I3,a,f6.1,a,f6.1,a,f4.1)') ' Day ',IDYP,
     &        ' Direct ',QDIR,' Diffuse ',QDIF,' Sun altitude ',SALT
              call edisp(iuout,outs)
            endif

C Check if direct sun, if yes then calculate its contribution
C to sensor illuminance.
            call eclose(QDIR,0.00,0.01,closea)
            if(closea)then
              dircon=0.0
            else
              CALL ECLOSE(RATIO,1.00,0.0001,closea)
              IF(CLOSEA)THEN
                DMIN=0.0
                DMAX=DCOEF(IDCFP,1,nelem)
              ELSE
                DMIN=DCOEF(IDCFP,2,nelem)
                DMAX=DCOEF(IDCFP,1,nelem)
              ENDIF
              DD2=DMIN + (DMAX - DMIN)*RATIO
              dircon=DD2*QDIR*suneff
            endif 

C Define sensed illumination at sensor.
            subT = sill + dircon
            suma=suma+subT

C Define selum for trace output.
            selum(N,J)=subT

C Trace output.
            if(IFT.EQ.1)then
              write(outs,113)N,J,dircon,sill,selum(N,J),SETPT(ICOMP,N)
  113         format(I6,I5,' N/A ','  N/A  ','   N/A   ','  N/A ',
     &               F8.1,F8.1,F7.1,F6.0,'   N/A   ')
              call edisp(itu,outs)
            endif
40        CONTINUE
        endif

C Average illumination for this light zone (in case of more than one
C sensor in lighting zone).
        zelum(N)=suma/float(NDF(ICOMP,N))

      ELSEIF(IDFST(ICOMP,N).EQ.4)THEN

C Time varying illuminances calculated by Radiance. Test if there
C is any diffuse irradiation and if solar altitude is positive. Do
C not carry out lighting simulation if not, and if in pre-calculation
C period (via call to bInStartup).
        call eclose(QDIF,0.00,0.01,closea)
        instart=.false.
        if (bInStartup())then
          instart=.true.
        endif
        if(closea.OR.(SALT.LT.0.).OR.(IDYP.LT.ISDS).OR.instart)then
          suma=0.
          DO 21 I=1,NDF(ICOMP,N)
            selum(N,I)=0.

C Trace output.
            if(IFT.EQ.1)then
              write(outs,111)N,I,selum(N,I),SETPT(ICOMP,N)
              call edisp(itu,outs)
            endif
   21     CONTINUE
        else

C Call RADIANCE only once for first lighting zone
C as all internal illuminances are now known.
          IF (N.EQ.1) then

C Transfer solar azimuth into Radiance coordinate system
C where South is 0 degrees and azimuth is +ve towards West.
            RSAZI=SAZI-180.
            
C Loop through all TMCs in the zone and determine blind and
C shuter control status. 
C If IBOFOT()=1, then blind/shutter is ON.
            NC=NCONST(ICOMP)
            IBLIND = 0
            RATIO = 0.0
            DO 31 I=1,NC
              IF(ITMCFL(ICOMP,I).GT.0)THEN
                IF(IBOFOT(ICOMP,I).GT.0)THEN
                  IBLIND = 1
                  RATIO = 1.0
                ENDIF
              ENDIF
31          CONTINUE

C Check if sensor values are already in illuminance database and set accuracies.
            ILLDBM=0
            AZIACC=5.
            QDRACC=5.

            write (66,*) ' ***  Solar data for this timestep *** '
            CALL DAYCLK(IDYP,BTIMEF,66)
            write (66,'(4f9.1,i3)') SALT,RSAZI,QDIR,QDIF,IBLIND

C Calculate differences. Blind state MUST match.
C Check for no direct solar (effectively use daylight factor).
            if (QDIR.lt.QDRACC) then
              NMATCH=0
              do 311 ILZ=1,MLCOM
                do 3111 IDF=1,MDF
                  SENTMP(ILZ,IDF)=0.0
                  QDFTMP(ILZ,IDF)=0.0
 3111           continue
 311          continue
              do 310 IDB=1,NDBENT
                if ((DBILLA(IDB,3).lt.QDRACC).and.
     &              (IBLIND.eq.nint(DBILLA(IDB,5)))) then

                  write (66,*) ' Found a diffuse match: '
            write (66,*) 'Dir: ',DBILLA(IDB,3),' Blind: ',DBILLA(IDB,5)

                  NMATCH=NMATCH+1
                  do 321 ILZ=1,MLCOM
                    do 3211 IDF=1,MDF
                      SENTMP(ILZ,IDF)=SENTMP(ILZ,IDF)+
     &                                DBILLS(IDB,ILZ,IDF)
                      QDFTMP(ILZ,IDF)=QDFTMP(ILZ,IDF)+DBILLA(IDB,4)
 3211               continue
 321              continue
                endif
 310          continue
              if (NMATCH.gt.1) then
                ILLDBM=1
                do 322 ILZ=1,MLCOM
                  do 3221 IDF=1,MDF
                    if (QDFTMP(ILZ,IDF).gt.0.0) then

C Assume that the diffuse regresses to a straight line: y=mx
C The SENTMP and QDFTMP both need divided by NMATCH hence NMATCH
C cancels if y=mx assumption holds.
                      SENTMP(ILZ,IDF)=SENTMP(ILZ,IDF)*QDIF
                      selum(ILZ,IDF)=SENTMP(ILZ,IDF)/QDFTMP(ILZ,IDF)
                    else
                      selum(ILZ,IDF)=0.
                    endif
 3221             continue
 322            continue

                write (66,*) 'Estimated illuminances:'
                write (66,'(9(f7.1,1x))') (selum(1,IDFT),IDFT=1,MDF)
                write (66,'(9(f7.1,1x))') (selum(2,IDFT),IDFT=1,MDF)
                write (66,'(9(f7.1,1x))') (selum(3,IDFT),IDFT=1,MDF)
                write (66,'(9(f7.1,1x))') (selum(4,IDFT),IDFT=1,MDF)
              endif
            else

C Check for similar sun position.
              NMATCH=0
              do 411 ILZ=1,MLCOM
                do 4111 IDF=1,MDF
                  SENTMP(ILZ,IDF)=0.0
                  QDFTMP(ILZ,IDF)=0.0
 4111           continue
 411          continue
              do 320 IDB=1,NDBENT
                SOLDIF=esind(90.0-DBILLA(IDB,1))*esind(90.0-SALT)*
     &                 ecosd(DBILLA(IDB,2)-RSAZI)
                SOLDIF=SOLDIF+ecosd(90.0-DBILLA(IDB,1))*ecosd(90.0-SALT)
                SOLDIF=acos(SOLDIF)*180.0/PI
                if ((SOLDIF.lt.AZIACC).and.
     &              (IBLIND.eq.nint(DBILLA(IDB,5)))) then
                  NMATCH=NMATCH+1

            write (66,*) ' Found a direct/ diffuse match: '
            write (66,*) 'Dir: ',DBILLA(IDB,3),' Dif: ',DBILLA(IDB,4),
     &                    ' Blind: ',DBILLA(IDB,5)

                  do 421 ILZ=1,MLCOM
                    do 4211 IDF=1,MDF

C Generate coefficients of least squares arrays.
                      if (ILZ.eq.1.and.IDF.eq.1) then
                        Aa(ILZ,IDF)=DBILLA(IDB,3)**2
                        Ab(ILZ,IDF)=DBILLA(IDB,3)*DBILLA(IDB,4)
                        Ac(ILZ,IDF)=DBILLA(IDB,4)**2
                        Ya(ILZ,IDF)=DBILLA(IDB,3)*DBILLS(IDB,ILZ,IDF)
                        Yb(ILZ,IDF)=DBILLA(IDB,4)*DBILLS(IDB,ILZ,IDF)
                      else
                        Aa(ILZ,IDF)=Aa(ILZ,IDF)+DBILLA(IDB,3)**2
                        Ab(ILZ,IDF)=Ab(ILZ,IDF)+DBILLA(IDB,3)*
     &                                                    DBILLA(IDB,4)
                        Ac(ILZ,IDF)=Ac(ILZ,IDF)+DBILLA(IDB,4)**2
                        Ya(ILZ,IDF)=Ya(ILZ,IDF)+DBILLA(IDB,3)*
     &                                              DBILLS(IDB,ILZ,IDF)
                        Yb(ILZ,IDF)=Yb(ILZ,IDF)+DBILLA(IDB,4)*
     &                                              DBILLS(IDB,ILZ,IDF)
                      endif
 4211               continue
 421              continue
                endif
 320          continue
              if (NMATCH.gt.2) then
                ILLDBM=1
                do 422 ILZ=1,MLCOM
                  do 4221 IDF=1,MDF

C Calculate regression coefficients GRADQR and GRADQF.
                    GRADQF= Ya(ILZ,IDF) * (Ab(ILZ,IDF) / Aa(ILZ,IDF))
                    GRADQF= GRADQF - Yb(ILZ,IDF)
                    GRADQF= GRADQF / 
     &                 ( ( (Ab(ILZ,IDF)**2) /Aa(ILZ,IDF)) - Ac(ILZ,IDF))
                    GRADQR= Ya(ILZ,IDF) - (Ab(ILZ,IDF) * GRADQF)
                    GRADQR= GRADQR / Aa(ILZ,IDF)
                    selum(ILZ,IDF)=(GRADQR*QDIR) + (GRADQF*QDIF)
 4221             continue
 422            continue

                write (66,*) 'Estimated illuminances:'
                write (66,'(9(f7.1,1x))') (selum(1,IDFT),IDFT=1,MDF)
                write (66,'(9(f7.1,1x))') (selum(2,IDFT),IDFT=1,MDF)
                write (66,'(9(f7.1,1x))') (selum(3,IDFT),IDFT=1,MDF)
                write (66,'(9(f7.1,1x))') (selum(4,IDFT),IDFT=1,MDF)
              endif
            endif

C Call e2r to calculate illuminances.
C << For coupling with casual gain control should write out control
C << state information to the transfer file.
            ILLDBM=0
            if (ILLDBM.eq.0) then

C If there are IES entities in the model, tell e2r to invoke editing
C of the IES transforms file.
              if(nbofies.gt.0)then
                e2redit='YES'
              else
                e2redit='NO'
              endif

C Create transfer file (format: .PID.tmp). LTMPFL is the file name (passed
C to e2r) and LTMPFLA is the absolute path/file name.
              call esppid(ipid)
              write(LTMPFLA,'(a,i7,a)') '.',ipid,'.dat'
              call st2file(LTMPFLA,LTMPFL)
              write (LTMPFLA,'(a,a,a)')upath(1:lnblnk(upath)),'/',
     &          LTMPFL(1:lnblnk(LTMPFL))
              INQUIRE (FILE=LTMPFLA(1:lnblnk(LTMPFLA)),EXIST=XST)
              if(XST)then
                call edisp(iuout,'Removing existing transfer file. ')
                call FPOPEN(IUNIT,ISTAT,1,1,LTMPFLA)
                call EFDELET(IUNIT,ISTAT)
              endif
              call FPOPEN(IUNIT,ISTAT,1,2,LTMPFLA)
              write (OUTCMD,'(4f9.1,i3,2a)') SALT,RSAZI,QDIR,QDIF,
     &          IBLIND,' ',e2redit
              write (IUNIT,'(a)') OUTCMD
              call ERPFREE(IUNIT,ISTAT)
              write(doit,'(3a,i3,3a)')'e2r -file ',
     &          LCFGF(:lnblnk(LCFGF)),' -purpose Coupling -zone ',
     &          ICOMP,' -act Calculate -actf ',LTMPFL(:lnblnk(LTMPFL)),
     &          ' -mode text'

C Debug.
C              write(6,'(3a,i3,3a)')'e2r -file ',
C     &          LCFGF(:lnblnk(LCFGF)),' -purpose Coupling -zone ',
C     &          ICOMP,' -act Calculate -actf ',LTMPFL(:lnblnk(LTMPFL)),
C     &          ' -mode text'
              write(outs,'(a,f8.3,a,i3,a)') 'Lighting simulation @ ',
     &          BTIMEF,' for zone',ICOMP,' commenced.'
              call edisp(icout,outs)
              call runit(doit,'-')

C Create new entry in illuminance database.
              NDBENT=NDBENT+1
              DBILLA(NDBENT,1)=SALT
              DBILLA(NDBENT,2)=RSAZI
              DBILLA(NDBENT,3)=QDIR
              DBILLA(NDBENT,4)=QDIF
              DBILLA(NDBENT,5)=float(IBLIND)

C Recover new illuminance data. Open transfer file.
              suma=0.
              CALL FPOPEN(IUNIT,ISTAT,1,1,LTMPFLA)
              if(ISTAT.lt.0)then
                IER=1
                write(outs,7) zname(icomp),N
    7           format(' Zone ',A,'light zone',I1,' time var illum err')
                call edisp(iuout,outs)
                close(ieout)
                CALL ERPFREE(ieout,ISTAT)
                CALL EPWAIT
                CALL EPAGEND
                stop
              endif

C Loop through till 'OUTCMD' string found.
 777          CALL STRIPC(IUNIT,OUTSTR,0,ND,1,'trn file line',IER)
              if (IER.ne.0) then
                write(outs,17) zname(icomp),N
   17           format(' Zone ',A,'light zone',I1,'trns file read err')
                call edisp(iuout,outs)
                close(ieout)
                CALL ERPFREE(ieout,ISTAT)
                CALL EPWAIT
                CALL EPAGEND
                stop
              endif
              if (OUTSTR(1:60).ne.OUTCMD(1:60)) goto 777

C Read sensor data.
              DO 19 I=1,NDF(ICOMP,N)
                sumJ=0.
                CALL STRIPC(IUNIT,OUTSTR,0,ND,1,'t var illum',IER)
                if (IER.ne.0) then
                 write(outs,'(2a)')' Fatal error: Radiance simulation',
     &              ' produced no data.'
                  call edisp(iuout,outs)
                  close(ieout)
                  CALL ERPFREE(ieout,ISTAT)
                  CALL EPWAIT
                  CALL EPAGEND
                  stop
                endif
                K=0
                si=0.0
                CALL EGETWR(OUTSTR,K,si,0.,0.,'-','sen illum',IER)
                subT=si
                suma=suma+subT
                selum(N,I)=subT

C Copy information to illuminance DB.
                DBILLS(NDBENT,N,I)=int(subT)

C Trace output.
                if(IFT.EQ.1)then
                  write(outs,111)N,I,selum(N,I),SETPT(ICOMP,N)
  111             format(I6,I5,' N/A ','  N/A  ','   N/A   ','  N/A ',
     &               '   N/A  ','   N/A  ',F7.1,F6.0,'   N/A   ')
                  call edisp(itu,outs)
                endif
   19         CONTINUE

C Close transfer file.
              call ERPFREE(IUNIT,ISTAT)

              write (66,*) 'Calculated illuminances:'
              write (66,'(9(f7.1,1x))') (selum(1,IDFT),IDFT=1,MDF)
              write (66,'(9(f7.1,1x))') (selum(2,IDFT),IDFT=1,MDF)
              write (66,'(9(f7.1,1x))') (selum(3,IDFT),IDFT=1,MDF)
              write (66,'(9(f7.1,1x))') (selum(4,IDFT),IDFT=1,MDF)
            else
              call edisp(iuout,'Using information from illum database.')
            endif
          ENDIF
        endif

C Average illumination for this light zone.
        zelum(N)=suma/float(NDF(ICOMP,N))
      ELSE

C IDST is somthing other than 3/4/5.
C Loop through each sensor point, calculate the illumination due
C to each external window and summate. Each window will
C have a different solar contribution depending on whether or
C not it can see the sun. To regain information on the associated
C surfaces, loop as in dayfac.f. Note suma is total sensed lux
C in zone for use in averaging, sumJ is the total sensed lux at
C each sensor.
        suma=0.
        DO 10 I=1,NDF(ICOMP,N)
          sumJ=0.
          XP=CGX(ICOMP,N,I)
          YP=CGY(ICOMP,N,I)
          ZP=CGH(ICOMP,N,I)

C Internal sensor, determine the 'other' zone and connection
C information.   
          DO 20 J=1,NSUR
            CALL SURADJ(ICOMP,J,IES,T,IZC,ISC,ICN,CSTR)
            NA=NVER(J)
            DO 201 K=1,NA
              XPL(K)=X(JVN(J,K))
              YPL(K)=Y(JVN(J,K))
              ZPL(K)=Z(JVN(J,K))
  201       CONTINUE

C Continue only if this surface is vertical, is TMC, has 4 vertices
C and is connected to either the exterior or another zone.  
C Keep track of total windows in zone with nt. Exclude skylights.
            IF(ITMCFL(ICOMP,J).GT.0)THEN
              call eclose(SPELV(ICOMP,J),0.00,0.70,close)
              IF(close)THEN
                IF((NVER(J).NE.4).and.(IDFST(ICOMP,N).ne.2))then
                  WRITE(ITU,843) zname(icomp),J
  843             format('Zone ',A,'TMC surf ',I3,' not 4 vertices!')

                ENDIF

C Test if the TMC is rectangular.
                call eclose(XPL(1),XPL(2),0.01,closea)
                call eclose(XPL(1),XPL(3),0.01,closeb)
                call eclose(XPL(1),XPL(4),0.01,closec)
                call eclose(YPL(1),YPL(2),0.01,closed)
                call eclose(YPL(1),YPL(3),0.01,closee)
                call eclose(YPL(1),YPL(4),0.01,closef)
                call eclose(ZPL(1),ZPL(2),0.01,closeg)
                call eclose(ZPL(1),ZPL(3),0.01,closeh)
                call eclose(ZPL(1),ZPL(4),0.01,closei)
                IF(closea.OR.closeb.OR.closec)THEN
                  IF(closed.OR.closee.OR.closef)THEN
                    IF(closeg.OR.closeh.OR.closei)THEN
                      CONTINUE
                    ELSE
                      WRITE(ITU,842)ICOMP,J
                    ENDIF      
                  ELSE
                    WRITE(ITU,842)ICOMP,J
                  ENDIF
                ELSE
                  if(IDFST(ICOMP,N).ne.2)then
                    WRITE(ITU,842) zname(icomp),J
  842             format('Zone ',A,'TMC',I3,' may not be rectangular!')
                  endif
                ENDIF 

                IF((IES.EQ.0).OR.(IES.EQ.3))THEN
                  IF((IES.EQ.3).AND.(IDFST(ICOMP,N).NE.2))THEN

C Calculate equivalent source illuminance via interior window.
C DIFT is in Watts/m^2 (similar to the outside radiation). DIRT is
C disregarded because we do not trace its direction in this zone.
C Note that the attenuation of visible transmittance through
C the glass is roughly accounted for in solar algorithm and DFDAT
C includes the maintenance factor.
                    diftzn=DIFT(IZC,ISC)*skyeff
                    subT=diftzn*DFDAT(ICOMP,N,J,I)
                    if(IFT.EQ.1)then
                      write(outs,103)I,J,
     &                      DFDAT(ICOMP,N,J,I)*100.0,diftzn,
     &                      subT,SETPT(ICOMP,N)
  103                 format(I5,' via internal ',I5,F9.2,'      ',
     &                     F8.1,F7.1,F6.0)
                      call edisp(itu,outs)
                    endif
                    
                  ELSEIF((IES.EQ.3).AND.(IDFST(ICOMP,N).EQ.2 ))THEN

C If user specified DF for internal window then calculate the
C same way as for external but do not check for sun direct illuminance.
                    suncon=0.
                    CALL DWINVT(ICOMP,J,V,RATIO)
                    skycon=esky*DFDAT(ICOMP,N,J,I)*V
                    subT=skycon
                    if(IFT.EQ.1)then
                      write(outs,107)N,I,J,
     &                 DFDAT(ICOMP,N,J,I)*V*100.0,V,
     &                 suncon,skycon,subT,SETPT(ICOMP,N),Ev
  107                 format(I6,I5,'  no ',I7,F9.2,F6.2,F8.1,F8.1,
     &                       F7.1,F6.0,F9.0)
                      call edisp(itu,outs)
                    endif
                  ELSE

C Determine the angles of view to window from the sensor point,
C convert to degrees between 0 & 360, and compare with the
C solar angle.
                    XX1=XPL(1)
                    DO 555 M=1,4
                      XPLTEM=XPL(M)
                      call eclose(XPL(1),XPLTEM,0.01,close)
                      IF(.NOT.close)XX2=XPL(M)
  555               CONTINUE
                    YY1=YPL(1)
                    DO 556 M=1,4
                      YPLTEM=YPL(M)
                      call eclose(YPL(1),YPLTEM,0.01,close)
                      IF(.NOT.close)YY2=YPL(M)
  556               CONTINUE
                    ZZ1=ZPL(1)              
                    DO 557 M=1,4
                      ZPLTEM=ZPL(M)
                      call eclose(ZPL(1),ZPLTEM,0.01,close)
                      IF(.NOT.close)ZZ2=ZPL(M)
  557               CONTINUE

C Only valid for a rectangle.
                    WIDTH=SQRT((XX1-XX2)**2+(YY1-YY2)**2)
                    HIGH=ABS(ZZ1-ZZ2)

C P1, P2, T1 & T2 as returned from ANGS are the angles (radians) from the 
C normal to the surface with the window (from the inside). Convert azimuth to
C degrees (clockwise from North) equivalent to SAZI by subtracting
C from the azimuth angle of the surface in question and keeping within the
C range 0 to 360 degrees.
                    CALL ANGS(NA,XPL,YPL,ZPL,WIDTH,HIGH,XP,YP,ZP)
                    PP1=SPAZI(ICOMP,J)-(P1*TODEG)
                    IF(PP1.LT.0.)PP1=360.+PP1
                    IF(PP1.GE.360.)PP1=PP1-360.
                    PP2=SPAZI(ICOMP,J)-(P2*TODEG)
                    IF(PP2.LT.0.)PP2=360.+PP2
                    IF(PP2.GE.360.)PP2=PP2-360.
                    TT1=T1*TODEG
                    TT2=T2*TODEG

C Calculate the vertical unobstructed illumination (Ev) at plane of
C the glazing and get current visible transmission of glazing.
                    CALL VRTILM(SPAZI(ICOMP,J),QDIR,QDIF,SRADDO,SKYDIF)
                    Ev=SRADDO*suneff+SKYDIF*skyeff
                    IF((PP1.LT.SAZI).AND.(PP2.GT.SAZI).AND.
     &                (TT1.LT.SALT).AND.(TT2.GT.SALT))THEN
                      CALL DWINVT(ICOMP,J,V,RATIO)
                      suncon=esun*V
                      skycon=esky*DFDAT(ICOMP,N,J,I)*V
                      subT=suncon+skycon
                      if(IFT.EQ.1)then
                        write(outs,101)N,I,J,
     &                   DFDAT(ICOMP,N,J,I)*V*100.0,V,
     &                    suncon,skycon,subT,SETPT(ICOMP,N),Ev
  101                   format(I6,I5,' yes ',I7,F9.2,F6.2,F8.1,F8.1,
     &                       F7.1,F6.0,F9.0)
                        call edisp(itu,outs)
                      endif
                    ELSE
                      suncon=0.
                      CALL DWINVT(ICOMP,J,V,RATIO)
                      skycon=esky*DFDAT(ICOMP,N,J,I)*V
                      subT=skycon
                      if(IFT.EQ.1)then
                        write(outs,102)N,I,J,
     &                   DFDAT(ICOMP,N,J,I)*V*100.0,V,
     &                   suncon,skycon,subT,SETPT(ICOMP,N),Ev
  102                   format(I6,I5,'  no ',I7,F9.2,F6.2,F8.1,F8.1,
     &                       F7.1,F6.0,F9.0)
                        call edisp(itu,outs)
                      endif
                    ENDIF
                  ENDIF
                  suma=suma+subT
                  sumJ=sumJ+subT
                ENDIF
              ELSE
                if(IDFST(ICOMP,N).ne.2)then
                  WRITE(ITU,844) zname(icomp),J
  844         format('Zone ',A,'TMC',I3,'is not vertical-no dir illum!')
                endif
                suncon=0.
                CALL DWINVT(ICOMP,J,V,RATIO)
                skycon=esky*DFDAT(ICOMP,N,J,I)*V
                subT=skycon
                if(IFT.EQ.1)then
                  write(outs,104)N,I,J,
     &              DFDAT(ICOMP,N,J,I)*V*100.0,V,
     &              suncon,skycon,subT,SETPT(ICOMP,N),Ev
  104             format(I6,I5,'  no ',I7,F9.2,F6.2,F8.1,F8.1,
     &                  F7.1,F6.0,F9.0)
                  call edisp(itu,outs)
                endif
                suma=suma+subT
                sumJ=sumJ+subT
              ENDIF
            ENDIF
   20     CONTINUE

C Sensed illumination at given sensor.
          selum(N,I)=sumJ
   10   CONTINUE

C Average illumination at lighting zone.
        zelum(N)=suma/float(NDF(ICOMP,N)) 
      ENDIF

      RETURN
      END

C ******************** DWINVT ****************************
C Returns the visible transmittance of glazing at the
C future timestep. IZONE is the zone, ISURF the surface and
C VT the returned visible transmittance.
C Check to see if blind/shutter control is enabled for this zone;
C if so, see if it is active for the current glazing and substitute
C the visible transmittance.

      SUBROUTINE DWINVT(IZONE,ISURF,VT,BLIND)
#include "building.h"

      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      COMMON/TMCB1/IBCMT(MCOM,MTMC)
      COMMON/TMCB2/NBCTMC(MCOM,MTMC),IBCST(MCOM,MTMC),
     &IBCFT(MCOM,MTMC),IBCSUR(MCOM,MTMC)
      COMMON/PRECTC/ITMCFL(MCOM,MS),TMCT(MCOM,MTMC,5),
     &       TMCA(MCOM,MTMC,ME,5),TMCREF(MCOM,MTMC),TVTR(MCOM,MTMC)
      COMMON/PRECT2/TMCT2(MCOM,MTMC,5),TMCA2(MCOM,MTMC,ME,5),
     &              TVTR2(MCOM,MTMC)

C TMC blind control flag.
      COMMON/TMCB4/IBOFOT(MCOM,MS),NBPONT(MCOM,MS),ECRAT(MCOM,MS)

C Test if blind/shutter control is on for this TMC.
      BLIND = 0.0
      ITVTR=ITMCFL(IZONE,ISURF)
      IF(ITVTR.GT.0.AND.IBCMT(IZONE,ITVTR).EQ.1)THEN

C TMC blind/shutter is in place; test if active.
C NP is number of control periods for this TMC type.
        NP=NBCTMC(IZONE,ITVTR)
        DO 43 KK=1,NP
          IT1=IBCST(IZONE,ITVTR)
          IT2=IBCFT(IZONE,ITVTR)
          IF(IHRF.GT.IT1.AND.IHRF.LE.IT2.AND.
     &      IBOFOT(IZONE,ISURF).EQ.1)THEN
            BLIND = 1.0
            VT=TVTR2(IZONE,ITVTR)
          ELSE
            BLIND = 0.0
            VT=TVTR(IZONE,ITVTR)
          ENDIF
   43   CONTINUE
      ELSE
        BLIND = 0.0
        VT=TVTR(IZONE,ITVTR)
      ENDIF

      RETURN
      END

C ******************** HUNTA ********************
C Calculate if current hour has scheduled casual gains, which are
C controlled via a casual gain control file.
C If ACTIVE = 1. then there is a potential casual gain/occupants.
C if ACTIVE = 0. then no casual gain/occupants.

      SUBROUTINE HUNTA(IZONE,IH,ACTIVE)
#include "building.h"
#include "schedule.h"

      common/simtim/ihrp,ihrf,idyp,idyf,idwp,idwf,nsinc,its,idynow
      common/caleni/nbdaytype,nbcaldays(MDTY),icalender(365)
      INTEGER NBDAYTYPE,NBCALDAYS,ICALENDER,IDTY

      COMMON/CGCIN2N/NCGTC(MCOM,MDTY),NLITZ(MCOM),IDFST(MCOM,MLCOM),
     & CGX(MCOM,MLCOM,MDF),CGY(MCOM,MLCOM,MDF),CGH(MCOM,MLCOM,MDF),
     & UX(MCOM,MLCOM,MDF),UY(MCOM,MLCOM,MDF),UH(MCOM,MLCOM,MDF),
     & SETPT(MCOM,MLCOM),SYSPER(MCOM,MLCOM),SOFFLL(MCOM,MLCOM),
     & IOFFDT(MCOM,MLCOM),SMLOUT(MCOM,MLCOM),SMEOUT(MCOM,MLCOM)

      ACTIVE=0.

C Determine which day of the week (6=Sat, 7=Sun etc.), or othwer if defined
C via the calendar function. Set the gain to control (IGC) to that requested
C in the casual gain control file. Assumes 3 day types are standard but check
C status of NBDAYTYPE and use ICALENDER if required. If matching gain(s)
C set ACTIVE=1.0.
      IF(NBDAYTYPE.LE.3)THEN
        IDTY=1
        IF(IDWP.EQ.6)IDTY=2
        IF(IDWP.EQ.7)IDTY=3
      ELSE
        IDTY=ICALENDER(IDYP)
      ENDIF
      IGC=NCGTC(IZONE,IDTY)
      IF(NCAS(IDTY).GE.1)THEN
        DO 30 ICNN=1,NCAS(IDTY)
          IF(ICGT(IDTY,ICNN).EQ.IGC)THEN
            IF(IH.GT.ICGS(IDTY,ICNN).AND.IH.LE.ICGF(IDTY,ICNN))ACTIVE=1.
          ENDIF
   30   CONTINUE
      ENDIF
      RETURN
      END

C ******************** VRTILM ********************
C Calculate the illumination (Ev) on a unobstructed vertical
C surface with an azimuth (AZMANG) in degrees clockwise from
C North for use with Lynes' daylighting methods. Use
C method in shadow.f to determine if surface is self shaded. 
C << Note not yet integrated into lighting calculations. >>
      SUBROUTINE VRTILM(AZMANG,QDIR,QDIF,SRADDO,SKYDIF)
      COMMON/SUNPOS/SAZI,SALT,ISUNUP

      PI = 4.0 * ATAN(1.0)
      R=PI/180.

C Determine surface-solar azimuth angle & cosine of angle of
C incidence. Surface is self-shaded if CAI is less than or
C equal to zero. SRADDO is the direct radiation incident on
C the surface.
      PSAZI=ABS(AZMANG-SAZI)
      CAI=(COS(SALT*R))*(SIN(90.*R))*(COS(PSAZI*R))+
     &    (SIN(SALT*R))*(COS(90.*R))
      IF(CAI.LE.0.)THEN
        SRADDO=0.
      ELSE
        SRADDO=QDIR*CAI
      ENDIF

C Sky diffuse incident on surface.
      SKYDIF=QDIF*0.5*(1.0+COS(90.*R))

      return
      end

C ******************** PATLUM ********************
C Calculate reference luminance of a sky patch.
C QDIF -   diffuse irradiance.
C QDIR -   direct normal irradiance.
C SALT -   solar altitude.
C SAZI -   solar azimith.
C IDYP -   year day number.
C npatch - number of the sky patch.
C plv  -   reference sky luminance - returned value.

      SUBROUTINE PATLUM(QDIF,QDIR,SALT,SAZI,IDYP,npatch,plv,ang)
      COMMON/SHOUT/ICOUT

      dimension npz(8),alt(8),azim(8)
      DIMENSION  A1(8),A2(8),A3(8),A4(8)
      DIMENSION  B1(8),B2(8),B3(8),B4(8)
      DIMENSION  C1(8),C2(8),C3(8),C4(8)
      DIMENSION  D1(8),D2(8),D3(8),D4(8)
      DIMENSION  E1(8),E2(8),E3(8),E4(8)

C Definitions for daylight coefficient patch calculation.
      data npz/30,60,84,108,126,138,144,145/
      data alt/6.,18.,30.,42.,54.,66.,78.,90./
      data azim/12.,12.,15.,15.,20.,30.,60.,0./

C Start definition of Perez all-weather sky luminance model coefficients
C (Solar Energy V50(3), pp. 235-245, 1993).
      DATA A1/1.3525,-1.2219,-1.1,-.5484,-.6,-1.0156,-1.0,-1.05/
      DATA A2/-.2567,-.773,-.2515,-.6654,-.3566,-.367,.0211,.0289/
      DATA A3/-.269,1.4148,.8952,-.2672,-2.5,1.0078,.5025,.426/
      DATA A4/-1.4366,1.1016,.0156,.7117,2.325,1.4051,-.5119,.359/

      DATA B1/-.767,-.2054,.2782,.7234,.2937,.2875,-.3,-.325/
      DATA B2/.0007,.0367,-.1812,-.6219,.0496,-.5328,.1922,.1156/
      DATA B3/1.2734,-3.9128,-4.5,-5.6812,-5.6812,-3.85,.7023,.7781/
      DATA B4/-.1233,.9156,1.1766,2.6297,1.8415,3.375,-1.6317,.0025/

      DATA C1/2.8,6.975,24.7219,33.3389,21.,14.,19.,31.0625/
      DATA C2/.6004,.1774,-13.0812,-18.3,-4.7656,-.9999,-5.,-14.5/
      DATA C3/1.2375,6.4477,-37.7,-62.25,-21.5906,-7.1406,1.2438,
     &       -46.1148/
      DATA C4/1.,-.1239,34.8438,52.0781,7.2492,7.5469,-1.9094,55.375/

      DATA D1/1.8734,-1.5798,-5.,-3.5,-3.5,-3.4,-4.,-7.2312/
      DATA D2/.6297,-.5081,1.5218,.0016,-.1554,-.1078,.025,.405/
      DATA D3/.9738,-1.7812,3.9229,1.1477,1.4062,-1.075,.3844,13.35/
      DATA D4/.2809,.108,-2.6204,.1062,.3988,1.5702,.2656,.6234/

      DATA E1/.0356,.2624,-.0156,.4659,.0032,-.0672,1.0468,1.5/
      DATA E2/-.1246,.0672,.1597,-.3296,.0766,.4016,-.3788,-.6426/
      DATA E3/-.5718,-.219,.4199,-.0876,-.0656,.3017,-2.4517,1.8564/
      DATA E4/.9938,-.4285,-.5562,-.0329,-.1294,-.4844,1.4656,.5636/

      PI = 4.0 * ATAN(1.0)
      RAD=PI/180.
      ZENN=(90.-SALT)*rad
      GONN=1367.*(1.+0.033*COS(0.017214*IDYP))
      CONN1=1./0.001572
      CONN2=(COS(ZENN))**2+0.0031465
      OPTM=CONN1*(SQRT(CONN2)-COS(ZENN))

C Sky brightness.
      DELTT=QDIF*OPTM/GONN

C Sky clearness.
      EPSS=((QDIF+QDIR)/QDIF+1.041*ZENN**3)/(1.+1.041*ZENN**3)

C Sky clearness bin (1-8).
      IF(EPSS.LT.1.)THEN
        IE=1
        call edisp(icout,'Sky clearness < 1.0 setting to 1.0')
        ELSEIF((EPSS.GE.1.).AND.(EPSS.LT.1.065))THEN
        IE=1
        ELSEIF((EPSS.GE.1.065).AND.(EPSS.LT.1.23))THEN
        IE=2
        ELSEIF((EPSS.GE.1.23).AND.(EPSS.LT.1.50))THEN
        IE=3
        ELSEIF((EPSS.GE.1.50).AND.(EPSS.LT.1.95))THEN
        IE=4
        ELSEIF((EPSS.GE.1.95).AND.(EPSS.LT.2.80))THEN
        IE=5
        ELSEIF((EPSS.GE.2.80).AND.(EPSS.LT.4.50))THEN
        IE=6
        ELSEIF((EPSS.GE.4.50).AND.(EPSS.LT.6.20))THEN
        IE=7
        ELSEIF(EPSS.GE.6.20)THEN 
        IE=8
      ENDIF

C Sky luminance coefficients.
      a=a1(ie)+a2(ie)*zenn+deltt*(a3(ie)+a4(ie)*zenn)
      b=b1(ie)+b2(ie)*zenn+deltt*(b3(ie)+b4(ie)*zenn)
      if(ie.eq.1)then
        c=exp((deltt*(c1(ie)+c2(ie)*zenn))**c3(ie))-c4(ie)
        d=-exp(deltt*(d1(ie)+d2(ie)*zenn))+d3(ie)+deltt*d4(ie)
      else
        c=c1(ie)+c2(ie)*zenn+deltt*(c3(ie)+c4(ie)*zenn)
        d=d1(ie)+d2(ie)*zenn+deltt*(d3(ie)+d4(ie)*zenn)
      endif
      e=e1(ie)+e2(ie)*zenn+deltt*(e3(ie)+e4(ie)*zenn)

C Sun position unit vector x, y, z coordinates.
C Transfer sun azimuth angle (clockwise from North) into sky
C patch azimuth angle (anticlockwise from East).
      if((sazi.gt.0.0).and.(sazi.lt.90.0))then
        aazim = 90.0 - sazi
      else
        aazim = 450.0 - sazi
      endif

      x3 = ecosd(aazim)*cos(salt*RAD)
      y3 = esind(aazim)*cos(salt*RAD)
      z3 = sin(salt*RAD)

C Origin coordinates for calculation of angle between sun
C and a given sky patch.
      x2=0.0
      y2=0.0
      z2=0.0

C Determie sky patch level (of 8 levels).
      if((npatch.ge.1).and.(npatch.le.30))then
        nzone = 1
      elseif((npatch.ge.31).and.(npatch.le.60))then
        nzone = 2
      elseif((npatch.ge.61).and.(npatch.le.84))then
        nzone = 3
      elseif((npatch.ge.85).and.(npatch.le.108))then
        nzone = 4
      elseif((npatch.ge.109).and.(npatch.le.126))then
        nzone = 5
      elseif((npatch.ge.127).and.(npatch.le.138))then
        nzone = 6
      elseif((npatch.ge.139).and.(npatch.le.144))then
        nzone = 7
      elseif(npatch.eq.145)then
        nzone = 8
      endif

C Calculate x, y, z coordinates for unit position vector of given
C sky patch.
      if(nzone.eq.8) then
         x1 = 0.0
         y1 = 0.0
         z1 = 1.0
      elseif(nzone.eq.1) then
         aazim = azim(nzone)*(npatch - 1)
         x1 = ecosd(aazim)*cos(alt(nzone)*RAD)
         y1 = esind(aazim)*cos(alt(nzone)*RAD)
         z1 = sin(alt(nzone)*RAD)
      else
         aazim = azim(nzone)*((npatch - npz(nzone-1)) - 1)
         x1 = ecosd(aazim)*cos(alt(nzone)*RAD)
         y1 = esind(aazim)*cos(alt(nzone)*RAD)
         z1 = sin(alt(nzone)*RAD)
      endif

C Find angle between given sky patch and sun position
C and convert to radians.
      call ang3vtx(x1,y1,z1,x2,y2,z2,x3,y3,z3,ang)
      ang = ang*rad

C Calculate reference luminance and zenith angle for given sky patch.
      zelm = (90.0 - alt(nzone))*rad
      plv=(1.+a*exp(b/cos(zelm)))*(1.+c*exp(d*ang)+e*(cos(ang))**2)
      return
      end

C ******************** REFILL ********************
C Calculate reference illuminance for model normalisation.
C QDIF -  diffuse irradiance.
C QDIR -  direct normal irradiance.
C SALT -  solar altitude.
C SAZI -  solar azimith.
C IDYP -  year day number.
C refell - reference sky illuminance.

      SUBROUTINE REFILL(QDIF,QDIR,SALT,SAZI,IDYP,refell)
      COMMON/SHOUT/ICOUT

      dimension npz(8),alt(8),azim(8)
      DIMENSION  A1(8),A2(8),A3(8),A4(8)
      DIMENSION  B1(8),B2(8),B3(8),B4(8)
      DIMENSION  C1(8),C2(8),C3(8),C4(8)
      DIMENSION  D1(8),D2(8),D3(8),D4(8)
      DIMENSION  E1(8),E2(8),E3(8),E4(8)

C Definitions for daylight coefficient patch calculation.
      data npz/30,60,84,108,126,138,144,145/
      data alt/6.,18.,30.,42.,54.,66.,78.,90./
      data azim/12.,12.,15.,15.,20.,30.,60.,0./

C Start definition of Perez all-weather sky luminance model coefficients
C (Solar Energy, V50(3), pp. 235-245, 1993).
      DATA A1/1.3525,-1.2219,-1.1,-.5484,-.6,-1.0156,-1.0,-1.05/
      DATA A2/-.2567,-.773,-.2515,-.6654,-.3566,-.367,.0211,.0289/
      DATA A3/-.269,1.4148,.8952,-.2672,-2.5,1.0078,.5025,.426/
      DATA A4/-1.4366,1.1016,.0156,.7117,2.325,1.4051,-.5119,.359/

      DATA B1/-.767,-.2054,.2782,.7234,.2937,.2875,-.3,-.325/
      DATA B2/.0007,.0367,-.1812,-.6219,.0496,-.5328,.1922,.1156/
      DATA B3/1.2734,-3.9128,-4.5,-5.6812,-5.6812,-3.85,.7023,.7781/
      DATA B4/-.1233,.9156,1.1766,2.6297,1.8415,3.375,-1.6317,.0025/

      DATA C1/2.8,6.975,24.7219,33.3389,21.,14.,19.,31.0625/
      DATA C2/.6004,.1774,-13.0812,-18.3,-4.7656,-.9999,-5.,-14.5/
      DATA C3/1.2375,6.4477,-37.7,-62.25,-21.5906,-7.1406,1.2438,
     &       -46.1148/
      DATA C4/1.,-.1239,34.8438,52.0781,7.2492,7.5469,-1.9094,55.375/

      DATA D1/1.8734,-1.5798,-5.,-3.5,-3.5,-3.4,-4.,-7.2312/
      DATA D2/.6297,-.5081,1.5218,.0016,-.1554,-.1078,.025,.405/
      DATA D3/.9738,-1.7812,3.9229,1.1477,1.4062,-1.075,.3844,13.35/
      DATA D4/.2809,.108,-2.6204,.1062,.3988,1.5702,.2656,.6234/

      DATA E1/.0356,.2624,-.0156,.4659,.0032,-.0672,1.0468,1.5/
      DATA E2/-.1246,.0672,.1597,-.3296,.0766,.4016,-.3788,-.6426/
      DATA E3/-.5718,-.219,.4199,-.0876,-.0656,.3017,-2.4517,1.8564/
      DATA E4/.9938,-.4285,-.5562,-.0329,-.1294,-.4844,1.4656,.5636/

      PI = 4.0 * ATAN(1.0)
      RAD=PI/180.
      ZENN=(90.-SALT)*rad
      GONN=1367.*(1.+0.033*COS(0.017214*IDYP))
      CONN1=1./0.001572
      CONN2=(COS(ZENN))**2+0.0031465
      OPTM=CONN1*(SQRT(CONN2)-COS(ZENN))

C Sky brightness.
      DELTT=QDIF*OPTM/GONN

C Sky clearness.
      EPSS=((QDIF+QDIR)/QDIF+1.041*ZENN**3)/(1.+1.041*ZENN**3)

C Sky clearness bin (1-8).
      IF(EPSS.LT.1.)THEN
        IE=1
        call edisp(icout,'Sky clearness < 1.0 setting to 1.0')
        ELSEIF((EPSS.GE.1.).AND.(EPSS.LT.1.065))THEN
        IE=1
        ELSEIF((EPSS.GE.1.065).AND.(EPSS.LT.1.23))THEN
        IE=2
        ELSEIF((EPSS.GE.1.23).AND.(EPSS.LT.1.50))THEN
        IE=3
        ELSEIF((EPSS.GE.1.50).AND.(EPSS.LT.1.95))THEN
        IE=4
        ELSEIF((EPSS.GE.1.95).AND.(EPSS.LT.2.80))THEN
        IE=5
        ELSEIF((EPSS.GE.2.80).AND.(EPSS.LT.4.50))THEN
        IE=6
        ELSEIF((EPSS.GE.4.50).AND.(EPSS.LT.6.20))THEN
        IE=7
        ELSEIF(EPSS.GE.6.20)THEN 
        IE=8
      ENDIF

C Sky luminaace coefficients.
      a=a1(ie)+a2(ie)*zenn+deltt*(a3(ie)+a4(ie)*zenn)
      b=b1(ie)+b2(ie)*zenn+deltt*(b3(ie)+b4(ie)*zenn)
      if(ie.eq.1)then
        c=exp((deltt*(c1(ie)+c2(ie)*zenn))**c3(ie))-c4(ie)
        d=-exp(deltt*(d1(ie)+d2(ie)*zenn))+d3(ie)+deltt*d4(ie)
      else
        c=c1(ie)+c2(ie)*zenn+deltt*(c3(ie)+c4(ie)*zenn)
        d=d1(ie)+d2(ie)*zenn+deltt*(d3(ie)+d4(ie)*zenn)
      endif
      e=e1(ie)+e2(ie)*zenn+deltt*(e3(ie)+e4(ie)*zenn)

C Calculate sun position unit vector x, y, z coordinates.
C Transfer sun azimuth angle (clockwise from North) into sky
C patch azimuth angle (anticlockwise from East).
      if((sazi.gt.0.0).and.(sazi.lt.90.0))then
        aazim = 90.0 - sazi
      else
        aazim = 450.0 - sazi
      endif

      x3 = ecosd(aazim)
      y3 = esind(aazim)
      z3 = tan(salt*RAD)

C Origin coordinates for calculation of angle between sun
C and given sky patch.
      x2=0.0
      y2=0.0
      z2=0.0
      
      refell = 0.0
      do 20 npatch = 1,145

C Find sky patch level (of 8 levels).
        if((npatch.ge.1).and.(npatch.le.30))then
          nzone = 1
        elseif((npatch.ge.31).and.(npatch.le.60))then
          nzone = 2
        elseif((npatch.ge.61).and.(npatch.le.84))then
          nzone = 3
        elseif((npatch.ge.85).and.(npatch.le.108))then
          nzone = 4
        elseif((npatch.ge.109).and.(npatch.le.126))then
          nzone = 5
        elseif((npatch.ge.127).and.(npatch.le.138))then
          nzone = 6
        elseif((npatch.ge.139).and.(npatch.le.144))then
          nzone = 7
        elseif(npatch.eq.145)then
          nzone = 8
        endif

C Calculate x, y, z coordinates for unit position vector of given
C sky patch.
        if(nzone.eq.8) then
           x1 = 0.0
           y1 = 0.0
           z1 = 1.0
        elseif(nzone.eq.1) then
           aazim = azim(nzone)*(npatch - 1)
           x1 = ecosd(aazim)
           y1 = esind(aazim)
           z1 = tan(alt(nzone)*RAD)
        else
           aazim = azim(nzone)*((npatch - npz(nzone-1)) - 1)
           x1 = ecosd(aazim)
           y1 = esind(aazim)
           z1 = tan(alt(nzone)*RAD)
        endif

C Find angle between given sky patch and sun position
C and convert to radians.
        call ang3vtx(x1,y1,z1,x2,y2,z2,x3,y3,z3,ang)
        ang = ang*rad

C Calculate reference luminance and zenith angle for given sky patch
        zelm = (90.0 - alt(nzone))*rad
        plv=(1.+a*exp(b/cos(zelm)))*(1.+c*exp(d*ang)+e*(cos(ang))**2)
      
C Calculate delta reference illuminance from npatch:
        refplv = plv*(2.*PI/145.0)*cos(zelm)
        refell = refell + refplv

20    continue

      return
      end

C ******************** LUMEFF ********************
C Calculate luminous efficacy of solar irradiance.
C QDIF  - diffuse irradiance.
C QDIR  - direct normal irradiance.
C SALT  - solar altitude.
C IDYP  - year day number.
C skyeff - luminous efficacy of diffuse irradiance.
C suneff - luminous efficacy of direct normal irradiance.

      SUBROUTINE LUMEFF(QDIF,QDIR,SALT,IDYP,skyeff,suneff)
      COMMON/SHOUT/ICOUT

C Definition of Perez model coefficients for luminous efficacy
C of solar irradiance and zenith luminance.
C (Solar Energy, V44(5), pp. 271-289, 1990)
      DIMENSION  ASUN(8),ASKY(8),CSUN(8),CSKY(8),DSUN(8),DSKY(8)

      logical closea

      DATA ASUN/57.20,98.99,109.83,110.34,106.36,107.19,105.75,101.18/
      DATA ASKY/97.24,107.22,104.97,102.39,100.71,106.42,141.88,152.23/
      DATA CSUN/-2.98,-1.21,-1.71,-1.99,-1.75,-1.51,-1.26,-1.10/
      DATA CSKY/12.00,0.59,-5.53,-13.95,-22.75,-36.15,-53.24,-45.27/
      DATA DSUN/117.12,12.38,-8.81,-4.56,-6.16,-26.73,-34.44,-8.29/
      DATA DSKY/-8.91,-3.95,-8.77,-13.90,-23.74,-28.83,-14.03,-7.98/

      PI = 4.0 * ATAN(1.0)
      RAD=PI/180.

C Check if any diffuse irradiance and if sun is up. If not, do not 
C calculate and set efficacy and zenith luminance to zero.
      call eclose(QDIF,0.00,0.01,closea)
      if(SALT.LT.0.)closea=.true.
      if(closea) then
        skyeff=0.
        suneff=0.     
      else

C Calculate efficacy of direct and diffuse irradiance 
C and zenith luminance.
        ZENN=(90.-SALT)*rad
        GONN=1367.*(1.+0.033*COS(0.017214*IDYP))
        CONN1=1./0.001572
        CONN2=(COS(ZENN))**2+0.0031465
        OPTM=CONN1*(SQRT(CONN2)-COS(ZENN))

C Sky brightness.
        DELTT=QDIF*OPTM/GONN

C Sky clearness.
        EPSS=((QDIF+QDIR)/QDIF+1.041*ZENN**3)/(1.+1.041*ZENN**3)

C Sky clearness bin (1-8).
        IF(EPSS.LT.1.)THEN
          IE=1
          call edisp(icout,'Sky clearness < 1.0 setting to 1.0')
          ELSEIF((EPSS.GE.1.).AND.(EPSS.LT.1.065))THEN
          IE=1
          ELSEIF((EPSS.GE.1.065).AND.(EPSS.LT.1.23))THEN
          IE=2
          ELSEIF((EPSS.GE.1.23).AND.(EPSS.LT.1.50))THEN
          IE=3
          ELSEIF((EPSS.GE.1.50).AND.(EPSS.LT.1.95))THEN
          IE=4
          ELSEIF((EPSS.GE.1.95).AND.(EPSS.LT.2.80))THEN
          IE=5
          ELSEIF((EPSS.GE.2.80).AND.(EPSS.LT.4.50))THEN
          IE=6
          ELSEIF((EPSS.GE.4.50).AND.(EPSS.LT.6.20))THEN
          IE=7
          ELSEIF(EPSS.GE.6.20)THEN 
          IE=8
        ENDIF

        call eclose(QDIR,0.00,0.01,closea)
        IF(closea)THEN
          suneff=0.
          skyeff=(ASKY(IE)+CSKY(IE)*COS(ZENN)+DSKY(IE)*LOG(DELTT))
        ELSE
          suneff=(ASUN(IE)+CSUN(IE)*EXP(5.73*ZENN-5.)+DSUN(IE)*DELTT)
          skyeff=(ASKY(IE)+CSKY(IE)*COS(ZENN)+DSKY(IE)*LOG(DELTT))
        ENDIF
      endif 
    
      return
      end
