C This file is part of the ESP-r system.
C Copyright Natural Resources Canada 1999
C Author: Ian Beausoleil-Morrison
C Revision Date: November 5, 1999

C ESP-r is free software.  You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation 
C (version 2 orlater).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.

C This file contains the following routines related to the BASESIMP
C foundation heat loss model:
C BSCOEFF      Controls setting of correlation coeffs for BASESIMP configuration.
C COEFBAS      BASESIMP coefficients for basic configurations.
C BSCORNER     Set coefficients for BASESIMP `corner-correction method'.


C ******************** BSCOEFF ********************
C BSCOEFF     Controls setting of correlation coeffs for BASESIMP configuration.

      SUBROUTINE BSCOEFF(iconfig)

C This subroutine controls the setting of the correlation coefficients and
C the `corner-correction column number' for the BASESIMP configuration under
C consideration. It examines the `iconfig' number can calls the appropriate
C subroutine to set the coefficients. The coefficients are placed in the
C Dsumuo, Dsumur, Datten, Dphase, and Dcorn common blocks.

C The correlation coefficients are described in:
C    Beausoleil-Morrison I., `BASESIMP: A Simplified Foundation Energy-Loss
C    Model Derived from BASECALC Simulations', NRCan internal report, 1996.

C The `corner-correction column number' refers to columns of Table 1 and
C Table 2 in:
C    Beausoleil-Morrison, Mitalas, and Chin (1992), `Estimating Three-Dimensional
C    Below-Grade Heat Losses from Houses Using Two-Dimensional Calculations',
C    Proceedings of Thermal Performance of the Exterior Envelopes of Buildings VI,
C    ASHRAE, pp. 95-99.
C icol = 1 for column 1 of Table 1.  icol = 5 for column1 of Table 2.


C Call the appropriate subroutine to set the coefficients in the common blocks.
C The basic configurations.
      IF( (iconfig.ge.1 .and. iconfig.le.10)
     &    .or. (iconfig.ge.28 .and. iconfig.le.35)
     &   .or. (iconfig.ge.52 .and. iconfig.le.53) ) THEN
      CALL COEFBAS(iconfig)

C Extended configurations from the original set of 67 configurations.
      ELSEIF( (iconfig.ge.11 .and. iconfig.le.27)
     &        .or. (iconfig.ge.36 .and. iconfig.le.51)
     &        .or. (iconfig.ge.54 .and. iconfig.le.67) )THEN
        CALL COEFEXT1(iconfig)

C Extended configurations for set generated by Debra Haltrecht in 1998.
      ELSEIF( iconfig.ge.68 .and. iconfig.le.72 ) THEN
        CALL COEFEXT2(iconfig)

C Extended configurations for set generated by Julia Purdy in summer of 1999.
      ELSEIF( (iconfig.ge.73 .and. iconfig.le.86)
     &        .or. (iconfig.ge.108 .and. iconfig.le.115)
     &        .or. (iconfig.eq.120) ) THEN
        CALL COEFEXT3(iconfig)

C Extended configurations for set generated by Kamel Haddad in summer of 1999.
      ELSEIF( (iconfig.ge.87 .and. iconfig.le.107)
     &        .or. (iconfig.ge.116 .and. iconfig.le.119) ) THEN
        CALL COEFEXT4(iconfig)

C Extended configurations for set generated by Julia Purdy October 1999 to
C support MNECH.
      ELSEIF( iconfig.ge.121 .and. iconfig.le.145 ) THEN
        CALL COEFEXT5(iconfig)

      ENDIF

      RETURN
      END


C ******************** COEFBAS ********************
C COEFBAS      BASESIMP coefficients for basic configurations.

C      #1   BCIN_1 conc Wl&Fl; intrn ins full-height; *any constr
C      #2   BCIN_2 conc Wl&Fl; intrn ins to 0.2 above floor; *any constr
C      #3   BCIN_3 conc Wl&Fl; intrn ins to 0.6 below grade; *brick veneer
C      #4   BCIN_4 conc Wl&Fl; intrn ins to 0.6 below grade; *non-brick veneer
C      #5   BCEN_1 conc Wl&Fl; extrn ins full-height; *brick veneer
C      #6   BCEN_2 conc Wl&Fl; extrn ins full-height; *non-brick veneer
C      #7   BCEN_3 conc Wl&Fl; extrn ins below grade; *brick veneer
C      #8   BCEN_4 conc Wl&Fl; extrn ins below grade; *non-brick veneer
C      #9   BCNN_1 conc Wl&Fl; no ins; *brick veneer
C      #10  BCNN_2 conc Wl&Fl; no ins; *non-brick veneer
C      #11  BCCN_1 conc Wl&Fl; extrn below grade with overlap; *brick on wall
C      #12  BCCN_2 conc Wl&Fl; extrn below grade with overlap; *non-brick
C      #13  BWNN_1 wood Wl&Fl; no ins; *any constr
C      #14  BWIN_1 wood Wl&Fl; intrn ins full-height; *any constr
C      #15  BWIN_2 wood Wl&Fl; intrn ins to 0.2 above floor; *any constr
C      #16  BWIN_3 wood Wl&Fl; intrn ins to 0.6 below grade; *any constr
C      #17  BWEN_1 wood Wl&Fl; extrn ins full-height; *any constr
C      #18  BWEN_2 wood Wl&Fl; extrn ins below grade; *any constr
C      #19  BCIB_1 conc Wl&Fl; intrn ins full; ins sub-surface; *any constr
C      #20  BCIB_2 conc Wl&Fl; intrn ins full; 0.6m perm floor ins; *any constr
C      #21  BCIB_3 conc Wl&Fl; intrn ins full; 1.0m perm floor ins; *any constr
C      #22  BCIB_4 conc Wl&Fl; intrn ins full; full floor ins; *any constr
C      #23  BCIB_5 conc Wl&Fl; intrn ins full; 0.6m perm floor ins; TB; *any
C      #24  BCIB_6 conc Wl&Fl; intrn ins full; 1.0m perm floor ins; TB; *any
C      #25  BCEB_1 conc Wl&Fl; extrn ins full; full floor ins; *any constr
C      #26  BCEB_2 conc Wl&Fl; extrn ins full; 0.6m perm floor ins; *any constr
C      #27  BCEB_3 conc Wl&Fl; extrn ins full; 1.0m perm floor ins; *any constr
C      #28  SCN_1 conc/soil FL; no ins; *non-brick veneer
C      #29  SCN_2 conc/soil FL; no ins; *brick on slab
C      #30  SCN_3 conc/soil FL; no ins; TB; *non-brick veneer
C      #31  SCN_4 conc/soil FL; no ins; TB; *brick veneer 
C      #32  SCN_7 conc/soil FL; no ins; vert skirt; *non-brick veneer
C      #33  SCN_8 conc/soil FL; no ins; vert skirt; TB; *brick veneer 
C      #34  SCB_1 conc/soil FL; 0.6m perm slab ins; *non-brick veneer
C      #35  SCB_2 conc/soil FL; 0.6m perm slab ins; *brick on slab
C      #36  SCB_3 conc/soil FL; 0.6m perm slab & footing ins; *non-brick veneer
C      #37  SCB_4 conc/soil FL; 0.6m perm slab & footing ins; *brick on slab
C      #38  SCB_5 conc/soil FL; no slab or footing ins; TB; *non-brick veneer
C      #39  SCB_6 conc/soil FL; no slab or footing ins; TB; *brick on slab
C      #40  SCB_9 conc/soil FL; 0.6m perm slab ins; TB; vert skirt; *non-brick
C      #41  SCB_10 conc/soil FL; 0.6m perm slab; TB; vert skirt;  *brick on slb
C      #42  SCB_11 conc/soil FL; 0.6m perm slab; horiz skirt; *non-brick veneer
C      #43  SCB_12 conc/soil FL; 0.6m perm slab; horiz skirt; *brick on slab
C      #44  SCB_13 conc/soil FL; 1.0m perm slab; horiz skirt; *brick on slab
C      #45  SCB_14 conc/soil FL; 1.0m perm slab; *brick on slab
C      #46  SCB_17 conc/soil FL; 1.0m perm slab; TB; *non-brick veneer
C      #47  SCB_18 conc/soil FL; 1.0m perm slab; TB; *brick on slab
C      #48  SCB_21 conc/soil FL; 1.0m perm slab; TB; vert skirt; *non-brick
C      #49  SCB_22 conc/soil FL; 1.0m perm slab; TB; vert skirt; *brick on slb
C      #50  SCB_23 conc/soil FL; 1.0m perm slab; TB; horiz skirt; *non-brick
C      #51  SCB_24 conc/soil FL; 1.0m perm slab; TB; horiz skirt; *brick on slb
C      #52  SCB_25 conc/soil FL; full slab ins; *non-brick veneer
C      #53  SCB_26 conc/soil FL; full slab ins; *brick on slab
C      #54  SCB_29 conc/soil FL; full slab ins; TB; *non-brick veneer
C      #55  SCB_30 conc/soil FL; full slab ins; TB; *brick on slab
C      #56  SCB_33 conc/soil FL; full slab ins; TB; vert skirt; *non-brick
C      #57  SCB_34 conc/soil FL; full slab ins; TB; vert skirt; *brick on slab
C      #58  SCB_35 conc/soil FL; full slab ins; horiz skirt; *non-brick veneer
C      #59  SCB_36 conc/soil FL; full slab ins; horiz skirt; *brick on slab
C      #60  SCA_17 conc/soil FL; full top slab; *non-brick veneer
C      #61  SCA_18 conc/soil FL; full top slab; *brick on slab
C      #62  SCA_19 conc/soil FL; full top slab; TB; *non-brick veneer
C      #63  SCA_20 conc/soil FL; full top slab; TB; *brick on slab
C      #64  SCA_21 conc/soil FL; full top slab; TB; vert skirt; *non-brick venr
C      #65  SCA_22 conc/soil FL; full top slab; TB; vert skirt; *brick on slab
C      #66  SCA_23 conc/soil FL; full top slab; TB; horiz skirt; *non-brick
C      #67  SCA_24 conc/soil FL; full top slab; TB; horiz skirt; brick on slab
C      #68  BCCN_3 conc Wl&Fl; no slab ins; TB; full ins both sides; *any
C      #69  BCCN_4 conc Wl&Fl; full slab; TB; full ins both sides; *any constr
C      #70  BCEA_1 conc Wl&Fl; full top slab; extrn ins full; *brick veneer
C      #71  BCEA_4 conc Wl&Fl; full top slab; extrn ins full; *non-brick veneer
C      #72  BCIA_1 conc Wl&Fl; full top slab; intern ins full; *brick veneer
C      #73  BCIA_4 conc Wl&Fl; full top slab; intern ins full; *non-brick
C      #74  BCEA_7 conc Wl&Fl; full top slab; extrn ins blw_grd; *brick on slb
C      #75  BCEA_8 conc Wl&Fl; full top slab; extrn blw_grd; *non-brick veneer
C      #76  BCEB_8 conc Wl&Fl; full slab ins; extrn ins blw_grd; *brick veneer
C      #77  BCEB_9 conc Wl&Fl; full slab ins; TB; extrn 0.6 blw_grd; *non-brick
C      #78  BWEN_3 wood Wl&Fl; extrn to 0.6 blw_grd; *any constr
C      #79  BBIB_3 wood Wl conc Fl; full slab; TB; intrn full; *any constr
C      #80  BBIB_4 wood Wl conc Fl; full slab; TB; intrn 0.6 blw_grd; *brick
C      #81  SCB_31 conc/soil FL; full slab & footings; TB; *non-brick veneer
C      #82  SCB_32 conc/soil FL; full slab & footings; TB; *brick on slab
C      #83  SCB_37 conc/soil FL; 0.35m perm slab & footings; TB; *non-brick
C      #84  SCB_38 conc/soil FL; 0.35m perm slab & footings; TB; *brick on slb
C      #85  SCB_39 conc/soil FL; 0.75m perm slab & footings; TB; *non-brick
C      #86  SCB_40 conc/soil FL; 0.75m perm slab & footings; TB; *brick on slb
C      #87  BBEB_1 wood Wl conc Fl; full slab; extrn 0.6 blw_grd; *brick on w
C      #88  BBEN_1 wood Wl conc Fl; no slab; extrn full; *brick on wall
C      #89  BBEN_2 wood Wl conc Fl; no slab; extrn 0.6 blw_grd; *brick on wal
C      #90  BBIA_2 wood Wl conc Fl; top slab; intrn ins full; *any constr
C      #91  BBIN_2 wood Wl conc Fl; intrn to 0.6 blw_grd; *non-brick veneer
C      #92  BCCB_8 conc Wl&Fl; 0.6m perm slab; intrn&extrn ins full; *any con
C      #93  BCCA_7 conc Wl&Fl; top slab; intrn full; ext 0.6 blw_grd; *any con
C      #94  BCCA_8 conc Wl&Fl; top slab; intrn above 0.2; extrn full; *any con
C      #95  BCCN_4 conc Wl&Fl; intrn full; extrn 0.6 blw_grd; *any constr
C      #96  BCCN_5 conc Wl&Fl; intrn top to 0.2; extrn ins full; *any constr
C      #97  BCEA_5 conc Wl&Fl; top slab; extrn 0.6 blw_grd; *brick veneer
C      #98  BCEA_6 conc Wl&Fl; top slab; extrn 0.6 blw_grd; *non-brick veneer
C      #99  BCEB_4 conc Wl&Fl; full slab ins; extrn full; *non-brick veneer
C      #100 BCEB_5 conc Wl&Fl; 0.6m perm slab; extrn full; *non-brick veneer
C      #101 BCEB_6 conc Wl&Fl; 1.0m perm slab; extrn full; *non-brick veneer
C      #102 BWIA_1 wood Wl&Fl; 0.6m perm top floor; intrn full; *any constr
C      #103 BWIA_2 wood Wl&Fl; top floor ins; intrn full; *any constr
C      #104 SCA_1 conc/soil FL; 0.6m perm top slab ins; *non-brick veneer
C      #105 SCA_2 conc/soil FL; 0.6m perm top slab ins; *brick on slab
C      #106 SCA_9 conc/soil FL; 1.0m perm top slab ins; *non-brick veneer
C      #107 SCA_10 conc/soil FL; 1.0m perm top slab ins; *brick on slab
C      #108 BBIN_1 wood Wl conc Fl; intrn ins full; *non-brick veneer
C      #109 BCEN_5 conc Wl&Fl; extrn ins to 0.6 blw_grd; *brick on wall
C      #110 BCEN_6 conc Wl&Fl; extrn ins to 0.6 blw_grd; *non-brick veneer
C      #111 BBIA_1 wood Wl conc Fl; top slab; intrn full; *any constr
C      #112 BBIB_1 wood Wl conc Fl; 0.6m perm slab; intrn full; *any constr
C      #113 BBIB_2 wood Wl conc Fl; full slab; intrn full; *any constr
C      #114 BCCB_9 conc Wl&Fl; full slab; intrn full; extrn 0.6 blw_grd; *any
C      #115 BCCB_10 conc Wl&Fl; 0.6m perm; intrn full; extrn 0.6 blw_grd; *any
C      #116 BCCA_1 conc Wl&Fl; full top; int overlap; extrn blw_grd; *brick
C      #117 BCCA_4 conc Wl&Fl; full top; int overlap; extrn blw_grd; *non-brick
C      #118 BCIB_7 conc Wl&Fl; full slab; TB; intrn 0.6 blw_grd; *brick on wall
C      #119 BCIB_8 conc Wl&Fl; full slab; TB; intrn 0.6 blw_grd; *non-brick
C      #120 BBEB_2 wood W conc F; full slab; TB; extrn 0.6 blw_grd; *brick on w
C      #121 BCIA_3 conc Wl&Fl; 1.0m perm top; intrn full; *brick on wall
C      #122 BCIA_5 conc Wl&Fl; 0.6m perm top; intrn 0.6 blw_grd; *brick on wall
C      #123 BCIA_6 conc Wl&Fl; 0.6m perm top; intrn 0.6 blw_grd; *non-brick
C      #124 BCIB_9 conc Wl&Fl; 0.6m perm; TB; intrn 0.6 blw_grd; *brick on wall
C      #125 BCIB_10 conc Wl&Fl; 0.6m perm; TB; intrn 0.6 blw_grd; *non-brick
C      #126 BCEB_10 conc Wl&Fl; 0.6m perm; TB; extrn 0.6 blw_grd; *brick on w
C      #127 BCEB_11 conc Wl&Fl; 0.6m perm; TB; extrn 0.6 blw_grd; *non-brick
C      #128 BCEA_3 conc Wl&Fl; 1.0m perm top; extrn full; *brick on wall
C      #129 BCEA_9 conc Wl&Fl; 1.0m perm top; extrn full; *non-brick
C      #130 BCEA_10 conc Wl&Fl; 0.6m perm top; extrn 0.6 blw_grd; *brick on w
C      #131 BCEA_11 conc Wl&Fl; 0.6m perm top; extrn 0.6 blw_grd; *non-brick
C      #132 BWIB_1 wood Wl&Fl; 1.0m perm floor; intrn full; *brick on wall
C      #133 BWIB_2 wood Wl&Fl; full floor; intrn full; *brick on wall
C      #134 BWIB_4 wood Wl&Fl; 0.6m perm floor; intrn full; *brick on wall
C      #135 BWIA_3 wood Wl&Fl; 0.6m perm top; intrn 0.6 blw_grd; *brick on wall
C      #136 BWEB_1 wood Wl&Fl; 1.0m perm floor; extrn full; *brick on wall
C      #137 BWEB_2 wood Wl&Fl; full floor; extrn full; *brick on wall
C      #138 BWEB_3 wood Wl&Fl; 0.6m perm floor; extrn 0.6 blw_grd; *brick on w
C      #139 BWEB_4 wood Wl&Fl; 0.6m perm floor; extrn full; * brick on wall
C      #140 BBIB_5 wood Wl conc Fl; 1.0m perm; intrn ins full; *brick on wall
C      #141 BBIB_6 wood Wl conc Fl; 1.0m perm; intrn 0.6 blw_grd; *brick on w
C      #142 BBEB_3 wood Wl conc Fl; 1.0m perm; extrn full;* brick on wall
C      #143 BBEB_4 wood Wl conc Fl; 0.6m perm; extrn full; *brick on wall
C      #144 BBEB_5 wood Wl conc Fl; 0.6m perm; extrn 0.6 blw_grd; *brick on w
C      #145 BBEA_2 wood Wl conc Fl; full top slab; extrn full; *brick on wall
C      

      SUBROUTINE COEFBAS(iconfig)

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/Dsumuo/BSa1,BSb1,BSc1,BSd1,BSe1,BSf1,BSg1,BSh1,BSi1,BSj1
      COMMON/Dsumur/BSq2,BSr2,BSu2,BSv2,BSw2,BSx2,BSs2,BSt2,BSy2,BSa2,
     &              BSb2,BSc2,BSd2,BSe2,BSf2,BSg2,BSh2
      COMMON/Datten/BSa3,BSb3,BSc3,BSe3,BSf3,BSg3,BSh3,BSi3
      COMMON/Dphase/BSa4,BSb4,BSc4
      COMMON/Dcorn/icol

C This subroutine sets the BASESIMP correlation coefficients for the basic
C configurations that are available in the ESP-r distribution. These coefficients
C were generated by Ian Beausoleil-Morrison in 1996.

      IF(iconfig.eq.1) THEN
C BCIN_1:
C    - concrete walls and floor;
C    - interior surface of wall insulated over full-height;
C    - any first storey construction type.
        BSa1 =     0.021400
        BSb1 =     0.706000
        BSc1 =     0.102000
        BSd1 =     0.704000
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =     0.291000
        BSr2 =     0.318000
        BSu2 =     0.229000
        BSv2 =     0.620000
        BSw2 =     0.711000
        BSx2 =     0.500000
        BSs2 =    -0.055000
        BSt2 =     0.023500
        BSy2 =     0.179000
        BSa2 =     0.749000
        BSb2 =     0.712000
        BSc2 =     0.452000
        BSd2 =     0.263000
        BSe2 =     3.000000
        BSf2 =    -0.035300
        BSg2 =    -1.011000
        BSh2 =     0.000000
        BSa3 =     0.258000
        BSb3 =     0.317000
        BSc3 =    -0.188000
        BSe3 =     0.006760
        BSf3 =     0.110000
        BSg3 =     0.219000
        BSh3 =     0.769000
        BSi3 =     0.000000
        BSa4 =     2.415000
        BSb4 =     0.488000
        BSc4 =     0.259000
        icol =  3

      ELSEIF(iconfig.eq.2) THEN
C BCIN_2:
C    - concrete walls and floor
C    - interior surface of wall insulated from top of wall to 0.2 m from floor
C    - any first storey construction type
        BSa1 =     0.022500
        BSb1 =     0.698000
        BSc1 =     0.117000
        BSd1 =     0.643000
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =     0.439000
        BSr2 =     0.281000
        BSu2 =     0.266000
        BSv2 =     0.620000
        BSw2 =     0.826000
        BSx2 =     0.500000
        BSs2 =    -0.072000
        BSt2 =     0.023000
        BSy2 =     0.178000
        BSa2 =     0.706000
        BSb2 =     0.865000
        BSc2 =     0.533000
        BSd2 =     0.358000
        BSe2 =     3.486000
        BSf2 =    -0.064700
        BSg2 =    -1.211000
        BSh2 =     0.000000
        BSa3 =     0.439000
        BSb3 =     0.377000
        BSc3 =    -0.278000
        BSe3 =    -0.045700
        BSf3 =     0.104000
        BSg3 =     0.227000
        BSh3 =     0.756000
        BSi3 =     0.000000
        BSa4 =     2.645000
        BSb4 =     0.284000
        BSc4 =     0.325000
        icol =  3

      ELSEIF(iconfig.eq.3) THEN
C BCIN_3:
C    - concrete walls and floor
C    - interior surface of wall insulated from top of wall to 0.6 m below grade
C    - first storey brick veneer placed directly on basement's concrete walls
        BSa1 =    -0.102000
        BSb1 =     0.735000
        BSc1 =     0.133000
        BSd1 =     0.764000
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.095200
        BSq2 =     1.253000
        BSr2 =     0.546000
        BSu2 =     0.256000
        BSv2 =     0.620000
        BSw2 =     0.177000
        BSx2 =     0.500000
        BSs2 =    -0.061700
        BSt2 =     0.021900
        BSy2 =     0.187000
        BSa2 =     0.104000
        BSb2 =     1.350000
        BSc2 =     0.765000
        BSd2 =    -0.233000
        BSe2 =     0.769000
        BSf2 =    -0.013000
        BSg2 =    -0.283000
        BSh2 =     0.000000
        BSa3 =    -0.218000
        BSb3 =     0.609000
        BSc3 =     0.270000
        BSe3 =     0.180000
        BSf3 =     0.064600
        BSg3 =    -0.030600
        BSh3 =     0.812000
        BSi3 =     0.000000
        BSa4 =     2.978000
        BSb4 =    -0.008520
        BSc4 =    -0.026600
        icol =  2

      ELSEIF(iconfig.eq.4) THEN
C BCIN_4:
C    - concrete walls and floor
C    - interior surface of wall insulated from top of wall to 0.6 m below grade
C    - first storey is non-brick veneer or bricks thermally broken from
C      basement's concrete walls
        BSa1 =    -0.107000
        BSb1 =     0.756000
        BSc1 =     0.120000
        BSd1 =     0.727000
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.080800
        BSq2 =     1.217000
        BSr2 =     0.533000
        BSu2 =     0.260000
        BSv2 =     0.620000
        BSw2 =     0.184000
        BSx2 =     0.500000
        BSs2 =    -0.072900
        BSt2 =     0.022100
        BSy2 =     0.188000
        BSa2 =     0.111000
        BSb2 =     1.446000
        BSc2 =     0.763000
        BSd2 =    -0.180000
        BSe2 =     0.810000
        BSf2 =    -0.014100
        BSg2 =    -0.297000
        BSh2 =     0.000000
        BSa3 =    -0.216000
        BSb3 =     0.608000
        BSc3 =     0.273000
        BSe3 =     0.182000
        BSf3 =     0.064900
        BSg3 =    -0.029500
        BSh3 =     0.811000
        BSi3 =     0.000000
        BSa4 =     3.052000
        BSb4 =    -0.092800
        BSc4 =    -0.135000
        icol =  2

      ELSEIF(iconfig.eq.5) THEN
C BCEN_1:
C    - concrete walls and floor
C    - exterior surface of wall insulated over full-height
C    - first storey brick veneer placed directly on basement's concrete walls
        BSa1 =    -0.083500
        BSb1 =     0.749000
        BSc1 =     0.003340
        BSd1 =     0.885000
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.357000
        BSq2 =     0.312000
        BSr2 =     0.349000
        BSu2 =     0.186000
        BSv2 =     0.620000
        BSw2 =     0.678000
        BSx2 =     0.500000
        BSs2 =    -0.036000
        BSt2 =     0.023200
        BSy2 =     0.184000
        BSa2 =     0.824000
        BSb2 =     0.722000
        BSc2 =     0.358000
        BSd2 =     0.271000
        BSe2 =     2.917000
        BSf2 =    -0.061300
        BSg2 =    -0.967000
        BSh2 =     0.000000
        BSa3 =     0.136000
        BSb3 =     0.330000
        BSc3 =    -0.181000
        BSe3 =     0.043400
        BSf3 =     0.082800
        BSg3 =     0.246000
        BSh3 =     0.728000
        BSi3 =     0.000000
        BSa4 =     1.574000
        BSb4 =     1.302000
        BSc4 =     0.108000
        icol =  5

      ELSEIF(iconfig.eq.6) THEN
C BCEN_2:
C    - concrete walls and floor
C    - exterior surface of wall insulated over full-height
C    - first storey is non-brick veneer or bricks thermally broken from
C      basement's concrete walls
        BSa1 =    -0.037600
        BSb1 =     0.765000
        BSc1 =     0.001760
        BSd1 =     0.888000
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.120000
        BSq2 =     0.313000
        BSr2 =     0.350000
        BSu2 =     0.187000
        BSv2 =     0.620000
        BSw2 =     0.675000
        BSx2 =     0.500000
        BSs2 =    -0.036400
        BSt2 =     0.023200
        BSy2 =     0.184000
        BSa2 =     0.821000
        BSb2 =     0.731000
        BSc2 =     0.354000
        BSd2 =     0.263000
        BSe2 =     2.888000
        BSf2 =    -0.060800
        BSg2 =    -0.953000
        BSh2 =     0.000000
        BSa3 =     0.137000
        BSb3 =     0.330000
        BSc3 =    -0.182000
        BSe3 =     0.039900
        BSf3 =     0.083300
        BSg3 =     0.255000
        BSh3 =     0.727000
        BSi3 =     0.000000
        BSa4 =     1.508000
        BSb4 =     1.371000
        BSc4 =     0.103000
        icol =  5

      ELSEIF(iconfig.eq.7) THEN
C BCEN_3:
C    - concrete walls and floor
C    - exterior surface of wall insulated below grade
C    - first storey brick veneer placed directly on basement's concrete walls
        BSa1 =     0.202000
        BSb1 =     2.921000
        BSc1 =     0.004110
        BSd1 =    -0.002890
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =     0.439000
        BSr2 =     0.268000
        BSu2 =     0.353000
        BSv2 =     0.620000
        BSw2 =     0.781000
        BSx2 =     0.500000
        BSs2 =    -0.070300
        BSt2 =     0.023100
        BSy2 =     0.170000
        BSa2 =     0.643000
        BSb2 =     0.878000
        BSc2 =     0.391000
        BSd2 =     0.266000
        BSe2 =     3.079000
        BSf2 =    -0.081900
        BSg2 =    -0.996000
        BSh2 =     0.000000
        BSa3 =     0.728000
        BSb3 =     0.318000
        BSc3 =    -0.349000
        BSe3 =    -0.078100
        BSf3 =     0.077200
        BSg3 =     0.299000
        BSh3 =     0.654000
        BSi3 =     0.000000
        BSa4 =     2.866000
        BSb4 =     0.116000
        BSc4 =     0.411000
        icol =  4

      ELSEIF(iconfig.eq.8) THEN
C BCEN_4:
C    - concrete walls and floor
C    - exterior surface of wall insulated below grade
C    - first storey is non-brick veneer or bricks thermally broken from
C      basement's concrete walls
        BSa1 =     0.128000
        BSb1 =     2.951000
        BSc1 =     0.003960
        BSd1 =    -0.002840
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =     0.443000
        BSr2 =     0.268000
        BSu2 =     0.355000
        BSv2 =     0.620000
        BSw2 =     0.777000
        BSx2 =     0.500000
        BSs2 =    -0.069800
        BSt2 =     0.023100
        BSy2 =     0.169000
        BSa2 =     0.631000
        BSb2 =     0.890000
        BSc2 =     0.386000
        BSd2 =     0.258000
        BSe2 =     3.021000
        BSf2 =    -0.071300
        BSg2 =    -0.980000
        BSh2 =     0.000000
        BSa3 =     0.726000
        BSb3 =     0.317000
        BSc3 =    -0.342000
        BSe3 =    -0.079500
        BSf3 =     0.077500
        BSg3 =     0.301000
        BSh3 =     0.653000
        BSi3 =     0.000000
        BSa4 =     2.874000
        BSb4 =     0.109000
        BSc4 =     0.426000
        icol =  4

      ELSEIF(iconfig.eq.9) THEN
C BCNN_1:
C    - concrete walls and floor
C    - no insulation
C    - first storey brick veneer placed directly on basement's concrete walls
        BSa1 =     0.001340
        BSb1 =     2.936000
        BSc1 =     0.095700
        BSd1 =     0.000000
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =     1.302000
        BSr2 =     0.279000
        BSu2 =     0.344000
        BSv2 =     0.620000
        BSw2 =     0.690000
        BSx2 =     0.500000
        BSs2 =    -0.034700
        BSt2 =     0.020300
        BSy2 =     0.086000
        BSa2 =     0.000000
        BSb2 =     0.000000
        BSc2 =     0.000000
        BSd2 =     0.000000
        BSe2 =     0.000000
        BSf2 =     0.000000
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =     0.459000
        BSb3 =     0.829000
        BSc3 =     0.174000
        BSe3 =     0.000000
        BSf3 =     0.000000
        BSg3 =     0.000000
        BSh3 =     0.000000
        BSi3 =     0.000000
        BSa4 =     3.064000
        BSb4 =     0.000000
        BSc4 =     0.000000
        icol =  1

      ELSEIF(iconfig.eq.10) THEN
C BCNN_2:
C    - concrete walls and floor
C    - no insulation
C    - first storey is non-brick veneer or bricks thermally broken from
C      basement's concrete walls
        BSa1 =    -0.050500
        BSb1 =     2.959000
        BSc1 =     0.083000
        BSd1 =     0.000000
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =     1.309000
        BSr2 =     0.279000
        BSu2 =     0.348000
        BSv2 =     0.620000
        BSw2 =     0.686000
        BSx2 =     0.500000
        BSs2 =    -0.034200
        BSt2 =     0.020200
        BSy2 =     0.085000
        BSa2 =     0.000000
        BSb2 =     0.000000
        BSc2 =     0.000000
        BSd2 =     0.000000
        BSe2 =     0.000000
        BSf2 =     0.000000
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =     0.453000
        BSb3 =     0.830000
        BSc3 =     0.186000
        BSe3 =     0.000000
        BSf3 =     0.000000
        BSg3 =     0.000000
        BSh3 =     0.000000
        BSi3 =     0.000000
        BSa4 =     3.065000
        BSb4 =     0.000000
        BSc4 =     0.000000
        icol =  1

      ELSEIF(iconfig.eq.28) THEN
C SCN_1:
C    - concrete or soil (for crawl space) floor
C    - no insulation
C    - first storey is non-brick veneer or bricks thermally broken from concrete
C      floor
        BSa1 =     0.024900
        BSb1 =     0.000000
        BSc1 =     0.004650
        BSd1 =     0.000000
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =    -3.398000
        BSr2 =     0.423000
        BSu2 =     1.000000
        BSv2 =     0.000000
        BSw2 =     1.000000
        BSx2 =     0.000000
        BSs2 =     0.592000
        BSt2 =     0.010300
        BSy2 =     0.000000
        BSa2 =     2.193000
        BSb2 =     0.000000
        BSc2 =     0.774000
        BSd2 =     0.176000
        BSe2 =     0.000000
        BSf2 =     0.000000
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =     0.144000
        BSb3 =     0.502000
        BSc3 =     0.000000
        BSe3 =     0.000000
        BSf3 =     0.000000
        BSg3 =     0.000000
        BSh3 =     0.000000
        BSi3 =     0.000000
        BSa4 =     2.878000
        BSb4 =     0.000000
        BSc4 =     0.000000
        icol = 98
 
      ELSEIF(iconfig.eq.29) THEN
C SCN_2:
C    - concrete or soil (for crawl space) floor
C    - no insulation
C    - first storey brick veneer placed directly on concrete slab
        BSa1 =     0.059300
        BSb1 =     0.000000
        BSc1 =     0.008790
        BSd1 =     0.000000
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =    -3.441000
        BSr2 =     0.423000
        BSu2 =     1.000000
        BSv2 =     0.000000
        BSw2 =     1.000000
        BSx2 =     0.000000
        BSs2 =     0.588000
        BSt2 =     0.010400
        BSy2 =     0.000000
        BSa2 =     2.241000
        BSb2 =     0.000000
        BSc2 =     0.747000
        BSd2 =     0.173000
        BSe2 =     0.000000
        BSf2 =     0.000000
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =     0.193000
        BSb3 =     0.496000
        BSc3 =     0.000000
        BSe3 =     0.000000
        BSf3 =     0.000000
        BSg3 =     0.000000
        BSh3 =     0.000000
        BSi3 =     0.000000
        BSa4 =     2.897000
        BSb4 =     0.000000
        BSc4 =     0.000000
        icol = 98

      ELSEIF(iconfig.eq.30) THEN
C SCN_3:
C    - concrete or soil (for crawl space) floor
C    - no insulation except for thermal break around edge of slab
C    - first storey is non-brick veneer or bricks thermally broken from concrete
C      floor
        BSa1 =     0.027200
        BSb1 =     0.000000
        BSc1 =     0.005820
        BSd1 =    -0.026400
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =    -3.424000
        BSr2 =     0.432000
        BSu2 =     1.000000
        BSv2 =     0.000000
        BSw2 =     1.000000
        BSx2 =     0.000000
        BSs2 =     0.597000
        BSt2 =     0.010800
        BSy2 =     0.000000
        BSa2 =     2.102000
        BSb2 =     0.000000
        BSc2 =     0.811000
        BSd2 =     0.180000
        BSe2 =     0.003440
        BSf2 =    -0.000649
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =     0.018200
        BSb3 =     0.517000
        BSc3 =     0.000000
        BSe3 =     0.049300
        BSf3 =    -0.011400
        BSg3 =     0.000000
        BSh3 =     0.495000
        BSi3 =     0.000000
        BSa4 =     2.821000
        BSb4 =     0.020400
        BSc4 =     0.512000
        icol = 98

      ELSEIF(iconfig.eq.31) THEN
C SCN_4:
C    - concrete or soil (for crawl space) floor
C    - thermal break around edge of slab
C    - first storey brick veneer placed directly on concrete slab
        BSa1 =     0.070100
        BSb1 =     0.000000
        BSc1 =     0.014500
        BSd1 =    -0.032800
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =    -3.366000
        BSr2 =     0.431000
        BSu2 =     1.000000
        BSv2 =     0.000000
        BSw2 =     1.000000
        BSx2 =     0.000000
        BSs2 =     0.602000
        BSt2 =     0.010500
        BSy2 =     0.000000
        BSa2 =     2.116000
        BSb2 =     0.000000
        BSc2 =     0.794000
        BSd2 =     0.177000
        BSe2 =     0.005250
        BSf2 =    -0.001490
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =     0.021700
        BSb3 =     0.510000
        BSc3 =     0.000000
        BSe3 =     0.066500
        BSf3 =    -0.010900
        BSg3 =     0.000000
        BSh3 =     0.466000
        BSi3 =     0.000000
        BSa4 =     2.826000
        BSb4 =     0.026400
        BSc4 =     0.455000
        icol = 98

      ELSEIF(iconfig.eq.32) THEN
C SCN_7:
C    - concrete or soil (for crawl space) floor
C    - thermal break around edge of slab
C    - vertical skirt extends from bottom of slab
C    - first storey is non-brick veneer or bricks thermally broken from concrete
C      floor
        BSa1 =     0.040300
        BSb1 =     0.000000
        BSc1 =     0.001420
        BSd1 =    -0.067100
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =    -3.354000
        BSr2 =     0.431000
        BSu2 =     1.000000
        BSv2 =     0.000000
        BSw2 =     1.000000
        BSx2 =     0.000000
        BSs2 =     0.613000
        BSt2 =     0.009350
        BSy2 =     0.000000
        BSa2 =     1.955000
        BSb2 =     0.000000
        BSc2 =     0.822000
        BSd2 =     0.219000
        BSe2 =     0.058300
        BSf2 =    -0.018500
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =    -0.101000
        BSb3 =     0.357000
        BSc3 =     0.000000
        BSe3 =     0.159000
        BSf3 =     0.003710
        BSg3 =     0.000000
        BSh3 =     0.669000
        BSi3 =     0.000000
        BSa4 =     2.428000
        BSb4 =     0.296000
        BSc4 =     0.362000
        icol = 98

      ELSEIF(iconfig.eq.33) THEN
C SCN_8:
C    - concrete or soil (for crawl space) floor
C    - thermal break around edge of slab
C    - vertical skirt extends from bottom of slab
C    - first storey brick veneer placed directly on concrete slab
        BSa1 =     0.106000
        BSb1 =     0.000000
        BSc1 =    -0.000446
        BSd1 =    -0.071100
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =    -3.350000
        BSr2 =     0.434000
        BSu2 =     1.000000
        BSv2 =     0.000000
        BSw2 =     1.000000
        BSx2 =     0.000000
        BSs2 =     0.623000
        BSt2 =     0.009410
        BSy2 =     0.000000
        BSa2 =     2.016000
        BSb2 =     0.000000
        BSc2 =     0.785000
        BSd2 =     0.212000
        BSe2 =     0.045800
        BSf2 =    -0.013600
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =    -0.072800
        BSb3 =     0.352000
        BSc3 =     0.000000
        BSe3 =     0.160000
        BSf3 =     0.007080
        BSg3 =     0.000000
        BSh3 =     0.627000
        BSi3 =     0.000000
        BSa4 =     2.511000
        BSb4 =     0.244000
        BSc4 =     0.356000
        icol = 98

      ELSEIF(iconfig.eq.34) THEN
C SCB_1:
C    - concrete or soil (for crawl space) floor
C    - bottom of slab insulated 0.6 m around perimeter
C    - no insulation under footing or foundation wall (ie. insulation starts
C      0.25 m from edge)
C    - first storey is non-brick veneer or bricks thermally broken from concrete
        BSa1 =     0.024300
        BSb1 =     0.000000
        BSc1 =     0.004060
        BSd1 =     0.002510
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =    -3.313000
        BSr2 =     0.412000
        BSu2 =     1.000000
        BSv2 =     0.000000
        BSw2 =     1.000000
        BSx2 =     0.000000
        BSs2 =     0.562000
        BSt2 =     0.011300
        BSy2 =     0.000000
        BSa2 =     2.075000
        BSb2 =     0.000000
        BSc2 =     0.763000
        BSd2 =     0.180000
        BSe2 =     0.010100
        BSf2 =    -0.001210
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =     0.108000
        BSb3 =     0.432000
        BSc3 =     0.000000
        BSe3 =     0.024800
        BSf3 =     0.009120
        BSg3 =     0.000000
        BSh3 =     0.685000
        BSi3 =     0.000000
        BSa4 =     2.826000
        BSb4 =     0.016200
        BSc4 =     0.636000
        icol =  3

      ELSEIF(iconfig.eq.35) THEN
C SCB_2:
C    - concrete or soil (for crawl space) floor
C    - bottom of slab insulated 0.6 m around perimeter
C    - no insulation under footing or foundation wall (ie. insulation starts
C      0.25 m from edge)
C    - first storey brick veneer placed directly on concrete slab
        BSa1 =     0.056500
        BSb1 =     0.000000
        BSc1 =     0.007810
        BSd1 =     0.004430
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =    -3.345000
        BSr2 =     0.411000
        BSu2 =     1.000000
        BSv2 =     0.000000
        BSw2 =     1.000000
        BSx2 =     0.000000
        BSs2 =     0.558000
        BSt2 =     0.011300
        BSy2 =     0.000000
        BSa2 =     2.120000
        BSb2 =     0.000000
        BSc2 =     0.737000
        BSd2 =     0.178000
        BSe2 =     0.011500
        BSf2 =    -0.002150
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =     0.148000
        BSb3 =     0.427000
        BSc3 =     0.000000
        BSe3 =     0.027900
        BSf3 =     0.008520
        BSg3 =     0.000000
        BSh3 =     0.671000
        BSi3 =     0.000000
        BSa4 =     2.848000
        BSb4 =     0.015100
        BSc4 =     0.615000
        icol =  3

      ELSEIF(iconfig.eq.52) THEN
C SCB_25:
C    - concrete or soil (for crawl space) floor
C    - bottom of slab fully insulated except under footing/foundation wall
C      (ie. insulation starts 0.25 m from edge)
C    - first storey is non-brick veneer or bricks thermally broken from concrete
C      floor
        BSa1 =     0.017300
        BSb1 =     0.000000
        BSc1 =     0.009190
        BSd1 =     0.083200
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =    -1.583000
        BSr2 =     0.113000
        BSu2 =     1.000000
        BSv2 =     0.000000
        BSw2 =     1.000000
        BSx2 =     0.000000
        BSs2 =     0.309000
        BSt2 =     0.007560
        BSy2 =     0.000000
        BSa2 =     2.130000
        BSb2 =     0.000000
        BSc2 =     0.423000
        BSd2 =     0.156000
        BSe2 =     0.120000
        BSf2 =     0.094800
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =     0.243000
        BSb3 =     0.010100
        BSc3 =     0.000000
        BSe3 =    -0.003700
        BSf3 =     0.226000
        BSg3 =     0.000000
        BSh3 =     0.610000
        BSi3 =     0.000000
        BSa4 =     2.644000
        BSb4 =     0.186000
        BSc4 =    -0.296000
        icol =  7

      ELSEIF(iconfig.eq.53) THEN
C SCB_26:
C    - concrete or soil (for crawl space) floor
C    - bottom of slab fully insulated except under footing/foundation wall
C      (ie. insulation starts 0.25 m from edge)
C    - first storey brick veneer placed directly on concrete slab
        BSa1 =     0.040200
        BSb1 =     0.000000
        BSc1 =     0.019900
        BSd1 =     0.088900
        BSe1 =     1.000000
        BSf1 =     1.000000
        BSg1 =     1.000000
        BSh1 =     1.000000
        BSi1 =     0.000000
        BSj1 =     0.000000
        BSq2 =    -1.611000
        BSr2 =     0.113000
        BSu2 =     1.000000
        BSv2 =     0.000000
        BSw2 =     1.000000
        BSx2 =     0.000000
        BSs2 =     0.308000
        BSt2 =     0.007390
        BSy2 =     0.000000
        BSa2 =     2.169000
        BSb2 =     0.000000
        BSc2 =     0.402000
        BSd2 =     0.153000
        BSe2 =     0.118000
        BSf2 =     0.094200
        BSg2 =     0.000000
        BSh2 =     0.000000
        BSa3 =     0.279000
        BSb3 =     0.001160
        BSc3 =     0.000000
        BSe3 =     0.003570
        BSf3 =     0.226000
        BSg3 =     0.000000
        BSh3 =     0.612000
        BSi3 =     0.000000
        BSa4 =     2.634000
        BSb4 =     0.220000
        BSc4 =    -0.263000
        icol =  7

      ELSE
        call edisp(IUOUT,' Not a valid iconfig.')

      ENDIF

      RETURN
      END


C ******************** BSCORNER ********************
C BSCORNER     Set coefficients for BASESIMP `corner-correction method'.

      SUBROUTINE BSCORNER

      COMMON/BSCORN/coeff(16,19)

C The `corner-correction method' is described in:
C    Beausoleil-Morrison, Mitalas, and Chin (1992), `Estimating Three-Dimensional
C    Below-Grade Heat Losses from Houses Using Two-Dimensional Calculations',
C    Proceedings of Thermal Performance of the Exterior Envelopes of Buildings VI,
C    ASHRAE, pp. 95-99.

      coeff( 1, 1) =  0.78404336
      coeff( 1, 2) =  0.00000000
      coeff( 1, 3) = -0.06269160
      coeff( 1, 4) =  0.01588997
      coeff( 1, 5) =  0.18362466
      coeff( 1, 6) = -0.00280010
      coeff( 1, 7) =  0.00000000
      coeff( 1, 8) =  0.00000000
      coeff( 1, 9) =  0.00000000
      coeff( 1,10) = -0.00058830
      coeff( 1,11) = -0.00323320
      coeff( 1,12) =  0.00000000
      coeff( 1,13) =  0.01234568
      coeff( 1,14) = -0.01884040
      coeff( 1,15) = -0.04484290
      coeff( 1,16) =  0.00000000
      coeff( 1,17) =  0.00115767
      coeff( 1,18) =  0.00186534
      coeff( 1,19) =  0.00302408
      coeff( 2, 1) =  0.75572785
      coeff( 2, 2) =  0.00000000
      coeff( 2, 3) = -0.04127260
      coeff( 2, 4) =  0.06788635
      coeff( 2, 5) =  0.17354284
      coeff( 2, 6) = -0.00142870
      coeff( 2, 7) =  0.00000000
      coeff( 2, 8) =  0.00000000
      coeff( 2, 9) =  0.00000000
      coeff( 2,10) =  0.00369666
      coeff( 2,11) = -0.00508140
      coeff( 2,12) =  0.00000000
      coeff( 2,13) =  0.01781545
      coeff( 2,14) = -0.01948280
      coeff( 2,15) = -0.03993470
      coeff( 2,16) =  0.00000000
      coeff( 2,17) = -0.00069420
      coeff( 2,18) = -0.00023500
      coeff( 2,19) =  0.00259176
      coeff( 3, 1) =  0.89231276
      coeff( 3, 2) = -0.00590050
      coeff( 3, 3) = -0.07535490
      coeff( 3, 4) = -0.02696010
      coeff( 3, 5) =  0.06201586
      coeff( 3, 6) = -0.00249220
      coeff( 3, 7) =  0.00039837
      coeff( 3, 8) = -0.00062370
      coeff( 3, 9) = -0.00089240
      coeff( 3,10) =  0.00308021
      coeff( 3,11) =  0.00185631
      coeff( 3,12) =  0.00237192
      coeff( 3,13) =  0.01246938
      coeff( 3,14) = -0.02075720
      coeff( 3,15) =  0.00000000
      coeff( 3,16) =  0.00023050
      coeff( 3,17) =  0.00063693
      coeff( 3,18) =  0.00189477
      coeff( 3,19) =  0.00490457
      coeff( 4, 1) =  0.85653204
      coeff( 4, 2) =  0.00560896
      coeff( 4, 3) = -0.04706910
      coeff( 4, 4) =  0.03933353
      coeff( 4, 5) =  0.07332308
      coeff( 4, 6) = -0.00260210
      coeff( 4, 7) = -0.00049210
      coeff( 4, 8) = -0.00142660
      coeff( 4, 9) = -0.00063020
      coeff( 4,10) =  0.00551828
      coeff( 4,11) = -0.00129990
      coeff( 4,12) =  0.00224003
      coeff( 4,13) =  0.01512380
      coeff( 4,14) = -0.01987420
      coeff( 4,15) =  0.00000000
      coeff( 4,16) = -0.00002580
      coeff( 4,17) = -0.00098480
      coeff( 4,18) = -0.00034900
      coeff( 4,19) =  0.00407644
      coeff( 5, 1) =  0.71725273
      coeff( 5, 2) = -0.02203810
      coeff( 5, 3) = -0.06561350
      coeff( 5, 4) =  0.03467739
      coeff( 5, 5) =  0.11542548
      coeff( 5, 6) = -0.00243060
      coeff( 5, 7) =  0.00239333
      coeff( 5, 8) =  0.00017063
      coeff( 5, 9) =  0.00078584
      coeff( 5,10) =  0.00538004
      coeff( 5,11) = -0.00690080
      coeff( 5,12) = -0.00684610
      coeff( 5,13) = -0.00592680
      coeff( 5,14) = -0.01821230
      coeff( 5,15) = -0.01805300
      coeff( 5,16) =  0.00021603
      coeff( 5,17) =  0.00007409
      coeff( 5,18) =  0.00219304
      coeff( 5,19) =  0.00495485
      coeff( 6, 1) =  0.66286115
      coeff( 6, 2) = -0.01117000
      coeff( 6, 3) = -0.04668110
      coeff( 6, 4) =  0.10415677
      coeff( 6, 5) =  0.15558250
      coeff( 6, 6) = -0.00099990
      coeff( 6, 7) =  0.00123304
      coeff( 6, 8) = -0.00138840
      coeff( 6, 9) =  0.00144637
      coeff( 6,10) =  0.00813609
      coeff( 6,11) = -0.01029190
      coeff( 6,12) = -0.00276280
      coeff( 6,13) =  0.00179876
      coeff( 6,14) = -0.01555330
      coeff( 6,15) = -0.02351600
      coeff( 6,16) = -0.00013170
      coeff( 6,17) = -0.00130010
      coeff( 6,18) = -0.00017010
      coeff( 6,19) =  0.00211170
      coeff( 7, 1) =  0.79399414
      coeff( 7, 2) = -0.00590040
      coeff( 7, 3) = -0.06301890
      coeff( 7, 4) =  0.01801793
      coeff( 7, 5) =  0.16194821
      coeff( 7, 6) = -0.00370510
      coeff( 7, 7) =  0.00081849
      coeff( 7, 8) = -0.00032200
      coeff( 7, 9) =  0.00039037
      coeff( 7,10) = -0.00088960
      coeff( 7,11) = -0.00412530
      coeff( 7,12) = -0.00348100
      coeff( 7,13) =  0.00266370
      coeff( 7,14) = -0.01568740
      coeff( 7,15) = -0.04236650
      coeff( 7,16) =  0.00004032
      coeff( 7,17) =  0.00137467
      coeff( 7,18) =  0.00205882
      coeff( 7,19) =  0.00345549
      coeff( 8, 1) =  0.75304947
      coeff( 8, 2) = -0.00394460
      coeff( 8, 3) = -0.03874130
      coeff( 8, 4) =  0.07195393
      coeff( 8, 5) =  0.15915441
      coeff( 8, 6) = -0.00103200
      coeff( 8, 7) =  0.00051130
      coeff( 8, 8) = -0.00007610
      coeff( 8, 9) =  0.00059102
      coeff( 8,10) =  0.00414472
      coeff( 8,11) = -0.00626760
      coeff( 8,12) = -0.00225830
      coeff( 8,13) =  0.01422931
      coeff( 8,14) = -0.01478020
      coeff( 8,15) = -0.03936080
      coeff( 8,16) = -0.00007220
      coeff( 8,17) = -0.00103310
      coeff( 8,18) = -0.00015610
      coeff( 8,19) =  0.00204608
      coeff( 9, 1) =  0.86826744
      coeff( 9, 2) = -0.01540410
      coeff( 9, 3) = -0.06757750
      coeff( 9, 4) = -0.03465790
      coeff( 9, 5) =  0.10827631
      coeff( 9, 6) = -0.00003390
      coeff( 9, 7) =  0.00177775
      coeff( 9, 8) =  0.00086900
      coeff( 9, 9) =  0.00039226
      coeff( 9,10) =  0.00834285
      coeff( 9,11) =  0.00051283
      coeff( 9,12) = -0.00852540
      coeff( 9,13) = -0.00794040
      coeff( 9,14) = -0.01895280
      coeff( 9,15) = -0.01331900
      coeff( 9,16) =  0.00024774
      coeff( 9,17) = -0.00044170
      coeff( 9,18) =  0.00193629
      coeff( 9,19) =  0.00554321
      coeff(10, 1) =  0.85992881
      coeff(10, 2) = -0.00117880
      coeff(10, 3) = -0.05706770
      coeff(10, 4) =  0.03259717
      coeff(10, 5) =  0.16789688
      coeff(10, 6) = -0.00017742
      coeff(10, 7) =  0.00041268
      coeff(10, 8) = -0.00120900
      coeff(10, 9) =  0.00075719
      coeff(10,10) =  0.01013219
      coeff(10,11) = -0.00276250
      coeff(10,12) = -0.00340710
      coeff(10,13) =  0.00021417
      coeff(10,14) = -0.01798040
      coeff(10,15) = -0.02177810
      coeff(10,16) = -0.00015070
      coeff(10,17) = -0.00123310
      coeff(10,18) = -0.00005570
      coeff(10,19) =  0.00222115
      coeff(11, 1) =  0.59593229
      coeff(11, 2) = -0.04732680
      coeff(11, 3) = -0.03112210
      coeff(11, 4) =  0.03256568
      coeff(11, 5) =  0.21272148
      coeff(11, 6) =  0.00202659
      coeff(11, 7) =  0.00500298
      coeff(11, 8) =  0.00590225
      coeff(11, 9) = -0.00119010
      coeff(11,10) = -0.00301830
      coeff(11,11) = -0.00349060
      coeff(11,12) = -0.00087780
      coeff(11,13) = -0.00436520
      coeff(11,14) = -0.01597470
      coeff(11,15) = -0.03786950
      coeff(11,16) = -0.00082380
      coeff(11,17) = -0.00140310
      coeff(11,18) =  0.00155153
      coeff(11,19) =  0.00220878
      coeff(12, 1) =  0.55161620
      coeff(12, 2) = -0.04071620
      coeff(12, 3) = -0.01822780
      coeff(12, 4) =  0.10359240
      coeff(12, 5) =  0.24491063
      coeff(12, 6) =  0.00285987
      coeff(12, 7) =  0.00422790
      coeff(12, 8) =  0.00407121
      coeff(12, 9) = -0.00045380
      coeff(12,10) = -0.00146840
      coeff(12,11) = -0.00608740
      coeff(12,12) =  0.00187184
      coeff(12,13) =  0.00199230
      coeff(12,14) = -0.01458080
      coeff(12,15) = -0.04651480
      coeff(12,16) = -0.00103380
      coeff(12,17) = -0.00224000
      coeff(12,18) = -0.00109160
      coeff(12,19) =  0.00074973
      coeff(13, 1) =  0.67369793
      coeff(13, 2) = -0.00747010
      coeff(13, 3) = -0.04190240
      coeff(13, 4) =  0.02388243
      coeff(13, 5) =  0.13744754
      coeff(13, 6) =  0.00223426
      coeff(13, 7) =  0.00279771
      coeff(13, 8) =  0.00605931
      coeff(13, 9) = -0.00144370
      coeff(13,10) = -0.00151650
      coeff(13,11) = -0.00385710
      coeff(13,12) = -0.00620850
      coeff(13,13) =  0.00184524
      coeff(13,14) = -0.01126230
      coeff(13,15) = -0.02505730
      coeff(13,16) = -0.00075940
      coeff(13,17) = -0.00146740
      coeff(13,18) =  0.00142162
      coeff(13,19) =  0.00275213
      coeff(14, 1) =  0.65043818
      coeff(14, 2) = -0.00626730
      coeff(14, 3) = -0.02744980
      coeff(14, 4) =  0.08847391
      coeff(14, 5) =  0.16448990
      coeff(14, 6) =  0.00098788
      coeff(14, 7) =  0.00209113
      coeff(14, 8) =  0.00376914
      coeff(14, 9) = -0.00137280
      coeff(14,10) =  0.00063718
      coeff(14,11) = -0.00626790
      coeff(14,12) = -0.00473840
      coeff(14,13) =  0.00404005
      coeff(14,14) = -0.00997580
      coeff(14,15) = -0.03069250
      coeff(14,16) = -0.00043140
      coeff(14,17) = -0.00210410
      coeff(14,18) = -0.00083370
      coeff(14,19) =  0.00182356
      coeff(15, 1) =  0.77184789
      coeff(15, 2) = -0.03865610
      coeff(15, 3) = -0.04342940
      coeff(15, 4) = -0.03465390
      coeff(15, 5) =  0.21522972
      coeff(15, 6) =  0.00393668
      coeff(15, 7) =  0.00409718
      coeff(15, 8) =  0.00543320
      coeff(15, 9) = -0.00118130
      coeff(15,10) =  0.00001733
      coeff(15,11) =  0.00349585
      coeff(15,12) =  0.00013448
      coeff(15,13) = -0.00067790
      coeff(15,14) = -0.01977780
      coeff(15,15) = -0.03873230
      coeff(15,16) = -0.00097200
      coeff(15,17) = -0.00200060
      coeff(15,18) =  0.00151529
      coeff(15,19) =  0.00269907
      coeff(16, 1) =  0.77682841
      coeff(16, 2) = -0.02884030
      coeff(16, 3) = -0.03734350
      coeff(16, 4) =  0.03387784
      coeff(16, 5) =  0.26006713
      coeff(16, 6) =  0.00155403
      coeff(16, 7) =  0.00309614
      coeff(16, 8) =  0.00338795
      coeff(16, 9) = -0.00062250
      coeff(16,10) =  0.00132589
      coeff(16,11) =  0.00080919
      coeff(16,12) =  0.00325629
      coeff(16,13) =  0.00538607
      coeff(16,14) = -0.01955600
      coeff(16,15) = -0.05034970
      coeff(16,16) = -0.00123340
      coeff(16,17) = -0.00247470
      coeff(16,18) = -0.00076300
      coeff(16,19) =  0.00120785

      RETURN
      END
