C This file is part of the ESP-r system.
C Copyright Energy Systems Research Unit, University of
C Strathclyde, Glasgow Scotland, 2001.

C ESP-r is free software.  You can redistribute it and/or
C modify it under the terms of the GNU General Public
C License as published by the Free Software Foundation
C (version 2 orlater).

C ESP-r is distributed in the hope that it will be useful
C but WITHOUT ANY WARRANTY; without even the implied
C warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
C PURPOSE. See the GNU General Public License for more
C details.

C egeometry.f provides the following facilities:
C  EGOMIN:  Reads legacy zone geometry data as ASCII strings, with or without
C           range checking and printed summary.
C  EMKGEO:  Write an annotated geometry file (GEN type) based
C           on common blocks G0 G1 G2 G2OPT G4
C  GEOREAD: Reads V1.1 zone geometry data as ASCII strings, with or without
C           range checking and printed summary.
C GEOWRITE: Write a geometry file (GEN V1.1 type) based on
C           information currently held in common blocks G0 G1 G4.
C GEOWRITE2: Write a geometry file (GEN V1.1 type) based on whole model commons.
C  ERECC:   Converts REC (rectilinear) into a G1 description.
C  ERECC3A: Converts REC (rectilinear with 2 rotations) into a G1 common block.
C  CNVBLK:  Converts REC (rectilinear) into GB1 common block.
C  CNVBLK3A:Converts REC (rectilinear with 2 rotations) into GB1 common block.
C  CNVVISP: converts a six sided visual 'visp' into GB1 common block.
C  EREGC:   Converts REG into a GEN description.
C  ESCROT:  Rotate a zone by ANG degrees around point x1,y1.
C  SURINFO: Display surface details and attributes in a tabular format.
C  ZINFO:   Returns descriptive information zone geometry.
C  ZGUPDATE: takes the geometry common G1 and updates G7 & PREC2
C  SURLEHI: Determines 2D bounding box size of surface (G1 required).
C  ZSURLEHI: Determines 2D bounding box size give a zone & surface.
C  SUREDGEADJ: detects polygon edge associations (how many assoc. surfs etc.)
C  SURREL: checks each surface I in a zone and relates it to
C          another surface J in that zone.
C  zdata:   scan zones to build master geometry structures.
C  FILSUR:  Fills the common block G5 with default assumptions.
C  INSINFO: English description of default insolation options.
C  VERINFO: Prints vertex/surface list summaries.
C  SURLIST: takes an list of surfaces (lstsf) returns descriptive
C           string (sdescr) to be used in headers.
C  MKVOBJILIST: Populates common VOBJILIST.


C ******************** EGOMIN
C Reads legacy zone geometry data from a user-constructed data
C file (LGEOMF). There are three levels of geometry data input corresponding
C to RECtangular (REC), REGular (REG) and GENeral (GEN) shaped zones.
C If IR=1 then range checking is enabled, otherwise only minimal checking
C is performed on the data as read in.

C REC - Each of the 4 component walls are rectangular, vertical and of
C       equal height. Ceiling and floor are rectangular and horizontal.

C REG - each of 'N' zone walls are rectangular, vertical and of equal
C       height. Ceiling and floor are horizontal polygons.

C GEN - any polyhedral-shaped zone comprised of 'M' planes.

C Zone geometry input requirements vary depending on the zone shape
C type but in each case is defined relative to some aebitrary site
C cartesian coordinate system. Geometry requirements are:

C REC - the X, Y and Z coordinates of the 'bottom left-hand' corner
C       (when viewed from the south), the length (from this point
C       towards east), width and height of the zone (all internal
C       dimensions) and the rotation angle (the angle between the
C       'length' side and east - anticlockwise +ve).

C REG - the number of walls, the X and Y coordinates of the bottom
C       corner of each wall in an anticlockwise direction, the floor
C       and ceiling heights and the rotation angle from the site
C       X-axis to east (anticlockwise +ve).

C GEN - the total number of vertices in the body, the number of
C       surfaces, the X, Y and Z coordinates of each vertex (in any
C       convenient order), the number of vertices in each surface,
C       an associated ordered list of the vertex numbers in an
C       anticlockwise order which comprise each surface and the
C       rotation angle from the site X-axis to east (anticlockwise
C       +ve).

C For GEN type input the 'G1' common block is filled immediately.
C With REC and REG type input the simpler input data is read and
C passed to subroutines ERECC and EREGC respectively where
C conversion to the 'G1' format takes place.

C Legacy geometry files hold only a subset of boundary condition
C attributes and rely on the 'connection' c3 data.

C Return ier=3 if icomp is greater than MCOM.
C Return ier=4 if number of surfaces is greater than MS.

       SUBROUTINE EGOMIN(IUNIT,LGEOMF,ICOMP,IR,ITRC,ITRU,IER)
#include "building.h"

C geometry.h provides commons G0/G2/G4/prec2/prec16/prec17/precz/c20.
#include "geometry.h"
#include "esprdbfile.h"
#include "material.h"
#include "espriou.h"

      integer lnblnk  ! function definition
      integer ISN
      COMMON/OUTIN/IUOUT,IUIN,IEOUT

C Include ishdirc common in legacy file read so that some inference
C can be made in case the data is saved to version 1.1 file.

C << add logic to check for existance of obstructions file and
C << shading file in subsequent read and then update the values
C << as appropriate

C SNAME (12 char) - surface name attribute. Other G5 common block
c variables are defined in geometry.h
C ZBASEA    - area of base (m^2)
C IBASES    - surfaces (up to MBL) associated with base area. If all 0 then
C             base area has been supplied by the user.

C ZNAME (12 char) - the zone name (from geometry.h).
C ZDESC (64 char) - zone notes (from geometry.h).

C zbasea(),ibases(),iuzbasea(),izbaselist() are defined in geometry.h

C Save total transparent surface area for surfaces connected to external BC.
      common/PREC18/ZTRANA(MCOM)
      COMMON/C3/IC1(MCON),IE1(MCON),ICT(MCON),IC2(MCON),IE2(MCON)
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

      COMMON/GR3D07/Y0S(MS),Y0SS(MSSZ),Y0SE(MSEZ)

      DIMENSION XX(MS),YY(MS),IVA(MS)

      CHARACTER LGEOMF*72,WORD*20,loutstr*248,lkoutstr*1000
      CHARACTER tmpvfc*4,tmpsot*32,tother*12
      character ZN*12,phrase*64,outs*124,outs2*124
      character OPT*32 ! for use with surface optical attributes
      character sbound_ty*12,sbound_c2*6,sbound_e2*6
      logical checkbase
      logical havec24
      real XO1,YO1,ZO1,DX1,DY1,DZ1 ! to prevent name clash with geometry.h
      integer lsn   ! length of currentfile
      integer iflag ! for read errors
      integer iCountWords ! to test legacy items on surface line

C Variables used in parsing of space-separated geo files
      character*248 cLnWords(124)! Array containing contents of a line
      integer iEGetArrW          ! Returns space-separated contents of
                                 !   a line as an array
      integer iWordCount         ! Number of words on a line.

      integer iCtoI_err          ! Function parsing a string and returning an integer
      logical bParseError
      logical bH3KExtentionsActive

      IER=0    ! initial values of read state
      iflag=0

      ZN=' '
      phrase=' '
      checkbase=.false.
      havec24=.false.

C Initialise geometry data file. and set currentfile.
      CALL EFOPSEQ(IUNIT,LGEOMF,1,IER)
      IF(IER.LT.0)THEN
        write(outs,'(3a)') 'Geometry file ',LGEOMF(1:lnblnk(LGEOMF)),
     &      ' could not be opened.'
        call edisp(itru,outs)
        IER=1
        RETURN
      endif
      write(currentfile,'(a)') LGEOMF(1:lnblnk(LGEOMF))

C Check that the zone index is within complexity limits.
      if(icomp.gt.MCOM)then
        write(outs,'(a)') 'EGOMIN: zone index beyond range...'
        lsn=MIN0(lnblnk(currentfile),110)
        write(outs2,'(2a)') 'in: ',currentfile(1:lsn)
        call edisp(iuout,outs)
        call edisp(iuout,outs2)
        IER=3
        CALL ERPFREE(IUNIT,ios)
        RETURN
      endif

C Read lines from file, if one item then assumed to
C be CTYPE, if 2 then the second is the zone name which
C is tested for illegal characters, if '*Geometry 1.1'
C the code should use georead to scan the file so return
C ier as a 3.
      call lstripc(IUNIT,loutstr,99,ND,1,'geo line 1',IER)
      IF(IER.NE.0)goto 1001
      if(loutstr(1:13).eq.'*Geometry 1.1')then
        K=13
        CALL EGETW(loutstr,K,WORD,'W','CTYPE',IFLAG)
        write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
        zname(ICOMP)=' '
        CALL EGETW(loutstr,K,WORD,'W','Z name',IFLAG)
        ZN=WORD(1:12)
        call st2name(ZN,zname(ICOMP))
        lnzname(ICOMP)=lnblnk(zname(ICOMP))  ! update string length
        gversion(icomp) = 1.1   ! set the version number.
        CALL ERPFREE(IUNIT,ios)
        write(outs,'(3a)') 'Geometry file ',LGEOMF(1:lnblnk(LGEOMF)),
     &      ' was newer format than egeomin can read.'
        call edisp(itru,outs)
        IER=2
        RETURN
      else

C We can deal with this format.
        K=0
        IF(ND.EQ.1)THEN
          CALL EGETW(loutstr,K,WORD,'W','CTYPE',IFLAG)
          write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
          zname(ICOMP)=' '; lnzname(ICOMP)=0
          zdesc(ICOMP)=' '
        ELSEIF(ND.EQ.2)THEN
          CALL EGETW(loutstr,K,WORD,'W','CTYPE',IFLAG)
          write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
          CALL EGETW(loutstr,K,WORD,'W','Z name',IFLAG)
          ZN=WORD(1:12)
          call st2name(ZN,zname(ICOMP))
          lnzname(ICOMP)=lnblnk(zname(ICOMP))  ! update string length
          zdesc(ICOMP)=' '
        ELSEIF(ND.gt.2)THEN
          CALL EGETW(loutstr,K,WORD,'W','CTYPE',IFLAG)
          write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
          CALL EGETW(loutstr,K,WORD,'W','Z name',IFLAG)
          ZN=WORD(1:12)
          call st2name(ZN,zname(ICOMP))

          lnzname(ICOMP)=lnblnk(zname(ICOMP))  ! update string length
          call egetrm(loutstr,K,phrase,'W','Z description',IER)
          ZDESC(ICOMP)=phrase
        ENDIF
        IF(IFLAG.NE.0)GOTO 1001
        gversion(icomp) = 1.0   ! set the version number.
      endif

C The file format is prior to 1.1 so continue scanning.
C Fill in a default zone name and description if blank.
      if(zname(ICOMP)(1:2).EQ.'  '.OR.
     &   zname(ICOMP)(1:7).EQ.'UNKNOWN')then
        IF(ICOMP.LE.9)WRITE(zname(ICOMP),'(A5,I1)')'Zone-',ICOMP
        IF(ICOMP.GT.9)WRITE(zname(ICOMP),'(A5,I2)')'Zone-',ICOMP
        IF(ICOMP.GT.99)WRITE(zname(ICOMP),'(A5,I3)')'Zone-',ICOMP
        lnzname(ICOMP)=lnblnk(zname(ICOMP))  ! update string length
      endif
      if(zdesc(ICOMP)(1:1).EQ.' ')then
        write(zdesc(ICOMP),'(2a)')
     &    zname(ICOMP)(1:lnzname(ICOMP)),' describes a '
      endif

C Zone is of type REC, look for 7 items on one or two lines.
      IF(CTYPE(icomp)(1:3).EQ.'REC')THEN
        call lstripc(IUNIT,loutstr,99,ND,1,'line 2',IER)
        IF(IER.NE.0)goto 1001
        K=0
        CALL EGETWR(loutstr,K,XO1,0.,0.,'-','rec X origin',IER)
        CALL EGETWR(loutstr,K,YO1,0.,0.,'-','rec Y origin',IER)
        CALL EGETWR(loutstr,K,ZO1,0.,0.,'-','rec Z origin',IER)
        zorigin(icomp,1)=XO1; zorigin(icomp,2)=YO1
        zorigin(icomp,3)=ZO1

C If only 3 items on first line read another loutstr and try to continue.
        IF(ND.EQ.3) THEN
          K=0
          call lstripc(IUNIT,loutstr,0,ND,1,'line 2',IER)
          IF(IER.NE.0)goto 1001
        ENDIF
        CALL EGETWR(loutstr,K,DX1,0.,0.,'-','length',IER)
        CALL EGETWR(loutstr,K,DY1,0.,0.,'-','width',IER)
        CALL EGETWR(loutstr,K,DZ1,0.,0.,'-','height',IER)
        zsize(icomp,1)=DX1; zsize(icomp,2)=DY1
        zsize(icomp,3)=DZ1
        CALL EGETWR(loutstr,K,AR1,-360.,360.,'W','rotation angle',IER)
        nbwalls(ICOMP)=4        ! there are 4 walls in box

C Now convert to a gen description. Pass in critical dimensions
C and expect the data back via common G1.
        CALL ERECC(XO1,YO1,ZO1,DX1,DY1,DZ1,AR1)
        NZSUR(ICOMP)=NSUR
        NZTV(ICOMP)=NTV
        CTYPE(icomp)='GEN '

C Zone is of type REG.
      ELSEIF(CTYPE(icomp)(1:3).EQ.'REG')THEN
        call lstripc(IUNIT,loutstr,4,ND,1,'line 2',IER)
        IF(IER.NE.0)goto 1001
        K=0
        CALL EGETWI(loutstr,K,NW,3,MS-2,'W','no of walls',IER)
        nbwalls(ICOMP)=NW
        CALL EGETWR(loutstr,K,Z1,0.,1000.,'W','floor height',IER)
        CALL EGETWR(loutstr,K,Z2,Z1,1000.,'F','ceiling ht',IER)
        zorigin(icomp,1)=Z1
        zorigin(icomp,2)=Z2
        zorigin(icomp,3)=0.0
        zsize(icomp,1)=0.0; zsize(icomp,2)=0.0; zsize(icomp,3)=0.0
        CALL EGETWR(loutstr,K,AR1,-360.,360.,'W','rot angle',IER)

C Read base vertex data.
        DO 60 IW=1,NW
          call lstripc(IUNIT,loutstr,2,ND,1,'vertex data',IER)
          IF(IER.NE.0)goto 1001
          K=0
          CALL EGETWR(loutstr,K,XX(IW),0.,0.,'-','Base X',IER)
          CALL EGETWR(loutstr,K,YY(IW),0.,0.,'-','Base Y',IER)
   60   CONTINUE

C Now convert to GEN body type and place via common block G1.
        CALL EREGC(NW,Z1,Z2,XX,YY)
        IF(AR1.LT.-.01.OR.AR1.GT..01)then
          x1=X(1)
          y1=Y(1)
          CALL ESCROT(AR1,x1,y1)  ! Apply rotation to G1 commons.
        endif
        NZSUR(ICOMP)=NSUR
        NZTV(ICOMP)=NTV
        CTYPE(icomp)='GEN '

C Zone is of type GEN.
      ELSEIF(CTYPE(icomp)(1:3).EQ.'GEN')THEN
        call lstripc(IUNIT,loutstr,3,ND,1,
     &       '# of verticies, surfices, rotation angle',IER)
        IF(IER.NE.0)goto 1001
        K=0

C Read vertices.
        CALL EGETWI(loutstr,K,NTV,4,MTV,'F','# of vertices',IER)

C Read surfaces
C (Shouldn't four planar surfaces be required to fully-bound a zone?)
C << need to gracefully fail is geometry is overly complex >>
        CALL EGETWI(loutstr,K,NSUR,3,MS,'F','# of surfaces',IER)

C Check that the number of surfaces is within complexity limits.
        if(NSUR.gt.MS)then
          write(outs,'(3a)') 'EGOMIN: surfaces beyond range...',
     &    loutstr(1:50),'...'
          lsn=MIN0(lnblnk(currentfile),110)
          write(outs2,'(2a)') 'in: ',currentfile(1:lsn)
          call edisp(iuout,outs)
          call edisp(iuout,outs2)
          IER=4
          CALL ERPFREE(IUNIT,ios)
          RETURN
        endif

        nbwalls(ICOMP)=NSUR   ! remember so can work with META
        NZSUR(ICOMP)=NSUR
        NZTV(ICOMP)=NTV
        zorigin(icomp,1)=0.0; zorigin(icomp,2)=0.0
        zorigin(icomp,3)=0.0
        zsize(icomp,1)=0.0; zsize(icomp,2)=0.0; zsize(icomp,3)=0.0

C Read rotation angle.  If there are enough items on the line also
C look for the rotation point.
        CALL EGETWR(loutstr,K,AR1,-360.,360.,'W','rotation angle',IER)
        if(ND.gt.3)then
          CALL EGETWR(loutstr,K,AR1,0.,0.,'-','rotation point X',IER)
          CALL EGETWR(loutstr,K,AR1,0.,0.,'-','rotation point Y',IER)
        endif

C Read each vertex data line, strip any comments, see if 3 items and
C place in X(),Y(),Z().
        DO 62 I=1,NZTV(ICOMP)
          call lstripc(IUNIT,loutstr,3,ND,1,'vertex data',IER)
          IF(IER.NE.0)goto 1001
          K=0
          CALL EGETWR(loutstr,K,X(I),0.,0.,'-','X coord',IER)
          CALL EGETWR(loutstr,K,Y(I),0.,0.,'-','Y coord',IER)
          CALL EGETWR(loutstr,K,Z(I),0.,0.,'-','Z coord',IER)
          szcoords(ICOMP,I,1)=X(I)
          szcoords(ICOMP,I,2)=Y(I)
          szcoords(ICOMP,I,3)=Z(I)

   62   CONTINUE

C Read vertex list for each surface, strip comments, begin by finding
C the number of expected vertices (first item on list). Vertex
C list can be quite long so use stripc1k.
        DO 10 I=1,NZSUR(ICOMP)
          call stripc1k(IUNIT,lkoutstr,99,ND,1,'vertex list',IER)
          IF(IER.NE.0)goto 1001
          IF(ND.GE.4)THEN
            K=0
            CALL EGETWI(lkoutstr,K,J,3,MV,'F','nb assoc vertices',IERV)
            NVER(I)=J
            isznver(icomp,i)=J

C Now proceed to read vertices on one or more lines.
            DO 12 KV=1,NVER(I)
              CALL EGETWI(lkoutstr,K,IVAL,0,MTV,'F','vertex',IERV)
              IF(IERV.NE.0) THEN
                call edisp(ITRU,' reading continuation line...')
                call stripc1k(IUNIT,lkoutstr,0,ND,0,'vertex XYZ',IER)
                IF(IER.NE.0)goto 1001
                K=0
                CALL EGETWI(lkoutstr,K,IVAL,0,MTV,'F','vertex',IERV)
              ENDIF
              IF(IERV.NE.0) GOTO 1001
              JVN(I,KV)=IVAL
              ISZJVN(icomp,i,kv)=IVAL
   12       CONTINUE
          ENDIF
   10   CONTINUE

C Rotate if required.
        if(AR1.LT.-.01.OR.AR1.GT..01)then
          x1=X(1)
          y1=Y(1)
          CALL ESCROT(AR1,x1,y1)
        endif

      ELSE

C Unrecognised keyword.
        write(outs,'(2a)') ' Geometry shape type illegal in ',
     &    LGEOMF(1:lnblnk(LGEOMF))
        call edisp(itru,outs)
        IER=1
        CALL ERPFREE(IUNIT,ios)
        RETURN
      endif

C Read line of unused indices (one per surface) if any index is
C non zero then it represents a depreciated default window.
      IRVA=NZSUR(ICOMP)
      CALL EGETWIA(IUNIT,IVA,IRVA,0,MG,'F','def window list',IER)

C Read the window dimension details and advise user (if IVA non-zero).
      DO 16 KS=1,NZSUR(ICOMP)
        IF( IVA(KS).GT.0 )THEN
          DO 18 KW=1,IVA(KS)
            call lstripc(IUNIT,loutstr,99,ND,1,'window data',IER)
            IF(IER.NE.0)goto 1001
            IF(ND.GE.4)THEN
              K=0
              CALL EGETWR(loutstr,K,VALX,0.,0.,'-','win X off',IER)
              CALL EGETWR(loutstr,K,VALZ,0.,0.,'-','win Z off',IER)
              CALL EGETWR(loutstr,K,VALW,0.,100.,'W','win width',IER)
              CALL EGETWR(loutstr,K,VALH,0.,100.,'W','win ht',IER)
              call usrmsg(
     &  'A default window description has been detected. This is not',
     &  'supported, note its demensions and make an equiv surface.','W')
              write(outs,'(a,f6.3,a,f6.3,a,f6.3,a,f6.3)')
     &          'Note def win: X off is ',VALX,' Z off is ',VALZ,
     &          ' width is ',VALW,' height is ',VALH
              call edisp(itru,outs)
            ENDIF
   18     CONTINUE
        ENDIF
   16 CONTINUE

C Read surface indentations (one per surface) associated
C with 3D conduction.
      IRVA=NZSUR(ICOMP)
      CALL EGETWRA(IUNIT,YY,IRVA,0.,1.,'W','surfaces recess',IER)
      DO 22 KS=1,NZSUR(ICOMP)
        Y0S(KS)=YY(KS)
   22 CONTINUE

C Read default insolation surface numbers. NN = insolation defining
C index:  1; one plane, 2; two planes, 3; all planes (diffuse).
C IDPN defines the default plane numbers.
      call lstripc(IUNIT,loutstr,4,ND,1,'insolation data',IER)
      IF(IER.NE.0)goto 1001
      K=0
      CALL EGETWI(loutstr,K,IV,1,3,'W','Def insol index',IER)
      NDP(ICOMP)=IV
      CALL EGETWI(loutstr,K,IV,0,NSUR,'W','1st recv surf',IER)
      IDPN(ICOMP,1)=IV
      CALL EGETWI(loutstr,K,IV,0,NSUR,'W','2nd recv surf',IER)
      IDPN(ICOMP,2)=IV
      CALL EGETWI(loutstr,K,IV,-1,NSUR,'W','3rd insol data',IER)
      IDPN(ICOMP,3)=IV
      IF(IR.EQ.1)THEN
        DO 40 I=1,3
          IX=0
          IF(NDP(ICOMP).EQ.1.AND.I.EQ.1)IX=1
          IF(NDP(ICOMP).EQ.2.AND.I.LE.2)IX=1
          if(IDPN(ICOMP,I).NE.-1.AND.IDPN(ICOMP,I).NE.0)then
            if(IDPN(ICOMP,I).LT.IX.OR.IDPN(ICOMP,I).GT.NSUR)then
              CALL USRMSG(' Nonexistent insolation plane!',loutstr,'W')
            endif
          endif
   40   CONTINUE
      ENDIF

C Surface attributes:
C 1st item - surface number ISN,
C 2nd item - 12 char name SNAME()
C 3rd item is a string (32 char) indicating whether the surface is
C     OPAQ/OPAQAUE (opaque) TRAN/optical set name or CFC.
C 4th item is a string (4 char) indicating whether the surface
C     is to be counted as a FLOR (floor), VERT (wall), CEIL (ceiling).
C 5th item is the name of the construction ( 32 char).
C 6th item is a 12 char string indicating connection type.
C     See notes about scanning conversion.
C If end of file then assume no names, do not error.

C Begin with default assumptions for each surface then overwrite
C this if user supplied information exists.

C Instantiate defaults for the surfaces to be read in.
      CALL FILSUR(ICOMP,0)

      icn1=izstocn(icomp,1)
      if(icn1.gt.0)then
      endif

      DO 42 I=1,NZSUR(ICOMP)

C Check and see if izstocn returns non-zeros and set havec24.
        icn1=izstocn(icomp,I)
        if(icn1.gt.0)then
          havec24=.true.
        else
          havec24=.false.
        endif
        call lstripc(IUNIT,loutstr,0,ND,0,'surface attributes',IER)
        ni = iCountWords(loutstr)
        IF(IER.EQ.2)THEN

C End of file sensed, however this is not an error at this point.
          IER=0
          GOTO 44
        ENDIF
        IF(IER.EQ.1)goto 1001


C Read surface attributes. Format depends on whether HOT3000 exensions
C are active (check bH3KExtentionsActive()).

        SurfFormat1: if ( .not. bH3KExtentionsActive() ) then

C If there are 6 tokens on the line then there are no added spaces
C in any of the names and we can just use the space separation.
          if(ni.eq.6)then
            K=0
            CALL EGETWI(loutstr,K,ISN,1,NSUR,'F','surface no',IER)
            IF(IER.NE.0)GOTO 44
            CALL EGETW(LOUTSTR,K,WORD,'W','surface name',IER)
            write(SNAME(ICOMP,ISN),'(a)') WORD(1:lnblnk(WORD))
            CALL EGETW(LOUTSTR,K,WORD,'W','surface optics',IER)
            write(tmpsot,'(a)') WORD(1:lnblnk(WORD))
            CALL EGETW(LOUTSTR,K,WORD,'W','surface vfc',IER)
            write(tmpvfc,'(a)') WORD(1:lnblnk(WORD))
            CALL EGETW(LOUTSTR,K,WORD,'W','surface construction',IER)
            write(SMLCN(ICOMP,ISN),'(a)') WORD(1:lnblnk(WORD))
            CALL EGETW(LOUTSTR,K,WORD,'W','surface construction',IER)
            write(tother,'(a)') WORD(1:lnblnk(WORD))

          else
C Assume that line is in fixed-format, and that attributes are defined
C according to this convention:
C Read in the various attributes as in the following line.  With this
C method both construction and surface names can have blanks.
C        1         2         3         4         5         6         7
C23456789012345678901234567890123456789012345678901234567890123456789012
C 1, Surf-1        OPAQ  VERT  UNKN         EXTERIOR
            K=0
            CALL EGETWI(loutstr,K,ISN,1,NSUR,'F','surface no',IER)
            IF(IER.NE.0)GOTO 44

C Recover Surface name.
            SNAME(ICOMP,ISN)=loutstr(6:17)

C Recover specified optics
            write(tmpsot,'(a)') loutstr(20:23)

C Interpret positional portion of the string.
            tmpvfc=loutstr(26:29)

C Interpret the construction name.
            write(SMLCN(ICOMP,ISN),'(a)') loutstr(32:43)

C Recover "to-other' parameter:
            write(tother,'(a)') loutstr(45:56)
          endif
        else
C Space-separated format (Used by HOT3000)
C Read surface attibutes. iEGetArrW returns the contents of a line
C as an array of comma, tab or space-sparated variables. [The old
C approach of using a space-sensitive format is too inflexible and
C error prone. Obstensibly, it was used to permit spaces in surface
C and consturction names, but it defies convention elsewhere in ESP-r.
C A (much) better approach would escape characters such as commas and
C spaces, as is done in the rest of the computing world.

          iWordCount = iEGetArrW ( loutstr, cLnWords )

C Line should contain a number, name, and four keywords ( e.g. OPAQ,
C VERT, UNKN, EXTERIOR )        svn
          read_attribute: if ( iWordCount .ne. 6 ) then
            write (outs, '(A,I3,A)')
     &        'Could not intrepret attributes for surface', I, '.'
              call edisp(itru,outs)
              IER = 1
          else

C Read surface number from iWordCount

            ISN = iCtoI_err ( cLnWords(1),
     &                        "Surface position in .geo file",
     &                         bParseError )
            if (bParseError) IER = 1

C Recover Surface name
            SNAME(ICOMP,ISN) = cLnWords(2)(1:12)

C Recover optics specification
            tmpsot = cLnWords(3)(1:4)

C Interpret positional portion of the string.
            tmpvfc=cLnWords(4)(1:4)

C Interpret the construction name.
            SMLCN(ICOMP,ISN) = cLnWords(5)(1:32)

C Recover "to-other' parameter:
            tother = cLnWords(6)(1:12)

          endif read_attribute

        endif SurfFormat1

C Now we've parsed the line. Process and error-trap the contents

C The value of I and ISN should be the same. If not warn.
        if(I.ne.ISN)then
           if(ITRC.ne.0) call edisp(itru,
     &       'Disagreement between surface index and position in list.')
        endif

         icn=izstocn(icomp,isn)

C Process positional attribute
        IF(tmpvfc.EQ.'VERT'.OR.tmpvfc.EQ.'SLOP')then
          write(SVFC(ICOMP,ISN),'(a)') tmpvfc
        ELSEIF(tmpvfc(1:4).EQ.'CEIL')then
          write(SVFC(ICOMP,ISN),'(a)') 'CEIL'
          izsceil(icomp)=ISN   ! identify as a ceiling
        ELSEIF(tmpvfc(1:4).EQ.'FLOR')then
          write(SVFC(ICOMP,ISN),'(a)') 'FLOR'
          izsfloor(icomp)=ISN ! identify as a floor
        ENDIF


C Check if there is a matching MLC name for given construction and which one it is.
        ICF=-1
        if(NMLC.gt.0)then
          lstr=lnblnk(SMLCN(icomp,ISN))
          DO 20 IC=1,NMLC
           if(mlcname(ic)(1:lnmlcname(ic)).EQ.
     &        SMLCN(icomp,ISN)(1:lstr))ICF=IC
  20      CONTINUE
        endif

C Now that we know the construction try and work out the optics.
        TMPSOT_Value: if(tmpsot(1:4).eq.'OPAQ')then
          write(SOTF(icomp,ISN),'(a)') 'OPAQUE'  ! write OPAQUE
        elseif(tmpsot(1:4).eq.'CFC ')then
          write(SOTF(icomp,ISN),'(a)') 'CFC '    !write CFC
        elseif(tmpsot(1:4).eq.'CFC2')then
          write(SOTF(icomp,ISN),'(a)') 'CFC2'    !write CFC2
        elseif(tmpsot(1:4).ne.'OPAQ'.and.tmpsot(1:3).ne.'CFC')then

C We need to find the name of the optical set for the construction.
C If there is a matching MLC entry attempt to discover
C the name of the optical property. Otherwise leave SOFT as TRAN
          NMLC_GT_0: if(NMLC.gt.0)then
            ICF_GT_0: if(ICF.gt.0)then
              lno=lnblnk(mlcoptical(ICF))
              write(OPT,'(a)') mlcoptical(ICF)(1:lno)

            else

C No matching database record. Use if file specifies 'TRAN', use that.
C otherwise, use 'OPAQUE'
              user_specd_TRAN: if ( tmpsot(1:4).eq.'TRAN' ) then
C Use user-provided TRAN definition. Ensures that
C the model can be read even if it contains optical
C data that does not correspond with the optics databaes.
C [Note that several applications now stream ESP-r input
C files independently of the optical databases, and must
C be supported.]

                OPT='TRAN'

              else

C Definition is not 'TRAN' or 'OPAQ', default to OPAQUE.
C [An even better approach would produce a fatal error, since
C  the user --- or third-party app --- has used an unsupported
C  keyword.]
                OPT='OPAQUE'

              endif user_specd_TRAN

            endif ICF_GT_0

            write(SOTF(icomp,ISN),'(a)') OPT(1:lnblnk(OPT))
          else
            write(SOTF(icomp,ISN),'(a)') 'TRAN'
          endif NMLC_GT_0
        endif TMPSOT_Value

C Depending on what the 'other' column is fill boundary arrays.
C Note when scanning an older file we will not yet know the
C C3 based information go get from connectins list.
        if(tother(1:7).eq.'UNKNOWN')then
          zboundarytype(icomp,isn,1)=-1
          zboundarytype(icomp,isn,2)=0
          zboundarytype(icomp,isn,3)=0
        elseif(tother(1:8).eq.'EXTERIOR')then
          zboundarytype(icomp,isn,1)=0
          zboundarytype(icomp,isn,2)=0
          zboundarytype(icomp,isn,3)=0
        elseif(tother(1:9).eq.'ADIABATIC')then
          zboundarytype(icomp,isn,1)=5
          zboundarytype(icomp,isn,2)=0
          zboundarytype(icomp,isn,3)=0
        elseif(tother(1:7).eq.'SIMILAR')then

C In this case the 2nd and 3rd values are not held in a legacy geometry
C file. And if this is not the first scan of the geometry file then
C the C24 and C3 data structures will have data that can be used.
          if(havec24)then
            zboundarytype(icomp,isn,1)=1
            zboundarytype(icomp,isn,2)=IC2(icn)
            zboundarytype(icomp,isn,3)=IE2(icn)
          else
            zboundarytype(icomp,isn,1)=-1
            zboundarytype(icomp,isn,2)=0
            zboundarytype(icomp,isn,3)=0
          endif
        elseif(tother(1:8).eq.'CONSTANT')then

C In this case the 2nd and 3rd values are not held in a legacy geometry
C file. And if this is not the first scan of the geometry file then
C the C24 and C3 data structures will have data that can be used.
          if(havec24)then
            zboundarytype(icomp,isn,1)=2
            zboundarytype(icomp,isn,2)=IC2(icn)
            zboundarytype(icomp,isn,3)=IE2(icn)
          else
            zboundarytype(icomp,isn,1)=-1
            zboundarytype(icomp,isn,2)=0
            zboundarytype(icomp,isn,3)=0
          endif
        elseif(tother(1:8).eq.'BASESIMP')then
          if(havec24)then
            zboundarytype(icomp,isn,1)=6
            zboundarytype(icomp,isn,2)=IC2(icn)
            zboundarytype(icomp,isn,3)=IE2(icn)
          else
            zboundarytype(icomp,isn,1)=-1
            zboundarytype(icomp,isn,2)=0
            zboundarytype(icomp,isn,3)=0
          endif
        elseif(tother(1:6).eq.'GROUND')then
          if(havec24)then
            zboundarytype(icomp,isn,1)=4
            zboundarytype(icomp,isn,2)=IC2(icn)
            zboundarytype(icomp,isn,3)=IE2(icn)
          else
            zboundarytype(icomp,isn,1)=-1
            zboundarytype(icomp,isn,2)=0
            zboundarytype(icomp,isn,3)=0
          endif
        elseif(tother(1:7).eq.'ANOTHER')then
          if(havec24)then
            zboundarytype(icomp,isn,1)=3
            zboundarytype(icomp,isn,2)=IC2(icn)
            zboundarytype(icomp,isn,3)=IE2(icn)
          else
            zboundarytype(icomp,isn,1)=-1
            zboundarytype(icomp,isn,2)=0
            zboundarytype(icomp,isn,3)=0
          endif
        elseif(tother(1:9).eq.'IDENT_CEN')then
          if(havec24)then
            zboundarytype(icomp,isn,1)=7
            zboundarytype(icomp,isn,2)=IC2(icn)
            zboundarytype(icomp,isn,3)=IE2(icn)
          else
            zboundarytype(icomp,isn,1)=-1
            zboundarytype(icomp,isn,2)=0
            zboundarytype(icomp,isn,3)=0
          endif
        else

C Assume partition so setup zboundarytype arrays with this in mind.
          if(havec24)then
            zboundarytype(icomp,isn,1)=3
            zboundarytype(icomp,isn,2)=IC2(icn)
            zboundarytype(icomp,isn,3)=IE2(icn)
          else
            zboundarytype(icomp,isn,1)=-1
            zboundarytype(icomp,isn,2)=0
            zboundarytype(icomp,isn,3)=0
          endif
        endif

C Decode the zboundarytype into sother.
        call decode_zsbound(icomp,isn,sbound_ty,sbound_c2,sbound_e2)

C Usage is unknown so write place holder.
        write(suse(icomp,isn,1),'(a)') '-'
        write(suse(icomp,isn,2),'(a)') '-'

C Child/parent unknown in legacy file so write place holder.
        write(sparent(icomp,isn),'(a)') '-'

   42 CONTINUE

C Check if base area has been defined.
      call lstripc(IUNIT,loutstr,99,ND,0,'base',IER)
      IF(IER.EQ.2)THEN

C End of file sensed before zone base area was defined. Estimate.
        checkbase=.true.
        lastlist=0
        do 61 ibcount=1,MBL
          IBASES(ICOMP,ibcount)=0
  61    continue
        ZBASEA(ICOMP)=0.00
        IER=0
        GOTO 44
      ENDIF

C Gather information about surfaces associated with floor area.
      do 59 ibcount=1,MBL
        IBASES(ICOMP,ibcount)=0
  59  continue
      ZBASEA(ICOMP)=0.00
      K=0
      CALL EGETWI(loutstr,K,IB1,0,MS,'-','zn base 1st index',IER)
      CALL EGETWI(loutstr,K,IB2,0,MS,'-','zn base 2nd index',IER)
      CALL EGETWI(loutstr,K,IB3,0,MS,'-','zn base 3rd index',IER)
      CALL EGETWI(loutstr,K,IB4,0,MS,'-','zn base 4th index',IER)
      if(ND.eq.6)then
        CALL EGETWI(loutstr,K,IB5,0,MS,'-','zn base 5th index',IER)
      elseif(ND.gt.6)then
        CALL EGETWI(loutstr,K,IB5,0,MS,'-','zn base 5th index',IER)
        CALL EGETWI(loutstr,K,IB6,0,MS,'-','zn base 6th index',IER)
      endif
      CALL EGETWR(loutstr,K,VAL,0.,99999.,'W','zn base area m2',IER)

C If there is an 8th item on the line it is the value for iuzbasea
C (user might have selected the list of surfaces assoc with base).
      if(ND.eq.8)then
        CALL EGETWI(loutstr,K,IUB,0,2,'-','zn base area user flag',IER)
      else
        IUB=0
      endif
      lastlist=0
      IBASES(ICOMP,1)=IB1
      if(ibases(icomp,1).ne.0) lastlist=1
      IBASES(ICOMP,2)=IB2
      if(ibases(icomp,2).ne.0) lastlist=2
      IBASES(ICOMP,3)=IB3
      if(ibases(icomp,3).ne.0) lastlist=3
      IBASES(ICOMP,4)=IB4
      if(ibases(icomp,4).ne.0) lastlist=4
      IBASES(ICOMP,5)=IB5
      if(ibases(icomp,5).ne.0) lastlist=5
      IBASES(ICOMP,6)=IB6
      if(ibases(icomp,6).ne.0) lastlist=6

C If list is full of zeros and there is a no-zero user defined area
C set iuzbasea() to 1.
      if(ib1.eq.0.and.ib2.eq.0.and.ib3.eq.0.and.ib4.eq.0.and.ib5.eq.0
     &           .and.ib6.eq.0)then
        if(VAL.gt.0.00)then
          iuzbasea(icomp)=1
          ZBASEA(ICOMP)=VAL
        else
          iuzbasea(icomp)=0
          ZBASEA(ICOMP)=0.00
          if(ITRC.ne.0) call edisp(itru,
     &      'Zone base area description missing.')
        endif
        izbaselist(icomp)=lastlist  ! remember how many items in list.
      else
        if(iub.eq.0)then
          iuzbasea(icomp)=0
        elseif(iub.eq.2)then
          iuzbasea(icomp)=2
        endif
        ZBASEA(ICOMP)=VAL
        izbaselist(icomp)=lastlist  ! remember how many items in list.
      endif

C Now close geometry data file.
   44 CALL ERPFREE(IUNIT,ios)

C If base area has not yet been calculated, do this now and calculate
C transparent area for surfaces connected to external boundary conditions.
C Get surface areas via call to zinfo.
      if(ITRC.ne.0)then
        call zinfo(icomp,zoa,zvol,'-')
      else
        call zinfo(icomp,zoa,zvol,'q')
      endif
      vol(icomp)=zvol
      zonetotsurfacearea(icomp)=zoa
      ZTRANA(ICOMP)=0.
      lastlist=0
      do 43 ijj=1,NZSUR(ICOMP)
        if(checkbase)then
          if(SVFC(ICOMP,ijj)(1:4).eq.'FLOR')then
            if(SUSE(ICOMP,ijj,1)(1:5).eq.'FURNI')then
              continue  ! ignore horizontal surfaces marked as furniture.
            elseif(SUSE(ICOMP,ijj,1)(1:6).eq.'REVEAL')then
              continue  ! ignore horizontal surfaces marked as reveal.
            else

C If surface `flor` not included in the list add it.
              lastlist=lastlist+1
              if(lastlist.le.6)then
                IBASES(ICOMP,lastlist)=ijj
                ZBASEA(ICOMP)=ZBASEA(ICOMP)+SNA(ICOMP,ijj)
              endif
            endif
          endif
        endif
        if (SOTF(ICOMP,ijj)(1:4).ne.'OPAQ'.and.
     &      SOTF(ICOMP,ijj)(1:3).ne.'CFC'.and.
     &      zboundarytype(ICOMP,ijj,1).eq.0) then
          ZTRANA(ICOMP)=ZTRANA(ICOMP)+SNA(ICOMP,ijj)
        endif
  43  continue
      if(checkbase)then
        write(outs,'(a,f6.2,3a)') 'Base area estimated at ',
     &     ZBASEA(ICOMP),'m^2 for ',zname(icomp),
     &    ' (probably an older file).'
        call edisp(itru,outs)
        if(lastlist.gt.0)then
          iuzbasea(icomp)=0           ! signal area from orientation scan
          izbaselist(icomp)=lastlist  ! remember how many items in list.
        endif
      endif

C Leave it to calling code to report on contents of zone geometry
C as extended reporting needs to know of the context of the zone.
C Close geometry data file before exiting.
      CALL ERPFREE(IUNIT,ios)
      RETURN

 1001 write(outs,'(3a)') 'EGOMIN: conversion error in...',
     &  loutstr(1:50),'...'
      lsn=MIN0(lnblnk(currentfile),110)
      write(outs2,'(2a)') 'in: ',currentfile(1:lsn)
      call edisp(iuout,outs)
      call edisp(iuout,outs2)
      IER=1
      CALL ERPFREE(IUNIT,ios)
      RETURN

      END


C ************* EMKGEO
C Writes a legacy geometry file (GEN type) based on information
C currently held in common blocks G0 G1 G4.  It is assumed
C that this information has been checked.
C GENFIL is the name of the file to be written to (any existing file
C by this name is overwritten).

      SUBROUTINE EMKGEO(IFILG,GENFIL,ICOMP,iwf,IER)
#include "building.h"

C geometry.h provides commons G0/G1/G2/G4/prec17/precz/c20.
#include "geometry.h"

      integer lnblnk  ! function definition

C Parameters
      integer IFILG        ! file unit
      character GENFIL*72  ! file name
      integer ICOMP        ! the zone number
      integer iwf          ! 3 create/overwrite, 4 confirm before overwriting.
      integer IER          ! IER 0 OK IER 1 problem

      COMMON/C3/IC1(MCON),IE1(MCON),ICT(MCON),IC2(MCON),IE2(MCON)
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

C NOTE: G1 is only valid for the zone that has currently been
C scanned and is being worked on. Common block ZNDATA
C holds this information globally.

C << todo: emkgeo could use ZNDATA if there was a way for the
C << calling code to signal that ZNDATA could be used instead.

      COMMON/GR3D07/Y0S(MS),Y0SS(MSSZ),Y0SE(MSEZ)

C Dummy array for default windows.
      dimension NGLD(MS)

      CHARACTER louts*248,outs*124,lkouts*1000
      CHARACTER SO*15    ! to write out legacy
      logical newgeo  ! to use for testing if new/old geometry file.
      character msg*48

      IER=0

C Clear dummy array.
      do 42 i=1,NZSUR(icomp)
        NGLD(i)=0
  42  continue
      newgeo=.false.  ! assume older format geometry.

C Open any existing file by this name (ask user for confirmation to
C over-write) or create a new file.
      if(iwf.eq.4)then
        CALL EFOPSEQ(IFILG,GENFIL,4,IER)
      else
        CALL EFOPSEQ(IFILG,GENFIL,3,IER)
      endif
      IF(IER.LT.0)THEN
        IER=1
        RETURN
      ENDIF

      msg='first comment line'
      WRITE(IFILG,30,IOSTAT=ios,ERR=13)
     &  zname(ICOMP)(1:lnzname(ICOMP)),GENFIL(1:lnblnk(GENFIL))
  30  FORMAT('# geometry of ',a,' defined in: ',a)
      lz=lnzname(ICOMP)

C File is always a GEN type as this is representation in common.
      msg='2nd line starting with GEN'
      WRITE(IFILG,'(a3,2x,A,2x,a,2x,a)',IOSTAT=ios,ERR=13) 'GEN',
     &  zname(ICOMP)(1:lz),zdesc(ICOMP)(1:lnblnk(zdesc(ICOMP))),
     &  '# type, name, descr'

C Vertex coordinates.
      AR=0.0   ! Assume any prior rotations are implied in coordinates.
      msg='3rd line with nb of surfs and vert'
      WRITE(IFILG,'(I8,I8,F8.3,4x,A)',IOSTAT=ios,ERR=13)
     &  NZTV(ICOMP),NZSUR(ICOMP),AR,
     &  '# vertices, surfaces, rotation angle'
      WRITE(IFILG,32,IOSTAT=ios,ERR=13)
  32  FORMAT('#  X co-ord, Y co-ord, Z co-ord')
      DO 960 I = 1,NZTV(ICOMP)
        msg='vertex data line'
        WRITE(IFILG,33,IOSTAT=ios,ERR=13)X(I),Y(I),Z(I),I
  33    FORMAT(1X,3F12.5,'  # vert ',I3)
960   CONTINUE

C Vertex list. << convert from G1 >>
      WRITE(IFILG,34,IOSTAT=ios,ERR=13)
  34  FORMAT('# no of vertices followed by list of associated vert')
      DO 970 I = 1,NZSUR(ICOMP)
      msg='edge list data line'
      WRITE(IFILG,5650,IOSTAT=ios,ERR=13)NVER(I),(JVN(I,J),J=1,NVER(I))
5650    FORMAT(1X,102(I3,','))
970   CONTINUE

C Unused index. Generate packed strings to write out on as many lines as reqd.
      WRITE(IFILG,'(a)',IOSTAT=ios,ERR=13) '# unused index'
      itrunc=1
      ipos=1
      do while (itrunc.ne.0)
        call ailist(ipos,nzsur(icomp),ngld,MS,'C',lkouts,loutlen,itrunc)
        write(ifilg,'(1x,a)',IOSTAT=ios,ERR=14) lkouts(1:loutlen)
        ipos=itrunc+1
      end do

C Surfaces indentation (depreciated). Call to arlist should generate long lines of
C packed text. Code should be good for any number of surfaces.
      WRITE(IFILG,'(a)',IOSTAT=ios,ERR=13) '# surfaces indentation (m)'
      itrunc=1
      ipos=1
      do while (itrunc.ne.0)
        call arlist(ipos,nzsur(icomp),Y0S,MS,'C',lkouts,loutln,itrunc)
        write(ifilg,'(1x,a)',IOSTAT=ios,ERR=14) lkouts(1:loutln)
        ipos=itrunc+1
      end do

C Default internal insolation.
      msg='default insolation line'
      WRITE(IFILG,'(I5,3I4,4x,a)',IOSTAT=ios,ERR=13)NDP(ICOMP),
     &  (IDPN(ICOMP,J),J=1,3),'# default insolation distribution'

C Surface attributes.  Begin by inserting default information
C if for some reason there are blanks remaining.
      WRITE(IFILG,41,IOSTAT=ios,ERR=13)
  41  FORMAT('# surface attributes follow: ',/,
     &  '# id  surface      geom  loc/  construction environment',/,
     &  '# no  name         type  posn  name         other side')
      DO 102 I=1,NZSUR(icomp)
        icc=izstocn(icomp,i)
        IF(SNAME(icomp,i)(1:2).EQ.'  ')THEN
          IF(I.LE.9)WRITE(SNAME(icomp,i),76)I
          IF(I.GT.9)WRITE(SNAME(icomp,i),77)I
          IF(I.GT.99)WRITE(SNAME(icomp,i),78)I
   76     FORMAT('Surf-',I1)
   77     FORMAT('Surf-',I2)
   78     FORMAT('Surf-',I3)
        ENDIF

C This version of the geometry does not have a slot for the full
C name of the optical property so if SOTF is not OPAQUE then
C set it to TRAN within the legacy geometry file.
        IF(SOTF(icomp,i)(1:2).EQ.'  ')then
          SOTF(icomp,i)='OPAQUE'
        ELSEIF(SOTF(icomp,i)(1:4).EQ.'OPAQ'.OR.
     &         SOTF(icomp,i)(1:4).ne.'CFC '.and.
     &         SOTF(icomp,i)(1:4).EQ.'CFC2')then
          continue
        ELSE
          SOTF(icomp,i)='TRAN'
        endif
        IF(SUSE(icomp,i,1)(1:2).EQ.'  ')SUSE(icomp,i,1)='-'
        IF(SUSE(icomp,i,2)(1:2).EQ.'  ')SUSE(icomp,i,2)='-'

C Based on the current contents of zboundarytype create a buffer SO
C to write out.
        IF(zboundarytype(icomp,i,1).EQ.-1)then
          SO='UNKNOWN'
        ELSEIF(zboundarytype(icomp,i,1).EQ.0)then
          SO='EXTERIOR'
        ELSEIF(zboundarytype(icomp,i,1).EQ.5)then
          SO='ADIABATIC'
        ELSEIF(zboundarytype(icomp,i,1).EQ.1)then
          SO='SIMILAR'
        ELSEIF(zboundarytype(icomp,i,1).EQ.2)then
          SO='CONSTANT'
        ELSEIF(zboundarytype(icomp,i,1).EQ.6)then
          SO='BASESIMP'
        ELSEIF(zboundarytype(icomp,i,1).EQ.4)then
          SO='GROUND'
        ELSEIF(zboundarytype(icomp,i,1).EQ.7)then
          SO='IDENT_CEN'
        ELSEIF(zboundarytype(icomp,i,1).EQ.3)then
          SO='ANOTHER'
        ELSE
          if(IC2(icc).ne.0)then
            write(SO,'(a)') 'ANOTHER' 
          else
            SO='UNKNOWN'
          endif
        ENDIF
        msg='surface data line'
        lnsmlcn=lnblnk(SMLCN(icomp,i))  ! set to at least 12 char
        if(lnsmlcn.lt.12)lnsmlcn=12
        WRITE(IFILG,75,IOSTAT=ios,ERR=13)I,SNAME(icomp,i),
     &    SOTF(icomp,i)(1:4),SVFC(icomp,i),SMLCN(icomp,i)(1:lnsmlcn),SO
   75   FORMAT(I3,', ',A,2X,A,2X,A,2X,A,1X,A)
  102 CONTINUE

C Surfaces associated with base.
      msg='base area data line'
      WRITE(IFILG,'(a)',IOSTAT=ios,ERR=13)'# base'
      WRITE(IFILG,'(6i3,F9.2,i2)',IOSTAT=ios,ERR=13)IBASES(ICOMP,1),
     &  IBASES(ICOMP,2),IBASES(ICOMP,3),IBASES(ICOMP,4),
     &  IBASES(ICOMP,5),IBASES(ICOMP,6),ZBASEA(ICOMP),IUZBASEA(ICOMP)

   99 CALL ERPFREE(IFILG,ios)

C If file written with this source and the version number is 1.1
C then reset it to 1.0.
      call eclose(gversion(icomp),1.1,0.01,newgeo)
      if(newgeo) gversion(icomp) = 1.0
      RETURN

C Error messages.
   13 if(IOS.eq.2)then
        write(outs,'(2a)')'No permission to write ',msg(1:lnblnk(msg))
        CALL USRMSG(outs,GENFIL,'W')
      else
        write(outs,'(2a)')'File write error in ',msg(1:lnblnk(msg))
        CALL USRMSG(outs,GENFIL,'W')
      endif
      IER=1
      GOTO 99
   14 if(IOS.eq.2)then
        CALL USRMSG('No prmission to write array in ',GENFIL,'W')
      else
        CALL USRMSG('Long arrary write error in ',GENFIL,'W')
      endif
      IER=1
      GOTO 99

      END

C ******************** GEOREAD
C GEOREAD reads V1.1 zone geometry data (LGEOMF) from a user-constructed data
C file. Data input is based on an extended GENeral (GEN) shaped zones.
C Code adapted to scan potentially long lines in geometry file.

C GEN - any polyhedral-shaped zone comprised of 'M' planes.

C Zone geometry input requirements vary depending on the zone
C shape relative to some aebitrary site cartesian coordinate system.

C Geometry requirements are:

C GEN - the total number of vertices in the body, the number of
C       surfaces, the X, Y and Z coordinates of each vertex (in any
C       convenient order), the number of vertices in each surface,
C       an associated ordered list of the vertex numbers in an
C       anticlockwise order which comprise each surface and the
C       rotation angle from the site X-axis to east (anticlockwise
C       +ve).

C Return ier=3 if icomp is greater than MCOM.
C Return ier=4 if number of surfaces is greater than MS.

      SUBROUTINE GEOREAD(IUNIT,LGEOMF,ICOMP,IR,ITRU,IER)
#include "building.h"
#include "model.h"

C geometry.h provides commons G0/G1/G2/G4/G5/prec2/prec17/precz/c20/GS5/GS6.
#include "geometry.h"
#include "predefined.h"
#include "esprdbfile.h"
#include "material.h"

C espriou.h provides currentfile.
#include "espriou.h"


      integer lnblnk  ! function definition

C Parameters
      integer IUNIT  ! file unit to read
      integer ICOMP  ! zone number
      integer IR     ! IR 1 range checking is enabled, otherwise only minimal checking
      integer ITRU   ! file unit for feedback
      integer IER    ! zero is ok

      integer iuout,iuin,ieout
      common/OUTIN/IUOUT,IUIN,IEOUT

C iaplic(zone,1) toggle for shading; iaplic(zone,2) toggle for insolation.
C   toggle for assessment where:
C   1 is 'all_applicable', 0 is manual selection of surfaces.
C nsurfcalc nb of shaded surfaces, lstsfcalc() list of applicable surfaces.
C nsurfinso nb of insolation sources, isurfinso() list of insolation sources.
      common/ishdirec/iaplic(MCOM,2),nsurfcalc(MCOM),lstsfcalc(MCOM,MS),
     &     nsurfinso(MCOM),isurfinso(MCOM,MS)

C SNAME (12 char) - surface name attribute
C SOTF (32 char) - surface attribute indicating OPAQUE or an optical property.
C SVFC (4 char) - string indicating whether the surface is to be
C     counted as a FLOR (face up), VERT (wall), CEIL (face down),
C     SLOP (not vertical) for purposes of floor area and daylighting.
C
C SMLCN (32 char) surface construction attribute and other G5 common block
C variables are defined in geometry.h  Some additonal information on the
C array SUSE follows.

C    Surfaces which function as opaque portions of a facade are
C    marked as `WALL`. If they are vertical they will be composed
C    of a code complient construction. If they are not vertical
C    e.g. within 20 degree of vertical they could be composed of
C    a code-complent roofing material. The 2nd array index is not
C    used.

C    The first array index for the case of doors is as follows:
C    `DOOR` marks a normal door (inside or outside) which might be
C       altered to conform to building regulations.
C    'P-DOOR' marks a door for personel egress (e.g. a fire door) which
C       has specific code requirments
C    'H-DOOR' marks an entrance or other high-usage door.
C    'V-DOOR' marks a door used for vehicle access
C    The 2nd array index for the case of doors indicates the users
C    preference for treating air movement and the allowed tags are:
C    CLOSED UNDERCUT  OPEN  BIDIR

C    Surfaces marked 'REVEAL' helps to resolve whether a thermal
C    bridge may or may not apply.

C    Surfaces marked 'BALCONY' indicates a thermal bridge may need
C    to be defined.

C    Thie first array index for a frame associated with a door or window:
C    `FRAME` marks a frame associated with an inside door or glazing,
C    'F-FRAME' marks a frame within the facade which might need to be
C       of a paricular area or construction for code compliance.
C    The 2nd array index for frames is CLOSED CRACK VENT

C    The first array index for a glazing:
C    'C-WINDOW' is a code complient window (e.g.construction or size)
C    'D-WINDOW' is a display window which is not typically resized to
C       meet code compliance
C   .'S-WINDOW' is a code complient skylight (e.g.construction or size)
C    'WINDOW' any other window inside or exterior
C    The 2nd array index for frames is CLOSED CRACK OPEN SASH BIDIR

C    The first array index for a surface which the user wishes to
C    mark as an air supply or extract is 'GRILL':
C    The 2nd array index for grills is CLOSED CRACK INLET EXTRACT OPEN

C    The opaque portion of roofs are maked as 'ROOF' and the 2nd
C    array index is 'FLAT' (0-20 degree incline) or 'PITCHED'. Note
C    that a wall which is not vertical may be converted to a roof
C    construction type.

C    Surfaces which `are-not-there` e.g. fictitious are marked 'FICT'
C    and the 2nd array index can be: CLOSED CRACK OPEN BIDIR

C    Surfaces which represent internal or ground-contact floors are
C    marked with 'FLOOR' and the 2nd index is 'EXTGRND' or '- ' to
C    signal ground connection or internal.  A raised floor above a
C    ground connected crawl-space is ambiguious.

C    Surfaces which represent internal mass are marked 'FURNI' and
C    the 2nd array index is un-used.

C    Surface which are used as internal partitions are marked 'PARTN'
C    and the 2nd array index is un-used.

C    Interior surfaces can be marked as 'STRUC' for structural elements
C    such as beams and columns and the 2nd array index is un-used.

C    IT equipment such as TVs, computers and monitors can be marked as
C    'ITEQUIP'. The 2nd array index is un-used.

C    Interior plants can be marked as 'PLANTS'. The 2nd array index is
C    un-used.

C    Light fittings can be marked as 'FIXTURE'. The 2nd array index can
C    be 'IES', which directs radiance to use a specific set of IES data
C    specified in an *ies entry in the model configuration file.

C    otherwise  -,-  is the initial state for newly created surfaces
C    and surfaces which the user has not yet expressed an opinion. This
C    can also be used for internal surfaces which may not be of concern
C    within building code complence assessments.

C ZNAME (12 char) - the zone name (from geometry.h).
C ZDESC (64 char) - zone notes (from geometry.h).

C prec17 common described in geometry.h

C Save total transparent surface area for surfaces connected to external BC.
      common/PREC18/ZTRANA(MCOM)
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

C Thermal bridge common block is described in esruprj/edgeo.F
      integer nbrdg, ibrdg
      real psi,lngth,losspercent,totheatloss,thbrpercent
      real uavtotal
      common/THRBRDG/nbrdg(MCOM),psi(MCOM,16),lngth(MCOM,16),
     &               ibrdg(MCOM,16),losspercent(MCOM),totheatloss(MCOM),
     &               thbrpercent,uavtotal(MCOM)
      real bridgelen       ! Length (m) of potential thermal bridges in each zone.
      integer nbridgevt    ! Number of verticies associted with this bridge.
      integer bridgevlst   ! List of vertices associated with bridge.
      common/gbridge/bridgelen(MCOM,16),nbridgevt(MCOM,16),
     &  bridgevlst(MCOM,16,MV*2)
      common/THBRSCH/tbregime
      character tbregime*36

C This information was previously/also held in zone obstruction
C file. The intent is that older models when scanned can be upgraded
C to hold such information in the zone geometry file. Common GS5/GS6
C GS7/GS8 are defined in geometry.h

      DIMENSION IVA(MS)
      CHARACTER LGEOMF*72,WORD*32
      CHARACTER tmpvfc*4,tother*12,tother1*12,tother2*12
      character ZN*12,phrase*64,outs*124,outs2*124
      character loutstr*248,lkout*1000 ! longer line for vertex list
      character dstmp*24
      character sbound_ty*12,sbound_c2*6,sbound_e2*6
      character OPT*32 ! for use with optical attributes
      logical checkbase
      logical havec24
      logical optcorrected      ! set to true if optics corrected
      integer nbo               ! local value of nbobs
      integer nbv               ! local value of nbvis
      integer lsn               ! length of currentfile
      integer lno,lnow          ! length of optics name
      integer lstrs,lstrd,lnsn  ! length of mlc names
      real VX,VY,VZ             ! local values for XOB YOB ZOB
      integer iflag             ! for read error state

C Set initial values.
      IER=0
      iflag=0
      itrct=0
      ZN=' '
      phrase=' '
      checkbase=.false.
      optcorrected=.false.
      havec24=.false.  ! true if connections list has data
      NTV=0    ! counter for vertices, assume no vertices
      NEDGE=0  ! counter for surfaces in the edge list, assume no surfaces
      NS=0     ! temporary array for counting surfaces.
      NSUR=0   ! reset surface count
      nmrtsen=0 ! reset mrt sensor count

C Treatment of obstructions needs to take into account several
C factors. The version 1.1 geometry file might not yet have
C embedded legacy zone obstruction file entities. If there
C is still a legacy zone obstruction file associated with this
C zone then the calling code will need to do that instantiation
C in a subsequent call and that will be updating nbobs.
      if(iobs(icomp).eq.1)then
        continue          ! scan of zone obs file will reset
      else
        nbobs(icomp)=0    ! reset obstructions
      endif

C Initialise geometry data file. and set currentfile.
      CALL EFOPSEQ(IUNIT,LGEOMF,1,IER)
      IF(IER.LT.0)THEN
        write(outs,'(3a)') 'Geometry file ',LGEOMF(1:lnblnk(LGEOMF)),
     &      ' could not be opened.'
        call edisp(itru,outs)
        IER=1
        RETURN
      endif
      write(currentfile,'(a)') LGEOMF(1:lnblnk(LGEOMF))

C Check that the zone index is within complexity limits.
      if(icomp.gt.MCOM)then
        write(outs,'(a)') 'EGOMIN: zone index beyond range...'

        lsn=MIN0(lnblnk(currentfile),110)
        write(outs2,'(2a)') 'in: ',currentfile(1:lsn)
        call edisp(iuout,outs)
        call edisp(iuout,outs2)
        IER=3
        CALL ERPFREE(IUNIT,ios)
        RETURN
      endif

C Read header lines from file, the 1.1 version looks like:
C *Geometry 1.1,GEN,manager  # tag version, format, zone name (tbd allow spaces)
C An older file header looks like:
C # geometry of manager defined in: ../zones/manager.geo
C GEN  manager  manager describes cellular office  # type, name, descr
C      22      10   0.000    # vertices, surfaces, rotation angle
      CALL LSTRIPC(IUNIT,LOUTSTR,99,ND,1,'geo line 1',IER)
      IF(IER.NE.0)goto 1002
      if(LOUTSTR(1:13).eq.'*Geometry 1.1')then

C Decode first line of version 1.1 geometry file.
        K=13
        CALL EGETW(LOUTSTR,K,WORD,'W','CTYPE',IFLAG)
        write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
        zname(ICOMP)=' '
        CALL EGETW(LOUTSTR,K,WORD,'W','Z name',IFLAG)
        ZN=WORD(1:12)
        call st2name(ZN,zname(ICOMP))
        lnzname(ICOMP)=lnblnk(zname(ICOMP))  ! update string length
        gversion(icomp) = 1.1   ! set the version number.
      else

C Check if it matches the syntax of older geometry file format.
        K=0
        IF(ND.EQ.1)THEN
          CALL EGETW(LOUTSTR,K,WORD,'W','CTYPE',IFLAG)
          write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
          zname(ICOMP)=' '
          lnzname(ICOMP)=0
          zdesc(ICOMP)=' '
        ELSEIF(ND.EQ.2)THEN
          CALL EGETW(LOUTSTR,K,WORD,'W','CTYPE',IFLAG)
          write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
          CALL EGETW(LOUTSTR,K,WORD,'W','Z name',IFLAG)
          ZN=WORD(1:12)
          call st2name(ZN,zname(ICOMP))
          lnzname(ICOMP)=lnblnk(zname(ICOMP))  ! update string length
          zdesc(ICOMP)=' '
        ELSEIF(ND.gt.2)THEN
          CALL EGETW(LOUTSTR,K,WORD,'W','CTYPE',IFLAG)
          write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
          CALL EGETW(LOUTSTR,K,WORD,'W','Z name',IFLAG)
          ZN=WORD(1:12)
          call st2name(ZN,zname(ICOMP))
          call egetrm(loutstr,K,phrase,'W','Z description',IER)
          ZDESC(ICOMP)=phrase
        ENDIF
        zoneLabel(icomp) = WORD(1:32)
        IF(IFLAG.NE.0)GOTO 1002

C If we have reached this position then it is an older geometry
C file that seems to be readable so close the file and call egomin.
        CALL ERPFREE(IUNIT,ios)
        gversion(icomp) = 1.0   ! set the version number.
        itrct=0                 ! set to silent
        call egomin(IUNIT,LGEOMF,ICOMP,IR,ITRCT,ITRU,IER)
        return
      endif

      IF(CTYPE(icomp)(1:3).EQ.'REC')THEN

C Zone is of type REC, subroutine EGOMIN should have been used.
        write(outs,'(3a)') 'Geometry file ',LGEOMF(1:lnblnk(LGEOMF)),
     &      ' has REC format so it must be a legacy file.'
        call edisp(itru,outs)
        IER=1
        RETURN
      ELSEIF(CTYPE(icomp)(1:3).EQ.'REG')THEN

C Zone is of type REG, subroutine EGOMIN should have been used.
        write(outs,'(3a)') 'Geometry file ',LGEOMF(1:lnblnk(LGEOMF)),
     &      ' has REG format so it must be a legacy file.'
        call edisp(itru,outs)
        IER=1
        RETURN
      ELSEIF(CTYPE(icomp)(1:3).EQ.'GEN')THEN

C Zone is of type GEN. First read the date stamp line followed
C by the zone description on the next line. For now zdesc string
C is short, but will be extended to 248 char.
        CALL LSTRIPC(IUNIT,LOUTSTR,0,ND,1,'date stamp',IER)
        IF(IER.NE.0) goto 1002
        K=0
        CALL EGETW(LOUTSTR,K,WORD,'W','header tags',IER)
        IF(IER.NE.0) goto 1002
        if(WORD(1:5).eq.'*Date'.or.WORD(1:5).eq.'*date')then
          CALL EGETRM(LOUTSTR,K,dstmp,'W','date stamp',IER)
        endif
        zdesc(ICOMP)=' '
        CALL STRIPC1K(IUNIT,lkout,0,ND,1,'Z description',IER)
        IF(IER.NE.0) goto 1002
        write(zdesc(ICOMP),'(a)') lkout(1:64)
        zorigin(icomp,1)=0.0   ! clear the zorigin and zsize arrays
        zorigin(icomp,2)=0.0
        zorigin(icomp,3)=0.0
        zsize(icomp,1)=0.0; zsize(icomp,2)=0.0; zsize(icomp,3)=0.0
        nbvis(icomp)=0  ! clear number of visual entities
        NBVOBJ(icomp)=0 ! clear number of objects (collections of entities)

C Next look for *vertex lines, incrementing the value of NTV
C as each of these lines is scanned. If the tag is *edges then
C increment the value of NSUR and then scan JVN data. If the
C tag is *surf then there should be 10 further phrases on the
C line. 
C If ier equals 2 then end of file reached so close.
   62   CALL STRIPC1K(IUNIT,lkout,99,ND,0,'*vertex *edges tags',IER)
        if(IER.EQ.2)then

C End of file sensed, however this is not an error at this point.
          IER=0
          goto 44
        elseif(IER.eq.0)then
          continue
        else
          goto 1002
        endif
        K=0
        CALL EGETW(lkout,K,WORD,'W','*vertex or *surface tags',IER)
        if(WORD(1:7).eq.'*vertex')then
          NTV=NTV+1
          if(NTV.le.MTV)then
            CALL EGETWR(lkout,K,X(NTV),0.,0.,'-','X coord',IER)
            CALL EGETWR(lkout,K,Y(NTV),0.,0.,'-','Y coord',IER)
            CALL EGETWR(lkout,K,Z(NTV),0.,0.,'-','Z coord',IER)
            szcoords(ICOMP,NTV,1)=X(NTV)
            szcoords(ICOMP,NTV,2)=Y(NTV)
            szcoords(ICOMP,NTV,3)=Z(NTV)
            NZTV(ICOMP)=NTV
          else
            call edisp(iuout,
     &        'Exceeded number of vertices. Skipping input line.')
          endif
          goto 62
        elseif(WORD(1:7).eq.'*rotate')then           ! Deprecated option.
          goto 62
        elseif(WORD(1:16).eq.'*previous_rotate')then ! Deprecated option.
          goto 62
        elseif(WORD(1:6).eq.'*edges')then

          NEDGE=NEDGE+1

C Check that the number of surfaces is within complexity limits.
          if(NEDGE.gt.MS)then
            write(outs,'(3a)') 'GEOREAD: surfaces beyond range...',
     &      lkout(1:50),'...'
            lsn=MIN0(lnblnk(currentfile),110)
            write(outs2,'(2a)') 'in: ',currentfile(1:lsn)
            call edisp(iuout,outs)
            call edisp(iuout,outs2)
            IER=4
            CALL ERPFREE(IUNIT,ios)
            RETURN
          endif
          if(NEDGE.le.MS)then
            CALL EGETWI(lkout,K,J,3,MV,'F','nb assoc vertices',IERV)
            NVER(NEDGE)=J
            isznver(icomp,NEDGE)=J

C Now proceed to read vertices on one or more lines.
            DO 12 KV=1,NVER(NEDGE)
              CALL EGETWI(lkout,K,IVAL,0,MTV,'F','vertex',IERV)
              IF(IERV.NE.0) THEN
                call edisp(ITRU,' reading continuation line...')
                CALL STRIPC1K(IUNIT,lkout,0,ND,0,'vertex',IER)
                IF(IER.NE.0)goto 1002
                K=0
                CALL EGETWI(lkout,K,IVAL,0,MTV,'F','vertex',IERV)
              ENDIF
              IF(IERV.NE.0) GOTO 1002
              JVN(NEDGE,KV)=IVAL
              iszjvn(icomp,nedge,kv)=IVAL
   12       CONTINUE

            NZSUR(ICOMP)=NEDGE  ! update the number of surfaces.
          else
            call edisp(iuout,
     &        'Exceeded number of surfaces. Skipping input line.')
          endif
          goto 62
        elseif(WORD(1:5).eq.'*surf')then

          NS=NS+1          ! Increment counter for surfaces (NS).
          if(ND.lt.11)then
            write(loutstr,'(a,i2,a,i3,a,i2,2a)') 
     &        'Insufficient tokens in line for zone ',ICOMP,
     &        ' surf ',NS,' tokens ',ND,' in ',lkout(1:lnblnk(lkout))
            call edisp248(iuout,loutstr,120)
            call pauses(2)
          endif
          if(NS.eq.1)then

C If this is the first *surf encountered then set NSUR equal to the
C number of edge sets and call filsur to instantiate default data.
            NSUR=NEDGE
            NZSUR(icomp)=NSUR
            CALL FILSUR(ICOMP,0)

C Check if the master connection list has been filled, if so
C this is a 'subsequent scan' and the connections based commons
C can probably be filled. Each surface needs to be tested
C in case the master list is corrupt.
            icn1=izstocn(icomp,1)
            if(icn1.gt.0)then
            endif
          endif
          if(NS.le.MS)then

C Check and see if izstocn returns non-zeros and set havec24.
            icn=izstocn(icomp,NS)
            if(icn.gt.0)then
              havec24=.true.
            else
              havec24=.false.
            endif

C Surface attributes in a typical line:
C *surf,door,VERT,-,DOOR,UNDERCUT,door,OPAQUE,ANOTHER,3,6 # to door in coridor

C (1) surf name (12 char, tbd - allow spaces)
C (2) surf posn (tags VERT, CEIL (faces down), FLOR (faces up), SLOP (other)
C (3) child of (name of surface, otherwise a -)
C (4-5) useage: a pair of tags for code complience and to assist with air
C  flow network creation. A full description is at the top of this source
C  file.
C (6) construction name (from MLC database)
C (7) optical name (OPAQUE or the name of the optical set name, (tbd allow spaces))
C (8-10) boundary condition (including indices from cnn file):
C   ANOTHER, i index of zone, j index of surface in that zone
C   EXTERIOR,i,j    where i is 0 and j is 0
C   SIMILAR,i,j     where i is offset degC and j is offset Watts
C   IDENT_CEN,i,j     where i is 0 and j is 0
C   CONSTANT,i,j     where i is degC and j is Watts
C   ADIABETIC,i,j     where i is 0 and j is 0
C   BASESIMP,i,j     where i is configuration index and j is % to this surface
C   GROUND_STD,i,j     where i is profile index and j is 0
C   GROUND_USR,i,j     where i is 0 and j is profile index
C   UNKNOWN,i,j     where i is 0 and j is 0

C Surface name, allow for future spaces in name.
            CALL EGETP(lkout,K,WORD,'W','surface name',IER)
            lnsn=MIN0(lnblnk(WORD),12)
            write(SNAME(ICOMP,NS),'(a)') WORD(1:lnsn)
            if(havec24)then
              write(sname(ICOMP,NS),'(a)') WORD(1:lnsn)
            endif

C Surface position
            CALL EGETW(lkout,K,tmpvfc,'W','surface position',IER)
            IF(tmpvfc.EQ.'VERT'.OR.tmpvfc.EQ.'SLOP')then
              write(SVFC(ICOMP,NS),'(a)') tmpvfc
            ELSEIF(tmpvfc(1:4).EQ.'CEIL')then
              write(SVFC(ICOMP,NS),'(a)') 'CEIL'
              izsceil(icomp)=NS   ! identify as a ceiling
            ELSEIF(tmpvfc(1:4).EQ.'FLOR')then
              write(SVFC(ICOMP,NS),'(a)') 'FLOR'
              izsfloor(icomp)=NS  ! identify as a floor
            ENDIF

C Parent name, allow for future spaces in parent name.
            CALL EGETP(lkout,K,WORD,'W','surface parent',IER)
            write(sparent(ICOMP,NS),'(a)') WORD(1:lnblnk(WORD))

C Surface usage - two tokens. If there was a fault then the token might
C be longer than SUSE and it is probably the construction. Check first.
            CALL EGETP(lkout,K,WORD,'W','surface use 1',IER)
            if(lnblnk(WORD).gt.12)then
              write(SUSE(ICOMP,NS,1),'(a)') '-'
              write(SUSE(ICOMP,NS,2),'(a)') '-'
              write(SMLCN(ICOMP,NS),'(a)') WORD(1:lnblnk(WORD))
            else
              write(SUSE(ICOMP,NS,1),'(a)') WORD(1:lnblnk(WORD))
              CALL EGETW(lkout,K,WORD,'W','surface use 2',IER)
              write(SUSE(ICOMP,NS,2),'(a)') WORD(1:lnblnk(WORD))

C Surface construction name, allow for spaces.
              CALL EGETP(lkout,K,WORD,'W','surface construction',IER)
            endif
            write(SMLCN(ICOMP,NS),'(a)') WORD(1:lnblnk(WORD))

C Check if there is a matching MLC name find out which one it is.
C Because many names may start similarly, check against actual widths.
            ICF=-1
            if(NMLC.gt.0)then
              lstrs=lnblnk(SMLCN(ICOMP,NS))  ! surface attribute length
              DO 20 IC=1,NMLC
                if(mlcname(ic)(1:lnmlcname(ic)).EQ.
     &             SMLCN(ICOMP,NS)(1:lstrs))ICF=IC
  20          CONTINUE
            endif

C Surface optics set name or OPAQ/TRAN, allow for spaces.
            CALL EGETP(lkout,K,WORD,'W','surface optics',IER)
            if(WORD(1:4).eq.'OPAQ')then
              write(SOTF(ICOMP,NS),'(a)') 'OPAQUE'  ! write OPAQUE
            elseif(WORD(1:4).eq.'CFC ')then
              write(SOTF(ICOMP,NS),'(a)') 'CFC '  ! write CFC
            elseif(WORD(1:4).eq.'CFC2')then
              write(SOTF(ICOMP,NS),'(a)') 'CFC2'  ! write CFC2
            elseif(WORD(1:4).ne.'OPAQ'.and.WORD(1:3).ne.'CFC')then

C This token could be the name of the optical property. Find out if it
C matches then name of the optical set for the construction.
C If the construction database has been scanned then attempt to discover
C the name of the optical property. Otherwise leave SOFT as TRAN
              if(NMLC.gt.0)then

C Find the optical name within the common constructions file.
                if(ICF.gt.0)then
                  lno=lnblnk(mlcoptical(ICF))
                  write(OPT,'(a)') mlcoptical(ICF)(1:lno)
                else
                  OPT='OPAQUE'
                endif

C When scanning geometry warn users if the optical description in the
C geometry file is different from that of the optical property associated
C with the construction (TRAN is an ok mis-match).
                if(ICF.gt.0)then
                  lnow=lnblnk(word)   ! length for token in file
                  lno=lnblnk(OPT)     ! length for mlc optics
                  if(WORD(1:lnow).ne.OPT(1:lno))then
                    if(WORD(1:4).eq.'TRAN')then
                      continue
                    else
                      lnz=lnblnk(zname(ICOMP))
                      lnsn=lnblnk(SNAME(ICOMP,NS))
                      write(outs,'(10a)') 'Optics ',WORD(1:lnow),
     &                  ' of ',zname(ICOMP)(1:lnz),' ',
     &                  SNAME(ICOMP,NS)(1:lnsn),
     &                  ' in geo file does not match optics of MLC ',
     &                  mlcname(ICF)(1:lnmlcname(ICF)),' ',OPT(1:lno)
                      call edisp(iuout,outs)
                      optcorrected=.true.
                    endif
                  endif
                endif

C The matched optical property is saved to memory. The geometry file
C would need to be written to record the matched property.
                write(SOTF(ICOMP,NS),'(a)') OPT(1:lnblnk(OPT))
              else

C During initial scan of configuration file the databases will
C not yet have been scanned and so this block of code will be
C active. Lets assume that this token really is the name of the
C optical set.
                write(SOTF(ICOMP,NS),'(a)') WORD(1:lnblnk(WORD))
              endif
            endif

C Surface other side - three tokens. Read first as string and
C 2nd and 3rd items could be read as integers
            CALL EGETW(lkout,K,tother,'W','surface other 1',IER)
C            CALL EGETW(lkout,K,tother1,'W','surface other 2',IER)
C            CALL EGETW(lkout,K,tother2,'W','surface other 3',IER)
            CALL EGETWI(lkout,K,io1,-3,MCOM,'W','surface other 2',IER)
            CALL EGETWI(lkout,K,io2,0,MS,'W','surface other 3',IER)
            if(tother(1:7).eq.'UNKNOWN')then
              zboundarytype(icomp,ns,1)=-1
              zboundarytype(icomp,ns,2)=io1
              zboundarytype(icomp,ns,3)=io2
            elseif(tother(1:8).eq.'EXTERIOR')then
              zboundarytype(icomp,ns,1)=0
              zboundarytype(icomp,ns,2)=io1
              zboundarytype(icomp,ns,3)=io2
            elseif(tother(1:9).eq.'ADIABATIC')then
              zboundarytype(icomp,ns,1)=5
              zboundarytype(icomp,ns,2)=io1
              zboundarytype(icomp,ns,3)=io2
            elseif(tother(1:7).eq.'SIMILAR')then
              zboundarytype(icomp,ns,1)=1
              zboundarytype(icomp,ns,2)=io1
              zboundarytype(icomp,ns,3)=io2
            elseif(tother(1:8).eq.'CONSTANT')then
              zboundarytype(icomp,ns,1)=2
              zboundarytype(icomp,ns,2)=io1
              zboundarytype(icomp,ns,3)=io2
            elseif(tother(1:8).eq.'BASESIMP')then
              zboundarytype(icomp,ns,1)=6
              zboundarytype(icomp,ns,2)=io1
              zboundarytype(icomp,ns,3)=io2
            elseif(tother(1:6).eq.'GROUND')then
              zboundarytype(icomp,ns,1)=4
              zboundarytype(icomp,ns,2)=io1
              zboundarytype(icomp,ns,3)=io2
            elseif(tother(1:9).eq.'IDENT_CEN')then
              zboundarytype(icomp,ns,1)=7
              zboundarytype(icomp,ns,2)=io1
              zboundarytype(icomp,ns,3)=io2
            elseif(tother(1:7).eq.'ANOTHER')then ! Reset if - -.
              if(io1.eq.0.or.io2.eq.0)then  ! If 2nd or 3rd atributes are zero reset.
                zboundarytype(icomp,ns,1)=-1
                zboundarytype(icomp,ns,2)= 0
                zboundarytype(icomp,ns,3)= 0
              else
                zboundarytype(icomp,ns,1)= 3
                zboundarytype(icomp,ns,2)=io1
                zboundarytype(icomp,ns,3)=io2
              endif
            else

C Assume partition so setup boundary arrays with this in mind.
              zboundarytype(icomp,ns,1)= 3
              zboundarytype(icomp,ns,2)=io1
              zboundarytype(icomp,ns,3)=io2
            endif

C Decode the zboundarytype into sother.
            call decode_zsbound(icomp,ns,sbound_ty,sbound_c2,sbound_e2)

          else
            call edisp(iuout,
     &        'Exceeded number of surfaces. Skipping input line.')
          endif
          goto 62
        elseif(WORD(1:10).eq.'*base_list')then

C The list-based definition of base surfaces.
C A typical entry would look like:
C *base_list,1,6,13.50 0
C where the 1st item is the number of associated surfaces, then the
C list of surfaces followed by the base area followed by user flag.
          CALL EGETWI(lkout,K,IB0,0,MBL,'-','zn base list',IER)
          izbaselist(icomp)=IB0
          if(IB0.gt.0)then

C There are items in the list so scan the surface indices.
            do 63 inthelist=1,IB0
              IBASES(ICOMP,inthelist)=0
              CALL EGETWI(lkout,K,IB1,0,0,'-',
     &          'zn base surface list item',IER)
              IBASES(ICOMP,inthelist)=IB1
  63        continue
          endif
          CALL EGETWR(lkout,K,VAL,0.,99999.,'W','zn base area',IER)
          ZBASEA(ICOMP)=VAL
          CALL EGETWI(lkout,K,IUB,0,2,'-',
     &      'zn base area user flag',IER)
          iuzbasea(icomp)=iub
          goto 62

        elseif(WORD(1:10).eq.'*shad_calc')then

C Shading calculation instructions.
          CALL EGETW(lkout,K,WORD,'W','shad directive',IFLAG)
          if(WORD(1:4).eq.'none')then
            iaplic(icomp,1)=0
            nsurfcalc(icomp)=0
          else
            if(WORD(1:14).eq.'all_applicable')then
              iaplic(icomp,1)=1
            elseif(WORD(1:4).eq.'list')then
              iaplic(icomp,1)=0
            endif

C Read number of surfaces and then the list
            CALL EGETWI(lkout,K,iv,0,MS,'-','nsurfcalc',IER)
            nsurfcalc(icomp)=iv
            IRVA=nsurfcalc(icomp)
            CALL EGETWIA(IUNIT,IVA,IRVA,0,MS,'F','lstsfcalc',IER)
            DO 342 ks=1,nsurfcalc(icomp)
              lstsfcalc(icomp,ks)=IVA(ks)
 342        CONTINUE
          endif
          goto 62
        elseif(WORD(1:11).eq.'*insol_calc')then

C Insolation calculation instructions.
          CALL EGETW(lkout,K,WORD,'W','insolation directive',IFLAG)
          if(WORD(1:4).eq.'none')then
            iaplic(icomp,2)=0
            nsurfinso(icomp)=0
          else
            if(WORD(1:14).eq.'all_applicable')then
              iaplic(icomp,2)=1
            elseif(WORD(1:4).eq.'list')then
              iaplic(icomp,2)=0
            endif

C Applicable surfaces are the current exterior facing surfaces.
            CALL EGETWI(lkout,K,iv,0,MS,'-','nsurfinso',IER)
            nsurfinso(icomp)=iv
            IRVA=nsurfinso(icomp)
            CALL EGETWIA(IUNIT,IVA,IRVA,0,MS,'F','isurfinso',IER)
            DO 344 ks=1,nsurfinso(icomp)
              isurfinso(icomp,ks)=IVA(ks)
 344        CONTINUE
          endif
          goto 62
        elseif(WORD(1:7).eq.'*insol ')then

C There are two tags that begin with *insol so include the space
C in the tag test to clarify.
C Read default insolation surface numbers. NN = insolation defining
C index:  1; one plane, 2; two planes, 3; all planes (diffuse).
C IDPN defines the default plane numbers.
          CALL EGETWI(lkout,K,IV,1,3,'W','Def insol index',IER)
          NDP(ICOMP)=IV
          CALL EGETWI(lkout,K,IV,0,NSUR,'W','1st recv surf',IER)
          IDPN(ICOMP,1)=IV
          CALL EGETWI(lkout,K,IV,0,NSUR,'W','2nd recv surf',IER)
          IDPN(ICOMP,2)=IV
          CALL EGETWI(lkout,K,IV,-1,NSUR,'W','3rd insol data',IER)
          IDPN(ICOMP,3)=IV
          IF(IR.EQ.1)THEN
            DO 40 I=1,3
              IX=0
              IF(NDP(ICOMP).EQ.1.AND.I.EQ.1)IX=1
              IF(NDP(ICOMP).EQ.2.AND.I.LE.2)IX=1
              if(IDPN(ICOMP,I).NE.-1.AND.IDPN(ICOMP,I).NE.0)then
                if(IDPN(ICOMP,I).LT.IX.OR.IDPN(ICOMP,I).GT.NSUR)then
                  CALL LUSRMSG(' Nonexistent insolation plane!',
     &              lkout,'W')
                endif
              endif
   40       CONTINUE
          ENDIF
          goto 62
        elseif(WORD(1:13).eq.'*bridge_start')then

C Simplified thermal bridge data starts here. Scan three reals after tag.
          nbrdg(icomp)=0
          CALL EGETWR(lkout,K,VAL,0.,0.,'-','user fraction thbr',IER)
          thbrpercent=VAL
          CALL EGETWR(lkout,K,VAL,0.,0.,'-','W/K from loss %',IER)
          losspercent(icomp)=VAL
          CALL EGETWR(lkout,K,VAL,0.,0.,'-','W/K total',IER)
          totheatloss(icomp)=VAL
          CALL EGETWR(lkout,K,VAL,0.,0.,'-','W/K total',IER)
          uavtotal(icomp)=VAL
          if(ND.gt.5)then
            call egetrm(lkout,K,tbregime,'W','tb regime',IER)
          endif
          goto 62
        elseif(WORD(1:11).eq.'*ukt_bridge')then

C Phrases for thermal bridges are checked and ibrdg index assigned
C based on exact match of characters in phrase. This token was
C used in older models which only held a summary for each thermal
C bridge type. Look in setbridgenames for where these are set.
          nbrdg(icomp)=nbrdg(icomp)+1
          CALL EGETP(lkout,K,phrase,'W','bridge tag',IER)
          if(phrase(1:10).eq.'roof-wall ')then
            ibrdg(icomp,nbrdg(icomp))=1
          elseif(phrase(1:18).eq.'wall-ground floor ')then
            ibrdg(icomp,nbrdg(icomp))=2
          elseif(phrase(1:25).eq.'wall-wall (convex corner)')then
            ibrdg(icomp,nbrdg(icomp))=3
          elseif(phrase(1:26).eq.'wall-wall (concave corner)')then
            ibrdg(icomp,nbrdg(icomp))=4
          elseif(phrase(1:26).eq.'wall-floor (exposed floor)')then
            ibrdg(icomp,nbrdg(icomp))=5
          elseif(phrase(1:28).eq.'lintel above window or door')then
            ibrdg(icomp,nbrdg(icomp))=6
          elseif(phrase(1:18).eq.'Sill below window '.or.
     &           phrase(1:18).eq.'sill below window ')then
            ibrdg(icomp,nbrdg(icomp))=7
          elseif(phrase(1:23).eq.'jamb at window or door ')then
            ibrdg(icomp,nbrdg(icomp))=8
          elseif(phrase(1:11).eq.'wall-gable ')then
            ibrdg(icomp,nbrdg(icomp))=9
          elseif(phrase(1:13).eq.'wall-parapet ')then
            ibrdg(icomp,nbrdg(icomp))=10
          elseif(phrase(1:24).eq.'wall-intermediate-floor ')then
            ibrdg(icomp,nbrdg(icomp))=11
          elseif(phrase(1:15).eq.'wall-partition ')then
            ibrdg(icomp,nbrdg(icomp))=12
          elseif(phrase(1:12).eq.'glass-frame ')then
            ibrdg(icomp,nbrdg(icomp))=13
          elseif(phrase(1:7).eq.'balcony')then
            ibrdg(icomp,nbrdg(icomp))=14
          elseif(phrase(1:14).eq.'user-defined-a')then
            ibrdg(icomp,nbrdg(icomp))=15
          else
            ibrdg(icomp,nbrdg(icomp))=16
          endif
          CALL EGETWR(lkout,K,VAL,0.,0.,'-','length of bridge',IER)
          lngth(icomp,nbrdg(icomp))=VAL
          bridgelen(icomp,ibrdg(icomp,nbrdg(icomp)))=VAL
          CALL EGETWR(lkout,K,VAL,0.,0.,'-','bridge psi',IER)
          psi(icomp,nbrdg(icomp))=VAL
          nbridgevt(icomp,ibrdg(icomp,nbrdg(icomp)))=0
          goto 62
        elseif(WORD(1:7).eq.'*bridge')then

C Thermal bridges with associated edges in 2nd line. Look in 
C subroutine setbridgenames for where these are set.
          nbrdg(icomp)=nbrdg(icomp)+1
          CALL EGETP(lkout,K,phrase,'W','bridge tag',IER)
          if(phrase(1:10).eq.'roof-wall ')then
            ibrdg(icomp,nbrdg(icomp))=1
          elseif(phrase(1:18).eq.'wall-ground floor ')then
            ibrdg(icomp,nbrdg(icomp))=2
          elseif(phrase(1:25).eq.'wall-wall (convex corner)')then
            ibrdg(icomp,nbrdg(icomp))=3
          elseif(phrase(1:26).eq.'wall-wall (concave corner)')then
            ibrdg(icomp,nbrdg(icomp))=4
          elseif(phrase(1:26).eq.'wall-floor (exposed floor)')then
            ibrdg(icomp,nbrdg(icomp))=5
          elseif(phrase(1:28).eq.'lintel above window or door')then
            ibrdg(icomp,nbrdg(icomp))=6
          elseif(phrase(1:18).eq.'Sill below window '.or.
     &           phrase(1:18).eq.'sill below window ')then
            ibrdg(icomp,nbrdg(icomp))=7
          elseif(phrase(1:23).eq.'jamb at window or door ')then
            ibrdg(icomp,nbrdg(icomp))=8
          elseif(phrase(1:11).eq.'wall-gable ')then
            ibrdg(icomp,nbrdg(icomp))=9
          elseif(phrase(1:13).eq.'wall-parapet ')then
            ibrdg(icomp,nbrdg(icomp))=10
          elseif(phrase(1:24).eq.'wall-intermediate-floor ')then
            ibrdg(icomp,nbrdg(icomp))=11
          elseif(phrase(1:15).eq.'wall-partition ')then
            ibrdg(icomp,nbrdg(icomp))=12
          elseif(phrase(1:12).eq.'glass-frame ')then
            ibrdg(icomp,nbrdg(icomp))=13
          elseif(phrase(1:7).eq.'balcony')then
            ibrdg(icomp,nbrdg(icomp))=14
          elseif(phrase(1:14).eq.'user-defined-a')then
            ibrdg(icomp,nbrdg(icomp))=15
          else
            ibrdg(icomp,nbrdg(icomp))=16
          endif
          CALL EGETWR(lkout,K,VAL,0.,0.,'-','length of bridge',IER)
          lngth(icomp,nbrdg(icomp))=VAL
          bridgelen(icomp,ibrdg(icomp,nbrdg(icomp)))=VAL
          CALL EGETWR(lkout,K,VAL,0.,0.,'-','bridge psi',IER)
          psi(icomp,nbrdg(icomp))=VAL
          CALL EGETWI(lkout,K,IV,1,52,'W','number assoc verts',IER)
          nbridgevt(icomp,ibrdg(icomp,nbrdg(icomp)))=IV

C Read next line IF nbridgevt non-zero.
          if(IV.gt.0)then
            IRVA=IV
            CALL EGETWIA(IUNIT,IVA,IRVA,0,MV*2,'F','bridge lst',IER)
            DO ks=1,IV
              bridgevlst(icomp,ibrdg(icomp,nbrdg(icomp)),ks)=IVA(ks)
            enddo
          endif
          goto 62
        elseif(WORD(1:11).eq.'*end_bridge')then
          goto 62
        elseif(WORD(1:12).eq.'*block_start')then

C A number of block shapes can be associated with a model as follows:
C *obs - solar obstruction blocks and
C *mrt - mean radiant temperature sensor blocks (echos vwf file)
C *photo - lighting sensor position and vector (not yet implemented)

C If there is a zone obstruction file (iobs() = 1) then there should
C not be *obs tags in this file.
C If iobs() = 2 then obstructions were previously imported from a
C zone obstructions file or are held in this geometry file.
C If iobs() = 0 then no obstructions yet defined so the value should
C be incremented to iobs() = 2.
          if(iobs(icomp).eq.1)then
            write(outs,'(3a)') 'In zone ',zname(icomp),
     &        ' there is a zone obstructions file as well as block'
            call usrmsg(outs(1:lnblnk(outs)),
     &      'definitions in this geometry file. Skipping data.','W')
          endif

C Following the tag *block_start will be the grid for X and Z. If
C there are insufficient tokens on line set both to 20. This is also
C the case if the blocks section includes only mrt sensor bodies.
          if(ND.gt.2)then
            CALL EGETWI(lkout,K,NOX(icomp),4,MOX,'F','opq grid X',IER)
            CALL EGETWI(lkout,K,NOZ(icomp),4,MOZ,'F','opq grid Z',IER)
          else
            NOX(icomp)=20
            NOZ(icomp)=20
          endif

C Read another line to get the actual blocks. There are slightly different
C formats depending on whether the key is '*obs' or '*obs3'.
   72     CALL STRIPC1K(IUNIT,lkout,99,ND,0,'*obs *mrt tags',IER)
          k=0
          CALL EGETP(lkout,K,phrase,'W','block tag',IER)
          if(phrase(1:5).eq.'*obs3')then
            if(iobs(icomp).eq.1) goto 72  ! keep reading till *end_block
            if(iobs(icomp).eq.0) iobs(icomp)=2
            nbobs(icomp)=nbobs(icomp)+1
            nbo=nbobs(icomp)
            CALL EGETWR(lkout,K,VX,-999.,998.,'W','obs X org',IER)
            CALL EGETWR(lkout,K,VY,-999.,998.,'W','obs Y org',IER)
            CALL EGETWR(lkout,K,VZ, -99., 99.,'W','obs Z org',IER)
            XOB(icomp,nbo)=VX
            YOB(icomp,nbo)=VY
            ZOB(icomp,nbo)=VZ
            CALL EGETWR(lkout,K,VX,0.,150.,'W','obs X dis',IER)
            CALL EGETWR(lkout,K,VY,0.,150.,'W','obs Y dis',IER)
            CALL EGETWR(lkout,K,VZ,0.,150.,'W','obs Z dis',IER)
            DXOB(icomp,nbo)=VX
            DYOB(icomp,nbo)=VY
            DZOB(icomp,nbo)=VZ
            CALL EGETWR(lkout,K,VX,-359.,359.,'W','obs rot ang a',IER)
            BANGOB(icomp,nbo,1)=VX
            CALL EGETWR(lkout,K,VX,-359.,359.,'W','obs rot ang b',IER)
            BANGOB(icomp,nbo,2)=VX
            CALL EGETWR(lkout,K,VX,-359.,359.,'W','obs rot ang c',IER)
            BANGOB(icomp,nbo,3)=VX
            if(ND.ge.13)then  ! if enough items for opacity
              CALL EGETWR(lkout,K,VX,0.,1.,'W','obs opacity',IER)
              OPOB(icomp,nbo)=VX
            else
              OPOB(icomp,nbo)=1.0  ! set to opaque if not specified
            endif
            CALL EGETW(lkout,K,WORD,'W','obs blk name',IFLAG)
            BLOCKNAME(icomp,nbo)=WORD(1:12)

C The name of the construction might contain spaces so use EGETP.
            CALL EGETP(lkout,K,WORD,'W','obs mat name',IFLAG)
            write(BLOCKMAT(icomp,nbo),'(a)') WORD(1:lnblnk(WORD))
            BLOCKTYP(icomp,nbo)='obs3'
            goto 72

          elseif(phrase(1:5).eq.'*obsp')then

C A general polygon obstruction to be associated with the current zone.
C The first line includes (current fixed) integer number of vertices
C followed by number of faces and the name and material.
C The 2nd line has the first 4 coordinates and the 3rd line has the
C next 4 coordinates. The surface and edge ordering is as in a
C standard obstruction block when converted into a GB1 common block.
            if(iobs(icomp).eq.1) goto 72  ! keep reading till *end_block
            if(iobs(icomp).eq.0) iobs(icomp)=2
            nbobs(icomp)=nbobs(icomp)+1
            nbo=nbobs(icomp)
            CALL EGETWI(lkout,K,ival,8,8,'F','obs nb vertices',IER)
            CALL EGETWI(lkout,K,ival,6,6,'F','obs nb faces',IER)
            if(ND.ge.6)then  ! if enough items for opacity
              CALL EGETWR(lkout,K,VX,0.,1.,'W','obs opacity',IER)
              OPOB(icomp,nbo)=VX
            else
              OPOB(icomp,nbo)=1.0  ! set to opaque if not specified
            endif
            CALL EGETW(lkout,K,WORD,'W','obs blk name',IFLAG)
            write(BLOCKNAME(icomp,nbo),'(a)') WORD(1:lnblnk(WORD))
            CALL EGETP(lkout,K,WORD,'W','obs mat name',IFLAG)
            write(BLOCKMAT(icomp,nbo),'(a)') WORD(1:lnblnk(WORD))
            BLOCKTYP(icomp,nbo)='obsp'

            CALL STRIPC1K(IUNIT,lkout,99,ND,1,'first 4 coord',IER)
            IF(IER.NE.0)goto 1002
            K=0
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XBP 1',IER)
            XBP(icomp,nbo,1)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YBP 1',IER)
            YBP(icomp,nbo,1)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZBP 1',IER)
            ZBP(icomp,nbo,1)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XBP 2',IER)
            XBP(icomp,nbo,2)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YBP 2',IER)
            YBP(icomp,nbo,2)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZBP 2',IER)
            ZBP(icomp,nbo,2)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XBP 3',IER)
            XBP(icomp,nbo,3)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YBP 3',IER)
            YBP(icomp,nbo,3)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZBP 3',IER)
            ZBP(icomp,nbo,3)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XBP 4',IER)
            XBP(icomp,nbo,4)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YBP 4',IER)
            YBP(icomp,nbo,4)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZBP 4',IER)
            ZBP(icomp,nbo,4)=val1

            CALL STRIPC1K(IUNIT,lkout,99,ND,1,'2nd 4 coord',IER)
            IF(IER.NE.0)goto 1002
            K=0
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XBP 5',IER)
            XBP(icomp,nbo,5)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YBP 5',IER)
            YBP(icomp,nbo,5)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZBP 5',IER)
            ZBP(icomp,nbo,5)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XBP 6',IER)
            XBP(icomp,nbo,6)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YBP 6',IER)
            YBP(icomp,nbo,6)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZBP 6',IER)
            ZBP(icomp,nbo,6)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XBP 7',IER)
            XBP(icomp,nbo,7)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YBP 7',IER)
            YBP(icomp,nbo,7)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZBP 7',IER)
            ZBP(icomp,nbo,7)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XBP 8',IER)
            XBP(icomp,nbo,8)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YBP 8',IER)
            YBP(icomp,nbo,8)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZBP 8',IER)
            ZBP(icomp,nbo,8)=val1
            goto 72

          elseif(phrase(1:4).eq.'*obs')then

            if(iobs(icomp).eq.1) goto 72  ! keep reading till *end_block
            if(iobs(icomp).eq.0) iobs(icomp)=2
            nbobs(icomp)=nbobs(icomp)+1
            nbo=nbobs(icomp)
            CALL EGETWR(lkout,K,VX,-999.,998.,'W','obs X org',IER)
            CALL EGETWR(lkout,K,VY,-999.,998.,'W','obs Y org',IER)
            CALL EGETWR(lkout,K,VZ, -99., 99.,'W','obs Z org',IER)
            XOB(icomp,nbo)=VX
            YOB(icomp,nbo)=VY
            ZOB(icomp,nbo)=VZ
            CALL EGETWR(lkout,K,VX,-150.,150.,'W','obs X dis',IER)
            CALL EGETWR(lkout,K,VY,-150.,150.,'W','obs Y dis',IER)
            CALL EGETWR(lkout,K,VZ,-150.,150.,'W','obs Z dis',IER)
            DXOB(icomp,nbo)=VX
            DYOB(icomp,nbo)=VY
            DZOB(icomp,nbo)=VZ
            CALL EGETWR(lkout,K,VX,-359.,359.,'W','obs rot ang',IER)
            BANGOB(icomp,nbo,1)=VX
            BANGOB(icomp,nbo,2)=0.0   ! there is no 2nd rotation
            BANGOB(icomp,nbo,3)=0.0   ! there is no 3rd rotation
            if(ND.ge.11)then  ! if enough items for opacity
              CALL EGETWR(lkout,K,VX,0.,1.,'W','obs opacity',IER)
              OPOB(icomp,nbo)=VX
            else
              OPOB(icomp,nbo)=1.0  ! set to opaque if not specified
            endif
            CALL EGETW(lkout,K,WORD,'W','obs blk name',IFLAG)
            BLOCKNAME(icomp,nbo)=WORD(1:12)

C The name of the construction might contain spaces so use EGETP.
            CALL EGETP(lkout,K,WORD,'W','obs mat name',IFLAG)
            write(BLOCKMAT(icomp,nbo),'(a)') WORD(1:lnblnk(WORD))
            BLOCKTYP(icomp,nbo)='obs '
            goto 72
          elseif(phrase(1:4).eq.'*mrt')then

C << somewhere need to ensure ncub is zero when model is read in.
            nmrtsen=nmrtsen+1
            nbo=nmrtsen
            CALL EGETWR(lkout,K,VX,-999.,998.,'W','mrt X org',IER)
            CALL EGETWR(lkout,K,VY,-999.,998.,'W','mrt Y org',IER)
            CALL EGETWR(lkout,K,VZ, -99., 99.,'W','mrt Z org',IER)
            XOC(nbo)=VX
            YOC(nbo)=VY
            ZOC(nbo)=VZ
            CALL EGETWR(lkout,K,VX,0.,150.,'W','mrt X dis',IER)
            CALL EGETWR(lkout,K,VY,0.,150.,'W','mrt Y dis',IER)
            CALL EGETWR(lkout,K,VZ,0.,150.,'W','mrt Z dis',IER)
            DXC(nbo)=VX
            DYC(nbo)=VY
            DZC(nbo)=VZ
            CALL EGETWR(lkout,K,VX,-359.,359.,'W','mrt rot ang',IER)
            CANG(nbo)=VX
            CALL EGETW(lkout,K,WORD,'W','mrt blk name',IFLAG)
            write(CUBN(nbo),'(a)') WORD(1:lnblnk(WORD))
            goto 72

          elseif(phrase(1:6).eq.'*photo')then

C << not yet implemented >>
            goto 72
          elseif(phrase(1:10).eq.'*end_block')then

C At end of block section check if ncub(icomp) matches nmrtsen.
C If nmrtsen > 0 and ncub = 0 then reset ncub.
            if(nmrtsen.gt.0.and.ncub(icomp).eq.0)then
              ncub(icomp)=nmrtsen
            endif
            goto 62
          endif
        elseif(WORD(1:13).eq.'*visual_start')then

C Read another line to get the visual entities. There are slightly different
C formats depending on whether the tag is:
C *vis  - block perpendicular to the floor
C *vis3 - block with arbitrary orientation
C *visp - general 6 sided polygon
   73     CALL STRIPC1K(IUNIT,lkout,99,ND,0,'*vis tags',IER)
          k=0
          CALL EGETP(lkout,K,phrase,'W','visual tag',IER)
          if(phrase(1:5).eq.'*vis3')then
            nbvis(icomp)=nbvis(icomp)+1
            nbv=nbvis(icomp)
            CALL EGETWR(lkout,K,VX,-999.,998.,'W','vis X org',IER)
            CALL EGETWR(lkout,K,VY,-999.,998.,'W','vis Y org',IER)
            CALL EGETWR(lkout,K,VZ, -99., 99.,'W','vis Z org',IER)
            XOV(icomp,nbv)=VX
            YOV(icomp,nbv)=VY
            ZOV(icomp,nbv)=VZ
            CALL EGETWR(lkout,K,VX,0.,150.,'W','vis X dis',IER)
            CALL EGETWR(lkout,K,VY,0.,150.,'W','vis Y dis',IER)
            CALL EGETWR(lkout,K,VZ,0.,150.,'W','vis Z dis',IER)
            DXOV(icomp,nbv)=VX
            DYOV(icomp,nbv)=VY
            DZOV(icomp,nbv)=VZ
            CALL EGETWR(lkout,K,VX,-359.,359.,'W','vis rot ang a',IER)
            BANGOV(icomp,nbv,1)=VX
            CALL EGETWR(lkout,K,VX,-359.,359.,'W','vis rot ang b',IER)
            BANGOV(icomp,nbv,2)=VX
            CALL EGETWR(lkout,K,VX,-359.,359.,'W','vis rot ang c',IER)
            BANGOV(icomp,nbv,3)=VX
            if(ND.ge.13)then  ! if enough items for opacity
              CALL EGETWR(lkout,K,VX,0.,1.,'W','vis opacity',IER)
              OPOV(icomp,nbv)=VX
            else
              OPOV(icomp,nbv)=1.0  ! set to opaque if not specified
            endif
            CALL EGETW(lkout,K,WORD,'W','vis blk name',IFLAG)
            VISNAME(icomp,nbv)=WORD(1:12)

C The name of the construction might contain spaces so use EGETP.
            CALL EGETP(lkout,K,WORD,'W','vis mat name',IFLAG)
            write(VISMAT(icomp,nbv),'(a)') WORD(1:lnblnk(WORD))
            VISTYP(icomp,nbv)='vis3'
            goto 73    ! check if there is another

          elseif(phrase(1:5).eq.'*visp')then

C A general polygon visual to be associated with the current zone.
C The first line includes (current fixed) integer number of vertices
C followed by number of faces and the name and material.
C The 2nd line has the first 4 coordinates and the 3rd line has the
C next 4 coordinates. The surface and edge ordering is as obsp.
            nbvis(icomp)=nbvis(icomp)+1
            nbv=nbvis(icomp)
            CALL EGETWI(lkout,K,ival,8,8,'F','vis nb vertices',IER)
            CALL EGETWI(lkout,K,ival,6,6,'F','vis nb faces',IER)
            if(ND.ge.6)then  ! if enough items for opacity
              CALL EGETWR(lkout,K,VX,0.,1.,'W','vis opacity',IER)
              OPOV(icomp,nbv)=VX
            else
              OPOV(icomp,nbv)=1.0  ! set to opaque if not specified
            endif
            CALL EGETW(lkout,K,WORD,'W','vis blk name',IFLAG)
            write(VISNAME(icomp,nbv),'(a)') WORD(1:lnblnk(WORD))
            CALL EGETP(lkout,K,WORD,'W','vis mat name',IFLAG)
            write(VISMAT(icomp,nbv),'(a)') WORD(1:lnblnk(WORD))
            VISTYP(icomp,nbv)='visp'

            CALL LSTRIPC(IUNIT,lkout,99,ND,1,'first 4 coord',IER)
            IF(IER.NE.0)goto 1002
            K=0
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XVP 1',IER)
            XVP(icomp,nbv,1)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YVP 1',IER)
            YVP(icomp,nbv,1)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZVP 1',IER)
            ZVP(icomp,nbv,1)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XVP 2',IER)
            XVP(icomp,nbv,2)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YVP 2',IER)
            YVP(icomp,nbv,2)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZVP 2',IER)
            ZVP(icomp,nbv,2)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XVP 3',IER)
            XVP(icomp,nbv,3)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YVP 3',IER)
            YVP(icomp,nbv,3)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZVP 3',IER)
            ZVP(icomp,nbv,3)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XVP 4',IER)
            XVP(icomp,nbv,4)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YVP 4',IER)
            YVP(icomp,nbv,4)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZVP 4',IER)
            ZVP(icomp,nbv,4)=val1

            CALL LSTRIPC(IUNIT,lkout,99,ND,1,'2nd 4 coord',IER)
            IF(IER.NE.0)goto 1002
            K=0
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XVP 5',IER)
            XVP(icomp,nbv,5)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YVP 5',IER)
            YVP(icomp,nbv,5)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZVP 5',IER)
            ZVP(icomp,nbv,5)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XVP 6',IER)
            XVP(icomp,nbv,6)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YVP 6',IER)
            YVP(icomp,nbv,6)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZVP 6',IER)
            ZVP(icomp,nbv,6)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XVP 7',IER)
            XVP(icomp,nbv,7)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YVP 7',IER)
            YVP(icomp,nbv,7)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZVP 7',IER)
            ZVP(icomp,nbv,7)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','XVP 8',IER)
            XVP(icomp,nbv,8)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','YVP 8',IER)
            YVP(icomp,nbv,8)=val1
            CALL EGETWR(lkout,K,val1,-999.,998.,'W','ZVP 8',IER)
            ZVP(icomp,nbv,8)=val1
            goto 73

          elseif(phrase(1:4).eq.'*vis')then

            nbvis(icomp)=nbvis(icomp)+1
            nbv=nbvis(icomp)
            CALL EGETWR(lkout,K,VX,-999.,998.,'W','vis X org',IER)
            CALL EGETWR(lkout,K,VY,-999.,998.,'W','vis Y org',IER)
            CALL EGETWR(lkout,K,VZ, -99., 99.,'W','vis Z org',IER)
            XOV(icomp,nbv)=VX
            YOV(icomp,nbv)=VY
            ZOV(icomp,nbv)=VZ
            CALL EGETWR(lkout,K,VX,0.,150.,'W','vis X dis',IER)
            CALL EGETWR(lkout,K,VY,0.,150.,'W','vis Y dis',IER)
            CALL EGETWR(lkout,K,VZ,0.,150.,'W','vis Z dis',IER)
            DXOV(icomp,nbv)=VX
            DYOV(icomp,nbv)=VY
            DZOV(icomp,nbv)=VZ
            CALL EGETWR(lkout,K,VX,-359.,359.,'W','vis rot ang',IER)
            BANGOV(icomp,nbv,1)=VX
            BANGOV(icomp,nbv,2)=0.0   ! there is no 2nd rotation
            BANGOV(icomp,nbv,3)=0.0   ! there is no 3rd rotation
            if(ND.ge.11)then  ! if enough items for opacity
              CALL EGETWR(lkout,K,VX,0.,1.,'W','vis opacity',IER)
              OPOV(icomp,nbv)=VX
            else
              OPOV(icomp,nbv)=1.0  ! set to opaque if not specified
            endif
            CALL EGETW(lkout,K,WORD,'W','vis blk name',IFLAG)
            VISNAME(icomp,nbv)=WORD(1:12)

C The name of the visual material might contain spaces so use EGETP.
            CALL EGETP(lkout,K,WORD,'W','vis mat name',IFLAG)
            write(VISMAT(icomp,nbv),'(a)') WORD(1:lnblnk(WORD))
            VISTYP(icomp,nbv)='vis '
            goto 73

          elseif(phrase(1:8).eq.'*vobject')then

C Collection of entities making up an object.
C *vobject,rubish-bskt,square wood rubish bin,4,basket_bk,basket_fr,basket_lf,basket_rt
            NBVOBJ(icomp)=NBVOBJ(icomp)+1
            nbvo=NBVOBJ(icomp)
            CALL EGETW(lkout,K,WORD,'W','vis obj name',IFLAG)
            VOBJNAME(icomp,nbvo)=WORD(1:12)
            CALL EGETP(lkout,K,WORD,'W','vis obj desc',IFLAG)
            write(VOBJDESC(icomp,nbvo),'(a)') WORD(1:lnblnk(WORD))
            CALL EGETWI(lkout,K,ival,1,MOMVB,'F','nb vis entities',
     &        IER)
            NBVOBJLIST(icomp,nbvo)=ival
            do ibvo = 1,ival
             CALL EGETW(lkout,K,WORD,'W','entity name',IFLAG)
             write(VOBJLIST(icomp,nbvo,ibvo),'(a)')WORD(1:lnblnk(WORD))
            enddo  ! ibvo
C Debug.
C            write(6,*) 'vis obj ',icomp,nbvo,ival,VOBJNAME(icomp,nbvo),
C     &        VOBJDESC(icomp,nbvo),
C     &        VOBJLIST(icomp,nbvo,1),VOBJLIST(icomp,nbvo,2)
            goto 73
          elseif(phrase(1:11).eq.'*end_visual')then
            goto 62
          endif

          goto 62   ! not recognised so jump

        elseif(WORD(1:22).eq.'*zone_viewfactor_start')then

C << to be done >>
          goto 62
        elseif(WORD(1:24).eq.'*sensor_viewfactor_start')then

C  << to be done >>
          goto 62
        elseif(WORD(1:24).eq.'*sensor_viewfactor_start')then

C  << to be done >>
        else

C Fall through position. Warn and loop back to see if more.
          write(outs,'(2a)') ' Unknown tag in geometry file ',word
          call edisp(iuout,outs)
          goto 62
        endif

      ELSE
        write(outs,'(2a)') ' Geometry shape type illegal in ',
     &    LGEOMF(1:lnblnk(LGEOMF))
        call edisp(itru,outs)
        IER=1
        CALL ERPFREE(IUNIT,ios)
        RETURN
      endif

C << to this point in the logic... >>

C Now close geometry data file.
   44 CALL ERPFREE(IUNIT,ios)

C If base area has not yet been calculated, do this now and calculate
C transparent area for surfaces connected to external boundary.
C Get surface areas via call to zinfo.
      call zinfo(icomp,zoa,zvol,'q')
      vol(ICOMP)=zvol
      zonetotsurfacearea(ICOMP)=zoa
      ZTRANA(ICOMP)=0.
      do 43 ijj=1,NSUR
        if(checkbase)then
          if(SVFC(ICOMP,ijj)(1:4).eq.'FLOR')then
            if(SUSE(ICOMP,ijj,1)(1:5).eq.'FURNI')then
              continue  ! ignore horizontal surfaces marked as furniture.
            elseif(SUSE(ICOMP,ijj,1)(1:6).eq.'REVEAL')then
              continue  ! ignore horizontal surfaces marked as reveal.
            else

C If surface `flor` not included in the list add it.
              lastlist=lastlist+1
              if(lastlist.le.10)then
                IBASES(ICOMP,lastlist)=ijj
                ZBASEA(ICOMP)=ZBASEA(ICOMP)+SNA(ICOMP,ijj)
              endif
            endif
          endif
        endif
        if (SOTF(ICOMP,ijj)(1:4).ne.'OPAQ'.and.
     &      SOTF(ICOMP,ijj)(1:3).ne.'CFC'.and.
     &      zboundarytype(ICOMP,ijj,1).eq.0) then
          ZTRANA(ICOMP)=ZTRANA(ICOMP)+SNA(ICOMP,ijj)
        endif
  43  continue
      if(checkbase)then
        write(outs,'(a,f6.2,3a)') 'Base area estimated at ',
     &     ZBASEA(ICOMP),'m^2 for ',zname(icomp),
     &    ' (probably an older file).'
        call edisp(itru,outs)
        if(lastlist.gt.0)iuzbasea(icomp)=0
      endif

C If optics were corrected during the scan advise user to save the
C geometry file to record the correction.
      if(optcorrected)then
        call edisp(iuout,
     &  ' Optic mismatch corrected in memory. Save geometry to record.')
        optcorrected=.false.
      endif

C Leave it to calling code to report on contents of zone geometry
C as extended reporting needs to know of the context of the zone.
C Close geometry data file before exiting.
      CALL ERPFREE(IUNIT,ios)
      RETURN

C Errors for lkout reads.
 1002 write(outs,'(3a)') 'GEOREAD: conversion error in...',
     &  lkout(1:50),'...'
      lsn=MIN0(lnblnk(currentfile),110)
      write(outs2,'(2a)') 'in: ',currentfile(1:lsn)
      call edisp(iuout,outs)
      call edisp(iuout,outs2)
      IER=1
      CALL ERPFREE(IUNIT,ios)
      RETURN

      END

C ************* GEOWRITE
C Writes a geometry file (GEN V1.1 type) based on information
C held in common blocks G0, G1, G4. It assumes that this
C information has been checked. G1 is valid only for the zone
C that has been scanned and is being worked on. ZNDATA holds
C this information globally.

C GENFIL is the name of the file to be written (an existing file
C is overwritten).
C ICOMP is the zone number, ITRC the unit number for user output,
C IER is an error status (=0 OK, =1 problem detected).
C IWF = 3 create/overwrite, = 4 check with user before overwriting.

C If iobs()=2 then there are blocks to write out within the
C geometry file.

      SUBROUTINE GEOWRITE(IFILG,GENFIL,ICOMP,ITRU,iwf,IER)
#include "building.h"
#include "model.h"
#include "geometry.h"
#include "prj3dv.h"
#include "espriou.h"

      integer lnblnk  ! function definition

C iaplic(1) toggle for shading; iaplic(2) toggle for insolation.
C   toggle for assessment where:
C   1 is 'all_applicable', 0 is manual selection of surfaces.
C nsurfcalc nb of shaded surfaces, lstsfcalc() list of applicable surfaces.
C nsurfinso nb of insolation sources, isurfinso() list of insolation sources.
      common/ishdirec/iaplic(MCOM,2),nsurfcalc(MCOM),lstsfcalc(MCOM,MS),
     &       nsurfinso(MCOM),isurfinso(MCOM,MS)

C ZNAME (12 char) - the zone name (from geometry.h).
C ZDESC (64 char) - zone notes (from geometry.h).
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)
      COMMON/C3/IC1(MCON),IE1(MCON),ICT(MCON),IC2(MCON),IE2(MCON)

C Thermal bridge common block is described in esruprj/edgeo.F
      integer nbrdg, ibrdg
      real psi,lngth,losspercent,totheatloss,thbrpercent
      real uavtotal
      common/THRBRDG/nbrdg(MCOM),psi(MCOM,16),lngth(MCOM,16),
     &               ibrdg(MCOM,16),losspercent(MCOM),totheatloss(MCOM),
     &               thbrpercent,uavtotal(MCOM)
      real bridgelen       ! Length (m) of potential thermal bridges in each zone.
      integer nbridgevt    ! Number of verticies associted with this bridge.
      integer bridgevlst   ! List of vertices associated with bridge.
      common/gbridge/bridgelen(MCOM,16),nbridgevt(MCOM,16),
     &  bridgevlst(MCOM,16,MV*2)
      common/THBRSCH/tbregime
      character tbregime*36

      integer mlcnamegt24         ! number of instances of long MLC names.
      integer mlcindex32          ! indices of long MLCs
      integer mlcindex24          ! indies of paired shorter MLCs
      integer mlcindexo32         ! indices of MLCs paired to long MLCs
      integer mlcindexo24         ! indies of short naed paired MLCs
      integer mlcx32refs          ! references to long MLCs
      common/mlclong/mlcnamegt24,mlcindex32(6),mlcindex24(6),
     &               mlcindexo32(6),mlcindexo24(6),mlcx32refs

C MRT sensors via geometry.h.
C Obstruction blocks common GS5/GS6 is found in geometry.h.

      CHARACTER GENFIL*72
      character outs*144,outsd*144,outs72*72,OTHSTR*30
      character left*72,leftd*72,right*62,rightd*62
      character louts*496,loutsd*496,outs2*144
      character tab*1    ! separator
      character dstmp*24
      character phrase*36 ! for thermal bridge tag
      character tokens*156,comment*76,aligned_str*156
      character sbound_ty*12,sbound_c2*6,sbound_e2*6  ! for c3 decode other strings
      integer  lso,lso2,lso3  ! string lengths of sother()
      integer  lsml           ! string length of SMLCN()
      integer  lspa           ! string length of sparent()
      integer lsna            ! string length of sname()
      integer lsot            ! string length of soft()
      integer luse1,luse2     ! string length of suse()
      logical newgeo          ! to use for testing if new/old geometry file.
      logical havebrdge       ! to use for thermal bridge existance.
      logical hasobstr        ! true if IOBS() is set to 1 and NB > 0
      logical all_short       ! true is uses short MLC names.
      logical firstcomment    ! only comment first thermal bridge

      IER=0
      tab=','    ! make the separator a comma.
      newgeo=.false.  ! assume older format geometry.
      havebrdge=.false.
      hasobstr=.false.

      call dstamp(dstmp)

C Open any existing file by this name (ask user for confirmation to
C over-write) or create a new file.
      if(iwf.eq.4)then
        CALL EFOPSEQ(IFILG,GENFIL,4,IER)
      else
        CALL EFOPSEQ(IFILG,GENFIL,3,IER)
      endif
      IF(IER.LT.0)THEN
        write(outs,'(3a)') 'Geometry file ',GENFIL(1:lnblnk(GENFIL)),
     &      ' could not be written.'
        call edisp(itru,outs)
        IER=1
        RETURN
      ENDIF
      write(currentfile,'(a)') GENFIL(1:lnblnk(GENFIL))

C Test for long MLC names.
      all_short=.true.
      if(mlcnamegt24.eq.0)then
        continue   ! no long MLC names in the database.
      else
        do I = 1,NZSUR(ICOMP)
          lnsmlcn=lnblnk(SMLCN(icomp,i))  ! set to at least 12 char
          if(lnsmlcn.gt.24) all_short=.false.
        enddo
      endif

C Write out the header including date stamp and documentation.
      lz=lnzname(ICOMP)
      write(tokens,'(5a)')  '*Geometry 1.1',tab,
     &  'GEN',tab,zname(ICOMP)(1:lz)
      write(comment,'(a)') ' tag file-version, file-format, zone-name'
      call align_comment(48,tokens,comment,aligned_str)
      write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
C      write(ifilg,'(6a)',IOSTAT=ios,ERR=13) '*Geometry 1.1',tab,
C     &  'GEN',tab,zname(ICOMP)(1:lz),
C     &  ' # tag version, format, zone name'
      write(ifilg,'(2a)',IOSTAT=IOS,ERR=13) '*date ',dstmp

      write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &  zdesc(ICOMP)(1:lnblnk(zdesc(ICOMP)))


C Scale the write statements for coordinates based on the bounding
C box of the zone.
      ilft=3
      if(ZXMN(icomp).lt.(-9.).or.ZYMN(icomp).lt.(-9.).or.
     &   ZZMN(icomp).lt.(-9.))then
        ilft=4
      elseif(ZXMX(icomp).gt.(9.).or.ZYMX(icomp).gt.(9.).or.
     &       ZZMX(icomp).gt.(9.))then
        ilft=4
      endif
      if(ZXMN(icomp).lt.(-99.).or.ZYMN(icomp).lt.(-99.).or.
     &   ZZMN(icomp).lt.(-99.))then
        ilft=5
      elseif(ZXMX(icomp).gt.(99.).or.ZYMX(icomp).gt.(99.).or.
     &       ZZMX(icomp).gt.(99.))then
        ilft=5
      endif
      if(ZXMN(icomp).lt.(-999.).or.ZYMN(icomp).lt.(-999.).or.
     &   ZZMN(icomp).lt.(-999.))then
        ilft=6
      elseif(ZXMX(icomp).gt.(999.).or.ZYMX(icomp).gt.(999.).or.
     &       ZZMX(icomp).gt.(999.))then
        ilft=6
      endif
      ilftd=ilft+6
      if(ilft.eq.3)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '# tag, X co-ord, Y co-ord, Z co-ord'
      elseif(ilft.eq.4)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '# tag, X co-ord,  Y co-ord,  Z co-ord'
      elseif(ilft.eq.5)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '# tag, X co-ord,   Y co-ord,   Z co-ord'
      elseif(ilft.eq.6)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '# tag, X co-ord,    Y co-ord,    Z co-ord'
      endif
      

C Write vertices with minimal white space comma separated. Comment
C vertex index every 5th.
      DO 960 I = 1,NZTV(ICOMP)
        im=mod(I,5)
        if(im.eq.0)then
          if(ilft.eq.3)then
            WRITE(outs,'(a,3F9.5)',IOSTAT=ios,ERR=13)'*vertex',
     &      X(I),Y(I),Z(I)
          elseif(ilft.eq.4)then
            WRITE(outs,'(a,3F10.5)',IOSTAT=ios,ERR=13)'*vertex',
     &      X(I),Y(I),Z(I)
          elseif(ilft.eq.5)then
            WRITE(outs,'(a,3F11.5)',IOSTAT=ios,ERR=13)'*vertex',
     &      X(I),Y(I),Z(I)
          elseif(ilft.eq.6)then
            WRITE(outs,'(a,3F12.5)',IOSTAT=ios,ERR=13)'*vertex',
     &      X(I),Y(I),Z(I)
          endif
C          call SDELIM(outs,outsd,'C',IW)
          write(comment,'(i3)') I
          call align_comment(48,outs,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
        else
          if(ilft.eq.3)then
            WRITE(outs,'(a,3F9.5)',IOSTAT=ios,ERR=13)'*vertex',
     &      X(I),Y(I),Z(I)
          elseif(ilft.eq.4)then
            WRITE(outs,'(a,3F10.5)',IOSTAT=ios,ERR=13)'*vertex',
     &      X(I),Y(I),Z(I)
          elseif(ilft.eq.5)then
            WRITE(outs,'(a,3F11.5)',IOSTAT=ios,ERR=13)'*vertex',
     &      X(I),Y(I),Z(I)
          elseif(ilft.eq.6)then
            WRITE(outs,'(a,3F12.5)',IOSTAT=ios,ERR=13)'*vertex',
     &      X(I),Y(I),Z(I)
          endif
C          call SDELIM(outs,outsd,'C',IW)
          write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &      outs(1:lnblnk(outs))
        endif
960   CONTINUE

C Edge list. Write as a packed list comma separated.
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &  '# tag, number of vertices followed by list of associated vert'

C If list of vertices is short enough then use align_comment.
      DO 970 I = 1,NZSUR(ICOMP)
        im=mod(I,5)
        if(im.eq.0)then
          write(louts,'(a,i4,102I4)',IOSTAT=ios,ERR=13) '*edges ',
     &      NVER(I),(JVN(I,J),J=1,NVER(I))
          call SDELIM(louts,loutsd,'C',IW)
          write(comment,'(i3)') I
          if(lnblnk(louts).lt.72)then    ! If short enough write to outs.
            write(outs,'(a,i4,102I4)',IOSTAT=ios,ERR=13) '*edges ',
     &        NVER(I),(JVN(I,J),J=1,NVER(I))
            call SDELIM(outs,outsd,'C',IW)
            call align_comment(48,outsd,comment,aligned_str)
            write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
          else
            write(ifilg,'(2a,i3)',IOSTAT=IOS,ERR=13)
     &        loutsd(1:lnblnk(loutsd)),'  # ',I
          endif
        else
          write(louts,'(a,i4,102I4)',IOSTAT=ios,ERR=13) '*edges ',
     &      NVER(I),(JVN(I,J),J=1,NVER(I))
          call SDELIM(louts,loutsd,'C',IW)
          write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &      loutsd(1:lnblnk(loutsd))
        endif
970   CONTINUE

C Surface list. Write as a packed list comma separated.
      if(all_short)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# surf attributes:'
        write(ifilg,'(4a)',IOSTAT=IOS,ERR=13)
     &  '#tag  Surf name    |Position|Child of     |',
     &  'USE type & subtype |',
     &  'Construction name        |Optical name     |',
     &  'Boundary  + attributes      Index & names'
      else
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# surf attributes:'
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &  '#  surf name, surf position VERT/CEIL/FLOR/SLOP/UNKN'
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &  '#  child of (surface name), useage (pair of tags) '
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &  '#  construction name, optical name'
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &  '#  boundary condition tag followed by two data items'
      endif

      DO 102 I=1,NZSUR(icomp)
        call OTHERINFO(icomp,i,OTHSTR)

C Form strings for boundary based on c3 commons.
        call decode_c3(icomp,i,sbound_ty,sbound_c2,sbound_e2)
C        write(6,*) 'geowrite: ',sbound_ty,sbound_c2,sbound_e2
        icc=izstocn(icomp,i)

C Based on the current contents of zboundarytype create string buffers.
C If any of the expected strings are blank write placeholders.
        if(icc.gt.0)then
          lso=lnblnk(sbound_ty)
          lso2=lnblnk(sbound_c2)
          if(lso2.eq.0)then
            sbound_c2='000'; lso2=3
          endif
          lso3=lnblnk(sbound_e2)
          if(lso3.eq.0)then
            sbound_e2='000'; lso3=3
          endif
          lsml=lnblnk(SMLCN(icomp,i))
          if(lsml.eq.0)then
            SMLCN(icomp,i)='UNKNOWN'; lsml=7
          endif
          lspa=lnblnk(SPARENT(icomp,i))
          if(lspa.gt.12) lspa=12
          lsna=lnblnk(SNAME(icomp,i))
          lsot=lnblnk(SOTF(icomp,i))
          loth=lnblnk(OTHSTR)
          luse1=lnblnk(SUSE(icomp,i,1))
          luse2=lnblnk(SUSE(icomp,i,2))

C Traps for possible blank strings if data structure corrupted.
C SMLCN might have spaces so write in several chunks and pack.
          if(SVFC(icomp,i)(1:4).eq.'    ') SVFC(icomp,i)='UNKN'
          if(SOTF(icomp,i)(1:4).eq.'    ') SOTF(icomp,i)='UNKN'
          if(SPARENT(icomp,i)(1:2).eq.'  ') SPARENT(icomp,i)='-'

C Debug.
C          WRITE(outs,'(a,10i3)',IOSTAT=ios,ERR=13) 'lengths ',lso,
C     &      lso2,lso3,lsml,lspa,lsna,lsot,loth,luse1,luse2
C          write(6,*) outs(1:lnblnk(outs))
C          WRITE(outs,'(20a)',IOSTAT=ios,ERR=13) '*surf ',
C     &      SNAME(icomp,i)(1:lsna),' ',SVFC(icomp,i),' ',
C     &      SPARENT(icomp,i)(1:lspa),' ',SUSE(icomp,i,1)(1:luse1),' ',
C     &      SUSE(icomp,i,2)(1:luse2),' ',SMLCN(icomp,i)(1:lsml),' ',
C     &      SOTF(icomp,i)(1:lsot),' ',sbound_ty(1:lso),' ',
C     &      sbound_c2(1:lso2),' ',sbound_e2(1:lso3)
C          call SDELIM(outs,outsd,'C',IW)
C          write(6,*) '2915 ',outsd(1:lnblnk(outsd))

C If no long names detected then use fixed column width.
          if(all_short)then
            WRITE(left,'(11a)',IOSTAT=ios,ERR=13) '*surf ',
     &        SNAME(icomp,i),' ,',SVFC(icomp,i),'    ,',
     &        SPARENT(icomp,i),' ,',SUSE(icomp,i,1)(1:8),' ,',
     &        SUSE(icomp,i,2)(1:8),' ,'
            WRITE(right,'(7a)',IOSTAT=ios,ERR=13)
     &        SOTF(icomp,i)(1:16),' ,',sbound_ty(1:12),' ',
     &        sbound_c2(1:6),' ',sbound_e2(1:6)
            write(ifilg,'(5a,i3,2a)',IOSTAT=IOS,ERR=13)
     &        left(1:lnblnk(left)),SMLCN(icomp,i)(1:24),' ,',
     &        right(1:lnblnk(right)),'  # ',I,' ',OTHSTR(1:loth)

          else
            WRITE(left,'(10a)',IOSTAT=ios,ERR=13) '*surf ',
     &        SNAME(icomp,i)(1:lsna),' ',SVFC(icomp,i),' ',
     &        SPARENT(icomp,i)(1:lspa),' ',SUSE(icomp,i,1)(1:luse1),' ',
     &        SUSE(icomp,i,2)(1:luse2)
            call SDELIM(left,leftd,'C',IW)  ! pack tokens before SMLCN
            WRITE(right,'(7a)',IOSTAT=ios,ERR=13)
     &        SOTF(icomp,i)(1:lsot),' ',sbound_ty(1:lso),' ',
     &        sbound_c2(1:lso2),' ',sbound_e2(1:lso3)
            call SDELIM(right,rightd,'C',IW) ! pack tokens after SMLCN
            write(ifilg,'(6a,i3,2a)',IOSTAT=IOS,ERR=13)
     &        leftd(1:lnblnk(leftd)),',',SMLCN(icomp,i)(1:lsml),',',
     &        rightd(1:lnblnk(rightd)),'  # ',I,' ',OTHSTR(1:loth)
          endif
        else

C There was an array fault so write out a placeholder surface. As
C there might be more than one fault, make name unique.

C Debug.
C          write(6,'(a,2i2,a)',IOSTAT=ios,ERR=13) '*surf,fault',i,
C     &      ',UNKN,-,-,-,UNKN,OPAQUE,EXTERIOR,000,000'

          write(ifilg,'(a,2i3,a)',IOSTAT=ios,ERR=13) '*surf,fault',i,
     &      ',UNKN,-,-,-,UNKN,OPAQUE,EXTERIOR,000,000'
        endif
  102 CONTINUE

C Default insolation distribution as packed string.
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
      WRITE(outs,'(a,4I4)',IOSTAT=ios,ERR=14)
     &  '*insol ',NDP(ICOMP),(IDPN(ICOMP,J),J=1,3)
      call SDELIM(outs,outsd,'S',IW)
      write(comment,'(a)') 'default insolation distribution '
      call align_comment(48,outsd,comment,aligned_str)
      write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))

C Shading calculation directives in the form of:
C *shad_calc,all_applic,20,20
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
      if(nsurfcalc(icomp).eq.0)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# shading directives'
        write(tokens,'(a)') '*shad_calc none'
        write(comment,'(a)') 'no temporal shading requested'
        call align_comment(48,tokens,comment,aligned_str)
        write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
      elseif(nsurfcalc(icomp).gt.0)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# shading directives'
        if(iaplic(icomp,1).eq.1)then
          write(tokens,'(a,i3)',IOSTAT=ios,ERR=13)
     &      '*shad_calc,all_applicable ',nsurfcalc(icomp)
          write(comment,'(a)') 'list of surfs'
          call align_comment(48,tokens,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
          write(louts,'(90i4)',IOSTAT=ios,ERR=14)
     &      (lstsfcalc(ICOMP,J),J=1,nsurfcalc(icomp))
          call SDELIM(louts,loutsd,'S',IW)
          write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &      loutsd(1:lnblnk(loutsd))
        elseif(iaplic(icomp,1).eq.0)then
          write(tokens,'(a,i3)',IOSTAT=ios,ERR=13)
     &      '*shad_calc,list',nsurfcalc(icomp)
          write(comment,'(a)') 'list of surfs'
          call align_comment(48,tokens,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
          write(louts,'(90i4)',IOSTAT=ios,ERR=14)
     &      (lstsfcalc(ICOMP,J),J=1,nsurfcalc(icomp))
          call SDELIM(louts,loutsd,'S',IW)
          write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &      loutsd(1:lnblnk(loutsd))
        endif
      endif

C Insolation calculation directives.
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
      if(nsurfinso(icomp).eq.0)then
        write(tokens,'(a)') '*insol_calc none'
        write(comment,'(a)') 'no insolation requested'
        call align_comment(48,tokens,comment,aligned_str)
        write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
      elseif(nsurfinso(icomp).gt.0)then
        if(iaplic(icomp,2).eq.1)then
          write(tokens,'(a,i3)',IOSTAT=ios,ERR=13)
     &      '*insol_calc all_applicable ',nsurfinso(icomp)
          write(comment,'(a)') 'insolation sources'
          call align_comment(48,tokens,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
          write(louts,'(90i4)',IOSTAT=ios,ERR=14)
     &      (isurfinso(ICOMP,J),J=1,nsurfinso(icomp))
          call SDELIM(louts,loutsd,'S',IW)
          write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &      loutsd(1:lnblnk(loutsd))
        elseif(iaplic(icomp,2).eq.0)then
          write(tokens,'(a,i3)',IOSTAT=ios,ERR=13)
     &      '*insol_calc list',nsurfinso(icomp)
          write(comment,'(a)') 'insolation sources'
          call align_comment(48,tokens,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
          write(louts,'(90i4)',IOSTAT=ios,ERR=14)
     &      (isurfinso(ICOMP,J),J=1,nsurfinso(icomp))
          call SDELIM(louts,loutsd,'S',IW)
          write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &      loutsd(1:lnblnk(loutsd))
        endif
      endif

C Surfaces associated with base.
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
      if(izbaselist(icomp).eq.0)then
        WRITE(outs,'(a,i2,F9.2,i2)',IOSTAT=ios,ERR=13) '*base_list ',
     &    izbaselist(icomp),ZBASEA(ICOMP),IUZBASEA(ICOMP)
        call SDELIM(outs,outsd,'S',IW)
        write(comment,'(a)') 'zone base'
        call align_comment(48,outsd,comment,aligned_str)
        write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
      else

C Write the list and then the data after the list.
        WRITE(outs,'(a,21i4)',IOSTAT=ios,ERR=13) '*base_list ',
     &    izbaselist(icomp),(IBASES(icomp,J),J=1,izbaselist(icomp))
        call SDELIM(outs,outsd,'S',IW)
        if(lnblnk(outsd).lt.124)then
          write(outs2,'(2a,f9.2,i2)') outsd(1:lnblnk(outsd)),',',
     &      ZBASEA(ICOMP),IUZBASEA(ICOMP)   ! combine list and area
          write(comment,'(a)') 'zone base list'
          call align_comment(48,outs2,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
        else
          write(outs72,'(F9.2,i2)') ZBASEA(ICOMP),IUZBASEA(ICOMP)
          write(ifilg,'(4a)',IOSTAT=IOS,ERR=13)
     &      outsd(1:lnblnk(outsd)),',',outs72(1:lnblnk(outs72)),
     &      '  # zone base list'
        endif
      endif

C Write thermal bridge info if detected or the losspercent is non-zero.
      if(losspercent(icomp).gt.0.0) havebrdge=.true.
      if(nbrdg(icomp).gt.0.or.havebrdge)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        write(ifilg,'(2a)',IOSTAT=IOS,ERR=13) 
     &    '# bridge:     user        loss%   total      total '
        write(ifilg,'(2a)',IOSTAT=IOS,ERR=13) 
     &    '#             fraction     W/K,   loss W/K   UA W/K'
        WRITE(outs,'(a,F7.3,3F10.3)',IOSTAT=ios,ERR=13) 
     &   '*bridge_start ',thbrpercent,losspercent(icomp),
     &    totheatloss(icomp),uavtotal(icomp)
        if(tbregime(1:2).ne.' ')then  ! If TB regime not blank include.
          write(ifilg,'(3a)',IOSTAT=IOS,ERR=13)
     &      outs(1:lnblnk(outs)),'  ',tbregime(1:lnblnk(tbregime))
        else
          write(ifilg,'(a)',IOSTAT=IOS,ERR=13) outs(1:lnblnk(outs))
        endif
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        firstcomment=.true.
        if(nbrdg(icomp).gt.0)then
          do 88 itb=1,nbrdg(icomp)
            if(ibrdg(icomp,itb).eq.1)then
              phrase='roof-wall '
            elseif(ibrdg(icomp,itb).eq.2)then
              phrase='wall-ground floor '
            elseif(ibrdg(icomp,itb).eq.3)then
              phrase='wall-wall (convex corner) '
            elseif(ibrdg(icomp,itb).eq.4)then
              phrase='wall-wall (concave corner) '
            elseif(ibrdg(icomp,itb).eq.5)then
              phrase='wall-floor (exposed floor) '
            elseif(ibrdg(icomp,itb).eq.6)then
              phrase='lintel above window or door '
            elseif(ibrdg(icomp,itb).eq.7)then
              phrase='sill below window '
            elseif(ibrdg(icomp,itb).eq.8)then
              phrase='jamb at window or door '
            elseif(ibrdg(icomp,itb).eq.9)then
              phrase='wall-gable '
            elseif(ibrdg(icomp,itb).eq.10)then
              phrase='wall-parapet '
            elseif(ibrdg(icomp,itb).eq.11)then
              phrase='wall-intermediate-floor '
            elseif(ibrdg(icomp,itb).eq.12)then
              phrase='wall-partition '
            elseif(ibrdg(icomp,itb).eq.13)then
              phrase='glass-frame '
            elseif(ibrdg(icomp,itb).eq.14)then
              phrase='balcony'
            elseif(ibrdg(icomp,itb).eq.15)then
              phrase='user-defined-a'
            elseif(ibrdg(icomp,itb).eq.16)then
              phrase='user-defined'
            endif

C The phrase can have spaces in it so do not pack string. If
C nbridgevt is nonzero also include additional line with
C associated vertices. Only comment the first instance.
            if(nbridgevt(icomp,ibrdg(icomp,itb)).eq.0)then
              WRITE(tokens,'(4a,F7.3,a,F8.4,a)',IOSTAT=ios,ERR=13)
     &          '*bridge','  ',phrase(1:lnblnk(phrase)),tab,
     &          lngth(icomp,itb),' ',psi(icomp,itb),'  0'
              write(comment,'(a)') 
     &          'type length psi-value nb of assoc verts'
              if(firstcomment)then
                call align_comment(48,tokens,comment,aligned_str)
                write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
                firstcomment=.false.
              else
                write(ifilg,'(a)',IOSTAT=IOS,ERR=13) 
     &            tokens(1:lnblnk(tokens))
              endif
            else
              WRITE(tokens,'(4a,F7.3,a,F8.4,i4)',IOSTAT=ios,ERR=13)
     &          '*bridge','  ',phrase(1:lnblnk(phrase)),tab,
     &          lngth(icomp,itb),' ',psi(icomp,itb),
     &          nbridgevt(icomp,ibrdg(icomp,itb))
              write(comment,'(a)') 
     &          'type length psi-value nb of assoc verts'
              if(firstcomment)then
                call align_comment(48,tokens,comment,aligned_str)
                write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
                firstcomment=.false.
              else
                write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &            tokens(1:lnblnk(tokens))
              endif
              iabv=nbridgevt(icomp,ibrdg(icomp,itb))
              write(louts,'(90i4)',IOSTAT=ios,ERR=14)
     &          (bridgevlst(icomp,ibrdg(icomp,itb),J),J=1,iabv)
              call SDELIM(louts,loutsd,'S',IW)
              if(lnblnk(loutsd).lt.100)then
                write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &            loutsd(1:lnblnk(loutsd))
              else
                write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &            loutsd(1:lnblnk(loutsd))
              endif
            endif
  88      continue
        endif
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '*end_bridge'
      endif

C iobs is 2 if there are block descriptions to include in this zone
C geometry file also write the NOX and NOZ values.
      if(IOBS(icomp).eq.2.and.nbobs(icomp).gt.0)then
        hasobstr=.true.
      endif
      if(hasobstr)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# block entities:'
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '#  *obs = obstructions'
        write(tokens,'(2a,i3,i3)',IOSTAT=IOS,ERR=13)
     &  '*block_start',tab,NOX(icomp),NOZ(icomp)
        write(comment,'(a)') 'geometric blocks'
        call align_comment(48,tokens,comment,aligned_str)
        write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
        do ib=1,nbobs(icomp)
          if(BLOCKTYP(icomp,ib)(1:4).eq.'obs3')then
            WRITE(outs,'(2a,6F9.4,3F8.2,F6.2,1X,A)',
     &        IOSTAT=ios,ERR=13)
     &        '*obs3',tab,XOB(icomp,ib),YOB(icomp,ib),ZOB(icomp,ib),
     &        DXOB(icomp,ib),DYOB(icomp,ib),DZOB(icomp,ib),
     &        BANGOB(icomp,ib,1),BANGOB(icomp,ib,2),BANGOB(icomp,ib,3),
     &        OPOB(icomp,ib),BLOCKNAME(icomp,ib)
            call SDELIM(outs,outsd,'C',IW)
            lnbm=lnblnk(BLOCKMAT(icomp,ib))
            write(ifilg,'(4a,i3)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',BLOCKMAT(icomp,ib)(1:lnbm),
     &        '  # block ',ib
          elseif(BLOCKTYP(icomp,ib)(1:4).eq.'obsp')then
            WRITE(outs,'(2a,F7.2,1X,A)',IOSTAT=ios,ERR=13)
     &        '*obsp',' 8 6 ',OPOB(icomp,ib),BLOCKNAME(icomp,ib)
            call SDELIM(outs,outsd,'C',IW)
            lnbm=lnblnk(BLOCKMAT(icomp,ib))
            write(ifilg,'(4a,i3,a)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',BLOCKMAT(icomp,ib)(1:lnbm),
     &        '  # block ',ib,' coords follow:'

            WRITE(outs,'(12F9.4)',IOSTAT=ios,ERR=13)
     &        XBP(icomp,ib,1),YBP(icomp,ib,1),ZBP(icomp,ib,1),
     &        XBP(icomp,ib,2),YBP(icomp,ib,2),ZBP(icomp,ib,2),
     &        XBP(icomp,ib,3),YBP(icomp,ib,3),ZBP(icomp,ib,3),
     &        XBP(icomp,ib,4),YBP(icomp,ib,4),ZBP(icomp,ib,4)
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(2A)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),'  # 1-4 '

            WRITE(outs,'(12F9.4)',IOSTAT=ios,ERR=13)
     &        XBP(icomp,ib,5),YBP(icomp,ib,5),ZBP(icomp,ib,5),
     &        XBP(icomp,ib,6),YBP(icomp,ib,6),ZBP(icomp,ib,6),
     &        XBP(icomp,ib,7),YBP(icomp,ib,7),ZBP(icomp,ib,7),
     &        XBP(icomp,ib,8),YBP(icomp,ib,8),ZBP(icomp,ib,8)
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(2A)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),'  # 5-8 '

          elseif(BLOCKTYP(icomp,ib)(1:4).eq.'obs ')then
            WRITE(outs,'(2a,6F9.4,F8.2,F6.2,1X,A)',
     &        IOSTAT=ios,ERR=13)
     &        '*obs',tab,XOB(icomp,ib),YOB(icomp,ib),ZOB(icomp,ib),
     &        DXOB(icomp,ib),DYOB(icomp,ib),DZOB(icomp,ib),
     &        BANGOB(icomp,ib,1),OPOB(icomp,ib),BLOCKNAME(icomp,ib)
            call SDELIM(outs,outsd,'C',IW)
            lnbm=lnblnk(BLOCKMAT(icomp,ib))
            write(ifilg,'(4a,i3)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',BLOCKMAT(icomp,ib)(1:lnbm),
     &        '  # block ',ib
          endif
        enddo  !  ib

C Adapt logic in case there are also mrt sensors.
        if(IVF(icomp).eq.1.and.ncub(icomp).gt.0)then
          continue
        else
          write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '*end_block'
        endif
      endif

C If there is a viewfactor file and mrt blocks echo that into the
C zone geometry file so block definitions are persisitent (they
C tend to get lost when surfaces get added and vwf file is scanned).
C If there were no obstructions then include a *block_start.
      if(IVF(icomp).eq.1)then
        if(ncub(icomp).gt.0)then
          if(hasobstr)then
            continue
          else
            write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
            write(tokens,'(3a)',IOSTAT=IOS,ERR=13)
     &        '*block_start',tab,' 20  20 '
            write(comment,'(a)') 'mrt blocks'
            call align_comment(48,tokens,comment,aligned_str)
            write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
C            write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# mrt blocks:'
          endif
          do ib=1,ncub(icomp)
            WRITE(outs,'(2a,6F9.4,F8.2,1X,A)',IOSTAT=ios,ERR=13)
     &        '*mrt',tab,XOC(ib),YOC(ib),ZOC(ib),
     &        DXC(ib),DYC(ib),DZC(ib),
     &        CANG(ib),CUBN(ib)
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(2a,i3)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),'  # mrt ',ib
          enddo
          write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '*end_block'
        endif
      endif

C Visual entities are in the next section.
      if(nbvis(icomp).gt.0)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        write(ifilg,'(3a)',IOSTAT=IOS,ERR=13) '# visual entities:',
     &    ' *vis = visual blocks  *vis3 = visual 3 axis',
     &    '  *visp = visual polys  *vobject = named collection'
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '*visual_start'
        do ib=1,nbvis(icomp)
          if(VISTYP(icomp,ib)(1:4).eq.'vis3')then
            WRITE(outs,'(2a,6F10.4,3F8.2,F6.2,1X,A)',
     &        IOSTAT=ios,ERR=13)
     &        '*vis3',tab,XOV(icomp,ib),YOV(icomp,ib),ZOV(icomp,ib),
     &        DXOV(icomp,ib),DYOV(icomp,ib),DZOV(icomp,ib),
     &        BANGOV(icomp,ib,1),BANGOV(icomp,ib,2),BANGOV(icomp,ib,3),
     &        OPOV(icomp,ib),VISNAME(icomp,ib)
            call SDELIM(outs,outsd,'C',IW)
            lnvm=lnblnk(VISMAT(icomp,ib))
            write(ifilg,'(4a,i3)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',VISMAT(icomp,ib)(1:lnvm),
     &        '  # visual block ',ib
          elseif(VISTYP(icomp,ib)(1:4).eq.'visp')then
            WRITE(outs,'(2a,F7.2,1X,A)',IOSTAT=ios,ERR=13)
     &        '*visp',' 8 6 ',OPOV(icomp,ib),VISNAME(icomp,ib)
            lnvm=lnblnk(VISMAT(icomp,ib))
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(4a,i3,a)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',VISMAT(icomp,ib)(1:lnvm),
     &        '  # visual ',ib,' coords follow:'

            WRITE(outs,'(12F9.4)',IOSTAT=ios,ERR=13)
     &        XVP(icomp,ib,1),YVP(icomp,ib,1),ZVP(icomp,ib,1),
     &        XVP(icomp,ib,2),YVP(icomp,ib,2),ZVP(icomp,ib,2),
     &        XVP(icomp,ib,3),YVP(icomp,ib,3),ZVP(icomp,ib,3),
     &        XVP(icomp,ib,4),YVP(icomp,ib,4),ZVP(icomp,ib,4)
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(2A)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),'  # 1-4 '

            WRITE(outs,'(12F9.4)',IOSTAT=ios,ERR=13)
     &        XVP(icomp,ib,5),YVP(icomp,ib,5),ZVP(icomp,ib,5),
     &        XVP(icomp,ib,6),YVP(icomp,ib,6),ZVP(icomp,ib,6),
     &        XVP(icomp,ib,7),YVP(icomp,ib,7),ZVP(icomp,ib,7),
     &        XVP(icomp,ib,8),YVP(icomp,ib,8),ZVP(icomp,ib,8)
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(2A)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),'  # 5-8 '

          elseif(VISTYP(icomp,ib)(1:4).eq.'vis ')then
            WRITE(outs,'(2a,6F10.4,F8.2,F6.2,1X,A)',
     &        IOSTAT=ios,ERR=13)
     &        '*vis',tab,XOV(icomp,ib),YOV(icomp,ib),ZOV(icomp,ib),
     &        DXOV(icomp,ib),DYOV(icomp,ib),DZOV(icomp,ib),
     &        BANGOV(icomp,ib,1),OPOV(icomp,ib),VISNAME(icomp,ib)
            lnvm=lnblnk(VISMAT(icomp,ib))
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(4a,i3)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',VISMAT(icomp,ib)(1:lnvm),
     &        '  # visual ',ib
          endif
        enddo   ! ib

C If there are objects (collections of entities) write them. Because there
C may be spaces in VOBJDESC take care.
        if(NBVOBJ(icomp).gt.0)then
          do ib=1,NBVOBJ(icomp)
            WRITE(outs,'(6a,i2,a)',IOSTAT=ios,ERR=13)
     &        '*vobject',tab,
     &        VOBJNAME(icomp,ib)(1:lnblnk(VOBJNAME(icomp,ib))),
     &        tab,VOBJDESC(icomp,ib)(1:lnblnk(VOBJDESC(icomp,ib))),tab,
     &        NBVOBJLIST(icomp,ib),tab

C Append the list of visual entities comma separated. Similar to logic
C in esru_lib.F subroutine aslist.
            outs2=' '; ix=1; ixl=0
            do ibo=1,NBVOBJLIST(icomp,ib)
              lna=lnblnk(VOBJLIST(icomp,ib,ibo))
              if(lna.eq.1)then
                ixl=ix
              else
                ixl=ix+(lna-1)
              endif
              write(outs2(ix:ixl),'(a)')VOBJLIST(icomp,ib,ibo)(1:lna)
              if(ibo.lt.NBVOBJLIST(icomp,ib))then
                write(outs2(ixl+1:ixl+1),'(a)') ','
                ix=ix+lna+1
              else
                ix=ix+lna+1
              endif
            enddo  ! of ibo
            write(ifilg,'(2a)',IOSTAT=IOS,ERR=13)
     &        outs(1:lnblnk(outs)),outs2(1:lnblnk(outs2))
          enddo    ! of ib
        endif      ! of nbvobj
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '*end_visual'
      endif

C If file written with this source and the version number is 1.0
C then reset it to 1.1.
      call eclose(gversion(icomp),1.1,0.01,newgeo)
      if(.NOT.newgeo) gversion(icomp) = 1.1

   99 CALL ERPFREE(IFILG,ios)
      RETURN

C Error messages.
   13 if(IOS.eq.2)then
        CALL USRMSG('No permission to write ',GENFIL,'W')
      else
        CALL USRMSG('File write error in ',GENFIL,'W')
      endif
      IER=1
      GOTO 99
   14 if(IOS.eq.2)then
        CALL USRMSG('No prmission to write array in ',GENFIL,'W')
      else
        CALL USRMSG('Long arrary write error in ',GENFIL,'W')
      endif
      IER=1
      GOTO 99

      END


C ************* GEOWRITE2
C Writes a geometry file (GEN V1.2 type) based on information currently
C held in model-wide common blocks.  
C GENFIL is the name of the file to be written to (any existing file
C is overwritten).
C ICOMP is the zone number,
C ITRC is the unit number for user output.
C IER=0 OK IER=1 problem.
C IWF = 3 create/overwrite, =4 check with user before overwriting.

C If iobs()=2 then there are blocks to write out within the
C geometry file.

      SUBROUTINE GEOWRITE2(IFILG,GENFIL,ICOMP,ITRU,iwf,IER)
#include "building.h"
#include "model.h"
#include "geometry.h"
#include "prj3dv.h"
#include "espriou.h"

      integer lnblnk  ! function definition

C iaplic(1) toggle for shading; iaplic(2) toggle for insolation.
C   toggle for assessment where:
C   1 is 'all_applicable', 0 is manual selection of surfaces.
C nsurfcalc nb of shaded surfaces, lstsfcalc() list of applicable surfaces.
C nsurfinso nb of insolation sources, isurfinso() list of insolation sources.
      common/ishdirec/iaplic(MCOM,2),nsurfcalc(MCOM),lstsfcalc(MCOM,MS),
     &       nsurfinso(MCOM),isurfinso(MCOM,MS)

C ZNAME (12 char) - the zone name (from geometry.h).
C ZDESC (64 char) - zone notes (from geometry.h).
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

C Thermal bridge common block is described in esruprj/edgeo.F
      integer nbrdg, ibrdg
      real psi,lngth,losspercent,totheatloss,thbrpercent
      real uavtotal
      common/THRBRDG/nbrdg(MCOM),psi(MCOM,16),lngth(MCOM,16),
     &               ibrdg(MCOM,16),losspercent(MCOM),totheatloss(MCOM),
     &               thbrpercent,uavtotal(MCOM)
      real bridgelen       ! Length (m) of potential thermal bridges in each zone.
      integer nbridgevt    ! Number of verticies associted with this bridge.
      integer bridgevlst   ! List of vertices associated with bridge.
      common/gbridge/bridgelen(MCOM,16),nbridgevt(MCOM,16),
     &  bridgevlst(MCOM,16,MV*2)
      common/THBRSCH/tbregime
      character tbregime*36

      integer mlcnamegt24         ! number of instances of long MLC names.
      integer mlcindex32          ! indices of long MLCs
      integer mlcindex24          ! indies of paired shorter MLCs
      integer mlcindexo32         ! indices of MLCs paired to long MLCs
      integer mlcindexo24         ! indies of short naed paired MLCs
      integer mlcx32refs          ! references to long MLCs
      common/mlclong/mlcnamegt24,mlcindex32(6),mlcindex24(6),
     &               mlcindexo32(6),mlcindexo24(6),mlcx32refs

C MRT sensors via geometry.h.

C Obstruction blocks common GS5/GS6 is found in geometry.h.

      CHARACTER GENFIL*72
      character outs*144,outsd*144,outs72*72,OTHSTR*30
      character sbound_ty*12,sbound_c2*6,sbound_e2*6  ! equiv to sother
      character left*72,leftd*72,right*62,rightd*62
      character louts*496,loutsd*496,outs2*144
      character tab*1    ! separator
      character dstmp*24
      character phrase*36 ! for thermal bridge tag
      character tokens*156,comment*76,aligned_str*156
      integer  lso,lso2,lso3  ! string lengths of sother()
      integer  lsml           ! string length of SMLCN()
      integer  lspa           ! string length of sparent()
      integer lsna            ! string length of sname()
      integer lsot            ! string length of soft()
      integer luse1,luse2     ! string length of suse()
      logical newgeo  ! to use for testing if new/old geometry file.
      logical havebrdge ! to use for thermal bridge existance.
      logical hasobstr  !  true if IOBS() is set to 1 and NB > 0
      logical all_short       ! true is uses short MLC names.
      logical firstcomment    ! only comment first thermal bridge

      IER=0
      tab=','    ! make the separator a comma.
      newgeo=.false.  ! assume older format geometry.
      havebrdge=.false.
      hasobstr=.false.

      call dstamp(dstmp)

C Open any existing file by this name (ask user for confirmation to
C over-write) or create a new file.
      if(iwf.eq.4)then
        CALL EFOPSEQ(IFILG,GENFIL,4,IER)
      else
        CALL EFOPSEQ(IFILG,GENFIL,3,IER)
      endif
      IF(IER.LT.0)THEN
        write(outs,'(3a)') 'Geometry file ',GENFIL(1:lnblnk(GENFIL)),
     &      ' could not be written.'
        call edisp(itru,outs)
        IER=1
        RETURN
      endif
      write(currentfile,'(a)') GENFIL(1:lnblnk(GENFIL))

C Test for long MLC names.
      all_short=.true.
      if(mlcnamegt24.eq.0)then
        continue   ! no long MLC names in the database.
      else
        do I = 1,NZSUR(ICOMP)
          lnsmlcn=lnblnk(SMLCN(icomp,i))  ! set to at least 12 char
          if(lnsmlcn.gt.24) all_short=.false.
        enddo
      endif

C Write out the header including date stamp and documentation.
      lz=lnzname(ICOMP)
      write(tokens,'(5a)')  '*Geometry 1.1',tab,
     &  'GEN',tab,zname(ICOMP)(1:lz)
      write(comment,'(a)') ' tag file-version, file-format, zone-name'
      call align_comment(48,tokens,comment,aligned_str)
      write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
      write(ifilg,'(2a)',IOSTAT=IOS,ERR=13) '*date ',dstmp
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &  zdesc(ICOMP)(1:lnblnk(zdesc(ICOMP)))

C Scale the write statements for coordinates based on the bounding
C box of the zone.
      ilft=3
      if(ZXMN(icomp).lt.(-9.).or.ZYMN(icomp).lt.(-9.).or.
     &   ZZMN(icomp).lt.(-9.))then
        ilft=4
      elseif(ZXMX(icomp).gt.(9.).or.ZYMX(icomp).gt.(9.).or.
     &       ZZMX(icomp).gt.(9.))then
        ilft=4
      endif
      if(ZXMN(icomp).lt.(-99.).or.ZYMN(icomp).lt.(-99.).or.
     &   ZZMN(icomp).lt.(-99.))then
        ilft=5
      elseif(ZXMX(icomp).gt.(99.).or.ZYMX(icomp).gt.(99.).or.
     &       ZZMX(icomp).gt.(99.))then
        ilft=5
      endif
      if(ZXMN(icomp).lt.(-999.).or.ZYMN(icomp).lt.(-999.).or.
     &   ZZMN(icomp).lt.(-999.))then
        ilft=6
      elseif(ZXMX(icomp).gt.(999.).or.ZYMX(icomp).gt.(999.).or.
     &       ZZMX(icomp).gt.(999.))then
        ilft=6
      endif
      ilftd=ilft+6
      if(ilft.eq.3)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '# tag, X co-ord, Y co-ord, Z co-ord'
      elseif(ilft.eq.4)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '# tag, X co-ord,  Y co-ord,  Z co-ord'
      elseif(ilft.eq.5)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '# tag, X co-ord,   Y co-ord,   Z co-ord'
      elseif(ilft.eq.6)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '# tag, X co-ord,    Y co-ord,    Z co-ord'
      endif

C Write vertices with minimal white space comma separated.
      DO 960 I = 1,NZTV(ICOMP)
        im=mod(I,5)
        if(im.eq.0)then
          if(ilft.eq.3)then
            WRITE(outs,'(a,3F9.5)',IOSTAT=ios,ERR=13)'*vertex',
     &        szcoords(ICOMP,I,1),szcoords(ICOMP,I,2),
     &        szcoords(ICOMP,I,3)
          elseif(ilft.eq.4)then
            WRITE(outs,'(a,3F10.5)',IOSTAT=ios,ERR=13)'*vertex',
     &        szcoords(ICOMP,I,1),szcoords(ICOMP,I,2),
     &        szcoords(ICOMP,I,3)
          elseif(ilft.eq.5)then
            WRITE(outs,'(a,3F11.5)',IOSTAT=ios,ERR=13)'*vertex',
     &        szcoords(ICOMP,I,1),szcoords(ICOMP,I,2),
     &        szcoords(ICOMP,I,3)
          elseif(ilft.eq.6)then
            WRITE(outs,'(a,3F12.5)',IOSTAT=ios,ERR=13)'*vertex',
     &        szcoords(ICOMP,I,1),szcoords(ICOMP,I,2),
     &        szcoords(ICOMP,I,3)
          endif
          write(comment,'(i3)') I
          call align_comment(48,outs,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
        else
          if(ilft.eq.3)then
            WRITE(outs,'(a,3F9.5)',IOSTAT=ios,ERR=13)'*vertex',
     &        szcoords(ICOMP,I,1),szcoords(ICOMP,I,2),
     &        szcoords(ICOMP,I,3)
          elseif(ilft.eq.4)then
            WRITE(outs,'(a,3F10.5)',IOSTAT=ios,ERR=13)'*vertex',
     &        szcoords(ICOMP,I,1),szcoords(ICOMP,I,2),
     &        szcoords(ICOMP,I,3)
          elseif(ilft.eq.5)then
            WRITE(outs,'(a,3F11.5)',IOSTAT=ios,ERR=13)'*vertex',
     &        szcoords(ICOMP,I,1),szcoords(ICOMP,I,2),
     &        szcoords(ICOMP,I,3)
          elseif(ilft.eq.6)then
            WRITE(outs,'(a,3F12.5)',IOSTAT=ios,ERR=13)'*vertex',
     &        szcoords(ICOMP,I,1),szcoords(ICOMP,I,2),
     &        szcoords(ICOMP,I,3)
          endif
          write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &      outs(1:lnblnk(outs))
        endif
960   CONTINUE

C Edge list. Write as a packed list comma separated.
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &  '# tag, number of vertices followed by list of associated vert'

      DO 970 I = 1,NZSUR(ICOMP)
        im=mod(I,5)
        if(im.eq.0)then
          write(louts,'(a,i4,102I4)',IOSTAT=ios,ERR=13) '*edges ',
     &      isznver(ICOMP,I),(iszjvn(ICOMP,I,J),J=1,isznver(ICOMP,I))
          call SDELIM(louts,loutsd,'C',IW)
          write(comment,'(i3)') I
          if(lnblnk(louts).lt.72)then    ! If short enough write to outs.
            write(outs,'(a,i4,102I4)',IOSTAT=ios,ERR=13) '*edges ',
     &        isznver(ICOMP,I),(iszjvn(ICOMP,I,J),J=1,isznver(ICOMP,I))
            call SDELIM(outs,outsd,'C',IW)
            call align_comment(48,outsd,comment,aligned_str)
            write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
          else
            write(ifilg,'(2a,i3)',IOSTAT=IOS,ERR=13)
     &        loutsd(1:lnblnk(loutsd)),'  # ',I
          endif
        else
          write(louts,'(a,i4,102I4)',IOSTAT=ios,ERR=13) '*edges ',
     &      isznver(ICOMP,I),(iszjvn(ICOMP,I,J),J=1,isznver(ICOMP,I))
          call SDELIM(louts,loutsd,'C',IW)
          write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &      loutsd(1:lnblnk(loutsd))
        endif
970   CONTINUE

C Surface list. Write as a packed list comma separated.
      if(all_short)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# surf attributes:'
        write(ifilg,'(4a)',IOSTAT=IOS,ERR=13)
     &  '#tag  Surf name    |Position|Child of     |',
     &  'USE type & subtype |',
     &  'Construction name        |Optical name     |',
     &  'Boundary  + attributes      Index & names'
      else
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# surf attributes:'
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '#  surf name, surf position VERT/CEIL/FLOR/SLOP/UNKN'
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '#  child of (surface name), useage (pair of tags) '
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '#  construction name, optical name'
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13)
     &    '#  boundary condition tag followed by two data items'
      endif

      DO 102 I=1,NZSUR(icomp)
        call OTHERINFO(icomp,i,OTHSTR)

C Form strings for boundary columns based on zboundarytype.  
        call decode_zsbound(icomp,i,sbound_ty,sbound_c2,sbound_e2)
C        write(6,*) 'geowrite2 ',sbound_ty,sbound_c2,sbound_e2
        icc=izstocn(icomp,i)

C Based on the current contents of zboundarytype create string buffer.
C If any of the expected strings are blank write placeholders.
        if(icc.gt.0)then
          lso=lnblnk(sbound_ty)
          lso2=lnblnk(sbound_c2)
          if(lso2.eq.0)then
            sbound_c2='000'; lso2=3
          endif
          lso3=lnblnk(sbound_e2)
          if(lso3.eq.0)then
            sbound_e2='000'; lso3=3
          endif

          lsml=lnblnk(SMLCN(icomp,i))
          if(lsml.eq.0)then
            SMLCN(icomp,i)='UNKNOWN'; lsml=7
          endif
          lspa=lnblnk(SPARENT(icomp,i))
          if(lspa.gt.12) lspa=12
          lsna=lnblnk(SNAME(icomp,i))
          lsot=lnblnk(SOTF(icomp,i))
          loth=lnblnk(OTHSTR)
          luse1=lnblnk(SUSE(icomp,i,1))
          luse2=lnblnk(SUSE(icomp,i,2))

C Traps for possible blank strings if data structure corrupted.
C SMLCN might have spaces so write in several chunks and pack.
          if(SVFC(icomp,i)(1:4).eq.'    ') SVFC(icomp,i)='UNKN'
          if(SOTF(icomp,i)(1:4).eq.'    ') SOTF(icomp,i)='UNKN'
          if(SPARENT(icomp,i)(1:2).eq.'  ') SPARENT(icomp,i)='-'

C Debug.
C          WRITE(outs,'(a,10i3)',IOSTAT=ios,ERR=13) 'lengths ',lso,
C     &      lso2,lso3,lsml,lspa,lsna,lsot,loth,luse1,luse2
C          write(6,*) outs(1:lnblnk(outs))
C          WRITE(outs,'(20a)',IOSTAT=ios,ERR=13) '*surf ',
C     &      SNAME(icomp,i)(1:lsna),' ',SVFC(icomp,i),' ',
C     &      SPARENT(icomp,i)(1:lspa),' ',SUSE(icomp,i,1)(1:luse1),' ',
C     &      SUSE(icomp,i,2)(1:luse2),' ',SMLCN(icomp,i)(1:lsml),' ',
C     &      SOTF(icomp,i)(1:lsot),' ',sbound_ty(1:lso),' ',
C     &      sbound_c2(1:lso2),' ',sbound_e2(1:lso3)
C          call SDELIM(outs,outsd,'C',IW)
C          write(6,*) '2915 ',outsd(1:lnblnk(outsd))


C If no long names detected then write in fixed column width mode.
          if(all_short)then
            WRITE(left,'(11a)',IOSTAT=ios,ERR=13) '*surf ',
     &        SNAME(icomp,i),' ,',SVFC(icomp,i),'    ,',
     &        SPARENT(icomp,i),' ,',SUSE(icomp,i,1)(1:8),' ,',
     &        SUSE(icomp,i,2)(1:8),' ,'
            WRITE(right,'(7a)',IOSTAT=ios,ERR=13)
     &        SOTF(icomp,i)(1:16),' ,',sbound_ty(1:12),' ',
     &        sbound_c2(1:6),' ',sbound_e2(1:6)
            write(ifilg,'(5a,i3,2a)',IOSTAT=IOS,ERR=13)
     &        left(1:lnblnk(left)),SMLCN(icomp,i)(1:24),' ,',
     &        right(1:lnblnk(right)),'  # ',I,' ',OTHSTR(1:loth)
          else

            WRITE(left,'(10a)',IOSTAT=ios,ERR=13) '*surf ',
     &        SNAME(icomp,i)(1:lsna),' ',SVFC(icomp,i),' ',
     &        SPARENT(icomp,i)(1:lspa),' ',SUSE(icomp,i,1)(1:luse1),' ',
     &        SUSE(icomp,i,2)(1:luse2)
            call SDELIM(left,leftd,'C',IW)  ! pack tokens before SMLCN
            WRITE(right,'(7a)',IOSTAT=ios,ERR=13)
     &        SOTF(icomp,i)(1:lsot),' ',sbound_ty(1:lso),' ',
     &         sbound_c2(1:lso2),' ',sbound_e2(1:lso3)
            call SDELIM(right,rightd,'C',IW) ! pack tokens after SMLCN
            write(ifilg,'(6a,i3,2a)',IOSTAT=IOS,ERR=13)
     &        leftd(1:lnblnk(leftd)),',',SMLCN(icomp,i)(1:lsml),',',
     &        rightd(1:lnblnk(rightd)),'  # ',I,' ',OTHSTR(1:loth)
          endif
        else

C There was an array fault so write out a placeholder surface. As
C there might be more than one fault make name unique.
          write(ifilg,'(a,2i3,a)',IOSTAT=ios,ERR=13) '*surf,fault',i,
     &      ',UNKN,-,-,-,UNKN,OPAQUE,EXTERIOR,00,000'
        endif
  102 CONTINUE

C Default insolation distribution as packed string.
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
      WRITE(outs,'(a,4I4)',IOSTAT=ios,ERR=14)
     &  '*insol ',NDP(ICOMP),(IDPN(ICOMP,J),J=1,3)
      call SDELIM(outs,outsd,'S',IW)
      write(comment,'(a)') 'default insolation distribution '
      call align_comment(48,outsd,comment,aligned_str)
      write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))

C Shading calculation directives in the form of:
C *shad_calc,all_applic,20,20
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
      if(nsurfcalc(icomp).eq.0)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# shading directives'
        write(tokens,'(a)') '*shad_calc none'
        write(comment,'(a)') 'no temporal shading requested'
        call align_comment(48,tokens,comment,aligned_str)
        write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
      elseif(nsurfcalc(icomp).gt.0)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# shading directives'
        if(iaplic(icomp,1).eq.1)then
          write(tokens,'(a,i3)',IOSTAT=ios,ERR=13)
     &      '*shad_calc,all_applicable ',nsurfcalc(icomp)
          write(comment,'(a)') 'list of surfs'
          call align_comment(48,tokens,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
          write(louts,'(90i4)',IOSTAT=ios,ERR=14)
     &      (lstsfcalc(ICOMP,J),J=1,nsurfcalc(icomp))
          call SDELIM(louts,loutsd,'S',IW)
          write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &      loutsd(1:lnblnk(loutsd))
        elseif(iaplic(icomp,1).eq.0)then
          write(tokens,'(a,i3)',IOSTAT=ios,ERR=13)
     &      '*shad_calc,list',nsurfcalc(icomp)
          write(comment,'(a)') 'list of surfs'
          call align_comment(48,tokens,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
          write(louts,'(90i4)',IOSTAT=ios,ERR=14)
     &      (lstsfcalc(ICOMP,J),J=1,nsurfcalc(icomp))
          call SDELIM(louts,loutsd,'S',IW)
          write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &      loutsd(1:lnblnk(loutsd))
        endif
      endif

C Insolation calculation directives.
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
      if(nsurfinso(icomp).eq.0)then
        write(tokens,'(a)') '*insol_calc none'
        write(comment,'(a)') 'no insolation requested'
        call align_comment(48,tokens,comment,aligned_str)
        write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
      elseif(nsurfinso(icomp).gt.0)then
        if(iaplic(icomp,2).eq.1)then
          write(tokens,'(a,i3)',IOSTAT=ios,ERR=13)
     &      '*insol_calc all_applicable ',nsurfinso(icomp)
          write(comment,'(a)') 'insolation sources'
          call align_comment(48,tokens,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
          write(louts,'(90i4)',IOSTAT=ios,ERR=14)
     &      (isurfinso(ICOMP,J),J=1,nsurfinso(icomp))
          call SDELIM(louts,loutsd,'S',IW)
          write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &      loutsd(1:lnblnk(loutsd))
        elseif(iaplic(icomp,2).eq.0)then
          write(tokens,'(a,i3)',IOSTAT=ios,ERR=13)
     &      '*insol_calc list',nsurfinso(icomp)
          write(comment,'(a)') 'insolation sources'
          call align_comment(48,tokens,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
          write(louts,'(90i4)',IOSTAT=ios,ERR=14)
     &      (isurfinso(ICOMP,J),J=1,nsurfinso(icomp))
          call SDELIM(louts,loutsd,'S',IW)
          write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &      loutsd(1:lnblnk(loutsd))
        endif
      endif

C Surfaces associated with base.
      write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
      if(izbaselist(icomp).eq.0)then
        WRITE(outs,'(a,i2,F9.2,i2)',IOSTAT=ios,ERR=13) '*base_list ',
     &    izbaselist(icomp),ZBASEA(ICOMP),IUZBASEA(ICOMP)
        call SDELIM(outs,outsd,'S',IW)
        write(comment,'(a)') 'zone base'
        call align_comment(48,outsd,comment,aligned_str)
        write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
      else

C Write the list and then the data after the list.
        WRITE(outs,'(a,21i4)',IOSTAT=ios,ERR=13) '*base_list ',
     &    izbaselist(icomp),(IBASES(icomp,J),J=1,izbaselist(icomp))
        call SDELIM(outs,outsd,'S',IW)
        if(lnblnk(outsd).lt.124)then
          write(outs2,'(2a,f9.2,i2)') outsd(1:lnblnk(outsd)),',',
     &      ZBASEA(ICOMP),IUZBASEA(ICOMP)   ! combine list and area
          write(comment,'(a)') 'zone base list'
          call align_comment(48,outs2,comment,aligned_str)
          write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
        else
          write(outs72,'(F9.2,i2)') ZBASEA(ICOMP),IUZBASEA(ICOMP)
          write(ifilg,'(4a)',IOSTAT=IOS,ERR=13)
     &      outsd(1:lnblnk(outsd)),',',outs72(1:lnblnk(outs72)),
     &      '  # zone base list'
        endif
      endif

C Write thermal bridge info if detected or the losspercent is non-zero.
      if(losspercent(icomp).gt.0.0) havebrdge=.true.
      if(nbrdg(icomp).gt.0.or.havebrdge)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        write(ifilg,'(2a)',IOSTAT=IOS,ERR=13) 
     &    '# bridge: user fraction,loss% W/K, ',
     &    'total loss W/K & total UA W/K'
        WRITE(outs,'(a,4F10.3)',IOSTAT=ios,ERR=13) 
     &    '*bridge_start ',thbrpercent,losspercent(icomp),
     &    totheatloss(icomp),uavtotal(icomp)
        if(tbregime(1:2).ne.' ')then  ! If TB regime not blank include.
          write(ifilg,'(3a)',IOSTAT=IOS,ERR=13)
     &      outs(1:lnblnk(outs)),'  ',tbregime(1:lnblnk(tbregime))
        else
          write(ifilg,'(a)',IOSTAT=IOS,ERR=13) outs(1:lnblnk(outs))
        endif
        if(nbrdg(icomp).gt.0)then
          do 88 itb=1,nbrdg(icomp)
            if(ibrdg(icomp,itb).eq.1)then
              phrase='roof-wall '
            elseif(ibrdg(icomp,itb).eq.2)then
              phrase='wall-ground floor '
            elseif(ibrdg(icomp,itb).eq.3)then
              phrase='wall-wall (convex corner) '
            elseif(ibrdg(icomp,itb).eq.4)then
              phrase='wall-wall (concave corner) '
            elseif(ibrdg(icomp,itb).eq.5)then
              phrase='wall-floor (exposed floor) '
            elseif(ibrdg(icomp,itb).eq.6)then
              phrase='lintel above window or door '
            elseif(ibrdg(icomp,itb).eq.7)then
              phrase='sill below window '
            elseif(ibrdg(icomp,itb).eq.8)then
              phrase='jamb at window or door '
            elseif(ibrdg(icomp,itb).eq.9)then
              phrase='wall-gable '
            elseif(ibrdg(icomp,itb).eq.10)then
              phrase='wall-parapet '
            elseif(ibrdg(icomp,itb).eq.11)then
              phrase='wall-intermediate-floor '
            elseif(ibrdg(icomp,itb).eq.12)then
              phrase='wall-partition '
            elseif(ibrdg(icomp,itb).eq.13)then
              phrase='glass-frame '
            elseif(ibrdg(icomp,itb).eq.14)then
              phrase='balcony'
            elseif(ibrdg(icomp,itb).eq.15)then
              phrase='user-defined-a'
            elseif(ibrdg(icomp,itb).eq.16)then
              phrase='user-defined-b'
            endif

C The phrase can have spaces in it so do not pack string.  If
C nbridgevt is nonzero also include additional line with
C associated vertices.
            if(nbridgevt(icomp,ibrdg(icomp,itb)).eq.0)then
              WRITE(outs,'(4a,F7.3,a,F8.4,a)',IOSTAT=ios,ERR=13)
     &          '*bridge',tab,phrase(1:lnblnk(phrase)),tab,
     &          lngth(icomp,itb),tab,psi(icomp,itb),
     &          '  0  # type length psi value assoc verts'
              write(ifilg,'(a)',IOSTAT=IOS,ERR=13) outs(1:lnblnk(outs))
            else
              WRITE(outs,'(4a,F7.3,a,F8.4,i4,a)',IOSTAT=ios,ERR=13)
     &          '*bridge',tab,phrase(1:lnblnk(phrase)),tab,
     &          lngth(icomp,itb),tab,psi(icomp,itb),
     &          nbridgevt(icomp,ibrdg(icomp,itb)),
     &          '  # type length psi value assoc verts'
              write(ifilg,'(a)',IOSTAT=IOS,ERR=13) outs(1:lnblnk(outs))
              iabv=nbridgevt(icomp,ibrdg(icomp,itb))
              write(louts,'(90i4)',IOSTAT=ios,ERR=14)
     &          (bridgevlst(icomp,ibrdg(icomp,itb),J),J=1,iabv)
              call SDELIM(louts,loutsd,'S',IW)
              if(lnblnk(loutsd).lt.100)then
                write(ifilg,'(2a)',IOSTAT=ios,ERR=14)
     &            loutsd(1:lnblnk(loutsd)),'  # associated vertices'
              else
                write(ifilg,'(a)',IOSTAT=ios,ERR=14)
     &            loutsd(1:lnblnk(loutsd))
              endif
            endif
  88      continue
        endif
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '*end_bridge'
      endif

C iobs is 2 if there are block descriptions to include in this zone
C geometry file also write the NOX and NOZ values.
      if(IOBS(icomp).eq.2.and.nbobs(icomp).gt.0)then
        hasobstr=.true.
      endif
      if(hasobstr)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# block entities:'
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '#  *obs = obstructions'
        write(tokens,'(2a,i3,i3)',IOSTAT=IOS,ERR=13)
     &  '*block_start',tab,NOX(icomp),NOZ(icomp)
        write(comment,'(a)') 'geometric blocks'
        call align_comment(48,tokens,comment,aligned_str)
        write(ifilg,'(a)') aligned_str(1:lnblnk(aligned_str))
        do ib=1,nbobs(icomp)
          if(BLOCKTYP(icomp,ib)(1:4).eq.'obs3')then
            WRITE(outs,'(2a,6F9.4,3F8.2,F6.2,1X,A)',
     &        IOSTAT=ios,ERR=13)
     &        '*obs3',tab,XOB(icomp,ib),YOB(icomp,ib),ZOB(icomp,ib),
     &        DXOB(icomp,ib),DYOB(icomp,ib),DZOB(icomp,ib),
     &        BANGOB(icomp,ib,1),BANGOB(icomp,ib,2),BANGOB(icomp,ib,3),
     &        OPOB(icomp,ib),BLOCKNAME(icomp,ib)
            call SDELIM(outs,outsd,'C',IW)
            lnbm=lnblnk(BLOCKMAT(icomp,ib))
            write(ifilg,'(4a,i3)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',BLOCKMAT(icomp,ib)(1:lnbm),
     &        '  # block ',ib
          elseif(BLOCKTYP(icomp,ib)(1:4).eq.'obsp')then
            WRITE(outs,'(2a,F7.2,1X,A)',IOSTAT=ios,ERR=13)
     &        '*obsp',' 8 6 ',OPOB(icomp,ib),BLOCKNAME(icomp,ib)
            call SDELIM(outs,outsd,'C',IW)
            lnbm=lnblnk(BLOCKMAT(icomp,ib))
            write(ifilg,'(4a,i3,a)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',BLOCKMAT(icomp,ib)(1:lnbm),
     &        '  # block ',ib,' coords follow:'

            WRITE(outs,'(12F9.4)',IOSTAT=ios,ERR=13)
     &        XBP(icomp,ib,1),YBP(icomp,ib,1),ZBP(icomp,ib,1),
     &        XBP(icomp,ib,2),YBP(icomp,ib,2),ZBP(icomp,ib,2),
     &        XBP(icomp,ib,3),YBP(icomp,ib,3),ZBP(icomp,ib,3),
     &        XBP(icomp,ib,4),YBP(icomp,ib,4),ZBP(icomp,ib,4)
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(2A)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),'  # 1-4 '

            WRITE(outs,'(12F9.4)',IOSTAT=ios,ERR=13)
     &        XBP(icomp,ib,5),YBP(icomp,ib,5),ZBP(icomp,ib,5),
     &        XBP(icomp,ib,6),YBP(icomp,ib,6),ZBP(icomp,ib,6),
     &        XBP(icomp,ib,7),YBP(icomp,ib,7),ZBP(icomp,ib,7),
     &        XBP(icomp,ib,8),YBP(icomp,ib,8),ZBP(icomp,ib,8)
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(2A)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),'  # 5-8 '

          elseif(BLOCKTYP(icomp,ib)(1:4).eq.'obs ')then
            WRITE(outs,'(2a,6F9.4,F8.2,F6.2,1X,A)',
     &        IOSTAT=ios,ERR=13)
     &        '*obs',tab,XOB(icomp,ib),YOB(icomp,ib),ZOB(icomp,ib),
     &        DXOB(icomp,ib),DYOB(icomp,ib),DZOB(icomp,ib),
     &        BANGOB(icomp,ib,1),OPOB(icomp,ib),BLOCKNAME(icomp,ib)
            call SDELIM(outs,outsd,'C',IW)
            lnbm=lnblnk(BLOCKMAT(icomp,ib))
            write(ifilg,'(4a,i3)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',BLOCKMAT(icomp,ib)(1:lnbm),
     &        '  # block ',ib
          endif
        enddo  !  ib

C Adapt logic in case there are also mrt sensors.
        if(IVF(icomp).eq.1.and.ncub(icomp).gt.0)then
          continue
        else
          write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '*end_block'
        endif
      endif

C If there is a viewfactor file and mrt blocks echo that into the
C zone geometry file so block definitions are persisitent (they
C tend to get lost when surfaces get added and vwf file is scanned).
C If there were no obstructions then include a *block_start.
      if(IVF(icomp).eq.1)then
        if(ncub(icomp).gt.0)then
          if(hasobstr)then
            continue
          else
            write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
            write(ifilg,'(3a)',IOSTAT=IOS,ERR=13)
     &        '*block_start',tab,' 20  20  # geometric blocks'
            write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# mrt blocks:'
          endif
          do ib=1,ncub(icomp)
            WRITE(outs,'(2a,6F9.4,F8.2,1X,A)',IOSTAT=ios,ERR=13)
     &        '*mrt',tab,XOC(ib),YOC(ib),ZOC(ib),
     &        DXC(ib),DYC(ib),DZC(ib),
     &        CANG(ib),CUBN(ib)
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(2a,i3)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),'  # mrt ',ib
          enddo
          write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '*end_block'
        endif
      endif

C Visual entities are in the next section.
      if(nbvis(icomp).gt.0)then
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '# '
        write(ifilg,'(3a)',IOSTAT=IOS,ERR=13) '# visual entities:',
     &    ' *vis = visual blocks  *vis3 = visual 3 axis',
     &    '  *visp = visual polys  *vobject = named collection'
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '*visual_start'

        do ib=1,nbvis(icomp)
          if(VISTYP(icomp,ib)(1:4).eq.'vis3')then
            WRITE(outs,'(2a,6F10.4,3F8.2,F6.2,1X,A)',
     &        IOSTAT=ios,ERR=13)
     &        '*vis3',tab,XOV(icomp,ib),YOV(icomp,ib),ZOV(icomp,ib),
     &        DXOV(icomp,ib),DYOV(icomp,ib),DZOV(icomp,ib),
     &        BANGOV(icomp,ib,1),BANGOV(icomp,ib,2),BANGOV(icomp,ib,3),
     &        OPOV(icomp,ib),VISNAME(icomp,ib)
            call SDELIM(outs,outsd,'C',IW)
            lnvm=lnblnk(VISMAT(icomp,ib))
            write(ifilg,'(4a,i3)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',VISMAT(icomp,ib)(1:lnvm),
     &        '  # visual block ',ib
          elseif(VISTYP(icomp,ib)(1:4).eq.'visp')then
            WRITE(outs,'(2a,F7.2,1X,A)',IOSTAT=ios,ERR=13)
     &        '*visp',' 8 6 ',OPOV(icomp,ib),VISNAME(icomp,ib)
            lnvm=lnblnk(VISMAT(icomp,ib))
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(4a,i3,a)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',VISMAT(icomp,ib)(1:lnvm),
     &        '  # visual ',ib,' coords follow:'

            WRITE(outs,'(12F9.4)',IOSTAT=ios,ERR=13)
     &        XVP(icomp,ib,1),YVP(icomp,ib,1),ZVP(icomp,ib,1),
     &        XVP(icomp,ib,2),YVP(icomp,ib,2),ZVP(icomp,ib,2),
     &        XVP(icomp,ib,3),YVP(icomp,ib,3),ZVP(icomp,ib,3),
     &        XVP(icomp,ib,4),YVP(icomp,ib,4),ZVP(icomp,ib,4)
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(2A)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),'  # 1-4 '

            WRITE(outs,'(12F9.4)',IOSTAT=ios,ERR=13)
     &        XVP(icomp,ib,5),YVP(icomp,ib,5),ZVP(icomp,ib,5),
     &        XVP(icomp,ib,6),YVP(icomp,ib,6),ZVP(icomp,ib,6),
     &        XVP(icomp,ib,7),YVP(icomp,ib,7),ZVP(icomp,ib,7),
     &        XVP(icomp,ib,8),YVP(icomp,ib,8),ZVP(icomp,ib,8)
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(2A)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),'  # 5-8 '

          elseif(VISTYP(icomp,ib)(1:4).eq.'vis ')then
            WRITE(outs,'(2a,6F10.4,F8.2,F6.2,1X,A)',
     &        IOSTAT=ios,ERR=13)
     &        '*vis',tab,XOV(icomp,ib),YOV(icomp,ib),ZOV(icomp,ib),
     &        DXOV(icomp,ib),DYOV(icomp,ib),DZOV(icomp,ib),
     &        BANGOV(icomp,ib,1),OPOV(icomp,ib),VISNAME(icomp,ib)
            lnvm=lnblnk(VISMAT(icomp,ib))
            call SDELIM(outs,outsd,'C',IW)
            write(ifilg,'(4a,i3)',IOSTAT=IOS,ERR=13)
     &        outsd(1:lnblnk(outsd)),' ',VISMAT(icomp,ib)(1:lnvm),
     &        '  # visual ',ib
          endif
        enddo   ! ib

C If there are objects (collections of entities) write them. Because there
C may be spaces in VOBJDESC take care.
        if(NBVOBJ(icomp).gt.0)then
          do ib=1,NBVOBJ(icomp)
            WRITE(outs,'(6a,i2,a)',IOSTAT=ios,ERR=13)
     &        '*vobject',tab,
     &        VOBJNAME(icomp,ib)(1:lnblnk(VOBJNAME(icomp,ib))),
     &        tab,VOBJDESC(icomp,ib)(1:lnblnk(VOBJDESC(icomp,ib))),tab,
     &        NBVOBJLIST(icomp,ib),tab

C Append the list of visual entities comma separated. Similar to logic
C in esru_lib.F subroutine aslist.
            outs2=' '; ix=1; ixl=0
            do ibo=1,NBVOBJLIST(icomp,ib)
              lna=lnblnk(VOBJLIST(icomp,ib,ibo))
              if(lna.eq.1)then
                ixl=ix
              else
                ixl=ix+(lna-1)
              endif
              write(outs2(ix:ixl),'(a)')VOBJLIST(icomp,ib,ibo)(1:lna)
              if(ibo.lt.NBVOBJLIST(icomp,ib))then
                write(outs2(ixl+1:ixl+1),'(a)') ','
                ix=ix+lna+1
              else
                ix=ix+lna+1
              endif
            enddo  ! of ibo
            write(ifilg,'(2a)',IOSTAT=IOS,ERR=13)
     &        outs(1:lnblnk(outs)),outs2(1:lnblnk(outs2))
          enddo    ! of ib
        endif      ! of nbvobj
        write(ifilg,'(a)',IOSTAT=IOS,ERR=13) '*end_visual'
      endif

C If file written with this source and the version number is 1.0
C then reset it to 1.1.
      call eclose(gversion(icomp),1.1,0.01,newgeo)
      if(.NOT.newgeo) gversion(icomp) = 1.1

   99 CALL ERPFREE(IFILG,ios)
      RETURN

C Error messages.
   13 if(IOS.eq.2)then
        CALL USRMSG('No permission to write ',GENFIL,'W')
      else
        CALL USRMSG('File write error in ',GENFIL,'W')
      endif
      IER=1
      GOTO 99
   14 if(IOS.eq.2)then
        CALL USRMSG('No prmission to write array in ',GENFIL,'W')
      else
        CALL USRMSG('Long arrary write error in ',GENFIL,'W')
      endif
      IER=1
      GOTO 99

      END


C ******************** ERECC
C ERECC converts a REC (rectilinear) description into a GEN description.
C The X,Y, & Z block origin coordinates of the 'Bottom left-hand' corner
C (when viewed from the south),
C the length (metres from the BLHC towards the east),
C width (metres from the BLHC towards the north),
C height (metres from the BLHC upwards),
C angle of rotation A degrees between the length side and due east axis
C (anticlockwise is positive).
C Information is converted into the common block G1.
C Note: calling subroutine should instanciate isznver and iszjvn.

      SUBROUTINE ERECC(XO,YO,ZO,DX,DY,DZ,A)
#include "building.h"
#include "geometry.h"
      PI = 4.0 * ATAN(1.0)
      R=PI/180.
      SA=SIN(A*R)
      CA=COS(A*R)
      NSUR=6
      DO 10 I=1,NSUR
        NVER(I)=4
   10 CONTINUE

C Assign bottom 4 points X(1) to X(4) etc. anticlockwise and assign the
C top 4 points X(5) to X(8) etc. again anticlockwise.  Point 1 is point
C X0,Y0,Z0 with point 5 is directly above.
      X(1)=XO
      X(2)=X(1)+(DX*CA)
      X(3)=X(2)-(DY*SA)
      X(4)=X(3)-(DX*CA)
      Y(1)=YO
      Y(2)=Y(1)+(DX*SA)
      Y(3)=Y(2)+(DY*CA)
      Y(4)=Y(3)-(DX*SA)
      DO 20 I=5,8
        X(I)=X(I-4)
        Y(I)=Y(I-4)
        Z(I-4)=ZO
        Z(I)=ZO+DZ
   20 CONTINUE

C Number the vertices in each face.  Vertical faces numbered anticlockwise
C from bottom left-hand point when face viewed from the outside.  Ceiling
C numbered anitclockwise from point 5 and when viewed from the outside.
C Floor numbered anticlockwise from point 1 when viewed from outside (below).
      DO 30 I=1,NSUR
        IF(I.LE.4)THEN
          DO 40 J=1,4
            IF(J.LE.2)THEN
              IF(J.EQ.1)JVN(I,J)=I
              IF(J.NE.1)JVN(I,J)=I+1
              IF(I.EQ.4.AND.J.EQ.2)JVN(I,J)=1
            ELSE
              JVN(I,J)=8+I-J
              IF(I.EQ.4.AND.J.EQ.3)JVN(I,J)=5
            ENDIF
   40     CONTINUE
        ELSEIF(I.EQ.6)THEN
          JVN(I,1)=1
          DO 50 J=2,4
            K=6-J
            JVN(I,J)=K
   50     CONTINUE
        ELSEIF(I.EQ.5)THEN
          K=I-1
          DO 60 J=1,4
            K=K+1
            JVN(I,J)=K
   60     CONTINUE
        ENDIF
   30 CONTINUE
      NTV=8
      RETURN
      END

C ******************** ERECC3A
C ERECC converts a REC (rectilinear) description into a GEN description.
C The X,Y, & Z block origin coordinates of the 'Bottom left-hand' corner
C (when viewed from the south),
C the length (metres from the BLHC towards the east),
C width (metres from the BLHC towards the north),
C height (metres from the BLHC upwards),
C angle of rotation A degrees between the length side and due east axis
C (anticlockwise is positive). This alters the X and Y coordinates but
C not the Z coordinate.
C The angle of rotation B is between the length side and the horizon
C (positive lifts the edge). This alters the X and Z coordinates but
C not the Y coordinate.
C The angle of rotation C is between the width side and the horizon
C (positive lifts the edge) NOT YET implemented. This alters the Y
C and Z coordinates but not the X.
C Information is converted into the common block G1.
C Note: calling subroutine should instanciate isznver and iszjvn.

      SUBROUTINE ERECC3A(XO,YO,ZO,DX,DY,DZ,A,B,C)
#include "building.h"
#include "geometry.h"
C Parameters
      real XO,YO,ZO  ! origin of the box
      real DX,DY,DZ  ! length, width, height of box
      real A,B,C     ! angles of rotation

      real PI,R,SA,CA,SB,CB ! local variables
C     real SC,CC
      logical CorrectAngle,bCorrectAngle,cCorrectAngle

      PI = 4.0 * ATAN(1.0)
      R=PI/180.
      SA=SIN(A*R)  ! for a angle
      CA=COS(A*R)
      SB=SIN(B*R)  ! for b angle
      CB=COS(B*R)
C      SC=SIN(C*R)  ! for c angle
C      CC=COS(C*R)

      NSUR=6
      DO 10 I=1,NSUR
        NVER(I)=4
   10 CONTINUE

C To prevent infinity set CB to zero if B close to 90 or 270 degrees.
      call eclose(B,90.0,0.02,CorrectAngle)
      call eclose(B,270.0,0.02,bCorrectAngle)
      if(CorrectAngle.or.bCorrectAngle)then
        CB=0.0
      endif

C To prevent infinity set SB to zero if B close to 0 180 or 360 degrees.
      call eclose(B,0.0,0.02,CorrectAngle)
      call eclose(B,180.0,0.02,bCorrectAngle)
      call eclose(B,360.0,0.02,cCorrectAngle)
      if(CorrectAngle.or.bCorrectAngle.or.cCorrectAngle)then
        SB=0.0
      endif

C To prevent infinity set CA to zero if A close to 90 or 270 degrees.
      call eclose(A,90.0,0.02,CorrectAngle)
      call eclose(A,270.0,0.02,bCorrectAngle)
      if(CorrectAngle.or.bCorrectAngle)then
        CA=0.0
      endif

C To prevent infinity set SA to zero if A close to 0 180 or 360 degrees.
      call eclose(A,0.0,0.02,CorrectAngle)
      call eclose(A,180.0,0.02,bCorrectAngle)
      call eclose(A,360.0,0.02,cCorrectAngle)
      if(CorrectAngle.or.bCorrectAngle.or.cCorrectAngle)then
        SA=0.0
      endif

C Assign corners of the box assuming no rotation.
      X(1)=XO; X(2)=X(1)+DX; X(3)=X(2); X(4)=X(3)-DX
      X(5)=XO; X(6)=X(2); X(7)=X(3); X(8)=X(4)
      Y(1)=YO; Y(2)=Y(1); Y(3)=Y(2)+DY; Y(4)=Y(3)
      Y(5)=YO; Y(6)=Y(2); Y(7)=Y(3); Y(8)=Y(4)
      Z(1)=ZO; Z(2)=ZO; Z(3)=ZO; Z(4)=ZO
      Z(5)=ZO+DZ; Z(6)=ZO+DZ; Z(7)=ZO+DZ; Z(8)=ZO+DZ

C Do a B angle of rotation first. No Y changes required.
C X(1) and X(4) do not change.
C X(2) and X(3) change based on DX length.


      X(2)=X(1)+(DX*CB); X(3)=X(4)+(DX*CB)

C X(5) & X(8) change based on DZ length.
      X(5)=X(5)-(DZ*SB); X(8)=X(8)-(DZ*SB)

C X(6) & X(7) change based on DX length.
      X(6)=X(5)+(DX*CB); X(7)=X(8)+(DX*CB)

C Z(1) & Z(4) do not change.
C Z(2) & Z(3) change based on DX length.
      Z(2)=Z(2)+(DX*SB); Z(3)=Z(3)+(DX*SB)

C Z(5) & Z(8) change based on DZ length.
      Z(5)=ZO+(CB*DZ); Z(8)=ZO+(CB*DZ)

C Z(6) & Z(7) change based on DZ length.
      Z(6)=Z(2)+(DZ*CB); Z(7)=Z(3)+(DZ*CB)

C Moving now to the Y rotation
C Do an A angle of rotation. No Z changes required.
C X(1) does not change.
C X(2) changes based on [X(2)-X(1)] length


      X(2)=X(1)+(CB*DX)*CA
      X(3)=X(2)-(SA*DY)
      X(4)=X(1)-(SA*DY)
      X(5)=X(1)-(SB*DZ)*CA
      X(6)=X(5)+(CB*DX)*CA
      X(8)=X(4)-(SB*DZ)*CA
      X(7)=X(8)+(CB*DX)*CA

C Y(1) does not change.
      Y(2)=Y(1)+((CB*DX)*SA)
      Y(4)=Y(1)+(CA*DY)
      Y(3)=Y(4)+((CB*DX)*SA)
      Y(5)=Y(1)-(SB*DZ)*SA
      Y(6)=Y(5)+(CB*DX)*SA
      Y(8)=Y(4)-(SB*DZ)*SA
      Y(7)=Y(8)+((CB*DX)*SA)


C Number the vertices in each face.  Vertical faces numbered anticlockwise
C from bottom left-hand point when face viewed from the outside.  Ceiling
C numbered anitclockwise from point 5 and when viewed from the outside.
C Floor numbered anticlockwise from point 1 when viewed from outside
C (below).
      DO 30 I=1,NSUR
        IF(I.LE.4)THEN
          DO 40 J=1,4
            IF(J.LE.2)THEN
              IF(J.EQ.1)JVN(I,J)=I
              IF(J.NE.1)JVN(I,J)=I+1
              IF(I.EQ.4.AND.J.EQ.2)JVN(I,J)=1
            ELSE
              JVN(I,J)=8+I-J
              IF(I.EQ.4.AND.J.EQ.3)JVN(I,J)=5
            ENDIF
   40     CONTINUE
        ELSEIF(I.EQ.6)THEN
          JVN(I,1)=1
          DO 50 J=2,4
            K=6-J
            JVN(I,J)=K
   50     CONTINUE
        ELSEIF(I.EQ.5)THEN
          K=I-1
          DO 60 J=1,4
            K=K+1
            JVN(I,J)=K
   60     CONTINUE
        ENDIF
   30 CONTINUE
      NTV=8
      RETURN
      END

C ******************** CNVBLK
C CNVBLK converts a REC (rectilinear) description into GB1 common block.
C The X,Y, & Z coordinates of the 'Bottom left-hand' corner (when viewed
C from the south), the length (from the BL-HC towards the east), width,
C height of the enclosure and angle of rotation between the length side
C and due east (anticlockwise positive) are converted.
      SUBROUTINE CNVBLK(XO,YO,ZO,DX,DY,DZ,A)
      COMMON/GB1/XB(12),YB(12),ZB(12),JVNB(6,4)
      PI = 4.0 * ATAN(1.0)
      R=PI/180.
      SA=SIN(A*R)
      CA=COS(A*R)
      NSB=6

C Assign bottom 4 points XB(1) to XB(4) etc. anticlockwise and assign the
C top 4 points XB(5) to XB(8) etc. again anticlockwise.  Point 1 is point
C X0,Y0,Z0 with point 5 is directly above.
      XB(1)=XO
      XB(2)=XB(1)+(DX*CA)
      XB(3)=XB(2)-(DY*SA)
      XB(4)=XB(3)-(DX*CA)
      YB(1)=YO
      YB(2)=YB(1)+(DX*SA)
      YB(3)=YB(2)+(DY*CA)
      YB(4)=YB(3)-(DX*SA)
      DO 20 I=5,8
        XB(I)=XB(I-4)
        YB(I)=YB(I-4)
        ZB(I-4)=ZO
        ZB(I)=ZO+DZ
   20 CONTINUE

C Number the vertices in each face anticlockwise from the bottom
C left-hand point when face viewed from the outside.  Top is
C numbered anitclockwise from point 5 and when viewed from the outside.
C base is numbered anticlockwise from point 1 when viewed from outside (below).
      DO 30 I=1,NSB
        IF(I.LE.4)THEN
          DO 40 J=1,4
            IF(J.LE.2)THEN
              IF(J.EQ.1)JVNB(I,J)=I
              IF(J.NE.1)JVNB(I,J)=I+1
              IF(I.EQ.4.AND.J.EQ.2)JVNB(I,J)=1
            ELSE
              JVNB(I,J)=8+I-J
              IF(I.EQ.4.AND.J.EQ.3)JVNB(I,J)=5
            ENDIF
   40     CONTINUE
        ELSEIF(I.EQ.6)THEN
          JVNB(I,1)=1
          DO 50 J=2,4
            K=6-J
            JVNB(I,J)=K
   50     CONTINUE
        ELSEIF(I.EQ.5)THEN
          K=I-1
          DO 60 J=1,4
            K=K+1
            JVNB(I,J)=K
   60     CONTINUE
        ENDIF
   30 CONTINUE
      RETURN
      END

C ******************** CNVBLK3A
C CNVBLK3A converts a REC (rectilinear) description into GB1 common block.
C The X,Y, & Z coordinates of the 'Bottom left-hand' corner (when viewed
C from the south), the length (from the BL-HC towards the east), width,
C height of the enclosure.
C The angle of rotation A is between the length side and due east
C (anticlockwise positive). This alters the X and Y coordinates but
C not the Z coordinate.
C The angle of rotation B is between the length side and the horizon
C (positive lifts the edge). This alters the X and Z coordinates but
C not the Y coordinate.
C The angle of rotation C is between the width side and the horizon
C (positive lifts the edge) not yet implemented. This alters the Y
C and Z coordinates but not the X.

      SUBROUTINE CNVBLK3A(XO,YO,ZO,DX,DY,DZ,A,B,C)

C Parameters
      real XO,YO,ZO  ! origin of the box
      real DX,DY,DZ  ! length, width, height of box
      real A,B,C     ! angles of rotation

      real XB,YB,ZB
      integer JVNB
      COMMON/GB1/XB(12),YB(12),ZB(12),JVNB(6,4)
      real PI,R,SA,CA,SB,CB ! local variables
C     real SC,CC
      integer NSB     ! number of sides
      logical CorrectAngle,bCorrectAngle,cCorrectAngle

      PI = 4.0 * ATAN(1.0)
      R=PI/180.
      SA=SIN(A*R)  ! for a angle
      CA=COS(A*R)
      SB=SIN(B*R)  ! for b angle
      CB=COS(B*R)
C      SC=SIN(C*R)  ! for c angle
C      CC=COS(C*R)
      NSB=6

C To prevent infinity set CB to zero if B close to 90 or 270 degrees.
      call eclose(B,90.0,0.02,CorrectAngle)
      call eclose(B,270.0,0.02,bCorrectAngle)
      if(CorrectAngle.or.bCorrectAngle)then
        CB=0.0
      endif

C To prevent infinity set SB to zero if B close to 0 180 or 360 degrees.
      call eclose(B,0.0,0.02,CorrectAngle)
      call eclose(B,180.0,0.02,bCorrectAngle)
      call eclose(B,360.0,0.02,cCorrectAngle)
      if(CorrectAngle.or.bCorrectAngle.or.cCorrectAngle)then
        SB=0.0
      endif

C To prevent infinity set CA to zero if A close to 90 or 270 degrees.
      call eclose(A,90.0,0.02,CorrectAngle)
      call eclose(A,270.0,0.02,bCorrectAngle)
      if(CorrectAngle.or.bCorrectAngle)then
        CA=0.0
      endif

C To prevent infinity set SA to zero if A close to 0 180 or 360 degrees.
      call eclose(A,0.0,0.02,CorrectAngle)
      call eclose(A,180.0,0.02,bCorrectAngle)
      call eclose(A,360.0,0.02,cCorrectAngle)
      if(CorrectAngle.or.bCorrectAngle.or.cCorrectAngle)then
        SA=0.0
      endif

C Assign corners of the box assuming no rotation.
      XB(1)=XO; XB(2)=XB(1)+DX; XB(3)=XB(2); XB(4)=XB(3)-DX
      XB(5)=XO; XB(6)=XB(2); XB(7)=XB(3); XB(8)=XB(4)
      YB(1)=YO; YB(2)=YB(1); YB(3)=YB(2)+DY; YB(4)=YB(3)
      YB(5)=YO; YB(6)=YB(2); YB(7)=YB(3); YB(8)=YB(4)
      ZB(1)=ZO; ZB(2)=ZO; ZB(3)=ZO; ZB(4)=ZO
      ZB(5)=ZO+DZ; ZB(6)=ZO+DZ; ZB(7)=ZO+DZ; ZB(8)=ZO+DZ

C Do a B angle of rotation first. No Y changes required.
C XB(1) and XB(4) do not change.
C XB(2) and XB (3) change based on DX length.


      XB(2)=XB(1)+(DX*CB); XB(3)=XB(4)+(DX*CB)

C XB(5) & XB(8) change based on DZ length.
      XB(5)=XB(5)-(DZ*SB); XB(8)=XB(8)-(DZ*SB)

C XB(6) & XB(7) change based on DX length.
      XB(6)=XB(5)+(DX*CB); XB(7)=XB(8)+(DX*CB)

C ZB(1) & ZB(4) do not change.
C ZB(2) & ZB(3) change based on DX length.
      ZB(2)=ZB(2)+(DX*SB); ZB(3)=ZB(3)+(DX*SB)

C ZB(5) & ZB(8) change based on DZ length.
      ZB(5)=ZO+(CB*DZ); ZB(8)=ZO+(CB*DZ)

C ZB(6) & ZB(7) change based on DZ length.
      ZB(6)=ZB(2)+(DZ*CB);  ZB(7)=ZB(3)+(DZ*CB)

C Moving now to the Y rotation
C Do an A angle of rotation. No Z changes required.
C XB(1) does not change.
C XB(2) changes based on [XB(2)-XB(1)] length


      XB(2)=XB(1)+(CB*DX)*CA
      XB(3)=XB(2)-(SA*DY)
      XB(4)=XB(1)-(SA*DY)
      XB(5)=XB(1)-(SB*DZ)*CA
      XB(6)=XB(5)+(CB*DX)*CA
      XB(8)=XB(4)-(SB*DZ)*CA
      XB(7)=XB(8)+(CB*DX)*CA

C YB(1) does not change.
      YB(2)=YB(1)+((CB*DX)*SA)
      YB(4)=YB(1)+(CA*DY)
      YB(3)=YB(4)+((CB*DX)*SA)
      YB(5)=YB(1)-(SB*DZ)*SA
      YB(6)=YB(5)+(CB*DX)*SA
      YB(8)=YB(4)-(SB*DZ)*SA
      YB(7)=YB(8)+((CB*DX)*SA)


C Number the vertices in each face anticlockwise from the bottom
C left-hand point when face viewed from the outside.  Top is
C numbered anitclockwise from point 5 and when viewed from the outside.
C base is numbered anticlockwise from point 1 when viewed from outside (below).
      DO 30 I=1,NSB
        IF(I.LE.4)THEN
          DO 40 J=1,4
            IF(J.LE.2)THEN
              IF(J.EQ.1)JVNB(I,J)=I
              IF(J.NE.1)JVNB(I,J)=I+1
              IF(I.EQ.4.AND.J.EQ.2)JVNB(I,J)=1
            ELSE
              JVNB(I,J)=8+I-J
              IF(I.EQ.4.AND.J.EQ.3)JVNB(I,J)=5
            ENDIF
   40     CONTINUE
        ELSEIF(I.EQ.6)THEN
          JVNB(I,1)=1
          DO 50 J=2,4
            K=6-J
            JVNB(I,J)=K
   50     CONTINUE
        ELSEIF(I.EQ.5)THEN
          K=I-1
          DO 60 J=1,4
            K=K+1
            JVNB(I,J)=K
   60     CONTINUE
        ENDIF
   30 CONTINUE
      RETURN
      END

C ******************** CNVBLKP
C CNVBLKP converts a six sided obstuction 'obsp' into GB1 common block.
C The first 8 point of common block GB1 are filled and the standard
C JVNB edges are filled.
      SUBROUTINE CNVBLKP(IZ,IB)
#include "building.h"
#include "geometry.h"

C Parameters
      integer iz,ib  ! IZ is the zone index and IB is the block inde

      real XB,YB,ZB
      integer JVNB
      COMMON/GB1/XB(12),YB(12),ZB(12),JVNB(6,4)
      integer NSB     ! number of sides

      NSB=6
      DO 20 I=1,8
        XB(I)=XBP(IZ,IB,I)
        YB(I)=YBP(IZ,IB,I)
        ZB(I)=ZBP(IZ,IB,I)
  20  CONTINUE

C Number the vertices in each face anticlockwise from the bottom
C left-hand point when face viewed from the outside.  Top is
C numbered anitclockwise from point 5 and when viewed from the outside.
C base is numbered anticlockwise from point 1 when viewed from outside (below).
      DO 30 I=1,NSB
        IF(I.LE.4)THEN
          DO 40 J=1,4
            IF(J.LE.2)THEN
              IF(J.EQ.1)JVNB(I,J)=I
              IF(J.NE.1)JVNB(I,J)=I+1
              IF(I.EQ.4.AND.J.EQ.2)JVNB(I,J)=1
            ELSE
              JVNB(I,J)=8+I-J
              IF(I.EQ.4.AND.J.EQ.3)JVNB(I,J)=5
            ENDIF
   40     CONTINUE
        ELSEIF(I.EQ.6)THEN
          JVNB(I,1)=1
          DO 50 J=2,4
            K=6-J
            JVNB(I,J)=K
   50     CONTINUE
        ELSEIF(I.EQ.5)THEN
          K=I-1
          DO 60 J=1,4
            K=K+1
            JVNB(I,J)=K
   60     CONTINUE
        ENDIF
   30 CONTINUE
      RETURN
      END


C ******************** CNVVISP
C CNVVISP converts a six sided visual 'visp' into GB1 common block.
C The first 8 point of common block GB1 are filled and the standard
C JVNB edges are filled.
      SUBROUTINE CNVVISP(IZ,IB)
#include "building.h"
#include "geometry.h"

C Parameters
      integer iz,ib  ! IZ is the zone index and IB is the block inde

      real XB,YB,ZB
      integer JVNB
      COMMON/GB1/XB(12),YB(12),ZB(12),JVNB(6,4)
      integer NSB     ! number of sides

      NSB=6
      DO 20 I=1,8
        XB(I)=XVP(IZ,IB,I)
        YB(I)=YVP(IZ,IB,I)
        ZB(I)=ZVP(IZ,IB,I)
  20  CONTINUE

C Number the vertices in each face anticlockwise from the bottom
C left-hand point when face viewed from the outside.  Top is
C numbered anitclockwise from point 5 and when viewed from the outside.
C base is numbered anticlockwise from point 1 when viewed from outside (below).
      DO 30 I=1,NSB
        IF(I.LE.4)THEN
          DO 40 J=1,4
            IF(J.LE.2)THEN
              IF(J.EQ.1)JVNB(I,J)=I
              IF(J.NE.1)JVNB(I,J)=I+1
              IF(I.EQ.4.AND.J.EQ.2)JVNB(I,J)=1
            ELSE
              JVNB(I,J)=8+I-J
              IF(I.EQ.4.AND.J.EQ.3)JVNB(I,J)=5
            ENDIF
   40     CONTINUE
        ELSEIF(I.EQ.6)THEN
          JVNB(I,1)=1
          DO 50 J=2,4
            K=6-J
            JVNB(I,J)=K
   50     CONTINUE
        ELSEIF(I.EQ.5)THEN
          K=I-1
          DO 60 J=1,4
            K=K+1
            JVNB(I,J)=K
   60     CONTINUE
        ENDIF
   30 CONTINUE
      RETURN
      END

C ******************** CNVOBJVISP
C CNVOBJVISP converts a six sided predefined visual 'visp' into GB1 common block.
C The first 8 point of common block GB1 are filled and the standard
C edges are filled.
      SUBROUTINE CNVOBJVISP(IB)
#include "building.h"
#include "geometry.h"
#include "predefined.h"

C Parameters
      integer ib  ! IB is the predefined block index

      real XB,YB,ZB
      integer JVNB
      COMMON/GB1/XB(12),YB(12),ZB(12),JVNB(6,4)
      integer NSB     ! number of sides

      NSB=6
      DO 20 I=1,8
        XB(I)=OBJXVP(IB,I)
        YB(I)=OBJYVP(IB,I)
        ZB(I)=OBJZVP(IB,I)
  20  CONTINUE

C Number the vertices in each face anticlockwise from the bottom
C left-hand point when face viewed from the outside.  Top is
C numbered anitclockwise from point 5 and when viewed from the outside.
C base is numbered anticlockwise from point 1 when viewed from outside (below).
      DO 30 I=1,NSB
        IF(I.LE.4)THEN
          DO 40 J=1,4
            IF(J.LE.2)THEN
              IF(J.EQ.1)JVNB(I,J)=I
              IF(J.NE.1)JVNB(I,J)=I+1
              IF(I.EQ.4.AND.J.EQ.2)JVNB(I,J)=1
            ELSE
              JVNB(I,J)=8+I-J
              IF(I.EQ.4.AND.J.EQ.3)JVNB(I,J)=5
            ENDIF
   40     CONTINUE
        ELSEIF(I.EQ.6)THEN
          JVNB(I,1)=1
          DO 50 J=2,4
            K=6-J
            JVNB(I,J)=K
   50     CONTINUE
        ELSEIF(I.EQ.5)THEN
          K=I-1
          DO 60 J=1,4
            K=K+1
            JVNB(I,J)=K
   60     CONTINUE
        ENDIF
   30 CONTINUE
      RETURN
      END

C << there are several quite similar - see if can revise/merge >>

C ******************** CNVPREOBS
C CNVPREOBS converts a six sided predefined obstruction 'obsp' into GB1 common block.
C The first 8 point of common block GB1 are filled and the standard
C edges are filled.
      SUBROUTINE CNVPREOBS(IB)
#include "building.h"
#include "geometry.h"
#include "predefined.h"

C Parameters
      integer ib  ! IB is the predefined block index

      real XB,YB,ZB
      integer JVNB
      COMMON/GB1/XB(12),YB(12),ZB(12),JVNB(6,4)
      integer NSB     ! number of sides

      NSB=6
      DO 20 I=1,8
        XB(I)=OBJXBP(IB,I)
        YB(I)=OBJYBP(IB,I)
        ZB(I)=OBJZBP(IB,I)
  20  CONTINUE

C Number the vertices in each face anticlockwise from the bottom
C left-hand point when face viewed from the outside.  Top is
C numbered anitclockwise from point 5 and when viewed from the outside.
C base is numbered anticlockwise from point 1 when viewed from outside (below).
      DO 30 I=1,NSB
        IF(I.LE.4)THEN
          DO 40 J=1,4
            IF(J.LE.2)THEN
              IF(J.EQ.1)JVNB(I,J)=I
              IF(J.NE.1)JVNB(I,J)=I+1
              IF(I.EQ.4.AND.J.EQ.2)JVNB(I,J)=1
            ELSE
              JVNB(I,J)=8+I-J
              IF(I.EQ.4.AND.J.EQ.3)JVNB(I,J)=5
            ENDIF
   40     CONTINUE
        ELSEIF(I.EQ.6)THEN
          JVNB(I,1)=1
          DO 50 J=2,4
            K=6-J
            JVNB(I,J)=K
   50     CONTINUE
        ELSEIF(I.EQ.5)THEN
          K=I-1
          DO 60 J=1,4
            K=K+1
            JVNB(I,J)=K
   60     CONTINUE
        ENDIF
   30 CONTINUE
      RETURN
      END



C ******************** EREGC
C EREGC converts a REG (extruded) geometry description into a Gen form.
C The number of vertical walls, X & Y coords of the bottom corner of
C each wall in an anticlockwise direction (defining a polygon in a plan
C view) as well as the floor & ceiling heights are passed to EREGC and
C the GEN description returned via common G1.
C Note: calling subroutine should instanciate isznver and iszjvn.

      SUBROUTINE EREGC(NW,Z1,Z2,XX,YY)
#include "building.h"
#include "geometry.h"
      DIMENSION XX(MS),YY(MS)

C Assign bottom points X(1) to X(NW) etc. anticlockwise and assign top
C points X(NW+1) to X(2*NW) etc. again anticlockwise.  Point 1 is point
C XX(1),YY(1),Z1 and point NW+1 is directly above.
      NSUR=NW+2
      DO 10 I=1,NW
        X(I)=XX(I)
        Y(I)=YY(I)
        Z(I)=Z1
   10 CONTINUE
      J1=NW+1
      J2=2*NW
      DO 20 I=J1,J2
        X(I)=X(I-NW)
        Y(I)=Y(I-NW)
        Z(I)=Z2
   20 CONTINUE

C Number the vertices in each face using the following conventions:
C Vertical surface are numbered anticlockwise from the bottom left-hand
C point when viewed from the outside; ceilings are numbered anticlockwise
C from point'NW+1' when viewed from the outside (above); floors are
C numbered anticlockwise from point 1 when viewed from the outside (below).
      J2=NSUR-1
      DO 30 I=1,NSUR
        IF(I.GE.J2)goto 1
        NVER(I)=4
        DO 40 J=1,4
          IF(J.GT.2)goto 2
          IF(J.EQ.1)JVN(I,J)=I
          IF(J.NE.1)JVN(I,J)=I+1
          IF(I.EQ.NW.AND.J.EQ.2)JVN(I,J)=1
          goto 40
    2     JVN(I,J)=NSUR+2+I-J
          IF(I.EQ.NW.AND.J.EQ.3)JVN(I,J)=NW+1
   40   CONTINUE
        goto 30

C In case the number of walls is greater than the number of
C edges that can be accommodated in a single floor or ceiling
C reduce NW.
    1   NTV=2*NW
        if(NW.gt.MV)then
          call edisp(iuout,
     &    'Reducing the number of edges in top/bottom (floor plan')
          call edisp(iuout,
     &    'extrusion had more walls than edges allowed.)')
          NVER(I)=MV
        else
          NVER(I)=NW
        endif
        IF(I.EQ.J2)goto 3
        JVN(I,1)=1
        DO 50 J=2,NVER(I)
          K=NW+2-J
          JVN(I,J)=K
   50   CONTINUE
        goto 30
    3   K=I-1
        DO 60 J=1,NVER(I)
          K=K+1
          JVN(I,J)=K
   60   CONTINUE
   30 CONTINUE
      RETURN
      END

C ******************** ESCROT
C Rotate a zone by ANG degrees and return information via common G1.
C X1,Y1 is the point to rotate about.

C << consider zone index so no longer necessary to use G1 >>

      SUBROUTINE ESCROT(ANGR,X1,Y1)
#include "building.h"
#include "geometry.h"

C Parameters
      real ANGR  ! angle of rotation
      real X1    ! X point for rotation
      real Y1    ! Y point for rotation

C Local variables
      real PI,A,CA,SA,XXX,YYY,XR,YR

      PI = 4.0 * ATAN(1.0)
      A=-ANGR*PI/180.
      CA=COS(A)
      SA=SIN(A)
      DO 10 I=1,NTV
        XXX=X(I)-X1
        YYY=Y(I)-Y1
        XR=XXX*CA+YYY*SA
        YR=YYY*CA-XXX*SA
        X(I)=XR+X1
        Y(I)=YR+Y1
   10 CONTINUE
      RETURN
      END


C ************* SURINFO
C Display surface details and attributes in a tabular format. Make use
C of information currently in Common blocks G0 G7 G20.
C If context is .true. then give verbose description of the boundary.
      SUBROUTINE SURINFO(ICOMP,ITRU,context)
#include "building.h"
#include "geometry.h"

      common/SFIG/NSIGFIG

C Markdown flag.
      logical markdown
      common/markdownflag/markdown

C Topic variables: surfcolumns modifies what is reported.
      integer siteinfo,databaseinfo,contextinfo,controlinfo,netinfo
      integer plantinfo,geominfo,schedinfo,zoneextrainfo,fileinfo
      integer spminfo,enetinfo
      integer zonecolumns,surfcolumns  ! patterns of zone and surface columns
      common/qatopics/siteinfo,databaseinfo,contextinfo,controlinfo,
     &  netinfo,plantinfo,geominfo,schedinfo,zoneextrainfo,fileinfo,
     &  spminfo,enetinfo,zonecolumns,surfcolumns

      logical context
      logical newgeo  ! to use for testing if new/old geometry file.

      CHARACTER OUTSTR*124,OTHSTR*30,outstr2*180,susecomb*17
      character sbound_ty*12,sbound_c2*6,sbound_e2*6
      CHARACTER SIGSTR*12
      integer lzn      ! length of zone name

C Check version number of file.
      newgeo=.false.  ! assume older format geometry.
      call eclose(gversion(icomp),1.1,0.01,newgeo)

C Surface summary: print header, followed by surface information.
      lzn=lnzname(ICOMP)
      call edisp(ITRU,' ')
      if(markdown)then
        write(outstr,'(3a,i2,a)')': A summary of the surfaces in ',
     &    zname(ICOMP)(1:lzn),'(',icomp,')'
      else
        write(outstr,'(3a,i2,a)')' A summary of the surfaces in ',
     &    zname(ICOMP)(1:lzn),'(',icomp,') follows:'
      endif
      call edisp(ITRU,outstr)
      call edisp(ITRU,' ')

C Write header based on current version number.
      if(newgeo)then
        if(markdown)then
          if(surfcolumns.eq.3)then    ! All items
            write(outstr2,'(3a)')
     &      'ID   Area (m^2^)  Azimuth   Elevation  Name          ',
     &      'Optics        Location  Use primary:detail  ',
     &      'Construction name          Environment other side'
            call edisp(ITRU,OUTSTR2)
            write(outstr2,'(3a)')
     &      '---  -----------  --------  ---------  ------------  ',
     &      '------------  --------  ------------------  ',
     &      '-------------------------  ------------------------'
            call edisp(ITRU,OUTSTR2)
          elseif(surfcolumns.eq.2)then
            write(outstr2,'(3a)')
     &      'ID   Area (m^2^)  Azimuth   Elevation  Name          ',
     &      'Optics        Use primary:detail  ',
     &      'Construction name          Environment other side'
            call edisp(ITRU,OUTSTR2)
            write(outstr2,'(3a)')
     &      '---  -----------  --------  ---------  ------------  ',
     &      '------------  ------------------  ',
     &      '-------------------------  ------------------------'
            call edisp(ITRU,OUTSTR2)
          elseif(surfcolumns.eq.1)then
            write(outstr2,'(3a)')
     &      'ID   Area (m^2^)  Azimuth   Elevation  Name          ',
     &      'Use primary:detail  ',
     &      'Construction name          Environment other side'
            call edisp(ITRU,OUTSTR2)
            write(outstr2,'(3a)')
     &      '---  -----------  --------  ---------  ------------  ',
     &      '------------------  ',
     &      '-------------------------  ------------------------'
            call edisp(ITRU,OUTSTR2)
          elseif(surfcolumns.eq.0)then
            write(outstr2,'(3a)')
     &      'ID   Area (m^2^)  Azimuth   Elevation  Name          ',
     &      'Use          ',
     &      'Construction name          Environment other side'
            call edisp(ITRU,OUTSTR2)
            write(outstr2,'(3a)')
     &      '---  -----------  --------  ---------  ------------  ',
     &      '-----------  ',
     &      '-------------------------  ------------------------'
            call edisp(ITRU,OUTSTR2)
          endif
        else
          if(surfcolumns.eq.3)then      ! All items
            write(outstr,'(2a)')
     &   ' Sur| Area  |Azim|Elev|          Surface                    ',
     &   '       | construction            |environment'
            call edisp(ITRU,OUTSTR)
            write(outstr,'(2a)')
     &   '    | m^2   |deg |deg | name       |optical|locat| use      ',
     &   '       | name                    |other side '
            call edisp(ITRU,OUTSTR)
          elseif(surfcolumns.eq.2)then  ! Subset B
            write(outstr,'(2a)')
     &   ' Sur| Area  |Azim|Elev|          Surface               ',
     &   '       | Construction           |Environment'
            call edisp(ITRU,OUTSTR)
            write(outstr,'(2a)')
     &   '    | m^2   |deg |deg | name       |optical| use       ',
     &   '       | name                   |other side '
            call edisp(ITRU,OUTSTR)
          elseif(surfcolumns.eq.1)then  ! Subset A
            write(outstr,'(2a)')
     &   ' Sur| Area  |Azim|Elev|        Surface        ',
     &   '       | Construction           |Environment'
            call edisp(ITRU,OUTSTR)
            write(outstr,'(2a)')
     &   '    | m^2   |deg |deg | name       | use      ',
     &   '       | name                   |other side '
            call edisp(ITRU,OUTSTR)
          elseif(surfcolumns.eq.0)then
            write(outstr,'(2a)')
     &   ' Sur| Area  |Azim|Elev|     Surface         ',
     &   '| Construction            |Environment'
            call edisp(ITRU,OUTSTR)
            write(outstr,'(2a)')
     &   '    | m^2   |deg |deg | name       |use     ',
     &   '| name                    |other side '
            call edisp(ITRU,OUTSTR)
          endif
        endif
      else
        write(outstr,'(2a)')
     &    ' Sur| Area  |Azim|Elev| Surface    | Geometry |',
     &    ' Construction         |Environment'
        call edisp(ITRU,OUTSTR)
        write(outstr,'(2a)')
     &    '    | m^2   |deg |deg | name       |type |loca|',
     &    ' name                 |other side '
        call edisp(ITRU,OUTSTR)
      endif
      if(.NOT.context)then
        DO 893 I=1,NZSUR(icomp)
          call decode_zsbound(icomp,i,sbound_ty,sbound_c2,sbound_e2)
          call SIGFIG(SNA(icomp,i),NSIGFIG,RNO,SIGSTR,LSTR)
          lnsmlcn=lnblnk(SMLCN(icomp,i))
          if(lnsmlcn.lt.19) lnsmlcn=19  ! at least 19 char for MLC
          if(newgeo)then
            lnsuse=lnblnk(SUSE(icomp,i,1))
            write(susecomb,'(3a)') SUSE(icomp,i,1)(1:lnsuse),':',
     &        SUSE(icomp,i,2)(1:8)          ! combined USE tokens
            if(markdown)then
              if(lnsmlcn.lt.25) lnsmlcn=24  ! at least 24 char for MLC
              if(lnsmlcn.gt.25) lnsmlcn=24  ! no more than 24 char for MLC
              if(surfcolumns.eq.3)then
                WRITE(OUTSTR2,895)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),SOTF(icomp,i)(1:10),
     &           SVFC(icomp,i)(1:4),susecomb,SMLCN(icomp,i)(1:lnsmlcn),
     &           sbound_ty(1:12)
                call edisp(ITRU,OUTSTR2)
              elseif(surfcolumns.eq.2)then
                WRITE(OUTSTR2,8955)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),SOTF(icomp,i)(1:10),
     &           susecomb,SMLCN(icomp,i)(1:lnsmlcn),sbound_ty(1:12)
                call edisp(ITRU,OUTSTR2)
              elseif(surfcolumns.eq.1)then
                WRITE(OUTSTR2,8956)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),
     &           susecomb,SMLCN(icomp,i)(1:lnsmlcn),sbound_ty(1:12)
                call edisp(ITRU,OUTSTR2)
              elseif(surfcolumns.eq.0)then
                WRITE(OUTSTR2,8957)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),
     &           SUSE(icomp,i,1)(1:8),SMLCN(icomp,i)(1:lnsmlcn),
     &           sbound_ty(1:12)
                call edisp(ITRU,OUTSTR2)
              endif
 895      FORMAT(I3,2X,A,F10.0,F11.0,6X,A,2X,A,5X,A,8X,A,4X,A,2X,A)
 8955     FORMAT(I3,2X,A,F10.0,F11.0,6X,A,2X,A,4X,A,4X,A,2X,A)
 8956     FORMAT(I3,2X,A,F10.0,F11.0,6X,A,2X,A,4X,A,2X,A)
 8957     FORMAT(I3,2X,A,F10.0,F11.0,6X,A,2X,A,4X,A,2X,A)
            else
              if(lnsmlcn.lt.24) lnsmlcn=24  ! at least 24 char for MLC
              if(lnsmlcn.gt.24) lnsmlcn=24  ! no more than 24 char for MLC
              if(surfcolumns.eq.3)then
                WRITE(OUTSTR,8951)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),SOTF(icomp,i)(1:8),
     &           SVFC(icomp,i)(1:4),susecomb,
     &           SMLCN(icomp,i)(1:lnsmlcn),sbound_ty(1:12)
              elseif(surfcolumns.eq.2)then
                WRITE(OUTSTR,8952)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),SOTF(icomp,i)(1:8),
     &           susecomb,SMLCN(icomp,i)(1:lnsmlcn),sbound_ty(1:12)
              elseif(surfcolumns.eq.1)then
                WRITE(OUTSTR,8953)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),
     &           susecomb,SMLCN(icomp,i)(1:lnsmlcn),sbound_ty(1:12)
              elseif(surfcolumns.eq.0)then
                WRITE(OUTSTR,8954)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),
     &           SUSE(icomp,i,1)(1:8),SMLCN(icomp,i)(1:lnsmlcn),
     &            sbound_ty(1:12)
              endif
              call edisp(ITRU,OUTSTR)
 8951     FORMAT(I3,2X,A,F5.0,F5.0,1X,A,1X,A,1X,A,1X,A,1X,A,2X,A)
 8952     FORMAT(I3,2X,A,F5.0,F5.0,1X,A,1X,A,1X,A,1X,A,2X,A)
 8953     FORMAT(I3,2X,A,F5.0,F5.0,1X,A,1X,A,1X,A,2X,A)
 8954     FORMAT(I3,2X,A,F5.0,F5.0,1X,A,1X,A,1X,A,2X,A)
            endif
          else
 894        FORMAT(I3,2X,A,F5.0,F5.0,1X,A,1X,A,1X,A,1X,A,2X,A)
            WRITE(OUTSTR,894)I,SIGSTR(1:7),SPAZI(icomp,i),
     &        SPELV(icomp,i),SNAME(icomp,i),SOTF(icomp,i)(1:6),
     &        SVFC(icomp,i),SMLCN(icomp,i)(1:lnsmlcn),
     &        sbound_ty(1:12)
            call edisp(ITRU,OUTSTR)
          endif
 893    CONTINUE
      else
        DO 793 I=1,NZSUR(icomp)
          lnsmlcn=lnblnk(SMLCN(icomp,i))
          if(lnsmlcn.lt.19) lnsmlcn=19  ! at least 19 char for MLC
          call OTHERINFO(icomp,i,OTHSTR)
          if(OTHSTR(1:3).eq.'||<')then
            ipo=4  ! jump past ||<
          else
            ipo=1  ! minimal offset
          endif
          call SIGFIG(SNA(icomp,i),NSIGFIG,RNO,SIGSTR,LSTR)
          if(newgeo)then
            if(markdown)then
              lnsuse=lnblnk(SUSE(icomp,i,1))
              write(susecomb,'(3a)') SUSE(icomp,i,1)(1:lnsuse),':',
     &          SUSE(icomp,i,2)(1:8)          ! combined USE tokens
              if(lnsmlcn.lt.25) lnsmlcn=25  ! at least 25 char for MLC
              if(lnsmlcn.gt.25) lnsmlcn=25  ! no more than 25 char for MLC
              if(surfcolumns.eq.3)then
                WRITE(OUTSTR2,7951)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),SOTF(icomp,i)(1:10),
     &           SVFC(icomp,i)(1:4),susecomb,SMLCN(icomp,i)(1:lnsmlcn),
     &           OTHSTR(ipo:lnblnk(OTHSTR))  ! skip the ||< characters
                call edisp(ITRU,OUTSTR2)
              elseif(surfcolumns.eq.2)then
                WRITE(OUTSTR2,7955)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),SOTF(icomp,i)(1:10),
     &           susecomb,SMLCN(icomp,i)(1:lnsmlcn),
     &           OTHSTR(ipo:lnblnk(OTHSTR))  ! skip the ||< characters
                call edisp(ITRU,OUTSTR2)
              elseif(surfcolumns.eq.1)then
                WRITE(OUTSTR2,7956)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),susecomb,
     &           SMLCN(icomp,i)(1:lnsmlcn),OTHSTR(ipo:lnblnk(OTHSTR))  ! skip the ||< characters
                call edisp(ITRU,OUTSTR2)
              elseif(surfcolumns.eq.0)then
                WRITE(OUTSTR2,7957)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),
     &           SUSE(icomp,i,1)(1:9),SMLCN(icomp,i)(1:lnsmlcn),
     &           OTHSTR(ipo:lnblnk(OTHSTR))  ! skip the ||< characters
                call edisp(ITRU,OUTSTR2)
              endif
 7951     FORMAT(I3,2X,A,F10.0,F10.0,7X,A,2X,A,4X,A,6X,A,4X,A,2X,A)
 7955     FORMAT(I3,2X,A,F10.0,F10.0,7X,A,2X,A,4X,A,3X,A,2X,A)
 7956     FORMAT(I3,2X,A,F10.0,F10.0,7X,A,2X,A,3X,A,2X,A)
 7957     FORMAT(I3,2X,A,F10.0,F10.0,7X,A,2X,A,4X,A,2X,A)
            else
              lnsuse=lnblnk(SUSE(icomp,i,1))
              write(susecomb,'(3a)') SUSE(icomp,i,1)(1:lnsuse),':',
     &          SUSE(icomp,i,2)(1:8)          ! combined USE tokens
              if(lnsmlcn.lt.24) lnsmlcn=24  ! at least 24 char for MLC
              if(lnsmlcn.gt.24) lnsmlcn=24  ! no more than 24 char for MLC
              if(surfcolumns.eq.3)then
                WRITE(OUTSTR,795)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),SOTF(icomp,i)(1:8),
     &           SVFC(icomp,i)(1:4),susecomb,SMLCN(icomp,i)(1:lnsmlcn),
     &           OTHSTR(1:lnblnk(OTHSTR))
              elseif(surfcolumns.eq.2)then
                WRITE(OUTSTR,7952)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),SOTF(icomp,i)(1:8),
     &           susecomb,SMLCN(icomp,i)(1:lnsmlcn),
     &           OTHSTR(1:lnblnk(OTHSTR))
              elseif(surfcolumns.eq.1)then
                WRITE(OUTSTR,7953)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),susecomb,
     &           SMLCN(icomp,i)(1:lnsmlcn),OTHSTR(1:lnblnk(OTHSTR))
              elseif(surfcolumns.eq.0)then
                WRITE(OUTSTR,7954)I,SIGSTR(1:7),SPAZI(icomp,i),
     &           SPELV(icomp,i),SNAME(icomp,i),
     &           SUSE(icomp,i,1)(1:8),SMLCN(icomp,i)(1:lnsmlcn),
     &           OTHSTR(1:lnblnk(OTHSTR))
              endif
              call edisp(ITRU,OUTSTR)
            endif
 795      FORMAT(I3,2X,A,F5.0,F5.0,1X,A,1X,A,1X,A,1X,A,1X,A,2X,A)
 7952     FORMAT(I3,2X,A,F5.0,F5.0,1X,A,1X,A,1X,A,1X,A,2X,A)
 7953     FORMAT(I3,2X,A,F5.0,F5.0,1X,A,1X,A,1X,A,2X,A)
 7954     FORMAT(I3,2X,A,F5.0,F5.0,1X,A,1X,A,1X,A,2X,A)
          else
            if(lnsmlcn.gt.24) lnsmlcn=24  ! no more than 24 char for MLC
            WRITE(OUTSTR,794)I,SIGSTR(1:7),SPAZI(icomp,i),
     &       SPELV(icomp,i),SNAME(icomp,i),SOTF(icomp,i)(1:6),
     &       SVFC(icomp,i),SMLCN(icomp,i)(1:lnsmlcn),
     &       OTHSTR(1:lnblnk(OTHSTR))
 794        FORMAT(I3,2X,A,F5.0,F5.0,1X,A,1X,A,1X,A,1X,A,2X,A)
            call edisp(ITRU,OUTSTR)
          endif
 793    CONTINUE
      endif
      call edisp(ITRU,' ')

      RETURN
      END

C **************** OTHERINFO
C OTHERINFO returns an English description in OTHSTR of the boundary
C condition at the other side of a surface (based on information in
C the connections file).

      SUBROUTINE OTHERINFO(izone,isurf,OTHSTR)
#include "building.h"
#include "geometry.h"

      integer lnblnk  ! function definition

      integer ncomp,ncon
      COMMON/C1/NCOMP,NCON
      COMMON/C3/IC1(MCON),IE1(MCON),ICT(MCON),IC2(MCON),IE2(MCON)
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)
      CHARACTER OTHSTR*30,SST2*12,ZST2*12

      if(izone.ne.0.and.izone.le.NCOMP)then
        if(isurf.le.NZSUR(izone).and.isurf.ne.0)then
          icon=IZSTOCN(izone,isurf)
          if(icon.eq.0)then
            WRITE(OTHSTR,'(a)') 'izstocn index unknown'
            return
          endif
          IF(ICT(ICON).EQ.-1)THEN
            WRITE(OTHSTR,'(a)') 'not yet defined'
          ELSEIF(ICT(ICON).EQ.0)THEN
            WRITE(OTHSTR,'(a)') 'external'
          ELSEIF(ICT(ICON).EQ.1)THEN
            if(IC2(ICON).eq.0.and.IE2(ICON).eq.0)then
              WRITE(OTHSTR,'(a)') 'identical environment '
            else
              WRITE(OTHSTR,'(a,i3,a,i4,a)') 'similar+- ',IC2(ICON),
     &        'dC &',IE2(ICON),'W rad'
            endif
          ELSEIF(ICT(ICON).EQ.2)THEN
            WRITE(OTHSTR,'(a,i3,a,i4,a)')   'constant @',IC2(ICON),
     &        'dC &',IE2(ICON),'W rad'
          ELSEIF(ICT(ICON).EQ.3)THEN

C If a connection know use it otherwise if zboundarytype known try
C that as an alternative (as in META file).
            icc=IZSTOCN(IC2(ICON),IE2(ICON))
            if(icc.gt.0)then
              SST2=SNAME(IC2(ICON),IE2(ICON))
              ZST2=zname(IC2(ICON))
            else
              if(zboundarytype(izone,isurf,2).gt.0.and.
     &           zboundarytype(izone,isurf,3).gt.0)then
                ioz=zboundarytype(izone,isurf,2)
                ios=zboundarytype(izone,isurf,3)
                SST2=SNAME(ioz,ios)
                ZST2=zname(ioz)
              else
                SST2='not_known'
                ZST2='not_known'
              endif
            endif
            WRITE(OTHSTR,'(4a)')'to ',SST2(1:LNBLNK(SST2)),':',
     &        ZST2(1:LNBLNK(ZST2))
          ELSEIF(ICT(ICON).EQ.4)THEN
            IF(IC2(ICON).GT.0)THEN
              WRITE(OTHSTR,'(a,i2)') 'ground profile ',IC2(ICON)
            ELSEIF(IC2(ICON).EQ.-3)THEN
              WRITE(OTHSTR,'(a)') '||< 3D ground model'
            ELSE
              WRITE(OTHSTR,'(a,i2)') 'user def grnd profile ',
     &          IE2(ICON)
            ENDIF
          ELSEIF(ICT(ICON).EQ.5)THEN
            WRITE(OTHSTR,'(a)') 'adiabatic'

C BASESIMP begin.
          ELSEIF(ICT(ICON).EQ.6)THEN
            WRITE(OTHSTR,'(a,i3)') 'BASESIMP config type ',IC2(ICON)
C BASESIMP end.

C CEN 13791 partition start.
          ELSEIF(ICT(ICON).EQ.7)THEN
            if(IC2(ICON).eq.0.and.IE2(ICON).eq.0)then
              WRITE(OTHSTR,'(a)') 'Identical CEN 13791 '
            else
              WRITE(OTHSTR,'(a,i3,a,i4,a)') 'CEN13791+- ',IC2(ICON),
     &        'dC &',IE2(ICON),'W rad'
            endif
C CEN 13791 partition end.
          ENDIF
        else

C The passed surface number was out of range.
          WRITE(OTHSTR,'(a)') 'surface unknown'
        endif
      else

C The passed zone number was out of range.
        WRITE(OTHSTR,'(a)') 'zone unknown'
      endif
      RETURN
      END

C ********************** ZINFOREP
C 'ZINFOREP' takes data from the zone geometry commons and produces
C a high level report channel (itu). It assumes that common blocks
C (G7,PREC2,PREC2,PREC17,C20,C24) have been filled. It does not
C require a recent scan of zone geometry.
      SUBROUTINE ZINFOREP(itu,icomp)
#include "building.h"
#include "geometry.h"

C Markdown flag.
      logical markdown
      common/markdownflag/markdown

      common/SFIG/NSIGFIG
      character outstr*124,t10*10,t10a*10,T12*12,T12A*12

      call edisp(itu,' ')

C Convert nzsur,nztv,zvol etc into strings with no leading spaces.
      CALL INTSTR(NZSUR(icomp),t10,lna,IER)
      CALL INTSTR(NZTV(icomp),t10a,lnaa,IER)
      if(markdown)then
        WRITE(outstr,'(3a,i2,5a)')'Zone ',
     &   zname(ICOMP)(1:lnzname(ICOMP)),
     &   ' (',ICOMP,') is composed of ',t10(1:lna),' surfaces and ',
     &   t10a(1:lnaa),' vertices.'
C        call edisp2tr(itu,outstr)
        call edisp(itu,outstr)
      else
        WRITE(outstr,'(3a,i2,5a)')' Zone ',
     &   zname(ICOMP)(1:lnzname(ICOMP)),
     &   ' (',ICOMP,') is composed of ',t10(1:lna),' surfaces and ',
     &   t10a(1:lnaa),' vertices.'
        call edisp(itu,outstr)
      endif

C Report data to required number of significant figures.
      call SIGFIG(VOL(icomp),NSIGFIG,RNO,T12,LSTR)
      if(markdown)then
        WRITE(outstr,'(3a)')'It encloses a volume of ',T12(1:LSTR),
     &   ' m^3^ of space, with a total surface '
C        call edisp2tr(itu,outstr)
        call edisp(itu,outstr)
      else
        WRITE(outstr,'(3a)')' It encloses a volume of ',T12(1:LSTR),
     &   'm^3 of space, with a total surface'
        call edisp(itu,outstr)
      endif
      call SIGFIG(zonetotsurfacearea(icomp),NSIGFIG,RNO,T12,LSTR)
      call SIGFIG(ZBASEA(ICOMP),NSIGFIG,RNO,T12A,LSTRA)
      if(IUZBASEA(icomp).eq.0)then
        if(markdown)then
          write(outstr,'(5a)') 'area of ',T12(1:LSTR),
     &     ' m^2^ & approx floor area of ',T12A(1:LSTRA),' m^2^.  '
        else
          write(outstr,'(5a)') ' area of ',T12(1:LSTR),
     &     'm^2 & approx floor area of ',T12A(1:LSTRA),'m^2'
        endif
      elseif(IUZBASEA(icomp).eq.1)then
        if(markdown)then
          write(outstr,'(5a)') ' area of ',T12(1:LSTR),
     &     ' m^2^ & user edited floor area of ',T12A(1:LSTRA),' m^2^.  '
        else
          write(outstr,'(5a)') ' area of ',T12(1:LSTR),
     &     'm^2 & user edited floor area of ',T12A(1:LSTRA),'m^2'
        endif
      elseif(IUZBASEA(icomp).eq.2)then
        if(markdown)then
          write(outstr,'(5a)') ' area of ',T12(1:LSTR),
     &     ' m^2^ & user list floor area of ',T12A(1:LSTRA),' m^2^.  '
        else
          write(outstr,'(5a)') ' area of ',T12(1:LSTR),
     &     'm^2 & user list floor area of ',T12A(1:LSTRA),'m^2'
        endif
      endif
      if(markdown)then
C        call edisp2tr(itu,outstr)
        call edisp(itu,outstr)
      else
        call edisp(itu,outstr)
      endif
      if(markdown)then
        WRITE(outstr,'(2A)') zdesc(ICOMP)(1:lnblnk(zdesc(ICOMP))),'. '
        call edisp2tr(itu,outstr)
      else
        WRITE(outstr,'(1x,A)') zdesc(ICOMP)(1:lnblnk(zdesc(ICOMP)))
        call edisp(itu,outstr)
      endif
      return
      end

C ********************** ZINFO
C Takes data from the zone geometry commons G1,G2 and returns
C several derived values.
      SUBROUTINE ZINFO(icomp,ZOA,ZVOL,act)
#include "building.h"
#include "geometry.h"
#include "sbem.h"

C Parameters
      integer ICOMP
      real ZOA  ! returned total surface area associated with zone
      real ZVOL ! returned zone bounded volume
      character act*1  ! q is quiet  - includes warnings

      DIMENSION XSUM(MS),YSUM(MS),ZSUM(MS)
      LOGICAL CLOSE
      character outs*124

C Set area summations to zero & consider each surface of the zone in turn.
      PI = 4.0 * ATAN(1.0)
      R=PI/180.
      ZOA=0.
      DO 10 I=1,NSUR
        XS=0.; YS=0.; ZS=0.; ZMAX=-1.E+10; ZMIN=1.E+10

C Compute gross surface area
        ADDL=0.
        DO 20 J=1,NVER(I)
          K=J+1
          IF(J.EQ.NVER(I))K=1
          IP1=JVN(I,J)
          IP2=JVN(I,K)
          if(IP1.gt.0.and.IP2.gt.0)then
            ZMAX=AMAX1(ZMAX,Z(IP1))
            ZMIN=AMIN1(ZMIN,Z(IP1))
            XS=XS+Y(IP1)*Z(IP2)-Z(IP1)*Y(IP2)
            YS=YS+Z(IP1)*X(IP2)-X(IP1)*Z(IP2)
            ZS=ZS+X(IP1)*Y(IP2)-Y(IP1)*X(IP2)
            ADDL=ADDL+SQRT((X(IP2)-X(IP1))**2+(Y(IP2)-Y(IP1))**2+
     &       (Z(IP2)-Z(IP1))**2)
          else
            write(outs,'(a,i3,a,i3,a,i3,a)') ' Edge ',J,' in ',icomp,
     &        ' surf ',i,' vertices are zero.'
            CALL USRMSG(outs,
     &        'Other geometric properties may be incorrect.','W')
          endif
   20   CONTINUE
        XSUM(I)=XS
        YSUM(I)=YS
        ZSUM(I)=ZS
        ZAREA=0.5*SQRT(XS*XS+YS*YS+ZS*ZS)

C Surface area is given by:
        SNA(ICOMP,I)=ZAREA

C Check for -ve surface area: windows wrongly specified.
C Suppress warning message if UK NCM notional model
        IF(SNA(ICOMP,I).LT.0.00001.AND.INOTI.NE.1)THEN
          write(outs,'(a,i3,a,i3,a)') ' Area of zone ',icomp,' surf ',
     &      i,' is less than 0.00001m^2!'
          CALL USRMSG(outs,
     &      'Other geometric properties may be incorrect.','W')
          RETURN
        ENDIF

C And surface area summations by:
        ZOA=ZOA+SNA(ICOMP,I)
   10 CONTINUE

C This section computes the volume of any polyhedral zone by computing
C the algebraic sum of the volumes of a prism formed by joining the
C coordinate system origin point (0,0,0) with each vertex of each
C face in turn.   The prism volume of a face whose associated outward
C normal (when the face vertices are defined in an anticlockwise
C direction when viewed from 'outside') faces away from the origin is
C computed as a positive value.   Conversely a negative value is
C obtained when the outward normal to the face points towards the origin.
      V=0.0
      DO 60 K=1,NSUR
        JJ=JVN(K,1)
        if(JJ.gt.0)then
          PV=.16667*(X(JJ)*XSUM(K)+Y(JJ)*YSUM(K)+Z(JJ)*ZSUM(K))
        else
          PV=0.0
        endif
        V=V+PV

C This section computes the azimuth and elevation angles of the outward
C facing normal for each plane in turn. The azimuth is the angle -
C measured from north (Y-axis) in degrees - clockwise positive.
C The elevation angle is the angle between the plane normal and the
C horizontal measured vertically in degrees.
        SPAZI(ICOMP,K)=90.0
        IF(XSUM(K).LT.0.0)SPAZI(ICOMP,K)=-90.0
        CALL ECLOSE(XSUM(K),0.0,0.0001,CLOSE)
        IF(CLOSE)SPAZI(ICOMP,K)=0.0
        SPELV(ICOMP,K)=90.0
        IF(ZSUM(K).LT.0.0)SPELV(ICOMP,K)=-90.0
        CALL ECLOSE(ZSUM(K),0.0,0.0001,CLOSE)
        IF(CLOSE)SPELV(ICOMP,K)=0.0
        CALL ECLOSE(YSUM(K),0.0,0.0001,CLOSE)
        IF(CLOSE)goto 61
        AZI=ATAN2(XSUM(K),YSUM(K))
        SPAZI(ICOMP,K)=AZI/R
   61   SRX2Y2=SQRT(XSUM(K)*XSUM(K)+YSUM(K)*YSUM(K))
        IF(SPAZI(ICOMP,K).LT.0.)SPAZI(ICOMP,K)=SPAZI(ICOMP,K)+360.
        CALL ECLOSE(SRX2Y2,0.0,0.0001,CLOSE)
        IF(CLOSE)goto 60
        ELV=ATAN2(ZSUM(K),SRX2Y2)
        SPELV(ICOMP,K)=ELV/R

C << this could be a place to insert special logic for horizontal
C << surfaces as done in pangs in esruish.
C << note logic is slightly different for the spazi and spelv in
C << pangs - it might be better?
C << also - why not include surface tilt angle?

   60 CONTINUE
      ZVOL=ABS(V)

C Check for small volume.
      IF(ZVOL.LE.0.00001.and.act.eq.'-')THEN
        CALL USRMSG(' Negative or zero zone volume detected. Vertex',
     &    ' information may be in error or surfaces incomplete.','W')
        RETURN
      ENDIF

      RETURN
      END

C ********************** ZGUPDATE
C 'ZGUPDATE' takes the geometry commons G1 and updates G7 & PREC2
C SUREQN(MCOM,MS,4)- equation of each polygon (from geometry.h)
C SURCOG(MCOM,MS,3)- vertex weighted COG of polygon (from geometry.h)
C SURVN(MCOM,MS,3) - unit normal vector from COG of polygon.
C VOL(MCOM)        - zone bounded volume
      SUBROUTINE ZGUPDATE(itrc,icomp,ier)
#include "building.h"
#include "geometry.h"
#include "sbem.h"

      integer lnblnk  ! function definition

C << could it use ?? instead of G1? >>

      common/OUTIN/IUOUT,IUIN,IEOUT
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)
      DIMENSION XSUM(MS),YSUM(MS),ZSUM(MS)
      DIMENSION XX(MV),YY(MV),ZZ(MV),CG(3),EQN(4)
      LOGICAL CLOSE,closeaz,closeel
      CHARACTER outs*124
      integer lln   ! length of surface name
      real distance,ofby

C ZGUPDATE assumes that the zone geometry file has been read prior to
C its invocation and that the read of the model configuration file has
C instantiated the common block.

C Set area summations to zero & consider each surface of the zone in turn.
      ier=0
      PI = 4.0 * ATAN(1.0)
      R=PI/180.

      DO 10 I=1,NZSUR(icomp)

C Recover the connection associated with this zone and surface.
        icc=IZSTOCN(icomp,i)

C Do reality checks on connection and number of edges.
        if(icc.eq.0)then
          write(outs,*) 'Zone ',icomp,' & surface ',i,
     &      ' did not resolve to a known item in connection list.'
          call edisp(iuout,outs)
          ier=3
          goto 10
        endif
        if(NVER(I).eq.0)then
          write(outs,*) 'Zone ',icomp,' & surface ',i,
     &      ' has no edges.'
          call edisp(iuout,outs)
          ier=4
          goto 10
        endif
        XS=0.
        YS=0.
        ZS=0.
        ZMAX=-1.E+10
        ZMIN=1.E+10

C Compute gross surface area
        ADDL=0.
        DO 20 J=1,NVER(I)
          K=J+1
          IF(J.EQ.NVER(I))K=1
          IP1=JVN(I,J)
          IP2=JVN(I,K)
          if(IP1.gt.0.and.IP2.gt.0)then
            ZMAX=AMAX1(ZMAX,Z(IP1))
            ZMIN=AMIN1(ZMIN,Z(IP1))
            XS=XS+Y(IP1)*Z(IP2)-Z(IP1)*Y(IP2)
            YS=YS+Z(IP1)*X(IP2)-X(IP1)*Z(IP2)
            ZS=ZS+X(IP1)*Y(IP2)-Y(IP1)*X(IP2)
            ADDL=ADDL+SQRT((X(IP2)-X(IP1))**2+(Y(IP2)-Y(IP1))**2+
     &        (Z(IP2)-Z(IP1))**2)
          endif
   20   CONTINUE
        XSUM(I)=XS
        YSUM(I)=YS
        ZSUM(I)=ZS

C Compute surface area and check for negative or near zero area.
C Suppress warning message if UK NCM notional model
        ZAREA=0.5*SQRT(XS*XS+YS*YS+ZS*ZS)
        SNA(icomp,i)=ZAREA
        IF(SNA(icomp,i).LT.0.00001.AND.INOTI.NE.1)THEN
          write(outs,'(a,i3,a)') ' Area of surf connection ',icc,
     &      ' is less than 0.00001m^2!'
          CALL USRMSG(outs,
     &      'Other geometric properties may be incorrect.','W')
          ier=2
        ENDIF

C Derive the equation of the polygon, if problems generating the
C equation echo warnings.
        DO 60 KK=1,NVER(i)
          if(JVN(I,KK).eq.0)then
            XX(KK) = 0.0; YY(KK) = 0.0; ZZ(KK) = 0.0
          else
            XX(KK) = X(JVN(I,KK))
            YY(KK) = Y(JVN(I,KK))
            ZZ(KK) = Z(JVN(I,KK))
          endif
   60   CONTINUE
        N = NVER(i)
        ierr=0
        call PLEQN(XX,YY,ZZ,N,CG,EQN,IERR)
        lln=lnblnk(sname(icomp,i))
        if(ierr.ne.0.and.itrc.gt.0)then
          if(ierr.eq.1)then
            write(outs,'(5a)') ' Surface ',sname(icomp,i)(1:lln),
     &        ' in ',zname(icomp)(1:lnzname(icomp)),
     &        ' not enough points for plneqn.'
          elseif(ierr.eq.-1)then
            write(outs,'(5a)') ' Surface ',sname(icomp,i)(1:lln),
     &        ' in ',zname(icomp)(1:lnzname(icomp)),
     &        ' issue with unit vector in plneqn.'
          else
            write(outs,'(5a)') ' Surface ',sname(icomp,i)(1:lln),
     &        ' in ',zname(icomp)(1:lnzname(icomp)),
     &        ' issue in plneqn.'
          endif
          call edisp(iuout,outs)
        endif

C See if any points are out of the plane. Note a point that is
C only a few mm off of the line can result in a warp which is
C reported as a larger number.
        ofby=0.0
        DO 64 KK=1,NVER(i)
          distance = EQN(1)*XX(KK) + EQN(2)*YY(KK) + EQN(3)*ZZ(KK)
     &      - EQN(4)
          if(distance.gt.ofby) ofby=distance
  64    continue
        lln=lnblnk(sname(icomp,i))
        if(ofby.gt.0.01.and.itrc.gt.0)then
          write(outs,'(5a,F6.4,a)') ' Surface ',sname(icomp,i)(1:lln),
     &      ' in ',zname(icomp)(1:lnzname(icomp)),
     &      ' is warped by ',ofby,
     &      'm. Please check via surface transforms.'
          call edisp(iuout,outs)
        endif

C Update the G7 common block.
        SUREQN(icomp,i,1)=EQN(1)
        SUREQN(icomp,i,2)=EQN(2)
        SUREQN(icomp,i,3)=EQN(3)
        SUREQN(icomp,i,4)=EQN(4)

        SURCOG(icomp,i,1)=CG(1)
        SURCOG(icomp,i,2)=CG(2)
        SURCOG(icomp,i,3)=CG(3)
        SURVN(icomp,i,1)=CG(1)+EQN(1)
        SURVN(icomp,i,2)=CG(2)+EQN(2)
        SURVN(icomp,i,3)=CG(3)+EQN(3)
   10 CONTINUE

C This section computes the volume of any polyhedral zone by computing
C the algebraic sum of the volumes of a prism formed by joining the
C coordinate system origin point (0,0,0) with each vertex of each
C face in turn.   The prism volume of a face whose associated outward
C normal (when the face vertices are defined in an anticlockwise
C direction when viewed from 'outside') faces away from the origin is
C computed as a positive value.   Conversely a negative value is
C obtained when the outward normal to the face points towards the origin.
      V=0.0
      DO 62 K=1,NZSUR(icomp)
        icc=IZSTOCN(icomp,k)
        if(icc.eq.0)goto 62
        JJ=JVN(K,1)
        if(JJ.gt.0)then
          PV=.16667*(X(JJ)*XSUM(K)+Y(JJ)*YSUM(K)+Z(JJ)*ZSUM(K))
        else
          PV=0.0
        endif
        V=V+PV

C This section computes the azimuth and elevation angles of the outward
C facing normal for each plane in turn. The azimuth is the angle -
C measured from north (Y-axis) in degrees - clockwise positive.
C The elevation angle is the angle between the plane normal and the
C horizontal measured vertically in degrees.
        SPAZI(icomp,k)=90.0
        IF(XSUM(K).LT.0.0) SPAZI(icomp,k)=-90.0
        CALL ECLOSE(XSUM(K),0.0,0.0001,CLOSE)
        IF(CLOSE) SPAZI(icomp,k)=0.0
        SPELV(icomp,k)=90.0
        IF(ZSUM(K).LT.0.0) SPELV(icomp,k)=-90.0
        CALL ECLOSE(ZSUM(K),0.0,0.0001,CLOSE)
        IF(CLOSE) SPELV(icomp,k)=0.0
        CALL ECLOSE(YSUM(K),0.0,0.0001,CLOSE)
        IF(CLOSE) goto 61
        AZI=ATAN2(XSUM(K),YSUM(K))
        SPAZI(icomp,k)=AZI/R
   61   SRX2Y2=SQRT(XSUM(K)*XSUM(K)+YSUM(K)*YSUM(K))
        IF(SPAZI(icomp,k).LT.0.) SPAZI(icomp,k)=SPAZI(icomp,k)+360.
        CALL ECLOSE(SRX2Y2,0.0,0.0001,CLOSE)
        IF(CLOSE) goto 63
        ELV=ATAN2(ZSUM(K),SRX2Y2)
        SPELV(icomp,k)=ELV/R

C Check that the azimuth and elevation based on surface normal are
C the same as computed above.
   63   vdx=SUREQN(icomp,k,1)
        vdy=SUREQN(icomp,k,2)
        vdz=SUREQN(icomp,k,3)
        call UV2AZ(vdx,vdy,vdz,uazim,uelev)
        call eclose(uazim,SPAZI(icomp,k),0.001,closeaz)
        call eclose(uelev,SPELV(icomp,k),0.001,closeel)

C << this could be a place to insert special logic for horizontal
C << surfaces as done in pangs in esruish.
C << note logic is slightly different for the spazi and spelv in
C << pangs - it might be better?
C << also - why not include surface tilt angle?

        if(ITRC.gt.1)then
          write(outs,*) 'name azi elv m2 ',SNAME(icomp,i),
     &      SPAZI(icomp,i),SPELV(icomp,i),SNA(icomp,i)
          call edisp(iuout,outs)
          write(outs,*) 'Unit vector azi elv         ',uazim,uelev,
     &      closeaz,closeel
          call edisp(iuout,outs)
          write(outs,*) 'SUREQN ',SUREQN(icomp,i,1),
     &      SUREQN(icomp,i,2),SUREQN(icomp,i,3),SUREQN(icomp,i,4)
          call edisp(iuout,outs)
          write(outs,*) 'SURCOG ',SURCOG(icomp,i,1),SURCOG(icomp,i,2),
     &      SURCOG(icomp,i,3),' SURVN ',SURVN(icomp,i,1),
     &      SURVN(icomp,i,2),SURVN(icomp,i,3)
          call edisp(iuout,outs)
        endif
   62 CONTINUE
      ZVOL=ABS(V)

C Check for small volume.
      if(ZVOL.LE.0.00001)then
        CALL USRMSG(' Negative or zero zone volume detected. Vertex',
     &    ' information may be in error or surfaces incomplete.','W')
        ier=4
      else
        VOL(icomp)=ZVOL
      endif

      RETURN
      END

C ********** getperimeter **********

      subroutine getperimeter(icomp,sperim)
#include "building.h"
#include "geometry.h"

      integer icomp        ! the current zone
      real sperim(MS)      ! values to return
      integer i,ii,jj,list ! for looping
      real vdis            ! summation of edge length

      do i=1,nzsur(icomp)

C Compute perimeter of the surface.
        sperim(i)=0.0
        list=isznver(icomp,i)-1
        do ii=1,list
          jj=iszjvn(icomp,i,ii)
          kk=iszjvn(icomp,i,ii+1)
          vdis=0.0
          if(jj.gt.0.and.kk.gt.0)then
            vdis= crowxyz(szcoords(icomp,jj,1),szcoords(icomp,jj,2),
     &        szcoords(icomp,jj,3),szcoords(icomp,kk,1),
     &        szcoords(icomp,kk,2),szcoords(icomp,kk,3))
          endif
          SPERIM(i)=SPERIM(i)+vdis
  42    enddo

C Link back to the start vertex.
        jj=iszjvn(icomp,i,isznver(icomp,i))
        kk=iszjvn(icomp,i,1)
        vdis=0.0
        if(jj.gt.0.and.kk.gt.0)then
          vdis= crowxyz(szcoords(icomp,jj,1),szcoords(icomp,jj,2),
     &      szcoords(icomp,jj,3),szcoords(icomp,kk,1),
     &      szcoords(icomp,kk,2),szcoords(icomp,kk,3))
        endif
        SPERIM(i)=SPERIM(i)+vdis
      enddo
      return
      end

C*******************************************************************
C SURLEHI Determines the overall length and height of a
C surface (bounding box) and passes the vaules back as XYMAX and ZMAX.
C This variant assumes current G1 common block.
C Uses a temporary transform into 2D to get this data so
C it work on surface of most orientations.

      SUBROUTINE SURLEHI(IS,XYMAX,ZMAX,llpos,lrpos,ulpos,urpos)
#include "building.h"
#include "geometry.h"

      DIMENSION  XX(MV),YY(MV),ZZ(MV)
      DIMENSION  VP(3),EP(3),EQN(4)
      DIMENSION  TMAT(4,4),RMAT(4,4)

      real XYMAX,ZMAX
      real XMIN,YMIN,XMAX,YMAX
      real shortest,shortestlr,shortestul,shortestur
      integer llpos,lrpos,ulpos,urpos     ! closest to BB corners for parent

C Set maximum values to zero.
      XYMAX = 0.0; ZMAX = 0.0

C Transform surface into into site coordinates in the
C surface of the plane. Make up XX,YY,ZZ to pass across to the
C transform routine.
      N = NVER(IS)
      DO 150 J = 1,N
        if(JVN(IS,J).gt.0)then
          XX(J) = X(JVN(IS,J))
          YY(J) = Y(JVN(IS,J))
          ZZ(J) = Z(JVN(IS,J))
        else
          XX(J) = 0.0; YY(J) = 0.0; ZZ(J) = 0.0
        endif
  150 CONTINUE

C Find transformation matrices that normalise face.
      call PLEQN(XX,YY,ZZ,N,VP,EQN,IERR)
      IF (IERR .LT. 0) return
      DO 250 J = 1,3
        EP(J) = VP(J) + EQN(J)
  250 CONTINUE
      CALL  EYEMAT(EP,VP,1.0,TMAT,RMAT)

C Transform all points in surface and find lower left corner
C and the upper right corner.
      XMIN=100.0; YMIN=100.0
      XMAX=0.0; YMAX=0.0
      DO 300 I=1,N
        CALL ORTTRN(XX(I),YY(I),ZZ(I),TMAT,X1,Y1,ZZZ,IERR)
        IF(X1.LT.XMIN)XMIN=X1
        IF(Y1.LT.YMIN)YMIN=Y1
        IF(X1.GT.XMAX)XMAX=X1
        IF(Y1.GT.YMAX)YMAX=Y1
  300 CONTINUE

C Determine the height difference between the vertices. If the height
C difference is greater than the previous values then define as new
C maximum.
      ZVAL = ABS(YMAX - YMIN)
      If (ZVAL.gt.ZMAX) ZMAX = ZVAL

C Determine the distance between the vertices on the x plane. If the
C distance is greater than the previous values then define as new
C maximum.
      XYVAL= ABS(XMAX - XMIN)
      if (XYVAL.gt.XYMAX) XYMAX = XYVAL

C Loop through the points again and figure out which one is closest
C to XMIN and YMIN (lower left).
      shortest=100.0; llpos=0
      DO I=1,N
        CALL ORTTRN(XX(I),YY(I),ZZ(I),TMAT,X1,Y1,ZZZ,IERR)
        vdis= crowxyz(xmin,ymin,0.0,x1,y1,0.0)
        if(vdis.lt.shortest)then
          shortest=vdis
          llpos=i
        endif
      ENDDO

C Loop through the points again and figure out which one is closest
C to XMAX and YMIN (lower right).
      shortestlr=100.0; lrpos=0
      DO I=1,N
        CALL ORTTRN(XX(I),YY(I),ZZ(I),TMAT,X1,Y1,ZZZ,IERR)
        vdis= crowxyz(xmax,ymin,0.0,x1,y1,0.0)
        if(vdis.lt.shortestlr)then
          shortestlr=vdis
          lrpos=i
        endif
      ENDDO

C Loop through the points again and figure out which one is closest
C to XMIN and YMAX (upper left).
      shortestul=100.0; ulpos=0
      DO I=1,N
        CALL ORTTRN(XX(I),YY(I),ZZ(I),TMAT,X1,Y1,ZZZ,IERR)
        vdis= crowxyz(xmin,ymax,0.0,x1,y1,0.0)
        if(vdis.lt.shortestul)then
          shortestul=vdis
          ulpos=i
        endif
      ENDDO

C Loop through the points again and figure out which one is closest
C to XMAX and YMAX (upper right).
      shortestur=100.0; urpos=0
      DO I=1,N
        CALL ORTTRN(XX(I),YY(I),ZZ(I),TMAT,X1,Y1,ZZZ,IERR)
        vdis= crowxyz(xmax,ymax,0.0,x1,y1,0.0)
        if(vdis.lt.shortestur)then
          shortestur=vdis
          urpos=i
        endif
      ENDDO

C Debug.
C      write(6,*) 'SURLEHI ',is
C      write(6,*) 'x and y ',XMIN,XMAX,YMIN,YMAX
C      write(6,*) ZMAX,XYMAX
C      write(6,*) 'llpos lrpos urpos ulpos',llpos,lrpos,urpos,ulpos
C      write(6,'(a,i3,4f5.1,a,i3,4f5.1,a,4i4)')
C     & 'lower left at ',llpos,shortest,
C     &  XX(llpos),YY(llpos),ZZ(llpos),
C     &  ' lr',lrpos,shortestlr,XX(lrpos),
C     &  YY(lrpos),ZZ(lrpos),
C     &  ' verts are ',JVN(IS,llpos),JVN(IS,lrpos),JVN(IS,urpos),
C     &  JVN(IS,ulpos)
C      write(6,'(a,i3,4f5.1,a,i3,4f5.1)')
C     & 'upper left at ',ulpos,shortestul,
C     &  XX(ulpos),YY(ulpos),ZZ(ulpos),
C     &  ' ur',urpos,shortestur,XX(urpos),
C     &  YY(urpos),ZZ(urpos)

      return
      END


C*******************************************************************
C ZSURLEHI Determines the overall length and height of a
C surface (bounding box) and passes the vaules back as XYMAX and ZMAX
C And the Z delta between the surface lower left and the zone minimum
C Z value.
C This variant does not use G1 common block.
C Uses a temporary transform into 2D to get this data.

      SUBROUTINE ZSURLEHI(IZ,IS,XYMAX,ZMAX,llpos,lrpos,ulpos,urpos,
     &  DZLLFF)
#include "building.h"
#include "geometry.h"
#include "prj3dv.h"

      DIMENSION  XX(MV),YY(MV),ZZ(MV)
      DIMENSION  VP(3),EP(3),EQN(4)
      DIMENSION  TMAT(4,4),RMAT(4,4)

      real XYMAX,ZMAX
      real XMIN,YMIN,XMAX,YMAX
      real shortest,shortestlr,shortestul,shortestur
      real DZLLFF                         ! Delta Z between lower left and zone minimum.
      integer llpos,lrpos,ulpos,urpos     ! closest to BB corners for parent

C Set maximum values to zero.
      XYMAX = 0.0; ZMAX = 0.0

C Transform surface into into site coordinates in the
C surface of the plane. Make up XX,YY,ZZ to pass across to the
C transform routine.
      N = isznver(IZ,IS)
      DO 150 J = 1,N
        if(iszjvn(IZ,IS,J).gt.0)then
          XX(J) = szcoords(iz,iszjvn(IZ,IS,j),1)
          YY(J) = szcoords(iz,iszjvn(IZ,IS,j),2)
          ZZ(J) = szcoords(iz,iszjvn(IZ,IS,j),3)
        else
          XX(J) = 0.0; YY(J) = 0.0; ZZ(J) = 0.0
        endif
  150 CONTINUE

C Find transformation matrices that normalise face.
      call PLEQN(XX,YY,ZZ,N,VP,EQN,IERR)
      IF (IERR .LT. 0) return
      DO 250 J = 1,3
        EP(J) = VP(J) + EQN(J)
  250 CONTINUE
      CALL  EYEMAT(EP,VP,1.0,TMAT,RMAT)

C Transform all points in surface and find lower left corner
C and the upper right corner.
      XMIN=100.0; YMIN=100.0; ZMin=100.0
      XMAX=0.0; YMAX=0.0
      DO 300 I=1,N
        CALL ORTTRN(XX(I),YY(I),ZZ(I),TMAT,X1,Y1,ZZZ,IERR)
        IF(X1.LT.XMIN)XMIN=X1
        IF(Y1.LT.YMIN)YMIN=Y1
        IF(X1.GT.XMAX)XMAX=X1
        IF(Y1.GT.YMAX)YMAX=Y1
        IF(ZZZ.LT.ZMIN)ZMIN=ZZZ
  300 CONTINUE

C Determine the height difference between the vertices. If the height
C difference is greater than the previous values then define as new
C maximum.
      ZVAL = ABS(YMAX - YMIN)
      If (ZVAL.gt.ZMAX) ZMAX = ZVAL

C Determine the distance between the vertices on the x plane. If the
C distance is greater than the previous values then define as new
C maximum.
      XYVAL= ABS(XMAX - XMIN)
      if (XYVAL.gt.XYMAX) XYMAX = XYVAL

C Loop through the points again and figure out which one is closest
C to XMIN and YMIN (lower left).
      shortest=100.0; llpos=0
      DO I=1,N
        CALL ORTTRN(XX(I),YY(I),ZZ(I),TMAT,X1,Y1,ZZZ,IERR)
        vdis= crowxyz(xmin,ymin,0.0,x1,y1,0.0)
        if(vdis.lt.shortest)then
          shortest=vdis
          llpos=i
        endif
      ENDDO

C Loop through the points again and figure out which one is closest
C to XMAX and YMIN (lower right).
      shortestlr=100.0; lrpos=0
      DO I=1,N
        CALL ORTTRN(XX(I),YY(I),ZZ(I),TMAT,X1,Y1,ZZZ,IERR)
        vdis= crowxyz(xmax,ymin,0.0,x1,y1,0.0)
        if(vdis.lt.shortestlr)then
          shortestlr=vdis
          lrpos=i
        endif
      ENDDO

C Loop through the points again and figure out which one is closest
C to XMIN and YMAX (upper left).
      shortestul=100.0; ulpos=0
      DO I=1,N
        CALL ORTTRN(XX(I),YY(I),ZZ(I),TMAT,X1,Y1,ZZZ,IERR)
        vdis= crowxyz(xmin,ymax,0.0,x1,y1,0.0)
        if(vdis.lt.shortestul)then
          shortestul=vdis
          ulpos=i
        endif
      ENDDO

C Loop through the points again and figure out which one is closest
C to XMAX and YMAX (upper right).
      shortestur=100.0; urpos=0
      DO I=1,N
        CALL ORTTRN(XX(I),YY(I),ZZ(I),TMAT,X1,Y1,ZZZ,IERR)
        vdis= crowxyz(xmax,ymax,0.0,x1,y1,0.0)
        if(vdis.lt.shortestur)then
          shortestur=vdis
          urpos=i
        endif
      ENDDO

C Debug.
C      write(6,'(a,2i4,a,6f7.2)') 'ZSURLEHI ',iz,is,
C     &  ' x & y &z',XMIN,XMAX,YMIN,YMAX,ZMAX,XYMAX
C      write(6,'(a,4i4)') 'llpos lrpos urpos ulpos',
C     &  llpos,lrpos,urpos,ulpos
C      write(6,'(a,i3,4f5.1,a,i3,4f5.1,a,4i4)')
C     & 'lower left at ',llpos,shortest,
C     &  XX(llpos),YY(llpos),ZZ(llpos),
C     &  ' lr',lrpos,shortestlr,XX(lrpos),
C     &  YY(lrpos),ZZ(lrpos),
C     &  ' verts are ',JVN(IS,llpos),JVN(IS,lrpos),JVN(IS,urpos),
C     &  JVN(IS,ulpos)
C      write(6,'(a,i3,4f5.1,a,i3,4f5.1)')
C     & 'upper left at ',ulpos,shortestul,
C     &  XX(ulpos),YY(ulpos),ZZ(ulpos),
C     &  ' ur',urpos,shortestur,XX(urpos),
C     &  YY(urpos),ZZ(urpos)
      DZLLFF=ZZ(llpos)-ZZMN(IZ)
C      write(6,'(a,4F8.2)') 'Delta Z over zone base to lower left ',
C     & ZZ(llpos),ZZMN(IZ),(ZZ(llpos)-ZZMN(IZ)),DZLLFF

      return
      END


C ******* suredgeadj
C Suredgeadj detects polygon edge associations (how many associated surfs etc.)
C Used to assist in drawing wireframe and for determining parent and child
c relationships. Fills common block G8. Assumes that the surface is
C in the current zones common blocks.
C act is '-' derive geometric data or 'c' also determine parent
C act is 't' also determine thermal bridges.
      subroutine suredgeadj(itrc,act,izone,ier)
#include "building.h"
#include "geometry.h"
#include "help.h"

      common/OUTIN/IUOUT,IUIN,IEOUT

      integer ncomp,ncon
      COMMON/C1/NCOMP,NCON
      COMMON/C3/IC1(MCON),IE1(MCON),ICT(MCON),IC2(MCON),IE2(MCON)
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

      CHARACTER outs*124,act*1
      logical closeazi,closeelv,foundone,ok,same

C iwhich is local variable to sort most often referenced adjacent surf.
      dimension iwhich(MV),iwhichs(MV)
C      real distb ! currently not used

C There is a file unit clash between gethelptext and file units
C in ish. Bypass call to gethelptext and set h() manually.
C      helpinsub='egeometry'  ! set for subroutine
C      helptopic='check_adjacency'
      h(1)='The shape & edges of the two surfaces suggest'
      h(2)='that one is a subsurface.'
      h(3)=' '
      h(4)='You may also be asked if the parent surface is in the'
      h(5)='current zone.'
      nbhelp=5
C      call gethelptext(helpinsub,helptopic,nbhelp)

      if(izone.ne.0.and.izone.le.NCOMP)then
        if(act.eq.'c')then

C Do test call to surrel to see if it picks up any parent child
C relationships. Use edge checking logic in surrel2.
          call SURREL2('s',izone,ier)
        endif

        do 43 isurf=1,NZSUR(izone)
          icc=izstocn(izone,isurf)
          if(icc.eq.0)then
            write(outs,'(a,i2,a,i3,a)') ' Zone ',izone,' surface ',
     &        isurf,' is an unknown connection, skipping suredgeadj.'
            call edisp(iuout,outs)
            ier=1
            return
          endif

C Clear assumptions.
          nbedgdup(icc)=0

C Confirm whether there are any duplicate references to vertices
c (indicates a polygon with a hole in it). Outer loop takes each
C edge in turn. Inner loop checks from next edge onwards. If a
C duplicate edge is found also remember it.
          do 40 m=1,NVER(isurf)
            jes=JVN(isurf,m)
            if(m.lt.NVER(isurf))then
              jee=JVN(isurf,m+1)
            else
              jee=JVN(isurf,1)
            endif
            im=m+1
            do 41 n=im,NVER(isurf)
              if(n.lt.NVER(isurf))then
                kes=JVN(isurf,n+1)
              else
                kes=JVN(isurf,1)
              endif
              kee=JVN(isurf,n)

              if(jes.EQ.kes.AND.jee.EQ.kee)then
                nbedgdup(icc)=nbedgdup(icc)+2
                iedgdup(icc,m)=1
                iedgdup(icc,n)=1
              endif
  41        continue   ! inner loop vertices in surf
  40      continue     ! outer loop vertices in surf

C Step through the vertices of each edge in turn and check to see
C what other surfaces and materials are associated with edge. j & k
C are the vertices at the ends of the edge being tested. When the
C last vertext is reached point the end to the start point.
          do 46 ijj=1,MV
            iwhich(ijj)=0
            iwhichs(ijj)=0
  46      continue
          list=NVER(isurf)
          nbedgshr(icc)=0     ! reset number of shared edges for this connection
          do 42 i=1,list      ! loop through each vertex
            iedgshr(icc,i)=0
            imatshr(icc,i)=0
            ibridgeshr(icc,i)=0 ! begin by assuming no bridge locations
            iwhich(i)=0
            j=JVN(isurf,i)
            if(i.lt.NVER(isurf))then
              k=JVN(isurf,i+1)
            else
              k=JVN(isurf,1)
            endif
            do 44 iosurf=1,NZSUR(izone)
              foundone=.false.
              icco=izstocn(izone,iosurf)
              if(icco.eq.0) goto 44
              if(iosurf.eq.isurf) goto 44

C For other surface, loop through each edge and check if they match.
C (by definition, they will be in reversed order so test jj against k).
C If an edge matches, test if other surface is facing the same way.
              listo=NVER(iosurf)
              do 45 mm=1,listo
                jj=JVN(iosurf,mm)
                if(mm.lt.NVER(iosurf))then
                  kk=JVN(iosurf,mm+1)
                else
                  kk=JVN(iosurf,1)
                endif
                if(j.eq.0.or.k.eq.0)then
                  continue   ! skip past zero indices
                elseif(j.eq.kk.and.k.eq.jj)then

C Test if azimuth are within two degrees. For the special case of the
C difference between the two being ~360deg call it closeazi
                  call eclose(SPAZI(izone,isurf),
     &                        SPAZI(izone,iosurf),2.0,closeazi)
                  if(.NOT.closeazi)then
                    zazi = abs(SPAZI(izone,isurf)-SPAZI(izone,iosurf))
                    call eclose(zazi,360.0,2.0,closeazi)
                  endif
                  call eclose(SPELV(izone,isurf),
     &                        SPELV(izone,iosurf),2.0,closeelv)
                  if(closeazi.and.closeelv)then

C Point the current edge to the connection of the other surface.
                    iedgshr(icc,i)=icco
                    iwhich(i)=icco

C Check if same material (but ignore UNKNOWN).
                    if(SMLCN(izone,isurf).eq.SMLCN(izone,iosurf))then
                      if(SMLCN(izone,isurf)(1:4).ne.'UNKN')then
                        imatshr(icc,i)=icco
                      endif
                    endif

C Find out if this other surface has already been noticed (so as to
C increment nbedgshr once even if other surface shares several edges).
                    if(.NOT.foundone)then
                      foundone=.true.
                      nbedgshr(icc)=nbedgshr(icc)+1
                    endif
                    goto 44
                  endif
                endif
  45          continue
  44        continue     ! loop surfaces in zone
  42      continue  ! loop vertices in surface

          if(itrc.gt.1)then
            write(outs,*) 'nbedgshr for ',
     &        sname(izone,isurf)(1:lnblnk(sname(izone,isurf))),
     &        ' is ',nbedgshr(icc)
            call edisp(iuout,outs)
          endif

C If most the edges are shared with other similarly facing surfaces
C then if act='c' (check for parent/child) and the call surrel did
C not find it ask the user what to do.
C If there are duplicate surfaces in the zone then the other
C surface might have exactly the same number of edges.
          if(act.eq.'c')then
            if(SPARENT(izone,isurf)(1:2).eq.'  ')then

C Parent attribute has not yet been defined so do some checking.
              if(nbedgshr(icc).eq.NVER(isurf).or.
     &           nbedgshr(icc).eq.(NVER(isurf)-1))then

C If the two surfaces have the same number of edges then it is
C likely that these are duplicate surfaces.
                if(NVER(iosurf).eq.NVER(isurf))then

                  continue
                else
                  KFLAG = -1
                  call SORTI(iwhich,iwhichs,list,KFLAG)

C If all of iwhich is the same number (excluding zero) then use it.
                  same=.true.
                  do 442 im=1,list-1
                    if(iwhich(im).ne.0.and.iwhich(im+1).ne.0)then
                      if(iwhich(im).ne.iwhich(im+1)) same=.false.
                    endif
  442             continue
                  if(same)then
                    ibinval=iwhich(1)  ! << also remember parent index>>
                  else
                    ibinval=0
                  endif

C If current surface already marked and is the same as ibinval then do not
C bother the user. If ibinval was set to zero then clear SPARENT.
                  if(ibinval.ne.0)then
                    if(SPARENT(izone,isurf).eq.
     &                 sname(IC1(ibinval),IE1(ibinval)))then
                      ok=.true.
                    else
                      write(outs,'(5a)') 'Is ',sname(izone,isurf),
     &                  ' a subsurface of ',
     &                  sname(IC1(ibinval),IE1(ibinval)),'?'
                      call easkok(' ',outs,ok,nbhelp)
                    endif
                    if(ok)then
                      SPARENT(izone,isurf)=
     &                  sname(IC1(ibinval),IE1(ibinval))
                    endif
                  else
                    izz=ic1(icc)
                    write(outs,'(5a)') 'Does ',sname(izz,isurf),
     &                ' have a parent surface in ',zname(izz),'?'
                    call easkok(' ',outs,ok,nbhelp)
                    if(ok)then
                      write(outs,'(5a)') 'Specify parent surface for ',
     &                  sname(izz,isurf),' in ',zname(izz),'.'
                      call easksur(izz,iss,'-',outs,' ',ier)
                      ibinval=izstocn(izz,iss)
                      if(ibinval.ne.0)then
                        SPARENT(izone,isurf)=
     &                    sname(IC1(ibinval),IE1(ibinval))  ! also remember parent index
                      else
                        SPARENT(izone,isurf)=' '
                      endif
                    else
                      SPARENT(izone,isurf)=' '
                    endif
                  endif
                endif
              endif
            endif
          endif
 43     continue    ! outer loop of surfaces in the zone
      endif
      return
      end

**************** SURREL ****************************
C This is older logic - to be depreciated in favour of SURREL2.

C Subroutine SURREL checks each surface I in a zone and relates it to
C another surface J in that zone. as parent if one less than all vertices
C in I are also in surface J. One less because the child surface may be
C at the edge of the parent as shown in the example below:
C   xxxxxxxxxx
C   x        x
C   x        x
C   xxxxx    x
C   x   x    x
C   x   x    x
C   xxxxxxxxxx
C Act = '-' a loose definition of child which allows for what is
C   drawn above as well as the case of a door normally inserted
C   which is not at the edge of the parent.
C Act = 's' for a strict definition where a child must be fully
C   bounded by the parent.

C Note: if surface I and J both have the name number of vertices then
C a match should be discounted because the two surfaces are likely to
C be duplicates.

      SUBROUTINE SURREL(ACT,ICOMP,IER)
#include "building.h"
#include "geometry.h"

      integer ncomp,ncon
      COMMON/C1/NCOMP,NCON
      COMMON/C3/IC1(MCON),IE1(MCON),ICT(MCON),IC2(MCON),IE2(MCON)
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

      DIMENSION IDVN(MV),nTbchild(MCON)
      DIMENSION ICOMVER(MS)  ! icomver is the number of vertices in common
      dimension idupv(MS,48)    ! duplicate vertex indices
      dimension iduplist(MS,48) ! position in list
      LOGICAL closeazi,closeelv
      CHARACTER ACT*1,CHILDRIN*12,PARENTIN*12
      integer isur   ! loop counter for parent surface loop
      integer jsur   ! loop counter for child surface loop
      integer iver   ! loop counter for parent surface vertices
      integer jver   ! loop counter for child surface vertices
      integer iiver,jjver,next

C Initialise various variables
      do 10 isur=1,NZSUR(ICOMP)
        icc=IZSTOCN(icomp,isur)
        if(icc.lt.1) goto 10  ! jump if corrupt connection list
        sparent(icomp,isur)='-'
        do i=1,48          ! loop must match dimension statement
          idupv(isur,i)=0     ! clear duplicate vertex index
          iduplist(isur,i)=0  ! clear duplicate position
        enddo  ! of i
        do i=1,MCHILD
          ichild(iCC,i)=0     ! clear array holding children
          igchild(iCC,i)=0    ! clear array holding grand children
        enddo  ! of i
        nbchild(iCC)=0     ! assume no children
        nbgchild(iCC)=0    ! assume no grand children
        iparent(iCC)=0     ! assume no parent
        igparent(iCC)=0     ! assume no grandparent
        next=0  ! counter to update to get to 2nd dimension.

C First identify any surfaces with vertices that are
C referenced more than once (e.g. surface with hole in it)
C Check for 2nd references of a vertex with the fast loop looping
C through only the untested part of the surface edge list.
        do iver=1,nver(isur)
          IIVER=JVN(isur,iver)  ! the vertex to check in slow loop
          if(IIVER.gt.0)then
            do jver=iver,nver(isur)  ! fast loop is diagonal check
              JJVER=JVN(isur,jver)  ! the fast loop vertex
              if(iver.ne.jver)then  ! don't compare with same position in jvn()
                if(JJVER.eq.IIVER)then
                  next=next+1
                  idupv(isur,next)=jjver  ! index of duplicate
                  iduplist(isur,next)=jver  ! duplicate position in list
                endif
              endif
            enddo  ! of jver
          endif
        enddo  ! of iver
  10  continue    ! of isur

      DO 101 ISUR=1,NZSUR(ICOMP)   ! for each possible parent
        DO 102 JSUR=1,NZSUR(ICOMP) ! for each possible child

C Initialise temporary holding array for common vertices.
          ICOMVER(JSUR)=0
          IF(JSUR.EQ.ISUR)GOTO 102   ! skip if same
          DO IVER=1,NVER(ISUR)   ! for each item in parent vert list
            do loop=1,48             ! should match dimension
              if(iver.eq.iduplist(isur,loop))then
                continue  ! this is a known reference
              else
                DO JVER=1,NVER(JSUR) ! for each item in child vert list

C Compare vertex numbers for possible parent and child avoiding any
C 2nd duplicate references (e.g. iver is in iduplist)
                  IIVER=JVN(ISUR,IVER)
                  JJVER=JVN(JSUR,JVER)
                  IF(IIVER.EQ.JJVER)ICOMVER(JSUR)=ICOMVER(JSUR)+1
                enddo  ! JVER
              endif
            enddo   ! loop
          enddo     ! of iver
          if(NVER(JSUR).eq.3)then

C If the number of vertices in jsur is 3 (triangular) then icomver
C must be >= nver(jsur).
            if(ICOMVER(JSUR).GE.NVER(JSUR))then
              continue
            else
              goto 105   ! jump to just before 102
            endif
          elseif(NVER(JSUR).gt.3)then

C If the number of vertices in jsur is 4 or more, icomver must be >= nver()
C for the strict case and >= nver()-1 for the loose or blank action.
C so as to pick up the door-in-corner.
            if(act.eq.'-'.and.(ICOMVER(JSUR).GE.NVER(JSUR)-1))then
              continue
            elseif(act.eq.' '.and.(ICOMVER(JSUR).GE.NVER(JSUR)-1))then
              continue
            elseif(act.eq.'s'.and.(ICOMVER(JSUR).GE.NVER(JSUR)))then
              continue
            else
              goto 105   ! jump to just before 102
            endif

C If isur and jsur both have the same number of edges then it is likely
C that we have duplicate surfaces.
            if(NVER(JSUR).eq.NVER(ISUR))then

              goto 105   ! jump to just before 102
            endif
          endif

C If there are the same shared vertices check orientation as well.
          icc=IZSTOCN(icomp,isur)  ! parent connection
          jcc=IZSTOCN(icomp,jsur)  ! possible child connection

C Parent and child must have the same azimuth and elevation angles,
C allow tolerance of 2deg and possible rotation of 360deg.
C<< Division mod 360 may be a more generic way for angle checking
          call eclose(SPAZI(icomp,isur),SPAZI(icomp,jsur),2.0,closeazi)
          if(.NOT.closeazi)then
            zazi = abs(SPAZI(icomp,isur)-SPAZI(icomp,jsur))
            call eclose(zazi,360.0,2.0,closeazi)
          endif
          call eclose(SPELV(icomp,isur),
     &                SPELV(icomp,jsur),2.0,closeelv)
          if(closeazi.and.closeelv)then

            SPARENT(icomp,jsur)=SNAME(icomp,isur)

C Having found a child add it to the list of children of
C the parent surface (current limit of 20 children).
            if(nbchild(iCC).lt.20)then
              nbchild(iCC)=nbchild(iCC)+1
              ichild(iCC,nbchild(iCC))=jCC ! remember child connection
            endif
            iparent(JCC)=icc   ! remember childs parent connection

          endif
 105      continue    ! jump point in case of dups
 102    CONTINUE
 101  CONTINUE

C Now that parents are known also work out grand parents. This is done
C by a similar logic to how parents are worked out (by inspecting vertex
C numbers of surfaces).
C Only need to work out whether parents are themselves children of some
C other surface (grandparent)
      DO ICC=1,MCON
        nTbchild(iCC)=nbchild(iCC)
      ENDDO  ! of ICC
      DO 110 ISUR=1,NZSUR(ICOMP)
        icc=IZSTOCN(icomp,isur)
        if(icc.lt.1) goto 110  ! jump past if corrupt connection list.
        DO IV=1,MV
          IDVN(IV)=0
        ENDDO  ! of IV
        DO 111 ICHLD=1,nTbchild(iCC)
          NCV=0
          icv=0 ; ido=0
          ichcon=ICHILD(ICC,ICHLD)
          CHILDRIN=SNAME(IC1(ichcon),IE1(ichcon))
          DO 112 JSUR=1,NZSUR(ICOMP)
            jcc=izstocn(icomp,jsur)
            if(jcc.lt.1) goto 112  ! jump past if corrupt connection list
            PARENTIN=SNAME(icomp,jsur)
            ISNL=LNBLNK(PARENTIN)
            ICHL=LNBLNK(CHILDRIN)
            IF(CHILDRIN(1:ICHL).EQ.PARENTIN(1:ISNL))THEN
              DO 113 IVER=1,NVER(ISUR)
                IIVER=JVN(ISUR,IVER)
                ICV=0
                DO JVER=1,NVER(JSUR)
                  JJVER=JVN(JSUR,JVER)
                  IF(IIVER.EQ.JJVER)THEN
                    ICV=1
                  ENDIF
                ENDDO  ! of JVER
                IF(ICV.NE.1)THEN
                  IDO=0
                  DO ICCV=1,MV
                    IF(IDVN(ICCV).EQ.IIVER)THEN
                      IDO=1
                    ENDIF
                  ENDDO  ! of ICCV
                  IF(IDO.EQ.0)THEN
                    NCV=NCV+1
                    IDVN(NCV)=IIVER
                  ENDIF
                ENDIF
 113          CONTINUE
            ENDIF
 112      CONTINUE

C Now IDVN holds list of all vertices that define the parents but are
C not contained in the child. Use these to work out parent of parent
          DO 115 JSUR=1,NZSUR(ICOMP)
            IF(ISUR.NE.JSUR)THEN
              NCOMV=0
              DO 116 ICV=1,NCV
                DO JVER=1,NVER(JSUR)
                  JJVER=JVN(JSUR,JVER)
                  IF(IDVN(ICV).EQ.JJVER)then
                    NCOMV=NCOMV+1
                  ENDIF
                ENDDO  ! of JVER
 116          CONTINUE
              IF(NCV.GT.0.AND.NCV.LE.NCOMV)THEN
                Jcc=IZSTOCN(icomp,Jsur)
                if(nbchild(jCC).lt.23)then ! match with geometry.h
                  SPARENT(icomp,isur)=SNAME(icomp,Jsur)
                  nbchild(jCC)=nbchild(jCC)+1
                  ichild(jCC,nbchild(jCC))=iCC
                endif
              ENDIF
            ENDIF
 115      CONTINUE
 111    CONTINUE
 110  CONTINUE

C Now fill grand children commons
      DO 201 ISUR=1,NZSUR(ICOMP)     ! for every surface in zone
        icc=IZSTOCN(icomp,isur)      ! connection of the surface
        if(icc.lt.1) goto 201
        DO 202 ICHL=1,NBCHILD(ICC)   ! for every child of that surface
          JCC=ICHILD(ICC,ICHL)       ! connection of the child
          DO 203 IGCD=1,NBCHILD(JCC) ! for every child of a child
            KCC=ICHILD(JCC,IGCD)     ! connection for that as well
            IGCHILD(ICC,IGCD)=KCC    ! remember grandchild connection
            NBGCHILD(ICC)=NBGCHILD(ICC)+1  ! increment nb granchildren
            igparent(KCC)=icc   ! remember grandchilds grandparent connection
 203      CONTINUE
 202    CONTINUE
 201  CONTINUE
      RETURN
      END

**************** SURREL2 ****************************
C Subroutine SURREL2 checks via edge lists how each surface I in a
C zone and relates it to another surface J in that zone. It scans
C child surfaces with children to establish grandchild relationships.
C Act = '-' standard verbose reporting.
C Act = 's' for a less verbose reporting.
C Act = 'u' update commons based on existing values of SPARENT
C           (e.g. read from geometry file), instead of scanning for
C           parent-child relationships.

      SUBROUTINE SURREL2(ACT,ICOMP,IER)
#include "building.h"
#include "geometry.h"

      integer iuout,iuin,ieout
      common/OUTIN/IUOUT,IUIN,IEOUT
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

      DIMENSION itv1(MS*MS*MV) ! 1st of every edge in every surface
      DIMENSION itv2(MS*MS*MV) ! 2nd of every edge in every surface
      DIMENSION irelsur(MS*MV) ! which surface assoc with every edge
      DIMENSION ihits(MS,MS)   ! bin number of hits

      LOGICAL closeazi,closeelv
      CHARACTER ACT*1
      CHARACTER outs*496
      integer llposa,lrposa,ulposa,urposa
      integer llposb,lrposb,ulposb,urposb
      real DZLLFF

C Clear arrays.
      ip=0
      DO loop=1,NZSUR(ICOMP)
        icc=IZSTOCN(icomp,loop)
        if(icc.gt.0)then     ! range che check
          nbchild(icc)=0     ! assume no children
          nbgchild(icc)=0    ! assume no grand children
          iparent(icc)=0     ! assume no parent
          igparent(icc)=0    ! assume no grandparent
          do i=1,MCHILD
            ichild(icc,i)=0     ! clear array holding children
            igchild(icc,i)=0    ! clear array holding grand children
          enddo  ! of i
          DO loop2=1,NZSUR(ICOMP)
            ihits(loop,loop2)=0
            ip=ip+1
            itv1(ip)=0; itv2(ip)=0
          ENDDO
        endif
      ENDDO

C Skip finding edge pairs if we are just updating commons.
      if (act(1:1).eq.'u') GOTO 145

C Generate edge pairs in the zone. itv1 & itv2 are the vertices at the
C start and end of each edge, its1 is the reference surface.
      ipair=0; iub=0
      DO IS=1,NZSUR(ICOMP)
        if(NVER(IS).eq.0)then
          continue   ! what about itv2 value for this?
        else
          DO IV=1,NVER(IS)  ! need to trap for NVER zero
            ipair=ipair+1
            itv1(ipair)=JVN(IS,IV)

C Also check the last vertex back to the first.
            if(IV.eq.NVER(IS))then
              itv2(ipair)=JVN(IS,1)
            else
              itv2(ipair)=JVN(IS,IV+1)
            endif
            irelsur(ipair)=IS
          ENDDO  ! of NVER
        endif
      ENDDO      ! of NSUR

C Locate edges mentioned twice (in both directions).
      do 144 ip=1,ipair
        matchab = 0; matchba = 0
        do ipck=1,ipair

C First see if reverse match has been found.
          if(itv1(ip).eq.itv2(ipck).and.itv2(ip).eq.itv1(ipck))then

C Found a match, add a hit to surfaces associated with ip and ipck.
C Exclude hits where loop & loop2 are the same.
            matchba = matchba +1
            loop=irelsur(ip); loop2=irelsur(ipck)
            if(loop.ne.loop2)then

C Parent and child must have the same azimuth and elevation angles,
C allow tolerance of 2deg and possible rotation of 360deg.
              icc=IZSTOCN(icomp,loop)  ! current surf connection
              jcc=IZSTOCN(icomp,loop2) ! comparison surf connection
              call eclose(SPAZI(icomp,loop),
     &                    SPAZI(icomp,loop2),2.0,closeazi)
              if(.NOT.closeazi)then
                zazi = abs(SPAZI(icomp,loop)-SPAZI(icomp,loop2))
                call eclose(zazi,360.0,2.0,closeazi)
              endif
              call eclose(SPELV(icomp,loop),
     &                    SPELV(icomp,loop2),2.0,closeelv)
              if(closeazi.and.closeelv)then
                ihits(loop,loop2)=ihits(loop,loop2)+1  ! increment bucket
              endif
            endif
          elseif(itv1(ip).eq.itv1(ipck).and.itv2(ip).eq.itv2(ipck))then
            matchab = matchab +1
            loop=irelsur(ip); loop2=irelsur(ipck)
            if(loop.ne.loop2)then

C Parent and child must have the same azimuth and elevation angles,
C allow tolerance of 2deg and possible rotation of 360deg.
              icc=IZSTOCN(icomp,loop)  ! current surf connection
              jcc=IZSTOCN(icomp,loop2)  ! possible child connection
              call eclose(SPAZI(icomp,loop),
     &                    SPAZI(icomp,loop2),2.0,closeazi)
              if(.NOT.closeazi)then
                zazi = abs(SPAZI(icomp,loop)-SPAZI(icomp,loop2))
                call eclose(zazi,360.0,2.0,closeazi)
              endif
              call eclose(SPELV(icomp,loop),
     &                    SPELV(icomp,loop2),2.0,closeelv)
              if(closeazi.and.closeelv)then
                ihits(loop,loop2)=ihits(loop,loop2)+1  ! increment bucket
              endif
            endif
          endif
        enddo
  144 continue
  145 continue

C The idea is that zero or small number of hits are unlikely to
C signal a child relationship.
      DO loop=1,NZSUR(ICOMP)

C Updating commons based on SPARENT.
C Skip to next surface if this surface is not a child.
        if (act(1:1).eq.'u') then
          if (SPARENT(icomp,loop)(1:1).eq.'-') CYCLE
        endif

        if(act(1:1).eq.'s')then
          continue
        else
          write(outs,'(a,99i3)')sname(icomp,loop),
     &      (ihits(loop,ii),ii=1,NZSUR(ICOMP))
          call edisp248(iuout,outs,120)
        endif
        DO loop2=1,NZSUR(ICOMP)
          icc=IZSTOCN(icomp,loop)   ! current surf connection
          jcc=IZSTOCN(icomp,loop2)  ! comparison surf connection
          if(loop2.ne.loop)then

C Updating commons based on SPARENT.
            if (act(1:1).eq.'u') then
              l=lnblnk(SPARENT(icomp,loop))
              if (SPARENT(icomp,loop)(1:l).eq.SNAME(icomp,loop2)(1:l))
     &          then
                nbchild(jcc)=nbchild(jcc)+1
                ichild(jcc,nbchild(jcc))=icc
                iparent(icc)=jcc
                EXIT
              endif
              CYCLE
            endif

C Strict definition that surface loop is a child of surface loop2.
            if(ihits(loop,loop2).eq.NVER(loop))then
              if(act(1:1).eq.'s' .or. act(1:1).eq.'u')then
                continue
              else
                write(outs,'(3a)') sname(icomp,loop),' is child of ',
     &            sname(icomp,loop2)
                call edisp248(iuout,outs,120)
              endif
              if(nbchild(jcc).lt.24)then
                nbchild(jcc)=nbchild(jcc)+1
                ichild(jcc,nbchild(jcc))=icc ! remember this
                SPARENT(icomp,loop)=SNAME(icomp,loop2)
                iparent(icc)=jcc   ! remember childs parent connection
              endif

C Looser definition e.g. at a door or glazing with shared edge.
            elseif(ihits(loop,loop2).eq.(NVER(loop)-1))then
              if(act(1:1).eq.'s' .or. act(1:1).eq.'u')then
                continue
              else
                write(outs,'(3a)') sname(icomp,loop),' is? child of ',
     &            sname(icomp,loop2)
                call edisp248(iuout,outs,120)
              endif
              if(nbchild(jcc).lt.24)then
                nbchild(jcc)=nbchild(jcc)+1
                ichild(jcc,nbchild(jcc))=icc ! remember this
                SPARENT(icomp,loop)=SNAME(icomp,loop2)
                iparent(icc)=jcc   ! remember childs parent connection
              endif

            else
              if((NVER(loop).ge.8).and.(ihits(loop,loop2).ge.4))then

C Do a bounding box test of both surfaces. If loop smaller then it
C is probably a frame child.
                CALL ZSURLEHI(icomp,loop,XYMAXa,ZMAXa,llposa,lrposa,
     &            ulposa,urposa,DZLLFF)
                CALL ZSURLEHI(icomp,loop2,XYMAXb,ZMAXa,llposb,lrposb,
     &            ulposb,urposb,DZLLFF)
                if(XYMAXa.lt.XYMAXb)then
                  write(outs,'(4a,2f7.3)') sname(icomp,loop),
     &              ' used as frame is child of ',sname(icomp,loop2),
     &              ' max dimensions ',XYMAXa,XYMAXb
                  call edisp248(iuout,outs,120)
                  if(nbchild(jcc).lt.24)then
                    nbchild(jcc)=nbchild(jcc)+1
                    ichild(jcc,nbchild(jcc))=icc ! remember this
                    SPARENT(icomp,loop)=SNAME(icomp,loop2)
                    iparent(icc)=jcc   ! remember childs parent connection
                  endif
                endif
              endif
            endif
          endif
        ENDDO  ! of loop2
      ENDDO    ! of loop

C Report on what was found.
      if(act(1:1).eq.'s' .or. act(1:1).eq.'u')then
        continue
      else
        call edisp(iuout,' ')
        call edisp(iuout,'Checking parent-child relationships...')
      endif
      DO loop=1,NZSUR(ICOMP)
        icc=IZSTOCN(icomp,loop)
        if(act(1:1).eq.'s' .or. act(1:1).eq.'u')then
          continue
        else
          write(outs,'(2a,4i4)') sname(icomp,loop),
     &      ' nbchild nbgchild iparent igparent',nbchild(icc),
     &      nbgchild(icc),iparent(icc),igparent(icc)
          call edisp248(iuout,outs,120)
        endif
      ENDDO

C Traverse the indicators to discover grandchildren. If a
C child surface has a child surface.
      DO loop=1,NZSUR(ICOMP)
        icc=IZSTOCN(icomp,loop)   ! current surf connection
        if(icc.gt.0)then          ! array range check
          if(nbchild(icc).gt.0)then
            DO loop3=1,NZSUR(ICOMP)     ! find iparent pointing to icc
              jcc=IZSTOCN(icomp,loop3)  ! possible child surf
              if(jcc.gt.0)then          ! array range check
                if(iparent(jcc).eq.icc)then
                  if(act(1:1).eq.'s' .or. act(1:1).eq.'u')then
                    continue
                  else
                    write(outs,'(4a,4i4)') sname(icomp,loop),
     &                ' is parent of ',sname(icomp,loop3),
     &                ' @',loop,icc,loop3,jcc
                    call edisp248(iuout,outs,120)
                  endif
                  if(nbchild(jcc).gt.0)then
                    DO loop4=1,NZSUR(ICOMP)
                      kcc=IZSTOCN(icomp,loop4)  ! possible grand child
                      if(iparent(kcc).eq.jcc)then
                        nbgchild(icc)=nbgchild(icc)+1
                        igchild(ICC,nbgchild(icc))=kcc
                        igparent(kcc)=icc
                        write(outs,'(3a)') sname(icomp,loop),
     &                    ' has grandchild ',sname(icomp,loop4)
                        call edisp248(iuout,outs,120)
                        EXIT
                      endif
                    ENDDO  ! of loop4
                  endif
                endif
              endif
            ENDDO  ! of loop3
          endif
        endif
      ENDDO    ! of loop

C Report on what was found.
      DO loop=1,NZSUR(ICOMP)
        icc=IZSTOCN(icomp,loop)
        if(act(1:1).eq.'s' .or. act(1:1).eq.'u')then
          continue
        else
          write(outs,'(3a,4i4)') 'revised ',sname(icomp,loop),
     &      ' nbchild nbgchild iparent igparent',nbchild(icc),
     &      nbgchild(icc),iparent(icc),igparent(icc)
          call edisp248(iuout,outs,120)
        endif
      ENDDO

      RETURN
      END


C ***************** ZDATA
C ZDATA reads NZONES geometry and uses common blocks G1 G2 etc to fill
C common blocks ZNDATA so wireframes can be drawn without the
C need to re-read geometry files. It is called as the model cfg file
C is initially read as well as prior to topology checks.

      SUBROUTINE ZDATA(ITRC,IER,NZONES)
#include "building.h"
#include "model.h"
#include "geometry.h"
#include "prj3dv.h"
C #include "help.h"

      integer lnblnk  ! function definition

      COMMON/OUTIN/IUOUT,IUIN,IEOUT
      COMMON/FILEP/IFIL
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

C Variables from geometry.h:
C SURCOG is centre of gravity of surface (vertex weighted).
C SUREQN is the surface equation A*X + B*Y + C*Z = D
C SURVN is point 1 unit vectpr off of surface COG along the normal.

      LOGICAL OK,dok
      CHARACTER outs*124

      integer lng     ! length of geometry file name
      integer itrct   ! temporary trace level

C Assuming this routine is called in order: read in the geometry file
C using georead or egomin and pass across into the appropriate array.
      DO 30 ICOMP=1,NZONES
   25   IER=0
        if(itrc.gt.1)then
          lng=lnblnk(LGEOM(ICOMP))
          write(outs,'(2a)') ' Reading : ',LGEOM(ICOMP)(1:lng)
          CALL USRMSG(' ',outs,'-')
        endif
        call georead(IFIL+1,LGEOM(ICOMP),ICOMP,1,iuout,ier)
        IF(IER.NE.0)THEN
          CALL EASKOK('Problem found!','Retry?',OK,nbhelp)
          IF(OK)goto 25
          goto 99
        ENDIF

C Find current zone geometric information and fill common G7 & PREC2
C NOTE: this was done in ersys call so it might not be needed here.
        itrct=0
        call zgupdate(itrct,icomp,ier)
        call zinfo(icomp,zoa,zvol,'q') ! Re-derive values

C Find co-planer surfaces and edges of similar materials.
        call suredgeadj(itrc,'-',icomp,ier)

C Report on derived variables.
        DO 50 J=1, nzsur(ICOMP)
          icc=IZSTOCN(icomp,j)
          if(icc.ne.0)then
            if(ITRC.gt.1)then
              write(outs,*) 'name azi elv m2 ',SNAME(icomp,j),
     &          SPAZI(icomp,j),SPELV(icomp,j),SNA(icomp,j)
              call edisp(iuout,outs)
              write(outs,*) 'SUREQN ',icc,SUREQN(icomp,j,1),
     &          SUREQN(icomp,j,2),
     &          SUREQN(icomp,j,3),SUREQN(icomp,j,4)
              call edisp(iuout,outs)
              write(outs,*) 'SURCOG ',SURCOG(icomp,j,1),
     &          SURCOG(icomp,j,2),SURCOG(icomp,j,3)
              call edisp(iuout,outs)
              write(outs,*) 'SURVN ',SURVN(icomp,j,1),
     &          SURVN(icomp,j,2),SURVN(icomp,j,3)
              call edisp(iuout,outs)
            endif
          endif
   50   CONTINUE
   30 CONTINUE

C Set to check zone bounds.
      INPIC=NZONES
      DO 42 I=1,INPIC
        iZBFLG(I)=0
  42  CONTINUE

      RETURN

   99 IER=1
      RETURN
      END

C ********************** FILSUR
C 'FILSUR' Fills the common block G5 with default assumptions.
C If ISUR=0 do all surfaces otherwise if ISUR>0 for one surface.
C Assumes that zone geometry file was recently scanned.

      SUBROUTINE FILSUR(ICOMP,ISUR)
#include "building.h"
#include "geometry.h"

C Parameters
      integer icomp ! zone number
      integer isur  ! zero do all surfaces otherwise index of a surface

      common/SurfInfo/iSurfOrient(MCOM,MS)
      integer iSurfOrient, iSurfWall, iSurfCeiling, iSurfFloor
      parameter (iSurfWall=1, iSurfFloor=2, iSurfCeiling=3)

C Begin with default assumptions for each surface then overwrite
C this if user supplied information exists.  Gather misc zone infor-
C mation so make a probable guess at locations.
      CALL ZINFO(ICOMP,ZOA,ZVOL,'q')
      if(ISUR.eq.0)then
        DO 44 I=1,NSUR
          SOTF(ICOMP,I)='OPAQUE'
          SMLCN(ICOMP,I)='UNKNOWN'

C Guess at location based on elevation of surface. Use same
C logic as in subroutine updatesvfc.
          SVFC(ICOMP,I)='UNKN'
          if(SPELV(ICOMP,I).GE.-1.5.AND.SPELV(ICOMP,I).LE.1.5)then
            SVFC(ICOMP,I)='VERT'
            IF(I.LE.9)WRITE(SNAME(ICOMP,I),'(A5,I1)') 'Wall-',I
            IF(I.GT.9)WRITE(SNAME(ICOMP,I),'(A5,I2)') 'Wall-',I
            IF(I.GT.99)WRITE(SNAME(ICOMP,I),'(A5,I3)') 'Wall-',I
          elseif(SPELV(ICOMP,I).GE.88.5.AND.SPELV(ICOMP,I).LE.91.5)then
            SVFC(ICOMP,I)='CEIL'
            IF(I.LE.9)WRITE(SNAME(ICOMP,I),'(A4,I1)') 'Top-',I
            IF(I.GT.9)WRITE(SNAME(ICOMP,I),'(A4,I2)') 'Top-',I
            IF(I.GT.99)WRITE(SNAME(ICOMP,I),'(A4,I3)') 'Top-',I
            izsceil(icomp)=I   ! identify as a ceiling
          elseif(SPELV(ICOMP,I).GE.-91.5.AND.
     &           SPELV(ICOMP,I).LE.-88.5)then
            SVFC(ICOMP,I)='FLOR'
            IF(I.LE.9)WRITE(SNAME(ICOMP,I),'(A5,I1)') 'Base-',I
            IF(I.GT.9)WRITE(SNAME(ICOMP,I),'(A5,I2)') 'Base-',I
            IF(I.GT.99)WRITE(SNAME(ICOMP,I),'(A5,I3)') 'Base-',I
            izsfloor(icomp)=I  ! identify as a floor
          else
            SVFC(ICOMP,I)='SLOP'
            IF(I.LE.9)WRITE(SNAME(ICOMP,I),'(A5,I1)') 'Surf-',I
            IF(I.GT.9)WRITE(SNAME(ICOMP,I),'(A5,I2)') 'Surf-',I
            IF(I.GT.99)WRITE(SNAME(ICOMP,I),'(A5,I3)') 'Surf-',I
          endif
          zboundarytype(icomp,i,1)=-1
          zboundarytype(icomp,i,2)=0
          zboundarytype(icomp,i,3)=0
          SUSE(ICOMP,I,1)='-'
          SUSE(ICOMP,I,2)='-'
          SPARENT(ICOMP,I)='-'

C Set flags for wall/celing/floor
          if(SPELV(ICOMP,I).GE.-0.02.AND.SPELV(ICOMP,I).LE.30.0)then
            iSurfOrient(ICOMP,I) = iSurfWall
          elseif(SPELV(ICOMP,I).GT.30.0.AND.SPELV(ICOMP,I).LE.90.02)then
            iSurfOrient(ICOMP,I) = iSurfCeiling
          elseif(SPELV(ICOMP,I).GE.-90.02.AND.
     &           SPELV(ICOMP,I).LE.-89.98)then
            iSurfOrient(ICOMP,I) = iSurfFloor
          endif

   44   CONTINUE
      elseif(ISUR.gt.0)then
        I=ISUR
        SOTF(ICOMP,I)='OPAQUE'
        SMLCN(ICOMP,I)='UNKNOWN'
        SVFC(ICOMP,I)='UNKN'
        if(SPELV(ICOMP,I).GE.-1.5.AND.SPELV(ICOMP,I).LE.1.5)then
          SVFC(ICOMP,I)='VERT'
          IF(I.LE.9)WRITE(SNAME(ICOMP,I),'(A5,I1)') 'Wall-',I
          IF(I.GT.9)WRITE(SNAME(ICOMP,I),'(A5,I2)') 'Wall-',I
          IF(I.GT.99)WRITE(SNAME(ICOMP,I),'(A5,I3)') 'Wall-',I
        elseif(SPELV(ICOMP,I).GE.88.5.AND.SPELV(ICOMP,I).LE.91.5)then
          SVFC(ICOMP,I)='CEIL'
          IF(I.LE.9)WRITE(SNAME(ICOMP,I),'(A4,I1)') 'Top-',I
          IF(I.GT.9)WRITE(SNAME(ICOMP,I),'(A4,I2)') 'Top-',I
          IF(I.GT.99)WRITE(SNAME(ICOMP,I),'(A4,I3)') 'Top-',I
          izsceil(icomp)=I   ! identify as a ceiling
        elseif(SPELV(ICOMP,I).GE.-91.5.AND.SPELV(ICOMP,I).LE.-88.5)then
          SVFC(ICOMP,I)='FLOR'
          IF(I.LE.9)WRITE(SNAME(ICOMP,I),'(A5,I1)') 'Base-',I
          IF(I.GT.9)WRITE(SNAME(ICOMP,I),'(A5,I2)') 'Base-',I
          IF(I.GT.99)WRITE(SNAME(ICOMP,I),'(A5,I3)') 'Base-',I
          izsfloor(icomp)=I  ! identify as a floor
        else
          SVFC(ICOMP,I)='SLOP'
          IF(I.LE.9)WRITE(SNAME(ICOMP,I),'(A5,I1)') 'Surf-',I
          IF(I.GT.9)WRITE(SNAME(ICOMP,I),'(A5,I2)') 'Surf-',I
          IF(I.GT.99)WRITE(SNAME(ICOMP,I),'(A5,I3)') 'Surf-',I
        endif
        SUSE(ICOMP,I,1)='-'
        SUSE(ICOMP,I,2)='-'
        SPARENT(ICOMP,I)='-'
      endif

      RETURN
      END

C ******************** INSINFO
C INSINFO provides an English description of default insolation
C options chosen by the user in the geometry file based on
C NDP(ICOMP) and IDPN(ICOMP,?) from common G4.

C Note: the version of the geometry file should be taken into
C account as the QA report can lead to conflicting information
C as an older geometry file does not have the ishdirec common
C data held in the geometry file but inferred from the obstructions
C file.

C << update documentation as well as reporting based on files >>
      SUBROUTINE INSINFO(ICOMP,ITRU)
#include "building.h"
#include "model.h"
#include "geometry.h"

      common/ishdirec/iaplic(MCOM,2),nsurfcalc(MCOM),lstsfcalc(MCOM,MS),
     &     nsurfinso(MCOM),isurfinso(MCOM,MS)

C Markdown flag.
      logical markdown
      common/markdownflag/markdown

      CHARACTER OUTSTR*124
      character sdescr*80

C Print out information about the default insolation planes if
C the user has not requested a shading analysis.
      call edisp(ITRU,' ')
      if(ISI(ICOMP).eq.1)then
        if(markdown)then
          call edispxtr(ITRU,
     &'An hourly solar radiation distribution is used for this zone.  ')
        else
          call edisp(ITRU,
     &' An hourly solar radiation distribution is used for this zone.')
        endif
      else
        IF(NDP(ICOMP).EQ.1)THEN
          WRITE(OUTSTR,'(3a)')' Solar insolation focused on surface ',
     &    SNAME(ICOMP,IDPN(ICOMP,1)),' (if shading not calculated).'
          call edisp(ITRU,OUTSTR)
        ELSEIF(NDP(ICOMP).EQ.2.AND.IDPN(ICOMP,3).EQ.0)THEN
          WRITE(OUTSTR,'(5a)')' Solar insolation focused on ',
     &    SNAME(ICOMP,IDPN(ICOMP,1)),' & ',SNAME(ICOMP,IDPN(ICOMP,2)),
     &    ' (if shading not calculated).'
          call edisp(ITRU,OUTSTR)
        ELSEIF(NDP(ICOMP).EQ.3)THEN
          if(markdown)then
            write(outstr,'(2a)')
     &      'All surfaces will receive diffuse insolation (if shading',
     &      ' not calculated).'
            call edisp2tr(ITRU,OUTSTR)
          else
            write(outstr,'(2a)')
     &      ' All surfaces will receive diffuse insolation (if shading',
     &      ' not calculated).'
            call edisp(ITRU,OUTSTR)
          endif
        ENDIF
      endif

C If there are existing ish directives print them out. An older
C geometry file will not have this information.
      if(nsurfcalc(icomp).gt.0)then
        ns=nsurfcalc(icomp)
        call surlist(icomp,ns,lstsfcalc,sdescr,length,ierr)
        if(iaplic(icomp,1).eq.0)then
          call edisp(ITRU,
     &      ' Surfaces (user list) for shading analysis: ')
        else
          if(markdown)then
            call edispxtr(ITRU,
     &      'Surfaces (all applicable) for shading analysis  ')
          else
            call edisp(ITRU,
     &      ' Surfaces (all applicable) for shading analysis: ')
          endif
        endif
        if(markdown)then
          call edisp2tr(ITRU,sdescr)
        else
          call edisp(ITRU,sdescr)
        endif
      else
        if(ISI(ICOMP).eq.1)then
          continue   ! obviously requested
        else
          if(markdown)then
            call edispxtr(ITRU,'No shading analysis requested.  ')
          else
            call edisp(ITRU,' No shading analysis requested.')
          endif
        endif
      endif

      if(nsurfinso(icomp).gt.0)then
        ns=nsurfinso(icomp)
        call surlist(icomp,ns,isurfinso,sdescr,length,ierr)
        if(iaplic(icomp,2).eq.0)then
          call edisp(ITRU,' Insolation sources (user list): ')
        else
          if(markdown)then
            call edispxtr(ITRU,'Insolation sources (all applicable) ')
          else
            call edisp(ITRU,' Insolation sources (all applicable):')
          endif
        endif
        if(markdown)then
          call edisp2tr(ITRU,sdescr)
        else
          call edisp(ITRU,sdescr)
        endif
      else
        if(ISI(ICOMP).eq.1)then
          continue    ! obviously requested
        else
          if(markdown)then
            call edispxtr(ITRU,'No insolation analysis requested.  ')
          else
            call edisp(ITRU,' No insolation analysis requested.')
          endif
        endif
      endif

      RETURN
      END

C ********************** VERINFO
C VERINFO prints two summaries, the first includes vertex coordinates
C and a list of surfaces associated with each vertex.  The second
C prints a list of surface numbers and the list of vertices which
C define them.  Information is gathered from common block G1.

      SUBROUTINE VERINFO(ICOMP,ITRU)
#include "building.h"
#include "geometry.h"

C Parameters
      integer icomp  ! zone index
      integer itru   ! unit to write report to
      DIMENSION ISASSO(MS)
      CHARACTER OUTSTR*124,LOUTSTR*420

      call edisp(ITRU,' ')
      call edisp(ITRU,
     &' The zone is defined by the following general polygon vertices:')
      call edisp(ITRU,
     &' Vertex|X-coord.| Y-coord.| Z-coord.|Associated surfaces     |')
      DO 8790 I=1,NZTV(ICOMP)
        IHIT=0
        DO 8792 IS=1,NZSUR(ICOMP)
          DO 8794 IV=1,NVER(IS)
            IF(I.EQ.JVN(IS,IV))THEN
              IHIT=IHIT+1
              ISASSO(IHIT)=IS
            ENDIF
 8794     CONTINUE
 8792   CONTINUE
        if(IHIT.lt.5)then

C If there are less than 5 surfaces write our their names.
          WRITE(OUTSTR,9994)I,X(I),Y(I),Z(I),
     &      (SNAME(icomp,ISASSO(IH)),IH=1,IHIT)
 9994     FORMAT(I5,3F10.4,'  ',5(A,' '))
        else
          WRITE(OUTSTR,9993)I,X(I),Y(I),Z(I),(ISASSO(IH),IH=1,IHIT)
 9993     FORMAT(I5,3F10.4,'  ',10(I3,', '))
        endif
        call edisp(ITRU,OUTSTR)
 8790 CONTINUE

C Surfaces with associated vertices.
      call edisp(ITRU,' ')
      call edisp(ITRU,
     &' Each surface (polygon) is composed of vertices as follows:')
      call edisp(ITRU,
     &' Surface|No.of   | vertex list (anticlockwise viewed from ext)')
      call edisp(ITRU,'        |Vertices|')
      DO 9991 I=1,NZSUR(ICOMP)
        NV=NVER(I)
        if(NV.le.26)then
          WRITE(OUTSTR,9990) SNAME(icomp,I),NV,(JVN(I,J),J=1,NV)
          call edisp(ITRU,OUTSTR)
        elseif(NV.le.100)then
          WRITE(LOUTSTR,9990)SNAME(icomp,I),NV,(JVN(I,J),J=1,NV)
          call edisp248(ITRU,LOUTSTR,100)
        else  ! echo the first 100 edges
          WRITE(LOUTSTR,9990)SNAME(icomp,I),NV,(JVN(I,J),J=1,100)
          call edisp248(ITRU,LOUTSTR,100)
        endif
 9990   FORMAT(A,I5,1x,124(',',I3))
 9991 CONTINUE
      RETURN
      END

C ********************** ZINDEX
C ZINDEX is passed a string (which might be a zone name or an index) and
C returns an index to the matching zone.  If no match then returns 0.
      SUBROUTINE ZINDEX(STRING,index)
#include "building.h"
#include "geometry.h"

      integer lnblnk  ! function definition

      integer ncomp,ncon
      COMMON/C1/NCOMP,NCON
      COMMON/FOPENED/CFGOK,MLDBOK,MATDBOK,CTLOK,OPTKOK,CFCDBOK
      LOGICAL        CFGOK,MLDBOK,MATDBOK,CTLOK,OPTKOK,CFCDBOK
      CHARACTER ZN*12
      character*(*) STRING

C If ALL then return negative of NCOMP.
      if(STRING(1:3).eq.'ALL')then
        index = 0 - NCOMP
        return
      endif

C Loop through each of the zones and see if a match.
      index = 0
      lstr = lnblnk(STRING)
      do 42 i = 1, NCOMP
        ZN = zname(i)
        lzn = lnzname(i)
        if(lstr.eq.lzn)then
          if(STRING(1:lstr).eq.ZN(1:lzn))then
            index = i
            return
          endif
        endif
42    continue

C If reached this point then STRING was not recognisable as a
C zone name.  See if UNKNOWN, if so present a list of zones.
      if(STRING(1:7).eq.'UNKNOWN')then
        IC=-1
 254    CALL EASKGEOF('Select a zone from the list.',CFGOK,IC,'-',
     &    34,IER)
        IF(IC.EQ.0.OR.IC.EQ.-1)then
          call usrmsg('Please select a zone... ',' ','W')
          goto 254
        else
          index = ic
          return
        endif
      else

C See if the string is actually a number.
        read(STRING,*,ERR=20)indext
        if(indext.eq.0)then
          IC=-1
 255      CALL EASKGEOF('Select a zone from list.',CFGOK,IC,'-',34,IER)
          IF(IC.EQ.0.OR.IC.EQ.-1)then
            call usrmsg('Please select a zone... ',' ','W')
            goto 255
          else
            index = ic
            return
          endif
        elseif(indext.le.NCOMP)then
          index = indext
          return
        else
          call USRMSG('index > number of zones...',STRING,'W')
          index = 0
          return
        endif
      endif

 20   CALL USRMSG('Did not understand zone id...',STRING,'W')
      index = 0
      return

      end

C ********************** SNAMDUP
C SNAMDUP is passed a proposed surface name and checks to see that
C it is unique (thus OK = TRUE) if duplicate then OK = FALSE.
C If 'is' is passed as 0 then check all, otherwise exclude surface is.
      SUBROUTINE SNAMDUP(STRING,icomp,is,ok)
#include "building.h"
#include "geometry.h"

      integer lnblnk  ! function definition

      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)
      character string*12,SN*12
      logical ok
      integer lstr  ! length of passed in surface name
      integer icn   ! which connection

      lstr=lnblnk(string)
      ok=.true.
      do 42 i=1,NZSUR(icomp)-1   ! last surface corresponds to new surface for
                                 ! which a connection does not yet exist so don't count it
        icn=izstocn(icomp,i)
        if(icn.ge.1)then
          SN=SNAME(icomp,i)           ! assign local name for other surface
          if(string(1:lstr).eq.SN(1:lnblnk(SN)))then
            if(is.eq.0)then
              ok=.false.
              return
            else
              if(i.ne.is)then
                ok=.false.
                return
              endif
            endif
          endif
        else
          ok=.false.
          return
        endif
  42  continue
      return
      end

C ********************** MATCHSNAME
C matchsname is passed a surface name and zone index and checks which
C surface index in the zone matches the name.
C If 'is' is returned as 0 if no match.
      subroutine matchsname(STRING,icomp,is)
#include "building.h"
#include "geometry.h"

      integer lnblnk  ! function definition

      character string*12,SN*12

      lstr=lnblnk(string)
      is=0
      do 42 i=1,NZSUR(icomp)
        SN=SNAME(icomp,i)
        if(string(1:lstr).eq.SN(1:lnblnk(SN)))then
          is=i
          return
        endif
  42  continue

      return
      end

C ******************** SURLIST ********************
C SURLIST takes an list of surfaces (lstsf) and builds a descriptive
C string (sdescr) to be used in headers.
C icomp is the zone, nbsur is the number of surfaces in the list,
C lstsf is the list of surface indices, sdescr is the string returned
C and length is its maximum length.
      subroutine surlist(icomp,nbsur,lstsf,sdescr,length,ierr)
#include "building.h"
#include "geometry.h"

      integer lnblnk  ! function definition

      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

      dimension lstsf(mcom,ms),iva(ms)
      CHARACTER sdescr*80,sn*12,msg*124
      logical unixok

C Find out overall length and build 1D array iva.
      length=0
      do 42 i=1,nbsur
        icc=izstocn(icomp,lstsf(icomp,i))
        if(icc.eq.0)then
          write(msg,'(a,i2,a,i2,a)') 'Surface list item ',i,
     &      ' in zone ',icomp,' not found.'
          call usrmsg(msg,'Check your model.','W')
          ierr=1
        else
          iva(i)=lstsf(icomp,i)
          lna=lnblnk(SNAME(icomp,lstsf(icomp,i)))
          length=length+lna+1
        endif
  42  continue
      if(length.lt.80)then

C Write out a packed string (space separated) of each surface name.
        sdescr=' '
        ix=1
        ixl=0
        do 43 i=1,nbsur
          ico=izstocn(icomp,lstsf(icomp,i))
          if(ico.ne.0)then
            sn=SNAME(icomp,lstsf(icomp,i))
            lna=max(1,lnblnk(sn))
            if(lna.eq.1)then
              ixl=ix
            else
              ixl=ix+(lna-1)
            endif
            write(sdescr(ix:ixl),'(a)',iostat=ios,err=1)sn(1:lna)
            if(i.lt.nbsur)then
              write(sdescr(ixl+1:ixl+1),'(a)') ' '
              ix=ix+lna+1
            else
              ix=ix+lna+1
            endif
          endif
  43    continue
        return
      else
        if(nbsur.eq.1)then
          ic1=izstocn(icomp,lstsf(icomp,1))
          if(ic1.ne.0)then
            WRITE(SDESCR,'(A)',iostat=ios,err=1)
     &        SNAME(icomp,lstsf(icomp,1))
          else
            WRITE(SDESCR,'(A)',iostat=ios,err=1) 'unknown surf'
          endif
        elseif(nbsur.gt.1)THEN

C Write indices into one packed string sdescr (i.e. truncate).
          ipos=1
          call ailist(ipos,nbsur,iva,MS,'S',sdescr,loutlen,itrunc)
        endif
        return
      endif

C Only give feedback if non-DOS.
   1  call isunix(unixok)
      if(unixok)then
        if(IOS.eq.2)then
          write(6,*) 'SURLIST: permission error writing surf names:',
     &      sdescr
        else
          write(6,*) 'SURLIST: error writing surface names: ',sdescr
        endif
      endif
      ierr=1
      return
      END

C *********** MKVOBJILIST
C This subroutine populates common VOBJILIST (in geometry.h) for zone
C iz. This is a list of indices of associated visual entities for
C each visual object.

C TODO: replace all uses of VOBJLIST with VOBJILIST.

      SUBROUTINE MKVOBJILIST(iz)

#include "building.h"
#include "geometry.h"

      character t12*12

      if (NBVOBJ(iz).le.0) goto 999
      if (NBVIS(iz).le.0) goto 999
      do ivo=1,NBVOBJ(iz)
        if (NBVOBJLIST(iz,ivo).le.0) CYCLE
        do ivoe=1,NBVOBJLIST(iz,ivo)
          t12=VOBJLIST(iz,ivo,ivoe)
          do ive=1,NBVIS(iz)
            if (VISNAME(iz,ive).eq.t12) then
              VOBJILIST(iz,ivo,ivoe)=ive
              EXIT
            endif
          enddo
        enddo
      enddo

  999 RETURN
      END

C ************* decode_zsbound
C Decodes zboundarytype values returning three strings.

      subroutine decode_zsbound(iz,is,sbound_ty,sbound_c2,sbound_e2)

#include "building.h"
#include "geometry.h"
      integer iz,is
      character sbound_ty*12,sbound_c2*6,sbound_e2*6

C Form strings for boundary columns based on zboundarytype.
      if(zboundarytype(iz,is,1).eq.-1)then
        write(sbound_ty,'(a)')'UNKNOWN'
      elseif(zboundarytype(iz,is,1).eq.0)then
        write(sbound_ty,'(a)') 'EXTERIOR'
      elseif(zboundarytype(iz,is,1).eq.1)then
        write(sbound_ty,'(a)') 'SIMILAR'
      elseif(zboundarytype(iz,is,1).eq.2)then
        write(sbound_ty,'(a)') 'CONSTANT'
      elseif(zboundarytype(iz,is,1).eq.3)then
        write(sbound_ty,'(a)') 'ANOTHER'
      elseif(zboundarytype(iz,is,1).eq.4)then
        write(sbound_ty,'(a)') 'GROUND'
      elseif(zboundarytype(iz,is,1).eq.5)then
        write(sbound_ty,'(a)') 'ADIABATIC'
      elseif(zboundarytype(iz,is,1).eq.6)then
        write(sbound_ty,'(a)') 'BASESIMP'
      elseif(zboundarytype(iz,is,1).eq.7)then
        write(sbound_ty,'(a)') 'IDENT_CEN'
      else
        write(sbound_ty,'(a)')'UNKNOWN'
      endif
      if(zboundarytype(iz,is,2).lt.0)then
        write(sbound_c2,'(i3)') zboundarytype(iz,is,2)
      else
        write(sbound_c2,'(i3.3)') zboundarytype(iz,is,2)
      endif
      write(sbound_e2,'(i3.3)') zboundarytype(iz,is,3)
      return
      end


C ************* decode_c3
C Decodes c3 common values returning three [other] strings.

      subroutine decode_c3(iz,is,sbound_ty,sbound_c2,sbound_e2)

#include "building.h"
#include "geometry.h"
      integer iz,is
      character sbound_ty*12,sbound_c2*6,sbound_e2*6

      COMMON/C3/IC1(MCON),IE1(MCON),ICT(MCON),IC2(MCON),IE2(MCON)
      integer IZSTOCN
      COMMON/C24/IZSTOCN(MCOM,MS)

C Form strings for boundary columns based on zboundarytype.
      icc=izstocn(iz,is)
      if(icc.eq.0)then
        write(sbound_ty,'(a)')'UNKNOWN'
        write(sbound_c2,'(a)') '000'
        write(sbound_e2,'(a)') '000'
        return
      endif
      if(ict(icc).eq.-1)then
        write(sbound_ty,'(a)')'UNKNOWN'
      elseif(ict(icc).eq.0)then
        write(sbound_ty,'(a)') 'EXTERIOR'
      elseif(ict(icc).eq.1)then
        write(sbound_ty,'(a)') 'SIMILAR'
      elseif(ict(icc).eq.2)then
        write(sbound_ty,'(a)') 'CONSTANT'
      elseif(ict(icc).eq.3)then
        write(sbound_ty,'(a)') 'ANOTHER'
      elseif(ict(icc).eq.4)then
        write(sbound_ty,'(a)') 'GROUND'
      elseif(ict(icc).eq.5)then
        write(sbound_ty,'(a)') 'ADIABATIC'
      elseif(ict(icc).eq.6)then
        write(sbound_ty,'(a)') 'BASESIMP'
      elseif(ict(icc).eq.7)then
        write(sbound_ty,'(a)') 'IDENT_CEN'
      else
        write(sbound_ty,'(a)')'UNKNOWN'
      endif
      if(IC2(icc).lt.0)then
        write(sbound_c2,'(i3)') IC2(icc)
      else
        write(sbound_c2,'(i3.3)') IC2(icc)
      endif
      write(sbound_e2,'(i3.3)') IE2(icc)
      return
      end


C ******************** scan_zone_name
C Scans zone geometry file (LGEOMF) for zname and zdesc. 

      SUBROUTINE scan_zone_name(IUNIT,LGEOMF,ICOMP,ITRU,IER)
#include "building.h"
C #include "model.h"
#include "geometry.h"
#include "esprdbfile.h"
#include "material.h"
#include "espriou.h"

      integer lnblnk  ! function definition

C Parameters
      integer IUNIT  ! file unit to read
      integer ICOMP  ! zone number
      integer ITRU   ! file unit for feedback
      integer IER    ! zero is ok

      integer iuout,iuin,ieout
      common/OUTIN/IUOUT,IUIN,IEOUT

C ZNAME (12 char) - the zone name (from geometry.h).
C ZDESC (64 char) - zone notes (from geometry.h).
      CHARACTER LGEOMF*72,WORD*32
      character ZN*12,phrase*64,outs*124,outs2*124
      character loutstr*248

C Set initial values.
      IER=0
      iflag=0
      ZN=' '
      phrase=' '

C Initialise geometry data file. and set currentfile.
      CALL EFOPSEQ(IUNIT,LGEOMF,1,IER)
      IF(IER.LT.0)THEN
        write(outs,'(3a)') 'Geometry file ',LGEOMF(1:lnblnk(LGEOMF)),
     &      ' could not be opened.'
        call edisp(itru,outs)
        IER=1
        RETURN
      endif
      write(currentfile,'(a)') LGEOMF(1:lnblnk(LGEOMF))

C Check that the zone index is within complexity limits.
      if(icomp.gt.MCOM)then
        write(outs,'(a)') 'scan_zone_name: zone index beyond range...'
        lsn=MIN0(lnblnk(currentfile),110)
        write(outs2,'(2a)') 'in: ',currentfile(1:lsn)
        call edisp(iuout,outs)
        call edisp(iuout,outs2)
        IER=3
        CALL ERPFREE(IUNIT,ios)
        RETURN
      endif

C Read header lines from file, the 1.1 version looks like:
C *Geometry 1.1,GEN,manager  # tag version, format, zone name (tbd allow spaces)
C An older file header looks like:
C # geometry of manager defined in: ../zones/manager.geo
C GEN  manager  manager describes cellular office  # type, name, descr
C      22      10   0.000    # vertices, surfaces, rotation angle
      CALL LSTRIPC(IUNIT,LOUTSTR,99,ND,1,'geo line 1',IER)
      IF(IER.NE.0)goto 1002
      if(LOUTSTR(1:13).eq.'*Geometry 1.1')then

C Decode first line of version 1.1 geometry file.
        K=13
        CALL EGETW(LOUTSTR,K,WORD,'W','CTYPE',IFLAG)
        write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
        zname(ICOMP)=' '
        CALL EGETW(LOUTSTR,K,WORD,'W','Z name',IFLAG)
        ZN=WORD(1:12)
        call st2name(ZN,zname(ICOMP))
        lnzname(ICOMP)=lnblnk(zname(ICOMP))  ! update string length
        gversion(icomp) = 1.1   ! set the version number.
        return
      else

C Check if it matches the syntax of older geometry file format.
        K=0
        IF(ND.EQ.1)THEN
          CALL EGETW(LOUTSTR,K,WORD,'W','CTYPE',IFLAG)
          write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
          zname(ICOMP)=' '
          lnzname(ICOMP)=0
          zdesc(ICOMP)=' '
        ELSEIF(ND.EQ.2)THEN
          CALL EGETW(LOUTSTR,K,WORD,'W','CTYPE',IFLAG)
          write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
          CALL EGETW(LOUTSTR,K,WORD,'W','Z name',IFLAG)
          ZN=WORD(1:12)
          call st2name(ZN,zname(ICOMP))
          lnzname(ICOMP)=lnblnk(zname(ICOMP))  ! update string length
          zdesc(ICOMP)=' '
        ELSEIF(ND.gt.2)THEN
          CALL EGETW(LOUTSTR,K,WORD,'W','CTYPE',IFLAG)
          write(CTYPE(icomp),'(a)') WORD(1:lnblnk(WORD))
          CALL EGETW(LOUTSTR,K,WORD,'W','Z name',IFLAG)
          ZN=WORD(1:12)
          call st2name(ZN,zname(ICOMP))
          call egetrm(loutstr,K,phrase,'W','Z description',IER)
          ZDESC(ICOMP)=phrase
        ENDIF
        zoneLabel(icomp) = WORD(1:32)
        return
      endif

C Errors for LOUTSTR reads.
 1002 write(outs,'(3a)') 'scan_zone_name: conversion error in...',
     &  LOUTSTR(1:50),'...'
      lsn=MIN0(lnblnk(currentfile),110)
      write(outs2,'(2a)') 'in: ',currentfile(1:lsn)
      call edisp(iuout,outs)
      call edisp(iuout,outs2)
      IER=1
      CALL ERPFREE(IUNIT,ios)
      RETURN
      end
