#include	"theory/Radiation_fd_1.h"

#include	"dimensions/Energy.h"
#include	"dimensions/SpecificHeatCapacity.h"
#include	"dimensions/ThermalResistance.h"
#include	"transport/Equation_set.h"
#include	"transport/Equation.h"
#include	"transport/State_variable.h"
#include	"building/Room_es.h"
#include	"building/Surface_es.h"

Type*	Radiation_fd_1_Type_pointer;
static	char*	class_name = "the_radiation_fd_1";


Radiation_fd_1::Radiation_fd_1(Metaclass* meta, Radiation_fd_1_def* def)
					: (meta, (Radiation_def*)def)
{
	DEBUG << "Radiation_fd_1::Radiation_fd_1(Metaclass* meta, Radiation_fd_1_def* def)\n";

	name(class_name);
	directType(TYPE_OF(Radiation_fd_1));
}

Radiation_fd_1::Radiation_fd_1(Radiation_fd_1& c) : ((Radiation&)c)
{
	DEBUG << "Radiation_fd_1::Radiation_fd_1(Radiation_fd_1& c)\n";

	name(class_name);
	directType(TYPE_OF(Radiation_fd_1));
}

Radiation_fd_1::Radiation_fd_1(APL* theAPL) : Radiation(theAPL)
{};

Radiation_fd_1::~Radiation_fd_1()
{
	DEBUG << "Radiation_fd_1::~Radiation_fd_1()\n";
}


float	Radiation_fd_1::heat_transfer_coeff(Surface*    surf_1,
                                            Surface*    surf_2,
                                            ViewFactor  view_factor)
{
	DEBUG1	<< "float Radiation_fd_1::heat_transfer_coef(Surface* surf_1, Surface* surf_2, ViewFactor view_factor)"\n;

        // This function returns the radiative heat transfer coefficient
        // which is recommented by CIBSE or ASHRAE.
        return 6.5;
};



Radiation_fd_1& Radiation_fd_1::operator=(Radiation_fd_1& c)
{
	DEBUG1 << "Radiation_fd_1& Radiation_fd_1::operator=(Radiation_fd_1& c)\n";

	return *this;
}


void    Radiation_fd_1::add_coupling(Equation_iterator  layer1_eqn_iter,
				     State_variable*  node_in_layer1,
				     Equation_iterator  layer2_eqn_iter,
				     State_variable*  node_in_layer2,
				     EKSObject*  data_supplier)
{
        DEBUG1 << "void Radiation_fd_1::add_coupling(Equation_iterator  layer1_eqn_iter,\n";
	DEBUG1 << "	  State_variable*  node_in_layer1, Equation_iterator  layer2_eqn_iter,\n";
	DEBUG1 << "	  State_variable*  node_in_layer2, EKSObject*  data_supplier)\n";

/*==============================================================================
Inputs:
        layer1_eqn_iter	- Equation_set for surface 1.
        layer2_eqn_iter	- Equation_set for surface 2.
	data_supplier		- Surface (gives area & inclination)
What the function does:
        For each surface, do the following:
        1. calculate the new coefficients.
        2. find the self coupling coefficient and get the value.
        3. modify the self coupling coefficient. c=c+new_value.
        4. create a cross coupling coefficient and add to the list in Equation.
Note:
        same as in Convection_fd.C.
==============================================================================*/

					// Get the equations to be coupled.
	Equation*	eqn1;
	layer1_eqn_iter(RESET);
	while (eqn1 = layer1_eqn_iter(FORWARD))
		if (node_in_layer1 == eqn1->state_variable())
			break;
	if (eqn1 == NULLPTR(Equation)) {
		cerr << "Radiation::add_coupling(): eqn for surface sv not found";
		exit(1);
	};

	Equation*	eqn2;
	layer2_eqn_iter(RESET);
	while (eqn2 = layer2_eqn_iter(FORWARD))
		if (node_in_layer2 == eqn2->state_variable())
			break;
	if (eqn2 == NULLPTR(Equation)) {
		cerr << "Radiation::add_coupling(): eqn for surface sv not found";
		exit(1);
	};


        SpecificHeatCapacity	surface1_capacity = eqn1->capacity();
        SpecificHeatCapacity	surface2_capacity = eqn2->capacity();
        ThermalResistance	surface1_resistance = eqn1->resistance();
        ThermalResistance	surface2_resistance = eqn2->resistance();

				// calculate heat transfer coeffs
	Surface_es*	s1 = (Surface_es*)node_in_layer1->owner();
	Surface_es*	s2 = (Surface_es*)node_in_layer2->owner();

	ViewFactor    vf1_2 = ((Room_es*)data_supplier)->viewFactor(s1, s2);
	float	htc1_2 = heat_transfer_coeff(s1, s2, vf1_2);
	ViewFactor    vf2_1 = ((Room_es*)data_supplier)->viewFactor(s1, s2);
	float	htc2_1 = heat_transfer_coeff(s2, s1, vf2_1);
		        	// For first surface.
        float	new_coeff = (1 / (surface1_resistance/2 + 1/htc1_2)
			       / surface1_capacity)
						.value();	// ditch type info
        float	old_coeff = eqn1->coefficient_value(node_in_layer1);
        float	modified_coeff = old_coeff - new_coeff;
        eqn1->change_coefficient(node_in_layer1, modified_coeff);
        eqn1->append_coefficient(node_in_layer2, new_coeff, TEMPERATURE);

        // For second surface.
        new_coeff = (1 / (surface2_resistance/2 + 1/htc2_1) / surface2_capacity)
						.value();	// ditch type info
        old_coeff = eqn2->coefficient_value(node_in_layer2);
        modified_coeff = old_coeff - new_coeff;
        eqn2->change_coefficient(node_in_layer2, modified_coeff);
        eqn2->append_coefficient(node_in_layer1, new_coeff, TEMPERATURE);
};


void    Radiation_fd_1::inject_energy(Equation_iterator  layer_eqn_iter,
				      State_variable*  node_in_layer,
				      Energy  gain)
{
	DEBUG1	<< "void   Radiation_fd_1::inject_energy(Equation_iterator  layer_eqn_iter,"\n;
	DEBUG1	<< "		     State_variable*  node_in_layer,"\n;
	DEBUG1	<< "		     Energy  gain)"\n;

					// Get the equation to be modified.
	Equation*	eqn;

	layer_eqn_iter(RESET);
	while (eqn = layer_eqn_iter(FORWARD))
		if (node_in_layer == eqn->state_variable())
			break;
	if (eqn == NULLPTR(Equation)) {
		cerr << "Radiation::add_coupling(): eqn for surface sv not found";
		exit(1);
	};

        eqn->add_gain(gain);
};
