#include	"theory/Conduction_fd.h"


#include	"dimensions/Density.h"
#include	"dimensions/ThermalConductivity.h"
#include	"transport/Equation_set.h"
#include	"building/Layer_es.h"
//#include	"theory/Conduction_fd_def.h"

Type*	Conduction_fd_Type_pointer;
static	char*	class_name = "the_conduction(fd)";


Conduction_fd::Conduction_fd(Metaclass* meta, Conduction_fd_def* def)
				: Conduction(meta, (Conduction_def*) def)
{
	DEBUG	<< "Conduction_fd::Conduction_fd(" << meta->oid() << ", "
	DEBUG	<< def->oid() << ")\n";

	name(class_name);
	directType(TYPE_OF(Conduction_fd));
};

Conduction_fd::Conduction_fd(Conduction_fd& c) : ((Conduction&)c)
{
	DEBUG << "Conduction_fd::Conduction_fd(" << c->oid() << ")\n";

	name(class_name);
	directType(TYPE_OF(Conduction_fd));
};

Conduction_fd::Conduction_fd(APL* theAPL) : Conduction(theAPL)
{};

Conduction_fd::~Conduction_fd()
{
	DEBUG << "Conduction_fd::~Conduction_fd()\n";
};

Conduction_fd& Conduction_fd::operator=(Conduction_fd& c)
{
	DEBUG1 << "Conduction& Conduction::operator=(" << c->oid() << ")\n";

	return *this;
};


/*
** One dimensional conduction heat flux equation with
** a finite difference approximation applied to the space
** derivative.
*/
Equation_set	Conduction_fd::generate_equations(State_vector* state_vector,
						  EKSObject*  layer,
						  Equation_iterator  layer_eqn_iter)
{
	DEBUG1 << "Equation_set*   generate_equations(State_vector* state_vector,\n";
	DEBUG1 << layer->oid() << ", Equation_iterator  layer_eqn_iter)\n";


// Create an equation set for generated equations
	Equation_set	eqn_set;

// Get the required data from layer.
	Layer_es*	lyr = (Layer_es*)layer; // cast guaranteed by Metaclass
	Length 			x_dimension = lyr->thickness().value();
	ThermalConductivity	cd = lyr->substance().thermalConductivity();
	Density 		lo = lyr->substance().density();
	SpecificHeatCapacity 	cp = lyr->substance().specificHeatCapacity();
	int 			noDivision = lyr->no_of_divisions();

// Initialise variables.
	int 		nnCoef = 1;
  	if(noDivision > 1)   nnCoef = noDivision*3-2;

	Length* 		dx = new Length[noDivision];
	ThermalResistance* 	resist = new ThermalResistance[noDivision];
	SpecificHeatCapacity* 	capacity = new SpecificHeatCapacity[noDivision];

 	for(int i=0; i<noDivision; i++) {
		dx[i] = x_dimension/noDivision;
		resist[i] = dx[i]/cd;
		capacity[i] = lo*cp*dx[i];
 	};

				// generate 1st equations - may be only eqn
	float	self_coupling_coeff = 0;
	Energy	gain(0);
	if (noDivision > 1)
		self_coupling_coeff = (-2/capacity[0]/(resist[0]+resist[1])).value();
	Equation*	equ = new Equation((*state_vector)[0], ENERGY,
					   self_coupling_coeff, gain,
					   resist[0], capacity[0]);
	

	if (noDivision > 1) {
				// forward couple to succeeding layer
	    float coupling_coeff = -self_coupling_coeff;
	    equ->append_coefficient((*state_vector)[1], coupling_coeff, TEMPERATURE);
	    eqn_set.append(equ);
				// generate internal equations
	    for (int nd = 1; nd < noDivision-1; nd++) {
		self_coupling_coeff = (-2/capacity[nd]*(1/(resist[nd-1]+resist[nd])+
					1/(resist[nd]+resist[nd+1]))).value();
		equ = new Equation((*state_vector)[nd], ENERGY, self_coupling_coeff, 0,
				   resist[nd], capacity[nd]);
					// back couple to preceeding layer
		coupling_coeff = (2/capacity[nd]/(resist[nd-1]+resist[nd])).value();
		equ->append_coefficient((*state_vector)[nd-1], coupling_coeff, TEMPERATURE);
					// forward couple to succeeding layer
		coupling_coeff = (2/capacity[nd]/(resist[nd]+resist[nd+1])).value();
		equ->append_coefficient((*state_vector)[nd+1], coupling_coeff, TEMPERATURE);
		eqn_set.append(equ);

	    };
					// generate last equation
	    self_coupling_coeff = (-2/capacity[noDivision-1].value()/(resist[noDivision-1]+resist[noDivision-2])).value();
	    equ = new Equation((*state_vector)[noDivision-1], ENERGY, self_coupling_coeff, 0,
			   resist[noDivision-1], capacity[noDivision-1]);
					// back couple to preceeding layer
	    coupling_coeff = -self_coupling_coeff;
	    equ->append_coefficient((*state_vector)[noDivision-2], coupling_coeff, TEMPERATURE);
	};
	eqn_set.append(equ);

					// delete old equations
	if (layer_eqn_iter.size() != 0) {
		layer_eqn_iter(RESET);
		while (equ = layer_eqn_iter(FORWARD))
			delete equ;
	};
					// delete temporaries
	delete dx, resist, capacity;

	return eqn_set;
};

Equation_set Conduction_fd::combine_equations(Equation_iterator  layer1_eqn_iter,
					       State_variable* layer1_surface_node,
					       Equation_iterator  layer2_eqn_iter,
					       State_variable* layer2_surface_node,
					       EKSObject* data_supplier)
{
	
	DEBUG1 << "Equation_set* Conduction_fd::combine_equations(Equation_iterator  layer1_eqn_iter,\n";
	DEBUG1 << "	  State_variable*  layer1_surface_node, Equation_iterator  layer2_eqn_iter,\n";
	DEBUG1 << "	  State_variable*  layer2_surface_node, EKSObject* data_supplier)\n";

	Equation*	eqn1;
					// probably last eqn of layer 1
	if (layer1_surface_node == layer1_eqn_iter.last()->state_variable())
		eqn1 = layer1_eqn_iter.last();
	else {
					// no, so search
		layer1_eqn_iter(RESET);
		while (eqn1 = layer1_eqn_iter(FORWARD))
			if (layer1_surface_node == eqn1->state_variable())
				break;
		if (eqn1 = NULLPTR(Equation)) {
			cerr << "Conduction::combine equations(): eqn for sv1 not found";
			exit(1);
		};
	};

	Equation*	eqn2;
					// probably 1st eqn of layer 2
	if (layer2_surface_node == layer2_eqn_iter[0]->state_variable())
		eqn2 = layer2_eqn_iter[0];
	else {
					// no, so set up iterator and search
		layer2_eqn_iter(RESET);
		while (eqn2 = layer2_eqn_iter(FORWARD))
			if (layer2_surface_node == eqn2->state_variable())
				break;
		if (eqn2 = NULLPTR(Equation)) {
			cerr << "Conduction::combine_equations(): eqn for sv2 not found";
			exit(1);
		};
	};

					// update self coupling coefficients
	float coeff1 = (- 8 / eqn1->capacity() / (eqn1->resistance() +
						        eqn2->resistance()))
			.value();	// ditch type info
	eqn1->change_coefficient(eqn1->state_variable(), coeff1);
	float coeff2 = (- 8 / eqn2->capacity() / (eqn2->resistance() +
						       eqn1->resistance()))
			.value();	// ditch type info
	eqn2->change_coefficient(eqn2->state_variable(), coeff2);

					// add new coupling coefficients
	eqn1->append_coefficient(eqn2->state_variable(), -coeff1, TEMPERATURE);
	eqn2->append_coefficient(eqn1->state_variable(), -coeff2, TEMPERATURE);

					// concatenate equation sets and return
	Equation_set	eqn_set;
	eqn_set = layer1_eqn_iter;
	eqn_set += layer2_eqn_iter;

	return eqn_set;
};
	
void	Conduction_fd::inject_energy(Equation_iterator  layer_eqn_iter,
				     State_variable*  node_in_layer,
				     Energy  gain)
{
	DEBUG1	<< "void   inject_energy(Equation_iterator  layer_eqn_iter, State_variable*  node_in_layer,"\n;
	DEBUG1	<< "		     Energy  gain)"\n;

	Equation*	eqn;
					// probably 1st eqn of layer
	if (node_in_layer == layer_eqn_iter[0]->state_variable())
		eqn = layer_eqn_iter[0];
					// ... or may be last eqn
	else if (node_in_layer == layer_eqn_iter.last()->state_variable())
		eqn = layer_eqn_iter.last();

	else {
					// ... no, so set up iterator and search
		layer_eqn_iter(RESET);
		while (eqn = layer_eqn_iter(FORWARD))
			if (node_in_layer == eqn->state_variable())
				break;
		if (eqn == NULLPTR(Equation)) {
			cerr << "Conduction::inject_energy(): eqn for sv not found";
			exit(1);
		};
	};

	eqn->add_gain(gain);
};
