#include	"site/Sun_basic.h"

#include	<string.h>
#include	<math.h>
#include	"transport/Sun_position.h"
#include	"transport/Location.h"
#include	"transport/Time_of_day.h"

Type*	Sun_basic_Type_pointer;
static	char*	class_name = "the_sun";


Sun_basic::Sun_basic(Metaclass* meta, Sun_basic_def* def)
				: Sun(meta, (Sun_def*)def)
{
	DEBUG	<< "Sun_basic::Sun_basic(" << meta->oid() << ", " << def->oid() << ")\n";

	name(class_name);			// assume only 1 sun object
	directType(TYPE_OF(Sun_basic));
};

Sun_basic::Sun_basic(Sun_basic& s)  : Sun((Sun&)s)
{
	DEBUG << "Sun_basic::Sun_basic(" << s->oid() << ")\n";

					// NB user will have to set new name
	directType(TYPE_OF(Sun_basic));
};

Sun_basic::Sun_basic(APL* theAPL) : Sun(theAPL)
{};


/* The following function calculates the local azimuth and
** elevation of the sun at a specific location and time using
** an approximation to equations used to generate tables in The
** Astronomical Almanac.  Refraction correction is added so sun
** position is the apparent one.  Source:
** The Astronomical Almanac's Algorithm for Approximate Solar
** Position (1950-2050) by Michalsky J J, Solar Energy V40(1),
** pp227-35, 1988.  Errata: Solar Energy V41 (1), p113, 1988.

** The code has been taken from a report from the PASSYS Project
** Model Development and Validation Subgroup - New Routine for the
** ESP - system concerning the calculation of the solar position
** by C.E.E. Pernot.
*/
Sun_position Sun_basic::position(Location& a_location, Time_of_day& a_time_of_day)
{
	DEBUG1 << "Sun_basic::position(" << a_location << ", " << a_time_of_day << ")\n";

	int leap, delta;
	double jd; /* Julian date. */

	/* Variables used in the calculation of the ecliptic
	   coordinates. */
	double mnlong, mnanom, eclong, oblqec;

	/* Variables used to calculate the right ascension and
	   declination. */
	double num, den, ra, dec, sdec, cdec;

	double eqt; /* Equation of time (hours). */
	double ha;  /* Hour angle. */

	/* Sin and cos of latitude. */
	double aztmp, slat, clat;
	double xx, yy, rad;

	/* Refraction correction for US standard atmosphere. */
	double refrac;
	double azimuth, elevation;
	double solartime;


	double latitude = (double)a_location.latitude();
	double longitude = (double)a_location.longitude();
	int year = a_time_of_day.year();
	int day = a_time_of_day.day();
	float sec=a_time_of_day.sec();
	float min=(float)a_time_of_day.min();
	float hr=(float)a_time_of_day.hour();
	double hour=(double)(hr+(min+sec/60)/60);
	
	rad=PI/180;

	/* Get the current Julian date (actually add 2,400,000
	   for jd). */
	delta=year-1949;

	/* In fortran the following was leap=dint(delta/4.). */
	leap = (delta)/4; //Integer division.

	jd = 32916.5 + delta*365 + leap + day + hour/24;
 
	/* 1st no. is mid. 0 jan 1949 minus 2.4e6; leap=leap days
	   since 1949. */

	/* Calculate ecliptic coordinates. */
	jd=jd - 51545.0;
	/* 51545.0 + 2.4e6 = noon 1 jan 2000 */

	/* Force mean longitude between 0 and 360 degs. */
	mnlong = 280.460 + 0.9856474 * jd;

	/* Mean anomaly in radians between 0 and 2*pi. */
	mnanom = 357.528 + 0.9856003 * jd;
	mnanom = mnanom * rad;

	/* Compute elliptic longitude and obliquity of ecliptic in
	   radians. */
	eclong = mnlong + 1.915*sin(mnanom) + 0.020*sin(2.*mnanom);
	oblqec = 23.429 - 0.0000004 * jd;
	eclong = eclong * rad;
	oblqec = oblqec * rad;

	/* Calculate right ascension and declination. */
	num = cos(oblqec) * sin(eclong);
	den = cos(eclong);
	ra = atan(num/den);

	/* dec in radians. */
	dec = asin(sin(oblqec) * sin(eclong));
	cdec = cos(dec);
	sdec = sin(dec);

	/* Calculate equation of time (hours). */
	/* Force eqt to be less than 12. */
	eqt = (mnlong - ra/rad)/15.0;
	/*** The following in fortran was:
	     eqt=eqt-idnint(eqt/12.)*12 ***/
	eqt = eqt - (int(eqt/12.0))*12.0;

	/* Calculate solar time. */
	solartime = hour + eqt + (longitude/15.0);
	if(solartime > 24.0) solartime = solartime - 24.0;

	/* Calculate hour angle in radians between -pi and pi. */
	ha = 15.0 * rad * (12.0 - solartime);

	/* Change latitude to radians. */
	latitude = latitude * rad;
	slat = sin(latitude);
	clat = cos(latitude);

	/* Calculate elevation and azimuth. */
	elevation = asin(sdec * slat + cdec * clat * cos(ha));

	if(elevation >= 0)
	/* Elevation is greater than 0. */
	{
		aztmp = cdec * sin(fabs(ha))/cos(elevation);
		if(aztmp < -1.0) aztmp = -1.0;
		if(aztmp > 1.0) aztmp = 1.0;
		azimuth = asin(aztmp)/rad;

		/* correct the azimuthal angle for time of day and
		   hemisphere */
		xx = cos(ha);

		if(latitude == 0)
			yy= 10 * (sdec/cdec);
		else if((latitude/rad) == 90)
			yy = 0;
		     else yy = (clat/slat)*(sdec/cdec);

		if((yy-xx) < 0) {
			if(latitude >= 0) {
				if(solartime<=12) azimuth=180-azimuth;
				if(solartime>12) azimuth=180+azimuth;
			}
			else if(solartime>12) azimuth=360-azimuth;
		}
		if((yy-xx) == 0) {
			if(solartime<=12) azimuth=90;
			if(solartime>12) azimuth=270;
		}
		if((yy-xx) > 0) {
			if(latitude < 0) {
				if(solartime<=12) azimuth=180-azimuth;
				if(solartime>12) azimuth=180+azimuth;
			}
			else if(solartime>12) azimuth=360-azimuth;
		}

		/* Calculate refraction correction for US standard
		   atmosphere.  Need to have el in degs before
		   calculating the correction. */
		elevation = elevation/rad;
		if(elevation > -0.56) refrac=3.51561*
			(0.1594+0.0196*elevation
			+0.00002*elevation*elevation)
			/(1.+0.505*elevation+
		    	0.0845*elevation*elevation);
		else refrac = 0.56;

		/* Note that 3.51561 = 1013.2 mb/288.2 K. */
		elevation = elevation + refrac;
		/* Elevation in degrees. */

		/* Convert lat back to degs before returning. */
		latitude = latitude/rad;

		/* mnlong, latitude and az in degs, jd in days
		   if 2.4e6 added; mnanom, eclong, oblqec, ra,
		   dec, lmst and ha in radians */

	} /* End of elevation greater than 0. */
	else azimuth = 0; /* Give a value to solar azimuth. */

	// Return azimuth, elevation and declination.
	return Sun_position(azimuth, elevation, dec);
};
