#include	"site/Berdahl_and_Martin.h"

#include	<math.h>
#include	"transport/ClimateRecord.h"
#include	"transport/Sun_position.h"
#include	"transport/Time_of_day.h"
#include	"dimensions/Temperature.h"

Type*	Berdahl_and_Martin_Type_pointer;
static	char*	class_name = "the_sky_temperature(Berdahl_and_Martin)";


Berdahl_and_Martin::Berdahl_and_Martin(Metaclass* meta, Berdahl_and_Martin_def* def)
				: Sky_temperature(meta, (Sky_temperature_def*)def)
{
	DEBUG << "Berdahl_and_Martin::Berdahl_and_Martin()\n";

	name(class_name);			// assume only 1 sky temp
	directType(TYPE_OF(Berdahl_and_Martin));

	/* Set initial cloud cover = 0.4. */
	CLOUDC=0.4;
	CLOUD1=0.4;
	CLOUD2=0.4;
	CLOUD3=0.4;
}

Berdahl_and_Martin::Berdahl_and_Martin(Berdahl_and_Martin& s)
					: Sky_temperature((Sky_temperature&)s)
{
	DEBUG << "Berdahl_and_Martin::Berdahl_and_Martin(Berdahl_and_Martin& s)\n";

						// NB user will have to set name
	directType(TYPE_OF(Berdahl_and_Martin));

	/* Set initial cloud cover = 0.4. */
	CLOUDC=0.4;
	CLOUD1=0.4;
	CLOUD2=0.4;
	CLOUD3=0.4;
}

Berdahl_and_Martin::Berdahl_and_Martin(APL* theAPL) : Sky_temperature(theAPL)
{};


Berdahl_and_Martin& Berdahl_and_Martin::operator=(Berdahl_and_Martin& s)
{
	DEBUG1 << "Berdahl_and_Martin::operator=(Berdahl_and_Martin&)\n";
	if(this==&s) return *this;

	/* Copy the Sky_temperature part. */
	(void) Sky_temperature::operator=((Sky_temperature&)s);
	/* Copy the Berdahl_and_Martin part (should be a member per member
	   copy, not an re-initialization). */
	CLOUDC=s.CLOUDC;
	CLOUD1=s.CLOUD1;
	CLOUD2=s.CLOUD2;
	CLOUD3=s.CLOUD3;
	return *this;
}

/*
** Determines the Sky temperature after Berdahl and Martin.
*/
Temperature Berdahl_and_Martin::execute(ClimateRecord& a_clm_rec,
		Sun_position& a_sun_pos,
		Time_of_day& a_time_of_day)
{
	DEBUG1 << "Berdahl_and_Martin::execute(ClimateRecord,\n";
	DEBUG1 << "               Sun_position&, Time_of_day&)\n";

	int IHRF=a_time_of_day.hour();
	float TF=a_clm_rec.dryBulb().value();
	Temperature rel_humidity = (a_clm_rec.dryBulb()-a_clm_rec.wetBulb())
					/ a_clm_rec.dryBulb();
////???	float HF=a_clm_rec.relative_humidity();
	float HF=rel_humidity.value();
	float QFF=a_clm_rec.diffuseHorizontalSolar().value();
	float QDF=a_clm_rec.directNormalSolar().value();
	float SALT=a_sun_pos.elevation();
	/* Stephan Boltmann Constant */
	float SBC=5.6697e-08;

	/* 0 degrees Centigrade in Kelvin. */
	float ZEROK=273.15;

	/* Evaluate the quantity of cloud cover. */
	float RCLOUD;
	if (SALT<0)
		CLOUD1=CLOUDC;
	else {
		if((QFF <= 0.0) || (QDF <= 0.0)) CLOUD1=CLOUDC;
		else {
			CLOUDC=(CLOUD1+CLOUD2+CLOUD3)/3.0;
			CLOUD2=CLOUD1;
			CLOUD3=CLOUD2;

			/* Check this part - need global etc. */
			float QT=QFF+QDF*sin(SALT*PI/180);
			float RADR=QFF/QT;
			if(RADR <= 0.0) {
				CLOUDC=0.0;
				RCLOUD=0.0;
			} else
				if (RADR > 1.0) {
					CLOUDC=1.0;
					CLOUD1=CLOUDC;
				} else {
					CLOUDC=RADR*RADR;
					CLOUD1=CLOUDC;
				}
		}
	}

	/* Atmospheric pressure (fixed, but should depend on site elevation). */
	float PATM=965.0;

	/* Compute the dew point temperature, TDP. */
	float PSAT=6.1078*exp(17.08085*TF/(234.175+TF));
	if(fabs(PSAT) < 1.E-6)   PSAT=1.e-6;
	float PREAL=HF/100*PSAT;

	/* For zero RF set dew point temperature to 293. */
	float TDP;
	if (fabs(PREAL) > 1e-6) 
		TDP=234.175*log(PREAL/6.1078)/(17.08085-log(PREAL/6.1078));
	else
		TDP=293.0;

	/* Clear sky condition. */
	float TIM=2*PI*IHRF/24;
	float E0=0.711+0.56*(TDP/100)+0.73*(pow((TDP/100),2.0))
     		+0.013*cos(TIM)+(12e-05)*(PATM-1000);

	/* Cloudy condition: assume cloud base is at 3km. */
	float CBH=3.0;
	float GAMMA=exp(CBH/(-8.2));

	/* Radiation from the atmosphere. */
	float Rsky=SBC*(E0+(1-E0)*CLOUDC*GAMMA)*(pow((TF+ZEROK),4));

	/* Assume the atmosphere is a black body emitter and
	   calculate the temperature of the sky. */
	float Tsky=pow((Rsky/SBC),0.25);
	DEBUG2 << " Cloud cover = " << CLOUDC << "\n";
	DEBUG2 << " Dewpoint temperature = " << TDP << "\n";
	if ((Tsky < 200.0) || (Tsky > 325.0)) {
		cout << "Berdahl_and_Martin a dubious sky \n";
		cout << "temperature has been calculated.\n";
	}
	return Temperature(Tsky);
};
