#include 	"EConnection.h"
#include 	"EComponent.h"

#include	"Metaclass.h"
#include	"Equation_set.h"
#include 	"Dimension.h"
#include 	"Temperature.h"
#include 	"Energy.h"
#include 	"Mixed_vector.h"

EConnection::EConnection(Metaclass* meta, EConnection_def* def)
			: Connection(meta, (Connection_def*) def)
{ 
	DEBUG << "EConnection::EConnection(" <<meta->oid() <<", " <<def->oid() ")\n";

	the_no_of_links = def->no_of_links();
	the_link_id = def->link_id();
	the_link_coef = def->link_coef();
	the_link_src_loc = def->link_pos_src();
	the_link_tgt_loc = def->link_pos_tgt();
};

EConnection::~EConnection()
{ 
	DEBUG << "EConnection::~EConnection()\n";

	delete the_link_id;
	delete the_link_coef;
	delete the_link_src_loc;
	delete the_link_tgt_loc;
};

EConnection& EConnection::operator=(EConnection& conns)
{
	DEBUG1 << "EConnection::operator=(" << conns->oid() << ")\n";

	if(this==&conns) return (*this);

	(void) Connection::operator=((Connection&) conns);

	the_no_of_links = conns.the_no_of_links;
	for(int i=0; i<the_no_of_links; i++) {
		the_link_id[i] = conns.the_link_id[i];
		the_link_coef[i] = conns.the_link_coef[i];
		the_link_src_loc[i] = conns.the_link_src_loc[i];
		the_link_tgt_loc[i] = conns.the_link_tgt_loc[i];
	}
	return (*this);
}

//for FLOW between boundary and internal nodes.
void	EConnection::modify_equation(Energy q, Equation* eq)
{
	eq->add_gain(q);
};

//for CONTACT between boundary and internal nodes
//or FLOW between internal nodes.
void	EConnection::modify_equation(Temperature ta, Dimension ha, Equation* eq)
{
	if (eq == NULLPTR(Equation)) {
		cerr << "EConnection::()modify_equation(): attempt to modify null eqn";
		exit(1);
	};

	SpecificHeatCapacity	cap = eq->capacity();
	ThermalResistance	res = eq->resistance().value();
	State_variable*		sv_of_eq = eq->state_variable();
	List_iterator(Coefficient) icoef = eq->coeff_iterator();
	Coefficient* 		coef;
	while (coef = icoef(FORWARD))
		if (sv_of_eq == coef->state_variable())
			break;

	float 		old_coeff = coef->value();
	float 		new_coeff = (ha/cap).value();
	float		modified_coeff = old_coeff - new_coeff;
	eq->change_coefficient(sv_of_eq, modified_coeff);
	Energy 		gain = ta*ha;
	eq->add_gain(gain);
};

//for contact between internal nodes.
void	EConnection::modify_equation(State_variable* sv, Dimension ha, Equation* eq)
{
	if (eq == NULLPTR(Equation)) {
		cerr << "EConnection::()modify_equation(): attempt to modify null eqn";
		exit(1);
	};
	if (sv == NULLPTR(State_variable)) {
		cerr << "EConnection::()modify_equation(): null state variable sent";
		exit(1);
	};

	float			ta = sv->state_variable();
	SpecificHeatCapacity	cap = eq->capacity();
	ThermalResistance	res = eq->resistance().value();
	State_variable*		sv_of_eq = eq->state_variable();
	List_iterator(Coefficient) icoef = eq->coeff_iterator();
	Coefficient* 		coef;
	while (coef = icoef(FORWARD))
		if (sv_of_eq == coef->state_variable())
			break;

	float 		old_coeff = coef->value();
	float 		new_coeff = (ha/cap).value();
	float		modified_coeff = old_coeff - new_coeff;
	eq->change_coefficient(sv_of_eq, modified_coeff);
	Energy 		gain = ta*ha;
	eq->append_coefficient(sv, new_coeff, TEMPERATURE);
};

void	EConnection::connect()
{ 
	DEBUG << "EConnection::connect()\n";

	EComponent*		src = (EComponent*)source();
	EComponent*		tgt = (EComponent*)target();

	for(int i=0; i<the_no_of_links; i++) {
		int 	tgt_loc = the_link_tgt_loc[i];
		int 	src_loc = the_link_src_loc[i];
		if(src->pid()==BOUNDARY && tgt->pid()==BOUNDARY) {
			cerr  <<"EConnection::connect() illegal connection: ";
			cerr  <<"attempt to link two boundary components\n";
			return;
		} else if(src->pid()==BOUNDARY && tgt->pid()==INTERNAL) {
			Mixed_vector 		src_vec = src->excitations();
			Equation_iterator 	itgt_eqs = tgt->equation_set().iterator();
			Temperature 		excitation = *src_vec[src_loc];
			modify_equation(excitation,
					the_link_coef[i],
					itgt_eqs[tgt_loc]);
		} else if(src->pid()==INTERNAL && tgt->pid()==INTERNAL) {
			Equation_iterator 	isrc_eqs = src->equation_set().iterator();
			Equation_iterator 	itgt_eqs = tgt->equation_set().iterator();
			if(the_link_id[i]==FLOW) {
				modify_equation(isrc_eqs[src_loc]->state_variable()->state_variable(),
						the_link_coef[i],
						itgt_eqs[tgt_loc]);
			} else if(the_link_id==CONTACT) {
				modify_equation(isrc_eqs[src_loc]->state_variable(),
						the_link_coef[i],
						itgt_eqs[tgt_loc]);
				modify_equation(itgt_eqs[tgt_loc]->state_variable(),
						the_link_coef[i],
						isrc_eqs[src_loc]);
			} else {
				cerr <<"EConnection::connect() illegal link type\n";
				return;
			}
		} else if(src->pid()==INTERNAL && tgt->pid()==BOUNDARY) {
			Equation_iterator 	isrc_eqs = src->equation_set().iterator();
			Mixed_vector 		tgt_vec  = tgt->excitations();
			if(the_link_id[i]==CONTACT) {
				modify_equation(*(tgt_vec[tgt_loc]),
						the_link_coef[i],
						isrc_eqs[src_loc]);
			}
		} else {
			cerr <<"EConnection::connect() illegal link position identified\n";
			return;
		};
	};
};
