#define		MAX(x,y) ( ((x)>(y))?(x):(y) )
#define		MIN(x,y) ( ((x)<(y))?(x):(y) )

#include "Ac_coil.h"

#include "EConnection.h"
#include "Metaclass.h"
#include "math.h"
#include "list.h"
#include "Temperature.h"
#include "Speed.h"

typedef	float	Velocity;

template_define(List,State_variable);

static	char*	class_name = "the_Ac_coil";
static	char*	class_type = "Ac_coil";

static	float	initial_temp[3] = { 15, 20, 10 };
static	List(State_variable)*	sv_list = new List(State_variable)();

int the_no_of_nodes;

void	Ac_coil::initialise()
{
		//initialise component system equation_set.
	for(int i=0; i<the_no_of_nodes; i++) {
		State_variable*	svp = new State_variable(this, initial_temp[i]);
		sv_list->append(svp);
	}
};

/*
			3 internal functions
*/

Heat_transfer_coeff	htc_water(Speed vw, Length dia, Temperature tw)
{
	float htc = 1400*(1+0.015*tw.value())*pow(vw.value(), 0.8)*pow(dia.value(), -0.2);
	return float(htc);
};

Heat_transfer_coeff	htc_air(Speed va)
{
	return 38*va.value();
};

Heat_transfer_coeff	htc_air_water(float cmin, float cmax, Area outside_area,
				ThermalResistance rair, ThermalResistance rwater,
				ThermalResistance rmetal )
{
	float ra = rair.value();
	float rw = rwater.value();
	float rm = rmetal.value();
	float ao = outside_area.value();

	float ntu = ao / (cmin * (ra+rm+rw));
	float c = cmin / cmax;

	float epslon = ( 1-exp(-ntu*(1-c)) ) / ( 1-c*exp(-ntu*(1-c)) );

	return cmin*epslon;
};


Ac_coil::Ac_coil(Metaclass* meta, Ac_coil_def* def)
					: EComponent(meta, (EComponent_def*) def),
					the_water("water"), the_air("air"), the_metal("metal")
{
	DEBUG << "Ac_coil::Ac_coil(" << meta->oid() << ", " << def->oid() << ")\n";

	char*	throwaway = class_name;

	name(def->unique_name());	//use unique_name
/**/	my_type = class_type;

	the_no_of_nodes = 3;

	the_outside_area = def->outside_area();
	the_inside_area = def->inside_area();
	the_coil_face_area = def->coil_face_area();
	the_total_mass = def->total_mass();
	the_water_mass = def->water_mass();
	the_tube_diameter = def->tube_diameter();
	the_metal_resistance = def->metal_resistance();

	initialise();
};

Ac_coil::~Ac_coil()
{
	DEBUG << "Ac_coil::~Ac_coil()\n";
};

void		Ac_coil::create_me(EConnection* caller)
{
	DEBUG << "Ac_coil::create_me(EConnection* caller)\n";

	//	workout Mass_flow_rate of water and air at run time by calling
	//	back to the EConnection.

	// temporalrily fixed to constants.

	Mass_flow_rate water_flow_rate(10.0);
	Mass_flow_rate air_flow_rate(5.0);

	create_me(water_flow_rate, air_flow_rate);
};

void		Ac_coil::create_me(Mass_flow_rate water_rate, Mass_flow_rate air_rate)
{
	DEBUG << "Ac_coil::create_me(Mass_flow_rate water_rate, Mass_flow_rate air_rate)\n";

	List_iterator(State_variable)	state_variable(sv_list);
	Equation_iterator	ieq = equation_set().iterator();

	SpecificHeatCapacity	capacity[3];
	capacity[0] = (total_mass()-water_mass()) * metal().specificHeatCapacity();
	capacity[1] = (0.01) * air().specificHeatCapacity();
	capacity[2] = water_mass() * water().specificHeatCapacity();

	float	htcs[3];
	float 	rho_a = (air().density() * coil_face_area()).value();
	ThermalResistance	ra = 1 / htc_air(air_rate/rho_a);
	htcs[0] = ( outside_area() / (ra + metal_resistance() / 2) ).value();

	rho_a = (water().density() * M_PI * tube_diameter() * tube_diameter() / 4).value();
	Temperature	tm = ( state_variable[1]->state_variable() +
				state_variable[2]->state_variable() ) /2;
	ThermalResistance	rw =  outside_area() / inside_area() /
				htc_water(water_rate/rho_a, tube_diameter(), tm);
	htcs[1] = ( outside_area() / (rw + metal_resistance() / 2) ).value();

	float	cw = ( water_rate * water().specificHeatCapacity() ).value();
	float	ca = ( air_rate * air().specificHeatCapacity() ).value();
	float cmax = MAX(cw, ca);
	float cmin = MIN(cw, ca);
	htcs[2] = htc_air_water(cmin, cmax, outside_area(), ra, rw, metal_resistance());

		//flush old equations.
	Equation*	eqp;
	while (eqp = ieq(FORWARD)) delete eqp;

		//building up 3 component equations.
	float coefficient = ( (htcs[0] + htcs[1]) / capacity[0] ).value();
	eqp = new Equation(state_variable[0], ENERGY, - coefficient, 0, 0, capacity[0]);
	coefficient = ( htcs[0] / capacity[0] ).value();
	eqp->append_coefficient(state_variable[1], coefficient, TEMPERATURE);
	coefficient = ( htcs[1] / capacity[1] ).value();
	eqp->append_coefficient(state_variable[2], coefficient, TEMPERATURE);
	equation_set().append(eqp);

	coefficient = ( (htcs[0] + htcs[2]) / capacity[1] ).value();
	eqp = new Equation(state_variable[1], ENERGY, - coefficient, 0, 0, capacity[1]);
	coefficient = ( htcs[0] / capacity[1] ).value();
	eqp->append_coefficient(state_variable[0], coefficient, TEMPERATURE);
	coefficient = ( htcs[2] / capacity[1] ).value();
	eqp->append_coefficient(state_variable[2], coefficient, TEMPERATURE);
	equation_set().append(eqp);

	coefficient = ( (htcs[1] + htcs[2]) / capacity[2] ).value();
	eqp = new Equation(state_variable[2], ENERGY, - coefficient, 0, 0, capacity[2]);
	coefficient = ( htcs[1] / capacity[2] ).value();
	eqp->append_coefficient(state_variable[0], coefficient, TEMPERATURE);
	coefficient = ( htcs[2] / capacity[2] ).value();
	eqp->append_coefficient(state_variable[1], coefficient, TEMPERATURE);
	equation_set().append(eqp);
};
