#include "Flow_network.h"
#include "Flow_network_def.h"
#include "Metaclass.h"
#include "Flow_node.h"
#include "Flow_arc.h"
#include  <math.h>

template_define(List,Component);
template_define(List,Flow_node);
template_define(List,Connection);
template_define(List,Flow_arc);

static	char*	class_name = "the_flow_network";
static	char*	class_type = "Flow_network";

// Utility functions.

Flow_network::Flow_network(Metaclass* meta, Flow_network_def* def)
					: System(meta, (System_def*) def), jacobi(0,0)
					
{
	DEBUG <<"Flow_network::Flow_network(" << meta->oid() << ", " << def->oid() << ")\n";

	char* hideaway = class_name;

	name(def->name());	//use unique_name in Connection_def
	my_type = class_type;

			//set parameters
	ferrel 	= def->relative_error();
	fermfl 	= def->absolute_error();
	maxitf 	= def->max_steps();
	pmax 	= def->max_p_correction();
	steffr 	= def->correction_bound();
	mslvtp 	= def->solver_type();

	the_flow_node_list = (List(Flow_node)*)&components();
	the_flow_arc_list = (List(Flow_arc)*)&connections();
	List_iterator(Flow_node) iter_node(the_flow_node_list);
	List_iterator(Flow_arc) iter_arc(the_flow_arc_list);
	no_of_nodes = iter_node.size();
	no_of_arcs = iter_arc.size();
	ok = 1;
	the_worst_node_abs = NULLPTR(Flow_node);
	the_worst_node_rel = NULLPTR(Flow_node);
	jacobi = *(new Matrix(no_of_nodes, no_of_nodes, 0));
};

Flow_network::~Flow_network()
{
	DEBUG << "Flow_network::~Flow_network()\n";

	delete the_worst_node_abs;
	delete the_worst_node_rel;
}

/****
Flow_network& Flow_network::operator=(Flow_network& net)
{
	DEBUG << "\nFlow_network::Flow_network(Flow_network& net)\n";

	if(this==&net) return (*this);

	(void) System::operator=((System&) a);

	the_flow_node_list = net.the_list_flow_node;
	the_flow_arc_list = net.the_list_flow_arc;
	no_of_nodes = net.no_of_nodes;
	no_of_arcs = net.no_of_arcs;
	ok = net.ok;
	the_worst_node_abs = net.the_worst_node_abs;
	the_worst_node_rel = net.the_worst_node_rel;
	jacobi = net.jacobi;
	//to be completed.
};
****/

List(Flow_node)* Flow_network::get_node_list()
{
	return the_flow_node_list;
};
	
List(Flow_arc)* Flow_network::get_arc_list()
{
	return the_flow_arc_list;
};

void Flow_network::add_node(Flow_node* n)
{
	/* When a node is added to the list,
	   no_of_nodes has to be re-evaluated.
	*/
};

void Flow_network::add_arc(Flow_arc* a)
{
	/* When aa arc is added to the list,
	   no_of_arcs has to be re-evaluated.
	*/
};

// Start Network physical functions.

void Flow_network::initialise()
{
/* First zeroise node residual, sum of flows and Jacobian matrix and
** set diagonal elements of the Jacobian matrix for known pressure nodes.
*/
	if(the_flow_node_list==NULL || no_of_nodes==0) {
		DEBUG <<"\n Bad Flow_node_list\n";
	}
	List_iterator(Flow_node) nl_iter(the_flow_node_list);
	for(int i=0; i<no_of_nodes; i++) {
		Flow_node* np = nl_iter[i];
		np->set_flow_coupled_sum(0);
		np->set_flow_residual(0);
		for(int j=0; j<no_of_nodes; j++)
			if(np->node_type()!=0 && i==j)
				jacobi[i][j] =  1;
			else
				jacobi[i][j] =  0;
	}
};

void Flow_network::arc_flow()
{

/* Start of connections fluid mass flow calculations. */
	if(the_flow_arc_list==NULL || no_of_arcs==0) {
		DEBUG <<"Bad Flow_arc_list" <<"\n";
	}
	List_iterator(Flow_arc) al_iter(the_flow_arc_list);
	for(int i=0; i<no_of_arcs; i++) {

			/* Set up node identifiers and connection type. */
		Flow_arc* ap = al_iter[i];
		// int component_type = ap->get_flow_type();

			/* Calculate, depending on flow component type, 
			** fluid mass flow(s) and partial derivative 
			** dFlow/d(dPressure).*/
		ap->arc_flow();
		float f1 = ap->get_flow1().value();
		float f2 = ap->get_flow2().value();
		float fr1 = ap->get_to_node()->get_flow_residual().value();
		float fr2 = ap->get_from_node()->get_flow_residual().value();
		float deriv = ap->arc_flow_derivative();
		float fc1 = ap->get_to_node()->get_flow_coupled_sum().value();
		float fc2 = ap->get_from_node()->get_flow_coupled_sum().value();

			/* Adjust fluid mass flow residual of connected nodes. */
		fr2 += -f1-f2;
		ap->get_from_node()->set_flow_residual(fr2);
		fr1 += f1+f2;
		ap->to_node->set_flow_residual(fr1);

			/* And their sum of absolute flows. */
		float fa = fabs(f1)+fabs(f2);
		fc2 += fa;
		ap->get_from_node()->set_flow_coupled_sum(fc2);
		fc1 += fa;
		ap->get_to_node()->set_flow_coupled_sum(fc1);
		
			/* Set up the elements of the Jacobian matrix. */
		int nt2 = ap->get_from_node()->node_type();
		int nt1 = ap->get_to_node()->node_type();
		int no2 = ap->get_from_node()->node_number();
		int no1 = ap->get_to_node()->node_number();
		if(nt2==0 && nt1==0) {
			float c = jacobi[no2][no2];
			c += deriv;
			jacobi[no2][no2] = c;
			c = jacobi[no2][no1];
			c -= deriv;
			jacobi[no2][no1] = c;
			c = jacobi[no1][no1];
			c += deriv;
			jacobi[no1][no1] = c;
			c = jacobi[no1][no2];
			c += deriv;
			jacobi[no1][no2] = c;
		} else if(nt2==0 && nt1!=0) {
			float c = jacobi[no2][no2];
			c += deriv;
			jacobi[no2][no2] = c;
			c = jacobi[no2][no1];
			c -= deriv;
			jacobi[no2][no1] = c;
		} else if(nt2!=0 && nt1==0) {
			float c = jacobi[no1][no1];
			c += deriv;
			jacobi[no1][no1] = c;
			c = jacobi[no1][no2];
			c += deriv;
			jacobi[no1][no2] = c;
		}
	}
}

int Flow_network::worst_node(int& iter)
{
	iter++;

	float resamx = 0;
	float resrmx = 0;
	float resr = 0;

	if(the_flow_node_list==NULL || no_of_nodes==0) {
		DEBUG <<"\n Bad Flow_node_list\n";
	}
	List_iterator(Flow_node) nl_iter(the_flow_node_list);
	for(int i=0; i<no_of_nodes; i++) {
		Flow_node* np = nl_iter[i];

			/* Internal nodes only. */
		if(np->node_type() > 1) continue;
		
			/* Calculate residual relative to sum of mass 
			** flow absolute values.*/
		float fr = np->get_flow_residual().value();
		float fc = np->get_flow_coupled_sum().value();
		if(fc > SMALL) resr = fr/fc;
		else resr = fr/SMALL;

			/* Update worst node values if this is the worst
			** node up to now. */
		if(fabs(resr) > fabs(resamx)) {
			the_worst_node_abs = np;
			resamx = fr;
		}
		if(fabs(resr) > fabs(resrmx)) {
			the_worst_node_rel = np;
			resrmx = resr;
		}
	}

		/* Iteration check:
	   	** If resrmx or flow_residule of the_worst_node_abs within 
	   	** tolerance, stop iteration. */
	float maxfr = the_worst_node_abs->get_flow_residual().value();
	if(fabs(resrmx) <= ferrel || fabs(maxfr) <= fermfl) return 1;
	if(iter >= maxitf) {
		ok = 0;
		return 1;
	} else
		return 0;
}

void Flow_network::solution()
{
	Matrix hjacobi(no_of_nodes, no_of_nodes, 0);
	hjacobi = jacobi;
	float* rhs = new float[no_of_nodes];
	float* prc = new float[no_of_nodes];
	int*   indx = new int[no_of_nodes];

	/* Set right hand side of matrix equation {PJMTX} {PRC} = {RESID}.
	** To avoid calculating a pressure correction for known pressure nodes
	** zeroise RHS elements for known pressure nodes.
	** In some cases it is possible that an unknown pressure node's diagonal
	** coefficient is equal to zero, e.g.:
	** - when there is no flow through its connections
	** - when the partial derivatives of its connections are effectively zero.
	** To avoid numerical problems when solving the matrix for these cases,
	** set the diagonal element to 1. and the RHS to 0. (ie. don't change
	** this node's pressure for the time being).
	*/

	if(the_flow_node_list==NULL || no_of_nodes==0) {
		DEBUG <<"\n Bad Flow_node_list\n";
	}
	List_iterator(Flow_node) nl_iter(the_flow_node_list);
	for(int i=0; i<no_of_nodes; i++) {
		Flow_node* np = nl_iter[i];
		int nn = np->node_number();
		if(np->node_type() == 0) {
			if(jacobi[nn][nn] != 0)
				rhs[nn] = np->get_flow_residual().value();
			else {
				jacobi[nn][nn] = 1;
				rhs[nn] = 0;
			}
		} else	rhs[nn] = 0;
	}

		/* Now solve {PRC} from matrix equation 
	   	** {PJMTX} {PRC} = {RHS}. */
	if(mslvtp == 1)
		jsolvd(jacobi, no_of_nodes, prc, rhs);
	else if(mslvtp == 2) {
		float d = 0;
		ludcmp(jacobi, no_of_nodes, indx, d);
		lubksb(jacobi, no_of_nodes, indx, rhs);
		for(int j=0; j<no_of_nodes; j++)
			prc[i] = rhs[i];
	} else
		cout <<"\n unresolvable error !\n";
	
		// assign prc to each node.
	for(i=0; i<no_of_nodes; i++) {
		Flow_node* np = nl_iter[i];
		int nn = np->node_number();
		Pressure p(prc[nn]);
		np->set_p_correction(p);
	}

		/* Dump {rhs} recalculated from {hjacobi}*{prc}. */
	for(i=0, rhs[i]=0; i<no_of_nodes; i++) 
		for(int j=0; j<no_of_nodes; j++)
			rhs[i] = hjacobi[i][j]*prc[j];
	
	for(i=0; i<no_of_nodes; i++)
		cout<<rhs[i]<<"\n";

	delete rhs, prc, indx;
}

void Flow_network::stefensen_optimisation()
{
		/* Revise pressure correction vector by Steffensen's
	   	** method to handle oscillating corrections.*/
	List_iterator(Flow_node) nl_iter(the_flow_node_list);
	float ratio = 0;
	for(int i=0; i<no_of_nodes; i++) {
		Flow_node* np = nl_iter[i];
		int nt = np->node_type();
		float r = np->get_relax();
		float prc = np->get_p_correction().value();
		float hprc = np->get_p_correction_previous().value();
		if(nt != 0) continue;
		if(r == 1) {
			if(hprc > SMALL)
				ratio = prc/hprc;
			else	ratio = prc/SMALL;
			if(ratio <= steffr && ratio != 1)
				r = 1/(1-ratio);
		} else	r = 1.0;

		hprc = prc;
		prc = prc * r;
		np->set_relax(r);
		Pressure p(prc);
		np->set_p_correction(p);
		Pressure hp(hprc);
		np->set_p_correction_previous(hp);
	}
}

void Flow_network::set_pressure()
{
		/* Adjust node pressure of node_type=0 nodes,
		** but limit correction to pmax. */
	List_iterator(Flow_node) nl_iter(the_flow_node_list);
	for(int i=0; i<no_of_nodes; i++) {
		Flow_node* np = nl_iter[i];
		int nt = np->node_type();
		if(nt != 0) continue;
		float prc = np->get_p_correction().value();
		if(fabs(prc) > pmax) {
			float r = np->get_relax();
			r *= (pmax/fabs(prc));
			if(prc > 0) prc = pmax;
			if(prc < 0) prc = -pmax;
		}
		float fp = np->get_current_pressure().value();
		fp += prc;
		Pressure p(fp);
		np->update_pressure(p);
	}
}

void Flow_network::simulate()
{
	int counter = 0;
	int bound_id = 0;

	while(!bound_id) {
		initialise();
		arc_flow();
		bound_id = worst_node(counter);
		if(bound_id == 1 && ok == 1) {
	 		cout<<"\nsuccessful\n";
			break;
		}
		if(bound_id == 1 && ok == 0) {
	 		cout<<"\nfailed    \n";
			break;
		}
		solution();
		stefensen_optimisation();
		set_pressure();
	}
}

void Flow_network::report()
{
	// to be completed.
}

/* JSOLVD solves the matrix equation {A} {X} = {B} based
** on Gaussian elimination with backsubstitution and no pivoting.
** An efficient matrix technique is used to forward reduce the
** N*N matrix A (physical size NP*NP) halfway, to a matrix whose
** components on the diagonal and above remain nontrivial.
** The solution vector X (size N) is then generated through
** backsubstitution of the known right hand side vector B (size N)
*/
void Flow_network::jsolvd(Matrix a, int n, float* x, float* b)
{
	float ab=0, ac=0, ad=0;
//coding note: index should start from 0.

		//Forward reduce matrix A.
	for(int i=2; i<=n; i++) {
		float f = a[1][i]/a[1][1];
		a[1][i] = f;
	}
	for(int k=2; k<=n; k++) {
		int k1 = k-1;
		for(int ik=k; ik<=n; ik++) {
			for(int j=1; j<=k1; j++) {
				ab = a[ik][j];
				ac = a[j][k];
				ad = ab * ac;
				float f = a[ik][k] - ad;
				a[ik][k] = f;
			}
		}
		int j1 = k+1;
		for(int j=j1; j<=n; j++) {
			for(int mj=1; mj<=k1; mj++) {
				ab = a[k][mj];
				ac = a[mj][j];
				ad = ab * ac;
				float f = a[k][j] - ad;
				a[k][j] = f;
			}
			float f = a[k][j] / a[k][k];
			a[k][j] = f;
		}
	}

		/* Conduct backward substitution to establish 
		** solution vextor X. */
	b[1] = b[1] / a[1][1];
	for(i=2; i<=n; i++) {
		int k1 = i - 1;
		for(int j=1; j<=k1; j++) {
			ab = b[j];
			ac = a[i][j]; 
			ad = ab * ac;
			b[i] = b[i] - ad;
		}
		b[i] = b[i] / a[i][i];
	}
	x[n] = b[n];
	int n1 = n - 1;
	for(int jj=1; jj<=n1; jj++) {
		i = n - jj;
		int ik = i + 1;
		for(int j=ik; j<=n; j++) {
			ab = a[i][j];
			ac = b[j];
			ad = ab * ac;
			b[i] = b[i] - ad;
		}
		x[i] = b[i];
	}
}

/* Given an N x N matrix A, with physical dimension NP, this routine
** replaces it by the LU decomposition of a row-wise permutation of
** itself. A and N are input. On output A is the LU decomposition of
** itself; INDX is an output vector which records the row permutation
** effected by the partial pivoting; D is output as +/- 1 depending
** on whether the number of row interchanges was even or odd, respect.
** This routine is used in combination with LUBKSB to solve linear
** equations or invert a matrix.

** See Press, W.H. et al. "Numerical Recipes. The Art of Scientific
** Computing", Cambridge University Press, Cambridge, 1988.
*/

#define NMAX 100

void Flow_network::ludcmp(Matrix a, int n, int*	indx, float d)
{
	float vv[NMAX], aamax, sum, dum;

	if(n > NMAX) {
		cout<<"\nlocal array VV too small !\n";
		exit(0);
	}
	
	d = 1;

		//Loop over rows to get implicit scaling information.
	for(int i=1; i<=n; i++) {
		aamax = 0;
		for(int j=1; j<=n; j++) {
			float f = a[i][j];
			if(fabs(f)>aamax) aamax = fabs(f);
		}
		if(aamax==0) {
			cout<<"\nLUDCMP: Singular matrix !\n";
			exit(0);
		}
		vv[i] = 1 / aamax;
	}

		// Start loop over columns of Crout's method.
	for(int j=1; j<=n; j++) {
		for(int i=1; i <= j-1; i++) {
			sum = a[i][j];
			for(int k=1; k<=i-1; k++)
				sum += a[i][k] * a[k][j];
			a[i][j] = sum;
		}

			// Start search for largest pivot element.
		int imax = 0;
		aamax = 0;
		for(i=j; i<=n; i++) {
			sum = a[i][j];
			for(int k=1; k<=j-1; k++)
				sum -= a[i][k] * a[k][j];
			a[i][j] = sum;
			float dum = vv[i] * fabs(sum);
			if(dum >= aamax) {
				imax = i;
				aamax = dum;
			}
		}

			// Perhaps interchange rows and scale factor.
		if(j!=imax) {
			for(int k=1; k<=n; k++) {
				dum = a[imax][k];
				a[imax][k] = a[j][k];
				a[j][k] = dum;
			}
			d = -d;
			vv[imax] = vv[j];
		}

		indx[j] = imax;
		if(a[j][j] == 0) a[j][j] = SMALL;
		if(j!=n) {
			dum = 1 / a[j][j];
			for(i=j+1; i<=n; i++) {
				float f = a[i][j] * dum;
				a[i][j] = f;
			}
		}
	}
}

/* This routine solves the set of N linear equations {A}{x}={b}.
** Here A is input, not as the matrix A but rather as its
** LU decompositions, determined by routine LUDCMP. INDX is
** input as the permutation vector returned by LUDCMP. B is
** input as the right-hand side vector B, and returns
** with the solution vector X. A, N, NP and INDX are not modified by
** this routine and can be left in place for successive calls with
** different right-hand sides B. This routine takes into account the
** possibility that B will begin with many zero elements, so it is
** efficient for use in matrix inversion.
*/
void Flow_network::lubksb(Matrix a, int n, int*	indx, float* b)
{
	float sum;

		//Start forward substitution.
	int ii = 0;
	for(int i=1; i<=n; i++) {
		int ll = indx[i];
		sum = b[ll];
		b[ll] = b[i];
		if(ii!=0)
			for(int j=ii; j<=i-1; j++)
				sum -= a[i][j] * b[j];
		else if(sum != 0) ii = i;
		else b[i] = sum;
	}

		// Start backsubstitution.
	for(i=n; i>=1; i--) {
		sum = b[i];
		if(i >= n)
			for(int j=i+1; j<=n; j++)
				sum -= a[i][j] *b [j];
		b[i] = sum / a[i][i];
	}
};

template_implement(List,Component);
template_implement(List,Flow_node);
template_implement(List,Connection);
template_implement(List,Flow_arc);
