#include	"transport/Equation.h"


Equation::Equation(State_variable* a_state_variable, Equation_enum a_equation_enum,
		 float self_coupling_coeff, Energy initial_gain,
		 ThermalResistance resist, SpecificHeatCapacity capac)
			: the_coefficients(), the_gain(initial_gain)
{
	DEBUG << "Equation::Equation(State_variable* a_state_variable, . . .)\n";

	the_state_variable = a_state_variable;
	the_equation_enum = a_equation_enum;

	Coefficient* coeff = new Coefficient(a_state_variable, self_coupling_coeff,
					     TEMPERATURE);
	the_coefficients.append(coeff);

	the_resistance = resist;
	the_capacity = capac;
};

Equation::Equation(Equation& e)  : the_state_variable(e.the_state_variable),
				   the_equation_enum(e.the_equation_enum),
				   the_coefficients(), the_gain(e.the_gain)
{
	DEBUG << "Equation::Equation(Equation& e)\n";

	/* Having created the coefficient list, we have to copy the
	 * coefficients from equation e into the_coefficients. */

	/* Get an iterator from the coefficient list in equation e. */
	List_iterator(Coefficient) iter(&(e.the_coefficients));
	Coefficient*	coefficient;

	/* Copy the coefficients and put them in the list. */
	while(coefficient = iter(FORWARD)) {
		the_coefficients.append(new Coefficient(*coefficient));
	};

	the_resistance = e.the_resistance;
	the_capacity = e.the_capacity;
}

Equation::~Equation()
{
	DEBUG << "Equation::~Equation()\n";
	
	/* Delete all the coefficients in this equation. */
	List_iterator(Coefficient) iter(&the_coefficients);
	Coefficient* coefficient;
	while(coefficient = iter(FORWARD))
		delete coefficient;
	the_coefficients.clear();
}

/*
** Copy the coefficient list by copying the coefficients from
** equation e into the coefficient list in the private data of
** "this" equation.
*/
Equation& Equation::operator=(Equation& e)
{
	DEBUG1 << "Equation& Equation::operator=(Equation& e)\n";

	if(this == &e) return *this;

	if (the_equation_enum != e.the_equation_enum) {
		cerr << "Equation::op=():  assigned eqn of different type!\n"
		     << "                  value not changed\n";
		return *this;
	};
	if (the_state_variable != e.the_state_variable) {
		cerr << "Equation::op=():  Warning!  assigned eqn "
		     << "is for a different state_variable\n"
		     << "                  value not changed\n";
		return *this;
	};

	/* Delete all existing coefficients in this equation. */
	List_iterator(Coefficient) iter(&the_coefficients);
	Coefficient* coefficient;
	while(coefficient = iter(FORWARD))
		delete coefficient;
	the_coefficients.clear();

	/* Get an iterator from the coefficient list in equation e. */
	iter = e.coeff_iterator();

	/* Copy the coefficients themselves and put them in the list */
	iter(RESET);
	while(coefficient = iter(FORWARD)) {
		the_coefficients.append(new Coefficient(*coefficient));
	}

	the_gain = e.the_gain;

	the_resistance = e.the_resistance;
	the_capacity = e.the_capacity;

	return *this;
}


/*
** returns the id of the node to which this equation refers
*/
State_variable* Equation::state_variable()
{
	DEBUG1 << "State_variable* Equation::state_variable()\n";

	return the_state_variable;
};

/*
** specifies the enum of equation e.g. energy
*/
Equation_enum Equation::equation_enum()
{
	DEBUG1 << "Equation_enum Equation::equation_enum()\n";

	return the_equation_enum;
};

Energy Equation::gain()
{
	DEBUG1 << "Energy Equation::gain()\n";

	return the_gain;
};

ThermalResistance Equation::resistance()
{
	DEBUG1 << "ThermalResistance Equation::resistance()\n";

	return the_resistance;
};

SpecificHeatCapacity Equation::capacity()
{
	DEBUG1 << "SpecificHeatCapacity Equation::capacity()\n";

	return the_capacity;
};


/*
** Append a coefficient by specifying its value, its state
** variable type and the node to which it refers.  This
** function creates the coefficient and adds it to the
** coefficient list the_coefficients.
*/
void Equation::append_coefficient(State_variable* a_state_variable, float a_value,
				State_variable_enum a_state)
{
	DEBUG1 << "void Equation::append_coefficient(State_variable* a_state_variable, float a_value,\n";
	DEBUG1 << "					State_variable_enum a_state)\n";

	Coefficient* coefficient = new Coefficient(a_state_variable, a_value, a_state);
	the_coefficients.append(coefficient);
}

/*
** Changes the value of an already existing coefficient.
** Perhaps a pointer to the Coefficient should be required
** instead of the integer position as at present.
** The State_variable* for the coefficient can not be changed.
*/
void Equation::change_coefficient(State_variable* a_state_variable, float a_value)
{
	DEBUG1 << "void Equation::change_coefficient(State_variable* a_state_variable, float a_value)\n";

	List_iterator(Coefficient) iter(&the_coefficients);
	Coefficient* coefficient;
	while (coefficient = iter(FORWARD))
		if(coefficient->state_variable() == a_state_variable) {
			// for efficiency, we've been made a friend of Coefficient
			// otherwise we would have had to ctor a new one & replace
			coefficient->the_value += a_value;
			break;
		};
	if (coefficient == NULLPTR(Coefficient))
		cerr << "Equation::change_coeff():	Warning!  attempt to "
		     << "change non existent coefficient\n"
		     << "                  value not changed\n";
}

/*
** Remove a coefficient as specified by its position.
** See previous comment.
*/
void Equation::remove_coefficient(State_variable* a_state_variable)
{
	DEBUG1 << "void Equation::remove_coefficient(int posCoefficient* coefficient)\n";

	List_iterator(Coefficient) iter(&the_coefficients);
	Coefficient* coefficient;
	while (coefficient = iter(FORWARD))
		if(coefficient->state_variable() == a_state_variable) {
			the_coefficients.remove(coefficient);
			delete coefficient;
			break;
		};
}

float Equation::coefficient_value(State_variable* a_state_variable)
{
	DEBUG1 << "float Equation::coefficient_value(State_variable* a_state_variable)\n";

	List_iterator(Coefficient) iter(&the_coefficients);
	Coefficient* coefficient;
	while (coefficient = iter(FORWARD))
		if(coefficient->state_variable() == a_state_variable)
			return coefficient->value();
	
};

void Equation::add_gain(Energy a_value)
{
	DEBUG1 << "void Equation::add_gain(Energy a_value))\n";

	the_gain = a_value/the_capacity;
};

ostream&	operator<<(ostream& s, Equation& e)
{
	Equation_enum	eqtype = e.the_equation_enum;

	s << (eqtype==ENERGY ? "Energy" : "unknown type")
	  << " eqn (sv_id " << e.the_state_variable->id()
	  << ")\n\t(gain " << e.the_gain
	  << ")\n\t(resist " << e.the_resistance
	  << "\t(l_capac " << e.the_capacity
	  << ")\n\t\t";
	List_iterator(Coefficient) coef_iter(&(e.the_coefficients));
	Coefficient* c;
	while (c = coef_iter(FORWARD))
		s << *c << "\t\t";
	s << "\n";

	return s;
};





/*
** Get the iterator for the list of coefficients.  This
** allows access to the list of coefficients without
** returning a pointer to the list itself.
*/
List_iterator(Coefficient)	Equation::coeff_iterator()
{
	DEBUG1 << "List_iterator(Coefficient)	Equation::get_iterator()\n";

	List_iterator(Coefficient)	iter(&the_coefficients);
	return iter;
}



template_implement(List,Coefficient);
